// IBM Open Class Library
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// (C) Copyright IBM Corporation, 1992, 1997 All Rights Reserved
//
#include <iikbhsh.h>
#include <iiexc.h>

#ifdef IC_PAGETUNE
  #define _IIKBHSH_CPP_
  #include <ipagetun.h>
#endif


#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (nocls, nocnd, nocns, nocnv, noext, nognr, novft)
#endif

#if defined (__SOM_ENABLED__) && defined (__RRBC_LIB__)
#pragma SOMDefine (IAKeyBagAsHshTableOps)
#endif

// ---------------------------
// IKeyBagAsHshTableCursorImpl
// ---------------------------

// public members

IKeyBagAsHshTableCursorImpl::
IKeyBagAsHshTableCursorImpl (IACollectionImpl const& collection)
: ICursorImpl (collection), ivNode (0)
{
}

IKeyBagAsHshTableCursorImpl::
IKeyBagAsHshTableCursorImpl
  (IKeyBagAsHshTableCursorImpl const& cursor)
: ICursorImpl (cursor), ivNode (cursor.ivNode)
{
}

IKeyBagAsHshTableCursorImpl::
~IKeyBagAsHshTableCursorImpl ()
{
}

ICursorImpl*
IKeyBagAsHshTableCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
IKeyBagAsHshTableCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

#if ! defined (__INO_MNGCURSOR__)
// ------------------------------
// IKeyBagAsHshTableMngCursorImpl
// ------------------------------

IAKeyBagAsHshTableNodeImpl const*const
IKeyBagAsHshTableMngCursorImpl::kIsInBetween =
  (IAKeyBagAsHshTableNodeImpl*)
   &IKeyBagAsHshTableMngCursorImpl::kIsInBetween;


// public members

IKeyBagAsHshTableMngCursorImpl::
IKeyBagAsHshTableMngCursorImpl (IKeyBagAsHshTableImpl const& collection)
: IKeyBagAsHshTableCursorImpl (collection),
  ivMngCollection ((IKeyBagAsHshTableImpl*)&collection)
{ ivNext = ivMngCollection->ivMngCursors->cursor;
  ivMngCollection->ivMngCursors->cursor = this;
}

IKeyBagAsHshTableMngCursorImpl::
~IKeyBagAsHshTableMngCursorImpl ()
{ Self*& first = ivMngCollection->ivMngCursors->cursor;
  if (this == first) {
     first = ivNext;
  }
  else {
    // find previous cursor
    for (Self* cursor = first;
         cursor->ivNext != this;
         cursor = cursor->ivNext);

    cursor->ivNext = ivNext;
  }
}

ICursorImpl*
IKeyBagAsHshTableMngCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
IKeyBagAsHshTableMngCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

bool
IKeyBagAsHshTableMngCursorImpl::
IsInBetween () const
{ return ivMngCollection->IsInBetween (*this);
}

bool
IKeyBagAsHshTableMngCursorImpl::
IsMngCursor () const
{ return true;
}

void
IKeyBagAsHshTableMngCursorImpl::
operator= (IKeyBagAsHshTableMngCursorImpl const& cursor)
{ Inherited::operator= (cursor);
  ivNextNode = cursor.ivNextNode;
  ivNextBucket = cursor.ivNextBucket;
  if (ivMngCollection != cursor.ivMngCollection) {

    // detach 'this' from current list of mngcursors
    Self*& first = ivMngCollection->ivMngCursors->cursor;
    if (this == first) {
       first = ivNext;
    }
    else {
      // find previous cursor
      for (Self* cursor = first;
           cursor->ivNext != this;
           cursor = cursor->ivNext);

      cursor->ivNext = ivNext;
    }


    // insert 'this' in new list of mngcursors
    ivMngCollection = cursor.ivMngCollection;
    ivNext = (cursor.ivMngCollection)->ivMngCursors->cursor;
    ivMngCollection->ivMngCursors->cursor = this;
  }
}
#endif

// ---------------------
// IKeyBagAsHshTableImpl
// ---------------------

char const*
IKeyBagAsHshTableImpl::cvClassName = "KeyBagAsHshTable";

// inline private members

inline IKeyBagAsHshTableImpl::Ops&
IKeyBagAsHshTableImpl::
OpsOf (IKeyBagAsHshTableImpl const& collection)
{ return (Ops&) Inherited::OpsOf (collection);
}

inline INumber&
IKeyBagAsHshTableImpl::
BucketOf (ICursorImpl const& cursor)
{ return ((Cursor&)cursor).ivBucket;
}

inline void*
IKeyBagAsHshTableImpl::
ElementOf (Node const* node) const
{
    size_t offset = 4;
    if (ivMngCursors)
        offset = ivMngCursors->offset;
    if (offset != 4 && offset != 8 && offset != 12 && offset != 16) {
        offset = 4;
    }
    size_t *elementPtr = (size_t *)((size_t)node + offset);
    size_t element = *elementPtr;

    return (OpsOf (*this).ivContainsDTSObjects ?
            (void *)element :
            (void *)elementPtr);
}

inline bool
IKeyBagAsHshTableImpl::
IsInBetween (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return (((Cursor const&)cursor).ivNode == MngCursor::kIsInBetween);
#else
  return false;
#endif
}

inline IKeyBagAsHshTableImpl::Node*&
IKeyBagAsHshTableImpl::
NodeOf (ICursorImpl const& cursor)
{ return ((Cursor&)cursor).ivNode;
}

inline INumber&
IKeyBagAsHshTableImpl::
NextBucketOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivNextBucket;
#else
  return ((Cursor&)cursor).ivBucket; // just a dummy
#endif
}

inline IKeyBagAsHshTableImpl::Node*&
IKeyBagAsHshTableImpl::
NextNodeOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivNextNode;
#else
  return ((Cursor&)cursor).ivNode; // just a dummy
#endif
}

// inline protected members

inline void
IKeyBagAsHshTableImpl::
Assign (void* element1, void const* element2) const
{ OpsOf (*this).Assign (element1, element2);
}

inline void*
IKeyBagAsHshTableImpl::
CreateNode (void const* element) const
{ return OpsOf (*this).CreateNode (element);
}

#if ! defined (__INO_STREAMING__)
inline void*
IKeyBagAsHshTableImpl::
CreateNodeWithStreamer (IDataStream& stream) const
{ return OpsOf (*this).CreateNodeWithStreamer (ivStreamer, stream);
}
#endif

inline void*
IKeyBagAsHshTableImpl::
CreateNodes (INumber n) const
{ return OpsOf (*this).CreateNodes (n, (size_t &)ivMngCursors->offset);
}

inline void
IKeyBagAsHshTableImpl::
DeleteNode (void* node) const
{ OpsOf (*this).DeleteNode (node);
}

inline void
IKeyBagAsHshTableImpl::
DeleteNodes (void* nodes) const
{ OpsOf (*this).DeleteNodes (nodes);
}

inline bool
IKeyBagAsHshTableImpl::
EqualKeys (void const* arg1, void const* arg2, IArgType argType) const
{ return OpsOf (*this).EqualKeys (arg1, arg2, argType);
}

inline unsigned long
IKeyBagAsHshTableImpl::
HashKey (void const* arg, unsigned long h, IArgType argType) const
{ return OpsOf (*this).HashKey (arg, h, argType);
}

// inline public members

#define inline

inline char const*
IKeyBagAsHshTableImpl::
ClassName () const
{ return cvClassName;
}

inline bool
IKeyBagAsHshTableImpl::
ContainsElementWithKey (void const* key) const
{ return Self::LocateElementWithKey (key, CursorOf (*this));
}

inline void*
IKeyBagAsHshTableImpl::
ElementAt (ICursorImpl const& cursor) const
{ IASSERT (NodeOf (cursor) != 0);
  return ElementOf (NodeOf (cursor));
}

inline bool
IKeyBagAsHshTableImpl::
IsBounded () const
{ return false;
}

inline bool
IKeyBagAsHshTableImpl::
IsConsistent () const
{ return true;
}

inline bool
IKeyBagAsHshTableImpl::
IsEmpty () const
{ return (ivNumberOfElements == 0);
}

inline bool
IKeyBagAsHshTableImpl::
IsFull () const
{ return false;
}

inline void*
IKeyBagAsHshTableImpl::
Key (void const* element) const
{ return OpsOf (*this).Key (element);
}

inline INumber
IKeyBagAsHshTableImpl::
NumberOfElements () const
{ return ivNumberOfElements;
}

inline void
IKeyBagAsHshTableImpl::
ReplaceAt (ICursorImpl const& cursor, void const* element)
{ IASSERT (NodeOf (cursor) != 0);
  Assign (Self::ElementAt (cursor), element);
}

#undef inline

// public members

IKeyBagAsHshTableImpl::
IKeyBagAsHshTableImpl (Ops& ops, INumber numberOfElements)
: IAKeyBagImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivTable (0), ivNumberOfBuckets (0), ivNumberOfElements (0)
{ 
  ivMngCursors = new MngOffsetCursor;
  ivMngCursors->cursor = 0;
  // Default offset
  typedef IKeyBagAsHshTableNodeImpl <long> TypedNode;
  TypedNode tmpNode(0);
  ivMngCursors->offset = (size_t)(&(tmpNode.ivElement)) - (size_t)(&tmpNode);
  Initialize (numberOfElements);
}

IKeyBagAsHshTableImpl::
IKeyBagAsHshTableImpl
  (Ops& ops, IKeyBagAsHshTableImpl const& collection)
: IAKeyBagImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivTable (0), ivNumberOfBuckets (0), ivNumberOfElements (0)
{
  ivMngCursors = new MngOffsetCursor;
  ivMngCursors->cursor = 0;
  ivMngCursors->offset = (collection.ivMngCursors)->offset;

  Initialize (collection.ivNumberOfBuckets);
  Self::AddAllFrom (collection);
#if ! defined (__INO_STREAMING__)
  ivStreamer = collection.ivStreamer;
#endif
}

IKeyBagAsHshTableImpl::
~IKeyBagAsHshTableImpl ()
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    cursor->Invalidate ();
  }
#endif
  if (ivTable != 0) {
    Self::RemoveAll ();
    DeleteNodes (ivTable);
  }
  delete ivMngCursors;
}

bool
IKeyBagAsHshTableImpl::
Add (void const* element, ICursorImpl& cursor)
{ if (ivNumberOfBuckets * 2 < Self::NumberOfElements ()) {
    GrowBy (1);
  }

  unsigned long hashValue =
    HashKey (element, ivNumberOfBuckets, kElement) % ivNumberOfBuckets;

  IASSERT (hashValue < ivNumberOfBuckets);

  Node* node = (Node*) CreateNode (element);

  //
  // link element in bucket
  //
  node->ivNext = ivTable [hashValue];
  ivTable [hashValue] = node;

  ivNumberOfElements++;

  //
  // Let cursor point to newly added element
  //
  BucketOf (cursor) = hashValue;
  NodeOf (cursor) = node;

  return true;
}

#if ! defined (__INO_STREAMING__)
void
IKeyBagAsHshTableImpl::
AddFromStream (IDataStream& fromWhere, ICursorImpl& cursor)
{ if (ivNumberOfBuckets * 2 < Self::NumberOfElements ()) {
    GrowBy (1);
  }

  StreamerNode* node = (StreamerNode*) CreateNodeWithStreamer (fromWhere);

  // Don't rely on node->ivElement because node's element is an long and does
  // not work if users use long long instead.

  unsigned long hashValue =
    HashKey((void *)((size_t)(node) + ivMngCursors->offset),
            ivNumberOfBuckets, kElement) % ivNumberOfBuckets;

  IASSERT (hashValue < ivNumberOfBuckets);

  //
  // link element in bucket
  //
  node->ivNext = ivTable [hashValue];
  ivTable [hashValue] = node;

  ivNumberOfElements++;

  //
  // Let cursor point to newly added element
  //
  BucketOf (cursor) = hashValue;
  NodeOf (cursor) = node;
}
#endif

void
IKeyBagAsHshTableImpl::
AddAllFrom (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ()) {
    Self const& kb = (Self const&)collection;
    // ivDummyCursor may not be initialized yet
    Cursor dummyCursor (*this);
    for (INumber i = 0; i < kb.ivNumberOfBuckets; i++) {
      Node* node = kb.ivTable [i];
      while (node != 0) {
        //
        // New element is added at the beginning of the bucket.
        //
        Self::Add (ElementOf (node), dummyCursor);
        node = node->ivNext;
      }
    }
  }
  else
    Inherited::AddAllFrom (collection);
}

bool
IKeyBagAsHshTableImpl::
AddOrReplaceElementWithKey (void const* element, ICursorImpl& cursor)
{ bool isContained = LocateElementWithKey (Key (element), cursor);

  if (isContained)
    Self::ReplaceAt (cursor, element);
  else
    Self::Add (element, cursor);

  return ! isContained;
}

bool
IKeyBagAsHshTableImpl::
AllElementsDo (IApplFunc applFunc, void* addArg)
{ if (Self::SetToFirst (CursorOf (*this)))
    while ((*applFunc) (Self::ElementAt (CursorOf (*this)), addArg) &&
           Self::SetToNext (CursorOf (*this)));

  return (NodeOf (CursorOf (*this)) == 0);
}

void*
IKeyBagAsHshTableImpl::
Any () const
{ Self::SetToFirst (CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

bool
IKeyBagAsHshTableImpl::
CheckCursor (ICursorImpl const& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    return true;
  }
  else
#endif
  {
    for (Self::SetToFirst (CursorOf (*this));
         NodeOf (CursorOf (*this)) != NodeOf (cursor);
         Self::SetToNext (CursorOf (*this)));

    return (NodeOf (CursorOf (*this)) != 0);
  }
}

bool
IKeyBagAsHshTableImpl::
CheckReplacement (ICursorImpl const& cursor, void const* element) const
{ return EqualKeys (Self::ElementAt (cursor), element, kElementElement);
}

IACollectionImpl*
IKeyBagAsHshTableImpl::
Clone () const
{ return (IACollectionImpl*) CheckPointer (OpsOf (*this).Clone ());
}

bool
IKeyBagAsHshTableImpl::
ContainsAllKeysFrom (IACollectionImpl const& collection) const
{ if (Self::ClassName () == collection.ClassName ()) {
    if (this != &collection) {
      Self const& kb = (Self const&)collection;
      ICursorImpl& cursor = (CursorOf (kb));
      if (kb.Self::SetToFirst (cursor))
        while (Self::ContainsElementWithKey
                 (Key (kb.Self::ElementAt (cursor)))
               && kb.Self::SetToNext (cursor));

      return (NodeOf (cursor) == 0);
    }
    else
      return true;
  }
  else
    return Inherited::ContainsAllKeysFrom (collection);
}

void
IKeyBagAsHshTableImpl::
Copy (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ())
    operator= ((Self const&)collection);
  else
    Inherited::Copy (collection);
}

IKeyBagAsHshTableImpl::Cursor*
IKeyBagAsHshTableImpl::
CreateCursor () const
{ return (Cursor*) CheckPointer (new Cursor (*this));
}

IKeyBagAsHshTableImpl::MngCursor*
IKeyBagAsHshTableImpl::
CreateMngCursor () const
{ return (MngCursor*) CheckPointer (new MngCursor (*(Self*)this));
}

void*
IKeyBagAsHshTableImpl::
ElementWithKey (void const* key) const
{ ICursorImpl& cursor = CursorOf (*this);
  ICHECK_ContainsKey (ClassName (), "ElementWithKey ()")
  return Self::ElementAt (cursor);
}

bool
IKeyBagAsHshTableImpl::
LocateElementWithKey (void const* key, ICursorImpl& cursor) const
{ BucketOf (cursor) =
    HashKey (key, ivNumberOfBuckets, kKey) % ivNumberOfBuckets;
  Node*& node = (NodeOf (cursor) = ivTable [BucketOf (cursor)]);

  while (! (node == 0 ||
            EqualKeys (ElementOf (node), key, kElementKey))) {
    //
    // while element available and key not equal to argument key,
    // try next one
    //
    node = node->ivNext;
  }

  return (node != 0);
}

bool
IKeyBagAsHshTableImpl::
LocateNextElementWithKey (void const* key, ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    NodeOf (cursor) = NextNodeOf (cursor);
    BucketOf (cursor) = NextBucketOf (cursor);

    // find previous element
    for (Node* previous = ivTable [BucketOf (cursor)];
         previous != NodeOf (cursor);
         previous = previous->ivNext);

    NodeOf (cursor) = previous;
  }
#endif

  Node*& node = NodeOf (cursor);

  IASSERT (node != 0);

  do {
    node = node->ivNext;
  } while (! (node == 0 ||
              EqualKeys (Self::ElementAt (cursor), key, kElementKey)));

  return (node != 0);
}

bool
IKeyBagAsHshTableImpl::
LocateOrAddElementWithKey (void const* element, ICursorImpl& cursor)
{ bool isContained = LocateElementWithKey (Key (element), cursor);

  if (! isContained)
     Self::Add (element, cursor);

  return isContained;
}

INumber
IKeyBagAsHshTableImpl::
NumberOfDifferentKeys () const
{ INumber result = 0;

  for (Self::SetToFirst (CursorOf (*this));
       NodeOf (CursorOf (*this)) != 0;
       Self::SetToNextWithDifferentKey (CursorOf (*this))) {
    result++;
  }

  return result;
}

INumber
IKeyBagAsHshTableImpl::
NumberOfElementsWithKey (void const* key) const
{ INumber n = 0;

  if (Self::LocateElementWithKey (key, CursorOf (*this))) {
    n = 1;
    while (Self::LocateNextElementWithKey (key, CursorOf (*this)))
      n++;
  }

  return n;
}

INumber
IKeyBagAsHshTableImpl::
RemoveAll ()
{ INumber n = Self::NumberOfElements ();
  Node *current,
       *next;
  for (INumber i = 0; i < ivNumberOfBuckets; i++) {
    current = ivTable [i];
    while (current != 0) {
      next = current->ivNext;
      DeleteNode (current);
      current = next;
    }
    ivTable [i] = 0;
  }
  ivNumberOfElements = 0;

#if ! defined (__INO_MNGCURSOR__)
   forMngCursors (cursor) {
     if (cursor->IsValid ()) {
       NodeOf(*cursor)          = (Node*) MngCursor::kIsInBetween;
       NextNodeOf (*cursor)     = 0;
     }
   }
#endif

  return n;
}

INumber
IKeyBagAsHshTableImpl::
RemoveAll (IPredFunc predFunc, void* addArg)
{ Cursor& current = (Cursor&) CursorOf (*this);
  Cursor cache (*this);
  INumber removed = 0;
  for (Self::SetToFirst (current), Self::SetToFirst (cache);
       NodeOf (current) != 0;
       current = cache) {
    Self::SetToNext (cache);
    if ((*predFunc) (Self::ElementAt (current), addArg)) {
      Self::RemoveAt (current);
      removed++;
    }
  }
  return removed;
}

INumber
IKeyBagAsHshTableImpl::
RemoveAllElementsWithKey (void const* key)
{ Cursor& current = (Cursor&) CursorOf (*this);
  Cursor cache (*this);
  bool removed = 0;

  unsigned long hashValue =
    HashKey (key, ivNumberOfBuckets, kKey) % ivNumberOfBuckets;

  IASSERT (hashValue < ivNumberOfBuckets);

  BucketOf (current) = hashValue;
  NodeOf (current) = ivTable [BucketOf (current)];

  if (NodeOf (current) != 0) {
    cache = current;
    do {
      Self::SetToNext (cache);
      if (EqualKeys (Self::ElementAt (current), key, kElementKey)) {
        Self::RemoveAt (current);
        removed++;
      }
      current = cache;
    }
    while (NodeOf (current) != 0 && BucketOf (current) == hashValue);
  }

  return removed;
}

void
IKeyBagAsHshTableImpl::
RemoveAt (ICursorImpl& cursor)
{ IASSERT (NodeOf (cursor) != 0);

  Node* nodeOfCursor = NodeOf (cursor);
  INumber bucketOfCursor = BucketOf (cursor);

#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor2) {
    if (nodeOfCursor == NextNodeOf (*cursor2)) {
      NodeOf (*cursor2) = NextNodeOf (*cursor2);
      SetInBetween (*cursor2);
    }
    else if (nodeOfCursor == NodeOf (*cursor2))
      SetInBetween (*cursor2);
  }
#endif

  if (nodeOfCursor == ivTable [bucketOfCursor])
    ivTable [bucketOfCursor] = nodeOfCursor->ivNext;
  else {
    Node* previous = ivTable [bucketOfCursor];
    while (previous->ivNext != nodeOfCursor) {
      //
      // find previous of element
      //
      previous = previous->ivNext;
      IASSERT (previous != 0);
    }

    previous->ivNext = nodeOfCursor->ivNext;
  }

  DeleteNode (nodeOfCursor);
  ivNumberOfElements--;

  if (! IsInBetween (cursor))
    cursor.Invalidate ();
}

bool
IKeyBagAsHshTableImpl::
RemoveElementWithKey (void const* key)
{ bool isContained =
    Self::LocateElementWithKey (key, CursorOf (*this));

  if (isContained) {
    Self::RemoveAt (CursorOf (*this));
  }

  return isContained;
}

bool
IKeyBagAsHshTableImpl::
ReplaceElementWithKey (void const* element, ICursorImpl& cursor)
{ bool isContained = LocateElementWithKey (Key (element), cursor);

  if (isContained)
    Self::ReplaceAt (cursor, element);

  return isContained;
}

bool
IKeyBagAsHshTableImpl::
SetToFirst (ICursorImpl& cursor) const
{ BucketOf (cursor) = 0;

  if (Self::IsEmpty()) {
    //
    // setToFirst will return very fast on an empty hashtable
    //
    NodeOf (cursor) = 0;
  }
  else {
    //
    // find first non-empty bucket
    //
    while (ivTable [BucketOf (cursor)] == 0 &&
           BucketOf (cursor) < ivNumberOfBuckets - 1) {
      BucketOf (cursor)++;
    }

    NodeOf (cursor) = ivTable [BucketOf (cursor)];
  }

  return (NodeOf (cursor) != 0);
}

bool
IKeyBagAsHshTableImpl::
SetToNext (ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    NodeOf (cursor) = NextNodeOf (cursor);
    BucketOf (cursor) = NextBucketOf (cursor);
    return (NodeOf (cursor) != 0);
  }
#endif

  Node*& node = NodeOf (cursor);

  IASSERT (node != 0);

  node = node->ivNext;
  if (node == 0) {
    //
    // end of the bucket,
    // try to find new non-empty bucket
    //
    while (BucketOf (cursor) < ivNumberOfBuckets - 1) {
      BucketOf (cursor)++;
      if (ivTable [BucketOf (cursor)] != 0) {
        node = ivTable [BucketOf (cursor)];
        break;
      }
    }
  }

  return (node != 0);
}

bool
IKeyBagAsHshTableImpl::
SetToNextWithDifferentKey (ICursorImpl& cursor) const
{ void const* key = Key (Self::ElementAt (cursor));

  do {
    Self::SetToNext (cursor);
  }
  while (NodeOf (cursor) != 0 &&
         EqualKeys (Self::ElementAt (cursor), key, kElementKey));

  return (NodeOf (cursor) != 0);
}

void
IKeyBagAsHshTableImpl::
operator= (IKeyBagAsHshTableImpl const& collection)
{ if (this != &collection) {
    Self::RemoveAll ();
    CopyFrom (collection);
#if ! defined (__INO_STREAMING__)
    ivStreamer = collection.ivStreamer;
#endif
  }
}

#if ! defined (__INO_STREAMING__)
void
IKeyBagAsHshTableImpl::
StreamIn (IDataStream& fromWhere)
{ ReadVersion (fromWhere);
  Self::RemoveAll ();
  INumber n;
  n <<= fromWhere;
  while (n--) {
    Self::AddFromStream (fromWhere, CursorOf (*this));
  }
}

void
IKeyBagAsHshTableImpl::
StreamOut (IDataStream& toWhere) const
{ WriteVersion (toWhere);
  Self::NumberOfElements () >>= toWhere;
  for (Self::SetToFirst (CursorOf (*this));
       CursorOf (*this).IsValid ();
       Self::SetToNext (CursorOf (*this))) {
    ivStreamer->streamOut (toWhere, Self::ElementAt (CursorOf (*this)));
  }
}

void
IKeyBagAsHshTableImpl::
AdoptStreamer (IACollectionStreamer *streamer)
{ if (streamer != ivStreamer.PtrOf())
    ivStreamer = IStreamerPtr(streamer);
}

IStreamerPtr&
IKeyBagAsHshTableImpl::
StreamerOf (IACollectionImpl const& collection)
{ return ivStreamer;
}
#endif

// private members

void
IKeyBagAsHshTableImpl::
CopyFrom (IKeyBagAsHshTableImpl const& collection)
{ IASSERT (Self::IsEmpty ());

  if (ivNumberOfBuckets == collection.ivNumberOfBuckets) {
    //
    // copy contents of hash table
    //
    for (INumber i = 0; i < ivNumberOfBuckets; i++) {
      Node* node;
      Node* thatNode;

      ivTable [i] = 0;

      thatNode = collection.ivTable [i];
      while (thatNode != 0) {

        ITRY {
          node = (Node*) CreateNode (ElementOf (thatNode));
        }
        IENDTRY

        ICATCH (IOutOfCollectionMemoryImpl exc) {
          for (++i; i < ivNumberOfBuckets; i++) {
            //
            // leave table in consistent state.
            //
            ivTable [i] = 0;
          }
          Self::RemoveAll();
          IRETHROW (exc);
        }
        IENDCATCH

        node->ivNext = ivTable [i];
        ivTable [i] = node;

        thatNode = thatNode->ivNext;
      }
    }
    ivNumberOfElements = collection.NumberOfElements ();
  }
  else {
    //
    // add per element,
    // hash value has to be computed for every element
    //
    Self::AddAllFrom (collection);
  }
}

INumber
IKeyBagAsHshTableImpl::
FindNearestPrime (INumber n)
{ n |= 0x01L;

  for (INumber i = 3; i * i <= n; i += 2) {
    if (n % i == 0) {
      n += 2;
      i = 1;
    }
  }
  return n;
}

void
IKeyBagAsHshTableImpl::
GrowBy (INumber)
{ INumber oldNumberOfBuckets = ivNumberOfBuckets;
  Node** oldTable = ivTable;

  ITRY {
    Initialize (ivNumberOfBuckets * 2);
  }
  IENDTRY

  ICATCH (IOutOfCollectionMemoryImpl exc) {
    ivTable = oldTable;
    ivNumberOfBuckets = oldNumberOfBuckets;
    IRETHROW (exc);
  }
  IENDCATCH

  for (INumber i = 0; i < oldNumberOfBuckets; i++) {
    while (oldTable [i] != 0) {
      Node* node = oldTable [i];
      oldTable [i] = node->ivNext;
      unsigned long hashValue =
        HashKey (ElementOf (node), ivNumberOfBuckets, kElement)
          % ivNumberOfBuckets;
#if ! defined (__INO_MNGCURSOR__)
      forMngCursors (cursor) {
        if (node == NodeOf(*cursor))
          BucketOf (*cursor) = hashValue;
        else if (node == NextNodeOf (*cursor))
          NextBucketOf (*cursor) = hashValue;
      }
#endif
      node->ivNext = ivTable [hashValue];
      ivTable [hashValue] = node;

      ivNumberOfElements++;
    }
  }

  DeleteNodes (oldTable);
}

void
IKeyBagAsHshTableImpl::
Initialize (INumber numberOfBuckets)
{ if (numberOfBuckets == 0)
    ivNumberOfBuckets = 1;
  else
    ivNumberOfBuckets = FindNearestPrime (numberOfBuckets);

  ivNumberOfElements = 0;
  ivTable = 0; // in case CreateNodes () fails

  ivTable = (Node**) CheckPointer (CreateNodes (ivNumberOfBuckets));
  for (INumber i = 0; i < ivNumberOfBuckets; i++) {
    ivTable [i] = 0;
  }
}

void
IKeyBagAsHshTableImpl::
SetInBetween (MngCursor& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (! IsInBetween (cursor)) {
    SetToNext (cursor);
    NextNodeOf (cursor) = NodeOf (cursor);
    NextBucketOf (cursor) = BucketOf (cursor);

    NodeOf (cursor) = (Node*) MngCursor::kIsInBetween;
  }
#endif
}

#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (restore)
#endif
