// ----------------------------------------------------------------------------
// FILE NAME: itrace.cpp
//
// DESCRIPTION:
//   This file contains the implementation of classes/functions declared
//   in itrace.hpp.
//
//
// COPYRIGHT:
//   IBM Open Class Library
//   Licensed Materials - Property of IBM
//
//   5645-001
//   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.
//
//   US Government Users Restricted Rights - Use, duplication, or
//   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
//
// Revision: 05  1.6.2.1  source/core/base/itrace.cpp, appsupport, ioc.v400, 001006  
// ----------------------------------------------------------------------------

#ifdef __MVS__
// Need _OPEN_THREADS for thread-safe version of USL
#include <ibhandle.hpp>
#endif // __MVS__

#include <itrace.hpp>
#include <ingthrd.hpp>

#ifdef IC_PMWIN
#define INCL_DOSSEMAPHORES
#define INCL_DOSINFOSEG
#define INCL_DOSPROCESS
#ifdef IC_WIN
#include <windows.h>
#endif
#ifdef IC_PM
#include <os2.h>
#endif
#endif

#include <istring.hpp>
#ifdef IC_WIN
#include <iplatfrm.hpp>
#endif


extern "C"
{
    #include <stdio.h>
    #include <stdlib.h>
    #include <string.h>
    #include <imprintf.h>

#ifdef IC_MOTIF
    #include <unistd.h>
    #include <sys/time.h>
    #include <sys/systemcfg.h>
#endif
}

#ifdef IC_SAFE_USL
// From USL Library
#include <irtllock.h>

// Defined in iexcbase.cpp for init
extern IRTLResource writeTrace_Lock;
#endif

#define IC_MAX_THREADS 255
int ITrace::iClState = ITrace::uninitialized;

#ifdef IC_PMWIN
ITrace::Destination ITrace::iClTraceLocation = ITrace::queue;
#endif

#ifndef IC_PMWIN
ITrace::Destination ITrace::iClTraceLocation = ITrace::standardOutput;
#endif

static unsigned ITrace__uClIndent[IC_MAX_THREADS] = {0};
static unsigned long ITrace__ulClProcessId =0;
static bool ITrace__fCheckMemory = false;
static bool ITrace__fCheckStack = false;
static const unsigned INDENT_LEVEL = 2;
#if defined(IC_WIN) || defined(IC_MOTIF) || defined(IC_PM)
static unsigned long indentThreadID[IC_MAX_THREADS] = {0};
#endif

#if defined (__SOM_ENABLED__)
#if defined (__RRBC_LIB__)
#pragma SOMAsDefault (on)
#else
#pragma SOMAsDefault (off)
#endif
#endif

class ITraceSetup
{
  public:
    static bool traceOn()
    {
        if (ITrace::iClState & ITrace::uninitialized)
            initialize();
        return (ITrace::iClState & ITrace::on);
    }
    static void initialize();
};

#ifdef IC_MOTIF
class ITracePrivateData
{
public:
    ITracePrivateData(const char *name) {
        function_name = new char[strlen(name) + 1];
        strcpy(function_name, name);
        read_real_time(&start_time, TIMEBASE_SZ);
    }
    ~ITracePrivateData() {
        if (function_name)
            delete function_name;
    }
    char *function_name;
    timebasestruct_t start_time;
};
#endif

#if defined (__SOM_ENABLED__)
#pragma SOMAsDefault(pop)
#endif


// Segment definitions
#ifdef IC_PAGETUNE
  #define _ITRACE_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| ITrace::ITrace                                                               |
|                                                                              |
| Default constructor for tracing.                                             |
------------------------------------------------------------------------------*/
ITrace :: ITrace(const char* pszTraceName,
                 long lLineNo)
        : pszClTraceName(0)
{
   if(ITraceSetup::traceOn())
   {
      if (pszTraceName!=0)
      {
#ifdef IC_MOTIF
          ITracePrivateData *data = new ITracePrivateData(pszTraceName);
          pszClTraceName = (char *)data;
#else
          pszClTraceName = (char*)pszTraceName;
#endif
          IString str(pszTraceName);
          if(isWriteLineNumberEnabled() && lLineNo>0)
          {
              str+=IString("(");
              str+=IString(lLineNo);
              str+=IString(")");
          }
          writeFormattedString(str, "+");
      }
      else
          pszClTraceName = 0;
   }
}


/*------------------------------------------------------------------------------
| ITrace::~ITrace                                                              |
|                                                                              |
| Write exit trace and flush the buffer.                                       |
------------------------------------------------------------------------------*/
ITrace :: ~ITrace()
{
    if(ITraceSetup::traceOn())
    {
        if (pszClTraceName!=0)
        {
#ifdef IC_MOTIF
            timebasestruct_t stop_time;
            read_real_time(&stop_time, TIMEBASE_SZ);
            
            ITracePrivateData *data = (ITracePrivateData *)pszClTraceName;

            time_base_to_time(&(data->start_time), TIMEBASE_SZ);
            time_base_to_time(&stop_time, TIMEBASE_SZ);
            
            int secs = stop_time.tb_high - (data->start_time).tb_high;
            int n_secs = stop_time.tb_low - (data->start_time).tb_low;

            /* If there was a carry from low-order to high-order during 
             * the measurement, we may have to undo it. 
             */
            if (n_secs < 0)  {
                secs--;
                n_secs += 1000000000;
            }

            IString outputString = IString(data->function_name);
            outputString += " [";
            outputString += IString(secs);
            outputString += " sec ";
            outputString += IString(n_secs);
            outputString += " nsec]";
            writeFormattedString(outputString,"-");
            delete data;
#else
            writeFormattedString(IString(pszClTraceName),"-");
#endif
        }
    }
}


/*------------------------------------------------------------------------------
| ITrace::write                                                                |
|                                                                              |
| Write an IString using writeFormattedString after adding the appropriate     |
| padding for indentation.                                                     |
------------------------------------------------------------------------------*/
void ITrace :: write(const IString& strString)
{
   if(ITraceSetup::traceOn())
   {
       writeFormattedString(strString, ">");
   }
#ifdef __DEBUG_ALLOC__
   if(ITrace__fCheckMemory)
     _heap_check();
#endif

}

/*------------------------------------------------------------------------------
| ITrace::write                                                                |
|                                                                              |
| Write a char array after adding the appropriate padding for indentation.     |
------------------------------------------------------------------------------*/
void ITrace::write(const char* pszText)
{
   write(IString(pszText));
}

/*------------------------------------------------------------------------------
| ITrace::writeFormattedString                                                 |
|                                                                              |
| Write a formatted IString using writeString.                                 |
------------------------------------------------------------------------------*/
void  ITrace :: writeFormattedString(const IString& strString,
                                     char* pszMarker)
{
#ifdef IC_SAFE_USL
   IRTLResourceLock aLock(writeTrace_Lock);
#endif // IC_SAFE_USL

   static unsigned long ulSequence = 0;

   unsigned long ulThreadId = threadId();
   int indentId;

#if defined(IC_WIN) || defined(IC_MOTIF) || defined(IC_PM)
   { 
       int index = ulThreadId % IC_MAX_THREADS;
       int count = IC_MAX_THREADS;
       while (count-- > 0 &&
              indentThreadID[index] != ulThreadId && indentThreadID[index] != 0)
       {
           if (++index >= IC_MAX_THREADS) index = 0;
       }
       if (count == 0)
           indentId = 0;
       else
       {
           indentThreadID[index] = ulThreadId;
           indentId = index;
       }
   }
#else
   indentId = ulThreadId;
#endif

   if(*pszMarker=='-' && ITrace__uClIndent[indentId] >= INDENT_LEVEL)
      ITrace__uClIndent[indentId]-=INDENT_LEVEL;


   IString strOut("");
   IString strPrefix("");
   strPrefix.insert(pszMarker, ITrace__uClIndent[indentId], ' ');

   if(isWritePrefixEnabled())
   {
      IString strThreadId = IString(ulThreadId).d2x().rightJustify(8, '0');
      IString strProcId   = IString(ITrace__ulClProcessId).rightJustify(6, '0');
      IString strLineNo   = IString(ulSequence).rightJustify(8, '0');
      if(ITrace__fCheckStack)
      {
        IString strStack =
                        IString(ITrace::remainingStack()).rightJustify(7, '0');
        strPrefix = IString(" ") + strStack + " " + strPrefix;
      }
      strPrefix = strLineNo + " " + strProcId + ":" + strThreadId + " " +
          strPrefix;
   }


   // copy the string and add the prefix info where a newline found
   strOut = IString::change(strString, '\n', "\n" + strPrefix);
   strOut = strPrefix+strOut+"\n";

   writeString((char*)strOut);

   ulSequence++;
   if(*pszMarker=='+')
      ITrace__uClIndent[indentId]+=INDENT_LEVEL;

}

/*------------------------------------------------------------------------------
| ITrace::threadId                                                             |
|                                                                              |
| Determine the threadId of the current thread                                 |
------------------------------------------------------------------------------*/
unsigned long ITrace::threadId()
{
#ifdef IC_PM
   unsigned long traceid;

   PTIB    ptib;      /*  Thread Information Block  */
   PPIB    ppib;      /*  Process Information Block */

   DosGetInfoBlocks(&ptib, &ppib);

   traceid = ptib->tib_ptib2->tib2_ultid;
   ITrace__ulClProcessId = ppib->pib_ulpid;
   return (traceid);
#endif

#ifdef IC_WIN
  ITrace__ulClProcessId = GetCurrentProcessId();
  unsigned long ourTid = GetCurrentThreadId();
  return (ourTid);
#endif

#ifdef IC_MOTIF
  ITrace__ulClProcessId = (unsigned long) getpid();
  unsigned long ourTid = INonGUIThread::currentId().asUnsigned();
  return (ourTid);
#endif 

#ifdef IC_400
  return 1;
#endif

#ifdef IC_MVS
   unsigned long traceid = 1;
   ITrace__ulClProcessId = 1;
   return (traceid);
#endif
}

/*------------------------------------------------------------------------------
| ITrace::remainingStack                                                       |
|                                                                              |
| Determine the size of the remaining stack of the current thread.             |
------------------------------------------------------------------------------*/
unsigned long ITrace::remainingStack()
{

#ifdef IC_PM
   PTIB    ptib;      /*  Thread Information Block  */
   PPIB    ppib;      /*  Process Information Block */
   DosGetInfoBlocks(&ptib, &ppib);
   void
     *base = ptib->tib_pstack,
     *cur  = &ptib;
   return (char*)cur - (char*)base;
#endif

#ifdef IC_WIN
   CONTEXT context;
   context.ContextFlags = CONTEXT_CONTROL;
   GetThreadContext(GetCurrentThread(), &context);
   void
     *base = (void *)context.Esp,
     *cur  = &context;
   return (char*)cur - (char*)base;
#endif

#ifdef IC_MVS
  return 0;
#endif

#ifdef IC_400
  return 0;
#endif

#ifdef IC_MOTIF
  return 0;
#endif
}

/*------------------------------------------------------------------------------
| ITrace::writeString                                                          |
|                                                                              |
| Write a text string to file.                                                 |
|                                                                              |
| Notes:                                                                       |
|                                                                              |
|   1) We cannot use cout or cerr. since these are C++ objects that may not    |
|      be initialized when this routine is called.                             |
|   2) For Unix, standardOutput and queue are treated identically.             |
------------------------------------------------------------------------------*/
void  ITrace :: writeString(char* pszString)
{

   switch(ITrace::iClTraceLocation)
   {
     case ITrace::standardError :
       {
#ifdef IC_WIN
         if (!IPlatform::isWinNT())
         {
           fprintf(stdout, "%s", pszString);
           fflush(stdout);
         }
         else
#endif
         {
           fprintf(stderr, "%s", pszString);
           fflush(stderr);
         }
         break;
       }

     case ITrace::standardOutput :
#if defined(IC_MVS) || defined(IC_MOTIF)
     case ITrace::queue :
#endif
       {
         fprintf(stdout, "%s", pszString);
         fflush(stdout);
         break;
       }

#if ( defined(IC_PM) || defined(IC_WIN) || defined(IC_400) )
     case ITrace::queue :
       {
         imprintf("%0.*s", IC_PRINTFMAXLEN, pszString);
         break;
       }
#endif

     case ITrace::file :
       {
         // check to see whether the environment variable
         // is defined
         char *envptr = getenv("ICLUI TRACEFILE");
         if (!envptr)
           envptr = getenv("ICLUI_TRACEFILE");

         // If (environment variable defined) {
         //   - strip the leading and trailing blanks from the value
         //   - use the value as filename
         //   - open the file
         //   - if (open successful) {
         //       - append the provided string the file
         //       - close the file
         //     } else {
         //       - do nothing
         //     }
         // } else {
         //   - do nothing
         // }
         if (envptr)
         {
           IString strTraceOutputFile(envptr);
           strTraceOutputFile.strip();

           FILE *fp = fopen(strTraceOutputFile, "a");
           if (fp)
           {
             fprintf(fp, "%s", pszString);
             fclose(fp);
           }
         }
         break;
       }
   }
}



/*------------------------------------------------------------------------------
| ITraceSetup::initialize                                                      |
|                                                                              |
| Initialize uses the following environment variables to set up the tracing    |
| options.                                                                     |
|                                                                              |
| Environment variables and possible values:                                   |
|   ICLUI TRACE                                                                |
|      ON                - turns tracing on                                    |
|      NOPREFIX          - no prefixes                                         |
|   ICLUI TRACETO                                                              |
|      STDERR | ERR      - standard error for tracing output                   |
|      STDOUT | OUT      - standard output for tracing output                  |
|      QUEUE  | PMPRINTF - tracing output to queue                             |
|   ICLUI CHECKMEMORY                                                          |
|      ON | TRUE         - check memory                                        |
|   ICLUI CHECKSTACK                                                           |
|      ON | TRUE         - check stack                                         |
------------------------------------------------------------------------------*/

void ITraceSetup::initialize()
{
   char *envptr;
   /******************************************************************/
   /* Reset uninitialized bit and setup trace defaults.  OR values   */
   /* in to prevent resetting other trace bits that may have been    */
   /* set PRIOR to initialization being called.                      */
   /******************************************************************/
   ITrace::iClState &= ~ITrace::uninitialized;
   ITrace::iClState |= ITrace::writeLineNumber | ITrace::writePrefix;

   /* Decide whether tracing is on */
   if (!(envptr = getenv("ICLUI TRACE")))
      envptr = getenv("ICLUI_TRACE");
   IString strTrace(envptr);
   strTrace.strip().upperCase();
#ifdef _MSC_VER
   if(operator==(strTrace, "ON"))
#else
   if(strTrace== "ON")
#endif
      ITrace::iClState |=ITrace::on;
#ifdef _MSC_VER
   else if(operator==(strTrace,"NOPREFIX"))
#else
   else if(strTrace == "NOPREFIX")
#endif
   {
      ITrace::iClState |=ITrace::on;
      ITrace::iClState &= ~ITrace::writePrefix;
   }

   /* decide where to send the output */
   if (!(envptr = getenv("ICLUI TRACETO")))
      envptr = getenv("ICLUI_TRACETO");
   IString strTraceTo(envptr);
   strTraceTo.strip().upperCase();

   if(strTraceTo.length() > 2)
   {
      /* Turn tracing on */
      ITrace::iClState |=ITrace::on;

      /* Setup output location; default is PM Queue  */
#ifdef _MSC_VER
      if(operator==(strTraceTo,"STDERR") || operator==(strTraceTo,"ERR"))
#else
      if(strTraceTo=="STDERR" || strTraceTo=="ERR")
#endif
        ITrace::iClTraceLocation =ITrace::standardError;
#ifdef _MSC_VER
      else if(operator==(strTraceTo,"STDOUT") || operator==(strTraceTo,"OUT"))
#else
      else if(strTraceTo=="STDOUT" || strTraceTo=="OUT")
#endif

        ITrace::iClTraceLocation=ITrace::standardOutput;
#ifdef _MSC_VER
      else if(operator==(strTraceTo,"FILE"))
#else
      else if(strTraceTo=="FILE")
#endif
        ITrace::iClTraceLocation=ITrace::file;
#ifdef _MSC_VER
      else if(operator==(strTraceTo,"QUEUE") || operator==(strTraceTo,"PMPRINTF"))
#else
      else if(strTraceTo=="QUEUE" || strTraceTo=="PMPRINTF")
#endif
        ITrace::iClTraceLocation=ITrace::queue;
      else
        ITrace::iClTraceLocation=ITrace::queue;
   }
#ifdef __DEBUG_ALLOC__
   /* decide whether to check memory */
   if (!(envptr = getenv("ICLUI CHECKMEMORY")))
      envptr = getenv("ICLUI_CHECKMEMORY");
   IString strCheckMemory(envptr);
   strTraceTo.strip().upperCase();
   if(strCheckMemory.length() > 1)
   {
      if(strCheckMemory=="ON" | strCheckMemory=="TRUE")
        ITrace__fCheckMemory = true;
   }
#endif

   /* decide whether to check stack usage */
   if (!(envptr = getenv("ICLUI CHECKSTACK")))
      envptr = getenv("ICLUI_CHECKSTACK");
   IString strCheckStack(envptr);
   strCheckStack.strip().upperCase();
   if(strCheckStack.length() > 1)
   {
#ifdef _MSC_VER
      if(operator==(strCheckStack,"ON") | operator==(strCheckStack,"TRUE"))
#else
      if(strCheckStack=="ON" | strCheckStack=="TRUE")
#endif
        ITrace__fCheckStack = true;
   }
}

// Note: The following are normally inlined but cannot be in the Windows environ
//       due to the variables they reference not being exported (Win32S private
#ifdef IC_WIN
void ITrace :: enableWriteLineNumber()
{
  ITrace::iClState |= ITrace::writeLineNumber;
}

void ITrace :: disableWriteLineNumber()

{
  ITrace::iClState &= ~ITrace::writeLineNumber;
}


bool ITrace :: isWriteLineNumberEnabled()
{
  return (ITrace::iClState & ITrace::writeLineNumber);
}


void ITrace::enableTrace()
{
  ITrace::iClState |= ITrace::on;
}

void ITrace::disableTrace()
{
  ITrace::iClState &= ~ITrace::on;
}

bool ITrace::isTraceEnabled()
{
  return (ITrace::iClState & ITrace::on);
}

void ITrace::writeToQueue()
{
 ITrace::iClTraceLocation = ITrace::queue;
}

void ITrace::writeToStandardError()
{
   ITrace::iClTraceLocation = ITrace::standardError;
}

void ITrace::writeToStandardOutput()
{
   ITrace::iClTraceLocation = ITrace::standardOutput;
}

void ITrace::writeToFile()
{
   ITrace::iClTraceLocation = ITrace::file;
}

ITrace::Destination ITrace::traceDestination()
{
   return(ITrace::iClTraceLocation);
}

void ITrace::enableWritePrefix()
{
  ITrace::iClState |= ITrace::writePrefix;
}

void ITrace::disableWritePrefix()
{
  ITrace::iClState &= ~ITrace::writePrefix;
}

bool ITrace::isWritePrefixEnabled()
{
  return (ITrace::iClState & ITrace::writePrefix);
}
#endif

void ITrace::writeDebugLocation(const char* str, const IExceptionLocation& location)
{
   IString text(str);
   text += IString("< file: ") + location.fileName();
   text += IString(" function: ") + location.functionName();
   text += IString(" line: ") + IString(location.lineNumber());
   ITrace::write(text);
}
