// ----------------------------------------------------------------------------
// FILE NAME: ireqproc.cpp
//
// DESCRIPTION:
//   This file contains the implementation of classes/functions declared
//   in ireqproc.hpp.
//
// COPYRIGHT:
//   IBM Open Class Library
//   (C) Copyright International Business Machines Corporation 1992, 1997
//   Licensed Material - Program-Property of IBM - All Rights Reserved.
//   US Government Users Restricted Rights - Use, duplication, or disclosure
//   restricted by GSA ADP Schedule Contract with IBM Corp.
//
// Revision: 68 1.9.2.2 source/core/base/ireqproc.cpp, notification, ioc.v400, 001006 
// ----------------------------------------------------------------------------

#include <ireqproc.hpp>
#include <ireslock.hpp>
#include <iconditn.hpp>
#include <ideque2.h>
#include <imaplst2.h>
#include <iexcept.hpp>
#include <ingthrd.hpp>
#include <iprimlck.hpp>
#include <istatics.hpp>

#ifdef IC_PAGETUNE
    #define _IREQPROC_CPP_
    #include <ipagetun.h>
#endif

class IObserverElement;

typedef IVPtrDeque<IRequest*> RequestDeque;
typedef IMapAsList<IObserverElement, IThreadId> ObserverMap;


// ----------------------------------------------------------------------------
class IObserverElement
{
public:
    IObserverElement(IRequestQueueObserver* observerToAlias, const IThreadId& threadId)
        :   fObserver(observerToAlias),
            fNewThreadId(threadId)
    {
    }

    IObserverElement(IObserverElement const& element)
        :   fObserver(element.fObserver),
            fNewThreadId(element.fNewThreadId)
    {
    }

    virtual             ~IObserverElement() {};

        // Required by collections
    IObserverElement&   operator=(const IObserverElement& element);
    bool                operator==(const IObserverElement& element) const;
    IObserverElement()
    {
    }

    bool operator<(const IObserverElement& element) const
    {
        return (this < &element);
    }

    IThreadId const&    getKey() const;

    IRequestQueueObserver* getObserver() const;

private:
    IRequestQueueObserver*  fObserver;
    IThreadId               fNewThreadId;
};

IObserverElement& IObserverElement::operator=(const IObserverElement& element)
{
    fObserver = element.fObserver;
    fNewThreadId = element.fNewThreadId;
        return *this;
}

bool IObserverElement::operator==(const IObserverElement& element) const
{
    return (fNewThreadId == element.fNewThreadId);
}

IThreadId const& IObserverElement::getKey() const
{
    return fNewThreadId;
}

IRequestQueueObserver* IObserverElement::getObserver() const
{
    return fObserver;
}

// global function required by collections
IThreadId const& key(IObserverElement const& element)
{
    return element.getKey();
}

// ----------------------------------------------------------------------------
class IRequestQueueImplementation
{
public:
    IRequestQueueImplementation();
    IRequestQueueImplementation(const IRequestQueueImplementation&);

    IRequestQueueImplementation& operator=(const IRequestQueueImplementation&) {return *this;};

    // Don't change relative order of these two fields -- the lock must be
    // initialized before the IPrivateCondition.
    IPrivateResource        fQueueLock;
    IPrivateCondition       fQueueNotEmpty;

    RequestDeque            fNormalPriorityQueue;
    RequestDeque            fHighPriorityQueue;

    bool                    fClosed;

    IRequestQueueObserver*  fObserver;

    static ObserverMap*     fgDefaultObservers;

//TDequeOf<IRequestExceptionHandler> fExceptionHandlers;
};

ObserverMap* IRequestQueueImplementation::fgDefaultObservers = 0;

static IPrivateResource& observersLock()
{
    static IPrivateResource* gObserversLock;

    if (!gObserversLock)
    {
        IPrimalLock lockInit;

        if (!gObserversLock) {
            gObserversLock = new IPrivateResource;
            adoptStaticObject(gObserversLock);
        }
    }
    return *gObserversLock;
}

IRequestQueueImplementation::IRequestQueueImplementation() :
    fQueueNotEmpty(),
    fClosed(false),
    fObserver(0)
//fExceptionHandlers()
{
}

IRequestQueueImplementation::IRequestQueueImplementation(const IRequestQueueImplementation&) :
    fQueueNotEmpty(),
    fClosed(false),
    fObserver(0)
//fExceptionHandlers()
{
}

// ----------------------------------------------------------------------------
IRequestHandler* IRequestQueueObserver::handler()
{
    return fHandler;
}

void IRequestQueueObserver::SetHandler(IRequestHandler* handler)
{
    fHandler = handler;
}

// ----------------------------------------------------------------------------
bool
IRequestProcessor::IsSameRequestProcessorAs(const IRequestProcessor& another) const
{
    return GetActualRequestProcessor() == another.GetActualRequestProcessor();
}


const IRequestProcessor* IRequestProcessor::GetActualRequestProcessor() const
{
    return this;
}


// ----------------------------------------------------------------------------
IRequestQueue::IRequestQueue() :

    IRequestProcessor(),
    fImplementation(new IRequestQueueImplementation())
{
}

IRequestQueue::IRequestQueue(IThreadId ourThread) :

    IRequestProcessor(),
    fImplementation(new IRequestQueueImplementation())
{
    {
        IResourceLock entry(observersLock());

        if (IRequestQueueImplementation::fgDefaultObservers)
        {
            if ( IRequestQueueImplementation::fgDefaultObservers->containsElementWithKey( ourThread ) )
            {
                fImplementation->fObserver
                    = IRequestQueueImplementation::fgDefaultObservers->elementWithKey( ourThread ).getObserver();
            }
        }
    }
}


IRequestQueue::~IRequestQueue()
{
    //
    // Note: If there any requests left in here to delete when
    // we execute this statement, then those requests are left
    // *unhandled*.  Within the semantics of this class, we are not able
    // to question the correctness or incorrectness of such a state.
    // What we *are* responsible for is properly managing the
    // storage of objects formally under our ownership; hence
    // this statement.
    //
    IRequest* nextRequest = 0;
    while (!fImplementation->fHighPriorityQueue.isEmpty())
    {
        IVPtrDeque<IRequest*>::Cursor cursor(fImplementation->fHighPriorityQueue);
        cursor.setToLast();
        nextRequest = fImplementation->fHighPriorityQueue.elementAt(cursor);
        fImplementation->fHighPriorityQueue.removeLast();
        delete nextRequest;
    }

    while (!fImplementation->fNormalPriorityQueue.isEmpty())
    {
        IVPtrDeque<IRequest*>::Cursor cursor( fImplementation->fNormalPriorityQueue);
        cursor.setToLast();
        nextRequest = fImplementation->fNormalPriorityQueue.elementAt(cursor);
        fImplementation->fNormalPriorityQueue.removeLast();
        delete nextRequest;
    }

    // fObserver was not adopted.
    delete fImplementation;
    fImplementation = 0;
}


//
// Called from an external thread to add a functor to the end of
// the normal priority request queue.  The requests are processed
// in FIFO order.
//
void IRequestQueue::AdoptRequest(IRequest* theRequest)
{
    AdoptRequest(theRequest, IRequestProcessor::kNormal);
}


// Called from an external thread to add a functor to the end of
// the request queue at a specific priority.  The requests are
// processed in FIFO order at each priority level, all kHighs are
// handled before any kNormals.
//
void IRequestQueue::AdoptRequest(IRequest* theRequest, ERequestPriority thePriority)
{

    IResourceLock entry(fImplementation->fQueueLock);

    bool wasEmpty = (fImplementation->fNormalPriorityQueue.isEmpty() &&
                     fImplementation->fHighPriorityQueue.isEmpty());

    if (thePriority == IRequestProcessor::kHigh)
        fImplementation->fHighPriorityQueue.addAsFirst(theRequest);
     else
        fImplementation->fNormalPriorityQueue.addAsFirst(theRequest);

    // Notify monitor - if we have one.
    if (fImplementation->fObserver)
        fImplementation->fObserver->ObserveNewRequest(*theRequest);

    // On empty => nonempty transition
    if (wasEmpty)
        fImplementation->fQueueNotEmpty.broadcast();
}


//
// Called from outside of the request queue/handlers to wait until all
// queues in an address space are blocking on input.  Returns when all
// queues are sleeping (immediately if they are all already sleeping)
//
void IRequestQueue::WaitUntilAllQueuesIdle()
{
}


//
// Called from the thread handling the requests, and typically called
// by the request handler.
//
IRequest* IRequestQueue::WaitAndOrphanNextRequest()
{
    IResourceLock entry(fImplementation->fQueueLock);

    InternalWaitForRequest(entry);
    return InternalOrphanNextPendingRequest();
}


void IRequestQueue::InternalWaitForRequest(IResourceLock& entry)
{
    while ( fImplementation->fNormalPriorityQueue.isEmpty() &&
            fImplementation->fHighPriorityQueue.isEmpty())
    {
        IASSERTSTATE(!fImplementation->fClosed);
        fImplementation->fQueueNotEmpty.wait(entry);
    }
}


IRequest* IRequestQueue::OrphanNextPendingRequest()
{
    IResourceLock entry(fImplementation->fQueueLock);

    return InternalOrphanNextPendingRequest();
}


IRequest* IRequestQueue::InternalOrphanNextPendingRequest()
{
    // Called within the scope of the fQueueLock lock.
    IRequest* nextRequest = 0;

    if (!fImplementation->fHighPriorityQueue.isEmpty())
    {
        IVPtrDeque<IRequest*>::Cursor cursor(fImplementation->fHighPriorityQueue);
        cursor.setToLast();
        nextRequest = fImplementation->fHighPriorityQueue.elementAt(cursor);
        fImplementation->fHighPriorityQueue.removeLast();
    }
     else
    {
        if (!fImplementation->fNormalPriorityQueue.isEmpty())
        {
            IVPtrDeque<IRequest*>::Cursor cursor(fImplementation->fNormalPriorityQueue);
            cursor.setToLast();
            nextRequest = fImplementation->fNormalPriorityQueue.elementAt(cursor);
            fImplementation->fNormalPriorityQueue.removeLast();
        }
    }

    if (nextRequest)
    {
        // Notify monitor - if we have one.
        if (fImplementation->fObserver)
            fImplementation->fObserver->ObserveNextRequest(*nextRequest);
    }

    return nextRequest;
}


bool IRequestQueue::IsEmpty()
{
    //
    // Should probably convert to maintaining an internal count, although
    // practically speaking, the queue size will always be small (way under
    // 20). Result can be stale immediately upon return
    //
    IResourceLock entry(fImplementation->fQueueLock);

    return ( fImplementation->fNormalPriorityQueue.isEmpty() &&
            fImplementation->fHighPriorityQueue.isEmpty());
}


void IRequestQueue::WaitForRequest()
{
    IResourceLock entry( fImplementation->fQueueLock );
    InternalWaitForRequest(entry);
}


void IRequestQueue::Close()
{
    IResourceLock entry( fImplementation->fQueueLock );

    fImplementation->fClosed = true;
    fImplementation->fQueueNotEmpty.broadcast();
}

IRequestQueueObserver* IRequestQueue::GetRequestQueueObserver()
{
    return fImplementation->fObserver;
}

IRequestQueueObserver*
IRequestQueue::SetRequestQueueObserver(IRequestQueueObserver* observerToAlias)
{
    IRequestQueueObserver* oldMonitor = fImplementation->fObserver;
    fImplementation->fObserver = observerToAlias;
    return oldMonitor;
}

IRequestQueueObserver*
IRequestQueue::SetDefaultRequestQueueObserver(IRequestQueueObserver* observerToAlias)
{
    IRequestQueueObserver* oldMonitor = 0;
    IThreadId ourThread = INonGUIThread::currentId();
    IObserverElement ourElement(observerToAlias, ourThread);
    {
        IResourceLock entry(observersLock());

        if (!IRequestQueueImplementation::fgDefaultObservers)
        {
            IRequestQueueImplementation::fgDefaultObservers = new ObserverMap();
            adoptStaticObject(IRequestQueueImplementation::fgDefaultObservers);
            IRequestQueueImplementation::fgDefaultObservers->add(ourElement);
        }
        else
        {
            if ( IRequestQueueImplementation::fgDefaultObservers->containsElementWithKey(ourThread))
            {
                oldMonitor = IRequestQueueImplementation::fgDefaultObservers->elementWithKey(ourThread).getObserver();
                IRequestQueueImplementation::fgDefaultObservers->replaceElementWithKey(ourElement);
            }
            else
            {
                IRequestQueueImplementation::fgDefaultObservers->add(ourElement);
            }
        }
    }
    return oldMonitor;
}

IRequestQueueObserver* IRequestQueue::defaultRequestQueueObserver()
{
    IRequestQueueObserver* oldMonitor = 0;
    IThreadId ourThread = INonGUIThread::currentId();
    {
        IResourceLock entry(observersLock());

        if (IRequestQueueImplementation::fgDefaultObservers)
        {
            if (IRequestQueueImplementation::fgDefaultObservers->containsElementWithKey(ourThread))
                oldMonitor = IRequestQueueImplementation::fgDefaultObservers->elementWithKey(ourThread).getObserver();
        }
    }
    return oldMonitor;
}

void
IRequestQueue::AddExceptionHandlerFirst(IRequestExceptionHandler* /* theHandler */)
{
//    fImplementation->fExceptionHandlers.AddFirst(theHandler);
}


void
IRequestQueue::AddExceptionHandlerLast(IRequestExceptionHandler* /* theHandler */)
{
//    fImplementation->fExceptionHandlers.AddLast(theHandler);
}

void
IRequestQueue::AddExceptionHandlerBefore(const IRequestExceptionHandler&, /* before */
                                         IRequestExceptionHandler* /* theHandler */)
{
//    fImplementation->fExceptionHandlers.AddBefore(before, theHandler);
}

void
IRequestQueue::AddExceptionHandlerAfter(const IRequestExceptionHandler&, /* after */
                                        IRequestExceptionHandler* /* theHandler */)
{
//    fImplementation->fExceptionHandlers.AddAfter(after, theHandler);
}

void
IRequestQueue::RemoveExceptionHandler(const IRequestExceptionHandler& /* theHandler */)
{
//    fImplementation->fExceptionHandlers.Remove(theHandler);
}

bool
IRequestQueue::HandleException(const IException& /* theException */)
{
    bool     handled = false;

/*
    TDequeOfIterator<IRequestExceptionHandler>    i( &fImplementation->fExceptionHandlers );
    for ( IRequestExceptionHandler* theHandler = i.First();
          theHandler != 0 && !handled;
          theHandler = i.Next() )
    {
        handled = theHandler->HandleException( theException );
    }
*/

    return handled;
}


// ----------------------------------------------------------------------------
// Implementation notes
//
// HandleRequest() is a private virtual method that is a bottleneck
// for all request handling.  It's not published publicly now but it's
// there if internal subclassing needs finds it useful to catch each
// request processing.
//
IRequestHandler::IRequestHandler(IRequestQueue* requestQueue) :
    fQueue( requestQueue )
{
    //IRequestQueueObserver* observer = IRequestQueue::defaultRequestQueueObserver();
    IRequestQueueObserver* observer = requestQueue->GetRequestQueueObserver();
    if (observer)
        observer->SetHandler(this);
}


IRequestHandler::~IRequestHandler()
{
}


void IRequestHandler::WaitAndDispatchNextRequest()
{
    HandleRequest( fQueue->WaitAndOrphanNextRequest() );
}


void IRequestHandler::DispatchNextPendingRequest()
{
    IRequest* request = fQueue->OrphanNextPendingRequest();
    if (request)
        HandleRequest(request);
}


// Dispatches all queued requests and then returns.
void IRequestHandler::DispatchAllPendingRequests()
{
    IRequest* request;
    while ((request = fQueue->OrphanNextPendingRequest()) != 0)
        HandleRequest(request);
}


// Operations provided by the request handler
bool IRequestHandler::IsQueueEmpty()
{
    return fQueue->IsEmpty();
}


IRequestQueue* IRequestHandler::GetRequestQueue()
{
    return ( fQueue );
}


void IRequestHandler::HandleRequest(IRequest* request)
{
    try
    {
        request->Do();
    }
    catch( IException& exception )
    {
        if ( !fQueue->HandleException(exception))
        {
            delete request;
            IRETHROW(exception);
        }
    }

#if 0
    catch ( ... )
    {
        // TBD: What low level service can log this.
        delete request;
//      throw;
    }
#endif

    delete request;
}
