// ----------------------------------------------------------------------------
// FILE NAME: ipoint.hpp
//
// DESCRIPTION:
//   Declaration of the classes:
//     IPair
//     IPoint
//     ISize
//     IRange
//
// COPYRIGHT:
//   IBM Open Class Library
//   Licensed Material - Property of IBM
//   (C) Copyright IBM Corp. 1992, 1997  All Rights Reserved
//
// Revision: 57 1.5.1.6 source/core/base/ipoint.hpp, appsupport, ioc.v400, 001006 
// ----------------------------------------------------------------------------

#ifndef _IPOINT_
#define _IPOINT_

#include <istream.hpp>
#include <ibase.hpp>
#include <istring.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

#pragma pack(push,4)
#pragma enum(4)

struct _POINTL;
struct _RECTL;
struct _SIZEL;
struct tagPOINT;
struct tagSIZE;
class  ISize;
class  IWindowHandle;

#ifdef IC_WIN
typedef struct tagSIZE SIZEL;
#endif
#ifdef IC_PM
typedef struct _SIZEL SIZEL;
#endif

#ifndef IC_BUILD
class IPair;
inline IPair operator+(const IPair& pair1, const IPair& pair2);
inline IPair operator*(const IPair& pair1, const IPair& pair2);
inline IPair operator*(const IPair& pair1, double multiplier);
inline IPair operator-(const IPair& pair1, const IPair& pair2);
inline IPair operator/(const IPair& pair1, const IPair& pair2);
inline IPair operator/(const IPair& pair1, double divisor);
inline IPair operator%(const IPair& pair1, const IPair& pair2);
inline IPair operator%(const IPair& aPair1, long divisor);
inline IPair transpose(const IPair& aPair);
inline ostream &operator<<(ostream& aStream, const IPair& aRectangle);
#endif

class IPair
{
public:
/*------------------------------ Related Types -------------------------------*/
typedef long Coord;

/*------------------------------ Constructors --------------------------------*/
  IPair ( );
  IPair ( Coord init );
  IPair ( Coord coord1,
          Coord coord2 );

/*-------------------------------- Accessors ---------------------------------*/
Coord
  coord1 ( )  const,
  coord2 ( )  const;

IPair
 &setCoord1 ( Coord coord1 ),
 &setCoord2 ( Coord coord2 );

/*----------------------------- Minimum and Maximum --------------------------*/
IPair
  minimum ( const IPair& aPair ) const,
  maximum ( const IPair& aPair ) const;

/*--------------------------------- Negation ---------------------------------*/
IPair
  operator - ( ) const;

/*--------------------------- Comparison Operators ---------------------------*/
bool
  operator == ( const IPair& aPair ) const,
  operator != ( const IPair& aPair ) const,
  operator <  ( const IPair& aPair ) const,
  operator >  ( const IPair& aPair ) const,
  operator <= ( const IPair& aPair ) const,
  operator >= ( const IPair& aPair ) const;

/*-------------------------- Manipulation Operators --------------------------*/
friend IPair
  operator +  ( const IPair& pair1,  const IPair& pair2       ),
  operator *  ( const IPair& pair1,  const IPair& pair2       ),
  operator *  ( const IPair& pair1,  double       multiplier  ),
  operator -  ( const IPair& pair1,  const IPair& pair2       ),
  operator /  ( const IPair& pair1,  const IPair& pair2       ),
  operator /  ( const IPair& pair1,  double       divisor     ),
  operator %  ( const IPair& pair1,  const IPair& pair2       ),
  operator %  ( const IPair& aPair1, long         divisor     );

IPair
 &operator += ( const IPair& aPair ),
 &operator -= ( const IPair& aPair ),
 &operator *= ( const IPair& aPair ),
 &operator *= ( double multiplier ),
 &operator /= ( const IPair& aPair ),
 &operator /= ( double divisor ),
 &operator %= ( const IPair& aPair ),
 &operator %= ( long  divisor );

/*-------------------------- Manipulation Functions --------------------------*/
IPair
 &scaleBy  ( double xFactor, double yFactor ),
  scaledBy ( double xFactor, double yFactor ) const;

/*------------------------------ Miscellaneous -------------------------------*/
double
  distanceFrom ( const IPair& aPair ) const;

long
  dotProduct ( const IPair& aPair ) const;

IPair
 &transpose ( );

friend IPair
  transpose ( const IPair& aPair );

/*-------------------------------- Conversions -------------------------------*/
IString
  asString ( ) const,
  asDebugInfo ( ) const;

/*-------------------------------- Displaying --------------------------------*/
friend ostream
 &operator << ( ostream&          aStream,
                const IPair&      aRectangle );

private:
/*--------------------------------- Private ----------------------------------*/
Coord
  coordCl1,
  coordCl2;
}; // class IPair

class IPoint : public IPair {
typedef IPair
  Inherited;

public:
/*------------------------------- Constructors -------------------------------*/
  IPoint ( );
  IPoint ( const IPair& pair );
  IPoint ( Coord x, Coord y );
#ifdef IC_PMWIN
  IPoint ( const struct _POINTL& ptl );
#endif
#ifdef IC_WIN
  IPoint ( const struct tagPOINT& ptl );
#endif

/*-------------------------------- Accessors ---------------------------------*/
Coord
  x      ( ) const,
  y      ( ) const;

IPoint
 &setX ( Coord X ),
 &setY ( Coord Y );

#ifdef IC_PMWIN
/*-------------------------------- Conversions -------------------------------*/
struct _POINTL
  asPOINTL ( ) const;
#endif // IC_PMWIN
};



class ISize  : public IPair
{
typedef IPair
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
  ISize ( );
  ISize ( const IPair& pair );
  ISize ( Coord width, Coord height );

#ifdef IC_PMWIN
  ISize ( const SIZEL& sizl );
  ISize ( const struct _RECTL& rcl );
#endif // IC_PMWIN

/*-------------------------------- Accessors ---------------------------------*/
Coord
  width  ( ) const,
  height ( ) const;

ISize
 &setWidth  ( Coord cx ),
 &setHeight ( Coord cy );

/*-------------------------------- Conversions -------------------------------*/
#ifdef IC_PMWIN
SIZEL
  asSIZEL ( ) const;
#endif // IC_PMWIN
};



class IRange : public IPair
{
typedef IPair
  Inherited;

public:
/*------------------------------- Constructors -------------------------------*/
  IRange ( );
  IRange ( const IPair& aPair );
  IRange ( Coord lower, Coord upper );

/*-------------------------------- Accessors ---------------------------------*/
Coord
  lowerBound ( ) const,
  upperBound ( ) const;

IRange
 &setUpperBound ( Coord upper ),
 &setLowerBound ( Coord lower );

/*--------------------------------- Testing ----------------------------------*/
bool
  includes ( Coord aValue ) const;
}; // class IRange

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

/*----------------------------- Inline Functions -----------------------------*/
#include <ipoint.inl>

#endif // _IPOINT_
