// ----------------------------------------------------------------------------
// FILE NAME: iobslist.cpp
//
// DESCRIPTION:
//   This file contains the implementation of classes/functions declared
//   in iobslist.hpp.
//
// COPYRIGHT:
//   IBM Open Class Library
//   (C) Copyright International Business Machines Corporation 1992, 1997
//   Licensed Material - Program-Property of IBM - All Rights Reserved.
//   US Government Users Restricted Rights - Use, duplication, or disclosure
//   restricted by GSA ADP Schedule Contract with IBM Corp.
//
// Revision: 48 1.6.2.1 source/core/base/iobslist.cpp, notification, ioc.v400, 001006 
// ----------------------------------------------------------------------------

#include <iobslist.hpp>
#include <iobservr.hpp>
#include <inotifev.hpp>
#include <iobsptr.hpp>
#include <iasynty.hpp>
#include <ireslock.hpp>
#include <iprimlck.hpp>
#include <iconditn.hpp>
#include <istatics.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
    #define _IOBSLIST_CPP_
    #include <ipagetun.h>
#endif

#include <ies2.h>


//
// Define the Collection of observer data
// Note:    We implement a collection of IObserverPtr objects since we must store
//          the observer unique data as well as the observer pointer. Most of the
//          functions in this class are a simple passthru to their ISequence
//          counterpart.
//
class IObserverPointers : public IEqualitySequence<IObserverPtr>
{
public:
    IObserverPointers()
       : IEqualitySequence<IObserverPtr>(50)
    {
    }

    ~IObserverPointers()
    {
    }
};


//
// Define the observer list extended data class.
// The cursor must be contained here so that when we declare it,
// the collection's nested Cursor class will have been declared.
// This is not the case in the hpp file.
//
class IObserverListData
{
public:
    IObserverListData(const IObserverList& observerList)
       : fCursor ( *(observerList.fObserverPointers)),
         fInUseThread(0),
         fListWaitCondition(0)
    {
    }

    ~IObserverListData()
    {
        if (fListWaitCondition)
            delete fListWaitCondition;
    }

    IPrivateCondition*        listWaitCondition();
    IObserverPointers::Cursor fCursor;
    IThreadId                 fInUseThread;
    IPrivateCondition*        fListWaitCondition;
};


/*------------------------------------------------------------------------------
| IObserverList::listWaitCondidion                                             |
------------------------------------------------------------------------------*/

IPrivateCondition* IObserverListData::listWaitCondition()
{
    // No race condition here because listWaitCondition is only called when
    // observerListKey() lock is held.
    if (! fListWaitCondition)
        fListWaitCondition = new IPrivateCondition;
    return fListWaitCondition;
}

// ----------------------------------------------------------------------------
//  Lazy evaluators for statics
// ----------------------------------------------------------------------------
static IPrivateResource& observerListKey()
{
    static IPrivateResource* gObserverListKey = 0;

    if (!gObserverListKey)
    {
        IPrimalLock lockInit;

        if (!gObserverListKey) {
            gObserverListKey = new IPrivateResource;
            adoptStaticObject(gObserverListKey);
        }
    }
    return *gObserverListKey;
}


/*------------------------------------------------------------------------------
| IObserverList::IObserverList                                                 |
------------------------------------------------------------------------------*/
IObserverList::IObserverList () :
    fObserverPointers(0),
    fNotifyDepth(0),
    fObserverListData(0)
{
    fObserverPointers = new IObserverPointers();
    fObserverListData = new IObserverListData ( *this );
}


/*------------------------------------------------------------------------------
| IObserverList::IObserverList {1}                                             |
------------------------------------------------------------------------------*/
IObserverList::IObserverList ( const IObserverList& obsList )
{
    // Check if this is correct
    fNotifyDepth = 0;

    fObserverPointers = new IObserverPointers(*obsList.fObserverPointers);
    fObserverListData = new IObserverListData ( *this );
}

/*------------------------------------------------------------------------------
| IObserverList::operator= {1}                                                 |
------------------------------------------------------------------------------*/
IObserverList& IObserverList::operator=(const IObserverList& obsList)
{
    if(this != &obsList) 
    {
        fNotifyDepth = 0;
        *fObserverPointers = *obsList.fObserverPointers;
    }
    return *this;
}

/*------------------------------------------------------------------------------
| IObserverList::~IObserverList                                                |
------------------------------------------------------------------------------*/
IObserverList::~IObserverList()
{
    delete fObserverListData;
    delete fObserverPointers;
}

/*------------------------------------------------------------------------------
| IObserverList::lockList                                                      |
------------------------------------------------------------------------------*/

void IObserverList::lockList()
{
    IResourceLock dispatchersLock(observerListKey());
    while ((fObserverListData->fInUseThread != IThreadId(0)) && 
           (fObserverListData->fInUseThread != INonGUIThread::currentId()))
        fObserverListData->listWaitCondition()->wait(dispatchersLock);
    fObserverListData->fInUseThread = INonGUIThread::currentId();
}

/*------------------------------------------------------------------------------
| IObserverList::unlockList                                                    |
------------------------------------------------------------------------------*/

void IObserverList::unlockList()
{
    if (fNotifyDepth == 0) {
        IResourceLock dispatchersLock(observerListKey());
        fObserverListData->fInUseThread = IThreadId(0);
        if (fObserverListData->fListWaitCondition)
            fObserverListData->fListWaitCondition->broadcast();
    }
}

/*------------------------------------------------------------------------------
| IObserverList::notifyObservers                                               |
|                                                                              |
| This function must only called on the thread on which the owning notifer     |
| was created.  Adds and removes may occur while we are in here.  Those are    |
| also run on the creation thread.  The unusual logic in here is looking for   |
| interactions with add and remove.  We could just use positions to run the    |
| list and avoid the unusual code, but cursors are faster.  The usual code     |
| makes sure the cursor stays defined correctly.                               |
|                                                                              |
| NOTE:  Any observers added during a call to dispatchNotificationEvent will   |
|        not be notified.  Only observers that existed at the time the         |
|        notification was generated are a candidate for notification.          |
|                                                                              |
| NOTE:  When certain functions are performed on a collection (e.g. add),      |
|        all cursors on that collection become undefined (not invalidated,     |
|        that is something different).  Therefore, using separate cursors for  |
|        each function does not gain anything with regard to reentrancy.  So,  |
|        we use a single cursor.  One cursor is easier to keep defined when    |
|        required than many cursors.                                           |
------------------------------------------------------------------------------*/
IObserverList& IObserverList::notifyObservers(const INotificationEvent& anEvent)
{
    unsigned long numberToRun = fObserverPointers->numberOfElements();

    if (numberToRun)
    {
        lockList();
            
        try {

            //
            // If we are winding to another depth of notification, save the current
            // position of the cursor, so it can be reset when we unwind.
            //
            IPosition oldPosition = 0;
            if (fNotifyDepth)
                oldPosition = fObserverPointers->position(fObserverListData->fCursor);
            
            // We are now notifying at some depth.
            fNotifyDepth++;
            
            //
            // Run the list calling the observers.  Using the number of elements
            // on entry to control iteration ensures that observers that are added
            // during our dispatches do not get this notification.
            //
            IPosition position;
            fObserverListData->fCursor.setToFirst();
            for (position = 1; position <= numberToRun; position++)
            {
                // If remove has not gotten rid of this observer, dispatch.
                IObserverPtr& obsptr = fObserverPointers->elementAt(fObserverListData->fCursor);
                if ( obsptr.obsrvr != 0 )
                {
                    ((INotificationEvent&)anEvent).setObserverData(obsptr.observerData);
                    
                    //
                    // Do not use obsptr after this call until next loop iteration.
                    // It could get set to 0 by remove during the dispatch.
                    //
                    obsptr.obsrvr->dispatchNotificationEvent(anEvent);
                }

                fObserverListData->fCursor.setToNext();
            }
            
            //
            //  We are now returning from this depth.  If we have unwound completely,
            //  remove the zeroed pointers from the list.
            //
            fNotifyDepth--;
            if (fNotifyDepth == 0)
            {
                position = 1;
                for (fObserverListData->fCursor.setToFirst(); fObserverListData->fCursor.isValid();)
                {
                    // If remove has not gotten rid of this observer, delete it.
                    IObserverPtr& obsptr = fObserverPointers->elementAt(fObserverListData->fCursor);
                    if (obsptr.obsrvr == 0)
                    {
                        fObserverPointers->removeAt(fObserverListData->fCursor);
                        if (position <= fObserverPointers->numberOfElements())
                        {
                            fObserverPointers->setToPosition(position,
                                                             fObserverListData->fCursor);
                        }
                    }
                    else
                    {
                        fObserverListData->fCursor.setToNext();
                        position++;
                    }
                }
            }
            else
            {
                // As we unwind, reset the cursor back to where it was before we started.
                fObserverPointers->setToPosition ( oldPosition,
                                                   fObserverListData->fCursor );
            }
        }
        catch (...) {
            unlockList();
            throw;
        }
        unlockList();
    }
  
    return *this;
}

/*------------------------------------------------------------------------------
| IObserverList::notifyObserversAsync
------------------------------------------------------------------------------*/
IObserverList& IObserverList::notifyObserversAsync(const INotificationEvent& anEvent)
{
    if (fObserverPointers->numberOfElements()) 
    {
        lockList();

        try {
            IObserverPointers::Cursor cursor(*fObserverPointers);
            for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
            {
                IObserverPtr& obsptr = fObserverPointers->elementAt(cursor);
                ((INotificationEvent&)anEvent).setObserverData(obsptr.observerData);
                
                IObserver*              pObserver = obsptr.obsrvr;
                INotificationProcessor* pRequestProcessor = pObserver->requestProcessor();
                
                //if same thread notification, then it is ok to set the observers request  processor
                if (pRequestProcessor)
                    pRequestProcessor->AdoptRequest (new INotificationRequest(*pObserver, anEvent));
                else
                    throw IException("IObserverList::notifyObserversAsync - No observer request processor exists for cross thread notification.");
            }
        }
        catch (...) {
            unlockList();
            throw;
        }

        unlockList();
    }
    
    return *this;
}


/*------------------------------------------------------------------------------
| IObserverList::add                                                           |
------------------------------------------------------------------------------*/
bool IObserverList::add ( IObserver& anObserver, void* observerData )
{
    bool result = false;

    lockList();
       
    try {
        // If we are wound inside notifyObservers, save the current
        // position of the cursor, so it can be reset on exit.
        IPosition oldPosition = 0;

        if (fNotifyDepth)
            oldPosition = fObserverPointers->position(fObserverListData->fCursor);
        
        // Do the add, if not already there
        IObserverPtr entry(&anObserver, observerData);
        if (!fObserverPointers->contains(entry))
            result = fObserverPointers->add(entry);
        
        // If wound, reset the cursor.
        if (fNotifyDepth)
            fObserverPointers->setToPosition(oldPosition, fObserverListData->fCursor);
    }
    catch (...) {
        unlockList();
        throw;
    }

    unlockList();
    return result;
}

/*------------------------------------------------------------------------------
| IObserverList::remove                                                        |
------------------------------------------------------------------------------*/
IObserverList& IObserverList::remove ( const IObserver& observer )
{
    lockList();
    
    try {
        //
        // When we are wound inside notifyObservers, this function does nothing
        // that will change fObserverListData->fCursor.  Instead of worrying about
        // fObserverListData->fCursor, this function uses its own cursor.
        //
        IObserverPointers::Cursor cursor(*fObserverPointers);
        for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
        {
            IObserverPtr& obsptr = fObserverPointers->elementAt(cursor);
            if (obsptr.obsrvr == &observer)
            {
                //
                // If we are wound inside notifyObservers, leave it to that
                // function to clean up the collection.  Otherwise, really
                // remove the observer.
                //
                if (fNotifyDepth)
                    obsptr.obsrvr = 0;
                else
                    fObserverPointers->removeAt(cursor);
                
                //
                // In either case, we have found the one we are looking for
                // and can leave the loop.
                //
                break;
            }
        }
    }
    catch (...) {
        unlockList();
        throw;
    }

    unlockList();
    return *this;
}

/*------------------------------------------------------------------------------
| IObserverList::removeAll                                                     |
------------------------------------------------------------------------------*/
IObserverList& IObserverList::removeAll()
{
    lockList();
    
    try {
        //
        // When we are wound inside notifyObservers, this function does nothing
        // that will change fObserverListData->fCursor.  Instead of worrying about
        // fObserverListData->fCursor, this function uses its own cursor.
        //
        
        //
        // If we are wound inside notifyObservers, leave it to that
        // function to clean up the collection.  Otherwise, really
        // remove all the observers.
        //
        if (fNotifyDepth)
        {
            IObserverPointers::Cursor cursor(*fObserverPointers);
            for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
            {
                IObserverPtr& obsptr = fObserverPointers->elementAt(cursor);
                obsptr.obsrvr = 0;
            }
        }
        else
        {
            fObserverPointers->removeAll();
        }
    }
    catch (...) {
        unlockList();
        throw;
    }

    unlockList();
    return *this;
}

bool IObserverList::isEmpty() const
{
    return fObserverPointers->isEmpty();
}
