/*NOSHIP*/
// ----------------------------------------------------------------------------
//  FILE NAME: ingthrdp.hpp
//
//  DESCRIPTION:
//   Declaration of the classes:
//      IStartedThread
//      IThreadsArrayDeleter
//
//  COPYRIGHT:
//   IBM Open Class Library
//   (C) Copyright International Business Machines Corporation 1992, 1997
//   Licensed Material - Program-Property of IBM - All Rights Reserved.
//   US Government Users Restricted Rights - Use, duplication, or disclosure
//   restricted by GSA ADP Schedule Contract with IBM Corp.
//
// Revision: 31 1.14.2.1 source/core/base/ingthrdp.hpp, oss, ioc.v400, 001006 
// ----------------------------------------------------------------------------

#ifndef _INGTHRDP_
#define _INGTHRDP_

#include <ibhandle.hpp>
#include <imrefcnt.hpp>
#include <ireslock.hpp>
#include <istring.hpp>
#include <ingthrd.hpp>
#include <iconditn.hpp>
#ifdef IC_WIN
    #include <windows.h>
#endif

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

class IThreadVariables;
class ITLSHandle;
class IStandardNotifier;
class IObserver;

#pragma enum(4)
#pragma pack(push,4)

//
// The IStartedThread class maintains information about started threads.
// An instance is allocated by IThread for each thread detected by IOC, and
// kept in a static array.
//
// If possible we will have only a single IStartedThread to serve both
// INonGUThread and IThread.  This means it will contain data needed by both.
// Data stored must be limited to the types found in ibase.  Do not bind
// code in the UI portion of the library.
//
class IStartedThread : public IMRefCounted
{
    typedef IMRefCounted Inherited;

public:
                            IStartedThread(const IThreadId& tid = IThreadId());
                            ~IStartedThread();

    IString                 asDebugInfo() const;
    bool                    isStarted();
    void                    end(unsigned long exitCode );
    unsigned long           exitCode() const;

#ifdef IC_PM
    tib_s*                  tib();
#endif

    IString                 variable (const IString& key) const;
    IStartedThread&         setVariable(const IString& key, const IString& value);

    static void             add(IStartedThread& aThread);
    static void             remove(IStartedThread& aThread);
    static void             initThread(IStartedThread* pThread);
    static IStartedThread*  locate(const IThreadId& tid );
    static bool             isTIDValid( const IThreadId& tid);
    static void             flushCache();
    static IStartedThread*  checkCacheFor( const IThreadId& tid);
    static void             cache(const IThreadId& tid, IStartedThread* pStarted);
    static void             watch();
    static IString          classDebugInfo();

    static IThreadId&       cachedTID();
    static IPrivateCondition& aThreadEnded();
    static IPrivateCondition& allThreadsEnded();
    static IThreadId&        idOfLastThreadRemoved();
    static ITLSHandle&       tlsSlot();

    enum WaitForEndOptions
    {
        waitSingle,         // Wait on single thread
        waitIOCOnly,        // Wait on IOC threads only
        waitAll             // Wait on all threads
    };

    static bool             waitForAllThreads(long timeout);
    static IThreadId        waitForAnyThread(long timeout);

    static unsigned long    fNumIOCRunningThreads;

#ifdef IC_WIN
    // creates a handle for current thread
    static IThreadHandle    currentHandle(bool create = true,
                                            IStartedThread* This = 0);
#endif

    static IThreadId        idWithHandle(const IThreadHandle& handle);
    static IThreadHandle    handleWithId(const IThreadId& tid);

#ifdef IC_WIN
    IThreadHandle handle() const
    {
        return fhandle;
    }
#endif

#ifdef IC_PM
    IThreadHandle handle() const
    {
        return id;
    }
#endif

#ifdef IC_POSIX
    //
    //  Default to windows version of IThreadHandle for POSIX. Not sure if its
    //  the right thing to do , but leave it for a while for the sake of testing.
    //
    IThreadHandle handle() const
    {
        return id;
    }
  // Utility member function for cleaning up system semaphores
  void removeUnixSemaphores();
#endif // IC_POSIX

    // --------------------------- Thread Local Storage -----------------------
    //  The functions localVariableFor and setLocalVariableFor provide a
    //  single void* slot of thread local storage for use by each IOC component.
    //  Access to this storage uses OS provided thread local storage facilities
    //  and is fast. A thread, however, cannot access the storage of another
    //  thread.  If such access is needed, use the INonGUIThread functions
    //  variable and setVariable. The LocalVariableIndex enumeration defines the
    //  components which have a slot. Each component is responsible for managing
    //  its own slot, including allocating and deleting storage.  Components can
    //  use IStartedThread::notifyOnCurrentThreadEnd to register themselves to be
    //  notified when a thread is ending.
    // ----------------------------------------------------------------------------
    enum LocalVariableIndex
    {
        base,
        ui,
        dde,
        multimedia,
        documentFramework,
        reservedMustBeLast
    };

    static void*        localVariableFor(LocalVariableIndex index );
    static void         setLocalVariableFor(LocalVariableIndex index, void* value);

    // Enables notification to the specified IObserver when the thread
    // is ending.
    static void         notifyOnCurrentThreadEnd( IObserver* observer );



    // ------------------------- Data Members ---------------------------------

    // TID.
    IThreadId id;

    // Left by ::exit method
    unsigned long fExitCode;

    // Started flag
    bool started;

    // Started-by-initThread flag
    bool startedHere;

    // In-threads-array flag
    bool inArray;

    // Used for wait operations
    IPrivateCondition fThreadEnded;

    #ifndef IC_POSIX
    // Creation order
    unsigned long fCount;
    #endif  // !IC_POSIX

    // Stack size
    unsigned long stkSize;

    //
    //  Thread priority class support is deprecated along with other
    //  non-portable priority methods. Only available under OS/2 for backwards
    //  compatibility and for storing the current priority class that the new
    //  levels might map to.
    //
    #ifdef IC_PM
    INonGUIApplication::PriorityClass priorityClass;
    #endif

    //
    //  Unfortunately the priority level is, under OS/2 still, overloaded to
    //  hold either the new EThreadPriority values or the old style 0..31
    //  values. Since these do not overlap, its is possible to know which
    //  style value it holds at any given time.
    //
    int priorityLevel;

    // Function object reference.
    IThreadFn*  fnObj;

    // Slot in threads-array.
    unsigned    slot;

    #ifdef IC_WIN
    // Thread handle (Same as id in PM)
    IThreadHandle   fhandle;
    #endif

    #ifdef IC_POSIX
    int                 fSemid;
    IPrivateResource    fSemaphoreMutex;
    #endif

    #ifdef IC_PM
    tib_s   *pTIB;          // Pointer to TIB structure.
    void    **ftlsPointer;  // pointer to thread local block for this thread
    #endif


    //
    //  Class for storage of data that is associated with the started thread.
    //  The data is stored as IStrings and can be set and retrieved through the
    //  functions setVariable and variable of IStartedThread and IThread.
    //
    IThreadVariables* vars;

    //
    // fNotifier can be used to send notifications when the IStartedThread
    // object is deleted.
    //
    IStandardNotifier* fNotifier;

    // Array of IStartedThread pointers.
    static IStartedThread** threads;

    // Size of threads array.
    static unsigned arraySize;

    // Cached ptr.
    static IStartedThread* cachedThread;

    // Watching for non-IThread terminations.
    static bool watching;

    static unsigned long fStartedThreadCount;
    static unsigned long cacheHits;
    static unsigned long cacheMisses;

    static IPrivateResource* pResource;
    static IPrivateResource& lock();

    // primary thread ID
    static IThreadId fgPrimaryTID;


#ifdef IC_WIN
    static unsigned long    tlsIndex;
#endif

private :
    // --------------------------- Hidden stuff -------------------------------
                            IStartedThread(const IStartedThread&);
    void                    operator=(const IStartedThread&);
};  //  End of class IStartedThread


//
// class IThreadsArrayDeleter
//       Class to manage the deletion of IStartedThread::threads
//       and clean-up of related resources at process termination.
class IThreadsArrayDeleter {
public:
  IThreadsArrayDeleter ( );
 ~IThreadsArrayDeleter ( );
}; // IThreadsArrayDeleter


#pragma pack(pop)
#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _INGTHRDP_
