// ----------------------------------------------------------------------------
// FILE NAME: ingthrd.cpp
//
// DESCRIPTION:
//   Implementation of the class(es):
//     IThreadFn
//     INonGUIThread
//     INonGUIThread::Cursor
//     ICurrentNonGUIThread
//     IStartedThread
//     IThreadsArrayDeleter
//     IThreadVariable
//     IThreadVariables
//     ITLSHandle
//
//
// COPYRIGHT:
//   IBM Open Class Library
//   Licensed Materials - Property of IBM
//
//   5645-001
//   (C) Copyright IBM Corporation 1992, 1997
//
//   US Government Users Restricted Rights - Use, duplication, or
//   disclosure restricted by GSA ADP Schedule Contract with IBM
//   Corp.
//
// Revision: 28 1.50.3.7 source/core/base/ingthrd.cpp, oss, ioc.v400, 001006 
//-----------------------------------------------------------------------------

#ifdef IC_POSIX
extern "C"
{
#include <pthread.h>
}
#endif

#include <ilanglvl.hpp>
#include <ibase.hpp>
#include <iprimlck.hpp>
#include <ingthrd.hpp>

extern "C"
{
#ifdef IC_POSIX
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/time.h>
#include <signal.h>
#ifdef IC_SUN
#include <sys/procfs.h>          // Needed for prpsinfo_t
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <ipfx.h>

#else

#include <stdlib.h>
#ifdef IC_PMWIN
#define INCL_DOSPROCESS
#define INCL_DOSSESMGR
#define INCL_DOSERRORS
#define INCL_DOSMISC
#define INCL_WINMESSAGEMGR
#define INCL_WINFRAMEMGR
#define INCL_WINHOOKS
#ifdef IC_WIN
#include <windows.h>
#endif
#ifdef IC_PM
#include <os2.h>
#endif
#endif // IC_PMWIN
#endif
}

#ifdef IC_POSIX
typedef unsigned long TID;
#endif // IC_POSIX

#include <ingthrdp.hpp>
#include <ingthrdv.hpp>
#include <icconsta.h>
#include <iexcept.hpp>
#include <ireslock.hpp>
#include <iobservr.hpp>
#include <istatics.hpp>
#include <istdntfy.hpp>
#include <istring.hpp>
#include <itrace.hpp>

#ifdef IC_WIN
#include <float.h>
#include <windows.h>
#endif

#ifndef IC_POSIX
#ifdef IC_WIN
#include <iplatfrm.hpp>
#endif  // IC_WIN
#include <iss2.h>
#endif

#ifdef IC_POSIX
#include <ipcondtn.hpp>
#include "del4cngt.hpp"
#endif

#ifdef IC_WIN
// Adjust a 0 centered value (-15,-2..2,15) to a 0..31 range
#define IC_THREAD_PRIORITY_ADJUST 15
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
#define _INGTHRD_CPP_
#include <ipagetun.h>
#endif

#ifdef IC_WIN
extern void __IThreadLocalBase__freeTlsIndex();
extern "C" void _Optlink __IInitTerm(unsigned long ulflag);
#endif

#ifdef IC_PM
// ----------------------------------------------------------------------------
//                      DLL termination thread cleanup
// ----------------------------------------------------------------------------
class IThreadWatcher {
public:
  IThreadWatcher ( )
  {
    ITRACE_DEVELOP( "IThreadWatcher constructor" );
  }
 ~IThreadWatcher ( )
  {
    ITRACE_DEVELOP( "IThreadWatcher destructor" );
    IThreadWatcher::threadWatcher_Terminate();
  }
static TID
  fgTid;
static void
  threadWatcher_Terminate ( );
}; // IThreadWatcher

void IThreadWatcher::threadWatcher_Terminate ( )
{
  ITRACE_DEVELOP( "IThreadWatcher terminating" );
  if ( IStartedThread::watching == true )
  {
     ITRACE_DEVELOP( IString( "Killing thread: " ) + IString( fgTid ) );
     DosKillThread( IThreadWatcher::fgTid );
     IStartedThread::watching = false;
  }
}

// This function is called by _DLL_InitTerm in core/dllinit/dde4cdll.cpp
// when the core DLL is terminating.
extern "C" void threadWatcher_Terminate ( )
{
  IThreadWatcher::threadWatcher_Terminate();
}

TID IThreadWatcher::fgTid = 0;
#endif // IC_PM


// ----------------------------------------------------------------------------
//  Static data
//
//  These are all lazily gen'ed up by the gDoStaticInit() method. The accessor
//  methods of these objects make sure that static init is done before using
//  them, otherwise they call gDoStaticInit().
// ----------------------------------------------------------------------------
static bool                 gStaticInitDone = false;
static IPrivateCondition*   gAThreadEnded = 0;
static IPrivateCondition*   gAllThreadsEnded = 0;
static IThreadId*           gCachedTID = 0;
static IThreadId*           gIdOfLastThreadRemoved = 0;
static ITLSHandle*          gTLSSlot = 0;

static IThreadsArrayDeleter* gThreadsArrayDeleter;
   // For clean up of the static IStartedThread::threads.

static ICurrentNonGUIThread* pCurrent = 0;
#ifdef IC_PM
static IThreadWatcher        gThreadWatcher;
#endif


// ----------------------------------------------------------------------------
//  Static Members
// ----------------------------------------------------------------------------

unsigned IStartedThread::arraySize = 0;

#ifdef IC_POSIX
unsigned long INonGUIThread::dfltStackSize = PTHREAD_STACK_MIN;
bool INonGUIThread::initInProgress = false;
#else
unsigned long INonGUIThread::dfltStackSize = 32768;
#endif


IStartedThread** IStartedThread::threads = 0;

IStartedThread* IStartedThread::cachedThread = 0;

bool IStartedThread::watching = false;

unsigned long IStartedThread::fStartedThreadCount = 0;
unsigned long IStartedThread::cacheHits = 0;
unsigned long IStartedThread::cacheMisses = 0;

IPrivateResource* IStartedThread::pResource = 0;

#ifdef IC_POSIX
IThreadId IStartedThread::fgPrimaryTID((unsigned long)pthread_self());
#endif

#ifdef IC_PM
IThreadId IStartedThread::fgPrimaryTID(1);
#endif

#ifdef IC_WIN
//
// We rely on static initialization of non thread local statics
// occurring in the context of the primary thread to initialize the following:
//
IThreadId IStartedThread::fgPrimaryTID(GetCurrentThreadId() );

unsigned long dummyStack = INonGUIThread::current().stackSize();

unsigned long IStartedThread::tlsIndex = (unsigned long)-1;

#endif

unsigned long IStartedThread::fNumIOCRunningThreads = 0;

#ifdef IC_WIN


/*------------------------------------------------------------------------------
| __IInitTerm                                                                  |
|   This function is used to process thread attach and detach events.          |
|   It is called by our DLL_InitTerm function when we are in a DLL             |
|   and by the C runtime's TLS callback in file tls.c when we are              |
|   in an .exe.                                                                |
|   In both cases "ulflag" takes the values                                    |
|     DLL_PROCESS_DETACH       0                                               |
|     DLL_PROCESS_ATTACH       1                                               |
|     DLL_THREAD_ATTACH        2                                               |
|     DLL_THREAD_DETACH        3                                               |
|                                                                              |
|   Note these are the same flag values that Dll_InitTerm takes.               |
|                                                                              |
|   This function must be packaged in Ingthrd.OBJ, so that it will             |
|   be linked in whenever this workaround is required.                         |
|                                                                              |
|   The PROCESS_ATTACH call occurs before any static initialization            |
|   and before main() is entered.                                              |
|                                                                              |
|   The PROCESS_DETACH call occurs after main() returns and after              |
|   all static destruction.                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
void _Optlink __IInitTerm(unsigned long ulflag)
{
    switch (ulflag)
    {
    case DLL_PROCESS_ATTACH :
        //
        //  We are called before static init.
        //
        break;

    case DLL_PROCESS_DETACH :
        //
        // We are called after static destructors.  Do nothing here...it is
        // handled in the destructors.
        // Since this is called after static destructors be carefull not use any static
        // variables.
        //
        if (gTLSSlot)
            ITLSHandle::release(*gTLSSlot);

        delete gTLSSlot;
        __IThreadLocalBase__freeTlsIndex();
        break;

    case DLL_THREAD_ATTACH :
        {
            ITRACE_DEVELOP("__IInitTerm - DLL_THREAD_ATTACH");
            if ( IStartedThread::currentHandle( false ).asUnsigned() == 0)
            {
                // have not seen this one yet.
                ITRACE_DEVELOP("__IInitTerm - new thread");

                //
                //  Force an object to be allocated in the list of threads if it
                //  is not already. This will set the currentHandle value and
                //  initialize the id member of IStartedThread.
                //
                IStartedThread::locate(IThreadId(GetCurrentThreadId()));
            }
         }
         break;

    case  DLL_THREAD_DETACH:
        {
            ITRACE_DEVELOP("__IInitTerm - DLL_THREAD_DETACH");

            //
            // Check if we have seen this thread.  Note that
            // IStartedThread::currentHandle has the side effect of allocating
            // a localVariable block if one does not exist.
            if ( IStartedThread::currentHandle( false ).asUnsigned() != 0)
            {
                //
                //  We have seen it.  Check if entry exists and remove it.  We do
                //  not want to do this to the primary thread or remove any threads
                //  during process termination.  Doing so may cause the thread to
                //  be reallocated during IOC static destructors.
                //
                unsigned long id = GetCurrentThreadId();

                // create ithreadid instance for comparison
                IThreadId  curtid = id;

                if ((IStartedThread::handleWithId(id).asUnsigned())
                &&  (curtid != IStartedThread::fgPrimaryTID)
                // &&  (!IStartedThread::fsProcessDetachInProgress)
                                    // Investigate why this code was inactive  (the flag was never being set.)
                                                                                    )
                {
                    ITRACE_DEVELOP("__IInitTerm - removing thread");

                    // Entry exists
                    IStartedThread* thread = IStartedThread::locate( id );
                    if (thread)
                    {
                        thread->end(0);
                        thread->removeRef();
                    }
                }
            }

            // last action in this thread is free memory it uses
            void** ptls( (void**) IStartedThread::tlsSlot().storage());
            if (ptls)
            {
                delete [] ptls;
                IStartedThread::tlsSlot().setStorage(0);
            }
         }
         break;
    }
    return;
}
#endif // IC_WIN


// ----------------------------------------------------------------------------
//  gHandleInit
// ----------------------------------------------------------------------------
static void gDoStaticInit()
{
    if (!gStaticInitDone)
    {
        IPrimalLock lockInit;

        if (!gStaticInitDone)
        {
            IStartedThread::pResource = new IPrivateResource;
            adoptStaticObject( IStartedThread::pResource );

            gAThreadEnded = new IPrivateCondition;
            adoptStaticObject( gAThreadEnded );

            gAllThreadsEnded = new IPrivateCondition;
            adoptStaticObject( gAllThreadsEnded );

            gCachedTID = new IThreadId;
            adoptStaticObject( gCachedTID );

            gIdOfLastThreadRemoved = new IThreadId;
            adoptStaticObject( gIdOfLastThreadRemoved );

            gTLSSlot = new ITLSHandle(ITLSHandle::allocate());

#ifndef IC_WIN
            adoptStaticObject( gTLSSlot );
#endif

            gStaticInitDone = True;
        }
    }
}


/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::timeStamp - Incremented each time a thread is started |
|                              or stopped.  A cursor is valid if its created   |
|                              data member is equal to this value.             |
------------------------------------------------------------------------------*/
unsigned INonGUIThread::Cursor::timeStamp = 1;

#ifdef IC_POSIX
/*------------------------------------------------------------------------------
| ITLSHandle_threadTerm                                                        |
| Called at thread termination when the thread has thread specific data.       |
------------------------------------------------------------------------------*/
void ITLSHandle_threadTerm( void * threadData)
{
    IFUNCTRACE_DEVELOP();

    // last action in this thread is free memory it uses
    void** ptls( (void**) threadData );
    if (ptls)
        delete [] ptls;
}
#endif


/*------------------------------------------------------------------------------
| ITLSHandle::allocate                                                         |
------------------------------------------------------------------------------*/
ITLSHandle  ITLSHandle::allocate()
{
    IFUNCTRACE_DEVELOP();
    Value result(0);
#ifdef IC_PM
    unsigned long rc( DosAllocThreadLocalMemory(1, &result) );
    if (rc)
    {
        ITHROWSYSTEMERROR(  rc,
                            "DosAllocThreadLocalMemory",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
#endif

#ifdef IC_WIN
    result = TlsAlloc();
    if (result == (Value)-1)
    {
        ITHROWGUIERROR2("TlsAlloc",
                        IBaseErrorInfo::accessError,
                        IException::recoverable );
    }
#endif

#ifdef IC_POSIX
    int rc (pthread_key_create( &result, ITLSHandle_threadTerm ) );
    if (rc)
    {
        ITHROWSYSTEMERROR(  rc,
                            "pthread_key_create",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
#endif

    return  ITLSHandle(result);
}

/*------------------------------------------------------------------------------
| ITLSHandle::release                                                          |
------------------------------------------------------------------------------*/
void ITLSHandle::release(ITLSHandle& handle)
{
#ifdef IC_PM
    unsigned long rc(DosFreeThreadLocalMemory((Value)handle));
    if (rc)
    {
        ITHROWSYSTEMERROR(  rc,
                            "DosAllocThreadLocalMemory",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
#endif

#ifdef IC_WIN
   if (!TlsFree((Value)handle))
   {
       ITHROWGUIERROR2("TlsFree",
                       IBaseErrorInfo::accessError,
                       IException::recoverable );
   }
#endif
#ifdef IC_POSIX
   int rc (pthread_key_delete((Value)handle));
   {
       ITHROWSYSTEMERROR( rc,
                         "pthread_key_delete",
                         IBaseErrorInfo::accessError,
                         IException::recoverable );
   }
#endif
}

/*------------------------------------------------------------------------------
| ITLSHandle::storage                                                          |
------------------------------------------------------------------------------*/
void* ITLSHandle::storage() const
{
#ifdef IC_PM
   return *((void**)this->fhandle);
#endif

#ifdef IC_WIN
    return  TlsGetValue( this->fhandle );
#endif

#ifdef IC_POSIX
    return  pthread_getspecific( this->fhandle );
#endif
}

/*------------------------------------------------------------------------------
| ITLSHandle::setStorage                                                       |
------------------------------------------------------------------------------*/
void* ITLSHandle::setStorage(void* value)
{
    IFUNCTRACE_DEVELOP();

#ifdef IC_PM
   void* result = *((void**)this->fhandle);
   *((void**)this->fhandle) = value;
#endif

#ifdef IC_WIN
   void* result = TlsGetValue(this->asUnsigned());
   if (!TlsSetValue( this->asUnsigned(), value ))
      {
      ITHROWGUIERROR2("TlsSetValue",
                      IBaseErrorInfo::accessError,
                      IException::recoverable );
      }
#endif
#ifdef IC_POSIX
    void* result = pthread_getspecific( this->fhandle );
    int rc ( pthread_setspecific( this->fhandle, value ));
    if (rc)
       {
       ITHROWSYSTEMERROR( rc,
                          "pthread_set_specific",
                          IBaseErrorInfo::accessError,
                          IException::recoverable );
       }
#endif
   return result;
}


/*------------------------------------------------------------------------------
| IThreadVariables::IThreadVariables                                           |
------------------------------------------------------------------------------*/
IThreadVariables::IThreadVariables() :
  IKeySetAsHshTable<IThreadVariable, IString>(7)  // 7 hash slots
{
}

/*------------------------------------------------------------------------------
| IThreadVariables::~IThreadVariables                                          |
------------------------------------------------------------------------------*/
IThreadVariables::~IThreadVariables()
{
}

/*------------------------------------------------------------------------------
| IThreadVariable::IThreadVariable                                             |
------------------------------------------------------------------------------*/
IThreadVariable::IThreadVariable( ) :
  fkey  ( ),
  fvalue( )
{ }

/*------------------------------------------------------------------------------
| IThreadVariable::IThreadVariable                                             |
------------------------------------------------------------------------------*/
IThreadVariable::IThreadVariable( const IThreadVariable& variable ) :
  fkey  ( variable.fkey ),
  fvalue( variable.fvalue )
{ }

/*------------------------------------------------------------------------------
| IThreadVariable::IThreadVariable                                             |
------------------------------------------------------------------------------*/
IThreadVariable::IThreadVariable( const IString &key, const IString &value ) :
  fkey  ( key ),
  fvalue( value )
{ }

/*------------------------------------------------------------------------------
| IThreadVariable::operator=                                                   |
------------------------------------------------------------------------------*/
IThreadVariable& IThreadVariable::operator=  (const IThreadVariable& variable)
{
  fkey = variable.fkey;
  fvalue = variable.fvalue;
  return *this;
}

/*------------------------------------------------------------------------------
| IThreadVariable::~IThreadVariable                                            |
------------------------------------------------------------------------------*/
IThreadVariable::~IThreadVariable( )
{ }

/*------------------------------------------------------------------------------
| IStartedThread::IStartedThread                                               |
|                                                                              |
| Initialize all the members.  If a thread ID is given, it is used.            |
------------------------------------------------------------------------------*/
IStartedThread::IStartedThread(const IThreadId& tid) :
        id(tid),
        inArray(false),
        started((tid.isValid()) ? true : false),
        startedHere(false),
#ifdef  IC_PM
        fCount(++IStartedThread::fStartedThreadCount),
#endif
        stkSize(INonGUIThread::defaultStackSize()),
#ifdef IC_PM
        priorityClass(INonGUIApplication::noChange),
#endif
        priorityLevel(INonGUIThread::normal),
        fnObj(0),
        slot(0),
#ifdef IC_WIN
        fhandle(0),
#endif
#ifdef IC_PM
        pTIB(0),
        ftlsPointer(0),
#endif
        vars(new IThreadVariables),
        fNotifier(0),
#ifdef IC_POSIX
        fSemid(-1),
#endif
        fExitCode(0)
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP( IString("tid=") + tid.asString().d2x());
}


/*------------------------------------------------------------------------------
| IStartedThread::~IStartedThread                                              |
|                                                                              |
|                                     |
------------------------------------------------------------------------------*/
IStartedThread::~IStartedThread()
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP(IString("tid=") + this->id.asString().d2x());

    delete fnObj;
    fnObj = 0;

#ifdef IC_POSIX
    if (-1 != fSemid)
        semctl(fSemid,0,IPC_RMID);
#endif
    delete vars;
}

/*------------------------------------------------------------------------------
| IStartedThread::isStarted                                                    |
|                                                                              |
| Return whether the thread represented by this object is currently started.   |
|                                                                              |
| If the "started here" flag is set, then the thread was started via           |
| IStartedThread::initThread and we know it hasn't ended.                      |
|                                                                              |
| Otherwise, we check the thread ID via a call to DosWaitThread and if         |
| not OK, return false (and set the "started" flag appropriately).             |
------------------------------------------------------------------------------*/
bool IStartedThread::isStarted()
{
    IFUNCTRACE_DEVELOP();

    bool result = true;
    if ( !this->startedHere )
    {
        // Verify it, though!
        if ( this->started )
            result = this->started = IStartedThread::isTIDValid( this->id );
        else
            result = false;
    }
#ifdef IC_WIN
    if (WAIT_OBJECT_0 == WaitForSingleObject(this->handle(),0)) {
        result = false;
    }
#endif
    return result;
}


/*------------------------------------------------------------------------------
| IStartedThread::exitCode                                                     |
|                                                                              |
| Returns the exit code member.                                                |
------------------------------------------------------------------------------*/
unsigned long IStartedThread::exitCode() const
{
    return fExitCode;
}


/*------------------------------------------------------------------------------
| IStartedThread::isTIDValid                                                   |
|                                                                              |
| Return whether the thread represented by this object is currently started.   |
------------------------------------------------------------------------------*/
bool IStartedThread::isTIDValid(const IThreadId &id)
{
    IFUNCTRACE_DEVELOP();

bool result = false;

#ifdef IC_PM
    IThreadId tid = id;

    if ((id == IStartedThread::fgPrimaryTID)
    ||  (id == INonGUIThread::currentId())
    ||  (DosWaitThread((PTID)&tid, DCWW_NOWAIT) == ERROR_THREAD_NOT_TERMINATED))
    {
        result = true;
    }
#endif

#ifdef IC_WIN
    if ((id == IStartedThread::fgPrimaryTID)
    ||  (id == INonGUIThread::currentId())
    ||  (WaitForSingleObject(IStartedThread::handleWithId(id),0) == WAIT_TIMEOUT))
    {
        result = true;
    }
#endif

#ifdef IC_POSIX
    result = true;
#endif

    return( result );
}

/*------------------------------------------------------------------------------
| IStartedThread::asDebugInfo                                                  |
------------------------------------------------------------------------------*/
IString IStartedThread::asDebugInfo() const
{
    IResourceLock lock(IStartedThread::lock());

    IString result( "IStartedThread(count=" );
    result += IString( this->count() );
    result += ",id=";
    result += this->id.asDebugInfo();
    result += ",inArray=";
    result += IString( this->inArray );
    result += ",started=";
    result += IString( this->started );
    result += ",startedHere=";
    result += IString( this->startedHere );
    result += ",slot=";
    result += IString( this->slot );
#ifdef IC_PM
    result += ",pTIB=";
    result += IString( (unsigned long) this->pTIB );
#endif
    result += ",stkSize=";
    result += IString( this->stkSize );
#ifndef IC_POSIX
    result += ",fCount=";
    result += IString( this->fCount );
#endif

    //
    // If priority was set via the old scheme, just format the value. If
    // it was set via the new scheme, then give a more descriptive text.
    // Only show the class if its in the old scheme, since the new scheme
    // implies any such information.
    //
    if ((this->priorityLevel >= INonGUIThread::minPriority)
    &&  (this->priorityLevel <= INonGUIThread::maxPriority))
    {
        result += ",priorityLevel=";

        switch(INonGUIThread::EThreadPriority(this->priorityLevel))
        {
        case INonGUIThread::idle :
            result += IString("Idle");
            break;

        case INonGUIThread::belowNormal :
            result += IString("Below Normal");
            break;

        case INonGUIThread::normal :
            result += IString("Normal");
            break;

        case INonGUIThread::aboveNormal :
            result += IString("Above Normal");
            break;

        case INonGUIThread::timeCritical :
            result += IString("Time Critical");
            break;

        default :
            result += IString("Uknown");
            break;
        }
    }
     else
    {
#ifdef IC_PM
        result += ",priorityClass=";
        result += IString( (int) this->priorityClass );
#endif

        result += ",priorityLevel=";
        result += IString( this->priorityLevel );
  }

    result += ",fnObj=";
    result += IString( (unsigned long) this->fnObj );
    result += ")";

    return( result );
}

/*------------------------------------------------------------------------------
| IStartedThread::classDebugInfo                                               |
------------------------------------------------------------------------------*/
IString IStartedThread::classDebugInfo()
{
    IResourceLock lock( IStartedThread::lock() );

    IString result( "IStartedThread::(cacheHits=" );
    result += IString( IStartedThread::cacheHits );
    result += ",cacheMisses=";
    result += IString( IStartedThread::cacheMisses );
    result += ",arraySize=";
    result += IString( IStartedThread::arraySize );
    result += ",threads=";
    result += IString( (unsigned long) IStartedThread::threads );
    result += "->[";
    IString comma;
    for (unsigned i = 0; i < IStartedThread::arraySize; i++)
    {
        result += comma;
        comma = ",";
        if ( IStartedThread::threads[i] )
            result += IStartedThread::threads[i]->asDebugInfo();
        else
            result += "0";
    }
    result += "])";
    return result;
}

void IStartedThread::flushCache()
{
    IResourceLock lock(IStartedThread::lock());
    IStartedThread::cachedTID() = IThreadId();

    IStartedThread::cachedThread = 0;
    INonGUIThread::Cursor::timeStamp++; // Invalidate any existing cursors.
}


IThreadId& IStartedThread::cachedTID()
{
    if (!gStaticInitDone)
        gDoStaticInit();

    return *gCachedTID;
}

IPrivateCondition& IStartedThread::aThreadEnded()
{
    if (!gStaticInitDone)
        gDoStaticInit();

    return *gAThreadEnded;
}

IPrivateCondition& IStartedThread::allThreadsEnded()
{
    if (!gStaticInitDone)
        gDoStaticInit();

    return *gAllThreadsEnded;
}

IThreadId& IStartedThread::idOfLastThreadRemoved()
{
    if (!gStaticInitDone)
        gDoStaticInit();

    return *gIdOfLastThreadRemoved;
}

ITLSHandle& IStartedThread::tlsSlot()
{
    if (!gStaticInitDone)
        gDoStaticInit();

    return *gTLSSlot;
}


/*------------------------------------------------------------------------------
| IStartedThread::cache                                                        |
|                                                                              |
| Set cached thread id/ptr.                                                    |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources protected.                                |
------------------------------------------------------------------------------*/
inline void IStartedThread::cache(const IThreadId& tid, IStartedThread* thread)
{
    IFUNCTRACE_DEVELOP();

    IStartedThread::cachedTID() = tid;
    IStartedThread::cachedThread = thread;
}

/*------------------------------------------------------------------------------
| IStartedThread::checkCacheFor                                                |
|                                                                              |
| Check cached thread id/ptr.                                                  |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources protected.                                |
------------------------------------------------------------------------------*/
inline IStartedThread* IStartedThread::checkCacheFor(const IThreadId& tid)
{
    IFUNCTRACE_DEVELOP();

    IStartedThread *result = 0;
    if (tid == IStartedThread::cachedTID())
    {
        result = IStartedThread::cachedThread;
        IStartedThread::cacheHits++;
    }
    else
    {
        IStartedThread::cacheMisses++;
    }
    return result;
}

/*------------------------------------------------------------------------------
| IStartedThread::add                                                          |
|                                                                              |
| Search the static array for a free slot and add the argument IStartedThread  |
| object there.  If there is no room, allocate a larger array.                 |
|                                                                              |
| Notes:                                                                       |
|   1. Access to the static data is protected via the IStartedThread::lock     |
|      mutex.   Other access to this data is similarly protected.              |
|   2. This function is intended to be called by INonGUIThread::start().       |
------------------------------------------------------------------------------*/
void IStartedThread::add(IStartedThread &aThread)
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP( IString("aThread.id=") + aThread.id.asString().d2x());

    // Lock static info.
    IResourceLock lock( IStartedThread::lock());

    // If flag set, simply return.
    if ( aThread.inArray )
        return;

    // Search for free slot.
    unsigned index = 0;
    for ( index = 0; index < IStartedThread::arraySize; index++ )
    {
        if (!IStartedThread::threads[index])
            break;
    }

    ITRACE_DEVELOP( IString("arraySize = ") +
                    IString((int) IStartedThread::arraySize) +
                    IString(" arrayAddress = ") +
                    IString::d2x( IString((int)IStartedThread::threads)) +
                    IString(" Next available Slot in Array =") +
                    IString(index));

    if (index == IStartedThread::arraySize)
    {
        // Extend array size by 32
        unsigned newSize = IStartedThread::arraySize + 32;
        unsigned oldSize = IStartedThread::arraySize;

        // Allocate new array.
        IStartedThread **newArray = new IStartedThread*[newSize];

        // Copy old portion.
        for (index = 0; index < oldSize; ++index)
            newArray[index] = IStartedThread::threads[index];


        // Zero remainder of the new array
        index = oldSize;
        while (index < newSize)
            newArray[index++] = 0;

        // Free old array.
        delete [] IStartedThread::threads;

        // Update static info.
        IStartedThread::threads   = newArray;
        index                     = oldSize;
        IStartedThread::arraySize = newSize;
        if ( gThreadsArrayDeleter == 0 )
        {  // Manage the clean up of IStartedThread::threads via
           // gThreadsArrayDeleter and IStaticObjectManager.
           gThreadsArrayDeleter = new IThreadsArrayDeleter;
           adoptStaticObject( gThreadsArrayDeleter );
        }
    }

    // Set i-th slot to point to aThread.
    IStartedThread::threads[index] = &aThread;
    aThread.inArray = true;
    aThread.slot    = index;
    aThread.started = true;
    aThread.addRef();
    if (aThread.startedHere)
        ++fNumIOCRunningThreads;

    ITRACE_DEVELOP( IString(" New arraySize = ") +
                    IString((int) IStartedThread::arraySize) +
                    IString(" arrayAddress = ") +
                    IString::d2x( IString( (int) IStartedThread::threads)) +
                    IString(" Next available Slot in Array =") +
                    IString(index));

}

/*------------------------------------------------------------------------------
| IStartedThread::remove                                                       |
|                                                                              |
| Search the static array for the argument IStartedThread object and reset the |
| slot to zero if found.                                                       |
|                                                                              |
| Notes:                                                                       |
|   1. Access to the static data is protected via a critical section.  Most    |
|      other IStartedThread methods that utilize this data do likewise.        |
------------------------------------------------------------------------------*/
void IStartedThread::remove(IStartedThread &aThread)
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP( IString("aThread.id=") +
                    aThread.id.asString().d2x() +
                    IString( " p=" ) +
                    IString::d2x(IString((int)&aThread)));

    IResourceLock lock( IStartedThread::lock() );
    if ( aThread.inArray )
    {
        IStartedThread::idOfLastThreadRemoved() = aThread.id;

        IStartedThread::threads[ aThread.slot ] = 0;
        aThread.inArray = false;
        aThread.slot    = 0;
        IStartedThread::flushCache();

        aThread.fThreadEnded.broadcast();
        aThreadEnded().broadcast();

        // Are there any threads running?
        if (aThread.startedHere)
        {
           aThread.started = false;
           aThread.startedHere = false;
           if (--fNumIOCRunningThreads == 0)
               allThreadsEnded().signal();
        }

        aThread.removeRef();
    }
}

/*------------------------------------------------------------------------------
| IStartedThread::locate                                                       |
|                                                                              |
| Search through the static array of started threads and return a pointer to   |
| the one with the argument threadID.                                          |
|                                                                              |
| If not found, then determine if the thread ID is valid and if so, allocate a |
| new IStartedThread and insert it into the array.                             |
|                                                                              |
| Notes:                                                                       |
|   1. The use count of the returned IStartedThread will be incremented        |
|      by one (or initialized to one if the object is allocated here).         |
|      The caller must ensure it gets decremented (INonGUIThread ensure this   |
|      by calling removeRef() at destruction).                                 |
|   2. Care is taken to ensure the static thread info is stable while          |
|      being examined/updated.                                                 |
|   3. This function throws an exception if the argument threadID is           |
|      not a valid OS/2 thread.                                                |
------------------------------------------------------------------------------*/
IStartedThread *IStartedThread::locate(const IThreadId &id)
{
    IFUNCTRACE_DEVELOP();
    ITRACE_ALL(id.asString().d2x());

    IThreadId tid = id;
    IStartedThread* result = 0;

    // Lock static information
    IResourceLock lock(IStartedThread::lock());

    // Check cache first:
    result = IStartedThread::checkCacheFor( id );

    if (result)
        result -> addRef();

    // If not found, search array for this thread...
    if (!result)
    {
        unsigned i;
        for (   i = 0;
                i < IStartedThread::arraySize
                    && ( IStartedThread::threads[i] == 0
                        || IStartedThread::threads[i]->id != id ); i++ )
        {
        }

        if ( i == IStartedThread::arraySize )
        {
            //
            // Thread not found, see if it is a real thread.
            // In Windows, the thread must be the current thread for
            // this logic to work correctly.  We need to get the handle.
            // To avoid the exception that will result if this is not the
            // case, use the second argument on the IThreadId constructor.
            //
#ifdef IC_WIN
            if (tid.asUnsigned() == GetCurrentThreadId())
#else
            if (IStartedThread::isTIDValid(tid))
#endif
            {
                // This thread looks OK.
                // Allocate new IStartedThread for this TID.
                result = new IStartedThread( id );
                result->addRef();
#ifdef IC_WIN
                result->fhandle = IStartedThread::currentHandle( true );
#endif
                // Insert into array.
                IStartedThread::add( *result );

                // Cache this thread.
                IStartedThread::cache( id, result );

                // Watch for when it ends...
                if ( tid !=  IStartedThread::fgPrimaryTID )
                    IStartedThread::watch();
            }
             else
            {
                ITHROWLIBRARYERROR( IC_INVALID_THREAD_ID,
                                    IBaseErrorInfo::invalidParameter,
                                    IException::recoverable );
            }
        }
        else
        {
            result = IStartedThread::threads[i];
            result -> addRef();
            IStartedThread::cache( id, result );
        }
    }
    return result;
}


bool IStartedThread::waitForAllThreads(long timeout)
{
    IFUNCTRACE_DEVELOP();

    IResourceLock lock(IStartedThread::lock());
    if (fNumIOCRunningThreads != 0)
        allThreadsEnded().wait(lock, timeout);

    return fNumIOCRunningThreads != 0;
}



IThreadId IStartedThread::waitForAnyThread(long timeout)
{
    IFUNCTRACE_DEVELOP();

    IThreadId result = 0;
    {
        IResourceLock lock(IStartedThread::lock());
        aThreadEnded().wait(lock,timeout);
        result = IStartedThread::idOfLastThreadRemoved();
    }
    return result;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IStartedThread::currentHandle                                                |
| Queries the handle for the current thread.  By default (create true),        |
| a new handle is created if one is not already allocated.                     |
------------------------------------------------------------------------------*/
IThreadHandle IStartedThread:: currentHandle(bool create, IStartedThread *This)
{
    IFUNCTRACE_ALL();

    //
    // Get the stored handle.  Note that the uninitialized value is set
    // to 0 by localVariableFor.   Currently, the only use of the tls slot
    // for the base component is the thread handle.
    //
    HANDLE tlsThreadHandle = (HANDLE)IStartedThread::localVariableFor( base );
    ITRACE_ALL( IString("tlsThreadHandle=") +
                IString((unsigned long)tlsThreadHandle));

    //
    // if the handle has not been set yet create one from the
    // pseudohandle given by GetCurrentThread().  Store in thread local
    // storage so we only duplicate the handle once per thread.
    if ((tlsThreadHandle == 0) && create )
    {
        if (0 == This)
        {
            HANDLE processHandle = GetCurrentProcess();
            DuplicateHandle
            (
                processHandle,          //HANDLE hSourceProcessHandle,
                GetCurrentThread(),     //HANDLE hSourceHandle,
                processHandle,          //HANDLE hTargetProcessHandle,
                &tlsThreadHandle,       //LPHANDLE lpTargetHandle,
                0,                      //DWORD dwDesiredAccess,
                0,                      //BOOL bInheritHandle,
                DUPLICATE_SAME_ACCESS   //DWORD dwOptions
            );
        }
        else
        {
            tlsThreadHandle = This->fhandle;
        }

        //
        // Store the duplicated handle in thread local storage.  Future
        // references use the stored value
        //
        IStartedThread::setLocalVariableFor(base, (void*)tlsThreadHandle );
        ITRACE_DEVELOP( IString("new tlsThreadHandle=") +
                        IString((unsigned long)tlsThreadHandle) +
                        IString(" id=") + IString(GetCurrentThreadId()) );
    }

    return IThreadHandle( tlsThreadHandle );
}
#endif


#ifndef IC_POSIX
/*------------------------------------------------------------------------------
| IStartedThread::idWithHandle                                                 |
------------------------------------------------------------------------------*/
IThreadId  IStartedThread::idWithHandle(const IThreadHandle& handle)
{
    //
    // In Windows, a thread handle and a thread id are 2 different things,
    // with no API support to convert between them.   In other environments
    // a thread handle and thread id are the same.
    //
    IFUNCTRACE_DEVELOP();

#ifdef IC_WIN
    IThreadId result = 0;

    // lock the static thread list
    IResourceLock lock( IStartedThread::lock() );

    // Create thread ID = 0
    IThreadId thidNULL = IThreadId();

    // search for a matching handle
    for (unsigned i = 0; (i < IStartedThread::arraySize) && (!result); i++)
    {
        if ((IStartedThread::threads[i] != 0)
        &&  (IStartedThread::threads[i]->handle() == handle))
        {
            // trickery to try to obtain the id value
            if ( IStartedThread::threads[i]->id == IThreadId(0))
                IStartedThread::threads[i]->id = GetCurrentThreadId();
            result = IStartedThread::threads[i]->id;
        }
    }
    return result;
#else
    return handle;
#endif
}

/*------------------------------------------------------------------------------
| IStartedThread::handleWithId                                                 |
------------------------------------------------------------------------------*/
IThreadHandle IStartedThread::handleWithId(const IThreadId& tid)
{
    //
    // In Windows, a thread handle and a thread id are 2 different things,
    // with no API support to convert between them.   In other environments
    // a thread handle and thread id are the same.
    //
    IFUNCTRACE_DEVELOP();

#ifdef IC_WIN
    IThreadHandle result = 0;

    // Lock the static thread list
    IResourceLock lock( IStartedThread::lock() );

    // Create thread ID = 0
    IThreadId thidNULL = IThreadId();

    // search for a matching id
    for (unsigned i = 0; (i < IStartedThread::arraySize) && (!result); i++)
    {
        if ((IStartedThread::threads[i] != 0)
        &&  (IStartedThread::threads[i]->id == tid))
        {
            // trickery to try to obtain the handle value
            if ((IStartedThread::threads[i]->handle() == 0)
            &&  (tid.asUnsigned() == GetCurrentThreadId()))
            {
                //
                // We are executing in the thread...one of the 2 places we
                // can get the handle.
                //
                IStartedThread::threads[i]->fhandle = IStartedThread::currentHandle( true );
            }
            result = IStartedThread::threads[i]->handle();
        }
    }
    return result;
#else
    return tid;
#endif
}
#endif



/*------------------------------------------------------------------------------
| IStartedThread::localVariableFor                                             |
------------------------------------------------------------------------------*/
void* IStartedThread::localVariableFor(LocalVariableIndex index)
{
    IFUNCTRACE_DEVELOP();

    void** result ( (void**) IStartedThread::tlsSlot().storage() );
    if (result == 0)
    {
        // Not allocated ... allocate the TLS block and set value to 0.
        IStartedThread::setLocalVariableFor( index, 0);
        return 0;
    }
    return result[index];
}

/*------------------------------------------------------------------------------
| IStartedThread::setLocalVariableFor                                          |
------------------------------------------------------------------------------*/
void IStartedThread::setLocalVariableFor(LocalVariableIndex index,
                                         void* value)
{
    IFUNCTRACE_DEVELOP();

    void** result ( (void**) IStartedThread::tlsSlot().storage() );
    if (result == 0)
    {
        //
        // Block not allocated, need to allocate the array of void* for this
        // thread.
        //
        result =  new void* [reservedMustBeLast];
        int i;
        for (i=0; i < reservedMustBeLast; i++)
            result[i] = 0;

        ITRACE_DEVELOP( IString("new IThreadLocalVariable=") +
                        IString( (unsigned long)(void*)result ).d2x() );
        IStartedThread::tlsSlot().setStorage( result );

#ifdef IC_PM
        //
        // For OS/2, need to store the pointer in the startedThread element
        // for this thread to be sure it is deleted.
        // In Windows, IInitTerm handles this delete.
        // In POSIX, ITLSHandle__threadTerm handles this delete.
        //
        IStartedThread* p = IStartedThread::locate( INonGUIThread::currentId() );
        if (!p->ftlsPointer)
            p->ftlsPointer = result;
        p -> removeRef();
#endif
    }
    result[index] = value;
}

/*------------------------------------------------------------------------------
| IStartedThread::notifyOnCurrentThreadEnd                                     |
| Enables notification to the specified IObserver when the thread              |
| is ending.                                                                   |
------------------------------------------------------------------------------*/
void IStartedThread::notifyOnCurrentThreadEnd(IObserver* observer)
{
    IFUNCTRACE_DEVELOP();

    if (observer)
    {
        IStartedThread* p = IStartedThread::locate( INonGUIThread::currentId() );
        if (! p->fNotifier )
        {
            // Create the notifier object and ensure it is enabled.
            p->fNotifier = new IStandardNotifier;
            p->fNotifier->enableNotification( true );
        }

        // Link the supplied observer object with the delete notifier.
        observer->handleNotificationsFor(*(p->fNotifier),
                                         INotifier::deleteId);
        p->removeRef();
    }
}


//
// initThread
//
#ifdef IC_WIN
#pragma handler(initThread)
extern "C" void* __stdcall initThread(void *aThread)
{
    IStartedThread::initThread( (IStartedThread *)aThread );
    return (void*)NULL;
}
#else
extern "C" void* initThread(void *aThread)
{
    IStartedThread::initThread( (IStartedThread *)aThread );
    return (void*)NULL;
}
#endif

//
// IStartedThread::initThread
//
// Implementation:
//   Dispatch the run() virtual function of the IThreadFn object pointed
//   to by the argument.
//
void IStartedThread::initThread(IStartedThread* aThread)
{
    IFUNCTRACE_DEVELOP();

#ifdef IC_WIN
    // Must reset 387 since using CreateThread
    _fpreset();
#endif

    aThread->id = INonGUIThread::currentId();

#ifdef IC_WIN
    // Initialize handle in IStartedThread
    IStartedThread::currentHandle( true, aThread );
#endif

#ifdef IC_PM
    aThread->tib(); // Set pTIB member.
#endif

    ITRACE_DEVELOP( IString("aThread->id=") + aThread->id.asString().d2x() +
                    IString( " p=" ) +
                    IString::d2x( IString( (int)aThread )));

#ifdef IC_POSIX
    // Dummy parameter, value not used
    int oldvalue;
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,&oldvalue);
    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED,&oldvalue);

    // Block SIGALRM normally and unblocks it during special
    // operations.
    sigset_t set;
    sigemptyset(&set);
    sigaddset(&set, SIGALRM);
    sigthreadmask(SIG_BLOCK, &set, 0);

#endif /*IC_POSIX*/

    //
    //  If priority is other than "default", set it via API... We have
    //  to handle a special case if its OS/2 because it could be using
    //  one of two schemes.
    //

#ifdef IC_PM
    // Convert the info to native OS/2 scheme
    INonGUIApplication::PriorityClass classToSet;
    int levelToSet;
    levelToSet = INonGUIThread::convertedPriority(aThread->priorityLevel,
                                                  classToSet);
    //
    // If its anything besides default OS/2 thread priority, then set it
    // via the old style API.
    //
    if (levelToSet && (classToSet != INonGUIApplication::noChange))
    {
         INonGUIThread::current().setPriority( classToSet, levelToSet);
    }
#else
    // Make sure something has not gone awry
    IASSERTSTATE((aThread->priorityLevel >= INonGUIThread::minPriority) &&
                 (aThread->priorityLevel <= INonGUIThread::maxPriority))

    // For other platforms, just call the portable API to do the work
    if (aThread->priorityLevel != INonGUIThread::normal)
    {
        INonGUIThread::current().setThreadPriority
                (INonGUIThread::EThreadPriority(aThread->priorityLevel));
    }
#endif
    // Dispatch user function...
    aThread->fnObj->run();

    // Terminate thread...
#ifdef IC_WIN
    if ( aThread->count() == 1 )
    {  // Time to do final clean up of the IStartedThread object.
       aThread->end( 0 );
    }
    else
    {  // The IStartedThread object still has multiple reference
       // counts remaining. Don't obliterate it yet by calling
       // IStartedThread::end. Instead, just decrement the reference
       // count and allow later code (such as the DLL_THREAD_DETACH
       // case of __IInitThread) to do further clean up.
       aThread->removeRef();
    }
    _endthread();
#else
    aThread->end(0);
#endif
}

//
// IStartedThread::end
//
// Reset the object to an "unstarted" state.  This function can only be called
// if this->startedHere is true.
//
void IStartedThread::end(unsigned long exitCode)
{
    IFUNCTRACE_DEVELOP();

    // Store the exit code
    fExitCode = exitCode;

    //
    // Generate a notification to any interested observers by deleting
    // the notifier object.   This needs to occur before the thread local
    // storage is deleted or the IStartedThread is removed from the list.
    //
    if (this->fNotifier)
    {
        delete this->fNotifier;
        this->fNotifier = 0;
    }

#ifdef IC_WIN
    if (this->id.asUnsigned() == GetCurrentThreadId())
    {
        // clear the tls stored handle value
         IStartedThread::setLocalVariableFor( IStartedThread::base, 0 );
    }

    if ( !CloseHandle( this->fhandle ) )
    {
        ITHROWSYSTEMERROR(  GetLastError(),
                            "CloseHandle",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
    this->fhandle = 0;

#endif

#ifdef IC_PM
    this->pTIB = 0;

    //
    // Free the pointer to TLS block.  We attempt to free it using the
    // tlsSlot().storage value, but this can only happen if we are running
    // in the thread in question.  We use the ftlsPointer member of IStartedThread
    // as a backup mechanism to avoid leaks.
    // In Windows, we do this cleanup in the __IInitTerm DLL_THREAD_DETACH
    // callback.
    // In POSIX, we do this cleanup in ITLSHandle__threadTerm.
    //
    void** ptls;
    if (this->id.asUnsigned() == INonGUIThread::currentId().asUnsigned())
    {
        ptls = (void**) IStartedThread::tlsSlot().storage();
        IStartedThread::tlsSlot().setStorage( 0 );
    }
    else
    {
        ptls = this->ftlsPointer;
    }

    this->ftlsPointer = 0;
    if (ptls)
    {
        delete [] ptls;
        ITRACE_DEVELOP( IString("IStartedThread deleting TLS block=") +
                        IString( (unsigned long)(void*)ptls).d2x() );
    }
#endif

    // Make sure id is invalid now
#ifdef IC_PMWIN
    id = IThreadId(0);
#endif

    IStartedThread::remove( *this );    //  Remove this IStartedThread from the
                                        //  global thread list.

    // Calling removeRef() is not necessary here because the
    // ::remove() call above already calls removeRef() and
    // ~INonGUIThread() also calls removeRef().
    // this->removeRef();       //  Decrement our reference count, which
                                //  MAY delete "this".  DO NOT MAKE ANY
                                //  REFERENCES TO this AFTER THIS POINT.
}

/*------------------------------------------------------------------------------
| IStartedThread::tib                                                          |
|                                                                              |
| Return pointer to system TIB structure.  If zero on entry, then this member  |
| is set.  This function can only be called for the current thread (if not,    |
| then 0 may be returned).                                                     |
------------------------------------------------------------------------------*/
#ifdef IC_PM
tib_s *IStartedThread::tib()
{
    if ( !this->pTIB )
    {
        PTIB ptib;
        PPIB ppib;
        DosGetInfoBlocks( &ptib, &ppib );
        if ( (unsigned long)this->id == ptib->tib_ptib2->tib2_ultid )
            this->pTIB = ptib;
    }
    return this->pTIB;
}
#endif

/*------------------------------------------------------------------------------
| IStartedThread::variable                                                     |
|                                                                              |
| Examine each element in the vars array.  If one with a matching key is found,|
| return the value associated with it, otherwise return null.                  |
------------------------------------------------------------------------------*/
IString IStartedThread::variable(const IString &key) const
{
    IResourceLock lock( IStartedThread::lock() );
    IThreadVariables::Cursor cursor( *(this->vars) );
    if (this->vars->locateElementWithKey( key, cursor ))
        return cursor.element().fvalue ;

    return IString();
}

/*------------------------------------------------------------------------------
| IStartedThread::setVariable                                                  |
|                                                                              |
| Search for a matching key.  If found, set the associated value to the new    |
| one. If no matching key is found and there is room in the array, fill it in  |
| with the argument key/value pair.                                            |
------------------------------------------------------------------------------*/
IStartedThread
&IStartedThread::setVariable(const IString &key, const IString &value)
{
    IResourceLock   lock( IStartedThread::lock() );
    this->vars->addOrReplaceElementWithKey(IThreadVariable(key, value));
    return *this;
}

/*------------------------------------------------------------------------------
| IStartedThread::lock                                                         |
|                                                                              |
| If the pointed-to private resource hasn't been initialized, do it.           |
|                                                                              |
| Return the private resource.                                                 |
------------------------------------------------------------------------------*/
IPrivateResource &IStartedThread::lock()
{
    if (!gStaticInitDone)
        gDoStaticInit();
    return *IStartedThread::pResource;
}

#ifdef IC_POSIX
void
IStartedThread::removeUnixSemaphores()
{
  semctl((fThreadEnded.fImpl)->fSemid,0,IPC_RMID);
  semctl(fSemid,0,IPC_RMID);
}
#endif

/*------------------------------------------------------------------------------
| INonGUIThread::INonGUIThread                                                 |
|                                                                              |
| The role of the INonGUIThread constructors is twofold:                       |
|   1. They must locate the appropriate IStartedThread instance,               |
|      creating one if necessary, set the "thread" member to point             |
|      to this instance, and bump the IStartedThread use count.                |
|   2. Start the OS/2 thread (if necessary).                                   |
------------------------------------------------------------------------------*/
INonGUIThread::INonGUIThread(const IThreadId& tid,
                             const IThreadHandle& handle) :
        thread(0)
{
    IFUNCTRACE_ALL();

#ifdef IC_WIN
    if (handle != IThreadHandle::noHandle)
    {
        // Handle given.  Assume valid and use it.
        if (tid == IStartedThread::idWithHandle( handle ) )
        {
            // Already known ... set pointer
            thread = IStartedThread::locate( tid );
        }
        else
        {
            // New thread to us...fully qualified.  Add to list
            thread = new IStartedThread( tid );
            thread->addRef();
            thread->fhandle = handle;
            IStartedThread::add(*thread );

            // Watch for when it ends...
//          if ( tid != IStartedThread::fgPrimaryTID )
//              IStartedThread::watch();
        }
    }
    else
    {
        //
        // Handle is default value.  The thread must already be known to
        // us or be the current thread.  locate() checks and will throw
        // if this is not the case.
        //
        thread = IStartedThread::locate( tid );
    }
#else
    // Dont need to worry about the handle ... ignore it.
    thread = IStartedThread::locate( tid );
#endif
}


INonGUIThread::INonGUIThread() : thread(INonGUIThread::newStartedThread())
{
    this->thread->addRef();
}

INonGUIThread::INonGUIThread(const INonGUIThread &aThread) :
    thread(aThread.startedThread())
{
    this->thread->addRef();
}

/*------------------------------------------------------------------------------
| INonGUIThread::newStartedThread                                                    |
------------------------------------------------------------------------------*/
IStartedThread *INonGUIThread::newStartedThread()
{
    return new IStartedThread;
}

/*------------------------------------------------------------------------------
| INonGUIThread::asString                                                      |
------------------------------------------------------------------------------*/
IString INonGUIThread::asString() const
{
    IString result( "INonGUIThread(" );
    result += this->id().asString().d2x();
    result += ")";
    return result;
}

/*------------------------------------------------------------------------------
| INonGUIThread::asDebugInfo                                                   |
------------------------------------------------------------------------------*/
IString INonGUIThread::asDebugInfo() const
{
    IString result( "INonGUIThread(" );
    result += "thread=";
    result += IString( (unsigned long) this->startedThread() );
    result += "->";
    result += this->startedThread()->asDebugInfo();
    result += ")";
    return result;
}

/*------------------------------------------------------------------------------
| INonGUIThread::~INonGUIThread                                                |
|                                                                              |
| The destructor must remove the reference to the associated IStartedThread.   |
------------------------------------------------------------------------------*/
INonGUIThread::~INonGUIThread()
{
    if ( this->thread )
        this->thread->removeRef();
}

/*------------------------------------------------------------------------------
| INonGUIThread::operator =                                                    |
|                                                                              |
| Attach this thread to aThread's associated started thread.                   |
------------------------------------------------------------------------------*/
INonGUIThread &INonGUIThread::operator=(const INonGUIThread &aThread)
{
    if (aThread.startedThread())
        aThread.startedThread()->addRef();
    if (this->thread)
        this->thread->removeRef();

    this->thread = aThread.startedThread();
    return *this;
}

/*------------------------------------------------------------------------------
| INonGUIThread::startedThread                                                 |
|                                                                              |
| Returns the thread member.                                                   |
------------------------------------------------------------------------------*/
IStartedThread* INonGUIThread::startedThread() const
{
    return this->thread;
}

/*------------------------------------------------------------------------------
| INonGUIThread::currentId                                                     |
|                                                                              |
| Construct an IThreadId from the system info blocks.                          |
------------------------------------------------------------------------------*/
IThreadId INonGUIThread::currentId()
{
#ifdef IC_PM
    PTIB ptib;
    PPIB ppib;
    DosGetInfoBlocks( &ptib, &ppib );
    return IThreadId( ptib->tib_ptib2->tib2_ultid );

#endif // IC_PM

#ifdef IC_WIN
    return IThreadId(GetCurrentThreadId());
#endif

#ifdef IC_POSIX
    return IThreadId ( (unsigned long)pthread_self() );
#endif
}


/*------------------------------------------------------------------------------
| INonGUIThread::exitCode                                                      |
------------------------------------------------------------------------------*/
unsigned long INonGUIThread::exitCode() const
{
    return this->startedThread()->exitCode();
}

/*------------------------------------------------------------------------------
| INonGUIThread::currentHandle                                                 |
------------------------------------------------------------------------------*/
IThreadHandle INonGUIThread::currentHandle()
{
#ifdef IC_WIN
    // We need to get the handle to the thread, since that is what we
    // need for all the thread and synchronization functions.
    return IStartedThread::currentHandle(true);
#else
    // the threadid is the handle
    return INonGUIThread::currentId();
#endif
}


//
// INonGUIThread::prepareToStart
//
// Check that thread is not currently started (if so, throw an exception).
// Otherwise, initialize the IStartedThread object appropriately and
// return its address.
//
IStartedThread
*INonGUIThread::prepareToStart(IThreadFn *pFnObj)
{
    IFUNCTRACE_DEVELOP();

    IStartedThread* p = this->startedThread();

    // If thread is already started, throw exception.
    if ( p->isStarted() )
    {
        ITHROWLIBRARYERROR( IC_THREAD_STARTED,
                            IBaseErrorInfo::invalidRequest,
                            IException::recoverable );
    }

    p->startedHere = true;
    if ( !p->inArray )
        IStartedThread::add( *p );

    p->fnObj       = pFnObj;
    return p;
}


//
// INonGUIThread::threadStarted
//
// Check the return code from beginthread (see ithreads.cpp).  If an error
// occurred, throw exception, else set thread id.
//
void INonGUIThread::threadStarted(IStartedThread* p, int rc, IThreadId threadId)
{
    IFUNCTRACE_DEVELOP();

    if ( rc == -1 )
    {
        p -> end(0);
#ifdef IC_POSIX
        ITHROWCLIBERROR("pthread_create", IBaseErrorInfo::outOfSystemResource,
                         IException::recoverable);
#else
        ITHROWSYSTEMERROR(  rc, "_beginthread", IBaseErrorInfo::outOfSystemResource,
                            IException::recoverable );
#endif
    }
      else
    {
#ifdef IC_POSIX
        ITRACE_DEVELOP( IString("Thread ID=") + threadId.asString().d2x() );
#else
        ITRACE_DEVELOP( IString("Thread ID=") + IString(rc).d2x() );
#endif
        p->id = threadId;
#ifdef IC_WIN
        p->fhandle = (void*)rc;
#endif
    }
}


//
// INonGUIThread::stop
//
// Issue DosKillThread.  If it fails, throw an exception.
//
void INonGUIThread::stop ( )
{
    IFUNCTRACE_DEVELOP();

#ifdef IC_PM
    unsigned long rc = DosKillThread( this->id() );
    if ( rc != NO_ERROR )
    {
        ITHROWSYSTEMERROR(  rc,
                            "DosKillThread",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
#endif

#ifdef IC_WIN
    // 0 might not be the correct exitcode in all cases
    if (!TerminateThread(this->handle(), 0))
    {
        ITHROWGUIERROR2("TerminateThread",
                        IBaseErrorInfo::accessError,
                        IException::recoverable );
    }
#endif


#ifdef IC_POSIX
//     pthread_cancel(startedThread()->handle());
    pthread_cancel((pthread_t)startedThread()->handle().asUnsigned());
#endif /*IC_POSIX*/


    this->startedThread()->end(0);

}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::~ICurrentNonGUIThread                                  |
|                                                                              |
| Empty dtor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
ICurrentNonGUIThread::~ICurrentNonGUIThread()
{
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::suspend                                                |
|                                                                              |
| Issue DosSuspendThread.  If it fails, throw an exception.                    |
|                                                                              |
| For POSIX:                                                                   |
| Suspend thread unless GUI was initialized by this thread. In that case       |
| throw an exception.                                                          |
------------------------------------------------------------------------------*/
void ICurrentNonGUIThread::suspend()
{
    IFUNCTRACE_DEVELOP();

#ifdef IC_PM
    unsigned long rc = DosSuspendThread(INonGUIThread::currentId());
    if ( rc != NO_ERROR ) {
        ITHROWSYSTEMERROR(rc, "DosSuspendThread",
                          IBaseErrorInfo::accessError,
                          IException::recoverable );
    }
#endif
#ifdef IC_WIN
    if (WAIT_FAILED == SuspendThread( INonGUIThread::currentHandle())) {
        ITHROWGUIERROR2("SuspendThread",
                        IBaseErrorInfo::accessError,
                        IException::recoverable );
    }
#endif
#ifdef IC_POSIX
    IStartedThread *currentStartedThread = this->startedThread();
    if (-1 == currentStartedThread->fSemid)
    {
        IResourceLock(currentStartedThread->fSemaphoreMutex);
        if (-1 == currentStartedThread->fSemid)
        {
            currentStartedThread->fSemid =
                semget(IPC_PRIVATE, 1, S_IRUSR|S_IWUSR);
            if (-1 == currentStartedThread->fSemid)
            {
                ITHROWGUIERROR2( "semget",
                    IBaseErrorInfo::outOfSystemResource,
                    IException::recoverable );
            }
            struct sembuf seminit = {0,1,0};
            semop(currentStartedThread->fSemid, &seminit, 1);
       }
    }

    struct sembuf semsuspend = {0,0,0};
    struct sembuf semcontinue = {0,1,0};
    semop(currentStartedThread->fSemid, &semsuspend, 1);
    {
        IResourceLock(currentStartedThread->fSemaphoreMutex);
        semop(currentStartedThread->fSemid, &semcontinue, 1);
    }
#endif
}

/*------------------------------------------------------------------------------
| INonGUIThread::suspend                                                       |
|                                                                              |
| Issue DosSuspendThread.  If it fails, throw an exception.                    |
------------------------------------------------------------------------------*/
void INonGUIThread::suspend()
{
    IFUNCTRACE_DEVELOP();

#ifdef IC_PM
    unsigned long rc = DosSuspendThread( this->id() );
    if ( rc != NO_ERROR )
    {
        ITHROWSYSTEMERROR(  rc,
                            "DosSuspendThread",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
#endif

#ifdef IC_WIN
    if ( WAIT_FAILED == SuspendThread( this->handle() ) )
    {
        ITHROWGUIERROR2("SuspendThread",
                        IBaseErrorInfo::accessError,
                        IException::recoverable );
    }
#endif

#ifdef IC_POSIX
    ITHROWLIBRARYERROR( IC_THREAD_FUNC_NOT_SUPPORTED_POSIX,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
#endif
}

//
// INonGUIThread::resume
//
// Throws an exception if the thread cannot be resumed or was not in the
//   suspended state.
//
void INonGUIThread::resume()
{
    IFUNCTRACE_DEVELOP();

    if (!this->isStarted())
    {
        ITHROWGUIERROR2("resume",
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
    }

#ifdef IC_PM
    unsigned long rc = DosResumeThread( this->id() );
    if ( rc != NO_ERROR )
    {
        IBaseErrorInfo::ExceptionType toThrow;
        if (rc == ERROR_NOT_FROZEN)
            toThrow = IBaseErrorInfo::accessError;
        else
            toThrow = IBaseErrorInfo::invalidRequest;

        ITHROWSYSTEMERROR(  rc,
                            "DosResumeThread",
                            toThrow,
                            IException::recoverable );
    }
#endif

#ifdef IC_WIN
    DWORD status = ResumeThread(this->handle());
    if ((0xFFFFFFFF == status) || !status)
    {
        IBaseErrorInfo::ExceptionType toThrow;
        if (!status)
            toThrow = IBaseErrorInfo::accessError;
        else if (status == 0xFFFFFFFF)
            toThrow = IBaseErrorInfo::invalidRequest;

        ITHROWGUIERROR2("ResumeThread",
                        toThrow,
                        IException::recoverable );
    }
#endif

#ifdef IC_POSIX
    int schedpolicy;
    struct sched_param schedparam;

    if (-1 == this->thread->fSemid)
    {
        ITHROWGUIERROR2("ResumeThread",
                        IBaseErrorInfo::accessError,
                        IException::recoverable );
    }

    // Faux block for resource lock
    {
        IResourceLock(thread->fSemaphoreMutex);
        if (0 == semctl(thread->fSemid,0,GETZCNT))
        {
            ITHROWGUIERROR2("ResumeThread",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
        }
         else
        {
            struct sembuf semresume = {0,-1,0};
            semop(thread->fSemid, &semresume, 1);
        }
    }
#endif
}

/*------------------------------------------------------------------------------
| INonGUIThread::id                                                            |
|                                                                              |
| Return the id from the associated started thread.                            |
------------------------------------------------------------------------------*/
IThreadId INonGUIThread::id() const
{
    return this->startedThread()->id;
}

/*------------------------------------------------------------------------------
| INonGUIThread::handle                                                        |
------------------------------------------------------------------------------*/
IThreadHandle INonGUIThread::handle() const
{
    // catch cases where handle is not yet setup
    IASSERTSTATE( this->startedThread()->handle() != IThreadHandle())
    return this->startedThread()->handle();
}

/*------------------------------------------------------------------------------
| INonGUIThread::stackSize                                                     |
|                                                                              |
| Return the stkSize member of the referenced IStartedThread.                  |
------------------------------------------------------------------------------*/
unsigned long INonGUIThread::stackSize() const
{
#ifdef IC_PM
    tib_t *ptib = this->startedThread()->tib();
    if (ptib)
        return (char*)ptib->tib_pstacklimit - (char*)ptib->tib_pstack;
    else
        return this->startedThread()->stkSize;
#endif

#ifdef IC_WIN
    //
    // The available stack for the thread is controlled by the
    // /STACK linker directive.  The stack will grow to this limit.
    // This member is used to control the committed stack allocated
    // when the thread is created.   We do not attempt to determine
    // the committed size of the stack.
    //
    return this->startedThread()->stkSize;
#endif

#if defined(IC_POSIX)
    return this->startedThread()->stkSize;
#endif
}

/*------------------------------------------------------------------------------
| INonGUIThread::setStackSize                                                  |
|                                                                              |
| Set the stkSize member of the referenced IStartedThread.                     |
------------------------------------------------------------------------------*/
INonGUIThread &INonGUIThread::setStackSize(unsigned long newStkSize)
{
    this->startedThread()->stkSize = newStkSize;
    return *this;
}

/*------------------------------------------------------------------------------
| INonGUIThread::isStarted                                                     |
|                                                                              |
| Returns true if the associated start thread has been started                 |
------------------------------------------------------------------------------*/
bool INonGUIThread::isStarted() const
{
    return this->startedThread()->isStarted();
}

//
// ICurrentNonGUIThread::ICurrentNonGUIThread
//
ICurrentNonGUIThread::ICurrentNonGUIThread()
{
    IFUNCTRACE_DEVELOP();
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::exit                                                   |
|                                                                              |
| Terminate the current thread with the given return code.  The thread is      |
| terminated via DosExit.                                                      |
------------------------------------------------------------------------------*/
void ICurrentNonGUIThread::exit(unsigned long rc)
{
    IFUNCTRACE_DEVELOP();

    this->startedThread()->end(rc);

    if ( id() == IStartedThread::fgPrimaryTID )
    {
        ::exit( (int)rc );
    }
    else
    {
#ifdef IC_PM
        DosExit( EXIT_THREAD, rc );
#endif
#ifdef IC_WIN
        ExitThread(rc);
#endif
#ifdef IC_POSIX
        pthread_exit( &rc );
#endif
    }
}


/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::sleep                                                  |
|                                                                              |
| Suspend the current thread for the given number of milliseconds.  The thread |
| is suspended via DosSleep.                                                   |
|                                                                              |
| For POSIX:                                                                   |
| Suspend the current thread for the given number of milliseconds.  The thread |
| is suspended via sleep.                                                      |
| Note:                                                                        |
|   1) Since sleep time is in seconds the millisecond value is divided by 1000.|
|      If the value is less than a 1000 then the thread will not sleep.        |
------------------------------------------------------------------------------*/
ICurrentNonGUIThread &ICurrentNonGUIThread::sleep(unsigned long msecs)
{
    IFUNCTRACE_DEVELOP();

#ifdef IC_POSIX
#ifdef _AIX
    // sleep function on AIX is scoped to process.
    // use non-portable thread level sleep function pthread_delay
    timespec  interval;
    interval.tv_sec = msecs/1000;
    interval.tv_nsec = 0;
    pthread_delay_np(&interval);
#else
    // sleep function on MVS is scoped to thread
    ::sleep( msecs / 1000 );
#endif
#else
#if defined(IC_WIN)
    Sleep(msecs);
#elif defined(IC_PM)
    DosSleep(msecs);
#else
#error "sleep undefined"
#endif
#endif   // else IC_POSIX
    return *this;
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::waitFor                                                |
|                                                                              |
| Issue DosWaitThread on the argument thread's id.                             |
------------------------------------------------------------------------------*/
ICurrentNonGUIThread&
ICurrentNonGUIThread::waitFor(const INonGUIThread& anotherThread)
{
    return waitFor(anotherThread,-1);
}

ICurrentNonGUIThread&
ICurrentNonGUIThread::waitFor(const INonGUIThread& anotherThread, long timeout)
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP(IString("anotherThread.id=") + anotherThread.id().asString().d2x());

    // If it ain't running then nothing to do
    if (!anotherThread.isStarted())
        return *this;

    IThreadId tid = anotherThread.id();

    IASSERTPARM( tid != IStartedThread::fgPrimaryTID )
    IASSERTPARM( tid != INonGUIThread::current().id() )

#ifdef IC_PM
    //
    //  Just in case, see if the id is -1 because thread ids under OS/2 are
    //  invalidated when the thread ends.
    //
    if (tid == IThreadId(-1))
        return *this;

    waitFor(anotherThread, timeout, NULL);
    return *this;
#endif

#ifdef IC_WIN
    if ( anotherThread.startedThread()->handle() )
    {
        if (WAIT_FAILED == WaitForSingleObject( anotherThread.handle(),
                                                (DWORD)timeout) )
        {
            ITHROWGUIERROR2("WaitForSingleObject",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
        }
    }
#endif

#ifdef IC_POSIX
    waitFor(anotherThread, timeout, NULL);
#endif /*IC_POSIX*/

    return( *this );
}

ICurrentNonGUIThread&
ICurrentNonGUIThread::waitFor(const INonGUIThread& anotherThread,
                              long timeout, bool *timedOut)
{
    IFUNCTRACE_DEVELOP();

    IResourceLock lock(IStartedThread::lock());
    IThreadId tid = anotherThread.id();
    IStartedThread *startedThread = anotherThread.startedThread();
    bool threadDone = false;
    IStartedThread *startedThreadForTid = IStartedThread::locate(tid);
    if (startedThreadForTid != startedThread)
        threadDone = true;

    if (!threadDone)
        threadDone = (startedThread->fThreadEnded).wait(lock,timeout);
    startedThread->removeRef();

    if (timedOut)
        *timedOut = !threadDone;

    return (*this);
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::waitForAllThreads                                      |
|                                                                              |
| Wait for any thread till there are no more...                                |
------------------------------------------------------------------------------*/

ICurrentNonGUIThread&
ICurrentNonGUIThread::waitForAllThreads(long timeout, bool *timedOut)
{
   IFUNCTRACE_ALL();
   IASSERTPARM( this->id() == IStartedThread::fgPrimaryTID );

   bool t = IStartedThread::waitForAllThreads(timeout);
   if (NULL != timedOut) *timedOut = t;

    return(*this);
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::waitForAnyThread                                       |
|                                                                              |
| Issue DosWaitThread for thread id 0 and return the thread actually           |
| terminating.  Throw an exception in case of errors.                          |
------------------------------------------------------------------------------*/
IThreadId ICurrentNonGUIThread::waitForAnyThread(long timeout)
{
    IFUNCTRACE_ALL();
    IThreadId tid;

    tid = IStartedThread::waitForAnyThread(timeout);
    return(tid);
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::id                                                     |
|                                                                              |
| Return the current thread's id.                                              |
------------------------------------------------------------------------------*/
IThreadId ICurrentNonGUIThread::id() const
{
    return INonGUIThread::currentId();
}

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::handle                                                 |
------------------------------------------------------------------------------*/
IThreadHandle ICurrentNonGUIThread::handle() const
{
    return INonGUIThread::currentHandle();
}


/*------------------------------------------------------------------------------
| INonGUIThread::threadPriority                                                |
|                                                                              |
| Returns the thread priority for this thread. Does not query to see if its    |
| running or not. Assumes that the thread, if running, has the priority that   |
| was last set.                                                                |
------------------------------------------------------------------------------*/
INonGUIThread::EThreadPriority INonGUIThread::threadPriority() const
{
    IFUNCTRACE_DEVELOP();

    // Get our started thread, which has the priority in it.
    IStartedThread* pThread = this->startedThread();

    // Make sure we really have one
    IASSERTSTATE(pThread)

    //
    //  Make sure that our priority has not been set to something that cannot
    //  be represented via the portable API. This can happen under OS/2, and
    //  could possibly in other platforms via programming error so check it.
    //
    IASSERTSTATE((pThread->priorityLevel >= INonGUIThread::minPriority) &&
                 (pThread->priorityLevel <= INonGUIThread::maxPriority))

    return INonGUIThread::EThreadPriority(pThread->priorityLevel);
}


/*------------------------------------------------------------------------------
| INonGUIThread::setThreadPriority                                             |
|                                                                              |
| Sets the thread priority. The portable priority is mapped to the correct     |
| platform specific info and used to set the thread priority if its running.   |
| In either case it is stored as the new priority.                             |
------------------------------------------------------------------------------*/

INonGUIThread&
INonGUIThread::setThreadPriority(const INonGUIThread::EThreadPriority toSet)
{
    IFUNCTRACE_DEVELOP();

    //
    //  Make sure that it is set to something that can be represented via the
    //  portable API. Catch this regardless of whether it would have had
    //  no effect anyway because of platform issues.
    //
    IASSERTPARM((toSet >= INonGUIThread::minPriority) &&
                (toSet <= INonGUIThread::maxPriority))

    // See if we have a started thread
    IStartedThread* pThread = this->startedThread();

    // Make sure we really do
    IASSERTSTATE(pThread)

#ifdef IC_POSIX
    if (pThread->isStarted()) {
        int thread_policy;
        struct sched_param thread_sched_param;
        int ulRC = 0;
        pthread_t tid = (pthread_t)(pThread->id.asUnsigned());

        // Don't throw anything because AIX 4.1 and 4.2 apparently doesn't support
        // setting thread schedule and always return errno on pthread_setschedparam unless
        // sched_priority is 1.
        pthread_getschedparam(tid, &thread_policy, &thread_sched_param);
        switch (toSet) {
            // priority is defined in sys/sched.h
            case INonGUIThread::idle:
                thread_sched_param.sched_priority = 1;
                break;
            case INonGUIThread::belowNormal:
                thread_sched_param.sched_priority = 32;
                break;
            case INonGUIThread::normal:
                thread_sched_param.sched_priority = 64;
                break;
            case INonGUIThread::aboveNormal:
                thread_sched_param.sched_priority = 96;
                break;
            case INonGUIThread::timeCritical:
                thread_sched_param.sched_priority = 127;
                break;
        }
        pthread_setschedparam(tid, thread_policy, &thread_sched_param);
    }
#endif

#ifdef IC_PM
    unsigned prioClass;
    unsigned prioLev;
    switch(toSet)
    {
    case INonGUIThread::idle :
        prioClass = PRTYC_IDLETIME;
        prioLev = 15;
        break;

    case INonGUIThread::belowNormal :
        prioClass = PRTYC_IDLETIME;
        prioLev = 31;
        break;

    case INonGUIThread::normal :
        prioClass = PRTYC_REGULAR;
        prioLev = 0;
        break;

    case INonGUIThread::aboveNormal :
        prioClass = PRTYC_REGULAR;
        prioLev = 15;
        break;

    case INonGUIThread::timeCritical :
        prioClass = PRTYC_TIMECRITICAL;
        prioLev = 0;
        break;
    }

    if (pThread->isStarted())
    {
        unsigned long errCode =
            DosSetPriority( PRTYS_THREAD, prioClass, prioLev, pThread->id);
        if (errCode != NO_ERROR)
        {
            ITHROWSYSTEMERROR(  errCode,
                                "DosSetPriority",
                                IBaseErrorInfo::accessError,
                                IException::recoverable);
        }
    }

    //
    //  For setting via this version we clear out the priority class because
    //  its not relevant for the new style (or its infered when needed.)
    //
    pThread->priorityClass = INonGUIApplication::noChange;
#endif

#ifdef IC_WIN
    int prioLev;
    switch(toSet)
    {
    case INonGUIThread::idle :
        prioLev = THREAD_PRIORITY_IDLE;
        break;

    case INonGUIThread::belowNormal :
        prioLev = THREAD_PRIORITY_BELOW_NORMAL;
        break;

    case INonGUIThread::normal :
        prioLev = THREAD_PRIORITY_NORMAL;
        break;

    case INonGUIThread::aboveNormal :
        prioLev = THREAD_PRIORITY_ABOVE_NORMAL;
        break;

    case INonGUIThread::timeCritical :
        prioLev = THREAD_PRIORITY_TIME_CRITICAL;
        break;
    }

    if (pThread->isStarted())
    {
        if (!SetThreadPriority(pThread->handle(), prioLev))
        {
            ITHROWSYSTEMERROR(  GetLastError(),
                                "SetThreadPriority",
                                IBaseErrorInfo::accessError,
                                IException::recoverable);
        }
    }
#endif

    //
    //  We made it so store the new priority. We just cast it into the
    //  int storage field (which we need to do until we can get rid of the
    //  old, unportable versions under OS/2).
    //
    pThread->priorityLevel = (int)toSet;

    return *this;
}


/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::remainingStack                                         |
|                                                                              |
| Approximate stack remaining as the distance between the stack base           |
| (from the tib structure) and the current stack pointer (approximately        |
| the address of this function's argument).                                    |
------------------------------------------------------------------------------*/
unsigned long ICurrentNonGUIThread::remainingStack() const
{
#ifdef IC_PM
    tib_s* tib = this->startedThread()->tib();
    void*  base = tib->tib_pstack,
    *cur  = &tib;
    return (char*)cur - (char*)base;
#endif

#ifdef IC_WIN
    return 1000L;
#endif

#ifdef IC_POSIX
    return 0;
#endif
}

/*------------------------------------------------------------------------------
| IThread__watcher                                                             |
------------------------------------------------------------------------------*/
#ifdef IC_WIN
// static unsigned long __stdcall IThread__watcher ( void * );
// not required now
#else
#ifdef IC_POSIX
static void _System watcher(unsigned long);
#else
static void _System IThread__watcher(unsigned long);
#endif
#endif

/*------------------------------------------------------------------------------
| IStartedThread::watch                                                        |
|                                                                              |
| This function is called when a non-INonGUIThread thread is encountered in    |
| IStartedThread::startedThread.  The first time it is called it launches      |
| a secondary thread to watch for thread terminations.  This thread will       |
| see if terminating threads are non-INonGUIThread and if so, call end() to    |
| clean them up and call removeRef() to free the IStartedThread resource       |
| (if no other references exist).                                              |
|                                                                              |
| The thread watching is done by the function IThread__watcher (see below).    |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources locked.                                   |
------------------------------------------------------------------------------*/
void IStartedThread::watch()
{
    IFUNCTRACE_DEVELOP();

  // This function does nothing in Windows
#ifdef IC_PM
    TID tid;

    if ( !IStartedThread::watching )
    {
        unsigned long rc = DosCreateThread( &tid,
                                            IThread__watcher,
                                            0,
                                            0,
                                            16384 );
        if ( rc )
        {
            ITHROWSYSTEMERROR(  rc,
                                "DosCreateThread",
                                IBaseErrorInfo::accessError,
                                IException::unrecoverable );
        }
        else
        {
            IStartedThread::watching = true;
#ifdef IC_PM
            IThreadWatcher::fgTid = tid;
#endif
        }
    }
#endif
}


/*------------------------------------------------------------------------------
| IThread__watcher                                                             |
|                                                                              |
| This function watches for thread terminations.  When a thread ends, the      |
| list of started threads is searched for this thread.  If found, then we      |
| remove it from the list and call end().  Also, the reference is count        |
| is decremented (twice if PM had been started).                               |
------------------------------------------------------------------------------*/
#ifdef IC_PM
void _System IThread__watcher(unsigned long)
{
    bool      watching = true;
    unsigned  long rc = 0;

    while ( watching && rc == 0 )
    {
        IThreadId tid = 0;
        ITRACE_DEVELOP( "Waiting for a thread to end..." );
        rc = DosWaitThread( (PTID)&tid, DCWW_WAIT );

        // Check to see if a thread really ended...
        if ( rc == 0 )
        {
            ITRACE_DEVELOP( "Thread " + IString( tid.asUnsigned() ) + " ended." );
            IResourceLock lock( IStartedThread::lock() );
            unsigned index = 0;

            // Check if the application managed to start another thread
            // with the same thread ID.
            bool reusedId = IStartedThread::isTIDValid( tid );
            bool threadEnded = false;
            IStartedThread* restartedThread = 0;

            // Kill this thread unless no more not "startedHere" threads.
            watching = false;

            // Examine all started threads...
            while ( index < IStartedThread::arraySize )
            {
                IStartedThread* t = IStartedThread::threads[ index ];

                // Make sure slot is filled...
                if ( t )
                {
                    // See if this is the entry for the thread that just ended.
                    if ( t->id == tid )
                    {
                        // Thread IDs match.
                        if ( reusedId )
                        {
                            // The ended thread has been replaced with
                            // another one with the same thread ID.  Be
                            // sure not to delete the wrong one.
                            if ( restartedThread )
                            {
                                if ( t->fCount > restartedThread->fCount )
                                {
                                    // Keep the newest IStartedThread.
                                    IStartedThread* swap = t;
                                    t = restartedThread;     // End the old one.
                                    restartedThread = swap;  // Keep the new one.
                                }
                            }
                             else
                            {
                                // Decide later whether to end this one.
                                restartedThread = t;
                            }
                        }

                        if ( ! t->startedHere && t != restartedThread )
                        {
                            // Need to "end" the IStartedThread.
                            if ( restartedThread )
                            {
                                ITRACE_DEVELOP( "\tEnding oldest thread with same ID." );
                            }
                            ITRACE_DEVELOP( "\tRemoving from IStartedThread array!" );
                            t -> end(0);
                            threadEnded = true;
                        }
                    }
                     else if ( ! t->startedHere )
                    {
                        // If not started here, keep watching...
                        watching = true;
                    }
                }
                index++;
            }  // endwhile

            if (restartedThread
            &&  (threadEnded == false)
            &&  ! restartedThread->startedHere)
            {
                // Found only one IStartedThread with the ID to delete,
                // even though another thread with the same ID has been
                // started.  The IStartedThread could either be the one to
                // end (if it was not started via INonGUIThread) or the newly
                // started thread (if the ended thread was created via
                // INonGUIThread, and the new one wasn't but is now wrappered
                // with an INonGUIThread).
                ITRACE_DEVELOP( "\tOnly one of two started threads found..." );
                ITRACE_DEVELOP( "\tRemoving from IStartedThread array!" );
                restartedThread->end(0);
            }
        }
        else
        {
            ITRACE_DEVELOP( "Error watching threads, rc=" + IString( rc ) );
        }
    }

    ITRACE_DEVELOP( "No more threads to watch." );
    IStartedThread::watching = false;
}
#endif //IC_PM

/*------------------------------------------------------------------------------
| ICurrentNonGUIThread::startedThread                                          |
|                                                                              |
| Returns the result of looking up the current TID.                            |
------------------------------------------------------------------------------*/
IStartedThread *ICurrentNonGUIThread::startedThread() const
{
    IStartedThread* result = IStartedThread::locate(this->id());
    result -> removeRef();
    return result;
}

/*------------------------------------------------------------------------------
| INonGUIThread::variable                                                      |
|                                                                              |
| Dispatch this request to the corresponding started thread.                   |
------------------------------------------------------------------------------*/
IString INonGUIThread::variable(const IString &key) const
{
    return this-> startedThread() -> variable( key );
}

/*------------------------------------------------------------------------------
| INonGUIThread::setVariable                                                   |
|                                                                              |
| Dispatch this request to the corresponding started thread.                   |
------------------------------------------------------------------------------*/
INonGUIThread &INonGUIThread::setVariable(const IString &key,
                                          const IString &value)
{
    this-> startedThread() -> setVariable( key, value );
    return *this;
}

/*------------------------------------------------------------------------------
| IThreadFn::IThreadFn                                                         |
|                                                                              |
| Empty ctor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
IThreadFn::IThreadFn()
{
}

/*------------------------------------------------------------------------------
| IThreadFn::~IThreadFn                                                        |
|                                                                              |
| Empty dtor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
IThreadFn::~IThreadFn()
{
}

/*------------------------------------------------------------------------------
| INonGUIThread::current                                                       |
|                                                                              |
| Returns the current thread                                                   |
------------------------------------------------------------------------------*/
ICurrentNonGUIThread &INonGUIThread::current()
{
    IFUNCTRACE_DEVELOP();

    if ( !pCurrent )
    {
        IResourceLock lock( IStartedThread::lock() );

        // check again with resources locked to insure we don't get 2 instances
        if ( !pCurrent )
        {
           pCurrent = new ICurrentNonGUIThread;
           adoptStaticObject( pCurrent );
        }
    }

#ifdef ASC_DEBUG
    ITRACE_DEVELOP( IString("INonGUIThread::pCurrent = ") +
                    IString(pCurrent));
#endif

    return *pCurrent;
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::Cursor                                                |
|                                                                              |
| Set data members:                                                            |
|   all      - set from constructor argument                                   |
|   created  - set to 0 (which is always invalid)                              |
------------------------------------------------------------------------------*/
INonGUIThread::Cursor::Cursor(bool) :

    all( true )
#ifndef IC_POSIX
    , iteratedAll( false )
#endif
    , created( 0 )
#ifndef IC_POSIX
    , fCursorData( 0 )
#endif
{
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::~Cursor                                               |
------------------------------------------------------------------------------*/
INonGUIThread::Cursor::~Cursor()
{
    //delete fCursorData; when it is defined.
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::setToFirst                                            |
|                                                                              |
| Point to first element of started thread array and reset this cursor's       |
| time stamp.                                                                  |
------------------------------------------------------------------------------*/
bool INonGUIThread::Cursor::setToFirst()
{
    this->created = INonGUIThread::Cursor::timeStamp;
    this->cursor  = (unsigned)-1;
#ifndef IC_POSIX
    this->iteratedAll = false;
#endif
    return this->advance();
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::setToNext                                             |
|                                                                              |
| Advance cursor and return whether it's still valid.                          |
------------------------------------------------------------------------------*/
bool INonGUIThread::Cursor::setToNext()
{
    return this->advance();
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::isValid                                               |
|                                                                              |
| A cursor is valid iff its timestamp equals the static one (i.e., no          |
| threads have been started/stopped since it was reset).                       |
------------------------------------------------------------------------------*/
bool INonGUIThread::Cursor::isValid() const
{
    return this->created == INonGUIThread::Cursor::timeStamp;
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::invalidate                                            |
|                                                                              |
| Invalidate the cursor by setting it's creation "timestamp" to an invalid     |
| value.                                                                       |
------------------------------------------------------------------------------*/
void INonGUIThread::Cursor::invalidate()
{
    this->created = 0;
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::threadId                                              |
|                                                                              |
| Check validity and return TID of started thread at cursor.                   |
------------------------------------------------------------------------------*/
IThreadId INonGUIThread::Cursor::threadId() const
{
    IResourceLock lock( IStartedThread::lock() );
    IASSERTPARM( this->isValid() )
    return IStartedThread::threads[ this->cursor ] -> id;
}

/*------------------------------------------------------------------------------
| INonGUIThread::Cursor::advance                                               |
|                                                                              |
| Advance cursor to next INonGUIThread.  Skip non-PM if required.  If we run   |
| off the end, invalidate the cursor.                                          |
------------------------------------------------------------------------------*/
bool INonGUIThread::Cursor::advance()
{
    IResourceLock lock( IStartedThread::lock() );
    if ( this->isValid() )
    {
        for (this->cursor++; this->cursor < IStartedThread::arraySize;
             this->cursor++) {
            IStartedThread* p = IStartedThread::threads[ this->cursor ];
            if (p && p->id.isValid())
                if ( this->all )
                    break;
        }

        if (this->cursor == IStartedThread::arraySize) {
            // Flag no more items
            this->iteratedAll = true;
            this->invalidate();
        }
    }
    return this->isValid();
}


// Note: These functions are inlined on all platforms except Windows due to
//       variables they reference not being exported.
#ifdef IC_WIN
/*------------------------------------------------------------------------------
| INonGUIThread::defaultStatckSize                                             |
------------------------------------------------------------------------------*/
unsigned long INonGUIThread::defaultStackSize ()
{
    return INonGUIThread::dfltStackSize;
}

/*------------------------------------------------------------------------------
| INonGUIThread::setDefaultStackSize                                           |
------------------------------------------------------------------------------*/
void INonGUIThread::setDefaultStackSize ( unsigned long newStkSize )
{
    INonGUIThread::dfltStackSize = newStkSize;
}
#endif // IC_WIN





// -----------------------------------------------------------------------------
//  These old style priority methods are deprecated, only on OS/2 for
//  backwards compatibility
// -----------------------------------------------------------------------------
#ifdef IC_PM

/*------------------------------------------------------------------------------
| INonGUIThread::priorityClass                                                 |
|                                                                              |
|  Query the actual thread class.  If not available (i.e., the thread isn't    |
|  started) return the priority class that's been set).                        |
------------------------------------------------------------------------------*/
INonGUIApplication::PriorityClass INonGUIThread::priorityClass() const
{
    IFUNCTRACE_DEVELOP();

    IStartedThread *p = this->startedThread();
    if ( p->isStarted() )
    {
      unsigned t = 0;
      tib_s*   ptib = p->tib();
      if ( ptib )
        t = (unsigned)ptib->tib_ptib2->tib2_ulpri / 256;

      INonGUIApplication::PriorityClass result = INonGUIApplication::noChange;
      switch ( t )
      {
      case PRTYC_IDLETIME:
          result = INonGUIApplication::idleTime;
          break;

      case PRTYC_REGULAR:
          result = INonGUIApplication::regular;
          break;

      case PRTYC_TIMECRITICAL:
          result = INonGUIApplication::timeCritical;
          break;

      case PRTYC_FOREGROUNDSERVER:
          result = INonGUIApplication::foregroundServer;
          break;
      }

      return( result );
    }
    return( p->priorityClass );
}

/*------------------------------------------------------------------------------
| INonGUIThread::priorityLevel                                                 |
|                                                                              |
| Query the actual thread priority.  If not available (i.e., the thread        |
| isn't started) return either the priority that's been set, or, 0).           |
------------------------------------------------------------------------------*/
unsigned INonGUIThread::priorityLevel() const
{
    IFUNCTRACE_DEVELOP();

    IStartedThread* p = this->startedThread();
    if ( p->isStarted() )
    {
        tib_s *t = p->tib();
        if ( t )
            return( (unsigned)(t->tib_ptib2->tib2_ulpri % 256) );
    }

    return( ( p->priorityLevel == -1 ) ? 0 : p->priorityLevel );
}

/*------------------------------------------------------------------------------
| INonGUIThread::adjustPriority                                                |
|                                                                              |
| Adjust the thread priority level by the given delta.                         |
------------------------------------------------------------------------------*/
INonGUIThread &INonGUIThread::adjustPriority(int delta)
{
    IFUNCTRACE_DEVELOP();

    IStartedThread *p = this->startedThread();

    p->priorityLevel = delta;
    if ( p->isStarted() )
    {
        unsigned long rc =
                    DosSetPriority( PRTYS_THREAD, PRTYC_NOCHANGE, delta, p->id );
        if ( rc != NO_ERROR )
        {
            ITHROWSYSTEMERROR(  rc,
                                "DosSetPriority",
                                IBaseErrorInfo::accessError,
                                IException::recoverable );
        }
    }
    return( *this );
}

/*------------------------------------------------------------------------------
| INonGUIThread::setPriority                                                   |
|                                                                              |
| Set the priority class member of the referenced IStartedThread.              |
| In addition, issue necessary API(s) (in case the thread is started).         |
------------------------------------------------------------------------------*/
INonGUIThread&
INonGUIThread::setPriority(INonGUIApplication::PriorityClass priorityClass,
                           unsigned priorityLevel)
{
    IFUNCTRACE_DEVELOP();

    if ( priorityClass == INonGUIApplication::noChange )
        priorityClass = this->priorityClass();

    IStartedThread* p = this->startedThread();
    p->priorityClass = priorityClass;
    p->priorityLevel = priorityLevel;

    unsigned long pclass = 0;
    if ( p->isStarted() )
    {
        switch ( priorityClass )
        {
        case INonGUIApplication::idleTime:
            pclass = PRTYC_IDLETIME;
            break;

        case INonGUIApplication::regular:
            pclass = PRTYC_REGULAR;
            break;

        case INonGUIApplication::timeCritical:
            pclass = PRTYC_TIMECRITICAL;
            break;

        case INonGUIApplication::foregroundServer:
            pclass = PRTYC_FOREGROUNDSERVER;
            break;
        }
    }

    unsigned long rc = DosSetPriority(  PRTYS_THREAD,
                                        pclass,
                                        priorityLevel,
                                        p->id );
    if ( rc != NO_ERROR )
    {
        ITHROWSYSTEMERROR(  rc,
                            "DosSetPriority",
                            IBaseErrorInfo::accessError,
                            IException::recoverable );
    }
    return *this;
}


/*------------------------------------------------------------------------------
| INonGUIThread::convertedPriority()                                            |
|                                                                              |
| Returns the priority in the native OS/2 style. It converts if its is in      |
| the new style, or just returns if its in the old style (with no change to    |
| the passed class.)                                                           |
------------------------------------------------------------------------------*/
int
INonGUIThread::convertedPriority(const int toConvert,
                                 INonGUIApplication::PriorityClass& prioClass)
{
    IFUNCTRACE_DEVELOP();

    int level = toConvert;
    if ((toConvert >= INonGUIThread::minPriority)
    &&  (toConvert >= INonGUIThread::maxPriority))
    {
        switch(toConvert)
        {
        case INonGUIThread::idle :
            prioClass = INonGUIApplication::idleTime;
            level = 15;
            break;

        case INonGUIThread::belowNormal :
            prioClass = INonGUIApplication::idleTime;
            level = 31;
            break;

        case INonGUIThread::normal :
            prioClass = INonGUIApplication::regular;
            level = 0;
            break;

        case INonGUIThread::aboveNormal :
            prioClass = INonGUIApplication::regular;
            level = 15;
            break;

        case INonGUIThread::timeCritical :
            prioClass = INonGUIApplication::timeCritical;
            level = 0;
            break;
        }
    }
    return level;
}

#endif // IC_PM


IThreadsArrayDeleter::IThreadsArrayDeleter ( )
{
}

IThreadsArrayDeleter::~IThreadsArrayDeleter ( )
{
    {
        IResourceLock lock( IStartedThread::lock() );
        IStartedThread *p;
        int index;

        // Perform clean-up on all IStartedThread objects that
        // haven't for some reason been deleted at this point.
        // These objects are reference counted, but it is unlikely
        // that code that still has a reference to them will still
        // be calling functions on them. More likely, they are
        // being leaked by the other code. We will simply delete
        // the IStartedThread objects here.
#ifdef IC_POSIX
        // Deleting the IStartedThread objects is necessary to
        // avoid leaking system-wide semaphores.
#endif
        for (index = 0; index < IStartedThread::arraySize; index++) {
            p = IStartedThread::threads[index];
            if (p)
               delete p;
        }

        delete [] IStartedThread::threads;
        IStartedThread::threads = 0;
        IStartedThread::arraySize = 0;
    }
}

