// ----------------------------------------------------------------------------
// FILE NAME: iexcbase.cpp                                                   
//                                                                           
// DESCRIPTION:                                                              
//   This file contains the implementation of classes/functions declared     
//   in iexcbase.hpp.                                                        
//                                                                           
//                                                                           
// COPYRIGHT:                                                                
//   IBM Open Class Library                                                  
//   Licensed Materials - Property of IBM                                    
//                                                                           
//   5645-001                                                                
//   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.          
//                                                                           
//   US Government Users Restricted Rights - Use, duplication, or            
//   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.       
//                                                                           
// Revision: 09 1.8.2.1 source/core/base/iexcbase.cpp, exception, ioc.v400, 001006   
// ----------------------------------------------------------------------------

#include <iexcbase.hpp>

#ifdef __WINDOWS__
    #include <iplatfrm.hpp>
#endif

#ifdef __MVS__
    #include <asclocal.h>
#endif

#include <ireslock.hpp>
#include <iprimlck.hpp>
#include <istatics.hpp>

#ifdef IC_SAFE_USL
    // From USL Library
    #include <irtllock.h>
    IRTLResource writeTrace_Lock;
#endif // IC_SAFE_USL

#include <new.h>

#if defined ( _MSC_VER )
    #include <eh.h>
#elif defined ( __IBMCPP__ )
    #if defined(__WINDOWS__) || defined(__OS2__)
        #include <terminat.h>
    #else
        #include <terminate.h>
    #endif
#elif defined(_AIX)
    #include <terminate.h>
#endif

extern "C"
{
    #include <stdio.h>
    #include <string.h>
}

// Segment definitions
#ifdef IC_PAGETUNE
    #define _IEXCBASE_CPP_
    #include <ipagetun.h>
#endif

// ----------------------------------------------------------------------------
// Initialize ErrorCodeGroup strings
// ----------------------------------------------------------------------------
IException::ErrorCodeGroup const IException::baseLibrary = "IBM Class Library";
IException::ErrorCodeGroup const IException::CLibrary = "Standard C Library";
IException::ErrorCodeGroup const IException::operatingSystem = "Operating System";
IException::ErrorCodeGroup const IException::presentationSystem = "Presentation System";
IException::ErrorCodeGroup const IException::other = "Other Error Group";

// ----------------------------------------------------------------------------
//  Template for logged text
// ----------------------------------------------------------------------------
static const char* headerText = "%s exception thrown.\n  function: %s\n  file: %s\n  line: %d";


// ----------------------------------------------------------------------------
//  A dummy derivative of the standard trace function
// ----------------------------------------------------------------------------
class IExceptTraceFn : public IException::TraceFn
{
};

// ----------------------------------------------------------------------------
//  Local, static data
// ----------------------------------------------------------------------------
static IException::TraceFn* pTraceFn;
static IException::TraceFn* pUserTraceFn;


// ----------------------------------------------------------------------------
//  Lazy evaluation access methods for the statics
// ----------------------------------------------------------------------------
static IPrivateResource &traceFunction_Lock() 
{
    static IPrivateResource *theLock = 0;
    if (theLock == 0)
    {
        IPrimalLock lockInit;
        if (theLock == 0) {
            theLock = new IPrivateResource;
            adoptStaticObject(theLock);
        }
    }
    return *theLock;
}

IException::TraceFn& traceFunction()
{
    static IExceptTraceFn* pDefaultTraceFn = 0;

    // Get access to the local lock
    IResourceLock aLock(traceFunction_Lock());

    if (!pDefaultTraceFn) {
        pDefaultTraceFn = new IExceptTraceFn;
        adoptStaticObject(pDefaultTraceFn);
    }
    if(pUserTraceFn)
        return *pUserTraceFn;

    if(!pTraceFn)
        pTraceFn = pDefaultTraceFn;

  return *pTraceFn;
}



// ----------------------------------------------------------------------------
// IExcText Class
// ----------------------------------------------------------------------------
class IExcText
{
public:
    IExcText(const char* errText, const IExcText* msgtxtOld);
    IExcText(const IExcText& msg);

    ~IExcText ( ) 
    {
        if (pszClMsg)
            delete [] pszClMsg;
        if (msgtxtClNext)
            delete msgtxtClNext; 
    }

    const char* text() const
    {
        return pszClMsg;
    }

    IExcText* next() const
    {
        return msgtxtClNext;
    }

    void appendText(const char* errText);

    char* pszClMsg;
    IExcText* msgtxtClNext;
}; // IExcText


// ----------------------------------------------------------------------------
// IExcText Member Functions
// ----------------------------------------------------------------------------
IExcText :: IExcText ( const char* errText,
                       const IExcText* msgtxtOld )
{
    pszClMsg = 0;
    try 
    {
        pszClMsg = new char[strlen(errText)+1];
        strcpy(pszClMsg, errText);
    }
    catch (...) { }

    msgtxtClNext = (IExcText*)msgtxtOld;
}


IExcText :: IExcText ( const IExcText& msg )
{
    try {
       pszClMsg = new char[strlen(msg.pszClMsg)+1];
       strcpy(pszClMsg, msg.pszClMsg);
       msgtxtClNext = 0;
       if (msg.msgtxtClNext)
          msgtxtClNext = new IExcText(*(msg.msgtxtClNext));
    }
    catch (...)
    {
       pszClMsg = msg.pszClMsg;
       msgtxtClNext = msg.msgtxtClNext;
    }
}

void IExcText :: appendText ( const char* errText )
{
    try 
    {
       char* pszText;
       if (pszClMsg)
       {
          pszText = new char[strlen(pszClMsg)+strlen(errText)+1];
          strcpy(pszText, pszClMsg);
          delete [] pszClMsg;
          strcat(pszText, errText);
          pszClMsg = pszText;
       }
       else
       {
          pszText = new char[strlen(errText)+1];
          strcpy(pszText, errText);
          pszClMsg = pszText;
       }
    }
    catch (...) {}
}

// ----------------------------------------------------------------------------
//  IExceptionLocation Member Functions
// ----------------------------------------------------------------------------


IExceptionLocation :: IExceptionLocation ( const char* fileName,
                                           const char* functionName,
                                           unsigned long lineNumber)
                      : pClFile(fileName), pClFunction(functionName),
                        ulClLineNo(lineNumber)
{
   ;
}

const char* IExceptionLocation :: fileName ( ) const
{
  return pClFile;
}

const char* IExceptionLocation :: functionName ( ) const
{
   if (pClFunction)
      return pClFunction;
   return "Unknown";
}

unsigned long IExceptionLocation :: lineNumber ( ) const
{
  return ulClLineNo;
}


// ----------------------------------------------------------------------------
//  IException Member Functions
// ----------------------------------------------------------------------------
IException::IException(const char* exMsgText,
                       unsigned long errorId,
                       Severity exSev)
                 : ulClErrorId(errorId), exsevCl(exSev),
                   ulexlocClCount(0), fErrorGroup(baseLibrary)
{
   ulClTxtLvlCount = 0;
   msgtxtClTop = 0;
   try {
      if (exMsgText != 0) {
          msgtxtClTop = new IExcText(exMsgText, 0);
          ulClTxtLvlCount = 1;
      }
   }
   catch (...) {};
}

IException::IException(const IException& exc)
{
   fErrorGroup = exc.fErrorGroup;
   exsevCl = exc.exsevCl;
   ulClErrorId = exc.ulClErrorId;
   ulexlocClCount = exc.ulexlocClCount;
   ulClTxtLvlCount = exc.ulClTxtLvlCount;
   if (exc.msgtxtClTop != 0)
     msgtxtClTop = new IExcText(*(exc.msgtxtClTop));
   else msgtxtClTop = 0;
   for (int i=0; i<exc.locationCount(); i++ )
   {
      exlocClArray[i] = exc.exlocClArray[i];
   } /* endfor */
}

IException::~IException()
{
   if (msgtxtClTop)
     delete msgtxtClTop;
}

IException& IException::setErrorCodeGroup ( ErrorCodeGroup errorGroup )
{
   fErrorGroup = errorGroup;
   return *this;
}

IException::ErrorCodeGroup IException::errorCodeGroup ( ) const
{
   return fErrorGroup;
}

IException& IException::appendText(const char* exMsgText)
{
   if (exMsgText != 0)
   {
      if (msgtxtClTop)
        msgtxtClTop->appendText(exMsgText);
      else
        setText(exMsgText);
   }
   return *this;
}

IException& IException::setText(const char* exMsgText)
{
    try
    {
       if (exMsgText != 0)
       {
          msgtxtClTop = new IExcText(exMsgText, msgtxtClTop);
          ulClTxtLvlCount++;
       }
    }
    catch (...) {};

    return *this;
}

IException& IException::setSeverity(Severity exsev)
{
   exsevCl = exsev;
   return *this;
}

IException& IException::setErrorId(unsigned long errorId)
{
   ulClErrorId = errorId;
   return *this;
}

IException& IException::addLocation(const IExceptionLocation& exloc)
{
   if(ulexlocClCount == 5)
      ulexlocClCount--;          // reset to fill in last entry
   exlocClArray[ulexlocClCount] = exloc;
   ulexlocClCount++;
   return *this;
}

int IException::isRecoverable() const
{
   if(exsevCl)
      return 1;

   return 0;
}

const char* IException::name() const
{
   return "IException";
}

const char* IException::text(unsigned long indexFromTop) const
/****************************************************************/
/* Returns the exception text from an entry in the stack.       */
/****************************************************************/
{
  IExcText* tmp = msgtxtClTop;
  if (textCount() > indexFromTop)
  {
     for (int i = 0; i < indexFromTop; ++i)
     {
        tmp = tmp->next();
     } /* endfor */
     return tmp->text();
  }
  else
     return 0;
}

unsigned long  IException::textCount() const
{
   return ulClTxtLvlCount;
}

unsigned long IException::errorId() const
{
  return ulClErrorId;
}

unsigned long IException :: locationCount() const
{
   return ulexlocClCount;
}

const IExceptionLocation*
              IException :: locationAtIndex(unsigned long locIndex) const
{
    if (locIndex < ulexlocClCount)
    {
        return &(exlocClArray[locIndex]);
    }
    return 0;
}

IException& IException :: logExceptionData ( )
{
    traceFunction().logData(*this);
    return *this;
}

IException::TraceFn*
IException::setTraceFunction(IException::TraceFn& newFunction)
{
  IResourceLock aLock(traceFunction_Lock());

  IException::TraceFn* oldFunction = &(traceFunction());
  pUserTraceFn = &newFunction;
  return oldFunction;
}

void IException::terminate()
{
    ::terminate();
}

void IException::assertParameter(   const char*         exceptionText,
                                    IExceptionLocation  location)
{
    IAssertionFailure invParm(exceptionText);
    invParm.addLocation(location);
    invParm.setErrorCodeGroup(other);
    invParm.logExceptionData();
    throw invParm;
}


// ----------------------------------------------------------------------------
// IException::TraceFn functions.
// ----------------------------------------------------------------------------
IException::TraceFn::TraceFn()
{
}

void IException::TraceFn::exceptionLogged()
{
}

void IException::TraceFn::write(const char* buffer)
{
#ifdef __WINDOWS__
    if (!IPlatform::isWinNT())
        fprintf(stdout, "%s\n", buffer);
    else
#endif
        fprintf(stderr,"%s\n", buffer);
}

void IException::TraceFn::logData(IException& exception)
{
  // Determine the largest buffer we will need.
  unsigned long maxLength = 200;
  unsigned long length = 0;
  if (exception.ulexlocClCount != 0)
     length = strlen(headerText) + 1 +
              strlen(exception.name()) +
  strlen(exception.exlocClArray[exception.ulexlocClCount - 1].functionName()) +
  strlen(exception.exlocClArray[exception.ulexlocClCount - 1].fileName()) + 8;
  if (length > maxLength)
     maxLength = length;
  if (exception.msgtxtClTop)
  {
     for (int i = (exception.textCount()-1); i >= 0 ;i-- )
       {
          length = strlen(exception.text(i)) + 6;
          if (length > maxLength)
             maxLength = length;
       } /* endfor */
  }

  char * pbuf = 0;
  try 
  {
      pbuf = new char[maxLength + 1];
  }
  catch (...) {};

  if (pbuf)
  {
        if (exception.ulexlocClCount != 0)
        {
            sprintf(pbuf,headerText,
            exception.name(),
            exception.exlocClArray[exception.ulexlocClCount - 1].functionName(),
            exception.exlocClArray[exception.ulexlocClCount - 1].fileName(),
            exception.exlocClArray[exception.ulexlocClCount - 1].lineNumber());
        }
         else
        {
            sprintf(pbuf,
                    "\n%s exception thrown.\n   Location information is unavailable.",
                    exception.name());
        }

        traceFunction().write(pbuf);
        if (exception.errorId())
            sprintf(pbuf,"   Error Id is %d", exception.errorId());
        else
            sprintf(pbuf,"   Error Id is unavailable.");
        traceFunction().write(pbuf);

        if (exception.errorCodeGroup())
            sprintf(pbuf,"   Error Code group is %s", exception.errorCodeGroup());
        else
            sprintf(pbuf,"    Error Code group is unavailable.");

        traceFunction().write(pbuf);
        if (exception.msgtxtClTop)
        {
            sprintf(pbuf,"   Exception text is:");
            traceFunction().write(pbuf);
            for (int i = (exception.textCount()-1); i >= 0 ;i-- )
                {
                sprintf(pbuf,"      %s", exception.text(i));
                traceFunction().write(pbuf);
                } /* endfor */
        }
        else
        {
            sprintf(pbuf,"   Exception text is unavailable.");
            traceFunction().write(pbuf);
        }
        delete [] pbuf;
        exceptionLogged();
    } 
    else
    {
        // New failed, no buffer available
#ifdef IC_WIN
        if (!IPlatform::isWinNT())
        {
            if (exception.ulexlocClCount != 0)
            {
                fprintf(stdout,
                        "%s exception thrown.\n   function: %s\n   file: %s\n   line: %d.\n",
                        exception.name(),
                        exception.exlocClArray[exception.ulexlocClCount - 1].functionName(),
                        exception.exlocClArray[exception.ulexlocClCount - 1].fileName(),
                        exception.exlocClArray[exception.ulexlocClCount - 1].lineNumber());
            }
             else
            {
                fprintf(stdout,
                        "\n%s exception thrown.\n   Location information is unavailable.\n",
                        exception.name());
            }

            if (exception.errorId())
                fprintf(stdout,"   Error ID is %d\n", exception.errorId());
            else
                fprintf(stdout,"   Error ID is unavailable.\n");

            if (exception.errorCodeGroup())
                fprintf(stdout, " Error Code group is %s\n", exception.errorCodeGroup());
            else
                fprintf(stdout,"   Error Code group is unavailable.\n");

            if (exception.msgtxtClTop)
            {
                fprintf(stdout,"   Exception text is:\n");
                for (int i = (exception.textCount()-1); i >= 0 ; i--)
                {
                    fprintf(stdout,"      %s\n", exception.text(i));
                }
            }
             else
                fprintf(stdout,"   Exception text is unavailable.\n");
        }
         else
        {
#endif
            if (exception.ulexlocClCount != 0)
            {
                fprintf(stderr,
                        "%s exception thrown.\n   function: %s\n   file: %s\n   line: %d.\n",
                        exception.name(),
                        exception.exlocClArray[exception.ulexlocClCount - 1].functionName(),
                        exception.exlocClArray[exception.ulexlocClCount - 1].fileName(),
                        exception.exlocClArray[exception.ulexlocClCount - 1].lineNumber());
            }
             else
            {
                fprintf(stderr,
                        "\n%s exception thrown.\n   Location information is unavailable.\n",
                        exception.name());
            }

            if (exception.errorId())
                fprintf(stderr,"   Error ID is %d\n", exception.errorId());
            else
                fprintf(stderr,"   Error ID is unavailable.\n");

            if (exception.errorCodeGroup())
                fprintf(stderr, " Error Code group is %s\n", exception.errorCodeGroup());
            else
                fprintf(stderr,"   Error Code group is unavailable.\n");

            if (exception.msgtxtClTop)
            {
                fprintf(stderr,"   Exception text is:\n");
                for (int i = (exception.textCount()-1); i >= 0 ;i-- )
                {
                    fprintf(stderr,"      %s\n", exception.text(i));
                }
            }
            else
            {
                fprintf(stderr,"   Exception text is unavailable.\n");
            }
#ifdef IC_WIN
        }
#endif
    }
}

// ----------------------------------------------------------------------------
// Definition of IException subclasses using macro.
// ----------------------------------------------------------------------------
IEXCLASSIMPLEMENT(IAccessError,IException);
IEXCLASSIMPLEMENT(IAssertionFailure,IException);
IEXCLASSIMPLEMENT(IDeviceError,IException);
IEXCLASSIMPLEMENT(IInvalidParameter,IException);
IEXCLASSIMPLEMENT(IInvalidRequest,IException);
IEXCLASSIMPLEMENT(IResourceExhausted,IException);
IEXCLASSIMPLEMENT(IOutOfMemory,IResourceExhausted);
IEXCLASSIMPLEMENT(IOutOfSystemResource,IResourceExhausted);
IEXCLASSIMPLEMENT(IOutOfWindowResource,IResourceExhausted);
IEXCLASSIMPLEMENT(IDecimalDataError,IException);
