/*----------------------------------------------------------------------------*/
/* FILE NAME: idecimal.cpp                                                    */
/*                                                                            */
/* DESCRIPTION:                                                               */
/*   This file contains the implementation of the utility functions of        */
/*   the class IBinaryCodedDecimal, declared in idecimal.hpp.                 */
/*                                                                            */
/*                                                                            */
/* COPYRIGHT:                                                                 */
/*   IBM Open Class Library                                                   */
/*   Licensed Materials - Property of IBM                                     */
/*                                                                            */
/*   5645-001                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
/*                                                                            */
/*   US Government Users Restricted Rights - Use, duplication, or             */
/*   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.        */
/*                                                                            */
/*----------------------------------------------------------------------------*/
#ifdef __MVS__
    #pragma csect(code,"IDECIMAL")
    #pragma csect(static,"SDECIMAL")
    #pragma comment (copyright, \
    "Licensed Materials - Property of IBM  5645-001 (C) \
    Copyright IBM Corp. 1992, 1997  All Rights Reserved. \
    US Government Users Restricted Rights - Use, duplication, or \
    disclosure restricted by GSA ADP Schedule Contract with IBM Corp.")
#endif

/*
This describes the design considerations for IBinaryCodedDecimal.

A). Storage Structures

First we look at the how the BCD is stored :

 - IBinaryCodedDecimal (15,5), the storage is :

   12|34|56|78|90|01|02|5F  represents 1234567890.01025
   00|00|00|00|01|41|40|0D  represents -1.414
                 ^
                 |
    decimal point is at byte boundary


 - IBinaryCodedDecimal (14,4), the storage is :
   01|23|45|67|89|00|10|2F  represents 1234567890.0102
   00|00|00|00|00|14|14|0D  represents -1.414
                  ^
                  |
    decimal point is in the middle of a byte

One observation that can be made from this is, when the precision of
a IBinaryCodedDecimal is odd, the decimal point is align at byte boundary.
This observation is used throughout the code when manipulating
IBinaryCodedDecimal.

Also it can be seen, operations between IBinaryCodedDecimal of different size
will require location and alignment of the decimal points as well as
determination of the proper sign.

In view of this, a utility class IDecimalUtil is created to store all
IBinaryCodedDecimal objects that are in transition (while it is being
being operated on). This utility class also owns all the functions
that are used in the implementation of IBinaryCodedDecimal.

All works except constructions of IBinaryCodedDecimal are delegated to
IDecimalUtil. During the conversion of IBinaryCodedDecimal to IDecimalUtil,
the decimal point is aligned in a fixed byte boundary and
the sign is stored seperately.

To keep maximum precision, IDecimalUtil is set to store the product
of any two IBinaryCodedDecimal of any size.

  the integral part of IDecimalUtil can store 62 digits
  the fraction part of IDecimalUtil can store 31 digits

So the storage structure of IDecimalUtil is :

   |  |  |...|  |  |.|  |  |...|  |  | , nitgrl, nfrtnl, positive

   <-  32 bytes  -> ^ <-  16 bytes  ->
                    |
            implicit decimal point

  nitgrl   - left most digit position at the integral part that is non zero
  nitgrl   - right most digit position at the fractional part that is non zero
  positive - if the number is positive, true, else false

B). Constructions of IBinaryCodedDecimal

IBinaryCodedDecimal can be constructed from any native data types. If no
size and precision information is given, the default will be used.

C). Constructions of IDecimalUtil

IDecimalUtil can be constructed from any native data types. These
constructors are to be used implicitly when native data types
and IBinaryCodedDecimal are used in the same operation.

D). Logical operations.

In logical operations, the values of nitgrl, nfrtnl and positive are first
checked in attempt to determine the result first. This saves time if
actual bitwise comparison can be avoided.

E). Input/Output operations.

Since in the convertion of IBinaryCodedDecimal to IDecimalUtil, the actual size
of IBinaryCodedDecimal is determined and the decimal point aligned, I/O
operation of IBinaryCodedDecimal actually converts itself to a IDecimalUtil
and triggers the I/O operations of the corresponding IDecimalUtil.

F). Arithmetic operations.

The core routine in IDecimalUtil that are being used in all arithmetic
operations are IDecimalUtil::decAddWord(). This routine is taken from
Rochester and it adds two long word and cause carry from one nibble
to another when the nibble value exceeds 10 and with the carryout discarded
for the most significant nibble like decimal addition.

so for 0x00412213 + 0x34314300, the result will be 0x34726513.
   and 0x70412813 + 0x34314300, the result will be 0x04727113.

To make use of this routine, 8 BCD nibbles are converted to long using shifts
and rotates to overcome indian-ness. (IDecimalUtil::genuLong() to generate and
IDecimalUtil::genBCD() to reverse it).

Lets look at each operation :

operator + (const IDecimalUtil &lv, const IDecimalUtil &rv)
 - Determines the final sign and whether the magnitudes of the operands
   should be added together or subtracted.
 - pick up the maximum # of digits to add and the maximum # of word (4 bytes)
   that will contain these digits

operator - (const IDecimalUtil &lv, const IDecimalUtil &rv)
 - Determines the final sign and whether the magnitudes of the operands
   should be added together or subtracted. If lv's magnitude is smaller
   then rv's magnitude, swap.

operator * (const IDecimalUtil &lv, const IDecimalUtil &rv)
 - Detemines the final sign and use the number with more significant
   digits as the multiplicant and the number with less significant
   digits as multiplier. This will mean less addition and less nibble
   shifting during operation.

operator / (const IDecimalUtil &lv, const IDecimalUtil &rv)
 - Determines the sign and the final decimal position. During division
   the two numbers are treated as a whole number, so after division,
   the decimal point has to be re-adjusted.

IDecimalUtil::addBCD()

 - add two words at a time start at the right hand side (least significant
   digits) and propagate the carry.
 - a carry is required if the result of the addition is smaller than
   either of the two operands.

IDecimalUtil::subBCD()
 - the routine expects the result is alway positive (that is the right
   hand side is alway bigger that the left hand side) so that borrow
   does not go beyond the last nibble.
 - generate the 10's complement of the right operand a word at a time.
   10's complement is = bitwise flip of the original value and then
                        subtract 6 from each nibble and add 1.

 - subtract two words at a time start at the right hand side. If borrow,
   add  10's complement of 1. A borrow is required if the result is
   bigger than the right operand.

IDecimalUtil::mulBCD()
 - generate 1x, 2x and 4x of multiplicant, this is the base of the
 - for each nibble of the multiplier starting from the most significant
   nibble, add the appropriate # of time the
   multiplicant, keeping carryouts, since we are doing decimal multiplication
   carryout will never exceed 1 nibble.
 - left shift which is the equivalent of 10x the resulting multiplicant.
   keeping the shifted out value in sync with the carryout values.
 - at the end, add the carryout and the shifted out digits, this becomes
   the most significant digits of the result.

IDecimalUtil::divBCD()
 - generate 10's complement of the denominator..
 - left align the two operand, compare and subtract (add 10's complement)
 - left shift the numerator until it is bigger than denominator.
 - Do this until we have DEC_PREC digits of precision or it result is 0.
*/

#ifdef __MVS__
#include <ibhandle.hpp>  // Need _OPEN_THREADS for thread-safe version of USL
#endif // __MVS__

extern "C"
   {
   #include <string.h>
   #include <float.h>
   #include <stdlib.h>
   }

#include <iexcbase.hpp>
#include <idecimal.hpp>
#include <icconsta.h>

#ifdef IC_PAGETUNE
   #define _IDECIMAL_CPP_
   #include <ipagetun.h>
#endif

IBinaryCodedDecimal::IBinaryCodedDecimal(const char *val) :

    numDig(0),
    numPrec(0)
{
  int len = strlen(val);
  char *cptr = new char[len+1];
  if (cptr) {
    memcpy(cptr, val, len+1);

    char *p = cptr;
    if ((*p == '-') || (*p == '+')) {
      ++p;
      --len;
    }

    int numOfDotSeen = 0;
    for (int idx = 0; idx < len; ++idx, ++p) {
      if ((*p > '9') || (*p < '0')) {
        if ((*p == '.') && (++numOfDotSeen < 2)) {
          continue;
        }
        *p = '\0';
        break;
      }
    }

    memset(buf, '\0', sizeof(buf));
    IDecimalUtil::cptrToDec(cptr, numDig, numPrec, buf, false);
    delete[] cptr;
  }
}

IBinaryCodedDecimal::IBinaryCodedDecimal(float afloat)
: numDig(0), numPrec(0)
{
   if ((DEC_MAX.asDouble() < afloat) ||
       (DEC_MIN.asDouble() > afloat)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   memset(buf, '\0', sizeof(buf));
   char tmpBuf[DEC_DIG+FLT_DIG+3];
   sprintf(tmpBuf, "%.*f", FLT_DIG, afloat);
   IDecimalUtil::cptrToDec(tmpBuf, numDig, numPrec, buf, false);
}

IBinaryCodedDecimal::IBinaryCodedDecimal(double aDouble)
: numDig(0), numPrec(0)
{
   if ((DEC_MAX.asDouble() < aDouble) ||
       (DEC_MIN.asDouble() > aDouble)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   memset(buf, '\0', sizeof(buf));
   char tmpBuf[DEC_DIG+DBL_DIG+3];
   sprintf(tmpBuf, "%.*f", DBL_DIG,  aDouble);
   IDecimalUtil::cptrToDec(tmpBuf, numDig, numPrec, buf, false);
}

IBinaryCodedDecimal::IBinaryCodedDecimal(long double alDouble)
: numDig(0), numPrec(0)
{
   if ((DEC_MAX.asDouble() < alDouble) ||
       (DEC_MIN.asDouble() > alDouble)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   memset(buf, '\0', sizeof(buf));
   char tmpBuf[DEC_DIG+LDBL_DIG+3];
   sprintf(tmpBuf, "%.*Lf", LDBL_DIG, alDouble);
   IDecimalUtil::cptrToDec(tmpBuf, numDig, numPrec, buf, false);
}

IBinaryCodedDecimal &
IBinaryCodedDecimal::operator=(const IDecimalUtil &dut)
{
   memset(buf, '\0', sizeof(buf));
   if (dut.nitgrl == 0 && dut.nfrtnl == 0) {
      setPositive();
      return *this;
   }
   char *cptr;
   const char *ccptr;
   int idx;
   int numItgrl = numDig - numPrec;

   if (dut.nitgrl > numItgrl) {
      IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
      ITHROW(exc);
   }
   if (numPrec & 0x0001) {
      // odd number of precision, so the integral & fraction part is aligned
      // at a byte boundary
      memcpy(buf, dut.bcdData + ((MAX_ITGRL + 1) >> 1) - ((numItgrl + 1) >> 1),
             (numDig >> 1) + 1);
      // mask off the leading nibble if odd # of integral part
      if (numItgrl & 0x0001)
         buf[0] &= 0x0F;
   } else {
      cptr = buf + (numDig >> 1);
      ccptr = dut.bcdData + ((MAX_ITGRL + 1)  >> 1) + (numPrec >> 1) - 1;
      *(cptr--) = *ccptr << 4;
      for (idx = numDig - 1; idx > 1; idx -= 2) {
          *cptr = (*(ccptr--) & 0xF0) >> 4;
          *(cptr--) |= *ccptr << 4;
      }
      if (idx) {
         *cptr = (*ccptr & 0xF0) >> 4;
      }
   }
   setPositive();
   if (!dut.positive)
      setNegative();
   return *this;
}

#ifdef IC_STREAMING
void IBinaryCodedDecimal :: writeToStream( IDataStream &toWhere ) const {
  toWhere.writeBytes(buf, sizeof(buf));
  (long) numDig  >>= toWhere;
  (long) numPrec >>= toWhere;
}

void IBinaryCodedDecimal :: readFromStream( IDataStream &fromWhere ) {
  fromWhere.readBytes(buf, sizeof(buf));
  (long&) numDig  <<= fromWhere;
  (long&) numPrec <<= fromWhere;
}
#endif

//IDecimalUtil functions

IDecimalUtil::IDecimalUtil(int anInt)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (anInt == 0) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if (anInt == 1 || anInt == -1) { // to speed up increment/decrement
      nitgrl = 1; nfrtnl = 0;
      bcdData[((MAX_ITGRL + 1) >> 1) - 1] = 0x01;
      positive = anInt > 0 ? true : false;
      return;
   }
   char tmpBuf[DEC_DIG+2];
   sprintf(tmpBuf,"%d", anInt);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}
IDecimalUtil::IDecimalUtil(unsigned int auInt)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (auInt == 0) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if (auInt == 1) {
      nitgrl = 1; nfrtnl = 0;
      bcdData[((MAX_ITGRL + 1) >> 1) - 1] = 0x01;
      positive = true;
      return;
   }
   char tmpBuf[DEC_DIG+2];
   sprintf(tmpBuf,"%u", auInt);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}
IDecimalUtil::IDecimalUtil(long aLong)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (aLong == 0L) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if (aLong == 1L || aLong == -1L) {
      nitgrl = 1; nfrtnl = 0;
      bcdData[((MAX_ITGRL + 1) >> 1) - 1] = 0x01;
      positive = aLong > 0L ? true : false;
      return;
   }
   char tmpBuf[DEC_DIG+2];
   sprintf(tmpBuf,"%ld", aLong);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}
IDecimalUtil::IDecimalUtil(unsigned long auLong)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (auLong == 0L) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if (auLong == 1L) {
      nitgrl = 1; nfrtnl = 0;
      bcdData[((MAX_ITGRL + 1) >> 1) - 1] = 0x01;
      positive = true;
      return;
   }
   char tmpBuf[DEC_DIG+2];
   sprintf(tmpBuf,"%lu", auLong);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}
IDecimalUtil::IDecimalUtil(float afloat)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (afloat == 0.0f) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if ((DEC_MAX.asDouble() < afloat) ||
       (DEC_MIN.asDouble() > afloat)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   char tmpBuf[DEC_DIG+FLT_DIG+3];
   sprintf(tmpBuf, "%.*f", FLT_DIG, afloat);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}
IDecimalUtil::IDecimalUtil(double aDouble)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (aDouble == 0.0) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if ((DEC_MAX.asDouble() < aDouble) ||
       (DEC_MIN.asDouble() > aDouble)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   char tmpBuf[DEC_DIG+DBL_DIG+3];
   sprintf(tmpBuf, "%.*f", DBL_DIG, aDouble);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}
IDecimalUtil::IDecimalUtil(long double alDouble)
{
   memset(bcdData, '\0', sizeof(bcdData));
   if (alDouble == 0.0L) {
      nitgrl = nfrtnl = 0;
      positive = true;
      return;
   }
   if ((DEC_MAX.asDouble() < alDouble) ||
       (DEC_MIN.asDouble() > alDouble)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   char tmpBuf[DEC_DIG+LDBL_DIG+3];
   sprintf(tmpBuf, "%.*Lf", LDBL_DIG, alDouble);
   positive = IDecimalUtil::cptrToDUT(tmpBuf, nitgrl, nfrtnl, bcdData);
   trimZero();
}

IDecimalUtil::IDecimalUtil(const IBinaryCodedDecimal &bcd)
{
   int idx;
   char *cptr;
   const char *ccptr;
   int nDig = bcd.digitsOf();

   positive = bcd.isPositive();
   nfrtnl = bcd.precisionOf();
   nitgrl = nDig - nfrtnl;
   memset(bcdData, '\0', sizeof(bcdData));

   if (nfrtnl & 0x0001) {
      memcpy(bcdData + ((MAX_ITGRL + 1) >> 1) - ((nitgrl + 1) >> 1),
             bcd.cData(), (nDig >> 1) + 1);
      bcdData[((MAX_ITGRL + 1) >> 1) + (nfrtnl >> 1)] &= 0xF0;
   } else {
      ccptr = bcd.cData() + (nDig >> 1);
      cptr = bcdData + ((MAX_ITGRL + 1) >> 1) + (nfrtnl >> 1) - 1;
      *cptr = *(ccptr--) >> 4;
      for (idx = nDig - 1; idx > 1; idx -= 2) {
             *(cptr--) |= (*ccptr & 0x0F) << 4;
             *cptr = *(ccptr--) >> 4;
      }
      if (idx) {
         *cptr |= (*ccptr & 0x0F) << 4;
      }
   }
   trimZero();
}

void IDecimalUtil::throwOverflow()
{
   IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
      IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
   ITHROW(exc);
}

void IDecimalUtil::trimZero()
{
   // trim the # of precision and significant digits
   const char *cptr;
   int frtnlByte = (nfrtnl + 1) >> 1;
   int itgrlByte = (nitgrl + 1) >> 1;
   if (frtnlByte) {
      cptr = bcdData + ((MAX_ITGRL + 1) >> 1) + frtnlByte - 1;
      while (frtnlByte > 0 && *cptr == '\0') {
         frtnlByte--;
         cptr--;
      }
      if (frtnlByte > 0) {
         nfrtnl = frtnlByte << 1;
         if (!(*cptr & 0x0F))
            nfrtnl--;
      } else {
         nfrtnl = 0;
      }
   }
   if (itgrlByte) {
      cptr = bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlByte;
      while (itgrlByte > 0 && *cptr == '\0') {
         itgrlByte--;
         cptr++;
      }
      if (itgrlByte > 0) {
         nitgrl = itgrlByte << 1;
         if (!(*cptr & 0xF0))
            nitgrl--;
      } else {
         nitgrl = 0;
      }
   }
}

IDecimalUtil::operator double() const
{
   char output[DEC_DIG + 10];
   const char *ccptr;
   char *cptr = output;
   int idx;
   if (!positive)
      *(cptr++) = '-';
   if (nitgrl | nfrtnl) {
      if (nitgrl) {
         ccptr = bcdData + ((MAX_ITGRL + 1) >> 1) - ((nitgrl + 1) >> 1);
         if (*ccptr & 0xF0)
            *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
         for (idx = nitgrl - 1; idx > 1; idx -= 2) {
            *(cptr++) = (*(ccptr++) & 0x0F) + '0';
            *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
         }
         *(cptr++) = (*(ccptr++) & 0x0F) + '0';
      } else {
         *(cptr++) = '0';
      }
      if (nfrtnl) {
         *(cptr++) = '.';
         ccptr = bcdData + ((MAX_ITGRL + 1) >> 1);
         for (idx = nfrtnl; idx > 2; idx -= 2) {
            *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
            *(cptr++) = (*(ccptr++) & 0x0F) + '0';
         }
         *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
         if (*ccptr & 0x0F)
            *(cptr++) = (*(ccptr++) & 0x0F) + '0';
      }
   } else {
      *(cptr++) = '0';
   }
   *cptr = '\0';
   return atof((char *)output);
}

IDecimalUtil::operator long () const
{
   char output[DEC_DIG + 10];
   const char *ccptr;
   char *cptr = output;
   int idx;
   if (!positive)
      *(cptr++) = '-';
   if (nitgrl) {
      ccptr = bcdData + ((MAX_ITGRL + 1) >> 1) - ((nitgrl + 1) >> 1);
      if (*ccptr & 0xF0)
         *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
      for (idx = nitgrl - 1; idx > 1; idx -= 2) {
         *(cptr++) = (*(ccptr++) & 0x0F) + '0';
         *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
      }
      *(cptr++) = (*(ccptr++) & 0x0F) + '0';
   } else {
      *(cptr++) = '0';
   }
   *cptr = '\0';
   return atol((char *)output);
}

IDecimalUtil::operator IBinaryCodedDecimal() const
{

   if (!(nitgrl | nfrtnl)) {
      return IBinaryCodedDecimal();
   }
   int tmpItgrl = nitgrl;
   int tmpFrtnl = nfrtnl;
   int tmpDig = nitgrl + nfrtnl;
   if (tmpDig > DEC_DIG)
      tmpDig = DEC_DIG;
   tmpFrtnl = tmpDig - tmpItgrl;

   if (tmpFrtnl < 0)
      tmpFrtnl = 0;
   if (tmpItgrl > DEC_DIG)
      tmpItgrl = DEC_DIG;

   IBinaryCodedDecimal tmpBCD(tmpDig, tmpFrtnl);
   tmpBCD = *this;
   return tmpBCD;
}

// static functions
// NOTE: - if updating numDig and numPrec is required,
//         set hasDigPrec to false (default is true).
//       - the required memory from buffer 'buf' is
//         assumed to be initialized to zero before
//         calling this function
//       - throw exception if numDig and/or numPrec are
//         not in the valid range
void IDecimalUtil::cptrToDec(const char *val, int &numDig,
                             int &numPrec, char *buf, bool hasDigPrec)
{
   if ( hasDigPrec &&
       ( (numDig  <= 0) || (numDig  > DEC_DIG) ||
         (numPrec < 0)  || (numPrec > DEC_DIG) ||
         (numPrec > numDig) ) ) {

     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }

   bool positive = true;
   char tmpBuf[DEC_DIG];
   const char *cptr;
   int   nDec, nDig, nPrec;
   int   notAllZero = 0;
   int   idx, idx2;

   // take care of the possible sign
   if (val[0] == '-') {
      positive = false;
      val++;
   } else if (val[0] == '+')
       val++;

   const char *decPosPtr = strchr(val, '.');
   nDig = strlen(val);
   memset(tmpBuf, '\0', sizeof(tmpBuf));


   if (decPosPtr == NULL) {
      nPrec = 0;
      for (; nDig > DEC_DIG && *val == '0'; val++, nDig--)
          ;
      if (nDig > DEC_DIG) {
         IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
         ITHROW(exc);
      }
      nDec = nDig;
      if (!positive) {
         for (idx = 0, cptr = val; idx < nDig; idx++, cptr++)
            notAllZero |= (tmpBuf[idx] = (*cptr - '0'));
      } else {
         for (idx = 0, cptr = val; idx < nDig; idx++, cptr++)
            tmpBuf[idx] = (*cptr - '0');
      }
   } else {
      // if decimal point exist, try to get as much precision as
      // possible
      nDig--;
      for (; nDig > DEC_DIG && *val == '0'; val++, nDig--)
          ;
      if (nDig > DEC_DIG) {
         nDig = DEC_DIG;
      }
      nPrec = nDig - (int)(decPosPtr - val);
      if (nPrec < 0)
      {
         IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
         ITHROW(exc);
      }
      nDec = nDig - nPrec;
      if (!positive) {
         for (idx = 0, cptr = val; idx < nDec; idx++, cptr++)
            notAllZero |= (tmpBuf[idx] = (*cptr - '0'));
         cptr++;
         for (; idx < nDig; idx++, cptr++)
            notAllZero |= (tmpBuf[idx] = (*cptr - '0'));
      } else {
         for (idx = 0, cptr = val; idx < nDec; idx++, cptr++)
            tmpBuf[idx] = (*cptr - '0');
         cptr++;
         for (; idx < nDig; idx++, cptr++)
            tmpBuf[idx] = (*cptr - '0');
      }
   }

   // Modify the value for numDig and numPrec only if we
   // are told to do so (i.e. hasDigPrec is false)
   if (hasDigPrec) {
     int numDec = numDig - numPrec;

     // Are we truncating the integral part of the number?
     if (nDec > numDec) {
       IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
       ITHROW(exc);
     }
   } else {
     numDig  = nDig;
     numPrec = nPrec;
   }

   if (numDig == 0) {
     // if no digit is given, exception is thrown!!
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   } else {
     int diffPrec = numPrec - nPrec;

     idx2 = numDig >> 1;
     if (diffPrec > 0) {
       idx = nDec + nPrec - 1;
       buf[idx2] = ((positive | !notAllZero) ? 0x0F : 0x0D);
       idx2 -= (diffPrec + 1) >> 1;
       if (!(diffPrec & 0x1)) { // is it even?
         buf[idx2--] = tmpBuf[idx--] << 4;
       }
     } else {
       idx = nDec + numPrec - 1;
       buf[idx2--] = (tmpBuf[idx--] << 4) |
                     ((positive | !notAllZero) ? 0x0F : 0x0D);
     }

     for (; idx > 0; --idx2, idx -= 2)
       buf[idx2] = (tmpBuf[idx]) | (tmpBuf[idx - 1] << 4);
     if (idx == 0)
       buf[idx2] = tmpBuf[0];
   }

}

bool IDecimalUtil::cptrToDUT(const char *val, int &nitgrl,
                                       int &nfrtnl, char *bcdData)
{
   bool isPositive = true;

   // take care of the possible sign
   if (val[0] == '-') {
      isPositive = false;
      val++;
   } else if (val[0] == '+')
       val++;

   const char *decPosPtr = strchr(val, '.');
   char *cptr;
   int idx;
   int nDig = strlen(val) - (decPosPtr != NULL ? 1 : 0);
   if (decPosPtr != NULL)
      nitgrl = (int)(decPosPtr - val);
   else
      nitgrl = strlen(val);
   nfrtnl = nDig - nitgrl;
   if (nfrtnl) {
      for (cptr = bcdData + ((MAX_ITGRL + 1) >> 1), idx = 2;
           idx <= nfrtnl; idx += 2) {
          *cptr = (*(++decPosPtr) - '0') << 4;
          *(cptr++) |= *(++decPosPtr) - '0';
      }
      if (nfrtnl & 0x0001)
         *cptr = (*(++decPosPtr) - '0') << 4;
   }
   decPosPtr = val + nitgrl - 1;
   for (cptr = bcdData + ((MAX_ITGRL + 1) >> 1) - 1, idx = 2;
        idx <= nitgrl; idx += 2) {
      *cptr = *(decPosPtr--) - '0';
      *(cptr--) |= (*(decPosPtr--) - '0') << 4;
   }
   if (nitgrl & 0x0001)
      *cptr = *decPosPtr - '0';
   return isPositive;
}

void IDecimalUtil::intToDec(int anInt, int nDig, int nPrec, char *decBuf)
{
   char tmpBuf[DEC_DIG + 2];
   sprintf(tmpBuf,"%d", anInt);
   IDecimalUtil::cptrToDec(tmpBuf, nDig, nPrec, decBuf);
}
void IDecimalUtil::uIntToDec(unsigned int auInt, int nDig, int nPrec,
                             char *decBuf)
{
   char tmpBuf[DEC_DIG + 2];
   sprintf(tmpBuf,"%u", auInt);
   IDecimalUtil::cptrToDec(tmpBuf, nDig, nPrec, decBuf);
}
void IDecimalUtil::longToDec(long aLong, int nDig, int nPrec, char *decBuf)
{
   char tmpBuf[DEC_DIG + 2];
   sprintf(tmpBuf,"%ld", aLong);
   IDecimalUtil::cptrToDec(tmpBuf, nDig, nPrec, decBuf);
}
void IDecimalUtil::uLongToDec(unsigned long auLong, int nDig, int nPrec,
                              char *decBuf)
{
   char tmpBuf[DEC_DIG + 2];
   sprintf(tmpBuf,"%lu", auLong);
   IDecimalUtil::cptrToDec(tmpBuf, nDig, nPrec, decBuf);
}

void IDecimalUtil::doubleToDec(double aDouble, int nDig, int nPrec,
                               char *decBuf)
{
   if ((DEC_MAX.asDouble() < aDouble) ||
       (DEC_MIN.asDouble() > aDouble)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   char tmpBuf[DEC_DIG+DEC_DIG+3];
   sprintf(tmpBuf, "%0*.*f", nDig, nPrec, aDouble);
   IDecimalUtil::cptrToDec(tmpBuf, nDig, nPrec, decBuf);
}

void IDecimalUtil::lDoubleToDec(long double alDouble, int nDig,
                                int nPrec, char *decBuf)
{
   if ((DEC_MAX.asDouble() < alDouble) ||
       (DEC_MIN.asDouble() > alDouble)) {
     IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                           IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
     ITHROW(exc);
   }
   char tmpBuf[DEC_DIG+DEC_DIG+3];
   sprintf(tmpBuf, "%0*.*Lf", nDig, nPrec, alDouble);
   IDecimalUtil::cptrToDec(tmpBuf, nDig, nPrec, decBuf);
}

IString IDecimalUtil::asString() const
{
   char output[DEC_DIG + 10];
   const char *ccptr;
   char *cptr = output;
   int idx;
   if (!positive)
      *(cptr++) = '-';
   if (nitgrl | nfrtnl) {
      if (nitgrl) {
         ccptr = bcdData + ((MAX_ITGRL + 1) >> 1) - ((nitgrl + 1) >> 1);
         if (*ccptr & 0xF0)
            *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
         for (idx = nitgrl - 1; idx > 1; idx -= 2) {
            *(cptr++) = (*(ccptr++) & 0x0F) + '0';
            *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
         }
         *(cptr++) = (*(ccptr++) & 0x0F) + '0';
      } else {
         *(cptr++) = '0';
      }
      if (nfrtnl) {
         *(cptr++) = '.';
         ccptr = bcdData + ((MAX_ITGRL + 1) >> 1);
         for (idx = nfrtnl; idx > 2; idx -= 2) {
            *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
            *(cptr++) = (*(ccptr++) & 0x0F) + '0';
         }
         *(cptr++) = ((*(ccptr) & 0xF0) >> 4) + '0';
         if (*ccptr & 0x0F)
            *(cptr++) = (*(ccptr++) & 0x0F) + '0';
      }
   } else {
      *(cptr++) = '0';
   }
   *cptr = '\0';
   return IString((char *)output);
}

int IDecimalUtil::operator > (const IDecimalUtil &dut) const
{
   if (positive && !dut.positive)
      return 1;
   if (!positive && dut.positive)
      return 0;
   bool swapSign = !(positive || dut.positive);
   if (nitgrl > dut.nitgrl)
      return swapSign ? 0 : 1;
   if (nitgrl < dut.nitgrl)
      return swapSign ? 1 : 0;

   // same # of digits in integral part

   int itgrlBt = (nitgrl + 1) >> 1;
   int frtnlBt = ((nfrtnl > dut.nfrtnl ? nfrtnl : dut.nfrtnl) + 1) >> 1;
   int result = memcmp(bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlBt,
                       dut.bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlBt,
                       itgrlBt + frtnlBt);
   if (result > 0 )
      return swapSign ? 0 : 1;
   if (result < 0 )
      return swapSign ? 1 : 0;
   return 0;
}

int IDecimalUtil::operator < (const IDecimalUtil &dut) const
{
   if (positive && !dut.positive)
      return 0;
   if (!positive && dut.positive)
      return 1;
   bool swapSign = !(positive || dut.positive);
   if (nitgrl < dut.nitgrl)
      return swapSign ? 0 : 1;
   if (nitgrl > dut.nitgrl)
      return swapSign ? 1 : 0;

   // same # of digits in integral part

   int itgrlBt = (nitgrl + 1) >> 1;
   int frtnlBt = ((nfrtnl > dut.nfrtnl ? nfrtnl : dut.nfrtnl) + 1) >> 1;
   int result = memcmp(bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlBt,
                       dut.bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlBt,
                       itgrlBt + frtnlBt);
   if (result < 0 )
      return swapSign ? 0 : 1;
   if (result > 0 )
      return swapSign ? 1 : 0;
   return 0;
}
int IDecimalUtil::operator == (const IDecimalUtil &dut) const
{
   if (positive && !dut.positive)
      return 0;
   if (!positive && dut.positive)
      return 0;
   if (nitgrl != dut.nitgrl || nfrtnl != dut.nfrtnl)
      return 0;
   int itgrlBt = (nitgrl + 1) >> 1;
   int frtnlBt = (nfrtnl + 1) >> 1;
   int result = memcmp(bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlBt,
                       dut.bcdData + ((MAX_ITGRL + 1) >> 1) - itgrlBt,
                       itgrlBt + frtnlBt);

   return (result ? 0 : 1);
}

IDecimalUtil operator +(const IDecimalUtil &lv, const IDecimalUtil &rv)
{
   IDecimalUtil tmpDUT(0);
   bool finalSign;
   char *cptr;
   const char *lvptr, *rvptr;
   int lvGTrv;

   int maxitgrl = lv.nitgrl > rv.nitgrl ? lv.nitgrl : rv.nitgrl;
   int maxfrtnl = lv.nfrtnl > rv.nfrtnl ? lv.nfrtnl : rv.nfrtnl;
   int itgrlByte = (maxitgrl + 1) >> 1;
   int frtnlByte = (maxfrtnl + 1) >> 1;
   int numOfWordfrtnl = (maxfrtnl + 7) >> 3;
   int numOfWorditgrl = (maxitgrl + 7) >> 3;
   int numOfWord = numOfWordfrtnl + numOfWorditgrl;

   cptr = tmpDUT.bcdData + ((MAX_ITGRL + 1) >> 1) - (numOfWorditgrl << 2);
   lvptr = lv.bcdData + ((MAX_ITGRL + 1) >> 1) - (numOfWorditgrl << 2);
   rvptr = rv.bcdData + ((MAX_ITGRL + 1) >> 1) - (numOfWorditgrl << 2);

   if (lv.positive != rv.positive) {
      lvGTrv = memcmp(lvptr, rvptr, numOfWord << 2);
      if (lvGTrv == 0) {
         finalSign = true;
      } else {
         finalSign = lvGTrv > 0 ? lv.positive : rv.positive;
         if (lvGTrv > 0) {
            IDecimalUtil::subBCD(cptr, lvptr, rvptr, numOfWord);
         } else {
            IDecimalUtil::subBCD(cptr, rvptr, lvptr, numOfWord);
         }
      }
   } else {
      finalSign = lv.positive;
      if (IDecimalUtil::addBCD(cptr, lvptr, rvptr, numOfWord)) {
         if (cptr == tmpDUT.bcdData) {
            IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
            ITHROW(exc);
         } else {
            *(--cptr) = 0x01;
            maxitgrl++;
         }
      } else {
        int digitsInFirstWord = maxitgrl % 8;
        bool hasCarry = false;
        switch (digitsInFirstWord) {
        case 1:
           if (cptr[3] & 0xF0) {
             hasCarry = true;
           }
           break;
        case 2:
           if (cptr[2] & 0x0F) {
             hasCarry = true;
           }
           break;
        case 3:
           if (cptr[2] & 0xF0) {
             hasCarry = true;
           }
           break;
        case 4:
           if (cptr[1] & 0x0F) {
             hasCarry = true;
           }
           break;
        case 5:
           if (cptr[1] & 0xF0) {
             hasCarry = true;
           }
           break;
        case 6:
           if (cptr[0] & 0x0F) {
             hasCarry = true;
           }
           break;
        case 7:
           if (cptr[0] & 0xF0) {
             hasCarry = true;
           }
           break;
        default:
          break;
        }

        if (hasCarry) {
          maxitgrl++;
        }
      }
   }
   tmpDUT.nitgrl = maxitgrl;
   tmpDUT.nfrtnl = maxfrtnl;
   tmpDUT.positive = finalSign;
   tmpDUT.trimZero();
   return tmpDUT;
}

IDecimalUtil operator -(const IDecimalUtil &lv, const IDecimalUtil &rv)
{
   IDecimalUtil tmpDUT(0);
   bool finalSign;
   char *cptr;
   const char *lvptr, *rvptr;
   int lvGTrv;

   int maxitgrl = lv.nitgrl > rv.nitgrl ? lv.nitgrl : rv.nitgrl;
   int maxfrtnl = lv.nfrtnl > rv.nfrtnl ? lv.nfrtnl : rv.nfrtnl;
   int itgrlByte = (maxitgrl + 1) >> 1;
   int frtnlByte = (maxfrtnl + 1) >> 1;
   int numOfWordfrtnl = (maxfrtnl + 7) >> 3;
   int numOfWorditgrl = (maxitgrl + 7) >> 3;
   int numOfWord = numOfWordfrtnl + numOfWorditgrl;

   cptr = tmpDUT.bcdData + ((MAX_ITGRL + 1) >> 1) - (numOfWorditgrl << 2);
   lvptr = lv.bcdData + ((MAX_ITGRL + 1) >> 1) - (numOfWorditgrl << 2);
   rvptr = rv.bcdData + ((MAX_ITGRL + 1) >> 1) - (numOfWorditgrl << 2);

   if (lv.positive) {
      if (rv.positive) {
         lvGTrv = memcmp(lvptr, rvptr, numOfWord << 2);
         if (lvGTrv == 0) {
            finalSign = true;
         } else {
            finalSign = lvGTrv > 0 ? lv.positive : !lv.positive;
            if (lvGTrv > 0)
               IDecimalUtil::subBCD(cptr, lvptr, rvptr, numOfWord);
            else
               IDecimalUtil::subBCD(cptr, rvptr, lvptr, numOfWord);
         }
      } else {
         finalSign = true;
         if (IDecimalUtil::addBCD(cptr, lvptr, rvptr, numOfWord)) {
            if (cptr == tmpDUT.bcdData) {
            IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
               ITHROW(exc);
            } else {
               *(--cptr) = 0x01;
               maxitgrl++;
            }
         } else {
             int digitsInFirstWord = maxitgrl % 8;
             bool hasCarry = false;
             switch (digitsInFirstWord) {
                 case 1:
                     if (cptr[3] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 case 2:
                     if (cptr[2] & 0x0F) {
                         hasCarry = true;
                     }
                     break;
                 case 3:
                     if (cptr[2] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 case 4:
                     if (cptr[1] & 0x0F) {
                         hasCarry = true;
                     }
                     break;
                 case 5:
                     if (cptr[1] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 case 6:
                     if (cptr[0] & 0x0F) {
                         hasCarry = true;
                     }
                     break;
                 case 7:
                     if (cptr[0] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 default:
                     break;
             }
             
             if (hasCarry) {
                 maxitgrl++;
             }
         }
         
      }
   } else {
      if (rv.positive) {
         finalSign = lv.positive;
         if (IDecimalUtil::addBCD(cptr, lvptr, rvptr, numOfWord)) {
            if (cptr == tmpDUT.bcdData) {
            IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                          IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
               ITHROW(exc);
            } else {
               *(--cptr) = 0x01;
               maxitgrl++;
            }
         } else {
             int digitsInFirstWord = maxitgrl % 8;
             bool hasCarry = false;
             switch (digitsInFirstWord) {
                 case 1:
                     if (cptr[3] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 case 2:
                     if (cptr[2] & 0x0F) {
                         hasCarry = true;
                     }
                     break;
                 case 3:
                     if (cptr[2] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 case 4:
                     if (cptr[1] & 0x0F) {
                         hasCarry = true;
                     }
                     break;
                 case 5:
                     if (cptr[1] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 case 6:
                     if (cptr[0] & 0x0F) {
                         hasCarry = true;
                     }
                     break;
                 case 7:
                     if (cptr[0] & 0xF0) {
                         hasCarry = true;
                     }
                     break;
                 default:
                     break;
             }
             
             if (hasCarry) {
                 maxitgrl++;
             }
         }
         
      } else {
         lvGTrv = memcmp(lvptr, rvptr, numOfWord << 2);
         if (lvGTrv == 0) {
            finalSign = true;
         } else {
            finalSign = lvGTrv > 0 ? lv.positive : !lv.positive;
            if (lvGTrv > 0)
               IDecimalUtil::subBCD(cptr, lvptr, rvptr, numOfWord);
            else
               IDecimalUtil::subBCD(cptr, rvptr, lvptr, numOfWord);
         }
      }
   }

   tmpDUT.nitgrl = maxitgrl;
   tmpDUT.nfrtnl = maxfrtnl;

   tmpDUT.positive = finalSign;
   tmpDUT.trimZero();
   return tmpDUT;
}

IDecimalUtil operator *(const IDecimalUtil &lv, const IDecimalUtil &rv)
{
   // multiply by zero
   if (!(lv.nitgrl | lv.nfrtnl | rv.nitgrl | rv.nfrtnl)) {
      return IDecimalUtil(0);
   }

   IDecimalUtil tmpDUT(0);
   char tmpBuf[sizeof(lv.bcdData) << 1];
   int finalSign = (lv.positive == rv.positive);
   char *cptr;
   const char *lvptr, *rvptr;
   int carryout = 0;
   int numOfWord;
   int dropFrtnlByte = 0;


   int lvByte = ((lv.nitgrl + 1) >> 1) + ((lv.nfrtnl + 1) >> 1);
   int rvByte = ((rv.nitgrl + 1) >> 1) + ((rv.nfrtnl + 1) >> 1);
   int div100Pow = ((lv.nfrtnl + 1) >> 1) + ((rv.nfrtnl + 1) >> 1);

   memset(tmpBuf, '\0', sizeof(tmpBuf));

   lvptr = lv.bcdData + ((MAX_ITGRL + 1) >> 1) - ((lv.nitgrl + 1) >> 1);
   rvptr = rv.bcdData + ((MAX_ITGRL + 1) >> 1) - ((rv.nitgrl + 1) >> 1);

   if (lvByte > rvByte)
      carryout = IDecimalUtil::mulBCD(tmpBuf, sizeof(tmpBuf),
                                  lvptr, lvByte,
                                  rvptr, rvByte);
   else
      carryout = IDecimalUtil::mulBCD(tmpBuf, sizeof(tmpBuf),
                                  rvptr, rvByte,
                                  lvptr, lvByte);
   // adjust the result to the correct decimal position

   if (carryout) {
      IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                         IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
      ITHROW(exc);
   }
   if (div100Pow > ((MAX_FRTNL + 1) >> 1))
      dropFrtnlByte = div100Pow - ((MAX_FRTNL + 1) >> 1);
   memcpy(tmpDUT.bcdData + ((MAX_ITGRL + 1) >> 1) - lvByte - rvByte +
          div100Pow, tmpBuf + sizeof(tmpBuf) - lvByte - rvByte,
          lvByte + rvByte - dropFrtnlByte);
   tmpDUT.nitgrl = (lvByte + rvByte - div100Pow) << 1;
   tmpDUT.nfrtnl = (div100Pow - dropFrtnlByte) << 1;
   tmpDUT.positive = finalSign;

   tmpDUT.trimZero();
   return tmpDUT;
}

IDecimalUtil operator /(const IDecimalUtil &lv, const IDecimalUtil &rv)
{
   // left value is zero
   if (!(lv.nitgrl | lv.nfrtnl)) {
      return IDecimalUtil(0);
   }
   // divide by zero
   if (!(rv.nitgrl | rv.nfrtnl)) {
      IDecimalDataError exc(IMessageText(IC_DECIMAL_DATA_OVERFLOW),
                         IC_DECIMAL_DATA_OVERFLOW, IException::unrecoverable);
      ITHROW(exc);
   }
   IDecimalUtil tmpDUT(0);
   int finalSign = (lv.positive == rv.positive);
   char *cptr;
   const char *lvptr, *rvptr;
   char *tmpBuf;
   int tmpBufSize;
   int carryout;

   int lvByte = ((lv.nitgrl + 1) >> 1) + ((lv.nfrtnl + 1) >> 1);
   int rvByte = ((rv.nitgrl + 1) >> 1) + ((rv.nfrtnl + 1) >> 1);
   int powOf10 = ((((lv.nitgrl + 1) >> 1) - ((rv.nitgrl + 1) >> 1)) << 1) + 1;

   lvptr = lv.bcdData + ((MAX_ITGRL + 1) >> 1) - ((lv.nitgrl + 1) >> 1);
   rvptr = rv.bcdData + ((MAX_ITGRL + 1) >> 1) - ((rv.nitgrl + 1) >> 1);

#ifdef __IOC_IDECIMAL_DEBUG
   {
       int i;
       printf("lvptr:  ");
       for (i = 0; i < lvByte; i++)
           printf("%02x ", lvptr[i]);
       printf("\n");
   }
#endif

   if (powOf10 > 0)
      tmpBufSize = ((powOf10 + 1) >> 1) + ((MAX_FRTNL + 1) >> 1);
   else
      tmpBufSize = (MAX_FRTNL + 1) >> 1;

   // adjust to word boundary
   tmpBufSize = ((tmpBufSize + 3) >> 2) << 2;

   tmpBuf = new char[tmpBufSize];
   memset(tmpBuf, '\0', tmpBufSize);

   int mulBy10 = IDecimalUtil::divBCD(tmpBuf, powOf10, lvptr,
                                      lvByte, rvptr, rvByte);

#ifdef __IOC_IDECIMAL_DEBUG
   {
       int i;
       printf("tmpBuf: ");
       for (i = 0; i < tmpBufSize; i++)
           printf("%02x ", tmpBuf[i]);
       printf("\n");
   }
#endif            

   // adjust the result to byte boundary
   carryout = 0;
   if (powOf10 & 0x01) {
      carryout = IDecimalUtil::times10(tmpBuf, tmpBufSize >> 2);
   }

#ifdef __IOC_IDECIMAL_DEBUG
   {
       int i;
       printf("carryout: %d\n", carryout);
       printf("tmpBuf: ");
       for (i = 0; i < tmpBufSize; i++)
           printf("%02x ", tmpBuf[i]);
       printf("\n");
   }
#endif 

   unsigned long offset = ((MAX_ITGRL + 1) >> 1) - ((powOf10 - 1) >> 1);
   char *tmp = tmpDUT.bcdData + offset;
   memcpy(tmp, tmpBuf, ((powOf10 - 1) >> 1) + ((MAX_FRTNL + 1) >> 1));
   *(tmp - 1) = carryout;

/*
  Defect 29142
  memcpy(tmpDUT.bcdData + ((MAX_ITGRL + 1) >> 1) - ((powOf10 - 1) >> 1),
  tmpBuf, ((powOf10 - 1) >> 1) + ((MAX_FRTNL + 1) >> 1));
  *(tmpDUT.bcdData - 1 + ((MAX_ITGRL + 1) >> 1) - ((powOf10 - 1) >> 1)) +=
                                                                  carryout;
*/

#ifdef __IOC_IDECIMAL_DEBUG
   {
       printf("offset: %d\n", ((MAX_ITGRL + 1) >> 1) - ((powOf10 - 1) >> 1));
   }
#endif

   if (powOf10 > 0) {
      tmpDUT.nitgrl = powOf10;
   } else {
      tmpDUT.nitgrl = 0;
   }
   tmpDUT.nfrtnl = MAX_FRTNL;
   delete [] tmpBuf;
   tmpDUT.positive = finalSign;
   tmpDUT.trimZero();
   if(mulBy10) return tmpDUT * IDecimalUtil(10);
   else return tmpDUT;
}

int IDecimalUtil::addBCD(char *resBCD, const char *lvptr,
                                    const char *rvptr, int numOfWord)
{
   unsigned long ln1, ln2, result;
   int carryin = 0;
   int carryout = 0;

   for (numOfWord--; numOfWord >= 0; numOfWord--) {
      carryout = 0;
      ln1 = IDecimalUtil::genuLong(lvptr + (numOfWord << 2));
      ln2 = IDecimalUtil::genuLong(rvptr + (numOfWord << 2));
      if (ln1 && ln2) {
         result = IDecimalUtil::decAddWord(ln1, ln2,
                                           MASK_OF_6, MASK_OF_8);
         if (result < ln1 || result < ln2)
            carryout = 1;
      } else {
         result = ln1 ? ln1 : ln2;
         carryout = 0;
      }
      if (carryin)
         result = IDecimalUtil::decAddWord(result, 1UL,
                                           MASK_OF_6, MASK_OF_8);
      if ((ln1 || ln2) && !result)
         carryout = 1;
      IDecimalUtil::genBCD(resBCD + (numOfWord << 2), result);
      carryin = carryout;
   }
   return carryin;
}

// assumption is DUT1 is bigger than DUT2 so result is alway positive
void IDecimalUtil::subBCD(char *resBCD, const char *lvptr,
                          const char *rvptr, int numOfWord)
{
   unsigned long ln1, ln2, result;
   int borrow = 0;

   for (numOfWord--; numOfWord >= 0; numOfWord--) {
      ln1 = IDecimalUtil::genuLong(lvptr + (numOfWord << 2));
      ln2 = IDecimalUtil::genuLong(rvptr + (numOfWord << 2));
      if (borrow) {
         result = IDecimalUtil::decAddWord(ln1, NEGATIVE_1, MASK_OF_6,
                                           MASK_OF_8);
         if (result > ln1)
            borrow = 1; // need to borrow from next group
         else
            borrow = 0;
         ln1 = result;
      }
      if (ln2) {
         ln2 = IDecimalUtil::gen10Complement(ln2);
         result = IDecimalUtil::decAddWord(ln1, ln2,
                                           MASK_OF_6, MASK_OF_8);
         if (result > ln1)
            borrow = 1;
      } else {
         result = ln1;
      }
      IDecimalUtil::genBCD(resBCD + (numOfWord << 2), result);
   }
}

// left operand always have more significant digits
int IDecimalUtil::mulBCD(char *resBCD, int resBCDSize,
                                    const char *lvptr, int lvByte,
                                    const char *rvptr, int rvByte)
{
   char *lvBuf, *lvBufx2, *lvBufx4;
   char *tmpBuf;
   char *cptr;
   char *cyptr;
   char *lsptr;
   char *cyArray;
   int numOfWordlv = (lvByte + 3) >> 2;
   int numOfWordlvx2;
   int numOfWordlvx4;
   int numOfWordrv = (rvByte + 3) >> 2;
   int carryout;
   int idx1, idx2;

   lvBuf = new char[numOfWordlv << 2];
   tmpBuf = new char[numOfWordlv << 2];

   memset(lvBuf, '\0', numOfWordlv << 2);

   // keep a copy of 1x.
   memcpy(lvBuf + (numOfWordlv << 2) - lvByte, lvptr, lvByte);

   // generate 2x times of lvalue
   carryout = IDecimalUtil::addBCD(tmpBuf, lvBuf, lvBuf, numOfWordlv);
   numOfWordlvx2 = numOfWordlv;
   if (carryout) {
      numOfWordlvx2++;
      lvBufx2 = new char[numOfWordlvx2 << 2];
      memset(lvBufx2, '\0', 4);
      memcpy(lvBufx2 + 4, tmpBuf, numOfWordlv << 2);
      *(lvBufx2 + 3) = 0x01;
      delete [] tmpBuf;
      tmpBuf = new char[numOfWordlvx2 << 2];
   } else {
      lvBufx2 = new char[numOfWordlvx2 << 2];
      memcpy(lvBufx2, tmpBuf, numOfWordlv << 2);
   }

   // generate 4 times of lvalue
   carryout = IDecimalUtil::addBCD(tmpBuf, lvBufx2, lvBufx2, numOfWordlvx2);
   numOfWordlvx4 = numOfWordlvx2;
   if (carryout) {
      numOfWordlvx4++;
      lvBufx4 = new char[numOfWordlvx4 << 2];
      memset(lvBufx4, '\0', 4);
      memcpy(lvBufx4 + 4, tmpBuf, numOfWordlvx2 << 2);
      *(lvBufx4 + 3) = 0x01;
      delete [] tmpBuf;
      tmpBuf = new char[numOfWordlvx4 << 2];
   } else {
      lvBufx4 = new char[numOfWordlvx4 << 2];
      memcpy(lvBufx4, tmpBuf, numOfWordlvx2 << 2);
   }

   // have everything the same length
   if (numOfWordlv != numOfWordlvx4) {
      memset(tmpBuf, '\0', numOfWordlvx4 << 2);
      memcpy(tmpBuf + ((numOfWordlvx4 - numOfWordlv) << 2), lvBuf,
             numOfWordlv << 2);
      delete [] lvBuf;
      lvBuf = new char[numOfWordlvx4 << 2];
      memcpy(lvBuf, tmpBuf, numOfWordlvx4 << 2);
   }

   if (numOfWordlvx2 != numOfWordlvx4) {
      memset(tmpBuf, '\0', numOfWordlvx4 << 2);
      memcpy(tmpBuf + ((numOfWordlvx4 - numOfWordlvx2) << 2), lvBufx2,
             numOfWordlvx2 << 2);
      delete [] lvBufx2;
      lvBufx2 = new char[numOfWordlvx4 << 2];
      memcpy(lvBufx2, tmpBuf, numOfWordlvx4 << 2);
   }

   // everything is ready
   delete [] tmpBuf;

#ifdef __IOC_IDECIMAL_DEBUG
   {
       int i;
       printf("1x: ");
       for (i = 0; i < numOfWordlvx4 << 2; i++)
           printf("%02x ", lvBuf[i]);
       printf("\n");
       printf("2x: ");
       for (i = 0; i < numOfWordlvx4 << 2; i++)
           printf("%02x ", lvBufx2[i]);
       printf("\n");
       printf("4x: ");
       for (i = 0; i < numOfWordlvx4 << 2; i++)
           printf("%02x ", lvBufx4[i]);
       printf("\n");
   }
#endif

   cptr = resBCD + resBCDSize - (numOfWordlvx4 << 2);

   // array to store carryouts latter to be added back to the result
   cyArray = new char[numOfWordrv << 2];
   memset(cyArray, '\0', numOfWordrv << 2);
   cyptr = cyArray + (numOfWordrv << 2) - rvByte;

   // shifted out variables are placed directly into resulting Buffer
   lsptr = cptr - rvByte;

#ifdef __IOC_IDECIMAL_DEBUG
   {
       printf("resBCD:  0x%0X resBCDSize: %d cptr: 0x%0X lsptr: 0x%0X\n",
              resBCD, resBCDSize, cptr, lsptr);
       printf("cyArray: 0x%0X cyArraySize: %d cyptr: 0x%0X\n",
              cyArray, numOfWordrv << 2, cyptr);
   }
#endif

   for (carryout = 0, idx1 = 0; idx1 < rvByte; idx1++) {
      idx2 = *(rvptr + idx1);
      // multiply by 10  = left shift 1 nibble, if something got shifted
      // out, place in the final result
      // reason of putting it here is to handle the last digits that
      // does not get shifted
      *lsptr = (IDecimalUtil::times10(cptr, numOfWordlvx4)) << 4;
      carryout = 0;
      switch (idx2 >> 4) {
         case 9: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 5: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 1: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBuf,
                                                  numOfWordlvx4);
                 break;
         case 7: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 3: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx2,
                                                  numOfWordlvx4);
                 carryout += IDecimalUtil::addBCD(cptr, cptr, lvBuf,
                                                  numOfWordlvx4);
                 break;
         case 6: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 2: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx2,
                                                  numOfWordlvx4);
                 break;
         case 8: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 4: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         default: break;
      }

      *cyptr |= carryout << 4;

#ifdef __IOC_IDECIMAL_DEBUG
      {
          printf("idx2: %d carryout: %d ", idx2 >> 4, carryout);
          printf("cyArray: ");
          int i;
          for (i = 0; i < numOfWordrv << 2; i++)
              printf("%02x ", cyArray[i]);
          printf("\n");
      }
#endif

      // times 10 only if we have more to come
      *lsptr |= IDecimalUtil::times10(cptr, numOfWordlvx4);
      carryout = 0;

      switch (idx2 & 0x0F ) {
         case 9: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 5: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 1: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBuf,
                                                  numOfWordlvx4);
                 break;
         case 7: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 3: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx2,
                                                  numOfWordlvx4);
                 carryout += IDecimalUtil::addBCD(cptr, cptr, lvBuf,
                                                  numOfWordlvx4);
                 break;
         case 6: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 2: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx2,
                                                  numOfWordlvx4);
                 break;
         case 8: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         case 4: carryout += IDecimalUtil::addBCD(cptr, cptr, lvBufx4,
                                                  numOfWordlvx4);
         default: break;
      }

      lsptr++;
      *cyptr |= carryout;
      cyptr++;

#ifdef __IOC_IDECIMAL_DEBUG
      {
          printf("idx2: %d carryout: %d ", idx2 & 0x0F, carryout);
          printf("cyArray: ");
          int i;
          for (i = 0; i < numOfWordrv << 2; i++)
              printf("%02x ", cyArray[i]);
          printf("\n");
      }
#endif

   }

#ifdef __IOC_IDECIMAL_DEBUG
   {
       int i;
       char *tmp = cptr - (numOfWordrv << 2);
       printf("Adding carryout to :");
       for (i = 0; i < numOfWordrv << 2; i++)
           printf("%02x ", tmp[i]);
       printf("\n");
   }
#endif

   carryout = IDecimalUtil::addBCD(cptr - (numOfWordrv << 2),
                                   cptr - (numOfWordrv << 2),
                                   cyArray, numOfWordrv);

   if (carryout && (cptr - (numOfWordrv << 2)) != resBCD) {
      *(cptr - (numOfWordrv << 2) - 1) = carryout;
      carryout = 0;
   }

   // clean up
   delete [] cyArray;
   delete [] lvBuf;
   delete [] lvBufx2;
   delete [] lvBufx4;

   return carryout;
}

int IDecimalUtil::divBCD(char *resBCD, int powOf10, //wgo
                      const char *lvptr, int lvByte,
                      const char *rvptr, int rvByte)
{
#ifdef __IOC_IDECIMAL_DEBUG
    {
        printf("divBCD()\n");
        printf("powOf10: %d\n", powOf10);
        int i;
        printf("lvptr:  ");
        for (i = 0; i < lvByte; i++)
            printf("%02x ", lvptr[i]);
        printf("\n");
        printf("rvptr:  ");
        for (i = 0; i < rvByte; i++)
           printf("%02x ", rvptr[i]);
        printf("\n");
    }
#endif

    int mulBy10 = 0;
    int idx;
    int result;
    int numOfWordlv = (lvByte + 3) >> 2;
    int numOfWordrv = (rvByte + 3) >> 2;
    int maxWord = numOfWordlv > numOfWordrv ? numOfWordlv : numOfWordrv;

    char *lvBuf = new char [(maxWord + 1) << 2];
    char *rvBuf = new char [(maxWord + 1) << 2];
    char *negrvBuf = new char [(maxWord + 1) << 2];
    char *emptyBuf = new char [maxWord << 2];
    char *oneBuf = new char [(maxWord + 1) << 2];

    memset(lvBuf, '\0', (maxWord + 1) << 2);
    memset(rvBuf, '\0', (maxWord + 1) << 2);
    memset(negrvBuf, '\0', (maxWord + 1) << 2);
    memset(emptyBuf, '\0', maxWord << 2);
    memset(oneBuf, '\0', (maxWord + 1) << 2);

    // set up the buffers for division
    memcpy(lvBuf + 4, lvptr, lvByte);
    memcpy(rvBuf + 4, rvptr, rvByte);
    if(((*lvptr)>0xF)&&((*rvptr)<=0xF)){
       IDecimalUtil::times10(rvBuf+4,maxWord);
       mulBy10 = 1;
    }

    // create a 10's complement representation of right value
    // for future subtraction
    memset(negrvBuf, '\x99', 4);
    for (idx = 0; idx < rvByte; idx++)
      *(negrvBuf + 4 + idx) = ~(*(rvBuf + 4 + idx)) - 0x66;
    for (;idx < (maxWord << 2); idx++)
      *(negrvBuf + 4 + idx) = 0x99;

    // defect 29142
    oneBuf[(maxWord << 2) - 1] = 0x01;
    addBCD(negrvBuf + 4, negrvBuf + 4, oneBuf, maxWord);
/*
    defect 29142
    *(lvBuf + (numOfWordrv << 2) - 1) = 0x01;
    addBCD(negrvBuf + 4, negrvBuf + 4, lvBuf, numOfWordrv);
      *(lvBuf + (numOfWordrv << 2) - 1) = '\0';
*/

#ifdef __IOC_IDECIMAL_DEBUG
    {
        int i;
        printf("lvBuf:    ");
        for (i = 0; i < (maxWord + 1) << 2; i++)
            printf("%02x ", lvBuf[i]);
        printf("\n");
        printf("rvBuf:    ");
        for (i = 0; i < (maxWord + 1) << 2; i++)
            printf("%02x ", rvBuf[i]);
        printf("\n");
        printf("negrvBuf: ");
        for (i = 0; i < (maxWord + 1) << 2; i++)
            printf("%02x ", negrvBuf[i]);
        printf("\n");
    }
#endif

    // find until MAX_FRTNL or fully divisible
    idx = 0;
    while (powOf10 > -MAX_FRTNL) {
       result = 0;
       while (memcmp(lvBuf, rvBuf, (1+maxWord)*4) >= 0) {
           addBCD(lvBuf, lvBuf, negrvBuf, maxWord + 1);
           result++;
#ifdef __IOC_IDECIMAL_DEBUG
            {
                int i;
                printf("result: %d\n", result);
                printf("lvbBuf: ");
                for (i = 0; i < (maxWord + 1) << 2; i++)
                    printf("%02x ", lvBuf[i]);
                printf("\n");
            }
#endif
       }
       *(resBCD + (idx >> 1)) |= (idx & 0x01) ? result : result << 4;
#ifdef __IOC_IDECIMAL_DEBUG
       {
           int i;
           printf("resBCD: ");
           for (i = 0; i < (maxWord + 1) << 2; i++)
               printf("%02x ", resBCD[i]);
           printf("\n");
       }
#endif
       if (memcmp(lvBuf + 4, emptyBuf, maxWord << 2) == 0) break;
       *(lvBuf + 3) = IDecimalUtil::times10(lvBuf + 4, maxWord);
#ifdef __IOC_IDECIMAL_DEBUG
       {
           int i;
           printf("lvbBuf: ");
           for (i = 0; i < (maxWord + 1) << 2; i++)
               printf("%02x ", lvBuf[i]);
           printf("\n");
       }
#endif
       idx++;
       powOf10--;
    }
    delete [] lvBuf;
    delete [] rvBuf;
    delete [] negrvBuf;
    delete [] emptyBuf;
    delete [] oneBuf;

#ifdef __IOC_IDECIMAL_DEBUG
    {
        int i;
        printf("resBCD: ");
        for (i = 0; i < (maxWord + 1) << 2; i++)
            printf("%02x ", resBCD[i]);
        printf("\n");
        printf("mulBy10: %d\n", mulBy10);
    }
#endif
    return mulBy10;
}

unsigned long IDecimalUtil::decAddWord(unsigned long in1, unsigned long in2,
                                   unsigned long mask6, unsigned long mask8)
{
    unsigned long s1, s2;
    unsigned long t1, t2, t5, t6, t7;
//
// mask6 is a word of bcd 6's (0x66666666)
// mask8 is a word of hex 8's (0x88888888)
// (These could be developed locally, but practically
// it is more efficient to have the caller provide them)
//
      s1 = in1 + mask6;       // 6-adjust
      s2 = s1 + in2;          // add 2nd operand
//
// now develop the "carry out of this nibble"
// (This is what the "dsixes" instruction does)
//
      t1 = (s1 & in2);         // "8" bit is carry generated
        // Propagte, and sum bit differs - must have carried
      t2 = (s1 ^ in2) & (~s2);
      t5 =  ~( t1 | t2);       // (actually no carry out)
      t6 = t5 & mask8;         // keep one bit: results is 8 or 0
      t7 = t6 - (t6>>2);       // develop word of 6's or 0's
//
// final correction to original sum
//
      return (s2 - t7);
}
int IDecimalUtil::times10(char *cptr, int numOfWord)
{
   int shiftout = *cptr >> 4;
   int idx;

   char tmpChar;
   for (idx = 0; idx < (numOfWord << 2) - 1; idx++) {
      tmpChar = *cptr << 4;
      tmpChar |= *(cptr + 1) >> 4;
      *(cptr++) = tmpChar;
   }
   *cptr <<= 4;
   return shiftout;
}
