/*----------------------------------------------------------------------------*/
/* FILE NAME: idbcsb6.cpp                                                     */
/*                                                                            */
/* DESCRIPTION:                                                               */
/*   This file contains the implementation of the IDBCSBuffer searching       */
/*   functions.                                                               */
/*                                                                            */
/*                                                                            */
/* COPYRIGHT:                                                                 */
/*   IBM Open Class Library                                                   */
/*   Licensed Materials - Property of IBM                                     */
/*                                                                            */
#ifdef __MVS__
/*   5645-001                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
#ifdef __OS400__
/*   ????-???                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
#ifdef __OS2__
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
#ifdef __WINDOWS__
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
#ifdef _AIX
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
#ifdef _sparc
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
#ifdef __sparc
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
#endif
/*                                                                            */
/*   US Government Users Restricted Rights - Use, duplication, or             */
/*   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.        */
/*                                                                            */
/*----------------------------------------------------------------------------*/
#ifdef __MVS__
    #pragma csect(code,"IDBCSB6")
    #pragma csect(static,"SDBCSB6")
    #pragma comment (copyright, \
    "Licensed Materials - Property of IBM  5645-001 (C) \
    Copyright IBM Corp. 1992, 1997  All Rights Reserved. \
    US Government Users Restricted Rights - Use, duplication, or \
    disclosure restricted by GSA ADP Schedule Contract with IBM Corp.")
#endif
#pragma priority( -2147482024 )

#include <istring.hpp>

#include <idbcsbuf.hpp>

#include <istrtest.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IDBCSB6_CPP_
  #include <ipagetun.h>
#endif

#include "inlsutil.hpp"

#ifdef IC_PAGETUNE
    #define _IDBCSB6_CPP_
    #include <ipagetun.h>
#endif

#ifdef IC_MVS400
  static const char SO = '\x0e';         // shift-out character
  static const char SI = '\x0f';         // shift-in  character
#endif

/*------------------------------------------------------------------------------
| IDBCSBuffer::startSearch                                                     |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: startSearch ( unsigned startPos,
                                      unsigned searchLen ) const
  {
  unsigned
    result = this->IBuffer::startSearch( startPos, searchLen );
  while ( this->charType( result ) > IStringEnum::mbcs1 )
    if ( ++result > this->length() )
    {
      result = 0;
      break;
    }
  return result;
  }

/*------------------------------------------------------------------------------
| IDBCSBuffer::startBackwardsSearch                                            |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: startBackwardsSearch ( unsigned startPos,
                                               unsigned searchLen ) const
  {
  unsigned
    result = this->IBuffer::startBackwardsSearch( startPos, searchLen );
  while ( result > 0 && this->charType( result ) > IStringEnum::mbcs1 )
    result--;
  return result;
  }

/*------------------------------------------------------------------------------
| IDBCSBuffer::indexOf                                                         |
|                                                                              |
| 1. First, check whether the search can succeed (and set the point at which   |
|    the search is to start) by calling startSearch().                         |
| 2. If the search cannot succeed, then return 0.                              |
| 3. Otherwise, compare the buffer contents starting at each index to the      |
|    search string until a match is found (in which case we return the index)  |
|    or the entire buffer is tested (in which case 0 is returned).             |
|                                                                              |
|                                                                              |
| Notes:                                                                       |
|   1. If the search string is a single character, then this function is       |
|      implemented via the equivalent invocation of indexOfAnyOf (which is     |
|      optimized for that case).  This saves the addtional logic of special    |
|      code to handle that case here, as well.                                 |
|   2. If the search string is null, then 0 is returned.                       |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: indexOf ( const char *pSearchString,
                                  unsigned    searchLen,
                                  unsigned    startPos ) const
{
#if defined(IC_NLS) && defined(IC_MVS400)
  if (IString::isInternationalized()) {
    if (searchLen == 1)
      return indexOfAnyOf(pSearchString, 1, startPos);

    if (searchLen == 0) return 0;

    if (startPos == 0) startPos = 1;

    unsigned numPos = findNextCharNum(contents(), length(), startPos);

    if (numPos) {
      IWBuffer wBuf(contents(), length());
      IWBuffer sBuf(pSearchString, searchLen);

      return findBytePos(contents(), length(),
                         wBuf.positionOf(sBuf, numPos));
    } else {
      return 0;
    }
  } else {
#endif
    if ( searchLen == 1 )
      return this->indexOfAnyOf( pSearchString, searchLen, startPos );

    startPos = this->startSearch( startPos, searchLen );
    if ( startPos && searchLen )
      while ( (startPos + searchLen - 1) <= this->length() )
        if ( memcmp( this->contents() + startPos - 1,
                     pSearchString,
                     searchLen ) == 0 )
          return startPos;
        else
          startPos += charLength ( startPos );
    return 0;
#if defined(IC_NLS) && defined(IC_MVS400)
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::indexOf                                                         |
|                                                                              |
| Simply invokes the functionally equivalent indexOf function.                 |
|                                                                              |
| Notes:                                                                       |
|   1. If the search string is null, then 0 is returned.                       |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: indexOf ( const IStringTest &aTest,
                                  unsigned           startPos ) const
{
#ifdef IC_NLS
  if (IString::isInternationalized()) {
    return indexOfAnyOf ( aTest, startPos );
  } else
#endif
    return this->IBuffer::indexOf( aTest, startPos );
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::indexOfAnyBut                                                   |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the starting       |
|    point) by calling startSearch().                                          |
| 2. If it can't succeed, return 0.                                            |
| 3. If it can succeed, the starting point is converted to an array (0-based)  |
|    index.                                                                    |
| 4. Search for the first character not in the set of valid characters, based  |
|    on the number of valid characters:                                        |
|      0  - return 1; i.e., the first character of the buffer                  |
|           isn't in the (empty) set of valid characters                       |
|      1  - compare each character of the buffer to the valid                  |
|           character until one doesn't match                                  |
|      >1 - call memchr for each character of the buffer to see if the         |
|           character is in the set of valid characters;                       |
|           return the index of the first that isn't found                     |
|                                                                              |
| If all characters pass the test, return 0.                                   |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: indexOfAnyBut ( const char *pValidChars,
                                        unsigned    numValidChars,
                                        unsigned    startPos ) const
{
#if defined(IC_NLS) && defined(IC_MVS400)
  if (IString::isInternationalized()) {
    if (startPos == 0) startPos = 1;

    if (numValidChars == 0) {
      // return index that points to the beginning of the next
      // valid character (i.e. if 'startPos' points to redundant SO/SI,
      // they are skipped).
      unsigned curPos = findCurPos(contents(), length(), startPos);
      if (startPos > curPos) {
        return findNextPos(contents(), length(), startPos);
      } else {
        return curPos;
      }
    }

    if ((numValidChars == 1) &&
        ((*pValidChars == SO) ||
         (*pValidChars == SI))) {
      for (const char *p = contents() + startPos - 1;
           startPos <= length();
           ++p, ++startPos) {
        if (*p != *pValidChars)
          return startPos;
      }

      return 0;
    } else {
        unsigned numPos = findNextCharNum(contents(), length(), startPos);

      if (numPos) {
        IWBuffer wBuf(contents(), length());
        IWBuffer sBuf(pValidChars, numValidChars);

        return findBytePos(contents(), length(),
                           wBuf.positionOfAnyBut(sBuf, numPos));
      } else {
        return 0;
      }

    }

  } else {
#endif
    startPos = this->startSearch( startPos, 1 );
    if ( startPos )
      {
      switch ( numValidChars )
        {
        case 0:
          return startPos;
        case 1:
          // No character can match single DBCS first-byte:
          if ( ! isSBC ( pValidChars ) )
            return startPos;
          // Examine each character of this string...
          while ( startPos <= this->length() )
            // Compare this character.  Note that if current character
            // is DBCS first-byte, this compare will fail.
            if ( this->contents()[ startPos - 1 ] != *pValidChars )
              return startPos;
            else
              startPos++;
          break;
        default:
          while ( startPos <= this->length() )
            {
            if ( !this->isCharValid( startPos, pValidChars, numValidChars ) )
              return startPos;
            else
              startPos += charLength ( startPos );
            }
        }
      }
    return 0;
#if defined(IC_NLS) && defined(IC_MVS400)
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::indexOfAnyBut                                                   |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the starting       |
|    point) by calling startSearch().                                          |
| 2. Invokes the IStringTest test method against each character in turn, until |
|    the argument IStringTest returns a false result.                          |
|                                                                              |
| If all characters pass the test, return 0.                                   |
------------------------------------------------------------------------------*/

unsigned IDBCSBuffer :: indexOfAnyBut ( const IStringTest &aTest,
                                        unsigned           startPos ) const
{
#ifdef IC_NLS
  if (IString::isInternationalized()) {
    if (startPos == 0) startPos = 1;

    unsigned numPos = findNextCharNum(contents(), length(), startPos);

    if (numPos) {
      IWBuffer wBuf(contents(), length());

      while (numPos <= wBuf.charNum()) {
        if (!aTest.test(wBuf[numPos]))
          return findBytePos(contents(), length(), numPos);
        ++numPos;
      }
      return 0;
    } else {
      return 0;
    }
  } else {
#endif
    startPos = this->startSearch( startPos, 1 );
    if ( startPos )
    {
      while ( startPos <= this->length() )
        {
        // Note:  This is broken!  Must somehow pass both bytes of DBCS!
        if ( !aTest.test( this->contents()[ startPos - 1 ] ) )
          return startPos;
        else
          startPos += charLength ( startPos );
        }
    }
    return 0;
#ifdef IC_NLS
  }
#endif
}

#ifdef IC_NLS
unsigned IDBCSBuffer :: windexOfAnyBut ( const IStringTest &aTest,
                                         unsigned           startPos ) const
{
  wchar_t    wc;
#ifdef IC_HAS_RESTARTABLE
  mbstate_t  ss = 0;
#endif

  startPos = this->startSearch( startPos, 1 );
  if ( startPos )
  {
    while ( startPos <= this->length() )
      {
#ifdef IC_HAS_RESTARTABLE
         mbrtowc( &wc, this->contents() + startPos -1 , MB_CUR_MAX , &ss );
#else
         mbtowc( &wc, this->contents() + startPos -1 , MB_CUR_MAX);
#endif
         if ( !aTest.test( wc ) )
             return startPos;
         else
            startPos += charLength ( startPos );
      }
  }
  return 0;
}
#endif

/*------------------------------------------------------------------------------
| IDBCSBuffer::indexOfAnyOf                                                    |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the starting       |
|    point) by calling startSearch().                                          |
| 2. If it can't succeed, return 0.                                            |
| 3. If it can succeed, the starting point is converted to an array (0-based)  |
|    index.                                                                    |
| 4. Search for the first character in the set of search characters, based     |
|    on the number of search characters:                                       |
|      0  - return 0; i.e., no characters of the buffer are in the (empty)     |
|           set of search characters                                           |
|      1  - search for a matching character in the buffer using memchr         |
|      >1 - call memchr for each character of the buffer to see if the         |
|           character is in the set of search characters;                      |
|           return the index of the first that is found                        |
|                                                                              |
|  If all characters fail the test, return 0.                                  |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: indexOfAnyOf ( const char *pValidChars,
                                       unsigned    numValidChars,
                                       unsigned    startPos ) const
{
#if defined(IC_NLS) && defined(IC_MVS400)
  if (IString::isInternationalized()) {
    if (numValidChars == 0) return 0;

    if (startPos == 0) startPos = 1;

    if ((numValidChars == 1) &&
        ((*pValidChars == SO) ||
         (*pValidChars == SI))) {
      for (const char *p = contents() + startPos - 1;
           startPos <= length();
           ++p, ++startPos) {
        if (*p == *pValidChars)
          return startPos;
      }

      return 0;
    } else {
      unsigned numPos = findNextCharNum(contents(), length(), startPos);

      if (numPos) {
        IWBuffer wBuf(contents(), length());
        IWBuffer sBuf(pValidChars, numValidChars);

        return findBytePos(contents(), length(),
                           wBuf.positionOfAnyOf(sBuf, numPos));
      } else {
        return 0;
      }

    }

  } else {
#endif
    startPos = this->startSearch( startPos, 1 );
    if ( startPos )
      {
      switch ( numValidChars )
        {
        case 0:
          break;
        case 1:
          while ( startPos <= this->length() )
            {
            if ( this->contents()[ startPos - 1 ] == *pValidChars )
              return startPos;
            startPos += charLength ( startPos );
            }
          break;
        default:
          while ( startPos <= this->length() )
            {
            if ( this->isCharValid( startPos,
                                    pValidChars,
                                    numValidChars ) )
              return startPos;
            else
              startPos += charLength ( startPos );
            }
        }
      }
    return 0;
#if defined(IC_NLS) && defined(IC_MVS400)
  }
#endif
}


/*------------------------------------------------------------------------------
| IDBCSBuffer::indexOfAnyOf                                                    |

|                                                                              |
| 1. Checking that the search can succeed (and establishing the starting       |
|    point) by calling startSearch().                                          |
| 2. Invokes the IStringTest test method against each character in turn, until |
|    the argument IStringTest returns a true result.                           |
|                                                                              |
| If all characters fail the test, return 0.                                   |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: indexOfAnyOf ( const IStringTest &aTest,
                                       unsigned           startPos ) const
{
#ifdef IC_NLS
  if (IString::isInternationalized()) {
    if (startPos == 0) startPos = 1;

    unsigned numPos = findNextCharNum(contents(), length(), startPos);

    if (numPos) {
      IWBuffer wBuf(contents(), length());

      while (numPos <= wBuf.charNum()) {
        if (aTest.test(wBuf[numPos]))
          return findBytePos(contents(), length(), numPos);
        ++numPos;
      }
      return 0;
    } else {
      return 0;
    }
  } else {
#endif
    startPos = this->startSearch( startPos, 1 );
    if ( startPos )
      {
      while ( startPos <= this->length() )
        if ( aTest.test( this->contents()[ startPos - 1 ] ) )
          return startPos;
        else
          startPos++;
      }
    return 0;
#ifdef IC_NLS
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::lastIndexOf                                                     |
|                                                                              |
|  1. Adjust starting position and see if the search can                       |
|     succeed by calling startBackwardsSearch().                               |
|  2. If a match isn't possible, return 0.                                     |
|  3. Otherwise, search (backward) according to length of search               |
|     string:                                                                  |
|       0  - return 0;                                                         |
|       1  - search for a matching character in the buffer by                  |
|            comparing one byte at a time                                      |
|       >1 - call memcmp for each character of the buffer to see               |
|            if the characters starting here match the search string.          |
|                                                                              |
| If no match is found, return 0.                                              |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: lastIndexOf ( const char *pSearchString,
                                      unsigned    searchLen,
                                      unsigned    startPos ) const
{
#if defined(IC_NLS) && defined(IC_MVS400)
  if (IString::isInternationalized()) {
    if (searchLen == 1)
      return lastIndexOfAnyOf(pSearchString, 1, startPos);

    if (searchLen == 0) return 0;

    if (startPos > length()) startPos = length();

    unsigned numPos = findCharNum(contents(), length(), startPos);

    if (numPos) {
      IWBuffer wBuf(contents(), length());
      IWBuffer sBuf(pSearchString, searchLen);

      return findBytePos(contents(),length(),
                         wBuf.lastPositionOf(sBuf, numPos));
    } else {
      return 0;
    }
  } else {
#endif
    startPos = this->startBackwardsSearch( startPos, searchLen );
    if ( startPos )
      {
      switch ( searchLen )
        {
        case 0:
          break;
        case 1:
          while ( startPos )
            if ( this->contents()[ startPos - 1 ] == *pSearchString )
              return startPos;
            else
              startPos -= prevCharLength ( startPos );
          break;
        default:
          while ( startPos )
            {
            if ( memcmp( this->contents() + startPos - 1,
                         pSearchString,
                         searchLen ) == 0 )
              return startPos;
            else
              startPos -= prevCharLength ( startPos );
            }
        }
      }
    return 0;
#if defined(IC_NLS) && defined(IC_MVS400)
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::lastIndexOf                                                     |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: lastIndexOf ( const IStringTest &aTest,
                                      unsigned           startPos ) const
{
#ifdef IC_NLS
  if (IString::isInternationalized()) {
    return lastIndexOfAnyOf( aTest, startPos );
  } else
#endif
    return this->IBuffer::lastIndexOf( aTest, startPos );
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::lastIndexOfAnyBut                                               |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the                |
|    starting point) by calling startBackwardsSearch().                        |
| 2. If it can't succeed, return 0.                                            |
| 3. If it can succeed, the starting point is converted to an                  |
|    array (0-based) index.                                                    |
| 4. Search (going backwards) for the first character not in the               |
|    set of valid characters, based on the number of valid                     |
|    characters:                                                               |
|      0  - return the buffer length; i.e., the last character of              |
|           the buffer isn't in the (empty) set of valid characters            |
|      1  - compare each character of the buffer to the valid                  |
|           character until one doesn't match                                  |
|      >1 - call memchr for each character of the buffer to see                |
|           if the character is in the set of valid characters;                |
|           return the index of the first that isn't found                     |
|                                                                              |
| If all characters pass the test, return 0.                                   |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: lastIndexOfAnyBut ( const char *pValidChars,
                                            unsigned    numValidChars,
                                            unsigned    startPos ) const
{
#if defined(IC_NLS) && defined(IC_MVS400)
  if (IString::isInternationalized()) {
    if (startPos > length()) startPos = length();

    if (numValidChars == 0)
      return findCurPos(contents(), length(), startPos);

    if ((numValidChars == 1) &&
        ((*pValidChars == SO) ||
         (*pValidChars == SI))) {
      for (const char *p = contents() + startPos - 1;
           startPos;
           --p, --startPos) {
        if (*p != *pValidChars)
          return startPos;
      }

      return 0;
    } else {
      unsigned numPos = findCharNum(contents(), length(), startPos);

      if (numPos) {
        IWBuffer wBuf(contents(), length());
        IWBuffer sBuf(pValidChars, numValidChars);

        return findBytePos(contents(), length(),
                           wBuf.lastPositionOfAnyBut(sBuf, numPos));
      } else {
        return 0;
      }

    }

  } else {
#endif
    startPos = this->startBackwardsSearch( startPos, 1 );
    if ( startPos )
      {
      switch ( numValidChars )
        {
        case 0:
          return startPos;
        case 1:
          if ( ! isSBC ( pValidChars ) )
            return 0;
          while ( startPos )
            if ( this->contents()[ startPos - 1 ] != *pValidChars )
              return startPos;
            else
              startPos -= prevCharLength ( startPos );
          break;
        default:
          while ( startPos )
            if ( !this->isCharValid( startPos, pValidChars, numValidChars ) )
              return startPos;
            else
              startPos -= prevCharLength ( startPos );
        }
      }
    return 0;
#if defined(IC_NLS) && defined(IC_MVS400)
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::lastIndexOfAnyBut                                               |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the starting       |
|    point) by calling startBackwardsSearch().                                 |
| 2. Invokes the IStringTest test method against each character in turn, until |
|    the argument IStringTest returns a false result.                          |
|                                                                              |
| If all characters pass the test, return 0.                                   |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: lastIndexOfAnyBut ( const IStringTest &aTest,
                                            unsigned   startPos ) const
{
#ifdef IC_NLS
  if (IString::isInternationalized()) {
    if (startPos > length()) startPos = length();

    unsigned numPos = findCharNum(contents(), length(), startPos);

    if (numPos) {
      IWBuffer wBuf(contents(), length());

      while (numPos) {
        if (!aTest.test(wBuf[numPos]))
          return findBytePos(contents(), length(), numPos);
        --numPos;
      }
      return 0;
    } else {
      return 0;
    }
  } else {
#endif
#ifdef IC_HAS_RESTARTABLE
    mbstate_t  ss = 0;
#endif
    
    unsigned int endPos = this->startBackwardsSearch( startPos, 1 );
    unsigned int newStartPos = 1;
    unsigned int lastPos = 0;
        
    if ( endPos )
    {
      while ( newStartPos <= endPos )
        {
           if ( !aTest.test( this->contents()[ newStartPos - 1 ] ) ) {
             lastPos = newStartPos;
           }
#ifdef IC_HAS_RESTARTABLE
           newStartPos += charLength ( newStartPos, &ss );
#else
           newStartPos += charLength ( newStartPos );
#endif
        }
    }
    return lastPos;
    
#ifdef IC_NLS
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer :: lastIndexOfAnyOf                                              |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the                |
|    starting point) by calling startBackwardsSearch().                        |
| 2. If it can't succeed, return 0.                                            |
| 3. Otherwise, search for the first character in the set of                   |
|    search characters, based on the number of search characters:              |
|      0  - return 0; i.e., no characters of the buffer are in                 |
|           the (empty) set of search characters                               |
|      1  - search for a matching character in the buffer by                   |
|           comparing buffer characters one-by-one (going backwards).          |
|      >1 - call memchr for each character of the buffer to see                |
|           if the character is in the set of search characters                |
|           return the index of the first that is found                        |
|                                                                              |
| If all characters fail the test, return 0.                                   |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: lastIndexOfAnyOf ( const char *pValidChars,
                                           unsigned    numValidChars,
                                           unsigned    startPos ) const
{
#if defined(IC_NLS) && defined(IC_MVS400)
  if (IString::isInternationalized()) {
    if (numValidChars == 0) return 0;

    if (startPos > length()) startPos = length();

    if ((numValidChars == 1) &&
        ((*pValidChars == SO) ||
         (*pValidChars == SI))) {
      for (const char *p = contents() + startPos - 1;
           startPos;
           --p, --startPos) {
        if (*p == *pValidChars)
          return startPos;
      }

      return 0;
    } else {
      unsigned numPos = findCharNum(contents(), length(), startPos);

      if (numPos) {
        IWBuffer wBuf(contents(), length());
        IWBuffer sBuf(pValidChars, numValidChars);

        return findBytePos(contents(), length(),
                           wBuf.lastPositionOfAnyOf(sBuf, numPos));
      } else {
        return 0;
      }

    }

  } else {
#endif
    startPos = this->startBackwardsSearch( startPos, 1 );
    if ( startPos )
      switch ( numValidChars )
        {
        case 0:
          break;
        case 1:
          while ( startPos )
            if ( this->contents()[ startPos - 1 ] == *pValidChars )
              return startPos;
            else
              startPos -= prevCharLength ( startPos );
          break;
        default:
          while ( startPos )
            if ( this->isCharValid( startPos, pValidChars, numValidChars ) )
              return startPos;
            else
              startPos -= prevCharLength ( startPos );
        }
    return 0;
#if defined(IC_NLS) && defined(IC_MVS400)
  }
#endif
}

/*------------------------------------------------------------------------------
| IDBCSBuffer::lastIndexOfAnyOf                                                |
|                                                                              |
| 1. Checking that the search can succeed (and establishing the starting       |
|    point) by calling startBackwardsSearch().                                 |
| 2. Invokes the IStringTest test method against each character in turn, until |
|    the argument IStringTest returns a true result.                           |
|                                                                              |
| If all characters fail the test, return 0.                                   |
------------------------------------------------------------------------------*/
unsigned IDBCSBuffer :: lastIndexOfAnyOf ( const IStringTest &aTest,
                                          unsigned      startPos ) const
{
#ifdef IC_NLS
  if (IString::isInternationalized()) {
    if (startPos > length()) startPos = length();

    unsigned numPos = findCharNum(contents(), length(), startPos);

    if (numPos) {
      IWBuffer wBuf(contents(), length());

      while (numPos) {
        if (aTest.test(wBuf[numPos]))
          return findBytePos(contents(), length(), numPos);
        --numPos;
      }
      return 0;
    } else {
      return 0;
    }
  } else {
#endif
    startPos = this->startBackwardsSearch( startPos, 1 );
    while ( startPos )
      if ( aTest.test( this->contents()[ startPos - 1 ] ) )
        return startPos;
      else
        startPos -= prevCharLength ( startPos );
    return 0;
#ifdef IC_NLS
  }
#endif
}
