/*----------------------------------------------------------------------------*/
/* FILE NAME: ibuffer0.cpp                                                    */
/*                                                                            */
/* DESCRIPTION:                                                               */
/*   This file contains the implementation of classes/functions declared      */
/*   in ibuffer.hpp.                                                          */
/*                                                                            */
/*                                                                            */
/* COPYRIGHT:                                                                 */
/*   IBM Open Class Library                                                   */
/*   Licensed Materials - Property of IBM                                     */
/*                                                                            */
/*   5645-001                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
/*                                                                            */
/*   US Government Users Restricted Rights - Use, duplication, or             */
/*   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.        */
/*                                                                            */
/*----------------------------------------------------------------------------*/
#ifdef __MVS__
    #pragma csect(code,"IBUFFER0")
    #pragma csect(static,"SBUFFER0")
    #pragma comment (copyright, \
    "Licensed Materials - Property of IBM  5645-001 (C) \
    Copyright IBM Corp. 1992, 1997  All Rights Reserved. \
    US Government Users Restricted Rights - Use, duplication, or \
    disclosure restricted by GSA ADP Schedule Contract with IBM Corp.")
#endif
#pragma priority( -2147482324 )

#include <istring.hpp>
#include <ibuffer.hpp>

extern "C" {
  #include <string.h>
  #include <stdlib.h>
  #include <stdio.h>
#ifdef IC_PMWIN
  #define INCL_DOSNLS
#ifdef IC_WIN
  #include <windows.h>
#endif
#ifdef IC_PM
  #include <os2.h>
#endif
#endif
  }


#include <idbcsbuf.hpp>
#include <iexcbase.hpp>
#include <icconsta.h>
#include <iexcept.hpp>
#ifdef IC_WIN
#include <iplatfrm.hpp>
#endif
#include <istatics.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IBUFFER0_CPP_
  #include <ipagetun.h>
#endif
#ifdef IC_400
  #include <qusrjobi.h>
#endif

#ifdef IC_PMWIN

char
  IBuffer::dbcsTable[ 256 ];
#endif

#ifdef IC_PMWIN
#endif


/*------------------------------------------------------------------------------
| IBuffer::className                                                           |
------------------------------------------------------------------------------*/
const char * IBuffer :: className ( ) const
  {
  return "IBuffer";
  }

/*------------------------------------------------------------------------------
| IBuffer::asDebugInfo                                                         |
------------------------------------------------------------------------------*/
IString IBuffer :: asDebugInfo ( ) const
  {
  IString result( this->className() );
  result += "(@";
  result += IString( (unsigned long) this );
  result += ",refs=";
  result += IString( this->useCount() );
  result += ",len=";
  result += IString( this->length() );
  result += ",data=[";
  if ( this->length() > 23 )
    {
    result += IString( this->contents(), 10 );
    result += "...";
    result += IString( this->contents() + this->length() - 10, 10 );
    }
  else
    result += this->contents();
  result += "])";
  return result;
  }


/*------------------------------------------------------------------------------
| IBuffer :: charType                                                          |
------------------------------------------------------------------------------*/
IStringEnum::CharType IBuffer :: charType ( unsigned ) const
{
    return IStringEnum::sbcs;
}

/*------------------------------------------------------------------------------
| IBuffer::allocate                                                            |
| Buffers are allocated in 32 bytes boundary. E.g. if                          |
| newLen = 1..31, the actual allocated size is 32 bytes.                       |
|                                                                              |
| To calculate the remaining (or reserved) allocated space, you can try:       |
|   total_size = checkAddition(sizeof(IBuffer), this->length());               |
|   allocate_size = (total_size | 31) + 1;                                     |
|   reserved_size = allocate_size - total_size;                                |
|                                                                              |
| This is the usable space and there is no need to account for the null char   |
| because the null char space is accounted for in the IBuffer object.          |
------------------------------------------------------------------------------*/
IBuffer *
IBuffer::allocate(unsigned newLen) const
{
    unsigned int allocate_size = 0;

    if (newLen)
        allocate_size = (newLen | 31) + 1;

#if defined(IC_MOTIF) || defined(IC_MVS)
  if (MB_CUR_MAX > 1)
      return new (allocate_size) IDBCSBuffer(newLen);
  else
      return new (allocate_size) IBuffer(newLen);
#endif
#if defined(IC_400) || defined(IC_PM) || defined(IC_WIN)
#ifdef IC_NLS
  if (IString::isInternationalized())
  {
      if (MB_CUR_MAX > 1)
          return new (allocate_size) IDBCSBuffer(newLen);
      else
          return new (allocate_size) IBuffer(newLen);
  } else
#endif
      return new (allocate_size) IBuffer(newLen);
#endif
}

/*------------------------------------------------------------------------------
| IBuffer::IBuffer                                                             |
------------------------------------------------------------------------------*/
IBuffer :: IBuffer ( unsigned length )
  : len( length )
  {
  addRef();
  data[ length ] = '\0';

  // printf ("IBuffer::IBuffer Address of IBuffer = %p \n ", this);
  // printf ("address of data = %p \n ", data);
  // printf ("address of Computed IBuffer  = %p \n", data - 12);

  // printf ("offset = %d \n", (int)  offsetof(IBuffer, data) );
  }

/*------------------------------------------------------------------------------
| IBuffer::~IBuffer                                                            |
------------------------------------------------------------------------------*/
IBuffer :: ~IBuffer ( )
  {
   // printf ("IBuffer::~IBuffer called \n ");
  }

/*------------------------------------------------------------------------------
| IBuffer::nullBuffer                                                          |
|                                                                              |
| IBuffer with "null" value.                                                   |
|                                                                              |
| This buffer is the "default" buffer used to construct IStrings.              |
|                                                                              |
| All "null" IStrings will point to this instance.  It's "contents"            |
| are a single null character.  This ensures that using a null                 |
| IString equates to using a null string (in more of the C sense).             |
| When running on a DBCS machine, the null buffer will be an instance of       |
| IDBCSBuffer and a new nullIBuffer is used when a null IBuffer instance is    |
| needed on DBCS machines.                                                     |
------------------------------------------------------------------------------*/
IBuffer *IBuffer::nullBuffer = 0;
IBuffer *IBuffer::nullIBuffer = 0;
bool IBuffer::is32bytesAligned = true;

/*------------------------------------------------------------------------------
| IBuffer::null                                                                |
------------------------------------------------------------------------------*/
IBuffer *IBuffer :: null ( ) const
  {
  if ( nullIBuffer && ( nullIBuffer->className() == this->className() ))
    {
    return nullIBuffer;
    }
  return IBuffer::defaultBuffer();
  }

/*------------------------------------------------------------------------------
| IBuffer::defaultBuffer                                                       |
------------------------------------------------------------------------------*/
IBuffer *IBuffer :: defaultBuffer ( )
{
    if (nullBuffer)
        return nullBuffer;
    nullBuffer = IBuffer::initialize();
    IStaticObjectManager::manageRefCountOf( nullBuffer );
    return nullBuffer;
}

/*------------------------------------------------------------------------------
| IBuffer::initialize                                                          |
|                                                                              |
| Build DBCS table and return pointer to appropriate (null) IBuffer.           |
------------------------------------------------------------------------------*/
IBuffer *IBuffer :: initialize ( )
  {
  unsigned      n = 0;
  signed int           i;
  bool       dbcs = false;

#if (defined(IC_MVS) || defined(IC_MOTIF))
  dbcs = (MB_CUR_MAX > 1);
#endif
#if (defined(IC_PM) || defined(IC_WIN) || defined(IC_400))

#ifdef IC_PM
  UCHAR         buffer[12];
  COUNTRYCODE   countryCode;
  countryCode.country = 0;     // 0 means return DBCS information for
  countryCode.codepage = 0;    //  the current process code page.
  DosQueryDBCSEnv( sizeof(buffer), &countryCode, (PCHAR)buffer );
#endif

#ifdef IC_WIN
  CPINFO        infoCP;
  UCHAR*        buffer = &infoCP.LeadByte[0];
   if (IPlatform::isWinNT())
      GetCPInfo(GetConsoleCP(), &infoCP);
   else
      GetCPInfo(CP_ACP, &infoCP);
#endif

#ifdef IC_400
  #define JOBIFORMAT "JOBI0600"                    // >= 8 in length
  #define THISJOB    "*                            " // >= 26 in length
  Qwc_JOBI0600 _jobinfo;
  short int _joblength;      // must be short int (4 bytes)
  char _jobiid[20];
  memset(_jobiid, ' ', 20);
  _joblength = sizeof(_jobinfo);
  QUSRJOBI(&_jobinfo, _joblength, JOBIFORMAT, THISJOB, _jobiid);
  dbcs = (_jobinfo.Dbcs_Enabled[0] == '1');
#endif
#if ( defined(IC_PM) || defined(IC_WIN)  )
  if ( buffer[0] | buffer[1] )
  {
    dbcs = true;
    for (i = 0; i < sizeof(dbcsTable); i++)
    {
      if ( i >= buffer[n] && i <= buffer[n+1] )
        dbcsTable[i] = 1;
      else
        dbcsTable[i] = 0;
      if ( i == buffer[n+1] )
        n += 2;
    }
  }
  else
  {
    for (i = 0; i < sizeof(dbcsTable); i++)
      dbcsTable[i] = 0;
    // Test code: Query IDBCSTABLE environment variable and
    // use its value as valid DBCS first byte codes.
    char *var = getenv( "IDBCSTABLE" );
    if ( var )
      while ( *var )
        {
        dbcs = true;
        dbcsTable[ (unsigned)(*var++) ] = 1;
        }
  }
#endif
#ifdef IC_NLS
  if (IString::isInternationalized())
  {
     dbcs = (MB_CUR_MAX > 1);
  }
#endif
#endif // IC_PM IC_WIN IC_400
  if ( dbcs )
    {
    nullIBuffer = new (0) IBuffer( 0 );
    return new (0) IDBCSBuffer( 0 );
    }
  else
    return new (0) IBuffer( 0 );
  }

/*------------------------------------------------------------------------------
| IBuffer::next                                                                |
------------------------------------------------------------------------------*/
char *IBuffer :: next ( const char *prev )
  {
  return (char*)(++prev);
  }

/*------------------------------------------------------------------------------
| IBuffer::next                                                                |
------------------------------------------------------------------------------*/
const char *IBuffer :: next ( const char *prev ) const
  {
  return ++prev;
  }

/*------------------------------------------------------------------------------
| IBuffer::IBuffer                                                             |
|                                                                              |
| Construct a buffer of the specified length.  The "data" member               |
| array must actually be of length 1 greater than the argument                 |
| value (this is achieved automatically via use of the overloaded              |
| operator new for class IBuffer).                                             |
|                                                                              |
| The terminating (extra) byte is set to null.                                 |
|                                                                              |
| Notes:                                                                       |
|   1. This method is protected.  IBuffers must be obtained by using           |
|      IBuffer::nullBuffer and subsequent newBuffer calls to existing          |
|      IBuffer objects.  The only non-heap instance of this class is           |
|      the static IBuffer::nullBuffer object.                                  |
|   2. This method should be inline (in context of IBuffer::newBuffer).        |
------------------------------------------------------------------------------*/
// See ibuffer.inl

IBuffer *IBuffer :: fromContents ( const char *p )
  {
   return (IBuffer*)( p - offsetof( IBuffer, data ) );
  }

/*------------------------------------------------------------------------------
| IBuffer::newBuffer                                                           |
|                                                                              |
| Allocate a new IBuffer of the same type as the receiver and initialize it    |
| with the provided data.                                                      |
------------------------------------------------------------------------------*/
IBuffer *IBuffer :: newBuffer ( const void *p1,
                                unsigned    len1,
                                const void *p2,
                                unsigned    len2,
                                const void *p3,
                                unsigned    len3,
                                char        padChar )
  const
  {
  unsigned
    newLen = checkAddition( checkAddition( len1, len2 ),  len3 );
  IBuffer *buffer;
  if ( newLen )
    {
    buffer = this -> allocate( newLen );
    char *p = buffer->contents();
    // Copy first portion (or pad).
    if ( p1 )
      memcpy( p, p1, len1 );
    else
      memset( p, padChar, len1 );
    p += len1;
    // Copy second portion (or pad).
    if ( p2 )
      memcpy( p, p2, len2 );
    else
      memset( p, padChar, len2 );
    p += len2;
    // Copy third portion (or pad).
    if ( p3 )
      memcpy( p, p3, len3 );
    else
      memset( p, padChar, len3 );
    }
  else
    {
    buffer = this->null();
    buffer -> addRef();
    }
  return buffer;
  }

/*------------------------------------------------------------------------------
| IBuffer :: operator new                                                      |
|                                                                              |
| Allocate an IBuffer object with "data" array of requested size.              |
|                                                                              |
| Notes:                                                                       |
|   1. The argument size is the size of the string to be placed in             |
|      the buffer, *excluding* the terminating null.  The total amount         |
|      of space is thus the size of the IBuffer header (use count and          |
|      length fields), plus the string length (as given by the argument),      |
|      plus 1 more byte (for the terminating null).  Since one byte of         |
|      the data array is declared in IBuffer, the proper amount of             |
|      space is calculated by the expression used below.                       |
------------------------------------------------------------------------------*/
// See ibuffer.inl

/*------------------------------------------------------------------------------
| IBuffer::operator delete                                                     |
|                                                                              |
| Delete the storage using the inverse of the operator used to allocate it.    |
------------------------------------------------------------------------------*/
// See ibuffer.inl

/*------------------------------------------------------------------------------
| IBuffer :: addRef                                                            |
|                                                                              |
| Increment use count for this IBuffer.                                        |
------------------------------------------------------------------------------*/
// See ibuffer.inl

/*------------------------------------------------------------------------------
| IBuffer :: removeRef                                                         |
|                                                                              |
| Decrement the use count of the receiver.                                     |
|                                                                              |
| If the resulting use count is now zero, release the IBuffer by               |
| calling delete against "this."                                               |
|                                                                              |
| Notes:                                                                       |
|   1. This function appears dangerous.  However, a call to this function      |
|      is interpreted as giving up access to the associated buffer.  At        |
|      that point, only other references are valid (and if there are none,     |
|      then the receiver can be deleted).                                      |
------------------------------------------------------------------------------*/
// See ibuffer.inl

/*------------------------------------------------------------------------------
| IBuffer :: rightJustify                                                      |
|                                                                              |
| This function right justifies the receiver in a new buffer of the given      |
| length and returns the address of the new buffer.                            |
|                                                                              |
| The right justification algorithm is as follows:                             |
|                                                                              |
|   1. Calculate how much padding will be needed on the left.                  |
|   2. Calculate how many bytes of the current buffer will be                  |
|      transferred to the right-justified result.                              |
|   3. return a new buffer comprised of the "prefix" pad bytes                 |
|      and the portion of the receiver calculated at step 2.                   |
|                                                                              |
| Notes:                                                                       |
|   1. This function (and likewise all the similar "editing" functions)        |
|      is usually called in the following context:                             |
|         IString::xxxxx ( )                                                   |
|           {                                                                  |
|           ...                                                                |
|           IString old( self.pBuffer );                                       |
|           self.pBuffer = old.pBuffer->rightJustify( ... );                   |
|           ...                                                                |
|           }                                                                  |
|   2. This function appears here (rather than with the other editing          |
|      functions in ibuffer7.cpp) because it is called from the conversion     |
|      functions in istring9/ibuffer9.  Placing it here avoids pulling in      |
|      all the editing functions when only conversion is needed.               |
------------------------------------------------------------------------------*/
IBuffer * IBuffer :: rightJustify ( unsigned newLength,
                                    char     padCharacter )
  {
  if ( newLength != this->length() )
     {
     unsigned prefix       = ( newLength > this->length() )
                               ? newLength - this->length()
                               : 0,
              fromReceiver = ( this->length() < newLength ) ?
                               this->length()
                               :
                               newLength;
     return this->newBuffer( 0,
                             prefix,
                             this->contents() + this->length() - fromReceiver,
                             fromReceiver,
                             0, 0,
                             padCharacter );
     } else {
       this->addRef();
     }

  return this;
  }

/*------------------------------------------------------------------------------
| IBuffer::remove                                                              |
|                                                                              |
| If the buffer will remain unmodified (starting index is beyond the end or    |
| receiver is null), then the receiver is returned unmodified.                 |
|                                                                              |
| Else, if the number of characters to delete is zero, or passes the end of    |
| the receiver, then it is defaulted to the rest of the buffer. Finally, the   |
| new buffer is allocated and filled in via a call to newBuffer().             |
|                                                                              |
| Notes:                                                                       |
|   1. See notes 1. and 2. under IBuffer::rightJustify.                        |
------------------------------------------------------------------------------*/
IBuffer * IBuffer :: remove ( unsigned startPos,
                              unsigned numChars )
  {
  if ( startPos <= this->length()
       &&
       this->length() != 0 )
    {
    // Consider startPos 0 to be 1:
    if ( startPos == 0 )
      startPos++;
    // Default numChars to rest of string:
    if ( numChars > this->length() - startPos )
      numChars = this->length() - startPos + 1;

    // Initialize from current contents before/after deleted chars:
    return this->newBuffer( this->contents(),
                            startPos - 1,
                            this->contents() + startPos + numChars - 1,
                            this->length() - numChars - startPos + 1,
                            0, 0,
                            0 );
    } else {
      this->addRef();
    }

  return this;
  }

/*------------------------------------------------------------------------------
| IBuffer::overflow                                                            |
|                                                                              |
| Throw "overflow" exception.                                                  |
------------------------------------------------------------------------------*/
unsigned IBuffer :: overflow ( )
  {
#ifdef IC_MOTIF
  IInvalidRequest exception(IMessageText(IC_ISTRING_OVERFLOW),
                           IC_ISTRING_OVERFLOW, IException::recoverable);
  ITHROW( exception );
#else
  ITHROWLIBRARYERROR( IC_ISTRING_OVERFLOW,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
#endif
  return 0;
  }

#ifdef IC_PMWIN
bool IBuffer::isDBCSLead(char inByte)
{
   return IBuffer::dbcsTable[inByte] ? true : false;
}
#endif
