/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
||
||  File:       XDevices.cpp
||
||  What:       Routines which support the graphics device on AIX platform
||
*/

// Revision: 07 1.46.3.2 source/albert/graph2d/xdevice.cpp, 2d, ioc.v400, 001006 

#include <ilanglvl.hpp>

#include <xdevice.hpp>
#include <igimage.hpp>
#include <grstate.hpp>
#include <regnextr.hpp>
#include <inumeric.hpp>
#include <igloop2d.hpp>
#include <igtxtrun.hpp>
#include <grassert.hpp>
#include <ibmpdata.hpp>
#include <iprimlck.hpp>
#include <igbidi.hpp>
#include <igrport.hpp>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <Xm/Xm.h>

#include <ibase.hpp>
#include <istring.hpp>
#include <itrace.hpp>

#include <pixbuf.hpp>
#include <ixfntset.hpp>
#include <ixlfdstr.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _XDEVICE_CPP_
#include <ipagetun.h>
#endif

#define NEAREST_INT(_val) ((int)(_val))

//This was added for the defect 27755. The fix for this defect is nessasary
//for albert but fixing it breaks Montana. So, till the adjustment of 
//lower-left coord origin (which was earlier adjusted to one pixel up) 
//is removed from the montana code this flag would remain. 

bool fLowerLeftCoordOriginHack = false;

//==============================================================
//     The device maker implementation -
//     this should probably in a saperate file itself when we have more than one device
//

IGrafDevice* IGrafDeviceMaker::makeDevice(
       const IPresSpaceHandle& deviceContext,
       const IGrafPort::EGrafPortType  grafPortType,
       EDeviceMappingMode mappingMode)
{
       IFUNCTRACE_DEVELOP();
       if (grafPortType == IGrafPort::kIExtendedRootGrafPort)
               return new IXDevice(deviceContext,mappingMode, false);
       else
               return new IXDevice(deviceContext,mappingMode, true);
}

//================================================================
//  helper class for font and text support...
//================================================================

#pragma pack(push,4)

class IXFontContext {
public:
       IXFontContext(
               IXDevice* theDevice,
               const IGTextRun& theText,
               const IGrafMatrix*,
               const IGrafMatrix*);

       virtual ~IXFontContext();

       void renderTextRunGlyphs(const IGTextRun& textRun,
                            IGPoint2D&,
                            const IGrafState& grafState) ;
       IGPoint2D calculateBoundsAndAdvancement(const IGTextRun& textRun ) const;

       bool cacheHit(
               IXDevice* theDevice,
               const IGTextRun& theText,
               const IGrafMatrix* model,
               const IGrafMatrix* flip) const;

       void loadFont();

       unsigned long fMatrixStamp;
       unsigned long fFlipStamp;

       IGrafMatrix fMatrixUsed;
       const IGrafState* fGrafState;

       XFontStruct* fFontStruct;

       IGRect2D* fBounds;

protected:
       virtual void renderGlyphs(const IGTextRun&,
                            IGPoint2D& ,
                            Drawable& xDrawable, GC gc ) const;
       IXDevice* fDevice;

private:
       bool IsBlack(int pixel_id );
       void strikeOutText(const IGTextRun&,
                IGPoint2D position,
                Drawable& xDrawable, IGRect2D*) const;

       const IGrafMatrix* fAnchorTransform;
       bool fTmCached;

       Font fFont;
	XFontSet fFontSet;
       XFontStruct** fPtrFontStruct;

       // for caching GdiFontContext:
       const IGrafMatrix* fMatrix;
       const IGrafMatrix* fFlip;

       IPresSpaceHandle fDeviceContext;
       IXDC* fIXDeviceContext;

       int fEscapement;
       IString fTypeFace;
       GCoordinate fPointSize;

       int fWeight;
       short fItalic;
       short fUnderline;
       short fStrikeOut;
       short fFixedPitch;
};

#pragma pack(pop)

//----------------------------------------------------------------

#if !IC_RTTI_ENABLED
GrafDeviceTypeDefinitionsMacro(IXDevice);
#endif // IC_RTTI_ENABLED

//==============================================================
//  Statics for IXDevice(info shared by all instances)
//

// min/max values in device coordinates ... initialised to LONG_MIN/LONG_MAX, only Win95 had SHORT...
GCoordinate IXDevice::gMaxCoord = (GCoordinate)LONG_MAX - 1;
GCoordinate IXDevice::gMinCoord = (GCoordinate)LONG_MIN + 1;

bool IXDevice::fInitialized = false;

IXDevice::EPlatform IXDevice::fPlatform = IXDevice::kUndefined;

#pragma enum(4)
enum EClipState {
       kClipAreaEmpty = 0,
       kClipAreaInfinite,
       kClipAreaRegular
};
#pragma enum(pop)


#if 1 //SAS
//==============================================================
//  Utility Routine for Translating from GCoordinate to integer
//

int IXDevice::GCoordinateToInt(GCoordinate value)
{
       if (value > 0)
       {
               value += 0.5;           // round up
               if (value > gMaxCoord)
                       value = gMaxCoord;
       }
       else if (value < 0)
       {
               value -= 0.5;           // round down
               if (value < gMinCoord)
                       value = gMinCoord;
       }
       return (short)floor(value);
}
#else //SAS

GCoordinateToInt(value)\
       (int)(((value) > 0) ? ((value) += 0.5,\
               (((value) > (double)(LONG_MAX -1)) ? (value=((double)(LONG_MAX -1)), (value)) : (value))) :\
               ((value) -= 0.5,\
               (((value) > (double)(LONG_MIN -1)) ? (value=((double)(LONG_MIN -1)), (value)) : (value))))
#endif //SAS

void IXDevice::IGPoint2DToPOINTL(const IGPoint2D &p, XPoint &ptl)
{
       ptl.x = GCoordinateToInt(p.fX);
       ptl.y = GCoordinateToInt(p.fY);
}

#ifdef IC_TRACE_ALL
static int IXDevice_allocated = 0;
#endif
//==============================================================
//     Constructors
//

IXDevice::IXDevice(const IPresSpaceHandle& deviceHandle) :
       fDrawOperation(IAttributeState::kUnset),
       fMatrix(),
       fOwnHDC(false),
       fResolution(0 /*NIL*/),
       fFontCache(0),
       fHDC(deviceHandle),
       fIXDC((IXDC*)deviceHandle),
       fBaseRootDevice(true),
       fColors(0)
{
       IFUNCTRACE_DEVELOP();
       initialise(kPixel);
}

IXDevice::IXDevice(const IPresSpaceHandle& deviceHandle,
                                       EDeviceMappingMode mappingMode,
                                       bool baseRootFlag ) :
       fDrawOperation(IAttributeState::kUnset),
       fMatrix(),
       fOwnHDC(false),
       fResolution(0 /*NIL*/),
       fFontCache(0),
       fHDC(deviceHandle),
       fIXDC((IXDC*)deviceHandle),
       fBaseRootDevice(baseRootFlag),
       fColors( 0 )
{
       IFUNCTRACE_DEVELOP();
       if (fBaseRootDevice)
               fResolution = resolutionMatrix() ;

       initialise(mappingMode);
}

IXDevice::IXDevice(
       const IPresSpaceHandle& deviceHandle,
       const IGRect2D& imagingRect,
       const IGrafMatrix& resolution) :
       fHDC(deviceHandle),
       fIXDC((IXDC*)deviceHandle),
       fDrawOperation(IAttributeState::kUnset),
       fMatrix(),
       fOwnHDC(false),
       fFontCache(0),
       fColors( 0 ),
       fResolution(new IGrafMatrix(resolution))
{
       IFUNCTRACE_DEVELOP();
       initialise(kPixel, &imagingRect);
}

void IXDevice::initialise(int mmode, const IGRect2D* clipRect)
{
       ITRACE_ALL(IString("IXDevice_allocated=") +
                  IString(++IXDevice_allocated));
       if (!fInitialized) {
        IPrimalLock lock;
        if (!fInitialized) {
               fInitialized = true;
               fPlatform = kUnix;
               fMatrixCacheUsed = false;
               fClipAreaSet = false;

               if ((!fIXDC) || (fIXDC->xDisplay == NULL))
            {
                Display* defDisplay = IXDisplay::display();
                Drawable  defDrawble = DefaultRootWindow( defDisplay);
                       IXDC newXDC(defDisplay, defDrawble);
                       fHDC = IPresSpaceHandle(&newXDC);
                       fIXDC = ((IXDC*)fHDC);
                fOwnHDC = true;
            }
        }
       }

       if (mmode != kPixel) {
               GCoordinate scale = 72.;
               switch (mmode) {
               default:
                       scale = 1.0;
                       break;
               case kLowMetric:
                       scale /= 254.;
                       break;
               case kHighMetric:
                       scale /= 2540.;
                       break;
               case kLowEnglish:
                       scale /= 100.;
                       break;
               case kHighEnglish:
                       scale /= 1000.;
                       break;
               case kTwips:
                       scale /= 1440.;
                       break;
               }

               IGrafMatrix mappingMode; // identity...
               mappingMode.scaleBy(IGPoint2D(scale, scale));
               if (fResolution)
                       fResolution->concatWith(mappingMode);
               else
                       fResolution = new IGrafMatrix(mappingMode);
       }

       if (!clipRect)
       {
               if (fIXDC->xDisplay)
               {
                       // find the height for current client window
                       int x, y;
                       unsigned int wid, ht, depth, border;
                       Window root;
                       XGetGeometry( fIXDC->xDisplay,
                               fIXDC->xDrawable,
                               &root,
                               &x, &y, &wid, &ht, & border, &depth);

                       setWorldBounds(IGRect2D(0, 0, wid, ht));
               }
       }
       else    // custom clipping ...
               setWorldBounds(*clipRect);

       fFrameMatrix = fModelMatrix = fViewMatrix = fPenMatrix = &fMatrix;

       queryColormap();
}


// these three functions are disabled...
IXDevice::IXDevice()
{}

IXDevice::IXDevice(const IXDevice&)
{}

IXDevice& IXDevice::operator=(const IXDevice& Src)
{
       return *this;
}

//==============================================================
//     Destructor
//

IXDevice::~IXDevice()
{
    IFUNCTRACE_DEVELOP();
    ITRACE_ALL(IString("IXDevice_allocated=") +
               IString(--IXDevice_allocated));
    if (fOwnHDC) {
                 fIXDC->decrementCount();
                 if (fIXDC->refCount() == 0)
                         //delete fHDC;
                         fHDC = NULL;
                 fIXDC = NULL;
    }
    delete fResolution;
    delete fFontCache;
}

//==============================================================
//  device inquitries
//

const IGrafMatrix& IXDevice::deviceResolution(const IGRect2D&) const
{
       if (!fResolution)
               return IGrafMatrix::identity();
       else
               return *fResolution;
}

IGrafMatrix* IXDevice::resolutionMatrix()
{
    IFUNCTRACE_DEVELOP();
    Display* display = fIXDC->xDisplay;
    int scrnNumber = DefaultScreen(display);

    int scrnHeight = XDisplayHeight( display, scrnNumber);
    int scrnHeightMM = XDisplayHeightMM( display, scrnNumber);

    int scrnWidth = XDisplayWidth( display, scrnNumber);
    int scrnWidthMM = XDisplayWidthMM( display, scrnNumber);

    GCoordinate pixelsPerInchX = scrnWidth/(scrnWidthMM * 0.04);
    GCoordinate pixelsPerInchY  = scrnHeight/(scrnHeightMM * 0.04);

    IGrafMatrix* resolution = new IGrafMatrix(IGPoint2D((int)(pixelsPerInchX + 0.5)/72,
                                                        (int)(pixelsPerInchY + 0.5)/72),
                                                        IGPoint2D::origin());
    return resolution ;

}

void IXDevice::flush()
{
       IFUNCTRACE_DEVELOP();
       XFlush( fIXDC->xDisplay );
}

// the coordinate system is inherited from the base class - lefthand, righthand on o/s2

const IPresSpaceHandle& IXDevice::deviceContext() const
{
       if (fDeviceHandle == 0 )
       {
               if (fHDC)
                       ((IXDevice*)this)->fDeviceHandle = fHDC;
       }

       return fDeviceHandle;
}

IPresSpaceHandle IXDevice::orphanDeviceContext()
{
       IFUNCTRACE_DEVELOP();
       if (isValid())
       {
               IPresSpaceHandle temp(fHDC);
               setValidity(false);

               // cleanup whatever necessary
               fOwnHDC = false;
               fUsedOwnHDC = fHDC;
               fHDC = 0 /*NIL*/;
               fIXDC = 0 /*NIL*/;

               delete fFontCache;
        fFontCache = 0 /*NIL*/;
               fMatrixCacheUsed = false;
               fClipAreaSet = false;
               return temp;
       }
       return fHDC;
}

void IXDevice::adoptDeviceContext(const IPresSpaceHandle& dc)
{
       fHDC = dc;

       if (!fHDC)
               GrafDeviceException("Invalid IPresSpaceHandle");

       fIXDC = (IXDC*)dc;

       if (fHDC == fUsedOwnHDC) {
               fUsedOwnHDC = 0 /*NIL*/;
       }
       setValidity(true);
}

//============================================================
//     IXDevice::queryColormap
//
void IXDevice::queryColormap()
{
    IColorMap* defColorMap = &( IColorMap::defaultColorMap() );
    if (defColorMap->hasColorMapSupport())
    {  // Make local pointer to colormap color table.
       fColors = defColorMap;
    }
    else
    {  // If there is no colormap support set the pointer to 0 to provide
       // a simple and fast test for colormap availability.
       fColors = 0;
    }

}

//============================================================
//     IXDevice::lookupColormap
//
unsigned long IXDevice::lookupColormap(const IBaseColor& bcolor)
{
    if (fColors)
       return fColors->colorIndex( bcolor );
    else
       return (IXDisplay::rgbToPixel(bcolor.redMix(),
				bcolor.greenMix(),
				bcolor.blueMix()));
}

static void destroyBrushPixmap(Display *dpy, GC gc)
{
    IFUNCTRACE_DEVELOP();
    unsigned long mask = 0x0;
	XGCValues gcval;

	if(XGetGCValues(dpy, gc, GCFillStyle|GCTile|GCStipple, &gcval)){
		if(((gcval.fill_style & FillTiled) == FillTiled) && 
				gcval.tile){
			XFreePixmap(dpy, gcval.tile);
			mask |= GCTile;
		}
		if(((gcval.fill_style & FillStippled) == FillStippled) && 
				gcval.stipple){
			XFreePixmap(dpy, gcval.stipple);
			mask |= GCStipple;
		}
		gcval.fill_style = FillSolid;
		mask |= GCFillStyle;
	}
	XChangeGC(dpy, gc, mask, &gcval);
}

static const GCoordinate kRoot2 = 1.414213562373;


//==============================================================
//      IXDevice::UpdateTransferMode
//

void IXDevice::UpdateTransferMode (const IGrafState& state)
{
       IFUNCTRACE_DEVELOP();
       XGCValues gcval;
       unsigned long mask =0;

       const IAttributeState* attState = state.attributeState();

       IColorTransferMode::EColorMode tmixMode = attState->fillTransferMode()->mode();

       switch (tmixMode)
       {
       default:
       case IColorTransferMode::kSourceCopy:
               gcval.function = GXcopy;
               break;

       case IColorTransferMode::kInvertSource:
               gcval.function = GXcopyInverted;
               break;

       case IColorTransferMode::kDestinationCopy:
               gcval.function = GXnoop;
               break;

       case IColorTransferMode::kInvertDestination:
               gcval.function = GXinvert;
               break;

       case IColorTransferMode::kOR:
               gcval.function = GXor;
               break;

       case IColorTransferMode::kAND:
               gcval.function = GXand;
               break;

       case IColorTransferMode::kXOR:
               gcval.function = GXxor;
               break;

       case IColorTransferMode::kInvertedSourceAND:
               gcval.function = GXandInverted;
               break;

       case IColorTransferMode::kInvertedSourceOR:
               gcval.function = GXorInverted;
               break;

       case IColorTransferMode::kInvertedDestinationAND:
               gcval.function = GXandReverse;
               break;

       case IColorTransferMode::kInvertedDestinationOR:
               gcval.function = GXorReverse;
               break;

       case IColorTransferMode::kInvertedAND:
               gcval.function = GXnand;
               break;

       case IColorTransferMode::kInvertedOR:
               gcval.function = GXnor;
               break;

       case IColorTransferMode::kInvertedXOR:
               gcval.function = GXequiv;
               break;

       case IColorTransferMode::kONES:
               gcval.function = GXset;
               break;

       case IColorTransferMode::kZEROS:
           gcval.function = GXclear;
               break;

       }
       mask = GCFunction;

///////////////////////////////////////////////////////////

       XChangeGC(fIXDC->xDisplay, fIXDC->gc(), mask, &gcval);

///////////////////////////////////////////////////////////
}



//==============================================================
//      IXDevice::UpdatePen
//

void IXDevice::UpdatePen (const IGrafState& state)
{
       IFUNCTRACE_DEVELOP();
       XGCValues gcval;
       unsigned long mask;

       // foreground color
       const IPaint* paint = state.attributeState()->framePaint();
       gcval.foreground = lookupColormap( *paint->color() );
       mask= GCForeground ;

       // undo the fill style set in UpdateBrush
       gcval.fill_style = FillSolid;
       mask = mask | GCFillStyle;

       //Line Style

       gcval.line_width = 1;
       const IPen* pen = state.attributeState()->framePen();
       IPen::EPenType penType = pen->penType();
       GCoordinate penWidth;
       if (pen->penType() == IPen::kHairline)
               penWidth = 0;
       else
               penWidth = pen->penWidth();
       gcval.line_width = GCoordinateToInt(penWidth);

       switch (penType) {
       default:
       case IPen::kSolid :
               gcval.line_style = LineSolid;
               break;
       case IPen::kDashed:
               gcval.line_style = LineOnOffDash;
               break;
       case IPen::kDot:
               gcval.line_style = LineOnOffDash;
               break;
       case IPen::kDashDot:
               gcval.line_style = LineOnOffDash;
               break;
       case IPen::kDashDotDot:
               gcval.line_style = LineOnOffDash;
               break;
       case IPen::kInvisible:
               break;
       }

       mask = mask | GCLineWidth |GCLineStyle;

       //Cap Style

       // default Cap
       gcval.cap_style = CapButt;
       const ICap* cap = state.attributeState()->frameEndCap();
       ICap::ECapType endCapType = cap->capType();
       switch (endCapType)
       {
       case ICap::kFlush:
       default:
               gcval.cap_style = CapButt;
               break;
       case ICap::kSquare:
               gcval.cap_style = CapProjecting;
               break;
       case ICap::kRound:
               gcval.cap_style = CapRound;
               break;
       }

       mask = mask | GCCapStyle;

       //Join Style

       // default Joint
       gcval.join_style = JoinMiter;
       const IJoint* joint  = state.attributeState()->frameJoint();
       IJoint::EJointType jointType  = joint->jointType();
       switch (jointType)
       {
       case IJoint::kMiterLimit:
       default:
               gcval.join_style = JoinMiter;
               break;
       case IJoint::kBevel:
               gcval.join_style = JoinBevel;
               break;
       case IJoint::kRound:
               gcval.join_style = JoinRound;
               break;
       }
       mask = mask | GCJoinStyle;

///////////////////////////////////////////////////////////

       XChangeGC(fIXDC->xDisplay, fIXDC->gc(), mask, &gcval);

///////////////////////////////////////////////////////////
       //to set the complicated line dash styles
       const int DotListLength = 2;
       const int DashDotListLength = 4;
       const int DashDotDotListLength = 6;

       const char gcDashed[DotListLength] = {5, 4};
       const char gcDot[DotListLength] = {1, 4};
       const char gcDashDot [DashDotListLength] = {5, 4, 1, 4};
       const char gcDashDotDot [DashDotDotListLength] = {5, 4, 1, 4, 1, 4};
       const char* dot[] = {gcDashed, gcDot, gcDashDot, gcDashDotDot};

       switch (penType) {
       case IPen::kDashed:
               XSetDashes(
                       fIXDC->xDisplay,
                       fIXDC->gc(),
                       0,
                       (const char*)dot[0],
                       DotListLength);

               break;
       case IPen::kDot:
               XSetDashes(
                       fIXDC->xDisplay,
                       fIXDC->gc(),
                       0,
                       (const char*)dot[1],
                       DotListLength);

               break;
       case IPen::kDashDot:
               XSetDashes(
                       fIXDC->xDisplay,
                       fIXDC->gc(),
                       0,
                       (const char*)dot[2],
                       DashDotListLength);
               break;
       case IPen::kDashDotDot:
               XSetDashes(
                       fIXDC->xDisplay,
                       fIXDC->gc(),
                       0,
                       (const char*)dot[3],
                       DashDotDotListLength);
               break;
       default:
               break;
       }

}

//==============================================================
//      IXDevice::UpdateBrush
//


void IXDevice::UpdateBrush(const IGrafState& state)
{
       IFUNCTRACE_DEVELOP();
       XGCValues gcval;
       unsigned long mask;

       const IAttributeState* attState = state.attributeState();
       const IPaint* paint = attState->fillPaint();


       // color
       gcval.foreground = lookupColormap( *paint->color() );
       //gcval.background = 0 ; //would be replaced by colors
       //mask= GCForeground | GCBackground;
       mask= GCForeground ;

       Pixmap tiledPattern;
       unsigned int width, height;

       //fill rules & styles
       if (!paint->imagePattern())   // not image but color paint
       {
               const IMaskPattern* paintPattern = paint->maskPattern();
               EMaskPattern pattern = maskPattern(paintPattern);


               if (pattern != kCustomPattern)
               {
                       gcval.fill_rule = EvenOddRule;
                       if (pattern == kSolidPattern)
                       {
                               gcval.fill_style = FillSolid;
                       }
               }
               else
               {
                       tiledPattern =
                               XCreateBitmapFromData(
                                       fIXDC->xDisplay,
                                       fIXDC->xDrawable,
                                       (char*)maskBits(paintPattern),
                                       8,
                                       8);

                       if (!tiledPattern)
                               GrafDeviceException("Pattern");

                       gcval.fill_rule = EvenOddRule;
                       gcval.fill_style = FillStippled;
                       gcval.stipple = tiledPattern;
                       mask = mask |GCStipple ;
               }

       }
       else
       {
               XImage* tileImage = (XImage*)(unsigned long)(paint->imagePattern()->imageHandle());

        tiledPattern = XCreatePixmap( fIXDC->xDisplay,
            fIXDC->xDrawable,
            tileImage->width, tileImage->height, tileImage->depth);

        XPutImage(
            fIXDC->xDisplay,
            tiledPattern,
            fIXDC->gc(),
            tileImage,
            0, 0, 0, 0, tileImage->width, tileImage->height);

        gcval.fill_rule = EvenOddRule;
        gcval.fill_style = FillTiled;
        gcval.tile = tiledPattern;
        const IGPoint2D *tileOrigin = paint->patternPhase();
        gcval.ts_x_origin = (int)tileOrigin->fX;
        gcval.ts_y_origin = (int)tileOrigin->fY;

        mask = mask | GCTile|GCTileStipXOrigin|GCTileStipYOrigin;

       }

       mask = mask | GCFillRule | GCFillStyle;
///////////////////////////////////////////////////////////

       XChangeGC(fIXDC->xDisplay, fIXDC->gc(), mask, &gcval);

///////////////////////////////////////////////////////////
}




//==============================================================
//      IXDevice::FrameEqualFillPaintModes
//

bool IXDevice::FrameEqualFillPaintModes(const IGrafState& state) const
{
       IFUNCTRACE_DEVELOP();
       const IPaint* fillPaint  = state.attributeState()->fillPaint();
       const IPaint* framePaint = state.attributeState()->framePaint();

       if (*fillPaint != *framePaint || !fillPaint->imagePattern())
               return false;  // different or at least one paint is a pattern paint

       return true;
}

//==============================================================
//      IXDevice::imageTransferMode
//

#define ICLUIHalfToneMode 0X00FC008A

int IXDevice::imageTransferMode(const int render_mode)
{
       IFUNCTRACE_DEVELOP();
       int gc_fn;

       // Warning: serveral GC functions are not sure.
       switch (render_mode)
       {
       case IImageTransferMode::kSrcCopy:
               gc_fn = GXcopy;
               break;
       case IImageTransferMode::kSrcORDst:
               gc_fn = GXor;
               break;
       case IImageTransferMode::kSrcANDDst:
               gc_fn = GXand;
               break;
       case IImageTransferMode::kSrcXORDst:
               gc_fn = GXxor;
               break;
       case IImageTransferMode::kSrcANDInvertDst:
               gc_fn = GXandReverse;
               break;
       case IImageTransferMode::kInvertSrc:
               gc_fn = GXcopyInverted;
               break;
       case IImageTransferMode::kInvertResultOfSrcORDst:
               gc_fn = GXnor;
               break;
       case IImageTransferMode::kSrcANDPattern:
               gc_fn = GXand;          // not sure
               break;
       case IImageTransferMode::kInvertSrcORDst:
               gc_fn = GXorInverted;
               break;
       case IImageTransferMode::kPatternCopy:
               gc_fn = GXcopy;         // not sure
               break;
       case IImageTransferMode::kPatternORInvertSrcORDst:
               gc_fn = GXand;          // not sure
               break;
       case IImageTransferMode::kPatternXORDst:
               gc_fn = GXxor;          // not sure
               break;
       case IImageTransferMode::kInvertDst:
               gc_fn = GXinvert;
               break;
       case IImageTransferMode::kAllZeros:
               gc_fn = GXclear;
               break;
       case IImageTransferMode::kAllOnes:
               gc_fn = GXset;
               break;
       case IImageTransferMode::kOCHalfTone:
               // do it just like in ICLUI, make it a special code - 0x00fc008a
               gc_fn = ICLUIHalfToneMode;
               break;
       }

       return gc_fn;
}


//==============================================================
//      IXDevice::imageSamplingControl
//

unsigned long IXDevice::imageSamplingControl(XImage* source, XImage* dest,
          IGPoint2D srcPoint, int destX, int destY, const int render_mode)
{
    unsigned long oldpixel, newpixel;

    switch (render_mode)
    {

    default :

    case IImageSamplingControl::kColorOnColor:
        oldpixel = XGetPixel(source, NEAREST_INT(srcPoint.fX),
                    NEAREST_INT(srcPoint.fY));
        break;

    case IImageSamplingControl::kBlackOnWhite:
        newpixel = XGetPixel( dest, destX, destY);
        oldpixel = XGetPixel( source, NEAREST_INT(srcPoint.fX),
                    NEAREST_INT(srcPoint.fY));
        oldpixel |= newpixel ;
        break;

    case IImageSamplingControl::kWhiteOnBlack:
        newpixel = XGetPixel( dest, destX, destY);
        oldpixel = XGetPixel( source, NEAREST_INT(srcPoint.fX),
                    NEAREST_INT(srcPoint.fY));
        oldpixel &= newpixel ;
        break;

    case IImageSamplingControl::kHalfTone:
       {

           struct tPoint {
               int tX;
               int tY;
           };

           tPoint pixelPoint[4];
		   unsigned char r, g, b;
			const IR8G8B8A8Color* color=0x0;
			if(fColors)
				fColors->fColors;

               //constructing the bounding pixels

               //top-left
               pixelPoint[0].tX = (int)floor(srcPoint.fX);
               pixelPoint[0].tY = (int)floor(srcPoint.fY);
               //bottom-right
                       pixelPoint[2].tX = (int)ceil(srcPoint.fX);
               pixelPoint[2].tY = (int)ceil(srcPoint.fY);
               //top-right
               pixelPoint[1].tX = pixelPoint[2].tX;
               pixelPoint[1].tY = pixelPoint[0].tY;
               //bottom-left
               pixelPoint[3].tX = pixelPoint[0].tX;
               pixelPoint[3].tY = pixelPoint[2].tY;


               // check if the point falls exactly on the pixel value

               if ( (double)pixelPoint[0].tX == srcPoint.fX )
               {
                   if( (double)pixelPoint[0].tY == srcPoint.fY ) //on pixel
                   {
                       oldpixel = XGetPixel( source,
                                           (int)pixelPoint[0].tX,
                                           (int)pixelPoint[0].tY) ;
                       return oldpixel;
                   }
                   else if ( (double)pixelPoint[3].tY == srcPoint.fY ) //on pixel
                   {
                       oldpixel = XGetPixel( source,
                                           (int)pixelPoint[3].tX,
                                           (int)pixelPoint[3].tY) ;
                       return oldpixel;
                   }
                   else
                   {
                       // its on  left y-edge
                       double redClr[2], greenClr[2], blueClr[2];
                       double ydist = srcPoint.fY - (double)pixelPoint[0].tY;
                       unsigned long pixel[2];

                       pixel[0] = XGetPixel( source,
                                           (int)pixelPoint[0].tX,
                                           (int)pixelPoint[0].tY) ;

                       {
						  if(color){
							  redClr[0]   = (double)color[pixel[0]].fRed;
							  greenClr[0] = (double)color[pixel[0]].fGreen;
							  blueClr[0]  = (double)color[pixel[0]].fBlue;
						  } else {
								IXDisplay::pixelToRGB(pixel[0], &r, &g, &b);
								redClr[0]   = (double)r;
								greenClr[0] = (double)g;
								blueClr[0]  = (double)b;
						  }
                       }

                       pixel[1] = XGetPixel( source,
                                             (int)pixelPoint[3].tX,
                                             (int)pixelPoint[3].tY) ;
                       {
						  if(color){
							  redClr[1]   = (double)color[pixel[1]].fRed;
							  greenClr[1] = (double)color[pixel[1]].fGreen;
							  blueClr[1 ] = (double)color[pixel[1]].fBlue;
						  } else {
								IXDisplay::pixelToRGB(pixel[1], &r, &g, &b);
								redClr[1]   = (double)r;
								greenClr[1] = (double)g;
								blueClr[1]  = (double)b;
						  }
                       }

                       redClr[0] = ( redClr[1] - redClr[0]) * ydist + redClr[0];
                       greenClr[0] = ( greenClr[1] - greenClr[0]) * ydist + greenClr[0];
                       blueClr[0] = ( blueClr[1] - blueClr[0]) * ydist + blueClr[0];

                       oldpixel = ConstructClrPixel(
                                           (unsigned char)redClr[0],
                                           (unsigned char)greenClr[0],
                                           (unsigned char)blueClr[0]);
                       return oldpixel;
                   }
               }
               else if ( (double)pixelPoint[0].tY == srcPoint.fY )
               {
                   if( (double)pixelPoint[1].tX == srcPoint.fX ) //on pixel
                   {
                       oldpixel = XGetPixel( source,
                                           (int)pixelPoint[1].tX,
                                           (int)pixelPoint[1].tY) ;
                       return oldpixel;
                   }
                   else
                   {
                       // its on  top x-edge
                       double redClr[2], greenClr[2], blueClr[2];
                       double xdist = srcPoint.fX - (double)pixelPoint[0].tX;
                       unsigned long pixel[2];

                       pixel[0] = XGetPixel( source,
                                           (int)pixelPoint[0].tX,
                                           (int)pixelPoint[0].tY) ;
                       {
						  if(color){
							  redClr[0]   = (double)color[pixel[0]].fRed;
							  greenClr[0] = (double)color[pixel[0]].fGreen;
							  blueClr[0]  = (double)color[pixel[0]].fBlue;
						  } else {
								IXDisplay::pixelToRGB(pixel[0], &r, &g, &b);
								redClr[0]   = (double)r;
								greenClr[0] = (double)g;
								blueClr[0]  = (double)b;
						  }
                       }

                       pixel[1] = XGetPixel( source,
                                             (int)pixelPoint[1].tX,
                                             (int)pixelPoint[1].tY) ;

                       {
						  if(color){
							  redClr[1]  =  (double)color[pixel[1]].fRed;
							  greenClr[1] = (double)color[pixel[1]].fGreen;
							  blueClr[1 ] = (double)color[pixel[1]].fBlue;
						  } else {
								IXDisplay::pixelToRGB(pixel[1], &r, &g, &b);
								redClr[1]   = (double)r;
								greenClr[1] = (double)g;
								blueClr[1]  = (double)b;
						  }
                       }

                       redClr[0] = ( redClr[1] - redClr[0]) * xdist + redClr[0];
                       greenClr[0] = ( greenClr[1] - greenClr[0]) * xdist + greenClr[0];
                       blueClr[0] = ( blueClr[1] - blueClr[0]) * xdist + blueClr[0];
                       oldpixel = ConstructClrPixel(
                                           (unsigned char)redClr[0],
                                           (unsigned char)greenClr[0],
                                           (unsigned char)blueClr[0]);
                       return oldpixel;
                   }
               }

               // now the point is neither on the pixel or on the edge. Just interpolate

               double xdist =0, ydist=0 ;
               double redClr[4], greenClr[4], blueClr[4];
               double totred =0, totgreen =0, totblue =0;
               unsigned long pixel[4];

               xdist =  srcPoint.fX - pixelPoint[0].tX ;
               ydist =  srcPoint.fY - pixelPoint[0].tY  ;

               for (int i =0; i<4; i++)
               {
                   pixel[i] = XGetPixel( source, pixelPoint[i].tX, pixelPoint[i].tY);

				  if(color){
					   redClr[i]   = (double)color[pixel[i]].fRed;
					   greenClr[i] = (double)color[pixel[i]].fGreen;
					   blueClr[i]  = (double)color[pixel[i]].fBlue;
				  } else {
						IXDisplay::pixelToRGB(pixel[i], &r, &g, &b);
					  redClr[i]   = (double)r;
					  greenClr[i] = (double)g;
					  blueClr[i]  = (double)b;
				  }

               }

               double red[2], green[2], blue[2];

                       red[0] = redClr[1]*xdist + redClr[0] * (1.0 - xdist);
               red[1] = redClr[2]*xdist + redClr[3] * (1.0 - xdist);

               green[0] = greenClr[1]*xdist + greenClr[0] * (1.0 - xdist);
               green[1] = greenClr[2]*xdist + greenClr[3] * (1.0 - xdist);

               blue[0] = blueClr[1]*xdist + blueClr[0] * (1.0 - xdist);
               blue[1] = blueClr[2]*xdist + blueClr[3] * (1.0 - xdist);

               totred = red[1]* ydist + red[0]* (1.0 - ydist);
               totgreen = green[1]* ydist + green[0]* (1.0 - ydist);
               totblue = blue[1]* ydist + blue[0]* (1.0 - ydist);

               oldpixel = ConstructClrPixel(   (unsigned char)totred,
                                               (unsigned char)totgreen,
                                               (unsigned char)totblue );
               }
        break;
    }
    return oldpixel;
}


//==============================================================
//      IXDevice::ConstructClrPixel
//
unsigned long IXDevice::ConstructClrPixel(unsigned char red,
                                          unsigned char green,
                                          unsigned char blue)
{
   if (fColors)
      return fColors->colorIndex( IBaseColor( red, green, blue) );
   else
       return (IXDisplay::rgbToPixel(red, green, blue));
}

//==============================================================
//      IXDevice::SetClipArea
//

int IXDevice::SetClipArea(const IGrafState& state)
{
       IFUNCTRACE_DEVELOP();

       const IGArea* clipArea = state.clipArea();
       if (clipArea->isEmpty())
               return(kClipAreaEmpty);

       if (fClipAreaSet) {
               if (!fStateChange.clipAreaChanged())
                       return kClipAreaRegular;
       }
       else
               fClipAreaSet = true;

       if (clipArea->bounds() == IGRect2D::infiniteRect())
       {
               IGRect2D bigRect = worldBounds();

               IGPoint2D points[4];
               points[0] = bigRect.topLeft();
               points[1] = bigRect.bottomLeft();
               points[2] = bigRect.bottomRight();
               points[3] = bigRect.topRight();

               XPoint gpts[4];

		if  ((fFrameMatrix->isIdentity()) &&
                           (::fLowerLeftCoordOriginHack) ){
                       for (int i = 0; i < 4 ; i++)
                               IGPoint2DToPOINTL(points[i], gpts[i]);
	       }
               else if ( ((fFrameMatrix->isIdentity()) ||
                           (fFrameMatrix->isRectilinear())) &&
                           (!::fLowerLeftCoordOriginHack) ) {
                       for (int i = 0; i < 4 ; i++)
                               IGPoint2DToPOINTL(points[i], gpts[i]);
               }
               else {
                       for (int i = 0; i < 4 ; i++)
                               IGPoint2DToPOINTL(fFrameMatrix->transformPoint(points[i]), gpts[i]);
               }

               Region bigRegion = XPolygonRegion(gpts, 4, EvenOddRule);

               if (!XSetRegion(
                               fIXDC->xDisplay,
                               fIXDC->gc(),
                               bigRegion))
                       GrafDeviceException("XSetRegion");
               XDestroyRegion(bigRegion);

               return kClipAreaInfinite;
       }
       else
       {
               IRegionGrafExtractor regionExtractor(this);
               ICAGRoot root;
               IRegionHandle handle;

               const IGrafMatrix* flip = state.root()->coordinateFlip();
               if (!flip || flip->isIdentity()) {
                       extract(*clipArea, regionExtractor, &root);
               }
               else {
                       IGArea tarea(*clipArea);  // make a copy of it
                       tarea.transformBy(*fFrameMatrix);
                       extract(tarea, regionExtractor, &root);
               }

               if (regionExtractor.region(handle) != NULL) {
                       if (!XSetRegion(
                                       fIXDC->xDisplay,
                                       fIXDC->gc(),
                                       (Region)(void*)handle))
                               GrafDeviceException("XSetRegion");
                       return kClipAreaRegular;
               }
               else
                       return kClipAreaEmpty;

       }
}


//==============================================================
//      IXDevice::SetWinWorldTransform
//

void IXDevice::SetWinWorldTransform( const IGrafMatrix* grafmatrix )
{
       IFUNCTRACE_DEVELOP();
       if (!grafmatrix || grafmatrix->isIdentity())
       {
               if (fMatrixCacheUsed) {
                       if (fMatrixCache.isIdentity())
                               return; // cache hit...
               }
               else
                       fMatrixCacheUsed = true;

               fMatrixCache.setToIdentity();
       }
       else { // not identity
               if (fMatrixCacheUsed) {
                       if (fMatrixCache == *grafmatrix)
                               return; // cache hit...
               }
               else
                       fMatrixCacheUsed = true;

               fMatrixCache = *grafmatrix;
       }
}


/*
__________

transform issue:

       T(model) = T(geometry) . T(view) (short: Tm = Tg.Tv)

       NT:
               For given geometry g,

               G = g.Tg;

               if (Tv ~perspective)
                       extract Tv = affine(Tv)

               render(G) with Tv

       W95:
               For given geometry g,
               G = g.Tm; set penwidth by using Tv
               render(G)
__________
*/
void IXDevice::setupTransforms(const IGrafState& state)
{
       IFUNCTRACE_DEVELOP();
       if (!fStateChange.matrixChanged())
               return;

       fModelMatrix = state.modelMatrix();

       fViewMatrix = 0 /*NIL*/;
       fFrameMatrix = fModelMatrix;
       fPenMatrix = state.viewMatrix();
}


//==============================================================
//      IXDevice::renderLine
//

void IXDevice::renderLine(const IGLine2D& geometry, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty)
          return;
       
       IGLine2D tline(geometry);

       if (!fFrameMatrix->isIdentity())
               tline.transformBy(*fFrameMatrix);

       SetWinWorldTransform( fViewMatrix);

       IGPoint2D p0, p1;
       tline.points(p0, p1);
       GCoordinate dX= p1.fX - p0.fX;
       GCoordinate dY= p1.fY - p0.fY;
       int deltaX = (int)floor(dX);
       int deltaY = (int)floor(dY);

       if ( deltaX == 0 && deltaY == 0) { // if line is actually an empty line
             return ;
       }
    
//
// AIX calls XDrawLine, which draws a line between two points,
// including start and end points.
// Windows calls LineTo which draws a line between two points,
// including the start point, but not including the end point.
// Only checking horizontal, vertical, and diagonal (slope == 1 or -1) lines for now
//

// Augment Endpoint pixel since trying to emulate Windows API LineTo and Polyline
//
       if (deltaY == 0) {  // horizontal line
          if (deltaX < 0)  // line goes from right to left (-tive x-dir)
              p1.fX++;
          else             // line goes from left to right (+tive x-dir)
              p1.fX--; 
       } else if (deltaX == 0 ) { // vertical line
          if (deltaY < 0)  // line goes from bottom to top (-tive y-dir)
              p1.fY++;
          else             // line goes from top to bottom (+tive y-dir)
              p1.fY--; 
       } else if (deltaX == deltaY) { // diagonal line (equal slopes)
          if (deltaY < 0){ // line goes from bottom right to top left
              p1.fX++;
              p1.fY++;
          }
          else {           // line goes from top left to bottom right
              p1.fX--;
              p1.fY--;
          }
       } else if (deltaX == -deltaY) { // diagonal line (equal -tive slopes)
          if (deltaX < 0){ // line goes from top right to bottom left
              p1.fX++;
              p1.fY--;
          }
          else {           // line goes from bottom left to top right
              p1.fX--;
              p1.fY++;
          }
       } // no augmentation done
       
       UpdateTransferMode(grafState);
       UpdatePen(grafState);

       XDrawLine(
               fIXDC->xDisplay,
               fIXDC->xDrawable,
               fIXDC->gc(),
               GCoordinateToInt(p0.fX),
               GCoordinateToInt(p0.fY),
               GCoordinateToInt(p1.fX),
               GCoordinateToInt(p1.fY));
}


//==============================================================
//      IXDevice::renderPolyline
//

void IXDevice::renderPolyline(const IGPolyline2D& geometry, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty)
               return;

       IGPolyline2D tpolyline(geometry);
       if (!fFrameMatrix->isIdentity())
               tpolyline.transformBy(*fFrameMatrix);

       // set up gdi points for polyline.
       long numPoints = tpolyline.numberOfPoints();
       IDeleterForArrayOf<XPoint> gpts = new XPoint[numPoints];


       // for pixel perfection by XL 10/15/97
       IGPoint2D pixel1(tpolyline.point(numPoints-2));
       IGPoint2D pixel2(tpolyline.point(numPoints-1)); // endpoint of polyline

       GCoordinate dX = pixel2.fX - pixel1.fX;
       GCoordinate dY = pixel2.fY - pixel1.fY;
       int deltaX = (int)floor(dX);
       int deltaY = (int)floor(dY);

//
// AIX calls XDrawLines, which draws a line between two points,
// including start and end points.
// Windows calls Polyline which draws a line between points,
// including the start point, but not including the end point.
// Only checking horizontal, vertical, and diagonal (slope == 1 or -1) lines for now
//

// Augment Endpoint pixel since trying to emulate Windows API LineTo and Polyline
//
       if (deltaY == 0) { // horizontal line
          if (deltaX < 0)  // line goes from right to left (-tive x-dir)
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX+1.0, pixel2.fY));
          else             // line goes from left to right (+tive x-dir)
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX-1.0, pixel2.fY));
       } else if (deltaX == 0 ) { // vertical line
          if (deltaY < 0)  // line goes from bottom to top (-tive y-dir)
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX, pixel2.fY+1.0));
          else             // line goes from top to bottom (+tive y-dir)
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX, pixel2.fY-1.0));
       } else if (deltaX == deltaY) { // diagonal line (equal slopes)
          if (deltaY < 0)  // line goes from bottom right to top left
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX+1.0, pixel2.fY+1.0));
          else             // line goes from top left to bottom right
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX-1.0, pixel2.fY-1.0));
       } else if (deltaX == -deltaY) { // diagonal line (equal -tive slopes)
          if (deltaX < 0)  // line goes from top right to bottom left
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX+1.0, pixel2.fY-1.0));
          else             // line goes from bottom left to top right
             tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX-1.0, pixel2.fY+1.0));
       }

       for (unsigned long i=0; i < numPoints; i++)
               IGPoint2DToPOINTL(tpolyline.point(i), gpts[i]);

       UpdateTransferMode(grafState);
       UpdatePen(grafState);

       XDrawLines(
               fIXDC->xDisplay,
               fIXDC->xDrawable,
               fIXDC->gc(),
               gpts,
               numPoints,
               CoordModeOrigin);
}

//==============================================================
//      IXDevice::renderRect
//

void IXDevice::renderRect(const IGRect2D& geometry, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty) return;

       IGRect2D trect(geometry);

       fDrawOperation = grafState.attributeState()->drawingOperation();

       if (!fFrameMatrix->isIdentity())
       {
               if (fFrameMatrix->isRectilinear())
               {
                       trect = fFrameMatrix->transformBounds(trect);
               }
               else
               {
                       IGPolygon2D polygon(4);
                       polygon.setPoint(0, trect.topLeft());
                       polygon.setPoint(1, trect.bottomLeft());
                       polygon.setPoint(2, trect.bottomRight());
                       polygon.setPoint(3, trect.topRight());

                       renderPolygon(polygon, grafState);
                       return;
               }
       }
       else  // just in case the rect is transposed
       {
               trect.orderPoints();
       }

       if (trect.contains(worldBounds()))      // for infinite rect
               trect = worldBounds();


       unsigned int width = GCoordinateToInt(trect.fRight) - GCoordinateToInt(trect.fLeft);
       unsigned int height = GCoordinateToInt(trect.fBottom) - GCoordinateToInt(trect.fTop);

       // draw rectangle

       UpdateTransferMode(grafState);

       // it is important to do fill before frame
       if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
       {
               UpdateBrush(grafState);
               XFillRectangle(
                       fIXDC->xDisplay,
                       fIXDC->xDrawable,
                       fIXDC->gc(),
                       GCoordinateToInt(trect.fLeft),
                       GCoordinateToInt(trect.fTop),
                       width,
                       height);
				destroyBrushPixmap(fIXDC->xDisplay, fIXDC->gc());
       }

       if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
       {
               UpdatePen(grafState);
               XDrawRectangle(
                       fIXDC->xDisplay,
                       fIXDC->xDrawable,
                       fIXDC->gc(),
                       GCoordinateToInt(trect.fLeft),
                       GCoordinateToInt(trect.fTop),
                       width,
                       height);
       }
}

//==============================================================
//      IXDevice::renderEllipse
//

void IXDevice::renderEllipse(const IGEllipse2D& geometry, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty)
               return;

       fDrawOperation = grafState.attributeState()->drawingOperation();
       IGRect2D trect(geometry.bounds());

       if (!fFrameMatrix->isIdentity())
       {
               if (fFrameMatrix->isRectilinear())
               {
                       trect = fFrameMatrix->transformBounds(trect);
               }
               else
               {
                       IGLoop2D elliptical(geometry);
                       renderLoop(elliptical, grafState);
                       return;
               }
       }


       trect.orderPoints(); //for an infinite rect

       XPoint gp1;
       XPoint gp2;
       IGPoint2DToPOINTL(trect.topLeft(), gp1);
       IGPoint2DToPOINTL(trect.bottomRight(), gp2);

       unsigned int width = GCoordinateToInt(trect.fRight) - GCoordinateToInt(trect.fLeft);
       unsigned int height = GCoordinateToInt(trect.fBottom) - GCoordinateToInt(trect.fTop);
       int start_angle =0;
       int path_angle = 360*64;

       //Draw Ellipse

       UpdateTransferMode(grafState);

       // it is important to do fill before frame
       if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
       {
               UpdateBrush(grafState);
               XFillArc(
                       fIXDC->xDisplay,
                       fIXDC->xDrawable,
                       fIXDC->gc(),
                       GCoordinateToInt(trect.fLeft),
                       GCoordinateToInt(trect.fTop),
                       width,
                       height,
                       start_angle,
                       path_angle);
				destroyBrushPixmap(fIXDC->xDisplay, fIXDC->gc());
       }

       if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
       {
               UpdatePen(grafState);
               XDrawArc(
                       fIXDC->xDisplay,
                       fIXDC->xDrawable,
                       fIXDC->gc(),
                       GCoordinateToInt(trect.fLeft),
                       GCoordinateToInt(trect.fTop),
                       width,
                       height,
                       start_angle,
                       path_angle);
       }
}

//==============================================================
//      IXDevice::renderPolygon
//

void IXDevice::renderPolygon(const IGPolygon2D& geometry, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty) return;

       IGPolygon2D tpolygon(geometry);
       if (!fFrameMatrix->isIdentity())
               tpolygon.transformBy(*fFrameMatrix);

       // set up gdi points for polygon.
       unsigned long numPoints = tpolygon.numberOfPoints();
       IDeleterForArrayOf<XPoint> gpts = new XPoint[numPoints + 1];

       for (unsigned long i=0; i < numPoints; i++)
               IGPoint2DToPOINTL(tpolygon.point(i), gpts[i]);
       IGPoint2DToPOINTL(tpolygon.point(0), gpts[numPoints]);

       fDrawOperation = grafState.attributeState()->drawingOperation();

       // draw polygon
       UpdateTransferMode(grafState);

       // it is important to do fill before frame
       if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
       {
               UpdateBrush(grafState);
               if (tpolygon.eOFill())
                       XSetFillRule(
                               fIXDC->xDisplay,
                               fIXDC->gc(),
                               EvenOddRule);
               else
                       XSetFillRule(
                               fIXDC->xDisplay,
                               fIXDC->gc(),
                               WindingRule);

               XFillPolygon(
                       fIXDC->xDisplay,
                       fIXDC->xDrawable,
                       fIXDC->gc(),
                       gpts,
                       numPoints,
                       Complex,
                       CoordModeOrigin);
				destroyBrushPixmap(fIXDC->xDisplay, fIXDC->gc());
       }

       if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
       {
               UpdatePen(grafState);
               XDrawLines(
                       fIXDC->xDisplay,
                       fIXDC->xDrawable,
                       fIXDC->gc(),
                       gpts,
                       numPoints+1,
                       CoordModeOrigin);
       }
}

//==============================================================
//      IXDevice::renderArea
//
#include <igarea2d.hpp>
#include <vrtxeng.hpp>
#include <sampextr.hpp>
#include <cag.hpp>


void IXDevice::renderArea(const IGArea& area, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       if (area.isEmpty())
               return;
       IGArea aCopy(area);

       IPolygonVertexEngine polygoner;
       IOutlineMakerVertexEngine outliner((IVertexEngine*)&polygoner);
       ISamplingExtractor sampler(&outliner, IGRect2D::infiniteRect());
       ICAGRoot root;
       extract(aCopy, sampler, (ICAGNode*)&root);
       sampler.render(root);
       IGLoop2D &tLoop = polygoner.loop();

       if (tLoop.isEmpty())
               return;

       renderLoop(tLoop, grafState);
}

#include <igcurv2d.hpp>
#include <inumeric.hpp>
#include <iknots.hpp>
#include "prspline.hpp"

#ifndef NO_CONIC_ACCELERATION_SUPPORT
#include "conics.hpp"

#pragma pack(push,4)

class ConicMaker {
private:
       int fX, fY, fWidth, fHeight;
       int fStartX, fStartY, fEndX, fEndY;
       int fStartAngle, fPathAngle;
       IGPoint2D fStartPoint, fEndPoint, fCentrePoint;
       IConicInfo::ECurveType fType;
       bool fIsLoop;
       bool fIsArc;

       void SetupForRendering(const IGCurve2D& curve) {
           IFUNCTRACE_DEVELOP();
           IConicInfo* ci = curve.fConicInfo;
           fType = ci->fType;
           fCentrePoint= ci->fArcBounds.center();
           
           fX = (int)(nearest( ci->fArcBounds.fLeft));
           fY = (int)(nearest( ci->fArcBounds.fTop));
           fWidth = (int)(nearest(ci->fArcBounds.fRight - ci->fArcBounds.fLeft));
           fHeight = (int)(nearest(ci->fArcBounds.fBottom - ci->fArcBounds.fTop));
           
           fIsArc = (ci->fType == IConicInfo::kArc);
           IGPoint2D startPoint, endPoint;
           if (fIsArc)
           {
               fStartPoint = curve.evaluate(curve.minParameter());
               fEndPoint = curve.evaluate(curve.maxParameter());
           }
           else
           {
               fStartPoint = ci->fStart;
               fEndPoint = ci->fEnd;
           }
           
           //calculating points to angle for circular/elliptical shaped objects
           double startAngle = (atan2((fStartPoint.fY -fCentrePoint.fY)*fWidth/fHeight, 
                                      fStartPoint.fX -fCentrePoint.fX)) *kRadiansToDegrees;
           double endAngle = (atan2((fEndPoint.fY -fCentrePoint.fY)*fWidth/fHeight, 
                                    fEndPoint.fX -fCentrePoint.fX)) * kRadiansToDegrees;
           endAngle= startAngle - endAngle;
           startAngle = -startAngle;
           
           if (startAngle < 0.)
		       startAngle += 360.;
           
           if (endAngle > 0. && ci->fDirection)
		       endAngle = endAngle - 360.;
           else if (endAngle < 0. && !ci->fDirection)
		       endAngle = endAngle + 360.;
           
           fStartAngle = (int)(nearest(startAngle)) * 64;
           fPathAngle = (int)(nearest(endAngle)) * 64;
       }

public:
       ConicMaker(const IGCurve2D& curve) {
               SetupForRendering(curve);
               fIsLoop = false;
       }
       ConicMaker(const IGLoop2D& loop) {
               SetupForRendering(loop);
               fIsLoop = true;
       }

       void render( IXDevice* xDevice, const IGrafState &grafState ) {

           IFUNCTRACE_DEVELOP();
           
           if (fType == IConicInfo::kEllipse) {
               xDevice->UpdateTransferMode(grafState);
               xDevice->UpdatePen (grafState);
               if (!XDrawArc(
                   xDevice->fIXDC->xDisplay,
                   xDevice->fIXDC->xDrawable,
                   xDevice->fIXDC->gc(),
                   fX,
                   fY,
                   fWidth,
                   fHeight,
                   0,
                   360*64))
                   GrafDeviceException("Ellipse");
               return;
           }
           else if (fIsLoop)
           {
               xDevice->UpdateTransferMode(grafState);
               xDevice->UpdateBrush (grafState);
               if (fType == IConicInfo::kPie)
               {       // pie:
                   XSetArcMode(xDevice->fIXDC->xDisplay, xDevice->fIXDC->gc(), ArcPieSlice);
               }
               else {  // chord:
                   XSetArcMode(xDevice->fIXDC->xDisplay,xDevice->fIXDC->gc(), ArcChord);
               }
               if (!XFillArc(
                   xDevice->fIXDC->xDisplay,
                   xDevice->fIXDC->xDrawable,
                   xDevice->fIXDC->gc(),
                   fX,
                   fY,
                   fWidth,
                   fHeight,
                   fStartAngle,
                   fPathAngle))
                   GrafDeviceException("Pie-Chord");
               destroyBrushPixmap(xDevice->fIXDC->xDisplay, 
                                  xDevice->fIXDC->gc());
               
               xDevice->UpdatePen(grafState);
               if (!XDrawArc(
                   xDevice->fIXDC->xDisplay,
                   xDevice->fIXDC->xDrawable,
                   xDevice->fIXDC->gc(),
                   fX,
                   fY,
                   fWidth,
                   fHeight,
                   fStartAngle,
                   fPathAngle))
                   GrafDeviceException("Pie-Chord");
               if (fType == IConicInfo::kPie) {
                   if (!XDrawLine(
                       xDevice->fIXDC->xDisplay,
                       xDevice->fIXDC->xDrawable,
                       xDevice->fIXDC->gc(),
                       xDevice->GCoordinateToInt(fStartPoint.fX),
                       xDevice->GCoordinateToInt(fStartPoint.fY),
                       xDevice->GCoordinateToInt(fCentrePoint.fX),
                       xDevice->GCoordinateToInt(fCentrePoint.fY)) )
                       GrafDeviceException("XDrawLine");
                   if (!XDrawLine(
                       xDevice->fIXDC->xDisplay,
                       xDevice->fIXDC->xDrawable,
                       xDevice->fIXDC->gc(),
                       xDevice->GCoordinateToInt(fEndPoint.fX),
                       xDevice->GCoordinateToInt(fEndPoint.fY),
                       xDevice->GCoordinateToInt(fCentrePoint.fX),
                       xDevice->GCoordinateToInt(fCentrePoint.fY)) )
                       GrafDeviceException("XDrawLine");
               } else {
                   if (!XDrawLine(
                       xDevice->fIXDC->xDisplay,
                       xDevice->fIXDC->xDrawable,
                       xDevice->fIXDC->gc(),
                       xDevice->GCoordinateToInt(fStartPoint.fX),
                       xDevice->GCoordinateToInt(fStartPoint.fY),
                       xDevice->GCoordinateToInt(fEndPoint.fX),
                       xDevice->GCoordinateToInt(fEndPoint.fY)) )
                       GrafDeviceException("XDrawLine");
               }
           }
           else  //  arc through 3 points...
           {
               xDevice->UpdateTransferMode(grafState);
               xDevice->UpdatePen (grafState);
               if (!XDrawArc(
                   xDevice->fIXDC->xDisplay,
                   xDevice->fIXDC->xDrawable,
                   xDevice->fIXDC->gc(),
                   fX,
                   fY,
                   fWidth,
                   fHeight,
                   fStartAngle,
                   fPathAngle))
                   GrafDeviceException("Arc");
           }
       }
};

#pragma pack(pop)

#endif // NO_CONIC_ACCELERATION_SUPPORT

//Discretizer classes

#include <nurbs.hpp>
#include <xpoint.hpp>

#pragma pack(push,4)

class LoopDiscretizer : public INurbDiscretiser
{
    IXDevice* fXDevice;
    IGrafState* fState;
    bool fDrawing;
    IRawArray<IXPoint> fLoopPts;
    IRawArray<IXPoint> fFillLoopPts;
    IRawArray<IXPoint> fFrameLoopPts;
    bool fInit;
    int fFillAndFrame;
    bool feoFill;
    int fNumPts;
    
public:
    LoopDiscretizer(int fillAndFrame, int eoFill, IXDevice* xDev , IGRect2D clipBounds) :
            INurbDiscretiser(clipBounds, 0.5)
        {
            fXDevice = xDev;
            fFillAndFrame = fillAndFrame;
            feoFill = eoFill;
            
        }
    
    LoopDiscretizer(
        int fillAndFrame,
        bool eoFill,
        IXDevice* xDev,
        IGRect2D clipBounds,
        double resolution) :
            
            INurbDiscretiser(clipBounds, resolution)
        {
            fFillAndFrame = fillAndFrame;
            fXDevice = xDev;
            fInit = false;
            feoFill = eoFill;
        }
    const IGrafState* state()
        {
            return fState;
        }
    
    void begin() {
        if (fInit) {
            fLoopPts.resize(0);
            fFrameLoopPts.resize(0);
            fFillLoopPts.resize(0);
        } else {
            fLoopPts.setGrowIncrement(16);
            fFrameLoopPts.setGrowIncrement(16);
            fFillLoopPts.setGrowIncrement(16);
            fInit = true;
        }
    }
    
    void end() {

        IFUNCTRACE_DEVELOP();
        
        int lengthOfFillLoop = fFillLoopPts.numberOfValues();
        int lengthOfFrameLoop = fFrameLoopPts.numberOfValues();
        XGCValues gcval, oldval;
        
        if (IGrafPort::restoreGC) {
            XGetGCValues(
                fXDevice->fIXDC->xDisplay,
                fXDevice->fIXDC->gc(),
                GCJoinStyle, &oldval);
        }

        gcval.join_style = JoinRound;

        XChangeGC(
            fXDevice->fIXDC->xDisplay,
            fXDevice->fIXDC->gc(),
            GCJoinStyle, &gcval);
        
        if ( lengthOfFillLoop > 2 && (fFillAndFrame == 1 || fFillAndFrame == 2) )
        {
            fXDevice->UpdateBrush(*(this->state()));
            if (feoFill)
                XSetFillRule(
                    fXDevice->fIXDC->xDisplay,
                    fXDevice->fIXDC->gc(),
                    EvenOddRule);
            else
                XSetFillRule(
                    fXDevice->fIXDC->xDisplay,
                    fXDevice->fIXDC->gc(),
                    WindingRule);
            
            IXPoint* ixPt = &fFillLoopPts[0];
            XPoint* xPts = &((*ixPt).point());
            XFillPolygon(
                fXDevice->fIXDC->xDisplay,
                fXDevice->fIXDC->xDrawable,
                fXDevice->fIXDC->gc(),
                xPts,
                lengthOfFillLoop,
                Complex,
                CoordModeOrigin);
			destroyBrushPixmap(fXDevice->fIXDC->xDisplay, fXDevice->fIXDC->gc());
        }
        if ( feoFill && (lengthOfFrameLoop > 2) && (fFillAndFrame == 0 || fFillAndFrame ==
                                                    2) )
        {

            fXDevice->UpdatePen(*(this->state()));
            unsigned long start = 0, end;

            IXPoint* ixPt = &fFrameLoopPts[start];
            XPoint* xPts = &((*ixPt).point());

            for (unsigned long i=0; i< lengthOfFrameLoop; i++)
            {

                if (fFrameLoopPts[start] == fFrameLoopPts[i] && (start != i))
                {
                    end = i;
                    if (!XDrawLines(
                            fXDevice->fIXDC->xDisplay,
                            fXDevice->fIXDC->xDrawable,
                            fXDevice->fIXDC->gc(),
                            &xPts[start],
                            (end - start+1),
                            CoordModeOrigin))
                        GrafDeviceException("Polyline");
                    start = end +1;
                }
            }
        }
        if (IGrafPort::restoreGC) {
            XChangeGC(
                fXDevice->fIXDC->xDisplay,
                fXDevice->fIXDC->gc(),
                GCJoinStyle, &oldval);
        }
    }

       void endPath() {

           IFUNCTRACE_DEVELOP();

           int length = fLoopPts.numberOfValues();
           int lengthOfFillLoop = fFillLoopPts.numberOfValues();
           
           if (length > 2)
           {
               if ( !feoFill && (fFillAndFrame != 1) )
               {
                fXDevice->UpdateTransferMode(*(this->state()));
                fXDevice->UpdatePen(*(this->state()));

                XGCValues gcval, oldval;

                if (IGrafPort::restoreGC) {
                    XGetGCValues(
                        fXDevice->fIXDC->xDisplay,
                        fXDevice->fIXDC->gc(),
                        GCJoinStyle, &oldval);
                }

                gcval.join_style = JoinRound;
                XChangeGC(fXDevice->fIXDC->xDisplay,
                          fXDevice->fIXDC->gc(),
                          GCJoinStyle, &gcval);

                IXPoint* ixPt = &fLoopPts[0];
                XPoint* xPts = &((*ixPt).point());
                XDrawLines(
                    fXDevice->fIXDC->xDisplay,
                    fXDevice->fIXDC->xDrawable,
                    fXDevice->fIXDC->gc(),
                    xPts,
                    length,
                    CoordModeOrigin);

                fLoopPts.resize(0);

                if (IGrafPort::restoreGC) {
                    XChangeGC(fXDevice->fIXDC->xDisplay,
                              fXDevice->fIXDC->gc(),
                              GCJoinStyle, &oldval);
                }

            }
            if (fFillAndFrame != 0 )
            {
                IXPoint aPt = fFillLoopPts.value(0);
                if ( aPt != fFillLoopPts.value(lengthOfFillLoop -1) )
                    fFillLoopPts.append(aPt);
            }
        }
       }


       void addPoint(IGPoint2D point)
       {
           IFUNCTRACE_DEVELOP();
           XPoint pt ;
           fXDevice->IGPoint2DToPOINTL(point, pt);
           IXPoint xpt(pt);
           fLoopPts.append( xpt );
           fFrameLoopPts.append( xpt );
           fFillLoopPts.append( xpt );
       }

       void DiscretizeLoop(
               unsigned long order,
               const GParametric* knotsVector,
               IGRPoint2DArray controlPoints, const IGrafState& state )
       {
           IFUNCTRACE_DEVELOP();
           fState = (IGrafState*)(&state);
           
           if (order > 2)
               DiscretizeNurb(order, knotsVector, controlPoints, true);
           else
               DiscretizeOrderTwo(knotsVector, controlPoints);
           
           end();
       }
};

#pragma pack(pop)

#include <ibcurve.hpp>

//================================================================
//     IXDevice::renderCurve
//
// Parameters:
//   geometry : the curve to be rendered
//   grafState: indicates the attributes/transformation to be applied
//
// Return value: NONE
//
// Purpose: renders the curve geometry to the display
//
//

void IXDevice::renderCurve(const IGCurve2D& geometry, const IGrafState& grafState)
{
       IFUNCTRACE_DEVELOP();
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty) return;
       IGCurve2D tcurve(geometry);

       if (!fFrameMatrix->isIdentity()) {
               tcurve.transformBy(*fFrameMatrix);
#ifndef NO_CONIC_ACCELERATION_SUPPORT
               if (conicInfo(tcurve)) {
                       if (!conicInfo(tcurve)->transformBy(*fFrameMatrix)) {
                               deleteConicInfo(tcurve);
                       }
               }
#endif // NO_CONIC_ACCELERATION_SUPPORT
       }

#ifndef NO_CONIC_ACCELERATION_SUPPORT
       if (conicInfo(tcurve))
       { // if curve is supported by windows and is an arc...
               ConicMaker(tcurve).render(this, grafState);
       }
       else
#endif // NO_CONIC_ACCELERATION_SUPPORT
       {
               unsigned long numPoints;
               unsigned long order = tcurve.order();

               if (order > 2) {  // make everything order == 4 beziers...

                       if (!tcurve.isBezier())
                               tcurve.refineToBeziers();

                       IGRPoint2DArray pts;
                       tcurve.points(pts);

                       IRawArray<GParametric> knotPts;
                       tcurve.knots(knotPts);


                       LoopDiscretizer discrete(0, 0, this, worldBounds());
            discrete.DiscretizeLoop(tcurve.order(), &knotPts[0], pts, grafState);

               }
               else {  // is a polyline

                       numPoints = tcurve.numberOfPoints();
                       IDeleterForArrayOf<XPoint> gpts = new XPoint[numPoints];
                       unsigned long start = 0;
                       unsigned long end;

                       for (unsigned long i = 0; i < numPoints; i++) {

                               IGPoint2DToPOINTL(tcurve.point(i).dropW(), gpts[i]);

                               if (tcurve.knot(i + 1) == tcurve.knot(i + 2)) {
                                       end = i + 1;

                                       UpdateTransferMode ( grafState );
                                       UpdatePen (grafState);

                                       if (!XDrawLines(
                                                       fIXDC->xDisplay,
                                                       fIXDC->xDrawable,
                                                       fIXDC->gc(),
                                                   (XPoint*)(gpts + start),
                                                   (end - start),
                                                   CoordModeOrigin))
                                               GrafDeviceException("Polyline");

                                       start = end;
                               }
                       }
               }
       }
}


#include <igloop2d.hpp>

//================================================================
//     IXDevice::renderLoop
//
// Parameters:
//   geometry : the loop to be rendered
//   grafState: indicates the attributes/transformation to be applied
//
// Return value: NONE
//
// Purpose: renders the loop geometry to the display
//
//

void IXDevice::renderLoop(const IGLoop2D& geometry, const IGrafState& grafState)
{
       fStateChange = grafState.registeredAsTheStateInUse();

       setupTransforms(grafState);

       if (SetClipArea(grafState) == kClipAreaEmpty)
       return;

       IGLoop2D tloop(geometry);
       int xFillAndFrameMode = 0; //frame = 0, fill = 1, fillandframe = 2

       if (!fFrameMatrix->isIdentity())
       {
               tloop.transformBy(*fFrameMatrix);
#ifndef NO_CONIC_ACCELERATION_SUPPORT
               if (conicInfo(tloop))
               {
                       if (!conicInfo(tloop)->transformBy(*fFrameMatrix))
                       {
                               deleteConicInfo(tloop);
                       }
               }
#endif // NO_CONIC_ACCELERATION_SUPPORT
       }

       SetWinWorldTransform(fViewMatrix);

       unsigned long order = tloop.order();
       fDrawOperation = grafState.attributeState()->drawingOperation();

       switch (fDrawOperation)
       {
       case (IAttributeState::kFillAndFrame) :
               xFillAndFrameMode = 2;
               break;
       case (IAttributeState::kFill) :
               xFillAndFrameMode = 1;
               break;
       default:
       case (IAttributeState::kFrame) :
               xFillAndFrameMode = 0;
               break;
       }


#ifndef NO_CONIC_ACCELERATION_SUPPORT
       if (conicInfo(tloop))
       {  // if loop is supported by windows...
               ConicMaker(tloop).render(this, grafState);
               return;
       }
       else
#endif //NO_CONIC_ACCELERATION_SUPPORT
       {
               IGRPoint2DArray pts;
               IRawArray<GParametric> knotPts;

               tloop.closeLoop();
               tloop.points(pts);
               tloop.knots(knotPts);

               if (!tloop.isBezier())
                       tloop.refineToBeziers();

               LoopDiscretizer discrete(xFillAndFrameMode, tloop.eOFill(), this, worldBounds());
               discrete.DiscretizeLoop(tloop.order(), &knotPts[0], pts, grafState);
       }
}



void IXDevice::imageTransferControl(unsigned long& pixelVal,
                                    const int render_mode)
{
	unsigned char redClr, greenClr, blueClr;

	if(fColors){ //has colormap support
		const IR8G8B8A8Color& color ( (*fColors)[pixelVal] );
		redClr = color.fRed;
		greenClr = color.fGreen;
		blueClr = color.fBlue;
	} else {
       IXDisplay::pixelToRGB(pixelVal, &redClr, &greenClr, &blueClr);
	}

	redClr += (255 - redClr)>>1;
	greenClr += (255 - greenClr)>>1;
	blueClr += (255 - blueClr)>>1;

	pixelVal = ConstructClrPixel( redClr, greenClr, blueClr);
}


//================================================================
//     modify the XImage
// We created the image with Root window as the destination window
// but it the destination window is now known. So, we have to modify
// the data bcause if Root window is 24bit and destination window is
// 8 bit we got to modify the data by matching the color.
// The cases possible are
// 8 -> 8              Do nothing
// 8 -> 24             Not possible as the Rootwindow is the superset for child
// 24 -> 8             Yes, this is what we got to handle
// 24 -> 24            Do nothing

XImage* IXDevice::convert24To8Image(Display *dpy, Window dstWin, const IGImage &org)
{
       IFUNCTRACE_DEVELOP();
       XImage *xi = (XImage*)(unsigned long)org.imageHandle();

       if (xi->depth != 24)
           return NULL;

       XWindowAttributes wattr;
       int i, j;
       memset(&wattr, 0x0, sizeof(XWindowAttributes));

       if (XGetGeometry(dpy, dstWin, &wattr.root,
                        &wattr.x, &wattr.y, 
                        (unsigned int*)(&wattr.width),
                        (unsigned int*)(&wattr.height), 
                        (unsigned int*)(&wattr.border_width), 
                        (unsigned int*)(&wattr.depth))) {
           wattr.colormap = IColorMap::defaultColorMap().nativeColorMap();
       } else {
           if (! XGetWindowAttributes(dpy, dstWin, &wattr))
               return NULL; //real problem
       }

       if (wattr.depth != 8)
           return NULL;

       IColorMap* defColorMap = &(IColorMap::defaultColorMap());

       XColor  clr_tmp;
       RGBAClr rgba;

       char *imgData = new char[xi->width * xi->height];

       unsigned long scanstride, curColor=0,prevColor=0, prevIndex=0;
       //init for the black color
       clr_tmp.red = clr_tmp.green = clr_tmp.blue = 0;
       prevIndex = defColorMap->colorIndex( IBaseColor(clr_tmp.red,
                                                       clr_tmp.green,
                                                       clr_tmp.blue) );

       for(i=0; i < xi->height; i++){
               scanstride = i*xi->width;
               for(j=0; j < xi->width; j++){
                       clr_tmp.pixel = XGetPixel(xi, j, i);
                       rgba = *((RGBAClr *)&clr_tmp.pixel);
                       clr_tmp.red = rgba.R;
                       clr_tmp.green = rgba.G;
                       clr_tmp.blue = rgba.B;
                       curColor = (clr_tmp.red <<16) |
                                  (clr_tmp.green <<8) |
                                  (clr_tmp.blue);

                       if(prevColor == curColor){
                               imgData[scanstride + j] = prevIndex;
                       } else {
                               imgData[scanstride + j] = prevIndex =
                                  defColorMap->colorIndex(
                                        IBaseColor(clr_tmp.red,
                                                   clr_tmp.green,
                                                   clr_tmp.blue) );
                               prevColor = curColor;
                       }
               }
       }

       return( XCreateImage(dpy, wattr.visual, wattr.depth, ZPixmap, 0,
                               imgData, xi->width, xi->height,8, xi->width) );

}


unsigned long IXDevice::transparencyIndex(Display *dpy, Window dstWin, const IGImage &org)
{
       IBaseColor transparencyClr = org.transparencyColor();


       if (IColorMap::hasColorMapSupport()){
               return(IColorMap::defaultColorMap().colorIndex(transparencyClr));
       } else {
               return(IXDisplay::rgbToPixel(transparencyClr.redMix(),
                               transparencyClr.greenMix(),
                               transparencyClr.blueMix()));
       }
}

//================================================================
//     renderImage
//
//     image : the pixels.
//     source : a rectangle in pixel space.
//     grafState : has the model matrix and attributes.
//
//     render the pixels contained in the source rectangle to
//     the device. Transform the position of the pixels using the model matrix.
//     Use the transfer mode and image filtering mode defined in the graf state.
//
//     The grafState contains the matrix that maps the image pixel space
//     to World Space. This matrix is set as the GDI WorldTransform to we can use
//     BitBlt with no transformation. If the World Transform is not supported, then
//     we must extract the components of the transformation from grafState, and
//     use them to set the parameters of StretchBlt.
//

//Due to some problem in XDisplay the display Ptr is not valid, for this
//reason open a new connection and close it when finished


void IXDevice::renderImage(
       const IGImage& image,
       const IGRect2D& source,
       const IGrafState& grafState)
{
    IFUNCTRACE_ALL();
    fStateChange = grafState.registeredAsTheStateInUse();

    setupTransforms(grafState);

    if (SetClipArea(grafState) == kClipAreaEmpty)
            return;

    IGRect2D trect = source;
    IGRect2D irect = image.rasterBounds();

    const IGrafMatrix* tMatrix = grafState.modelMatrix();
    IGrafMatrix fMatrixUsed;

    trect.orderPoints();

    // to avoid images to go upside down on flip coordinates
    //fMatrixUsed is being used.
    const IGrafMatrix* fFlip = grafState.root()->coordinateFlip();
    if (fFlip)
    {
        fMatrixUsed = *fFlip;
        fMatrixUsed.invert();
        fMatrixUsed.concatWith(*tMatrix);
        SetWinWorldTransform( &fMatrixUsed);

        trect = fFlip->transformBounds(trect);
    }
    else
    {
        if (tMatrix)
            fMatrixUsed = *tMatrix;
        else
            fMatrixUsed.setToIdentity();
        SetWinWorldTransform( tMatrix );
    }
    if (!fMatrixUsed.isIdentity())
        trect = fMatrixUsed.transformBounds(trect);

    IGRect2D worldRect = worldBounds();
    if (fFlip)
       worldRect.offset(IGPoint2D(0.0, 1.0)); //as worldBounds() have been shifted by one in IGrafPort::setupCoordinates()
    trect.intersectWith(worldRect);

    if (trect.isEmpty())
        return;

    Display *dstDpy = fIXDC->xDisplay;
    Window  dstWin = fIXDC->xDrawable;
    GC              dstGC = fIXDC->gc();

    XImage *orgXImage, *newXImage;
    bool flag=true;

    orgXImage = convert24To8Image(dstDpy, dstWin, image);

    if(orgXImage == NULL ){
       orgXImage = (XImage*)(unsigned long)image.imageHandle();
       flag = false;
    }

    if (fFlip)
        irect = fFlip->transformBounds(irect);

    int left = GCoordinateToInt(trect.fLeft);
    int top = GCoordinateToInt(trect.fTop);
    int width = GCoordinateToInt(trect.width()) ;
    int height = GCoordinateToInt(trect.height()) ;

    int sample_mode = grafState.attributeState()->imageSampling()->mode();
    int render_mode = grafState.attributeState()->imageTransferMode()->mode();
    bool transparency = image.transparencyFlag();
    unsigned long tIndex;
    if(transparency)
        tIndex = transparencyIndex(dstDpy, dstWin, image);

    if ((width != orgXImage->width) ||
            (height != orgXImage->height) ||
            (sample_mode == IImageSamplingControl::kHalfTone) ||
            (render_mode == IImageTransferMode::kOCHalfTone))
    {
            unsigned long  oldpixel;
			//XGetImage fails if the window is partially visiblei, but this
			//behaviour is true on Windows and not for pixmaps.
            newXImage = XGetImage( dstDpy, dstWin, left, top,
                                   width, height, AllPlanes, ZPixmap);

			if(!newXImage){ //If the drawable is a pixmap, it should not enter
				unsigned long pixstride;
				XWindowAttributes wattr;
				memset(&wattr, 0x0, sizeof(XWindowAttributes));
				XGetWindowAttributes(dstDpy, dstWin, &wattr);
				if (IColorMap::hasColorMapSupport())
					pixstride = PIXELSTRIDE(width, 8);
				else
					pixstride = PIXELSTRIDE(width, 32);
				char *imgData = new char[height * pixstride];
				memset(imgData, 0x0, (height * pixstride));
				newXImage = XCreateImage(dstDpy, wattr.visual, wattr.depth, 
								ZPixmap, 0, imgData, width, height,
								32, pixstride);
			}

            if ((sample_mode == IImageSamplingControl::kHalfTone) ||
                    (render_mode == IImageTransferMode::kOCHalfTone))
            {
                    if (IColorMap::hasColorMapSupport())
                            queryColormap();
            }

            IGrafMatrix invMatrix = fMatrixUsed;
            invMatrix.invert();
            IGPoint2D ipoint, newPoint;

            for (int i = 0; i < height; i++)
            {
                    ipoint.fY =  trect.fTop + i;
                    for (int j = 0; j < width; j++)
                    {
                            ipoint.fX = trect.fLeft + j;
                            newPoint = invMatrix.transformPoint(ipoint);
                            if (irect.contains(newPoint))
                            {
                                    IGPoint2D transformedPoint;

                                    transformedPoint.fX = newPoint.fX - irect.fLeft;
                                    transformedPoint.fY = newPoint.fY - irect.fTop;

                                    oldpixel = imageSamplingControl(orgXImage,
                                                                    newXImage,
                                                                    transformedPoint,
                                                                    j, i,
                                                                    sample_mode);
                                    if (render_mode == IImageTransferMode::kOCHalfTone)
                                            imageTransferControl(oldpixel, render_mode);
                                    if(transparency){
                                            if(oldpixel != tIndex)
                                                    XPutPixel(newXImage, j, i, oldpixel);
                                    } else
                                            XPutPixel(newXImage, j, i, oldpixel);
                            }
                    }
            }
    } else {
            if(transparency){
                    newXImage = XGetImage( dstDpy, dstWin, left, top,
                                           width, height, AllPlanes, ZPixmap);
                    unsigned long  pixel;
                    for (int i = 0; i < height; i++)
                            for (int j = 0; j < width; j++)
                            {
                                    pixel = XGetPixel(orgXImage, j, i);
                                    if(pixel != tIndex)
                                            pixel = XPutPixel(newXImage, j, i, pixel);
                            }
            } else
                    newXImage = orgXImage;
    }

    XGCValues gc_val, old_val;

    if( render_mode == IImageTransferMode::kOCHalfTone)
    {
            gc_val.function = GXcopy;
    }
    else
    {
            gc_val.function = imageTransferMode(render_mode);
    }

    if (IGrafPort::restoreGC)
        XGetGCValues(dstDpy, dstGC, GCFunction, &old_val);

    XChangeGC(dstDpy, dstGC, GCFunction, &gc_val);

    XPutImage(dstDpy, dstWin, dstGC,
              newXImage,
              0,
              0,
              left,
              top,
              newXImage->width ,
              newXImage->height  );
    
#if 0 //SAS
    XPutImage(dstDpy, DefaultRootWindow(dstDpy),
              DefaultGC(dstDpy, DefaultScreen(dstDpy)),
              newXImage,
              0,
              0,
              0,
              200,
              newXImage->width ,
              newXImage->height  );
    
#endif //SAS
    
    if (IGrafPort::restoreGC)
        XChangeGC(dstDpy, dstGC, GCFunction, &old_val);
    
    if ((width != orgXImage->width) ||
            (height != orgXImage->height) ||
            transparency ||
            (sample_mode == IImageSamplingControl::kHalfTone) ||
            (render_mode == IImageTransferMode::kOCHalfTone))
            XDestroyImage(newXImage);

    if(flag)
            XDestroyImage(orgXImage);
}

//================================================================
//     IXDevice::renderMaskedImage
//

void IXDevice::renderMaskedImage(
       const IGImage& image,
       const IGImage& mask,
       const IGRect2D& source,
       const IGrafState& grafState)
{
    IFUNCTRACE_ALL();
    fStateChange = grafState.registeredAsTheStateInUse();

    setupTransforms(grafState);

    if (SetClipArea(grafState) == kClipAreaEmpty)
            return;

    IGRect2D trect = source;
    IGRect2D irect = image.rasterBounds();

    //assuming the image and mask are of the same size
    if (irect != mask.rasterBounds())
        return;

    irect.orderPoints();

    int iwidth = GCoordinateToInt(irect.width());
    int iheight = GCoordinateToInt(irect.height());

    const IGrafMatrix* tMatrix = grafState.modelMatrix();
       IGrafMatrix fMatrixUsed;

    trect.orderPoints();

       // to avoid images to go upside down on flip coordinates
    //fMatrixUsed is being used.
    const IGrafMatrix* fFlip = grafState.root()->coordinateFlip();
    if (fFlip)
    {
        fMatrixUsed = *fFlip;
        fMatrixUsed.invert();
        fMatrixUsed.concatWith(*tMatrix);
        SetWinWorldTransform( &fMatrixUsed);

        trect = fFlip->transformBounds(trect);
    }
    else
    {
        if (tMatrix)
            fMatrixUsed = *tMatrix;
        else
            fMatrixUsed.setToIdentity();
        SetWinWorldTransform( tMatrix );

    }
    if (!fMatrixUsed.isIdentity())
        trect = fMatrixUsed.transformBounds(trect);

    IGRect2D worldRect = worldBounds();
    if (fFlip)
       worldRect.offset(IGPoint2D(0.0, 1.0)); //as worldBounds() have been shifted by one in IGrafPort::setupCoordinates()
    trect.intersectWith(worldRect);

    if (trect.isEmpty())
        return;

    Display    *dstDpy = fIXDC->xDisplay;
    Window     dstWin  = fIXDC->xDrawable;
    GC         dstGC   = fIXDC->gc();

    XImage *orgXImage=NULL, *maskXImage=NULL, *newXImage=NULL;
    bool flag1=true, flag2=true;

    //assuming the image and mask are of the same size
    if (irect == mask.rasterBounds())
    {
        orgXImage = convert24To8Image(dstDpy, dstWin, image);
        maskXImage = convert24To8Image(dstDpy, dstWin, mask);

        if(!orgXImage){
           orgXImage = (XImage*)(unsigned long)image.imageHandle();
           flag1 = false;
        }

        if(!maskXImage){
           maskXImage = (XImage*)(unsigned long)mask.imageHandle();
           flag2 = false;
        }
    }else {
        IGImage orgImage(image, irect);
        orgImage.sizeTo(IGPoint2D(irect.width(), irect.height() ));
        IGImage maskImage(mask, irect);
        maskImage.sizeTo(IGPoint2D(irect.width(), irect.height() ));

        orgXImage = convert24To8Image(dstDpy, dstWin, orgImage);
        maskXImage = convert24To8Image(dstDpy, dstWin, maskImage);

        if(!orgXImage){
           orgXImage = (XImage*)(unsigned long)orgImage.imageHandle();
           flag1 = false;
        }

        if(!maskXImage){
           maskXImage = (XImage*)(unsigned long)maskImage.imageHandle();
           flag2 = false;
        }
    }

    if (fFlip)
        irect = fFlip->transformBounds(irect);

    int left = GCoordinateToInt(trect.fLeft);
    int top = GCoordinateToInt(trect.fTop);
    int width = GCoordinateToInt(trect.width()) ;
    int height = GCoordinateToInt(trect.height()) ;

    unsigned long  oldpixel, maskPixel = 0L;
    newXImage = XGetImage(dstDpy, dstWin,
                          left, top, width, height, AllPlanes, ZPixmap);

    int sample_mode = grafState.attributeState()->imageSampling()->mode();
    int render_mode = grafState.attributeState()->imageTransferMode()->mode();
    bool transparency = mask.transparencyFlag();
    unsigned long tIndex;
    if(transparency)
       tIndex = transparencyIndex(dstDpy, dstWin, mask);


    if ((sample_mode != IImageSamplingControl::kHalfTone) &&
            (render_mode == IImageTransferMode::kOCHalfTone))
    {
            if (IColorMap::hasColorMapSupport())
                    queryColormap();
    }

    IGrafMatrix invMatrix = *tMatrix;
    invMatrix.invert();
    IGPoint2D ipoint, newPoint;

    for (int i = 0; i < height; i++)
    {
            ipoint.fY = trect.fTop + i;
            for (int j = 0; j < width; j++)
            {
                    ipoint.fX = trect.fLeft + j;
                    newPoint = invMatrix.transformPoint(ipoint);
                    if (irect.contains(newPoint))
                    {
                            IGPoint2D transformedPoint;

                            transformedPoint.fX = newPoint.fX - irect.fLeft;
                            transformedPoint.fY = newPoint.fY - irect.fTop;

                            oldpixel = imageSamplingControl(orgXImage,
                                                            newXImage,
                                                            transformedPoint,
                                                            j, i,
                                                            sample_mode);
                            maskPixel = imageSamplingControl(maskXImage,
                                                             newXImage,
                                                             transformedPoint,
                                                             j, i,
                                                             sample_mode);
                    }

                    if(transparency){
                            if(maskPixel == tIndex)
                                    XPutPixel(newXImage, j, i, oldpixel);
                    } else
                            XPutPixel(newXImage, j, i, oldpixel);
            }
    }

    XGCValues gc_val, old_val;
    gc_val.function = imageTransferMode(render_mode);

    if (IGrafPort::restoreGC)
        XGetGCValues(dstDpy, dstGC, GCFunction, &old_val);

    XChangeGC(dstDpy, dstGC, GCFunction, &gc_val);

    XPutImage(dstDpy, dstWin, dstGC,
        newXImage,
        0,
        0,
        left,
        top,
        newXImage->width ,
        newXImage->height  );

    if (IGrafPort::restoreGC)
        XChangeGC(dstDpy, dstGC, GCFunction, &old_val);

    if(flag1)
            XDestroyImage(orgXImage);
    if(flag2)
            XDestroyImage(maskXImage);

    XDestroyImage(newXImage);
}



IXFontContext::IXFontContext(
       IXDevice* theDevice,
       const IGTextRun& theText,
       const IGrafMatrix* modelMatrix,
       const IGrafMatrix* coordFlip) :

       fDevice(theDevice),
       fTmCached(false),
       fAnchorTransform(0 /*NIL*/),

       fDeviceContext(theDevice->fHDC),
       fIXDeviceContext((IXDC*)theDevice->fHDC),

       fPointSize(theText.fPointSize),
       fEscapement(theText.fEscapement),
       fWeight(theText.fWeight),
       fItalic((short)theText.fItalic),
       fTypeFace(theText.fTypeFace),
       fUnderline((short)theText.fUnderline),
       fStrikeOut((short)theText.fStrikeOut),
       fFixedPitch((short)theText.fFixedPitch),
       fFontSet(0),
       fPtrFontStruct(0),

       fBounds( new IGRect2D()),

       fMatrix(modelMatrix),
       fMatrixStamp(modelMatrix ? modelMatrix->timeStamp() : 0),
       fFlip(coordFlip),
       fFlipStamp(coordFlip ? coordFlip->timeStamp() : 0)
{
       IFUNCTRACE_DEVELOP();
       // transform the adjusted origin

       if (!fFlip) {
               if (fMatrix)
                       fMatrixUsed = *fMatrix;
               else
                 fMatrixUsed.setToIdentity();
               fDevice->SetWinWorldTransform(fMatrix);
       }
       else
       {
    // the anchor point should be transformed as flipped, the rest of the grometry is not...
               fAnchorTransform = fFlip;
               fMatrixUsed = *fFlip;
               fMatrixUsed.invert();
               fMatrixUsed.concatWith(*fMatrix); // undo the coordinate flip from the model matrix...

               fDevice->SetWinWorldTransform(&fMatrixUsed);
       }

       loadFont();
}

void IXFontContext::loadFont()
{
    IFUNCTRACE_DEVELOP();
    unsigned long maxNumFonts=5000; // Arbitrarily large limit for XListFonts
    int resX, resY;
    Display* disp = fIXDeviceContext->xDisplay;
    int screen = DefaultScreen(disp);

    resX = DisplayWidth(disp,screen)/(DisplayWidthMM(disp, screen)/25.4);
    resY = DisplayHeight(disp,screen)/(DisplayHeightMM(disp, screen)/25.4);

    resX = (resX < 88 ? 75 : 100);
    resY = (resY < 88 ? 75 : 100);

  IXLFDString xlfdStr(fTypeFace);

  if( fWeight )
    xlfdStr.setWord("bold",IXLFDString::WEIGHT);
  else  
    xlfdStr.setWord("medium",IXLFDString::WEIGHT);

  if( fItalic )
    xlfdStr.setWord("i",IXLFDString::SLANT);     //since we can have a couple of values
  else  
     xlfdStr.setWord("r",IXLFDString::SLANT);

  if (fFixedPitch)
     xlfdStr.setWord("m",IXLFDString::SPACING); //since we can have a couple of values
  else
    xlfdStr.setWord("p",IXLFDString::SPACING);

    xlfdStr.setWord(IString(resX),IXLFDString::RESX);
    xlfdStr.setWord(IString(resY),IXLFDString::RESY);    
    xlfdStr.setWord(IString(fPointSize*10),IXLFDString::POINTSIZE);    
    fFontSet = IXCreateFontSet(disp, xlfdStr,false);

    if (!fFontSet) 
    {
      //check for valid font    
      IFontList* validFontList = IXLFDString::getFontListForCurrentLocale(disp);    
      IFontList::Cursor myCursor( *validFontList );
  
      IXLFDString IXLFDTestStr("-*-fixed");
        myCursor.setToFirst();
        
      IString theKey = IXLFDTestStr.keyName();
      if( validFontList->locateElementWithKey(theKey,myCursor) )
         fFontSet = IXCreateFontSet(disp, "fixed");

      if( !fFontSet)
      {
        myCursor.setToFirst();
        IXLFDString baseXLFDstr = validFontList->elementAt(myCursor);
                
        baseXLFDstr.setWord("*",IXLFDString::PIXELS);  //indicates we want a bitmap font
        baseXLFDstr.setWord("*",IXLFDString::AVGWIDTH);    
    
        if( fPointSize>0)
          baseXLFDstr.setWord(IString(fPointSize*10),IXLFDString::POINTSIZE);
        else if( baseXLFDstr.getWord(IXLFDString::POINTSIZE) == "0" )   
          baseXLFDstr.setWord("100",IXLFDString::POINTSIZE);  //default a font size to 10

        fFontSet = IXCreateFontSet(disp, baseXLFDstr,false);
      }
    }

	char **fontNames;
	int fontListLength = XFontsOfFontSet(fFontSet,
                                         &fPtrFontStruct,
                                         &fontNames);
	fFontStruct = fPtrFontStruct[0];

}

IXFontContext::~IXFontContext()
{ 
    IFUNCTRACE_DEVELOP();
//    XFreeFontSet(fIXDeviceContext->xDisplay, fFontSet);
}

bool IXFontContext::IsBlack(int pixel_id )
{
    if (fDevice->fColors)
    {   // has colormap
        const IR8G8B8A8Color& color ( (*fDevice->fColors)[pixel_id] );
        return ( color.fRed + color.fGreen + color.fBlue ) ? False : True;
    }
    else
    {   // True color.
        return pixel_id ? False : True;
    }
}

void IXFontContext::renderTextRunGlyphs(const IGTextRun& textRun,
                                IGPoint2D& anchorP,
                                const IGrafState& grafState)
{
   IFUNCTRACE_DEVELOP();
   if (!textRun.fSubTextCached)
       ((IGTextRun*)&textRun)->cacheRenderSubtext(false);

    const IText text1 = textRun.fSubText;
    const IText* text = &text1;
    int length = textRun.fSubText.length();
    const char* cText = (const char*)(*text);

    if (fMatrixUsed.isIdentity())      // transform the anchor if need to ...
    {                                       //for rendering
        renderGlyphs(
            textRun,
            anchorP,
            fIXDeviceContext->xDrawable,
            fIXDeviceContext->gc());
    }
    else { //for rendering text as pixmaps

        IGPoint2D newloc = calculateBoundsAndAdvancement( textRun);
        IGRect2D irect;
        irect  = *fBounds;
               if (irect.isEmpty())
                return;

        int iwidth = fDevice->GCoordinateToInt(irect.width());
        int iheight = fDevice->GCoordinateToInt(irect.height());

        unsigned long black = BlackPixel( fIXDeviceContext->xDisplay,
                                DefaultScreen( fIXDeviceContext->xDisplay));
        unsigned long white = WhitePixel( fIXDeviceContext->xDisplay,
                                DefaultScreen( fIXDeviceContext->xDisplay));

        irect.orderPoints();

        int x, y;
        unsigned int wid, ht, border, depth;
        Window root;
        XGetGeometry(fIXDeviceContext->xDisplay,
                fIXDeviceContext->xDrawable, &root,
               &x, &y, &wid, &ht, &border, &depth);

        XGCValues gcval;

        if (IColorMap::hasColorMapSupport())
               fDevice->queryColormap();


        Pixmap pix1 = XCreatePixmap( fIXDeviceContext->xDisplay,
                fIXDeviceContext->xDrawable,
                iwidth+1, iheight+1, depth);

        GC oldGC = XCreateGC( fIXDeviceContext->xDisplay,
                    fIXDeviceContext->xDrawable, 0L, NULL);

        XGetGCValues(fIXDeviceContext->xDisplay, fIXDeviceContext->gc(),
                GCForeground, &gcval);


        if(!IsBlack(gcval.foreground))
            XSetForeground(fIXDeviceContext->xDisplay, oldGC, black);
        else 
            XSetForeground(fIXDeviceContext->xDisplay, oldGC, white);

        XFillRectangle(fIXDeviceContext->xDisplay,
                               pix1,
                               oldGC,
                               0,0, iwidth, iheight);

        //make sure that fBounds have the right data before this call

               XSetForeground(fIXDeviceContext->xDisplay, oldGC, gcval.foreground);

        IGPoint2D pixmapPoint(0.,0.);
        renderGlyphs( textRun, pixmapPoint, pix1, oldGC);


        XImage* x1 = XGetImage( fIXDeviceContext->xDisplay,
                    pix1, 0,0, iwidth, iheight,
                    AllPlanes, ZPixmap);


        //for the dest image
        IGRect2D lrect(0., 0., irect.width(), irect.height());
        IGRect2D trect;

               //transform irect to trect
        if (fAnchorTransform)
            irect = fAnchorTransform->transformBounds(irect);

        IGrafMatrix iMatrix(fMatrixUsed);
        trect  = iMatrix.transformBounds(irect);

        trect.orderPoints();
       IGRect2D worldRect = fDevice->worldBounds();
       if (fFlip)
           worldRect.offset(IGPoint2D(0.0, 1.0)); //as worldBounds() have been shifted by one in IGrafPort::setupCoordinates()
       trect.intersectWith(worldRect);

        if (trect.isEmpty())
                return;

        int left = fDevice->GCoordinateToInt(trect.fLeft);
        int top = fDevice->GCoordinateToInt(trect.fTop);
               int right = fDevice->GCoordinateToInt(trect.fRight);
               int bottom = fDevice->GCoordinateToInt(trect.fBottom);
        int width = fDevice->GCoordinateToInt(trect.width()) ;
        int height = fDevice->GCoordinateToInt(trect.height()) ;

               if ((double)bottom >= worldRect.fBottom)
            height--;
        if ((double)right >= worldRect.fRight)
            width--;

        XImage* x2 = XGetImage(
                            fIXDeviceContext->xDisplay,
                                                       fIXDeviceContext->xDrawable,
                            left, top,
                            width ,
                            height,
                            AllPlanes,
                            ZPixmap);

        IGrafMatrix invMatrix = fMatrixUsed;
        invMatrix.invert();

        unsigned long oldpixel;

        for (unsigned long i = 0; i < height; i++)
        {
            for (unsigned long j = 0; j < width; j++)
            {
                IGPoint2D ipoint(trect.fLeft + j, trect.fTop + i);
                IGPoint2D newPoint = invMatrix.transformPoint(ipoint);
                newPoint.fX -= irect.fLeft;
                newPoint.fY -= irect.fTop;

                if ((lrect.contains(newPoint)) &&
                    (NEAREST_INT(newPoint.fX) <
                            fDevice->GCoordinateToInt(irect.width())) &&
                    (NEAREST_INT(newPoint.fY) <
                            fDevice->GCoordinateToInt(irect.height())))
                {
                    oldpixel = XGetPixel(x1,
                                    NEAREST_INT(newPoint.fX),
                                    NEAREST_INT(newPoint.fY));
                    if(IsBlack(gcval.foreground) &&
                        IsBlack(oldpixel)){
                            XPutPixel(x2, j, i, oldpixel);
                    } else if(!IsBlack(gcval.foreground) &&
                                !IsBlack(oldpixel)) {
                                XPutPixel(x2, j, i, oldpixel);
                    }
                }
            }
        }

        XPutImage(
            fIXDeviceContext->xDisplay,
            fIXDeviceContext->xDrawable,
            fIXDeviceContext->gc(),
            x2, 0, 0, left, top, width , height  );

        XFreePixmap(fIXDeviceContext->xDisplay, pix1);
               XDestroyImage( x1 );
        XDestroyImage( x2 );
        XFreeGC(fIXDeviceContext->xDisplay, oldGC);

        anchorP = newloc;
    }
}



void IXFontContext::renderGlyphs(
       const IGTextRun& textRun,
       IGPoint2D& anchorP,
       Drawable& xDrawable, GC xGC) const
{
    IFUNCTRACE_DEVELOP();
    int x= fDevice->GCoordinateToInt(anchorP.fX);
    int y= fDevice->GCoordinateToInt(anchorP.fY);

    IGRect2D worldRect = fDevice->worldBounds();
    if (fFlip)
       worldRect.offset(IGPoint2D(0.0, 1.0)); //as worldBounds() have been shifted by one in IGrafPort::setupCoordinates()

    if (fFlip && (xDrawable == fIXDeviceContext->xDrawable) )
       y = worldRect.fBottom - y;
    else
    {
        x++; y++; //to avoid loosing of pixel values of the text on the edges
    }

    const IText text1 = textRun.fSubText;
    const IText* text = &text1;
    int length = textRun.fSubText.length();
    const char* cText = (const char*)(*text);
    char* tCharOfText = (char*)cText;
    char tempChar[2];

    XFontStruct* tempFont;
    tempFont = fFontStruct;


    XmFontListEntry entry = XmFontListEntryCreate( XmFONTLIST_DEFAULT_TAG,
                                                XmFONT_IS_FONTSET, fFontSet);
    XmString tempString;
    int attachment = textRun.attachment();

    if (IGBidiSettings::isBidiSupported() ) 
		IGBidiSettings::layoutObjectTransform(cText, 
				(size_t*)&length, 
				(void*)cText,
				(size_t*)&length);

    if( attachment == IGTextRun::kRightToLeft)
        tempString = XmStringSegmentCreate( ( char*)cText,
                             XmFONTLIST_DEFAULT_TAG ,
                              XmSTRING_DIRECTION_R_TO_L, 1);
    else
        tempString = XmStringSegmentCreate( ( char*)cText,
                             XmFONTLIST_DEFAULT_TAG ,
                              XmSTRING_DIRECTION_L_TO_R, 1);

    //XSetFont ( fIXDeviceContext->xDisplay, xGC, fFontStruct->fid); 

    XmFontList fontlist = 0;
    fontlist = XmFontListAppendEntry(fontlist, entry);
    Dimension width = XmStringWidth(fontlist, tempString);

    int xPos = x;
    int yPos = y;
    int heightascent   = tempFont->ascent;
    int heightdescent  = tempFont->descent;
    int height         = tempFont->ascent + tempFont->descent;

    if (xDrawable == fIXDeviceContext->xDrawable)
        yPos = y- height;

    unsigned long cnt;
    int tmp_x=0;

    if (fFlip && (xDrawable == fIXDeviceContext->xDrawable) )
    {
        if (attachment == IGTextRun::kTopToBottom)
            attachment = IGTextRun::kBottomToTop;

        if (attachment == IGTextRun::kBottomToTop)
            attachment = IGTextRun::kTopToBottom;
    }

    switch (attachment)
    {
        default:
        case IGTextRun::kLeftToRight:

            fFontStruct->direction = FontLeftToRight;

            if (fUnderline)
            {
                XmStringDrawUnderline (fIXDeviceContext->xDisplay,
                                  xDrawable,
                                  fontlist,
                                  tempString,
                                  xGC,
                                  xPos,yPos, width, XmALIGNMENT_BEGINNING,
                                  XmSTRING_DIRECTION_R_TO_L,
                                  NULL, tempString);
            }
            else
            {
                XmStringDraw(fIXDeviceContext->xDisplay,
                                  xDrawable,
                                  fontlist,
                                  tempString,
                                  xGC,
                                  xPos,yPos, width, XmALIGNMENT_BEGINNING,
                                  XmSTRING_DIRECTION_L_TO_R,
                                  NULL);
            }
            break;
        case IGTextRun::kRightToLeft:

            //fFontStruct->direction = FontRightToLeft;
            if (xDrawable != fIXDeviceContext->xDrawable)
            {
                xPos = x;
                yPos = y;
                tmp_x = x + XTextWidth(fFontStruct, cText, text->length());
            } else {
                xPos = xPos - XTextWidth(fFontStruct, cText, text->length());
                tmp_x = x;
            }

            XmStringDraw(fIXDeviceContext->xDisplay,
                              xDrawable,
                              fontlist,
                                                       tempString,
                              xGC,
                              xPos,yPos, width, XmALIGNMENT_BEGINNING,
                              XmSTRING_DIRECTION_R_TO_L,
                              NULL);
            if(fUnderline)
                XDrawLine(fIXDeviceContext->xDisplay,
                            xDrawable,
                            xGC,
                            xPos, yPos+height, tmp_x, yPos+height);
            break;

                       case IGTextRun::kTopToBottom:

            yPos = y ;

            for ( cnt = 0; cnt< length; cnt++)
            {
                tempChar[0] =  *tCharOfText;
                tempChar[1] =  0x0;

                tempString = XmStringSegmentCreate(  tempChar,
                             XmFONTLIST_DEFAULT_TAG ,
                              XmSTRING_DIRECTION_L_TO_R, 1);
                width = XmStringWidth(fontlist, tempString);

                if (xDrawable == fIXDeviceContext->xDrawable)
                    xPos = (int)(x - (ceil(float(width/2.))) +1);
                else
                {
                    xPos = (int)(ceil(float( (
                            (fFontStruct->per_char[(int)'W'].width)
                            - width)/2. ))  +1);
                }
                if (fUnderline)
                {
                    if (tempChar[0] != 0x20)
                        XmStringDrawUnderline (fIXDeviceContext->xDisplay,
                                xDrawable,
                                fontlist,
                                tempString,
                                xGC,
                                xPos,
                                yPos,
                                width,
                                XmALIGNMENT_CENTER,
                                XmSTRING_DIRECTION_L_TO_R,
                                NULL, tempString);
                }
                else
                {
                    XmStringDraw ( fIXDeviceContext->xDisplay,
                                xDrawable,
                                fontlist,
                                tempString,
                                xGC,
                                xPos,
                                yPos,
                                                               width,
                                XmALIGNMENT_CENTER,
                                XmSTRING_DIRECTION_L_TO_R,
                                NULL);
                }
                yPos += height;

                tCharOfText++;
            }
            break;
        case IGTextRun::kBottomToTop:

            if (xDrawable != fIXDeviceContext->xDrawable)
            {
                yPos = y + fBounds->height() - height -3;
            }
            else
                yPos -=3;
            for (cnt=0; cnt<length; cnt++)
            {
                tempChar[0] =  *tCharOfText;
                tempChar[1] =  0x0;

                tempString = XmStringSegmentCreate(  tempChar,
                                XmFONTLIST_DEFAULT_TAG ,
                                XmSTRING_DIRECTION_L_TO_R, 1);
                width = XmStringWidth(fontlist, tempString);

                if (xDrawable == fIXDeviceContext->xDrawable)
                    xPos = (int)(x - (ceil(float(width/2.))) +1);
                else
                {
                    xPos = (int)(ceil(float( (
                            (fFontStruct->per_char[(int)'W'].width)
                            - width)/2. ))  +1);
                }

                if (fUnderline)
                {
                    if (tempChar[0] != 0x20)
                        XmStringDrawUnderline (fIXDeviceContext->xDisplay,
                                xDrawable,
                                fontlist,
                                tempString,
                                                               xGC,
                                xPos,
                                yPos,
                                width,
                                XmALIGNMENT_CENTER,
                                XmSTRING_DIRECTION_L_TO_R,
                                NULL, tempString);
                }
                else
                {
                    XmStringDraw ( fIXDeviceContext->xDisplay,
                                xDrawable,
                                fontlist,
                                tempString,
                                xGC,
                                xPos,
                                yPos,
                                width,
                                XmALIGNMENT_CENTER,
                                XmSTRING_DIRECTION_L_TO_R,
                                NULL);
                }

                yPos -=height;
                tCharOfText++;
            }
            break;
    }

    IGPoint2D nextPoint  = calculateBoundsAndAdvancement( textRun);
    if (fStrikeOut)
            strikeOutText( textRun, anchorP, xDrawable, fBounds);
    anchorP = nextPoint;

	XmFontListEntryFree(&entry);
	XmFontListFree(fontlist);
    XmStringFree(tempString);
}


IGPoint2D IXFontContext::calculateBoundsAndAdvancement(
       const IGTextRun& textRun ) const
{
    IFUNCTRACE_DEVELOP();
    const IText text1 = textRun.fSubText;
    const IText* text = &text1;
    IGPoint2D anchor = textRun.anchor();

    if (fAnchorTransform)
        anchor = fAnchorTransform->transformPoint(anchor);

    int height = fFontStruct->ascent + fFontStruct->descent;

    int yOffset = anchor.fY - height; //XmStringDraw draws the text from
								   //the bottom of the anchor point

    int length = text->length(), cnt, shift;

    switch (textRun.attachment())
    {
        default:
        case IGTextRun::kLeftToRight:
            fBounds->fLeft = anchor.fX - 1;
            fBounds->fTop = yOffset;
            fBounds->fRight = anchor.fX +
                                XTextWidth(fFontStruct,
                                        (const char*)(*text),
                                        text->length())+1;
            //the bottom takes care of the underline space too
                       fBounds->fBottom = yOffset +
                            fFontStruct->ascent +
                            fFontStruct->descent +3;

                       anchor.fX += (XTextWidth(fFontStruct,
                                    (const char*)(*text),
                                    text->length()) + 2);
            break;
        case IGTextRun::kRightToLeft:
            fBounds->fLeft = anchor.fX -
                                (XTextWidth(fFontStruct,
                                        (const char*)(*text),
                                        text->length() ) +1);
            fBounds->fTop = yOffset;
            fBounds->fRight = anchor.fX +1;

            //the bottom takes care of the underline space too
                       fBounds->fBottom = yOffset +
                            fFontStruct->ascent +
                            fFontStruct->descent +3;

            anchor.fX -= (XTextWidth(fFontStruct,
                                    (const char*)(*text),
                                    text->length()) +2);
            break;
               case IGTextRun::kTopToBottom:

            shift = ceil((float)(fFontStruct->per_char[(int)'W'].width/2.0))
                     +2;
            fBounds->fLeft = anchor.fX - shift;
            fBounds->fTop = anchor.fY;
            fBounds->fBottom = fBounds->fTop + length * height +3;
            fBounds->fRight = anchor.fX + shift;
            anchor.fY = fBounds->fBottom;
            break;

        case IGTextRun::kBottomToTop:

            shift = ceil((float)(fFontStruct->per_char[(int)'W'].width/2.0))
                     +2;

            fBounds->fLeft  = anchor.fX - shift;
            fBounds->fRight = anchor.fX + shift;
            fBounds->fBottom = anchor.fY;
            fBounds->fTop =  fBounds->fBottom -  length * height -3;

            anchor.fY = fBounds->fTop;
            break;
    }
    if (fAnchorTransform)
    {
        *fBounds = fAnchorTransform->untransformBounds(*fBounds);
        anchor = fAnchorTransform->untransformPoint(anchor);
    }

    return anchor;

}

void IXFontContext::strikeOutText(const IGTextRun& textRun,
                                    IGPoint2D position,
                                    Drawable& xDrawable,
                                    IGRect2D* bounds) const
{
    IFUNCTRACE_DEVELOP();
    if (fAnchorTransform)
    {
        *bounds = fAnchorTransform->transformBounds(*bounds);
        position = fAnchorTransform->transformPoint(position);
    }

    int x1Pos, y1Pos,x2Pos, y2Pos;
    int height = fDevice->GCoordinateToInt(bounds->height());
    int width = fDevice->GCoordinateToInt(bounds->width());

    switch(textRun.attachment())
    {
        default:
        case IGTextRun::kLeftToRight:
        case IGTextRun::kRightToLeft:
            if (xDrawable == fIXDeviceContext->xDrawable)
            {
                x1Pos = fDevice->GCoordinateToInt(bounds->fLeft);
                y1Pos = fDevice->GCoordinateToInt(bounds->fBottom -
                            (bounds->fBottom - bounds->fTop)/2 );
                x2Pos = fDevice->GCoordinateToInt(bounds->fRight);
                y2Pos = y1Pos;
            }
            else
            {
                x1Pos = fDevice->GCoordinateToInt(position.fX);
                y1Pos = fDevice->GCoordinateToInt(position.fY +
                                height/2.);
                x2Pos = fDevice->GCoordinateToInt(position.fX + width);
                y2Pos =  y1Pos;
            }

            XDrawLine(fIXDeviceContext->xDisplay,
                    xDrawable,
                    fIXDeviceContext->gc(),
                    x1Pos, y1Pos, x2Pos, y2Pos);
            break;

               case IGTextRun::kTopToBottom:
               case IGTextRun::kBottomToTop:
            if (xDrawable == fIXDeviceContext->xDrawable)
            {
                x1Pos = fDevice->GCoordinateToInt(bounds->fLeft +
                        (bounds->fRight -
                        bounds->fLeft)/2);
                y1Pos = fDevice->GCoordinateToInt(bounds->fTop );
                x2Pos = x1Pos;
                y2Pos = fDevice->GCoordinateToInt(bounds->fBottom );
            }
            else
            {
                x1Pos = fDevice->GCoordinateToInt(position.fX +
                            width/2.);
                y1Pos = fDevice->GCoordinateToInt(position.fY);
                x2Pos = x1Pos;
                y2Pos = fDevice->GCoordinateToInt(position.fY +
                            height);
            }

            XDrawLine(fIXDeviceContext->xDisplay,
                    xDrawable,
                    fIXDeviceContext->gc(),
                    x1Pos, y1Pos, x2Pos, y2Pos);
            break;
    }
    if (fAnchorTransform)
    {
        *bounds = fAnchorTransform->untransformBounds(*bounds);
        position = fAnchorTransform->untransformPoint(position);
    }
}

inline bool IXFontContext::cacheHit(
       IXDevice* theDevice,
       const IGTextRun& theText,
       const IGrafMatrix* model,
       const IGrafMatrix* flip) const
{

       return (
               fDeviceContext == theDevice->fHDC &&
               fMatrix == model &&
               fMatrixStamp == (model ? model->timeStamp() : 0) &&
               fFlip == flip &&
               fFlipStamp == (flip ? flip->timeStamp() : 0) &&
               fPointSize == theText.fPointSize &&
               fEscapement == theText.fEscapement &&
               fWeight == theText.fWeight &&
               fItalic == (short)theText.fItalic &&
               fUnderline == (short)theText.fUnderline &&
               fStrikeOut == (short)theText.fStrikeOut &&
               fTypeFace == theText.fTypeFace);
}


IXFontContext& IXDevice::makeFontContext(
       const IGTextRun& theText,
       const IGrafMatrix* modelMatrix,
       const IGrafState* state)
{
       IFUNCTRACE_DEVELOP();
       const IGrafMatrix* coordFlip = (state == NULL ? 0 : state->root()->coordinateFlip());

       if (!theText.fStylesCached)
               ((IGTextRun*)&theText)->cacheRenderStyles();

       if (fFontCache) {
               if (fFontCache->cacheHit(this, theText, modelMatrix, coordFlip)) {

                       SetWinWorldTransform(&fFontCache->fMatrixUsed);

                       if (fFontCache->fGrafState != state)
                fFontCache->fGrafState = state;

                       return *fFontCache;
               }
               delete fFontCache;
	       fFontCache = 0;
       }

       fFontCache = new IXFontContext(this, theText, modelMatrix, coordFlip);

       return *fFontCache;

}

//================================================================
//  IXDevice::renderTextRun
//

IGPoint2D IXDevice::renderTextRun(
       const IGTextRun& textRun,
       const IGrafState& grafState)
{

    IFUNCTRACE_DEVELOP();
    const IText text1 = textRun.fSubText;
    const IText* text = &text1;

    if (!text)
       return textRun.anchor();

    IGPoint2D nextLocation = textRun.anchor();

    fStateChange = grafState.registeredAsTheStateInUse();

    setupTransforms(grafState);

    if (SetClipArea(grafState) == kClipAreaEmpty)   // before transformation
       return  nextLocation;

// get all attributes for creating a font from the style set
    IXFontContext& fontContext =
               makeFontContext(
                       textRun,
                       fModelMatrix,
                       &grafState);

//////////////////////////////////////////////////////////////
    UpdateTransferMode( grafState);
    UpdatePen (grafState);

    const IBaseColor* theColor = textRun.fTextColor;
    if (theColor)
       XSetForeground(fIXDC->xDisplay, fIXDC->gc(), lookupColormap(*theColor));

       fontContext.renderTextRunGlyphs(textRun, nextLocation, grafState);
//////////////////////////////////////////////////////////////

    return nextLocation;
}


// static function to persuit rendering text metrics... Does not work for righthand...
void IXDevice::determineTextMetrics(
       IGTextRun& textRun,
       IXDevice* dev,
       const IGrafState* grafState)
{
       IFUNCTRACE_DEVELOP();
       const IGrafMatrix* xform = 0 /*NIL*/;
       const IGrafMatrix* flip = 0 /*NIL*/;
       unsigned long matStamp = 0;
       unsigned long flipStamp = 0;

       IGPoint2D anchor = textRun.anchor();

       if (grafState) {
               xform = grafState->modelMatrix();
               flip = grafState->root()->coordinateFlip();
               if (xform)
                       matStamp = xform->timeStamp();
               if (flip)
                       flipStamp = flip->timeStamp();

               // we should eventually take into account of the bundle's time stamp...
       }

       if ( !dev)
       {
               Display* xdisp = IXDisplay::display();
               IXDC xdc(xdisp, DefaultRootWindow(xdisp));
               IPresSpaceHandle newIpsh(&xdc);
               IXDevice* xdev;
               xdev  = new IXDevice(newIpsh);

               if (!textRun.fStylesCached)
                       textRun.cacheRenderStyles();
               if (!textRun.fSubTextCached)
                       textRun.cacheRenderSubtext(false);

               IXFontContext& fontContext = xdev->makeFontContext(
                                                       textRun,
                                                       xform,
                                                       grafState);

               IGPoint2D next = fontContext.calculateBoundsAndAdvancement(textRun);

               textRun.updateMetrics(*( fontContext.fBounds),
                                       next,
                                       0,
                                       matStamp,
                                       flipStamp);
               delete (xdev);
               return;
       }
       if (!textRun.hadMetrics(0, matStamp, flipStamp))
       {
               //For a pixel position (x,y) the upper left corner is (x, y+ascent)
               //and lower left corner is (x, y-descent+1)

               if (!textRun.fStylesCached)
                       textRun.cacheRenderStyles();
               if (!textRun.fSubTextCached)
                       textRun.cacheRenderSubtext(false);


               IXFontContext fontContext( dev, textRun, xform, flip);
               IGPoint2D next = fontContext.calculateBoundsAndAdvancement(textRun);

               textRun.updateMetrics(*( fontContext.fBounds),
                                       next,
                                       0,
                                       matStamp,
                                       flipStamp);
       }
}

