/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 30 1.54.2.1 source/albert/graph2d/winemul.cpp, 2d, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: winemul.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in winemul.h.                                                              *
*                                                                              *
*******************************************************************************/
#define INCL_PM
#define INCL_DOSMODULEMGR
#define INCL_DOSERRORS
#include <winemul.h>

#include <ipmdc.hpp>

#include <irect.hpp>
#include <ipoint.hpp>
#include <iexcept.hpp>
#include <istring.hpp>
#include <inumeric.hpp>
#include <iprimlck.hpp>

#include <grassert.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _WINEMUL_CPP_
#include <ipagetun.h>
#endif

#ifdef GRAPH2D_DEBUG
	#include <imprintf.h>
	#include <dumpPS.hpp>
#endif

#pragma pack(push,4)

//	these are a set of list macros to use with storage of
// 	brushes and pens for HPS ajd 5/30/1997

#define LISTANCHOR(TYPE) \
	struct {TYPE * Next; TYPE * Prior; TYPE * Root;}

#define LISTLINKAGE(TYPE) \
	struct {TYPE * Next; TYPE * Prior;}

#define ListInitialize(TYPE,LIST,LINK) \
	LIST.Next = LIST.Prior = LIST.Root = \
	(TYPE *)(((char *)&(LIST)) - (size_t)&(((TYPE *)0)->LINK))

#define ListRoot(LIST) \
	(LIST.Root)

#define ListHead(LIST) \
	(LIST.Next)

#define ListTail(LIST) \
	(LIST.Prior)

#define ListIsEmpty(LIST) \
	(LIST.Next == LIST.Root)

#define ListNext(ELEM,LINK) \
	((ELEM)->LINK.Next)

#define ListPrior(ELEM,LINK) \
	((ELEM)->LINK.Prior)

#define ListInsert(ELEM,LINK,BEFORE) \
	( \
	 ((ELEM)->LINK.Prior = (BEFORE)->LINK.Prior)->LINK.Next= (ELEM), \
	 ( (BEFORE)->LINK.Prior= (ELEM))->LINK.Next= (BEFORE) \
	)

#define ListAppend(ELEM,LINK,AFTER) \
	( \
	 ((ELEM)->LINK.Next = (AFTER)->LINK.Next)->LINK.Prior= (ELEM), \
	 ( (AFTER)->LINK.Next= (ELEM))->LINK.Prior= (AFTER) \
	)

#define ListQueue(ELEM,LINK,LIST) \
 ( \
  ((ELEM)->LINK.Prior = (LIST.Root)->LINK.Prior)->LINK.Next= (ELEM), \
  ( (LIST.Root)->LINK.Prior= (ELEM))->LINK.Next= (LIST.Root) \
 )

#define ListPush(ELEM,LINK,LIST) \
 ( \
  ((ELEM)->LINK.Next = (LIST.Root)->LINK.Next)->LINK.Prior= (ELEM), \
  ( (LIST.Root)->LINK.Next= (ELEM))->LINK.Prior= (LIST.Root) \
 )

#define ListDelete(ELEM,LINK) \
	( (ELEM)->LINK.Next->LINK.Prior= (ELEM)->LINK.Prior,\
   (ELEM)->LINK.Prior->LINK.Next= (ELEM)->LINK.Next)

typedef struct BRUSHLIST
{
	LISTLINKAGE(struct BRUSHLIST) list;
	IPresSpaceHandle	hps;
	HBRUSH				hBrush;
	BRUSHLIST();
	~BRUSHLIST();
}	BRUSHLIST, *PBRUSHLIST;
BRUSHLIST::BRUSHLIST()
{}
BRUSHLIST::~BRUSHLIST()
{}

typedef struct PENLIST
{
	LISTLINKAGE(struct PENLIST) list;
	IPresSpaceHandle	hps;
	HPEN				hPen;
	HBRUSH				hBrush;
	PENLIST();
	~PENLIST();
}	PENLIST, *PPENLIST;
PENLIST::PENLIST()
{}

PENLIST::~PENLIST()
{}

typedef struct BANCHOR
{
	struct BRUSHLIST * Next;
    struct BRUSHLIST * Prior;
    struct BRUSHLIST * Root;
}	BANCHOR;

typedef struct PANCHOR
{
	struct PENLIST * Next;
    struct PENLIST * Prior;
    struct PENLIST * Root;
}	PANCHOR;

void DeletePenList( void );
void DeleteBrushList( void );

#pragma pack(pop)

// Note: This needs to be added to some OCL header (and a corresponding
//       message to the library resource DLL).

/*--------------------------- ModifyWorldTransform -----------------------------
| Invoke equivalent GpiSetModelTransformMatrix.                                |
|                                                                              |
| Note that right now only iMode==MWT_IDENTITY is supported.                   |
------------------------------------------------------------------------------*/
int ModifyWorldTransform( const IPresSpaceHandle&  hps,
						  XFORM            *pMatrixLF,
						  DWORD             iMode ) {
	int
	result = false;

	switch( iMode )
	{
		case MWT_IDENTITY:
			result = GpiSetModelTransformMatrix( hps, 0, 0, TRANSFORM_REPLACE );
			break;

		default:
			break;
	}

	return result;
}

/*-------------------------------- BeginPath -----------------------------------
| Invoke equivalent GpiBeginPath.                                              |
------------------------------------------------------------------------------*/
int BeginPath( const IPresSpaceHandle& hps ) {
	int
	result = GpiBeginPath( hps, 1 );

	return result;
}

/*--------------------------------- EndPath ------------------------------------
| Invoke equivalent GpiEndPath.                                                |
------------------------------------------------------------------------------*/
int EndPath( const IPresSpaceHandle& hps ) {
	int
	result = GpiEndPath( hps );

	return result;
}

int CloseFigure( const IPresSpaceHandle& hps ) {
	int
	result = GpiCloseFigure( hps );

	return result;
}

/*------------------------------ GetStockObject --------------------------------
| There are two versions of this function, one each for pens and brushes.      |
|                                                                              |
| When first called for a given object tag, a static instance of the required  |
| object type is allocated and a pointer returned.                             |
|                                                                              |
| If an unsupported object is requested, this function throws a "library       |
| error" exception.                                                            |
------------------------------------------------------------------------------*/
static HBRUSH gPenBrush;

static HPEN
nullPen = 0;

void setPenBrush(HBRUSH penBrush) {
   if (gPenBrush)
      delete gPenBrush;
   gPenBrush = penBrush;
}


HPEN GetStockObject( EPenTag penType ) {
	HPEN
	result = 0;

	switch( penType )
	{
		case NULL_PEN:
			if( !nullPen )
			{
                IPrimalLock lock;
                if (!nullPen) {
    				nullPen = new PEN;

    				nullPen->lColor        = 0;
    				nullPen->lBackColor    = 0;
    				nullPen->usMixMode     = FM_LEAVEALONE;
    				nullPen->usBackMixMode = BM_LEAVEALONE;
    				nullPen->fxWidth       = 0;
    				nullPen->lGeomWidth    = 0;
    				nullPen->usType        = LINETYPE_INVISIBLE;
    				nullPen->usEnd         = LINEEND_DEFAULT;
    				nullPen->usJoin        = LINEJOIN_DEFAULT;
    				nullPen->usReserved    = 0;
                }
			}
			result = nullPen;
			break;

		default:
			ParameterException(false, IGraphicException::kInternalExeption);
			break;
	}

	return result;
}

static HBRUSH
hollowBrush = 0;

HBRUSH GetStockObject( EBrushTag brushType ) {
	HBRUSH
	result = 0;

	switch( brushType )
	{
		case HOLLOW_BRUSH:
			if( !hollowBrush )
			{
                IPrimalLock lock;
                if (!hollowBrush) {
    				hollowBrush = new BRUSH;

    				hollowBrush->lColor        = 0;
    				hollowBrush->lBackColor    = 0;
    				hollowBrush->usMixMode     = FM_LEAVEALONE;
    				hollowBrush->usBackMixMode = BM_LEAVEALONE;
    				hollowBrush->usSet         = 0;
    				hollowBrush->usSymbol      = 0;
    				hollowBrush->ptlRefPoint   = IPoint().asPOINTL();
                }
			}
			result = hollowBrush;
			break;

		default:
			ParameterException(false, IGraphicException::kInternalExeption);
			break;
	}

	return result;
}
                                        // these list anchors are used for storing
                                        // the previous object, so it can be returned
                                        // the old one as a new object is selected.
                                        //
                                        // NOTE: this has been explained to me that
                                        //       we need to do this on a hps basis
                                        //       so we will store the hps along with
                                        //       the brush and/or pen in a linked list
                                        //       and search/delete list when DeleteObject
                                        //       is called.

BANCHOR	bAnchor;                        // brush anchor
bool fBListInitialized = false;         // brush anchor init flag

PANCHOR pAnchor;                        // pen anchor
BANCHOR pbAnchor;                       // pen brush anchor
bool fPListInitialized = false;         // pen anchor init flag

int DeleteObject2( HPEN hPen, const IPresSpaceHandle& hps )
{
	if( hPen != nullPen )
	{
	 	delete hPen;
		hPen = 0;
	}
	else
	{
		PPENLIST	ppWork;
		if( !ListIsEmpty( pAnchor ) )
		{
			IPrimalLock lock;
			ppWork = ListHead( pAnchor );
			if( ppWork )
			{
				do
				{
					if( ppWork->hps == hps )
					{
						ListDelete( ppWork, list );
						if( ppWork )
							free( ppWork );
						if( ListIsEmpty( pAnchor ) )
							fPListInitialized = false;
						ppWork = ListTail( pAnchor );
					}
				}while( ( (	ppWork = (PPENLIST) ListNext( ppWork, list )) != ListRoot( ( pAnchor ) ) ) );
			}
		}
	}
	return 1;
}

int DeleteObject2( HBRUSH hBrush, const IPresSpaceHandle& hps )
{
	if( hBrush != hollowBrush )
	{
		delete hBrush;
		hBrush = 0;
	}
	else
	{
		PBRUSHLIST	pbWork;
		if( !ListIsEmpty( bAnchor ) )
		{
			IPrimalLock lock;
			pbWork = ListHead( bAnchor );
			if( pbWork )
			{
				do
				{
					if( pbWork->hps == hps )
					{
						ListDelete( pbWork, list );
						if( pbWork )
							free( pbWork );
						if( ListIsEmpty( bAnchor ) )
							fBListInitialized = false;
						pbWork = ListTail( bAnchor );
					}
				}while( ( (	pbWork = (PBRUSHLIST) ListNext( pbWork, list )) != ListRoot( ( bAnchor ) ) ) );
			}
		}
	}
	return 1;
}

#pragma pack(push,4)

/*------------------------------- OptionalPath ---------------------------------
| Objects of this class is used to implement optional                          |
| GpiBeginPath/GpiEndPath/GpiStrokePath bracketing of line drawing to          |
| handle cases where the current line width is > 1.                            |
------------------------------------------------------------------------------*/
struct OptionalPath
{
	OptionalPath ( const IPresSpaceHandle& hps, int &result );

	~OptionalPath ( ) ;

	IPresSpaceHandle hps;
	int hasPath, &result;
};	// OptionalPath

OptionalPath::OptionalPath ( const IPresSpaceHandle& hps, int &result )
	: hps( hps ),
	hasPath( false ),
	result( result )
{
	if( GpiQueryLineWidthGeom( hps ) > 1 )
	{
		// Drawing must occur within path.
		result = hasPath = GpiBeginPath( hps, 1 );
		// If already in a path, then that's OK, too.
		if( !result && SHORT1FROMMP( WinGetLastError(0) ) == PMERR_ALREADY_IN_PATH )
		{
			result = true;
			hasPath = false;
		}
	}
}

OptionalPath::~OptionalPath ( )
{
	// If we created a path, close it.
	if( hasPath )
	{
		if( !GpiEndPath( hps )
			||
			// Note: Only stroke the path if the drawing worked (as
			// implied by "result").
			(result && (GpiStrokePath(hps,1,0) == GPI_ERROR)))
		{
			result = false;
		}
	}
}
/*------------------------------- SavePosition ---------------------------------
| Objects of this class save and restore the current position for the HPS      |
| provided the constructor.                                                    |
------------------------------------------------------------------------------*/
struct SavePosition
{
	SavePosition ( const IPresSpaceHandle& hps ) ;
	~SavePosition ( ) ;
	IPresSpaceHandle hps;
	POINTL pos;
};	// SavePosition

SavePosition::SavePosition ( const IPresSpaceHandle& hps )
: hps( hps ) {
	GpiQueryCurrentPosition( hps, &pos );
}

SavePosition::~SavePosition ( ) {
	GpiSetCurrentPosition( hps, &pos );
}

#pragma pack(pop)

/*-------------------------------- Rectangle -----------------------------------
| Draw the equivalent rectangle using GpiMove+GpiBox.                          |
|                                                                              |
| If the current line width is > 1, then we must draw within a path.           |
|                                                                              |
| We preserve the current position just in case (Windows doesn't use or        |
| update it in Rectangle).                                                     |
------------------------------------------------------------------------------*/
int Rectangle( const IPresSpaceHandle& hps,
			   int              left,
			   int              top,
			   int              right,
			   int              bottom ) {
	int
	result = true;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			SavePosition
			savePos( hps );

			POINTL
			pt1 = { left, top};

			POINTL
			pt2 = { right, bottom};

			if( !GpiMove( hps, &pt1 )
				||
				GpiBox( hps, DRO_OUTLINEFILL, &pt2, 0, 0 ) == GPI_ERROR )
			{
				unsigned long
				err = WinGetLastError(WinQueryAnchorBlock(HWND_DESKTOP));
				if( ( err & 0x0000ffff ) == PMERR_INV_IN_PATH )
				{
					if( GpiBox( hps, DRO_OUTLINE, &pt2, 0, 0 ) == GPI_ERROR )
					{
						err = WinGetLastError(WinQueryAnchorBlock(HWND_DESKTOP));
						result = false;
					}
				}
				else
				{
					result = false;
				}
			}
		}
	}

	return result;
}

/*--------------------------------- Ellipse ------------------------------------
| Draw an ellipse using GpiSetArcParams+GpiFullArc.                            |
|                                                                              |
| See notes above (for Rectangle) concerning paths and preserving the current  |
| position.                                                                    |
------------------------------------------------------------------------------*/
int Ellipse( const IPresSpaceHandle& hps,
			 int              left,
			 int              top,
			 int              right,
			 int              bottom ) {
	int
	result = true;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			SavePosition
			savePos( hps );

			// Get bounding rectangle.
			IRectangle
			rect = IRectangle( IPoint( left, top ), IPoint( right, bottom ) );

			// Get center (we'll be GpiMove-ing there).
			POINTL
			center( rect.center().asPOINTL() );

			// Arc params match bounding rectangle.
			ARCPARAMS
			parms = { rect.width()/2, rect.height()/2, 0, 0};

			if( !GpiSetArcParams( hps, &parms )
				||
				!GpiMove( hps, &center )
				||
				GpiFullArc( hps, DRO_OUTLINEFILL, 1<<16 ) == GPI_ERROR )
			{
				result = false;
			}
		}
	}

	return result;
}


/*--------------------------------- Polygon ------------------------------------
| Draw the equivalent polygon using GpiPolygons.                               |
|                                                                              |
| We need to construct a POLYGON structure from the input arguments and use    |
| this as argument to GpiPolygons.  Since GpiPolygons implicitly adds the      |
| current position as the first element of the array of points, we move        |
| to the real first point to avoid extra drawing.  The current position is     |
| maintained.                                                                  |
|                                                                              |
| GpiPolygons requires an argument specifying alternate vs. winding mode.      |
| Under Windows, this will have been specified on a separate call to           |
| SetPolyFillMode.  To reconcile this, we must implement SetPolyFillMode       |
| to record alternate vs. winding in some place we can find from here.         |
|                                                                              |
| We stash the "poly fill mode" in the markerbundle's symbol field (which we   |
| don't use otherwise).  Note that there is some risk in doing this as a user  |
| might be doing real Gpi (including marker stuff).                            |
|                                                                              |
| Note: ALTERNATE should be left == 0 so that the default "poly fill mode"     |
|       appears to be ALTERNATE.                                               |
------------------------------------------------------------------------------*/
int SetPolyFillMode ( const IPresSpaceHandle& hps, int iMode ) {
	int
	result = GpiQueryMarker( hps );

	result = GpiSetMarker( hps, iMode );

	return result;
}

int Polygon( const IPresSpaceHandle&  hps,
			 const POINTL     *points,
			 int               numPoints ) {
	int
	result = true;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			SavePosition
			savePos( hps );

			POLYGON
			polygon = { numPoints, (POINTL*)points};

			int
			fillMode = ( GpiQueryMarker(hps) == ALTERNATE )
					   ? POLYGON_ALTERNATE : POLYGON_WINDING;

#if 1
			// Just draw lines.  The figure is enclosed in a path that will be
			// stroked and/or filled.
			if( !GpiMove( hps, (POINTL*)points+numPoints-1 )
				||
				GpiPolyLine( hps, numPoints, (POINTL*)points ) == GPI_ERROR )
			{
				result = false;
			}
#else
			if( !GpiMove( hps, (POINTL*)points )
				||
				GpiPolygons( hps,
							 1,
							 &polygon,
							 POLYGON_BOUNDARY | fillMode,
							 POLYGON_INCL ) == GPI_ERROR )
			{
				result = false;
			}
#endif
		}
	}

	return result;
}

/*------------------------------ SelectClipPath --------------------------------
| This function is implemented using GpiSetClipRegion if the mode is           |
| RGN_COPY (which simply makes the current path the clip region).              |
|                                                                              |
| Other modes TBD.                                                             |
------------------------------------------------------------------------------*/
int SelectClipPath( const IPresSpaceHandle& hps, int mode ) {
	int
	result = false;

	switch( mode )
	{
		case RGN_COPY:
			{
				IRegionHandle
				old,
				hrgn = GpiPathToRegion( hps,
										1,
										( GpiQueryMarker( hps ) == ALTERNATE )
										? FPATH_ALTERNATE : FPATH_WINDING );

				if( hrgn != RGN_ERROR )
				{
					// Path converted to region OK.  Make it the clip region.
					if( GpiSetClipRegion( hps, hrgn, (PHRGN)&old ) != RGN_ERROR )
					{
						// Region reset.  Clean up previous region.
						if( old != NULLHANDLE )
						{
							if( GpiDestroyRegion( hps, old ) )
							{
								// Old region destroyed OK.  Indicate success.
								result = true;
							}
						}
						else
						{
							// No old region to destroy, indicate success.
							result = true;
						}
					}
				}
			}
			break;

		default:
			break;
	}

	return result;
}

/*------------------------------- PathToRegion ---------------------------------
| This function is implemented by calling GpiPathToRegion.                     |
------------------------------------------------------------------------------*/
IRegionHandle PathToRegion( const IPresSpaceHandle& hps ) {
	IRegionHandle
	result = GpiPathToRegion( hps,
							  1,
							  ( GpiQueryMarker( hps ) == ALTERNATE )
							  ? FPATH_ALTERNATE : FPATH_WINDING );

	// Map RGN_ERROR -> 0 (to match Windows semantics).  Actually, I think
	// RGN_ERROR==0 but better safe than sorry.
	if( result == RGN_ERROR )
	{
		result = 0;
	}

	return result;
}

/*-------------------------------- CombineRgn ----------------------------------
| This function is implemented via GpiCombineRegion.                           |
|                                                                              |
| Note that the input combineMode is presumed to be a valid GPI CRGN_* value.  |
| Also, the resulting GPI complexity indicators (RGN_*) must map to the        |
| values given the Windows constants NULLREGION, SIMPLEREGION, etc.            |
------------------------------------------------------------------------------*/
#undef CombineRgn
int CombineRgn( IRegionHandle    resultRgn,
				IRegionHandle    source1,
				IRegionHandle    source2,
				int              combineMode,
				const IPresSpaceHandle& hps ) {
	int
	result = GpiCombineRegion( hps, resultRgn, source1, source2, combineMode );

	// Map RGN_ERROR -> ERROR.
	if( result == RGN_ERROR )
	{
		result = ERROR;
	}

	return result;
}

/*---------------------------------- GetDC -------------------------------------
| This function maps directly to WinGetPS.                                     |
|                                                                              |
| Note: This returns a micro-cached PS!                                        |
------------------------------------------------------------------------------*/
IPresSpaceHandle GetDC( INativeWindowHandle hwnd ) {
	IPresSpaceHandle
	result = WinGetPS( hwnd ? hwnd : INativeWindowHandle( HWND_DESKTOP ) );

	return result;
}

/*-------------------------------- SetMapMode ----------------------------------
| This function is implemented by calling GpiSetPS specifying flags that       |
| change the PS_UNITS attribute.  The mode argument is mapped from the Windows |
| MM_* value straight to the corresponding GPI PU_* value (see winemul.h).     |
|                                                                              |
| Note: This function causes the PS attributes to be reset to their default    |
|       values.                                                                |
------------------------------------------------------------------------------*/
DWORD SetMapMode( const IPresSpaceHandle& hps, DWORD mode ) {
	SIZEL
	size;

	// Get prior value to return.
	DWORD
	result = GpiQueryPS( hps, &size );

	// Set new value.
	if( !GpiSetPS( hps, &size, ( result & ~PS_UNITS ) | PS_NORESET | mode ) )
	{
		// Set failed, return 0.
		result = 0;
	}
	else
	{
		result &= PS_UNITS;
	}

	return result;
}

/*------------------------ GetMapMode --------------------------
| Inverse of SetMapMode (see above).                           |
--------------------------------------------------------------*/
DWORD GetMapMode( const IPresSpaceHandle& hps ) {
	DWORD
	result;

	SIZEL
	ignore;

	result = GpiQueryPS( hps, &ignore ) & PS_UNITS;

	return result;
}

/*------------------------------ GetWindowRect ---------------------------------
| This function maps directly to WinQueryWindowRect.                           |
------------------------------------------------------------------------------*/
int GetWindowRect( INativeWindowHandle hwnd, RECT *pRect ) {
	RECTL
	rect;
	int
	result = WinQueryWindowRect( hwnd, &rect );

	if( result )
	{
		pRect->left   = rect.xLeft;
		pRect->bottom = rect.yTop;
		pRect->right  = rect.xRight;
		pRect->top    = rect.yBottom;
	}

	return result;
}

/*------------------------------- WindowFromDC ---------------------------------
| This function presents a problem.  OS/2 provides no way to go from an HPS    |
| to an HWND unless the HPS is associated with a DC allocated using            |
| WinOpenWindowDC (which would be unlikely).                                   |
|                                                                              |
| The only fix is to change the code that calls this function.  To help        |
| find such code, we'll just return 0 (which indicates an error).              |
------------------------------------------------------------------------------*/
INativeWindowHandle WindowFromDC( const IPresSpaceHandle& hps ) {
	if (hps.fhandle) {
  		IPMDC* hdc = (IPMDC*)hps.fhandle;
		if (hdc->fHPS)
			return hdc->fHWND;
	}
	return HWND_DESKTOP;
}


/*------------------------------- ColorToIndex ---------------------------------
| This utility function obtains the color index for a given RGB value.         |
|                                                                              |
| The input is in Windows-ese so we have to swap the red/blue bytes.           |
------------------------------------------------------------------------------*/
static COLORREF swapRedBlue( COLORREF in ) {
	ULONG
	red      = in & 0x000000ff,
			   green    = in & 0x0000ff00,
						  blue     = in & 0x00ff0000;

	COLORREF
	result = ( red << 16 ) + green + ( blue >> 16 );

	return result;
}

long ColorToIndex( const IPresSpaceHandle& hps, COLORREF color ) {
	LONG
	os2Color = swapRedBlue( color );

	return GpiQueryColorIndex( hps, 0, os2Color );
}

/*------------------------------ SetBrushOrgEx ---------------------------------
  |his function is implemented using GpiSetPatternRefPoint.                    |
------------------------------------------------------------------------------*/
int SetBrushOrgEx( const IPresSpaceHandle& hps, int x, int y, POINTL *old ) {
	int
	result = false;

	POINTL
	pt = { x, y};

	// Get old point, if requested.
	if( old )
	{
		GpiQueryPatternRefPoint( hps, old );
	}

	result = GpiSetPatternRefPoint( hps, &pt );

	return result;
}

/*-------------------------------- SetBkMode -----------------------------------
| This function maps to GpiSetBackMix.                                         |
|                                                                              |
| We have to do some work to ensure that the previous back mix mode is         |
| returned (0 if error).                                                       |
------------------------------------------------------------------------------*/
int SetBkMode( const IPresSpaceHandle& hps, int mode ) {
	int
	result = GpiQueryBackMix( hps );

	if( result != BM_ERROR )
	{
		if( GpiSetBackMix( hps, mode ) == BM_ERROR )
		{
			result = 0;
		}
		else
		{
			// Map BM_DEFAULT to real value ('cause 0 means error to Windoze).
			if( result == BM_DEFAULT )
			{
				AREABUNDLE
				bundle;
				GpiQueryDefAttrs( hps, PRIM_AREA, ABB_BACK_MIX_MODE, &bundle );
				result = bundle.usBackMixMode;
			}
		}
	}
	else
	{
		result = 0;
	}

	return result;
}

/*--------------------------------- SetROP2 ------------------------------------
| Simply implement by calling GpiSetMix.                                       |
------------------------------------------------------------------------------*/
int SetROP2( const IPresSpaceHandle& hps, int mode ) {
	int
	result = GpiSetMix( hps, mode );

	return result;
}

/*----------------------------- ExtSelectClipRgn -------------------------------
| Depending on the mode, we implement this function as follows:                |
|   RGN_COPY - Use GpiSetClipRegion to set the clip region to the argument.    |
|   other    - Get the current clip region, combine it with the argument       |
|              region, and reset it as the current clip region.                |
------------------------------------------------------------------------------*/
int ExtSelectClipRgn( const IPresSpaceHandle& hps, IRegionHandle hrgn, int mode ) {
	int
	result = ERROR;

	IRegionHandle
	old;

	if( mode == RGN_COPY )
	{
		// Make argument region the current clipping region.
		result = GpiSetClipRegion( hps, hrgn, (PHRGN)&old );

		// Free old clip region (if there was one).
		if( old )
			GpiDestroyRegion( hps, old );

	}
	else
	{
		// Get current clip region (and un-set it).
		if( GpiSetClipRegion( hps, 0, (PHRGN)&old ) )
		{
			// Combine old region with argument.
			GpiCombineRegion( hps, old, old, hrgn, mode );
			// Make result the new clip region.
			IRegionHandle
			temp;
			result = GpiSetClipRegion( hps, old, (PHRGN)&temp );
		}
	}

	return result;
}

/*----------------------------------- Fix --------------------------------------
| This structure provides a more convenient means of manipulating Gpi          |
| FIXED values.                                                                |
------------------------------------------------------------------------------*/

#pragma pack(push,4)

struct Fix
{
	Fix( short i, short f );
	Fix( double d );
	Fix operator-(const Fix &fix2);
	operator long ( ) const ;
	operator unsigned long ( ) const ;
	operator double ( ) const ;
	long fixed;
};

Fix::Fix( short i, short f )
: fixed( i << 16 + f )
{
}
Fix::Fix( double d )
: fixed( d * 0x10000u )
{
}
Fix Fix::operator-(const   Fix   &fix2)
{
	return Fix( (double)(*this) - (double)fix2 );
}
Fix::operator long ( ) const {
	return fixed;
}
Fix::operator unsigned long ( ) const {
	return (long)(*this);
}
Fix::operator double ( ) const {
	return (double)( fixed>>16 ) + (double)( fixed & 0xffffu ) / (double)0x10000u;
}

#pragma pack(pop)

/*---------------------------- SetWorldTransform -------------------------------
| Build a MATRIXLF from the argument XFORM and call                            |
| GpiSetModelTransformMatrix.                                                  |
------------------------------------------------------------------------------*/
int SetWorldTransform( const IPresSpaceHandle& hps, XFORM *xform ) {
	MATRIXLF
	matrix = { Fix( xform->eM11 ), Fix( xform->eM12 ), 0,
		Fix( xform->eM21 ), Fix( xform->eM22 ), 0,
		xform->eDx,         xform->eDy,         1};

	int
	result = GpiSetModelTransformMatrix( hps, 9, &matrix, TRANSFORM_REPLACE );

	return result;
}

/*------------------------------- SelectObject ---------------------------------
| Each overload of this function is implemented by applying the argument       |
| attributes/object to the presentation space.                                 |
------------------------------------------------------------------------------*/
HBRUSH SelectObject( const IPresSpaceHandle& hps, HBRUSH hBrush )
{
	IPrimalLock lock;
	if( !fBListInitialized )
	{
		ListInitialize( BRUSHLIST, bAnchor, list );
		fBListInitialized = true;
	}
	// ajd 5/19/1997 added to replace SelectObject
	// code so that SelectObject( HPEN ) can use it.
	return( SelectBrush( hps, hBrush, false ) );
}
	
HBRUSH SelectBrush( const IPresSpaceHandle& hps, HBRUSH hBrush, bool fPenFlag )
{
	ULONG
	masks[] = { ABB_COLOR, ABB_BACK_COLOR, ABB_MIX_MODE, ABB_BACK_MIX_MODE, ABB_SET, ABB_SYMBOL},
	rc = 0;

	// Set attributs one at a time.  Some fail (w/PMERR_INV_MIX_ATTR)
	// for some reason and this ensures the others "stick."
	// We gather the return codes (but don't check them).  This is
	// solely to aid debugging.
	if( hBrush )                        // ajd 6/5/1997 GpiSetAttrs doesn't lie hollow brush
	{
	for( int i = 0; i < 6; i++ )
	{
		rc <<= 1;
		rc |= GpiSetAttrs( hps, PRIM_AREA, masks[i], 0, hBrush );
	}

	if( hBrush->usSet )
	{
			GpiSetPatternSet( hps, hBrush->usSet );
			//  if( !GpiSetPatternSet( hps, hBrush->usSet ) )
			//  {
			//  	delete brushSelResult;
			//  	brushSelResult = 0;
			//  }
		}
	}


	BRUSHLIST	bWork;
	PBRUSHLIST	pbWork;
	HBRUSH		brushTemp;
	BANCHOR		anchorWork;

	if( fPenFlag )
		anchorWork = pbAnchor;
	else
		anchorWork = bAnchor;
                                        // check brush anchor to see if
                                        // list is empty.
	if( !ListIsEmpty( anchorWork ) )
	{
		bool found = false;

		pbWork = ListHead( anchorWork );

		do
		{
                                        // if we find our hps, we already stored
                                        // an old brush (as we selected one of
                                        // the same hps.
			if( pbWork->hps == hps )
			{
				brushTemp = pbWork->hBrush;
				pbWork->hBrush = hBrush;
				found = true;
				pbWork = ListTail( anchorWork );
}

		}while( ( (	pbWork = (PBRUSHLIST) ListNext( pbWork, list )) != ListRoot( ( anchorWork ) ) ) );
		                                // if we didn't find one, create a new one
                                        // and insert it into the brush anchor
		if( !found )
		{
                                        // allocate a list
			pbWork = (PBRUSHLIST)calloc( 1, sizeof( BRUSHLIST ) );
			pbWork->hps = hps;
			pbWork->hBrush = hBrush;
			ListInsert( pbWork, list, ListRoot( anchorWork ) );
			brushTemp = (HBRUSH)0;
		}
	}
	else                                // we have an empty list
	{
		pbWork = (PBRUSHLIST)calloc( 1, sizeof( BRUSHLIST ) );
		pbWork->hps = hps;
		pbWork->hBrush = hBrush;
		ListInsert( pbWork, list, ListRoot( anchorWork ) );
		brushTemp = (HBRUSH)0;
	}
	return brushTemp;
}

HPEN SelectObject( const IPresSpaceHandle& hps, HPEN hPen )
{
	IPrimalLock lock;
	if( !fPListInitialized )
	{                                   // init pen list anchor once
		ListInitialize( PENLIST, pAnchor, list );
		ListInitialize( BRUSHLIST, pbAnchor, list );
		fPListInitialized = true;
	}

	ULONG
	masks[] = { LBB_COLOR, LBB_BACK_COLOR, LBB_MIX_MODE, LBB_BACK_MIX_MODE,
		LBB_WIDTH, LBB_GEOM_WIDTH, LBB_TYPE, LBB_END, LBB_JOIN},
	rc = 0;

	// Set attributs one at a time.  Some fail (w/PMERR_INV_MIX_ATTR)
	// for some reason and this ensures the others "stick."
	// We gather the return codes (but don't check them).  This is
	// solely to aid debugging.
	if( hPen )                          // ajd 6/5/1997 GpiSetAttrs HATES
                                        // a null pen
	{
	for( int i = 0; i < 9; i++ )
	{
		rc <<= 1;
		rc |= GpiSetAttrs( hps, PRIM_LINE, masks[i], 0, hPen );
		}
	}

	PENLIST		pWork;
	PPENLIST	ppWork;
	HPEN		penTemp;
                                        // check brush anchor to see if
                                        // list is empty.
	if( !ListIsEmpty( pAnchor ) )
	{
		bool found = false;

		ppWork = ListHead( pAnchor );
	
		do
		{
                                        // if we find our hps, we already stored
                                        // an old brush (as we selected one of
                                        // the same hps.
			if( ppWork->hps == hps )
			{
				penTemp = ppWork->hPen;
				ppWork->hPen = hPen;
				if( gPenBrush )
					ppWork->hBrush = SelectBrush( hps, gPenBrush, true );
				found = true;
				ppWork = ListTail( pAnchor );
			}

		}while( ( (	ppWork = (PPENLIST) ListNext( ppWork, list )) != ListRoot( ( pAnchor ) ) ) );
		                                // if we didn't find one, create a new one
                                        // and insert it into the brush anchor
		if( !found )
		{
                                        // allocate a list
			ppWork = (PPENLIST)calloc( 1, sizeof( PENLIST ) );
			ppWork->hps = hps;
			ppWork->hPen = hPen;
			if( gPenBrush )
				ppWork->hBrush = SelectBrush( hps, gPenBrush, true );
			ListInsert( ppWork, list, ListRoot( pAnchor ) );
			penTemp = (HPEN)0;
		}
	}
	else                                // we have an empty list
	{
		ppWork = (PPENLIST)calloc( 1, sizeof( PENLIST ) );
		ppWork->hps = hps;
		ppWork->hPen = hPen;
		if( gPenBrush )
			ppWork->hBrush = SelectBrush( hps, gPenBrush, true );
		ListInsert( ppWork, list, ListRoot( pAnchor ) );
		penTemp = (HPEN)0;
	}
	return penTemp;
}

static  bool  fontSelResult_Init = false;
static  HFONT fontSelResult = 0;

HFONT SelectObject( const IPresSpaceHandle& hps, HFONT hFont )
{
    if (!fontSelResult_Init) {
        IPrimalLock lock;
        if (!fontSelResult_Init) {
            fontSelResult = new FATTRS;
            fontSelResult_Init = true;
        }
    }
	HFONT temp = fontSelResult;

	fontSelResult = hFont;

	return temp;
}

IBitmapHandle SelectObject( const IPresSpaceHandle& hps, IBitmapHandle bmp )
{                                       // ajd 6/6/1997
	//we cannot select the same bitmap into a presSpace
	//so select 0, to see what is set previously. If both prev and fBitmap 
	//are same then nothing to do, else select fBitmap.
        IBitmapHandle result = GpiSetBitmap(hps,  NULL);
	if(result == HBM_ERROR)
	{
                ITRACE_DEVELOP(IString("SelectObject 1:") + IString(WinGetLastError(0)).d2x());
		GrafDeviceException("SelectObject: Phase 1");
	}
	HBITMAP junk=GpiSetBitmap( hps, bmp );
	if(junk == HBM_ERROR)
	{
                ITRACE_DEVELOP(IString("SelectObject 2:") + IString(WinGetLastError(0)).d2x());
		GrafDeviceException("SelectObject: Phase 2");
	}

	return result;
}

/*--------------------------------- MoveToEx -----------------------------------
| Use GpiMove.                                                                 |
------------------------------------------------------------------------------*/
int MoveToEx( const IPresSpaceHandle& hps, int x, int y, POINTL *old ) {
	int
	result = false;

	if( old )
	{
		// Get current position.
		GpiQueryCurrentPosition( hps, old );
	}

	POINTL
	pos = { x, y};

	result = GpiMove( hps, &pos );

	return result;
}

/*---------------------------------- LineTo ------------------------------------
| Use GpiLine.                                                                 |
------------------------------------------------------------------------------*/
int LineTo( const IPresSpaceHandle& hps, int x, int y ) {
	int result;

	// create a path for geometry line
	OptionalPath
	path( hps, result );

	POINTL
	pt = { x, y};

	result = GpiLine( hps, &pt );

	return result;
}

/*--------------------------------- Polyline -----------------------------------
| Use GpiPolyLines.                                                            |
------------------------------------------------------------------------------*/
int Polyline( const IPresSpaceHandle& hps, POINTL *pts, int numPoints ) {
	int
	result;

	OptionalPath
	path( hps, result );

	if( result )
	{
		GpiMove( hps, pts );	 // move current position to the first point
		result = GpiPolyLine( hps, numPoints, pts );
	}

	return result;
}

/*---------------------------- SetStretchBltMode -------------------------------
| We stash this mode in the PSs IMAGEBUNDLE attribute (and use that when       |
| we need it during blitting).                                                 |
------------------------------------------------------------------------------*/
int SetStretchBltMode( const IPresSpaceHandle& hps, int mode ) {
	int
	result = false;


	IMAGEBUNDLE
	bundle;

	// Get current image bundle (to preserve other settings).
	GpiQueryAttrs( hps, PRIM_IMAGE, -1, &bundle );

	switch( mode )
	{
		case COLORONCOLOR:
			bundle.usMixMode = BBO_IGNORE;
			break;

		case BLACKONWHITE:
			bundle.usMixMode = BBO_AND;
			break;

		case WHITEONBLACK:
			bundle.usMixMode = BBO_OR;
			break;

		case HALFTONE:
			// Not supported in Gpi!
			break;

		default:
			break;
	}

	result = GpiSetAttrs( hps, PRIM_IMAGE, IBB_MIX_MODE, 0, &bundle );

	return result;
}

/*--------------------------------- FillPath -----------------------------------
| Use GpiFillPath.                                                             |
------------------------------------------------------------------------------*/
int FillPath( const IPresSpaceHandle& hps ) {
	int
	result = GpiFillPath( hps, 1, 0 );

	return result;
}

/*-------------------------------- StrokePath ----------------------------------
| Use GpiStrokePath.                                                           |
------------------------------------------------------------------------------*/
int StrokePath( const IPresSpaceHandle& hps ) {
	int result;

	if( GpiQueryLineWidthGeom( hps ) > 1)
		result = GpiStrokePath( hps, 1, 0 );
	else
		result = GpiOutlinePath( hps, 1, 0 );

	return result;
}

/*--------------------------------- FillRgn ------------------------------------
------------------------------------------------------------------------------*/
int FillRgn( const IPresSpaceHandle& hps, IRegionHandle hrgn, HBRUSH brush ) {
	int
	result = true;

	// Get current brush and select argument one.
	HBRUSH
	hOld = SelectObject( hps, brush );

	// Make lines invisible so no borders get drawn.
	LONG
	savedType = GpiQueryLineType( hps );

	GpiSetLineType( hps, LINETYPE_INVISIBLE );

	result = GpiPaintRegion( hps, hrgn );

	// Restore line type.
	GpiSetLineType( hps, savedType );

	// Replace prior brush.
	SelectObject( hps, hOld );

	return result;
}

/*--------------------------------- FrameRgn -----------------------------------
| Use GpiFrameRegion.                                                          |
------------------------------------------------------------------------------*/
int FrameRgn( const IPresSpaceHandle& hps, IRegionHandle hrgn, HBRUSH brush, int cx, int cy ) {
	SIZEL
	size = { cx, cy};

	HBRUSH
	hOld = SelectObject( hps, brush );

	// this code needs to emulate above for FillRgn, and get all rects
	// but if that is done, all rect frames will be drawn and we don't
	// want that.
	int
	result = ( GpiFrameRegion( hps, hrgn, &size ) != GPI_ERROR );

	SelectObject( hps, hOld );

	return result;
}

/*---------------------------------- angle -------------------------------------
| This utility functions calculate the angle of a line from the origin to the  |
| argument point.                                                              |
------------------------------------------------------------------------------*/
static Fix angle( double y, double x ) {
	double
	radians = ArcTan( y, x ),
			  degrees = radians * kRadiansToDegrees;

	return Fix( degrees );
}
static Fix angle( const POINTL &pt ) {
	return angle( pt.y, pt.x );
}
static Fix angle( const POINTL &pt1, const POINTL &pt2 ) {
	return angle( pt1.y - pt2.y, pt1.x - pt2.x );
}

/*----------------------------------- Pie --------------------------------------
| We draw the specified Pie slice using GpiPartialArc.                         |
|                                                                              |
| Specifically, we set the Arc parameters to match the ellipse's bounding      |
| rectangle.  We then move to the center of the ellipse to start the figure.   |
|                                                                              |
| We must begin an area definition to cause the area to get filled in.         |
|                                                                              |
| We then draw the partial arc defined by the angles to points pt1=(x2,y2)     |
| and pt2=(x1,y1).  This draws from the center to pt1 and around the arc       |
| to pt2.  We then draw back to the center to complete the pie slice, and      |
| end the area.                                                                |
|                                                                              |
| We must enclose this in a "optional" path (in case the current Pen has       |
| wide lines) and preserve the current position.                               |
|                                                                              |
| One more thing.  The center point is presumed to be in native coordinate     |
| space.  The other points are relative to the center from the Windows         |
| perspective.  Thus, to get the slice swept out the right way, we need to     |
| flip the start/end points.                                                   |
------------------------------------------------------------------------------*/
int Pie( const IPresSpaceHandle& hps,
		 int left, int top, int right, int bottom,
		 int x1, int y1, int x2, int y2 ) {
	int
	result = 1;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			// Save current position.
			SavePosition
			savedPos( hps );

			// Get bounding rectangle.
			IRectangle
			rect = IRectangle( IPoint( left, top ), IPoint( right, bottom ) );
			// Get center (we'll need it later for some Gpi calls).
			POINTL
			center( rect.center().asPOINTL() );
			// Calculate start/end points relative to center.
			// We swap start/end so arc sweeps the right way on OS/2.
			IPoint
			start ( x2, y2 ),
			end   ( x1, y1 );
			// Adjust points per OS/2 coordinate system.
			//start.setY( 2 * rect.center().y() - start.y() );
			//end.setY( 2 * rect.center().y() - end.y() );
			// Arc params match bounding rectangle.
			ARCPARAMS
			parms    =    {    rect.width(   )/2,    rect.height(   )/2,    0,    0};
			result = 0;
			if( GpiSetArcParams( hps, &parms ) )
			{
				if( GpiMove( hps, (PPOINTL)&center ) )
				{
					// Begin area (so pie gets filled in).
					if( GpiBeginArea( hps, 0 ) )
					{
						// Draw from center and around arc.
						Fix
						startAngle = angle( start.asPOINTL(), center ),
									 endAngle   = angle( end.asPOINTL(), center );
						if( (double)startAngle < 0.0 )
						{
							startAngle = (double)startAngle + 360.0;
						}
						if( (double)endAngle < 0.0 )
						{
							endAngle = (double)endAngle + 360.0;
						}
						Fix
						sweepAngle = endAngle - startAngle;
						if( (double)sweepAngle < 0.0 )
						{
							sweepAngle = (double)sweepAngle + 360.0;;
						}
#ifdef GRAPH2D_DEBUG
						imprintf( "Pie; rect=%s, center=(%d,%d), (x1,y1)=(%d,%d), (x2,y2)=(%d,%d)\n",
								  (char*)rect.asString(),
								  (int)center.x,
								  (int)center.y,
								  x1, y1,
								  x2, y2 );
						imprintf( "     start=%s, end=%s\n",
								  (char*)start.asString(),
								  (char*)end.asString() );
						imprintf( "     startAngle=%g, endAngle= %g, sweepAngle=%g\n",
								  (double)startAngle,
								  (double)endAngle,
								  (double)sweepAngle );
#endif
						if( GpiPartialArc( hps,
										   &center,
										   Fix(1,0),
										   startAngle,
										   sweepAngle ) != GPI_ERROR )
						{
							// Complete pie slice by drawing back to center.
							if( GpiLine( hps, &center ) )
							{
								result = 1;
							}
						}
						if( !GpiEndArea( hps ) && result )
						{
							result = 0;
						}
					}
				}
			}
		}
	}

	return result;
}

/*---------------------------------- Chord -------------------------------------
| Draw the specified chord (using GpiPartialArc).                              |
|                                                                              |
| This is similar to Pie but requires that we turn off drawing, go to the      |
| start of the arc, reset the line type (to resume drawing), and then draw     |
| the arc.  We define an area to close it off.                                 |
------------------------------------------------------------------------------*/
int Chord( const IPresSpaceHandle& hps,
		   int left, int top, int right, int bottom,
		   int x1, int y1, int x2, int y2 ) {
	int
	result = 1;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			// Save current position.
			SavePosition
			savedPos( hps );

			// Get bounding rectangle.
			IRectangle
			rect = IRectangle( IPoint( left, top ), IPoint( right, bottom ) );

			// Get center (we'll need it later for some Gpi calls).
			POINTL
			center( rect.center().asPOINTL() );

			// Start/end points for chord.
			IPoint
			start ( x2, y2 ),
			end   ( x1, y1 );

			// Save current line type (we'll be fiddling with it).
			LONG
			savedLineType = GpiQueryLineType( hps );

			// Arc params match bounding rectangle.
			ARCPARAMS
			parms = { rect.width()/2, rect.height()/2, 0, 0};

			result = 0;
			if( GpiSetArcParams( hps, &parms ) )
			{
				// Turn off drawing till we get to starting point.
				if( GpiSetLineType( hps, LINETYPE_INVISIBLE ) )
				{
					Fix
					startAngle = angle( start.asPOINTL(), center ),
								 endAngle   = angle( end.asPOINTL(), center );

					if( (double)startAngle < 0.0 )
					{
						startAngle = (double)startAngle + 360.0;
					}
					if( (double)endAngle < 0.0 )
					{
						endAngle = (double)endAngle + 360.0;
					}
					// Draw partial arc to get to starting point.
					if( GpiPartialArc( hps,
									   &center,
									   Fix(1,0),
									   startAngle,
									   0 ) != GPI_ERROR )
					{
						// Reset line type.
						GpiSetLineType( hps, savedLineType );
						// Begin area (so chord gets filled in).
						if( GpiBeginArea( hps, 0 ) )
						{
							// Draw around arc.
							Fix
							sweepAngle = endAngle - startAngle;
							if( (double)sweepAngle < 0.0 )
							{
								sweepAngle = (double)sweepAngle + 360.0;;
							}

							if( GpiPartialArc( hps,
											   &center,
											   Fix(1,0),
											   startAngle,
											   sweepAngle ) != GPI_ERROR )
							{
								result = 1;
							}
							if( !GpiEndArea( hps ) && result )
							{
								result = 0;
							}
						}
					}
				}
			}
		}
	}

	return result;
}

/*----------------------------------- Arc --------------------------------------
| The specified arc is drawn using GpiPartialArc.                              |
|                                                                              |
| This works just like Chord but we don't close the figure.                    |
------------------------------------------------------------------------------*/
int Arc( const IPresSpaceHandle& hps,
		 int left, int top, int right, int bottom,
		 int x1, int y1, int x2, int y2 ) {
	int
	result = 1;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			SavePosition
			savedPos( hps );

			// Get bounding rectangle.
			IRectangle
			rect = IRectangle( IPoint( left, top ), IPoint( bottom, right ) );

			// Get center (we'll need it later for some Gpi calls).
			POINTL
			center( rect.center().asPOINTL() ),
			start ( IPoint( x1, y1 ).asPOINTL() ),
			end   ( IPoint( x2, y2 ).asPOINTL() );

			// Save current line type (we'll be fiddling with it).
			LONG
			savedLineType = GpiQueryLineType( hps );

			// Arc params match bounding rectangle.
			ARCPARAMS
			parms = { rect.width()/2, rect.height()/2, 0, 0};

			result = 0;

			if( GpiSetArcParams( hps, &parms ) )
			{
				// Turn off drawing till we get to starting point.
				if( GpiSetLineType( hps, LINETYPE_INVISIBLE ) )
				{
					// Draw partial arc to get to starting point.
					if( GpiPartialArc( hps,
									   &center,
									   Fix(1,0),
									   angle(start),
									   0 ) != GPI_ERROR )
					{
						// Reset line type.
						GpiSetLineType( hps, savedLineType );
						// Draw around arc.
						if( GpiPartialArc( hps,
										   &center,
										   Fix(1,0),
										   angle( start ),
										   angle( end ) - angle( start ) ) != GPI_ERROR )
						{
							result = 1;
						}
					}
				}
			}
		}
	}

	return result;
}

/*-------------------------------- PolyBezier ----------------------------------
| Draws the specified curve using GpiPolySpline.                               |
|                                                                              |
| Since Gpi uses current position as the first point, we move there and        |
| pass GpiPolySpline the other n-1 points.                                     |
------------------------------------------------------------------------------*/
int PolyBezier( const IPresSpaceHandle& hps, POINTL *pts, DWORD numPoints ) {
	int
	result = 1;

	{
		OptionalPath
		path( hps, result );

		if( result )
		{
			IPrimalLock lock;
			SavePosition
			savedPos( hps );

			result = 0;

			// Set current position to the first point.
			if( GpiMove( hps, &pts[0] ) )
			{
				// Draw the curves.  PM uses current position rather than getting
				// starting point from first array element.
				if( GpiPolySpline( hps, numPoints-1, &pts[1] ) != GPI_ERROR )
				{
					result = 1;
				}
			}
		}
	}

	return result;
}

/*-------------------------------- SetBkColor ----------------------------------
| Call GpiSetBackColor passing the color index corresponding to the argument   |
| RGB value.                                                                   |
------------------------------------------------------------------------------*/
int SetBkColor( const IPresSpaceHandle& hps, COLORREF color ) {
	int
	result = GpiSetBackColor( hps, ColorToIndex( hps, color ) );

	return result;
}

/*-------------------------------- StretchBlt ----------------------------------
| Blit the source to the target using GpiBitBlt.                               |
|                                                                              |
| The flOptions is inferred from the current IMAGEBUNDLEs usMixMode (see       |
| SetStretchBltMode).                                                          |
------------------------------------------------------------------------------*/
int StretchBlt( const IPresSpaceHandle& hpsDest,
				int destX, int destY, int destWidth, int destHeight,
				const IPresSpaceHandle& hpsSrc,
				int srcX, int srcY, int srcWidth, int srcHeight,
				DWORD rop ) {
	int
	result = 0;

	// Define bounding rectangles.
	IRectangle
	source = IRectangle( IPoint( srcX, srcY ),
						 ISize( srcWidth, srcHeight ) ),
			 target = IRectangle( IPoint( destX, destY ),
								  ISize( destWidth -1, destHeight -1) );
	          //Decrease the widht and height by 1 as the last pixel is
             //included while filling the bitmap, if not this leads to
             //an output image having 1 pixel more in its width and height.

	// Build point array from those rectangles.
	POINTL
	points[4] = { target.bottomLeft().asPOINTL(),
		target.topRight().asPOINTL(),
		source.bottomLeft().asPOINTL(),
		source.topRight().asPOINTL()};

	// Get flOptions.
	IMAGEBUNDLE
	bundle;

	GpiQueryAttrs( hpsDest, PRIM_IMAGE, IBB_MIX_MODE, &bundle );

	unsigned long
	flOptions = bundle.usMixMode;

	LONG
	realRop = rop;

	// Convert funny ICLUIHalftone rop value to SRC (OR) PAT.
#define ICLUIHalfToneMode 0x00fc008a // From gdidev.c
	if( realRop == ICLUIHalfToneMode )
	{
		realRop = 0x000000fc;
		GpiSetPattern( hpsDest, PATSYM_HALFTONE );
	}

	// Get source bitmap and detach bitmap from device.
	IBitmapHandle
	bmp = GpiSetBitmap( hpsSrc, 0 );

	// Blit the bits.
	result = ( GpiWCBitBlt( hpsDest, bmp, 4, points, realRop, flOptions ) != GPI_ERROR );

	// Set the source bitmap back the way it was.
	GpiSetBitmap( hpsSrc, bmp );

	return result;
}

/*--------------------------------- MaskBlt ------------------------------------
| This function is not supported directly by Gpi.                              |
|                                                                              |
| We'll have to emulate it but that's tricky so we'll get to it later.  For    |
| now, just return error.                                                      |
------------------------------------------------------------------------------*/
int MaskBlt( const IPresSpaceHandle& hpsDest,
			 int destX, int destY, int destWidth, int destHeight,
			 const IPresSpaceHandle& hpsSrc,
			 int srcX, int srcY,
			 IBitmapHandle maskBmp, int maskX, int maskY, DWORD rop ) {
	return 0;
}


/*------------------------------- SetTextAlign ---------------------------------
| Set the text alignment by calling GpiSetTextAlignment with appropriate       |
| arguments.                                                                   |
------------------------------------------------------------------------------*/
unsigned int SetTextAlign( const IPresSpaceHandle& hps, unsigned int alignment ) {
	long
	horiz,
	vert;

	GpiQueryTextAlignment( hps, &horiz, &vert );

	unsigned int
	result = vert + horiz;

	GpiSetTextAlignment( hps, alignment & 0x000000FF, alignment & 0x0000FF00 );

	return result;
}

/*-------------------------------- ExtTextOut ----------------------------------
| Draw the argument text using GpiCharStringPos.                               |
------------------------------------------------------------------------------*/
int ExtTextOut( const IPresSpaceHandle& hps, int x, int y, unsigned int opts,
				const RECTL *pRect,
				const char*chars, unsigned int count,
				int *pDeltas ) {
	int
	result = 0;

	POINTL pt = {x,y};
	GpiMove( hps, &pt );

	if( GpiCharStringPos( hps, (PRECTL)pRect, opts, count, (PCH)chars,
						  (PLONG)pDeltas ) != GPI_ERROR )
	{
		result = 1;
	}

	return result;
}

/*--------------------------- GetTextExtentPoint32 -----------------------------
| Returns the rectangle in which the argument text would be drawn.  This       |
| is calculated by calling GpiQueryCharBox.                                    |
------------------------------------------------------------------------------*/
int GetTextExtentPoint32( const IPresSpaceHandle& hps,
						  const char *chars, int count,
						  SIZEL *size ) {
	int
	result = 0;

	POINTL
	points[4];

	result = GpiQueryTextBox( hps, count, (PCH)chars, 4, points );

	if( result )
	{
		*size = IRectangle( points[0], points[3] ).size().asSIZEL();
	}

	return result;
}

/*------------------------------- SetTextColor ---------------------------------
| Set the text foreground color attribute using GpiSetAttrs.                   |
------------------------------------------------------------------------------*/
COLORREF SetTextColor( const IPresSpaceHandle& hps, COLORREF color ) {
	CHARBUNDLE
	bundle;

	COLORREF
	result;

	GpiQueryAttrs( hps, PRIM_CHAR, -1, &bundle );

	result = swapRedBlue( GpiQueryRGBColor( hps,
											LCOLOPT_REALIZED,
											bundle.lColor ) );

	bundle.lColor = ColorToIndex( hps, color );

	if( !GpiSetAttrs( hps, PRIM_CHAR, CBB_COLOR, 0, &bundle ) )
	{
		result = CLR_INVALID;
	}

	return result;
}

/*-------------------------------- GetObject -----------------------------------
| Return the BITMAP header which is obtained via GpiQueryBitmapParameters.     |
------------------------------------------------------------------------------*/
#undef BITMAPINFOHEADER
int GetObject( IBitmapHandle hbmp, int count, void *p ) {
	int
	result;

	BITMAP
	&bmp = *(BITMAP*)p;

	BITMAPINFOHEADER
	info = { sizeof info};

	if( GpiQueryBitmapParameters( hbmp, &info ) )
	{
		// Copy data to result BITMAP structure.
		bmp.bmWidth     = info.cx;
		bmp.bmHeight    = info.cy;
		bmp.bmPlanes    = info.cPlanes;
		bmp.bmBitsPixel = info.cBitCount;

		result = sizeof( BITMAP );
	}
	else
	{
		result = 0;
	}

	return result;
}

/*--------------------- GetModuleHandle ------------------------
| This function is implemented using DosQueryModuleHandle.     |
--------------------------------------------------------------*/
HMODULE GetModuleHandle( const char *name ) {
	HMODULE
	result = 0;

	unsigned long
	rc = DosQueryModuleHandle( name, &result );

	return result;
}

/*-------------------------- SaveDC ----------------------------
| Implement directly, using GpiSavePS.                         |
--------------------------------------------------------------*/
int SaveDC( const IPresSpaceHandle& hps ) {
	int
	result = GpiSavePS( hps );

	return result;
}

/*------------------------ RestoreDC ---------------------------
| Implement directly, using GpiRestorePS.                      |
--------------------------------------------------------------*/
int RestoreDC( const IPresSpaceHandle& hps, int id ) {
	int
	result = GpiRestorePS( hps, id );

	return result;
}

/*------------------------- GetPixel ---------------------------
| We get the requested pixel using GpiQueryPel.  That          |
| returns a color index that we then convert to RGB.           |
--------------------------------------------------------------*/
COLORREF GetPixel( const IPresSpaceHandle& hps, int x, int y ) {
	COLORREF
	result = CLR_INVALID;

	POINTL
	pt = { x, y};

	long
	rgb,
	index = GpiQueryPel( hps, &pt );

	if( index >= 0 )
	{
		rgb = GpiQueryRGBColor( hps, LCOLOPT_REALIZED, index );
		result = rgb; // ?Are the red, green, blue bytes in the right place?
	}

	return result;
}

/*------------------------- SetPixel ---------------------------
| We take the specified color, convert to an index, and        |
| set it using GpiSetPel.                                      |
--------------------------------------------------------------*/
int SetPixel( const IPresSpaceHandle& hps, int x, int y, COLORREF color ) {
	int
	result;

	long
	index = GpiQueryColorIndex( hps, 0, color );

	POINTL
	pt = { x, y};

	// Save current color/mix.
	int
	id = SaveDC( hps );

	// Set color/mix to achieve desired result.
	GpiSetColor( hps, index );
	GpiSetMix  ( hps, FM_OVERPAINT );

	// Draw the pixel.
	result = GpiSetPel( hps, &pt );

	// Restore the color/mix.
	RestoreDC( hps, id );

	return result;
}

/*------------------ CreateCompatibleBitmap --------------------
| We create a bitmap using GpiCreateBitmap.  We query the      |
| formats supported by the device to find out what color       |
| depth is required.                                           |
--------------------------------------------------------------*/
IBitmapHandle CreateCompatibleBitmap( const IPresSpaceHandle& hps,
									  int width,
									  int height ) {
	IBitmapHandle
	result;

	// Get best bitmap format for the device with which the
	// argument PS is associated.
	LONG
	format[2];
	BOOL
	rc = GpiQueryDeviceBitmapFormats( hps, 2, format );

	// Build a bitmap info header using this format and the argument
	// width and height.
	BITMAPINFOHEADER2
	hdr = { sizeof hdr};

	// Fill in all the info header fields.
	hdr.cx              = width;
	hdr.cy              = height;
	hdr.cPlanes         = format[0];
	hdr.cBitCount       = format[1];
	hdr.ulCompression   = BCA_UNCOMP;
	hdr.cbImage         = 0;
	hdr.cxResolution    = 0;
	hdr.cyResolution    = 0;
	hdr.cclrUsed        = 0;
	hdr.cclrImportant   = 0;
	hdr.usUnits         = BRU_METRIC;
	hdr.usReserved      = 0;
	hdr.usRecording     = BRA_BOTTOMUP;
	hdr.usRendering     = BRH_NOTHALFTONED;
	hdr.cSize1          = 0;
	hdr.cSize2          = 0;
	hdr.ulColorEncoding = BCE_RGB;
	hdr.ulIdentifier    = 0;

	// Create the bitmap.
	result = GpiCreateBitmap( hps, &hdr, 0, 0, 0 );

	return result;
}

/*-------------------------- BitBlt ----------------------------
| Build an array of points describing the source/target        |
| rectangles and do a GpiBitBlt.                               |
--------------------------------------------------------------*/
int BitBlt( const IPresSpaceHandle& hpsDest, int xDest, int yDest,
			int width, int height,
			const IPresSpaceHandle& hpsSrc, int xSrc, int ySrc,
			DWORD rop ) {
	int
	result;

	IRectangle
	destination = IRectangle( IPoint( xDest, yDest ),
							  ISize(   width,   height   )   ),
				  source      = IRectangle( IPoint( xSrc, ySrc ),
											ISize( width, height ) );

	// Array of points describing src/dest for Gpi.
	POINTL
	points[] = { { destination.left(), destination.bottom()},
		{ destination.width(), destination.height()},
		{ source.left(), source.bottom()}};

	unsigned long
	rc = GpiBitBlt( hpsDest, hpsSrc, 3, points, rop, BBO_IGNORE );

	result = ( rc != GPI_ERROR );

	return result;
}

/*------------------------ ReleaseDC ---------------------------
| "Release" the argument PS using GpiDestroyPS.                |
--------------------------------------------------------------*/
int ReleaseDC( INativeWindowHandle, const IPresSpaceHandle& hps )
{
	int
	result = WinReleasePS( hps );
	
	return result;
}

/*------------------------ GetDIBits ---------------------------
| Get the bits for the argument bitmap via GpiQueryBitmapBits. |
--------------------------------------------------------------*/
int GetDIBits( const IPresSpaceHandle& hps, IBitmapHandle bmp,
			   unsigned int start, unsigned int num,
			   void *buffer,
			   LPBITMAPINFO info,
			   unsigned int colorMode ) {
	int
	result;

	result = GpiQueryBitmapBits( hps,
								 start,
								 num,
								 (PBYTE)buffer,
								 (PBITMAPINFO2)info );

	return result;
}

/*---------------------- GetDeviceCaps -------------------------
  Issue corresponding DevQueryCaps call.  We only support
  a small number of
--------------------------------------------------------------*/
int GetDeviceCaps( const IPresSpaceHandle& hps, int valueId ) {
	int
	result;

	HDC
	hdc = GpiQueryDevice( hps );

	long
	value,
	lStart;

	// Map enum values to Gpi values.
	switch( valueId )
	{
		case HORZRES:
			lStart = CAPS_WIDTH;
			break;
		case VERTRES:
			lStart = CAPS_HEIGHT;
			break;
		case LOGPIXELSY:
			//lStart = CAPS_VERTICAL_RESOLUTION;
			lStart = CAPS_VERTICAL_FONT_RES;
			break;
		case LOGPIXELSX:
			//lStart = CAPS_HORIZONTAL_RESOLUTION;
			lStart = CAPS_HORIZONTAL_FONT_RES;
			break;
		case TECHNOLOGY:
			lStart =    CAPS_TECHNOLOGY;
			break;
	}

	// Get value.
	unsigned long
	rc = DevQueryCaps( hdc, lStart, 1, &value );
	// check for technology and if so, convert to windows return
	// in the case of a non supported device, return 7 (DT_UNKNOWN)
	// as there is no such return for Windows.
	if( lStart == CAPS_TECHNOLOGY )
	{
		switch(value)
		{
			case CAPS_TECH_VECTOR_PLOTTER:
				result = DT_PLOTTER;
				break;
			case CAPS_TECH_RASTER_DISPLAY:
				result = DT_RASDISPLAY;
				break;
			case CAPS_TECH_RASTER_PRINTER:
				result = DT_RASPRINTER;
				break;
			case CAPS_TECH_RASTER_CAMERA:
				result = DT_RASCAMERA;
				break;
			default:
				result = DT_UNKNOWN; // no such return for windows, so use it for all errors
				break;
		}
	}
	else if( lStart == CAPS_HORIZONTAL_RESOLUTION || lStart == CAPS_VERTICAL_RESOLUTION )
	{
		double temp = value;				// assign the value to our temp
		temp *= .0254;						// multiply it to convert to inches
		temp += .5;							// add .5 so when we round down we get correct value
		result = temp;						// assign our real value for windows to result
	}
	else
		result = value;

	return result;
}

/*----------------------- GetWindowDC --------------------------
| Use WinGetPS to get a PS for the argument window.            |
--------------------------------------------------------------*/
IPresSpaceHandle GetWindowDC( INativeWindowHandle hwnd ) {
	IPresSpaceHandle
	result = WinGetPS( hwnd ? hwnd : INativeWindowHandle( HWND_DESKTOP ) );

	return result;
}

/*-------------------- CreateCompatibleDC ----------------------
| Create a new HPS associated with a memory device context.    |
|                                                              |
| We get the device associated with the argument HPS and       |
| use it to open a memory HD.  A new HPS is then created       |
| associated with that memory DC.                              |
--------------------------------------------------------------*/
IPresSpaceHandle CreateCompatibleDC( const IPresSpaceHandle& hps ) {
	IPresSpaceHandle
	result;

	// Get device associated with source HPS.
	HDC
	hdc = GpiQueryDevice( hps );

	if( hdc == HDC_ERROR )
		hdc = 0;

	// Open a memory DC using the source hdc.
	HDC
	hdcMem = DevOpenDC( 0,
						OD_MEMORY,
						"*",
						0,
						0,
						hdc );

	if( hdcMem != DEV_ERROR )
	{
		// Create a PS associated with the new memory hdc.
		SIZEL
		dimensions = { 0, 0}; // Causes default to be used.
		ULONG
		opts = PU_PELS | GPIF_DEFAULT | GPIT_NORMAL  | GPIA_ASSOC;

		result = GpiCreatePS( 0, hdcMem, &dimensions, opts );

		if( result == GPI_ERROR )
		{
			result = 0;
		}
		//Associate the palette to the comaptible DC
		HPAL oldPal = GpiSelectPalette(hdc, NULL);
		GpiSelectPalette(hdcMem, oldPal);
		GpiSelectPalette(hdc, oldPal);

	}

	return result;
}

/*------------------------- DeleteDC ---------------------------
| This function is implemented directly using GpiDestroyPS.    |
--------------------------------------------------------------*/
int DeleteDC( const IPresSpaceHandle& hps )
{
	GpiSelectPalette(hps, NULL);
	int result = GpiDestroyPS( hps );

	return result;
}

/*------------------------ LoadBitmap --------------------------
| This function is implemented directly via WinLoadBitmap.     |
|                                                              |
| We convert the (purportedly char*) resource name to a        |
| numeric resource id.                                         |
--------------------------------------------------------------*/
#undef LoadBitmap
IBitmapHandle LoadBitmap( HMODULE           hmod,
						  const char       *resName,
						  const IPresSpaceHandle&  hps ) {
	IBitmapHandle
	result;

	unsigned long
	resId = (unsigned long)resName;

	result = GpiLoadBitmap( hps, hmod, resId, 0, 0 );

	if( result == GPI_ERROR )
	{
		result = 0;
	}

	return result;
}

BOOL GetTextMetrics( const IPresSpaceHandle& pres, LPTEXTMETRIC ptm )
{
	FONTMETRICS fm;

	bool fReturn = GpiQueryFontMetrics( pres, sizeof( FONTMETRICS ), &fm );
	if( fReturn)
	{
		ptm->tmHeight = fm.lMaxBaselineExt;
		ptm->tmAscent = fm.lMaxAscender;
		ptm->tmDescent = fm.lMaxDescender;
		ptm->tmInternalLeading = fm.lInternalLeading;
		ptm->tmExternalLeading = fm.lExternalLeading;
		ptm->tmAveCharWidth = fm.lAveCharWidth;
		ptm->tmMaxCharWidth = fm.lMaxCharInc;
		ptm->tmWeight = fm.usWeightClass;
		ptm->tmOverhang = 0;
		ptm->tmDigitizedAspectX = 0;
		ptm->tmDigitizedAspectY = 0;
		ptm->tmFirstChar = fm.sFirstChar;
		ptm->tmLastChar = fm.sLastChar;
		ptm->tmDefaultChar = fm.sDefaultChar;
		ptm->tmBreakChar = fm.sBreakChar;
		ptm->tmItalic = 0;
		ptm->tmUnderlined = fm.lUnderscoreSize;
		ptm->tmStruckOut = fm.lStrikeoutSize;
		ptm->tmPitchAndFamily = fm.sFamilyClass;
		ptm->tmCharSet = 0;
	}
	return( fReturn );
}

#if 0// No longer needed because changes are made to IPresSpaceHandle to pass a windowhandle
/*
 * Function Name:getDCRectSize
 *
 * Parameters: IPresSpaceHandle
 *
 * Description:
 * the following call is used to get the clipping area.
 * this appears that it will work on OS/2 to determine how
 * big the window is. On Windows they have the luxury of
 * using WindowFromDC(), but OS/2 doesn't store the window
 * and pres space together, so it can't return the window.
 * Windows uses that to determine the size of the DC which
 * is passed into the constructor of IGdiDevice. This
 * function does appear to do as we want, and until there
 * is something to prove it doesn't work we will continue
 * to do so. ajd 1/27/1997
 *
 * Returns: size of pres space handle which equates to
 *              device context for Windows
 */
RECT getDCRectSize( const IPresSpaceHandle& myHPS )
{
	RECT  returnRect;
	RECTL rclWork;
	LONG  lComplex = 0L;

	lComplex = GpiQueryClipBox( myHPS, &rclWork );

	returnRect.left   = rclWork.xLeft;
	returnRect.bottom = rclWork.yTop;
	returnRect.right  = rclWork.xRight;
	returnRect.top    = rclWork.yBottom;

	return returnRect;
}
#endif // 0

/*
 *
 *Function Name: MulDiv
 *Parameters: m1, 32 bit signed integer
 *            m2, 32 bit signed integer
 *            d , 32 bit signed integer
 *Desciption:
 *This is a function emulates a window function MulDiv which multiplies
 *two 32-bit values and then divides the 64 bit result by a third 32 bit
 *value. The return value is rounded up or down to the nearest integer.
 *
 */

int MulDiv(int m1, int m2, int d)
{
	return (int) (0.5 + ((double)m1 * (double)m2)/d);
}

void DeletePenList( void )
{
	if( !ListIsEmpty( pAnchor ) )
	{
		IPrimalLock lock;
		PPENLIST ppWork;
		ppWork = ListTail( pAnchor );

		do
		{			
			ListDelete( ppWork, list );
			if( ppWork )
				free( ppWork );
		}
		while( ( (	ppWork = (PPENLIST) ListPrior( ppWork, list )) != ListRoot( ( pAnchor ) ) ) );
	}

	if( !ListIsEmpty( pbAnchor ) )
	{
		IPrimalLock lock;
		PBRUSHLIST pbWork;
		pbWork = ListTail( pbAnchor );

		do
		{
			ListDelete( pbWork, list );
			if( pbWork )
				free( pbWork );
		}
        while( ( (	pbWork = (PBRUSHLIST) ListPrior( pbWork, list )) != ListRoot( ( pbAnchor ) ) ) );
	}
	fPListInitialized = false;
}

void DeleteBrushList( void )
{
	if( !ListIsEmpty( bAnchor ) )
	{
		IPrimalLock lock;
		PBRUSHLIST pbWork;
		pbWork = ListTail( bAnchor );
	
		do
		{
			ListDelete( pbWork, list );
			if( pbWork )
				free( pbWork );
		}
		while( ( (	pbWork = (PBRUSHLIST) ListPrior( pbWork, list )) != ListRoot( ( bAnchor ) ) ) );
	}
	fBListInitialized = false;
}



/*-------------------------------- CreateFont2 ---------------------------------
| We implement this function by returning an "HFONT" (FATTRS structure)        |
| filled in according to the myriad font attributes pass as arguments.         |
|                                                                              |
| Note that there is a vast discrepancy between Gpi and Gdi in this arena.     |
------------------------------------------------------------------------------*/
HFONT CreateFont2( const IPresSpaceHandle& fHDC, int height, int width,
				   int escapement, int orientation,
				   int weight, DWORD italic, DWORD underline, DWORD strikeout,
					 DWORD outline,
					 DWORD bitmap,
				   DWORD charSet,
				   DWORD precision, DWORD clipPrecision, DWORD quality,
				   DWORD pitchAndFamily,
                   const char *faceName )
{
	HFONT  fat; //font attribute

	// get a non const char*
   char theFaceName[30];
   strcpy(theFaceName, faceName);

    if ( bitmap )
      // for bitmap font, the height is actually in pointsize
   	  fat = checkForBitmapMatch(fHDC, height, theFaceName);
    else{
	   // for non_bitmap font, the height is in pixel size
	  fat = checkForTransformableMatch(fHDC, height, outline, theFaceName);
      fat->lAveCharWidth   = abs(width);
	}
	
  if( weight )
		fat->fsSelection |= FATTR_SEL_BOLD;

	if( italic )
		fat->fsSelection |= FATTR_SEL_ITALIC;

	if( underline )
		fat->fsSelection |= FATTR_SEL_UNDERSCORE;

	if( strikeout )
		fat->fsSelection |=    FATTR_SEL_STRIKEOUT;
 	

	long lMatch = GpiCreateLogFont( fHDC, 0, 99L, fat );

	// must set the id 99L in order to use the created font in current presentation space
    GpiSetCharSet( fHDC, 99L );


	SIZEF box;

	if(fat->lMaxBaselineExt > 0)
	{
		box.cx = box.cy = fat->lMaxBaselineExt << 16;

		GpiSetCharBox( fHDC, &box );
	}
	else
	{
		BOOL fReturn = FALSE;
		box.cx = box.cy = 12L << 16;		// set for 12 as default
		fReturn = GpiSetCharBox( fHDC, &box );
	}
	

	long lDirection = 0L; // set direction of text
	switch(escapement)
	{
		case 900:
			lDirection = CHDIRN_BOTTOMTOP;
			break;
		case 1800:
			lDirection = CHDIRN_RIGHTLEFT;
			break;
		case 2700:
			lDirection = CHDIRN_TOPBOTTOM;
			break;
		default:
			lDirection = CHDIRN_LEFTRIGHT;
			break;
	}
	BOOL rc = GpiSetCharDirection( fHDC, lDirection );

	return fat;
}



// First look for an exact matched bitmap font. If an exact match can not be found,
// then look for a closest matched bitmap font
HFONT  checkForBitmapMatch (const IPresSpaceHandle& hps,int hPointSize, char* faceName)
{
   long lNumFonts;
   long lReqFonts = 0;
   long lBitmapMatch = 0;
   long lNextBitmap = 0;
   long lHowClose = 99999;
   unsigned long
     Index = 0,
     BitmapIndex = 0,
     NextBitmapIndex = 0;

   if(hPointSize < 0)
	    hPointSize = -hPointSize;

   if(hPointSize == 0)
	   hPointSize = 12;

   lNumFonts = GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE,
                  (PSZ)faceName, &lReqFonts, 0L, 0);

   if ( lNumFonts <= 0 ){
	   if (strcmp(faceName, (char*)"Courier"))
		 {  // use "Courier" as default	
        strcpy(faceName, (char*)"Courier"); 				
        lNumFonts = GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE,
                  (PSZ)faceName, &lReqFonts, 0L, 0);
	   }
	 }


   FONTMETRICS* pfm;
   pfm = new FONTMETRICS[lNumFonts];

   GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE, (PSZ)faceName,
         &lNumFonts, sizeof(FONTMETRICS), pfm);

   long lHorzRes, lVertRes;
   HDC hdc = GpiQueryDevice(hps);
	 DevQueryCaps(hdc, CAPS_HORIZONTAL_FONT_RES, 1L, &lHorzRes);
   DevQueryCaps(hdc, CAPS_VERTICAL_FONT_RES, 1L, &lVertRes);

	
   for (Index = 0; Index < lNumFonts ; Index++)
   {
      if (!(pfm[Index].fsDefn & FM_DEFN_OUTLINE))
			{
				// Check  device resolution and pointSize for bitmap font.
	      if ((pfm[Index].sXDeviceRes == lHorzRes) &&
	          (pfm[Index].sYDeviceRes == lVertRes))
	      {
	        if (pfm[Index].sNominalPointSize == (hPointSize*10))
	        {
	           lBitmapMatch = pfm[Index].lMatch;
	           BitmapIndex = Index;
	        }
	        else
	        {
	           long currentHowClose =
	               abs((hPointSize*10) - pfm[Index].sNominalPointSize);
	           if (!lBitmapMatch &&
	               currentHowClose <= lHowClose)
	           {
	               lNextBitmap = pfm[Index].lMatch;
	               NextBitmapIndex = Index;
	               lHowClose = currentHowClose;
	           } /* endif */
	        }
	      }
		  }		
   } /* endfor */


   HFONT  result = new FATTRS;
   memset( result, 0, sizeof( FATTRS ) );		

	 // We'll reset Index below to the position in pfm that's the best match.
	 // First look for an exact matched bitmap font.
	 if (lBitmapMatch) {
     result->lMatch = lBitmapMatch;
     result->fsFontUse = result->fsFontUse & ~FATTR_FONTUSE_OUTLINE;
     Index = BitmapIndex;
	 }
	 else {// If no exact match is found, then look for a closest matched bitmap font
        result->lMatch = lNextBitmap;
        result->fsFontUse = result->fsFontUse & ~FATTR_FONTUSE_OUTLINE;
        Index = NextBitmapIndex;
		} /* endif */
		


      // Get the FONTMETRICS of the font that we found above
	  FONTMETRICS *fontMet= new FONTMETRICS;
    memcpy(fontMet, &pfm[Index], sizeof(FONTMETRICS));

    result->usRecordLength = sizeof( FATTRS );
	
	  // copy from FONTMETRICS  to a FATTR
	  result->idRegistry = fontMet->idRegistry;
	  result->usCodePage = fontMet->usCodePage;
			
	  result->lMaxBaselineExt = fontMet->lMaxBaselineExt;
	  result->lAveCharWidth = fontMet->lAveCharWidth;
    strncpy( result->szFacename, fontMet->szFacename, FACESIZE );

		delete fontMet;
    delete []pfm;
    return result;
}


//Find a transformable font match
HFONT  checkForTransformableMatch (const IPresSpaceHandle& hps,int height, DWORD outline, char* faceName)
{

   long lNumFonts;
   long lReqFonts = 0;

   lNumFonts = GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE,
				   (PSZ)faceName, &lReqFonts, 0L, 0);

   if ( lNumFonts <= 0 )
	    strcpy(faceName, (char*)"Courier"); 				
        	
   HFONT  result = new FATTRS;
   memset( result, 0, sizeof( FATTRS ) );
	
	 result->usRecordLength = sizeof( FATTRS );
	
	 if ( outline )
	    // Note: this is a key setting for outline font(font which is hollowed, not filled).
 			result->fsSelection |= FATTR_SEL_OUTLINE;

   result->lMatch = 0;

	 //?? It seems to me that FATTR_FONT_TRANSFORMABLE has the same effect as FATTR_FONTUSE_OUTLINE
   //result->fsFontUse = result->fsFontUse | FATTR_FONTUSE_TRANSFORMABLE;
  result->fsFontUse = result->fsFontUse | FATTR_FONTUSE_OUTLINE;
	      															
  result->lMaxBaselineExt = abs(height);

  strncpy( result->szFacename, faceName, FACESIZE );
  return result;
}


int DeleteObjectOS ( HPEN hPen, const IPresSpaceHandle& hps ) {
   DeleteObject2( hPen, hps );    // moved to winemul.cpp - XZ
   return 1;
}

int DeleteObjectOS ( HBRUSH hBrush, const IPresSpaceHandle& hps ) {
   DeleteObject2( hBrush, hps );  // moved to winemul.cpp - XZ
   return 1;
}

int DeleteObjectOS ( IBitmapHandle hbm, const IPresSpaceHandle& ) {
  return (int)GpiDeleteBitmap( hbm );
}

int DeleteObjectOS ( HFONT hFont, const IPresSpaceHandle& ) {
  delete hFont;
  return 1;
}

// From now on, uses of HRGN must really be IRegionHandles.
// Otherwise, DeleteObject( hrgn ) is ambiguous.
int DeleteObjectOS ( IRegionHandle hrgn, const IPresSpaceHandle& hps ) {
  return (int)GpiDestroyRegion( hps, hrgn );
}
            										
