/*
******************************************************************************
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright Taligent, Inc.,  1996                                        *
*   (C) Copyright International Business Machines Corporation,  1997           *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
********************************************************************************
||
||  File:       pixbuf.cpp
||
||  What:       Routines which stores the image data read in both original
||                              data and also in native image format
||
||              Copyright(C) 1996 Taligent, Inc. All rights reserved.
||
*/

// Revision: 22 1.27.2.4 source/albert/graph2d/pixbuf.cpp, 2d, ioc.v400, 001006 

#define INIT_BUFFER ((iBYTE *)0xFFFFFFFF)

#ifdef IC_PM
#define INCL_WINSYS
#define INCL_WINERRORS
#define INCL_GPIBITMAPS
#define INCL_DOSPROCESS
#define INCL_GPI
#include <winemul.h>
#endif //IC_PM

#include <ixdc.hpp>
#include <pixbuf.hpp>

#include <grassert.hpp>

#include <istring.hpp>
#include <itrace.hpp>
#include <iprimlck.hpp>
#include <grstrmod.hpp>

#include <ibmpdata.hpp>
#include <ibmpstat.hpp>


// Segment definitions
#ifdef IC_PAGETUNE
#define _PIXBUF_CPP_
#include <ipagetun.h>
#endif

StreamableDefinitionsMacro(IPixelBuffer, gGraphicsStreamModule);

#ifdef IC_MOTIF
#define FLIPEDY(yPOS) (yPOS)

//In NT we maintain the data in BGR format and that in AIX is RGB
#define RGBA2IBASECOLOR(_R, _G, _B, _A)\
	IBaseColor((iBYTE)(_R), (iBYTE)(_G), (iBYTE)(_B), (iBYTE)(_A))

#define IBASECOLOR2RGBA(_rgba, _ibasecolor)\
{\
	(_rgba).R = (_ibasecolor).redMix();\
	(_rgba).G = (_ibasecolor).greenMix();\
	(_rgba).B = (_ibasecolor).blueMix();\
	(_rgba).A = (_ibasecolor).opacity();\
}

#define IBASECOLOR2RGB(_rgb, _ibasecolor)\
{\
	(_rgb).R = (_ibasecolor).redMix();\
	(_rgb).G = (_ibasecolor).greenMix();\
	(_rgb).B = (_ibasecolor).blueMix();\
}

#define IBASECOLOR2R5G6B5(_rgb, _ibasecolor)\
{\
	(_rgb).R = (_ibasecolor).redMix();\
	(_rgb).G = (_ibasecolor).greenMix();\
	(_rgb).B = (_ibasecolor).blueMix();\
}

#else
#define FLIPEDY(yPOS)	(fHeight - 1 - (yPOS))

#define RGBA2IBASECOLOR(_R, _G, _B, _A)\
	IBaseColor((iBYTE)(_B), (iBYTE)(_G), (iBYTE)(_R), (iBYTE)(_A))

#define IBASECOLOR2RGBA(_rgba, _ibasecolor)\
{\
	(_rgba).B = (_ibasecolor).redMix();\
	(_rgba).G = (_ibasecolor).greenMix();\
	(_rgba).R = (_ibasecolor).blueMix();\
	(_rgba).A = (_ibasecolor).opacity();\
}

#define IBASECOLOR2RGB(_rgb, _ibasecolor)\
{\
	(_rgb).B = (_ibasecolor).redMix();\
	(_rgb).G = (_ibasecolor).greenMix();\
	(_rgb).R = (_ibasecolor).blueMix();\
}

#define IBASECOLOR2R5G6B5(_rgb, _ibasecolor)\
{\
	(_rgb).B = (_ibasecolor).redMix();\
	(_rgb).G = (_ibasecolor).greenMix();\
	(_rgb).R = (_ibasecolor).blueMix();\
}

#endif //IC_MOTIF


#define GETBIT(Data, bit) \
                (iBYTE)(((Data) & (0x1 << (7-(bit)))) >> (7-(bit)))

#define SETBIT(Out, Data, bit) \
                ((Out) = (iBYTE)(((Out) & (~(0x1 << (7-(bit))))) | (((Data) & 0x1) << (7-(bit)))))

#define GETNIBBLE(Data, od) \
                (((Data) & ((od) ? 0x0F : 0xF0)) >> ((od) ? 0 : 4))

#define SETNIBBLE(Out, Data, od) \
                ((Out) = (((Out) & ((od) ? 0xF0 : 0x0F)) | ((Data) << ((od) ? 0 : 4))))

#define SASGC


#ifndef IC_MOTIF


#ifndef SASGC
static HDC getDC=NULL;

static HDC getDefaultSystemDC(){
	if (getDC == NULL) {
        IPrimalLock lock;
        if (getDC == NULL) {
		getDC = GetDC(0);
	}
	}
	return(getDC);
}

#endif //SASGC
#endif //IC_MOTIF

#ifdef IC_WIN

#define DEFAUT_SYSTEM_COLORS 20

//on as system palette of Windows with 256 colors
iBYTE DefaultSystemColors[DEFAUT_SYSTEM_COLORS][3] =
{
//first 10 system colors [0 -> 9]
        {0 , 0 , 0},
        {128 , 0 , 0},
        {0 , 128 , 0},
        {128 , 128 , 0},
        {0 , 0 , 128},
        {128 , 0 , 128},
        {0 , 128 , 128},
        {192 , 192 , 192},
        {192 , 220 , 192},
        {166 , 202 , 240},
//
//      The rest 236 user defined colors.
//
//last 10 system colors  [246 -> 255]
        {255 , 251 , 240},
        {160 , 160 , 164},
        {128 , 128 , 128},
        {255 , 0 , 0},
        {0 , 255 , 0},
        {255 , 255 , 0},
        {0 , 0 , 255},
        {255 , 0 , 255},
        {0 , 255 , 255},
        {255 , 255 , 255}
};
#endif //IC_WIN

iBYTE DefaultColorTable[256][3] =
{
	{0 , 0 , 0},
	{255 , 255 , 255},
	{0 , 0 , 51},
	{0 , 0 , 102},
	{0 , 0 , 153},
	{0 , 0 , 204},
	{0 , 0 , 255},
	{0 , 51 , 0},
	{0 , 51 , 51},
	{0 , 51 , 102},
	{0 , 51 , 153},
	{0 , 51 , 204},
	{0 , 51 , 255},
	{0 , 102 , 0},
	{0 , 102 , 51},
	{0 , 102 , 102},
	{0 , 102 , 153},
	{0 , 102 , 204},
	{0 , 102 , 255},
	{0 , 153 , 0},
	{0 , 153 , 51},
	{0 , 153 , 102},
	{0 , 153 , 153},
	{0 , 153 , 204},
	{0 , 153 , 255},
	{0 , 204 , 0},
	{0 , 204 , 51},
	{0 , 204 , 102},
	{0 , 204 , 153},
	{0 , 204 , 204},
	{0 , 204 , 255},
	{0 , 255 , 0},
	{0 , 255 , 51},
	{0 , 255 , 102},
	{0 , 255 , 153},
	{0 , 255 , 204},
	{0 , 255 , 255},
	{51 , 0 , 0},
	{51 , 0 , 51},
	{51 , 0 , 102},
	{51 , 0 , 153},
	{51 , 0 , 204},
	{51 , 0 , 255},
	{51 , 51 , 0},
	{51 , 51 , 51},
	{51 , 51 , 102},
	{51 , 51 , 153},
	{51 , 51 , 204},
	{51 , 51 , 255},
	{51 , 102 , 0},
	{51 , 102 , 51},
	{51 , 102 , 102},
	{51 , 102 , 153},
	{51 , 102 , 204},
	{51 , 102 , 255},
	{51 , 153 , 0},
	{51 , 153 , 51},
	{51 , 153 , 102},
	{51 , 153 , 153},
	{51 , 153 , 204},
	{51 , 153 , 255},
	{51 , 204 , 0},
	{51 , 204 , 51},
	{51 , 204 , 102},
	{51 , 204 , 153},
	{51 , 204 , 204},
	{51 , 204 , 255},
	{51 , 255 , 0},
	{51 , 255 , 51},
	{51 , 255 , 102},
	{51 , 255 , 153},
	{51 , 255 , 204},
	{51 , 255 , 255},
	{102 , 0 , 0},
	{102 , 0 , 51},
	{102 , 0 , 102},
	{102 , 0 , 153},
	{102 , 0 , 204},
	{102 , 0 , 255},
	{102 , 51 , 0},
	{102 , 51 , 51},
	{102 , 51 , 102},
	{102 , 51 , 153},
	{102 , 51 , 204},
	{102 , 51 , 255},
	{102 , 102 , 0},
	{102 , 102 , 51},
	{102 , 102 , 102},
	{102 , 102 , 153},
	{102 , 102 , 204},
	{102 , 102 , 255},
	{102 , 153 , 0},
	{102 , 153 , 51},
	{102 , 153 , 102},
	{102 , 153 , 153},
	{102 , 153 , 204},
	{102 , 153 , 255},
	{102 , 204 , 0},
	{102 , 204 , 51},
	{102 , 204 , 102},
	{102 , 204 , 153},
	{102 , 204 , 204},
	{102 , 204 , 255},
	{102 , 255 , 0},
	{102 , 255 , 51},
	{102 , 255 , 102},
	{102 , 255 , 153},
	{102 , 255 , 204},
	{102 , 255 , 255},
	{153 , 0 , 0},
	{153 , 0 , 51},
	{153 , 0 , 102},
	{153 , 0 , 153},
	{153 , 0 , 204},
	{153 , 0 , 255},
	{153 , 51 , 0},
	{153 , 51 , 51},
	{153 , 51 , 102},
	{153 , 51 , 153},
	{153 , 51 , 204},
	{153 , 51 , 255},
	{153 , 102 , 0},
	{153 , 102 , 51},
	{153 , 102 , 102},
	{153 , 102 , 153},
	{153 , 102 , 204},
	{153 , 102 , 255},
	{153 , 153 , 0},
	{153 , 153 , 51},
	{153 , 153 , 102},
	{153 , 153 , 153},
	{153 , 153 , 204},
	{153 , 153 , 255},
	{153 , 204 , 0},
	{153 , 204 , 51},
	{153 , 204 , 102},
	{153 , 204 , 153},
	{153 , 204 , 204},
	{153 , 204 , 255},
	{153 , 255 , 0},
	{153 , 255 , 51},
	{153 , 255 , 102},
	{153 , 255 , 153},
	{153 , 255 , 204},
	{153 , 255 , 255},
	{204 , 0 , 0},
	{204 , 0 , 51},
	{204 , 0 , 102},
	{204 , 0 , 153},
	{204 , 0 , 204},
	{204 , 0 , 255},
	{204 , 51 , 0},
	{204 , 51 , 51},
	{204 , 51 , 102},
	{204 , 51 , 153},
	{204 , 51 , 204},
	{204 , 51 , 255},
	{204 , 102 , 0},
	{204 , 102 , 51},
	{204 , 102 , 102},
	{204 , 102 , 153},
	{204 , 102 , 204},
	{204 , 102 , 255},
	{204 , 153 , 0},
	{204 , 153 , 51},
	{204 , 153 , 102},
	{204 , 153 , 153},
	{204 , 153 , 204},
	{204 , 153 , 255},
	{204 , 204 , 0},
	{204 , 204 , 51},
	{204 , 204 , 102},
	{204 , 204 , 153},
	{204 , 204 , 204},
	{204 , 204 , 255},
	{204 , 255 , 0},
	{204 , 255 , 51},
	{204 , 255 , 102},
	{204 , 255 , 153},
	{204 , 255 , 204},
	{204 , 255 , 255},
	{255 , 0 , 0},
	{255 , 0 , 51},
	{255 , 0 , 102},
	{255 , 0 , 153},
	{255 , 0 , 204},
	{255 , 0 , 255},
	{255 , 51 , 0},
	{255 , 51 , 51},
	{255 , 51 , 102},
	{255 , 51 , 153},
	{255 , 51 , 204},
	{255 , 51 , 255},
	{255 , 102 , 0},
	{255 , 102 , 51},
	{255 , 102 , 102},
	{255 , 102 , 153},
	{255 , 102 , 204},
	{255 , 102 , 255},
	{255 , 153 , 0},
	{255 , 153 , 51},
	{255 , 153 , 102},
	{255 , 153 , 153},
	{255 , 153 , 204},
	{255 , 153 , 255},
	{255 , 204 , 0},
	{255 , 204 , 51},
	{255 , 204 , 102},
	{255 , 204 , 153},
	{255 , 204 , 204},
	{255 , 204 , 255},
	{255 , 255 , 0},
	{255 , 255 , 51},
	{255 , 255 , 102},
	{255 , 255 , 153},
	{255 , 255 , 204},
	{17 , 17 , 17},
	{34 , 34 , 34},
	{68 , 68 , 68},
	{85 , 85 , 85},
	{119 , 119 , 119},
	{136 , 136 , 136},
	{170 , 170 , 170},
	{187 , 187 , 187},
	{221 , 221 , 221},
	{238 , 238 , 238},
	{128 , 77 , 77},
	{128 , 128 , 77},
	{77 , 128 , 77},
	{77 , 128 , 128},
	{77 , 77 , 128},
	{128 , 77 , 128},
	{179 , 128 , 128},
	{179 , 179 , 128},
	{128 , 179 , 128},
	{128 , 179 , 179},
	{128 , 128 , 179},
	{179 , 128 , 179},
	{230 , 179 , 179},
	{230 , 230 , 179},
	{179 , 230 , 179},
	{179 , 230 , 230},
	{179 , 179 , 230},
	{230 , 179 , 230},
	{230 , 212 , 212},
	{230 , 230 , 212},
	{212 , 230 , 212},
	{212 , 230 , 230},
	{212 , 212 , 230},
	{230 , 212 , 230},
	{255 , 242 , 242},
	{255 , 255 , 242},
	{242 , 255 , 242},
	{242 , 255 , 255},
	{242 , 242 , 255},
	{255 , 242 , 255}
};


//***********************************************************************
//IPixelBuffer

#ifdef IC_MOTIF

#define XImageCopy(_dst, _src)                                                                  \
{                                                                                                                               \
        if(((_dst)->width == (_src)->width) &&                                          \
                ((_dst)->height == (_src)->height) &&                                   \
                ((_dst)->depth == (_src)->depth)){                                              \
                (_dst)->width = (_src)->width;                                                  \
                (_dst)->height = (_src)->height;                                                \
                (_dst)->xoffset = (_src)->xoffset;                                              \
                (_dst)->format = (_src)->format;                                                \
                memcpy((_dst)->data, (_src)->data,                                              \
                                        (_src)->bytes_per_line * (_src)->height);       \
                (_dst)->byte_order = (_src)->byte_order;                                \
                (_dst)->bitmap_unit = (_src)->bitmap_unit;                              \
                (_dst)->bitmap_bit_order = (_src)->bitmap_bit_order;    \
                (_dst)->bitmap_pad = (_src)->bitmap_pad;                                \
                (_dst)->depth = (_src)->depth;                                                  \
                (_dst)->bytes_per_line = (_src)->bytes_per_line;                \
                (_dst)->bits_per_pixel = (_src)->bits_per_pixel;                \
                (_dst)->red_mask = (_src)->red_mask;                                    \
                (_dst)->green_mask = (_src)->green_mask;                                \
                (_dst)->blue_mask = (_src)->blue_mask;                                  \
        }                                                                                                                       \
}

#endif //IC_MOTIF


IPixelBuffer::~IPixelBuffer ()
{
        IFUNCTRACE_DEVELOP();
#ifdef IC_MOTIF

#if 0
	This is commented for the reason that the display would have been used
	by other objects and destroying this display leads to invalid objects
	which cannot be deleted. All X functions need display for interface.
	The display should have been refrence counted by some mechanism to
	overcome this problem - SAS.
		//This is true only for unittest case where UI is not inited
		if(bPrivateDisplay)
			IXDisplay::closeDisplay();
#endif

	if(fImage){
		delete [] fImage;
		fImage = NULL;
	}

        if( fMaskBitmap )
        {
               // destroy image
                XImage* xi;
                xi = (XImage*)(unsigned long)fMaskBitmap;
                XDestroyImage( xi );
	}

#else

		if (fBitmapHDC) {
			if (!DeleteDC(fBitmapHDC))
			GrafDeviceException("DeleteDC");
		}

        if( fBitmap )
        {
#ifdef IC_WIN
		//In Nt fImage is data storeage for bitmap so delete only bitmap
		DeleteObject(fBitmap);
#else
		GpiDeleteBitmap(fBitmap);
#endif //IC_WIN
		fBitmap = 0;
	}
#ifdef IC_PM
    	if(fImage){
			delete [] fImage;
			fImage = NULL;
		}
#endif
        if( fMaskBitmap )
        {
#ifdef IC_WIN
		DeleteObject(fMaskBitmap);
#else
		GpiDeleteBitmap(fMaskBitmap);
#endif
		fMaskBitmap = 0;
	}

#endif
}

void IPixelBuffer::buildMaskImage(const IBaseColor& )
{}

nativeImage IPixelBuffer::convertToNativeImage(const iBYTE *srcImg)
{ return NULL; }

unsigned long IPixelBuffer::createPatternImage()
{ return 0; }

IPixelBuffer* IPixelBuffer::clonePixelBuffer()
{ return 0; }

IRGBAColorArray* IPixelBuffer::colorTable()
{ return NULL; }

void IPixelBuffer::setColorTable(const IRGBAColorArray& ary )
{}

void IPixelBuffer::pixel(const IGPoint2D& p, IBaseColor& clr) const
{}

void IPixelBuffer::pixelRow(const IGPoint2D& p, IRGBAColorArray& ary) const
{}

IBaseColor* IPixelBuffer::pixelRect(const IGRect2D& r) const
{return NULL; }

void IPixelBuffer::setPixel(const IGPoint2D& p, const IBaseColor& Clr)
{}

void IPixelBuffer::setPixelRow(const IGPoint2D& p, const IRGBAColorArray&  ary)
{}

void IPixelBuffer::setPixelRect(const IGRect2D& r, const IBaseColor*  clr)
{}


IPixelBuffer* IPixelBuffer::createPixbufFromNativeImage(nativeImage natImg)
{
	int imageType;
        IFUNCTRACE_DEVELOP();
#ifdef IC_MOTIF
		imageType = natImg->depth;
#else
	BITMAP bm;
	if (!GetObject((HBITMAP)natImg, sizeof(bm), (LPSTR)&bm))
		GrafDeviceException("GetObject");

	imageType = (int)bm.bmBitsPixel;
#endif
IPixelBuffer *pixbuf=NULL;

	switch(imageType)
	{
	case 1:
		pixbuf = new IMonochromePixelBuffer(natImg);
		break;
	case 4:
		pixbuf = new ICLUT4PixelBuffer(natImg);
		break;
	case 8:
		pixbuf = new ICLUT8PixelBuffer(natImg);
		break;
	case 16:
		pixbuf = new IR5G6B5PixelBuffer(natImg);
		break;
	case 24:
		pixbuf = new IR8G8B8PixelBuffer(natImg);
		break;
	case 32:
		pixbuf = new IR8G8B8A8PixelBuffer(natImg);
		break;
	}
	return(pixbuf);
}

void IPixelBuffer::updateDeviceContext(nativeImage bitmap)
{
        IFUNCTRACE_DEVELOP();
#if defined(IC_MOTIF)
		if(bSyncXImageFromPixmap)
			syncXImageFromPixmap();
        // copy XImage into Pixmap
        XPutImage(((IXDC*)fBitmapHDC)->xDisplay,
                                ((IXDC*)fBitmapHDC)->xDrawable,
                                ((IXDC*)fBitmapHDC)->gc(),
                        bitmap, 0, 0, 0, 0, fWidth, fHeight);
#elif defined(IC_WIN)
        HDC hdcMem1  = CreateCompatibleDC (fBitmapHDC) ;
        HDC hdcMem2  = CreateCompatibleDC (fBitmapHDC) ;

        HBITMAP old_bit1 = (HBITMAP) SelectObject( hdcMem1, bitmap );
        HBITMAP old_bit2 = (HBITMAP) SelectObject( hdcMem2, fBitmap );
        StretchBlt (hdcMem2, 0, 0, fWidth, fHeight, hdcMem1,  0, 0, fWidth, fHeight, SRCCOPY) ;
        SelectObject(hdcMem1, old_bit1);
        SelectObject(hdcMem2, old_bit2);
        DeleteDC(hdcMem1);
        DeleteDC(hdcMem2);
#elif defined(IC_PM)
        IPresSpaceHandle hdcMem1  = CreateCompatibleDC (fBitmapHDC) ;
        IPresSpaceHandle hdcMem2  = CreateCompatibleDC (fBitmapHDC) ;

        HBITMAP old_bit1 = SelectObject(hdcMem1, bitmap);
        HBITMAP old_bit2 = SelectObject(hdcMem2, fBitmap);
        StretchBlt (hdcMem2, 0, 0, fWidth, fHeight, hdcMem1,  0, 0, fWidth, fHeight, SRCCOPY) ;
        SelectObject(hdcMem1, old_bit1);
        SelectObject(hdcMem2, old_bit2);
        DeleteDC(hdcMem1);
        DeleteDC(hdcMem2);
#endif
}

// Create a memory device context for the image geometry.
void IPixelBuffer::initializeDeviceContext()
{
        IFUNCTRACE_DEVELOP();
#ifdef IC_MOTIF

        Display *dpy = getDisplay();
        Pixmap pixmap = XCreatePixmap( dpy, DefaultRootWindow( dpy),
                                                fWidth, fHeight,
                                                DefaultDepth(dpy, DefaultScreen(dpy)));

        // construct new IPresSpaceHandle
        fBitmapHDC = IPresSpaceHandle( new IXDC(dpy, pixmap) );
#else
        // get the display device context for the screen
        HDC hdcDisplay = GetWindowDC(HWND_DESKTOP);
        if (hdcDisplay == 0)
                GrafDeviceException("GetWindowDC");

        fBitmapHDC = CreateCompatibleDC(hdcDisplay);
        if (!fBitmapHDC)
                GrafDeviceException("CreateCompatibleDC");

        if (fBitmapHDC)
        {
                // copy the Map Mode from hdcDisplay to hdcMem
                int mapMode = GetMapMode(hdcDisplay);
                SetMapMode(fBitmapHDC, mapMode);
        }

        if (!ReleaseDC(HWND_DESKTOP, hdcDisplay))
                GrafDeviceException("ReleaseDC");

#ifdef IC_PM
	if((fBitsPerPixel < 16) && IColorMap::hasColorMapSupport()){
		HPAL palette = (IColorMap::defaultColorMap()).nativeColorMap();
		GpiSelectPalette(fBitmapHDC, palette);
	}
#endif
#endif //IC_MOTIF
}


bool IPixelBuffer::isEqual (const IPixelBuffer& obj) const
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();

		obj.syncPixelBuffer();
			
        if ((fWidth == obj.fWidth) &&
                (fWidth == obj.fWidth) &&
                (fHeight == obj.fHeight) &&
                (xHot == obj.xHot) &&
                (yHot == obj.yHot) &&
                (fBitsPerPixel == obj.fBitsPerPixel) &&
                (fPixelStride == obj.fPixelStride) &&
                (srcImageFormat == obj.srcImageFormat) &&
                (fImage) && (obj.fImage) &&
				(memcmp(fImage, obj.fImage, (size_t)(fPixelStride * fHeight)) == 0))
                return true;

        return false;
}

void IPixelBuffer::syncPixelBuffer() const
{
        IFUNCTRACE_DEVELOP();
	if(!bSyncImage)
		return;

   ((IPixelBuffer*)this)->syncPixelBufferRect(IGRect2D(0, 0, fWidth, fHeight));
}

void IPixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        (long)srcImageFormat >>= toWhere;
        fWidth >>= toWhere;
        fHeight >>= toWhere;
        xHot >>= toWhere;
        yHot >>= toWhere;
        fBitsPerPixel >>= toWhere;
        fPixelStride >>= toWhere;
        (iBYTE) bImgUpdated >>= toWhere;
        (iBYTE) bSyncImage >>= toWhere;
        (iBYTE) bTransparencyEnabled >>= toWhere;
        transparencyClr >>= toWhere;
	for(unsigned long i=0; i<(fPixelStride * fHeight); i++)
		fImage[i] >>= toWhere;
}

void IPixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        long tmp;
        tmp <<= fromWhere;
        srcImageFormat = (IGImagePixelAccessor::EImageFormat)tmp;
        fWidth <<= fromWhere;
        fHeight <<= fromWhere;
        xHot <<= fromWhere;
        yHot <<= fromWhere;
        fBitsPerPixel <<= fromWhere;
        fPixelStride <<= fromWhere;
        iBYTE flag;
        flag <<= fromWhere;
        bImgUpdated = (bool)flag;
        flag <<= fromWhere;
        bSyncImage = (bool) flag;
        flag <<= fromWhere;
        bTransparencyEnabled  = (bool)flag;
        transparencyClr <<= fromWhere;
        fImage = new iBYTE[fHeight * fPixelStride];
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init
	for(unsigned long i=0; i<(fPixelStride * fHeight); i++)
		fImage[i] <<= fromWhere;
        initializeDeviceContext();
	convertToNativeImage();
}

IPixelBuffer::IPixelBuffer () :
        fRefCount(1),
        srcImageFormat(IGImagePixelAccessor::kUNKNOWN),
        fWidth(0),
        fHeight(0),
		xHot(0),
		yHot(0),
        fBitsPerPixel(0),
        fPixelStride(0),
        bImgUpdated(false),
        bSyncImage(false),
        bTransparencyEnabled(false),
        transparencyClr(0, 0, 0, 0xff),
        fBitmap(0),
        fMaskBitmap(0),
#ifdef IC_MOTIF
		bSyncXImageFromPixmap(false),
#endif //IC_MOTIF
        fImage(NULL)
{
        IFUNCTRACE_DEVELOP();
}

#ifdef IC_MOTIF
void IPixelBuffer::syncXImageFromPixmap()
{
        IFUNCTRACE_DEVELOP();
	if(bSyncXImageFromPixmap){
        Display *dpy = getDisplay();

        XImage *nImg = XGetImage(dpy, ((IXDC*)fBitmapHDC)->xDrawable, 0, 0,
                fWidth, fHeight, AllPlanes, ZPixmap);
		XImageCopy(fBitmap, nImg);
		XDestroyImage(nImg);
		bSyncXImageFromPixmap=false;
	}
}


void IPixelBuffer::matchXColor( XColor* clrs, int num, XColor* color)
{
    int cnt;
    int min=999; //max value
    int nearest_match=0;
	int diff;
    for (cnt = 0; cnt < num; cnt++)
    {
        diff =
            ABS(clrs[cnt].red-color->red) +
            ABS(clrs[cnt].green-color->green) +
            ABS(clrs[cnt].blue-color->blue);

        if (diff == 0){ //perfect match
            color->pixel = cnt;
            return;
        }

        if(min > diff){
            min = diff;
            nearest_match = cnt;
        }

    }
    color->pixel = nearest_match; //nearest match
}

#endif //IC_MOTIF

unsigned long IPixelBuffer::buildPatternImage(const IRGBAColorArray *clrAry)
{
        IFUNCTRACE_DEVELOP();
#if defined(IC_WIN)

        HBITMAP         phbm;

        HDC             hdc;
        HPALETTE        phpal;
        int             i;
	unsigned long 	nClr;

        struct
        {
                BITMAPINFOHEADER bmp;
                RGBQUAD rgbaClr[0x100];
        } bm;

        /* Got the data in memory, now make bitmap */

        memset(&(bm.bmp), 0, (size_t)(sizeof(bm.bmp)));
        bm.bmp.biSize     = sizeof(BITMAPINFOHEADER);
        bm.bmp.biWidth    = fWidth;
        bm.bmp.biHeight   = fHeight;
        bm.bmp.biBitCount = fBitsPerPixel;
        bm.bmp.biPlanes   = 1;

        if ( clrAry )
        {
		IR8G8B8A8Color *clrData = clrAry->fColors;
                nClr = clrAry->fNumberOfColors;
                for ( i = 0; i < nClr; i++ )
                {
                        bm.rgbaClr[i].rgbRed      = clrData[i].fRed;
                        bm.rgbaClr[i].rgbGreen    = clrData[i].fGreen;
                        bm.rgbaClr[i].rgbBlue     = clrData[i].fBlue;
                        bm.rgbaClr[i].rgbReserved = 0;
                }
        }

#ifdef SASGC
        if ( (hdc = GetDC(0)) == (HDC) NULL )
#else //SASGC
        if ( (hdc = getDefaultSystemDC()) == (HDC) NULL )
#endif //SASGC
                return 0; //MOD_ERR_HDC;

	//Does the display has a palette
        bool palEnb = IColorMap::hasColorMapSupport();

        /*handle 1bpp case*/
        if ( fBitsPerPixel == 1 )
        /*
        1bpp presentation spaces have a reset or background colour.
        They also have a contrast or foreground colour.
        When data is mapped into a 1bpp presentation space :-
        Data which is the reset colour, remains reset, and is stored as 0's.
        All other data becomes contrast, and is stored as 1's.
        The reset colour for 1bpp screen HPSs is white.
        I want 1's in the source data to become 1's in the HPS.
        We seem to have to reverse the ordering here to get the desired effect.
        */

        {
                static RGBQUAD rgba2Black = { 0x00, 0x00, 0x00, 0x00 };
                static RGBQUAD rgba2White = { 0xff, 0xff, 0xff, 0x00 };
                bm.rgbaClr[0] = rgba2Black; /* Contrast */
                bm.rgbaClr[1] = rgba2White; /* Reset */
        }

	phpal = (HPALETTE)(IColorMap::defaultColorMap().nativeColorMap());

        HPALETTE oldPal;

        if (phpal)
        {
                oldPal = SelectPalette(hdc, phpal, false);
                i=RealizePalette(hdc);
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"RealizePalette %d\n", i);
#endif
        }


        if ( (phbm = CreateDIBitmap( hdc, &(bm.bmp), CBM_INIT, fImage, (BITMAPINFO*)&bm, DIB_RGB_COLORS )) == (HBITMAP)NULL)
        {
#ifdef SASGC
                ReleaseDC(0, hdc );
#endif //SASGC
                return 0; //MOD_ERR_HBITMAP;
        }

        if (phpal != 0)
        {
                phpal = SelectPalette( hdc, oldPal, false );
		if(phpal)
			DeleteObject(phpal);
        }

#ifdef SASGC
        ReleaseDC(0, hdc );
#endif //SASGC
	return (unsigned long)phbm;

#elif defined(IC_PM) //IC_PM

	return((unsigned long)createNativeImage(NULL, clrAry));
#elif defined(IC_MOTIF) //IC_MOTIF
	return(0);
#endif //IC_MOTIF

}

nativeImage IPixelBuffer::createNativeImage(const iBYTE* srcImg, const IRGBAColorArray *clrAry)
{
        IFUNCTRACE_DEVELOP();
#if 0 //SAS
        int fd = open("sas1.dat", O_CREAT|O_WRONLY|O_BINARY, S_IREAD|S_IWRITE);
        if(fd > 0) {
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"File opened %d\n",fd);
#endif
                write(fd, (char *)fImage,
                        fPixelStride * fHeight);
                close(fd);
        } else {
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"File creation error\n");
#endif
        }
#endif //SAS

                bImgUpdated = false;

#if defined(IC_WIN)

        HBITMAP         phbm;

        phbm = createDIBSection(srcImg, clrAry);
	return phbm;

#elif defined(IC_PM) //IC_PM

        SIZEL           sizl;
        HDC             hdc;
        HBITMAP         phbm;
        HPS             hps;
        HAB             hab=0; //fixed value used also in old implementation
	int i;

        struct
        {
                BITMAPINFOHEADER2 bmp2;
                RGB2 argb2Color[0x100];
        } bm;

        /* Got the data in memory, now make bitmap */

        memset(&(bm.bmp2), 0, (size_t)(sizeof(bm.bmp2)));
        bm.bmp2.cbFix     = sizeof(BITMAPINFOHEADER2);
        bm.bmp2.cx        = fWidth;
        bm.bmp2.cy        = fHeight;
        bm.bmp2.cBitCount = fBitsPerPixel;
        bm.bmp2.cPlanes   = 1;

        if ( clrAry )
        {
                bm.bmp2.cclrUsed = clrAry->fNumberOfColors;
                bm.bmp2.cclrImportant = bm.bmp2.cclrUsed;
		IR8G8B8A8Color *clrData = clrAry->fColors;
                for ( i = 0; i < bm.bmp2.cclrUsed; i++ )
                {
                        bm.argb2Color[i].bRed      = clrData[i].fRed;
                        bm.argb2Color[i].bGreen    = clrData[i].fGreen;
                        bm.argb2Color[i].bBlue     = clrData[i].fBlue;
                        bm.argb2Color[i].fcOptions = 0;
                }
        }

        if ( (hdc = DevOpenDC(hab, OD_MEMORY, "*", 0L, (PDEVOPENDATA) NULL, (HDC) NULL)) == (HDC) NULL )
                return 0; //MOD_ERR_HDC;

        sizl.cx = fWidth;
        sizl.cy = fHeight;
        if ( (hps = GpiCreatePS(hab, hdc, &sizl, PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC)) == (HPS) NULL )
        {
                DevCloseDC(hdc);
                return 0; //MOD_ERR_HPS;
        }

	HPAL palette = (IColorMap::defaultColorMap()).nativeColorMap();
	GpiSelectPalette(hps, palette);

        if ( fBitsPerPixel == 1 ) /*handle 1bpp case*/
        /*
        1bpp presentation spaces have a reset or background colour.
        They also have a contrast or foreground colour.
        When data is mapped into a 1bpp presentation space :-
        Data which is the reset colour, remains reset, and is stored as 0's.
        All other data becomes contrast, and is stored as 1's.
        The reset colour for 1bpp screen HPSs is white.
        I want 1's in the source data to become 1's in the HPS.
        We seem to have to reverse the ordering here to get the desired effect.
        */

        {
                static RGB2 argb2Black = { 0x00, 0x00, 0x00 };
                static RGB2 argb2White = { 0xff, 0xff, 0xff };
                bm.argb2Color[0] = argb2Black; /* Contrast */
                bm.argb2Color[1] = argb2White; /* Reset */
        }

	const iBYTE *dataBuf = NULL;
	if(srcImg){
		if(srcImg != fImage) //if both not of same data
			fImage = new iBYTE[fPixelStride * fHeight];
		if (srcImg != INIT_BUFFER){
			dataBuf = srcImg;
			if(srcImg != fImage) //if both not of same data
				memcpy(fImage, srcImg, (fPixelStride * fHeight));
		} else if(srcImg == INIT_BUFFER) {
			memset(fImage, 0x0, (fPixelStride * fHeight));
			dataBuf = fImage;
		}
	} else
		dataBuf = fImage;
	


        if ( (phbm = GpiCreateBitmap(hps, &(bm.bmp2), CBM_INIT, (PBYTE)dataBuf, (BITMAPINFO2 *) &(bm.bmp2))) == (HBITMAP) NULL )
        {
		GpiSelectPalette(hps, NULL);
                GpiAssociate(hps, NULLHANDLE);
                GpiDeletePalette(palette);
                GpiDestroyPS(hps);
                DevCloseDC(hdc);
                return 0; //MOD_ERR_HBITMAP;
        }
        GpiSetBitmap(hps, (HBITMAP) NULL);
	GpiSelectPalette(hps, NULL);
        GpiAssociate(hps, NULLHANDLE);
        GpiDeletePalette(palette);
        GpiDestroyPS(hps);
        DevCloseDC(hdc);

	return phbm;

#else //IC_MOTIF

        XImage  *xi;
        int i, j;
        Display *dpy = getDisplay();
	int defDepth = DefaultDepth(dpy, DefaultScreen(dpy));
	Visual* defVisual = DefaultVisual(dpy, DefaultScreen(dpy));

	XColor clr_tmp;
	int *map_cmap=0x0;

	IR8G8B8A8Color *clrData=NULL;
	int noOfClrs=0;
	if ( clrAry ){
		noOfClrs = clrAry->fNumberOfColors;
		clrData = clrAry->fColors;
	}

	if (IColorMap::hasColorMapSupport()) {
		IColorMap* defColorMap = &(IColorMap::defaultColorMap());
		int total_count = defColorMap->numberOfColors();
		IR8G8B8A8Color *defClrData=defColorMap->fColors;

		//build the LUT
                if ( clrAry ){
			map_cmap = new int[noOfClrs];
                        for(i=0; i<noOfClrs; i++){
                                map_cmap[i] = defColorMap->colorIndex(clrAry->color(i));
                        }
                }
        }


	iBYTE tmp;
	unsigned long read_scan=0, write_scan=0, pixstride;

	if (IColorMap::hasColorMapSupport())
		pixstride = PIXELSTRIDE(fWidth, 8);
	else
		pixstride = PIXELSTRIDE(fWidth, 32);
        char *xiData = new char[fHeight * pixstride];
	R5G6B5 *rgb16;
	RGBClr *rgb24;
	RGBAClr *rgba;

	const iBYTE *dataBuf = NULL;
	if(srcImg){
		if(srcImg != fImage) //if both not of same data
			fImage = new iBYTE[fPixelStride * fHeight];
		if (srcImg != INIT_BUFFER){
			dataBuf = srcImg;
			if(srcImg != fImage) //if both not of same data
				memcpy(fImage, srcImg, (fPixelStride * fHeight));
		} else if(srcImg == INIT_BUFFER) {
			memset(fImage, 0x0, (fPixelStride * fHeight));
			dataBuf = fImage;
		}
	} else
		dataBuf = fImage;

	//if not creating an image with data then just init
	if((srcImg) && (srcImg == INIT_BUFFER)){
		memset(xiData, 0x0, (fHeight * pixstride));
	} else {
	if (!IColorMap::hasColorMapSupport()) { //assume to be 24 bit display
        RGBClr *cclr=0x0;
		if(fBitsPerPixel < 16){ //image data with colorTable
			cclr = new RGBClr[noOfClrs];
			for(i=0; i<noOfClrs; i++){
				cclr[i].R = clrData[i].fRed;
				cclr[i].G = clrData[i].fGreen;
				cclr[i].B = clrData[i].fBlue;
			}
		}

		RGBClr tmp24Clr;
		unsigned long *scan;
		for(i=0; i< fHeight; i++){
			read_scan = i*fPixelStride;
			write_scan = i*pixstride;

			scan = (unsigned long *) &xiData[write_scan];

			rgb16 = (R5G6B5 *) &fImage[read_scan];
			rgb24 = (RGBClr *) &fImage[read_scan];
			rgba = (RGBAClr *) &fImage[read_scan];

			for(j=0; j< fWidth; j++){

				switch( fBitsPerPixel){
					case 1:
						tmp = GETBIT(fImage[read_scan+(j>>3)], (j&0x7));
						tmp24Clr.R = cclr[tmp].R;
						tmp24Clr.G = cclr[tmp].G;
						tmp24Clr.B = cclr[tmp].B;
						break;
					case 4:
						tmp = GETNIBBLE(fImage[(read_scan+j/2)], (j&0x1));
						tmp24Clr.R = cclr[tmp].R;
						tmp24Clr.G = cclr[tmp].G;
						tmp24Clr.B = cclr[tmp].B;
						break;
					case 8:
						tmp = fImage[read_scan+j];
						tmp24Clr.R = cclr[tmp].R;
						tmp24Clr.G = cclr[tmp].G;
						tmp24Clr.B = cclr[tmp].B;
						break;
					case 16:
						tmp24Clr.R = rgb16[j].R;
						tmp24Clr.G = rgb16[j].G;
						tmp24Clr.B = rgb16[j].B;
					    break;
					case 24:
						tmp24Clr.R = rgb24[j].R;
						tmp24Clr.G = rgb24[j].G;
						tmp24Clr.B = rgb24[j].B;
					    break;
					case 32:
						tmp24Clr.R = rgba[j].R;
						tmp24Clr.G = rgba[j].G;
						tmp24Clr.B = rgba[j].B;
					    break;
				}
				scan[j]=IXDisplay::rgbToPixel(tmp24Clr.R, tmp24Clr.G, tmp24Clr.B);
			}
		}

		if(cclr)
			delete [] cclr;
	} else { // 8-bit mode
		unsigned long curColor=0,prevColor=0, prevIndex=0;
		IColorMap* defColorMap = &(IColorMap::defaultColorMap());

		//init for the black color
		prevIndex = defColorMap->colorIndex(IBaseColor(0,0,0));

		for(i=0; i< fHeight; i++){
			read_scan = i*fPixelStride;
			write_scan = i*pixstride;

			rgb16 = (R5G6B5 *) &fImage[read_scan];
			rgb24 = (RGBClr *) &fImage[read_scan];
			rgba = (RGBAClr *) &fImage[read_scan];

			for(j=0; j< fWidth; j++){

				switch( fBitsPerPixel){
					case 1:
						tmp = GETBIT(fImage[read_scan+(j>>3)], (j&0x7));

						xiData[write_scan+j] = map_cmap[(unsigned long)tmp];
						break;
					case 4:
						tmp = GETNIBBLE(fImage[(read_scan+j/2)], (j&0x1));
						xiData[write_scan+j] = map_cmap[(unsigned long)tmp];
						break;
					case 8:
						tmp = fImage[read_scan+j];
						xiData[write_scan+j] = map_cmap[(unsigned long)tmp];
						break;
					case 16:
						clr_tmp.red = rgb16[j].R;
						clr_tmp.green = rgb16[j].G;
						clr_tmp.blue = rgb16[j].B;

						curColor = IXDisplay::rgbToPixel(clr_tmp.red,
 								clr_tmp.green,
								clr_tmp.blue);

						if(prevColor == curColor){
							xiData[write_scan+j] = prevIndex;
						} else {
							prevIndex = defColorMap->colorIndex(IBaseColor(
												clr_tmp.red,
												clr_tmp.green,
												clr_tmp.blue));
							xiData[write_scan+j] = prevIndex;
							prevColor = curColor;
						}
						break;
					case 24:
						clr_tmp.red = rgb24[j].R;
						clr_tmp.green = rgb24[j].G;
						clr_tmp.blue = rgb24[j].B;

						curColor = IXDisplay::rgbToPixel(clr_tmp.red,
 								clr_tmp.green,
								clr_tmp.blue);

						if(prevColor == curColor){
							xiData[write_scan+j] = prevIndex;
						} else {
							prevIndex = defColorMap->colorIndex(IBaseColor(
												clr_tmp.red,
												clr_tmp.green,
												clr_tmp.blue));
							xiData[write_scan+j] = prevIndex;
							prevColor = curColor;
						}
						break;
					case 32:
						clr_tmp.red = rgba[j].R;
						clr_tmp.green = rgba[j].G;
						clr_tmp.blue = rgba[j].B;

						curColor = IXDisplay::rgbToPixel(clr_tmp.red,
 								clr_tmp.green,
								clr_tmp.blue);

						if(prevColor == curColor){
							xiData[write_scan+j] = prevIndex;
						} else {
							prevIndex = defColorMap->colorIndex(IBaseColor(
												clr_tmp.red,
												clr_tmp.green,
												clr_tmp.blue));
							xiData[write_scan+j] = prevIndex;
							prevColor = curColor;
						}
						break;
				} //switch
			} //width for loop
		} //height for loop
	} //depth check
	} //buffer init

        xi = XCreateImage(dpy, defVisual, defDepth, ZPixmap, 0,
                        xiData, fWidth, fHeight, 32,
                        pixstride);

#if 0 //SAS
XPutImage( dpy, win, DefaultGC(dpy, DefaultScreen(dpy)), xi, 0, 0,
                                        0, 0, fWidth, fHeight );
XFlush(dpy);
#endif //SAS

        if(xi == NULL)
                return 0;

                if ( clrAry )
			delete [] map_cmap;
        return xi;

#endif //IC_MOTIF

}

/*
This function is needed only in the case of os2 and AIX where in the
native image and pixelbuffer image data storage area are not same.
But in WINDOWS since we use CreateDIBSection for creating HBITMAP which makes
both the native image and pixel buffer storage area the same.
*/

void IPixelBuffer::updateNativeImage()
{
        IFUNCTRACE_DEVELOP();
#ifndef IC_WIN

	IRGBAColorArray *clrAry=NULL;

	if(fBitsPerPixel < 16)
	{
			clrAry = colorTable();
	}

	nativeImage natImg;

        natImg = createNativeImage(NULL, clrAry);
	
	if(natImg)
	{

#if defined(IC_PM)
// do copying of natImg on to fBitmap
        IPresSpaceHandle hdcMem1  = CreateCompatibleDC (fBitmapHDC) ;
        IPresSpaceHandle hdcMem2  = CreateCompatibleDC (fBitmapHDC) ;

        HBITMAP old_bit1 = SelectObject(hdcMem1, natImg);
        HBITMAP old_bit2 = SelectObject(hdcMem2, fBitmap);
        StretchBlt (hdcMem2, 0, 0, fWidth, fHeight, hdcMem1,  0, 0, fWidth, fHeight, SRCCOPY) ;
        SelectObject(hdcMem1, old_bit1);
        SelectObject(hdcMem2, old_bit2);
        DeleteDC(hdcMem1);
        DeleteDC(hdcMem2);

		GpiDeleteBitmap(natImg);
#elif defined(IC_MOTIF)
		XImageCopy(fBitmap, natImg);
		XDestroyImage(natImg);
	
#endif
	}

#endif //IC_WIN
}

nativeImage IPixelBuffer::buildNativeImage(const iBYTE *srcImg, const IRGBAColorArray *clrAry)
{
        IFUNCTRACE_DEVELOP();
	if(fBitmap){
#if defined(IC_WIN)
		//Since fImage is data storage for bitmap in WINDOWS
		//before deleteing the old we need to bakup the data.

		iBYTE *tmpData = new iBYTE[fPixelStride * fHeight];
		memcpy(tmpData, fImage, (size_t)(fPixelStride*fHeight));
		DeleteObject(fBitmap);
		fBitmap = NULL;
                // No need to copy twice. Use the copy as the new temp buffer
                // passed to createNativeImage.
		//fImage = new iBYTE[fPixelStride * fHeight];
		//memcpy(fImage, tmpData, (size_t)(fPixelStride*fHeight));
		delete [] tmpData;
		fBitmap = createNativeImage(NULL, clrAry);
#elif defined(IC_PM)
		GpiDeleteBitmap(fBitmap);
		fBitmap = NULL;
		fBitmap = createNativeImage(srcImg, clrAry);
#elif defined(IC_MOTIF)
                //We cannot destroy fBitmap, since it is associated with a fBitmapHDC

                nativeImage tmpBitmap = createNativeImage(srcImg, clrAry);
                XImageCopy(fBitmap, tmpBitmap);
                XDestroyImage(tmpBitmap);
                updateDeviceContext(fBitmap);

#endif
	} else {

		fBitmap = createNativeImage(srcImg, clrAry);

#if defined(IC_MOTIF)
		updateDeviceContext(fBitmap);
#endif
	}
	return fBitmap;
}

void IPixelBuffer::syncColormap(IRGBAColorArray *clrAry)
{
        IFUNCTRACE_DEVELOP();
	if(fBitsPerPixel < 16){
		unsigned long numberOfColors = clrAry->fNumberOfColors;
		IR8G8B8A8Color *clrData = clrAry->fColors;

#if defined(IC_WIN)
		HDC hdc = GetDC(0);
		SelectObject(hdc, fBitmap);
		RGBQUAD *rgb = new RGBQUAD[numberOfColors];
		for(int i=0; i<numberOfColors; i++){
			rgb[i].rgbRed = clrData[i].fRed;
			rgb[i].rgbGreen = clrData[i].fGreen;
			rgb[i].rgbBlue = clrData[i].fBlue;
			rgb[i].rgbReserved = 0;
		}
		SetDIBColorTable(hdc, 0, numberOfColors, rgb);
		delete [] rgb;
		SelectObject(hdc, 0x0);
		ReleaseDC(0, hdc);
#elif defined(IC_PM)
//TODO
#elif defined(IC_MOTIF)
                //We cannot destroy fBitmap, since it is associated with a fBitmapHDC

                nativeImage tmpBitmap = createNativeImage(fImage, clrAry);
                XImageCopy(fBitmap, tmpBitmap);
                XDestroyImage(tmpBitmap);
                updateDeviceContext(fBitmap);
#endif
	
	}
}

void IPixelBuffer::syncPixelBufferRect(const IGRect2D rect)
{
        IFUNCTRACE_DEVELOP();
	IGRect2D imgRect(0, 0, fWidth, fHeight);
	if(!imgRect.intersects(rect)) //No common area
                return;
#if defined(IC_MOTIF)
	bool hasColorMapSupport = IColorMap::hasColorMapSupport();
	int i, j;
	XColor *xclr=0x0;

	syncXImageFromPixmap();

	if(hasColorMapSupport){
		IColorMap* defColorMap = &(IColorMap::defaultColorMap());
		int total_count = defColorMap->numberOfColors();
		xclr = new XColor[total_count];
		IR8G8B8A8Color *defClrData=defColorMap->fColors;

		for(i=0; i< total_count; i++){
			xclr[i].pixel = i;
			xclr[i].flags = DoRed | DoGreen | DoBlue;
                        xclr[i].red = defClrData[i].fRed;
                        xclr[i].green = defClrData[i].fGreen;
                        xclr[i].blue = defClrData[i].fBlue;
		}
	}

	IRGBAColorArray *clrAry=NULL;
	unsigned long numberOfColors = 0;

	if(fBitsPerPixel < 16){
		clrAry = colorTable();
		numberOfColors = clrAry->fNumberOfColors;
	}

	XColor *pclr=NULL;
	int *map_cmap = new int[256];
	XColor  clr_tmp;

	if(numberOfColors){
		pclr = new XColor[numberOfColors];
		IR8G8B8A8Color *clrData = clrAry->fColors;
		for(i=0; i<numberOfColors; i++){
			pclr[i].red = clrData[i].fRed;
			pclr[i].green = clrData[i].fGreen;
			pclr[i].blue = clrData[i].fBlue;
			pclr[i].pixel = i;
		}
	} else if(hasColorMapSupport){ // image is a 24bit, and so does not have a colormap
		pclr = new XColor[256];
		memcpy(pclr, xclr, sizeof(XColor) << 8);
	}


	if(hasColorMapSupport){
		for(i=0; i<256; i++){
			matchXColor( pclr, numberOfColors, &xclr[i]);
			map_cmap[i] = xclr[i].pixel;
		}
	} else {
		for(i=0; i<256; i++)
			map_cmap[i] = i;
	}

	unsigned long curColor=0,prevColor=0, prevIndex=0;
	//init for the black color
	clr_tmp.red = clr_tmp.green = clr_tmp.blue = 0;
	matchXColor( pclr, 256, &clr_tmp);
	prevIndex = clr_tmp.pixel;

	unsigned long tmp, scanstride;
	imgRect.intersectWith(rect);

	R5G6B5 *rgb16;
	RGBClr *rgb24;
	RGBAClr *rgba;
	RGBClr tmp24Clr;
	for(i=(int)(imgRect.fTop); i< (int)(imgRect.fBottom); i++)
	{
		scanstride = i * fPixelStride;
		rgb16 = (R5G6B5 *) &fImage[scanstride];
		rgb24 = (RGBClr *) &fImage[scanstride];
		rgba = (RGBAClr *) &fImage[scanstride];
		for(j=(int)(imgRect.fLeft); j< (int)(imgRect.fRight); j++)
		{
			tmp = XGetPixel(fBitmap, j, i);
			if(!hasColorMapSupport){
				if(prevColor == tmp){
					tmp = prevIndex;
				} else {
					IXDisplay::pixelToRGB(tmp, &tmp24Clr.R,
						&tmp24Clr.G, &tmp24Clr.B);
					clr_tmp.red = tmp24Clr.R;
					clr_tmp.green = tmp24Clr.G;
					clr_tmp.blue = tmp24Clr.B;
					matchXColor( pclr, numberOfColors, &clr_tmp);
					prevColor = tmp;
					prevIndex = tmp = clr_tmp.pixel;
				}
			}
			switch(fBitsPerPixel){
				case 1:
					SETBIT(fImage[scanstride+j/8], map_cmap[tmp],  j&0x7);
					break;
				case 4:
					SETNIBBLE(fImage[scanstride+j/2], map_cmap[tmp],  j&0x1);
					break;
				case 8:
					fImage[scanstride + j] = map_cmap[tmp];
					break;
				case 16:
					rgb16[j].R = pclr[tmp].red;
					rgb16[j].G = pclr[tmp].green;
					rgb16[j].B = pclr[tmp].blue;
					break;
				case 24:
					rgb24[j].R = pclr[tmp].red;
					rgb24[j].G = pclr[tmp].green;
					rgb24[j].B = pclr[tmp].blue;
					break;
				case 32:
					rgba[j].R = pclr[tmp].red;
					rgba[j].G = pclr[tmp].green;
					rgba[j].B = pclr[tmp].blue;
					rgba[j].A = 0xFF;
					break;
			}
		}
	}
	if(hasColorMapSupport){
		delete [] xclr;
	}
	delete [] pclr;
	delete [] map_cmap;

#elif defined(IC_PM)
	int bmSize, i, j;
	bmSize = sizeof(BITMAPINFO2);
	if(fBitsPerPixel < 16) //for colortable
		bmSize += (sizeof(RGB2)*((1<< fBitsPerPixel)-1));

        BITMAPINFO2 *bm = (PBITMAPINFO2)malloc(bmSize);
        HBITMAP oldbm = SelectObject(fBitmapHDC,  fBitmap);
        if(oldbm == HBM_ERROR)
        {
                ITRACE_DEVELOP(IString("syncPixelBufferRect::SelectObject"));
                GrafDeviceException("syncPixelBufferRect::SelectObject");
        }
	
	memset(bm, 0x0, (size_t)bmSize);
        bm->cbFix = sizeof(BITMAPINFO2);
        bm->cBitCount = fBitsPerPixel;
        bm->cPlanes = 1;

        // if the bitmap has 24 bits per pixel then there will be no colour table information
        // so set cbFix to 16 (length of BITMAPINFOHEADER through to cBitCount)
        if (bm->cPlanes * bm->cBitCount == 24)
           bm->cbFix = 16;

        iBYTE *tmpData = new iBYTE[fPixelStride * fHeight];
	memset(tmpData, 0x0, (size_t)(fPixelStride * fHeight)); //Init

        i=GpiQueryBitmapBits(fBitmapHDC, 0, fHeight, (PBYTE)tmpData, bm);
        if( i < 0)
        {
                ITRACE_DEVELOP(IString("GpiQueryBitmapBits returned: ") + IString(WinGetLastError(0)).d2x());
                GrafDeviceException("GpiQueryBitmapBits");
        }
	SelectObject(fBitmapHDC, oldbm);

        if(fBitsPerPixel < 16){
		long noOfColors = 1<<fBitsPerPixel;
		IRGBAColorArray* clrTbl = colorTable();

            unsigned long scanstride;
            iBYTE indx;
            int *mClr = new int[noOfColors];
            for(i=0; i< noOfColors; i++){
               mClr[i] = matchRGBAColor(clrTbl,
                        bm->argbColor[i].bRed,
						bm->argbColor[i].bGreen,
						bm->argbColor[i].bBlue, 0xff);
			}

            for(i=0; i<fHeight; i++){
               scanstride = i * fPixelStride;
               for(j=0; j<fWidth; j++){
                  switch(fBitsPerPixel){
                  case 1:
                     indx=GETBIT(tmpData[scanstride+j/8], j&0x7);
                     SETBIT(fImage[scanstride+j/8], mClr[indx],  j&0x7);
                     break;
                  case 4:
                     indx = GETNIBBLE(tmpData[scanstride+j/2], j&0x1);
                     SETNIBBLE(fImage[scanstride+j/2], mClr[indx],  j&0x1);
                     break;
                  case 8:
                     indx = tmpData[scanstride + j];
                     fImage[scanstride + j] = mClr[indx];
                     break;
                  }
               }
           }
			delete [] mClr;
        } else {
		memcpy(fImage, tmpData, (size_t)(fPixelStride * fHeight));
        }
        free(bm);
        delete [] tmpData;


#endif //IC_MOTIF

        bImgUpdated=false; //setPixel is used only for sync
        bSyncImage = false;
}

nativeImage IPixelBuffer::createDIBSection(const iBYTE* srcImg, const IRGBAColorArray* clrAry)
{
        IFUNCTRACE_DEVELOP();
#if defined(IC_WIN)

        HDC hdc=NULL;
        int i;

        BITMAPINFO *fBmpInfo;

	//SAS - check this bug, this kind of manuplation is not required
        if ( clrAry )
		// (26177) i = sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*clrAry->numberOfColors();
		i = sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD) * (((clrAry->numberOfColors() < 2) &&  (fBitsPerPixel == 1)) ? 2 : clrAry->numberOfColors());
        else
		// (26177) i = sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD);
		i = sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD) * (fBitsPerPixel == 1 ? 2 : 1);
        fBmpInfo = (BITMAPINFO*)malloc(i);
        memset(fBmpInfo, 0x0, (size_t)i);
        fBmpInfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
        fBmpInfo->bmiHeader.biWidth = fWidth;
        fBmpInfo->bmiHeader.biHeight = fHeight;
        fBmpInfo->bmiHeader.biPlanes = 1;
        fBmpInfo->bmiHeader.biBitCount = fBitsPerPixel;
        fBmpInfo->bmiHeader.biCompression = BI_RGB;

        if ( clrAry )
        {
                fBmpInfo->bmiHeader.biClrUsed = clrAry->fNumberOfColors;
                fBmpInfo->bmiHeader.biClrImportant = fBmpInfo->bmiHeader.biClrUsed;

		IR8G8B8A8Color *clrData = clrAry->fColors;

                for ( i = 0; i < fBmpInfo->bmiHeader.biClrUsed; i++ )
                {
                        fBmpInfo->bmiColors[i].rgbRed      = clrData[i].fRed;
                        fBmpInfo->bmiColors[i].rgbGreen    = clrData[i].fGreen;
                        fBmpInfo->bmiColors[i].rgbBlue     = clrData[i].fBlue;
                        fBmpInfo->bmiColors[i].rgbReserved = 0;
                }
        }


#ifdef SASGC
        hdc = GetDC(0);
#else //SASGC
        hdc = getDefaultSystemDC();
#endif //SASGC

        /*handle 1bpp case*/
        if ( fBitsPerPixel == 1 )
        /*
        1bpp presentation spaces have a reset or background colour.
        They also have a contrast or foreground colour.
        When data is mapped into a 1bpp presentation space :-
        Data which is the reset colour, remains reset, and is stored as 0's.
        All other data becomes contrast, and is stored as 1's.
        The reset colour for 1bpp screen HPSs is white.
        I want 1's in the source data to become 1's in the HPS.
        We seem to have to reverse the ordering here to get the desired effect.
        */

        {
                static RGBQUAD rgba2Black = { 0x00, 0x00, 0x00, 0x00 };
                static RGBQUAD rgba2White = { 0xff, 0xff, 0xff, 0x00 };
                fBmpInfo->bmiColors[0] = rgba2Black; /* Contrast */
                fBmpInfo->bmiColors[1] = rgba2White; /* Reset */
        }

        iBYTE *tData = NULL;

	if(!srcImg){
		//data is stored in fImage copy into a temp location
		tData = new iBYTE[fHeight * fPixelStride];

		memcpy(tData, fImage, (size_t)(fHeight * fPixelStride));

		if(fBitmap){
			DeleteObject(fBitmap);
		}

		//delete the fImage
		if(fImage)
			delete [] fImage;
	}


        //Let DIB Section create a common area for Bitmap and pixel buffer
        HBITMAP fBMP=CreateDIBSection(hdc,
                        fBmpInfo,
                        DIB_RGB_COLORS,
                        (void **)&fImage, NULL, 0);
        if(!fBMP){
#ifdef SASGC
                ReleaseDC(0, hdc );
#endif //SASGC
                return 0; //MOD_ERR_HBITMAP;
        }

	if((srcImg) && (srcImg != INIT_BUFFER))
		memcpy(fImage, srcImg, (size_t)(fHeight * fPixelStride));
	else if((srcImg) && (srcImg == INIT_BUFFER))
		memset(fImage, 0x0, (size_t)(fHeight * fPixelStride));
	else
		memcpy(fImage, tData, (size_t)(fHeight * fPixelStride));

        free(fBmpInfo);

	if(!srcImg)
		delete [] tData;

        bImgUpdated = false;

#ifdef SASGC
        ReleaseDC(0, hdc );
#endif //SASGC

	return fBMP;
#endif //IC_WIN
return 0; //should not come for other platforms here at all
}


//if L2R = true then write from left to right
//if L2R = false then write from right to left
//if T2B = true then write from top to bottom
//if T2B = false then write from bottom to top
//where as the reading is always from left to right and top to bottom
void IPixelBuffer::framebufferFlip(bool L2R, bool T2B)
{
        IFUNCTRACE_DEVELOP();
        unsigned long i, j;
        iBYTE *tmpData1=0x0, *tmpData2=0x0, *swap=0x0;

        if(L2R && T2B)
                return;

        if(!T2B)
                swap = new iBYTE[fPixelStride];

	//before resizing, sync the image data
	syncPixelBuffer();

        for(i=0; i<fHeight/2; i++){
                tmpData1 = &fImage[i*fPixelStride];
                tmpData2 = &fImage[(fHeight - i -1)*fPixelStride];
                if(!L2R)
                for(j=0; j<fWidth/2; j++){
                        switch(fBitsPerPixel){
                                case 1:
                                        {
                                                iBYTE tmp;
                                                unsigned long i1=j>>3, i2=(fWidth - j-1)>>3;
                                                iBYTE od1 = (iBYTE)j&0x07;
                                                iBYTE od2 = (iBYTE)((fWidth - j -1)&0x07);
                                                //1 scan line
                                                tmp = GETBIT(tmpData1[i1],od1);
                                                SETBIT(tmpData1[i1], GETBIT(tmpData1[i2], od2), od1);
                                                SETBIT(tmpData1[i2], tmp, od2);

                                                //2 scan line
                                                tmp = GETBIT(tmpData2[i1],od1);
                                                SETBIT(tmpData2[i1], GETBIT(tmpData2[i2], od2), od1);
                                                SETBIT(tmpData2[i2], tmp, od2);

                                        }
                                        break;
                                case 4:
                                        {
                                                iBYTE tmp;
                                                unsigned long i1=j>>1, i2=(fWidth - j -1)>>1;
                                                iBYTE od1 = (iBYTE)j&0x01;
                                                iBYTE od2 = (iBYTE)((fWidth - j -1)&0x01);
                                                //1 scan line
                                                tmp = GETNIBBLE(tmpData1[i1],od1);
                                                SETNIBBLE(tmpData1[i1], GETNIBBLE(tmpData1[i2], od2), od1);
                                                SETNIBBLE(tmpData1[i2], tmp, od2);

                                                //2 scan line
                                                tmp = GETNIBBLE(tmpData2[i1],od1);
                                                SETNIBBLE(tmpData2[i1], GETNIBBLE(tmpData2[i2], od2), od1);
                                                SETNIBBLE(tmpData2[i2], tmp, od2);

                                        }
                                        break;
                                case 8:
                                        {
                                                iBYTE tmp;
                                                //1 scan line
                                                tmp = tmpData1[j];
                                                tmpData1[j] = tmpData1[(fWidth - j -1)];
                                                tmpData1[(fWidth - j -1)] = tmp;

                                                //2 scan line
                                                tmp = tmpData2[j];
                                                tmpData2[j] = tmpData2[(fWidth - j -1)];
                                                tmpData2[(fWidth - j -1)] = tmp;

                                        }
                                        break;
                                case 16:
                                        {
                                                R5G6B5 tmp;
                                                //1 scan line
                                                memcpy(&tmp, &tmpData1[j*2], (size_t)2);
                                                memcpy(&tmpData1[j*2], &tmpData1[(fWidth - j -1)*2], (size_t)2);
                                                memcpy(&tmpData1[(fWidth - j -1)*2], &tmp, (size_t)2);

                                                //2 scan line
                                                memcpy(&tmp, &tmpData2[j*2], (size_t)2);
                                                memcpy(&tmpData2[j*2], &tmpData2[(fWidth - j -1)*2], (size_t)2);
                                                memcpy(&tmpData2[(fWidth - j -1)*2], &tmp, (size_t)2);

                                        }
                                        break;
                                case 24:
                                        {
                                                RGBClr tmp;
                                                //1 scan line
                                                memcpy(&tmp, &tmpData1[j*3], (size_t)3);
                                                memcpy(&tmpData1[j*3], &tmpData1[(fWidth - j -1)*3], (size_t)3);
                                                memcpy(&tmpData1[(fWidth - j -1)*3], &tmp, (size_t)3);

                                                //2 scan line
                                                memcpy(&tmp, &tmpData2[j*3], (size_t)3);
                                                memcpy(&tmpData2[j*3], &tmpData2[(fWidth - j -1)*3], (size_t)3);
                                                memcpy(&tmpData2[(fWidth - j -1)*3], &tmp, (size_t)3);

                                        }
                                        break;
                                case 32:
                                        {
                                                RGBClr tmp;
                                                //1 scan line
                                                memcpy(&tmp, &tmpData1[j*4], (size_t)4);
                                                memcpy(&tmpData1[j*4], &tmpData1[(fWidth - j -1)*4], (size_t)4);
                                                memcpy(&tmpData1[(fWidth - j -1)*4], &tmp, (size_t)4);

                                                //2 scan line
                                                memcpy(&tmp, &tmpData2[j*4], (size_t)4);
                                                memcpy(&tmpData2[j*4], &tmpData2[(fWidth - j -1)*4], (size_t)4);
                                                memcpy(&tmpData2[(fWidth - j -1)*4], &tmp, (size_t)4);

                                        }
                                        break;
                        }
                }
                if(!T2B){
                        memcpy(swap, tmpData1, (size_t)fPixelStride);
                        memcpy(tmpData1, tmpData2, (size_t)fPixelStride);
                        memcpy(tmpData2, swap, (size_t)fPixelStride);
                }
        }
        if(!T2B)
                delete [] swap;
}

void IPixelBuffer::framebufferRotate(bool flip)
{
        IFUNCTRACE_DEVELOP();
        int i, j;
        unsigned long pixelStride = PIXELSTRIDE(fHeight, fBitsPerPixel);
        iBYTE *oldData = new iBYTE[fHeight * fPixelStride];

	//before resizing, sync the image data
	syncPixelBuffer();

        //copy all the data to temp area
        memcpy(oldData, fImage, (size_t)(fHeight * fPixelStride));
#if defined(IC_WIN)
        DeleteObject(fBitmap);
        fBitmap= NULL;
#else
        //delete the fImage (On NT this is deleted by DeleteObject above).
        delete [] fImage;
#endif
        //allocate for new data area
        fImage = new iBYTE[fWidth * pixelStride];
	memset(fImage, 0x0, (size_t)(pixelStride * fWidth)); //Init

        unsigned long readstride, writestride;
        for(i=0; i<fHeight; i++){
                readstride = i * fPixelStride;
                for(j=0; j<fWidth; j++){
                        if(flip)
                                writestride = (fWidth - j -1) * pixelStride;
                        else
                                writestride = j * pixelStride;
                        switch(fBitsPerPixel){
                                case 1:
                                        SETBIT(fImage[writestride + (i>>3)],
                                                GETBIT(oldData[readstride + (j>>3)], j&0x7),
                                                i&0x7);
                                        break;
                                case 4:
                                        SETNIBBLE(fImage[writestride + (i>>1)],
                                                GETNIBBLE(oldData[readstride + (j>>1)], j&0x1),
                                                i&0x1);
                                        break;
                                case 8:
                                        fImage[writestride + i]=oldData[readstride + j];
                                        break;
                                case 16:
                                        memcpy(&fImage[writestride + i*2],
                                                        &oldData[readstride + j*2], 2);
                                        break;
                                case 24:
                                        memcpy(&fImage[writestride + i*3],
                                                        &oldData[readstride + j*3], 3);
                                        break;
                                case 32:
                                        memcpy(&fImage[writestride + i*4],
                                                        &oldData[readstride + j*4], 4);
                                        break;
                        }
                }
        }

        delete [] oldData;
        i = fWidth;
        fWidth = fHeight;
        fHeight = i;
        fPixelStride = pixelStride;
        initializeDeviceContext();
	convertToNativeImage();
        updateDeviceContext(fBitmap);
}
/*
Using Nearest Neighbourhood method, later make changes for cubic convolution
*/
void IPixelBuffer::resize(unsigned long width, unsigned long height)
{
        IFUNCTRACE_DEVELOP();
        //if requested dimnesions match, ignore resize
        if((width == fWidth) && (height == fHeight))
                return;

        unsigned long i, j, x;
        unsigned long pixelStride = PIXELSTRIDE(width, fBitsPerPixel);
        iBYTE *oldData = new iBYTE[fHeight * fPixelStride];

	//before resizing, sync the image data
	syncPixelBuffer();

        //copy all the data to temp area
        memcpy(oldData, fImage, (size_t)(fHeight * fPixelStride));
#ifndef IC_WIN
        if(fImage)
                delete [] fImage;
#else
	//delete the Ximage and pixmap in AIX thru BitmapData

        if(fBitmap){
#ifdef IC_WIN
		DeleteObject(fBitmap);
#else
                GpiDeleteBitmap(fBitmap);
#endif
		fBitmap = NULL;
        }
#endif

        //allocate for new data area
        fImage = new iBYTE[height * pixelStride];
	memset(fImage, 0x0, (size_t)(pixelStride * height)); //Init

        double xfac = (double)fWidth/(double)width;
        double yfac = (double)fHeight/(double)height;

        unsigned long readstride, writestride;
        for(i=0; i<height; i++){
                readstride = ((int)((double)i*yfac)) * fPixelStride;
                writestride = i * pixelStride;
                for(j=0; j<width; j++){
                        x = (int)((double)j * xfac);
                        switch(fBitsPerPixel)
                        {
                                case 1:
                                        SETBIT(fImage[writestride + (j>>3)],
                                                GETBIT(oldData[readstride + (x>>3)], x&0x7),
                                                j&0x7);
                                        break;
                                case 4:
                                        SETNIBBLE(fImage[writestride + (j>>1)],
                                                GETNIBBLE(oldData[readstride + (x>>1)], x&0x1),
                                                j&0x1);
                                        break;
                                case 8:
                                        fImage[writestride + j] = oldData[readstride + x];
                                        break;
                                case 16:
                                        memcpy(&fImage[writestride + j*2], &oldData[readstride + x*2], 2);
                                        break;
                                case 24:
                                        memcpy(&fImage[writestride + j*3], &oldData[readstride + x*3], 3);
                                        break;
                                case 32:
                                        memcpy(&fImage[writestride + j*4], &oldData[readstride + x*4], 4);
                                        break;
                        }
                }
        }

        delete [] oldData;
        fWidth = width;
        fHeight = height;
        fPixelStride = pixelStride;
#ifdef IC_MOTIF
	XFreePixmap(((IXDC*)fBitmapHDC)->xDisplay,
	              ((IXDC*)fBitmapHDC)->xDrawable);

// SAS	delete fBitmapHDC;
	XDestroyImage(fBitmap);

	fBitmapHDC = NULL;
	fBitmap = NULL;

        initializeDeviceContext();
#endif
	convertToNativeImage();
#ifdef IC_MOTIF
        updateDeviceContext(fBitmap);
#endif

}

iBYTE IPixelBuffer::matchRGBAColor(const IRGBAColorArray* colorTable,
				const long tR, const long tG, const long tB, const long tA)
{
        IFUNCTRACE_DEVELOP();
        int min=999; //max value
        int cnt, min_index=0;
        int diff, numberOfColors = colorTable->fNumberOfColors;
        long sR, sG, sB;

	IR8G8B8A8Color *clrData = colorTable->fColors;

        for (cnt = 0; cnt < numberOfColors; cnt++)
        {
                sR = clrData[cnt].fRed;
                sG = clrData[cnt].fGreen;
                sB = clrData[cnt].fBlue;
                diff =
                        ABS(sR - tR) +
                        ABS(sG - tG) +
                        ABS(sB - tB);

                if (diff == 0){ //perfect match
                        return (iBYTE)cnt;
                }

                if(min > diff){
                        min = diff;
                        min_index = cnt;
                }

        }
        return (iBYTE)min_index; //nearest match

}

#if defined(IC_WIN)
iBYTE IPixelBuffer::matchRGBAColor(const RGBQUAD *colorTable, const long noOfColors,
				const long tR, const long tG, const long tB, const long tA)
{
        IFUNCTRACE_DEVELOP();
        int min=999; //max value
        int cnt, min_index=0;
        long diff;
        long sR, sG, sB;

        for (cnt = 0; cnt < noOfColors; cnt++)
        {
                sR = colorTable[cnt].rgbRed;
                sG = colorTable[cnt].rgbGreen;
                sB = colorTable[cnt].rgbBlue;
                diff =
                        ABS(sR - tR) +
                        ABS(sG - tG) +
                        ABS(sB - tB);

                if (diff == 0){ //perfect match
                        return (iBYTE)cnt;
                }

                if(min > diff){
                        min = diff;
                        min_index = cnt;
                }

        }
        return (iBYTE)min_index; //nearest match

}

#elif defined(IC_PM)

iBYTE IPixelBuffer::matchRGBAColor(const RGB2 *colorTable, const long noOfColors,
				const long tR, const long tG, const long tB, const long tA)
{
        IFUNCTRACE_DEVELOP();
        int min=999; //max value
        int cnt, min_index=0;
        int diff;
        long sR, sG, sB;

        for (cnt = 0; cnt < noOfColors; cnt++)
        {
                sR = colorTable[cnt].bRed;
                sG = colorTable[cnt].bGreen;
                sB = colorTable[cnt].bBlue;
                diff =
                        ABS(sR - tR) +
                        ABS(sG - tG) +
                        ABS(sB - tB);

                if (diff == 0){ //perfect match
                        return (iBYTE)cnt;
                }

                if(min > diff){
                        min = diff;
                        min_index = cnt;
                }

        }
        return (iBYTE)min_index; //nearest match

}
#endif


IRGBAColorArray IPixelBuffer::getSystemColors(unsigned long max)
{
        IFUNCTRACE_DEVELOP();
	if (IColorMap::hasColorMapSupport()) {
		IColorMap* defColorMap = &(IColorMap::defaultColorMap());
		IR8G8B8A8Color *defClrData=defColorMap->fColors;
		unsigned long num = defColorMap->numberOfColors();

		num = (num < max) ? num : max;
		IRGBAColorArray rgbaClrArr(num);
		IR8G8B8A8Color *clrData = rgbaClrArr.fColors;
		memcpy(clrData, defClrData, sizeof(IR8G8B8A8Color) * num);
		return rgbaClrArr;
	} else {
		IRGBAColorArray rgbaClrArr(max);
		IR8G8B8A8Color *clrData = rgbaClrArr.fColors;
		//fill the RGBA Color Array
		for(int i=0; i<max;i++) {
			clrData[i].fRed = DefaultColorTable[i][0];
			clrData[i].fGreen = DefaultColorTable[i][1];
			clrData[i].fBlue = DefaultColorTable[i][2];
			clrData[i].fOpacity = 0xFF;
		}
		return rgbaClrArr;
	}


#if 0 //SAS

#if defined(IC_MOTIF)

        Display *dpy = getDisplay();
	Colormap defColormap = IXColormap::colormap();

	if (IColorMap::hasColorMapSupport()) {
		IColorMap* defColorMap = &(IColorMap::defaultColorMap());
		IR8G8B8A8Color *defClrData=defColorMap->fColors;

		num = defColorMap->numberOfColors();
                num = (num < max) ? num : max;
                XColor *xclr = new XColor[num];
                for(i=0; i< num; i++){
                        xclr[i].pixel = i;
                        xclr[i].flags = DoRed | DoGreen | DoBlue;
                        xclr[i].red = defClrData[i].fRed;
                        xclr[i].green = defClrData[i].fGreen;
                        xclr[i].blue = defClrData[i].fBlue;
                }
	
                //fill the RGBA Color Array
                for(i=0; i<num;i++)
                {
			clrData[i].fRed = xclr[i].red;
			clrData[i].fGreen = xclr[i].green;
			clrData[i].fBlue = xclr[i].blue;
			clrData[i].fOpacity = 0xFF;
                }
                //Fill the rest with 0
                for(i=num;i < max; i++)
                {
			clrData[i].fRed = 0xFF;
			clrData[i].fGreen = 0xFF;
			clrData[i].fBlue = 0xFF;
			clrData[i].fOpacity = 0xFF;
                }

		delete []xclr;

        } else {
		num = max;
                //fill the RGBA Color Array
                for(i=0; i<num;i++)
                {
			clrData[i].fRed = DefaultColorTable[i][0];
			clrData[i].fGreen = DefaultColorTable[i][1];
			clrData[i].fBlue = DefaultColorTable[i][2];
			clrData[i].fOpacity = 0xFF;
                }
	}

#elif defined(IC_WIN)

        //ColorPalette Init
#ifdef SASGC
        HDC sys = GetDC(0);
#else //SASGC
        HDC sys = getDefaultSystemDC();
#endif //SASGC

	if (IColorMap::hasColorMapSupport()) {

                num = GetDeviceCaps(sys, SIZEPALETTE);
                num = (num < max) ? num : max;
                PALETTEENTRY *pal = new PALETTEENTRY[num];
                GetSystemPaletteEntries(sys, 0, num, pal);

                //fill the RGBA Color Array
                for(i=0; i<num;i++)
                {
			clrData[i].fRed = pal[i].peRed;
			clrData[i].fGreen = pal[i].peGreen;
			clrData[i].fBlue = pal[i].peBlue;
			clrData[i].fOpacity = 0xFF;
                }
		delete [] pal;

		//Fill the rest with 0
		for(i=num;i < max; i++)
		{
			clrData[i].fRed = 0xFF;
			clrData[i].fGreen = 0xFF;
			clrData[i].fBlue = 0xFF;
			clrData[i].fOpacity = 0xFF;
		}
        } else {
		num = max;
                //fill the RGBA Color Array
                for(i=0; i<num;i++)
                {
			clrData[i].fRed = DefaultColorTable[i][0];
			clrData[i].fGreen = DefaultColorTable[i][1];
			clrData[i].fBlue = DefaultColorTable[i][2];
			clrData[i].fOpacity = 0xFF;
                }
	}

#ifdef SASGC
        ReleaseDC(0, sys);
#endif //SASGC

#elif defined(IC_PM)

        HPS hps = WinGetPS(HWND_DESKTOP);
        HDC hdc = GpiQueryDevice( hps );
        long defaultEntries, rgbColor=0;

        DevQueryCaps(hdc, CAPS_COLORS, 1, &defaultEntries);
	if(defaultEntries <= 256)
	{
		num = (max < defaultEntries) ? max : defaultEntries;

		long *alArray = new long[num];
		GpiQueryRealColors(hps, 0L, 0L, num, alArray);

#define IQUERYRGBCOLOR(hps,idx,pPe) (*pPe = GpiQueryRGBColor(hps, 0L, idx))
#define IRGB_BLUE(rgb)   ((unsigned char)(rgb & 0x000000FF) )
#define IRGB_GREEN(rgb)  ((unsigned char)((rgb & 0x0000FF00) >> 8) )
#define IRGB_RED(rgb)    ((unsigned char)((rgb & 0x00FF0000) >> 16) )

		for (i = 0; i < num; i++)
		{
			if (i < 16)
			{
				//filling in the first 16 system colors
				IQUERYRGBCOLOR(hps,i,&rgbColor);
			} else{
				rgbColor = alArray[i];
			}
				clrData[i].fRed = IRGB_RED(rgbColor);
				clrData[i].fGreen = IRGB_GREEN(rgbColor);
				clrData[i].fBlue = IRGB_BLUE(rgbColor);
				clrData[i].fOpacity = 0xFF;
                }
		//Fill the rest with 0
		for(i=num; i< max; i++)
		{
			clrData[i].fRed = 0x0;
			clrData[i].fGreen = 0x0;
			clrData[i].fBlue = 0x0;
			clrData[i].fOpacity = 0xFF;
		}
        } else {
		num = max;
                //fill the RGBA Color Array
                for(i=0; i<num;i++)
                {
			clrData[i].fRed = DefaultColorTable[i][0];
			clrData[i].fGreen = DefaultColorTable[i][1];
			clrData[i].fBlue = DefaultColorTable[i][2];
			clrData[i].fOpacity = 0xFF;
                }
	}
	WinReleasePS(hps);
#endif
	if(num > 0)
		return rgbaClrArr;
	else
#endif //SAS
		return NULL;
}

void IPixelBuffer::imageToMem(const nativeImage nImg, unsigned long depth, unsigned long nClr, IRGBAColorArray* colorTable)
{
        IFUNCTRACE_DEVELOP();
        int i, j, k;
#if defined(IC_MOTIF)

        fWidth = nImg->width;
        fHeight = nImg->height;

        fBitsPerPixel = depth;
        fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
        if (fImage)
           delete [] fImage;
        fImage = new iBYTE[fPixelStride * fHeight];
	if(nImg->depth > 8){ // 32 bit word
		if(IXDisplay::isRGBFormat())
			memcpy(fImage, nImg->data, (size_t)(fHeight * fPixelStride));
		else {
			RGBAClr *pTmp;
			RGBAClr *rgba;
			RGBClr *rgb;
			for(i=0; i<fHeight; i++){
				rgba = (RGBAClr *)&fImage[i*fPixelStride];
				rgb = (RGBClr *)&fImage[i*fPixelStride];
				pTmp = (RGBAClr *)&nImg->data[i * nImg->bytes_per_line];
				for(j=0; j<fWidth; j++){
					//The ordering of RGBAClr is RGBA but the data in XImage
					//is ARGB of ABGR
					if(depth == 32){
						rgba[j].R = pTmp[j].A;
						rgba[j].G = pTmp[j].B;
						rgba[j].B = pTmp[j].G;
					} else if(depth == 24){
						rgb[j].R = pTmp[j].A;
						rgb[j].G = pTmp[j].B;
						rgb[j].B = pTmp[j].G;
					}
				}
			}
		}
	} else {
		memset(fImage, 0x0, (size_t)(fHeight * fPixelStride));
		for(i=0; i<fHeight; i++){
			memcpy(&fImage[i * fPixelStride],
				&nImg->data[i * nImg->bytes_per_line],
				(size_t)nImg->bytes_per_line);
		}
	}


        if(depth < 16)
                *colorTable = getSystemColors(nClr);

#elif defined(IC_WIN)

        //ColorPalette Init
        struct
        {
                BITMAPINFOHEADER bih;
                RGBQUAD          argb2Color[0x100];
        } bm;

#ifdef SASGC
        HDC sys = GetDC(0);
#else //SASGC
        HDC sys = getDefaultSystemDC();
#endif //SASGC
        BITMAP bmp;
        if (!GetObject(nImg, sizeof(bmp), (LPSTR)&bmp))
                GrafDeviceException("GetObject");

        fWidth = (int)bmp.bmWidth;
        fHeight = (int)bmp.bmHeight;
        fBitsPerPixel = depth;
        fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
        if (fImage)
           delete [] fImage;
        fImage = new iBYTE[fPixelStride * fHeight];
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init

        memset(&(bm.bih), 0, (size_t)sizeof(bm.bih));
        bm.bih.biSize          = sizeof( BITMAPINFOHEADER );
        bm.bih.biWidth         = fWidth;
        bm.bih.biHeight        = fHeight;
        bm.bih.biPlanes        = 1;
        bm.bih.biBitCount      = fBitsPerPixel;
        bm.bih.biCompression   = BI_RGB;

        HBITMAP oldbm;
        oldbm = (HBITMAP) SelectObject( sys, nImg );

        GetDIBits( sys, nImg, 0, fHeight,
                fImage, (BITMAPINFO*)&(bm.bih), DIB_RGB_COLORS);

        if(depth < 16){
			IRGBAColorArray rgbaClrArr(nClr);
			IR8G8B8A8Color *clrData = rgbaClrArr.fColors;
			//fill the RGBA Color Array
			for(i=0; i<nClr; i++) {
				clrData[i].fRed = bm.argb2Color[i].rgbRed;
				clrData[i].fGreen = bm.argb2Color[i].rgbGreen;
				clrData[i].fBlue = bm.argb2Color[i].rgbBlue;
				clrData[i].fOpacity = 0xFF;
			}
			*colorTable = rgbaClrArr;
        }

        SelectObject(sys, oldbm);
#ifdef SASGC
        ReleaseDC(0, sys);
#endif //SASGC

#elif defined(IC_PM)

        //ColorPalette Init
        SIZEL           sizl;
        HDC             hdc;
        HPS             hps;
        HAB             hab=0; //fixed value used also in old implementation

        if ( (hdc = DevOpenDC(hab, OD_MEMORY, "*", 0L, (PDEVOPENDATA) NULL, (HDC) NULL)) == (HDC) NULL )
                GrafDeviceException("DevOpenDC");

        BITMAP bmp;
        if (!GetObject(nImg, sizeof(bmp), (LPSTR)&bmp))
                GrafDeviceException("GetObject");

        sizl.cx = bmp.bmWidth;
        sizl.cy = bmp.bmHeight;
        if ( (hps = GpiCreatePS(hab, hdc, &sizl, PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC)) == (HPS) NULL )
        {
                DevCloseDC(hdc);
                GrafDeviceException("GpiCreatePS");
        }

        int bmSize;
	if(depth < 16){
		bmSize = sizeof(BITMAPINFOHEADER2) + (sizeof(RGB2)  << depth);
	} else {
		bmSize = sizeof(BITMAPINFOHEADER2) + sizeof(RGB2);
	}
        BITMAPINFO2 *bm = (PBITMAPINFO2)malloc(bmSize);
        HBITMAP oldbm = SelectObject(hps,  nImg);
        if(oldbm == HBM_ERROR)
        {
                ITRACE_DEVELOP(IString("GpiSetBitmap returned: ") + IString(WinGetLastError(hab)).d2x());
                GrafDeviceException("imageToMemory::SelectObject");
        }
	memset(bm, 0x0, (size_t)bmSize);
        bm->cbFix = sizeof(BITMAPINFOHEADER2);

        fBitsPerPixel = depth;
        fPixelStride =  PIXELSTRIDE(bmp.bmWidth, fBitsPerPixel);
        if (fImage)
           delete [] fImage;
        fImage = new iBYTE[fPixelStride * bmp.bmHeight];
	memset(fImage, 0x0, (size_t)(fPixelStride * bmp.bmHeight)); //Init

        bm->cBitCount = bmp.bmBitsPixel;
        bm->cPlanes = bmp.bmPlanes;

        // if the bitmap has 24 bits per pixel then there will be no colour table information
        // so set cbFix to 16 (length of BITMAPINFOHEADER through to cBitCount)
        if (bm->cPlanes * bm->cBitCount == 24)
           bm->cbFix = 16;

        i=GpiQueryBitmapBits(hps, 0, bmp.bmHeight, (PBYTE) fImage, bm);
        if( i < 0)
        {
                ITRACE_DEVELOP(IString("GpiQueryBitmapBits returned: ") + IString(WinGetLastError(hab)).d2x());
                GrafDeviceException("GpiQueryBitmapBits");
        }
        SelectObject(hps, oldbm);
        GpiDestroyPS(hps);
        DevCloseDC(hdc);
        fWidth = (int)bm->cx;
        fHeight = (int)bm->cy;
        fHeight = (fHeight < 0) ? -fHeight : fHeight;

        if(depth < 16)
        {
                IRGBAColorArray rgbaClrArr(nClr);
		IR8G8B8A8Color *clrData = rgbaClrArr.fColors;
                //fill the RGBA Color Array
                for(i=0; i<(1<<depth);i++)
                {
			clrData[i].fRed = bm->argbColor[i].bRed;
			clrData[i].fGreen = bm->argbColor[i].bGreen;
			clrData[i].fBlue = bm->argbColor[i].bBlue;
			clrData[i].fOpacity = 0xFF;
                }
                *colorTable = rgbaClrArr;

        }
        free(bm);
#endif
}

//Transparency Related functions
unsigned long IPixelBuffer::maskImage() const
{
        IFUNCTRACE_DEVELOP();
	if(!fMaskBitmap)
	{
	//In AIX the Masking effect is done in renderImage bi us
#ifndef IC_MOTIF
		IPixelBuffer* tmpPix = (IPixelBuffer*)this;
#if defined(IC_WIN)
//In NT both these implementations work exactly the same the only advantage is
//that we know what we R doing and minimized GDI calls
		int bmSize, i, j;
		if(fBitsPerPixel < 16){
			bmSize = sizeof(BITMAPINFOHEADER) + (sizeof(RGBQUAD)  << fBitsPerPixel);
		} else {
			bmSize = sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD);
		}
		BITMAPINFO *bm = (PBITMAPINFO)malloc(bmSize);

		iBYTE *tmpData = new iBYTE[fPixelStride * fHeight];

		memset(bm, 0x0, (size_t)bmSize);
		bm->bmiHeader.biSize          = sizeof( BITMAPINFOHEADER );
		bm->bmiHeader.biWidth         = fWidth;
		bm->bmiHeader.biHeight        = fHeight;
		bm->bmiHeader.biPlanes        = 1;
		bm->bmiHeader.biBitCount      = fBitsPerPixel;
		bm->bmiHeader.biCompression   = BI_RGB;

      HBITMAP oldbm = (HBITMAP) SelectObject( fBitmapHDC, fBitmap );

		i=GetDIBits(fBitmapHDC, fBitmap, 0, fHeight, tmpData, bm, DIB_RGB_COLORS);

		if( i == 0)
			GrafDeviceException("maskImage::GetDIBits");

		SelectObject(fBitmapHDC, oldbm);

		unsigned long scanstride;
		if(bTransparencyEnabled){
			if(fBitsPerPixel < 16){
				iBYTE indx;

				iBYTE tClr = tmpPix->matchRGBAColor(bm->bmiColors,
					(1<<fBitsPerPixel),
					transparencyClr.redMix(),
					transparencyClr.greenMix(),
					transparencyClr.blueMix(),
					transparencyClr.opacity());
				iBYTE black = tmpPix->matchRGBAColor(bm->bmiColors,
					(1<<fBitsPerPixel),
					0x0, 0x0, 0x0, 0xFF);
				iBYTE white = tmpPix->matchRGBAColor(bm->bmiColors,
					(1<<fBitsPerPixel),
					0xFF, 0xFF, 0xFF, 0xFF);

				for(i=0; i<fHeight; i++){
					scanstride = i * fPixelStride;
					for(j=0; j<fWidth; j++){
						switch(fBitsPerPixel){
						case 1:
						     indx=GETBIT(tmpData[scanstride+j/8], j&0x7);
						     indx = (indx == tClr) ? white : black;
						     SETBIT(tmpData[scanstride+j/8], indx,  j&0x7);
						break;
						case 4:
							indx = GETNIBBLE(tmpData[scanstride+j/2], j&0x1);
							indx = (indx == tClr) ? white : black;
							SETNIBBLE(tmpData[scanstride+j/2], indx,  j&0x1);
						break;
						case 8:
							indx = tmpData[scanstride + j];
							indx = (indx == tClr) ? white : black;
							tmpData[scanstride + j] = indx;
						break;
						}
					}
				}

			} else {
				memcpy(tmpData, fImage, (size_t)(fPixelStride * fHeight));
				R5G6B5 *clr16 = (R5G6B5 *)(tmpData);
				RGBClr *clr24 = (RGBClr *)(tmpData);
				RGBAClr *clr32 = (RGBAClr *)(tmpData);
				iBYTE tR = transparencyClr.redMix(),
					tG = transparencyClr.greenMix(),
					tB = transparencyClr.blueMix();
				for(i=0; i<fHeight; i++){
					scanstride = i * fPixelStride;
					clr16 = (R5G6B5 *)&(tmpData[scanstride]);
					clr24 = (RGBClr *)&(tmpData[scanstride]);
					clr32 = (RGBAClr *)&(tmpData[scanstride]);
					for(j=0; j<fWidth; j++){
						switch(fBitsPerPixel){
							case 16:
								if((clr16[j].B == tR) &&
									(clr16[j].G == tG) &&
									(clr16[j].R == tB))
									clr16[j].R=clr16[j].B=clr16[j].G = 0xff;
								else
									clr16[j].R=clr16[j].B=clr16[j].B = 0x00;
							break;
							case 24:
								if((clr24[j].B == tR) &&
									(clr24[j].G == tG) &&
									(clr24[j].R == tB))
									clr24[j].R=clr24[j].G=clr24[j].B = 0xff;
								else
									clr24[j].R=clr24[j].G=clr24[j].B = 0x00;
							break;
							case 32:
								if((clr32[j].B == tR) &&
									(clr32[j].G == tG) &&
									(clr32[j].R == tB))
									clr32[j].R=clr32[j].G=clr32[j].B = 0xff;
								else
									clr32[j].R=clr32[j].G=clr32[j].B = 0x00;
							break;
						}
					}
				}
			}
		} else {
			memset(tmpData, 0x00, (fPixelStride * fHeight));
		}

		HDC hdc1 = CreateCompatibleDC(0);
		tmpPix->fMaskBitmap = CreateCompatibleBitmap(hdc1, fWidth, fHeight);

      oldbm = (HBITMAP) SelectObject( hdc1,  tmpPix->fMaskBitmap );
		i=SetDIBits(hdc1, tmpPix->fMaskBitmap, 0, fHeight, tmpData, bm, DIB_RGB_COLORS);

		if(i == 0)
			GrafDeviceException("maskImage::SetDIBits");

		SelectObject(hdc1, oldbm);
		DeleteDC(hdc1);
		free(bm);
		delete [] tmpData;
#elif defined(IC_PM)
		int bmSize, i, j;
		if(fBitsPerPixel < 16){
			bmSize = sizeof(BITMAPINFOHEADER2) + (sizeof(RGB2)  << fBitsPerPixel);
		} else {
			bmSize = sizeof(BITMAPINFOHEADER2) + sizeof(RGB2);
		}
		BITMAPINFO2 *bm = (PBITMAPINFO2)malloc(bmSize);
		HBITMAP oldbm = SelectObject(fBitmapHDC,  fBitmap);
		if(oldbm == HBM_ERROR)
		{
			GrafDeviceException("maskImage::SelectObject");
		}

		iBYTE *tmpData = new iBYTE[fPixelStride * fHeight];

		memset(bm, 0x0, (size_t)bmSize);
		bm->cbFix = sizeof(BITMAPINFOHEADER2);
		bm->cBitCount = fBitsPerPixel;
		bm->cPlanes = 1;

                // if the bitmap has 24 bits per pixel then there will be no colour table information
                // so set cbFix to 16 (length of BITMAPINFOHEADER through to cBitCount)
                if (bm->cPlanes * bm->cBitCount == 24)
                   bm->cbFix = 16;

		i=GpiQueryBitmapBits(fBitmapHDC, 0, fHeight, (PBYTE) tmpData, bm);
		if( i < 0)
		{
                        ITRACE_DEVELOP(IString("GpiQueryBitmapBits returned: ") + IString(WinGetLastError(0)).d2x());
			GrafDeviceException("GpiQueryBitmapBits");
		}

		tmpPix->fMaskBitmap = CreateCompatibleBitmap(fBitmapHDC, fWidth, fHeight);

		SelectObject(fBitmapHDC, oldbm);

		unsigned long scanstride;
		if(bTransparencyEnabled){
			if(fBitsPerPixel < 16){
				iBYTE indx;

				iBYTE tClr = tmpPix->matchRGBAColor(bm->argbColor,
					(1<<fBitsPerPixel),
					transparencyClr.redMix(),
					transparencyClr.greenMix(),
					transparencyClr.blueMix(),
					transparencyClr.opacity());

				iBYTE black = tmpPix->matchRGBAColor(bm->argbColor,
					(1<<fBitsPerPixel),
					0x0, 0x0, 0x0, 0xFF);

				iBYTE white = tmpPix->matchRGBAColor(bm->argbColor,
					(1<<fBitsPerPixel),
					0xFF, 0xFF, 0xFF, 0xFF);

				for(i=0; i<fHeight; i++){
					scanstride = i * fPixelStride;
					for(j=0; j<fWidth; j++){
						switch(fBitsPerPixel){
						case 1:
						     indx=GETBIT(tmpData[scanstride+j/8], j&0x7);
						     indx = (indx == tClr) ? white : black;
						     SETBIT(tmpData[scanstride+j/8], indx,  j&0x7);
						break;
						case 4:
							indx = GETNIBBLE(tmpData[scanstride+j/2], j&0x1);
							indx = (indx == tClr) ? white : black;
							SETNIBBLE(tmpData[scanstride+j/2], indx,  j&0x1);
						break;
						case 8:
							indx = tmpData[scanstride + j];
							indx = (indx == tClr) ? white : black;
							tmpData[scanstride + j] = indx;
						break;
						}
					}
				}
			} else {
				R5G6B5 *clr16 = (R5G6B5 *)(tmpData);
				RGBClr *clr24 = (RGBClr *)(tmpData);
				RGBAClr *clr32 = (RGBAClr *)(tmpData);
				iBYTE tR = transparencyClr.redMix(),
					tG = transparencyClr.greenMix(),
					tB = transparencyClr.blueMix();
				for(i=0; i<fHeight; i++){
					scanstride = i * fPixelStride;
					clr16 = (R5G6B5 *)&(tmpData[scanstride]);
					clr24 = (RGBClr *)&(tmpData[scanstride]);
					clr32 = (RGBAClr *)&(tmpData[scanstride]);
					for(j=0; j<fWidth; j++){
						switch(fBitsPerPixel){
							case 16:
								if((clr16[j].R == tR) &&
									(clr16[j].G == tG) &&
									(clr16[j].B == tB))
									clr16[j].R=clr16[j].G=clr16[j].B = 0xff;
								else
									clr16[j].R=clr16[j].G=clr16[j].B = 0x00;
							break;
							case 24:
								if((clr24[j].R == tR) &&
									(clr24[j].G == tG) &&
									(clr24[j].B == tB))
									clr24[j].R=clr24[j].G=clr24[j].B = 0xff;
								else
									clr24[j].R=clr24[j].G=clr24[j].B = 0x00;
							break;
							case 32:
								if((clr32[j].R == tR) &&
									(clr32[j].G == tG) &&
									(clr32[j].B == tB))
									clr32[j].R=clr32[j].G=clr32[j].B = 0xff;
								else
									clr32[j].R=clr32[j].G=clr32[j].B = 0x00;
							break;
						}
					}
				}
			}
		} else {
			memset(tmpData, 0x00, (fPixelStride * fHeight));
		}
		POINTL aptl[4]={0, 0, fWidth-1, fHeight-1, 0, 0, fWidth, fHeight};
		oldbm = SelectObject(fBitmapHDC,  tmpPix->fMaskBitmap);
		i=GpiDrawBits(fBitmapHDC, (PVOID)tmpData, bm, 4L, aptl, ROP_SRCCOPY, BBO_IGNORE);
		if( i < 0)
			GrafDeviceException("maskImage::GpiDrawBits");

		SelectObject(fBitmapHDC, oldbm);
		free(bm);

		delete [] tmpData;
#endif //IC_PM
#endif //IC_MOTIF
	}
	return(unsigned long)fMaskBitmap;
}

#ifdef IC_MOTIF
nativeImage IPixelBuffer::maskImageOf1Bit() const
{
	IFUNCTRACE_DEVELOP();
	Display *dpy = getDisplay();
	unsigned long pixstride = PIXELSTRIDE(fWidth, 1);
	char *xiData = new char[fHeight * pixstride];
	Visual* defVisual = DefaultVisual(dpy, DefaultScreen(dpy));
	//set all for no transparency
	memset(xiData, 0xFF, (fHeight*pixstride));
	XImage* maskImage = XCreateImage(dpy, defVisual, 1, XYPixmap, 0,
					xiData, fWidth, fHeight, 8,
					pixstride);
        if(bTransparencyEnabled){
		unsigned long  mask=0, black=0, white=1, pixel;
		if (IColorMap::hasColorMapSupport()){
			mask = IColorMap::defaultColorMap().colorIndex(transparencyClr);
			black = IColorMap::defaultColorMap().colorIndex(IBaseColor(0,0,0));
			white = IColorMap::defaultColorMap().colorIndex(IBaseColor(0xff,0xff,0xff));
		} else {
			mask =IXDisplay::rgbToPixel(transparencyClr.redMix(),
					transparencyClr.greenMix(),
					transparencyClr.blueMix());
			black = IXDisplay::rgbToPixel(0,0,0);
			white = IXDisplay::rgbToPixel(0xff,0xff,0xff);
		}

		//sync image drawn on the Pixmap to Ximage
		((IPixelBuffer*)this)->syncXImageFromPixmap();
		for(int i=0; i<fHeight; i++){
			for(int j=0; j<fWidth; j++){
				pixel = XGetPixel(fBitmap, j, i);
				XPutPixel(maskImage, j, i, ((pixel == mask) ? 0 : 1));
			}
		}
	}
	return maskImage;
}

nativeImage IPixelBuffer::colorImageOf1Bit() const
{
	IFUNCTRACE_DEVELOP();
	Display *dpy = getDisplay();
	unsigned long pixel, pixstride = PIXELSTRIDE(fWidth, 1);
	char *xiData = new char[fHeight * pixstride];
	Visual* defVisual = DefaultVisual(dpy, DefaultScreen(dpy));
	XImage* colorImage = XCreateImage(dpy, defVisual, 1, XYPixmap, 0,
					xiData, fWidth, fHeight, 8,
					pixstride);
#ifdef IC_MOTIF
	//sync image drawn on the Pixmap to Ximage
	((IPixelBuffer*)this)->syncXImageFromPixmap();
#endif //IC_MOTIF
	for(int i=0; i<fHeight; i++){
		for(int j=0; j<fWidth; j++){
			pixel = XGetPixel(fBitmap, j, i);
			XPutPixel(colorImage, j, i, ((pixel) ? 1 : 0));
		}
	}
	return colorImage;
}
#endif //IC_MOTIF

#ifdef IC_PM
IBitmapHandle IPixelBuffer::desktopCompatibleBitmap()
{
        SIZEL           sizl;
        HDC             hdc;
        HBITMAP         phbm;
        HPS             hps;
        HAB             hab=0; //fixed value used also in old implementation
	int i;

        struct
        {
                BITMAPINFOHEADER2 bmp2;
                RGB2 argb2Color[0x100];
        } bm;

        memset(&(bm.bmp2), 0, (size_t)(sizeof(bm.bmp2)));
        bm.bmp2.cbFix     = sizeof(BITMAPINFOHEADER2);
        bm.bmp2.cx        = fWidth;
        bm.bmp2.cy        = fHeight;
        bm.bmp2.cBitCount = fBitsPerPixel;
        bm.bmp2.cPlanes   = 1;
	
	IRGBAColorArray *clrAry = colorTable();
        if ( clrAry )
        {
                bm.bmp2.cclrUsed = clrAry->fNumberOfColors;
                bm.bmp2.cclrImportant = bm.bmp2.cclrUsed;
		IR8G8B8A8Color *clrData = clrAry->fColors;
                for ( i = 0; i < bm.bmp2.cclrUsed; i++ )
                {
                        bm.argb2Color[i].bRed      = clrData[i].fRed;
                        bm.argb2Color[i].bGreen    = clrData[i].fGreen;
                        bm.argb2Color[i].bBlue     = clrData[i].fBlue;
                        bm.argb2Color[i].fcOptions = 0;
                }
        }

        if ( (hdc = DevOpenDC(hab, OD_MEMORY, "*", 0L, (PDEVOPENDATA) NULL, (HDC) NULL)) == (HDC) NULL )
                return 0; //MOD_ERR_HDC;

        sizl.cx = fWidth;
        sizl.cy = fHeight;
        if ( (hps = GpiCreatePS(hab, hdc, &sizl, PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC)) == (HPS) NULL )
        {
                DevCloseDC(hdc);
                return 0; //MOD_ERR_HPS;
        }

        if ( fBitsPerPixel == 1 ) /*handle 1bpp case*/
        /*
        1bpp presentation spaces have a reset or background colour.
        They also have a contrast or foreground colour.
        When data is mapped into a 1bpp presentation space :-
        Data which is the reset colour, remains reset, and is stored as 0's.
        All other data becomes contrast, and is stored as 1's.
        The reset colour for 1bpp screen HPSs is white.
        I want 1's in the source data to become 1's in the HPS.
        We seem to have to reverse the ordering here to get the desired effect.
        */

        {
                static RGB2 argb2Black = { 0x00, 0x00, 0x00 };
                static RGB2 argb2White = { 0xff, 0xff, 0xff };
                bm.argb2Color[0] = argb2Black; /* Contrast */
                bm.argb2Color[1] = argb2White; /* Reset */
        }

        if ( (phbm = GpiCreateBitmap(hps, &(bm.bmp2), CBM_INIT, (PBYTE)fImage, (BITMAPINFO2 *) &(bm.bmp2))) == (HBITMAP) NULL )
        {
                GpiDestroyPS(hps);
                DevCloseDC(hdc);
                return 0; //MOD_ERR_HBITMAP;
        }

        GpiSetBitmap(hps, (HBITMAP) NULL);
        GpiDestroyPS(hps);
        DevCloseDC(hdc);

	return phbm;
}

#endif //IC_PM

//***********************************************************************

//IR8G8B8A8PixelBuffer

IR8G8B8A8PixelBuffer::IR8G8B8A8PixelBuffer ()
{
        fBitsPerPixel = 32;
}
IR8G8B8A8PixelBuffer::~IR8G8B8A8PixelBuffer ()
{
}

IR8G8B8A8PixelBuffer::IR8G8B8A8PixelBuffer (unsigned long width,
				unsigned long height,
				const iBYTE* imageData,
				const IRGBAColorArray* clrArry)

{
        IFUNCTRACE_DEVELOP();
        if(width && height){
                fWidth = width;
                fHeight = height;
                fBitsPerPixel = 32;
                fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);

                initializeDeviceContext();
		if(imageData)
			convertToNativeImage(imageData);
		else
			convertToNativeImage(INIT_BUFFER);
        }
}

IR8G8B8A8PixelBuffer::IR8G8B8A8PixelBuffer (const nativeImage bitmap)
{
        IFUNCTRACE_DEVELOP();
	imageToMem(bitmap, 32, 0, NULL);
	initializeDeviceContext();
	convertToNativeImage();

}

IR8G8B8A8PixelBuffer::IR8G8B8A8PixelBuffer (
                const IR8G8B8A8PixelBuffer& source,
                const IGRect2D& rect)
{
        IFUNCTRACE_DEVELOP();
        int i, k;
        fRefCount=1;
        srcImageFormat = source.srcImageFormat;
        bTransparencyEnabled  = source.bTransparencyEnabled;
        transparencyClr = source.transparencyClr;

        fWidth = (int)rect.width();
        fHeight = (int)rect.height();
        fBitsPerPixel = 32;
        fPixelStride = PIXELSTRIDE(fWidth, fBitsPerPixel);
        fImage = new iBYTE[fPixelStride * fHeight]; //RGBA * 8
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init

        IGRect2D imgRect(0, 0, source.fWidth, source.fHeight);

        if(imgRect.intersects(rect))
        {
                imgRect.intersectWith(rect);

                for(i=(int)(imgRect.fTop), k=0; i< (int)(imgRect.fBottom); i++, k++)
                {
                        memcpy(&fImage[(k*fPixelStride)],
                                &source.fImage[(i*source.fPixelStride) + (((int)(imgRect.fLeft)) * 4)],
                                        (size_t)(imgRect.width() * 4));
                }
        }
        initializeDeviceContext();
	convertToNativeImage();
}

IR8G8B8A8PixelBuffer* IR8G8B8A8PixelBuffer::clonePixelBuffer()
{
        IFUNCTRACE_DEVELOP();
	syncPixelBuffer();
        IR8G8B8A8PixelBuffer *newPixbuf = new IR8G8B8A8PixelBuffer();
	newPixbuf->copyPixelBuffer(*this);
        return newPixbuf;
}

void IR8G8B8A8PixelBuffer::pixel(const IGPoint2D& pnt, IBaseColor& iClr) const
{
        IFUNCTRACE_DEVELOP();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }

	syncPixelBuffer();
        unsigned long index = ((FLIPEDY((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 2);
        RGBAClr* rgba = (RGBAClr *) &fImage[index];
        iClr = RGBA2IBASECOLOR(rgba[0].R,
                                rgba[0].G,
                                rgba[0].B,
                                rgba[0].A);

}

void IR8G8B8A8PixelBuffer::pixelRow(const IGPoint2D& pnt, IRGBAColorArray& Clr) const
{
        IFUNCTRACE_DEVELOP();
        unsigned long length = Clr.numberOfColors();

        if (!length) return;    // we are done!

        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");

        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);

                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;

        length = (unsigned long)targetRect.width();

	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 2);
        long x, i;
        RGBAClr *rgba = (RGBAClr *) &fImage[index];

        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                Clr.setColor(i, RGBA2IBASECOLOR(rgba[i].R,
                                        rgba[i].G,
                                        rgba[i].B,
                                        rgba[i].A));
        }

}

IBaseColor* IR8G8B8A8PixelBuffer::pixelRect(const IGRect2D& rect) const
{
        IFUNCTRACE_DEVELOP();
       if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return NULL;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.width(),
                h = (long) rect.height();
        unsigned long i, j, index;

        IBaseColor* iClr = new IBaseColor[w * h];
        RGBAClr *rgba;
	syncPixelBuffer();

        for(i=0; i < h; i++){
                index = (FLIPEDY(y) * fPixelStride) + (x << 2);
                rgba = (RGBAClr *) &fImage[index];
                for(j=0; j < w; j++){
                        iClr[(i*w)+j] = RGBA2IBASECOLOR(rgba[j].R,
                                                rgba[j].G,
                                                rgba[j].B,
                                                rgba[j].A);
                }
        }
        return iClr;
}

void IR8G8B8A8PixelBuffer::setPixel(const IGPoint2D& pnt, const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 2);
        RGBAClr *rgba = (RGBAClr *) &fImage[index];
        IBASECOLOR2RGBA(rgba[0], Clr);
        bImgUpdated=true;

}

void IR8G8B8A8PixelBuffer::setPixelRow(const IGPoint2D& pnt, const IRGBAColorArray& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        unsigned long length = Clr.numberOfColors();

        if (!length) return;    // we are done!

        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");

        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);

                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;

        length = (unsigned long)targetRect.width();


        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 2);
        long x, i;
        RGBAClr *rgba = (RGBAClr *) &fImage[index];

        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                IBASECOLOR2RGBA(rgba[i], Clr.color(offset+i));
        }
        bImgUpdated=true;

}

void IR8G8B8A8PixelBuffer::setPixelRect(const IGRect2D& rect, const IBaseColor* Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return ;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.fRight,
                h = (long) rect.fBottom,
                wid = (long) rect.width();
        unsigned long i, j, index, k;

        RGBAClr *rgba;

        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride) + (x << 2);
                rgba = (RGBAClr *) &fImage[index];
                k=i*wid;
                for(j=x; j < w; j++){
                        IBASECOLOR2RGBA(rgba[j], Clr[k+j]);
                }
        }
        bImgUpdated=true;
}

void IR8G8B8A8PixelBuffer::buildMaskImage(const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
        unsigned long index, i, j;
        RGBAClr *rgb;
        iBYTE   r=Clr.redMix(),
                        g=Clr.greenMix(),
                        b=Clr.blueMix();

        for(i=0; i < fHeight; i++){
                index = (FLIPEDY(i) * fPixelStride);
                rgb = (RGBAClr *)&fImage[index];
                for(j=0; j < fWidth; j++){
                        if((rgb[j].R == r) &&
                                (rgb[j].G == g) &&
                                (rgb[j].B == b)){
                                        rgb[j].R = 0xff;
                                        rgb[j].G = 0xff;
                                        rgb[j].B = 0xff;
                        } else {
                                        rgb[j].R = 0x00;
                                        rgb[j].G = 0x00;
                                        rgb[j].B = 0x00;
                        }
                }
        }
        bImgUpdated=true;
}

void IR8G8B8A8PixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        IPixelBuffer::writeToStream(toWhere);


}
void IR8G8B8A8PixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        IPixelBuffer::readFromStream(fromWhere);

}


//IR8G8B8PixelBuffer

IR8G8B8PixelBuffer::IR8G8B8PixelBuffer ()
{
        fBitsPerPixel = 24;
}

IR8G8B8PixelBuffer::~IR8G8B8PixelBuffer ()
{
}
IR8G8B8PixelBuffer::IR8G8B8PixelBuffer (unsigned long width,
				unsigned long height,
				const iBYTE* imageData,
				const IRGBAColorArray* clrArry)
{
        IFUNCTRACE_DEVELOP();
        if(width && height){
                fWidth = width;
                fHeight = height;
                fBitsPerPixel = 24;
                fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
                initializeDeviceContext();
		if(imageData)
			convertToNativeImage(imageData);
		else
			convertToNativeImage(INIT_BUFFER);
        }
}

IR8G8B8PixelBuffer::IR8G8B8PixelBuffer (const nativeImage bitmap)
{
        IFUNCTRACE_DEVELOP();
        imageToMem(bitmap, 24, 0, NULL);
        initializeDeviceContext();
		convertToNativeImage();
}


IR8G8B8PixelBuffer::IR8G8B8PixelBuffer (
                const IR8G8B8PixelBuffer& source,
                const IGRect2D& rect)
{
        IFUNCTRACE_DEVELOP();
        int i, k;
        fRefCount=1;
        srcImageFormat = source.srcImageFormat;
        bTransparencyEnabled  = source.bTransparencyEnabled;
        transparencyClr = source.transparencyClr;

        fWidth = (int)rect.width();
        fHeight = (int)rect.height();
        fBitsPerPixel = 24;
        fPixelStride = PIXELSTRIDE(fWidth, fBitsPerPixel);
        fImage = new iBYTE[fPixelStride * fHeight]; //RGBA * 8
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init

        IGRect2D imgRect(0, 0, source.fWidth, source.fHeight);

        if(imgRect.intersects(rect))
        {
                imgRect.intersectWith(rect);

                for(i=(int)(imgRect.fTop), k=0; i< (int)(imgRect.fBottom); i++, k++)
                {
                        memcpy(&fImage[(k*fPixelStride)],
                                &source.fImage[(i*source.fPixelStride) + (((int)(imgRect.fLeft)) * 3)],
                                        (size_t)(imgRect.width() * 3));
                }
        }
        initializeDeviceContext();
	convertToNativeImage();
}

IR8G8B8PixelBuffer* IR8G8B8PixelBuffer::clonePixelBuffer()
{
        IFUNCTRACE_DEVELOP();
	syncPixelBuffer();
        IR8G8B8PixelBuffer *newPixbuf = new IR8G8B8PixelBuffer();
	newPixbuf->copyPixelBuffer(*this);
        return newPixbuf;
}

void IR8G8B8PixelBuffer::pixel(const IGPoint2D& pnt, IBaseColor& Clr) const
{
        IFUNCTRACE_DEVELOP();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }

	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) * 3);
        RGBClr* rgb = (RGBClr *) &fImage[index];

        Clr = RGBA2IBASECOLOR(rgb[0].R, rgb[0].G, rgb[0].B, 0xFF);
}

void IR8G8B8PixelBuffer::pixelRow(const IGPoint2D& pnt, IRGBAColorArray& Clr) const
{
        IFUNCTRACE_DEVELOP();
        unsigned long length = Clr.numberOfColors();

        if (!length) return;    // we are done!

        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");

        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);

                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;

        length = (unsigned long)targetRect.width();


	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) * 3);
        long x, i;
        IRGBAColorArray iClr(fWidth - (unsigned long)(pnt.fX));
        RGBClr *rgb;

        rgb = (RGBClr *) &fImage[index];

        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                Clr.setColor(i, RGBA2IBASECOLOR(rgb[i].R,
                                        rgb[i].G,
                                        rgb[i].B,
                                        0xFF));
        }
                return ;

}

IBaseColor* IR8G8B8PixelBuffer::pixelRect(const IGRect2D& rect) const
{
        IFUNCTRACE_DEVELOP();
       if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return NULL;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.width(),
                h = (long) rect.height();
        unsigned long i, j, index;

	syncPixelBuffer();
        IBaseColor* iClr = new IBaseColor[w * h];
        RGBClr *rgb;

        for(i=0; i < h; i++){
                index = (FLIPEDY(y) * fPixelStride) + (x * 3);
                rgb = (RGBClr *) &fImage[index];
                for(j=0; j < w; j++){
                        iClr[(i*w)+j] = RGBA2IBASECOLOR(rgb[j].R,
                                                rgb[j].G,
                                                rgb[j].B,
                                                0xFF);
                }
        }
        return iClr;
}

void IR8G8B8PixelBuffer::setPixel(const IGPoint2D& pnt, const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) * 3);
        RGBClr *rgb = (RGBClr *) &fImage[index];
        IBASECOLOR2RGB(rgb[0], Clr);
        bImgUpdated=true;

}

void IR8G8B8PixelBuffer::setPixelRow(const IGPoint2D& pnt, const IRGBAColorArray& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        unsigned long length = Clr.numberOfColors();

        if (!length) return;    // we are done!

        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");

        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);

                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;

        length = (unsigned long)targetRect.width();

        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) * 3);
        long x, i;
        RGBClr *rgb = (RGBClr *) &fImage[index];

        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                IBASECOLOR2RGB(rgb[i], Clr.color(i));
        }
        bImgUpdated=true;

}

void IR8G8B8PixelBuffer::setPixelRect(const IGRect2D& rect, const IBaseColor* Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return ;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.fRight,
                h = (long) rect.fBottom,
                wid = (long) rect.width();
        unsigned long i, j, index, k;

        RGBClr *rgb;

        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride) + (x * 3);
                rgb = (RGBClr *) &fImage[index];
                k=i*wid;
                for(j=x; j < w; j++){
                        IBASECOLOR2RGB(rgb[j], Clr[k+j]);
                }
        }
        bImgUpdated=true;
}

void IR8G8B8PixelBuffer::buildMaskImage(const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
        unsigned long index, i, j;
        RGBClr *rgb;
        iBYTE   r=Clr.redMix(),
                        g=Clr.greenMix(),
                        b=Clr.blueMix();

        for(i=0; i < fHeight; i++){
                index = (FLIPEDY(i) * fPixelStride);
                rgb = (RGBClr *)&fImage[index];
                for(j=0; j < fWidth; j++){
                        if((rgb[j].R == r) &&
                                (rgb[j].G == g) &&
                                (rgb[j].B == b)){
                                        rgb[j].R = 0xff;
                                        rgb[j].G = 0xff;
                                        rgb[j].B = 0xff;
                        } else {
                                        rgb[j].R = 0x00;
                                        rgb[j].G = 0x00;
                                        rgb[j].B = 0x00;
                        }
                }
        }
        bImgUpdated=true;
}

void IR8G8B8PixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        IPixelBuffer::writeToStream(toWhere);

}
void IR8G8B8PixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        IPixelBuffer::readFromStream(fromWhere);
}


//IR5G6B5PixelBuffer
IR5G6B5PixelBuffer::IR5G6B5PixelBuffer ()
{
        fBitsPerPixel = 16;
}
IR5G6B5PixelBuffer::~IR5G6B5PixelBuffer ()
{
}
IR5G6B5PixelBuffer::IR5G6B5PixelBuffer (unsigned long width,
				unsigned long height,
				const iBYTE* imageData,
				const IRGBAColorArray* clrArry)
{
        IFUNCTRACE_DEVELOP();
        if(width && height){
                fWidth = width;
                fHeight = height;
                fBitsPerPixel = 16;
                fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
                initializeDeviceContext();
		if(imageData)
			convertToNativeImage(imageData);
		else
			convertToNativeImage(INIT_BUFFER);
        }
}
IR5G6B5PixelBuffer::IR5G6B5PixelBuffer (const nativeImage bitmap)
{
        IFUNCTRACE_DEVELOP();
        imageToMem(bitmap, 16, 0, NULL);
        initializeDeviceContext();
		convertToNativeImage();
}

IR5G6B5PixelBuffer::IR5G6B5PixelBuffer (
                const IR5G6B5PixelBuffer& source,
                const IGRect2D& rect)
{
        IFUNCTRACE_DEVELOP();
        int i, k;
        fRefCount=1;
        srcImageFormat = source.srcImageFormat;
        bTransparencyEnabled  = source.bTransparencyEnabled;
        transparencyClr = source.transparencyClr;

        fWidth = (int)rect.width();
        fHeight = (int)rect.height();
        fBitsPerPixel = 16;
        fPixelStride = PIXELSTRIDE(fWidth, fBitsPerPixel);
        fImage = new iBYTE[fPixelStride * fHeight]; //RGBA * 8
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init
        IGRect2D imgRect(0, 0, source.fWidth, source.fHeight);
        if(imgRect.intersects(rect))
        {
                imgRect.intersectWith(rect);
                for(i=(int)(imgRect.fTop), k=0; i< (int)(imgRect.fBottom); i++, k++)
                {
                        memcpy(&fImage[(k*fPixelStride)],
                                &source.fImage[(i*source.fPixelStride) + (((int)(imgRect.fLeft)) * 2)],
                                        (size_t)(imgRect.width() * 2));
                }
        }
        initializeDeviceContext();
	convertToNativeImage();
}
IR5G6B5PixelBuffer* IR5G6B5PixelBuffer::clonePixelBuffer()
{
        IFUNCTRACE_DEVELOP();
	syncPixelBuffer();
        IR5G6B5PixelBuffer *newPixbuf = new IR5G6B5PixelBuffer();
	newPixbuf->copyPixelBuffer(*this);
        return newPixbuf;
}

void IR5G6B5PixelBuffer::pixel(const IGPoint2D& pnt, IBaseColor& Clr) const
{
        IFUNCTRACE_DEVELOP();
       if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 1);
        R5G6B5 *rgb = (R5G6B5 *) &fImage[index];
        Clr = RGBA2IBASECOLOR(rgb[0].R, rgb[0].G, rgb[0].B, 0xFF);
}

void IR5G6B5PixelBuffer::pixelRow(const IGPoint2D& pnt, IRGBAColorArray& Clr) const
{
        IFUNCTRACE_DEVELOP();
        unsigned long length = Clr.numberOfColors();
       if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();

	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 1);
        long x, i;
        IRGBAColorArray iClr(fWidth - (unsigned long)(pnt.fX));
        R5G6B5 *rgb = (R5G6B5 *) &fImage[index];
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                Clr.setColor(i, RGBA2IBASECOLOR(rgb[i].R, rgb[i].G, rgb[i].B, 0xFF));
        }
}

IBaseColor* IR5G6B5PixelBuffer::pixelRect(const IGRect2D& rect) const
{
        IFUNCTRACE_DEVELOP();
       if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return NULL;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.width(),
                h = (long) rect.height();
	syncPixelBuffer();
        unsigned long i, j, index;
        IBaseColor* iClr = new IBaseColor[w * h];
        R5G6B5 *rgb;
        for(i=0; i < h; i++){
                index = (FLIPEDY(y) * fPixelStride) + (x << 1);
                rgb = (R5G6B5 *) &fImage[index];
                for(j=0; j < w; j++){
                        iClr[(i*w)+j] = RGBA2IBASECOLOR(rgb[j].R, rgb[j].G, rgb[j].B, 0xFF);
                }
        }
        return iClr;
}

void IR5G6B5PixelBuffer::setPixel(const IGPoint2D& pnt, const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 1);
        R5G6B5 *rgb = (R5G6B5 *) &fImage[index];
        IBASECOLOR2R5G6B5(rgb[0], Clr);
        bImgUpdated=true;
}

void IR5G6B5PixelBuffer::setPixelRow(const IGPoint2D& pnt, const IRGBAColorArray& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        unsigned long length = Clr.numberOfColors();
        if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) << 1);
        long x, i;
        R5G6B5 *rgb = (R5G6B5 *) &fImage[index];
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                IBASECOLOR2R5G6B5(rgb[i], Clr.color(i));
        }
        bImgUpdated=true;
}

void IR5G6B5PixelBuffer::setPixelRect(const IGRect2D& rect, const IBaseColor* Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return ;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.fRight,
                h = (long) rect.fBottom,
                wid = (long) rect.width();
        unsigned long i, j, index, k;
        R5G6B5 *rgb;
        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride) + (x << 1);
                rgb = (R5G6B5 *) &fImage[index];
                k=i*wid;
                for(j=x; j < w; j++){
                        IBASECOLOR2R5G6B5(rgb[j], Clr[k+j]);
                }
        }
        bImgUpdated=true;
}
void IR5G6B5PixelBuffer::buildMaskImage(const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
        unsigned long index, i, j;
        R5G6B5 *rgb;
        iBYTE   r=Clr.redMix(),
                        g=Clr.greenMix(),
                        b=Clr.blueMix();
        for(i=0; i < fHeight; i++){
                index = (FLIPEDY(i) * fPixelStride);
                rgb = (R5G6B5 *)&fImage[index];
                for(j=0; j < fWidth; j++){
                        if((rgb[j].R == r) &&
                                (rgb[j].G == g) &&
                                (rgb[j].B == b)){
                                        rgb[j].R = rgb[j].B = rgb[j].G = 0xff;
                        } else {
                                        rgb[j].R = rgb[j].B = rgb[j].G = 0x00;
                        }
                }
        }
        bImgUpdated=true;
}

void IR5G6B5PixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        IPixelBuffer::writeToStream(toWhere);
}
void IR5G6B5PixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        IPixelBuffer::readFromStream(fromWhere);
}

//ICLUT8PixelBuffer
ICLUT8PixelBuffer::ICLUT8PixelBuffer () :
        fColorTable(0)
{
        fBitsPerPixel = 8;
}
ICLUT8PixelBuffer::~ICLUT8PixelBuffer ()
{
}
ICLUT8PixelBuffer::ICLUT8PixelBuffer (unsigned long width,
				unsigned long height,
				const iBYTE* imageData,
				const IRGBAColorArray* clrArry):
	fColorTable(0)
{
        IFUNCTRACE_DEVELOP();
        if(width && height){
                fWidth = width;
                fHeight = height;
                fBitsPerPixel = 8;
                fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
		if(clrArry)
			fColorTable = *clrArry;
		else
			fColorTable = getSystemColors(256);
                initializeDeviceContext();
		if(imageData)
			convertToNativeImage(imageData);
		else
			convertToNativeImage(INIT_BUFFER);
        }
}
ICLUT8PixelBuffer::ICLUT8PixelBuffer (const nativeImage bitmap) :
        fColorTable(0)
{
        IFUNCTRACE_DEVELOP();
        imageToMem(bitmap, 8, 256, &fColorTable);
        initializeDeviceContext();
		convertToNativeImage();
}


ICLUT8PixelBuffer::ICLUT8PixelBuffer (
                const ICLUT8PixelBuffer& source,
                const IGRect2D& rect)
{
        IFUNCTRACE_DEVELOP();
        int i, k;
        fRefCount=1;
        srcImageFormat = source.srcImageFormat;
        bTransparencyEnabled  = source.bTransparencyEnabled;
        transparencyClr = source.transparencyClr;

        fWidth = (int)rect.width();
        fHeight = (int)rect.height();
        fBitsPerPixel = 8;
        fPixelStride = PIXELSTRIDE(fWidth, fBitsPerPixel);
        fImage = new iBYTE[fPixelStride * fHeight]; //RGBA * 8
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init
        IGRect2D imgRect(0, 0, source.fWidth, source.fHeight);
        if(imgRect.intersects(rect))
        {
                imgRect.intersectWith(rect);
                for(i=(int)(imgRect.fTop), k=0; i< (int)(imgRect.fBottom); i++, k++)
                {
                        memcpy(&fImage[(k*fPixelStride)],
                                &source.fImage[(i*source.fPixelStride) + ((int)(imgRect.fLeft))],
                                        (size_t)imgRect.width());
                }
        }
        //fColorTable = *(new IRGBAColorArray(source.fColorTable));
        fColorTable = source.fColorTable;
        initializeDeviceContext();
	convertToNativeImage();
}

ICLUT8PixelBuffer* ICLUT8PixelBuffer::clonePixelBuffer()
{
	IFUNCTRACE_DEVELOP();
	syncPixelBuffer();
	ICLUT8PixelBuffer *newPixbuf = new ICLUT8PixelBuffer();
	newPixbuf->fColorTable = fColorTable;
	newPixbuf->copyPixelBuffer(*this);
	return newPixbuf;
}

void ICLUT8PixelBuffer::pixel(const IGPoint2D& pnt, IBaseColor& Clr) const
{
        IFUNCTRACE_DEVELOP();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + ((unsigned long)pnt.fX);
        Clr = fColorTable.color( fImage[index]);
}
void ICLUT8PixelBuffer::pixelRow(const IGPoint2D& pnt, IRGBAColorArray& Clr) const
{
        IFUNCTRACE_DEVELOP();
        unsigned long length = Clr.numberOfColors();
       if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();
	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + ((unsigned long)pnt.fX);
        long x, i;
        IRGBAColorArray iClr(fWidth - (unsigned long)(pnt.fX));
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                Clr.setColor(i, fColorTable.color( fImage[index]));
                index += 1;
        }
}

IBaseColor* ICLUT8PixelBuffer::pixelRect(const IGRect2D& rect) const
{
        IFUNCTRACE_DEVELOP();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return NULL;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.width(),
                h = (long) rect.height();
	syncPixelBuffer();
        unsigned long i, j, index;
        IBaseColor* iClr = new IBaseColor[w * h];
        for(i=0; i < h; i++){
                index = (FLIPEDY(y) * fPixelStride) + x;
                for(j=0; j < w; j++){
                        iClr[(i*w)+j] = fColorTable.color( fImage[index]);
                        index += 1;
                }
        }
        return iClr;
}

void ICLUT8PixelBuffer::setPixel(const IGPoint2D& pnt, const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + ((unsigned long)pnt.fX);
        iBYTE data = matchRGBAColor(&fColorTable,
						Clr.redMix(),
						Clr.greenMix(),
						Clr.blueMix(),
						Clr.opacity());
        fImage[index] = data;
        bImgUpdated=true;
}

void ICLUT8PixelBuffer::setPixelRow(const IGPoint2D& pnt, const IRGBAColorArray& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        unsigned long length = Clr.numberOfColors();
        if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + ((unsigned long)pnt.fX);
        long x, i;
        iBYTE data;
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                data = matchRGBAColor(&fColorTable,
					Clr.color(i).redMix(),
					Clr.color(i).greenMix(),
					Clr.color(i).blueMix(),
					Clr.color(i).opacity());
                fImage[index + i] = data;
        }
        bImgUpdated=true;
}

void ICLUT8PixelBuffer::setPixelRect(const IGRect2D& rect, const IBaseColor* Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return ;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.fRight,
                h = (long) rect.fBottom,
                wid = (long) rect.width();
        unsigned long i, j, index, k;
        iBYTE data;
        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride) + x ;
                k=i*wid;
                for(j=x; j < w; j++){
                        data = matchRGBAColor(&fColorTable,
						Clr[k+j].redMix(),
						Clr[k+j].greenMix(),
						Clr[k+j].blueMix(),
						Clr[k+j].opacity());
                        fImage[index + j] = data;
                }
        }
        bImgUpdated=true;
}
void ICLUT8PixelBuffer::buildMaskImage(const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
			iBYTE mask = matchRGBAColor(&fColorTable,
							Clr.redMix(),
							Clr.greenMix(),
							Clr.blueMix(),
							Clr.opacity());
			iBYTE black = matchRGBAColor(&fColorTable, 0,0,0,0xFF);
			iBYTE white = matchRGBAColor(&fColorTable, 0xFF,0xFF,0xFF,0xFF);
        unsigned long index, i, j;
        for(i=0; i < fHeight; i++){
                index = (FLIPEDY(i) * fPixelStride);
                for(j=0; j < fWidth; j++){
                        fImage[index + j] = (fImage[index + j] == mask) ? white : black;
                }
        }
        bImgUpdated=true;
}

void ICLUT8PixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        IPixelBuffer::writeToStream(toWhere);
        fColorTable >>= toWhere;
}
void ICLUT8PixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        IPixelBuffer::readFromStream(fromWhere);
        fColorTable <<= fromWhere;
}


//ICLUT4PixelBuffer
ICLUT4PixelBuffer::ICLUT4PixelBuffer () :
        fColorTable(0)
{
        fBitsPerPixel = 4;
}
ICLUT4PixelBuffer::~ICLUT4PixelBuffer ()
{
}
ICLUT4PixelBuffer::ICLUT4PixelBuffer (unsigned long width,
				unsigned long height,
				const iBYTE* imageData,
				const IRGBAColorArray* clrArry):
        fColorTable(0)
{
        IFUNCTRACE_DEVELOP();
        if(width && height){
                fWidth = width;
                fHeight = height;
                fBitsPerPixel = 4;
                fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
		if(clrArry)
			fColorTable = *clrArry;
		else
			fColorTable = getSystemColors(16);
                initializeDeviceContext();
		if(imageData)
			convertToNativeImage(imageData);
		else
			convertToNativeImage(INIT_BUFFER);
        }
}
ICLUT4PixelBuffer::ICLUT4PixelBuffer (const nativeImage bitmap) :
        fColorTable(0)
{
        IFUNCTRACE_DEVELOP();
        imageToMem(bitmap, 4, 16, &fColorTable);
        initializeDeviceContext();
		convertToNativeImage();
}

ICLUT4PixelBuffer::ICLUT4PixelBuffer (
                const ICLUT4PixelBuffer& source,
                const IGRect2D& rect)
{
        IFUNCTRACE_DEVELOP();
        unsigned int i, j, k, l;
        fRefCount=1;
        srcImageFormat = source.srcImageFormat;
        bTransparencyEnabled  = source.bTransparencyEnabled;
        transparencyClr = source.transparencyClr;

        fWidth = (int)rect.width();
        fHeight = (int)rect.height();
        fBitsPerPixel = 4;
        fPixelStride = PIXELSTRIDE(fWidth, fBitsPerPixel);
        fImage = new iBYTE[fPixelStride * fHeight]; //RGBA * 8
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init
        IGRect2D imgRect(0, 0, source.fWidth, source.fHeight);
        if(imgRect.intersects(rect))
        {
                imgRect.intersectWith(rect);
                unsigned long readstride, writestride;
                for(i=(int)(imgRect.fTop), k=0; i< (int)(imgRect.fBottom); i++, k++)
                {
                        readstride = i*source.fPixelStride;
                        writestride = k*fPixelStride;
                        for(j=(int)(imgRect.fLeft), l=0; j< (int)(imgRect.fRight); j++, l++)
                        {
                                SETNIBBLE(fImage[writestride + (l>>1)],
                                                GETNIBBLE(source.fImage[readstride + (j>>1)], j&0x1),
                                                l&0x1);
                        }
                }
        }
        fColorTable = source.fColorTable;
        //fColorTable = *(new IRGBAColorArray(source.fColorTable));
        initializeDeviceContext();
	convertToNativeImage();
}
ICLUT4PixelBuffer* ICLUT4PixelBuffer::clonePixelBuffer()
{
        IFUNCTRACE_DEVELOP();
	syncPixelBuffer();
        ICLUT4PixelBuffer *newPixbuf = new ICLUT4PixelBuffer();
        newPixbuf->fColorTable = fColorTable;
	newPixbuf->copyPixelBuffer(*this);
        return newPixbuf;
}
void ICLUT4PixelBuffer::pixel(const IGPoint2D& pnt, IBaseColor& Clr) const
{
        IFUNCTRACE_DEVELOP();
       if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) >> 1);
        iBYTE dTmp = fImage[index];
        dTmp = GETNIBBLE(dTmp, (((unsigned long)pnt.fX) & 0x1));
        Clr =  fColorTable.color(dTmp);
}
void ICLUT4PixelBuffer::pixelRow(const IGPoint2D& pnt, IRGBAColorArray& Clr) const
{
        IFUNCTRACE_DEVELOP();
        unsigned long length = Clr.numberOfColors();
        if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();

	syncPixelBuffer();
        unsigned long index = (FLIPEDY((unsigned long)pnt.fY)) * fPixelStride;
        long x, i;
        IRGBAColorArray iClr(fWidth - (unsigned long)(pnt.fX));
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                Clr.setColor(i, fColorTable.color(GETNIBBLE(fImage[index+(x>>1)], x&0x1)));
                if(x & 1)
                        index += 1;
        }
}

IBaseColor* ICLUT4PixelBuffer::pixelRect(const IGRect2D& rect) const
{
        IFUNCTRACE_DEVELOP();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return NULL;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.width(),
                h = (long) rect.height();
	syncPixelBuffer();
        unsigned long i, j, index;
        IBaseColor* iClr = new IBaseColor[w * h];
        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride);
                for(j=x; j < w; j++){
                        iClr[(i*w)+j]= fColorTable.color(GETNIBBLE(fImage[index + (j >>1)], j&0x1));
                        if(j & 1)
                                index += 1;
                }
        }
        return iClr;
}

void ICLUT4PixelBuffer::setPixel(const IGPoint2D& pnt, const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) >> 1);
        iBYTE data = matchRGBAColor(&fColorTable,
					Clr.redMix(),
					Clr.greenMix(),
					Clr.blueMix(),
					Clr.opacity());
        SETNIBBLE(fImage[index], data, (((unsigned long)pnt.fX) & 0x1));
        bImgUpdated=true;
}

void ICLUT4PixelBuffer::setPixelRow(const IGPoint2D& pnt, const IRGBAColorArray& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        unsigned long length = Clr.numberOfColors();
        if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();

        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride);
        long x, i;
        iBYTE data;
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                data = matchRGBAColor(&fColorTable,
					Clr.color(i).redMix(),
					Clr.color(i).greenMix(),
					Clr.color(i).blueMix(),
					Clr.color(i).opacity());
                SETNIBBLE(fImage[index+(x>>1)], data, x&0x1);
        }
        bImgUpdated=true;
}

void ICLUT4PixelBuffer::setPixelRect(const IGRect2D& rect, const IBaseColor* Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return ;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.fRight,
                h = (long) rect.fBottom,
                wid = (long) rect.width();
        unsigned long i, j, index, k;
        iBYTE data;
        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride);
                k=i*wid;
                for(j=x; j < w; j++){
                        data = matchRGBAColor(&fColorTable,
				Clr[k+j].redMix(),
				Clr[k+j].greenMix(),
				Clr[k+j].blueMix(),
				Clr[k+j].opacity());
                        SETNIBBLE(fImage[index+((x+j)>>1)], data, (x+j)&0x1);
                }
        }
        bImgUpdated=true;
}
void ICLUT4PixelBuffer::buildMaskImage(const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		iBYTE mask = matchRGBAColor(&fColorTable,
						Clr.redMix(),
						Clr.greenMix(),
						Clr.blueMix(),
						Clr.opacity());
		iBYTE black = matchRGBAColor(&fColorTable, 0,0,0,0xFF);
		iBYTE white = matchRGBAColor(&fColorTable, 0xFF,0xFF,0xFF,0xFF);
        unsigned long index, i, j;
        for(i=0; i < fHeight; i++){
                index = (FLIPEDY(i) * fPixelStride);
                for(j=0; j < fWidth; j++){
                        if(GETNIBBLE(fImage[index + (j>>1)], j&0x1) == mask)
                                SETNIBBLE(fImage[index + (j>>1)], white, j&0x1);
                        else
                                SETNIBBLE(fImage[index + (j>>1)], black, j&0x1);
                }
        }
        bImgUpdated=true;
}

void ICLUT4PixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        IPixelBuffer::writeToStream(toWhere);
        fColorTable >>= toWhere;
}
void ICLUT4PixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        IPixelBuffer::readFromStream(fromWhere);
        fColorTable <<= fromWhere;
}


//IMonochromePixelBuffer
IMonochromePixelBuffer::IMonochromePixelBuffer () :
        fColorTable(0)
{
        fBitsPerPixel = 1;
}
IMonochromePixelBuffer::~IMonochromePixelBuffer ()
{
}
IMonochromePixelBuffer::IMonochromePixelBuffer (unsigned long width,
				unsigned long height,
				const iBYTE* imageData,
				const IRGBAColorArray* clrArry) :
        fColorTable(0)
{
        IFUNCTRACE_DEVELOP();
        if(width && height){
                fWidth = width;
                fHeight = height;
                fBitsPerPixel = 1;
                fPixelStride =  PIXELSTRIDE(fWidth, fBitsPerPixel);
		if(clrArry)
			fColorTable = *clrArry;
		else
			fColorTable = getSystemColors(2);
                initializeDeviceContext();
		if(imageData)
			convertToNativeImage(imageData);
		else
			convertToNativeImage(INIT_BUFFER);
        }
}

IMonochromePixelBuffer::IMonochromePixelBuffer (const nativeImage bitmap)
:
        fColorTable(0)
{
        IFUNCTRACE_DEVELOP();
        imageToMem(bitmap, 1, 2, &fColorTable);
        initializeDeviceContext();
		convertToNativeImage();
}

IMonochromePixelBuffer::IMonochromePixelBuffer (
                const IMonochromePixelBuffer& source,
                const IGRect2D& rect)
{
        IFUNCTRACE_DEVELOP();
        int i, j, k, l;
        fRefCount=1;
        srcImageFormat = source.srcImageFormat;
        bTransparencyEnabled  = source.bTransparencyEnabled;
        transparencyClr = source.transparencyClr;

        fWidth = (int)rect.width();
        fHeight = (int)rect.height();
        fBitsPerPixel = 1;
        fPixelStride = PIXELSTRIDE(fWidth, fBitsPerPixel);
        fImage = new iBYTE[fPixelStride * fHeight]; //RGBA * 8
	memset(fImage, 0x0, (size_t)(fPixelStride * fHeight)); //Init
        IGRect2D imgRect(0, 0, source.fWidth, source.fHeight);
        unsigned long readstride, writestride;
        if(imgRect.intersects(rect))
        {
                imgRect.intersectWith(rect);
                for(i=(int)(imgRect.fTop), k=0; i< (int)(imgRect.fBottom); i++, k++)
                {
                        readstride = i*source.fPixelStride;
                        writestride = k*fPixelStride;
                        for(j=(int)(imgRect.fLeft), l=0; j< (int)(imgRect.fRight); j++, l++)
                        {
                                SETBIT(fImage[writestride + (l>>3)],
                                                GETBIT(source.fImage[readstride + (j>>3)], j&0x7),
                                                l&0x7);
                        }
                }
        }
        fColorTable = source.fColorTable;
        //fColorTable = *(new IRGBAColorArray(source.fColorTable));
        initializeDeviceContext();
	convertToNativeImage();
}
IMonochromePixelBuffer* IMonochromePixelBuffer::clonePixelBuffer()
{
        IFUNCTRACE_DEVELOP();
	syncPixelBuffer();
        IMonochromePixelBuffer *newPixbuf = new IMonochromePixelBuffer();
        newPixbuf->fColorTable = fColorTable;
	newPixbuf->copyPixelBuffer(*this);
        return newPixbuf;
}
void IMonochromePixelBuffer::pixel(const IGPoint2D& pnt, IBaseColor& Clr) const
{
        IFUNCTRACE_DEVELOP();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) >> 3);
        iBYTE dTmp = fImage[index], mod;
        mod = ((unsigned long) pnt.fX) & 0x7;
        Clr = fColorTable.color( GETBIT(fImage[index], mod));
}
void IMonochromePixelBuffer::pixelRow(const IGPoint2D& pnt, IRGBAColorArray& Clr) const
{
        IFUNCTRACE_DEVELOP();
        unsigned long length = Clr.numberOfColors();
        if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();

	syncPixelBuffer();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride);
        long x, i, mod;
        IRGBAColorArray iClr(fWidth - (unsigned long)(pnt.fX));
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                mod = x & 0x7;
                Clr.setColor(i, fColorTable.color( GETBIT(fImage[index+x>>3], mod)));
                if(mod == 7)
                        index ++;
        }
}

IBaseColor* IMonochromePixelBuffer::pixelRect(const IGRect2D& rect) const
{
        IFUNCTRACE_DEVELOP();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return NULL;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.width(),
                h = (long) rect.height();
	syncPixelBuffer();
        unsigned long i, j, index, mod;
        IBaseColor* iClr = new IBaseColor[w * h];
        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride);
                for(j=0; j < w; j++){
                        mod = (x + j) & 0x7;
                        iClr[(i*w)+j]= fColorTable.color(GETBIT(fImage[index+((x + j)>>3)],mod));
                        if(mod == 7)
                                index ++;
                }
        }
        return iClr;
}

void IMonochromePixelBuffer::setPixel(const IGPoint2D& pnt, const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!pointInImage(pnt)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Point (%f, %f)out of Image Bounds \n", pnt.fX, pnt.fY);
#endif
                return ;
        }
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride) + (((unsigned long)pnt.fX) >> 3);
        if(fColorTable.color(1) == Clr)
                SETBIT(fImage[index], 1, (((unsigned long) pnt.fX) & 0x7));
        else
                SETBIT(fImage[index], 0, (((unsigned long) pnt.fX) & 0x7));
        bImgUpdated=true;
}

void IMonochromePixelBuffer::setPixelRow(const IGPoint2D& pnt, const IRGBAColorArray& Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        unsigned long length = Clr.numberOfColors();
        if (!length) return;    // we are done!
        IGRect2D requestedRect(
                pnt.fX, pnt.fY, pnt.fX + (GCoordinate)length, pnt.fY+1);
        IGRect2D fBounds(0, 0, fWidth, fHeight);
        ParameterAssert(
                fBounds.intersects(requestedRect),
                "Position of target pixel row is outside the target image.");
        IGRect2D targetRect;
        long offset = 0;
        bool clipped = !fBounds.contains(requestedRect);
        if (clipped) {
                targetRect = fBounds;
                targetRect.intersectWith(requestedRect);
                if (targetRect.fLeft < fBounds.fLeft)   // left clipped ...
                        offset = (long)(fBounds.fLeft - targetRect.fLeft);
        }
        else
                targetRect = requestedRect;
        length = (unsigned long)targetRect.width();
        unsigned long index = (FLIPEDY(((unsigned long)pnt.fY)) * fPixelStride);
        long x, i;
        for(x = (unsigned long)(pnt.fX), i=offset; i < length; x++, i++)
        {
                if(fColorTable.color(1) == Clr.color(i))
                        SETBIT(fImage[index + x>>3], 1, (x & 0x7));
                else
                        SETBIT(fImage[index + x>>3], 0, (x & 0x7));
        }
        bImgUpdated=true;
}

void IMonochromePixelBuffer::setPixelRect(const IGRect2D& rect, const IBaseColor* Clr)
{
        IFUNCTRACE_DEVELOP();
		syncPixelBuffer();
        if(!rectInImage(rect)){
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "Rect out of Image Bounds \n");
#endif
                return ;
        }
        long x = (long) rect.fLeft,
                y = (long) rect.fTop,
                w = (long) rect.fRight,
                h = (long) rect.fBottom,
                wid = (long) rect.width();
        unsigned long i, j, index, k;
        for(i=y; i < h; i++){
                index = (FLIPEDY(i) * fPixelStride);
                k=i*wid;
                for(j=x; j < w; j++){
                        if(fColorTable.color(1) == Clr[k+j])
                                SETBIT(fImage[index+((x+j)>>3)], 1, ((x + j) & 0x7));
                        else
                                SETBIT(fImage[index+((x+j)>>3)], 0, ((x + j) & 0x7));
                }
        }
        bImgUpdated=true;
}

void IMonochromePixelBuffer::buildMaskImage(const IBaseColor& Clr)
{
        IFUNCTRACE_DEVELOP();
//what do we do in the case of 2 colors only
#if 0
		iBYTE mask = matchRGBAColor(&fColorTable,
						Clr.redMix(),
						Clr.greenMix(),
						Clr.blueMix(),
						Clr.opacity());
		iBYTE black = matchRGBAColor(&fColorTable, 0,0,0,0xFF);
		iBYTE white = matchRGBAColor(&fColorTable, 0xFF,0xFF,0xFF,0xFF);
        unsigned long index, i, j;
        for(i=0; i < fHeight; i++){
                index = (FLIPEDY(i) * fPixelStride);
                for(j=0; j < fWidth; j++){
                        if(GETBIT(fImage[index + (j>>3)], j&0x1) == mask)
                                SETBIT(fImage[index + (j>>3)], white, j&0x7);
                        else
                                SETBIT(fImage[index + (j>>3)], black, j&0x7);
                }
        }
        bImgUpdated=true;
#endif
}

void IMonochromePixelBuffer::writeToStream  (IDataStream& toWhere) const
{
        IFUNCTRACE_DEVELOP();
        IStreamOutFrame streamFrame(toWhere);
        IPixelBuffer::writeToStream(toWhere);
        fColorTable >>= toWhere;
}
void IMonochromePixelBuffer::readFromStream (IDataStream& fromWhere)
{
        IFUNCTRACE_DEVELOP();
        IStreamInFrame streamFrame(fromWhere);
        IPixelBuffer::readFromStream(fromWhere);
        fColorTable <<= fromWhere;
}
