// Revision: 02 1.4 source/albert/graph2d/ixlfdstr.cpp, 2d, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ixlfdstr.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ixlfdstr.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   Licensed Materials - Property of IBM                                       *
*   (C) Copyright IBM Corporation 1992, 1997 All Rights Reserved.              *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

// <include-files>
#include <ixlfdstr.hpp>
#include <ixdc.hpp>
#include <iexcept.hpp>
#include <itrace.hpp>
#include <istparse.hpp>

#include <istatics.hpp>
#include <nl_types.h>
#include <langinfo.h>  

// Segment definitions
#ifdef IC_PAGETUNE
#define _IXLFDSTR_CPP_
#include <ipagetun.h>
#endif
// </include-files>

IString const& key(IXLFDString const& f)
{
   return f.keyName();
}

static IFontList* gValidFontList = 0;
IString IXLFDString::fontListLocale("");

/*------------------------------------------------------------------------------
| IXLFDString::getFontListForCurrentLocale                                            |
------------------------------------------------------------------------------*/
IFontList* IXLFDString::getFontListForCurrentLocale( Display* display )
{  
  IString currentLocale = nl_langinfo(CODESET);
  if( currentLocale != IXLFDString::fontListLocale )
  {
    if (gValidFontList) 
    {
      gValidFontList->removeAll();
    }
  }

  IXLFDString::fontListLocale = currentLocale;

  if ( (! gValidFontList) || (gValidFontList->isEmpty ()) )
  {
    if( !gValidFontList )
      gValidFontList = new IFontList;    
     // Create a list of Valis fonts for this locale
    char** missing_charset_list_return;
    int missing_charset_count_return;
    char* def_string_return;
    XFontSet aFontSet = XCreateFontSet(display
                            ,"-*"								
                            ,&missing_charset_list_return
                            ,&missing_charset_count_return
                            ,&def_string_return);
    
    XFontStruct **fontStructList;
    char **fontNameList;
    int fntCount = XFontsOfFontSet(aFontSet,&fontStructList,&fontNameList);
    IXLFDString XLFDValidStr;
        
    if( fntCount > 0 )
    {
      int patGuess=0;
      if( fntCount > 1 )
        patGuess=1;
      
      IXLFDString patternXLFD( fontNameList[patGuess] );
      XLFDValidStr.setWord(  patternXLFD.getWord(IXLFDString::CHARSETREGISTRY)
                        , IXLFDString::CHARSETREGISTRY);
      XLFDValidStr.setWord(  patternXLFD.getWord(IXLFDString::CHARSETENCODING)
                        , IXLFDString::CHARSETENCODING);
    }
    
    int     nameEntries;
    char    **nameList;
    nameList = XListFonts(  display
                          , XLFDValidStr.asString()
                          , 5000          // arbitrary large value
                          , &nameEntries);
                          
    for (int i = 0; i < nameEntries ; i++)
    {
      IXLFDString newFont(nameList[i]);
      // Don't add it to the list if there is no family name.  (Usability)
      if ( (newFont.familyName().length() != 0) 
        || (newFont.familyName() =="*") )
      {
        gValidFontList->add(newFont);
      }
    }
    if (aFontSet)
      XFreeFontSet(display, aFontSet);

    XFreeFontNames(nameList);
      
    adoptStaticObject(gValidFontList);
  }  //if !gValidFontList
 
 return gValidFontList;
}

/*------------------------------------------------------------------------------
| IXLFDString::IXLFDString                                                     |
------------------------------------------------------------------------------*/
IXLFDString::IXLFDString() :
   xlfdCl(),
   key1(),
   keyLong()
{
  for(int i=1;i<=14;i++)
      xlfdCl += IString("*") + " ";

  xlfdCl.stripTrailing(" ");
}

/*------------------------------------------------------------------------------
| IXLFDString::IXLFDString                                                     |
------------------------------------------------------------------------------*/
IXLFDString::IXLFDString( const char * rawXLFD ) :
   xlfdCl( rawXLFD ),
   key1(),
   keyLong()
{  
   xlfdCl.lowerCase();
   xlfdCl.stripTrailing(" ");
   
   xlfdCl.translate(" ", "|");
   xlfdCl.change("--", "-*-");
   xlfdCl.translate("-", " ");
   xlfdCl.stripLeading(" ");
   unsigned long numOfWords = xlfdCl.numWords();

  if( numOfWords == 1 )
  {
    setWord("*",1);
    setWord(rawXLFD,2);
  }  
  numOfWords = xlfdCl.numWords();
   for( int i=1;i<=14;i++)
   {
      if( i > numOfWords)
      {
        setWord("*",i);      
      }
   }

   // Set the key returned to the collection.  The key is the text in the
   // font description combo box.
   key1 = familyName();   
   keyLong = foundryName() + " " + familyName();   
}

IXLFDString::IXLFDString(const IXLFDString& orig):
        xlfdCl(orig.xlfdCl),
        key1(orig.key1),
        keyLong(orig.keyLong)
{
}

/*------------------------------------------------------------------------------
| IFontPrivateData::operator=                                                  |
------------------------------------------------------------------------------*/
IXLFDString& IXLFDString::operator=(const IXLFDString& orig)
{
   if (&orig != this)
      {        
        xlfdCl      = orig.xlfdCl;
        key1         = orig.key1;
        keyLong     = orig.keyLong;
      }
   return *this;
}


bool  IXLFDString :: operator == ( const IXLFDString &rhs ) const
{
  return( this->xlfdCl == rhs.xlfdCl);
}


/*------------------------------------------------------------------------------
| IXLFDString::~IXLFDString                                                    |
------------------------------------------------------------------------------*/
IXLFDString::~IXLFDString( )
{
}

/*------------------------------------------------------------------------------
| bool IXLFDString::isBold                                                     |
------------------------------------------------------------------------------*/
bool IXLFDString::isBold() const
{
   return( (weightName() == "bold") || (weightName() == "demibold") );
}

/*------------------------------------------------------------------------------
| bool IXLFDString::isItalic                                                   |
------------------------------------------------------------------------------*/
bool IXLFDString::isItalic() const
{
  return( (slant() == "i") || (slant() == "o") );
}

/*------------------------------------------------------------------------------
| bool IXLFDString::isScalable                                                 |
------------------------------------------------------------------------------*/
bool IXLFDString::isScalable() const
{
  return ( (xlfdCl.word(PIXELS) == "0") && (xlfdCl.word(AVGWIDTH) == "0") );
}

bool IXLFDString::isFixed() const
{
  return ( (xlfdCl.word(SPACING) == "m") || (xlfdCl.word(SPACING) == "c") );
}

/*------------------------------------------------------------------------------
| IXLFDString::foundryName                                                     |
------------------------------------------------------------------------------*/
IString IXLFDString::foundryName() const
{
  IString retVal = ChangeToXLFDFormat(xlfdCl.word(FOUNDRY));
  return retVal;
}

/*------------------------------------------------------------------------------
| IXLFDString::familyName                                                      |
------------------------------------------------------------------------------*/
IString IXLFDString::familyName() const
{
   return ChangeToXLFDFormat(xlfdCl.word(NAME));
}

/*------------------------------------------------------------------------------
| IXLFDString::weightName                                                      |
------------------------------------------------------------------------------*/
IString IXLFDString::weightName() const
{
   return ChangeToXLFDFormat(xlfdCl.word(WEIGHT));
}

/*------------------------------------------------------------------------------
| IXLFDString::slant                                                           |
------------------------------------------------------------------------------*/
IString IXLFDString::slant() const
{
   return ChangeToXLFDFormat(xlfdCl.word(SLANT));
}

/*------------------------------------------------------------------------------
| IXLFDString::spacing                                                         |
------------------------------------------------------------------------------*/
IString IXLFDString::spacing() const
{
   return ChangeToXLFDFormat(xlfdCl.word(SPACING));
}

/*------------------------------------------------------------------------------
| IXLFDString::charsetRegistry                                                 |
------------------------------------------------------------------------------*/
IString IXLFDString::charsetRegistry() const
{
   return ChangeToXLFDFormat(xlfdCl.word(CHARSETREGISTRY));
}

/*------------------------------------------------------------------------------
| IXLFDString::charsetEncoding                                                 |
------------------------------------------------------------------------------*/
IString IXLFDString::charsetEncoding() const
{
   return ChangeToXLFDFormat(xlfdCl.word(CHARSETENCODING));
}

/*------------------------------------------------------------------------------
| IXLFDString::keyName                                                         |
------------------------------------------------------------------------------*/
IString const& IXLFDString::keyName() const
{
   return key1;
}

/*------------------------------------------------------------------------------
| IXLFDString::keyNameLong                                                         |
------------------------------------------------------------------------------*/
IString const& IXLFDString::keyNameLong() const
{
   return keyLong;
}

/*------------------------------------------------------------------------------
| IXLFDString::asString                                                         |
------------------------------------------------------------------------------*/
IString const IXLFDString::asString() const
{
  return value();
}

/*------------------------------------------------------------------------------
| IXLFDString::pixelSize                                                       |
------------------------------------------------------------------------------*/
unsigned long IXLFDString::pixelSize() const
{
   return xlfdCl.word(PIXELS).asUnsigned();
}

/*------------------------------------------------------------------------------
| IXLFDString::pointSize                                                       |
------------------------------------------------------------------------------*/
unsigned long IXLFDString::pointSize() const
{
  unsigned long pointSize = xlfdCl.word(POINTSIZE).asUnsigned();
  return pointSize/10;
}


/*------------------------------------------------------------------------------
| IXLFDString::averageWidth                                                    |
------------------------------------------------------------------------------*/
unsigned long IXLFDString::averageWidth() const
{
  return xlfdCl.word(AVGWIDTH).asUnsigned();
}

/*------------------------------------------------------------------------------
| IXLFDString::getWord                                                    |
------------------------------------------------------------------------------*/
IString IXLFDString::getWord( short index)
{
  return( xlfdCl.word(index) );
}

/*------------------------------------------------------------------------------
| IXLFDString::setWord                                                    |
------------------------------------------------------------------------------*/
IXLFDString& IXLFDString::setWord( IString val, short index)
{
  IString newString;
  val.lowerCase();
  val.stripTrailing(" ");
  val.translate(" ", "|");
  val.stripLeading(" ");
  
  for(int i=1;i<=14;i++)
  {
    if( i==index)
      newString += val + " ";
    else    
      newString += xlfdCl.word(i) + " ";
  }
  newString.stripTrailing(" ");
  xlfdCl = newString;
      
   // Set the key returned to the collection.  The key is the text in the
   // font description combo box.
   key1 = familyName();
   keyLong = foundryName() + " " + familyName();

  return *this;
}

/*------------------------------------------------------------------------------
| IXLFDString::setPointSize                                                    |
------------------------------------------------------------------------------*/
IXLFDString& IXLFDString::setPointSize( unsigned long newPointSize )
{
   setWord(IString(newPointSize * 10),POINTSIZE);       
   return *this;
}

IString IXLFDString::value() const
{
  IString retVal = xlfdCl;
  retVal.translate(" ", "-");
  retVal.change("--", "-*-");
  retVal.translate("|", " ");  
  return (IString("-") + retVal);
}

IString IXLFDString::ChangeToXLFDFormat(const IString& val) const
{
  IString retVal = val;
  retVal.stripTrailing(" ");
  retVal.translate(" ", "-");
  retVal.change("--", "-*-");
  retVal.translate("|", " ");
  return retVal;
}

bool IXLFDString::isVector() const
{
  return( isScalable());
}
