/*******************************************************************************
* FILE NAME: ixfntset.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ixfntset.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   Licensed Materials - Property of IBM                                       *
*   (C) Copyright IBM Corporation 1992, 1997 All Rights Reserved.              *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Revision: 50 1.2.1.2 source/albert/graph2d/ixfntset.cpp, 2d, ioc.v400, 001006 

#ifdef IC_MOTIF
  #include <nl_types.h>
  #include <langinfo.h>  
#endif //IC_MOTIF
#include <ixfntset.hpp>
#include <istring.hpp>
#include <iprimlck.hpp>

XFontSet
IXFontSet::getFontSet(Display *dpy, const char *xlfd)
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP(IString("Requesting font: ") + IString(xlfd));
    if (! xlfd_cache.isEmpty()) {
        ISortedSet<IXLFDFontSet>::Cursor cursor(xlfd_cache);
        forICursor (cursor) {
            if ((strcasecmp((const char *)(xlfd_cache.elementAt(cursor).xlfd()),
                            xlfd) == 0) &&
                (xlfd_cache.elementAt(cursor).display() == dpy)) {

                ITRACE_DEVELOP("Using cached font.");
                return xlfd_cache.elementAt(cursor).fontset();
            }
        }
    }
    
    ITRACE_DEVELOP("Font not cached, use XCreateFontSet.");
    
    // nothing is found, create a new one.
    char** missing_charset_list_return;
    int missing_charset_count_return;
    char* def_string_return;
    XFontSet aFontSet = NULL;
    
    aFontSet = XCreateFontSet(dpy, xlfd,
                              &missing_charset_list_return,
                              &missing_charset_count_return,
                              &def_string_return);
    
    if (missing_charset_list_return)
        XFreeStringList(missing_charset_list_return);
    
    if (aFontSet)
        xlfd_cache.add(IXLFDFontSet(dpy, IString(xlfd), aFontSet));
    
    // aFontSet can be null
    return aFontSet;
}

IXFontSet::~IXFontSet() {
    if (! xlfd_cache.isEmpty()) {
        ISortedSet<IXLFDFontSet>::Cursor cursor(xlfd_cache);
        forICursor (cursor) {
            XFreeFontSet(xlfd_cache.elementAt(cursor).display(), 
                         xlfd_cache.elementAt(cursor).fontset());
        }
    }
}

static IXFontSet *ixfontset = 0;

XFontSet
IXCreateFontSet(Display *display, const char *xlfd)
{
    if (! ixfontset) {
        IPrimalLock lockinit;
        if (! ixfontset)
            ixfontset = new IXFontSet;
    }
    return ixfontset->getFontSet(display, xlfd);
}

XFontSet IXCreateFontSet( Display *display
                        , IXLFDString XLFDStr
                        , bool strict
                        , IXFontSet::PRIORITY priority)
{    
  IFontList* validFontList = IXLFDString::getFontListForCurrentLocale(display);
  
  XFontSet aFontSet=0;
  IXLFDString rateXLFD = XLFDStr;  //original XLFD to compare against
  XLFDStr.setWord("*",IXLFDString::RESX);
  XLFDStr.setWord("*",IXLFDString::RESY);    

  //used for querying the fonts on the system
  unsigned long maxNumFonts=5000; // Arbitrarily large limit for XListFonts.
  char** namesArray;
  int foundCount = 0;

  //check for valid font
  IFontList::Cursor myCursor( *validFontList );
  IString theKey = XLFDStr.keyName();
  
  if( validFontList->locateElementWithKey(XLFDStr.keyName(),myCursor) )
  {
  //a search for a quite specific font
    // ok - at this stage we should have a font - we were given a very poor
    // font choice so we will return "fixed font"
    int maxRate = 0;
    int maxRateIndex = 0;
    int tmpRate=0;
  
    //set the attributes in the IXLFDString to get a rate matching    
    //this will be less generis than the search string.                         

    myCursor.setToFirst();
    IXLFDString baseXLFDstr;
    for (validFontList->locateElementWithKey(theKey,myCursor)
        ; myCursor.isValid()
        ; validFontList->locateNextElementWithKey(theKey,myCursor))
    {
       IXLFDString possibleXLFD = validFontList->elementAt(myCursor);
       tmpRate = IXRateMatch(rateXLFD,possibleXLFD,priority);            
      if( tmpRate > maxRate )
      {       
        baseXLFDstr = possibleXLFD;
        maxRate = tmpRate;
      }
    }
  
    if( baseXLFDstr.isVector())
    {
      baseXLFDstr.setWord("*",IXLFDString::PIXELS);
    }
            
    // we do this to allow the internationaliztion features to work  
    baseXLFDstr.setWord("*",IXLFDString::AVGWIDTH);
    baseXLFDstr.setWord("*",IXLFDString::CHARSETREGISTRY);
    baseXLFDstr.setWord("*",IXLFDString::CHARSETENCODING);     
    
    if( baseXLFDstr.pointSize() == 0 )
      baseXLFDstr.setPointSize(rateXLFD.pointSize());
      
    baseXLFDstr.setWord("*",IXLFDString::PIXELS);
    baseXLFDstr.setWord("*",IXLFDString::SIXTHWORD);
    baseXLFDstr.setWord("*",IXLFDString::RESX);
    baseXLFDstr.setWord("*",IXLFDString::RESY);    
    
    // Search for and load a fontSet that satisfies the current locale.
    aFontSet = IXCreateFontSet(display, baseXLFDstr.asString());    
  }
  return( aFontSet );
}

int IXRateMatch(  IXLFDString& desiredXLFD
                , IXLFDString& possibleXLFD
                , IXFontSet::PRIORITY priority )
{
    int score = 0;    
    if (desiredXLFD.foundryName() == possibleXLFD.foundryName()) 
      score+=2;

    //special case if we deal with a vector font
    if( desiredXLFD.isVector() )
    {
      if ( possibleXLFD.isVector()) 
        score += 14;
    }
    else
    {      
      if (desiredXLFD.pointSize() == possibleXLFD.pointSize()) 
        score += 14;
      else if ( possibleXLFD.isVector())       
        score += 11;
      else if ( labs( desiredXLFD.pointSize() -
                        possibleXLFD.pointSize() ) < 3 )
      {
        score += 5;
      }
      else if ( labs( desiredXLFD.pointSize() -
                        possibleXLFD.pointSize() ) < 5 )
      {
        score += 4;
      }
      else if ( labs( desiredXLFD.pointSize() -
                        possibleXLFD.pointSize() ) < 10 )
      {
        score += 3;
      }
    }    

    if (desiredXLFD.isFixed() == possibleXLFD.isFixed()) 
    {
      if( priority == IXFontSet::PROP_PRIORITY)
        score+=6;
      else      
        score+=3;
    }
        
    if (desiredXLFD.familyName() == possibleXLFD.familyName()) 
      score += 25;

    
    if (desiredXLFD.isBold() == possibleXLFD.isBold()) 
      score+=1;

    if (desiredXLFD.isItalic() == possibleXLFD.isItalic())     
      score+=1;

    if (desiredXLFD.isVector() == possibleXLFD.isVector())
    {
      if( priority == IXFontSet::VECTOR_PRIORITY)
        score+=6;
      else      
        score+=3;
    }
    
    if (desiredXLFD.getWord(IXLFDString::RESX) == possibleXLFD.getWord(IXLFDString::RESX)) 
      score+=1;
    if (desiredXLFD.getWord(IXLFDString::RESY) == possibleXLFD.getWord(IXLFDString::RESY)) 
      score+=1;

    return score;
}

#pragma enum(4)
#pragma pack(push,4)

class IXFontSetStatic
{
public:
    ~IXFontSetStatic() { if (ixfontset) delete ixfontset; }
};

#pragma pack(pop)
#pragma enum(pop)

