// IBM Open Class Library
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// (C) Copyright IBM Corporation, 1992, 1997 All Rights Reserved
//
// Revision: 05 1.9.1.10 source/albert/graph2d/ixdc.cpp, 2d, ioc.v400, 001006 

#include <ixdc.hpp>
#include <iprimlck.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _IXDC_CPP_
#include <ipagetun.h>
#endif

IXDC::IXDC(Display* disp,  Drawable dra) : fRefCount(1)
{
    IFUNCTRACE_DEVELOP();
	ITRACE_DEVELOP(IString("IXDC new this=") +
                       IString( (unsigned long)(void*)this ).d2x() );	
	xDisplay = disp;
	xDrawable = dra; 

	xGC = NULL;
}

IXDC::IXDC() : fRefCount(1)
{
    IFUNCTRACE_DEVELOP();
	ITRACE_DEVELOP(IString("IXDC new this=") +
                       IString( (unsigned long)(void*)this ).d2x() );

	xDisplay = NULL;
	xDrawable = 0;
	xGC = NULL;
}


IXDC::IXDC(const IXDC& xDC) : fRefCount(((IXDC*)&xDC)->refCount())
{
	xDisplay = xDC.xDisplay;
    xDrawable = xDC.xDrawable;
    xGC = xDC.xGC;
}

IXDC& IXDC::operator =(const IXDC& xDC)
{
	fRefCount = xDC.fRefCount;
	xDisplay = xDC.xDisplay;
    xDrawable = xDC.xDrawable;
    xGC = xDC.xGC;
	return *this;
}

IXDC::~IXDC()
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP(IString("IXDC delete this=") +
                   IString( (unsigned long)(void*)this ).d2x() );
    
	if (xDisplay && xGC){
		XFreeGC(xDisplay, xGC);
		xGC = NULL;
	}
}

void IXDC::setGC(GC nGC)
{
    if(nGC){
        if(xGC){
            XFreeGC(xDisplay, xGC);
            xGC = NULL;
        }
        xGC = nGC;
    }
}

static XGCValues defaultXGCValues;
static bool defaultXGCValuesInit = false;
        
GC
IXDC::gc()
{
    IFUNCTRACE_DEVELOP();
	if (xGC == NULL){
        if (! defaultXGCValuesInit) {
            IPrimalLock lockinit;
            if (! defaultXGCValuesInit) {
                GC defaultGC = DefaultGC(xDisplay, DefaultScreen(xDisplay));
                XGetGCValues(xDisplay, defaultGC,
                             (GCForeground | GCBackground | GCFont),
                             &defaultXGCValues);
                defaultXGCValuesInit = true;
            }
        }
		xGC = XCreateGC(xDisplay, xDrawable, 
                        (GCForeground | GCBackground | GCFont),
                        &defaultXGCValues);
	}
    return(xGC);
}

// reference counting stuff

void IXDC::incrementCount()
{
    IFUNCTRACE_DEVELOP();
	++fRefCount;
	ITRACE_DEVELOP(IString("IXDC incrementCount new count=")+
                       IString(fRefCount) +
                       IString(" this=") +
                       IString( (unsigned long)(void*)this ).d2x() );

}

void IXDC::decrementCount()
{
    IFUNCTRACE_DEVELOP();
    ITRACE_DEVELOP(IString("IXDC decrementCount old count=")+
                   IString(fRefCount) +
                   IString(" this=") +
                   IString( (unsigned long)(void*)this ).d2x() );
    
	--fRefCount;
}

//-------------------------
//Class IXDisplay
//-------------------------

#define GetRed(_pixVal) (((_pixVal) & (redMask))>>(redShift))
#define GetGreen(_pixVal) (((_pixVal) & (greenMask))>>(greenShift))
#define GetBlue(_pixVal) (((_pixVal) & (blueMask))>>(blueShift))
#define PutRed(_clr) (((_clr) << (redShift)) & (redMask))
#define PutGreen(_clr) (((_clr) << (greenShift)) & (greenMask))
#define PutBlue(_clr) (((_clr) << (blueShift)) & (blueMask))

Display* IXDisplay::fDisplay = 0;
unsigned long IXDisplay::redMask = 0;
unsigned long IXDisplay::greenMask = 0;
unsigned long IXDisplay::blueMask = 0;
unsigned long IXDisplay::rgbDepth = 0;
unsigned long IXDisplay::redShift = 0;
unsigned long IXDisplay::greenShift = 0;
unsigned long IXDisplay::blueShift = 0;
bool IXDisplay::fDisplayOpened  = false;

IXDisplay::IXDisplay(Display* display)
{
  IFUNCTRACE_DEVELOP();
  if (display == NULL)
  {
    fDisplayOpened = true;
    fDisplay =  XOpenDisplay("");
	if(fDisplay == NULL){
		printf("Error: Can't open display:%s\n", getenv("DISPLAY"));
		exit(-1);
	}
  }
  else
  {
    fDisplayOpened = false;
    fDisplay = display;
  }
  if(DefaultDepth(fDisplay, DefaultScreen(fDisplay)) > 8){
      Visual* vis = DefaultVisual(fDisplay, DefaultScreen(fDisplay));
      redMask = vis->red_mask;
      greenMask = vis->green_mask;
      blueMask = vis->blue_mask;
      rgbDepth = vis->bits_per_rgb;
      //find the bit shifts
      for(int i=0; i<3; i++){
          unsigned long mask = (1<<rgbDepth) -1;
          mask <<= (i*rgbDepth);
          if(redMask & mask)
              redShift = i*rgbDepth;
          else if(greenMask & mask)
              greenShift = i*rgbDepth;
          else if(blueMask & mask)
              blueShift = i*rgbDepth;
      }
  } else {
      redMask = greenMask = blueMask = 0xff;
      rgbDepth = 8;
      redShift = greenShift = blueShift = 0;
  }
}

IXDisplay::~IXDisplay()
{
}

Display* IXDisplay::display()
{
  return fDisplay;
}

void IXDisplay::closeDisplay()
{
  IFUNCTRACE_DEVELOP();
  if (fDisplayOpened && fDisplay)
  {
    XCloseDisplay(fDisplay);
  }
  fDisplay = NULL;
  fDisplayOpened = false;
}

void IXDisplay::queryMasks(unsigned long* rM,
		unsigned long* gM,
		unsigned long* bM,
		unsigned long* cs,
		unsigned long* rS,
		unsigned long* gS,
		unsigned long* bS)
{
	if(!fDisplay) //if not inited 
		IXDisplay tmp(NULL);

	*rM = redMask;
	*gM = greenMask;
	*bM = blueMask;
	*cs = rgbDepth;
	*rS = redShift;
	*gS = greenShift;
	*bS = blueShift;
}

unsigned long IXDisplay::rgbToPixel(unsigned char r, 
		unsigned char g, 
		unsigned char b)
{
	if(!fDisplay) //if not inited 
		IXDisplay tmp(NULL);

	return (PutRed(r)|PutGreen(g)|PutBlue(b));
}

void IXDisplay::pixelToRGB(unsigned long pixel, 
		unsigned char* r, 
		unsigned char* g, 
		unsigned char* b)
{
	if(!fDisplay) //if not inited 
		IXDisplay tmp(NULL);

	*r = GetRed(pixel);
	*g = GetGreen(pixel);
	*b = GetBlue(pixel);
}

bool IXDisplay::isRGBFormat()
{
	if(!fDisplay) //if not inited 
		IXDisplay tmp(NULL);

	if((redShift==16) && (greenShift==8) && (blueShift==0))
		return true;
	return false;
}
