/*
******************************************************************************
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation,  1997           *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
********************************************************************************
||
||  File:       Imgconv.cpp
||
||  What:       Routines which support loading of Different Image formats
||
*/

// Revision: 19 1.16.3.3 source/albert/graph2d/imgconv.cpp, 2d, ioc.v400, 001006 

#define BLOCKREADSIZE 1024

#ifdef IC_PM
#define INCL_WINSYS
#define INCL_WINERRORS
#define INCL_GPIBITMAPS
#define INCL_DOSPROCESS
#define INCL_GPI
#include <winemul.h>
#endif //IC_PM

#include <ixdc.hpp>
#include <ibmpdata.hpp>
#include <imgconv.hpp>
#include <igifconv.hpp>
#include <ibmpconv.hpp>
#include <ixpmconv.hpp>
#include <pixbuf.hpp>
#include <iexcept.hpp>
#include <grassert.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#include <ijpgconv.hpp>  // Include this header file last to avoid
                         // conflicting definitions of INT32.

#ifdef IC_PAGETUNE
#define _IMGCONV_CPP_
#include <ipagetun.h>
#endif

#pragma pack(push,4)


//***********************************************************************
//IImageConverter

// ----- Constructors and Destructors -----

IImageConverter::~IImageConverter()
{
        if(fHasColormap)
                delete  fPal;
}

IImageConverter::IImageConverter()
        :fBitsPerPixel(0),
        fWidth(0),
        fHeight(0),
        xHot(0),
        yHot(0),
        fImageFileSize(0),
        fHasColormap(False),
        fColormapSize(0),
        fInterlace(False),
        fIsTransparencyEnabled(false),
        fTransparency(0, 0, 0),
        fPal(0x0),
        bFlipImage(false)
{
}

IGImage* IImageConverter::readImage(const char *inFile)
{
        char *fromFile, msg[128];

        IFUNCTRACE_DEVELOP();
        if(inFile == NULL){
                sprintf(msg,"readImage::NULL filename Specified");
                GrafDeviceException(msg);
        }

        IGImagePixelAccessor::EImageFormat imgtype = imageType(inFile);
        if(imgtype == -1){
                sprintf(msg,"readImage::Could not open image file \"%s\"",inFile);
                GrafDeviceException(msg);
        }
        if(imgtype == IGImagePixelAccessor::kUNKNOWN){
                sprintf(msg,"readImage::Unknown image format file \"%s\"",inFile);
                GrafDeviceException(msg);
        }


        if(imgtype == IGImagePixelAccessor::kCOMPRESS){
                fromFile = unCompressFile(inFile);
                if(fromFile == NULL)
                        return NULL;

                imgtype = imageType(fromFile);
        } else
                fromFile = (char *)inFile;


        IImageConverter* fImageConv;

        switch(imgtype)
        {
        case IGImagePixelAccessor::kBMP:
                fImageConv = new IBMPConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Bmp Image\n");
#endif
                break;
        case IGImagePixelAccessor::kGIF:
                fImageConv = new IGIFConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Gif Image\n");
#endif
                break;
#ifdef IC_MOTIF
        case IGImagePixelAccessor::kJPG:
                fImageConv = new IJPGConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Jpeg Image\n");
#endif
                break;
#endif
        case IGImagePixelAccessor::kXPM:
                fImageConv = new IXPMConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Xpm Image\n");
#endif
                break;
        default:
                sprintf(msg,"Un-supported file format of image file %s", inFile);
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"%s\n", msg);
#endif
                GrafDeviceException(msg);
                return NULL;
        }
        fImageConv->fImgFormat = imgtype;

        IGImage *igimage = fImageConv->createFromFile(fromFile);

        delete fImageConv;

        return igimage;
}

int IImageConverter::writeImage(char *toFile, IGImage* igimage, IGImagePixelAccessor::EImageFormat fmt)
{
        char msg[128];

        IFUNCTRACE_DEVELOP();
        if(toFile == NULL){
                sprintf(msg,"writeImage::Image File name NULL specified");
                GrafDeviceException(msg);
        }
        if(igimage == NULL){
                sprintf(msg,"writeImage::NULL Ptr to IGImage specified");
                GrafDeviceException(msg);
        }

        IImageConverter* fImageConv;
        IPixelBuffer* pixbuf = igimage->fBmpData->pixelBuffer();

        switch(fmt)
        {
        case IGImagePixelAccessor::kBMP:
                fImageConv = new IBMPConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Bmp Image\n");
#endif
                break;
        case IGImagePixelAccessor::kGIF:
                fImageConv = new IGIFConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Gif Image\n");
#endif
                break;
#ifdef IC_MOTIF
   case IGImagePixelAccessor::kJPG:
                fImageConv = new IJPGConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Jpeg Image\n");
#endif
                break;
#endif
        case IGImagePixelAccessor::kXPM:
                fImageConv = new IXPMConverter();
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"Xpm Image\n");
#endif
                break;
        default:
                sprintf(msg,"Un-supported file format of image file %s", toFile);
#ifdef GRAPH2D_DEBUG
                fprintf(stderr,"%s\n", msg);
#endif
                GrafDeviceException(msg);
                return -1;

        }


        fImageConv->fWidth = pixbuf->fWidth;
        fImageConv->fHeight = pixbuf->fHeight;
        fImageConv->xHot = pixbuf->xHot;
        fImageConv->yHot = pixbuf->yHot;

        iBYTE *data=NULL;
#ifdef IC_MOTIF
        if(fmt == IGImagePixelAccessor::kBMP){
#else
        if(fmt != IGImagePixelAccessor::kBMP){
#endif
                fImageConv->bFlipImage = true;
                data = new iBYTE[pixbuf->fHeight * pixbuf->fPixelStride];
                for(int y=0; y<pixbuf->fHeight; y++)
                {
                        memcpy(&(data[y * pixbuf->fPixelStride]),
                                &(pixbuf->fImage[(pixbuf->fHeight - 1 - y) * pixbuf->fPixelStride]),
                                (size_t)pixbuf->fPixelStride);
                }
                swapImageRBData(data, pixbuf->fWidth, pixbuf->fHeight, pixbuf->fBitsPerPixel);
        } else {
                data = pixbuf->fImage;
        }


        fImageConv->fBitsPerPixel = pixbuf->fBitsPerPixel;
        if(pixbuf->fBitsPerPixel <= 8)
                fImageConv->fHasColormap = true;
        else
                fImageConv->fHasColormap = false;
        fImageConv->fInterlace = false;

        if(fImageConv->fHasColormap)
        {
                fImageConv->fPal = new IRGBAColorArray(1<<fImageConv->fBitsPerPixel);
                *(fImageConv->fPal) = *(pixbuf->colorTable());
        }

        //get the transparency details
        fImageConv->fIsTransparencyEnabled = pixbuf->transparencyFlag();
        fImageConv->fTransparency = pixbuf->transparencyColor();

        int ret = fImageConv->writeToFile(toFile, data);

        delete fImageConv;
#ifdef IC_MOTIF
        if(fmt == IGImagePixelAccessor::kBMP)
#else
        if(fmt != IGImagePixelAccessor::kBMP)
#endif
                delete [] data;

        return ret;
}

IGImagePixelAccessor::EImageFormat IImageConverter::imageType( const char *fromFile)
{
  /* opens fname (which *better* be an actual file by this point!) and
     reads the first couple o' bytes.  Figures out what the file's likely
     to be, and returns the appropriate *** code */


  int  fd;
  iBYTE  magicno[30];    /* first 30 bytes of file */
  long   n;
  IGImagePixelAccessor::EImageFormat rv;

  if (!fromFile) return IGImagePixelAccessor::kUNKNOWN; //ERROR;   /* shouldn't happen */

  fd = open(fromFile, O_RDONLY|BINARYOPEN);
  if (fd < 0) return IGImagePixelAccessor::kUNKNOWN; //ERROR;

  n = read(fd, magicno, (size_t) 30);
  close(fd);

  if (n<30) return IGImagePixelAccessor::kUNKNOWN;    /* files less than 30 bytes long... */

  rv = IGImagePixelAccessor::kUNKNOWN;

  if (strncmp((char *) magicno,"GIF87a", (size_t) 6)==0 ||
      strncmp((char *) magicno,"GIF89a", (size_t) 6)==0)        rv = IGImagePixelAccessor::kGIF;

  else if (strncmp((char *) magicno,"VIEW", (size_t) 4)==0 ||
           strncmp((char *) magicno,"WEIV", (size_t) 4)==0)     rv = IGImagePixelAccessor::kUNKNOWN;//kPM;

  else if (magicno[0] == 'P' && magicno[1]>='1' &&
           magicno[1]<='6')                             rv = IGImagePixelAccessor::kUNKNOWN;//kPBM;

  /* note: have to check XPM before XBM, as first 2 chars are the same */
  else if ((strncmp((char *) magicno, "/* XPM */", (size_t) 9)==0) ||
                (strncmp((char *) magicno, "/* XPM2 C */", (size_t) 12)==0)) rv = IGImagePixelAccessor::kXPM;

  else if (strncmp((char *) magicno,"#define", (size_t) 7)==0 ||
           (magicno[0] == '/' && magicno[1] == '*'))    rv = IGImagePixelAccessor::kUNKNOWN;//kXBM;

  else if (magicno[0]==0x59 && (magicno[1]&0x7f)==0x26 &&
           magicno[2]==0x6a && (magicno[3]&0x7f)==0x15) rv = IGImagePixelAccessor::kUNKNOWN;//kSUNRAS;

  else if (magicno[0] == 'B' && magicno[1] == 'M')      rv = IGImagePixelAccessor::kBMP;

  else if (magicno[0]==0x52 && magicno[1]==0xcc)                rv = IGImagePixelAccessor::kUNKNOWN;//kUTAHRLE;

  else if ((magicno[0]==0x01 && magicno[1]==0xda) ||
                        (magicno[0]==0xda && magicno[1]==0x01))         rv = IGImagePixelAccessor::kUNKNOWN;//kIRIS;

        else if ((magicno[0]==0x1f && magicno[1]==0x9d) ||
                        (magicno[0]==0x1f && magicno[1]==0x8b))         rv = IGImagePixelAccessor::kCOMPRESS;

  else if (magicno[0]==0x0a && magicno[1] <= 5)         rv = IGImagePixelAccessor::kUNKNOWN;//kPCX;

  else if (magicno[0]==0 && magicno[1]==0 &&
           magicno[2]==2 && magicno[3]==0 &&
           magicno[4]==0 && magicno[5]==0 &&
           magicno[6]==0 && magicno[7]==0)              rv = IGImagePixelAccessor::kUNKNOWN;//kTARGA;

  else if (magicno[4]==0x00 && magicno[5]==0x00 &&
           magicno[6]==0x00 && magicno[7]==0x07)        rv = IGImagePixelAccessor::kUNKNOWN;//kXWD;

  else if (strncmp((char *) magicno,"SIMPLE  ", (size_t) 8)==0 &&
           magicno[29] == 'T')                          rv = IGImagePixelAccessor::kUNKNOWN;//kFITS;


  else if (magicno[0]==0xff && magicno[1]==0xd8 &&
           magicno[2]==0xff)
#ifdef IC_MOTIF
                                                   rv = IGImagePixelAccessor::kJPG;
#else
                                                   rv = IGImagePixelAccessor::kUNKNOWN;
#endif

  else if ((magicno[0]=='M' && magicno[1]=='M') ||
           (magicno[0]=='I' && magicno[1]=='I'))        rv = IGImagePixelAccessor::kUNKNOWN;//kTIFF;

  return rv;
}

#define GUNZIP "/usr/local/bin/gunzip -q"

/* returns filename on success, with name of uncompressed file
   returns 'NULL' on failure
*/
char *IImageConverter::unCompressFile(const char* compname)
{
#ifdef IC_MOTIF
        char namez[128], *fname, buf[512], *uncompname;

        fname = (char *)compname;
        namez[0] = '\0';
#if 0
        /* see if compressed file name ends with '.Z'.  If it *doesn't* we need
                temporarily rename it so it *does*, uncompress it, and rename *back*
                to what it was.  necessary because uncompress doesn't handle files
                that don't end with '.Z'
        */

        if (strlen(compname) >= (size_t) 2            &&
        strcmp(compname + strlen(compname)-2,".Z")!=0 &&
        strcmp(compname + strlen(compname)-2,".z")!=0) {
                strcpy(namez, compname);
                strcat(namez,".Z");

                if (rename(compname, namez) < 0) {
#ifdef GRAPH2D_DEBUG
                        fprintf(stderr, "Error renaming '%s' to '%s'\n", compname, namez);
#endif
                        GrafDeviceException("Error renaming '%s' to '%s'\n", compname, namez);
                        return NULL;
                }

                fname = namez;
        }
#endif

        uncompname = tempnam("/tmp","VA");
        sprintf(buf,"%s -c %s >%s", GUNZIP, fname, uncompname);

#ifdef GRAPH2D_DEBUG
        fprintf(stderr, "Uncompressing '%s'...\n", fname);
#endif
        if (system(buf)) {
                sprintf(buf, "Unable to uncompress '%s'.\n", fname);
#ifdef GRAPH2D_DEBUG
                fprintf(stderr, "%s", buf);
#endif
                GrafDeviceException(buf);
                return NULL;
        }

        /* if we renamed the file to end with a .Z for the sake of 'uncompress',
        rename it back to what it once was... */

        if (strlen(namez)) {
                if (rename(namez, compname) < 0) {
                        sprintf(buf, "Error renaming '%s' to '%s'\n", namez, compname);
#ifdef GRAPH2D_DEBUG
                        fprintf(stderr, "%s", buf);
#endif
                        GrafDeviceException(buf);
                        return NULL;
                }
        }

        return uncompname;
#else
        return NULL;
#endif
}

void IImageConverter::swapImageRBData( unsigned char* fImage,
                                const unsigned long width,
                                const unsigned long height,
                                const unsigned long bitsPerPixel)
{
        iBYTE cTmp;
        long x, y;
        R5G6B5 *clr16 = (R5G6B5 *)(fImage);
        RGBClr *clr24 = (RGBClr *)(fImage);
        RGBAClr *clr32 = (RGBAClr *)(fImage);
        unsigned long pixelStride = PIXELSTRIDE(width, bitsPerPixel);

        if(bitsPerPixel < 24)
                return;

        for(y=0; y<height; y++){
                clr16 = (R5G6B5 *)&((fImage)[y * pixelStride]);
                clr24 = (RGBClr *)&((fImage)[y * pixelStride]);
                clr32 = (RGBAClr *)&((fImage)[y * pixelStride]);
                for(x=0; x<width; x++){
                        if(bitsPerPixel == 16){ //RGB
                                cTmp = clr16[x].R;
                                clr16[x].R = clr16[x].B;
                                clr16[x].B = cTmp;
                        } else if(bitsPerPixel == 24){ //RGB
                                cTmp = clr24[x].R;
                                clr24[x].R = clr24[x].B;
                                clr24[x].B = cTmp;
                        } else { //32 bit RGBA
                                cTmp = clr32[x].R;
                                clr32[x].R = clr32[x].B;
                                clr32[x].B = cTmp;
                        }
                }
        }
}

IPixelBuffer* IImageConverter::constructPixelBuffer(iBYTE *fImgData)
{
        int i;

        IFUNCTRACE_DEVELOP();
#ifdef GRAPH2D_DEBUG
        fprintf(stderr,"BitsPerPixel = %d\n",fBitsPerPixel);
#endif
        //Flip the image data
        if(bFlipImage) {
                unsigned long y, stride=PIXELSTRIDE(fWidth,fBitsPerPixel);
                iBYTE *tmpData = new iBYTE[stride];
                for(y=0; y<fHeight/2; y++)
                {
                        memcpy(tmpData, &fImgData[y * stride], (size_t)stride);
                        memcpy(&fImgData[y * stride], &fImgData[(fHeight -y -1) * stride], (size_t)stride);
                        memcpy(&fImgData[(fHeight - y -1) * stride], tmpData, (size_t)stride);
                }
                delete [] tmpData;
        }

        switch(fBitsPerPixel)
        {
        case 1: //monochrome
        {
                IMonochromePixelBuffer *fPixBuf = new IMonochromePixelBuffer (fWidth, fHeight, fImgData, fPal);

                //set transparency
                fPixBuf->setTransparencyFlag(fIsTransparencyEnabled);
                fPixBuf->setTransparencyColor(fTransparency);
                fPixBuf->xHot = xHot;
                fPixBuf->yHot = yHot;
                fPixBuf->srcImageFormat = fImgFormat;
                return fPixBuf;
        }
        case 4: //4 bit image
        {
                ICLUT4PixelBuffer *fPixBuf = new ICLUT4PixelBuffer (fWidth, fHeight, fImgData, fPal);

                //set transparency
                fPixBuf->setTransparencyFlag(fIsTransparencyEnabled);
                fPixBuf->setTransparencyColor(fTransparency);
                fPixBuf->xHot = xHot;
                fPixBuf->yHot = yHot;
                fPixBuf->srcImageFormat = fImgFormat;
                return fPixBuf;

        }
        case 8: //8 bit image
        {
                ICLUT8PixelBuffer *fPixBuf = new ICLUT8PixelBuffer (fWidth, fHeight, fImgData, fPal);

                //set transparency
                fPixBuf->setTransparencyFlag(fIsTransparencyEnabled);
                fPixBuf->setTransparencyColor(fTransparency);
                fPixBuf->xHot = xHot;
                fPixBuf->yHot = yHot;
                fPixBuf->srcImageFormat = fImgFormat;
                return fPixBuf;

        }
        case 16: //16 bit image R5G6B5
        {
                IR5G6B5PixelBuffer *fPixBuf = new IR5G6B5PixelBuffer (fWidth, fHeight, fImgData);
                //set transparency
                fPixBuf->setTransparencyFlag(fIsTransparencyEnabled);
                fPixBuf->setTransparencyColor(fTransparency);
                fPixBuf->xHot = xHot;
                fPixBuf->yHot = yHot;
                fPixBuf->srcImageFormat = fImgFormat;
                return fPixBuf;
        }
        case 24: //24 bit image R8G8B8
        {
#ifdef IC_MOTIF
                if(fImgFormat == IGImagePixelAccessor::kBMP)
                        swapImageRBData(fImgData, fWidth, fHeight, fBitsPerPixel);
#else
                if(fImgFormat != IGImagePixelAccessor::kBMP)
                        swapImageRBData(fImgData, fWidth, fHeight, fBitsPerPixel);
#endif
                IR8G8B8PixelBuffer *fPixBuf = new IR8G8B8PixelBuffer (fWidth, fHeight, fImgData);

                //set transparency
                fPixBuf->setTransparencyFlag(fIsTransparencyEnabled);
                fPixBuf->setTransparencyColor(fTransparency);
                fPixBuf->xHot = xHot;
                fPixBuf->yHot = yHot;
                fPixBuf->srcImageFormat = fImgFormat;
                return fPixBuf;
        }
        case 32: //32 bit image R8G8B8A8
        {
#ifdef IC_MOTIF
                if(fImgFormat == IGImagePixelAccessor::kBMP)
                        swapImageRBData(fImgData, fWidth, fHeight, fBitsPerPixel);
#else
                if(fImgFormat != IGImagePixelAccessor::kBMP)
                        swapImageRBData(fImgData, fWidth, fHeight, fBitsPerPixel);
#endif
                IR8G8B8A8PixelBuffer *fPixBuf = new IR8G8B8A8PixelBuffer (fWidth, fHeight, fImgData);

                //set transparency
                fPixBuf->setTransparencyFlag(fIsTransparencyEnabled);
                fPixBuf->setTransparencyColor(fTransparency);
                fPixBuf->xHot = xHot;
                fPixBuf->yHot = yHot;
                fPixBuf->srcImageFormat = fImgFormat;
                return fPixBuf;
        }
        }
        return NULL; // should not have reached here
}

void IImageConverter::attachPixelBuffer(IGImage *image, IPixelBuffer *pixbuf)
{
        image->adoptBitmapData(image->createBmpDataFromPixBuf(pixbuf));
        // defect: 29186
        // both createBmpDataFromPixBuf and adoptBitmapData addRef()
        // for the image.  Thus in this case we need to removeRef one
        // of them.
        image->removeRef();
}

//***********************************************************************
//General Helpers
//***********************************************************************

inline void imageSkip(IDataStream &fromWhere, long count)
{
        for(long i=0; i<count; i++)
                fromWhere.readByte();
}

inline void read_usword(IDataStream &fromWhere, iWORD *w)
{
        iBYTE low = 0, high = 0;
        low = fromWhere.readByte();
        high = fromWhere.readByte();
        *w = (iWORD)make_usword(low, high);
}
inline void read_sword(IDataStream &fromWhere, short *w)
{
        iBYTE low = 0, high = 0;
        low = fromWhere.readByte();
        high = fromWhere.readByte();
        *w = (short)make_usword(low, high);
}
inline void read_usdword(IDataStream &fromWhere, iDWORD *d)
{
        iWORD low, high;
        read_usword(fromWhere, &low);
        read_usword(fromWhere, &high);
        *d = (iDWORD)make_usdword(low, high);
}
inline void read_sdword(IDataStream &fromWhere, long *d)
{
        short low, high;
        read_sword(fromWhere, &low);
        read_sword(fromWhere, &high);
        *d = (long)make_sdword(low, high);
}
static bool write_word(int fd, iWORD w)
{
        iBYTE    low  = (iBYTE) w;
        iBYTE    high = (iBYTE) (w >> 8);

        write(fd, &low, 1);
        write(fd, &high, 1);
        return true;
}
static bool write_dword(int fd, iDWORD d)
{
        write_word(fd, (iWORD) d);
        write_word(fd, (iWORD) (d >> 16));
        return true;
}

//***********************************************************************
//IGIFConverter
//***********************************************************************

IGIFConverter::IGIFConverter()
{
	
}
IGIFConverter::~IGIFConverter()
{
}

void IGIFConverter::writeTo( const IGImage& fromImage,
				IDataStream& toWhere) const
{
}


IGImage* IGIFConverter::createFromStream( IDataStream& fromWhere )
{
	long err;
	iBYTE *fImgData;

	IFUNCTRACE_DEVELOP();
	err = readGif(fromWhere, &fImgData); //read gif header

	if(err != GBM_ERR_OK){ //exeception
		GrafDeviceException("Error in reading GIF Image header");
	}

#ifndef IC_MOTIF
		bFlipImage = true;
#endif

	IPixelBuffer *tmpPix = constructPixelBuffer(fImgData);

	IGImage *igimage = new IGImage();

	attachPixelBuffer(igimage, tmpPix);

	delete [] fImgData;

	return(igimage);
}

IGImage* IGIFConverter::createFromFile( const char *fromFile)
{
	IFUNCTRACE_DEVELOP();
	int fd = open(fromFile, O_RDONLY|BINARYOPEN);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file read error", fromFile);
#endif
		GrafDeviceException("Error in opening GIF Image file");
	}

	IDataStream *theStream = IDataStream::createMemoryStream ();
	theStream->reset();
	char tdata[BLOCKREADSIZE];
	long i,length, cnt;

	lseek(fd, 0L, SEEK_END);
#ifdef IC_MOTIF
	length=lseek(fd, 0L, SEEK_CUR);
#else
	length = tell( fd );
#endif //IC_MOTIF
	fImageFileSize = length;
	lseek(fd, 0L, SEEK_SET);

	//dump the file data into stream
	for(cnt=0, i=0; cnt<length; cnt+= i){
		i=read(fd, tdata, BLOCKREADSIZE);
		theStream->writeBytes(tdata, i);
		if(i<BLOCKREADSIZE)
			break;
	}
	close(fd);
	theStream->reset ();
	IGImage *igimage = createFromStream(*theStream);
	delete theStream;


	return( igimage);
}

int IGIFConverter::writeToFile( char *toFile, iBYTE* data)
{
	long err;

	IFUNCTRACE_DEVELOP();
	int fd = open(toFile, O_CREAT|O_WRONLY|BINARYOPEN|O_TRUNC, S_IREAD|S_IWRITE);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file create error", toFile);
#endif
		GrafDeviceException("Error in creating GIF Image file");
	}

	err = writeGif(fd, data);
	close(fd);

	return err;

}

//GIF Reader helpers

int IGIFConverter:: readGif(IDataStream &fromWhere, iBYTE **imgData)
{

	iBYTE signiture[6], scn_desc[7], image_tag;
	int img = 0, img_want = 1;
	int i;
	iBYTE cTmp[3];

	/* Read and validate signiture block */

	IFUNCTRACE_DEVELOP();
	fromWhere.readBytes((char*)&signiture[0], 6);
	if ( memcmp(signiture, "GIF87a", 6) &&
		memcmp(signiture, "GIF89a", 6) )
		return GBM_ERR_BAD_MAGIC;

	/* Read screen descriptor */

	fromWhere.readBytes((char*)&scn_desc[0], 7);

	fBitsPerPixel = (scn_desc[4] & 7) + 1;
	fHasColormap = ( scn_desc[4] & GIF_HASCOLORMAPMASK ) ? True : False;
	fColormapSize = 1 << fBitsPerPixel;


	/* Global colour table follows screen descriptor */
	if ( fHasColormap )
	{
		fPal = new IRGBAColorArray(fColormapSize);
		IR8G8B8A8Color *clrData = fPal->fColors;

		for(i=0; i<fColormapSize; i++)
		{
			fromWhere.readBytes((char*)&cTmp[0], 3);

                        clrData[i].fRed = cTmp[0];
                        clrData[i].fGreen = cTmp[1];
                        clrData[i].fBlue = cTmp[2];
                        clrData[i].fOpacity = 0xFF;
		}
	}

	/* Expected image descriptors / extension blocks / terminator */

	while ( img < img_want )
	{
		image_tag = fromWhere.readByte();
		switch ( image_tag )
		{
		case 0x2c:
			if ( ++img != img_want ) /* Skip the image data */
			{
				iBYTE image_desc[9];

				fromWhere.readBytes((char*)&image_desc[0], 9);

				//skip colormap if one
				if ( image_desc[8] & GIF_HASCOLORMAPMASK ) /* Local colour table follows */
				{
					for(i=0; i<(image_desc[8] & 7)+1; i++)
					{
						fromWhere.readBytes((char*)&cTmp[0], 3);
					}
				}

				iBYTE code_size, block_size;

				code_size=fromWhere.readByte();
				do
				{
					block_size=fromWhere.readByte();
					imageSkip(fromWhere, block_size);
				} while ( block_size );
			}
			else //get the desired image
			{	
				gif_rdata(fromWhere, imgData);
			}

			break;
		case 0x21: /*0x21 extension block*/
		{
			iBYTE func_code, byte_count;

			func_code = fromWhere.readByte();
			if(func_code == 'R'){ //aspect extension
				byte_count = fromWhere.readByte();
				if(byte_count == 2){
					fromWhere.readByte(); //aspect extension
					fromWhere.readByte(); //aspect extension
				} else {
					imageSkip(fromWhere, byte_count);
				}
				do {
					byte_count = fromWhere.readByte();
					imageSkip(fromWhere, byte_count);
				} while ( byte_count );
			} else if (func_code == 0xFE) { //comment extension
				do { //skip comment
					byte_count = fromWhere.readByte();
					imageSkip(fromWhere, byte_count);
				} while ( byte_count );
			} else if (func_code == 0x01) { //plaintext extension
				//first subblock
				byte_count = fromWhere.readByte();
				imageSkip(fromWhere, byte_count);
				do { //ignore sub-block
					byte_count = fromWhere.readByte();
					imageSkip(fromWhere, byte_count);
				} while ( byte_count );
			} else if (func_code == 0xF9) { //graphic control extension
				iBYTE gce[8];
				fromWhere.readBytes((char*)&gce[0], 6);

				//both share the same bit for control
				fIsTransparencyEnabled = gce[1] & 0x01;

				//Set the Transparency color
				fTransparency = fPal->color(gce[4]);

			} else if (func_code == 0xFF) { // Application Extension
				do { //ignore sub-block
					byte_count = fromWhere.readByte();
					imageSkip(fromWhere, byte_count);
				} while ( byte_count );
			} else { //unknown extention
				do { //ignore sub-block
					byte_count = fromWhere.readByte();
					imageSkip(fromWhere, byte_count);
				} while ( byte_count );
			}
		}
			break;
		case 0x3b:
			/*0x3b terminator*/
			/* Oi, we were hoping to get an image descriptor! */
			return GBM_ERR_GIF_TERM;
		}
	}


	return GBM_ERR_OK;

}


iCWORD IGIFConverter::read_code(READ_CONTEXT *c)
{
        int raw_code, byte_inx;

	IFUNCTRACE_DEVELOP();
        if ( c->inx + c->code_size > c->size )
	/*not enough bits in buffer refill it*/
	/* Not very efficient, but infrequently called */

	{
		int bytes_to_lose = (int)(c->inx >> 3);
		iBYTE bytes;

		/* Note biggest code size is 12 bits */
		/* And this can at worst span 3 bytes */
		memcpy(c->buf, c->buf + bytes_to_lose, 3);
		(c->inx) &= 7;
		(c->size) -= (bytes_to_lose << 3);
		bytes=c->fromWhere->readByte();
		c->fromWhere->readBytes((char *)(c->buf + (c->size >> 3)), bytes);
		(c->size) += (bytes << 3);
	}

	byte_inx = (int)(c->inx >> 3);
	raw_code = c->buf[byte_inx] + ((c->buf[byte_inx + 1]) << 8);
	if ( c->code_size > 8 )
		raw_code += ((c->buf[byte_inx + 2]) << 16);
        raw_code >>= ((c->inx) & 7);
        (c->inx) += (iBYTE) (c->code_size);

        return (iCWORD) raw_code & c->read_mask;
}

void IGIFConverter::output(iBYTE value, OUTPUT_CONTEXT *o)
{
        if ( o->y >= o->h )
                return;

        switch ( o->bpp )
	{
		case 1:
			o->data_this_line[(o->x) >> 3] |= (value << (7 - ((o->x) & 7)));
                        break;
                case 2:
                case 3:
                case 4:
                        if ( (o->x) & 1 )
                                o->data_this_line[(o->x) >> 1] |= value & 0x0F;
                        else
                                o->data_this_line[(o->x) >> 1] = ((value & 0x0F) << 4);
                        break;
                case 5:
                case 6:
                case 7:
                case 8:
                        o->data_this_line[o->x] = value;
                        break;
	}

        if ( ++(o->x) < o->w )
                return;

        o->x = 0;
        if ( o->ilace )
	{
                switch ( o->pass )
		{
                        case 0:
                                (o->y) += 8;
                                if ( o->y >= o->h )
				{
                                        (o->pass)++;
                                        o->y = 4;
				}
                                break;
                        case 1:
                                (o->y) += 8;
                                if ( o->y >= o->h )
				{
                                        (o->pass)++;
                                        o->y = 2;
				}
                                break;
                        case 2:
                                (o->y) += 4;
                                if ( o->y >= o->h )
				{
                                        (o->pass)++;
                                        o->y = 1;
				}
                                break;
                        case 3:
                                (o->y) += 2;
                                break;
		}
                o->data_this_line = o->data + ( o->y * o->stride);
	}
        else
	{
                (o->y)++;
                o->data_this_line += (o->stride);
	}
}

int IGIFConverter:: gif_rdata(IDataStream &fromWhere, iBYTE **data)
{
        iBYTE  fin_char,min_code_size;   /* As read from the file */
        int init_code_size;              /* Initial code size */
        iCWORD max_code;                 /* 1 << code_size */
        iCWORD clear_code;               /* Code to clear table */
        iCWORD eoi_code;                 /* End of information code */
        iCWORD first_free_code;          /* First free code */
        iCWORD free_code;                /* Next available free code slot */
        int bit_mask;                    /* Output pixel mask */
        int i, out_count = 0;
        iCWORD code, cur_code, old_code, in_code;
        iCWORD prefix[4096], suffix[4096], outcode[4096];
        READ_CONTEXT c;
        OUTPUT_CONTEXT o;
        bool table_full = false;     /* To help implement deferred clear */
	iBYTE	image_desc[9], cTmp[3];

	IFUNCTRACE_DEVELOP();
	fromWhere.readBytes((char*)&image_desc[0], 9);
	fWidth = make_usword(image_desc[4], image_desc[5]);
	fHeight = make_usword(image_desc[6], image_desc[7]);
	fInterlace = ( (image_desc[8] & GIF_INTERLACEMASK) != 0 );

	if ( image_desc[8] & GIF_HASCOLORMAPMASK ) /* Local colour table follows */
	{
		IR8G8B8A8Color *clrData = fPal->fColors;
		for(i=0; i<(image_desc[8] & 0x07) + 1; i++)
		{
			fromWhere.readBytes((char*)&cTmp[0], 3);
                        clrData[i].fRed = cTmp[0];
                        clrData[i].fGreen = cTmp[1];
                        clrData[i].fBlue = cTmp[2];
                        clrData[i].fOpacity = 0xFF;
		}
	}

	min_code_size = fromWhere.readByte();

        if ( min_code_size < 2 || min_code_size > 9 )
	{
                return GBM_ERR_GIF_CODE_SIZE;
	}
	//Construct the bitmask before modulating the fBitsPerPixel
        bit_mask = (int) ((1 << fBitsPerPixel) - 1);
	
#ifdef GRAPH2D_DEBUG
	fprintf(stderr,"Original BitsPerPixel = %d\n",fBitsPerPixel);
#endif
	switch(fBitsPerPixel){
		case 1:	
			fBitsPerPixel=1;
			break;
		case 2:
		case 3:
		case 4:
			fBitsPerPixel=4;
			break;
		case 5:
		case 6:
		case 7:
		case 8:
			fBitsPerPixel=8;
			break;
	}

	*data = new iBYTE[PIXELSTRIDE(fWidth, fBitsPerPixel) * fHeight];



        /* Initial read context */

        c.inx            = 0;
        c.size           = 0;
        c.fromWhere      = &fromWhere;
        c.code_size      = min_code_size + 1;
        c.read_mask      = (iCWORD) (( 1 << c.code_size ) - 1);

        /* Initialise pixel-output context */

        o.x              = 0;
        o.y              = 0;
        o.pass           = 0;
        o.w              = fWidth;
        o.h              = fHeight;
        o.bpp            = fBitsPerPixel;
        o.ilace          = fInterlace;
        o.stride         = PIXELSTRIDE(fWidth, fBitsPerPixel);
        o.data           = *data;
        o.data_this_line = *data;


        /* 2^min_code size accounts for all colours in file */

        clear_code = (iCWORD) ( 1 << min_code_size );
        eoi_code = (iCWORD) (clear_code + 1);
        free_code = first_free_code = (iCWORD) (clear_code + 2);

        /* 2^(min_code_size+1) includes clear and eoi code and space too */

        init_code_size = c.code_size;
        max_code = (iCWORD) ( 1 << c.code_size );

        while ( (code = read_code(&c)) != eoi_code && code != 0xffff && o.y < o.h )
	{
                if ( code == clear_code )
		{
                        c.code_size = init_code_size;
                        max_code = (iCWORD) ( 1 << c.code_size );
                        c.read_mask = (iCWORD) (max_code - 1);
                        free_code = first_free_code;
                        cur_code = old_code = code = read_code(&c);
                        if ( code == 0xffff )
                                break;
                        fin_char = (cur_code & bit_mask);
                        output(fin_char, &o);
                        table_full = false;
		}
                else
		{
                        cur_code = in_code = code;
                        if ( cur_code >= free_code )
			{
                                cur_code = old_code;
                                outcode[out_count++] = fin_char;
			}
                        while ( cur_code > bit_mask )
			{
                                if ( out_count > 4096 )
				{
                                        return GBM_ERR_GIF_CORRUPT;
				}
                                outcode[out_count++] = suffix[cur_code];
                                cur_code = prefix[cur_code];
			}
                        fin_char = (cur_code & bit_mask);
                        outcode[out_count++] = fin_char;
                        for ( i = out_count - 1; i >= 0; i-- )
                                output(outcode[i], &o);
                        out_count = 0;

                        /* Update dictionary */

                        if ( !table_full )
			{
                                prefix[free_code] = old_code;
                                suffix[free_code] = fin_char;

                                /* Advance to next free slot */

                                if ( ++free_code >= max_code )
				{
                                        if ( c.code_size < 12 )
					{
                                                c.code_size++;
                                                max_code <<= 1;
                                                c.read_mask = (iCWORD) (( 1 << c.code_size ) - 1);
					}
                                        else
                                                table_full = true;
				}
			}
                        old_code = in_code;
		}
	}


        if ( code == 0xffff )
                return GBM_ERR_READ;
        return GBM_ERR_OK;
}


/*
We won't write any GIF89a or higher extensions into file.
Write palette as global colour table, not local.
*/

bool IGIFConverter::write_code(int code, WRITE_CONTEXT *w)
{
	iBYTE *buf = w->buf + (w->inx >> 3);

	code <<= ((w->inx) & 7);
	*buf++ |= (iBYTE)  code       ;
	*buf++  = (iBYTE) (code >>  8);
	*buf    = (iBYTE) (code >> 16);

	(w->inx) += (w->code_size);
	if ( w->inx >= (255 << 3) ) /* Flush out full buffer */
	{
		iBYTE bytes = 255;

		if ( write(w->fd, &bytes, 1) != 1 )
			return false;
		if ( write(w->fd, w->buf, 255) != 255 )
			return false;

		memcpy(w->buf, w->buf + 255, 2);
		memset(w->buf + 2, 0, 255);
		(w->inx) -= (255 * 8);
	}

	return true;
}

bool IGIFConverter::flush_code(WRITE_CONTEXT *w)
{
	iBYTE bytes = ((w->inx + 7) >> 3);

	if ( bytes )
	{
		if ( write(w->fd, &bytes, 1) != 1 )
			return false;
		if ( write(w->fd, w->buf, bytes) != bytes )
			return false;
	}

	/* Data block terminator - a block of zero size */

	bytes = 0;
	return write(w->fd, &bytes, 1) == 1;
}

int IGIFConverter:: writeGif(int fd, iBYTE *data)
{
	int xpos = 0, ypos = 0;
	int xscreen = fWidth, yscreen = fHeight;
	int inx_background = 0, inx_transcol = -1;

	IFUNCTRACE_DEVELOP();
	/*write header etc*/
	{
		iBYTE scn_desc[7], image_desc[10]; int p;
		char *sig = inx_transcol != -1 ? "GIF89a" : "GIF87a";

		/* Write signiture */

		if ( write(fd, sig, 6) != 6 )
			return GBM_ERR_WRITE;

		/* Write screen descriptor */

		scn_desc[0] = low_byte(xscreen);
		scn_desc[1] = high_byte(xscreen);
		scn_desc[2] = low_byte(yscreen);
		scn_desc[3] = high_byte(yscreen);
		scn_desc[4] = (0x80 | ((fBitsPerPixel - 1) * 0x11));
		/* Global colour table follows */
		/* Quality bpp == gct bpp == fBitsPerPixel */
		scn_desc[5] = (iBYTE) inx_background;
		scn_desc[6] = 0;
		if ( write(fd, scn_desc, 7) != 7 )
			return GBM_ERR_WRITE;

		/* Write global colour table */

		IR8G8B8A8Color *clrData = fPal->fColors;
		iBYTE pal[3];

		for ( p = 0; p < (1 << fBitsPerPixel); p++ )
		{
                        pal[0] = clrData[p].fRed;
                        pal[1] = clrData[p].fGreen;
                        pal[2] = clrData[p].fBlue;
			if ( write(fd, pal, 3) != 3 )
				return GBM_ERR_WRITE;
		}

		if ( inx_transcol != -1 )
		/* Do GIF89a "Graphic Control Extension" application extension block */
		{
			char gce[8];
			gce[0] = 0x21;                  /* Extension Introducer */
			gce[1] = 0xf9;                  /* Graphic Control Label */
			gce[2] = 4;                     /* Block size */
			gce[3] = 0x01;                  /* No 'disposal', no 'user input' */
			/* Just transparent index present */
			gce[4] = 0;                     /* Delay time, 0 => not set */
			gce[5] = 0;                     /* Delay time, 0 => not set */
			gce[6] = (char) inx_transcol;   /* Transparent colour index */
			gce[7] = 0;                     /* Block size, 0 => end of extension */
			if ( write(fd, gce, 8) != 8 )
				return GBM_ERR_WRITE;
		}

		/* Do image descriptor block */

		image_desc[0] = (iBYTE) 0x2c;
		image_desc[1] = low_byte(xpos);
		image_desc[2] = high_byte(xpos);
		image_desc[3] = low_byte(ypos);
		image_desc[4] = high_byte(ypos);
		image_desc[5] = low_byte(fWidth);
		image_desc[6] = high_byte(fWidth);
		image_desc[7] = low_byte(fHeight);
		image_desc[8] = high_byte(fHeight);
		image_desc[9] = fBitsPerPixel - 1;
		/* Non-interlaced, no local colour map, no sorted palette */
		if ( write(fd, image_desc, 10) != 10 )
			return GBM_ERR_WRITE;
	}
	/*LZW encode data tail col lookup version*/
	{
		int stride = PIXELSTRIDE(fWidth, fBitsPerPixel);
		iBYTE min_code_size, *pdata;
		int init_code_size, x, y;
		iCWORD clear_code, eoi_code, last_code, max_code, tail;
		iCWORD hashvalue, lenstring, j;
		DICT *dict, **hashtable;
		WRITE_CONTEXT w;

		/* Now LZW encode data */

		if ( (dict = (DICT *) malloc(MAX_DICT * sizeof(DICT))) == NULL )
			return GBM_ERR_MEM;

		if ( (hashtable = (DICT **) malloc(MAX_HASH * sizeof(DICT *))) == NULL )
		{
			free(dict);
			return GBM_ERR_MEM;
		}

		/* Initialise encoder variables */

		init_code_size = fBitsPerPixel + 1;
		if ( init_code_size == 2 )
		/* Room for col0, col1, cc, eoi, but no others! */
			init_code_size++;

		min_code_size = init_code_size - 1;
		if ( write(fd, &min_code_size, 1) != 1 )
		{
			free(hashtable);
			free(dict);
			return GBM_ERR_WRITE;
		}

		clear_code = ( 1 << min_code_size );
		eoi_code   = clear_code + 1;
		last_code  = eoi_code;
		max_code   = ( 1 << init_code_size );
		lenstring  = 0;

		/* Setup write context */

		w.fd        = fd;
		w.inx       = 0;
		w.code_size = init_code_size;
		memset(w.buf, 0, sizeof(w.buf));

		if ( !write_code(clear_code, &w) )
		{
			free(hashtable);
			free(dict);
			return GBM_ERR_WRITE;
		}

		for ( j = 0; j < MAX_HASH; j++ )
			hashtable[j] = NULL;

//		data += ( (fHeight - 1) * stride );
		for ( y = 0; y < fHeight; y++, data += stride )
			for ( x = 0, pdata = data; x < fWidth; x++ )
			{
				iBYTE col;
				switch ( fBitsPerPixel )
				{
				case 8:
					col = *pdata++;
					break;
				case 4:
					if ( x & 1 )
						col = (*pdata++ & 0x0f);
					else
						col = (*pdata >> 4);
					break;
				default: /* must be 1 */
					if ( (x & 7) == 7 )
						col = (*pdata++ & 0x01);
					else
						col = ((*pdata >> (7-(x&7))) & 0x01);
					break;
				}
				/*LZW encode*/
				if ( ++lenstring == 1 )
				{
					tail      = col;
					hashvalue = INIT_HASH(col);
				} else {
					hashvalue *= ( col + lenstring + 4 );
					j = ( hashvalue %= MAX_HASH );
					while ( hashtable[j] != NULL &&
						( hashtable[j]->tail != tail ||
						hashtable[j]->col  != col  ) )
						if ( ++j >= MAX_HASH )
							j = 0;

					if ( hashtable[j] != NULL )
					/* Found in the strings table */
						tail = (hashtable[j]-dict);
					else /* Not found */
					{
						if ( !write_code(tail, &w) )
						{
							free(hashtable);
							free(dict);
							return GBM_ERR_WRITE;
						}
						hashtable[j]       = dict + ++last_code;
						hashtable[j]->tail = tail;
						hashtable[j]->col  = col;
						tail               = col;
						hashvalue          = INIT_HASH(col);
						lenstring          = 1;

						if ( last_code >= max_code )
						/* Next code will be written longer */
						{
							max_code <<= 1;
							w.code_size++;
						}
						else if ( last_code >= MAX_DICT-2 )
						/* Reset tables */
						{
							if ( !write_code(tail      , &w) ||
								!write_code(clear_code, &w) )
							{
								free(hashtable);
								free(dict);
								return GBM_ERR_WRITE;
							}
							lenstring   = 0;
							last_code   = eoi_code;
							w.code_size = init_code_size;
							max_code    = ( 1 << init_code_size );
							for ( j = 0; j < MAX_HASH; j++ )
								hashtable[j] = NULL;
						}
					}
				}
			}

		free(hashtable);
		free(dict);

		if ( !write_code(tail    , &w) ||
			!write_code(eoi_code, &w) ||
			!flush_code(          &w) )
				return GBM_ERR_WRITE;
	}
	/*write terminator*/
	{
		iBYTE term = 0x3b;
		if ( write(fd, &term, 1) != 1 )
			return GBM_ERR_WRITE;
	}

	return GBM_ERR_OK;
}






//***********************************************************************

//IBMPConverter

IBMPConverter::IBMPConverter()
{

}
IBMPConverter::~IBMPConverter()
{
}

void IBMPConverter::writeTo( const IGImage& fromImage,
	IDataStream& toWhere) const
{
}

IGImage* IBMPConverter::createFromStream( IDataStream& fromWhere )
{
	long err;

	IFUNCTRACE_DEVELOP();
	err=readBmp(fromWhere); //read bmp header

	if(err != GBM_ERR_OK){ //exeception
		GrafDeviceException("Error in reading BMP Image header");
	}

	iBYTE *fImgData = new iBYTE[PIXELSTRIDE(fWidth,fBitsPerPixel) * fHeight];

	err=bmp_rdata(fromWhere, fImgData); //read gif data

	if(err != GBM_ERR_OK) //exeception
		GrafDeviceException("Error in reading BMP Image data");

	IPixelBuffer *tmpPix = constructPixelBuffer(fImgData);

	IGImage *igimage = new IGImage();

	attachPixelBuffer(igimage, tmpPix);

	delete [] fImgData;

	return( igimage);
}

#ifdef IC_WIN
IPixelBuffer* IBMPConverter::createDIBFromResource(char *bmpInfo, unsigned long resSize)
{
/*
1. Dump the data into a stream
2. Read/interpret the data.
3. Create a DIB.
*///Phase 1
	IFUNCTRACE_DEVELOP();
	IDataStream *theStream = IDataStream::createMemoryStream ();
	theStream->reset ();
	theStream->writeBytes(bmpInfo, resSize);
	theStream->reset ();

	fImageFileSize = resSize;
	fImgFormat = IGImagePixelAccessor::kBMP; // always BMP files from resources
	long err;

//Phase 2
	err=readBmp(*theStream, false); //read bmp header
	iBYTE *fImgData = new iBYTE[PIXELSTRIDE(fWidth,fBitsPerPixel) * fHeight];
	err=bmp_rdata(*theStream, fImgData); //read gif data

//Phase 3
	if(err != GBM_ERR_OK) //exeception
		GrafDeviceException("Error in reading BMP Image data");

	IPixelBuffer *tmpPix = constructPixelBuffer(fImgData);

	delete [] fImgData;
	delete theStream;

	return(tmpPix);
}
#endif //IC_WIN

IGImage* IBMPConverter::createFromFile( const char *fromFile)
{
	long err;
	IFUNCTRACE_DEVELOP();
	int fd = open(fromFile, O_RDONLY|BINARYOPEN);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file read error", fromFile);
#endif
		GrafDeviceException("Error in opening BMP Image file");
	}
	IDataStream *theStream = IDataStream::createMemoryStream ();
	theStream->reset();
	char tdata[BLOCKREADSIZE];
	long i,length, cnt;

	lseek(fd, 0L, SEEK_END);
#ifdef IC_MOTIF
	length=lseek(fd, 0L, SEEK_CUR);
#else
	length = tell( fd );
#endif //IC_MOTIF
	fImageFileSize = length;
	lseek(fd, 0L, SEEK_SET);

	//dump the file data into stream
	for(cnt=0, i=0; cnt<length; cnt+= i){
		i=read(fd, tdata, BLOCKREADSIZE);
		theStream->writeBytes(tdata, i);
		if(i<BLOCKREADSIZE)
			break;
	}
	close(fd);
	theStream->reset();
	IGImage *igimage = createFromStream(*theStream);
	delete theStream;

	return( igimage);
}


int IBMPConverter::writeToFile( char *toFile, iBYTE *data)
{
	long err=-1;

	IFUNCTRACE_DEVELOP();
	int fd = open(toFile, O_CREAT|O_WRONLY|BINARYOPEN|O_TRUNC, S_IREAD|S_IWRITE);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file create error", toFile);
#endif
		GrafDeviceException("Error in creating GIF Image file");
	}

	err = writeBmp(fd, data);
	close(fd);

	return err;

}


//BMP Reader helpers

#define WIN_OS2_OLD	12
#define WIN_NEW		40
#define OS2_NEW		64

/*bmp_rhdr*/
int IBMPConverter::readBmp(IDataStream& fromWhere, bool skipper)
{
	iWORD usType, xHotspot, yHotspot;
	iDWORD cbSize, cbFix;
	int winSkip = 14, os2Skip=26;
	long    curPos=0, i;


	IFUNCTRACE_DEVELOP();
	if(skipper){ //should be set if reading a mem dump of resource
	read_usword(fromWhere, &usType);

	if ( usType == BFT_BITMAPARRAY ) /*handle bitmap arrays*/
	{
		bool match=false;
		iDWORD   cbSize2, offNext;
#ifdef IC_MOTIF
		Display *dpy = IXDisplay::display();

		iWORD    cxScreen(DisplayWidth(dpy,DefaultScreen(dpy))),
			cyScreen(DisplayHeight(dpy,DefaultScreen(dpy)));
#else
		iWORD    cxScreen(IQUERYSYSVALUE(SV_CXSCREEN)),
		cyScreen(IQUERYSYSVALUE(SV_CYSCREEN));
#endif //IC_MOTIF

		iWORD	 xDelta(0xFFFF), yDelta(0xFFFF), cxDisplay, cyDisplay;

		read_usdword(fromWhere, &cbSize2);
		read_usdword(fromWhere, &offNext);
		read_usword(fromWhere, &cxDisplay);
		read_usword(fromWhere, &cyDisplay);
		curPos = 14;

		while (!match)
		{
			if ( cxScreen == cxDisplay &&
				cyScreen == cyDisplay )
			{
				match = true;
				break;
			}
			else if (!( cxDisplay > cxScreen && cyDisplay > cyScreen)  &&
					(((cxScreen - cxDisplay) < xDelta) &&
				((cyScreen - cyDisplay) < yDelta)))
			{
				xDelta = cxScreen - cxDisplay;
				yDelta = cyScreen - cyDisplay;
			}
			if ( offNext == 0L )
				break;
			fromWhere.reset();
			for(i=0;i<offNext;i++)
				fromWhere.readByte();

			curPos += offNext;

			read_usword(fromWhere, &usType);
			read_usdword(fromWhere, &cbSize2);
			read_usdword(fromWhere, &offNext);
			read_usword(fromWhere, &cxDisplay);
			read_usword(fromWhere, &cyDisplay);
			curPos += 14;

			if ( usType != BFT_BITMAPARRAY )
				return GBM_ERR_BAD_MAGIC;
		}

		if (!match)
		{
			fromWhere.reset();
			for(i=0;i<curPos;i++)
				fromWhere.readByte();
		}

		read_usword(fromWhere, &usType);
	}


	if ( usType != BFT_BMAP )
		return GBM_ERR_BAD_MAGIC;

	iBase = curPos;

	read_usdword(fromWhere, &cbSize);
	read_usword(fromWhere, &xHotspot);
	read_usword(fromWhere, &yHotspot);
	xHot = xHotspot;
	yHot = yHotspot;
	read_usdword(fromWhere, &offBits);
	} else { //skipper
		iBase = 0;
		offBits = 0;
		winSkip = 4, os2Skip=16;
	}
	read_usdword(fromWhere, &cbFix);


	if ( cbFix == WIN_OS2_OLD ) /* OS/2 1.x uncompressed bitmap */
	{
		iWORD cx, cPlanes, cBitCount;
		short cy;

		read_usword(fromWhere, &cx);
		read_sword(fromWhere, &cy);
		read_usword(fromWhere, &cPlanes);
		read_usword(fromWhere, &cBitCount);
		fWidth   = (iWORD) cx;
		fHeight   =  cy;
		fBitsPerPixel = cBitCount;

		if ( cx == 0 || cy == 0 )
			return GBM_ERR_BAD_SIZE;
		if ( cPlanes != 1 )
			return GBM_ERR_BMP_PLANES;
		if ( fBitsPerPixel != 1 && fBitsPerPixel != 4 && fBitsPerPixel != 8 && fBitsPerPixel != 24 )
			return GBM_ERR_BPP;

		fColormapSize = (1 << fBitsPerPixel);

#if defined(IC_MOTIF)
			if(fHeight > 0)
				bFlipImage=true;
			else
				fHeight = -fHeight;
#else
			if(fHeight < 0){
				bFlipImage=true;
				fHeight = -fHeight;
			}
#endif

		bWindows = false;
	}
	else if ( cbFix >= 16 && cbFix <= OS2_NEW &&
		((cbFix & 3) == 0 || cbFix == 42 || cbFix == 46) )
		/*OS\47\2 2\46\0 and Windows 3\46\0*/
	{
		iWORD cPlanes,
			cBitCount,
			usUnits,
			usReserved,
			usRecording,
			usRendering;
		iDWORD ulWidth,
			ulSizeImage,
			ulXPelsPerMeter,
			ulYPelsPerMeter;
		iDWORD cclrUsed,
			cclrImportant,
			cSize1,
			cSize2,
			ulColorEncoding,
			ulIdentifier;
		long lHeight;

		read_usdword(fromWhere, &ulWidth);
		read_sdword(fromWhere, &lHeight);
		fWidth   = (long) ulWidth;
		fHeight   = (long) lHeight;

		if(cbFix == WIN_NEW){
#if defined(IC_MOTIF)
			if(fHeight > 0)
				bFlipImage=true;
			else
				fHeight = -fHeight;
#else
			if(fHeight < 0){
				bFlipImage=true;
				fHeight = -fHeight;
			}
#endif
		}


		read_usword(fromWhere, &cPlanes);
		read_usword(fromWhere, &cBitCount);
		fBitsPerPixel = cBitCount;

		if ( cbFix > 16 )
			read_usdword(fromWhere, &ulCompression);
		else
			ulCompression = BCA_UNCOMP;

		if ( cbFix > 20 )
			read_usdword(fromWhere, &ulSizeImage);
		if ( cbFix > 24 )
			read_usdword(fromWhere, &ulXPelsPerMeter);
		if ( cbFix > 28 )
			read_usdword(fromWhere, &ulYPelsPerMeter);
		if ( cbFix > 32 )
		{
			read_usdword(fromWhere, &cclrUsed);
			fColormapSize = cclrUsed;
		}
		else
			fColormapSize = ( 1 << fBitsPerPixel );

		if ( fBitsPerPixel != 24 && fColormapSize == 0 )
			fColormapSize = ( 1 << fBitsPerPixel );

		if ( cbFix > 36 )
			read_usdword(fromWhere, &cclrImportant);
		if ( cbFix > 40 )
			read_usword(fromWhere, &usUnits);
		if ( cbFix > 42 )
			read_usword(fromWhere, &usReserved);
		if ( cbFix > 44 )
			read_usword(fromWhere, &usRecording);
		if ( cbFix > 46 )
			read_usword(fromWhere, &usRendering);
		if ( cbFix > 48 )
			read_usdword(fromWhere, &cSize1);
		if ( cbFix > 52 )
			read_usdword(fromWhere, &cSize2);
		if ( cbFix > 56 )
			read_usdword(fromWhere, &ulColorEncoding);
		if ( cbFix > 60 )
			read_usdword(fromWhere, &ulIdentifier);

		if ( fWidth == 0L || fHeight == 0L )
			return GBM_ERR_BAD_SIZE;

		if ( cPlanes != 1 )
			return GBM_ERR_BMP_PLANES;
		if ( fBitsPerPixel != 1 && fBitsPerPixel != 4 && fBitsPerPixel != 8 && fBitsPerPixel != 24 )
			return GBM_ERR_BPP;


		bWindows       = true;
	}
	else
		return GBM_ERR_BMP_CBFIX;

	if ( fBitsPerPixel != 24 )
	{
		int i;
		iBYTE b[4];

		fHasColormap = true; //has colormap associated
		fPal = new IRGBAColorArray(fColormapSize);
		IR8G8B8A8Color *clrData = fPal->fColors;

		if ( bWindows ) /*OS\47\2 2\46\0 and Windows 3\46\0*/
		{
			for ( i = 0; i < fColormapSize; i++ )
			{
				fromWhere.readBytes((char*)&b[0], 4);
				
				clrData[i].fRed = b[2];
				clrData[i].fGreen = b[1];
				clrData[i].fBlue = b[0];
				clrData[i].fOpacity = 0xFF;
			}
		}
		else /*OS\47\2 1\46\1\44\ 1\46\2*/
		{
			for ( i = 0; i < fColormapSize; i++ )
			{
				fromWhere.readBytes((char*)&b[0], 3);
				clrData[i].fRed = b[2];
				clrData[i].fGreen = b[1];
				clrData[i].fBlue = b[0];
				clrData[i].fOpacity = 0xFF;
			}
		}
	}

	if ( fBitsPerPixel == 1 ) //&& !bmp_priv->inv ) SAS
	{
		IBaseColor cTmp = fPal->color(0);
		fPal->setColor(0, fPal->color(1));
		fPal->setColor(1, cTmp);
	}

	return GBM_ERR_OK;
}

AHEAD* IBMPConverter::gbm_create_ahead(IDataStream &fromWhere)
{
	AHEAD *ahead;

	if ( (ahead = (AHEAD *)malloc(sizeof(AHEAD))) == NULL ) // mjs
		return NULL;

	ahead->inx = 0;
	ahead->cnt = 0;
	ahead->fromWhere  = &fromWhere;

        return ahead;
}

void IBMPConverter::gbm_destroy_ahead(AHEAD *ahead)
{
	free(ahead);
}

int IBMPConverter::gbm_read_ahead(AHEAD *ahead)
{
	if ( ahead->inx >= ahead->cnt )
	{
		ahead->fromWhere->readBytes((char *)ahead->buf, AHEAD_BUF);
		ahead->cnt = AHEAD_BUF;
		if ( ahead->cnt <= 0 )
			return -1;
		ahead->inx = 0;
	}
	return (int) (iCWORD) ahead->buf[ahead->inx++];
}

/*bmp_rdata*/
int IBMPConverter::bmp_rdata(IDataStream &fromWhere, iBYTE *data)
{
	IFUNCTRACE_DEVELOP();
	int cLinesWorth = PIXELSTRIDE(fWidth, fBitsPerPixel);

	if ( bWindows ) /*OS\47\2 2\46\0 and Windows 3\46\0*/
	{
		switch ( (int)ulCompression )
		{
		case BCA_UNCOMP:
		{
			fromWhere.readBytes((char *)data, fHeight * cLinesWorth);
		}
			break;
		case BCA_RLE8:
		{
			AHEAD *ahead;
			int stride = ((fWidth + 3) & ~3);
			iWORD x = 0, y = 0;
			bool eof8 = false;

			if ( (ahead = gbm_create_ahead(fromWhere)) == NULL )
				return GBM_ERR_MEM;

			while ( !eof8 )
			{
				iBYTE c = (iBYTE) gbm_read_ahead(ahead);
				iBYTE d = (iBYTE) gbm_read_ahead(ahead);

				if ( c )
				{
					memset(data, d, c);
					x += c;
					data += c;
				}
				else
					switch ( d )
					{
					case MSWCC_EOL:
					{
						iWORD     to_eol = stride - x;

						memset(data, 0, to_eol);
						x = 0; y++;
						data += to_eol;
					}
						break;
					case MSWCC_EOB:
						if ( y < fHeight )
						{
							iWORD     to_eol = stride - x;

							memset(data, 0, to_eol);
							x = 0; y++;
							data += to_eol;
							while ( y < fHeight )
							{
								memset(data, 0, stride);
								data += stride;
								y++;
							}
						}
						eof8 = true;
						break;
					case MSWCC_DELTA:
					{
						iBYTE dx = (iBYTE) gbm_read_ahead(ahead);
						iBYTE dy = (iBYTE) gbm_read_ahead(ahead);
						iWORD fill = dx + dy * stride;

						x += dx; y += dy;

						memset(data, 0, fill);
						data += fill;
					}
						break;
					default:
						{
						iWORD     n = (int) d;

						while ( n-- > 0 )
						*data++ = (iBYTE) gbm_read_ahead(ahead);
						x += d;
						if ( d & 1 )
							gbm_read_ahead(ahead); /* Align */
						}
						break;
					}
			}

			gbm_destroy_ahead(ahead);
		}
			break;
		case BCA_RLE4:
		{
			AHEAD *ahead;
			iWORD x = 0, y = 0;
			bool eof4 = false;
			iWORD inx = 0;

			if ( (ahead = gbm_create_ahead(fromWhere)) == NULL )
				return GBM_ERR_MEM;

			memset(data, 0, fHeight * cLinesWorth);

			while ( !eof4 )
			{
				iBYTE c = (iBYTE) gbm_read_ahead(ahead);
				iBYTE d = (iBYTE) gbm_read_ahead(ahead);

				if ( c )
				{
					iBYTE h, l;
					int i;

					if ( x & 1 )
					{	h = (iBYTE) (d >> 4); l = (iBYTE) (d << 4);}
					else
					{	h = (iBYTE) (d & 0xf0); l = (iBYTE) (d & 0x0f);}
					for ( i = 0; i < (iWORD) c; i++, x++ )
					if ( x & 1 )
						data[inx++] |= l;
					else
						data[inx]   |= h;
				}
				else
				switch ( d )
				{
					case MSWCC_EOL:
						for ( ; x < fWidth; x++ )
							if ( x & 1 )
								inx++;
						x = 0; y++;
						inx = ((inx + 3) & ~3); /* Align output */
						break;
					case MSWCC_EOB:
						eof4 = true;
						break;
					case MSWCC_DELTA:
					{
						iBYTE dx = (iBYTE) gbm_read_ahead(ahead);
						iBYTE dy = (iBYTE) gbm_read_ahead(ahead);

						y   += dy;
						inx += dy * cLinesWorth;

						if ( dx > 0 )
						{
							if ( x & 1 )
							{
								inx++;
								x++;
								dx--;
							}

							inx += (dx / 2);
							x += dx;
						}

					}
					break;
					default:
					{
						int     i, nr = 0;

						if ( x & 1 )
						{
							for ( i = 0; i < (iWORD) d; i += 2 )
							{
								iBYTE b = (iBYTE) gbm_read_ahead(ahead);

								data[inx++] |= (b >> 4);
								data[inx  ] |= (b << 4);
								nr++;
							}
							if ( i < (iWORD) d )
							{
								data[inx++] |= ((iBYTE) gbm_read_ahead(ahead) >> 4);
								nr++;
							}
						}
						else
						{
							for ( i = 0; i < (iWORD) d; i += 2 )
							{
								data[inx++] = (iBYTE) gbm_read_ahead(ahead);
								nr++;
							}
							if ( i < (iWORD) d )
							{
								data[inx] = (iBYTE) gbm_read_ahead(ahead);
								nr++;
							}
						}
						x += d;

						if ( nr & 1 )
							gbm_read_ahead(ahead); /* Align input stream to next word */
					}
					break;
				}
			}
			gbm_destroy_ahead(ahead);
		}
			break;
		default:
			return GBM_ERR_BMP_COMP;
		}
	}
        else
	{
		fromWhere.readBytes((char *)data, cLinesWorth * fHeight);
	}

        return GBM_ERR_OK;
}



/*write_inv*/
iCWORD IBMPConverter::writeInv(int fd, iBYTE* buffer, iCWORD count)
{
	iBYTE small_buf[1024];
	iCWORD so_far = 0, this_go, written;

#if 0
	while ( so_far < count )
	{
		this_go = min(count - so_far, 1024);
		memcpy(small_buf, buffer + so_far, this_go);
		invert((iBYTE *)small_buf, this_go);       // mjs
		if ( (written = write(fd, small_buf, this_go)) != this_go )
			return so_far + written;
		so_far += written;
	}
#endif

	return so_far;
}

int IBMPConverter::writeBmp(int fd, iBYTE* data)
{
	int cRGB;
	RGBAClr rgb_1bpp[2];
        bool pm11=false, win, invb=false;

	IFUNCTRACE_DEVELOP();
	cRGB = ( (1 << fBitsPerPixel) & 0x1ff );
	/* 1->2, 4->16, 8->256, 24->0 */

	if ( cRGB == 2 ) /*handle messy 1bpp case*/
	{
		/*
		The palette entries inside a 1bpp PM bitmap are not honored, or handled
		correctly by most programs. Current thinking is that they have no actual
		meaning. Under OS/2 PM, bitmap 1's re fg and 0's are bg, and it is the job of
		the displayer to pick fg and bg. We will pick fg=black, bg=white in the bitmap
		file we save. If we do not write black and white, we find that most programs
		will incorrectly honor these entries giving unpredicatable (and often black on
		a black background!) results.
		*/

		rgb_1bpp[0].R = rgb_1bpp[0].G = rgb_1bpp[0].B = 0xff;
		rgb_1bpp[1].R = rgb_1bpp[1].G = rgb_1bpp[1].B = 0x00;

		/*
		We observe these values must be the wrong way around to keep most PM
		programs happy, such as WorkPlace Shell WPFolder backgrounds.
		*/

	}

	if ( pm11 )
	{
		iWORD usType, xHotspot, yHotspot, cx, cy, cPlanes, cBitCount;
		iDWORD cbSize, offBits, cbFix;
		iWORD cLinesWorth, i, total, actual;

		usType      = BFT_BMAP;
		xHotspot    = xHot;
		yHotspot    = yHot;
		cbFix       = 12L;
		cx          = fWidth;
		cy          = fHeight;
		cPlanes     = 1;
		cBitCount   = fBitsPerPixel;
		cLinesWorth = PIXELSTRIDE(fWidth, fBitsPerPixel) * cPlanes;
				// (((cBitCount * cx + 31) / 32) * cPlanes) * 4;
		offBits     = 26L + cRGB * 3L;
		cbSize      = offBits + (iDWORD) cy * (iDWORD) cLinesWorth;

		write_word(fd, usType);
		write_dword(fd, cbSize);
		write_word(fd, xHotspot);
		write_word(fd, yHotspot);
		write_dword(fd, offBits);
		write_dword(fd, cbFix);
		write_word(fd, cx);
		write_word(fd, cy);
		write_word(fd, cPlanes);
		write_word(fd, cBitCount);

		IR8G8B8A8Color *clrData = fPal->fColors;
		iBYTE b[3];
		for ( i = 0; i < cRGB; i++ )
		{
                        b[2] = clrData[i].fRed;
                        b[1] = clrData[i].fGreen;
                        b[0] = clrData[i].fBlue;
			if ( write(fd, b, 3) != 3 )
				return GBM_ERR_WRITE;
		}

		total = fHeight * cLinesWorth;
#if 0
		if ( invb )
			actual = write_inv(fd, data, total); // mjs
		else
#endif
			actual = write(fd, data, total);     // mjs
		if ( actual != total )
			return GBM_ERR_WRITE;
	} else {
		iWORD usType, xHotspot, yHotspot, cPlanes, cBitCount;
		iDWORD cx, cbSize, offBits, cbFix, ulCompression, cbImage;
		iDWORD cxResolution, cyResolution, cclrUsed, cclrImportant;
		int cLinesWorth, i, total, actual;
		long cy;

		usType      = BFT_BMAP;
		xHotspot    = xHot;
		yHotspot    = yHot;
		cbFix       = 40L;
		cx          = fWidth;
		cy = fHeight;
		cPlanes     = 1;
		cBitCount   = fBitsPerPixel;
		cLinesWorth = PIXELSTRIDE(fWidth, fBitsPerPixel) * cPlanes;
				// (((cBitCount * (iWORD) cx + 31) / 32) * cPlanes) * 4;
		offBits     = 54L + cRGB * 4L;
		cbSize      = offBits + (iDWORD) cy * (iDWORD) cLinesWorth;

		ulCompression = BCA_UNCOMP;
		cbImage       = cLinesWorth * fHeight;
		cxResolution  = 0;
		cyResolution  = 0;
		cclrUsed      = 0;
		cclrImportant = 0;

		write_word(fd, usType);
		write_dword(fd, cbSize);
		write_word(fd, xHotspot);
		write_word(fd, yHotspot);
		write_dword(fd, offBits);

		write_dword(fd, cbFix);
		write_dword(fd, cx);
		write_dword(fd, cy);

		write_word(fd, cPlanes);
		write_word(fd, cBitCount);
		write_dword(fd, ulCompression);
		write_dword(fd, cbImage);
		write_dword(fd, cxResolution);
		write_dword(fd, cyResolution);
		write_dword(fd, cclrUsed);
		write_dword(fd, cclrImportant);

		IR8G8B8A8Color *clrData = fPal->fColors;
		iBYTE b[4];
		for ( i = 0; i < cRGB; i++ )
		{

                        b[2] = clrData[i].fRed;
                        b[1] = clrData[i].fGreen;
                        b[0] = clrData[i].fBlue;
			b[3] = 0;
			if ( write(fd, b, 4) != 4 )
				return GBM_ERR_WRITE;
		}

		total = fHeight * cLinesWorth;
#if 0
		if ( invb )
			actual = write_inv(fd, (char*)data, total);  // mjs
		else
#endif
			actual = write(fd, (char*)data, total);      // mjs
		if ( actual != total )
			return GBM_ERR_WRITE;
	}

	return GBM_ERR_OK;
}


//***********************************************************************
//IJPGConverter
//***********************************************************************

IJPGConverter::IJPGConverter()
{
	
}
IJPGConverter::~IJPGConverter()
{
}

void IJPGConverter::writeTo( const IGImage& fromImage,
				IDataStream& toWhere) const
{
}


IGImage* IJPGConverter::createFromStream( IDataStream& fromWhere )
{
#ifndef IC_MOTIF
	GrafDeviceException("JPG not supported");
#else
   iBYTE *fImgData;
	long err;

	IFUNCTRACE_DEVELOP();
	err = readJpg(fromWhere, &fImgData); //read jpg image file

	if(err != GBM_ERR_OK) //exeception
		GrafDeviceException("Error in reading JPG Image header");

#ifndef IC_MOTIF
	bFlipImage = true;
#endif

	IPixelBuffer *tmpPix = constructPixelBuffer(fImgData);

	IGImage *igimage = new IGImage();

	attachPixelBuffer(igimage, tmpPix);

	delete [] fImgData;
	return( igimage);
#endif
}

IGImage* IJPGConverter::createFromFile( const char *fromFile)
{
	IFUNCTRACE_DEVELOP();
#ifndef IC_MOTIF
	GrafDeviceException("JPG not supported");
#else
	int fd = open(fromFile, O_RDONLY|BINARYOPEN);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file read error", fromFile);
#endif
		GrafDeviceException("Error in opening JPG Image file");
	}

	IDataStream *theStream = IDataStream::createMemoryStream ();
	theStream->reset();
	char tdata[BLOCKREADSIZE];
	long i,length, cnt;

	lseek(fd, 0L, SEEK_END);
#ifdef IC_MOTIF
	length=lseek(fd, 0L, SEEK_CUR);
#else
	length = tell( fd );
#endif //IC_MOTIF
	fImageFileSize = length;
	lseek(fd, 0L, SEEK_SET);

	//dump the file data into stream
	for(cnt=0, i=0; cnt<length; cnt+= i){
		i=read(fd, tdata, BLOCKREADSIZE);
		theStream->writeBytes(tdata, i);
		if(i<BLOCKREADSIZE)
			break;
	}
	close(fd);
	theStream->reset ();
	IGImage *igimage = createFromStream(*theStream);
	delete theStream;

	return( igimage);
#endif
}


int IJPGConverter::writeToFile( char *toFile, iBYTE *data)
{
#ifndef IC_MOTIF
	GrafDeviceException("JPG not supported");
#else
	long err;

	IFUNCTRACE_DEVELOP();
	int fd = open(toFile, O_CREAT|O_WRONLY|BINARYOPEN|O_TRUNC, S_IREAD|S_IWRITE);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file create error", toFile);
#endif
		GrafDeviceException("Error in creating GIF Image file");
	}

	err = writeJpg(fd, data);
	close(fd);

	return err;
#endif
}

long IJPGConverter::readJpg(IDataStream &fromWhere, iBYTE **data)
{
#ifndef IC_MOTIF
	GrafDeviceException("JPG not supported");
#else

	JSAMPROW	rowptr[1];
	int		i, row_stride;

	jpgErrMgr jerr;

	IFUNCTRACE_DEVELOP();
	dInfo.err = jpeg_std_error(&jerr.pub);

	if (setjmp(jerr.setJmpBuffer)) {
		/* if we're here, it blew up... */
		jpeg_destroy_decompress(&dInfo);
		return GBM_ERR_BAD_SIZE;
	}

	jpeg_create_decompress(&dInfo);

	jpeg_stdio_src(&dInfo, fromWhere, fImageFileSize);
	(void) jpeg_read_header(&dInfo, TRUE);

	jpeg_calc_output_dimensions(&dInfo);

#if 0
	// fill up the palette
	if (dInfo.jpeg_color_space == JCS_GRAYSCALE) {

		dInfo.out_color_space = JCS_GRAYSCALE;
		dInfo.quantize_colors = FALSE;
	} else {

		dInfo.out_color_space = JCS_RGB;
		dInfo.quantize_colors = TRUE;
		dInfo.desired_number_of_colors = 256;
		dInfo.two_pass_quantize = TRUE;

	}
#endif

	jpeg_calc_output_dimensions(&dInfo);

	fWidth   = dInfo.output_width;
	fHeight   = dInfo.output_height;
	fBitsPerPixel = dInfo.output_components << 3;
	fHasColormap = False;
	fColormapSize = 0;
	jpeg_start_decompress(&dInfo);



#ifdef GRAPH2D_DEBUG
	if (dInfo.jpeg_color_space == JCS_GRAYSCALE) {
		fprintf(stderr,"Loading %dx%d Greyscale JPEG ...\n", fWidth, fHeight);
	} else {
		fprintf(stderr,"Loading %dx%d %d-Color JPEG ...\n", fWidth, fHeight, dInfo.actual_number_of_colors);
	}
#endif

	row_stride = PIXELSTRIDE(fWidth,  dInfo.output_components << 3);
#ifdef GRAPH2D_DEBUG
	fprintf(stderr,"output_components %dx%d \n", dInfo.output_components,row_stride);
#endif

	*data = new iBYTE[row_stride * fHeight];

	while (dInfo.output_scanline < fHeight) {
		rowptr[0] = (JSAMPROW) &((*data)[dInfo.output_scanline * row_stride]);
		(void) jpeg_read_scanlines(&dInfo, rowptr, 1);
	}

	// fill up the palette
	if (dInfo.jpeg_color_space == JCS_GRAYSCALE) {
		fHasColormap = True;
		fColormapSize = 256;
		fPal = new IRGBAColorArray(fColormapSize);
		IR8G8B8A8Color *clrData = fPal->fColors;
		for (i=0; i<256; i++) {
                        clrData[i].fRed = i;
                        clrData[i].fGreen = i;
                        clrData[i].fBlue = i;
                        clrData[i].fOpacity = 0xFF;
		}
	} else {
		if (dInfo.quantize_colors) {
			fHasColormap = True;
			fColormapSize = dInfo.actual_number_of_colors;
			fPal = new IRGBAColorArray(fColormapSize);
			IR8G8B8A8Color *clrData = fPal->fColors;

			for (i=0; i<fColormapSize; i++) {
				clrData[i].fRed = dInfo.colormap[0][i];
				clrData[i].fGreen = dInfo.colormap[1][i];
				clrData[i].fBlue = dInfo.colormap[2][i];
				clrData[i].fOpacity = 0xFF;
			}
		}
	}

	jpeg_finish_decompress(&dInfo);
	jpeg_destroy_decompress(&dInfo);

	return GBM_ERR_OK;
#endif
}

int IJPGConverter::writeJpg(int fd, iBYTE* data)
{
#ifndef IC_MOTIF
	GrafDeviceException("JPG not supported");
#else
	JSAMPROW row_pointer[1];
	int row_stride;
	iBYTE *t_data = NULL;

	jpgErrMgr jerr;

	IFUNCTRACE_DEVELOP();
	cInfo.err = jpeg_std_error(&jerr.pub);
#if 0
	jerr->pub.error_exit     = jpg_error_exit;
	jerr->pub.output_message = jpg_error_output;
#endif

	jpeg_create_compress(&cInfo);
	jpeg_stdio_dest(&cInfo, fd);

	cInfo.image_width = fWidth;  /* image width and height, in pixels */
	cInfo.image_height = fHeight;

	//we will be storing all images in 24 bit mode i.e R G B
	cInfo.input_components = 3;       /* # of color components per pixel */
	cInfo.in_color_space = JCS_RGB;   /* colorspace of input image */
	row_stride = PIXELSTRIDE(cInfo.image_width, cInfo.input_components << 3);

	jpeg_set_defaults(&cInfo);
	jpeg_start_compress(&cInfo, TRUE);

	//convert the 8 bit color image to 24 bit color.
	//as in jpg 8 bit image is grey scale
	if(fBitsPerPixel < 24){
		t_data = new iBYTE[row_stride * cInfo.image_height];
		unsigned long i, j, w_scan, r_scan, scan_stride;
		char t_byte;
		scan_stride = PIXELSTRIDE(cInfo.image_width, fBitsPerPixel);
		IR8G8B8A8Color *clrData = fPal->fColors;
		for(i=0; i<cInfo.image_height; i++){
			w_scan = i * row_stride;
			r_scan = i * scan_stride;
			for(j=0; j<cInfo.image_width; j++){
				if(fBitsPerPixel == 8){
					t_byte = data[r_scan + j];
				} else {
					if(j & 1)
						t_byte = (data[r_scan + (j>>1)] & 0x0F);
					else
						t_byte = (data[r_scan + (j>>1)] & 0xF0) >> 4;
				}
				t_data[w_scan + (j*3)]     = clrData[t_byte].fRed;
				t_data[w_scan + (j*3) + 1] = clrData[t_byte].fGreen;
				t_data[w_scan + (j*3) + 2] = clrData[t_byte].fBlue;
			}
		}
	}

	while (cInfo.next_scanline < cInfo.image_height) {
		row_pointer[0] = &t_data[cInfo.next_scanline * row_stride];
		(void) jpeg_write_scanlines(&cInfo, row_pointer, 1);
	}
	if(fBitsPerPixel < 24){
		delete [] t_data;
	}

	jpeg_finish_compress(&cInfo);
	jpeg_destroy_compress(&cInfo);


	return GBM_ERR_OK;
#endif
}

//***********************************************************************

IXPMConverter::IXPMConverter()
{
	
}
IXPMConverter::~IXPMConverter()
{
}

void IXPMConverter::writeTo( const IGImage& fromImage,
				IDataStream& toWhere) const
{
}

IPixelBuffer* IXPMConverter::createPixBufFromResource(char *xpmData, unsigned long resSize)
{
/*
1. Dump the data into a stream
2. Read/interpret the data.
3. Create a PixelBuffer.
*///Phase 1
	IFUNCTRACE_DEVELOP();
	IDataStream *theStream = IDataStream::createMemoryStream ();
	theStream->reset ();
	theStream->writeBytes(xpmData, resSize);
	theStream->reset ();

	fImageFileSize = resSize;
	long err;

//Phase 2
	iBYTE *fImgData;
	err=readXpm(*theStream, &fImgData); //read xpm data

//Phase 3
	if(err != GBM_ERR_OK) //exeception
		GrafDeviceException("Error in reading XPM Image data");

	IPixelBuffer *tmpPix = constructPixelBuffer(fImgData);

	delete [] fImgData;
	delete theStream;

	return(tmpPix);
}

IGImage* IXPMConverter::createFromStream( IDataStream& fromWhere )
{
	long err;

	iBYTE *fImgData;

	IFUNCTRACE_DEVELOP();
	err = readXpm(fromWhere, &fImgData); //read gif header

	if(err != GBM_ERR_OK){ //exeception
		GrafDeviceException("Error in reading XPM Image header");
	}

#ifndef IC_MOTIF
	bFlipImage = true;
#endif

	IPixelBuffer *tmpPix = constructPixelBuffer(fImgData);

	IGImage *igimage = new IGImage();

	attachPixelBuffer(igimage, tmpPix);

	delete [] fImgData;
	return(igimage);
}

IGImage* IXPMConverter::createFromFile( const char *fromFile)
{
	int fd = open(fromFile, O_RDONLY|BINARYOPEN);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file read error", fromFile);
#endif
		GrafDeviceException("Error in opening GIF Image file");
	}
	IDataStream *theStream = IDataStream::createMemoryStream ();
	theStream->reset();
	char tdata[BLOCKREADSIZE];
	long i,length, cnt;

	lseek(fd, 0L, SEEK_END);
#ifdef IC_MOTIF
	length=lseek(fd, 0L, SEEK_CUR);
#else
	length = tell( fd );
#endif //IC_MOTIF
	fImageFileSize = length;
	lseek(fd, 0L, SEEK_SET);

	//dump the file data into stream
	for(cnt=0, i=0; cnt<length; cnt+= i){
		i=read(fd, tdata, BLOCKREADSIZE);
		theStream->writeBytes(tdata, i);
		if(i<BLOCKREADSIZE)
			break;
	}
	close(fd);
	theStream->reset ();
	IGImage *igimage = createFromStream(*theStream);
	delete theStream;

	return( igimage);

}

int IXPMConverter::writeToFile( char *toFile, iBYTE* data)
{
	long err;

	int fd = open(toFile, O_CREAT|O_WRONLY|BINARYOPEN|O_TRUNC, S_IREAD|S_IWRITE);
	if(fd < 0 ){
#ifdef GRAPH2D_DEBUG
		fprintf(stderr, "%s file create error", toFile);
#endif
		GrafDeviceException("Error in creating GIF Image file");
	}

	err = writeXpm(fd, data);
	close(fd);

	return err;

}

//XPM Reader helpers
iBYTE IXPMConverter::getChar(IDataStream &fromWhere)
{
	iBYTE ch=fromWhere.readByte();
#ifdef GRAPH2D_DEBUG
	fprintf(stderr, "%c",ch);
#endif
	return(ch);
}

int IXPMConverter::getString(IDataStream &fromWhere, iBYTE *str)
{
	int i, j, cnt;
	// First, get to the first string
	do{
		str[0] = fromWhere.readByte();

	} while (str[0] != '"');


	j=-1;
	do{
		str[++j] = fromWhere.readByte();

	//find for the terminating '"'
	} while (str[j] != '"');

	str[j] = 0x0;

	return (1);
}
static void toLowerCase(char* str)
{
	int len = strlen(str);
	for(int i=0; i<len; i++){
		if((str[i] >= 'A') && (str[i] <= 'Z'))
			str[i] = 'a' + (str[i] - 'A');
	}
}

//**************************************
//         hashing functions           *
//**************************************


int IXPMConverter::hash(char* token)
{
	int i, sum;

	for (i=sum=0; token[i] != '\0'; i++)
		sum += token[i];

	sum = sum % hash_len;
	return (sum);
}


int IXPMConverter::hashInit(int hsize)
{
	//
	// hashInit() - This function takes an arg, but doesn't do anything with
	// it.  It could easily be expanded to figure out the "optimal" number of
	// buckets.  But, for now, a hard-coded 257 will do.  (Until I finish the
	// 24-bit image writing code.  :-)
	//

	int i;

	hash_len = 257;

	hashtab = (hEntry **) malloc(sizeof(hEntry *) * hash_len);
	if (!hashtab) {
		fprintf(stderr, "Couldn't malloc hashtable in LoadXPM()!\n");
		return 0;
	}

	for (i = 0 ; i < hash_len ; i++)
		hashtab[i] = NULL;

	return 1;
}


int IXPMConverter::hashInsert(hEntry* entry)
{
	int     key;
	hEntry *tmp;

	key = hash(entry->token);

	tmp = (hEntry *) malloc(sizeof(hEntry));
	if (!tmp) {
		fprintf(stderr, "Couldn't malloc hash entry in LoadXPM()!\n");
		return 0;
	}

	memcpy((char *)tmp, (char *)entry, sizeof(hEntry));

	if (hashtab[key])
		tmp->next = hashtab[key];
	else
		tmp->next = NULL;

	hashtab[key] = tmp;

	return 1;
}


hEntry * IXPMConverter::hashSearch(char* token)
{
	int     key;
	hEntry *tmp;

	key = hash(token);

	tmp = hashtab[key];
	while (tmp && strcmp(token, tmp->token)) {
		tmp = tmp->next;
	}

	return tmp;
}


void IXPMConverter::hashDestroy()
{
	int     i;
	hEntry *tmp;

	for (i=0; i<hash_len; i++) {
		while (hashtab[i]) {
			tmp = hashtab[i]->next;
			free(hashtab[i]);
			hashtab[i] = tmp;
		}
	}

	free(hashtab);
	return;
}

int IXPMConverter::parseColor(char* color, XColor* col)
{
	if(color == NULL)
		return 0;

	if(color[0] == '#'){
		int len = strlen(color);
		int cpc = (len -1) / 3; //# of chars per color
		int start=1; //ignore the '#'

		col->red = (hex[color[start]]<<12) | (hex[color[start+1]]<<8);
		start += cpc;
		col->green = (hex[color[start]]<<12) | (hex[color[start+1]]<<8);
		start += cpc;
		col->blue = (hex[color[start]]<<12) | (hex[color[start+1]]<<8);
		return 1;
	}
	return 0;
}


int IXPMConverter::readXpm(IDataStream &fromWhere, iBYTE **data)
{
	iBYTE c, tmp[128], *tmpPtr;
	unsigned long i, j, k, pixScan, cpp;
	hEntry   item;
	XColor col;

	IFUNCTRACE_DEVELOP();
	// Read in the values line.  It is the first string in the
	// xpm, and contains four numbers.  w, h, num_colors, and
	// chars_per_pixel.
	//
	if((getString(fromWhere, tmp)) < 0)
		return -1;

	//skip the x & y hotspot
	sscanf((char *)tmp, "%d%d%d%d%d%d", &fWidth, &fHeight, &fColormapSize, &cpp, &xHot, &yHot);
#ifdef GRAPH2D_DEBUG
	fprintf(stderr, "Width=%d Height=%d # Colors=%d BytesPerPixel=%d HotSpot=%d, %d\n",
				fWidth, fHeight, fColormapSize, cpp, xHot, yHot);
#endif
	if(cpp <= 2)
		fBitsPerPixel = 8;
	else
		fBitsPerPixel = 24;
	
	pixScan = PIXELSTRIDE(fWidth, fBitsPerPixel);
	tmpPtr = new iBYTE[fHeight * pixScan];
	if(fColormapSize <= 256){
		fHasColormap = true;
		fPal = new IRGBAColorArray(fColormapSize);
	} else {
		fHasColormap = false;
		fColormapSize = 0;
	}
	IR8G8B8A8Color *clrData = fPal->fColors;

	if (!hashInit(fColormapSize))
		return -1;

	clmp =  new hEntry[fColormapSize]; // Holds the colormap
	c_sptr = clmp;


	// initialize the 'hex' array for zippy ASCII-hex -> int conversion
	memset(hex, 0x0, 256);
	for (i = '0'; i <= '9' ; i++) hex[i] = (iBYTE)(i - '0');
	for (i = 'a'; i <= 'f' ; i++) hex[i] = (iBYTE)(i - 'a' + 10);
	for (i = 'A'; i <= 'F' ; i++) hex[i] = (iBYTE)(i - 'A' + 10);

	for(i=0; i<fColormapSize; i++)
	{
		while ((c = getChar(fromWhere)) && (c != '"')) ;

		//get the color token
		for (j = 0 ; j < cpp ; j++)
			c_sptr->token[j] = getChar(fromWhere);
		c_sptr->token[j] = '\0';

		while ((c = getChar(fromWhere)) && ((c == ' ') || (c == '\t'))) ;

		do {
			char  key[3];
			char  color[40];	// Need to figure a good size for this...
			short hd;			// Hex digits per R, G, or B
			int ret = 0;
            bool isTransparencyColor = false;

			//get the key (color type)
			for (j=0; j<2 && (c != ' ') && (c != '\t'); j++) {
				key[j] = c;
				c = getChar(fromWhere);
			}
			key[j] = '\0';

			while ((c = getChar(fromWhere)) && ((c == ' ') || (c == '\t'))) ;

			//get the actual color value
			for (j=0; j<39 && (c!=' ') && (c!='\t') && (c!='"'); j++) {
				color[j] = c;
				c = getChar(fromWhere);
			}
			color[j]='\0';

			//skip all the blank characters if any
			while ((c == ' ') || (c == '\t'))
				c = getChar(fromWhere);

			//to be removed once we R ok - SAS
			if (key[0] == 's')	// Don't find a color for a symbolic name
				continue;

            if ((color[0] == 'n'  || color[0] == 'N') &&
                (color[1] == 'o'  || color[1] == 'O') &&
                (color[2] == 'n'  || color[2] == 'N') &&
                (color[3] == 'e'  || color[3] == 'E') && (color[4] == '\0'))
                isTransparencyColor = true;

			//see if color is an hex representation
			if (! isTransparencyColor)
                ret = parseColor(color, &col);

#ifdef IC_MOTIF
			//if not transparency color and not hex repsentation then call for X help
			if (! isTransparencyColor && ! ret){
				ret = XParseColor(IXDisplay::display(),
                                  IColorMap::defaultColorMap().nativeColorMap(),
                                  color, &col);
			}
#endif
			if (ret) {
				if (fHasColormap) {
					c_sptr->cv_index = i;
					clrData[i].fRed = (iBYTE)(col.red >> 8);
					clrData[i].fGreen = (iBYTE)(col.green >> 8);
					clrData[i].fBlue = (iBYTE)(col.blue >> 8);
					clrData[i].fOpacity = 0xFF;
				} else {   // 24 bit
					c_sptr->cv_rgb[0] = (iBYTE)(col.red >> 8);
					c_sptr->cv_rgb[1] = (iBYTE)(col.green >> 8);
					c_sptr->cv_rgb[2] = (iBYTE)(col.blue >> 8);
				}
			} else {      // 'None' or unrecognized color spec
				int rgb;

				if (isTransparencyColor){
					fIsTransparencyEnabled = true;
					rgb = 0xabcdef;  // assume
				} else { //unknown color format specified
					rgb = 0;
				}

				if (fHasColormap) {
					c_sptr->cv_index = i;

					clrData[i].fRed = (iBYTE)((rgb>>16) & 0xff);
					clrData[i].fGreen = (iBYTE)((rgb>> 8) & 0xff);
					clrData[i].fBlue = (iBYTE)(rgb & 0xff);
					clrData[i].fOpacity = 0xFF;
					if(fIsTransparencyEnabled){
						fTransparency = IBaseColor(
								clrData[i].fRed,
								clrData[i].fGreen,
								clrData[i].fBlue,
								clrData[i].fOpacity);
					}
				} else {
					c_sptr->cv_rgb[0] = (iBYTE)((rgb>>16) & 0xff);
					c_sptr->cv_rgb[1] = (iBYTE)((rgb>> 8) & 0xff);
					c_sptr->cv_rgb[2] = (iBYTE)(rgb & 0xff);
				}
			}


			memcpy((char *) &item, (char *) c_sptr, sizeof(item));
			hashInsert(&item);


			if (*key == 'c') {	// This is the color entry, keep it.
				while (c!='"') c = getChar(fromWhere);
				break;
			}
#ifdef GRAPH2D_DEBUG
			if (fHasColormap) {
			fprintf(stderr, "Cmap entry %d, 0x%02x 0x%02x 0x%02x, token '%s'\n",
					i, clrData[i].fRed,
					clrData[i].fGreen,
					clrData[i].fBlue,
					c_sptr->token);
			}
#endif

		} while (c != '"');
		c_sptr++;

	} // for


#ifdef GRAPH2D_DEBUG
	fprintf(stderr, "LoadXPM(): Read and stored colormap.\n");
#endif

	/* Now, read the pixmap. */
	
	iBYTE *ptr, *scan_data;
	scan_data = new iBYTE[(fWidth*cpp) +10];
	for (i = 0 ; i < fHeight ; i++) {
		ptr=&tmpPtr[i*pixScan];
		if((getString(fromWhere, scan_data)) < 0)
			return -1;

		for (j = 0 ; j < fWidth ; j++) {
			char pixel[TOKEN_LEN];
			hEntry *mapentry;

			for (k = 0 ; k < cpp ; k++)
				pixel[k] = scan_data[(j*cpp)+k];
			pixel[k] = '\0';

			if (!(mapentry = (hEntry *) hashSearch(pixel))) {
				// No colormap entry found.  What do we do? return error
				return -1;
			}

			if (fHasColormap)
				ptr[j] = mapentry->cv_index;
			else {
				ptr[(j*3)] = mapentry->cv_rgb[0];
				ptr[(j*3)+1] = mapentry->cv_rgb[1];
				ptr[(j*3)+2] = mapentry->cv_rgb[2];
			}
		}  // for ( j < w )
			(void)getChar(fromWhere);		// Throw away the close "

	}  // for ( i < h )

	//destroy the hash table
	hashDestroy();
	delete [] clmp;
	delete [] scan_data;

	*data = tmpPtr;

	return GBM_ERR_OK;
}

int IXPMConverter::writeXpm(int fd, iBYTE *data)
{

	IFUNCTRACE_DEVELOP();
	if(fBitsPerPixel != 8 )
	{
#ifdef GRAPH2D_DEBUG
		fprintf(stderr,"\nUnsupported depth for XPM writing \n");
#endif
		return(-1);
	}

	char  *tokenchars = "'abcdefghijklmnopqrstuvwxyz";
	unsigned long TOKEN_CNT = strlen(tokenchars);
	char  buf[256], rtemp[256], gtemp[256], btemp[256], hist[256], trans[256];
	unsigned long i, j, k, nClrs=0, scanStride=PIXELSTRIDE(fWidth, fBitsPerPixel);
	unsigned long readscan, cpp=1; //default CharPerPixel = 1
	unsigned long transparencyToken=0;

	memset(hist, 0x0, 256);
	for(i=0; i<fHeight; i++)
	{
		readscan = i*scanStride;
		for(j=0; j<fWidth; j++)
		{
			hist[data[readscan + j]] = 1;
		}
	}
	if(fPal) {
		IR8G8B8A8Color *clrData = fPal->fColors;
		unsigned long noOfClrs = fPal->numberOfColors();
		for(i=0; i<noOfClrs; i++)
		{
			trans[i] = nClrs;
			rtemp[nClrs] = clrData[i].fRed;
			gtemp[nClrs] = clrData[i].fGreen;
			btemp[nClrs] = clrData[i].fBlue;
			if(fIsTransparencyEnabled &&
					(fTransparency == fPal->color(i)))
				transparencyToken=nClrs;
			nClrs += hist[i];

		}
	}

	if(nClrs > TOKEN_CNT)
		cpp = 2;

	iBYTE *scan = new iBYTE[fWidth*cpp];

	sprintf(buf, "/* XPM2 C */\nstatic char *XPMImage[] = {\n/* width height num_colors chars_per_pixel */\n\" %3d %3d %3d %2d %3d %3d\",\n/* colors */\n",
		fWidth, fHeight, nClrs, cpp, xHot, yHot);

	i=write(fd, buf, strlen(buf));
	if(i<strlen(buf)){
		delete [] scan;
		return(-1);
	}

	switch(cpp){
		case 1: // 1 char per pixel
			{
			for (i = 0 ; i < nClrs ; i ++){
				if(fIsTransparencyEnabled && (transparencyToken==i)){
					sprintf(buf, "\"%c c None\",\n", tokenchars[i]);
				}else{
					sprintf(buf, "\"%c c #%02X%02X%02X\",\n", tokenchars[i],
						rtemp[i], gtemp[i], btemp[i]);
				}
				j=write(fd, buf, strlen(buf));
				if(j<strlen(buf)){
					delete [] scan;
					return(-1);
				}
			}
			sprintf(buf, "/* pixels */\n");
			i=write(fd, buf, strlen(buf));
			if(i<strlen(buf)){
				delete [] scan;
				return(-1);
			}

			for (i=0; i < fHeight; i++)
			{
				readscan = i*scanStride;
				sprintf(buf, "\"");
				k=write(fd, buf, strlen(buf));
				if(k<strlen(buf)){
					delete [] scan;
					return(-1);
				}

				for (j=0; j < fWidth; j++)
					scan[j] = tokenchars[trans[data[readscan+j]]];

				//currently we write only 8-bit data	
				k=write(fd, scan, fWidth);
				if(k<fWidth){
					delete [] scan;
					return(-1);
				}
			
				if (i < fHeight-1)
				{
					sprintf(buf, "\",\n");
					k=write(fd, buf, strlen(buf));
					if(k<strlen(buf)){
						delete [] scan;
						return(-1);
					}
				}
			}
			sprintf(buf, "\"\n};\n");
			k=write(fd, buf, strlen(buf));
			if(k<strlen(buf)){
				delete [] scan;
				return(-1);
			}

			delete [] scan;
			}
			break;
		case 2: // 2 char per pixel
			{
			iBYTE *tokens = new iBYTE[(nClrs<<1) + 1];
			unsigned long imax = (nClrs/TOKEN_CNT) + 1;
			for (i = 0; i < imax; i++)
				for (j = 0; j < TOKEN_CNT && ((i*TOKEN_CNT)+j) < nClrs; j++) {
					tokens[((i*TOKEN_CNT)+j)<<1] = tokenchars[i];
					tokens[(((i*TOKEN_CNT)+j)<<1)+1] = tokenchars[j];
				}

			for (i = 0 ; i < nClrs ; i ++){
				if(fIsTransparencyEnabled && (transparencyToken==i)){
					sprintf(buf, "\"%c%c c None\",\n",
						tokens[i<<1], tokens[(i<<1)+1]);
				} else {
					sprintf(buf, "\"%c%c c #%02X%02X%02X\",\n",
						tokens[i<<1], tokens[(i<<1)+1],
						rtemp[i], gtemp[i], btemp[i]);
				}
				j=write(fd, buf, strlen(buf));
				if(j<strlen(buf)){
					delete [] scan;
					delete [] tokens;
					return(-1);
				}
			}
			sprintf(buf, "/* pixels */\n");
			i=write(fd, buf, strlen(buf));
			if(i<strlen(buf)){
				delete [] scan;
				delete [] tokens;
				return(-1);
			}

			for (i=0; i < fHeight; i++)
			{
				readscan = i*scanStride;
				sprintf(buf, "\"");
				k=write(fd, buf, strlen(buf));
				if(k<strlen(buf)){
					delete [] scan;
					delete [] tokens;
					return(-1);
				}

				for (j=0; j < fWidth; j++){
					scan[j*2] = tokens[trans[data[readscan+j]]*2];
					scan[(j*2) + 1] = tokens[(trans[data[readscan+j]]*2) + 1];
				}

				//currently we write only 8-bit data	
				k=write(fd, scan, fWidth*2);
				if(k<(fWidth*2)){
					delete [] scan;
					delete [] tokens;
					return(-1);
				}
			
				if (i < fHeight-1)
				{
					sprintf(buf, "\",\n");
					k=write(fd, buf, strlen(buf));
					if(k<strlen(buf)){
						delete [] scan;
						delete [] tokens;
						return(-1);
					}
				}
			}
			sprintf(buf, "\"\n};\n");
			k=write(fd, buf, strlen(buf));
			if(k<strlen(buf)){
				delete [] scan;
				delete [] tokens;
				return(-1);
			}

			delete [] scan;
			delete [] tokens;
			}
			break;
		default:
			break;
	}
	return 1;
}
//***********************************************************************

#pragma pack(pop)
