/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 95 1.49.2.1 source/albert/graph2d/iimage.cpp, 2d, ioc.v400, 001006 
/*================
||
||  File:  GraphicImages.c
||
||  What:   Albert 2D Area Primitives code
||          For IMDrawable Areas
||
||      Change History
||
*/

#include <iimage.hpp>
#include <stdio.h>
#include <igrport.hpp>

#if defined(IC_WIN)
#include <gdidev.hpp>
#elif defined(IC_PM)
#include <gpidev.hpp>
#include <winemul.h>
#elif defined(IC_MOTIF)
#include <xdevice.hpp>
#endif


#ifndef IC_MOTIF
#include <gdiimdev.hpp>    // IGdiImageDevice V102
#else //IC_MOTIF
#include <ximdev.hpp>
#endif //IC_MOTIF

#include <igimage.hpp>
#include <imatrix.hpp>        // IGrafMatrix V103
#ifndef IC_MOTIF
#include <ibmpstat.hpp>
#include <ibmpdata.hpp>
#endif //IC_MOTIF

#include <grtype.hpp>
#include <grassert.hpp>
#include <idatstrm.hpp>
#include <grstrmod.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _IIMAGE_CPP_
#include <ipagetun.h>
#endif

/*================================
||      IImage::
||
*/

StreamableDefinitionsMacro(IImage, gGraphicsStreamModule);

// === Data access ===
const IGrafMatrix* IImage::matrix() const
{
	return fMatrix;
}

IGRect2D IImage::geometricBounds() const
{
	if (!fBounds) ((IImage*)this)->fBounds = new IGRect2D();
	if (timeStamp() != fBoundsTimeStamp)
	{
		*(((IImage*)this)->fBounds) = fMatrix->transformBounds(IGImage::imageBounds(*fImage));
		((IImage*)this)->fBoundsTimeStamp = timeStamp();
	}
	return *fBounds;
}

IGRect2D IImage::sourceBounds() const
{
	IGRect2D theBounds = IGImage::imageBounds(*fImage);
	theBounds.offset(sourceOrigin());
	return theBounds;
}

IImage& IImage::operator=(const IImage& other)
{
	if (&other != this)
	{
		delete fBounds;

		IMGraphic::operator=(other);
		delete fBounds;
		fBounds = (other.fBounds) ? new IGRect2D(*other.fBounds) : 0 /*NIL*/;
		fBoundsTimeStamp = other.fBoundsTimeStamp;
		// PTOM fMatrix = (other.fMatrix) ? GraphicCopy(*other.fMatrix) : new IGrafMatrix();
		*fMatrix = *other.fMatrix;

		//if(fImage){
		//	fImage->removeRef();
		//	if(fImage->useCount() == 1)
		//		delete fImage;
		//}
		//fImage = (IGImage*)(other.image());
		//fImage->addRef();
		
                // We must have ownership of the IGImage object so we create a copy of the
                // one used by the other IImage.
                if(fImage){
                   delete fImage;
                   fImage= NULL;
                }
                if (other.image()) {
                   fImage= new IGImage(*other.image());
                }

		if (fBundle != other.fBundle) {
			delete fBundle;
			if (other.fBundle)
				fBundle = GraphicCopy(*other.fBundle);
			else
				fBundle = 0 /*NIL*/;
		}
	}
	return (*this);
}

bool IImage::operator==(const IMDrawable& m) const
{
	bool result;
	if (!IMGraphic::operator==(m))
		return false;

	const IImage* other = (const IImage*) &m;

	result = (*fMatrix == *(other->fMatrix));

	if (result)
	{
		if (fImage)
			result = (other->fImage && fImage->operator== (*other->fImage));
		else
			result = (other->fImage == 0 /*NIL*/);
	}

	if (result) {
		if (fBundle && other->fBundle) {
			result = (*other->fBundle == *fBundle);
		}
		else if (fBundle || other->fBundle) {
			result = false;
		}
	}

	return result;
}

void IImage::writeToStream(IDataStream& toWhere) const
{
	IStreamOutFrame streamFrame(toWhere);
	IMGraphic::writeToStream(toWhere);
	::writeObject(fBundle, toWhere);
	bool hasMatrix = fMatrix != 0 /*NIL*/;
	(unsigned char) hasMatrix >>=(toWhere);
	// PTOM if (hasMatrix)  toWhere.writeObject(fMatrix);
	if (hasMatrix) *fMatrix >>= toWhere;
	// PTOM toWhere.writeObject(fImage);
	*fImage >>= toWhere;
}

void IImage::readFromStream(IDataStream& fromWhere)
{
	IStreamInFrame streamFrame(fromWhere);
	IMGraphic::readFromStream(fromWhere);
	delete fBundle;
	IAttributeState* theBundle;
	::readObject(theBundle, fromWhere);
	fBundle = theBundle;
	delete fMatrix;
	fMatrix = 0 /*NIL*/;
	bool hasMatrix;
	unsigned char temp;
	temp <<= (fromWhere);
	hasMatrix = bool(temp);

	if (hasMatrix) {
		fMatrix = new IGrafMatrix();
		*fMatrix <<= fromWhere;
	}
	delete fBounds;
	fBounds = 0 /*NIL*/;

	// PTOM IGImage* theImage;
	//::readObject(theImage, fromWhere);
    //delete fImage;
	//fImage = theImage;
    if (fImage){
		fImage->removeRef();
		if(fImage->useCount() == 1)
			delete fImage;
		fImage = 0;
	}
	fImage = new IGImage();
	*fImage <<= fromWhere;
	fImage->addRef();

}

void IImage::draw(IGrafPort& port) const
{
	const IGImage* theImage = image();

	const IAttributeState* state = attributeState();
	if (!state) {
		if (fMatrix->isIdentity())
			port.draw(*theImage);
		else
		ILinkedGrafPort(&port, kModelMatrix, fMatrix).draw(*theImage);
	}
	else {
		if (fMatrix->isIdentity())
			port.draw(*theImage, *attributeState());
		else
			port.draw(*theImage, *attributeState(), *fMatrix);
	}
}

void IImage::drawPartial(IGrafPort& port, const IGRect2D& areaOfInterestInSource) const
{
	const IGImage* theImage = image();

	const IAttributeState* state = attributeState();
	if (!state) {
		if (fMatrix->isIdentity())
			port.draw(*theImage, areaOfInterestInSource);
		else
		ILinkedGrafPort(&port, kModelMatrix, fMatrix).draw(*theImage, areaOfInterestInSource);
	}
	else {
		if (fMatrix->isIdentity())
			port.draw(*theImage, areaOfInterestInSource, *attributeState());
		else
			port.draw(*theImage, areaOfInterestInSource, *attributeState(), *fMatrix);
	}
}


void IImage::adoptImage(IGImage* thePixels)
{
   // We take ownership of the new IGImage object and responsibility for deleting it
   // when we are done. This is an internal-use function and should onle be called
   // with an IGImage constructed on the heap.
	if(fImage){
		//fImage->removeRef();
		//if(fImage->useCount() == 1)
			delete fImage;
	}
	fImage = thePixels;
	//fImage->addRef();
    updateTimeStamp();
}

const IGImage* IImage::image() const
{
	return fImage;
}

IGPoint2D IImage::sourceOrigin() const
{
	return IGPoint2D(0.,0.);
}

IGrafPort* IImage::grafPort()
{
	return fImage->grafPort();
}

void IImage::adoptAttributeState(IAttributeState* adoptedBundle)
{
    delete fBundle;
    fBundle = adoptedBundle;
    updateTimeStamp();
}

IAttributeState* IImage::orphanAttributeState()
{
    IAttributeState* bundle = fBundle;
    fBundle = 0 /*NIL*/;
    updateTimeStamp();
    return bundle;
}

const IAttributeState* IImage::attributeState() const
{
    return (fBundle);
}


/////integrated IGImage code

// ----- Constructors and Destructors -----

IImage::IImage() :
	IMGraphic(),
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
	updateTimeStamp();
}

IImage::IImage(const IImage& graphicImage) :
	IMGraphic(graphicImage),
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{

	if (graphicImage.fBundle)
		fBundle = GraphicCopy(*graphicImage.fBundle);
	else
	fBundle = 0 /*NIL*/;

	fBounds = (graphicImage.fBounds) ? new IGRect2D(*graphicImage.fBounds) : 0 /*NIL*/;
	fBoundsTimeStamp = graphicImage.fBoundsTimeStamp;
	*fMatrix = *graphicImage.fMatrix;
	fImage = (graphicImage.fImage) ? new IGImage(*graphicImage.fImage) : 0 /*NIL*/;
        // We own the IGImage object and do not need to reference count it further.
	//fImage->addRef();
	updateTimeStamp();
}

IImage::IImage(const IGImage& theImage) :
	IMGraphic(),
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
        // We must have ownership of the IGImage object so we create a copy of the
        // one passed to us in the constructor.
        fImage= new IGImage(theImage);
	//fImage = (IGImage*)(&theImage);
	//fImage->addRef();
	updateTimeStamp();
}


IImage::~IImage()
{
	if(fImage){
		delete fImage;
	}
	delete fBounds;
	delete fMatrix;
	delete fBundle;
}

/*

	This constructor creates a bitmap compatible with the specified device context.

	hdc : device context
	nWidth : width in pixels
	nHeight : height in pixels

*/

IImage::IImage(long width, long height, IGImage::EImageType imageType,
	ICoordinateSystem::EOrientation orientation, IColorMap* colmap) :
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
	adoptImage(new IGImage(width, height, imageType, orientation, colmap));
}

IImage::IImage(const IBitmapHandle& bitmapHandle) :
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
	adoptImage(new IGImage(bitmapHandle));
}

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
IImage::IImage(IPresSpaceHandle hdc, int nWidth, int nHeight) :
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
#if 0
	printf( " IImage::IImage(IPresSpaceHandle hdc, int nWidth, int nHeight ) is obsoleted\n" );
	printf( " Please use IGImage::captureImage instead\n" );
#endif
#ifndef IC_MOTIF
	adoptImage(new IGImage(IGImage::captureImage(hdc, IGRect2D(0,0,nWidth,nHeight))));
#endif //IC_MOTIF
}

IImage::IImage(const IString& imageFilename, bool, EDitherType ditherStyle) :
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
#if 0
	printf( " IImage::IImage( const IString&, bool, EDitherType) is obsoleted\n" );
	printf( " Please use IGImagePixelAccessor::loadFromFile instead\n" );
#endif
	
	adoptImage(new IGImage(IGImagePixelAccessor::loadFromFile(imageFilename,
			IGImagePixelAccessor::kUNKNOWN,
			(IGImagePixelAccessor::EDitherType)ditherStyle)));
}

IImage::IImage(
	const IString& imageFilename,
	EImageFormat imageFormat,
	bool,
	EDitherType ditherStyle) :

	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
#if 0
	printf( " IImage::IImage( const IString&, EImageFormat, bool, EDitherType) is obsoleted\n" );
	printf( " Please use IGImagePixelAccessor::loadFromFile instead\n" );
#endif
	
	adoptImage(new IGImage(IGImagePixelAccessor::loadFromFile(imageFilename,
				(IGImagePixelAccessor::EImageFormat)imageFormat,
				(IGImagePixelAccessor::EDitherType)ditherStyle)));
}

IImage& IImage::loadFromFile(const IString& filename, EImageFormat fileType, bool, EDitherType ditherStyle)
{
#if 0
	printf( " IImage::loadFromFile( const IString&, EImageFormat, bool, EDitherType) is obsoleted\n" );
	printf( " Please use IGImagePixelAccessor::loadFromFile instead\n" );
	
#endif
	adoptImage( new IGImage(IGImagePixelAccessor::loadFromFile(
			filename, (IGImagePixelAccessor::EImageFormat)fileType,
			(IGImagePixelAccessor::EDitherType)ditherStyle) ));
	return *this;
}

IImage& IImage::writeToFile(const IString& imageFilename, EImageFormat imageFormat)
{
#if 0
	printf( " IImage::writeToFile( const IString&, EImageFormat) is obsoleted\n" );
	printf( " Please use IGImagePixelAccessor::writeToFile instead\n" );
#endif
	
	IGImagePixelAccessor::writeToFile( imageFilename,
			(IGImagePixelAccessor::EImageFormat)imageFormat, fImage->handle());
	return *this;
}

#endif //(IC_OBSOLETE <= IC_OBSOLETE_3)

IImage::IImage(const IImage& bitmap, const IGRect2D& rectangle) :
	fBounds(0 /*NIL*/),
	fMatrix(new IGrafMatrix()),
	fImage(0 /*NIL*/),
	fBundle(0 /*NIL*/),
   fSeed(0),
   fBoundsTimeStamp(0)
{
	adoptImage(new IGImage(*(bitmap.fImage), rectangle));
}


void IImage::transformBy(const IGrafMatrix& matrix)
{
	fMatrix->concatWith(matrix);
	updateTimeStamp();
}


// V104 Rotations and Reflections

IImage& IImage::reflectHorizontally()
{
	IGrafMatrix reflectHor(IGPoint2D(-1., 1.), geometricBounds().center());
	transformBy(reflectHor);
	// time stamp is updated in transformBy
	return *this;
}

IImage& IImage::reflectVertically()
{
	IGrafMatrix reflectVer(IGPoint2D(1., -1.), geometricBounds().center());
	transformBy(reflectVer);
	return *this;
}

IImage& IImage::rotateBy180()
{
	IGrafMatrix rotate180(GDegrees(180.), geometricBounds().center());
	transformBy(rotate180);
	return *this;
}

IImage& IImage::rotateBy270()
{
	IGrafMatrix rotate270(GDegrees(270.), geometricBounds().center());
	transformBy(rotate270);
	return *this;
}

IImage& IImage::rotateBy90()
{
	IGrafMatrix rotate90(GDegrees(90.), geometricBounds().center());
	transformBy(rotate90);
	return *this;
}

IImage& IImage::transposeXForY()
{
	IGPoint2D centre = IGImage::imageBounds(*fImage).topLeft();
	IGrafMatrix transposeXY(GDegrees(90.), centre);
	transposeXY.scaleBy(IGPoint2D(-1., 1.), centre);
	fMatrix->preConcatWith(transposeXY);
	updateTimeStamp();
	return *this;
}

IImage& IImage::sizeTo(const IGPoint2D& newSize)      // V106
{
	fImage->sizeTo(newSize);
	updateTimeStamp();
	return *this;
}

IImage& IImage::moveTo(const IGPoint2D& delta)
{
	fMatrix->setElement(IGrafMatrix::kTranslateX, delta.fX );
    fMatrix->setElement(IGrafMatrix::kTranslateY, delta.fY );
	updateTimeStamp();
	return *this;
}

bool IImage::hasTransparentColor() const       // V106
{
	return(fImage->transparencyFlag());
}

IImage& IImage::resetTransparentColor()   // V106
{
//	fMaskNeeded = !fMaskNeeded;
	updateTimeStamp();
	return *this;
}

IImage& IImage::setTransparentColor(const IBaseColor& aColor) // V106
{
	fImage->setTransparencyColor(aColor);
	fImage->setTransparencyFlag(true);

	updateTimeStamp();
	return *this;
}


IBaseColor IImage::transparentColor() const       // V106
{
	 return(fImage->transparencyColor());
}
