/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 77 1.58.2.4 source/albert/graph2d/igimage.cpp, 2d, ioc.v400, 001006 
/*================
||
||  File:   IGImage.cpp
||
||  What:   Basic Albert 2D Primitive for images
||
||      Classes defined in this file:
||
||      IGImage
||
||      Change History:
||
*/

#include <igimage.hpp>
#include <istring.hpp>

#ifdef IC_MOTIF
#include <ximdev.hpp>    // IXImageDevice
#else
#include <gdiimdev.hpp>    // IGdiImageDevice V103
#ifdef IC_PM
#include <winemul.h>
#endif
#endif

#include <imagedc.hpp>
#include <igrport.hpp>
#include <imatrix.hpp>
#include <grassert.hpp>
#include <limits.h>
#include <inumeric.hpp>
#include <idatstrm.hpp>
#include <stdio.h>
#include <pixbuf.hpp>
#include <imgconv.hpp>
#include <ibmpdata.hpp>
#include <ibmpstat.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _IGIMAGE_CPP_
#include <ipagetun.h>
#endif

IGImage::IGImage() :
	fGrafPort(0 /*NIL*/),
	fBmpData(0),
	fColorMap(&(IColorMap::defaultColorMap())),
	fOrientation(ICoordinateSystem::kOriginUpperLeft)
{
    IFUNCTRACE_DEVELOP();
}

IGImage::~IGImage()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData){
		fBmpData->removeRef();
		if(fBmpData->useCount() == 0){
			//first remove it from the set
			IBitmapStaticPtr::bitmapSet().remove(fBmpData);
			//now delete the object
			delete fBmpData;
			fBmpData = NULL;
		}
	}
	//referencing count dosent apply to fGrafPort
	//we keep the bitmapData arround but fGRafPort is created fro each IGImage
	if(fGrafPort){
		delete fGrafPort;
		fGrafPort = 0;
	}
}

void IGImage::setTransparencyFlag(bool flag)
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
		fBmpData->pixelBuffer()->setTransparencyFlag(flag);
}

bool IGImage::transparencyFlag() const
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
		return(fBmpData->pixelBuffer()->transparencyFlag());
	else
		return(false);
}

void IGImage::setTransparencyColor(const IBaseColor& clr)
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
		fBmpData->pixelBuffer()->setTransparencyColor(clr);
}

IBaseColor IGImage::transparencyColor() const
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
		return(fBmpData->pixelBuffer()->transparencyColor());
	else
		return NULL;
}

IGRect2D IGImage::rasterBounds()const
{
    IFUNCTRACE_DEVELOP();
	if (imageHandle())
	{
		return IGRect2D(0, 0, fBmpData->pixelBuffer()->fWidth, fBmpData->pixelBuffer()->fHeight );
	}
	else
		return IGRect2D::zeroRect();
}

/*================================
||      Utilities
||
*/
IRGBAColorArray IGImage::imageColorTable() const
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData){
		return(*(fBmpData->pixelBuffer()->colorTable()));
	}
	return NULL;
}

IGRect2D IGImage::imageBounds (const IGImage& theImage)
{
    IFUNCTRACE_DEVELOP();
	IGRect2D rasterBounds = theImage.rasterBounds();

	return rasterBounds;
}


bool IGImage::operator!=(const IGImage& obj) const
{
    IFUNCTRACE_DEVELOP();
	return !operator==(obj);
}

bool IGImage::operator==(const IGImage& src) const
{
    IFUNCTRACE_DEVELOP();
	if (this != &src) {
		
		if (this->fOrientation != src.fOrientation)
			return false;

		if((!fBmpData) || (!src.fBmpData))
			return false;

		if (rasterBounds() != src.rasterBounds())
			return false;

		IPixelBuffer *srcPixBuf = src.fBmpData->pixelBuffer();
		IPixelBuffer *dstPixBuf = fBmpData->pixelBuffer();

		if (dstPixBuf != srcPixBuf)
		{
			if(dstPixBuf->bitsPerPixel() == srcPixBuf->bitsPerPixel())
			{
				if (*dstPixBuf != *srcPixBuf)
					return false;
			} else {

				long width = (long)dstPixBuf->fWidth	;
				long height = (long)dstPixBuf->fHeight;	
				IBaseColor thePixel, thatPixel;
				for (long j = 0; j < height; j++) {
					for (long i = 0; i < width; i++) {
						IGPoint2D pos(i, j);
						dstPixBuf->pixel(pos, thePixel);
						srcPixBuf->pixel(pos, thatPixel);

						if (thePixel != thatPixel)
							return false;
					}
				}
			}
		}
	}
	return true;
}

//
//
///
//
//      extract of igbitmap
//
//
//

IRGBAColorArray* IGImage::createColorTable()const
{
    IFUNCTRACE_DEVELOP();
	return (IRGBAColorArray*)NULL;
#if 0//SAS
#ifdef IC_MOTIF
	return (IRGBAColorArray*)NULL;
#else
	if (!fBitmap)
		return 0 /*NIL*/;       // no image here...

	HBITMAP theBits = fBitmap;

	struct
	{
		BITMAPINFOHEADER bmp;
		RGBQUAD argb2Color[256];
	} bmdetails;

	BITMAP bm;
	if (!GetObject(theBits,sizeof(bm),(LPSTR)&bm))
		GrafDeviceException("GetObject");

	if (bm.bmBitsPixel > 8)
	{
		return 0 /*NIL*/;
	}
	else
	{
		IRGBColorArray* result = new IRGBColorArray(256);

		GetDIBits(
			fBitmapHDC,
			theBits,
			0,
			UINT(bm.bmHeight),
			0,
			(BITMAPINFO*)&bmdetails,
			(DWORD)DIB_RGB_COLORS);

		char* bmhdrPtr = (char*)&bmdetails;

		// at this point we may have retrieved a new or old style info header so
		// care must be taken in pointing into this data
		bool isQuad = (bmdetails.bmp.biSize == sizeof(BITMAPCOREHEADER));

		RGBQUAD *pRgb = (RGBQUAD FAR *)(bmhdrPtr + bmdetails.bmp.biSize);
		int i;
		for (i = 0; i < (1<<bm.bmBitsPixel); i++)
		{
			IBaseColor temp(pRgb->rgbRed, pRgb->rgbGreen, pRgb->rgbBlue);
			if (isQuad)
				pRgb++;
			else
			{       LPSTR tmpPtr = ((LPSTR)pRgb);
				tmpPtr+= sizeof(RGBTRIPLE);
				pRgb = (RGBQUAD FAR *)tmpPtr;
			}
			result->setColor(i, temp);
		}
		return result;
	}
#endif //IC_MOTIF
#endif //SAS
}


IGImage::IGImage(
	const IGImage& origImage,
	const IBaseColor& transparentColor) :

	fGrafPort(0 /*NIL*/),
	fBmpData(0),
	fColorMap(&(IColorMap::defaultColorMap())),
	fOrientation(ICoordinateSystem::kOriginUpperLeft)
{

    IFUNCTRACE_DEVELOP();
	IPixelBuffer *tmpPix = (origImage.fBmpData->pixelBuffer())->clonePixelBuffer();

	tmpPix->setTransparencyColor(transparentColor);
	tmpPix->setTransparencyFlag(true);
	fBmpData = createBmpDataFromPixBuf(tmpPix);
}


const IGImage& IGImage::operator=(const IGImage& src)
{
    IFUNCTRACE_DEVELOP();
	if (this != &src)
	{
		// out goes the old ...
		delete fGrafPort;
		fGrafPort = 0 /*NIL*/;	// reset the port, new port will be created
		adoptBitmapData(src.fBmpData);
		this->fOrientation = src.fOrientation;
		this->fColorMap = src.fColorMap;
	}
	return (*this);
}

IDataStream& IGImage::operator>>=(IDataStream& toWhere) const
{
    IFUNCTRACE_DEVELOP();
	writeToStream(toWhere);
	return toWhere;
}

IDataStream& IGImage::operator<<=(IDataStream& fromWhere)
{
    IFUNCTRACE_DEVELOP();
	this->readFromStream(fromWhere);
	return fromWhere;
}

void IGImage::writeToStream(IDataStream& toWhere) const
{
    IFUNCTRACE_DEVELOP();
	fBmpData->pixelBuffer()->bitsPerPixel() >>= toWhere;
	long temp;
	temp = fOrientation;
	temp >>= toWhere;
	*(fBmpData->pixelBuffer()) >>= toWhere;
}

void IGImage::readFromStream(IDataStream& fromWhere)
{
    IFUNCTRACE_DEVELOP();
	unsigned char depth;
	IPixelBuffer	*pixbuf;
	long temp;

	depth <<= fromWhere;
	temp <<= fromWhere;
	fOrientation = (ICoordinateSystem::EOrientation)temp;
	switch(depth)
	{
	case 1:
		pixbuf = new IMonochromePixelBuffer();
		break;
	case 4:
		pixbuf = new ICLUT4PixelBuffer();
		break;
	case 8:
		pixbuf = new ICLUT8PixelBuffer();
		break;
	case 16:
		pixbuf = new IR5G6B5PixelBuffer();
		break;
	case 24:
		pixbuf = new IR8G8B8PixelBuffer();
		break;
	case 32:
		pixbuf = new IR8G8B8A8PixelBuffer();
		break;
	}
	*pixbuf <<= fromWhere;

	fBmpData = createBmpDataFromPixBuf(pixbuf);

}

IGImage::IGImage(
	unsigned long width, unsigned long height, EImageType imageType,
	ICoordinateSystem::EOrientation orientation, IColorMap* colmap) :
		fGrafPort(0 /*NIL*/),
		fBmpData(0),
		fColorMap(colmap),
		fOrientation(orientation)
{
    IFUNCTRACE_DEVELOP();
	if((width == 0) || (height == 0))
		GrafDeviceException("IGImage:Invalid Dimensions specified");

    IPixelBuffer *pixbuf;

	switch(imageType)
	{
	case kMonochrome1Bit:
		pixbuf = new IMonochromePixelBuffer(width, height);
		break;
	case k16Color4Bit:
		pixbuf = new ICLUT4PixelBuffer(width, height);
		break;
	case k256Color8Bit:
		pixbuf = new ICLUT8PixelBuffer(width, height);
		break;
	case kTrueColor24Bit:
		pixbuf = new IR8G8B8PixelBuffer(width, height);
		break;
	}
	fBmpData = createBmpDataFromPixBuf(pixbuf);
}

IGImage::IGImage(
	const IBitmapHandle& bitmapHandle) :
	fGrafPort(0 /*NIL*/),
	fBmpData(0),
	fColorMap(&(IColorMap::defaultColorMap())),
	fOrientation(ICoordinateSystem::kOriginUpperLeft)
{
    IFUNCTRACE_DEVELOP();
	if(bitmapHandle == 0)
		GrafDeviceException("IGImage:Invalid IBitmapHandle specified");

	IBitmapData* node = IBitmapStaticPtr::bitmapSet().find(bitmapHandle);
	//if node is not found in bitmap set try  in pointer set
	if (!node)
   {
      IPointerHandle
        ptrHandle( (IPointerHandle::Value)
                   (IBitmapHandle::Value) bitmapHandle );
   }

	if (node)
	{
		adoptBitmapData(node);
	} else { //not in the collection
		unsigned int depth;
#ifdef IC_MOTIF
		int x, y;
		unsigned int wid, ht, border;
		Window root;

		if (!XGetGeometry(IXDisplay::display(), bitmapHandle, &root,
					&x, &y, &wid, &ht, &border, &depth))
			GrafDeviceException("XGetGeometry");

		XImage *hand = XGetImage(IXDisplay::display(), bitmapHandle,
			0, 0, wid, ht, AllPlanes, ZPixmap);
		depth = hand->bits_per_pixel;
#else
		BITMAP bm;
		if (!GetObject((HBITMAP)bitmapHandle, sizeof(bm), (LPSTR)&bm))
			GrafDeviceException("GetObject");

		depth = (int)bm.bmBitsPixel;
		HBITMAP hand = bitmapHandle;
#endif
		IPixelBuffer *pixbuf=NULL;

		switch(depth)
		{
		case 1:
			pixbuf = new IMonochromePixelBuffer(hand);
			break;
		case 4:
			pixbuf = new ICLUT4PixelBuffer(hand);
			break;
		case 8:
			pixbuf = new ICLUT8PixelBuffer(hand);
			break;
		case 16:
			pixbuf = new IR5G6B5PixelBuffer(hand);
			break;
		case 24:
			pixbuf = new IR8G8B8PixelBuffer(hand);
			break;
		case 32:
			pixbuf = new IR8G8B8A8PixelBuffer(hand);
			break;
		}
		fBmpData = createBmpDataFromPixBuf(pixbuf);
#ifdef IC_MOTIF
		XDestroyImage(hand);
#endif
	}
}

IGImage::IGImage(const IGImage& src) :
	fGrafPort(0 /*NIL*/),
	fBmpData(0),
	fColorMap(src.fColorMap),
	fOrientation(src.fOrientation)
{
    IFUNCTRACE_DEVELOP();
    // adoptBitmapData(src.fBmpData);
    // Don't point to src.fBmpData; create your own copy so that
    // you can modify it without affecting others pointing to src.fBmpData
    if(src.fBmpData){
      IPixelBuffer *tmpPix = src.fBmpData->pixelBuffer()->clonePixelBuffer();
      fBmpData = createBmpDataFromPixBuf(tmpPix);
    }
}

IGrafPort* IGImage::grafPort()
{
    IFUNCTRACE_DEVELOP();
	if (!fGrafPort) {
		IPresSpaceHandle myPress = fBmpData->pixelBuffer()->fBitmapHDC;
		IGrafDevice *imgDev =
#ifndef IC_MOTIF
			new IGdiImageDevice(this, myPress);
#else
			new IXImageDevice(this, myPress);
#endif
		//if the depth of the image is >=16, no palette is supported
		fGrafPort = new IBaseRootGrafPort( imgDev,
					((fBmpData->pixelBuffer()->fBitsPerPixel < 16) ? fColorMap : 0),
					fOrientation);
	}
	return fGrafPort;
}

IGImageDeviceContext* IGImage::createDeviceContext() const
{
    IFUNCTRACE_DEVELOP();
	return new IGImageDeviceContext(*this);
}

IPresSpaceHandle IGImage::deviceContext() const
{
	// printf("IGImage::deviceContext(): this is going to be obsolete, use IGImage::createDeviceContext() instead!!!\n");
    IFUNCTRACE_DEVELOP();
	return fBmpData->pixelBuffer()->fBitmapHDC;
}


IGImage& IGImage::reflectHorizontally()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();
		if(fBmpData->useCount() != 1)
		{
			fPixBuf = fPixBuf->clonePixelBuffer();
         		fBmpData->removeRef();
			fBmpData = createBmpDataFromPixBuf(fPixBuf);
		}
		fPixBuf->reflectHorizontaly();
	}
	return *this;
}

IGImage& IGImage::reflectVertically()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();
		if(fBmpData->useCount() != 1)
		{
			fPixBuf = fPixBuf->clonePixelBuffer();
        		fBmpData->removeRef();
			fBmpData = createBmpDataFromPixBuf(fPixBuf);
		}
		fPixBuf->reflectVerticaly();
	}
	return *this;
}

IGImage& IGImage::rotateBy180()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();
		if(fBmpData->useCount() != 1)
		{
			fPixBuf = fPixBuf->clonePixelBuffer();
        		fBmpData->removeRef();
			fBmpData = createBmpDataFromPixBuf(fPixBuf);
		}
		fPixBuf->rotate180();
	}
	return *this;
}

// The following operations swap the image resolution.

IGImage& IGImage::rotateBy270()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();
		if(fBmpData->useCount() != 1)
		{
			fPixBuf = fPixBuf->clonePixelBuffer();
        		fBmpData->removeRef();
			fBmpData = createBmpDataFromPixBuf(fPixBuf);
		}
                // The handle will change on the rotate270 call so we have to prepare
                // to remove the old fBmpData and add the new fBmpData to the bitmapSet
                // because it is indexed by the handle. -DCS
   	        IBitmapStaticPtr::bitmapSet().remove(fBmpData);
		
		fPixBuf->rotate270();
		
                fBmpData->setHandle((IBitmapHandle::Value)fPixBuf->handle());
                fBmpData->setSize(ISize(fPixBuf->fWidth, fPixBuf->fHeight));
                IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
	        IBitmapStaticPtr::bitmapSet().add(fBmpData);
	}
	return *this;
}

IGImage& IGImage::rotateBy90()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();
		if(fBmpData->useCount() != 1)
		{
			fPixBuf = fPixBuf->clonePixelBuffer();
         		fBmpData->removeRef();
			fBmpData = createBmpDataFromPixBuf(fPixBuf);
		}
                // The handle will change on the rotate90 call so we have to prepare
                // to remove the old fBmpData and add the new fBmpData to the bitmapSet
                // because it is indexed by the handle. -DCS
   	        IBitmapStaticPtr::bitmapSet().remove(fBmpData);
		
		fPixBuf->rotate270();
		
                fBmpData->setHandle((IBitmapHandle::Value)fPixBuf->handle());
                fBmpData->setSize(ISize(fPixBuf->fWidth, fPixBuf->fHeight));
                IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
	        IBitmapStaticPtr::bitmapSet().add(fBmpData);
	}
	return *this;
}

IGImage& IGImage::transposeXForY()
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();
		if(fBmpData->useCount() != 1)
		{
			fPixBuf = fPixBuf->clonePixelBuffer();
         		fBmpData->removeRef();
			fBmpData = createBmpDataFromPixBuf(fPixBuf);
		}
                // The handle will change on the transposeXForY call so we have to prepare
                // to remove the old fBmpData and add the new fBmpData to the bitmapSet
                // because it is indexed by the handle.
   	        IBitmapStaticPtr::bitmapSet().remove(fBmpData);
		
		fPixBuf->rotate270();

                fBmpData->setHandle((IBitmapHandle::Value)fPixBuf->handle());
                fBmpData->setSize(ISize(fPixBuf->fWidth, fPixBuf->fHeight));
                IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
	        IBitmapStaticPtr::bitmapSet().add(fBmpData);
	}
	return *this;
}


/*------------------------------------------------------------------------------
| IGBitmap::copyBitmap                                                         |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IBitmapHandle IGImage::copyBitmap() const
{
    IFUNCTRACE_DEVELOP();
	unsigned long hand = 0;
	if(fBmpData)
	{
		hand = fBmpData->pixelBuffer()->createPatternImage();
	}
#if defined(IC_WIN)
   return IBitmapHandle( (IBitmapHandle::Value) hand );
#else
	return IBitmapHandle(hand);
#endif
}

IBitmapHandle IGImage::maskImageHandle() const
{
	unsigned long hand = 0;
	if(fBmpData)
	{
		hand = fBmpData->pixelBuffer()->maskImage();
	}
#if defined(IC_WIN)
   return IBitmapHandle( (IBitmapHandle::Value) hand );
#else
	return IBitmapHandle(hand);
#endif
}

IBitmapHandle IGImage::handle() const
{
    IFUNCTRACE_DEVELOP();
	unsigned long hand = 0;
	if(fBmpData)
	{
		hand = fBmpData->pixelBuffer()->handle();
	}
#if defined(IC_WIN)
   return IBitmapHandle( (IBitmapHandle::Value) hand );
#else
	return IBitmapHandle(hand);
#endif
}

unsigned long IGImage::imageHandle() const
{
    IFUNCTRACE_DEVELOP();
	unsigned long hand = 0;
	if(fBmpData)
	{
		hand = (unsigned long)fBmpData->pixelBuffer()->imageHandle();
	}
	return hand;
}



// to capture the bitmap image from a given HDC, such as a screen device...
IBitmapHandle IGImage::captureImage(IPresSpaceHandle& srcHDC, const IGRect2D& imageArea)
{
    IFUNCTRACE_DEVELOP();
	unsigned int depth;
#ifdef IC_MOTIF
	IXDC* xDC = srcHDC;

	// get information from selection
	int left = (int)imageArea.fLeft;
	int top = (int)imageArea.fTop;
	int width = imageArea.width();
	int height = imageArea.height();

	int x, y;
	unsigned int wid, ht, border;
	Window root;

	if (!XGetGeometry(xDC->xDisplay, xDC->xDrawable,&root,
				&x, &y, &wid, &ht, &border, &depth))
		GrafDeviceException("XGetGeometry");

        Display *dpy = xDC->xDisplay;
        //get the copy of image
        XImage *hand = XGetImage( dpy, xDC->xDrawable,
                left, top, width, height, AllPlanes, ZPixmap );

#else

	HBITMAP hand;

	HDC hdcDisplay = GetWindowDC(HWND_DESKTOP);
	if (hdcDisplay == 0)
		GrafDeviceException("GetWindowDC");

	HDC dstHDC = CreateCompatibleDC(hdcDisplay);
	if (!dstHDC)
		GrafDeviceException("CreateCompatibleDC");

	if (!ReleaseDC(HWND_DESKTOP, hdcDisplay))
		GrafDeviceException("ReleaseDC");

	int width  = (int)imageArea.width();
	int height = (int)imageArea.height();
	hand = CreateCompatibleBitmap(srcHDC, width, height);

	BITMAP bm;
	if (!GetObject((HBITMAP)hand, sizeof(bm),(LPSTR)&bm))
		GrafDeviceException("GetObject");
	depth = (unsigned int)bm.bmBitsPixel;

	if (!hand)
		GrafDeviceException("CreateCompatibleBitmap, Cannot capture image per device!");

	int left  = (int)imageArea.fLeft;
	int top = (int)imageArea.fTop;

	HBITMAP hbmOld;
   hbmOld = (HBITMAP) SelectObject( dstHDC, hand );

	if (!BitBlt(dstHDC, 0, 0, width, height, srcHDC, left, top, SRCCOPY))
		GrafDeviceException("BitBlt, Cannot capture image per device!");

	SelectObject(dstHDC, hbmOld);

	if (!DeleteDC(dstHDC))
		GrafDeviceException("DeleteDC");

#endif //IC_MOTIF

	IGImage igimage;

	IPixelBuffer *pixbuf=NULL;

	switch(depth)
	{
	case 1:
		pixbuf = new IMonochromePixelBuffer(hand);
		break;
	case 4:
		pixbuf = new ICLUT4PixelBuffer(hand);
		break;
	case 8:
		pixbuf = new ICLUT8PixelBuffer(hand);
		break;
	case 16:
		pixbuf = new IR5G6B5PixelBuffer(hand);
		break;
	case 24:
		pixbuf = new IR8G8B8PixelBuffer(hand);
		break;
	case 32:
		pixbuf = new IR8G8B8A8PixelBuffer(hand);
		break;
	}

	igimage.fBmpData = igimage.createBmpDataFromPixBuf(pixbuf);


#if defined(IC_MOTIF)
	XDestroyImage(hand);
#elif defined(IC_WIN)
	DeleteObject(hand);
#elif defined(IC_PM)
	GpiDeleteBitmap(hand);
#endif

	return(igimage.handle());
}

IGImage& IGImage::sizeTo(const IGPoint2D& newSize)
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
	{
		int width = (int)newSize.fX;
		if (width < 0)  width = -width;
		int height = (int)newSize.fY;
		if (height < 0)  height = -height;

		IPixelBuffer *fPixBuf = fBmpData->pixelBuffer();

		if(fPixBuf)
			if((width != fPixBuf->fWidth) || (height != fPixBuf->fHeight))
			{
				bool newPB=false;
				if(fBmpData->useCount() != 1)
				{
				   fPixBuf = fPixBuf->clonePixelBuffer();
     				   fBmpData->removeRef();
                                   newPB=true;
				}
                                else
                                {
                                   // The handle will change on the resize call so we have to prepare
                                   // to remove the old fBmpData and add the new fBmpData to the bitmapSet
                                   // because it is indexed by the handle.
         	                   IBitmapStaticPtr::bitmapSet().remove(fBmpData);
                                }
				fPixBuf->resize(width, height);
                                if(newPB)
			           fBmpData = createBmpDataFromPixBuf(fPixBuf);
                                else
                                {
                                   fBmpData->setHandle((IBitmapHandle::Value)fPixBuf->handle());
                                   fBmpData->setSize(ISize(fPixBuf->fWidth, fPixBuf->fHeight));

                                   // Add the handle to the bitmapSet. We were either the only user and
                                   // temporarily removed the old handle or this is a new clone and the
                                   // handle was never added.
                                   IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
         	                   IBitmapStaticPtr::bitmapSet().add(fBmpData);
                                }

			}
	}

	return *this;
}

#ifdef NO_IGIMAGEPIXELACCESSOR
// these are not efficient and adding load to the ill-designed class IGImage/IImage...

// I added the IGImagePixelAccessor class below, so that if the functionality of
// read/write at the pixel level of image cannot be implemented on a platform, we can
// always take out that class.

bool IGImage::pixel(const IGPoint2D& position, IBaseColor& valueOfThePixelReturned) const
{
    IFUNCTRACE_DEVELOP();
	ParameterAssert(
		rasterBounds().contains(position),
		"Position of target pixel is outside the target image.");

	if(fBmpData)
		fBmpData->pixelBuffer()->pixel(position, valueOfThePixelReturned);

	return true;

}

void IGImage::setPixel(const IGPoint2D& position, const IBaseColor& valueOfThePixel)
{
    IFUNCTRACE_DEVELOP();
	ParameterAssert(
		rasterBounds().contains(position),
		"Position of target pixel lie is the target image.");

	if(fBmpData)
		fBmpData->pixelBuffer()->setPixel(position, valueOfThePixel);
}

#endif // NO_IGIMAGEPIXELACCESSOR

 //--------------------------------------------
//--------------------------------------------


IGImage::IGImage(
	const IGImage& srcImage,
	const IGRect2D& areaToCopy) :

	fGrafPort(0 /*NIL*/),
	fBmpData(0),
	fColorMap(&(IColorMap::defaultColorMap())),
	fOrientation(ICoordinateSystem::kOriginUpperLeft)
{
    IFUNCTRACE_DEVELOP();
	if(areaToCopy.isEmpty())
		GrafDeviceException("IGImage:IGRect2D is Empty");

	IPixelBuffer *pixbuf, *tmpPix = srcImage.fBmpData->pixelBuffer();
	if(tmpPix){
		switch(tmpPix->bitsPerPixel())
		{
		case 1:
			pixbuf = new IMonochromePixelBuffer(*((IMonochromePixelBuffer*)tmpPix),
						areaToCopy);
			break;
		case 4:
			pixbuf = new ICLUT4PixelBuffer(*((ICLUT4PixelBuffer*)tmpPix),
						areaToCopy);
			break;
		case 8:
			pixbuf = new ICLUT8PixelBuffer(*((ICLUT8PixelBuffer*)tmpPix),
						areaToCopy);
			break;
		case 16:
			pixbuf = new IR5G6B5PixelBuffer(*((IR5G6B5PixelBuffer*)tmpPix),
						areaToCopy);
			break;
		case 24:
			pixbuf = new IR8G8B8PixelBuffer(*((IR8G8B8PixelBuffer*)tmpPix),
						areaToCopy);
			break;
		case 32:
			pixbuf = new IR8G8B8A8PixelBuffer(*((IR8G8B8A8PixelBuffer*)tmpPix),
						areaToCopy);
			break;
		}
		fBmpData = createBmpDataFromPixBuf(pixbuf);
	}
}

void IGImage::adoptBitmapData(IBitmapData *src)
{
    IFUNCTRACE_DEVELOP();
	//delete the old and assign a new one
	if(fBmpData)
	{
	   fBmpData->removeRef();
           if (fBmpData->useCount() == 0) {
  		//first remove it from the set
		IBitmapStaticPtr::bitmapSet().remove(fBmpData);
		//now delete the object
		delete fBmpData;
           }
	}
	fBmpData = src;
        if (fBmpData)
           fBmpData->addRef();
}

unsigned long IGImage::addRef()
{
	return(fBmpData->addRef());
}

unsigned long IGImage::removeRef()
{
	return(fBmpData->removeRef());
}

unsigned long IGImage::useCount() const
{
	return(fBmpData->useCount());
}

void IGImage::setImageSyncFlag(bool flag)
{
    IFUNCTRACE_DEVELOP();
	if(fBmpData)
		fBmpData->pixelBuffer()->bSyncImage = flag;
}

IBitmapData* IGImage::createBmpDataFromPixBuf(IPixelBuffer* pixBuf)
{
    IFUNCTRACE_DEVELOP();

    IBitmapData *pBmpData;
#if defined(IC_PMWIN)
	pBmpData = new IBitmapData (0,  (IBitmapHandle::Value)pixBuf->handle(), NULL,
			 ISize(pixBuf->fWidth, pixBuf->fHeight), pixBuf);
#elif defined(IC_MOTIF)
	pBmpData = new IBitmapData (0, ISize(pixBuf->fWidth, pixBuf->fHeight),
				pixBuf->imageHandle(), pixBuf->handle(), pixBuf);
#endif
	pBmpData->addRef();
        IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
	IBitmapStaticPtr::bitmapSet().add(pBmpData);

	return(pBmpData);
}


//--------------------------------------------
// IGImagePixelAccessor class

// disabled...
IGImagePixelAccessor::IGImagePixelAccessor()
{}

// disabled...
IGImagePixelAccessor::IGImagePixelAccessor(const IGImagePixelAccessor&)
{}

IGImagePixelAccessor& IGImagePixelAccessor::operator=(const IGImagePixelAccessor& src) {
	return *this;
}

IGImagePixelAccessor::IGImagePixelAccessor(const IGImage& targetImage) :
	fDummy(0),
	fImage((IGImage*)(&targetImage))
{
	fImage->fBmpData->addRef();
	
//If the bitmap is once selected, it cannot be selected again in OS2.
//#ifndef IC_MOTIF
//	HBITMAP bitmap = (HBITMAP) fImage->handle();
//	fDummy = IBitmapHandle(SelectObject((HDC)fImage->deviceContext(), bitmap));
//#endif
}

IGImagePixelAccessor::~IGImagePixelAccessor()
{
    IFUNCTRACE_DEVELOP();
	if(fImage->fBmpData){
		fImage->fBmpData->removeRef();
	}
//#ifndef IC_MOTIF
//	if(fDummy != 0)
//	SelectObject((HDC)fImage->deviceContext(), fDummy);
//#endif
}

bool
IGImagePixelAccessor::pixel(const IGPoint2D& position,
                            IBaseColor& valueOfThePixelReturned) const
{
    IFUNCTRACE_DEVELOP();
    fImage->fBmpData->pixelBuffer()->pixel(position, valueOfThePixelReturned);
	return true;
}

void IGImagePixelAccessor::setPixel(
	const IGPoint2D& position,
	const IBaseColor& valueOfThePixel)
{
    IFUNCTRACE_DEVELOP();
	fImage->fBmpData->pixelBuffer()->setPixel(position, valueOfThePixel);
}

bool IGImagePixelAccessor::pixelRow(
	const IGPoint2D& position,
	IRGBAColorArray& valueOfThePixelsReturned) const
{
    IFUNCTRACE_DEVELOP();
	bool clipped = false;
	fImage->fBmpData->pixelBuffer()->pixelRow(position, valueOfThePixelsReturned);
	return clipped;
}

void IGImagePixelAccessor::setPixelRow(
	const IGPoint2D& position,
	const IRGBAColorArray& valueOfThePixels)
{
    IFUNCTRACE_DEVELOP();
	fImage->fBmpData->pixelBuffer()->setPixelRow(position,valueOfThePixels);
}

IBitmapHandle IGImagePixelAccessor::loadFromFile(const IString& filename,
		EImageFormat fileType, EDitherType ditherStyle)
{
    IFUNCTRACE_DEVELOP();
	IGImage *image = IImageConverter::readImage((char *)filename);	
        IBitmapHandle result(image->handle());
        delete image;
        return result;
}

IGImage* IGImagePixelAccessor::loadImageFromFile(const IString& filename)
{
    IFUNCTRACE_DEVELOP();
	IGImage *image = IImageConverter::readImage((char *)filename);	
	return image;
}

void IGImagePixelAccessor::writeToFile(const IString& imageFilename,
		EImageFormat imageFormat, IBitmapHandle bmp)
{
    IFUNCTRACE_DEVELOP();
	IGImage image(bmp);
	IImageConverter::writeImage((char *)imageFilename, &image, imageFormat);
}

void IGImagePixelAccessor::writeImageToFile(const IString& imageFilename,
                                            EImageFormat imageFormat, IGImage *image)
{
    IFUNCTRACE_DEVELOP();
	IImageConverter::writeImage((char *)imageFilename, image, imageFormat);
}


