// Revision: 27 1.5 source/albert/graph2d/igbidi.cpp, 2d, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: igbidi.cpp                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions used for        *
*   bidirectional support.                                                     *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

#include <ibase.hpp>

extern "C" {
#ifdef IC_WIN
  #include <windows.h>
#endif
#ifdef IC_PM
  #define INCL_DOSERRORS     // NO_ERROR, ERROR_CPLIST_TOO_SMALL
  #define INCL_DOSNLS        // DosQueryCp
  #include <os2.h>
  #include <pmbidi.h>
#endif
#ifdef IC_MOTIF
  #define BIDI               // For macros in XmStrDefs.h
#endif
}

#include <igbidi.hpp>
#include <igrexept.hpp>
#include <itrace.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IGBIDI_CPP_
  #include <ipagetun.h>
#endif

extern "C" {
#ifdef IC_MOTIF
#include <sys/lc_layout.h>
#include <locale.h>

  // Add these function prototypes, because lc_layout.h lacks them.
  int layout_object_create    ( const char*    locale_name,
                                LayoutObject*  layout_object );
  int layout_object_getvalue  ( LayoutObject   layout_object,
                                LayoutValues   values,
                                int*           index );
  int layout_object_setvalue  ( LayoutObject   layout_object,
                                LayoutValues   values,
                                int*           index );
  int layout_object_free      ( LayoutObject   layout_object );
  int layout_object_transform ( LayoutObject   layout_object,
                                const char*    inpBuf,
                                size_t*        inpSize,
                                void*          outBuf,
                                size_t*        outSize,
                                size_t*        inpToOut,
                                size_t*        outToInp,
                                unsigned char* bidiLvl);
#endif // IC_MOTIF
}

#ifndef IC_PM
/*------------------------------------------------------------------------------
| The following and defined as part of the OS/2 bidectional-support toolkit    |
| (pmbidi.h), and are included here for use in portable code.                  |
------------------------------------------------------------------------------*/
#define BDA_TEXTTYPE_VISUAL     0x00000000
#define BDA_TEXTTYPE_IMPLICIT   0x01000000
#define BDA_WND_ORIENT_LTR      0x00000000
#define BDA_WND_ORIENT_RTL      0x00100000
#define BDA_TEXT_ORIENT_LTR     0x00000000
#define BDA_TEXT_ORIENT_RTL     0x00010000
#define BDA_TEXT_ORIENT_CONTEXT 0x00020000
#define BDA_NUMERALS_NOMINAL    0x00000000
#define BDA_NUMERALS_PASSTHRU   0x00001000
#define BDA_NUMERALS_NATIONAL   0x00002000
#define BDA_NUMERALS_CONTEXTUAL 0x00003000
#define BDA_SYM_SWAP_OFF        0x00000000
#define BDA_SYM_SWAP_ON         0x00000100
#define BDA_WORD_BREAK_OFF      0x00000000
#define BDA_WORD_BREAK_ON       0x00000200
#define BDA_TEXT_DISPLAY_SHAPED 0x00000000
#define BDA_TEXT_SAVE_SHAPED    0x00000001
#define BDA_TEXT_NOMINAL        0x00000010
#define BDA_TEXT_INITIAL        0x00000011
#define BDA_TEXT_MIDDLE         0x00000012
#define BDA_TEXT_FINAL          0x00000013
#define BDA_TEXT_ISOLATED       0x00000014
#endif // ! IC_PM


/*------------------------------------------------------------------------------
| class IGBidiData                                                             |
|                                                                              |
| Private class used for implementation of bidi support.                       |
| Only a single instance of this class is needed per process, which is         |
| managed by IGBidiSettingsData.                                               |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IGBidiData {
public:
  IGBidiData ( );
 ~IGBidiData ( );
bool
  fSupport;
#ifdef IC_MOTIF
bool
  fLayoutObjectCreated;
LayoutObject
  fLayoutObject;
#endif
}; // IGBidiData

#pragma pack(pop)
#pragma enum(pop)

IGBidiData::IGBidiData ( )
  : fSupport( false )
#ifdef IC_MOTIF
  , fLayoutObjectCreated( false )
#endif
{ }

IGBidiData::~IGBidiData ( )
{
#ifdef IC_MOTIF
  if ( fLayoutObjectCreated )
  {
     layout_object_free( fLayoutObject );
     fLayoutObjectCreated = false;
  }
#endif
}


/*------------------------------------------------------------------------------
| class IGBidiSettingsData                                                     |
|                                                                              |
| Private data class for IGBidiSettings.                                       |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IGBidiSettingsData {
public:
  IGBidiSettingsData ( );
IGBidiSettingsData
 &operator=          ( const IGBidiSettingsData& data );
bool
  operator==         ( const IGBidiSettingsData& data ) const;
static IGBidiData
 *fBidi;
unsigned long
  fData;
#ifndef IC_PM
unsigned long
  fExplicitlySetFlags;
enum ESetFlags {
  kTextType             = 1,
  kTextOrientation      = 2,
  kWindowLayout         = 4,
  kNumeralDisplay       = 8,
  kSymmetricSwapping    = 16,
  kWordByWordReordering = 32,
  kTextShape            = 64
  };
#endif
}; // IGBidiSettingsData

#pragma pack(pop)
#pragma enum(pop)

IGBidiSettingsData::IGBidiSettingsData ( )
  : fData( 0 )
#ifndef IC_PM
  , fExplicitlySetFlags( 0 )
#endif
{ }

IGBidiSettingsData&
  IGBidiSettingsData::operator= ( const IGBidiSettingsData& data )
{
  if ( this != &data )
  {  // Not copying self.
     this->fData = data.fData;
#ifndef IC_PM
     this->fExplicitlySetFlags = data.fExplicitlySetFlags;
#endif
  }
  return *this;
}

bool IGBidiSettingsData::operator== ( const IGBidiSettingsData& data ) const
{
  return ( this->fData == data.fData
#ifndef IC_PM
           &&
           this->fExplicitlySetFlags == data.fExplicitlySetFlags
#endif
         );
}


/*------------------------------------------------------------------------------
| Static data                                                                  |
------------------------------------------------------------------------------*/
IGBidiData
 *IGBidiSettingsData::fBidi( 0 );


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IGBidiSettings::layoutObjectTransform ( const char* inpBuf, 
                                    			size_t*     inpSize, 
                                    			void*       outBuf, 
                                    			size_t*     outSize )
{
   IASSERTSTATE( IGBidiSettings::isBidiSupported() == true );

   int rc = 0;
   if ( ! IGBidiSettingsData::fBidi->fLayoutObjectCreated )
   {
      // Create a layout object that can transform a buffer containing
      // bidi characters into a buffer that can be passed to a Motif
      // drawing API. We delay creating the layout object until it is
      // needed -- when IXFontContext::renderGlyphs first calls this
      // routine. This allows the application to first call XtAppInitialize
      // (typically through ICurrentThread::anchorBlock), which allows
      // X to set up language information, needed by layout_object_create.
      // Otherwise if IGBidiSettings::layoutObjectTransform were to use
      // the layout object created by IGBidiSettings::isBidiSupported,
      // and isBidiSupported was called before XtAppInitialize, the
      // transformations may not do character shaping of Arabic text,
      // reversal of Arabic and Hebrew strings, and changing of
      // numerals from Arabic to Hindi.
      rc = layout_object_create( setlocale( LC_CTYPE, "" ),
                                 &( IGBidiSettingsData::fBidi
                                      ->fLayoutObject ) );
      if ( rc != 0 )
      {  // Error from layout_object_create.
         throw IGraphicException( "layout_object_create failed.",
                                  rc,
                                  IException::unrecoverable );
      }
      IGBidiSettingsData::fBidi->fLayoutObjectCreated = true;

      int index;
      LayoutValueRec
        layoutValues[ 2 ];
      LayoutTextDescriptorRec
        descriptor;
#ifdef IC_TRACE_DEVELOP
      // Query the bidi attributes of the layout object associated
      // with the current locale for debugging.
      LayoutTextDescriptorRec
       *getDescriptor;

      layoutValues[ 0 ].name  = AllTextDescptors;
      layoutValues[ 0 ].value = (caddr_t) &getDescriptor;
      layoutValues[ 1 ].name  = 0;   // Delimiter.
      layoutValues[ 1 ].value = 0;
      rc = layout_object_getvalue( IGBidiSettingsData::fBidi->fLayoutObject,
                                   layoutValues,
                                   &index );
      if ( rc != 0 )
      {  // Error from layout_object_getvalue.
         layout_object_free( IGBidiSettingsData::fBidi->fLayoutObject );
         IGBidiSettingsData::fBidi->fLayoutObjectCreated = false;
         throw IGraphicException( "IGBidiSettings::layoutObjectTransform"
                                  " - getvalue failed, rc="
                                  + IString( rc )
                                  + ", index=" + IString( index ),
                                  rc,
                                  IException::unrecoverable );
      }
      ITRACE_DEVELOP( "IGBidiSettings::layoutObjectTransform"
                        " (before setvalue):"
                        " in=" + IString( getDescriptor->in ).d2x() +
                        ", out=" + IString( getDescriptor->out ).d2x() );
      free( getDescriptor );
#endif // IC_TRACE_DEVELOP

      // Until we add the capability to set bidi attributes of a
      // grafport, assume that the string should always have an overall
      // right-to-left orientation, like that of an RTL XmText widget.
      // This only affects the drawing of text that contains both RTL
      // and LTR segments.
      descriptor.in  = ORIENTATION_RTL;
      descriptor.out = ORIENTATION_LTR;
      layoutValues[ 0 ].name  = Orientation;
      layoutValues[ 0 ].value = (caddr_t) &descriptor;
      layoutValues[ 1 ].name  = 0;   // Delimiter.
      layoutValues[ 1 ].value = 0;
      rc = layout_object_setvalue( IGBidiSettingsData::fBidi->fLayoutObject,
                                   layoutValues,
                                   &index );
      if ( rc != 0 )
      {  // Error from layout_object_setvalue.
         layout_object_free( IGBidiSettingsData::fBidi->fLayoutObject );
         IGBidiSettingsData::fBidi->fLayoutObjectCreated = false;
         throw IGraphicException( "IGBidiSettings::layoutObjectTransform -"
                                  " setvalue(Orientation) failed, rc="
                                  + IString( rc )
                                  + ", index=" + IString( index ),
                                  rc,
                                  IException::unrecoverable );
      }

#ifdef IC_TRACE_DEVELOP
      // Query the bidi attributes of the layout object again.
      layoutValues[ 0 ].name  = AllTextDescptors;
      layoutValues[ 0 ].value = (caddr_t) &getDescriptor;
      layoutValues[ 1 ].name  = 0;   // Delimiter.
      layoutValues[ 1 ].value = 0;
      rc = layout_object_getvalue( IGBidiSettingsData::fBidi->fLayoutObject,
                                   layoutValues,
                                   &index );
      if ( rc != 0 )
      {  // Error from layout_object_getvalue.
         layout_object_free( IGBidiSettingsData::fBidi->fLayoutObject );
         IGBidiSettingsData::fBidi->fLayoutObjectCreated = false;
         throw IGraphicException( "IGBidiSettings::layoutObjectTransform"
                                  " - getvalue failed, rc="
                                  + IString( rc )
                                  + ", index=" + IString( index ),
                                  rc,
                                  IException::unrecoverable );
      }
      ITRACE_DEVELOP( "IGBidiSettings::layoutObjectTransform"
                        " (after setvalue):"
                        " in=" + IString( getDescriptor->in ).d2x() +
                        ", out=" + IString( getDescriptor->out ).d2x() );
      free( getDescriptor );
#endif // IC_TRACE_DEVELOP
   }

   rc = layout_object_transform( IGBidiSettingsData::fBidi
                                    ->fLayoutObject,
                                 inpBuf, inpSize, outBuf, outSize,
                                 NULL, NULL, NULL );
	if ( rc != 0 )
   {  // Error from layout_object_transform.
      layout_object_free( IGBidiSettingsData::fBidi->fLayoutObject );
      IGBidiSettingsData::fBidi->fLayoutObjectCreated = false;
      throw IGraphicException( "layout_object_transform failed.",
		                         rc,
                               IException::unrecoverable );
   }
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
bool IGBidiSettings::isBidiSupported ( )
{
#ifdef IC_PM
  if ( ! IGBidiSettingsData::fBidi )
  {
     IGBidiSettingsData::fBidi = new IGBidiData;
     IGBidiSettingsData::fBidi->fSupport = false;

     // Check if the current process codepage is 862 (Hebrew) or
     // 864 (Arabic).
     unsigned long
       codePages [ 5 ];
     unsigned long
       bytes = 0;
     unsigned long
       rc = DosQueryCp( sizeof( codePages ), codePages, &bytes );
     if ( rc == NO_ERROR  ||  rc == ERROR_CPLIST_TOO_SMALL )
     {
#ifdef IC_TRACE_DEVELOP
        ITRACE_DEVELOP( "IGBidiSettings::isBidiSupported");
        for ( unsigned long i = 0; i < bytes; i += sizeof( unsigned long ) )
        {
            ITRACE_DEVELOP( " > codepage[" +
                              IString( i / sizeof( unsigned long ) ) +
                              "] == " + IString( codePages[ i ] ) );
        }
#endif
        if ( codePages[ 0 ] == 862  ||  codePages[ 0 ] == 864 )
        {
           // Note: We need to grow the above check if OS/2 adds
           //       code pages supporting Arabic or Hebrew.
           IGBidiSettingsData::fBidi->fSupport = true;
        }
     }
     else
     {
        throw IGraphicException( "DosQueryCp failed.",
                                 rc,
                                 IException::unrecoverable );
     }
  }
  return IGBidiSettingsData::fBidi->fSupport;
#endif // IC_PM

#ifdef IC_WIN
  if ( ! IGBidiSettingsData::fBidi )
  {
     IGBidiSettingsData::fBidi = new IGBidiData;
     IGBidiSettingsData::fBidi->fSupport = false;

     // Check if the system is mideast enabled.
     if ( GetSystemMetrics( SM_MIDEASTENABLED ) )
     {
        IGBidiSettingsData::fBidi->fSupport = true;
     }
#ifdef IC_DEVELOP
     else      // Code for development testing.
     {
        const char* bidiVar = getenv( "ICLUI_BIDI" );
        if ( bidiVar )
        {
           IGBidiSettingsData::fBidi->fSupport = true;
        }
     }
#endif
  }
  return IGBidiSettingsData::fBidi->fSupport;
#endif // IC_WIN

#ifdef IC_MOTIF
  if ( !IGBidiSettingsData::fBidi )
  {
     IGBidiSettingsData::fBidi = new IGBidiData;
     IGBidiSettingsData::fBidi->fSupport = false;

     // Don't use IGBidiData::fLayoutObject, so this function can
     // be called before XtAppInitialize (otherwise
     // IGBidiSettings::layoutObjectTransform may not work properly).
     LayoutObject
       layoutObject;
     int rc = layout_object_create( setlocale( LC_CTYPE, "" ),
                                    &layoutObject );
     if ( rc != 0 )
     {  // Error from layout_object_create.
        throw IGraphicException( "layout_object_create failed.",
                                 rc,
                                 IException::unrecoverable );
     }

     int index;
     LayoutValueRec
       layoutValues[ 2 ];
     layoutValues[ 0 ].name = ActiveBidirection;
     layoutValues[ 1 ].name = 0;   // Delimiter.
     rc = layout_object_getvalue( layoutObject, layoutValues, &index );
     layout_object_free( layoutObject );
     if ( rc != 0 )
     {  // Error from layout_object_getvalue.
        throw IGraphicException( "layout_object_getvalue failed.",
                                 rc,
                                 IException::unrecoverable );
     }
     IGBidiSettingsData::fBidi->fSupport = *( layoutValues[ 0 ].value );

  }
  return IGBidiSettingsData::fBidi->fSupport;
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| Default constructor.                                                         |
------------------------------------------------------------------------------*/
IGBidiSettings::IGBidiSettings ( )
  : fBidiSettingsData( new IGBidiSettingsData )
{ }

/*------------------------------------------------------------------------------
| Copy constructor.                                                            |
------------------------------------------------------------------------------*/
IGBidiSettings::IGBidiSettings ( const IGBidiSettings& settings )
  : fBidiSettingsData( new IGBidiSettingsData )
{
  *( this->fBidiSettingsData ) = *( settings.fBidiSettingsData );
}

/*------------------------------------------------------------------------------
| Assignment operator.                                                         |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::operator= ( const IGBidiSettings& settings )
{
  *( this->fBidiSettingsData ) = *( settings.fBidiSettingsData );
  return *this;
}

/*------------------------------------------------------------------------------
| Comparison operator.                                                         |
------------------------------------------------------------------------------*/
bool IGBidiSettings::operator== ( const IGBidiSettings& settings ) const
{
  return ( *( this->fBidiSettingsData ) ==
                       *( settings.fBidiSettingsData ) );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings::~IGBidiSettings ( )
{
  if ( fBidiSettingsData )
  {
     delete fBidiSettingsData;
  }
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::setTextType
                  ( IGBidiSettings::EBidiTextType textType )
{
  ITRACE_WIN_NOP();

  switch ( textType )
  {
     case kVisual:
       fBidiSettingsData->fData &= (unsigned long) ~BDA_TEXTTYPE_IMPLICIT;
       fBidiSettingsData->fData |= BDA_TEXTTYPE_VISUAL;
       break;
     case kImplicit:
       fBidiSettingsData->fData &= (unsigned long) ~BDA_TEXTTYPE_VISUAL;
       fBidiSettingsData->fData |= BDA_TEXTTYPE_IMPLICIT;
       break;
     default:
       throw IGraphicException( "The specified enumerator value"
                                " is not valid.",
                                IGraphicException::kUnknownEnumValue,
                                IException::recoverable );
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |= IGBidiSettingsData::kTextType;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::setTextOrientation
                  ( IGBidiSettings::EBidiTextOrientation textOrientation )
{
  switch ( textOrientation )
  {
     case kTextLeftToRight:
       fBidiSettingsData->fData &= (unsigned long)
                            ~( BDA_TEXT_ORIENT_RTL | BDA_TEXT_ORIENT_CONTEXT );
       fBidiSettingsData->fData |= BDA_TEXT_ORIENT_LTR;
       break;
     case kTextRightToLeft:
       fBidiSettingsData->fData &= (unsigned long)
                            ~( BDA_TEXT_ORIENT_LTR | BDA_TEXT_ORIENT_CONTEXT );
       fBidiSettingsData->fData |= BDA_TEXT_ORIENT_RTL;
       break;
     case kTextContextual:
       fBidiSettingsData->fData &= (unsigned long)
                            ~( BDA_TEXT_ORIENT_LTR | BDA_TEXT_ORIENT_RTL );
       fBidiSettingsData->fData |= BDA_TEXT_ORIENT_CONTEXT;
       break;
     default:
       throw IGraphicException( "The specified enumerator value"
                                " is not valid.",
                                IGraphicException::kUnknownEnumValue,
                                IException::recoverable );
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IGBidiSettingsData::kTextOrientation;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::setWindowLayout
                  ( IGBidiSettings::EBidiLayout windowLayout )
{
  switch ( windowLayout )
  {
     case kLayoutLeftToRight:
       fBidiSettingsData->fData &= (unsigned long) ~BDA_WND_ORIENT_RTL;
       fBidiSettingsData->fData |= BDA_WND_ORIENT_LTR;
       break;
     case kLayoutRightToLeft:
       fBidiSettingsData->fData &= (unsigned long) ~BDA_WND_ORIENT_LTR;
       fBidiSettingsData->fData |= BDA_WND_ORIENT_RTL;
       break;
     default:
       throw IGraphicException( "The specified enumerator value"
                                " is not valid.",
                                IGraphicException::kUnknownEnumValue,
                                IException::recoverable );
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IGBidiSettingsData::kWindowLayout;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::setNumeralDisplay
                  ( IGBidiSettings::EBidiNumeralType numeralDisplay )
{
  ITRACE_WIN_NOP();

  fBidiSettingsData->fData &= (unsigned long)
                          ~( BDA_NUMERALS_NOMINAL | BDA_NUMERALS_PASSTHRU |
                             BDA_NUMERALS_NATIONAL | BDA_NUMERALS_CONTEXTUAL );

  switch ( numeralDisplay )
  {
     case kArabic:
       fBidiSettingsData->fData |= BDA_NUMERALS_NOMINAL;
       break;
     case kAsStored:
       fBidiSettingsData->fData |= BDA_NUMERALS_PASSTHRU;
       break;
     case kNational:
       fBidiSettingsData->fData |= BDA_NUMERALS_NATIONAL;
       break;
     case kContextual:
       fBidiSettingsData->fData |= BDA_NUMERALS_CONTEXTUAL;
       break;
     default:
       throw IGraphicException( "The specified enumerator value"
                                " is not valid.",
                                IGraphicException::kUnknownEnumValue,
                                IException::recoverable );
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IGBidiSettingsData::kNumeralDisplay;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::enableSymmetricSwapping ( bool enable )
{
  ITRACE_WIN_NOP();

  if ( enable )
  {
     fBidiSettingsData->fData &= (unsigned long) ~BDA_SYM_SWAP_OFF;
     fBidiSettingsData->fData |= BDA_SYM_SWAP_ON;
  }
  else
  {
     fBidiSettingsData->fData &= (unsigned long) ~BDA_SYM_SWAP_ON;
     fBidiSettingsData->fData |= BDA_SYM_SWAP_OFF;
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IGBidiSettingsData::kSymmetricSwapping;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::disableSymmetricSwapping ( )
{
  return this->enableSymmetricSwapping( false );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::enableWordByWordReordering ( bool enable )
{
  ITRACE_WIN_NOP();

  if ( enable )
  {
     fBidiSettingsData->fData &= (unsigned long) ~BDA_WORD_BREAK_OFF;
     fBidiSettingsData->fData |= BDA_WORD_BREAK_ON;
  }
  else
  {
     fBidiSettingsData->fData &= (unsigned long ) ~BDA_WORD_BREAK_ON;
     fBidiSettingsData->fData |= BDA_WORD_BREAK_OFF;
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IGBidiSettingsData::kWordByWordReordering;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::disableWordByWordReordering ( )
{
  return this->enableWordByWordReordering( false );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings& IGBidiSettings::setTextShape
                  ( IGBidiSettings::EBidiTextShape textShape )
{
  ITRACE_WIN_NOP();

  fBidiSettingsData->fData &= (unsigned long)
                          ~( BDA_TEXT_DISPLAY_SHAPED | BDA_TEXT_SAVE_SHAPED |
                             BDA_TEXT_NOMINAL | BDA_TEXT_MIDDLE | BDA_TEXT_FINAL |
                             BDA_TEXT_ISOLATED );

  switch ( textShape )
  {
     case kDisplayShaped:
       fBidiSettingsData->fData |= BDA_TEXT_DISPLAY_SHAPED;
       break;
     case kSaveShaped:
       fBidiSettingsData->fData |= BDA_TEXT_SAVE_SHAPED;
       break;
     case kNominalShape:
       fBidiSettingsData->fData |= BDA_TEXT_NOMINAL;
       break;
     case kInitialShape:
       fBidiSettingsData->fData |= BDA_TEXT_INITIAL;
       break;
     case kMiddleShape:
       fBidiSettingsData->fData |= BDA_TEXT_MIDDLE;
       break;
     case kFinalShape:
       fBidiSettingsData->fData |= BDA_TEXT_FINAL;
       break;
     case kIsolatedShape:
       fBidiSettingsData->fData |= BDA_TEXT_ISOLATED;
       break;
     default:
       throw IGraphicException( "The specified enumerator value"
                                " is not valid.",
                                IGraphicException::kUnknownEnumValue,
                                IException::recoverable );
  }

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |= IGBidiSettingsData::kTextShape;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings::EBidiTextType IGBidiSettings::textType ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kTextType ) )
  {
     return kImplicit;
  }
  else
#endif
  if ( fBidiSettingsData->fData & BDA_TEXTTYPE_IMPLICIT )
  {
     return kImplicit;
  }
  return kVisual;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings::EBidiTextOrientation IGBidiSettings::textOrientation ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kTextOrientation ) )
  {
     return kTextLeftToRight;
  }
  else
#endif
  if ( fBidiSettingsData->fData & BDA_TEXT_ORIENT_CONTEXT )
  {
     return kTextContextual;
  }
  else if ( fBidiSettingsData->fData & BDA_TEXT_ORIENT_RTL )
  {
     return kTextRightToLeft;
  }
  return kTextLeftToRight;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings::EBidiLayout IGBidiSettings::windowLayout ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kWindowLayout ) )
  {
     return kLayoutLeftToRight;
  }
  else
#endif
  if ( fBidiSettingsData->fData & BDA_WND_ORIENT_RTL )
  {
     return kLayoutRightToLeft;
  }
  return kLayoutLeftToRight;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings::EBidiNumeralType IGBidiSettings::numeralDisplay ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kNumeralDisplay ) )
  {
     return kContextual;
  }
  else
#endif
  if ( ( fBidiSettingsData->fData & BDA_NUMERALS_CONTEXTUAL ) ==
                BDA_NUMERALS_CONTEXTUAL )
  {
     return kContextual;
  }
  else if ( ( fBidiSettingsData->fData & BDA_NUMERALS_PASSTHRU ) ==
                BDA_NUMERALS_PASSTHRU )
  {
     return kAsStored;
  }
  else if ( ( fBidiSettingsData->fData & BDA_NUMERALS_NATIONAL ) ==
                BDA_NUMERALS_NATIONAL )
  {
     return kNational;
  }
  return kArabic;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
bool IGBidiSettings::isSymmetricSwappingEnabled ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kSymmetricSwapping ) )
  {
     return false;
  }
  else
#endif
  if ( fBidiSettingsData->fData & BDA_SYM_SWAP_ON )
  {
     return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
bool IGBidiSettings::isWordByWordReorderingEnabled ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kWordByWordReordering ) )
  {
     return false;
  }
  else
#endif
  if ( fBidiSettingsData->fData & BDA_WORD_BREAK_ON )
  {
     return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IGBidiSettings::EBidiTextShape IGBidiSettings::textShape ( ) const
{
#ifndef IC_PM
  if ( ! ( fBidiSettingsData->fExplicitlySetFlags &
                        IGBidiSettingsData::kTextShape ) )
  {
     return kDisplayShaped;
  }
  else
#endif
  if ( ( fBidiSettingsData->fData & BDA_TEXT_ISOLATED ) == BDA_TEXT_ISOLATED )
  {
     return kIsolatedShape;
  }
  else if ( ( fBidiSettingsData->fData & BDA_TEXT_FINAL ) == BDA_TEXT_FINAL )
  {
     return kFinalShape;
  }
  else if ( ( fBidiSettingsData->fData & BDA_TEXT_MIDDLE ) == BDA_TEXT_MIDDLE )
  {
     return kMiddleShape;
  }
  else if ( ( fBidiSettingsData->fData & BDA_TEXT_INITIAL ) == BDA_TEXT_INITIAL )
  {
     return kInitialShape;
  }
  else if ( ( fBidiSettingsData->fData & BDA_TEXT_NOMINAL ) == BDA_TEXT_NOMINAL )
  {
     return kNominalShape;
  }
  else if ( ( fBidiSettingsData->fData & BDA_TEXT_SAVE_SHAPED ) == BDA_TEXT_SAVE_SHAPED )
  {
     return kSaveShaped;
  }
  return kDisplayShaped;
}

#ifdef IC_PM
IGBidiSettings& IGBidiSettings::setBidiFlags ( unsigned long flags )
{
  fBidiSettingsData->fData = flags;
  return *this;
}

unsigned long IGBidiSettings::bidiFlags ( ) const
{
  return fBidiSettingsData->fData;
}
#endif // IC_PM
