/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright Taligent, Inc.,  1996                                                 *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 80 1.14.2.2 source/albert/graph2d/icolmap.cpp, 2d, ioc.v400, 001006 
/*==========================================
||
||      File:   icolmap.c
||
||  Class:  IRGBAColorArray
||          IColorMap
||          IColorMapContext
||
||                      Copyright (C) 1995 Taligent, Inc. All rights reserved.
||
||      Change History:
||
*/

//#define IC_TRACE_ALL

#ifdef IC_PM
#define INCL_BASE
#define INCL_GPI
#define INCL_WINSYS
#define INCL_WINPALETTE
#include <winemul.h>
#define IQUERYRGBCOLOR(hps,idx,pPe) (*pPe = GpiQueryRGBColor(hps, 0L, idx))
#define IRGB_BLUE(rgb)   ((unsigned char)(rgb & 0x000000FF) )
#define IRGB_GREEN(rgb)  ((unsigned char)((rgb & 0x0000FF00) >> 8) )
#define IRGB_RED(rgb)    ((unsigned char)((rgb & 0x00FF0000) >> 16) )
#endif // IC_PM

#include <stdlib.h>

#include <icolmap.hpp>
#include <icolmapp.hpp>
#include <igrexept.hpp>
#include <iprimlck.hpp>

#include <istring.hpp>
#include <itrace.hpp>

#ifdef IC_WIN
#include <windows.h>
#endif

#ifdef IC_MOTIF

#include <ixdc.hpp>
#include <pixbuf.hpp>

#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>

#include <i2dghand.hpp>
#endif // IC_MOTIF

// Segment definitions
#ifdef IC_PAGETUNE
#define _ICOLMAP_CPP_
#include <ipagetun.h>
#endif

#define ABS(_val) (((_val) < 0) ? -(_val) : (_val))

#if 1 //SAS

/*------------------------------------------------------------------------------
| IR8G8B8A8Color::operator IBaseColor                                          |
------------------------------------------------------------------------------*/
IR8G8B8A8Color::operator IBaseColor() const {
        return IBaseColor(fRed, fGreen, fBlue, fOpacity);
}


/*------------------------------------------------------------------------------
| IRGBAColorArray::IRGBAColorArray                                             |
------------------------------------------------------------------------------*/
IRGBAColorArray::IRGBAColorArray(unsigned long numberOfColors)
{
    fNumberOfColors = numberOfColors;
    fColors = fNumberOfColors ? new IR8G8B8A8Color[numberOfColors] : 0 /*NIL*/;

}


/*------------------------------------------------------------------------------
| IRGBAColorArray::IRGBAColorArray                                             |
------------------------------------------------------------------------------*/
IRGBAColorArray::IRGBAColorArray(const IRGBAColorArray& source)
{
    fNumberOfColors = source.fNumberOfColors;

    if (!source.fNumberOfColors) {
                    delete [] fColors;
        fColors = 0 /*NIL*/;
        }
    else {
        fColors = new IR8G8B8A8Color[source.fNumberOfColors];
                                memcpy(fColors, source.fColors, sizeof(IR8G8B8A8Color)*fNumberOfColors);
    }

}


/*------------------------------------------------------------------------------
| IRGBAColorArray::~IRGBAColorArray                                            |
------------------------------------------------------------------------------*/
IRGBAColorArray::~IRGBAColorArray()
{
        delete [] fColors;
}



/*------------------------------------------------------------------------------
| IRGBAColorArray::color                                                       |
------------------------------------------------------------------------------*/
IBaseColor IRGBAColorArray::color(unsigned long i) const
{
        return IBaseColor(fColors[i].fRed, fColors[i].fGreen, fColors[i].fBlue, fColors[i].fOpacity);
}


/*------------------------------------------------------------------------------
| IRGBAColorArray::setColor                                                    |
------------------------------------------------------------------------------*/
void IRGBAColorArray::setColor(unsigned long i, const IBaseColor& p)
{
        IGraphicException::InternalAssert(i < fNumberOfColors, "IRGBAColorArray index out of range");
        fColors[i].fRed = p.redMix();
        fColors[i].fGreen = p.greenMix();
        fColors[i].fBlue = p.blueMix();
        fColors[i].fOpacity = p.opacity();
}


/*------------------------------------------------------------------------------
| IRGBAColorArray::numberOfColors                                              |
------------------------------------------------------------------------------*/
unsigned long IRGBAColorArray::numberOfColors()const
{
        return fNumberOfColors;
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::operator=                                                   |
------------------------------------------------------------------------------*/
IRGBAColorArray& IRGBAColorArray::operator=(const IRGBAColorArray& source)
{
        if (&source != this)
        {
                delete [] fColors;

                fNumberOfColors = source.fNumberOfColors;

    if (fNumberOfColors == 0)
        fColors = 0 /*NIL*/;
    else
    {
            fColors = new IR8G8B8A8Color[fNumberOfColors];
            memcpy(fColors, source.fColors, sizeof(IR8G8B8A8Color)*fNumberOfColors);
    }
        }
        return *this;
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::operator==                                                  |
------------------------------------------------------------------------------*/
bool IRGBAColorArray::operator==(const IRGBAColorArray& source) const
{
        if (fNumberOfColors != source.fNumberOfColors)
                return false;

        if (fNumberOfColors == 0)
        {
                return (fColors == 0 /*NIL*/) && (source.fColors == 0 /*NIL*/);
        }

        return !memcmp(fColors, source.fColors, sizeof(IR8G8B8A8Color))*fNumberOfColors;
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::operator[]                                                  |
------------------------------------------------------------------------------*/
IR8G8B8A8Color& IRGBAColorArray::operator[](unsigned long i)
{
        IGraphicException::InternalAssert(i < fNumberOfColors, "IRGBAColorArray index out of range");
        return fColors[i];
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::operator[]                                                  |
------------------------------------------------------------------------------*/
const IR8G8B8A8Color& IRGBAColorArray::operator[](unsigned long i) const
{
        IGraphicException::InternalAssert(i < fNumberOfColors, "IRGBAColorArray index out of range");
        return fColors[i];
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::orphanColorArray                                            |
------------------------------------------------------------------------------*/
IR8G8B8A8Color* IRGBAColorArray::orphanColorArray()
{
        IR8G8B8A8Color* result = fColors;
        fColors = 0 /*NIL*/;
        fNumberOfColors = 0;
        return result;
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::adoptColorArray                                             |
------------------------------------------------------------------------------*/
void IRGBAColorArray::adoptColorArray(unsigned long size, IR8G8B8A8Color* colorArrayToAdopt)
{
        delete [] fColors;
        fColors = colorArrayToAdopt ;
        fNumberOfColors = size;
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::operator>>=                                                 |
------------------------------------------------------------------------------*/
IDataStream& IRGBAColorArray::operator>>=(IDataStream& toWhere) const
{
        {
                IStreamOutFrame streamFrame(toWhere);

                register unsigned long i;
                fNumberOfColors >>= toWhere;

                long numberColors = fNumberOfColors;
                for (i = 0; i < numberColors; i++)
                {
                  fColors[i].fRed >>= toWhere;
                        fColors[i].fGreen >>= toWhere;
                        fColors[i].fBlue >>= toWhere;
                        fColors[i].fOpacity >>= toWhere;
                }
        }
        return toWhere;
}

/*------------------------------------------------------------------------------
| IRGBAColorArray::operator<<=                                                 |
------------------------------------------------------------------------------*/
IDataStream& IRGBAColorArray::operator<<=(IDataStream& fromWhere)
{
        {
                IStreamInFrame streamFrame(fromWhere);

                delete [] fColors;

                fNumberOfColors <<= fromWhere;

                long numberColors = fNumberOfColors;

                fColors = numberColors ? new IR8G8B8A8Color[numberColors] : 0 /*NIL*/;
                for (long i = 0; i < numberColors; i++)
                {
                  fColors[i].fRed <<= fromWhere;
                        fColors[i].fGreen <<= fromWhere;
                        fColors[i].fBlue <<= fromWhere;
                        fColors[i].fOpacity <<= fromWhere;
                }
        }
        return fromWhere;
}

#endif //SAS

// IRGBAColorArray class is the basic storage for cross platform pixel data or
// color data such as a  color lookup table (palette in Windows and OS/2).
//
// When used as a color lookup table there may be native platform specific entries
// in the table that's not available for the user to set(which will be ignored by
// the host.


/*------------------------------------------------------------------------------
| IColorMap::IColorMap                                                         |
------------------------------------------------------------------------------*/
IColorMap::IColorMap( EColorMapType colorMapType)
      : fNumberOfColors   (256),
        fData(0),
        fColorMapType( colorMapType ),
       fColorMapData(0)
{
        IFUNCTRACE_DEVELOP();
        fColors = new IR8G8B8A8Color[fNumberOfColors];
        fCellStatus = new EColorCellStatus[fNumberOfColors];

        unsigned long i;
        for (i = 0; i < fNumberOfColors; i++)
                fCellStatus[i] = kFree;
        fNumberOfFreeCells = fNumberOfColors;

#if defined(IC_WIN)
       if (IColorMap::hasColorMapSupport()) {

                HDC hdc = GetDC(HWND_DESKTOP);
                unsigned long defaultEntries = GetDeviceCaps(hdc, SIZEPALETTE);

                PALETTEENTRY* pal = new PALETTEENTRY[defaultEntries];
                memset(pal, 0x0,(defaultEntries * sizeof(PALETTEENTRY)));

                GetSystemPaletteEntries(hdc, 0L, defaultEntries, pal);

                ReleaseDC(HWND_DESKTOP, hdc);

                if(defaultEntries < 256){
                        for (i = 0; i < defaultEntries; i++)
                        {
                                fColors[i].fRed = pal[i].peRed ;
                                fColors[i].fGreen = pal[i].peGreen ;
                                fColors[i].fBlue = pal[i].peBlue;
                                fColors[i].fOpacity = 0xff;
                                fCellStatus[i] = kReserved;
                        }
                        fNumberOfFreeCells -= defaultEntries;
                } else {
                        //copy the system palette
                        for (i = 0; i < 10; i++)
                        {
                                fColors[i].fRed = pal[i].peRed ;
                                fColors[i].fGreen = pal[i].peGreen ;
                                fColors[i].fBlue = pal[i].peBlue;
                                fColors[i].fOpacity = 0xff;
                                fCellStatus[i] = kReserved;
                        }

                        for (i = 246; i < 256; i++)
                        {
                                fColors[i].fRed = pal[i].peRed ;
                                fColors[i].fGreen = pal[i].peGreen ;
                                fColors[i].fBlue = pal[i].peBlue;
                                fColors[i].fOpacity = 0xff;
                                fCellStatus[i] = kReserved;
                        }
                        fNumberOfFreeCells -= 20;
                }
                delete [] pal;
        } else { //no palette support

        }

#elif defined(IC_PM)

        HPS hps = WinGetPS(HWND_DESKTOP);
        HDC hdc = GpiQueryDevice( hps );
        long defaultEntries;
        long rgbColor=0;
        long capsClr;

        //check if there's palette support
       if (IColorMap::hasColorMapSupport()){
                //to get the no. of colors supported by the current device
                DevQueryCaps(hdc, CAPS_PHYS_COLORS, 1, &defaultEntries); //max. no. of colors supported

                long* alArray = new long[defaultEntries];
                GpiQueryRealColors(hps, 0L, 0L, defaultEntries, alArray);

                if(defaultEntries < 256){
                        for (i = 0; i < defaultEntries; i++)
                        {
                                fColors[i].fRed = IRGB_RED(alArray[i]);
                                fColors[i].fGreen = IRGB_GREEN(alArray[i]);
                                fColors[i].fBlue = IRGB_BLUE(alArray[i]);
                                fColors[i].fOpacity = 0xff;
                                fCellStatus[i] = kReserved;
                        }
                        fNumberOfFreeCells -= defaultEntries;
                } else {
                        //copy the system palette
                        for (i = 0; i < 10; i++)
                        {
                                fColors[i].fRed = IRGB_RED(alArray[i]);
                                fColors[i].fGreen = IRGB_GREEN(alArray[i]);
                                fColors[i].fBlue = IRGB_BLUE(alArray[i]);
                                fColors[i].fOpacity = 0xff;
                                fCellStatus[i] = kReserved;
                        }

                        for (i = 246; i < 256; i++)
                        {
                                fColors[i].fRed = IRGB_RED(alArray[i]);
                                fColors[i].fGreen = IRGB_GREEN(alArray[i]);
                                fColors[i].fBlue = IRGB_BLUE(alArray[i]);
                                fColors[i].fOpacity = 0xff;
                                fCellStatus[i] = kReserved;
                        }
                        fNumberOfFreeCells -= 20;
                }
		delete [] alArray;
        } else { //no palette support
        }
        WinReleasePS(hps); //this needs to be here (defect 23817)

#elif defined(IC_MOTIF)

#define XSYSTEMCOLORS 36 //just a hack
    // at present only default colormap is considered.

		if (IColorMap::hasColorMapSupport())
		{
  			int i, ncolors = 0;
			Display* disp = getDisplay(); //assuming its not null
			ncolors = CellsOfScreen(DefaultScreenOfDisplay(disp));
			Colormap defClrmap = XDefaultColormap(disp, DefaultScreen(disp));
			// Set the handle of the colormap so that nativeColormap does
			// not attempt to re-allocate colors.
			this->fData = defClrmap;

			XColor *xcolor = new XColor[ncolors];
			memset(xcolor, 0x0, sizeof(XColor)*ncolors);

			for (i=0; i< ncolors; i++)
			{
			   xcolor[i].pixel = i;
			   xcolor[i].flags = DoRed|DoGreen|DoBlue;
			}
			XQueryColors(disp, defClrmap, xcolor, ncolors);

			// Initialize the color array from the colors returned.
			for (i=0; i< ncolors; i++)
			{
				fColors[i].fRed   = xcolor[i].red >>8;
				fColors[i].fGreen = xcolor[i].green >>8;
				fColors[i].fBlue  = xcolor[i].blue >>8;
				fColors[i].fOpacity  = 0xff;
				fCellStatus[i] = kUsed;
				fNumberOfFreeCells--;
			}
			if (IColorMap::canChangeColorMap())
			{
				if (fColorMapType == kSystem)
				{
					//Used as array of pixels and reused as a flag array.
					unsigned long* pixels = new unsigned long[ ncolors ];

					// Make a pass through the color table.  For each cell, attempt to
					// allocate a color not in the list.  Cells which succeed in this
					// are not actually in use.
					unsigned long rgb(0);
					int           npixels (0);      // Number to free.
					// Assumption is any initially free cells are at the end.
					for (i=0; i<ncolors; i++)
					{
						XColor junkcolor;
						junkcolor.flags = DoRed|DoGreen|DoBlue;
						int currentIndex;

						// Find a color not in the list.
						do
						{
							rgb++;
							junkcolor.red =   (rgb & 0x00ff0000) >> 16;
							junkcolor.green = (rgb & 0x0000ff00) >> 8;
							junkcolor.blue =  (rgb & 0x000000ff);
						}
						while ( kNotFound != this->colorExists(
													   junkcolor.red,
													   junkcolor.green,
													   junkcolor.blue,
													   &currentIndex) );
						// Convert to XColor and attempt to allocate this color.  If
						// XAllocColor succeeds it will return a new pixel that is
						// in fact not allocated.
						junkcolor.red   <<= 8;
						junkcolor.green <<= 8;
						junkcolor.blue  <<= 8;
						if ( XAllocColor( disp, defClrmap, &junkcolor) )
						{  // Cell not allocated.  Mark it as free.
						   this->fCellStatus[junkcolor.pixel] = kFree;
						   //reset the free cell colors
						   this->fColors[junkcolor.pixel].fRed = 0x0;
						   this->fColors[junkcolor.pixel].fGreen = 0x0;
						   this->fColors[junkcolor.pixel].fBlue = 0x0;
						   npixels++;
						}
						else
						{  // Colormap must be full.  Quit.
						   i = ncolors;
						}
					}  //for

					//Free the junk colors we just allocated.
					if (npixels)
					{
					   unsigned long pindex(0);
					   for (i = 0; i < ncolors; i++)
					   {
						   if (this->fCellStatus[i] == kFree)
						   {
							  pixels[pindex] = i;
							  pindex++;
						   }
					   }
					   XFreeColors(disp, defClrmap, pixels, npixels, 0);
					   this->fNumberOfFreeCells += npixels;
					}

					#ifdef IC_TRACE_DEVELOP
					{
					   char colorArray[256];
					   for (i=0; i<256; i++)
					   {
						  switch (this->fCellStatus[i])
						  {
							  case kFree:
								 colorArray[i] = 'F';
								 break;
							  case kReserved:
								 colorArray[i] = 'R';
								 break;
							  case kUsed:
								 colorArray[i] = 'U';
								 break;
							  default:
								 colorArray[i] = ' ';
						  }
					   }
					   ITRACE_DEVELOP( IString("Initial colormap") );
					   ITRACE_DEVELOP( IString(&colorArray[0], 64) );
					   ITRACE_DEVELOP( IString(&colorArray[64], 64) );
					   ITRACE_DEVELOP( IString(&colorArray[128], 64) );
					   ITRACE_DEVELOP( IString(&colorArray[192], 64) );
					}
					#endif  //IC_TRACE_DEVELOP

					//Now we know the colors that are really allocated.
					// It is possible that colors are duplicated in the
					// color table. We flag the duplicates as reserved so that
					// we can get colors Motif is using into the color table.

					// Reset pixels array so that we can use it as flags.
					memset(pixels, 0x0, sizeof(unsigned long)*ncolors);
					for (i=0; i<ncolors; i++)
					{
						if (fCellStatus[i] == kUsed)
						{
						   int currentIndex(-1);
						   IR8G8B8A8Color& color(fColors[i]);
						   while ( (currentIndex < i) &&
								   (kNotFound != this->colorExists( color.fRed,
															color.fGreen,
															color.fBlue,
															&currentIndex)) )
						   {
							  if (currentIndex < i)
							  {
								  // Flag previous color as free temporarily so
								  // that colorExists will not examine it again.
								  fCellStatus[currentIndex] = kFree;
								  pixels[currentIndex] = 1;    //Flag to mark as reserved.
							  }
						   }
						}
					}

					// Flag all the duplicate colors as reserved.  They must have
					// been allocated R/W in order for the dups to get into the table.
					for (i = 0; i < ncolors; i++)
					{
					   if (pixels[i] != 0)
						  fCellStatus[i] = kReserved;
					}

					// Call XAllocColor to insure that all colors currently
					// in the table remain there for the duration of this object
					// so that we do not have stale data.
					#ifdef IC_TRACE_DEVELOP
					   int traceAllocColor(0);
					#endif
					for (i=0; i<ncolors; i++)
					{
						if (this->fCellStatus[i] == kUsed)
						{
							int status = XAllocColor( disp, defClrmap, &xcolor[i]);
							#ifdef IC_TRACE_DEVELOP
								 traceAllocColor++;
							#endif
							#ifdef IC_TRACE_ALL
							  ITRACE_ALL("XAllocColor");

							  if (!status)
								 ITRACE_ALL( IString("XAllocColor failed index=") +
											 IString(i) );
							  unsigned long rgb = (fColors[i].fRed << 16) |
												  (fColors[i].fGreen << 8) |
												   fColors[i].fBlue;
							  ITRACE_ALL( IString("pixel=") +
										  IString(xcolor[i].pixel) +
										  IString(" i=") + IString(i) +
										  IString(" rgb=") + IString(rgb).d2x() );
							#endif
						}  // if not free
					}  // for

					delete [] pixels;

					#ifdef IC_TRACE_DEVELOP
					{
					   char colorArray[256];
					   for (i=0; i<256; i++)
					   {
						  switch (this->fCellStatus[i])
						  {
							  case kFree:
								 colorArray[i] = 'F';
								 break;
							  case kReserved:
								 colorArray[i] = 'R';
								 break;
							  case kUsed:
								 colorArray[i] = 'U';
								 break;
							  default:
								 colorArray[i] = ' ';
						  }
					   }
					   ITRACE_DEVELOP( IString("Filtered colormap") );
					   ITRACE_DEVELOP( IString(&colorArray[0], 64) );
					   ITRACE_DEVELOP( IString(&colorArray[64], 64) );
					   ITRACE_DEVELOP( IString(&colorArray[128], 64) );
					   ITRACE_DEVELOP( IString(&colorArray[192], 64) );
					   ITRACE_DEVELOP(
						  IString("system colormap fData=") +
						  IString(this->fData).d2x() +
						  IString(" XAllocColor count=") + IString(traceAllocColor) );
					}
					#endif  //IC_TRACE_DEVELOP
				}
				else
				{   // Not system.  Copy the system map and reset color map type.
					*this = defaultSystemColorMap();
					fColorMapType = colorMapType;
				}

			}
#ifdef IC_TRACE_DEVELOP
			for (i=0; i< ncolors; i++)
			{
		   ITRACE_DEVELOP(
			  IString("fColors ") + IString(i).d2x() + IString(" ")+
			  IString((fColors[i].fRed << 16) |
						  (fColors[i].fGreen << 8) |
						   fColors[i].fBlue).d2x() + IString(" ")+
				IString(fCellStatus[i]).d2x());
			}
#endif  //IC_TRACE_DEVELOP
			delete [] xcolor;
				
		} else { //no palette support
		}
#endif // IC_MOTIF

}

/*------------------------------------------------------------------------------
| IColorMap::IColorMap                                                         |
------------------------------------------------------------------------------*/
IColorMap::IColorMap(unsigned long count, const IR8G8B8A8Color* clrAry)
      : fNumberOfColors(0),
        fNumberOfFreeCells(0),
        fData(0),
        fColors(0),
        fCellStatus(0),
        fColorMapType( kUserDefined ),
       fColorMapData(0)
{
        IFUNCTRACE_DEVELOP();
        *this = IColorMap(kUserDefined); //make a copy of system colormap
        mapColors(count, clrAry); //map these colors now
}

/*------------------------------------------------------------------------------
| IColorMap::IColorMap                                                         |
------------------------------------------------------------------------------*/
IColorMap::IColorMap(const IColorMap& src)
      : fNumberOfColors(src.fNumberOfColors),
        fNumberOfFreeCells( src.fNumberOfFreeCells),
        fData(0),
        fColorMapType( src.fColorMapType ),
        fColorMapData( 0 )  // Do not copy cache.  Re-initialize w/ new colormap.
{
        IFUNCTRACE_DEVELOP();
        fColors = new IR8G8B8A8Color[fNumberOfColors];
        memcpy(fColors, src.fColors, ( sizeof(IR8G8B8A8Color) * fNumberOfColors));
        fCellStatus = new EColorCellStatus[fNumberOfColors];
        memcpy(fCellStatus, src.fCellStatus, ( sizeof(EColorCellStatus) * fNumberOfColors));
}

/*------------------------------------------------------------------------------
| IColorMap::~IColorMap                                                        |
------------------------------------------------------------------------------*/
IColorMap::~IColorMap()
{
        IFUNCTRACE_DEVELOP();
        if (fData) {
#if defined(IC_WIN)
                DeleteObject((HPALETTE)fData);
#elif defined(IC_PM)
                GpiDeletePalette((HPAL)fData);
#elif defined(IC_MOTIF)
                if ( this->colorMapType() != kSystem )
                {
                   if (!XFreeColormap(getDisplay(), (Colormap)fData))
                           IGraphicException(" XFreeColormap ");
                }
                else
                {
                   // Free all colors that we allocated.
                   int npixels(0);
                   unsigned long* pixels = new unsigned long [this->fNumberOfColors];
                   for (int i=0; i < this->fNumberOfColors; i++)
                   {
                       if (this->isColorIndexAllocated(i))
                       {
                           pixels[npixels] = i;
                           npixels++;
                       }
                   }
                   if (npixels)
                      XFreeColors(getDisplay(), fData, pixels, npixels, 0);

                   delete [] pixels;
                }
#endif
        }
        delete fColorMapData;
        delete [] fColors;
        delete [] fCellStatus;
}

/*------------------------------------------------------------------------------
| IColorMap::operator=                                                         |
------------------------------------------------------------------------------*/
IColorMap& IColorMap::operator=(const IColorMap& src)
{
        IFUNCTRACE_DEVELOP();
        if (!(this == &src))
        {
                delete [] fColors;

                fNumberOfColors = src.fNumberOfColors;
                fColorMapType  =  src.fColorMapType;
                fColorMapData  =  0;   //Do not copy.  Re-evaluate cache.

                if (fNumberOfColors == 0)
                        fColors = 0 /*NIL*/;
                else
                {
                        fColors = new IR8G8B8A8Color[fNumberOfColors];
                        memcpy(fColors, src.fColors, (sizeof(IR8G8B8A8Color)*fNumberOfColors));
                        fCellStatus = new EColorCellStatus[fNumberOfColors];
                        memcpy(fCellStatus, src.fCellStatus, (sizeof(IColorMap::EColorCellStatus)*fNumberOfColors));
                        fNumberOfFreeCells = src.fNumberOfFreeCells;
                }

                fData = 0 /*NIL*/;      // need reevaluation...
        }
        return *this;
}

/*------------------------------------------------------------------------------
| IColorMap::operator==                                                        |
------------------------------------------------------------------------------*/
bool IColorMap::operator==(const IColorMap& source) const
{
        IFUNCTRACE_ALL();
        if (fNumberOfColors != source.fNumberOfColors)
                return false;

        if (fNumberOfColors == 0)
        {
                return (fColors == 0 /*NIL*/) && (source.fColors == 0 /*NIL*/);
        }

        return !memcmp(fColors, source.fColors, (sizeof(IR8G8B8A8Color)*fNumberOfColors));
}

/*------------------------------------------------------------------------------
| IColorMap::operator[]                                                        |
------------------------------------------------------------------------------*/
const IR8G8B8A8Color& IColorMap::operator[](unsigned long i) const
{
        IGraphicException::InternalAssert(i < fNumberOfColors, "IRGBAColorArray index out of range");
        return fColors[i];
}

/*------------------------------------------------------------------------------
| IColorMap::isColorIndexAllocated                                             |
------------------------------------------------------------------------------*/
bool IColorMap::isColorIndexAllocated(unsigned long index) const
{
    if ((index < fNumberOfColors) &&
        (fCellStatus[index] != kFree) )
        return true;
    else
        return false;
}


/*------------------------------------------------------------------------------
| IColorMap::numberOfColors                                                    |
------------------------------------------------------------------------------*/
unsigned long IColorMap::numberOfColors()const
{
        return fNumberOfColors;
}

/*------------------------------------------------------------------------------
| IColorMap::isSystemColor                                                     |
------------------------------------------------------------------------------*/
bool IColorMap::isSystemColor(unsigned long i) const
{
   return  (((i < fNumberOfColors) && (fCellStatus[i]==kReserved)) ? true : false);
}

/*------------------------------------------------------------------------------
| IColorMap::EColorMapType IColorMap::colorMapType                             |
------------------------------------------------------------------------------*/
IColorMap::EColorMapType IColorMap::colorMapType() const
{
        return(fColorMapType);
}

/*------------------------------------------------------------------------------
| IColorMap::setColorMapType                                                   |
------------------------------------------------------------------------------*/
void IColorMap::setColorMapType(EColorMapType cmapType)
{
        fColorMapType = cmapType;
}

static bool bDefautSystemColorMap_Init = false;
static bool bTrueColorMap_Init = false;
static bool bWebSafeColorMap_Init = false;
IColorMap* IColorMap::DefaultSystemColorMap = NULL;
IColorMap* IColorMap::TrueColorMap = NULL;
IColorMap* IColorMap::WebSafeColorMap = NULL;

/*------------------------------------------------------------------------------
| IColorMap::defaultSystemColorMap                                             |
------------------------------------------------------------------------------*/
IColorMap& IColorMap::defaultSystemColorMap()
{
   IFUNCTRACE_ALL();
   if (!bDefautSystemColorMap_Init)
   {
      IPrimalLock lock;
      if (!bDefautSystemColorMap_Init)
      {
          IMODTRACE_DEVELOP("initializing DefaultSystemColorMap");
          DefaultSystemColorMap = new IColorMap(kSystem);
          bDefautSystemColorMap_Init = true;
      }
   }
   return(*DefaultSystemColorMap);
}

/*------------------------------------------------------------------------------
| IColorMap::trueColorMap                                                      |
------------------------------------------------------------------------------*/
IColorMap& IColorMap::trueColorMap()
{
   IFUNCTRACE_ALL();
   if (!bTrueColorMap_Init)
   {
      IPrimalLock lock;
      if (!bTrueColorMap_Init)
      {
          TrueColorMap = new IColorMap();
          TrueColorMap->fColorMapType=kTrueColor;
          bTrueColorMap_Init = true;
      }
   }
   return(*TrueColorMap);
}


static bool bDefaultColorMap_Init = false;
IColorMap* IColorMap::fDefaultColorMap = NULL;
/*------------------------------------------------------------------------------
| IColorMap::setDefaultColorMap                                                |
------------------------------------------------------------------------------*/
void IColorMap::setDefaultColorMap(IColorMap *colormapToSet)
{
        IFUNCTRACE_DEVELOP();
        fDefaultColorMap = colormapToSet;
		bDefaultColorMap_Init = true;
}

/*------------------------------------------------------------------------------
| IColorMap::defaultColorMap                                                   |
------------------------------------------------------------------------------*/
IColorMap& IColorMap::defaultColorMap()
{
        IFUNCTRACE_DEVELOP();
        if (!bDefaultColorMap_Init) {
            IPrimalLock lock;
            if(!bDefaultColorMap_Init){
            	if(canChangeColorMap()){
                    //call and init the websafe color palette
                    webSafeColorMap();
                    //use websafecolmap as default
                    fDefaultColorMap = WebSafeColorMap;
				} else {
                    fDefaultColorMap = new IColorMap(kSystem);
				}
                bDefaultColorMap_Init = true;
            }
        }
        return(*fDefaultColorMap);
}

static bool bColorMapSupport_Init = false;
static bool bColorMapSupport = false;
static bool bColorMapReadOnly = false;
/*------------------------------------------------------------------------------
| IColorMap::canChangeColorMap                                                |
------------------------------------------------------------------------------*/
bool IColorMap::canChangeColorMap()
{
	IColorMap::hasColorMapSupport();
	return(!bColorMapReadOnly);
}
/*------------------------------------------------------------------------------
| IColorMap::hasColorMapSupport                                                |
------------------------------------------------------------------------------*/
bool IColorMap::hasColorMapSupport()
{
       if (!bColorMapSupport_Init) {
       IPrimalLock lock;
       if (!bColorMapSupport_Init) {
		   bColorMapSupport_Init = true;
               bColorMapSupport = false;
               bColorMapReadOnly = true;
#if defined (IC_WIN)
               HDC sys = GetDC(0);
               if((GetDeviceCaps(sys, RASTERCAPS) & RC_PALETTE) == RC_PALETTE){
                       bColorMapSupport = true;
                       bColorMapReadOnly = false;
               }
               ReleaseDC(0, sys);
#elif defined (IC_PM)
		//For the defect 28080 the palette has problems in OS2
		//With user defined palette. The palette support was added in
                //AIX and OS2 for better image view and to maintain consistency
                //accross out app. But for some reason OS2 always behaved
                //properly for any given image, but to maintain consistency
                //accross platforms palette support was added to OS2 too.
                //Since we some problems now due to palette being installed
                //I after talikng to Robert Cecco have decided to remove
                //palette support in OS2. If in future support neaded then
                //enable the code in "if 0".
#if 0
               HPS hps = WinGetPS(HWND_DESKTOP);
               HDC hdc = GpiQueryDevice( hps );
               long lPalSupport;

               DevQueryCaps(hdc, CAPS_ADDITIONAL_GRAPHICS, 1, &lPalSupport);
               if ((lPalSupport & CAPS_PALETTE_MANAGER) == CAPS_PALETTE_MANAGER){
                       DevQueryCaps(hdc,CAPS_COLORS, 1L, &lPalSupport);
                       if (lPalSupport <= 256 ){
                               bColorMapSupport = true;
                               bColorMapReadOnly = false;
                       }
               }
               WinReleasePS(hps);
#else
               bColorMapSupport = false;
               bColorMapReadOnly = true;
#endif
#elif defined (IC_MOTIF)
               Display *dpy = getDisplay();
               int defDepth = DefaultDepth(dpy, DefaultScreen(dpy));
               if(defDepth <= 8){
                    bColorMapSupport = true;
					Visual* visual = DefaultVisual(dpy, DefaultScreen(dpy));
					if ( (visual->c_class == PseudoColor) ||
					  (visual->c_class == DirectColor) ||
					  (visual->c_class == GrayScale))
						bColorMapReadOnly = false;
				}
#endif
       }
       }
       return bColorMapSupport;
}

#define TOTAL_STATIC_COLORS     264
IR8G8B8A8Color StaticColorTable[TOTAL_STATIC_COLORS] =
{
        {0 , 0 , 0, 0},         //Black
        {255 , 255 , 255, 0},   //White
        {255 , 0 , 0, 0},               //Red
        {0 , 255 , 0, 0},               //Green
        {0 , 0 , 255, 0},               //Blue
        {255 , 0 , 255, 0},     //Pink
        {0 , 255 , 255, 0},     //Cyan
        {255 , 255 , 0, 0},     //Yellow
        {128 , 0 , 0, 0},               //Dark Red
		{128 , 128 , 0},	//Dark Brown
#ifdef IC_WIN
        {0 , 0 , 128, 0},               //Dark Blue
        {128 , 128 , 128, 0},   //Dark Gray
        {128 , 0 , 128, 0},     //Dark Pink
        {0 , 128, 0, 0},                //Dark Green
        {0 , 128, 128, 0},              //Dark Cyan
        {192 , 192, 192, 0},    //Pale Gray
#else
        {0 , 0 , 170, 0},               //Dark Blue
        {131 , 131 , 131, 0},   //Dark Gray
        {170 , 0 , 170, 0},     //Dark Pink
        {0 , 146, 0, 0},                //Dark Green
        {0 , 146, 170, 0},              //Dark Cyan
        {204 , 204, 204, 0},    //Pale Gray
#endif
        {0 , 0 , 51, 0},
        {0 , 0 , 102, 0},
        {0 , 0 , 153, 0},
        {0 , 0 , 204, 0},
        {0 , 51 , 0, 0},
        {0 , 51 , 51, 0},
        {0 , 51 , 102, 0},
        {0 , 51 , 153, 0},
        {0 , 51 , 204, 0},
        {0 , 51 , 255, 0},
        {0 , 102 , 0, 0},
        {0 , 102 , 51, 0},
        {0 , 102 , 102, 0},
        {0 , 102 , 153, 0},
        {0 , 102 , 204, 0},
        {0 , 102 , 255, 0},
        {0 , 153 , 0, 0},
        {0 , 153 , 51, 0},
        {0 , 153 , 102, 0},
        {0 , 153 , 153, 0},
        {0 , 153 , 204, 0},
        {0 , 153 , 255, 0},
        {0 , 204 , 0, 0},
        {0 , 204 , 51, 0},
        {0 , 204 , 102, 0},
        {0 , 204 , 153, 0},
        {0 , 204 , 204, 0},
        {0 , 204 , 255, 0},
        {0 , 255 , 51, 0},
        {0 , 255 , 102, 0},
        {0 , 255 , 153, 0},
        {0 , 255 , 204, 0},
        {51 , 0 , 0, 0},
        {51 , 0 , 51, 0},
        {51 , 0 , 102, 0},
        {51 , 0 , 153, 0},
        {51 , 0 , 204, 0},
        {51 , 0 , 255, 0},
        {51 , 51 , 0, 0},
        {51 , 51 , 51, 0},
        {51 , 51 , 102, 0},
        {51 , 51 , 153, 0},
        {51 , 51 , 204, 0},
        {51 , 51 , 255, 0},
        {51 , 102 , 0, 0},
        {51 , 102 , 51, 0},
        {51 , 102 , 102, 0},
        {51 , 102 , 153, 0},
        {51 , 102 , 204, 0},
        {51 , 102 , 255, 0},
        {51 , 153 , 0, 0},
        {51 , 153 , 51, 0},
        {51 , 153 , 102, 0},
        {51 , 153 , 153, 0},
        {51 , 153 , 204, 0},
        {51 , 153 , 255, 0},
        {51 , 204 , 0, 0},
        {51 , 204 , 51, 0},
        {51 , 204 , 102, 0},
        {51 , 204 , 153, 0},
        {51 , 204 , 204, 0},
        {51 , 204 , 255, 0},
        {51 , 255 , 0, 0},
        {51 , 255 , 51, 0},
        {51 , 255 , 102, 0},
        {51 , 255 , 153, 0},
        {51 , 255 , 204, 0},
        {51 , 255 , 255, 0},
        {102 , 0 , 0, 0},
        {102 , 0 , 51, 0},
        {102 , 0 , 102, 0},
        {102 , 0 , 153, 0},
        {102 , 0 , 204, 0},
        {102 , 0 , 255, 0},
        {102 , 51 , 0, 0},
        {102 , 51 , 51, 0},
        {102 , 51 , 102, 0},
        {102 , 51 , 153, 0},
        {102 , 51 , 204, 0},
        {102 , 51 , 255, 0},
        {102 , 102 , 0, 0},
        {102 , 102 , 51, 0},
        {102 , 102 , 102, 0},
        {102 , 102 , 153, 0},
        {102 , 102 , 204, 0},
        {102 , 102 , 255, 0},
        {102 , 153 , 0, 0},
        {102 , 153 , 51, 0},
        {102 , 153 , 102, 0},
        {102 , 153 , 153, 0},
        {102 , 153 , 204, 0},
        {102 , 153 , 255, 0},
        {102 , 204 , 0, 0},
        {102 , 204 , 51, 0},
        {102 , 204 , 102, 0},
        {102 , 204 , 153, 0},
        {102 , 204 , 204, 0},
        {102 , 204 , 255, 0},
        {102 , 255 , 0, 0},
        {102 , 255 , 51, 0},
        {102 , 255 , 102, 0},
        {102 , 255 , 153, 0},
        {102 , 255 , 204, 0},
        {102 , 255 , 255, 0},
        {153 , 0 , 0, 0},
        {153 , 0 , 51, 0},
        {153 , 0 , 102, 0},
        {153 , 0 , 153, 0},
        {153 , 0 , 204, 0},
        {153 , 0 , 255, 0},
        {153 , 51 , 0, 0},
        {153 , 51 , 51, 0},
        {153 , 51 , 102, 0},
        {153 , 51 , 153, 0},
        {153 , 51 , 204, 0},
        {153 , 51 , 255, 0},
        {153 , 102 , 0, 0},
        {153 , 102 , 51, 0},
        {153 , 102 , 102, 0},
        {153 , 102 , 153, 0},
        {153 , 102 , 204, 0},
        {153 , 102 , 255, 0},
        {153 , 153 , 0, 0},
        {153 , 153 , 51, 0},
        {153 , 153 , 102, 0},
        {153 , 153 , 153, 0},
        {153 , 153 , 204, 0},
        {153 , 153 , 255, 0},
        {153 , 204 , 0, 0},
        {153 , 204 , 51, 0},
        {153 , 204 , 102, 0},
        {153 , 204 , 153, 0},
        {153 , 204 , 204, 0},
        {153 , 204 , 255, 0},
        {153 , 255 , 0, 0},
        {153 , 255 , 51, 0},
        {153 , 255 , 102, 0},
        {153 , 255 , 153, 0},
        {153 , 255 , 204, 0},
        {153 , 255 , 255, 0},
        {204 , 0 , 0, 0},
        {204 , 0 , 51, 0},
        {204 , 0 , 102, 0},
        {204 , 0 , 153, 0},
        {204 , 0 , 204, 0},
        {204 , 0 , 255, 0},
        {204 , 51 , 0, 0},
        {204 , 51 , 51, 0},
        {204 , 51 , 102, 0},
        {204 , 51 , 153, 0},
        {204 , 51 , 204, 0},
        {204 , 51 , 255, 0},
        {204 , 102 , 0, 0},
        {204 , 102 , 51, 0},
        {204 , 102 , 102, 0},
        {204 , 102 , 153, 0},
        {204 , 102 , 204, 0},
        {204 , 102 , 255, 0},
        {204 , 153 , 0, 0},
        {204 , 153 , 51, 0},
        {204 , 153 , 102, 0},
        {204 , 153 , 153, 0},
        {204 , 153 , 204, 0},
        {204 , 153 , 255, 0},
        {204 , 204 , 0, 0},
        {204 , 204 , 51, 0},
        {204 , 204 , 102, 0},
        {204 , 204 , 153, 0},
        {204 , 204 , 204, 0},
        {204 , 204 , 255, 0},
        {204 , 255 , 0, 0},
        {204 , 255 , 51, 0},
        {204 , 255 , 102, 0},
        {204 , 255 , 153, 0},
        {204 , 255 , 204, 0},
        {204 , 255 , 255, 0},
        {255 , 0 , 51, 0},
        {255 , 0 , 102, 0},
        {255 , 0 , 153, 0},
        {255 , 0 , 204, 0},
        {255 , 51 , 0, 0},
        {255 , 51 , 51, 0},
        {255 , 51 , 102, 0},
        {255 , 51 , 153, 0},
        {255 , 51 , 204, 0},
        {255 , 51 , 255, 0},
        {255 , 102 , 0, 0},
        {255 , 102 , 51, 0},
        {255 , 102 , 102, 0},
        {255 , 102 , 153, 0},
        {255 , 102 , 204, 0},
        {255 , 102 , 255, 0},
        {255 , 153 , 0, 0},
        {255 , 153 , 51, 0},
        {255 , 153 , 102, 0},
        {255 , 153 , 153, 0},
        {255 , 153 , 204, 0},
        {255 , 153 , 255, 0},
        {255 , 204 , 0, 0},
        {255 , 204 , 51, 0},
        {255 , 204 , 102, 0},
        {255 , 204 , 153, 0},
        {255 , 204 , 204, 0},
        {255 , 204 , 255, 0},
        {255 , 255 , 51, 0},
        {255 , 255 , 102, 0},
        {255 , 255 , 153, 0},
        {255 , 255 , 204, 0},
        {17 , 17 , 17, 0},
        {34 , 34 , 34, 0},
        {68 , 68 , 68, 0},
        {85 , 85 , 85, 0},
        {119 , 119 , 119, 0},
        {136 , 136 , 136, 0},
        {170 , 170 , 170, 0},
        {187 , 187 , 187, 0},
        {221 , 221 , 221, 0},
        {238 , 238 , 238, 0},
        {128 , 77 , 77, 0},
        {128 , 128 , 77, 0},
        {77 , 128 , 77, 0},
        {77 , 128 , 128, 0},
        {77 , 77 , 128, 0},
        {128 , 77 , 128, 0},
        {179 , 128 , 128, 0},
        {179 , 179 , 128, 0},
        {128 , 179 , 128, 0},
        {128 , 179 , 179, 0},
        {128 , 128 , 179, 0},
        {179 , 128 , 179, 0},
        {230 , 179 , 179, 0},
        {230 , 230 , 179, 0},
        {179 , 230 , 179, 0},
        {179 , 230 , 230, 0},
        {179 , 179 , 230, 0},
        {230 , 179 , 230, 0},
        {230 , 212 , 212, 0},
        {230 , 230 , 212, 0},
        {212 , 230 , 212, 0},
        {212 , 230 , 230, 0},
        {212 , 212 , 230, 0},
        {230 , 212 , 230, 0},
        {255 , 242 , 242, 0},
        {255 , 255 , 242, 0},
        {242 , 255 , 242, 0},
        {242 , 255 , 255, 0},
        {242 , 242 , 255, 0},
        {255 , 242 , 255, 0}
};

/*------------------------------------------------------------------------------
| IColorMap::releaseCells                                                      |
| returns number of Cells released                                             |
------------------------------------------------------------------------------*/
unsigned long IColorMap::releaseCells()
{
        unsigned long i, j=0;
        for(i=0;i<fNumberOfColors; i++){
                if(fCellStatus[i] != kReserved){
                        fCellStatus[i]=kFree;
                        ++j;
                       ++fNumberOfFreeCells; //increment the availability
                }
        }
        return j;
}

/*------------------------------------------------------------------------------
| IColorMap::EColorCellStatus IColorMap::colorExists                           |
------------------------------------------------------------------------------*/
IColorMap::EColorCellStatus IColorMap::colorExists(int r, int g, int b, int *index)
{
        int min=999; //max value
        int i;
        int diff;
        *index = -1; //set to default not found

        for (i = 0; i < fNumberOfColors; i++)
        {
                if (fCellStatus[i] != kFree)
                {
                   diff =
                           ABS(fColors[i].fRed - r) +
                           ABS(fColors[i].fGreen - g) +
                           ABS(fColors[i].fBlue - b);

                   if (diff == 0){ //nearest match
                           *index = i; //exists and its index
                           return fCellStatus[i];
                   }

                   if(min > diff){
                           min = diff;
                           *index = i;
                   }
                }
        }
        return(kNotFound); //doesn't exist
}

/*------------------------------------------------------------------------------
| IColorMap::insertColor                                                       |
------------------------------------------------------------------------------*/
int IColorMap::insertColor(int r, int g, int b, int opacity)
{
#ifdef IC_MOTIF
   if (fData)
   {
        // Have to use XAllocColor because the colormap is already realized.
        XColor xcolor;
        xcolor.red   = r << 8;
        xcolor.green = g << 8;
        xcolor.blue  = b << 8;
        xcolor.flags = DoRed|DoGreen|DoBlue;
        if (XAllocColor( getDisplay(), fData, &xcolor))
        {
           ITRACE_ALL(IString("XAllocColor pixel=")+ IString(xcolor.pixel));
           fColors[xcolor.pixel].fRed   = r;
           fColors[xcolor.pixel].fGreen = g;
           fColors[xcolor.pixel].fBlue  = b;
           fColors[xcolor.pixel].fOpacity  = opacity;
           --fNumberOfFreeCells; //decrement the availability
           fCellStatus[xcolor.pixel]=kUsed;
           return xcolor.pixel;
        }
   }
   else
#endif
   {
        int i;
        for (i = 0; i < fNumberOfColors; i++)
        {
                if(fCellStatus[i] == kFree){
                        fColors[i].fRed = r;
                        fColors[i].fGreen = g;
                        fColors[i].fBlue = b;
                        fColors[i].fOpacity = opacity;

                        --fNumberOfFreeCells; //decrement the avilability
                        fCellStatus[i]=kUsed;
                        return i;
                }
        }
    }
    return -1; //should never come here
}

/*------------------------------------------------------------------------------
| IColorMap::mapColors                                                         |
------------------------------------------------------------------------------*/
unsigned long IColorMap::mapColors(unsigned long size, const IR8G8B8A8Color* colorToMap)
{
        IFUNCTRACE_ALL();
        int i, index, cellsMapped=0;
        EColorCellStatus err;

        if(!fNumberOfFreeCells)
                return 0; //all cells filled

        for (i = 0; (i < size) && (fNumberOfFreeCells); i++)
        {
                err=colorExists( colorToMap[i].fRed,
                                        colorToMap[i].fGreen,
                                        colorToMap[i].fBlue, &index);
                if(err == kNotFound){ //not in the list
                        index = insertColor(colorToMap[i].fRed,
                                        colorToMap[i].fGreen,
                                        colorToMap[i].fBlue,
                                        0xff);
                        ++cellsMapped;
                } else if(err == kFree){ //in the list as a part of defaults
                        fCellStatus[index] = kUsed;
                        --fNumberOfFreeCells;
                        ++cellsMapped;
                }
        }
#if 0
        for (i = 0; i < fNumberOfColors; i++)
        fprintf(stderr,"%d = %d %d %d\n",i,
                                fColors[i].fRed,
                                fColors[i].fGreen,
                                fColors[i].fBlue);
#endif

        return cellsMapped;
}

/*------------------------------------------------------------------------------
| IColorMap::webSafeColorMap                                                   |
------------------------------------------------------------------------------*/
IColorMap& IColorMap::webSafeColorMap()
{
    IFUNCTRACE_ALL();
    if (!bWebSafeColorMap_Init)
    {
       IColorMap& systemColorMap( defaultSystemColorMap() );
       IPrimalLock lock;
       if (!bWebSafeColorMap_Init)
       {
           bWebSafeColorMap_Init = true;
           // Copy the system default color map and then add colors to it.
           WebSafeColorMap = new IColorMap( systemColorMap );
           WebSafeColorMap->setColorMapType ( kUserDefined );
           WebSafeColorMap->mapColors(TOTAL_STATIC_COLORS, StaticColorTable);
       }
    }
    return *WebSafeColorMap;
}

/*------------------------------------------------------------------------------
| IColorMap::systemColorRGB                                                    |
------------------------------------------------------------------------------*/
unsigned long IColorMap::systemColorRGB(const long index)
{
        unsigned long rgb=0;
#if defined(IC_WIN)
        HDC hdc = GetDC(0);
        if (IColorMap::hasColorMapSupport()) {
                PALETTEENTRY pal;
                GetSystemPaletteEntries(hdc, index, 1, &pal);
                rgb = RGB(pal.peRed, pal.peGreen, pal.peBlue);
        } else {
                rgb = index;
        }
        ReleaseDC(0, hdc);
#elif defined(IC_PM)
        HPS hps = WinGetPS(HWND_DESKTOP);
        HDC hdc = GpiQueryDevice( hps );
        long capsClr;

        //check if there's palette support
        DevQueryCaps(hdc,CAPS_COLORS, 1L, &capsClr);
       if (capsClr <= 256 ) {
                GpiQueryRealColors(hps, 0L, index, 1, (PLONG)&rgb);
        } else {
                rgb = index;
        }
        WinReleasePS(hps);
#elif defined(IC_MOTIF)
        if (IColorMap::hasColorMapSupport()) {
                IColorMap* defColorMap = &(IColorMap::defaultColorMap());
                int total_count = defColorMap->numberOfColors();
                IR8G8B8A8Color *defClrData=defColorMap->fColors;

                if(index < total_count)
                        rgb = (defClrData[index].fRed <<16) |
                                (defClrData[index].fGreen <<8) |
                                (defClrData[index].fBlue);
        } else {
                rgb = index;
        }
#endif
        return (rgb);
}

#ifdef IC_WIN
typedef struct _sysdefaclrs{
COLORREF clr;
long clrIndex;
} SysDefaClrs;
SysDefaClrs sysclrs[16] = {
        {0x00FFFFFF, 255}, //kWhite
        {0x00FF0000, 252}, //kBlue
        {0x000000FF, 249}, //kRed,
        {0x00FF00FF, 253}, //kPink
        {0x0000FF00, 250}, //kGreen
        {0x00FFFF00, 254}, //kCyan
        {0x0000FFFF, 251}, //kYellow
        {0x00000000, 0}, //kBlack
        {0x00808080, 248}, //kDarkGray
        {0x00800000, 4}, //kDarkBlue
        {0x00000080, 1}, //kDarkRed
        {0x00800080, 5}, //kDarkPink
        {0x00008000, 2}, //kDarkGreen
        {0x00808000, 6}, //kDarkCyan
        {0x00008080, 3}, //kBrown
        {0x00C0C0C0, 7} //kPaleGray
};
#endif //IC_WIN


/*------------------------------------------------------------------------------
| IColorMap::systemColorIndex                                                  |
------------------------------------------------------------------------------*/
unsigned long IColorMap::systemColorIndex(const IBaseColor& p)
{
        unsigned long i, index = 0;
#if defined(IC_WIN)
        COLORREF clrf = RGB(p.redMix(), p.greenMix(), p.blueMix());
        HDC hdc = GetDC(0);
        index = clrf;
        if (IColorMap::hasColorMapSupport()) {
                for(i=0; i<16; i++){
                        if((clrf & 0x00FFFFFF) == sysclrs[i].clr){
                                index = sysclrs[i].clrIndex;
                                return (index);
                        }
                }
                HPALETTE newpal, oldpal;
                oldpal = (HPALETTE) SelectObject( hdc, newpal );
                index = GetNearestPaletteIndex(oldpal, clrf);
                SelectObject(hdc, oldpal);
        }
        ReleaseDC(0, hdc);
        return (index);
#elif defined(IC_PM)
        HDC hdc = GetDC(0);
        unsigned long clrf = RGB(p.redMix(), p.greenMix(), p.blueMix());
        index = GpiQueryColorIndex(hdc, 0, clrf);
        ReleaseDC(0, hdc);
        return (index);
#elif defined(IC_MOTIF)
        if (IColorMap::hasColorMapSupport()) {
                index = IColorMap::defaultColorMap().colorIndex( p );
        } else {
                index = IXDisplay::rgbToPixel(p.redMix(), p.greenMix(), p.blueMix());
        }
        return (index);
#endif
}


/*------------------------------------------------------------------------------
| IColorMap::colorIndex                                                        |
------------------------------------------------------------------------------*/
unsigned long IColorMap::colorIndex(const IBaseColor& p) const
{
    unsigned long index(kINVALIDINDEX);

    CharIntensity red = p.redMix();
    CharIntensity green = p.greenMix();
    CharIntensity blue = p.blueMix();

    unsigned long requestedRGB = (red << 16) + (green << 8) + blue;

    // First, look in the cache.
    if (fColorMapData)
    {
       index = fColorMapData->colorIndex( requestedRGB );
    }
    else if (fData)
    {  // Colormap realized...create a color lookup cache.
       ((IColorMap*)this)->fColorMapData = new IColorMapData();
    }

    if (index == kINVALIDINDEX)
    {
       // Not in cache.  Search table for closest match.
       unsigned long diff(256*3);
       unsigned long i;
       for ( i = 0; (i < fNumberOfColors) && (diff > 0); i++)
       {
           // Find closeness of current color and compare to best match
           // so far.   If better, set result to current color.
           unsigned long tmp =
                   ABS(red - fColors[i].fRed) +
                   ABS(green - fColors[i].fGreen) +
                   ABS(blue - fColors[i].fBlue);
           if (diff > tmp)
           {
               diff = tmp;
               index = i;
           }
       }

       // Now, index contains the matched colortable index, diff contains
       // the closeness (0 if exact), and requestedRGB is the requested
       // color.  Add rgb/index pair to the cache for next time.
       if (fColorMapData)
          fColorMapData->add( requestedRGB, index, diff ? false : true);
    }
    return index; //closest match
}


/*------------------------------------------------------------------------------
| IColorMap::nativeColorMap                                                    |
------------------------------------------------------------------------------*/
unsigned long IColorMap::nativeColorMap()
{
    IFUNCTRACE_ALL();
    if ( (!fData) && IColorMap::hasColorMapSupport() )
    {
        unsigned long entries = fNumberOfColors;

#if defined(IC_WIN)

        LOGPALETTE* pal = (LOGPALETTE*)new char[2 * sizeof(WORD) + entries * sizeof(PALETTEENTRY)];
        pal->palVersion = 0x300;
        pal->palNumEntries = entries;

        for (long i = 0; i < entries; i++)
        {
                pal->palPalEntry[i].peRed = fColors[i].fRed;
                pal->palPalEntry[i].peGreen = fColors[i].fGreen;
                pal->palPalEntry[i].peBlue = fColors[i].fBlue;
                // windows does not do alpha, but use the byte to set some flags...
                pal->palPalEntry[i].peFlags = 0;  // default flag
        }
        fData = (unsigned long)CreatePalette(pal);
        delete pal;

#elif defined(IC_PM)

        HWND  hwnd  = HWND_DESKTOP;
        HAB   hab   = WinQueryAnchorBlock(hwnd);
        HPS   hps   = WinGetPS(hwnd);
        HDC   hdc   = GpiQueryDevice(hps);
        RGB2  *tColorData = new RGB2[entries];

        //max. no. of colors supported
        DevQueryCaps(hdc, CAPS_PHYS_COLORS, 1, (long*)&entries);

        for (unsigned long i=0; i< entries; i++) {
                tColorData[i].bRed = fColors[i].fRed;
                tColorData[i].bGreen = fColors[i].fGreen;
                tColorData[i].bBlue = fColors[i].fBlue;
                tColorData[i].fcOptions = 0L;
        }

       fData = GpiCreatePalette(hab, LCOL_PURECOLOR, LCOLF_CONSECRGB, entries, (PULONG)tColorData);

        WinReleasePS(hps);
        delete [] tColorData;

#elif defined(IC_MOTIF)

        if (canChangeColorMap())
        {
           Display *dpy =getDisplay();
           Window win = DefaultRootWindow(dpy);

           XWindowAttributes wattr;
           if(!XGetWindowAttributes(dpy, win, &wattr))
                IGraphicException(" XGetWindowAttributes ");

           unsigned long i;
           // Create a new color map.
           fData = (unsigned long)XCreateColormap(dpy,
                                                  win,
                                                  wattr.visual,
                                                  AllocNone);

           // Prepare a new color and status array which will be
           // initialized based on XAllocColor results.
           IR8G8B8A8Color*   newColors = new IR8G8B8A8Color[entries];
           EColorCellStatus* newStatus = new EColorCellStatus[entries];
           for (i=0; i<entries; i++)
              newStatus[i] = kFree;

           unsigned long     newFreeCells = entries;
           // Allocate the colors present in the color table.  For typical
           // entries we use XAllocColor so that sharable entries are created.
           // This causes XAllocColor to succeed when other components
           // (notably Motif) use it.  For cells marked with kReserved, we
           // use XAllocColorCells.
           #ifdef IC_TRACE_DEVELOP
              int traceAllocColor(0), traceAllocColorCells(0);
           #endif
           for (i=0; i<entries; i++)
           {
              if (fCellStatus[i] != kFree)
              {
                  XColor xcolor;
                  xcolor.red   = fColors[i].fRed << 8;
                  xcolor.green = fColors[i].fGreen << 8;
                  xcolor.blue  = fColors[i].fBlue << 8;
                  xcolor.flags = DoRed|DoGreen|DoBlue;
                  if (fCellStatus[i] == kReserved)
                  {
                     ITRACE_ALL("XAllocColorCells");
                     #ifdef IC_TRACE_DEVELOP
                        traceAllocColorCells++;
                     #endif
                     unsigned long pmr[1];
                     unsigned long pixel[1];
                     if (!XAllocColorCells(dpy, fData, True, pmr, 0, pixel, 1))
                         IGraphicException(" XAllocColorCells ");

                     xcolor.pixel = pixel[0];
                     XStoreColor( dpy, fData, &xcolor);
                  }
                  else
                  {
                     ITRACE_ALL("XAllocColor");
                     #ifdef IC_TRACE_DEVELOP
                        traceAllocColor++;
                     #endif
                     if (!XAllocColor( dpy, fData, &xcolor) )
                         IGraphicException(" XAllocColor ");
                  }
                  #ifdef IC_TRACE_ALL
                    unsigned long rgb = (fColors[i].fRed << 16) |
                                        (fColors[i].fGreen << 8) |
                                         fColors[i].fBlue;
                    ITRACE_ALL( IString("pixel=") +
                                IString(xcolor.pixel) +
                                IString(" i=") + IString(i) +
                                IString(" rgb=") + IString(rgb).d2x() );
                  #endif
                  newColors[xcolor.pixel] = fColors[i];
                  newStatus[xcolor.pixel] = fCellStatus[i];
                  newFreeCells--;
              }  // if not free
           }  // for

           // Now, replace the color and status arrays of the IColorMap
           // with the new lists just generated.
           delete [] fColors;
           delete [] fCellStatus;
           fColors = newColors;
           fCellStatus = newStatus;
           fNumberOfFreeCells = newFreeCells;

           XFlush(dpy);
           ITRACE_DEVELOP(
              IString("new colormap fData=") + IString(fData).d2x() +
              IString(" XAllocColor=") + IString(traceAllocColor) +
              IString(" XAllocColorCells=") + IString(traceAllocColorCells)  );
        }
#endif //IC_MOTIF
    }
    return fData;
}

/*------------------------------------------------------------------------------
| IColorMap::operator>>=                                                       |
------------------------------------------------------------------------------*/
IDataStream& IColorMap::operator>>=(IDataStream& toWhere) const
{
        IStreamOutFrame streamFrame(toWhere);
        long temp;

        register unsigned long i;
        fNumberOfColors >>= toWhere; //# of colors
        fNumberOfFreeCells >>= toWhere; //# of free colors cells
        temp = fColorMapType;
        temp >>= toWhere;

        long numberColors = fNumberOfColors;
        for (i = 0; i < numberColors; i++)
        {
                fColors[i].fRed >>= toWhere;
                fColors[i].fGreen >>= toWhere;
                fColors[i].fBlue >>= toWhere;
                fColors[i].fOpacity >>= toWhere;
                temp = fCellStatus[i];
                temp >>= toWhere;
        }
        return toWhere;
}

/*------------------------------------------------------------------------------
| IColorMap::operator<<=                                                       |
------------------------------------------------------------------------------*/
IDataStream& IColorMap::operator<<=(IDataStream& fromWhere)
{
        IStreamInFrame streamFrame(fromWhere);
        long temp;

        delete [] fColors;

        fNumberOfColors <<= fromWhere; //# of colors
        fNumberOfFreeCells <<= fromWhere; //# of free color cells
        temp <<= fromWhere;
        fColorMapType = (IColorMap::EColorMapType)temp;

        fColors = (fNumberOfColors) ? new IR8G8B8A8Color[fNumberOfColors] : 0 /*NIL*/;
        fCellStatus = (fNumberOfColors) ? new EColorCellStatus[fNumberOfColors] : 0 /*NIL*/;

        for (long i = 0; i < fNumberOfColors; i++)
        {
                fColors[i].fRed <<= fromWhere;
                fColors[i].fGreen <<= fromWhere;
                fColors[i].fBlue <<= fromWhere;
                fColors[i].fOpacity <<= fromWhere;
                temp <<= fromWhere;
                fCellStatus[i] = (IColorMap::EColorCellStatus)temp;
        }
        return fromWhere;
}

/*------------------------------------------------------------------------------
| IColorMapData::IColorMapData                                                 |
------------------------------------------------------------------------------*/
IColorMapData::IColorMapData(unsigned long size)
  : fMaxSize     ( size ),
    fCache       ( size ),   // for sorted set
    fCursor      ( fCache )
{
   IFUNCTRACE_DEVELOP();
   IASSERTSTATE( size > 0 );
}

/*------------------------------------------------------------------------------
| IColorMapData::~IColorMapData                                                |
------------------------------------------------------------------------------*/
IColorMapData::~IColorMapData( )
{
   IFUNCTRACE_DEVELOP();
}

/*------------------------------------------------------------------------------
| IColorMapData::add                                                           |
| Add color to cache.  RGB is 00rrggbb, index is colormap index.  Exactmatch   |
| should be true if it is an exact match, false if a closest match.            |
------------------------------------------------------------------------------*/
void IColorMapData::add( unsigned long rgb,
                         unsigned long index,
                         bool          exactMatch )
{
   // Initialize new record for cache.  Set the hit count for exact matches
   // to a larger value so they tend to stay in the cache.
   IColorMapCacheRecord element( rgb,
                                 index,
                                 exactMatch ? 2 : 0);

   // Keep the cache from getting too big.  If the maximum number of elements
   // have already been added, look for some low use ones to get rid of.
   unsigned long threshold = 1;
   while (fCache.numberOfElements() >= fMaxSize)
   {
      ITRACE_DEVELOP("flushing IColorMapCache");
      // Make a copy of the cache so that we can remove elements from the
      // original but maintain a cursor for iteration.
      IColorMapCache tempCache( fCache );
      IColorMapCache::Cursor cursor(tempCache);
      for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
      {
         // Remove elements with low usage.
         const IColorMapCacheRecord& tempElement( tempCache.elementAt( cursor ) );
         if (tempElement.fHitCount <= threshold)
            fCache.removeElementWithKey( tempElement.fRGB );
      }
      threshold <<= 1;    //Double threshold for next pass if needed.
   }
   fCache.add( element );
}

/*------------------------------------------------------------------------------
| IColorMapData::remove                                                        |
| Remove color from cache.  RGB is 00rrggbb.                                   |
------------------------------------------------------------------------------*/
void IColorMapData::remove     ( unsigned long rgb )
{
   fCache.removeElementWithKey( rgb );
}

/*------------------------------------------------------------------------------
| IColorMapData::colorIndex                                                    |
| Returns index of color with given RGB value.  Returns unsigned long(-1) if
| not found.
------------------------------------------------------------------------------*/
unsigned long  IColorMapData::colorIndex ( unsigned long rgb )
{
#ifdef IC_TRACE_ALL
   static unsigned long hits(0), misses(0);
   misses++;
#endif
   unsigned long index(kINVALIDINDEX);
//  IColorMapCache::Cursor cursor( fCache );
//   Make the cursor a member of the object to avoid constructing/destructing
//   it many times.  The cost of this exceeds the cost of the actual
//   lookup.
   if (fCache.locateElementWithKey( rgb, fCursor ) )
   {
       IColorMapCacheRecord& element (fCache.elementAt(fCursor) );
       index = element.fIndex;
       if (element.fHitCount < kMAXCACHEHITCOUNT)
          element.fHitCount++;
       #ifdef IC_TRACE_ALL
          hits++;
          misses--;
       #endif
   }
   ITRACE_ALL(IString("IColorMapData cache hits=") + IString(hits) +
              IString(" misses=") + IString(misses) );
   return index;
}


#if (IC_OBSOLETE <= IC_OBSOLETE_3)
//The class IColorMapContext was inadvertently made public
//in the portapak release.  Its function is superceded by
//the function IFrameWindow::adoptColorMap and so the class
//should be obsoleted.

//---------------------------------------------------------------------------

IColorMapContext::IColorMapContext(const INativeWindowHandle& theWindow, IColorMap* adoptColorMap) :
   fWindow(theWindow),
   fReleaseDC(True),
   fMap(adoptColorMap)
{

#if defined(IC_WIN)
        fPS = GetDC(fWindow);
        HPALETTE palette = (HPALETTE)fMap->nativeColorMap();
        fPrevious = (unsigned long)SelectPalette(fPS, palette, false);
        RealizePalette(fPS);
#elif defined(IC_PM)

        fPS = GetDC(fWindow);
        HPAL palette = (HPAL)fMap->nativeColorMap();
        fPrevious = (unsigned long)GpiSelectPalette(fPS, palette);
        unsigned long clr;
        WinRealizePalette(fWindow, fPS, &clr);


#elif defined(IC_MOTIF)

        Display* disp = getDisplay();

               if(!bPrivateDisplay){ //if not private display, means no GUI
                       Window win = XtWindow(fWindow);

                       XWindowAttributes wattr;
                       if(!XGetWindowAttributes(disp, win, &wattr))
                                       IGraphicException(" XGetWindowAttributes ");

                       fPrevious = (unsigned long)wattr.colormap;

                       Colormap xcmap = (Colormap)fMap->nativeColorMap();


                       XtVaSetValues(fWindow, XtNcolormap, xcmap, NULL);
                       if (!XSetWindowColormap(disp, win, xcmap))
                                       IGraphicException(" XSetWindowColormap ");

                       fPS = IPresSpaceHandle(new IXDC(disp, win));

               }
#endif
}

IColorMapContext::IColorMapContext(const IPresSpaceHandle& fPSH, IColorMap* adoptColorMap) :
   fWindow(0),
   fReleaseDC(False),
   fMap(adoptColorMap)
{

#if defined(IC_WIN)
        fPS = fPSH;
        HPALETTE palette = (HPALETTE)fMap->nativeColorMap();
        fPrevious = (unsigned long)SelectPalette(fPS, palette, false);
        RealizePalette(fPS);

#elif defined(IC_PM)

        fPS = fPSH;
        HPAL palette = (HPAL)fMap->nativeColorMap();
        fPrevious = (unsigned long)GpiSelectPalette(fPS, palette);
       fWindow = WindowFromDC(fPS); //get the window handle from presSpace
        unsigned long clr;
        WinRealizePalette(fWindow, fPS, &clr);


#elif defined(IC_MOTIF)

        Display* disp = getDisplay();
               if(!bPrivateDisplay){ //if not private display, means no GUI
                       Window win = ((IXDC*)fPSH)->xDrawable;

                       XWindowAttributes wattr;
                       if(!XGetWindowAttributes(disp, win, &wattr))
                                       IGraphicException(" XGetWindowAttributes ");

                       fPrevious = (unsigned long)wattr.colormap;

                       Colormap xcmap = (Colormap)fMap->nativeColorMap();

                       XtVaSetValues(fWindow, XtNcolormap, xcmap, NULL);
                       if (!XSetWindowColormap(disp, win, xcmap))
                                       IGraphicException(" XSetWindowColormap ");

                       fPS = fPSH;

               }
#endif
}

IColorMap& IColorMapContext::colormap() const
{
        return(*fMap);
}


IColorMapContext::~IColorMapContext()
{
#if 0
 ITRACE_DEVELOP(IString("old colormap realized "));
#endif

#if defined(IC_WIN)
        SelectPalette(fPS, (HPALETTE)fPrevious, false);
        RealizePalette(fPS);
        if(fReleaseDC)
                ReleaseDC(fWindow, fPS);
#elif defined(IC_PM)
        GpiSelectPalette(fPS, (HPAL)fPrevious);
        unsigned long clr;
        WinRealizePalette(fWindow, fPS, &clr);
        if(fReleaseDC)
                ReleaseDC(fWindow, fPS);
#elif defined(IC_MOTIF)
        if (fPrevious)
                if (!XSetWindowColormap(((IXDC*)fPS)->xDisplay, ((IXDC*)fPS)->xDrawable, fPrevious))
                        IGraphicException(" XSetWindowColormap ");
#endif
}
#endif //(IC_OBSOLETE <= IC_OBSOLETE_3)
