// Revision: 85 1.13.2.1 source/albert/graph2d/ibmpdata.cpp, 2d, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ibmpdata.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IBitmapData                                                              *
*     IBitmapSet                                                               *
*     IPointerData                                                             *
*     IPointerSet                                                              *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// For Windows, force compiling with STRICT, to ensure the compiler will
// generate mangled names that match the ones we are exporting. This must
// precede the inclusion of ibase.hpp.
#define STRICT  1

#include <ibmpdata.hpp>
#include <ibmpstat.hpp>
#include <irect.hpp>
#include <ireslock.hpp>
#include <istring.hpp>
#include <ingthrd.hpp>

#ifdef IC_MOTIF
  #include <stdlib.h>   // for getenv()
  #include <X11/Xlib.h>
  #include <Xm/Xm.h>
  #include <xpm.h>
  #include <ixdc.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IBMPDATA_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_PM
#define INCL_WINPOINTERS
  #define INCL_DOSPROCESS
  #define INCL_GPI
  #include <os2.h>
  #define IDELETEBITMAP(hbm)              (GpiDeleteBitmap( hbm ))
  #define IDESTROYPOINTER(hPtr)           (WinDestroyPointer( hPtr ))
#endif // IC_PM


#ifdef IC_WIN
  #include <windows.h>       // RDL: convert to winemul.h

  #define IDELETEBITMAP(hbm)              (DeleteObject( (LPVOID)hbm ))
  #define IDESTROYPOINTER(hPtr)           (DestroyIcon( hPtr ))
#endif // IC_WIN

#include <pixbuf.hpp>
#include <grassert.hpp>

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IBitmapData::IBitmapData                                                     |
------------------------------------------------------------------------------*/
IBitmapData::IBitmapData(unsigned long         bitmapId,
                         IBitmapHandle::Value  bitmapHandle,
                         const IString&        resourceFileName,
                         const ISize&          bitmapSize,
                         IPixelBuffer*         pixelBuffer)
  : fResourceId(bitmapId),
    fHandle((unsigned long)bitmapHandle),
    fResourceFileName(resourceFileName),
    fBitmapSize(bitmapSize),
    fUseCount(0),
    fPixBuf(pixelBuffer)
{
  IFUNCTRACE_DEVELOP();
  IASSERTPARM(bitmapHandle!=0);
}
#endif

/*------------------------------------------------------------------------------
| IBitmapData::~IBitmapData                                                    |
------------------------------------------------------------------------------*/
IBitmapData::~IBitmapData()
{
   IFUNCTRACE_DEVELOP();
#ifdef IC_PMWIN
   if(fHandle)
   {
     bool fSuccess = (bool)IDELETEBITMAP((IBitmapHandle::Value)fHandle);
     if(!fSuccess)
         GrafDeviceException("IDELETEBITMAP");
   }
#endif // IC_PMWIN

#ifdef IC_MOTIF
   // do real Motif delete here
   Display* display = IXDisplay::display();
   XFreePixmap( display, (Pixmap)fHandle );

   if ( fInsensitivePixmap)
     XFreePixmap( display, fInsensitivePixmap );

   XDestroyImage(fImage);
#endif

	if(fPixBuf)
		delete fPixBuf;

}

/*------------------------------------------------------------------------------
| IBitmapSet::IBitmapSet                                                       |
------------------------------------------------------------------------------*/
IBitmapSet::IBitmapSet()
{}

/*------------------------------------------------------------------------------
| IBitmapSet::~IBitmapSet                                                      |
------------------------------------------------------------------------------*/
IBitmapSet::~IBitmapSet() {}


/*------------------------------------------------------------------------------
| IBitmapSet::find                                                             |
------------------------------------------------------------------------------*/
IBitmapData* IBitmapSet::find( const unsigned long bitmapId,
                               const IString&      fileName,
                               const ISize&        bitmapSize )
{
  IFUNCTRACE_DEVELOP();
  IBitmapData* bitmapDataReturn(0);
  IBitmapData* bitmapData;

     /* Lock Resource Table */
  IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
  IBitmapSet::Cursor cursor(*this);
  for(cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
  {
     bitmapData = this->elementAt(cursor);
     if( (bitmapData->id() == bitmapId)               &&
         (bitmapData->resourceFileName() == fileName) &&
         (bitmapData->size() == bitmapSize))
     {
       ITRACE_ALL("BMP Match found");
       bitmapDataReturn = bitmapData;
       break;
     }
  }
  return bitmapDataReturn;
}

/*------------------------------------------------------------------------------
| IBitmapSet::find                                                             |
------------------------------------------------------------------------------*/
IBitmapData* IBitmapSet::find( const IBitmapHandle& bitmapHandle )
{
  return find((IBitmapHandle::Value)bitmapHandle);
}


/*------------------------------------------------------------------------------
| IBitmapSet::find                                                             |
------------------------------------------------------------------------------*/
IBitmapData* IBitmapSet::find( IBitmapHandle::Value bitmapHandle )
{

   IFUNCTRACE_DEVELOP();
  /* Lock Resource Table and locate the bitmap */
  IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
  IBitmapSet::Cursor cursor(*this);
  bool found = locateElementWithKey((unsigned long)bitmapHandle,
                                    cursor);
  if(found)
     return elementAt(cursor);
  else
     return 0;
}

/*------------------------------------------------------------------------------
| IBitmapSet::remove                                                           |
------------------------------------------------------------------------------*/
IBitmapSet& IBitmapSet::remove( IBitmapData* bitmapData )
{
     /* Lock Resource Table */
  IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
  removeElementWithKey((unsigned long)bitmapData->handle());
  return *this;
}


void IBitmapData::adoptPixelBuffer( IPixelBuffer* pixelBuffer)
{
  if (fPixBuf)
  {
	delete fPixBuf;
  }
  fPixBuf = pixelBuffer;
}


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IBitmapData::IBitmapData                                                     |
------------------------------------------------------------------------------*/
IBitmapData::IBitmapData(unsigned long   resourceId,
                         const IString&  resourceFileName,
                         const ISize&    imageSize,
                         const ISize&    desiredBitmapSize,
                         XImage*         image,
                         IPixelBuffer*   pixelBuffer)
          : fResourceId(resourceId),
            fHandle(0),
            fBitmapSize(desiredBitmapSize),
            fUseCount(0),
            fResourceFileName(resourceFileName),
            fImageSize(imageSize),
            fImage(image),
            fInsensitivePixmap(0),
            fPixBuf(pixelBuffer)
{
   IFUNCTRACE_DEVELOP();
  if(fPixBuf){
	if(imageSize != desiredBitmapSize) {
		fPixBuf->resize(desiredBitmapSize.width(), desiredBitmapSize.height());
		fImage = fPixBuf->imageHandle();
	}
	fHandle = fPixBuf->handle();

  } else if(fImage)
    createPixmapFromImage();
}


/*------------------------------------------------------------------------------
| IBitmapData::IBitmapData                                                     |
------------------------------------------------------------------------------*/
IBitmapData::IBitmapData(unsigned long   resourceId,
                         const ISize&    imageSize,
                         XImage*         image,
                         unsigned long   pixmap,
                         IPixelBuffer*   pixelBuffer)
          : fResourceId(resourceId),
            fHandle((unsigned long)pixmap),
            fBitmapSize(imageSize),
            fUseCount(0),
            fResourceFileName(),
            fImageSize(imageSize),
            fImage(image),
            fInsensitivePixmap(0),
            fPixBuf(pixelBuffer)
{
   IFUNCTRACE_DEVELOP();
   IASSERTPARM(pixmap!=0);
}

//
// I used the "bitmap" sample application to create this 2x2 50% bitmap.
//
#define stipple50_width 2
#define stipple50_height 2
static char stipple50_bits[] = {
   0x02, 0x01};


/*------------------------------------------------------------------------------
| IBitmapData::insensitivePixmap                                               |
------------------------------------------------------------------------------*/
unsigned long IBitmapData::insensitivePixmap()
{
  if(!fInsensitivePixmap)
     createInsensitivePixmap();
  return fInsensitivePixmap;
}

/*------------------------------------------------------------------------------
| IBitmapData::createInsensitivePixmap                                         |
------------------------------------------------------------------------------*/
Pixmap IBitmapData::createInsensitivePixmap()
{
   IFUNCTRACE_DEVELOP();
 //
 // Create, if necessary, and return a Pixmap which is a stippled
 // version of pixmap.
 //
 if ( !fInsensitivePixmap )
  {
   Display* display = IXDisplay::display();
   Screen* screen = DefaultScreenOfDisplay(display);
   unsigned int depth = XDefaultDepth(display, XDefaultScreen(display));

//   Pixmap stipple = XmGetPixmapByDepth( screen, "50_foreground", 1, 0, 1 );
   Pixmap stipple = XCreateBitmapFromData( display, (IBitmapHandle::Value)fHandle,
                                           stipple50_bits,
                                           stipple50_width, stipple50_height );

   fInsensitivePixmap
     = XCreatePixmap( display, (IBitmapHandle::Value)fHandle,
                      fBitmapSize.width(), fBitmapSize.height(), depth );

   XGCValues gcv;
   GC gc;
   gc = XCreateGC(display, fInsensitivePixmap, 0, &gcv );
   int result = 0;
   result = XCopyArea( display, (IBitmapHandle::Value)fHandle,
                       fInsensitivePixmap, gc, 0, 0,
              fBitmapSize.width(), fBitmapSize.height(), 0, 0 );
   result = XSetStipple( display, gc, stipple );
   result = XSetFillStyle( display, gc, FillStippled );
   result = XSetForeground( display, gc, XWhitePixelOfScreen( screen ) );
   result = XFillRectangle( display, fInsensitivePixmap, gc, 0, 0,
                   fBitmapSize.width(), fBitmapSize.height() );

   XFreeGC( display, gc );
   XFreePixmap( display, stipple );
  }
 return fInsensitivePixmap;
}

/*------------------------------------------------------------------------------
| IBitmapData::createXImage                                                    |
|    used only when we scale bitmaps using sizeImage()                         |
------------------------------------------------------------------------------*/
XImage* IBitmapData::createXImage(const ISize& size, unsigned long depth)
{
   IFUNCTRACE_DEVELOP();
    XImage* image_return;
    Display* display = IXDisplay::display();
    Screen* screen = DefaultScreenOfDisplay(display);
    Visual *visual = XDefaultVisualOfScreen(screen);
    unsigned int width = size.width();
    unsigned int height = size.height();
    int bitmap_pad;

    /* first get bitmap_pad */
    if (depth > 16)
        bitmap_pad = 32;
    else if (depth > 8)
        bitmap_pad = 16;
    else
        bitmap_pad = 8;

    /* then create the XImage with data = NULL and bytes_per_line = 0 */

    image_return = XCreateImage(display, visual, depth, ZPixmap, 0, 0,
                                 width, height, bitmap_pad, 0);
    if (!image_return)
		GrafDeviceException("XCreateImage");

    /* now that bytes_per_line must have been set properly alloc data */

    (image_return)->data =
        (char *) new char[ image_return->bytes_per_line * height];

    return image_return;
}

/*------------------------------------------------------------------------------
| IBitmapData::copyImage                                                       |
------------------------------------------------------------------------------*/
XImage* IBitmapData::copyImage(XImage* source, const ISize& sizeBitmap)
{
   IFUNCTRACE_DEVELOP();
  XImage* target = createXImage(sizeBitmap, source->depth);

  if (source)
  {
    // if the source was 100x100 and the target 200x200 then both
    // xCorrection and yCorrection would be .5
    // The correction factor is used in the loop below.
    int width = sizeBitmap.width(), height = sizeBitmap.height();
    double xCorrection = (double)source->width / (double)width;
    double yCorrection = (double)source->height / (double)height;

    // Loop through the target image setting each pixel to the corresponding
    // value in the source image. xCorrection and yCorrection find the
    // "corresponding" pixel in the source.
    for(int y=0; y<height; y++)
      for(int x=0; x<width; x++)
        XPutPixel(target, x, y, XGetPixel(source, x*xCorrection, y*yCorrection));

  }

  return target;
}

/*------------------------------------------------------------------------------
| IBitmapData::createPixmapFromImage                                           |
------------------------------------------------------------------------------*/
void IBitmapData::createPixmapFromImage()
{
   IFUNCTRACE_DEVELOP();
  Display* display = IXDisplay::display();
  Screen* screen = DefaultScreenOfDisplay(display);

  // If a non-zero size was passed in on the desiredSize parameter,
  // we may need to scale the Pixmap
  XImage* newImage = fImage;

  if (fBitmapSize.width() && fBitmapSize.height())
  {
    // Only do the scaling (an expensive operation) if
    // the requested non-zero size is different than the actual
    // size that was just loaded into image
    if (fBitmapSize.width()  != trueSize( ).width() ||
        fBitmapSize.height() != trueSize( ).height())
    {
      newImage = copyImage(fImage, fBitmapSize);
    }
  }
  else
  {
    // didn't need to resize, set currentSize to trueSize
    fBitmapSize = fImageSize;
  }

  XGCValues gcv;
  GC gc;
  gcv.function = GXcopy;

  fHandle = (unsigned long)XCreatePixmap(display,
                             XRootWindowOfScreen(screen),
                             newImage->width, newImage->height,
                             newImage->depth);

  gc = XCreateGC(display, (IBitmapHandle::Value)fHandle,
                 GCFunction, &gcv);

  XPutImage(display, (IBitmapHandle::Value)fHandle, gc, newImage,
            0, 0, 0, 0, newImage->width, newImage->height);

  XFreeGC(display, gc);



  // If the image was scaled, we allocated the data using new in
  // createXImage(). If this is the case delete it and set data to NULL.
  if (newImage != fImage)
  {
     delete [] newImage->data;
     newImage->data = NULL;
     XDestroyImage(newImage);
  }

}


#endif //IC_MOTIF

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IPointerData::IPointerData                                                   |
------------------------------------------------------------------------------*/
IPointerData::IPointerData(unsigned long  resourceId,
                           IPointerHandle::Value  pointerHandle,
                           const IString& resourceFileName,
                           IPixelBuffer*  pixelBuffer )
          : IBitmapData(resourceId, (IBitmapHandle::Value)pointerHandle, resourceFileName,
                        ISize(0,0), pixelBuffer)
{
   IFUNCTRACE_DEVELOP();
}
#endif // IC_PMWIN


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IPointerData::IPointerData                                                   |
------------------------------------------------------------------------------*/
IPointerData::IPointerData(unsigned long resourceId,
                         const IString&  resourceFileName,
                         const ISize&    imageSize,
                         const ISize&    desiredBitmapSize,
                         XImage*         image,
                         XImage*         maskImage,
                         const IPoint&   hotSpot,
                         IPixelBuffer*   pixelBuffer)
          : IBitmapData( resourceId, resourceFileName, imageSize, desiredBitmapSize,
                         image, pixelBuffer),
            fMaskImage(maskImage),
            fHotSpot(hotSpot)
{
   IFUNCTRACE_DEVELOP();
  createMaskPixmapFromImage();
}

IPointerData::IPointerData(unsigned long  bitmapId,
                         const IString& xpmFileName,
                         const ISize&   desiredSize,
                         IPixelBuffer*  pixelBuffer)

          : IBitmapData( bitmapId, xpmFileName, ISize(0,0), desiredSize,
                         0, pixelBuffer),
            fMaskImage(0),
            fHotSpot(0,0),
            fXpmFileName(xpmFileName)
{
   IFUNCTRACE_DEVELOP();
  _XImage* image = 0;
  ISize imageSize;
  loadXpmImageFile(xpmFileName, imageSize, &image, &fMaskImage, fHotSpot);
  IBitmapData::setImage(image);
  IBitmapData::setImageSize(imageSize);
  createPixmapFromImage();
  createMaskPixmapFromImage();
}

/*------------------------------------------------------------------------------
| IPointerData::createMaskPixmapFromImage                                      |
------------------------------------------------------------------------------*/
void IPointerData::createMaskPixmapFromImage( )
{
   IFUNCTRACE_DEVELOP();
  Display* display = IXDisplay::display();
  Screen* screen = DefaultScreenOfDisplay(display);

  // If a non-zero size was passed in on the desiredSize parameter,
  // we may need to scale the Pixmap
  XImage* newShapeImage = fMaskImage;

  if (size().width() && size().height())
  {
    // Only do the scaling (an expensive operation) if
    // the requested non-zero size is different than the actual
    // size that was just loaded into image
    if (size().width()  != trueSize( ).width() ||
        size().height() != trueSize( ).height())
    {
      // Only size the shape image if we have one
      if (newShapeImage)
        newShapeImage = copyImage(fMaskImage, size());
    }
  }

  if (newShapeImage)
  {
    GC gcm;
    XGCValues gcv;
    gcv.function = GXcopy;

    fMaskPixmap = XCreatePixmap(display,
                             XRootWindowOfScreen(screen),
                             newShapeImage->width, newShapeImage->height,
                             newShapeImage->depth);
    gcm = XCreateGC(display, fMaskPixmap, GCFunction, &gcv);
    XPutImage(display, fMaskPixmap, gcm, newShapeImage, 0, 0, 0, 0,
      newShapeImage->width, newShapeImage->height);
    XFreeGC(display, gcm);
  }


  // If the image was scaled, we allocated the data using new in
  // createXImage(). If this is the case delete it and set data to NULL.
  if (newShapeImage && (newShapeImage != fMaskImage))
  {
     delete [] newShapeImage->data;
     newShapeImage->data = 0;
     XDestroyImage(newShapeImage);
  }
}

/*------------------------------------------------------------------------------
| IPointerData::loadXpmImageFile                                               |
------------------------------------------------------------------------------*/
void IPointerData::loadXpmImageFile(const IString& pixFileName,
                                   ISize& imageSize,
                                   _XImage** image,
                                   _XImage** maskImage,
                                   IPoint&    hotSpot)
{
   IFUNCTRACE_DEVELOP();
  int result = 0;
  Display* display = IXDisplay::display();
  Screen* screen = DefaultScreenOfDisplay(display);
  XpmAttributes xpmAtt;
  xpmAtt.valuemask = XpmVisual | XpmColormap | XpmCloseness;
  xpmAtt.visual = XDefaultVisualOfScreen(screen);
  xpmAtt.colormap = IColorMap::defaultColorMap().nativeColorMap();
  // By giving a value for closeness, we cause the Xpm code to look
  // for the closest matching color if the DefaultColormap has no more
  // free cells. The is much better than just having XpmReadFileToImage
  // fail and for us to then throw an exception.
  // The Xpm documentation recommended 40000 as a good value for finding
  // a match that is resonable. If you change closeness you get:
  //    > 40000 - closer match but chance of not finding a match
  //    < 40000 - better at finding a match, color may not be close

  // The following is to allow for exceptions that occur when using IOC
  // from emulators, like Exceed, from VGA systems.
  unsigned int depth = XDefaultDepth(display, XDefaultScreen(display));
  if(depth <= 4)
    xpmAtt.closeness = 65535;
  else
    xpmAtt.closeness = 40000;

  // This will find the Xpm file if pixFileName is fully qualified
  // with its directory path.
  // It will also find it if there is no directory path specified
  // and the file itself is in the current directory.
  result = XpmReadFileToImage(
             display,
             pixFileName,
             image,
             maskImage,
             &xpmAtt);

  // If we didn't find the file and the name wasn't fully qualified,
  // see if we can load it from XAPPLRESDIR
  if (result == XpmOpenFailed && pixFileName[1] != '/') {
    IString betterFileName(getenv("XAPPLRESDIR"));
    if (betterFileName.size() > 0) {  // XAPPLRESDIR was set
      // ensure directory spec ends with a delimiter
      betterFileName += "/";
      betterFileName.change("//", "/");
      betterFileName += pixFileName;
      result = XpmReadFileToImage(
                 display,
                 betterFileName,
                 image,
                 maskImage,
                 &xpmAtt);
    }
  }

  if (result < 0)
    if (result == -1)
      GrafDeviceException(IString("Can't read XPM file: ") +
                     IString(pixFileName) );
    else
      GrafDeviceException(IString("XpmReadFileToImage failed with: ") +
                     IString(result) +
                     IString(" for: ") +
                     IString(pixFileName) );

  imageSize.setWidth(xpmAtt.width);
  imageSize.setHeight(xpmAtt.height);
  hotSpot.setX(xpmAtt.x_hotspot);
  hotSpot.setY(xpmAtt.y_hotspot);
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| IPointerData::~IPointerData                                                          |
------------------------------------------------------------------------------*/
IPointerData::~IPointerData()
{
#ifdef IC_PMWIN
   bool fSuccess = (bool)
      IDESTROYPOINTER( (IPointerHandle::Value) this->handle() );
   if(!fSuccess)
       GrafDeviceException("IDELETEBITMAP");
   setHandle(0);
#endif

#ifdef IC_MOTIF
   // do real Motif delete here
   Display* display = IXDisplay::display();
   if(fMaskPixmap)
     XFreePixmap( display, (Pixmap)fMaskPixmap );
   if(fMaskImage)
     XDestroyImage(fMaskImage);
#endif  // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IPointerSet::IPointerSet                                                     |
------------------------------------------------------------------------------*/
IPointerSet::IPointerSet ( ) {}

/*------------------------------------------------------------------------------
| IPointerSet::~IPointerSet                                                    |
------------------------------------------------------------------------------*/
IPointerSet::~IPointerSet ( ) {}


/*------------------------------------------------------------------------------
| IPointerSet::find                                                            |
------------------------------------------------------------------------------*/
IPointerData* IPointerSet::find( IPointerHandle::Value pointerHandle )
{

   IFUNCTRACE_DEVELOP();

  /* Lock Resource Table and find pointerdata */
  IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());

  IPointerSet::Cursor cursor(*this);
  bool found = locateElementWithKey((unsigned long)pointerHandle,
                                    cursor);
  if(found)
     return elementAt(cursor);
  else
     return 0;

}


/*------------------------------------------------------------------------------
| IPointerSet::remove                                                           |
------------------------------------------------------------------------------*/
IPointerSet& IPointerSet::remove( IPointerData* pointerData )
{
   IFUNCTRACE_DEVELOP();
     /* Lock Resource Table */
  IResourceLock reslibLock(IBitmapStaticPtr::bitmapKey());
  removeElementWithKey((unsigned long)pointerData->handle());
  return *this;
}
