/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 23 1.22.1.4 source/albert/graph2d/grstate.cpp, 2d, ioc.v400, 001006 
/*================
||
||  File:   GrafState.c
||
||  What:   Routines for the Albert TOldGrafState class
||
||  Change History:
||  18 Apr 96   tt      First version in OpenClass
||
*/

#include <grstate.hpp>
#include <ibcolor.hpp>
#include <iexgrprt.hpp>
#include <grstrmod.hpp>
#include <iprimlck.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _GRSTATE_CPP_
#include <ipagetun.h>
#endif

//---------------------------------------------------
// IStateChange - bit fields of state changes

IStateChange* IStateChange::fgNullChange = 0;

const IStateChange& IStateChange::nullStateChange()
{
    if (!fgNullChange) {
        IPrimalLock lock;
        if (!fgNullChange) {
             fgNullChange = new IStateChange();
        }
    }
    return *fgNullChange;
}

unsigned long IStateChange::attrbuteStateChange(const IAttributeState* bundle)
{
	unsigned long sc = 0;
	if (bundle)
	{
		IAttributeState::EDrawOperation dop = bundle->drawingOperation();
		
		if (dop != IAttributeState::kUnset)
			sc |= kDrawingOperation;
		if (dop == IAttributeState::kFill || dop == IAttributeState::kFillAndFrame)
		{
			if (bundle->fillPaint())
				sc |= kFillPaint;
			if (bundle->fillTransferMode())
				sc |= kFillTransferMode;
		}
		if (dop == IAttributeState::kFrame || dop == IAttributeState::kFillAndFrame)
		{
			if (bundle->framePaint())
				sc |= kFramePaint;
			if (bundle->frameTransferMode())
				sc |= kFrameTransferMode;
			if (bundle->framePen())
				sc |= kFramePen;
			if (bundle->frameEndCap())
				sc |= kFrameEndCap;
			if (bundle->frameJoint())
				sc |= kFrameJoint;
		}
		if (bundle->imageTransferMode())
			sc |= kImageTransferMode;
		if (bundle->imageSampling())
			sc |= kImageSamplin;
	}
	return sc;
}


/*================
||  IGrafState
*/

IGrafState::~IGrafState()
{}

IGrafState::IGrafState() :
	fLinkLevel(0),
	fReferenceCount(0)
{}

IStateChange IGrafState::registeredAsTheStateInUse() const
{
	IRootGrafState* rootState = (IRootGrafState*)root();
	IStateChange delta = rootState->fOrphanedStateChange;
	rootState->fOrphanedStateChange = 0;    // reset it to 0

	const IGrafState* prev = stateInUse();
	const IGrafState* curr = this;

	if (prev == this)
		return delta;
	else
	{
		((IGrafState*)this)->setStateInUse(this);	// this is the StateInUse now ...

		if (!prev)	// this is the first time the chain of grafport is used...
			return IStateChange((unsigned long)IStateChange::kAllChanges);
		
		unsigned int prevCount = prev->fLinkLevel;
		unsigned int currCount = curr->fLinkLevel;
		
		if (currCount > prevCount)
		{
			do
			{	// from curr up ...
				delta |= curr->stateChange();
				curr = curr->parent();
				--currCount;
			}
			while (currCount > prevCount);
		}
		else
		{
			while (prevCount > currCount)
			{	// from prev up ...
				delta |= prev->stateChange();
				prev = prev->parent();
				--prevCount;
			}
		}
		
		// now that the two has the same count...
		while (curr != prev) {
			delta |= curr->stateChange();
			curr = curr->parent();
			delta |= prev->stateChange();
			prev = prev->parent();
		}
		return delta;
	}
}

void IGrafState::setLinkLevel(unsigned int level)
{
	fLinkLevel = level;
}

//================
// IRootGrafState
//================

IRootGrafState::IRootGrafState() :
	fBundle(IBaseColor(IBaseColor::kBlack)),
						// the default bundle - black frame only
	fViewMatrix(),		// the identity view matrix
	fCoordinateFlip(0 /*NIL*/),
	fModelMatrixPointer(&IGrafMatrix::identity()),		// the identity model matrix
	fClipArea(IGRect2D::infiniteRect())	// the infinity clip retangular area, stateful port
										// makes assumptions that this is so...
{
	fStateInUse = 0 /*NIL*/;	// the first time, it's null
	fOrphanedStateChange = 0;
	
	// set the default attributes:
	fBundle.setFillColor(IBaseColor(IBaseColor::kBlack));
	fBundle.setFillTransferMode(IColorTransferMode());
	fBundle.setFrameTransferMode(IColorTransferMode());
	fBundle.setFramePen(IPen(IPen::kSolid));
	fBundle.setFrameEndCap(ICap());
	fBundle.setFrameJoint(IJoint());
	fBundle.setImageTransferMode(IImageTransferMode());
	fBundle.setImageSampling(IImageSamplingControl());
}

IRootGrafState::~IRootGrafState()
{
	delete fCoordinateFlip;
}

void IRootGrafState::setDefaultViewMatrix(const IGrafMatrix& matrix)
{	// this is suppose to be called only once at owner construction time...
	fViewMatrix = matrix;
    fModelMatrixPointer = &fViewMatrix;
}	

void IRootGrafState::setCoordinateFlip(const IGrafMatrix& matrix)
{
	if (!matrix.isIdentity()){
		delete fCoordinateFlip;
		fCoordinateFlip = new IGrafMatrix(matrix);
	}	
	else
	{
		delete fCoordinateFlip;             // ajd 3/27/1997
		fCoordinateFlip = 0 /*NIL*/;
	}
}

//================
// ILinkedGrafState
//================

ILinkedGrafState::ILinkedGrafState(
	const IGrafState* parent,
	unsigned long stateChange) :
    fParent(parent),
	fAttributes(parent->attributeState()),
	fViewMatrix(parent->viewMatrix()),
	fModelMatrix(parent->modelMatrix()),
	fClipArea(parent->clipArea()),
	fRoot(parent->root()),
	fStateChange(stateChange)
{
	setLinkLevel(parent->linkLevel() + 1);
	((IGrafState*)fParent)->addReference();
}

ILinkedGrafState::~ILinkedGrafState()
{
	((IGrafState*)fParent)->removeReference();
	if (this == stateInUse()) {
		setStateInUse(fParent); // parent is the StateInUse now ... orphan the stateChange
		((IRootGrafState*)fRoot)->fOrphanedStateChange |= fStateChange;
	}
}

void ILinkedGrafState::setStateInUse(const IGrafState* state)
{
	IRootGrafState* theRoot = (IRootGrafState*)fRoot;
	theRoot->setStateInUse(state);
}


//================
// ILinkedAttributeGrafState
//================

ILinkedAttributeGrafState::ILinkedAttributeGrafState(
	const IGrafState* parent,
	const IAttributeState* referencedAttributes) :

    ILinkedGrafState(
		parent,
		IStateChange::attrbuteStateChange(referencedAttributes)),
	fLocalAttributes(referencedAttributes),
	fAttributes(parent->attributeState(), referencedAttributes)
{
	setAttributeState(&fAttributes);
}

ILinkedAttributeGrafState::~ILinkedAttributeGrafState()
{}


//================
// ILinkedViewMatrixGrafState
//================

ILinkedViewMatrixGrafState::ILinkedViewMatrixGrafState(
	const IGrafState* parent,
	const IGrafMatrix* referencedMatrix) :

    ILinkedGrafState(parent, IStateChange::kViewMatrixChange | IStateChange::kModelMatrixChange),
    fLocalMatrix(referencedMatrix),
	fViewMatrix(*parent->viewMatrix()),
	fModelMatrix()
{
	if (fLocalMatrix)
	{
		fViewMatrix.preConcatWith(*fLocalMatrix);
		setViewMatrix(&fViewMatrix);

		const IGrafMatrix* parentModelMatrix = parent->modelMatrix();
		const IGrafMatrix* parentViewMatrix = parent->viewMatrix();

		if (parentModelMatrix == parentViewMatrix || parentModelMatrix->isIdentity())
		{	// only view matrices in the chain or parent is an identity matrix...
			setModelMatrix(&fViewMatrix);
		}
		else
		{	// evaluate the model matrix...
			fModelMatrix = *parentModelMatrix;
    		fModelMatrix.preConcatWith(*fLocalMatrix);
			setModelMatrix(&fModelMatrix);
		}
	}
}

ILinkedViewMatrixGrafState::~ILinkedViewMatrixGrafState()
{}


//================
// ILinkedModelMatrixGrafState
//================

ILinkedModelMatrixGrafState::ILinkedModelMatrixGrafState(
	const IGrafState* parent,
	const IGrafMatrix* referencedMatrix) :

    ILinkedGrafState(parent, IStateChange::kModelMatrixChange),
    fLocalMatrix(referencedMatrix),
	fMatrix(*parent->modelMatrix())
{
	if (fLocalMatrix)
	{
    	fMatrix.preConcatWith(*fLocalMatrix);
		setModelMatrix(&fMatrix);
	}
}

ILinkedModelMatrixGrafState::~ILinkedModelMatrixGrafState()
{}


//================
// ILinkedClipGrafState
//================

ILinkedClipGrafState::ILinkedClipGrafState(
	const IGrafState* parent,
	const IGArea* referencedArea) :

    ILinkedGrafState(parent, IStateChange::kClipAreaChange),
    fLocalClipArea(referencedArea),
	fClipArea()
{
	if (fLocalClipArea)
	{
		const IGArea* parentClipArea = parent->clipArea();
		unsigned long timeStamp = fLocalClipArea->timeStamp();
		if (timeStamp != parentClipArea->timeStamp() && timeStamp != IGArea::infiniteArea().timeStamp())
		{
			fClipArea = *fLocalClipArea;
			fClipArea.transformBy(*parent->modelMatrix());
			fClipArea.intersect(*parentClipArea);
			setClipArea(&fClipArea);
		}
	}
}

ILinkedClipGrafState::~ILinkedClipGrafState()
{}


//================
// ILinkedBundleState routines
//================

StreamableDefinitionsMacro(ILinkedBundleState, gGraphicsStreamModule);

ILinkedBundleState::ILinkedBundleState(
    const IAttributeState* parent,
    const IAttributeState* local) :
	fParent(parent),
	fLocal(local),
	fTimeStamp(parent->timeStamp())
{
	if (local) {
    	updateTimeStamp();
		fTimeStamp = timeStamp();
	}
}

ILinkedBundleState::~ILinkedBundleState()
{}

unsigned long ILinkedBundleState::timeStamp() const
{
    return fTimeStamp;
}

const IPaint* ILinkedBundleState::fillPaint() const
{
    const IPaint* temp;
	if (!fLocal || !(temp = fLocal->fillPaint()))
		temp = fParent->fillPaint();
	return temp;
}

const IColorTransferMode* ILinkedBundleState::fillTransferMode() const
{
    const IColorTransferMode* temp;
	if (!fLocal || !(temp = fLocal->fillTransferMode()))
		temp = fParent->fillTransferMode();
    return temp;
}

const IPaint* ILinkedBundleState::framePaint() const
{
    const IPaint* temp;
	if (!fLocal || !(temp = fLocal->framePaint()))
		temp = fParent->framePaint();
    return temp;
}

const IColorTransferMode* ILinkedBundleState::frameTransferMode() const
{
    const IColorTransferMode* temp;
	if (!fLocal || !(temp = fLocal->frameTransferMode()))
		temp = fParent->frameTransferMode();
    return temp;
}

const IPen* ILinkedBundleState::framePen() const
{
    const IPen* temp;
	if (!fLocal || !(temp = fLocal->framePen()))
		temp = fParent->framePen();
    return temp;
}

const ICap* ILinkedBundleState::frameEndCap() const
{
    const ICap* temp;
	if (!fLocal || !(temp = fLocal->frameEndCap()))
		temp = fParent->frameEndCap();
    return temp;
}

const IJoint* ILinkedBundleState::frameJoint() const
{
    const IJoint* temp;
	if (!fLocal || !(temp = fLocal->frameJoint()))
		temp = fParent->frameJoint();
    return temp;
}

const IImageTransferMode* ILinkedBundleState::imageTransferMode() const
{
    const IImageTransferMode* temp;
	if (!fLocal || !(temp = fLocal->imageTransferMode()))
		temp = fParent->imageTransferMode();
    return temp;
}

const IImageSamplingControl* ILinkedBundleState::imageSampling() const
{
    const IImageSamplingControl* temp;
	if (!fLocal || !(temp = fLocal->imageSampling()))
		temp = fParent->imageSampling();
    return temp;
}

IAttributeState::EDrawOperation ILinkedBundleState::drawingOperation() const
{
    IAttributeState::EDrawOperation temp;
	if (!fLocal || (temp = fLocal->drawingOperation()) == IAttributeState::kUnset)
		temp = fParent->drawingOperation();
    return temp;
}

// Explicitly disabled:

ILinkedBundleState::ILinkedBundleState()
{}

ILinkedBundleState::ILinkedBundleState(const ILinkedBundleState&)
{}

void ILinkedBundleState::writeToStream(IDataStream&) const
{}

void ILinkedBundleState::readFromStream(IDataStream&)
{}

ILinkedBundleState& ILinkedBundleState::operator=(const ILinkedBundleState&)
{
	return *this;
}

bool ILinkedBundleState::operator==(const IAttributeState&) const
{
	return false;
}

IAttributeState::EPrintingState ILinkedBundleState::printingFlag() const {
    IAttributeState::EPrintingState temp;
	if (!fLocal ||
		(temp = fLocal->printingFlag()) == IAttributeState::kPrintingUndefined)
		temp = fParent->printingFlag();
    return temp;
}


//================
//	IStatefulGrafState
//================

IStatefulGrafState::IStatefulGrafState(const IGrafState* root) :
	ILinkedGrafState(root),
	fBundle(),
	fViewMatrix(),
	fModelMatrix(),
	fClipArea(IGRect2D::infiniteRect()),	// the infinity clip retangular area
	fConcatenatedBundle( root->attributeState(), &fBundle ),
	fConcatenatedViewMatrix(),
	fConcatenatedModelMatrix()
{}

IStatefulGrafState::~IStatefulGrafState()
{}

void IStatefulGrafState::setBundle(const IGrafBundle& bundle)
{
	fBundle = bundle;

	IStateChange sc(stateChange());
	sc &= IStateChange::kMatrixAndClipAreaChange;
	sc |= IStateChange::attrbuteStateChange(&bundle);
	setStateChange(sc);
}

void IStatefulGrafState::setModelMatrix(const IGrafMatrix& matrix)
{
	IStateChange sc(stateChange());
	fModelMatrix = matrix;
	// concatenation:
	fConcatenatedModelMatrix = *parent()->modelMatrix();
	fConcatenatedModelMatrix.preConcatWith(fViewMatrix);
	fConcatenatedModelMatrix.preConcatWith(matrix);
	sc |= IStateChange::kModelMatrixChange;
	setStateChange(sc);
}

void IStatefulGrafState::setViewMatrix(const IGrafMatrix& matrix)
{
	IStateChange sc(stateChange());
	fViewMatrix = matrix;
	// concatenation:
	fConcatenatedViewMatrix = *parent()->viewMatrix();
	fConcatenatedViewMatrix.preConcatWith(matrix);
	fConcatenatedModelMatrix = *parent()->modelMatrix();
	fConcatenatedModelMatrix.preConcatWith(matrix);
	fConcatenatedModelMatrix.preConcatWith(fModelMatrix);
	sc |= IStateChange::kViewMatrixChange | IStateChange::kModelMatrixChange;
	setStateChange(sc);
}

void IStatefulGrafState::setClipArea(const IGArea& clipArea)
{
	fClipArea = clipArea;	// because we know that the root have a default infinite rect!
	IStateChange sc(stateChange());
	sc |= IStateChange::kClipAreaChange;
	setStateChange(sc);
}
