/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 16 1.21.2.1 source/albert/graph2d/gpidev.cpp, 2d, ioc.v400, 001006 
/*================
||
||	File:	GpiDevices.C
||
||	What:	Routines which support the graphics device on OS2 GPI platform
||
||	Change History:
||
*/
#ifndef INCL_WIN
#define INCL_WIN
#include <winemul.h>
#undef INCL_WIN
#endif
#include <gpidev.hpp>
#include <pmvtxeng.hpp>
#include <igimage.hpp>
#include <imagedc.hpp>
#include <grstate.hpp>
#include <regnextr.hpp>
#include <inumeric.hpp>
#include <igloop2d.hpp>
#include <igtxtrun.hpp>
#include <grassert.hpp>
#include <iprimlck.hpp>
#include <iplatfrm.hpp>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <ibase.hpp>
#include <istring.hpp>
#include <deleter.hpp>
#include <igcurv2d.hpp>
#include <iknots.hpp>
#include <prspline.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _GDIDEV_CPP_
#include <ipagetun.h>
#endif

class IGpiDevice;
class IGpiEmulator;

static const GCoordinate kRoot2 =  1.414213562373;

//==============================================================
// IGpiEmulator is basically made to avoid inclusion of os2.h
// in exported header files
//==============================================================
class IGpiEmulator {
   public:
      IGpiEmulator(IGpiDevice* theDevice);
      ~IGpiEmulator();

      static COLORREF convertToWRGB(const IBaseColor& color);
      void IGPoint2DToPOINTL(const IGPoint2D &p, POINTL &ptl);
      HPEN CreatePen(const IGrafState& state);
      HBRUSH CreateFillBrush(const IGrafState& state);
      HBRUSH CreateFrameBrush(const IGrafState& state);
      HBRUSH CreateBrush(const LOGBRUSH& logbrush);
      LOGBRUSH CreateBrushStructure(const IPaint& paint);
      IBitmapHandle CreateBitmap( int width,
                            int height,
                            unsigned int planes,
                            unsigned int bitsPerPixel,
                            const void*  pBits,
                            const IPresSpaceHandle& hps );
      HBRUSH CreatePatternBrush( IBitmapHandle bmp, const IPresSpaceHandle& hps );
      XFORM fXFORM;                     // GPI matrix
   private:
      HBRUSH CreateSolidBrush( COLORREF color, const IPresSpaceHandle& hps );
      HBRUSH CreateHatchBrush( int type, COLORREF color, const IPresSpaceHandle& hps );
      HPEN ExtCreatePen( DWORD, DWORD, LOGBRUSH*, DWORD, const DWORD[], const IPresSpaceHandle& );
      IGpiDevice* fDevice;
      static long fPatternBrushId;
      HBRUSH fPenBrush;
};

IGpiEmulator::IGpiEmulator(IGpiDevice* theDevice)
: fDevice(theDevice),
   fPenBrush(0)
{
}

IGpiEmulator::~IGpiEmulator()
{
   fPenBrush = NULL;
}

long IGpiEmulator::fPatternBrushId = 254L;

COLORREF IGpiEmulator::convertToWRGB(const IBaseColor& color)
{
    return RGB(color.redMix(), color.greenMix(), color.blueMix());
}


void IGpiEmulator::IGPoint2DToPOINTL(const IGPoint2D &p, POINTL &ptl)
{
    ptl.x = fDevice->IGpiDevice::GCoordinateToInt(p.fX);
    ptl.y = fDevice->IGpiDevice::GCoordinateToInt(p.fY);
}


/*---------------------------- CreateXXXXXXBrush -------------------------------
| The 3 flavors of brush creation routines creates an AREABUNDLE with the      |
| appropriate attributes and returns it.                                       |
------------------------------------------------------------------------------*/

HBRUSH IGpiEmulator::CreateSolidBrush( COLORREF color, const IPresSpaceHandle& hps )
{

    HBRUSH solidResult = new BRUSH;

    solidResult->lColor        = ColorToIndex( hps, color );
    solidResult->lBackColor    = CLR_FALSE;
    solidResult->usMixMode     = FM_OVERPAINT;
    solidResult->usBackMixMode = BM_LEAVEALONE;
    solidResult->usSet         = 0;
    solidResult->usSymbol      = PATSYM_SOLID;
    solidResult->ptlRefPoint   = IPoint().asPOINTL();

    return solidResult;
}

/*------------------------------- ExtCreatePen ---------------------------------
| Create a pseudo-pen (LINEBUNDLE) with attributes gleaned from all the        |
| arguments to this function.                                                  |
|                                                                              |
| To map the color value to color index, we must have the HPS.  This is        |
| obtained by adding an extra argument to the function and defining a macro    |
| that automatically provides "fPs" as this argument.                          |
------------------------------------------------------------------------------*/

HPEN IGpiEmulator::ExtCreatePen( DWORD        dwPenStyle,
                   DWORD        dwWidth,
                   LOGBRUSH    *lb,
                   DWORD,
                   const DWORD[],
                   const IPresSpaceHandle& hps ) {
    HPEN
    result = new PEN;

    result->lColor        = ColorToIndex( hps, lb->lbColor );
    result->lBackColor    = CLR_FALSE;
    result->usMixMode     = FM_OVERPAINT;
    result->usBackMixMode = BM_LEAVEALONE;
    result->fxWidth       = dwWidth << 16;
    result->lGeomWidth    = dwWidth;
    result->usType        = dwPenStyle & LINETYPE_BIT_MASK;
    result->usEnd         = (dwPenStyle & LINEEND_BIT_MASK) >> LINEEND_BIT_OFFSET;
    result->usJoin        = (dwPenStyle & LINEJOIN_BIT_MASK) >> LINEJOIN_BIT_OFFSET;

//    HPS fHDC = hps;
    if(lb->lbStyle == BS_SOLID)
        fPenBrush = CreateSolidBrush(lb->lbColor, fDevice->fHDC);
    else if(lb->lbStyle == BS_HATCHED)
        fPenBrush = CreateHatchBrush(int(lb->lbHatch), lb->lbColor, fDevice->fHDC);
    else
        fPenBrush = CreatePatternBrush((HBITMAP) lb->lbHatch, fDevice->fHDC);

    setPenBrush(fPenBrush);

    return result;
}

HBRUSH IGpiEmulator::CreateHatchBrush( int type, COLORREF color, const IPresSpaceHandle& hps )
{

    HBRUSH hatchResult = new BRUSH;

    hatchResult->lColor        = ColorToIndex( hps, color );
    hatchResult->lBackColor    = CLR_FALSE;
    hatchResult->usMixMode     = FM_OVERPAINT;
    hatchResult->usBackMixMode = BM_LEAVEALONE;
    hatchResult->usSet         = 0;
    hatchResult->usSymbol      = type;
    hatchResult->ptlRefPoint   = IPoint().asPOINTL();

    return hatchResult;
}

HBRUSH IGpiEmulator::CreatePatternBrush( IBitmapHandle bmp, const IPresSpaceHandle& hps ) {
    // This counter hopefully gives us a good bitmap ID!

    HBRUSH patternResult = new BRUSH;

    if (patternResult) {
	    IPrimalLock lock;
	    // First, set bitmap ID for this bitmap.
	    if( GpiSetBitmapId( hps, bmp, IGpiEmulator::fPatternBrushId ) )
	    {
//		patternResult = new BRUSH;

		patternResult->lColor        = CLR_NEUTRAL; //??
		patternResult->lBackColor    = CLR_NEUTRAL; //??
		patternResult->usMixMode     = FM_OVERPAINT;
		patternResult->usBackMixMode = BM_LEAVEALONE;
		patternResult->usSet         = IGpiEmulator::fPatternBrushId--;
		patternResult->usSymbol      = 0;
		patternResult->ptlRefPoint   = IPoint().asPOINTL();

		// Wrap these at 254.  Somehow we need to GpiDeleteSetId.
		if( IGpiEmulator::fPatternBrushId < 0 ) IGpiEmulator::fPatternBrushId = 254;
	    }
	    else
	    {
		unsigned long rc = WinGetLastError( 0 );
	    }
    }

    return patternResult;
}


/*------------------------------- CreateBitmap ---------------------------------
| This function maps pretty closely to GpiCreateBitmap.  We also need to       |
| have the HPS.                                                                |
|                                                                              |
| Note that if pBits is non-zero, then Gpi requires us to describe the format. |
| We do that by presuming it's a monochrome bitmap.  This seems to match the   |
| current usage as called from gdidev.cpp.                                     |
------------------------------------------------------------------------------*/
IBitmapHandle IGpiEmulator::CreateBitmap( int width,
                            int height,
                            unsigned int planes,
                            unsigned int bitsPerPixel,
                            const void*  pBits,
                            const IPresSpaceHandle& hps ) {
	IBitmapHandle
    result = 0;

    LONG
    xResYRes[2];

    HDC
    hdc = GpiQueryDevice( hps );

    DevQueryCaps( hdc, CAPS_HORIZONTAL_RESOLUTION, 2, xResYRes );

    // Build BITMAPINFOHEADER with argument values.
    BITMAPINFOHEADER2
    bmpHeader = { sizeof bmpHeader,
        width, height,           // cx, cy
        planes, bitsPerPixel, // cPlanes, cBitCount
        BCA_UNCOMP,              // No compression.
        0,                           // Image data uncompressed.
        xResYRes[0],             // Horizontal resolution.
        xResYRes[1],             // Vertical resolution.
        0,                           // All color indices used.
        0,                           // All color indices are important.
        BRU_METRIC,              // Pels per meter.
        0,                           // Reserved.
        BRA_BOTTOMUP,            // Scan lines from bottom up.
        BRH_NOTHALFTONED,        // No halftoning algorithm.
        0,                           // cSize1 n/a.
        0,                           // cSize2 n/a.
        BCE_RGB,                     // Color array holds RGB2 values.
        0                                // User id.
    };

    CHAR
    infoBuffer[ sizeof( BITMAPINFO2 ) + sizeof( RGB2 ) ];


    BITMAPINFO2
    *bmpInfo = (BITMAPINFO2*)infoBuffer;

    RGB2
    white = { 255, 255, 255, 0};

    RGB2
    black = { 0, 0, 0, 0};

    memcpy( bmpInfo, &bmpHeader, sizeof bmpHeader );

    bmpInfo->argbColor[0] = white;
	bmpInfo->argbColor[1] = black;

    result = GpiCreateBitmap( hps,
                              (PBITMAPINFOHEADER2)&bmpHeader,
                              CBM_INIT,
                              (PBYTE)pBits,
                              (PBITMAPINFO2)bmpInfo );

    if( result == 0 )
    {
        GrafDeviceException("GpiCreateBitmap");
    }

    return result;
}

//==============================================================
//  IGpiEmulator::CreatePen
//

HPEN IGpiEmulator::CreatePen(const IGrafState& state)
{
    DWORD dwPenType;
    DWORD dwPenWidth;
    DWORD dwPenStyle = PS_SOLID;
    DWORD dwJoinStyle = 0;
    DWORD dwEndCap = 0;

    // set pen attributes from Frame Paint(color, pattern)
    const IPaint* paint = state.attributeState()->framePaint();
    LOGBRUSH lb = CreateBrushStructure(*paint);

    // set pen type: cosmetic or geometric(wide) and pen width
    const IPen* pen = state.attributeState()->framePen();

    IPen::EPenType penType = pen->penType();

    if (penType == IPen::kHairline)
    {
        dwPenType = PS_COSMETIC;
        dwPenWidth = 1;
        lb.lbStyle = PS_SOLID;
    }
    else
    {
        dwPenType = PS_GEOMETRIC;

		 if (fDevice->fPenMatrix && !fDevice->fPenMatrix->isIdentity())
        {
            // pen width calculation ... only done on Win95
            IGPoint2D pts[2];   // unit vector
            pts[0] = IGPoint2D::origin();
            pts[1] = IGPoint2D(1, 1);
            fDevice->fPenMatrix->transformPoints(pts, pts, 2);
            dwPenWidth = fDevice->IGpiDevice::GCoordinateToInt((pts[1] - pts[0]).vectorLength() * pen->penWidth() / kRoot2);
        }
        else
            dwPenWidth = fDevice->IGpiDevice::GCoordinateToInt(pen->penWidth());

        switch (penType) {
        default:
        case IPen::kSolid :
            // dwPenStyle = PS_SOLID;
            break;
        case IPen::kDot:
            dwPenStyle = PS_DOT;
            break;
        case IPen::kDashed:
            dwPenStyle = PS_DASH;
            break;
        case IPen::kDashDot:
            dwPenStyle = PS_DASHDOT;
            break;
        case IPen::kDashDotDot:
            dwPenStyle = PS_DASHDOTDOT;
            break;
        case IPen::kInvisible:
            dwPenStyle = PS_NULL;
            break;
        }

        const ICap* cap = state.attributeState()->frameEndCap();
        ICap::ECapType endCapType = cap->capType();
        switch (endCapType)
        {
        case ICap::kFlush:
        default:
            dwEndCap = PS_ENDCAP_FLAT;
            break;
        case ICap::kSquare:
            dwEndCap = PS_ENDCAP_SQUARE;
            break;
        case ICap::kRound:
			 dwEndCap = PS_ENDCAP_ROUND;
            break;
        }

        const IJoint* joint = state.attributeState()->frameJoint();
        IJoint::EJointType jointType = joint->jointType();
        switch (jointType)
        {
        case IJoint::kMiterLimit:
        default:
            dwJoinStyle = PS_JOIN_MITER;
            if (!SetMiterLimit(fHDC, FLOAT(joint->miterLimit()), NULL))
                GrafDeviceException("SetMiterLimit");
            break;
        case IJoint::kBevel:
            dwJoinStyle = PS_JOIN_BEVEL;
            break;
        case IJoint::kRound:
            dwJoinStyle = PS_JOIN_ROUND;
            break;
        }
    }
     IPrimalLock lock; //locking for gPenBrush; defect 24186
    // create pen
    HPEN hPen = ExtCreatePen(
        dwPenType   |        // Cosmetic or Geometric
        dwPenStyle  |        // dash/dot style
        dwJoinStyle |        // miter/bevel/round
        dwEndCap,            // flat/square/round
        dwPenWidth,          // logical width for Geometric or 1 for cosmetic
        &lb,                 // structure for brush attributes
        0, NULL, fDevice->fHDC);            // no custom style

    if (!hPen)
        GrafDeviceException("ExtCreatePen");

    return(hPen);
}


//==============================================================
//  IGpiEmulator::CreateFillBrush
//

HBRUSH IGpiEmulator::CreateFillBrush(const IGrafState& state)
{
    const IPaint* paint = state.attributeState()->fillPaint();
    LOGBRUSH lb = CreateBrushStructure(*paint);
    HBRUSH hBrush = CreateBrush(lb);

    return(hBrush);

}

//==============================================================
//  IGpiEmulator::CreateFrameBrush
//

HBRUSH IGpiEmulator::CreateFrameBrush(const IGrafState& state)
{
    const IPaint* paint = state.attributeState()->framePaint();
    LOGBRUSH lb = CreateBrushStructure(*paint);
    HBRUSH hBrush = CreateBrush(lb);

    return(hBrush);
}

//==============================================================
//  IGpiEmulator::CreateBrush
//

HBRUSH IGpiEmulator::CreateBrush(const LOGBRUSH& lb)
{
    HBRUSH hBrush;

    if (lb.lbStyle == BS_SOLID) {
        hBrush = CreateSolidBrush(lb.lbColor, fDevice->fHDC);
        if (hBrush == 0)
            GrafDeviceException("CreateSolidBrush");
    }
    else if (lb.lbStyle == BS_HATCHED) {
        hBrush = CreateHatchBrush(int(lb.lbHatch), lb.lbColor, fDevice->fHDC);
        if (hBrush == 0)
            GrafDeviceException("CreateHatchBrush");
    }
	else { // pattern brush
        hBrush = CreatePatternBrush((HBITMAP) lb.lbHatch, fDevice->fHDC);
        if (hBrush == 0)
            GrafDeviceException("CreatePatternBrush");
    }

    return(hBrush);
}

//==============================================================
//  IGpiEmulator::CreateBrushStructure (I need this for CreatePen)
//

LOGBRUSH IGpiEmulator::CreateBrushStructure(const IPaint& paint)
{
    LOGBRUSH lb;

    // color
    lb.lbColor = convertToWRGB(*paint.color());

    // pattern
    if (!paint.imagePattern()) {    // not image but color paint
        const IMaskPattern* paintPattern = paint.maskPattern();
        IGpiDevice::EMaskPattern pattern = fDevice->maskPattern(paintPattern);

        if (pattern != fDevice->kCustomPattern)
        {
             lb.lbStyle = BS_HATCHED;   // hatch brush and WinNT

             switch (pattern)
             {
             default:
             case fDevice->kSolidPattern:
                 lb.lbStyle = BS_SOLID;
                 break;
             case fDevice->kDiagonalDownPattern:
                 lb.lbHatch = HS_BDIAGONAL;
                 break;
             case fDevice->kCrossPattern:
                 lb.lbHatch = HS_CROSS;
                 break;
             case fDevice->kDiagonalCrossPattern:
                 lb.lbHatch = HS_DIAGCROSS;
                 break;
             case fDevice->kDiagonalUpPattern:
                 lb.lbHatch = HS_FDIAGONAL;
                 break;
			case fDevice->kHorizontalPattern:
                 lb.lbHatch = HS_HORIZONTAL;
                 break;
             case fDevice->kVerticalPattern:
                 lb.lbHatch = HS_VERTICAL;
                 break;
             }
        }
        else {  // do the custom hatch pattern...
            lb.lbStyle = BS_PATTERN;
            HBITMAP bitmap = CreateBitmap(8, 8, 1, 1,(const void*)(fDevice->maskBits(paintPattern)), fDevice->fHDC);
            lb.lbHatch = (long)bitmap;
        }
    }
    else
    {   // do the image pattern paint
        lb.lbStyle = BS_PATTERN;
        HBITMAP bitmap = paint.imagePattern()->handle();
        lb.lbHatch =(long)bitmap;
    }

    return(lb);

}

//==============================================================
//	The device maker implementation -
//	this should probably in a saperate file itself when we have more than one device
//

IGrafDevice* IGrafDeviceMaker::makeDevice(
	const IPresSpaceHandle& deviceContext,
	const IGrafPort::EGrafPortType  grafPortType,
	EDeviceMappingMode mappingMode)
{
  if (grafPortType == IGrafPort::kIExtendedRootGrafPort)
	   return new IGpiDevice(deviceContext,mappingMode, false);
	else
	   return new IGpiDevice(deviceContext,mappingMode, true);
}
//===================================================================
// A class that can reselect and delete a PEN object automatically
#pragma pack(push,4)
class IPenContext {
public:
	IPenContext(const IPresSpaceHandle& dc, HPEN handle);
	~IPenContext();
	void setPen(HPEN handle);

private:
	HPEN fPenOld;
	IPresSpaceHandle fHDC;
	// always save the HPEN that is currently selected to the fHDC.
  // This is  to guarantee that the destructor of IPenContext will
	// destroy the exact HPEN when it is not selected any more.
	HPEN fSavePen;
	
	// The following functions are declared private to prevent multiple copies
	// of a HPEN from being created
	IPenContext(const IPenContext&);
	IPenContext& operator=(const IPenContext&);
};
#pragma pack(pop)

IPenContext::IPenContext(const IPresSpaceHandle& dc, HPEN handle) :
	fHDC(dc)
{
	fPenOld = SelectObject(fHDC, handle);
	fSavePen = handle ;
}

IPenContext::~IPenContext()
{
	//DeleteObject(SelectObject(fHDC, fPenOld));
	SelectObject(fHDC, fPenOld);
	DeleteObject(fSavePen);
	
}

void IPenContext::setPen(HPEN handle)
{
	//DeleteObject(SelectObject(fHDC, handle));
	SelectObject(fHDC, handle);
	DeleteObject(fSavePen); //first deleted the HPEN that has just been deselected.
	fSavePen = handle ;   // then save the HPEN that is currently selected.
}


//===================================================================
// A class that can reselect and delete a HBRUSH object automatically
#pragma pack(push,4)
class IBrushContext{
public:
	IBrushContext(const IPresSpaceHandle& dc, HBRUSH handle, bool select = true);
	~IBrushContext();
	void setBrush(HBRUSH handle);

	HBRUSH fBrush;	// used for os2

private:
	bool fSelected;
	HBRUSH fBrushOld;
	IPresSpaceHandle fHDC;
	
  // always save the HBRUSH that is currently selected to the fHDC.
  // This is  to guarantee that the destructor of IBrushContext will
	// destroy the exact HBRUSH when it is not selected any more.
	HBRUSH fSaveBrush;
	
	
	// The following functions are declared private to prevent multiple copies
	// of a HBRUSH from being created
	IBrushContext(const IBrushContext&);
	IBrushContext& operator=(const IBrushContext&);
	void deleteBrush(HBRUSH) ;
};
#pragma pack(pop)

void IBrushContext::deleteBrush(HBRUSH brush)
{
	if(brush->usSet > 0 ) //pattern image associated
       GpiDeleteSetId(fHDC, brush->usSet);
	DeleteObject(brush);
}

IBrushContext::IBrushContext(const IPresSpaceHandle& dc, HBRUSH handle, bool select) :
	fHDC(dc),
	fBrush(handle),
	fSelected(select)
{
	if (fSelected)
		fBrushOld = SelectObject(fHDC, fBrush);
		fSaveBrush = fBrush;
}

IBrushContext::~IBrushContext()
{
	if (fSelected){
		//deleteBrush(SelectObject(fHDC, fBrushOld));
		SelectObject(fHDC, fBrushOld);
		deleteBrush(fSaveBrush);
  }		
	else
		deleteBrush(fBrush);
}

void IBrushContext::setBrush(HBRUSH handle)
{
	fBrush = handle;
	if (fSelected){
		//deleteBrush(SelectObject(fHDC, fBrush));
		SelectObject(fHDC, fBrush);
		deleteBrush(fSaveBrush);  //first deleted the HBRUSH that has just been deselected.
		fSaveBrush = fBrush;   // then save the HBRUSH that is currently selected.   }		
	}
	else {
		fBrushOld = SelectObject(fHDC, fBrush);
		fSaveBrush = fBrush;
		fSelected = true;
	}
}



//================================================================
//	helper class for font and text support...
//----------------------------------------------------------------
#pragma pack(push,4)
class IGdiFontContext {	// implements NT specific behaviou...
public:
	IGdiFontContext(
		IGpiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip,
		const IGrafState* state);
	virtual ~IGdiFontContext();

	void initialise();		// called by the IGpiDevice::makeFontContext...
	virtual void renderGlyphs(const IGTextRun&) const;
	virtual IGPoint2D calculateBoundsAndAdvancement(const IGTextRun&, IGRect2D& bounds);

	bool cacheHit(
		IGpiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip) const;

	unsigned long fMatrixStamp;
	unsigned long fFlipStamp;
	IGrafMatrix fMatrixUsed;				// used for NT transform, and normalised model matrix for w95
	const IGrafState* fGrafState;

protected:
	virtual void makeFont();	// called by initialise...
	virtual void extentPoint(const IGTextRun&, SIZE& extent) const;		// called by calculateBoundsAndAdvancement...

	IGpiDevice* fDevice;

	const IGrafMatrix* fAnchorTransform;	// for coordinate system flip prupose only...
	TEXTMETRIC fTm;
	bool fTmCached;
	HFONT fFont;
	HFONT fFontOld;
	
	HDC fDeviceContext;
	int fEscapement;
	IString fTypeFace;
	GCoordinate fPointSize;
	int fWeight;
	BYTE fItalic;
	BYTE fUnderline;
	BYTE fStrikeOut;

	// for caching GdiFontContext:
	const IGrafMatrix* fMatrix;
	const IGrafMatrix* fFlip;

  BYTE fOutline; // outline font
	BYTE fBitmap;	 //bitmap font
};

#pragma pack(pop)

// implements OS/2 and maybe Win32s behaviou...
#pragma pack(push,4)
class IGdiNonUnicodeFontContext : public IGdiFontContext {
public:
	IGdiNonUnicodeFontContext(
		IGpiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) ;

	virtual ~IGdiNonUnicodeFontContext() ;

	virtual void renderGlyphs(const IGTextRun&) const;

protected:
	virtual void extentPoint(const IGTextRun&, SIZE& extent) const;
};

IGdiNonUnicodeFontContext::~IGdiNonUnicodeFontContext()
{}

#pragma pack(pop)

//================================================================
#if !IC_RTTI_ENABLED
GrafDeviceTypeDefinitionsMacro(IGpiDevice);
#endif // IC_RTTI_ENABLED

//==============================================================
//	Statics for IGpiDevice(info shared by all instances)
//

// min/max values in device coordinates ... initialised to LONG_MIN/LONG_MAX, only Win95 had SHORT...
GCoordinate IGpiDevice::gMaxCoord = (GCoordinate)LONG_MAX - 1;
GCoordinate IGpiDevice::gMinCoord = (GCoordinate)LONG_MIN + 1;

bool IGpiDevice::fInitialized = false;

// IGpiDevice::EPlatform IGpiDevice::fPlatform = IGpiDevice::kUndefined;

#pragma enum(4)
enum EClipState {
	kClipAreaEmpty = 0,
	kClipAreaInfinite,
	kClipAreaRegular
};
#pragma enum(pop)

//==============================================================
//	Utility Routine for Translating from GCoordinate to integer
//

int IGpiDevice::GCoordinateToInt(GCoordinate value)
{
	if (value > 0)
	{
		value += 0.5;		// round up
		if (value > gMaxCoord)
			value = gMaxCoord;
	}
	else if (value < 0)
	{
		value -= 0.5;		// round down
		if (value < gMinCoord)
			value = gMinCoord;
	}

	return (int)floor(value);
}

//==============================================================
//	Constructors
//

IGpiDevice::IGpiDevice(const IPresSpaceHandle& deviceHandle,
		bool baseRootFlag) :
	fHDC(deviceHandle),
	fHDCOwned(0 /*NIL*/),
	fDrawOperation(IAttributeState::kUnset),
	fMatrix(),
	fResolution(0 /*NIL*/),
	fFontCache(0 /*NIL*/),
	fMatrixCacheUsed(false),
	fCachePaintModeStateChange(0),
	fCacheTransferMode(-1),
	fBackModeSet(false),
	fCachePenBrushStateChange(0),
	fNullPenWhenFill(false),
	fHollowBrushWhenFrame(false),
	fPenCache(0 /*NIL*/),
	fBrushCache(0 /*NIL*/),
	fClipAreaSet(false),
	fTextColourCached(false),
	fBaseRootDevice(baseRootFlag),
	fGpiEmulator(new IGpiEmulator(this))
{
  if (fBaseRootDevice)
	    fResolution = resolutionMatrix() ;
			
	setupDevice(deviceHandle);
	
	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	// unit = pels, origin = top-left
	initialise(MM_TEXT);
}

// I just hope mappingMode will die out... so I can remove this ctr
IGpiDevice::IGpiDevice(
	const IPresSpaceHandle& deviceHandle,
	EDeviceMappingMode mappingMode,
	bool baseRootFlag) :
	
	fHDC(deviceHandle),
	fHDCOwned(0 /*NIL*/),
	fDrawOperation(IAttributeState::kUnset),
	fMatrix(),
	fResolution(0),
	fFontCache(0 /*NIL*/),
	fMatrixCacheUsed(false),
	fCachePaintModeStateChange(0),
	fCacheTransferMode(-1),
	fBackModeSet(false),
	fCachePenBrushStateChange(0),
	fNullPenWhenFill(false),
	fHollowBrushWhenFrame(false),
	fPenCache(0 /*NIL*/),
	fBrushCache(0 /*NIL*/),
	fClipAreaSet(false),
	fTextColourCached(false),
	fBaseRootDevice(baseRootFlag),
	fGpiEmulator(new IGpiEmulator(this))
{
   if (fBaseRootDevice)
	    fResolution = resolutionMatrix() ;

	setupDevice(deviceHandle);

	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	int mmode;
	switch (mappingMode)
	{
	case kPixel:
	default:
		mmode = MM_TEXT;
		break;
	case kLowMetric:
		mmode = MM_LOMETRIC;
		break;
	case kHighMetric:
		mmode = MM_HIMETRIC;
		break;
	case kLowEnglish:
		mmode = MM_LOENGLISH;
		break;
	case kHighEnglish:
		mmode = MM_HIENGLISH;
		break;
	case kTwips:
		mmode = MM_TWIPS;
		break;
	}
	
	initialise(mmode);
}

IGpiDevice::IGpiDevice(
	const IPresSpaceHandle& deviceHandle,
	const IGRect2D& imagingRect,
	const IGrafMatrix& resolution) :

	fHDC(deviceHandle),
	fHDCOwned(0 /*NIL*/),
	fDrawOperation(IAttributeState::kUnset),
	fMatrix(),
	fResolution(new IGrafMatrix(resolution)),
	fFontCache(0 /*NIL*/),
	fMatrixCacheUsed(false),
	fCachePaintModeStateChange(0),
	fCacheTransferMode(-1),
	fBackModeSet(false),
	fCachePenBrushStateChange(0),
	fNullPenWhenFill(false),
	fHollowBrushWhenFrame(false),
	fPenCache(0 /*NIL*/),
	fBrushCache(0 /*NIL*/),
	fClipAreaSet(false),
	fTextColourCached(false),
	fGpiEmulator(new IGpiEmulator(this))
{
	// don't call setupDevice here because it is supposely owned by the caller...

	if (fResolution->isIdentity())
		fBaseRootDevice = false ;
	else
		fBaseRootDevice = true ;
		
	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	initialise(MM_TEXT, &imagingRect);
}

ICoordinateSystem::EOrientation IGpiDevice::coordinateSystem() const
{
	return ICoordinateSystem::kOriginLowerLeft;	// for OS/2, return default coordinate system
}

void IGpiDevice::setupDevice(
	const IPresSpaceHandle& deviceHandle)
{
	if (deviceHandle == 0 /*NIL*/) {
		// this way anyone can construct a desktop screen device...
		fHDC = GetWindowDC(HWND_DESKTOP);
		if (fHDC == 0 /*NIL*/)
			GrafDeviceException("GetWindowDC");
		fHDCOwned = fHDC;
	}
}

void IGpiDevice::initialise(int mmode, const IGRect2D* clipRect)
{
	if (!SetMapMode(fHDC, mmode))
		GrafDeviceException("SetMapMode");
	
	if (!fInitialized)
	{
		fInitialized = true;
//		fPlatform = kOS2;
	}
	
	if (!clipRect)
	{
		int width;
		int height;
		INativeWindowHandle hw = WindowFromDC(fHDC);
		if (hw)
		{
			// find the height for current client window
			RECT wrect;
			GetWindowRect(hw, &wrect);

			width = wrect.right - wrect.left;
			height = wrect.bottom - wrect.top;
		}
		else	// this may not be a device that has a Window (not on screen...)
		{
			width = GetDeviceCaps(fHDC, HORZRES);	// number of pixels per raster line...
			height = GetDeviceCaps(fHDC, VERTRES);	// number of raster lines per device...
		}
			
	  IGRect2D boundInPixel= IGRect2D(0, 0, width, height);
#if 0
	  if ( fBaseRootDevice){
		  IGrafMatrix matrix(*fResolution);
			matrix.invert();    // get the matrix that will convert pixels to points
			setWorldBounds(matrix.transformBounds(boundInPixel));  // worldBound should be in points
		}	
	  else
#endif
	    setWorldBounds(boundInPixel);		
			
	}
	else	// custom clipping ...
	{
		setWorldBounds(*clipRect);
	}
	
	fFrameMatrix = fModelMatrix = fViewMatrix = fPenMatrix = &fMatrix;
                                        // set hps to RGB mode to allow > 16 colors
	GpiCreateLogColorTable( fHDC, 0L, LCOLF_RGB, 0L, 0L, (PLONG)0 );
}


// these three functions are disabled...
IGpiDevice::IGpiDevice()
{}

IGpiDevice::IGpiDevice(const IGpiDevice&)
{}

IGpiDevice& IGpiDevice::operator=(const IGpiDevice& Src)
{
	return *this;
}

//==============================================================
//	Destructor
//

IGpiDevice::~IGpiDevice()
{
	if (isValid()) {
		RestoreDC(fHDC, fSavedDC);
		// this fails on a printer HDC, if print job has been dispatched. Don't throw exception
	}

	delete fPenCache;
	delete fBrushCache;
	delete fFontCache;
	delete fResolution;
        delete fGpiEmulator;

	if (fHDCOwned && fHDCOwned == fHDC) {
		ReleaseDC(WindowFromDC(fHDC), fHDC);
		// don't throw an exception here because we tried our best,
		// but we don't want disturb anyone, don't we?
	}
}

//==============================================================
//	device inquitries
//

const IGrafMatrix& IGpiDevice::deviceResolution(const IGRect2D&) const
{
	if (!fResolution)
		return IGrafMatrix::identity();
	else
		return *fResolution;
}


IGrafMatrix* IGpiDevice::resolutionMatrix()
{
    long HerRes;
	 long VerRes;
		
	  HDC hdc = GpiQueryDevice(fHDC);
		
    if( ! DevQueryCaps(hdc,CAPS_HORIZONTAL_FONT_RES, 1L, &HerRes))	
                 GrafDeviceException("DevQueryCaps");
								
    if( ! DevQueryCaps(hdc,CAPS_VERTICAL_FONT_RES, 1L, &VerRes))	
						    GrafDeviceException("DevQueryCaps");
	

	 // resolution is a scale matrix
	IGrafMatrix*   resolution = new IGrafMatrix(IGPoint2D(HerRes/72.0, VerRes/72.0),IGPoint2D::origin());

	return resolution ;
}		


bool IGpiDevice::baseRootDevice() const
{
  return fBaseRootDevice ;
}

// the coordinate system is inherited from the base class - lefthand, righthand on o/s2

const IPresSpaceHandle& IGpiDevice::deviceContext() const
{
	return fHDC;
}

IPresSpaceHandle IGpiDevice::orphanDeviceContext()
{
	if (isValid())
	{
		setValidity(false);
		// ??? save our graf-state of the device...

		RestoreDC(fHDC, fSavedDC);
		// this fails on a printer HDC, if print job has been dispatched. Don't throw exception
		IPresSpaceHandle temp = fHDC;	// save for return...

		// reset the caches... font cache knows how to deal with a new HDC
		fHDC = 0 /*NIL*/;
		fCachePaintModeStateChange = 0;
		fCacheTransferMode = -1;
		fBackModeSet = false;
		fCachePenBrushStateChange = 0;
		fNullPenWhenFill = false;
		fHollowBrushWhenFrame = false;
		delete fPenCache;
		fPenCache = 0 /*NIL*/;
		delete fBrushCache;
		fBrushCache = 0 /*NIL*/;
		delete fFontCache;
		fFontCache = 0 /*NIL*/;
		fMatrixCacheUsed = false;
		fClipAreaSet = false;
		fTextColourCached = false;
		return temp;
	}
	return fHDC;
}

void IGpiDevice::adoptDeviceContext(const IPresSpaceHandle& dc)
{
	fHDC = dc;

	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	setValidity(true);
}

//==============================================================
//	IGpiDevice::SetFramePaint(...)
//
void IGpiDevice::SetFillPaint(const IGrafState& grafState, bool nullPen) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFillChanges).attributeStateChanged() ||
		fCachePenBrushStateChange != IStateChange::kFillChanges)
#endif
	{
		if (fNullPenWhenFill != nullPen) {
			if (nullPen) {
				if (fPenCache)
					fPenCache->setPen(GetStockObject(NULL_PEN));
				else
					fPenCache = new IPenContext(fHDC, GetStockObject(NULL_PEN));
			}
			fNullPenWhenFill = nullPen;
		}
		if (fBrushCache)
			fBrushCache->setBrush(fGpiEmulator->CreateFillBrush(grafState));
		else
			fBrushCache = new IBrushContext(fHDC, fGpiEmulator->CreateFillBrush(grafState));
		fHollowBrushWhenFrame = false;
		fCachePenBrushStateChange = IStateChange::kFillChanges;
	}
}

//==============================================================
//	IGpiDevice::SetFillPaint(...)
//
void IGpiDevice::SetFramePaint(const IGrafState& grafState, bool hollowBrush) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFrameChanges).attributeStateChanged() ||
		fCachePenBrushStateChange != IStateChange::kFrameChanges)
#endif
	{
		if (fHollowBrushWhenFrame != hollowBrush) {
			if (hollowBrush) {
				if (fBrushCache)
					fBrushCache->setBrush(GetStockObject(HOLLOW_BRUSH));
				else
					fBrushCache = new IBrushContext(fHDC, GetStockObject(HOLLOW_BRUSH));
			}
			fHollowBrushWhenFrame = hollowBrush;
		}
      if (fPenCache)
			fPenCache->setPen(fGpiEmulator->CreatePen(grafState));
      else
			fPenCache = new IPenContext(fHDC, fGpiEmulator->CreatePen(grafState));
		fNullPenWhenFill = false;
		fCachePenBrushStateChange = IStateChange::kFrameChanges;
	}
}

//==============================================================
//	IGpiDevice::SetFillAndFramePaint(...)
//
void IGpiDevice::SetFillAndFramePaint(const IGrafState& grafState) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFillOrFrameChanges).attributeStateChanged() ||
		fCachePenBrushStateChange != IStateChange::kFillOrFrameChanges)
#endif
	{
		if (fPenCache)
			fPenCache->setPen(fGpiEmulator->CreatePen(grafState));
		else
			fPenCache = new IPenContext(fHDC, fGpiEmulator->CreatePen(grafState));
		if (fBrushCache)
			fBrushCache->setBrush(fGpiEmulator->CreateFillBrush(grafState));
		else
			fBrushCache = new IBrushContext(fHDC, fGpiEmulator->CreateFillBrush(grafState));
		fNullPenWhenFill = false;
		fHollowBrushWhenFrame = false;
		fCachePenBrushStateChange = IStateChange::kFillOrFrameChanges;
	}
}

//==============================================================
//	IGpiDevice::SetFillPaintModes(...)
//

void IGpiDevice::SetFillPaintModes(const IGrafState& state) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFillChanges).attributeStateChanged() ||
		fCachePaintModeStateChange != IStateChange::kFillChanges)
#endif
	{
		const IAttributeState* attState = state.attributeState();
		SetPaintModes(attState->fillPaint(), attState->fillTransferMode());
		fCachePaintModeStateChange = IStateChange::kFillChanges;
	}
}

//==============================================================
//	IGpiDevice::SetFramePaintModes(...)
//

void IGpiDevice::SetFramePaintModes(const IGrafState& state) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFrameModeChanges).attributeStateChanged() ||
		fCachePaintModeStateChange != IStateChange::kFrameModeChanges)
#endif
	{
		const IAttributeState* attState = state.attributeState();
		SetPaintModes(attState->framePaint(), attState->frameTransferMode());
		fCachePaintModeStateChange = IStateChange::kFrameModeChanges;
	}
}

//==============================================================
//	IGpiDevice::SetPaintModes(...)
//

void IGpiDevice::SetPaintModes(
	const IPaint* paint,
	const IColorTransferMode* xmode)
{
	if (paint->maskPattern() != &IMaskPattern::solid() || paint->imagePattern())
	{
		const IGPoint2D* patPhase = paint->patternPhase();

		// set Pattern Origin
		if (!SetBrushOrgEx(fHDC, ((int)patPhase->fX) % 8, ((int)patPhase->fY) % 8, NULL))
			GrafDeviceException("SetBrushOrgEx");

    if (maskPattern(paint->maskPattern()) == kCustomPattern)
      SetBkColor(fHDC, fGpiEmulator->convertToWRGB(*(paint->color())));

	}
	
	if (!fBackModeSet) {
		// set Background Mode
		if (!SetBkMode(fHDC, TRANSPARENT))
			GrafDeviceException("SetBkMode");

		fBackModeSet = true;
	}

	unsigned long theMode = xmode->mode();
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if (fCacheTransferMode != theMode)
#endif
		 {

		fCacheTransferMode = theMode;
	
		// set transfermode
		int wmixMode;
		
		// in Win SDK MASK=AND, MERGE=OR, XOR=XOR
		switch (fCacheTransferMode)
		{
		default:
		case IColorTransferMode::kSourceCopy:
			wmixMode = R2_COPYPEN;
			break;
		case IColorTransferMode::kInvertSource:
			wmixMode = R2_NOTCOPYPEN;
			break;
		case IColorTransferMode::kDestinationCopy:
			wmixMode = R2_NOP;
			break;
		case IColorTransferMode::kInvertDestination:
			wmixMode = R2_NOT;
			break;
		case IColorTransferMode::kOR:
			wmixMode = R2_MERGEPEN;
			break;
		case IColorTransferMode::kAND:
			wmixMode = R2_MASKPEN;
			break;
		case IColorTransferMode::kXOR:
			wmixMode = R2_XORPEN;
			break;
		case IColorTransferMode::kInvertedSourceAND:
			wmixMode = R2_MASKNOTPEN;
			break;
		case IColorTransferMode::kInvertedSourceOR:
			wmixMode = R2_MERGENOTPEN;
			break;
		case IColorTransferMode::kInvertedDestinationAND:
			wmixMode = R2_MASKPENNOT;
			break;
		case IColorTransferMode::kInvertedDestinationOR:
			wmixMode = R2_MERGEPENNOT;
			break;
		case IColorTransferMode::kInvertedAND:
			wmixMode = R2_NOTMASKPEN;
			break;
		case IColorTransferMode::kInvertedOR:
			wmixMode = R2_NOTMERGEPEN;
			break;
		case IColorTransferMode::kInvertedXOR:
			wmixMode = R2_NOTXORPEN;
			break;
		case IColorTransferMode::kONES:
			wmixMode = R2_WHITE;
			break;
		case IColorTransferMode::kZEROS:
			wmixMode = R2_BLACK;
			break;
		}
		
		if (!SetROP2(fHDC, wmixMode))
			GrafDeviceException("SetROP2");
	}
}


//==============================================================
//	IGpiDevice::FrameEqualFillPaintModes
//

bool IGpiDevice::FrameEqualFillPaintModes(const IGrafState& state) const
{
	const IAttributeState* attState = state.attributeState();

	if (*attState->fillTransferMode() != *attState->frameTransferMode())
		return false;

	const IPaint* fillPaint = attState->fillPaint();
	const IPaint* framePaint = attState->framePaint();

	if (*fillPaint != *framePaint || !fillPaint->imagePattern())
		return false;	// different or at least one paint is a pattern paint
	
	return true;
}

//==============================================================
//	IGpiDevice::imageTransferMode
//

#define ICLUIHalfToneMode 0X00FC008A

int IGpiDevice::imageTransferMode(const IGrafState& state)
{
	int GdiMode = SRCCOPY;
	
	switch (state.attributeState()->imageTransferMode()->mode())
	{
	case IImageTransferMode::kSrcCopy:
		GdiMode = SRCCOPY;
		break;
	case IImageTransferMode::kSrcORDst:
		GdiMode = SRCPAINT;
		break;
	case IImageTransferMode::kSrcANDDst:
		GdiMode = SRCAND;
		break;
	case IImageTransferMode::kSrcXORDst:
		GdiMode = SRCINVERT;
		break;
	case IImageTransferMode::kSrcANDInvertDst:
		GdiMode = SRCERASE;
		break;
	case IImageTransferMode::kInvertSrc:
		GdiMode = NOTSRCCOPY;
		break;
	case IImageTransferMode::kInvertResultOfSrcORDst:
		GdiMode = NOTSRCERASE;
		break;
	case IImageTransferMode::kSrcANDPattern:
		GdiMode = MERGECOPY;
		break;
	case IImageTransferMode::kInvertSrcORDst:
		GdiMode = MERGEPAINT;
		break;
	case IImageTransferMode::kPatternCopy:
		GdiMode = PATCOPY;
		break;
	case IImageTransferMode::kPatternORInvertSrcORDst:
		GdiMode = PATPAINT;
		break;
	case IImageTransferMode::kPatternXORDst:
		GdiMode = PATINVERT;
		break;
	case IImageTransferMode::kInvertDst:
		GdiMode = DSTINVERT;
		break;
	case IImageTransferMode::kAllZeros:
		GdiMode = BLACKNESS;
		break;
	case IImageTransferMode::kAllOnes:
		GdiMode = WHITENESS;
		break;
	case IImageTransferMode::kOCHalfTone:
		// do it just like in ICLUI, make it a special code - 0x00fc008a
		GdiMode = ICLUIHalfToneMode;
		break;
	}
	
	return GdiMode;
}


//==============================================================
//	IGpiDevice::imageSamplingControl
//

int IGpiDevice::imageSamplingControl(const IGrafState& state)
{
	int GdiMode = COLORONCOLOR;
	switch (state.attributeState()->imageSampling()->mode())
	{
	case IImageSamplingControl::kBlackOnWhite:
		GdiMode = BLACKONWHITE;
		break;
	case IImageSamplingControl::kWhiteOnBlack:
		GdiMode = WHITEONBLACK;
		break;
	case IImageSamplingControl::kColorOnColor:
		GdiMode = COLORONCOLOR;
		break;
	case IImageSamplingControl::kHalfTone:
		GdiMode = HALFTONE;
		break;
	}
	return GdiMode;
}



//==============================================================
//	IGpiDevice::SetClipArea
//
//	Note: setupTransforms has to be called prior to calling this function, because this uses
//			fFrameMatrix.
//		returns: kClipAreaEmpty (shouldn't draw anything - clipping is not set)
//		         kClipAreaInfinite (no clipping - clipping reset to NULL)
//		         kClipAreaRegular (other - clipping set)

int IGpiDevice::SetClipArea(const IGrafState& state)
{
	// clipArea is transformed in GrafState.
	const IGArea* clipArea = state.clipArea();
		
	if (clipArea->isEmpty()) return(kClipAreaEmpty);

	if (fClipAreaSet) {
		if (!fStateChange.clipAreaChanged())
			return kClipAreaRegular;
	}
	else
		fClipAreaSet = true;
		
	if (clipArea->bounds() == IGRect2D::infiniteRect())
	{
		if (ExtSelectClipRgn(fHDC, NULL, RGN_COPY) == ERROR)
		{	// set clip region to NULL
			GrafDeviceException("ExtSelectClipRgn");
		}
		return kClipAreaInfinite;
	}
	else {
	
		IPmClipPathVertextEngine* polygoner= new IPmClipPathVertextEngine(fHDC);
		IOutlineMakerVertexEngine *outliner = new IOutlineMakerVertexEngine(polygoner);
		ISamplingExtractor sampler(outliner, IGRect2D::infiniteRect());

		// set to identity
		SetWinWorldTransform(&IGrafMatrix::identity());

		ICAGRoot root;
		extract(*clipArea, sampler, &root);
		if( sampler.isExtractedRectEmpty() ) return(kClipAreaEmpty);

		sampler.render(root);
		return kClipAreaRegular;	
	}
}


//==============================================================
//	IGpiDevice::SetWinWorldTransform
//

void IGpiDevice::SetWinWorldTransform(const IGrafMatrix* grafmatrix)
{
//	if (fPlatform != kWin95)
//	{
		if (!SetGraphicsMode(fHDC, GM_ADVANCED))	// to enable world transform
			GrafDeviceException("SetGraphicsMode");
		
		if (!grafmatrix || grafmatrix->isIdentity())
		{
			if (fMatrixCacheUsed) {
				if (fMatrixCache.isIdentity())
					return; // cache hit...
			}
			else
				fMatrixCacheUsed = true;

			fMatrixCache.setToIdentity();

			if (!ModifyWorldTransform(fHDC, 0, MWT_IDENTITY))
				GrafDeviceException("ModifyWorldTransform");
		}
		else { // not identity
			if (fMatrixCacheUsed) {
				if (fMatrixCache == *grafmatrix)
					return; // cache hit...
			}
			else
				fMatrixCacheUsed = true;

			fMatrixCache = *grafmatrix;
			fGpiEmulator->fXFORM.eM11 = FLOAT(fMatrixCache.element(IGrafMatrix::kScaleX));
			fGpiEmulator->fXFORM.eM12 = FLOAT(fMatrixCache.element(IGrafMatrix::kShearY));
			fGpiEmulator->fXFORM.eM21 = FLOAT(fMatrixCache.element(IGrafMatrix::kShearX));
			fGpiEmulator->fXFORM.eM22 = FLOAT(fMatrixCache.element(IGrafMatrix::kScaleY));
			fGpiEmulator->fXFORM.eDx = FLOAT(fMatrixCache.element(IGrafMatrix::kTranslateX));
			fGpiEmulator->fXFORM.eDy = FLOAT(fMatrixCache.element(IGrafMatrix::kTranslateY));
		
			if (!SetWorldTransform(fHDC, &(fGpiEmulator->fXFORM)))
				GrafDeviceException("SetWorldTransform");
		}
//	}
	}


/* ------------------------------------------------------------
transform issue:

	T(model) = T(geometry) . T(view) (short: Tm = Tg.Tv)

	NT:
		For given geometry g,

		G = g.Tg;
		
		if (Tv ~perspective)
			extract Tv = affine(Tv)
		
		render(G) with Tv

	W95:
		For given geometry g,
		G = g.Tm; set penwidth by using Tv
		render(G)

------------------------------------------------------------ */

void IGpiDevice::setupTransforms(const IGrafState& state)
{
	if (!fStateChange.matrixChanged())
		return;
	fModelMatrix = state.modelMatrix();
//	if (fPlatform == kWin95) {
//		fViewMatrix = 0 /*NIL*/;
//		fFrameMatrix = fModelMatrix;
//		fPenMatrix = state.viewMatrix();
//	}
//	else {	// WinNT & OS2
		fViewMatrix = state.viewMatrix();
		fMatrix = *fViewMatrix;
		fMatrix.invert();
		fMatrix.preConcatWith(*fModelMatrix);
		fFrameMatrix = &fMatrix;
      fPenMatrix = 0 /*NIL*/;
//	}
}


//==============================================================
//	IGpiDevice::renderLine
//

void IGpiDevice::renderLine(const IGLine2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;

	IGLine2D tline(geometry);
	if (!fFrameMatrix->isIdentity())
		tline.transformBy(*fFrameMatrix);
	
	SetWinWorldTransform(fViewMatrix);
	
	IGPoint2D pts[2];
	tline.points(pts[0], pts[1]);

   GCoordinate deltaX= pts[1].fX - pts[0].fX;
   GCoordinate deltaY= pts[1].fY - pts[0].fY;

   if ( deltaX == 0 && deltaY == 0) { // if line is actually an emppty line
         return ;
   }


	// create Pen and select it to the DC
	SetFramePaint(grafState);
	SetFramePaintModes(grafState);
	
	// draw Line
	if (!MoveToEx(fHDC, GCoordinateToInt(pts[0].fX), GCoordinateToInt(pts[0].fY), NULL))
		GrafDeviceException("MoveToEx");

	//Remove the last pixel point for the line to make it pixel perfect on all
	// platform. However, its not possible to make pixel perfect in all direction
	// so, decided to make it only on horizontal, vertical and diagonal directions.


   if (deltaY == 0) { //if line's in horizontal direction
      if (deltaX <0) {
          if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX+1.0), GCoordinateToInt(pts[1].fY)))
            GrafDeviceException("LineTo");
      }
      else {
         if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX-1.0), GCoordinateToInt(pts[1].fY)))
            GrafDeviceException("LineTo");
      }
   } else if (deltaX==0 ) { //if in vertical direction
      if (deltaY <0) {
            if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX), GCoordinateToInt(pts[1].fY+1.0)))
               GrafDeviceException("LineTo");
      }
      else {
            if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX), GCoordinateToInt(pts[1].fY-1.0)))
               GrafDeviceException("LineTo");
     }
   } else if (deltaX == deltaY) { //if diagonal line
      if (deltaY <0) {
         if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX+1.0), GCoordinateToInt(pts[1].fY+1.0)))
            GrafDeviceException("LineTo");
      }
      else {
         if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX-1.0), GCoordinateToInt(pts[1].fY-1.0)))
            GrafDeviceException("LineTo");
      }
   } else if (deltaX == -deltaY) { //if diagonal line
      if (deltaX <0) {
         if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX-1.0), GCoordinateToInt(pts[1].fY+1.0)))
            GrafDeviceException("LineTo");
      }
      else {
         if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX+1.0), GCoordinateToInt(pts[1].fY-1.0)))
            GrafDeviceException("LineTo");
      }
   } else
      if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX), GCoordinateToInt(pts[1].fY)))
         GrafDeviceException("LineTo");
}


//==============================================================
//	IGpiDevice::renderPolyline
//

void IGpiDevice::renderPolyline(const IGPolyline2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGPolyline2D tpolyline(geometry);
	if (!fFrameMatrix->isIdentity())
		tpolyline.transformBy(*fFrameMatrix);
	
	SetWinWorldTransform(fViewMatrix);
	
	// set up gdi points for polyline.
	long numPoints = tpolyline.numberOfPoints();
	IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
	IGPoint2D pixel1(tpolyline.point(numPoints-2));
	IGPoint2D pixel2(tpolyline.point(numPoints-1));

   GCoordinate deltaX = pixel2.fX - pixel1.fX;
   GCoordinate deltaY = pixel2.fY - pixel1.fY;

   if (deltaY == 0) { //if line's in horizontal direction
      if (deltaX <0)
         tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX+1.0, pixel2.fY));
      else tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX-1.0, pixel2.fY));
   } else if (deltaX==0 ) { //if in vertical direction
         if (deltaY <0)
            tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX, pixel2.fY+1.0));
         else tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX, pixel2.fY-1.0));
   } else if (deltaX == deltaY) { //if diagonal line
      if (deltaY <0)
         tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX+1.0, pixel2.fY+1.0));
      else
         tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX-1.0, pixel2.fY-1.0));
   } else if (deltaX == -deltaY) { //if diagonal line
      if (deltaX <0)
         tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX-1.0, pixel2.fY+1.0));
	  else
         tpolyline.setPoint(numPoints-1, IGPoint2D(pixel2.fX+1.0, pixel2.fY-1.0));
   }


	for (unsigned long i=0; i < numPoints; i++)
		fGpiEmulator->IGPoint2DToPOINTL(tpolyline.point(i), sp[i]);

	// create Pen and select it to the DC
	SetFramePaint(grafState);
	SetFramePaintModes(grafState);
	
	// draw Polyline
	if (!Polyline(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))	
		GrafDeviceException("Polyline");
}

//==============================================================
//	IGpiDevice::renderRect
//

void IGpiDevice::renderRect(const IGRect2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;

	IGRect2D trect(geometry);
	
	fDrawOperation = grafState.attributeState()->drawingOperation();

//	if (fPlatform == kWin95 ||
	if (
		fDrawOperation == IAttributeState::kFillAndFrame ||
		fDrawOperation == IAttributeState::kFrame)
	{
		if (!fFrameMatrix->isIdentity())
		{
			if (fFrameMatrix->isRectilinear())
			{
				trect = fFrameMatrix->transformBounds(trect);
			}
			else
			{
				IGPolygon2D polygon(4);
				polygon.setPoint(0, trect.topLeft());
				polygon.setPoint(1, trect.bottomLeft());
				polygon.setPoint(2, trect.bottomRight());
				polygon.setPoint(3, trect.topRight());
	
				renderPolygon(polygon, grafState);
						
				return;
			}
		}
		else
		{
			// just incase someone made the transposed rectangle or it is transformed to be transposed:
			trect.orderPoints();
		}
		SetWinWorldTransform(fViewMatrix);
	}
	else	// not Win95 and fDrawOperation is kFill only
	{
		// just incase someone made the transposed rectangle or it is transformed to be transposed:
		trect.orderPoints();
		SetWinWorldTransform(fModelMatrix);
	}

	if (trect.contains(worldBounds()))	// for infinite rect or somewhat huge rect...
		trect = worldBounds();

	POINTL gp1;
	POINTL gp2;
	fGpiEmulator->IGPoint2DToPOINTL(trect.topLeft(), gp1);
	fGpiEmulator->IGPoint2DToPOINTL(trect.bottomRight(), gp2);

	
	// draw rectangle
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState))
	{
		// create a PEN and select it in DC
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);
		
		if (!Rectangle(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
			GrafDeviceException("Rectangle");
	}
	else
	{
		// it is important to do fill before frame
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState, true);
			SetFillPaintModes(grafState);
			if (!Rectangle(fHDC, int(gp1.x), int(gp1.y), int(gp2.x)-1, int(gp2.y)-1 ))
				GrafDeviceException("Rectangle");
		}
		// fill in OS2 is one pixel more than fill in NT at the right and bottom boundary
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState);
			SetFramePaintModes(grafState);
			
			BeginPath(fHDC);
			if (!Rectangle(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
				GrafDeviceException("Rectangle");
				CloseFigure(fHDC);		
			EndPath(fHDC);

			if (!StrokePath(fHDC))
				GrafDeviceException("StrokePath");
		}
	}
}

//==============================================================
//	IGpiDevice::renderEllipse
//

void IGpiDevice::renderEllipse(const IGEllipse2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	fDrawOperation = grafState.attributeState()->drawingOperation();
	
	IGRect2D trect(geometry.bounds());
	
	// Try to fix defect 3102
	// Avoid using Win32 functions when drawing Arc, Chord or Ellipse on Win95
	// platform since those calls result in crashes sometimes
	// Crash Example: Ellipse(hdc, -670, -3696, 3531, 505)
//	if (fPlatform == kWin95 || fPlatform == kOS2)
//{
		IGLoop2D elliptical(geometry);
		renderLoop(elliptical, grafState);
		return;
//	}	
	
/*	//commmented to fix defect 3102
	if (fPlatform == kWin95 ||
		fDrawOperation == IAttributeState::kFillAndFrame ||
		fDrawOperation == IAttributeState::kFrame)
*/		
//	if (fDrawOperation == IAttributeState::kFillAndFrame ||
//		fDrawOperation == IAttributeState::kFrame)
//	{
//		if (!fFrameMatrix->isIdentity())
//		{
//			if (fFrameMatrix->isRectilinear())
//			{
//				trect = fFrameMatrix->transformBounds(trect);
//			}
//			else
//			{
//				IGLoop2D elliptical(geometry);
//				renderLoop(elliptical, grafState);
//						
//				return;
//			}
//		}
//		SetWinWorldTransform(fViewMatrix);
//	}
//	else	// not Win95 and fDrawOperation is kFill only
//		SetWinWorldTransform(fModelMatrix);
//	
//	// just incase someone made the transposed rectangle or it is transformed to be transposed:
//	trect.orderPoints();
//	
//	POINTL gp1;
//	POINTL gp2;
//	fGpiEmulator->IGPoint2DToPOINTL(trect.topLeft(), gp1);
//	fGpiEmulator->IGPoint2DToPOINTL(trect.bottomRight(), gp2);
//
//	// draw ellipse
//	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState))
//	{
//		SetFillAndFramePaint(grafState);
//		SetFillPaintModes(grafState);
//
//		if (!Ellipse(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
//			GrafDeviceException("Ellipse");
//	}
//	else
//	{
//		// it is important to do fill before frame
//		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
//		{
//			SetFillPaint(grafState, true);
//			SetFillPaintModes(grafState);
//			
//			if (!Ellipse(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
//				GrafDeviceException("Ellipse");
//		}
//		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
//		{
//			SetFramePaint(grafState, true);
//			SetFramePaintModes(grafState);
//			
//			if (!Ellipse(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
//				GrafDeviceException("Ellipse");
//		}
//	}
	}
	
//==============================================================
//	IGpiDevice::renderPolygon
//

void IGpiDevice::renderPolygon(const IGPolygon2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGPolygon2D tpolygon(geometry);
	if (!fFrameMatrix->isIdentity())
		tpolygon.transformBy(*fFrameMatrix);
	
	SetWinWorldTransform(fViewMatrix);
	
	// set up gdi points for polygon.
	long numPoints = tpolygon.numberOfPoints();
	IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
	for (unsigned long i=0; i < numPoints; i++)
		fGpiEmulator->IGPoint2DToPOINTL(tpolygon.point(i), sp[i]);
	
	fDrawOperation = grafState.attributeState()->drawingOperation();
	
	// draw polygon
	if (tpolygon.eOFill())
		SetPolyFillMode(fHDC, ALTERNATE);	// even-odd rule
	else
		SetPolyFillMode(fHDC, WINDING);		// non-zero winding rule
	
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState))
	{
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);
		
		if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
			GrafDeviceException("Polygon");
	}
	else
	{
		// it is important to do fill before frame
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState);
			SetFillPaintModes(grafState);
			
			BeginPath(fHDC);
			if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
				GrafDeviceException("Polygon");
			CloseFigure(fHDC); // to make sure the endings of the final intersection
			                   // are drawn with joints (need to verify)
			EndPath(fHDC);

			if (!FillPath(fHDC))
				GrafDeviceException("FillPath");

		}
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState);
			SetFramePaintModes(grafState);
			
			BeginPath(fHDC);
			if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
				GrafDeviceException("Ellipse");
			CloseFigure(fHDC); // to make sure the endings of the final intersection
			                   // are drawn with joints (need to verify)
			EndPath(fHDC);

			if (!StrokePath(fHDC))
				GrafDeviceException("StrokePath");

		}
	}
}

//==============================================================
//	IGpiDevice::renderArea
//

void IGpiDevice::renderArea(const IGArea& area, const IGrafState& grafState)
{
	if (area.isEmpty())
		return;
		
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;

	SetWinWorldTransform(fViewMatrix);
	
	IPmRenderPathVertextEngine *polygoner = new IPmRenderPathVertextEngine(fHDC);
	IOutlineMakerVertexEngine *outliner= new IOutlineMakerVertexEngine(polygoner);
	ISamplingExtractor sampler(outliner, IGRect2D::infiniteRect());

	ICAGRoot root;
	extract(area, sampler, &root, *fFrameMatrix);
	if( sampler.isExtractedRectEmpty() ) return;

	sampler.render(root);

	fDrawOperation = grafState.attributeState()->drawingOperation();
	
	// paint the resulting region(fill)
	if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
	{
		SetFillPaint(grafState);
		SetFillPaintModes(grafState);
		
		if (!polygoner->FillArea(fHDC))
			GrafDeviceException("pm?FillArea");
	}
	
	// frame the resulting region
	if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
	{
		const IPen* pen = grafState.attributeState()->framePen();
		IPen::EPenType penType = pen->penType();
		int frameWidth, frameHeight;			// must be logical units
		if (penType == IPen::kHairline)
		{
			frameWidth = frameHeight = 1;
		}
		else if (penType == IPen::kSolid)
		{
			if (fPenMatrix && !fPenMatrix->isIdentity())
			{
				// pen width calculation ... only done on Win95
				IGPoint2D pts[2];	// unit vector
				pts[0] = IGPoint2D::origin();
				pts[1] = IGPoint2D(1, 1);
				fPenMatrix->transformPoints(pts, pts, 2);
				pts[1] -= pts[0];
				pts[0].fX = pen->penWidth();
				frameWidth = GCoordinateToInt(pts[1].fX * pts[0].fX);
				frameHeight = GCoordinateToInt(pts[1].fY * pts[0].fX);
			}
			else
				frameWidth = frameHeight = GCoordinateToInt(pen->penWidth());

			frameWidth = frameWidth > 1 ? frameWidth : 1;	// must always be >= 1
			frameHeight = frameHeight > 1 ? frameHeight : 1;	// must always be >= 1
		}
		else	// else punt on other kinds of pens ...???
			frameWidth = frameHeight = 1;
		
		SetFramePaint(grafState);
		SetFramePaintModes(grafState);
		
		if(fDrawOperation == IAttributeState::kFillAndFrame)
			polygoner->Render(root);

		if (!polygoner->StrokeArea(fHDC))
			GrafDeviceException("pm?StrokeArea");
	}
}

#ifndef NO_CONIC_ACCELERATION_SUPPORT
#include "conics.hpp"

#pragma pack(push,4)
class ConicMaker {
private:
	int fLeft, fTop, fRight, fBottom;
	int fStartX, fStartY, fEndX, fEndY;
	IConicInfo::ECurveType fType;
	bool fIsLoop;

	void SetupForRendering(const IGCurve2D& curve) ;

public:
	ConicMaker(const IGCurve2D& curve) ;
	ConicMaker(const IGLoop2D& loop) ;

	void render(const IPresSpaceHandle& dc) ;
};

void ConicMaker::SetupForRendering(const IGCurve2D& curve) {
	IConicInfo* ci = curve.fConicInfo;
	fType = ci->fType;

	fLeft = IGpiDevice::GCoordinateToInt(ci->fArcBounds.fLeft);
	fTop = IGpiDevice::GCoordinateToInt(ci->fArcBounds.fTop);
	fRight = IGpiDevice::GCoordinateToInt(ci->fArcBounds.fRight);
	fBottom = IGpiDevice::GCoordinateToInt(ci->fArcBounds.fBottom);

	if (fType != IConicInfo::kEllipse) {
		IGPoint2D startPoint, endPoint;
		if (fType == IConicInfo::kArc) {
			startPoint = curve.evaluate(curve.minParameter());
			endPoint = curve.evaluate(curve.maxParameter());
		}
		else {
			startPoint = ci->fStart;
			endPoint = ci->fEnd;
		}

		if (ci->fDirection) {
			fStartX = IGpiDevice::GCoordinateToInt(endPoint.fX);
			fStartY = IGpiDevice::GCoordinateToInt(endPoint.fY);
			fEndX = IGpiDevice::GCoordinateToInt(startPoint.fX);
			fEndY = IGpiDevice::GCoordinateToInt(startPoint.fY);
		}
		else {
			fStartX = IGpiDevice::GCoordinateToInt(startPoint.fX);
			fStartY = IGpiDevice::GCoordinateToInt(startPoint.fY);
			fEndX = IGpiDevice::GCoordinateToInt(endPoint.fX);
			fEndY = IGpiDevice::GCoordinateToInt(endPoint.fY);
		}
	}
}

ConicMaker::ConicMaker(const IGCurve2D& curve) {
	SetupForRendering(curve);
	fIsLoop = false;
}

ConicMaker::ConicMaker(const IGLoop2D& loop) {
	SetupForRendering(loop);
	fIsLoop = true;
}

void ConicMaker::render(const IPresSpaceHandle& dc) {
	if (fType == IConicInfo::kEllipse) {
		if (!Ellipse(dc, fLeft, fTop, fRight, fBottom))
			GrafDeviceException("Ellipse");
	}
	else if (fIsLoop) {
		if (fType == IConicInfo::kPie) {
			if (!Pie(dc, fLeft, fTop, fRight, fBottom, fStartX, fStartY, fEndX, fEndY))
				GrafDeviceException("Pie");
		}
		else {
			if (!Chord(dc, fLeft, fTop, fRight, fBottom, fStartX, fStartY, fEndX, fEndY))
				GrafDeviceException("Chord");
		}
	}
	else {
		if (!Arc(dc, fLeft, fTop, fRight, fBottom, fStartX, fStartY, fEndX, fEndY))
			GrafDeviceException("Arc");
	}
}

#pragma pack(pop)
#endif // NO_CONIC_ACCELERATION_SUPPORT

#pragma pack(push,4)
class IPmPolySplineException : public IGraphicException {
public:
	IPmPolySplineException(const char* string) ;
	~IPmPolySplineException() ;
};

IPmPolySplineException:: IPmPolySplineException(const char* string) :
IGraphicException(string, IGraphicException::kGenericGraphicException)
{}

IPmPolySplineException::~IPmPolySplineException()
{}

#pragma pack(pop)

//================================================================
//	IGpiDevice::renderCurve
//
// Parameters:
//	geometry : the curve to be rendered
//	grafState: indicates the attributes/transformation to be applied
//
// Return value: NONE
//
// Purpose: renders the curve geometry to the display
//
//
void IGpiDevice::renderCurve(const IGCurve2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGCurve2D tcurve(geometry);
	// Try to fix defect 3102
	// Avoid using Win32 functions when drawing Arc, Chord or Ellipse on Win95
	// platform since those calls result in crashes sometimes
	// Crash Example: Ellipse(hdc, -670, -3696, 3531, 505)
	//                Arc(hdc, -670, -3696, 3531, 505, 290, 168, 132, 55)
//	if ((fPlatform == kWin95 || fPlatform == kOS2) && conicInfo(tcurve))
	if (conicInfo(tcurve))
#ifdef MAYBE_A_BUG
		tcurve.fConicInfo = 0 /*NIL*/;
#else
		deleteConicInfo(tcurve);
#endif // MAYBE_A_BUG
	
	if (!fFrameMatrix->isIdentity()) {
		tcurve.transformBy(*fFrameMatrix);
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		if (conicInfo(tcurve)) {
			if (!conicInfo(tcurve)->transformBy(*fFrameMatrix)) {
				deleteConicInfo(tcurve);
			}
		}
#endif // NO_CONIC_ACCELERATION_SUPPORT
	}
	
	SetWinWorldTransform(fViewMatrix);
	
	// create Pen and select it to the DC
	SetFramePaint(grafState);
	SetFramePaintModes(grafState);
	
#ifndef NO_CONIC_ACCELERATION_SUPPORT
	if (conicInfo(tcurve))
	{	// if curve is supported by windows and is an arc...
		ConicMaker(tcurve).render(fHDC);
	}
	else
#endif // NO_CONIC_ACCELERATION_SUPPORT
	{
		try {
		unsigned long numPoints;
		unsigned long order = tcurve.order();

		if (order > 2) {	// make everything order == 4 beziers...
			if (order < 4)
				tcurve.raiseOrder(4);
			else if (order > 4)
				tcurve.approximateLowerOrder(4, 0.5);	// bezified first and then tessolated...

			if (!tcurve.isBezier())
				tcurve.refineToBeziers();

			numPoints = tcurve.numberOfPoints();
			IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
			
			for (unsigned long i = 0; i < numPoints; i++) {
				fGpiEmulator->IGPoint2DToPOINTL(tcurve.point(i).divW(), sp[i]);
			}
			
			unsigned long numKnots = tcurve.numberOfKnots() - 1;
			unsigned long start = 0;
			unsigned long end;
			while ((end = tcurve.nextDiscontinuity(start, IGCurve2D::kBreak)) < numKnots)
			{
				if (!PolyBezier(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
					throw IPmPolySplineException("PolySpline-Curve");
				start = end;
			}
			
		}
		else {	// is a polyline
			numPoints = tcurve.numberOfPoints();
			IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
			
			unsigned long start = 0;
			unsigned long end;
			
			for (unsigned long i = 0; i < numPoints; i++) {
				fGpiEmulator->IGPoint2DToPOINTL(tcurve.point(i).dropW(), sp[i]);
				if (tcurve.knot(i + 1) == tcurve.knot(i + 2)) {
					end = i + 1;
					if (!Polyline(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
						GrafDeviceException("Polyline");
					start = end;
				}
			}
		}
		}	
		catch (const IPmPolySplineException&) {
			GParametric minPara = tcurve.minParameter();
			GParametric maxPara = tcurve.maxParameter();
			IGRect2D bounds = worldBounds();
			bounds.inset(IGPoint2D(-1.,-1.));	// one pixel larger than it was just to be save...

			IGrafMatrix frameMatrix(*fFrameMatrix);
			frameMatrix.invert();
			ILinkedModelMatrixGrafState theState(&grafState, &frameMatrix);

			// step is two pixels
			GParametric step = tcurve.approximateParameterFromArcLength(2.0) - minPara;
			GParametric startU;
			bool onCurve = false;
			for (GParametric u = minPara; u <= maxPara; u += step) {
				bool inside = bounds.contains(tcurve.evaluate(u));
				if (onCurve != inside) {
					onCurve = inside;
					if (inside) {	// off to on...
						if (u == minPara)
							startU = minPara;
						else
							startU = u - step;	// start at the last u that is outside of the rect
					}
					else {	// on to off...
						IGCurve2D sectionCurve;
						tcurve.sectionOfCurve(startU, u, sectionCurve);
						renderCurve(sectionCurve, theState);
					}
				}
			}
		}		
	}

}

//================================================================
//	IGpiDevice::renderLoop
//
// Parameters:
//	geometry : the loop to be rendered
//	grafState: indicates the attributes/transformation to be applied
//
// Return value: NONE
//
// Purpose: renders the loop geometry to the display
//
//

inline static bool isSingularOrder2(const IGLoop2D& loop) {
	unsigned long numPoints = loop.numberOfPoints();
	for (unsigned long i = 0; i++ < numPoints; ) {
		if (loop.knot(i) == loop.knot(i + 1))
			return (i == numPoints);
	}
	return false;
}

void IGpiDevice::renderLoop(const IGLoop2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGLoop2D tloop(geometry);
	
	// Try to fix defect 3102
	// Avoid using Win32 functions when drawing Arc, Chord or Ellipse on Win95
	// platform since those calls result in crashes sometimes
	// Crash Example: Chord(hdc, -670, -3696, 3531, 505, 290, 168, 132, 55)
//	if ((fPlatform == kWin95 || fPlatform == kOS2) && conicInfo(tcurve))
	if (conicInfo(tloop))
#ifdef MAYBE_A_BUG
		tloop.fConicInfo = 0 /*NIL*/;
#else
		deleteConicInfo(tloop);
#endif // MAYBE_A_BUG
	
	if (!fFrameMatrix->isIdentity()) {
		tloop.transformBy(*fFrameMatrix);
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		if (conicInfo(tloop)) {
			if (!conicInfo(tloop)->transformBy(*fFrameMatrix)) {
				deleteConicInfo(tloop);
			}
		}
#endif // NO_CONIC_ACCELERATION_SUPPORT
	}
	
	SetWinWorldTransform(fViewMatrix);
	
		
	if (tloop.eOFill())
		SetPolyFillMode(fHDC, ALTERNATE);
	else
		SetPolyFillMode(fHDC, WINDING);

	unsigned long numPoints;
	fDrawOperation = grafState.attributeState()->drawingOperation();

	unsigned long order = tloop.order();

	// note: order 4 bezier loop cannot be done with only one pass
	// note: loop cannot be done with only one pass.
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState) && (
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		conicInfo(tloop) ||
#endif // NO_CONIC_ACCELERATION_SUPPORT
		(order == 2 && isSingularOrder2(tloop))))
	{
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);
	
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		if (conicInfo(tloop)) {	// if loop is supported by windows...
			ConicMaker(tloop).render(fHDC);
		}
		else
#endif // NO_CONIC_ACCELERATION_SUPPORT
		{ // order = 2, single polygon... no c0 discontinuities
			tloop.closeLoop();

			numPoints = tloop.numberOfPoints();
			IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
			for (unsigned long i = 0; i < numPoints; i++)
				fGpiEmulator->IGPoint2DToPOINTL(tloop.point(i).dropW(), sp[i]);

			if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
				GrafDeviceException("Polygon");
		}
	}
	else
	{	// fill & frame paint modes are different, fill first...
		tloop.closeLoop();

	try {
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState, true);
			SetFillPaintModes(grafState);
			
#ifndef NO_CONIC_ACCELERATION_SUPPORT
			if (conicInfo(tloop)) {
				IPenContext aPen(fHDC, GetStockObject(NULL_PEN));
				ConicMaker(tloop).render(fHDC);
			}
			else
#endif // NO_CONIC_ACCELERATION_SUPPORT
			{
				BeginPath(fHDC);

				if (order > 2) {	// make everything order == 4 beziers...
					if (order < 4)
						tloop.raiseOrder(4);
					else if (order > 4)
						tloop.approximateLowerOrder(4, 0.5);	// bezified first and then tessolated...
		
					if (!tloop.isBezier())
						tloop.refineToBeziers();
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
					for (unsigned long i = 0; i < numPoints; i++) {
						fGpiEmulator->IGPoint2DToPOINTL(tloop.point(i).divW(), sp[i]);
					}

					unsigned long numKnots = tloop.numberOfKnots() - 1;
					unsigned long start = 0;
					unsigned long end;
					while ((end = tloop.nextDiscontinuity(start, IGCurve2D::kBreak)) < numKnots)
					{
						if (!MoveToEx(fHDC, int(((POINTL*)sp)[start].x), int(((POINTL*)sp)[start].y), NULL))
							GrafDeviceException("MoveToEx");

						if (!PolyBezier(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
							throw IPmPolySplineException("PolySpline-Loop");

						if (!LineTo(fHDC, int(((POINTL*)sp)[end-1].x), int(((POINTL*)sp)[end-1].y)))
							GrafDeviceException("LineTo");

						CloseFigure(fHDC);
							
						start = end;
					} // while


				}
				else { // order = 2, polygon...
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
					
					unsigned long start = 0;
					unsigned long end;
			
					for (unsigned long i = 0; i < numPoints; i++) {
						fGpiEmulator->IGPoint2DToPOINTL(tloop.point(i).dropW(), sp[i]);
						if (tloop.knot(i + 1) == tloop.knot(i + 2)) {
							end = i + 1;
							if (!Polygon(fHDC, (POINT*)(((POINTL*)sp) + start), int(end - start)))
								GrafDeviceException("Polygon");
							start = end;
						}
					}

				}

				EndPath(fHDC);

				if (!FillPath(fHDC))
					GrafDeviceException("FillPath");

			} // end not arc

		} // end fill

		// frame...		
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState);
			SetFramePaintModes(grafState);
			
#ifndef NO_CONIC_ACCELERATION_SUPPORT
			if (conicInfo(tloop)) {
				IBrushContext aBrush(fHDC,GetStockObject(HOLLOW_BRUSH));
				ConicMaker(tloop).render(fHDC);
			}
			else
#endif // NO_CONIC_ACCELERATION_SUPPORT
			{
				BeginPath(fHDC);

				if (order > 2) {	// make everything order == 4 beziers...
					if (order < 4)
						tloop.raiseOrder(4);
					else if (order > 4)
						tloop.approximateLowerOrder(4, 0.5);	// bezified first and then tessolated...
		
					if (!tloop.isBezier())
						tloop.refineToBeziers();
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
					
					for (unsigned long i = 0; i < numPoints; i++) {
						fGpiEmulator->IGPoint2DToPOINTL(tloop.point(i).divW(), sp[i]);
					}
					
					unsigned long numKnots = tloop.numberOfKnots() - 1;
					unsigned long start = 0;
					unsigned long end;
					while ((end = tloop.nextDiscontinuity(start, IGCurve2D::kBreak)) < numKnots)
					{
						if (!MoveToEx(fHDC, int(((POINTL*)sp)[start].x), int(((POINTL*)sp)[start].y), NULL))
							GrafDeviceException("MoveToEx");
	
						if (!PolyBezier(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
							throw IPmPolySplineException("PolySpline-Loop");

						if (!LineTo(fHDC, int(((POINTL*)sp)[end-1].x), int(((POINTL*)sp)[end-1].y)))
							GrafDeviceException("LineTo");

						CloseFigure(fHDC);

						start = end;
					} // while
					
				}
				else { // order = 2, polygon...
					tloop.closeLoop();
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
					
					unsigned long start = 0;
					unsigned long end;
			
					for (unsigned long i = 0; i < numPoints; i++) {
						fGpiEmulator->IGPoint2DToPOINTL(tloop.point(i).dropW(), sp[i]);
						if (tloop.knot(i + 1) == tloop.knot(i + 2)) {
							end = i + 1;
							if (!Polygon(fHDC, (POINT*)(((POINTL*)sp) + start), int(end - start)))
								GrafDeviceException("Polygon");
							start = end;
						}
					}
					
				}

				EndPath(fHDC);

				if (!StrokePath(fHDC))
					GrafDeviceException("StrokePath");
			} // end of not arc
		} // end of frame
		}	
		catch (const IPmPolySplineException&) {
			IGRect2D theBox(worldBounds());
			theBox.inset(IGPoint2D(-1.,-1.));	// expand by one
			IGArea theShape(fModelMatrix->untransformBounds(theBox));
			theShape.intersect(IGArea(geometry));
			renderArea(theShape, grafState);
		}
	} // end of different frame & fill paints

}

//================================================================
//	renderImage
//	
//	image : the pixels.
//	source : a rectangle in pixel space.
//	grafState : has the model matrix and attributes.
//	
//	render the pixels contained in the source rectangle to
//	the device. Transform the position of the pixels using the model matrix.
//	Use the transfer mode and image filtering mode defined in the graf state.
//	
//	The grafState contains the matrix that maps the image pixel space
//	to World Space. This matrix is set as the GDI WorldTransform to we can use
//	BitBlt with no transformation. If the World Transform is not supported, then
//	we must extract the components of the transformation from grafState, and
//	use them to set the parameters of StretchBlt.
//

void IGpiDevice::renderImage(
	const IGImage& image,
	const IGRect2D& source,
	const IGrafState& grafState)
{

	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	int rasterOperation = imageTransferMode(grafState);
	
	// draw the image on the presentation space
	HBITMAP srcImage = image.handle();
	IGImageDeviceContext srcHDC(image);
	
	IGRect2D  rect(source);
	const IGrafMatrix* coordFlip = grafState.root()->coordinateFlip();
	const IGrafMatrix* worldXForm = fModelMatrix;
	IGrafMatrix fImageMatrixUsed;

	//
	// Prior to the fix of defect #22234, the images were being
	// drawn in the correct location, but its orientation was
	// upside down in cases, where the default native platform coordinate
	// system origin was different from the one the application was using.
	//
	// The fix was to keep the location the same, but to just flip
	// the image.
	//
	// Doing this is somewhat convoluted. The SetWinWorldTransform
	// affects the entire device context. And it is the last one
	// that is applied. Nothing can be done about it. So if we see the
	// image is being drawn upside down, then we need to flip the image
	// one more time so that when the device context flips it, it will
	// appear right side up.
	//
	// The srcX, srcY values should not change. They refer to the image
	// device context which is different from the grafPort device context.
	//
	
	IGPoint2D pts[4];
	pts[2] = rect.topLeft();	// src rect
	pts[3] = rect.bottomRight();

	if (coordFlip) {
		fImageMatrixUsed = *coordFlip;
		rect = fImageMatrixUsed.transformBounds(rect);
    // need some adjust for coordinate flipping since image is exclusive.
    rect.offset(IGPoint2D(0.0, 1.0));

		fImageMatrixUsed.invert();
		fImageMatrixUsed.concatWith(*fModelMatrix);
		worldXForm = &fImageMatrixUsed;
	}
	
	pts[0] = rect.topLeft();	// dst rect
	pts[1] = rect.bottomRight();

	SetWinWorldTransform(worldXForm);

	int dstX = GCoordinateToInt(pts[0].fX);
	int dstY = GCoordinateToInt(pts[0].fY);
	int dstW = GCoordinateToInt(pts[1].fX - pts[0].fX);
	int dstH = GCoordinateToInt(pts[1].fY - pts[0].fY);

	int srcX = GCoordinateToInt(pts[2].fX);
	int srcY = GCoordinateToInt(pts[2].fY);
	int srcW = GCoordinateToInt(pts[3].fX - pts[2].fX);
	int srcH = GCoordinateToInt(pts[3].fY - pts[2].fY);
	
	SetStretchBltMode(fHDC, imageSamplingControl(grafState));
	HBITMAP fOldBitmap = SelectObject(srcHDC, srcImage);
	
	if(!image.transparencyFlag())
	{
	    if (rasterOperation != ICLUIHalfToneMode)
	    {
	    	if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, rasterOperation))
     			GrafDeviceException("StretchBlt");
    	}
    	else
    	{   // If the raster-op is halftone, set the pattern to a checker-board ...
		setTextColour(IBaseColor::kWhite); // white foreground
		SetBkColor(fHDC, 0x00000000); // black background

		WORD checker[] = { 0x55,0xaa,0x55,0xaa,0x55,0xaa,0x55,0xaa };
		HBITMAP bitmap = fGpiEmulator->CreateBitmap(8, 8, 1, 1, (const void*)checker, fHDC);
		IBrushContext aBrush(fHDC, fGpiEmulator->CreatePatternBrush(bitmap, fHDC));
	    	if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, rasterOperation))
		    	GrafDeviceException("StretchBlt");		
        }
	} else {
		HBITMAP srcMatte = image.maskImageHandle();
		if (dstW == srcW &&
			dstH == srcH &&
			!MaskBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcMatte, 0, 0, MAKEROP4(0x00AA0029, SRCCOPY)))
		{
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
				GrafDeviceException("StretchBlt");
			
			SelectObject(srcHDC, srcMatte);
			
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCAND))
				GrafDeviceException("StretchBlt");
			
			SelectObject(srcHDC, srcImage);
			
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
				GrafDeviceException("StretchBlt");
		}
	}

	SelectObject(srcHDC, fOldBitmap);


}

//================================================================
//	IGpiDevice::renderMaskedImage
//

void IGpiDevice::renderMaskedImage(
	const IGImage& image,
	const IGImage& mask,
	const IGRect2D& source,
	const IGrafState& grafState)
{

	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	//int rasterOperation = imageTransferMode(grafState);
	
	// draw the image on the presentation space
	HBITMAP srcImage = image.handle();
	HBITMAP srcMatte = mask.handle();
	IGImageDeviceContext srcHDC(image);
	
	IGRect2D rect(source);			
	const IGrafMatrix* coordFlip = grafState.root()->coordinateFlip();
	const IGrafMatrix* worldXForm = fModelMatrix;
	IGrafMatrix fImageMatrixUsed;

	//
	// Prior to the fix of defect #22234, the images were being
	// drawn in the correct location, but its orientation was
	// upside down in cases, where the default native platform coordinate
	// system origin was different from the one the application was using.
	//
	// The fix was to keep the location the same, but to just flip
	// the image.
	//
	// Doing this is somewhat convoluted. The SetWinWorldTransform
	// affects the entire device context. And it is the last one
	// that is applied. Nothing can be done about it. So if we see the
	// image is being drawn upside down, then we need to flip the image
	// one more time so that when the device context flips it, it will
	// appear right side up.
	//
	// The srcX, srcY values should not change. They refer to the image
	// device context which is different from the grafPort device context.
	//
	
	IGPoint2D pts[4];
	pts[2] = rect.topLeft();	// src rect
	pts[3] = rect.bottomRight();

	if (coordFlip) {
		fImageMatrixUsed = *coordFlip;
		rect = fImageMatrixUsed.transformBounds(rect);

		fImageMatrixUsed.invert();
		fImageMatrixUsed.concatWith(*fModelMatrix);
		worldXForm = &fImageMatrixUsed;
	}
	
	pts[0] = rect.topLeft();	// dst rect
	pts[1] = rect.bottomRight();

	SetWinWorldTransform(worldXForm);
	SetStretchBltMode(fHDC, imageSamplingControl(grafState));
	
	int dstX = GCoordinateToInt(pts[0].fX);
	int dstY = GCoordinateToInt(pts[0].fY);
	int dstW = GCoordinateToInt(pts[1].fX - pts[0].fX);
	int dstH = GCoordinateToInt(pts[1].fY - pts[0].fY);

	int srcX = GCoordinateToInt(pts[2].fX);
	int srcY = GCoordinateToInt(pts[2].fY);
	int srcW = GCoordinateToInt(pts[3].fX - pts[2].fX);
	int srcH = GCoordinateToInt(pts[3].fY - pts[2].fY);
	
	HBITMAP fOldBitmap = SelectObject(srcHDC, srcImage);
	
	if (dstW == srcW &&
		dstH == srcH &&
		!MaskBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcMatte, 0, 0, MAKEROP4(0x00AA0029, SRCCOPY)))
	{
		if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
			GrafDeviceException("StretchBlt");
		
		SelectObject(srcHDC, srcMatte);
		
		if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCAND))
			GrafDeviceException("StretchBlt");
		
		SelectObject(srcHDC, srcImage);
		
		if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
			GrafDeviceException("StretchBlt");
	}

	SelectObject(srcHDC, fOldBitmap);

}


//================================================================
//	helper class for font and text support...
//----------------------------------------------------------------

IGdiFontContext::IGdiFontContext(
	IGpiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* modelMatrix,
	const IGrafMatrix* coordFlip,
	const IGrafState* state) :

	fFont(0 /*NIL*/),
	fDevice(theDevice),
	fTmCached(false),
	fAnchorTransform(0 /*NIL*/),

	fDeviceContext(theDevice->fHDC),
	fPointSize(theText.fPointSize),
	fEscapement(theText.fEscapement),
	fWeight(theText.fWeight),
	fItalic(theText.fItalic),
	fUnderline(theText.fUnderline),
	fStrikeOut(theText.fStrikeOut),
	fTypeFace(theText.fTypeFace),
	fMatrix(modelMatrix),
	fMatrixStamp(modelMatrix ? modelMatrix->timeStamp() : 0),
	fFlip(coordFlip),
	fFlipStamp(coordFlip ? coordFlip->timeStamp() : 0),
	fGrafState(state)
{
	if (!fFlip) {
		if (fMatrix)
			fMatrixUsed = *fMatrix;
		else
			fMatrixUsed.setToIdentity();
		fDevice->SetWinWorldTransform(fMatrix);
	}	
	else
	{
		// the anchor point should be transformed as flipped, the rest of the grometry is not...
		fAnchorTransform = fFlip;
		fMatrixUsed = *fFlip;
		fMatrixUsed.invert();
		fMatrixUsed.concatWith(*fMatrix);	// undo the coordinate flip from the model matrix...
		fDevice->SetWinWorldTransform(&fMatrixUsed);
	}
}

void IGdiFontContext::initialise()
{
	makeFont();
}

void IGdiFontContext::makeFont()
{
   IPresSpaceHandle   fHDC =  fDeviceContext;
   int fontHeight;
	
  if ( fDevice->baseRootDevice())
	 {  //GrafPort that uses point size as world coordinate
	  	// In this case, the device resolution matrix is a scaling matrix .
					
		  if( fBitmap )
				   fontHeight = fPointSize; //bitmap font is not affected by device resolution matrix
		  else{
			  if(fPointSize)
			      fontHeight = fPointSize;
			  else
				  fontHeight = 12;  //default value
		  }
	 }
		else{	

			// GrafPort that uses pixel size as world coordinate
	      if( fBitmap)
				    fontHeight = fPointSize; //bitmap font, passing pointsize
		    else {
			     long   VerRes;
			
			      HDC hdc = GpiQueryDevice(fHDC);
			      // effective verticaldevice resolution is pels per inch for the purpose  of selecting fong
			      if( ! DevQueryCaps(hdc,CAPS_VERTICAL_FONT_RES, 1L, &VerRes))	
						    GrafDeviceException("DevQueryCaps");
					
					 if(fPointSize == 0)
				          fPointSize = 12; //default value
								
				    fontHeight = MulDiv (fPointSize, VerRes, 72.0);
			   }
 	}
	 fFont =
		CreateFont(
			fontHeight,
			0,	// FontWidth,
			fEscapement,
			0,	// Orientation,
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
      fOutline ,
			fBitmap,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			fTypeFace);

	if (fFont)
		fFontOld = SelectObject(fDevice->fHDC, fFont);
}

IGdiFontContext::~IGdiFontContext()
{
   if (fFont)
      delete fFont;
}

void IGdiFontContext::renderGlyphs(const IGTextRun& textRun) const
{
	IGPoint2D anchorP = textRun.anchor();

	if (fAnchorTransform)	// transform the anchor if need to ... for rendering
		anchorP = fAnchorTransform->transformPoint(anchorP);
}

void IGdiFontContext::extentPoint(
	const IGTextRun& textRun,
	SIZE& theSize) const
{
}

IGPoint2D IGdiFontContext::calculateBoundsAndAdvancement(
	const IGTextRun& textRun,
	IGRect2D& bounds)
{
	IGPoint2D leftTop, rightBottom;
	IGPoint2D anchor = textRun.anchor();

	if (fAnchorTransform)	// transform the anchor if need to ... for rendering
		anchor = fAnchorTransform->transformPoint(anchor);

	SIZE theSize;
	extentPoint(textRun, theSize);

	if (!fTmCached) {
		if (!GetTextMetrics(fDevice->fHDC, &fTm))
			GrafDeviceException("GetTextMetrics");
		fTmCached = true;
	}

	// It's better not to do this, specially at small pointsizes...	
	//	fBBoxSize.cx -= fTm.tmOverhang;

	IGPoint2D advance = leftTop = IGPoint2D::origin();
	switch (textRun.attachment())
	{
	default:
	case IGTextRun::kLeftToRight:
		advance.fX += theSize.cx;
		leftTop.fY -= fTm.tmAscent;
		break;
	case IGTextRun::kRightToLeft:
		advance.fX -= theSize.cx;
		leftTop.fX -= theSize.cx;
		leftTop.fY -= fTm.tmAscent;
		break;
	case IGTextRun::kTopToBottom:
		advance.fY += theSize.cy;
		leftTop.fX -= theSize.cx >> 1;
		break;
	case IGTextRun::kBottomToTop:
		advance.fY -= theSize.cy;
		leftTop.fX -= theSize.cx >> 1;
		leftTop.fY -= theSize.cy;
		break;
	}

#ifdef GRAPH2D_DEBUG
// printf("_ (%.2f, %.2f), extent:%d, %d (%.2f, %.2f)\n", anchor.fX, anchor.fY, theSize.cx, theSize.cy, leftTop.fX, leftTop.fY);	
#endif
	leftTop += anchor;
	rightBottom = leftTop + IGPoint2D(theSize.cx, theSize.cy);
	anchor += advance;

	if (!fAnchorTransform)	// transform the anchor if need to ... for rendering
	{
		bounds = IGRect2D(leftTop, rightBottom);
		return anchor;
	}
	else
	{
		bounds = fAnchorTransform->untransformBounds(IGRect2D(leftTop, rightBottom));
		return fAnchorTransform->untransformPoint(anchor);
	}
}	

inline bool IGdiFontContext::cacheHit(
	IGpiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* model,
	const IGrafMatrix* flip) const
{
   	return (
		fDeviceContext == theDevice->fHDC &&
		fMatrix == model &&
		fMatrixStamp == (model ? model->timeStamp() : 0) &&
		fFlip == flip &&
		fFlipStamp == (flip ? flip->timeStamp() : 0) &&
		fPointSize == theText.fPointSize &&
		fEscapement == theText.fEscapement &&
		fWeight == theText.fWeight &&
		fItalic == theText.fItalic &&
		fUnderline == theText.fUnderline &&
		fStrikeOut == theText.fStrikeOut &&
		fTypeFace == theText.fTypeFace  &&
    fOutline == theText.fOutline &&
		fBitmap == theText.fBitmap);
}

void IGpiDevice::setTextColour(const IBaseColor& target) {
	if (fTextColourCached) {
		if (fTextColour == target) {
			return;
		}
	}
	else
		fTextColourCached = true;

	fTextColour = target;
	if (SetTextColor(fHDC, fGpiEmulator->convertToWRGB(fTextColour)) == CLR_INVALID)
		GrafDeviceException("SetTextColor");
}

//----------------------------------------------------------------

IGdiNonUnicodeFontContext::IGdiNonUnicodeFontContext(
		IGpiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) :
		IGdiFontContext(theDevice, theText, model, flip, state)
{
	fOutline = theText.fOutline;
	fBitmap = theText.fBitmap;
}		

void IGdiNonUnicodeFontContext::renderGlyphs(const IGTextRun& textRun) const
{
	IGPoint2D anchorP = textRun.anchor();

	if (fAnchorTransform)	// transform the anchor if need to ... for rendering
		anchorP = fAnchorTransform->transformPoint(anchorP);

	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(false);

	// using ExTextOut is faster than TextOut according to MSDN bulletin...
	if (!ExtTextOut(
		fDevice->fHDC,
		IGpiDevice::GCoordinateToInt(anchorP.fX),
		IGpiDevice::GCoordinateToInt(anchorP.fY),
		0,			// wOptions
		0 /*NIL*/,	// lpRect
		textRun.fSubText,	//this transcodes to host character set
		textRun.fSubText.length(),
		0 /*NIL*/))

		GrafDeviceException("ExtTextOut");
}

void IGdiNonUnicodeFontContext::extentPoint(
	const IGTextRun& run,
	SIZE& theSize) const
{
	if (!run.fSubTextCached)
		((IGTextRun*)&run)->cacheRenderSubtext(false);

	if (!GetTextExtentPoint32(fDevice->fHDC, run.fSubText, run.fSubText.length(), &theSize))
		GrafDeviceException("GetTextExtentPoint32");
}


// create different types of font context according to the types of transformation matrices.
IGdiFontContext& IGpiDevice::makeFontContext(
	const IGTextRun& theText,
	const IGrafMatrix* modelMatrix,
	const IGrafState* state,
	bool unicodeSupported)
{
	if (!theText.fStylesCached)
		((IGTextRun*)&theText)->cacheRenderStyles();

	const IGrafMatrix* coordFlip = state->root()->coordinateFlip();

	if (fFontCache) {
		if (fFontCache->cacheHit(this, theText, modelMatrix, coordFlip)) {
			SetWinWorldTransform(&fFontCache->fMatrixUsed);
			if (fFontCache->fGrafState != state) // this may be a problem, we may need to do a contents comparison
				fFontCache->fGrafState = state;
			return *fFontCache;
		}
		delete fFontCache;
	}
	
	if (unicodeSupported)
		fFontCache = new IGdiFontContext(this, theText, modelMatrix, coordFlip, state);
//	else if (fPlatform != IGpiDevice::kWin95)
	else if (!(IPlatform::isWin9x()))
		fFontCache = new IGdiNonUnicodeFontContext(this, theText, modelMatrix, coordFlip, state);
	fFontCache->initialise();
	return *fFontCache;
}

//================================================================
//	IGpiDevice::renderTextRun
//

IGPoint2D IGpiDevice::renderTextRun(
	const IGTextRun& textRun,
	const IGrafState& grafState)
{
	const IText* text = textRun.text();
	if (!text)
		return textRun.anchor();

	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)		// before transformation
		return textRun.anchor();
	
	// get all attributes for creating a font from the style set
	IGdiFontContext& fontContext =
		makeFontContext(
			textRun,
			fModelMatrix,
			&grafState,
//			fPlatform == IGpiDevice::kWinNT);
         IPlatform::isWinNT());

	SetFillPaintModes(grafState);

	const IBaseColor* theColor = textRun.fTextColor;
	if (!theColor)
		theColor = grafState.attributeState()->fillPaint()->color();

	setTextColour(*theColor);

	fontContext.renderGlyphs(textRun);

	IGPoint2D nextLocation;
	if (textRun.hadMetrics(fHDC, fontContext.fMatrixStamp, fontContext.fFlipStamp)) {
		nextLocation = textRun.nextAnchor();
	}
	else {
		IGRect2D bounds;
		nextLocation = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		((IGTextRun*)&textRun)->updateMetrics(
			bounds,
			nextLocation,
			fHDC,
			fontContext.fMatrixStamp,
			fontContext.fFlipStamp);
	}

#ifdef DEBUG_BBOX
IGRect2D bounds = textRun.bounds();
IGPoint2D anc = textRun.anchor();
printf(" anchor(%.1f, %.1f), bounds(%.1f, %.1f)(%.1f, %.1f), next(%.1f, %.1f)\n",
 anc.fX, anc.fY, bounds.fLeft, bounds.fTop, bounds.width(), bounds.height(),
 nextLocation.fX, nextLocation.fY);
IFrameBundle redFrame(IBaseColor::kRed);
ILinkedAttributeGrafState tempState(&grafState, &redFrame);
renderRect(bounds, tempState);
IFrameBundle bluFrame(IBaseColor::kBlue);
ILinkedAttributeGrafState temp1State(&grafState, &bluFrame);
renderLine(IGLine2D(anc, nextLocation), temp1State);
IGPoint2D se(1.0, 1.0);
renderRect(IGRect2D(anc - se, anc + se), temp1State);
IFrameBundle grnFrame(IBaseColor::kGreen);
ILinkedAttributeGrafState temp2State(&grafState, &grnFrame);
renderRect(IGRect2D(nextLocation - se, nextLocation + se), temp2State);
#endif

	return nextLocation;
}

// static function to persuit rendering text metrics...
// but it does not work for righthand, caller has to manage this

void IGpiDevice::determineTextMetrics(
	IGTextRun& textRun,
	IGpiDevice* dev,
	const IGrafState* grafState)
{
  IPresSpaceHandle deviceContext(0);
	bool baseRootFlag = true;
	
	if(dev){
	 deviceContext = dev->deviceContext();
	 baseRootFlag = dev->baseRootDevice();
	}
	
	const IGrafMatrix* xform = 0 /*NIL*/;
	const IGrafMatrix* flip = 0 /*NIL*/;
	unsigned long matStamp = 0;
	unsigned long flipStamp = 0;

	if (grafState) {
		xform = grafState->modelMatrix();
		flip = grafState->root()->coordinateFlip();
		if (xform)
			matStamp = xform->timeStamp();
		if (flip)
			flipStamp = flip->timeStamp();

		// we should eventually take into account of the bundle's time stamp...
	}

	if (!textRun.hadMetrics(deviceContext, matStamp, flipStamp))
	{
		IGpiDevice device(deviceContext, baseRootFlag);

		if (!textRun.fStylesCached)
			textRun.cacheRenderStyles();

		IGRect2D bounds;
		IGPoint2D next;
		
//		if (device.fPlatform == IGpiDevice::kWinNT) {
		if (IPlatform::isWinNT()) {
			IGdiFontContext fontContext(&device, textRun, xform, flip, grafState);
			fontContext.initialise();
			next = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		}
//		else if (fPlatform != IGpiDevice::kWin95){
      else if (!(IPlatform::isWin9x())){
         IGdiNonUnicodeFontContext fontContext(&device, textRun, xform, flip, grafState);
			fontContext.initialise();
			next = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		}
		textRun.updateMetrics(bounds, next, deviceContext, matStamp, flipStamp);
	}
	// else already determined...
}

