// Revision: 17 1.100.2.1 source/albert/graph2d/gdidev.cpp, 2d, ioc.v400, 001006 
/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
/*================
||
||	File:	GdiDevices.C
||
||	What:	Routines which support the graphics device on Windows GDI platform
||
||	Change History:
||		?           Isai Scheinberg -   Original
||		20 May 96   tt              -   First rewrite
||		16 Jan 97   xl              -	Added Win95 Transform Text Support
||
*/

#include <gdidev.hpp>
#include <igimage.hpp>
#include <imagedc.hpp>
#include <grstate.hpp>
#include <regnextr.hpp>
#include <inumeric.hpp>
#include <igloop2d.hpp>
#include <igtxtrun.hpp>
#include <grassert.hpp>
#include <igbidi.hpp>
#include <iplatfrm.hpp>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <ibase.hpp>
#include <istring.hpp>
#include <igcurv2d.hpp>
#include <iknots.hpp>
#include <prspline.hpp>
#include <deleter.hpp>
#include <assert.h>
#include <itrace.hpp>

#ifdef GRAPH2D_DEBUG
#include <stream.h>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
#define _GDIDEV_CPP_
#include <ipagetun.h>
#endif

//==============================================================
//	The device maker implementation -
//	this should probably in a saperate file itself when we have more than one device
//

IGrafDevice* IGrafDeviceMaker::makeDevice(
	const IPresSpaceHandle& deviceContext,
	const IGrafPort::EGrafPortType  grafPortType,
	EDeviceMappingMode mappingMode)
{
  if (grafPortType == IGrafPort::kIExtendedRootGrafPort)
	   return new IGdiDevice(deviceContext,mappingMode, false);
	else
	   return new IGdiDevice(deviceContext,mappingMode, true);
}
//===================================================================
// A class that can reselect and delete a PEN object automatically

#pragma pack(push,4)
class IPenContext {
public:
  // NOTE: IPenContext is responsible to delete the HPEN handle. Don't call
	// windows function DeleteObject() to delete this handle.
	IPenContext(HDC dc, HPEN handle);
	~IPenContext();
	void setPen(HPEN handle);

private:
	HPEN fPenOld;
	HDC fHDC;
	// always save the HPEN that is currently selected to the fHDC.
  // This is  to guarantee that the destructor of IPenContext will
	// destroy the exact HPEN when it is not selected any more.
	HPEN fSavePen;
	
	// The following functions are declared private to prevent multiple copies
	// of a HPEN from being created
	IPenContext(const IPenContext&);
	IPenContext& operator=(const IPenContext&);
};
#pragma pack(pop)

inline IPenContext::IPenContext(HDC dc, HPEN handle) :
	fHDC(dc)
{
  	fPenOld = (HPEN) SelectObject( fHDC, handle );
	fSavePen = handle ;
}

inline IPenContext::~IPenContext()
{
	//DeleteObject(SelectObject(fHDC, fPenOld));
	SelectObject(fHDC, fPenOld);
	DeleteObject(fSavePen);
	
}

inline void IPenContext::setPen(HPEN handle)
{
	//DeleteObject(SelectObject(fHDC, handle));
	SelectObject(fHDC, handle);
	DeleteObject(fSavePen); //first deleted the HPEN that has just been deselected.
	fSavePen = handle ;   // then save the HPEN that is currently selected.
}


//===================================================================
// A class that can reselect and delete a HBRUSH object automatically
#pragma pack(push,4)
class IBrushContext{
public:
	// NOTE: IPenContext is responsible to delete the HBrush handle. Don't call
	// windows function DeleteObject() to delete this handle.
	IBrushContext(HDC dc, HBRUSH handle, bool select = true);
	~IBrushContext();
	void setBrush(HBRUSH handle);

	HBRUSH fBrush;	// used for os2

private:
	bool fSelected;
	HBRUSH fBrushOld;
	HDC fHDC;
	
  // always save the HBRUSH that is currently selected to the fHDC.
  // This is  to guarantee that the destructor of IBrushContext will
	// destroy the exact HBRUSH when it is not selected any more.
	HBRUSH fSaveBrush;
	
	
	// The following functions are declared private to prevent multiple copies
	// of a HBRUSH from being created
	IBrushContext(const IBrushContext&);
	IBrushContext& operator=(const IBrushContext&);
	void deleteBrush(HBRUSH) ;
};
#pragma pack(pop)

void IBrushContext::deleteBrush(HBRUSH brush)
{
	LOGBRUSH lb;
	if(!GetObject(brush, sizeof(lb), (LPSTR)&lb))
		GrafDeviceException("GetObject");
	if(lb.lbStyle == BS_PATTERN){
		DeleteObject((HBITMAP)lb.lbHatch);
	}
	DeleteObject(brush);
}

inline IBrushContext::IBrushContext(HDC dc, HBRUSH handle, bool select) :
	fHDC(dc),
	fBrush(handle),
	fSelected(select)
{
	if (fSelected)
      fBrushOld = (HBRUSH) SelectObject( fHDC, fBrush );
		fSaveBrush = fBrush;
}

inline IBrushContext::~IBrushContext()
{
	if (fSelected){
		//deleteBrush(SelectObject(fHDC, fBrushOld));
		SelectObject(fHDC, fBrushOld);
		deleteBrush(fSaveBrush);
  }		
	else
		deleteBrush(fBrush);
}

inline void IBrushContext::setBrush(HBRUSH handle)
{
	fBrush = handle;
	if (fSelected){
		//deleteBrush(SelectObject(fHDC, fBrush));
		SelectObject(fHDC, fBrush);
		deleteBrush(fSaveBrush);  //first deleted the HBRUSH that has just been deselected.
		fSaveBrush = fBrush;   // then save the HBRUSH that is currently selected.   }		
	}
	else {
  	   fBrushOld = (HBRUSH) SelectObject( fHDC, fBrush );
		fSaveBrush = fBrush;
		fSelected = true;
	}
}


//================================================================
//	helper class for font and text support...
//----------------------------------------------------------------
#pragma pack(push,4)
class IGdiFontContext {	// implements NT specific behaviou...
public:
	IGdiFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip,
		const IGrafState* state);
	virtual ~IGdiFontContext();

	void initialise();		// called by the IGdiDevice::makeFontContext...
	virtual void renderGlyphs(const IGTextRun&) const;
	virtual IGPoint2D calculateBoundsAndAdvancement(const IGTextRun&, IGRect2D& bounds);

	bool cacheHit(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip) const;

	unsigned long fMatrixStamp;
	unsigned long fFlipStamp;
	IGrafMatrix fMatrixUsed;				// used for NT transform, and normalised model matrix for w95
	const IGrafState* fGrafState;

protected:
	virtual void makeFont();	// called by initialise...
	virtual void extentPoint(const IGTextRun&, SIZE& extent) const;		// called by calculateBoundsAndAdvancement...

	IGdiDevice* fDevice;

	const IGrafMatrix* fAnchorTransform;	// for coordinate system flip prupose only...
	TEXTMETRIC fTm;
	bool fTmCached;
	HFONT fFont;
	HFONT fFontOld;
	
	HDC fDeviceContext;
	int fEscapement;
	IString fTypeFace;
	GCoordinate fPointSize;
	int fWeight;
	BYTE fItalic;
	BYTE fUnderline;
	BYTE fStrikeOut;

	// for caching GdiFontContext:
	const IGrafMatrix* fMatrix;
	const IGrafMatrix* fFlip;
};
#pragma pack(pop)

// implements OS/2 and maybe Win32s behaviou...
#pragma pack(push,4)
class IGdiNonUnicodeFontContext : public IGdiFontContext {
public:
IGdiNonUnicodeFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) ;

	virtual ~IGdiNonUnicodeFontContext() ;

	virtual void renderGlyphs(const IGTextRun&) const;

protected:
	virtual void extentPoint(const IGTextRun&, SIZE& extent) const;
};
#pragma pack(pop)

//================================================================
#if !IC_RTTI_ENABLED
GrafDeviceTypeDefinitionsMacro(IGdiDevice);
#endif // IC_RTTI_ENABLED

//==============================================================
//	Statics for IGdiDevice(info shared by all instances)
//

// min/max values in device coordinates ... initialised to LONG_MIN/LONG_MAX, only Win95 had SHORT...
GCoordinate IGdiDevice::gMaxCoord = (GCoordinate)LONG_MAX - 1;
GCoordinate IGdiDevice::gMinCoord = (GCoordinate)LONG_MIN + 1;

bool IGdiDevice::fInitialized = false;

// IGdiDevice::EPlatform IGdiDevice::fPlatform = IGdiDevice::kUndefined;

#pragma enum(4)
enum EClipState {
	kClipAreaEmpty = 0,
	kClipAreaInfinite,
	kClipAreaRegular
};
#pragma enum(pop)

//==============================================================
//	Utility Routine for Translating from GCoordinate to integer
//

int IGdiDevice::GCoordinateToInt(GCoordinate value)
{
	if (value > 0)
	{
		value += 0.5;		// round up
		if (value > gMaxCoord)
			value = gMaxCoord;
	}
	else if (value < 0)
	{
		value -= 0.5;		// round down
		if (value < gMinCoord)
			value = gMinCoord;
	}

	return (int)floor(value);
}

void IGdiDevice::IGPoint2DToPOINTL(const IGPoint2D &p, POINTL &ptl)
{
	ptl.x = GCoordinateToInt(p.fX);
	ptl.y = GCoordinateToInt(p.fY);
}


//==============================================================
//	Constructors
//

IGdiDevice::IGdiDevice(
    const IPresSpaceHandle& deviceHandle,
		bool baseRootFlag) :
	fHDC(deviceHandle),
	fHDCOwned(0 /*NIL*/),
	fDrawOperation(IAttributeState::kUnset),
	fMatrix(),
	fResolution(0 /*NIL*/),
	fFontCache(0 /*NIL*/),
	fMatrixCacheUsed(false),
	fCachePaintModeStateChange(0),
	fCacheTransferMode(-1),
	fBackModeSet(false),
	fCachePenBrushStateChange(0),
	fNullPenWhenFill(false),
	fHollowBrushWhenFrame(false),
	fPenCache(0 /*NIL*/),
	fBrushCache(0 /*NIL*/),
	fClipAreaSet(false),
	fTextColourCached(false),
	fBaseRootDevice(baseRootFlag) //default
{
  if (fBaseRootDevice)
	    fResolution = resolutionMatrix() ;
			
	setupDevice(deviceHandle);
	fPsCache = fHDC;

	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	// unit = pels, origin = top-left
	initialise(MM_TEXT);
}

// I just hope mappingMode will die out... so I can remove this ctr
IGdiDevice::IGdiDevice(
	const IPresSpaceHandle& deviceHandle,
	EDeviceMappingMode mappingMode,
	bool baseRootFlag) :
	
	fHDC(deviceHandle),
	fHDCOwned(0 /*NIL*/),
	fDrawOperation(IAttributeState::kUnset),
	fMatrix(),
	fResolution(0),
	fFontCache(0 /*NIL*/),
	fMatrixCacheUsed(false),
	fCachePaintModeStateChange(0),
	fCacheTransferMode(-1),
	fBackModeSet(false),
	fCachePenBrushStateChange(0),
	fNullPenWhenFill(false),
	fHollowBrushWhenFrame(false),
	fPenCache(0 /*NIL*/),
	fBrushCache(0 /*NIL*/),
	fClipAreaSet(false),
	fTextColourCached(false),
	fBaseRootDevice(baseRootFlag)
{
   if (fBaseRootDevice)
	    fResolution = resolutionMatrix() ;

	setupDevice(deviceHandle);

	fPsCache = fHDC;

	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	int mmode;
	switch (mappingMode)
	{
	case kPixel:
	default:
		mmode = MM_TEXT;
		break;
	case kLowMetric:
		mmode = MM_LOMETRIC;
		break;
	case kHighMetric:
		mmode = MM_HIMETRIC;
		break;
	case kLowEnglish:
		mmode = MM_LOENGLISH;
		break;
	case kHighEnglish:
		mmode = MM_HIENGLISH;
		break;
	case kTwips:
		mmode = MM_TWIPS;
		break;
	}
	
	initialise(mmode);
}

IGdiDevice::IGdiDevice(
	const IPresSpaceHandle& deviceHandle,
	const IGRect2D& imagingRect,
	const IGrafMatrix& resolution) :

	fHDC(deviceHandle),
	fHDCOwned(0 /*NIL*/),
	fDrawOperation(IAttributeState::kUnset),
	fMatrix(),
	fResolution(new IGrafMatrix(resolution)),
	fFontCache(0 /*NIL*/),
	fMatrixCacheUsed(false),
	fCachePaintModeStateChange(0),
	fCacheTransferMode(-1),
	fBackModeSet(false),
	fCachePenBrushStateChange(0),
	fNullPenWhenFill(false),
	fHollowBrushWhenFrame(false),
	fPenCache(0 /*NIL*/),
	fBrushCache(0 /*NIL*/),
	fClipAreaSet(false),
	fTextColourCached(false)
{
	// don't call setupDevice here because it is supposely owned by the caller...

	if (fResolution->isIdentity())
		fBaseRootDevice = false ;
	else
		fBaseRootDevice = true ;
		
	fPsCache = fHDC;


	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	initialise(MM_TEXT, &imagingRect);
}

void IGdiDevice::setupDevice(
	const IPresSpaceHandle& deviceHandle)
{
	if (deviceHandle == 0 /*NIL*/) {
		// this way anyone can construct a desktop screen device...
		fHDC = GetWindowDC(HWND_DESKTOP);
		if (fHDC == 0 /*NIL*/)
			GrafDeviceException("GetWindowDC");
		fHDCOwned = fHDC;
	}
}

void IGdiDevice::initialise(int mmode, const IGRect2D* clipRect)
{
	if (!SetMapMode(fHDC, mmode))
		GrafDeviceException("SetMapMode");
	
	if (!fInitialized)
	{
		fInitialized = true;
		
//		OSVERSIONINFO osversion;
//		osversion.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
		
//		if (!GetVersionEx(&osversion))
//		GrafDeviceException("GetVersionEx");
		
//		if (osversion.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS) {
      if (IPlatform::isWin9x()) {
//			fPlatform = IGdiDevice::kWin95;
			gMaxCoord =(GCoordinate)SHRT_MAX - 1;	// Win 95 coordinates is short ... *!&@^%#$ Microsoft!
			gMinCoord =(GCoordinate)SHRT_MIN + 1;
		}
//		}
//		else if (osversion.dwPlatformId == VER_PLATFORM_WIN32_NT) {
//			fPlatform = IGdiDevice::kWinNT;
			// Win NT coordinates is long
//		}
	}
	
	if (!clipRect)
	{
		int width;
		int height;
		INativeWindowHandle hw = WindowFromDC(fHDC);
		if (hw)
		{
			// find the height for current client window
			RECT wrect;
			GetWindowRect(hw, &wrect);

			width = wrect.right - wrect.left;
			height = wrect.bottom - wrect.top;
		}
		else	// this may not be a device that has a Window (not on screen...)
		{
			width = GetDeviceCaps(fHDC, HORZRES);	// number of pixels per raster line...
			height = GetDeviceCaps(fHDC, VERTRES);	// number of raster lines per device...
		}
		
	  IGRect2D boundInPixel= IGRect2D(0, 0, width, height);
#if 0	
	  if ( fBaseRootDevice){
		  IGrafMatrix matrix(*fResolution);
			matrix.invert();    // get the matrix that will convert pixels to points
			setWorldBounds(matrix.transformBounds(boundInPixel));  // worldBound should be in points
		}	
	  else
#else		
	    setWorldBounds(boundInPixel);		
#endif			
	}
	else	// custom clipping ...
	{
		setWorldBounds(*clipRect);
	}
	
	fFrameMatrix = fModelMatrix = fViewMatrix = fPenMatrix = &fMatrix;
}


// these three functions are disabled...
IGdiDevice::IGdiDevice()
{}

IGdiDevice::IGdiDevice(const IGdiDevice&)
{}

IGdiDevice& IGdiDevice::operator=(const IGdiDevice& Src)
{
	return *this;
}

//==============================================================
//	Destructor
//

IGdiDevice::~IGdiDevice()
{
	if (isValid()) {
		RestoreDC(fHDC, fSavedDC);
		// this fails on a printer HDC, if print job has been dispatched. Don't throw exception
	}

	delete fPenCache;
	delete fBrushCache;
	delete fFontCache;
	delete fResolution;

	if (fHDCOwned && fHDCOwned == fHDC) {
		ReleaseDC(WindowFromDC(fHDC), fHDC);
		// don't throw an exception here because we tried our best,
		// but we don't want disturb anyone, don't we?
	}
}

//==============================================================
//	device inquitries
//

const IGrafMatrix& IGdiDevice::deviceResolution(const IGRect2D&) const
{
	if (!fResolution)
		return IGrafMatrix::identity();
	else
		return *fResolution;
}


IGrafMatrix* IGdiDevice::resolutionMatrix()
{
	 GCoordinate pixelsPerInchX;
	 GCoordinate pixelsPerInchY ;
	
	//Number of pixels per logical inch along the screen width, height
	 if(!(pixelsPerInchX = GetDeviceCaps(fHDC, LOGPIXELSX)))
		GrafDeviceException("GetDeviceCaps");
	 if(!(pixelsPerInchY = GetDeviceCaps(fHDC , LOGPIXELSY)))
		GrafDeviceException("GetDeviceCaps");
	
	 // resolution is a scale matrix
	IGrafMatrix*   resolution = new IGrafMatrix(IGPoint2D(pixelsPerInchX/72, pixelsPerInchY/72),IGPoint2D::origin());
    return resolution ;
}		


bool IGdiDevice::baseRootDevice() const
{
  return fBaseRootDevice ;
}

// the coordinate system is inherited from the base class - lefthand, righthand on o/s2

const IPresSpaceHandle& IGdiDevice::deviceContext() const
{
    if (fPsCache != fHDC)
	    ((IGdiDevice *)this)->fPsCache =  fHDC;
	return fPsCache;
 }

IPresSpaceHandle IGdiDevice::orphanDeviceContext()
{
	if (isValid())
	{
		setValidity(false);
		// ??? save our graf-state of the device...

		RestoreDC(fHDC, fSavedDC);
		// this fails on a printer HDC, if print job has been dispatched. Don't throw exception
		IPresSpaceHandle temp = fHDC;	// save for return...

		// reset the caches... font cache knows how to deal with a new HDC
		fHDC = 0 /*NIL*/;
		fCachePaintModeStateChange = 0;
		fCacheTransferMode = -1;
		fBackModeSet = false;
		fCachePenBrushStateChange = 0;
		fNullPenWhenFill = false;
		fHollowBrushWhenFrame = false;
		delete fPenCache;
		fPenCache = 0 /*NIL*/;
		delete fBrushCache;
		fBrushCache = 0 /*NIL*/;
		delete fFontCache;
		fFontCache = 0 /*NIL*/;
		fMatrixCacheUsed = false;
		fClipAreaSet = false;
		fTextColourCached = false;
		return temp;
	}
	return fHDC;
}

void IGdiDevice::adoptDeviceContext(const IPresSpaceHandle& dc)
{
	fHDC = dc;

	if (!(fSavedDC = SaveDC(fHDC)))
		GrafDeviceException("SaveDC");
	
	setValidity(true);
}


static const GCoordinate kRoot2 =  1.414213562373;

//==============================================================
//	IGdiDevice::CreatePen
//

HPEN IGdiDevice::CreatePen(const IGrafState& state)
{
	DWORD dwPenType;
	DWORD dwPenWidth;
	DWORD dwPenStyle = PS_SOLID;
	DWORD dwJoinStyle = 0;
	DWORD dwEndCap = 0;
	
	// set pen attributes from Frame Paint(color, pattern)
	const IPaint* paint = state.attributeState()->framePaint();
	LOGBRUSH lb = CreateBrushStructure(*paint);
	
	// set pen type: cosmetic or geometric(wide) and pen width
	const IPen* pen = state.attributeState()->framePen();
	
	IPen::EPenType penType = pen->penType();
	
	if (penType == IPen::kHairline)
	{
		dwPenType = PS_COSMETIC;
		dwPenWidth = 1;
		lb.lbStyle = PS_SOLID;
	}
	else
	{
		dwPenType = PS_GEOMETRIC;

		if (fPenMatrix && !fPenMatrix->isIdentity())
		{
			// pen width calculation ... only done on Win95
			IGPoint2D pts[2];	// unit vector
			pts[0] = IGPoint2D::origin();
			pts[1] = IGPoint2D(1, 1);
			fPenMatrix->transformPoints(pts, pts, 2);
			dwPenWidth = GCoordinateToInt((pts[1] - pts[0]).vectorLength() * pen->penWidth() / kRoot2);
		}
		else
			dwPenWidth = GCoordinateToInt(pen->penWidth());
		
		switch (penType) {
		default:
		case IPen::kSolid :
			// dwPenStyle = PS_SOLID;
			break;
		case IPen::kDot:
			dwPenStyle = PS_DOT;
			break;
		case IPen::kDashed:
			dwPenStyle = PS_DASH;
			break;
		case IPen::kDashDot:
			dwPenStyle = PS_DASHDOT;
			break;
		case IPen::kDashDotDot:
			dwPenStyle = PS_DASHDOTDOT;
			break;
		case IPen::kInvisible:
			dwPenStyle = PS_NULL;
			break;
		}
		
		const ICap* cap = state.attributeState()->frameEndCap();
		ICap::ECapType endCapType = cap->capType();
		switch (endCapType)
		{
		case ICap::kFlush:
		default:
			dwEndCap = PS_ENDCAP_FLAT;
			break;
		case ICap::kSquare:
			dwEndCap = PS_ENDCAP_SQUARE;
			break;
		case ICap::kRound:
			dwEndCap = PS_ENDCAP_ROUND;
			break;
		}

		const IJoint* joint = state.attributeState()->frameJoint();
		IJoint::EJointType jointType = joint->jointType();
		switch (jointType)
		{
		case IJoint::kMiterLimit:
		default:
			dwJoinStyle = PS_JOIN_MITER;
			if (!SetMiterLimit(fHDC, FLOAT(joint->miterLimit()), NULL))
				GrafDeviceException("SetMiterLimit");
			break;
		case IJoint::kBevel:
			dwJoinStyle = PS_JOIN_BEVEL;
			break;
		case IJoint::kRound:
			dwJoinStyle = PS_JOIN_ROUND;
			break;
		}
	}
	
	// create pen
	HPEN hPen = ExtCreatePen(
		dwPenType   |        // Cosmetic or Geometric
		dwPenStyle  |        // dash/dot style
		dwJoinStyle |        // miter/bevel/round
		dwEndCap,            // flat/square/round
		dwPenWidth,          // logical width for Geometric or 1 for cosmetic
		&lb,                 // structure for brush attributes
		0, NULL);            // no custom style
	
	if (!hPen)
		GrafDeviceException("ExtCreatePen");
	
	return(hPen);
}


//==============================================================
//	IGdiDevice::CreateFillBrush
//

HBRUSH IGdiDevice::CreateFillBrush(const IGrafState& state)
{
	const IPaint* paint = state.attributeState()->fillPaint();
	LOGBRUSH lb = CreateBrushStructure(*paint);
	HBRUSH hBrush = CreateBrush(lb);
	
	return(hBrush);
	
}

//==============================================================
//	IGdiDevice::CreateFrameBrush
//

HBRUSH IGdiDevice::CreateFrameBrush(const IGrafState& state)
{
	const IPaint* paint = state.attributeState()->framePaint();
	LOGBRUSH lb = CreateBrushStructure(*paint);
	HBRUSH hBrush = CreateBrush(lb);
	
	return(hBrush);
}

//==============================================================
//	IGdiDevice::CreateBrush
//

HBRUSH IGdiDevice::CreateBrush(const LOGBRUSH& lb)
{
	HBRUSH hBrush;
	
	if (lb.lbStyle == BS_SOLID) {
		hBrush = CreateSolidBrush(lb.lbColor);
		if (hBrush == 0)
			GrafDeviceException("CreateSolidBrush");
	}
	else if (lb.lbStyle == BS_HATCHED) {
		hBrush = CreateHatchBrush(int(lb.lbHatch), lb.lbColor);
		if (hBrush == 0)
			GrafDeviceException("CreateHatchBrush");
	}
	else { // pattern brush
		hBrush = CreatePatternBrush((HBITMAP) lb.lbHatch);
		if (hBrush == 0)
			GrafDeviceException("CreatePatternBrush");
	}
	
	return(hBrush);
}

//==============================================================
//	IGdiDevice::CreateBrushStructure (I need this for CreatePen)
//

LOGBRUSH IGdiDevice::CreateBrushStructure(const IPaint& paint)
{
	LOGBRUSH lb;
	
	// color
	lb.lbColor = convertToWRGB(*paint.color());
	
	// pattern
	if (!paint.imagePattern()) {	// not image but color paint
		const IMaskPattern* paintPattern = paint.maskPattern();
		EMaskPattern pattern = maskPattern(paintPattern);
		
		if (pattern != kCustomPattern)
		{
//			if (fPlatform != kWin95)
			if (!(IPlatform::isWin9x()))
			{
				lb.lbStyle = BS_HATCHED;	// hatch brush and WinNT

				switch (pattern)
				{
				default:
				case kSolidPattern:
					lb.lbStyle = BS_SOLID;
					break;
				case kDiagonalDownPattern:
					lb.lbHatch = HS_BDIAGONAL;
					break;
				case kCrossPattern:
					lb.lbHatch = HS_CROSS;
					break;
				case kDiagonalCrossPattern:
					lb.lbHatch = HS_DIAGCROSS;
					break;
				case kDiagonalUpPattern:
					lb.lbHatch = HS_FDIAGONAL;
					break;
				case kHorizontalPattern:
					lb.lbHatch = HS_HORIZONTAL;
					break;
				case kVerticalPattern:
					lb.lbHatch = HS_VERTICAL;
					break;
				}
			}
			else
			{
				// punt hatch patterns on Win95??? or we can create custom patterns? ...
				lb.lbStyle = BS_SOLID;
			}
		}
		else {	// do the custom hatch pattern...
			lb.lbStyle = BS_PATTERN;
			HBITMAP bitmap = CreateBitmap(8, 8, 1, 1,(const void*)maskBits(paintPattern));
			lb.lbHatch = (long)bitmap;
		}
	}
	else
	{	// do the image pattern paint
		lb.lbStyle = BS_PATTERN;
		HBITMAP bitmap = (HBITMAP)paint.imagePattern()->copyBitmap();
		lb.lbHatch =(long)bitmap;
	}
	
	return(lb);
	
}

//==============================================================
//	IGdiDevice::SetFramePaint(...)
//
void IGdiDevice::SetFillPaint(const IGrafState& grafState, bool nullPen) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFillChanges).attributeStateChanged() ||
		fCachePenBrushStateChange != IStateChange::kFillChanges)
#endif
	{
		if (fNullPenWhenFill != nullPen) {
			if (nullPen) {
				if (fPenCache)
               fPenCache->setPen( (HPEN) GetStockObject( NULL_PEN ) );
				else
               fPenCache = new IPenContext( fHDC,
                                            (HPEN) GetStockObject( NULL_PEN ) );
			}
			fNullPenWhenFill = nullPen;
		}
		if (fBrushCache)
			fBrushCache->setBrush(CreateFillBrush(grafState));
		else
			fBrushCache = new IBrushContext(fHDC, CreateFillBrush(grafState));
		fHollowBrushWhenFrame = false;
		fCachePenBrushStateChange = IStateChange::kFillChanges;
	}
}

//==============================================================
//	IGdiDevice::SetFillPaint(...)
//
void IGdiDevice::SetFramePaint(const IGrafState& grafState, bool hollowBrush) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFrameChanges).attributeStateChanged() ||
		fCachePenBrushStateChange != IStateChange::kFrameChanges)
#endif
	{
		if (fHollowBrushWhenFrame != hollowBrush) {
			if (hollowBrush) {
				if (fBrushCache)
               fBrushCache->setBrush( (HBRUSH)
                                      GetStockObject( HOLLOW_BRUSH ) );
				else
               fBrushCache =
                  new IBrushContext( fHDC,
                                     (HBRUSH) GetStockObject( HOLLOW_BRUSH ) );
			}
			fHollowBrushWhenFrame = hollowBrush;
		}
		if (fPenCache)
			fPenCache->setPen(CreatePen(grafState));
		else
			fPenCache = new IPenContext(fHDC, CreatePen(grafState));
		fNullPenWhenFill = false;
		fCachePenBrushStateChange = IStateChange::kFrameChanges;
	}
}

//==============================================================
//	IGdiDevice::SetFillAndFramePaint(...)
//
void IGdiDevice::SetFillAndFramePaint(const IGrafState& grafState) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFillOrFrameChanges).attributeStateChanged() ||
		fCachePenBrushStateChange != IStateChange::kFillOrFrameChanges)
#endif
	{
		if (fPenCache)
			fPenCache->setPen(CreatePen(grafState));
		else
			fPenCache = new IPenContext(fHDC, CreatePen(grafState));
		if (fBrushCache)
			fBrushCache->setBrush(CreateFillBrush(grafState));
		else
			fBrushCache = new IBrushContext(fHDC, CreateFillBrush(grafState));
		fNullPenWhenFill = false;
		fHollowBrushWhenFrame = false;
		fCachePenBrushStateChange = IStateChange::kFillOrFrameChanges;
	}
}

//==============================================================
//	IGdiDevice::SetFillPaintModes(...)
//

void IGdiDevice::SetFillPaintModes(const IGrafState& state) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFillChanges).attributeStateChanged() ||
		fCachePaintModeStateChange != IStateChange::kFillChanges)
#endif
	{
		const IAttributeState* attState = state.attributeState();
		SetPaintModes(attState->fillPaint(), attState->fillTransferMode());
		fCachePaintModeStateChange = IStateChange::kFillChanges;
	}
}

//==============================================================
//	IGdiDevice::SetFramePaintModes(...)
//

void IGdiDevice::SetFramePaintModes(const IGrafState& state) {
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if ((IStateChange(fStateChange) &= IStateChange::kFrameModeChanges).attributeStateChanged() ||
		fCachePaintModeStateChange != IStateChange::kFrameModeChanges)
#endif
	{
		const IAttributeState* attState = state.attributeState();
		SetPaintModes(attState->framePaint(), attState->frameTransferMode());
		fCachePaintModeStateChange = IStateChange::kFrameModeChanges;
	}
}

//==============================================================
//	IGdiDevice::SetPaintModes(...)
//

void IGdiDevice::SetPaintModes(
	const IPaint* paint,
	const IColorTransferMode* xmode)
{
	if (paint->maskPattern() != &IMaskPattern::solid() || paint->imagePattern())
	{
		const IGPoint2D* patPhase = paint->patternPhase();

		// set Pattern Origin
		if (!SetBrushOrgEx(fHDC, ((int)patPhase->fX) % 8, ((int)patPhase->fY) % 8, NULL))
			GrafDeviceException("SetBrushOrgEx");

    if (maskPattern(paint->maskPattern()) == kCustomPattern)
      SetBkColor(fHDC, convertToWRGB(*(paint->color())));

	}
	
	if (!fBackModeSet) {
		// set Background Mode
		if (!SetBkMode(fHDC, TRANSPARENT))
			GrafDeviceException("SetBkMode");

		fBackModeSet = true;
	}

	unsigned long theMode = xmode->mode();
#if !(defined(__OS2__) && defined(OS2CHANGES))
	if (fCacheTransferMode != theMode)
#endif
		 {

		fCacheTransferMode = theMode;
	
		// set transfermode
		int wmixMode;
		
		// in Win SDK MASK=AND, MERGE=OR, XOR=XOR
		switch (fCacheTransferMode)
		{
		default:
		case IColorTransferMode::kSourceCopy:
			wmixMode = R2_COPYPEN;
			break;
		case IColorTransferMode::kInvertSource:
			wmixMode = R2_NOTCOPYPEN;
			break;
		case IColorTransferMode::kDestinationCopy:
			wmixMode = R2_NOP;
			break;
		case IColorTransferMode::kInvertDestination:
			wmixMode = R2_NOT;
			break;
		case IColorTransferMode::kOR:
			wmixMode = R2_MERGEPEN;
			break;
		case IColorTransferMode::kAND:
			wmixMode = R2_MASKPEN;
			break;
		case IColorTransferMode::kXOR:
			wmixMode = R2_XORPEN;
			break;
		case IColorTransferMode::kInvertedSourceAND:
			wmixMode = R2_MASKNOTPEN;
			break;
		case IColorTransferMode::kInvertedSourceOR:
			wmixMode = R2_MERGENOTPEN;
			break;
		case IColorTransferMode::kInvertedDestinationAND:
			wmixMode = R2_MASKPENNOT;
			break;
		case IColorTransferMode::kInvertedDestinationOR:
			wmixMode = R2_MERGEPENNOT;
			break;
		case IColorTransferMode::kInvertedAND:
			wmixMode = R2_NOTMASKPEN;
			break;
		case IColorTransferMode::kInvertedOR:
			wmixMode = R2_NOTMERGEPEN;
			break;
		case IColorTransferMode::kInvertedXOR:
			wmixMode = R2_NOTXORPEN;
			break;
		case IColorTransferMode::kONES:
			wmixMode = R2_WHITE;
			break;
		case IColorTransferMode::kZEROS:
			wmixMode = R2_BLACK;
			break;
		}
		
		if (!SetROP2(fHDC, wmixMode))
			GrafDeviceException("SetROP2");
	}
}


//==============================================================
//	IGdiDevice::FrameEqualFillPaintModes
//

bool IGdiDevice::FrameEqualFillPaintModes(const IGrafState& state) const
{
	const IAttributeState* attState = state.attributeState();

	if (*attState->fillTransferMode() != *attState->frameTransferMode())
		return false;

	const IPaint* fillPaint = attState->fillPaint();
	const IPaint* framePaint = attState->framePaint();

	return (*fillPaint->patternPhase() == *framePaint->patternPhase());
}

//==============================================================
//	IGdiDevice::imageTransferMode
//

#define ICLUIHalfToneMode 0X00FC008A

int IGdiDevice::imageTransferMode(const IGrafState& state)
{
	int GdiMode = SRCCOPY;
	
	switch (state.attributeState()->imageTransferMode()->mode())
	{
	case IImageTransferMode::kSrcCopy:
		GdiMode = SRCCOPY;
		break;
	case IImageTransferMode::kSrcORDst:
		GdiMode = SRCPAINT;
		break;
	case IImageTransferMode::kSrcANDDst:
		GdiMode = SRCAND;
		break;
	case IImageTransferMode::kSrcXORDst:
		GdiMode = SRCINVERT;
		break;
	case IImageTransferMode::kSrcANDInvertDst:
		GdiMode = SRCERASE;
		break;
	case IImageTransferMode::kInvertSrc:
		GdiMode = NOTSRCCOPY;
		break;
	case IImageTransferMode::kInvertResultOfSrcORDst:
		GdiMode = NOTSRCERASE;
		break;
	case IImageTransferMode::kSrcANDPattern:
		GdiMode = MERGECOPY;
		break;
	case IImageTransferMode::kInvertSrcORDst:
		GdiMode = MERGEPAINT;
		break;
	case IImageTransferMode::kPatternCopy:
		GdiMode = PATCOPY;
		break;
	case IImageTransferMode::kPatternORInvertSrcORDst:
		GdiMode = PATPAINT;
		break;
	case IImageTransferMode::kPatternXORDst:
		GdiMode = PATINVERT;
		break;
	case IImageTransferMode::kInvertDst:
		GdiMode = DSTINVERT;
		break;
	case IImageTransferMode::kAllZeros:
		GdiMode = BLACKNESS;
		break;
	case IImageTransferMode::kAllOnes:
		GdiMode = WHITENESS;
		break;
	case IImageTransferMode::kOCHalfTone:
		// do it just like in ICLUI, make it a special code - 0x00fc008a
		GdiMode = ICLUIHalfToneMode;
		break;
	}
	
	return GdiMode;
}


//==============================================================
//	IGdiDevice::imageSamplingControl
//

int IGdiDevice::imageSamplingControl(const IGrafState& state)
{
	int GdiMode = COLORONCOLOR;
	switch (state.attributeState()->imageSampling()->mode())
	{
	case IImageSamplingControl::kBlackOnWhite:
		GdiMode = BLACKONWHITE;
		break;
	case IImageSamplingControl::kWhiteOnBlack:
		GdiMode = WHITEONBLACK;
		break;
	case IImageSamplingControl::kColorOnColor:
		GdiMode = COLORONCOLOR;
		break;
	case IImageSamplingControl::kHalfTone:
		GdiMode = HALFTONE;
		break;
	}
	return GdiMode;
}



//==============================================================
//	IGdiDevice::SetClipArea
//
//	Note: setupTransforms has to be called prior to calling this function, because this uses
//			fFrameMatrix.
//		returns: kClipAreaEmpty (shouldn't draw anything - clipping is not set)
//		         kClipAreaInfinite (no clipping - clipping reset to NULL)
//		         kClipAreaRegular (other - clipping set)

int IGdiDevice::SetClipArea(const IGrafState& state)
{
	// clipArea is transformed in GrafState.
	const IGArea* clipArea = state.clipArea();
		
	if (clipArea->isEmpty()) return(kClipAreaEmpty);

	if (fClipAreaSet) {
		if (!fStateChange.clipAreaChanged())
			return kClipAreaRegular;
	}
	else
		fClipAreaSet = true;
		
	if (clipArea->bounds() == IGRect2D::infiniteRect())
	{
		if (ExtSelectClipRgn(fHDC, NULL, RGN_COPY) == ERROR)
		{	// set clip region to NULL
			GrafDeviceException("ExtSelectClipRgn");
		}
		return kClipAreaInfinite;
	}
	else {
		IRegionGrafExtractor regionExtractor(this);
		ICAGRoot root;
		IRegionHandle handle;

		extract(*clipArea, regionExtractor, &root);
		
		if (regionExtractor.region(handle) != NULLREGION) {
			if (ExtSelectClipRgn(fHDC, (HRGN)handle, RGN_COPY) == ERROR)
				GrafDeviceException("ExtSelectClipRgn");
			return kClipAreaRegular;
		}
		else {
			return kClipAreaEmpty;
		}
	}
}


//==============================================================
//	IGdiDevice::SetWinWorldTransform
//

void IGdiDevice::SetWinWorldTransform(const IGrafMatrix* grafmatrix)
{
//	if (fPlatform != kWin95)
	if (!(IPlatform::isWin9x()))
	{
		if (!SetGraphicsMode(fHDC, GM_ADVANCED))	// to enable world transform
			GrafDeviceException("SetGraphicsMode");
		
		if (!grafmatrix || grafmatrix->isIdentity())
		{
			if (fMatrixCacheUsed) {
				if (fMatrixCache.isIdentity())
					return; // cache hit...
			}
			else
				fMatrixCacheUsed = true;

			fMatrixCache.setToIdentity();

			if (!ModifyWorldTransform(fHDC, 0, MWT_IDENTITY))
				GrafDeviceException("ModifyWorldTransform");
		}
		else { // not identity
			if (fMatrixCacheUsed) {
				if (fMatrixCache == *grafmatrix)
					return; // cache hit...
			}
			else
				fMatrixCacheUsed = true;

			fMatrixCache = *grafmatrix;
			fXFORM.eM11 = FLOAT(fMatrixCache.element(IGrafMatrix::kScaleX));
			fXFORM.eM12 = FLOAT(fMatrixCache.element(IGrafMatrix::kShearY));
			fXFORM.eM21 = FLOAT(fMatrixCache.element(IGrafMatrix::kShearX));
			fXFORM.eM22 = FLOAT(fMatrixCache.element(IGrafMatrix::kScaleY));
			fXFORM.eDx = FLOAT(fMatrixCache.element(IGrafMatrix::kTranslateX));
			fXFORM.eDy = FLOAT(fMatrixCache.element(IGrafMatrix::kTranslateY));
		
			if (!SetWorldTransform(fHDC, &fXFORM))
				GrafDeviceException("SetWorldTransform");
		}
	}
}


/* ------------------------------------------------------------
transform issue:

	T(model) = T(geometry) . T(view) (short: Tm = Tg.Tv)

	NT:
		For given geometry g,

		G = g.Tg;
		
		if (Tv ~perspective)
			extract Tv = affine(Tv)
		
		render(G) with Tv

	W95:
		For given geometry g,
		G = g.Tm; set penwidth by using Tv
		render(G)

------------------------------------------------------------ */

void IGdiDevice::setupTransforms(const IGrafState& state)
{
	if (!fStateChange.matrixChanged())
		return;
	fModelMatrix = state.modelMatrix();
//	if (fPlatform == kWin95) {
	if (IPlatform::isWin9x()) {
		fViewMatrix = 0 /*NIL*/;
		fFrameMatrix = fModelMatrix;
		fPenMatrix = state.viewMatrix();
	}
	else {	// WinNT
		fViewMatrix = state.viewMatrix();
		fMatrix = *fViewMatrix;
		fMatrix.invert();
		fMatrix.preConcatWith(*fModelMatrix);
		fFrameMatrix = &fMatrix;
		fPenMatrix = 0 /*NIL*/;
	}
}


//==============================================================
//	IGdiDevice::renderLine
//

void IGdiDevice::renderLine(const IGLine2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;

	IGLine2D tline(geometry);
	if (!fFrameMatrix->isIdentity())
		tline.transformBy(*fFrameMatrix);
	
	SetWinWorldTransform(fViewMatrix);
	
	IGPoint2D pts[2];
	tline.points(pts[0], pts[1]);
	
	// create Pen and select it to the DC
	SetFramePaint(grafState);
	SetFramePaintModes(grafState);
	
	// draw Line
	if (!MoveToEx(fHDC, GCoordinateToInt(pts[0].fX), GCoordinateToInt(pts[0].fY), NULL))
		GrafDeviceException("MoveToEx");
	if (!LineTo(fHDC, GCoordinateToInt(pts[1].fX), GCoordinateToInt(pts[1].fY)))
		GrafDeviceException("LineTo");
	
}


//==============================================================
//	IGdiDevice::renderPolyline
//

void IGdiDevice::renderPolyline(const IGPolyline2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGPolyline2D tpolyline(geometry);
	if (!fFrameMatrix->isIdentity())
		tpolyline.transformBy(*fFrameMatrix);
	
	SetWinWorldTransform(fViewMatrix);
	
	// set up gdi points for polyline.
	long numPoints = tpolyline.numberOfPoints();
	IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
	for (unsigned long i=0; i < numPoints; i++)
		IGPoint2DToPOINTL(tpolyline.point(i), sp[i]);

	// create Pen and select it to the DC
	SetFramePaint(grafState);
	SetFramePaintModes(grafState);
	
	// draw Polyline
	if (!Polyline(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))	
		GrafDeviceException("Polyline");
}

//==============================================================
//	IGdiDevice::renderRect
//

void IGdiDevice::renderRect(const IGRect2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;

	IGRect2D trect(geometry);
	
	fDrawOperation = grafState.attributeState()->drawingOperation();

//	if (fPlatform == kWin95 ||
	if (IPlatform::isWin9x() ||
		fDrawOperation == IAttributeState::kFillAndFrame ||
		fDrawOperation == IAttributeState::kFrame)
	{
		if (!fFrameMatrix->isIdentity())
		{
			if (fFrameMatrix->isRectilinear())
			{
				trect = fFrameMatrix->transformBounds(trect);
			}
			else
			{
				IGPolygon2D polygon(4);
				polygon.setPoint(0, trect.topLeft());
				polygon.setPoint(1, trect.bottomLeft());
				polygon.setPoint(2, trect.bottomRight());
				polygon.setPoint(3, trect.topRight());
	
				renderPolygon(polygon, grafState);
						
				return;
			}
		}
		else
		{
			// just incase someone made the transposed rectangle or it is transformed to be transposed:
			trect.orderPoints();
		}
		SetWinWorldTransform(fViewMatrix);
	}
	else	// not Win95 and fDrawOperation is kFill only
	{
		// just incase someone made the transposed rectangle or it is transformed to be transposed:
		trect.orderPoints();
		SetWinWorldTransform(fModelMatrix);
	}

//	if( fPlatform == kWin95 && fFrameMatrix->isRectilinear() )
	if( IPlatform::isWin9x() && fFrameMatrix->isRectilinear() )
	{
		IGRect2D transformedWorld( fFrameMatrix->transformBounds( worldBounds() ));
		if (trect.contains( transformedWorld ))	// for infinite rect or somewhat huge rect...
			trect = transformedWorld;
	}
 	else
 	{
		if (trect.contains(worldBounds()) )	// for infinite rect or somewhat huge rect...
			trect = worldBounds();
	}

	POINTL gp1;
	POINTL gp2;
	IGPoint2DToPOINTL(trect.topLeft(), gp1);
	IGPoint2DToPOINTL(trect.bottomRight(), gp2);

//	if (fPlatform == kWin95)	//  since Win95 is one pixel less than NT at right and bottom boundary
	if (IPlatform::isWin9x())	//  since Win95 is one pixel less than NT at right and bottom boundary
	{
		gp2.x += 1.;
		gp2.y += 1.;
	}

	
	// draw rectangle
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState))
	{
		// create a PEN and select it in DC
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);
		
		if (!Rectangle(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
			GrafDeviceException("Rectangle");
	}
	else
	{
		// it is important to do fill before frame
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState, true);
			SetFillPaintModes(grafState);
			if (!Rectangle(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
				GrafDeviceException("Rectangle");

		}
		
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState, true);
			SetFramePaintModes(grafState);

			if (!Rectangle(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
				GrafDeviceException("Rectangle");

		}
	}
}

//==============================================================
//	IGdiDevice::renderEllipse
//

void IGdiDevice::renderEllipse(const IGEllipse2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	fDrawOperation = grafState.attributeState()->drawingOperation();
	
	IGRect2D trect(geometry.bounds());
	
	// Try to fix defect 3102
	// Avoid using Win32 functions when drawing Arc, Chord or Ellipse on Win95
	// platform since those calls result in crashes sometimes
	// Crash Example: Ellipse(hdc, -670, -3696, 3531, 505)
//	if (fPlatform == kWin95)
	if (IPlatform::isWin9x())
	{
		IGLoop2D elliptical(geometry);
		renderLoop(elliptical, grafState);
		return;
	}	
	
/*	//commmented to fix defect 3102
	if (fPlatform == kWin95 ||
		fDrawOperation == IAttributeState::kFillAndFrame ||
		fDrawOperation == IAttributeState::kFrame)
*/		
	if (fDrawOperation == IAttributeState::kFillAndFrame ||
		fDrawOperation == IAttributeState::kFrame)
	{
		if (!fFrameMatrix->isIdentity())
		{
			if (fFrameMatrix->isRectilinear())
			{
				trect = fFrameMatrix->transformBounds(trect);
			}
			else
			{
				IGLoop2D elliptical(geometry);
				renderLoop(elliptical, grafState);
						
				return;
			}
		}
		SetWinWorldTransform(fViewMatrix);
	}
	else	// not Win95 and fDrawOperation is kFill only
		SetWinWorldTransform(fModelMatrix);
	
	// just incase someone made the transposed rectangle or it is transformed to be transposed:
	trect.orderPoints();
	
	POINTL gp1;
	POINTL gp2;
	IGPoint2DToPOINTL(trect.topLeft(), gp1);
	IGPoint2DToPOINTL(trect.bottomRight(), gp2);

	// draw ellipse
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState))
	{
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);

		if (!Ellipse(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
			GrafDeviceException("Ellipse");
	}
	else
	{
		// it is important to do fill before frame
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState, true);
			SetFillPaintModes(grafState);
			
			if (!Ellipse(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
				GrafDeviceException("Ellipse");
		}
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState, true);
			SetFramePaintModes(grafState);
			
			if (!Ellipse(fHDC, int(gp1.x), int(gp1.y), int(gp2.x), int(gp2.y)))
				GrafDeviceException("Ellipse");
		}
	}
}

//==============================================================
//	IGdiDevice::renderPolygon
//

void IGdiDevice::renderPolygon(const IGPolygon2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGPolygon2D tpolygon(geometry);
	if (!fFrameMatrix->isIdentity())
		tpolygon.transformBy(*fFrameMatrix);
	
	SetWinWorldTransform(fViewMatrix);
	
	// set up gdi points for polygon.
	long numPoints = tpolygon.numberOfPoints();
	IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
	for (unsigned long i=0; i < numPoints; i++)
		IGPoint2DToPOINTL(tpolygon.point(i), sp[i]);
	
	fDrawOperation = grafState.attributeState()->drawingOperation();
	
	// draw polygon
	if (tpolygon.eOFill())
		SetPolyFillMode(fHDC, ALTERNATE);	// even-odd rule
	else
		SetPolyFillMode(fHDC, WINDING);		// non-zero winding rule
	
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState))
	{
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);
		
		if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
			GrafDeviceException("Polygon");
	}
	else
	{
		// it is important to do fill before frame
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState);
			SetFillPaintModes(grafState);
			
			BeginPath(fHDC);
			if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
				GrafDeviceException("Polygon");
			CloseFigure(fHDC); // to make sure the endings of the final intersection
			                   // are drawn with joints (need to verify)
			EndPath(fHDC);

			if (!FillPath(fHDC))
				GrafDeviceException("FillPath");

		}
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState);
			SetFramePaintModes(grafState);
			
			BeginPath(fHDC);
			if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
				GrafDeviceException("Ellipse");
			CloseFigure(fHDC); // to make sure the endings of the final intersection
			                   // are drawn with joints (need to verify)
			EndPath(fHDC);

			if (!StrokePath(fHDC))
				GrafDeviceException("StrokePath");

		}
	}
}

//==============================================================
//	IGdiDevice::renderArea
//

void IGdiDevice::renderArea(const IGArea& area, const IGrafState& grafState)
{
	if (area.isEmpty())
		return;
		
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;

	IRegionGrafExtractor regionExtractor(this);
	ICAGRoot root;
	extract(area, regionExtractor, &root, *fFrameMatrix);

	IRegionHandle handle;
	if (regionExtractor.region(handle) == NULLREGION)
		return;

	SetWinWorldTransform(fViewMatrix);
	fDrawOperation = grafState.attributeState()->drawingOperation();
	
	// paint the resulting region(fill)
	if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
	{
		// leave this one dangling ... look at it later
		IBrushContext aBrush(fHDC, CreateFillBrush(grafState), false);
		SetFillPaintModes(grafState);
		
		if (!FillRgn(fHDC, (HRGN)handle, aBrush.fBrush))
			GrafDeviceException("FillRgn");
	}
	
	// frame the resulting region
	if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
	{
		const IPen* pen = grafState.attributeState()->framePen();
		IPen::EPenType penType = pen->penType();
		int frameWidth, frameHeight;			// must be logical units
		if (penType == IPen::kHairline)
		{
			frameWidth = frameHeight = 1;
		}
		else if (penType == IPen::kSolid)
		{
			if (fPenMatrix && !fPenMatrix->isIdentity())
			{
				// pen width calculation ... only done on Win95
				IGPoint2D pts[2];	// unit vector
				pts[0] = IGPoint2D::origin();
				pts[1] = IGPoint2D(1, 1);
				fPenMatrix->transformPoints(pts, pts, 2);
				pts[1] -= pts[0];
				pts[0].fX = pen->penWidth();
				frameWidth = GCoordinateToInt(pts[1].fX * pts[0].fX);
				frameHeight = GCoordinateToInt(pts[1].fY * pts[0].fX);
			}
			else
				frameWidth = frameHeight = GCoordinateToInt(pen->penWidth());

			frameWidth = frameWidth > 1 ? frameWidth : 1;	// must always be >= 1
			frameHeight = frameHeight > 1 ? frameHeight : 1;	// must always be >= 1
		}
		else	// else punt on other kinds of pens ...???
			frameWidth = frameHeight = 1;
		
		// leave this one dangling ... look at it later
		IBrushContext aBrush(fHDC, CreateFrameBrush(grafState), false);
		SetFramePaintModes(grafState);
		
		if (!FrameRgn(fHDC, (HRGN)handle, aBrush.fBrush, frameWidth, frameHeight))
			GrafDeviceException("FrameRgn");
	}
}

#ifndef NO_CONIC_ACCELERATION_SUPPORT
#include <conics.hpp>

#pragma pack(push,4)
class ConicMaker {
private:
	int fLeft, fTop, fRight, fBottom;
	int fStartX, fStartY, fEndX, fEndY;
	IConicInfo::ECurveType fType;
	bool fIsLoop;

	void SetupForRendering(const IGCurve2D& curve) ;

public:
	ConicMaker(const IGCurve2D& curve) ;
	ConicMaker(const IGLoop2D& loop) ;

	void render(HDC dc) ;
};

void ConicMaker::SetupForRendering(const IGCurve2D& curve) {
	IConicInfo* ci = curve.fConicInfo;
	fType = ci->fType;

	fLeft = IGdiDevice::GCoordinateToInt(ci->fArcBounds.fLeft);
	fTop = IGdiDevice::GCoordinateToInt(ci->fArcBounds.fTop);
	fRight = IGdiDevice::GCoordinateToInt(ci->fArcBounds.fRight);
	fBottom = IGdiDevice::GCoordinateToInt(ci->fArcBounds.fBottom);

	if (fType != IConicInfo::kEllipse) {
		IGPoint2D startPoint, endPoint;
		if (fType == IConicInfo::kArc) {
			startPoint = curve.evaluate(curve.minParameter());
			endPoint = curve.evaluate(curve.maxParameter());
		}
		else {
			startPoint = ci->fStart;
			endPoint = ci->fEnd;
		}

		if (ci->fDirection) {
			fStartX = IGdiDevice::GCoordinateToInt(endPoint.fX);
			fStartY = IGdiDevice::GCoordinateToInt(endPoint.fY);
			fEndX = IGdiDevice::GCoordinateToInt(startPoint.fX);
			fEndY = IGdiDevice::GCoordinateToInt(startPoint.fY);
		}
		else {
			fStartX = IGdiDevice::GCoordinateToInt(startPoint.fX);
			fStartY = IGdiDevice::GCoordinateToInt(startPoint.fY);
			fEndX = IGdiDevice::GCoordinateToInt(endPoint.fX);
			fEndY = IGdiDevice::GCoordinateToInt(endPoint.fY);
		}
	}
}

ConicMaker::ConicMaker(const IGCurve2D& curve) {
	SetupForRendering(curve);
	fIsLoop = false;
}

ConicMaker::ConicMaker(const IGLoop2D& loop) {
	SetupForRendering(loop);
	fIsLoop = true;
}

void ConicMaker::render(HDC dc) {
	if (fType == IConicInfo::kEllipse) {
		if (!Ellipse(dc, fLeft, fTop, fRight, fBottom))
			GrafDeviceException("Ellipse");
	}
	else if (fIsLoop) {
		if (fType == IConicInfo::kPie) {
			if (!Pie(dc, fLeft, fTop, fRight, fBottom, fStartX, fStartY, fEndX, fEndY))
				GrafDeviceException("Pie");
		}
		else {
			if (!Chord(dc, fLeft, fTop, fRight, fBottom, fStartX, fStartY, fEndX, fEndY))
				GrafDeviceException("Chord");
		}
	}
	else {
		if (!Arc(dc, fLeft, fTop, fRight, fBottom, fStartX, fStartY, fEndX, fEndY))
			GrafDeviceException("Arc");
	}
}

#pragma pack(pop)
#endif // NO_CONIC_ACCELERATION_SUPPORT


//================================================================
//	IGdiDevice::renderCurve
//
// Parameters:
//	geometry : the curve to be rendered
//	grafState: indicates the attributes/transformation to be applied
//
// Return value: NONE
//
// Purpose: renders the curve geometry to the display
//
//
void IGdiDevice::renderCurve(const IGCurve2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGCurve2D tcurve(geometry);
	// Try to fix defect 3102
	// Avoid using Win32 functions when drawing Arc, Chord or Ellipse on Win95
	// platform since those calls result in crashes sometimes
	// Crash Example: Ellipse(hdc, -670, -3696, 3531, 505)
	//                Arc(hdc, -670, -3696, 3531, 505, 290, 168, 132, 55)
//	if ((fPlatform == kWin95) && conicInfo(tcurve))
	if ((IPlatform::isWin9x()) && conicInfo(tcurve))
#ifdef MAYBE_A_BUG
		tcurve.fConicInfo = 0 /*NIL*/;
#else
		deleteConicInfo(tcurve);
#endif // MAYBE_A_BUG
	
	if (!fFrameMatrix->isIdentity()) {
		tcurve.transformBy(*fFrameMatrix);
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		if (conicInfo(tcurve)) {
			if (!conicInfo(tcurve)->transformBy(*fFrameMatrix)) {
				deleteConicInfo(tcurve);
			}
		}
#endif // NO_CONIC_ACCELERATION_SUPPORT
	}
	
	SetWinWorldTransform(fViewMatrix);
	
	// create Pen and select it to the DC
	SetFramePaint(grafState);
	SetFramePaintModes(grafState);
	
#ifndef NO_CONIC_ACCELERATION_SUPPORT
	if (conicInfo(tcurve))
	{	// if curve is supported by windows and is an arc...
		ConicMaker(tcurve).render(fHDC);
	}
	else
#endif // NO_CONIC_ACCELERATION_SUPPORT
	{
		unsigned long numPoints;
		unsigned long order = tcurve.order();

		if (order > 2) {	// make everything order == 4 beziers...
			if (order < 4)
				tcurve.raiseOrder(4);
			else if (order > 4)
				tcurve.approximateLowerOrder(4, 0.5);	// bezified first and then tessolated...

			if (!tcurve.isBezier())
				tcurve.refineToBeziers();

			numPoints = tcurve.numberOfPoints();
			IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
			
			for (unsigned long i = 0; i < numPoints; i++) {
				IGPoint2DToPOINTL(tcurve.point(i).divW(), sp[i]);
			}
			
			unsigned long numKnots = tcurve.numberOfKnots() - 1;
			unsigned long start = 0;
			unsigned long end;
			while ((end = tcurve.nextDiscontinuity(start, IGCurve2D::kBreak)) < numKnots)
			{
				if (!PolyBezier(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
					GrafDeviceException("PolyBezier");
				start = end;
			}
			
		}
		else {	// is a polyline
			numPoints = tcurve.numberOfPoints();
			IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
			
			unsigned long start = 0;
			unsigned long end;
			
			for (unsigned long i = 0; i < numPoints; i++) {
				IGPoint2DToPOINTL(tcurve.point(i).dropW(), sp[i]);
				if (tcurve.knot(i + 1) == tcurve.knot(i + 2)) {
					end = i + 1;
					if (!Polyline(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
						GrafDeviceException("Polyline");
					start = end;
				}
			}
		}
	}

}


//================================================================
//	IGdiDevice::renderLoop
//
// Parameters:
//	geometry : the loop to be rendered
//	grafState: indicates the attributes/transformation to be applied
//
// Return value: NONE
//
// Purpose: renders the loop geometry to the display
//
//

inline static bool isSingularOrder2(const IGLoop2D& loop) {
	unsigned long numPoints = loop.numberOfPoints();
	for (unsigned long i = 0; i++ < numPoints; ) {
		if (loop.knot(i) == loop.knot(i + 1))
			return (i == numPoints);
	}
	return false;
}

void IGdiDevice::renderLoop(const IGLoop2D& geometry, const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGLoop2D tloop(geometry);
	
	// Try to fix defect 3102
	// Avoid using Win32 functions when drawing Arc, Chord or Ellipse on Win95
	// platform since those calls result in crashes sometimes
	// Crash Example: Chord(hdc, -670, -3696, 3531, 505, 290, 168, 132, 55)
//	if ((fPlatform == kWin95) && conicInfo(tloop))
	if ((IPlatform::isWin9x()) && conicInfo(tloop))
#ifdef MAYBE_A_BUG
		tloop.fConicInfo = 0 /*NIL*/;
#else
		deleteConicInfo(tloop);
#endif // MAYBE_A_BUG
	
	if (!fFrameMatrix->isIdentity()) {
		tloop.transformBy(*fFrameMatrix);
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		if (conicInfo(tloop)) {
			if (!conicInfo(tloop)->transformBy(*fFrameMatrix)) {
				deleteConicInfo(tloop);
			}
		}
#endif // NO_CONIC_ACCELERATION_SUPPORT
	}
	
	SetWinWorldTransform(fViewMatrix);
	
		
	if (tloop.eOFill())
		SetPolyFillMode(fHDC, ALTERNATE);
	else
		SetPolyFillMode(fHDC, WINDING);

	unsigned long numPoints;
	fDrawOperation = grafState.attributeState()->drawingOperation();

	unsigned long order = tloop.order();

	// note: order 4 bezier loop cannot be done with only one pass
	// note: loop cannot be done with only one pass.
	if (fDrawOperation == IAttributeState::kFillAndFrame && FrameEqualFillPaintModes(grafState) && (
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		conicInfo(tloop) ||
#endif // NO_CONIC_ACCELERATION_SUPPORT
		(order == 2 && isSingularOrder2(tloop))))
	{
		SetFillAndFramePaint(grafState);
		SetFillPaintModes(grafState);
	
#ifndef NO_CONIC_ACCELERATION_SUPPORT
		if (conicInfo(tloop)) {	// if loop is supported by windows...
			ConicMaker(tloop).render(fHDC);
		}
		else
#endif // NO_CONIC_ACCELERATION_SUPPORT
		{ // order = 2, single polygon... no c0 discontinuities
			tloop.closeLoop();

			numPoints = tloop.numberOfPoints();
			IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
			for (unsigned long i = 0; i < numPoints; i++)
				IGPoint2DToPOINTL(tloop.point(i).dropW(), sp[i]);

			if (!Polygon(fHDC, (POINT*)((POINTL*)sp), int(numPoints)))
				GrafDeviceException("Polygon");
		}
	}
	else
	{	// fill & frame paint modes are different, fill first...
		tloop.closeLoop();

		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFill)
		{
			SetFillPaint(grafState, true);
			SetFillPaintModes(grafState);
			
#ifndef NO_CONIC_ACCELERATION_SUPPORT
			if (conicInfo(tloop)) {
            IPenContext aPen( fHDC, (HPEN) GetStockObject( NULL_PEN ) );
				ConicMaker(tloop).render(fHDC);
			}
			else
#endif // NO_CONIC_ACCELERATION_SUPPORT
			{
				BeginPath(fHDC);

				if (order > 2) {	// make everything order == 4 beziers...
					if (order < 4)
						tloop.raiseOrder(4);
					else if (order > 4)
						tloop.approximateLowerOrder(4, 0.5);	// bezified first and then tessolated...
		
					if (!tloop.isBezier())
						tloop.refineToBeziers();
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
	
					for (unsigned long i = 0; i < numPoints; i++) {
						IGPoint2DToPOINTL(tloop.point(i).divW(), sp[i]);
					}

					unsigned long numKnots = tloop.numberOfKnots() - 1;
					unsigned long start = 0;
					unsigned long end;
					while ((end = tloop.nextDiscontinuity(start, IGCurve2D::kBreak)) < numKnots)
					{
						if (!MoveToEx(fHDC, int(((POINTL*)sp)[start].x), int(((POINTL*)sp)[start].y), NULL))
							GrafDeviceException("MoveToEx");

						if (!PolyBezier(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
							GrafDeviceException("PolyBezier");

						if (!LineTo(fHDC, int(((POINTL*)sp)[end-1].x), int(((POINTL*)sp)[end-1].y)))
							GrafDeviceException("LineTo");

						CloseFigure(fHDC);
							
						start = end;
					} // while


				}
				else { // order = 2, polygon...
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
					
					unsigned long start = 0;
					unsigned long end;
			
					for (unsigned long i = 0; i < numPoints; i++) {
						IGPoint2DToPOINTL(tloop.point(i).dropW(), sp[i]);
						if (tloop.knot(i + 1) == tloop.knot(i + 2)) {
							end = i + 1;
							if (!Polygon(fHDC, (POINT*)(((POINTL*)sp) + start), int(end - start)))
								GrafDeviceException("Polygon");
							start = end;
						}
					}

				}

				EndPath(fHDC);

				if (!FillPath(fHDC))
					GrafDeviceException("FillPath");

			} // end not arc

		} // end fill

		// frame...		
		if (fDrawOperation == IAttributeState::kFillAndFrame || fDrawOperation == IAttributeState::kFrame)
		{
			SetFramePaint(grafState);
			SetFramePaintModes(grafState);
			
#ifndef NO_CONIC_ACCELERATION_SUPPORT
			if (conicInfo(tloop)) {
            IBrushContext aBrush( fHDC,
                                  (HBRUSH) GetStockObject( HOLLOW_BRUSH ) );
				ConicMaker(tloop).render(fHDC);
			}
			else
#endif // NO_CONIC_ACCELERATION_SUPPORT
			{
				BeginPath(fHDC);

				if (order > 2) {	// make everything order == 4 beziers...
					if (order < 4)
						tloop.raiseOrder(4);
					else if (order > 4)
						tloop.approximateLowerOrder(4, 0.5);	// bezified first and then tessolated...
		
					if (!tloop.isBezier())
						tloop.refineToBeziers();
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
					
					for (unsigned long i = 0; i < numPoints; i++) {
						IGPoint2DToPOINTL(tloop.point(i).divW(), sp[i]);
					}
					
					unsigned long numKnots = tloop.numberOfKnots() - 1;
					unsigned long start = 0;
					unsigned long end;
					while ((end = tloop.nextDiscontinuity(start, IGCurve2D::kBreak)) < numKnots)
					{
						if (!MoveToEx(fHDC, int(((POINTL*)sp)[start].x), int(((POINTL*)sp)[start].y), NULL))
							GrafDeviceException("MoveToEx");
	
						if (!PolyBezier(fHDC, (POINT*)(((POINTL*)sp) + start), end - start))
							GrafDeviceException("PolyBezier");

						if (!LineTo(fHDC, int(((POINTL*)sp)[end-1].x), int(((POINTL*)sp)[end-1].y)))
							GrafDeviceException("LineTo");

						CloseFigure(fHDC);

						start = end;
					} // while
					
				}
				else { // order = 2, polygon...
					tloop.closeLoop();
		
					numPoints = tloop.numberOfPoints();
					IDeleterForArrayOf<POINTL> sp(new POINTL[numPoints]);
					
					unsigned long start = 0;
					unsigned long end;
			
					for (unsigned long i = 0; i < numPoints; i++) {
						IGPoint2DToPOINTL(tloop.point(i).dropW(), sp[i]);
						if (tloop.knot(i + 1) == tloop.knot(i + 2)) {
							end = i + 1;
							if (!Polygon(fHDC, (POINT*)(((POINTL*)sp) + start), int(end - start)))
								GrafDeviceException("Polygon");
							start = end;
						}
					}
					
				}

				EndPath(fHDC);

				if (!StrokePath(fHDC))
					GrafDeviceException("StrokePath");
			} // end of not arc
		} // end of frame
	} // end of different frame & fill paints

}


//----------------------------------------------------------------------------
IBaseColor IGdiDevice::Win95ImageSamplingControl(
	IGImagePixelAccessor& source,
	IGImagePixelAccessor& dest,
	IGPoint2D srcPoint,
	int destX,
	int destY,
	const IGrafState& state)
{
	IBaseColor oldpixel, newpixel;

	struct tPoint {
        int tX;
        int tY;
    };
    tPoint pixelPoint[4];
	IBaseColor thePixel;

	unsigned long oldPixelIndex ;
	unsigned long newPixelIndex ;

	switch (state.attributeState()->imageSampling()->mode())
    {

    default :

    case IImageSamplingControl::kColorOnColor:
        source.pixel(srcPoint, oldpixel);
        break;

    case IImageSamplingControl::kBlackOnWhite:
        dest.pixel(IGPoint2D(destX, destY), newpixel);
        source.pixel(srcPoint, oldpixel);

        oldPixelIndex = RGB(oldpixel.redMix(), oldpixel.greenMix(), oldpixel.blueMix());
				newPixelIndex = RGB(newpixel.redMix(), newpixel.greenMix(), newpixel.blueMix());
				
        oldPixelIndex |= newPixelIndex ;
				
				oldpixel.setRed(GetRValue(oldPixelIndex));
				oldpixel.setGreen(GetGValue(oldPixelIndex));
				oldpixel.setBlue(GetBValue(oldPixelIndex));
        break;

    case IImageSamplingControl::kWhiteOnBlack:
        dest.pixel(IGPoint2D(destX, destY), newpixel);
        source.pixel(srcPoint, thePixel);

        oldpixel.setRed( (CharIntensity)(newpixel.redMix() & thePixel.redMix()));
        oldpixel.setBlue( (CharIntensity)(newpixel.blueMix() & thePixel.blueMix()));
        oldpixel.setGreen( (CharIntensity)(newpixel.greenMix() & thePixel.greenMix()));
        break;

	
    case IImageSamplingControl::kHalfTone:

        //constructing the bounding pixels

        //top-left
        pixelPoint[0].tX = (int)floor(srcPoint.fX);
        pixelPoint[0].tY = (int)floor(srcPoint.fY);
        //bottom-right
	    	pixelPoint[2].tX = (int)ceil(srcPoint.fX);
        pixelPoint[2].tY = (int)ceil(srcPoint.fY);
        //top-right
        pixelPoint[1].tX = pixelPoint[2].tX;
        pixelPoint[1].tY = pixelPoint[0].tY;
        //bottom-left
        pixelPoint[3].tX = pixelPoint[0].tX;
        pixelPoint[3].tY = pixelPoint[2].tY;


        // check if the point falls exactly on the pixel value

        if ( (double)pixelPoint[0].tX == srcPoint.fX )
        {
            if( (double)pixelPoint[0].tY == srcPoint.fY ) //on pixel
            {
						    source.pixel(IGPoint2D((int)pixelPoint[0].tX,
								                       (int)pixelPoint[0].tY),
															oldpixel);				
								
                return oldpixel;
            }
            else if ( (double)pixelPoint[3].tY == srcPoint.fY ) //on pixel
            {
						    source.pixel(IGPoint2D((int)pixelPoint[3].tX,
								                       (int)pixelPoint[3].tY),
														  oldpixel);

                return oldpixel;
            }
            else
            {
                // its on  left y-edge
                double ydist = srcPoint.fY - (double)pixelPoint[0].tY;
                IBaseColor pixel[2];

			   				 source.pixel(IGPoint2D((int)pixelPoint[0].tX,
								                        (int)pixelPoint[0].tY),
														  pixel[0]);
															
								 source.pixel(IGPoint2D((int)pixelPoint[3].tX,
								                        (int)pixelPoint[3].tY),
														  pixel[1]);
				
                oldpixel.setRed((CharIntensity)((pixel[1].redMix() - pixel[0].redMix()) * ydist + pixel[0].redMix()));
                oldpixel.setGreen((CharIntensity)((pixel[1].greenMix() - pixel[0].greenMix()) * ydist + pixel[0].greenMix()));
                oldpixel.setBlue((CharIntensity)((pixel[1].blueMix() - pixel[0].blueMix()) * ydist + pixel[0].blueMix()));

                return oldpixel;
            }
        }
        else if ( (double)pixelPoint[0].tY == srcPoint.fY )
        {
            if( (double)pixelPoint[1].tX == srcPoint.fX ) //on pixel
            {
						
			   				 source.pixel(IGPoint2D((int)pixelPoint[1].tX,
								                        (int)pixelPoint[1].tY),
														  oldpixel);

                return oldpixel;
            }
            else
            {
                // its on  top x-edge
                double xdist = srcPoint.fX - (double)pixelPoint[0].tX;
                IBaseColor pixel[2];

			   				 source.pixel(IGPoint2D((int)pixelPoint[0].tX,
								                        (int)pixelPoint[0].tY),
														  pixel[0]);
															
								 source.pixel(IGPoint2D((int)pixelPoint[1].tX,
								                        (int)pixelPoint[1].tY),
														  pixel[1]);
				
                oldpixel.setRed((CharIntensity)((pixel[1].redMix() - pixel[0].redMix()) * xdist + pixel[0].redMix()));
                oldpixel.setGreen((CharIntensity)((pixel[1].greenMix() - pixel[0].greenMix()) * xdist + pixel[0].greenMix()));
                oldpixel.setBlue((CharIntensity)((pixel[1].blueMix() - pixel[0].blueMix()) * xdist + pixel[0].blueMix()));

                return oldpixel;
            }
        }

        // now the point is neither on the pixel or on the edge. Just interpolate

        double xdist =0, ydist=0 ;
        double redClr[4], greenClr[4], blueClr[4];
        double totred =0, totgreen =0, totblue =0;
        IBaseColor pixel[4];

        xdist =  srcPoint.fX - pixelPoint[0].tX ;
        ydist =  srcPoint.fY - pixelPoint[0].tY  ;

        for (int i =0; i<4; i++)
        {
           source.pixel(IGPoint2D((int)pixelPoint[i].tX,
								                  (int)pixelPoint[i].tY),
												pixel[i]);

        }

        double red[2], green[2], blue[2];

		red[0] = pixel[1].redMix() *xdist + pixel[0].redMix() * (1.0 - xdist);
        red[1] = pixel[2].redMix() *xdist + pixel[3].redMix() * (1.0 - xdist);

		green[0] = pixel[1].greenMix() *xdist + pixel[0].greenMix() * (1.0 - xdist);
        green[1] = pixel[2].greenMix() *xdist + pixel[3].greenMix() * (1.0 - xdist);

		blue[0] = pixel[1].blueMix() *xdist + pixel[0].blueMix() * (1.0 - xdist);
        blue[1] = pixel[2].blueMix() *xdist + pixel[3].blueMix() * (1.0 - xdist);
	
        totred = red[1]* ydist + red[0]* (1.0 - ydist);
        totgreen = green[1]* ydist + green[0]* (1.0 - ydist);
        totblue = blue[1]* ydist + blue[0]* (1.0 - ydist);

        oldpixel.setRed((CharIntensity)totred);
        oldpixel.setGreen((CharIntensity)totgreen);
        oldpixel.setBlue((CharIntensity)totblue);

        break;
   }
    return oldpixel;
}

void IGdiDevice::Win95RenderImage(
	const IGImage& image,
	const IGRect2D& source,
	const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();

	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGRect2D trect = source;
	IGRect2D irect = image.rasterBounds();
	
	const IGrafMatrix* tMatrix = grafState.modelMatrix();
	trect.orderPoints();
	if (!tMatrix->isIdentity())
		trect = tMatrix->transformBounds(trect);

//	if (fPlatform == kWin95 && !fFrameMatrix->isIdentity())
	if (IPlatform::isWin9x() && !fFrameMatrix->isIdentity())
	{
		trect.intersectWith(fFrameMatrix->transformBounds(worldBounds()));
	}
	else
	{
		trect.intersectWith(worldBounds());
	}
	
	if (trect.isEmpty())
		return;

    int left = GCoordinateToInt(trect.fLeft);
    int top = GCoordinateToInt(trect.fTop);
    int width = GCoordinateToInt(trect.width()) ;
    int height = GCoordinateToInt(trect.height()) ;

	IGRect2D theRect( trect.topLeft(), trect.bottomRight()+IGPoint2D(1,1) );	
	IPresSpaceHandle dc(fHDC);
	IGImage destImage(IGImage::captureImage(dc, theRect));

	{ //scope for IGImagePixelAccessor
		IGImagePixelAccessor dest(destImage);

		IGImage original(image);
		original.sizeTo(IGPoint2D(irect.width()+1, irect.height()+1 ));

		IGImagePixelAccessor src(original);

		IBaseColor oldpixel;

		IGrafMatrix invMatrix = *tMatrix;
		invMatrix.invert();

		for (int i = 0; i < height; i++)
		{
			for (int j = 0; j < width; j++)
			{
				IGPoint2D ipoint(trect.fLeft + j, trect.fTop + i);
				IGPoint2D newPoint = invMatrix.transformPoint(ipoint);
				if (irect.contains(newPoint))
				{
					IGPoint2D transformedPoint = newPoint - irect.topLeft();
					oldpixel =
						Win95ImageSamplingControl(
							src,
							dest,
							transformedPoint,
							j,
							i,
							grafState);
					dest.setPixel(IGPoint2D(j, i), oldpixel);
				}
			}
		}
	}//scope for IGImagePixelAccessor

	HBITMAP  destHandle = destImage.handle();
	IGImageDeviceContext theDC(destImage);
	HDC destHDC = theDC;

	int rasterOperation = imageTransferMode(grafState);
   HBITMAP fOldBitmap = (HBITMAP) SelectObject( destHDC, destHandle );
	
	if (rasterOperation != ICLUIHalfToneMode)
	{
		if (!BitBlt(fHDC, left, top, width, height, destHDC, 0, 0, rasterOperation))
			GrafDeviceException("BitBlt");
	}
	else
	{	// If the raster-op is halftone, set the pattern to a checker-board ...
		setTextColour(IBaseColor::kWhite); // white foreground
		SetBkColor(fHDC, 0x00000000); // black background

		WORD checker[] = { 0x55,0xaa,0x55,0xaa,0x55,0xaa,0x55,0xaa };
		HBITMAP bitmap = CreateBitmap(8, 8, 1, 1, (const void*)checker);
		IBrushContext aBrush(fHDC, CreatePatternBrush(bitmap));
		
		if (!BitBlt(fHDC, left, top, width, height, destHDC, 0, 0, rasterOperation))
			GrafDeviceException("BitBlt");
	}

	SelectObject(destHDC, fOldBitmap);
}

void IGdiDevice::Win95RenderMaskedImage(
	const IGImage& image,
  const IGImage& maskImage,
	const IGRect2D& source,
	const IGrafState& grafState)
{
	fStateChange = grafState.registeredAsTheStateInUse();

	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	IGRect2D trect = source;
	IGRect2D irect = image.rasterBounds();
	
	const IGrafMatrix* tMatrix = grafState.modelMatrix();
	trect.orderPoints();
	if (!tMatrix->isIdentity())
		trect = tMatrix->transformBounds(trect);

	trect.intersectWith(worldBounds());
	if (trect.isEmpty())
		return;

    int left = GCoordinateToInt(trect.fLeft);
    int top = GCoordinateToInt(trect.fTop);
    int width = GCoordinateToInt(trect.width()) ;
    int height = GCoordinateToInt(trect.height()) ;

	IGRect2D theRect( trect.topLeft(), trect.bottomRight()+IGPoint2D(1,1) );	
	IPresSpaceHandle dc(fHDC);
	IGImage destImage(IGImage::captureImage(dc, theRect));

	IGImage maskdestImage(theRect.width(), theRect.height(), IGImage::k256Color8Bit);

	{ //scope for IGImagePixelAccessor
	IGImagePixelAccessor dest(destImage);
	IGImagePixelAccessor maskdest(maskdestImage);

	IGImage  original(image);
	original.sizeTo(IGPoint2D(irect.width()+1, irect.height()+1 ));
	IGImagePixelAccessor src(original);

	IGImage  orimask(maskImage);
	orimask.sizeTo(IGPoint2D(irect.width()+1, irect.height()+1 ));
	IGImagePixelAccessor mask(orimask);

	IBaseColor oldpixel, maskpixel;

	IGrafMatrix invMatrix = *tMatrix;
    invMatrix.invert();

	for (int i = 0; i < height; i++)
	{
		for (int j = 0; j < width; j++)
		{
			IGPoint2D ipoint(trect.fLeft + j, trect.fTop + i);
			IGPoint2D newPoint = invMatrix.transformPoint(ipoint);
			if (irect.contains(newPoint))
			{
				IGPoint2D transformedPoint = newPoint - irect.topLeft();
				oldpixel =
					Win95ImageSamplingControl(
						src,
						dest,
						transformedPoint,
						j,
						i,
						grafState);

				maskpixel =
					Win95ImageSamplingControl(
					mask,
					maskdest,
					transformedPoint,
					j,
					i,
					grafState);
				dest.setPixel(IGPoint2D(j, i), oldpixel);
				maskdest.setPixel(IGPoint2D(j, i), maskpixel);
			}
			else
                 maskdest.setPixel(IGPoint2D(j, i), IBaseColor::kWhite);
        }
    }
	}//scope for IGImagePixelAccessor

	int rasterOperation = imageTransferMode(grafState);
	
	HBITMAP destHandle = destImage.handle();
	IGImageDeviceContext theDC(destImage);
	HDC destHDC = theDC;
   HBITMAP fOldBitmap = (HBITMAP) SelectObject( destHDC, destHandle );
		
	
	HBITMAP maskdestHandle = maskdestImage.handle();
		
	if (!MaskBlt(fHDC, left, top, width, height,  destHDC, 0, 0, maskdestHandle, 0, 0, MAKEROP4(0x00AA0029, SRCCOPY)))
	{
		if (!BitBlt(fHDC, left, top, width, height,  destHDC, 0, 0, SRCINVERT))
			GrafDeviceException("BitBlt");
			
		SelectObject(destHDC, maskdestHandle);
			
		if (!BitBlt(fHDC, left, top, width, height, destHDC, 0, 0, SRCAND))
			GrafDeviceException("BitBlt");
			
		SelectObject(destHDC, destHandle);
			
		if (!BitBlt(fHDC, left, top, width, height, destHDC, 0, 0, SRCINVERT))
			GrafDeviceException("BitBlt");
	}
	SelectObject(destHDC, fOldBitmap);
}


IRGBAColorArray getSystemColors(unsigned long max)
{
        unsigned long i, num=0;
	IRGBAColorArray rgbaClrArr(max);

        //ColorPalette Init
        HDC sys = GetDC(0);

        if((GetDeviceCaps(sys, RASTERCAPS) & RC_PALETTE)){
                num = GetDeviceCaps(sys, SIZEPALETTE);
                num = (num < max) ? num : max;
                PALETTEENTRY *pal = new PALETTEENTRY[num];
                GetSystemPaletteEntries(sys, 0, num, pal);

                //fill the RGBA Color Array
                for(i=0; i<num;i++)
                {
                        IBaseColor bClr(pal[i].peRed, pal[i].peGreen, pal[i].peBlue, 0xff);
                        rgbaClrArr.setColor(i, bClr);
                }
                delete [] pal;

		//Fill the rest with 0
		for(i=num;i < max; i++)
		{
			IBaseColor bClr(0, 0, 0, 0xff);
			rgbaClrArr.setColor(i, bClr);
		}
	}

        ReleaseDC(0, sys);
	return(rgbaClrArr);

}



//================================================================
//	renderImage
//	
//	image : the pixels.
//	source : a rectangle in pixel space.
//	grafState : has the model matrix and attributes.
//	
//	render the pixels contained in the source rectangle to
//	the device. Transform the position of the pixels using the model matrix.
//	Use the transfer mode and image filtering mode defined in the graf state.
//	
//	The grafState contains the matrix that maps the image pixel space
//	to World Space. This matrix is set as the GDI WorldTransform to we can use
//	BitBlt with no transformation. If the World Transform is not supported, then
//	we must extract the components of the transformation from grafState, and
//	use them to set the parameters of StretchBlt.
//

void IGdiDevice::renderImage(
	const IGImage& image,
	const IGRect2D& source,
	const IGrafState& grafState)
{

// if (fPlatform == kWin95 && !(grafState.modelMatrix()->isRectilinear())){
 if (IPlatform::isWin9x() && !(grafState.modelMatrix()->isRectilinear())){
      Win95RenderImage(image, source, grafState);
      return;
  }

	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	int rasterOperation = imageTransferMode(grafState);
	
	// draw the image on the presentation space
	HBITMAP srcImage = image.handle();
	IGImageDeviceContext theDC(image);
	HDC srcHDC = theDC;

	IGRect2D  rect(source);
	const IGrafMatrix* coordFlip = grafState.root()->coordinateFlip();
	const IGrafMatrix* worldXForm = fModelMatrix;
	IGrafMatrix fImageMatrixUsed;

	//
	// Prior to the fix of defect #22234, the images were being
	// drawn in the correct location, but its orientation was
	// upside down in cases, where the default native platform coordinate
	// system origin was different from the one the application was using.
	//
	// The fix was to keep the location the same, but to just flip
	// the image.
	//
	// Doing this is somewhat convoluted. The SetWinWorldTransform
	// affects the entire device context. And it is the last one
	// that is applied. Nothing can be done about it. So if we see the
	// image is being drawn upside down, then we need to flip the image
	// one more time so that when the device context flips it, it will
	// appear right side up.
	//
	// The srcX, srcY values should not change. They refer to the image
	// device context which is different from the grafPort device context.
	//
	
	IGPoint2D pts[4];
	pts[2] = rect.topLeft();	// src rect
	pts[3] = rect.bottomRight();

	if (coordFlip) {
		fImageMatrixUsed = *coordFlip;
		rect = fImageMatrixUsed.transformBounds(rect);
		// need some adjust for coordinate flipping since image is exclusive.
    rect.offset(IGPoint2D(0.0, 1.0));
		
		fImageMatrixUsed.invert();
		fImageMatrixUsed.concatWith(*fModelMatrix);
		worldXForm = &fImageMatrixUsed;
	}
	
	pts[0] = rect.topLeft();	// dst rect
	pts[1] = rect.bottomRight();

	SetWinWorldTransform(worldXForm);

//  if (fPlatform == kWin95 )
  if (IPlatform::isWin9x() )
	{
		if (!fModelMatrix->isIdentity())
			{
				fModelMatrix->transformPoints(pts, pts, 2);
			}
	}
	
	int dstX = GCoordinateToInt(pts[0].fX);
	int dstY = GCoordinateToInt(pts[0].fY);
	int dstW = GCoordinateToInt(pts[1].fX - pts[0].fX);
	int dstH = GCoordinateToInt(pts[1].fY - pts[0].fY);

	int srcX = GCoordinateToInt(pts[2].fX);
	int srcY = GCoordinateToInt(pts[2].fY);
	int srcW = GCoordinateToInt(pts[3].fX - pts[2].fX);
	int srcH = GCoordinateToInt(pts[3].fY - pts[2].fY);
	
	SetStretchBltMode(fHDC, imageSamplingControl(grafState));
   HBITMAP fOldBitmap = (HBITMAP) SelectObject( srcHDC, srcImage );
	
	if(!image.transparencyFlag())
	{
		if (rasterOperation != ICLUIHalfToneMode)
		{
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, rasterOperation))
				GrafDeviceException("StretchBlt");
		}
		else
		{	// If the raster-op is halftone, set the pattern to a checker-board ...
			setTextColour(IBaseColor::kWhite); // white foreground
			SetBkColor(fHDC, 0x00000000); // black background

			WORD checker[] = { 0x55,0xaa,0x55,0xaa,0x55,0xaa,0x55,0xaa };
			HBITMAP bitmap = CreateBitmap(8, 8, 1, 1, (const void*)checker);
			IBrushContext aBrush(fHDC, CreatePatternBrush(bitmap));
			
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, rasterOperation))
				GrafDeviceException("StretchBlt");		
		
			DeleteObject(bitmap);
		  }
	} else {
		HBITMAP srcMatte = image.maskImageHandle();
		if (dstW == srcW &&
			dstH == srcH &&
			!MaskBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcMatte, 0, 0, MAKEROP4(0x00AA0029, SRCCOPY)))
		{
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
				GrafDeviceException("StretchBlt");
			
			SelectObject(srcHDC, srcMatte);
			
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCAND))
				GrafDeviceException("StretchBlt");
			
			SelectObject(srcHDC, srcImage);
			
			if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
				GrafDeviceException("StretchBlt");
		}
	}

	SelectObject(srcHDC, fOldBitmap);

	if (coordFlip)
		SetWinWorldTransform(grafState.modelMatrix()); //this was added for defect 26169

}

//================================================================
//	IGdiDevice::renderMaskedImage
//

void IGdiDevice::renderMaskedImage(
	const IGImage& image,
	const IGImage& mask,
	const IGRect2D& source,
	const IGrafState& grafState)
{

// if (fPlatform == kWin95 && !(grafState.modelMatrix()->isRectilinear())){
 if (IPlatform::isWin9x() && !(grafState.modelMatrix()->isRectilinear())){
      Win95RenderMaskedImage(image,mask, source, grafState);
      return;
  }

	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);

	if (SetClipArea(grafState) == kClipAreaEmpty)
		return;
	
	//int rasterOperation = imageTransferMode(grafState);
	
	// draw the image on the presentation space
	HBITMAP srcImage = image.handle();
	HBITMAP srcMatte = mask.handle();
	IGImageDeviceContext theDC(image);
	HDC srcHDC = theDC;
	
	IGRect2D rect(source);
	const IGrafMatrix* coordFlip = grafState.root()->coordinateFlip();
	const IGrafMatrix* worldXForm = fModelMatrix;
	IGrafMatrix fImageMatrixUsed;

	//
	// Prior to the fix of defect #22234, the images were being
	// drawn in the correct location, but its orientation was
	// upside down in cases, where the default native platform coordinate
	// system origin was different from the one the application was using.
	//
	// The fix was to keep the location the same, but to just flip
	// the image.
	//
	// Doing this is somewhat convoluted. The SetWinWorldTransform
	// affects the entire device context. And it is the last one
	// that is applied. Nothing can be done about it. So if we see the
	// image is being drawn upside down, then we need to flip the image
	// one more time so that when the device context flips it, it will
	// appear right side up.
	//
	// The srcX, srcY values should not change. They refer to the image
	// device context which is different from the grafPort device context.
	//
	
	IGPoint2D pts[4];
	pts[2] = rect.topLeft();	// src rect
	pts[3] = rect.bottomRight();

	if (coordFlip) {
		fImageMatrixUsed = *coordFlip;
		rect = fImageMatrixUsed.transformBounds(rect);

		fImageMatrixUsed.invert();
		fImageMatrixUsed.concatWith(*fModelMatrix);
		worldXForm = &fImageMatrixUsed;
	}
	
	pts[0] = rect.topLeft();	// dst rect
	pts[1] = rect.bottomRight();

	SetWinWorldTransform(worldXForm);
	
//	 if (fPlatform == kWin95 )
	 if (IPlatform::isWin9x() )
	{
		if (!fModelMatrix->isIdentity())
			{
				fModelMatrix->transformPoints(pts, pts, 2);
			}
	}
	
	SetStretchBltMode(fHDC, imageSamplingControl(grafState));
	
	int dstX = GCoordinateToInt(pts[0].fX);
	int dstY = GCoordinateToInt(pts[0].fY);
	int dstW = GCoordinateToInt(pts[1].fX - pts[0].fX);
	int dstH = GCoordinateToInt(pts[1].fY - pts[0].fY);

	int srcX = GCoordinateToInt(pts[2].fX);
	int srcY = GCoordinateToInt(pts[2].fY);
	int srcW = GCoordinateToInt(pts[3].fX - pts[2].fX);
	int srcH = GCoordinateToInt(pts[3].fY - pts[2].fY);
	
   HBITMAP fOldBitmap = (HBITMAP) SelectObject( srcHDC, srcImage );
	
	if (dstW == srcW &&
		dstH == srcH &&
		!MaskBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcMatte, 0, 0, MAKEROP4(0x00AA0029, SRCCOPY)))
	{
		if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
			GrafDeviceException("StretchBlt");
		
		SelectObject(srcHDC, srcMatte);
		
		if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCAND))
			GrafDeviceException("StretchBlt");
		
		SelectObject(srcHDC, srcImage);
		
		if (!StretchBlt(fHDC, dstX, dstY, dstW, dstH, srcHDC, srcX, srcY, srcW, srcH, SRCINVERT))
			GrafDeviceException("StretchBlt");
	}

	SelectObject(srcHDC, fOldBitmap);

	if (coordFlip)
		SetWinWorldTransform(grafState.modelMatrix()); //this was added for defect 26169
}


//================================================================
//	helper class for font and text support...
//----------------------------------------------------------------

IGdiFontContext::IGdiFontContext(
	IGdiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* modelMatrix,
	const IGrafMatrix* coordFlip,
	const IGrafState* state) :

	fFont(0 /*NIL*/),
	fDevice(theDevice),
	fTmCached(false),
	fAnchorTransform(0 /*NIL*/),

	fDeviceContext(theDevice->fHDC),
	fPointSize(theText.fPointSize),
	fEscapement(theText.fEscapement),
	fWeight(theText.fWeight),
	fItalic(theText.fItalic),
	fUnderline(theText.fUnderline),
	fStrikeOut(theText.fStrikeOut),
	fTypeFace(theText.fTypeFace),
	fMatrix(modelMatrix),
	fMatrixStamp(modelMatrix ? modelMatrix->timeStamp() : 0),
	fFlip(coordFlip),
	fFlipStamp(coordFlip ? coordFlip->timeStamp() : 0),
	fGrafState(state)
{
	if (!fFlip) {
		if (fMatrix)
			fMatrixUsed = *fMatrix;
		else
			fMatrixUsed.setToIdentity();
		fDevice->SetWinWorldTransform(fMatrix);
	}	
	else
	{
		// the anchor point should be transformed as flipped, the rest of the grometry is not...
		fAnchorTransform = fFlip;
		fMatrixUsed = *fFlip;
		fMatrixUsed.invert();
		fMatrixUsed.concatWith(*fMatrix);	// undo the coordinate flip from the model matrix...
		fDevice->SetWinWorldTransform(&fMatrixUsed);
	}
}

void IGdiFontContext::initialise()
{
	makeFont();

	if (SetTextAlign(fDeviceContext, TA_BASELINE|TA_LEFT|TA_NOUPDATECP) == GDI_ERROR)
		GrafDeviceException("SetTextAlign");
}

 void IGdiFontContext::makeFont()
{
  int fontHeight;
	
	if ( fDevice->baseRootDevice())
	  fontHeight = -fPointSize;
  else
		fontHeight = -MulDiv(fPointSize, GetDeviceCaps(fDevice->fHDC, LOGPIXELSY), 72) ;

	DWORD   charSet = DEFAULT_CHARSET;
	LOGFONT sysLogfont;
   HFONT   systemFont( (HFONT) GetStockObject( SYSTEM_FONT ) );
	if ( GetObject( systemFont, sizeof( sysLogfont ), &sysLogfont ) ) {
	   if ( sysLogfont.lfCharSet ) {
	       charSet = sysLogfont.lfCharSet;
	   }
	}

	fFont = CreateFont(
			fontHeight,
			0,	// FontWidth,
			fEscapement,
			0,	// Orientation,
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			charSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			fTypeFace);

	if (fFont)
      fFontOld = (HFONT) SelectObject( fDevice->fHDC, fFont );

#ifdef IC_DEVELOP
  unsigned long fontInfo = GetFontLanguageInfo( fDevice->fHDC );
  ITRACE_DEVELOP( ">>> GetFontLanguageInfo=" + IString( fontInfo ).d2x() );
#endif
		
}

IGdiFontContext::~IGdiFontContext()
{
	if (fFont) {
		SelectObject(fDevice->fHDC, fFontOld);
		DeleteObject(fFont);
	}
}

void IGdiFontContext::renderGlyphs(const IGTextRun& textRun) const
{
	IGPoint2D anchorP = textRun.anchor();

	if (fAnchorTransform)	// transform the anchor if need to ... for rendering
		anchorP = fAnchorTransform->transformPoint(anchorP);

	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(true);

   // Check if the window is right-to-left, in which case any
   // drawing of strings should be right-to-left.
   bool rtl = false;
   if ( IGBidiSettings::isBidiSupported() )
   {
      HWND window = WindowFromDC( fDevice->fHDC );
      unsigned long
        extStyle = GetWindowLong( window, GWL_EXSTYLE );
      if ( extStyle & WS_EX_RTLREADING )
      {
         rtl = true;
      }
   }
   if ( rtl )
   {
      if (!ExtTextOutW(
            fDevice->fHDC,
            IGdiDevice::GCoordinateToInt(anchorP.fX),
            IGdiDevice::GCoordinateToInt(anchorP.fY),
            ETO_RTLREADING, // wOptions
            0 /*NIL*/,      // lpRect
            (LPCWSTR)textRun.fUnicodeString, // this is unicode storage
            textRun.fUnicodeStringLength,
            NULL ))
           	
         GrafDeviceException("renderGlyphs ExtTextOutW");
   }
   else
   {
      // for unicode, TextOutW is faster ExtTextOut at least from the numbers I got
      if (!TextOutW(
            fDevice->fHDC,
            IGdiDevice::GCoordinateToInt(anchorP.fX),
            IGdiDevice::GCoordinateToInt(anchorP.fY),
            (LPCWSTR)textRun.fUnicodeString,	// this is unicode storage
            textRun.fUnicodeStringLength))

         GrafDeviceException("renderGlyphs TextOutW");
   }
}

void IGdiFontContext::extentPoint(
	const IGTextRun& textRun,
	SIZE& theSize) const
{
	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(true);

	if (!GetTextExtentPoint32W(
			fDevice->fHDC,
			(LPCWSTR)textRun.fUnicodeString,	// this is unicode storage
			textRun.fUnicodeStringLength,
			&theSize))
		GrafDeviceException("GetTextExtentPoint32");
}

IGPoint2D IGdiFontContext::calculateBoundsAndAdvancement(
	const IGTextRun& textRun,
	IGRect2D& bounds)
{
	IGPoint2D leftTop, rightBottom;
	IGPoint2D anchor = textRun.anchor();

	if (fAnchorTransform)	// transform the anchor if need to ... for rendering
		anchor = fAnchorTransform->transformPoint(anchor);

	SIZE theSize;
	extentPoint(textRun, theSize);

	if (!fTmCached) {
		if (!GetTextMetrics(fDevice->fHDC, &fTm))
			GrafDeviceException("GetTextMetrics");
		fTmCached = true;
	}

	// It's better not to do this, specially at small pointsizes...	
	//	fBBoxSize.cx -= fTm.tmOverhang;

	IGPoint2D advance = leftTop = IGPoint2D::origin();
	switch (textRun.attachment())
	{
	default:
	case IGTextRun::kLeftToRight:
		advance.fX += theSize.cx;
		leftTop.fY -= fTm.tmAscent;
		break;
	case IGTextRun::kRightToLeft:
		advance.fX -= theSize.cx;
		leftTop.fX -= theSize.cx;
		leftTop.fY -= fTm.tmAscent;
		break;
	case IGTextRun::kTopToBottom:
		advance.fY += theSize.cy;
		leftTop.fX -= theSize.cx >> 1;
		break;
	case IGTextRun::kBottomToTop:
		advance.fY -= theSize.cy;
		leftTop.fX -= theSize.cx >> 1;
		leftTop.fY -= theSize.cy;
		break;
	}

#ifdef GRAPH2D_DEBUG
// printf("_ (%.2f, %.2f), extent:%d, %d (%.2f, %.2f)\n", anchor.fX, anchor.fY, theSize.cx, theSize.cy, leftTop.fX, leftTop.fY);	
#endif
	leftTop += anchor;
	rightBottom = leftTop + IGPoint2D(theSize.cx, theSize.cy);
	anchor += advance;

	if (!fAnchorTransform)	// transform the anchor if need to ... for rendering
	{
		bounds = IGRect2D(leftTop, rightBottom);
		return anchor;
	}
	else
	{
		bounds = fAnchorTransform->untransformBounds(IGRect2D(leftTop, rightBottom));
		return fAnchorTransform->untransformPoint(anchor);
	}
}	

inline bool IGdiFontContext::cacheHit(
	IGdiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* model,
	const IGrafMatrix* flip) const
{
	return (
		fDeviceContext == theDevice->fHDC &&
		fMatrix == model &&
		fMatrixStamp == (model ? model->timeStamp() : 0) &&
		fFlip == flip &&
		fFlipStamp == (flip ? flip->timeStamp() : 0) &&
		fPointSize == theText.fPointSize &&
		fEscapement == theText.fEscapement &&
		fWeight == theText.fWeight &&
		fItalic == theText.fItalic &&
		fUnderline == theText.fUnderline &&
		fStrikeOut == theText.fStrikeOut &&
		fTypeFace == theText.fTypeFace);
}

void IGdiDevice::setTextColour(const IBaseColor& target) {
	if (fTextColourCached) {
		if (fTextColour == target) {
			return;
		}
	}
	else
		fTextColourCached = true;

	fTextColour = target;
	if (SetTextColor(fHDC, convertToWRGB(fTextColour)) == CLR_INVALID)
		GrafDeviceException("SetTextColor");
}

//----------------------------------------------------------------

IGdiNonUnicodeFontContext::IGdiNonUnicodeFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) :
		IGdiFontContext(theDevice, theText, model, flip, state)
{
}		

IGdiNonUnicodeFontContext::~IGdiNonUnicodeFontContext() {}

void IGdiNonUnicodeFontContext::renderGlyphs(const IGTextRun& textRun) const
{
	IGPoint2D anchorP = textRun.anchor();

	if (fAnchorTransform)	// transform the anchor if need to ... for rendering
		anchorP = fAnchorTransform->transformPoint(anchorP);

	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(false);

   // Check if the window is right-to-left, in which case any
   // drawing of strings should be right-to-left.
   bool rtl = false;
   if ( IGBidiSettings::isBidiSupported() )
   {
      HWND window = WindowFromDC( fDevice->fHDC );
      unsigned long
        extStyle = GetWindowLong( window, GWL_EXSTYLE );
      if ( extStyle & WS_EX_RTLREADING )
      {
         rtl = true;
      }
   }
	// using ExTextOut is faster than TextOut according to MSDN bulletin...
   if (!ExtTextOutA(
         fDevice->fHDC,
         IGdiDevice::GCoordinateToInt(anchorP.fX),
         IGdiDevice::GCoordinateToInt(anchorP.fY),
         rtl ? ETO_RTLREADING : 0,  // wOptions
         0 /*NIL*/,         // lpRect
         textRun.fSubText,  // this transcodes to host character set
         textRun.fSubText.length(),
         0 /*NIL*/))

      GrafDeviceException("GDINonUnicodeFontContext ExtTextOut");
}

void IGdiNonUnicodeFontContext::extentPoint(
	const IGTextRun& run,
	SIZE& theSize) const
{
	if (!run.fSubTextCached)
		((IGTextRun*)&run)->cacheRenderSubtext(false);

	if (!GetTextExtentPoint32(fDevice->fHDC, run.fSubText, run.fSubText.length(), &theSize))
		GrafDeviceException("GDINonUnicodeFontContext GetTextExtentPoint32");
}

//======================================================
// helper class to extract outline for text rendering
// GlyphBuilder is from Rahul's glyph3d.hpp
//======================================================

#define MAX_PTS_IN_GLYPH 1024

#pragma pack(push,4)
class GlyphBuilder
{
public:
	GlyphBuilder();
	~GlyphBuilder();

	void startAfresh();
	void startNewContour(const GCoordinate x, const GCoordinate y);
	void addLineTo(const GCoordinate endPtX, const GCoordinate endPtY);
	void addCurve(const GCoordinate x2, const GCoordinate y2, const GCoordinate x3, const GCoordinate y3);
	IGLoop2D glyph() const;


private:
	void addKnotsForThisSegment();

	IRawArray<GParametric> fKnots;
	IGRPoint2DArray fCurvePtArray;
	short fPtCount;
	short fKVCount;
	short fKnotValue;
	bool fContourJustStarted;
}; // GlyphBuilder
#pragma pack(pop)


//----------------------------------------------------------
//	Subclasses of IGdiNonUnicodeFontContext
//	Class name:
//		IGdiWin95FontContext
//		IGdiWin95RotateFontContext
//		IGdiWin95OutlineFontContext
//		IGdiWin95ScaleFontContext
//----------------------------------------------------------

#pragma pack(push,4)
class IGdiWin95FontContext : public IGdiNonUnicodeFontContext {
public:
	IGdiWin95FontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) ;
	virtual ~IGdiWin95FontContext() ;
	virtual void renderGlyphs(const IGTextRun&) const;

protected:
	const IGrafMatrix* fModelTransform;		// for win95 model matrix only...
  virtual void makeFont();

};

IGdiWin95FontContext::IGdiWin95FontContext(
	IGdiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* model,
	const IGrafMatrix* flip ,
	const IGrafState* state) :
	IGdiNonUnicodeFontContext(theDevice, theText, model, flip, state),
	fModelTransform(model)
{}

IGdiWin95FontContext::~IGdiWin95FontContext() {}

// Rotate transformation
class IGdiWin95RotateFontContext : public IGdiWin95FontContext {
public:
	IGdiWin95RotateFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) ;
	virtual ~IGdiWin95RotateFontContext() ;

protected:
	virtual void makeFont();
};

IGdiWin95RotateFontContext::IGdiWin95RotateFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip ,
		const IGrafState* state) :
		IGdiWin95FontContext(theDevice, theText, model, flip, state) {}

IGdiWin95RotateFontContext::~IGdiWin95RotateFontContext() {}

//for positive scaling
class IGdiWin95ScaleFontContext : public IGdiWin95FontContext {
public:
	IGdiWin95ScaleFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip,
		const IGrafState* state);
	virtual ~IGdiWin95ScaleFontContext() ;
	virtual IGPoint2D calculateBoundsAndAdvancement(const IGTextRun&, IGRect2D& bounds);

protected:
	virtual void makeFont();
	virtual void renderGlyphs(const IGTextRun&) const;

private:
	bool fNegativeXScale;
	bool fNegativeYScale;
	bool fFliped;
};

IGdiWin95ScaleFontContext::~IGdiWin95ScaleFontContext() {}

// extract and render the outline
class IGdiWin95OutlineFontContext : public IGdiWin95FontContext {
public:
	IGdiWin95OutlineFontContext(
		IGdiDevice* theDevice,
		const IGTextRun& theText,
		const IGrafMatrix* model,
		const IGrafMatrix* flip,
		const IGrafState* state) ;
	virtual ~IGdiWin95OutlineFontContext() ;

protected:
	virtual void makeFont();
	virtual void renderGlyphs(const IGTextRun&) const;	
};

IGdiWin95OutlineFontContext::IGdiWin95OutlineFontContext(
	IGdiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* model,
	const IGrafMatrix* flip,
	const IGrafState* state) :
IGdiWin95FontContext(theDevice, theText, model, flip, state) {}

IGdiWin95OutlineFontContext:: ~IGdiWin95OutlineFontContext() {}
#pragma pack(pop)

//-----------------------------------------------------------


// create different types of font context according to the types of transformation matrices.
IGdiFontContext& IGdiDevice::makeFontContext(
	const IGTextRun& theText,
	const IGrafMatrix* modelMatrix,
	const IGrafState* state,
	bool unicodeSupported)
{
	if (!theText.fStylesCached)
		((IGTextRun*)&theText)->cacheRenderStyles();

	const IGrafMatrix* coordFlip = state->root()->coordinateFlip();

	if (fFontCache) {
		if (fFontCache->cacheHit(this, theText, modelMatrix, coordFlip)) {
			SetWinWorldTransform(&fFontCache->fMatrixUsed);
			if (fFontCache->fGrafState != state) // this may be a problem, we may need to do a contents comparison
				fFontCache->fGrafState = state;
			return *fFontCache;
		}
		delete fFontCache;
	}
	
	if (unicodeSupported)
		fFontCache = new IGdiFontContext(this, theText, modelMatrix, coordFlip, state);
//	else if (fPlatform != IGdiDevice::kWin95)
	else if (!(IPlatform::isWin9x()))
		fFontCache = new IGdiNonUnicodeFontContext(this, theText, modelMatrix, coordFlip, state);
	else {
		IGrafMatrix::EMatrixType matType = modelMatrix->type();
		switch (matType) {
			case IGrafMatrix::kIdentity:
			case IGrafMatrix::kTranslate:
				fFontCache = new IGdiWin95FontContext(this, theText, modelMatrix, coordFlip, state);
				break;
			case IGrafMatrix::kRotate:
				fFontCache = new IGdiWin95RotateFontContext(this, theText, modelMatrix, coordFlip, state);
				break;
			case IGrafMatrix::kScale:
				fFontCache = new IGdiWin95ScaleFontContext(this, theText, modelMatrix, coordFlip, state);
				break;
			default:
				fFontCache = new IGdiWin95OutlineFontContext(this, theText, modelMatrix , coordFlip, state);
		}	
	}
	fFontCache->initialise();
	return *fFontCache;
}

//================================================================
//	IGdiDevice::renderTextRun
//

IGPoint2D IGdiDevice::renderTextRun(
	const IGTextRun& textRun,
	const IGrafState& grafState)
{
/*
	static unsigned long kAnyPaintModeChange  =
		IStateChange::kFillPaint|
		IStateChange::kFillTransferMode|
		IStateChange::kFramePaint|
		IStateChange::kFrameTransferMode;
*/

	const IText* text = textRun.text();
	if (!text)
		return textRun.anchor();

	fStateChange = grafState.registeredAsTheStateInUse();
	setupTransforms(grafState);
	
	if (SetClipArea(grafState) == kClipAreaEmpty)		// before transformation
		return textRun.anchor();
	
	// get all attributes for creating a font from the style set
	IGdiFontContext& fontContext =
		makeFontContext(
			textRun,
			fModelMatrix,
			&grafState,
//			fPlatform == IGdiDevice::kWinNT);
         IPlatform::isWinNT());

	SetFillPaintModes(grafState);

	const IBaseColor* theColor = textRun.fTextColor;
	if (!theColor)
		theColor = grafState.attributeState()->fillPaint()->color();

	setTextColour(*theColor);

	fontContext.renderGlyphs(textRun);

	IGPoint2D nextLocation;
	if (textRun.hadMetrics(fHDC, fontContext.fMatrixStamp, fontContext.fFlipStamp)) {
		nextLocation = textRun.nextAnchor();
	}
	else {
		IGRect2D bounds;
		nextLocation = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		((IGTextRun*)&textRun)->updateMetrics(
			bounds,
			nextLocation,
			fHDC,
			fontContext.fMatrixStamp,
			fontContext.fFlipStamp);
	}

#ifdef DEBUG_BBOX
IGRect2D bounds = textRun.bounds();
IGPoint2D anc = textRun.anchor();
printf(" anchor(%.1f, %.1f), bounds(%.1f, %.1f)(%.1f, %.1f), next(%.1f, %.1f)\n",
 anc.fX, anc.fY, bounds.fLeft, bounds.fTop, bounds.width(), bounds.height(),
 nextLocation.fX, nextLocation.fY);
IFrameBundle redFrame(IBaseColor::kRed);
ILinkedAttributeGrafState tempState(&grafState, &redFrame);
renderRect(bounds, tempState);
IFrameBundle bluFrame(IBaseColor::kBlue);
ILinkedAttributeGrafState temp1State(&grafState, &bluFrame);
renderLine(IGLine2D(anc, nextLocation), temp1State);
IGPoint2D se(1.0, 1.0);
renderRect(IGRect2D(anc - se, anc + se), temp1State);
IFrameBundle grnFrame(IBaseColor::kGreen);
ILinkedAttributeGrafState temp2State(&grafState, &grnFrame);
renderRect(IGRect2D(nextLocation - se, nextLocation + se), temp2State);
#endif

	return nextLocation;
}

// static function to persuit rendering text metrics...
// but it does not work for righthand, caller has to manage this

void IGdiDevice::determineTextMetrics(
	IGTextRun& textRun,
	IGdiDevice* dev,
	const IGrafState* grafState)
{

  IPresSpaceHandle deviceContext(0);
	bool baseRootFlag = true;
	
	if(dev){
	 deviceContext = dev->deviceContext();
	 baseRootFlag = dev->baseRootDevice();
	}
	
	
	const IGrafMatrix* xform = 0 /*NIL*/;
	const IGrafMatrix* flip = 0 /*NIL*/;
	unsigned long matStamp = 0;
	unsigned long flipStamp = 0;

	if (grafState) {
		xform = grafState->modelMatrix();
		flip = grafState->root()->coordinateFlip();
		if (xform)
			matStamp = xform->timeStamp();
		if (flip)
			flipStamp = flip->timeStamp();

		// we should eventually take into account of the bundle's time stamp...
	}

	if (!textRun.hadMetrics(deviceContext, matStamp, flipStamp))
	{
	  // a flag should be passed in to determine whether
		// the device is an baseRootDevice or not.
		IGdiDevice device(deviceContext, baseRootFlag );

		if (!textRun.fStylesCached)
			textRun.cacheRenderStyles();

		IGRect2D bounds;
		IGPoint2D next;
		
//		if (device.fPlatform == IGdiDevice::kWinNT) {
		if (IPlatform::isWinNT()) {
			IGdiFontContext fontContext(&device, textRun, xform, flip, grafState);
			fontContext.initialise();
			next = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		}
//		else if (fPlatform != IGdiDevice::kWin95){
		else if (!(IPlatform::isWin9x())){
			IGdiNonUnicodeFontContext fontContext(&device, textRun, xform, flip, grafState);
			fontContext.initialise();
			next = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		}
		else { //XL : ?? needs to rewrite later
			IGdiWin95FontContext fontContext(&device, textRun, xform, flip, grafState);
			fontContext.initialise();
			next = fontContext.calculateBoundsAndAdvancement(textRun, bounds);
		}
		textRun.updateMetrics(bounds, next, deviceContext, matStamp, flipStamp);
	}
	// else already determined...
}


//---------------------------------------------------------
//	Class name:
//		IGdiWin95FontContext
//		IGdiWin95RotateFontContext
//		IGdiWin95OutlineFontContext
//		IGdiWin95ScaleFontContext
//----------------------------------------------------------


void IGdiWin95FontContext::makeFont()
{

int fontHeight;
	
	if ( fDevice->baseRootDevice())
	  fontHeight = -fPointSize;
  else
		fontHeight = -MulDiv(fPointSize, GetDeviceCaps(fDevice->fHDC, LOGPIXELSY), 72) ;

	fFont =
		CreateFont(
			fontHeight,
			0,	// FontWidth,
			0, //fEscapement,
			0,	// Orientation,
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			fTypeFace);

	if (fFont)
      fFontOld = (HFONT) SelectObject( fDevice->fHDC, fFont );
}

void IGdiWin95FontContext::renderGlyphs(const IGTextRun& textRun) const
{
	IGPoint2D anchorP = textRun.anchor();

	if (fModelTransform)	// transform the anchor if need to ... for rendering
		anchorP = fModelTransform->transformPoint(anchorP);

	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(false);

  if (textRun.attachment() == IGTextRun::kLeftToRight )
  {
  	// using ExTextOut is faster than TextOut according to MSDN bulletin...
  	if (!ExtTextOut(
  		fDeviceContext,
  		IGdiDevice::GCoordinateToInt(anchorP.fX),
  		IGdiDevice::GCoordinateToInt(anchorP.fY),
  		0,			// wOptions
  		0 /*NIL*/,	// lpRect
  		textRun.fSubText,	//this transcodes to host character set
  		textRun.fSubText.length(),
  		0 /*NIL*/))

  		GrafDeviceException("ExtTextOut");
  }
  else
  {
	 int length = textRun.fSubText.length();
   IText myIText(textRun.fSubText);
	 const char* str =(const char*)myIText;

	 TEXTMETRIC tm;
   GetTextMetrics(fDeviceContext, &tm);

   IGPoint2D  nextCharAnchor = anchorP;

   SIZE theSize ;
   for (int j = 0; j < length; j++) {
      // get the size of the character that is going to display
      if (! GetTextExtentPoint32(fDevice->fHDC,myIText.substr(j,1), 1, &theSize))
  		  GrafDeviceException("GetTextExtentPoint32");

			switch (textRun.attachment())
			{
				case IGTextRun::kRightToLeft:
					nextCharAnchor.fX  -= theSize.cx + 1;
          break;
				case IGTextRun::kTopToBottom:
          nextCharAnchor.fX = (int)(anchorP.fX - (ceil(float(theSize.cx/2.))) +1);
					nextCharAnchor.fY += tm.tmAscent;
					break;
				case IGTextRun::kBottomToTop:
          nextCharAnchor.fX = (int)(anchorP.fX - (ceil(float(theSize.cx/2.))) +1);
					nextCharAnchor.fY -= tm.tmAscent;
					break;
			}
      ExtTextOut(
    		fDeviceContext,
    		IGdiDevice::GCoordinateToInt(nextCharAnchor.fX),
    		IGdiDevice::GCoordinateToInt(nextCharAnchor.fY),
    		0,			// wOptions
    		0 /*NIL*/,	// lpRect
        myIText.substr(j,1),
    		1,
    		0 /*NIL*/);
    } //end for
 }  //end else
}



//============================================================================
void IGdiWin95RotateFontContext::makeFont()
{
	HDC dc = fDevice->fHDC;
	
  int fontHeight;
	if ( fDevice->baseRootDevice())
	  fontHeight = -fPointSize;
  else
		fontHeight = -MulDiv(fPointSize, GetDeviceCaps(fDevice->fHDC, LOGPIXELSY), 72) ;

	int fontWidth = 0;

	GDegrees degrees;
	IGPoint2D centerOfRotate;
	if (fMatrixUsed.rotate(degrees, centerOfRotate))
		fEscapement = -int(degrees*10);
	else
		fEscapement = 0;

	int orientation = fEscapement;

	// for rotation, need to get a true type font by the following:
	//(1) fTypeface of textRun is not used when creating a tempFont
	//(2) let the fontHeight to be a large data.
	HFONT tempFont;
	HFONT tempFontOld;
	
	tempFont =
		CreateFont(
			100, //fontHeight,
			fontWidth,
			0,  //Escapement
			0,  //Orientation
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			0  // TypeFace
			);
			
   tempFontOld = (HFONT) SelectObject( dc, tempFont );

	// get facename of a true type font
	int trueTypeFaceLen = 100;
	char trueTypeFaceName[100];
	if (!GetTextFace(dc, trueTypeFaceLen, trueTypeFaceName)) {
		SelectObject(dc, tempFontOld);
		DeleteObject(tempFont);
		GrafDeviceException("GetTextFace");
	}
	SelectObject(dc, tempFontOld);
	DeleteObject(tempFont);
	
	//create the actual font
	fFont =
		CreateFont(
			fontHeight,
			fontWidth,
			fEscapement,
			orientation,
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			trueTypeFaceName
			);

#ifdef GRAPH2D_DEBUG
	cout << "trueTypeFaceName: "<< trueTypeFaceName<< endl;
#endif
   fFontOld = (HFONT) SelectObject( dc, fFont );
}	


//==============================================================================

IGdiWin95ScaleFontContext::IGdiWin95ScaleFontContext(
	IGdiDevice* theDevice,
	const IGTextRun& theText,
	const IGrafMatrix* model,
	const IGrafMatrix* flip,
	const IGrafState* state) :

	IGdiWin95FontContext(theDevice, theText, model, flip, state)
{
	GCoordinate xScale = fMatrixUsed.element(IGrafMatrix::kScaleX);
	GCoordinate yScale = fMatrixUsed.element(IGrafMatrix::kScaleY);
	fNegativeXScale = xScale < 0;
	fNegativeYScale = yScale < 0;
	// fMatrixUsed is modified to always have positive scales
	if (fNegativeXScale)
		fMatrixUsed.setElement(IGrafMatrix::kScaleX, -xScale);
	if (fNegativeYScale)
		fMatrixUsed.setElement(IGrafMatrix::kScaleY, -yScale);
	fFliped = (fNegativeXScale || fNegativeYScale);
}


void IGdiWin95ScaleFontContext::makeFont()
{
	HDC dc = fDevice->fHDC;
	
int fontHeight;
	if ( fDevice->baseRootDevice())
	  fontHeight = -fPointSize;
  else
		fontHeight = -MulDiv(fPointSize, GetDeviceCaps(fDevice->fHDC, LOGPIXELSY), 72) ;

  int scaleFontWidth ;
	int scaleFontHeight ;
	int fontWidth = 0;
	
	// orientation is always the same as escapement in Win95
	int orientation = fEscapement;
		
	GCoordinate xScale = fMatrixUsed.element(IGrafMatrix::kScaleX);
	GCoordinate yScale = fMatrixUsed.element(IGrafMatrix::kScaleY);

	if (fabs(xScale) != 1. || fabs(yScale) != 1.)
	{
		//create a temporary font to find out the average width of the font used in scaling.
		HFONT tempFont;
		HFONT tempFontOld;
		
		tempFont =
			CreateFont(
				fontHeight,
				fontWidth,
				fEscapement,
				orientation,
				fWeight,
				fItalic,
				fUnderline,
				fStrikeOut,
				0, // CharSet,
				0, // OutputPrecision,
				0, // ClipPrecision,
				0, // Quality,
				0, // PitchAndFamily,
				fTypeFace);
				
      tempFontOld = (HFONT) SelectObject( dc, tempFont );

		TEXTMETRIC temptm;
		if (!GetTextMetrics(dc, &temptm)){
			SelectObject(dc, tempFontOld);
		DeleteObject(tempFont);		
			GrafDeviceException("GetTextMetrics");
		}	

		SelectObject(dc, tempFontOld);
		DeleteObject(tempFont);
	
		//scale the font
		scaleFontWidth = IGdiDevice::GCoordinateToInt(xScale * (GCoordinate)temptm.tmAveCharWidth);
		scaleFontHeight = IGdiDevice::GCoordinateToInt(yScale * (GCoordinate)temptm.tmHeight);
	}

	//create the actual font
	fFont =
		CreateFont(
			scaleFontHeight,
			scaleFontWidth,
			fEscapement,
			orientation,
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			fTypeFace);
						
   fFontOld = (HFONT) SelectObject( dc, fFont );
}	



IGPoint2D IGdiWin95ScaleFontContext::calculateBoundsAndAdvancement(
	const IGTextRun& textRun,
	IGRect2D& bounds)
{
	IGPoint2D leftTop, rightBottom;
	IGPoint2D anchor = textRun.anchor();

	if (! fMatrixUsed.isIdentity())
		anchor = fMatrixUsed.transformPoint(anchor);

	SIZE theSize;
	extentPoint(textRun, theSize);

	if (!fTmCached) {
		if (!GetTextMetrics(fDevice->fHDC, &fTm))
			GrafDeviceException("GetTextMetrics");
		fTmCached = true;
	}

	// It's better not to do this, specially at small pointsizes...	
	//	fBBoxSize.cx -= fTm.tmOverhang;

	IGPoint2D advance = leftTop = IGPoint2D::origin();
	switch (textRun.attachment())
	{
	default:
	case IGTextRun::kLeftToRight:
		advance.fX += theSize.cx;
		leftTop.fY -= fTm.tmAscent;
		break;
	case IGTextRun::kRightToLeft:
		advance.fX -= theSize.cx;
		leftTop.fX -= theSize.cx;
		leftTop.fY -= fTm.tmAscent;
		break;
	case IGTextRun::kTopToBottom:
		advance.fY += theSize.cy;
		leftTop.fX -= theSize.cx >> 1;
		break;
	case IGTextRun::kBottomToTop:
		advance.fY -= theSize.cy;
		leftTop.fX -= theSize.cx >> 1;
		leftTop.fY -= theSize.cy;
		break;
	}

#ifdef GRAPH2D_DEBUG
// printf("_ (%.2f, %.2f), extent:%d, %d (%.2f, %.2f)\n", anchor.fX, anchor.fY, theSize.cx, theSize.cy, leftTop.fX, leftTop.fY);	
#endif
	leftTop += anchor;
	rightBottom = leftTop + IGPoint2D(theSize.cx, theSize.cy);
	anchor += advance;

	if (fMatrixUsed.isIdentity())
	{
		bounds = IGRect2D(leftTop, rightBottom);
		return anchor;
	}
	else
	{
		bounds = fMatrixUsed.untransformBounds(IGRect2D(leftTop, rightBottom));
		return fMatrixUsed.untransformPoint(anchor);
	}
}	


	
void IGdiWin95ScaleFontContext::renderGlyphs(const IGTextRun& textRun) const
{
	if (! fFliped)
	{
		IGdiWin95FontContext::renderGlyphs(textRun);
		return;
	}
	
	const IAttributeState* attState = fGrafState->attributeState();
	const IPaint* paint = attState->fillPaint();
	
	const IBaseColor* theColor = textRun.fTextColor;
	if (!theColor)
		theColor = paint->color();

	IGRect2D box(textRun.bounds());

	// textMatrix is used to decide the text anchor inside the text bounds
	IGrafMatrix textMatrix(fMatrixUsed);
	IGRect2D textBox = textMatrix.transformBounds(box);
	textMatrix.translateBy(-textBox.topLeft());

	IGImage textImage(
		(unsigned long)textBox.width(),
		(unsigned long)textBox.height(),
		IGImage::kTrueColor24Bit);	

	// clear the background of the memDC with color that is different from text color
#ifdef DONT_DO_MASKBLT
	// Use GetPixel and SetPixel to render the text - the slow the sure way!
	IBaseColor backColor(
		(CharIntensity)(theColor->redMix() + 128),
		(CharIntensity)(theColor->greenMix() + 128),
		(CharIntensity)(theColor->blueMix() + 128));
#else
	//use mask image to render the text

	// find the nearest the solid color here... add half and hope we land on a very different colour
	COLORREF solid =
		GetNearestColor(
			fDeviceContext,
			RGB(
				CharIntensity(theColor->redMix() + 128),
				CharIntensity(theColor->greenMix() + 128),
				CharIntensity(theColor->blueMix() + 128)));
	IBaseColor backColor(GetRValue(solid), GetGValue(solid), GetBValue(solid));
#endif
	textImage.grafPort()->draw(IGRect2D::infiniteRect(), IFillAndFrameBundle(backColor, backColor));

	// create an image GdiDevice
	IGImageDeviceContext theDC(textImage);
	HDC dc = theDC;
	IGdiDevice imageDevice(dc, textBox);
	
	SelectObject(dc, textImage.handle());	
   HFONT oldFont = (HFONT) SelectObject( dc, fFont );
	SetTextColor(dc, IGdiDevice::convertToWRGB(*theColor));
	
	if (SetTextAlign(dc, TA_BASELINE|TA_LEFT|TA_NOUPDATECP) == GDI_ERROR)
		GrafDeviceException("SetTextAlign");
	
	imageDevice.SetFramePaintModes(*fGrafState);

	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(false);
	
	IGPoint2D newAnchor = textRun.anchor();
	newAnchor = textMatrix.transformPoint(newAnchor);
	
	// using ExTextOut is faster than TextOut according to MSDN bulletin...
	if (!ExtTextOut(
		dc,
		IGdiDevice::GCoordinateToInt(newAnchor.fX),
		IGdiDevice::GCoordinateToInt(newAnchor.fY),
		0,					// wOptions
		0 /*NIL*/,			// lpRect
		textRun.fSubText,	//this transcodes to host character set
		textRun.fSubText.length(),
		0 /*NIL*/))
		GrafDeviceException("ExtTextOut");

	SelectObject(dc, oldFont);	

	// now the grand finale - flips!
	if (fNegativeXScale) {
		if (fNegativeYScale)
			textImage.rotateBy180();
		else
			textImage.reflectHorizontally();
	}
	else if (fNegativeYScale)
		textImage.reflectVertically();

	// posBox is used to decide the output position of the image on the screen
	IGRect2D posBox = fModelTransform->transformBounds(box);
	IGPoint2D off = posBox.topLeft(); // offset on the screen
	
	// draw the image on the presentation space
	int dstX = IGdiDevice::GCoordinateToInt(off.fX);
	int dstY = IGdiDevice::GCoordinateToInt(off.fY);
	int dstW = IGdiDevice::GCoordinateToInt(textBox.width());
	int dstH = IGdiDevice::GCoordinateToInt(textBox.height());

#ifdef DONT_DO_MASKBLT
	// Use GetPixel and SetPixel to render the text - the slow the sure way!
	unsigned long textColorIndex = RGB(theColor->redMix(), theColor->greenMix(), theColor->blueMix());
	unsigned long nearTextColorIndex = GetNearestColor(fDeviceContext, textColorIndex);

	// after image reflection or rotation, a new bitmap is created with a different handle
	SelectObject(dc, textImage.handle());

	// display only the pixel that has the same color of the text on the presentation space
	unsigned long lColor;
	for (int i = 0; i < dstW; i++) {
		for (int j = 0; j < dstH; j++) {
			lColor = GetPixel(dc, i, j);
			if (lColor == nearTextColorIndex)
				SetPixel(fDeviceContext, dstX + i, dstY + j, lColor);
		}
	}	
#else
	//use mask image to render the text
	IGImage mask(textImage, backColor);	// create a mask for textImage
	HDC mdc = mask.deviceContext();

	int srcX = 0;
	int srcY = 0;

	HBITMAP srcImage = textImage.handle();

	// maskBlt...
	SelectObject(mdc, srcImage);
	if (!BitBlt(fDeviceContext, dstX, dstY, dstW, dstH, mdc, srcX, srcY, SRCINVERT))
		GrafDeviceException("BitBlt");

	SelectObject(mdc, mask.handle());
	if (!BitBlt(fDeviceContext, dstX, dstY, dstW, dstH, mdc, srcX, srcY, SRCAND))
		GrafDeviceException("BitBlt");

	SelectObject(mdc, srcImage);
	if (!BitBlt(fDeviceContext, dstX, dstY, dstW, dstH, mdc, srcX, srcY, SRCINVERT))
		GrafDeviceException("BitBlt");

#endif
}	

//=============================================================

void IGdiWin95OutlineFontContext::makeFont()
{
	HDC dc = fDevice->fHDC;
	
  int fontHeight;
 	if ( fDevice->baseRootDevice())
	  fontHeight = -fPointSize;
  else
		fontHeight = -MulDiv(fPointSize, GetDeviceCaps(fDevice->fHDC, LOGPIXELSY), 72) ;

	int fontWidth = 0;
	
	// orientation and fEscapement are both zero for outline rendering
	int orientation = 0;
	fEscapement = orientation;

	// for rotation, need to get a true type font by the following:
	//(1) fTypeface of textRun is not used when creating a tempFont
	//(2) let the fontHeight to be a large data.
	HFONT tempFont;
	HFONT tempFontOld;
	
	tempFont =
		CreateFont(
			100, //fontHeight,
			fontWidth,
			0,  //Escapement
			0,  //Orientation
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			0  // TypeFace
			);
			
  	tempFontOld = (HFONT) SelectObject( dc, tempFont );

	// get facename of a true type font
	const int trueTypeFaceLen = 100;
	char trueTypeFaceName[trueTypeFaceLen];
	if (!GetTextFace(dc, trueTypeFaceLen, trueTypeFaceName)) {
		SelectObject(dc, tempFontOld);
		DeleteObject(tempFont);
		GrafDeviceException("GetTextFace");
	}
	
	SelectObject(dc, tempFontOld);
	DeleteObject(tempFont);

	//create the actual font
	fFont =
		CreateFont(
			fontHeight,
			fontWidth,
			fEscapement,
			orientation,
			fWeight,
			fItalic,
			fUnderline,
			fStrikeOut,
			0, // CharSet,
			0, // OutputPrecision,
			0, // ClipPrecision,
			0, // Quality,
			0, // PitchAndFamily,
			trueTypeFaceName
			);
  	fFontOld = (HFONT) SelectObject( dc, fFont );
}	
	
IGLoop2D* BufIntoGlyph(unsigned long size, char *chBuf);


// Later, an outline extracting function needs to be implemented IGTextRun.
void IGdiWin95OutlineFontContext::renderGlyphs(const IGTextRun& textRun) const
{
	if (!textRun.fSubTextCached)
		((IGTextRun*)&textRun)->cacheRenderSubtext(false);

	HDC hdc = fDevice->fHDC;

	const IAttributeState* attState = fGrafState->attributeState();
	const IPaint* paint = attState->fillPaint();
	const IBaseColor* theColor = textRun.fTextColor;
	if (!theColor)
		theColor = paint->color();
				
	IFillAndFrameBundle bundle(*theColor, *theColor);
	ILinkedAttributeGrafState tempState(fGrafState, &bundle);

  // specified  that the width of a space character since the outline can
  // not be extracted for a space.
  TEXTMETRIC temptm;
	if (!GetTextMetrics(hdc, &temptm))
			GrafDeviceException("GetTextMetrics");

  GCoordinate  spaceWidth = (GCoordinate)temptm.tmAveCharWidth;


	GLYPHMETRICS gm;
	MAT2 mat;
	// initialize mat to be an identity matrix
	mat.eM11.value = 1;
	mat.eM12.value = 0;
	mat.eM21.value = 0;
	mat.eM22.value = 1;

	mat.eM11.fract = 0;
	mat.eM12.fract = 0;
	mat.eM21.fract = 0;
	mat.eM22.fract = 0;

	int length = textRun.fSubText.length();
	IText myIText(textRun.fSubText);
	const char* str =(const char*)myIText;
	
	IGPoint2D nextCharAnchor;
	nextCharAnchor = textRun.anchor();
		
	IGrafMatrix vertRecfMat(IGPoint2D(1, -1), IGPoint2D::origin());
	for (int j = 0; j < length; j++) {
    if  ( str[j] == 32 /* ANSII Code for space*/ )
    { // no outline for a space character
    	switch (textRun.attachment())
  			{
  				default:
  				case IGTextRun::kLeftToRight:
  					 nextCharAnchor.fX += spaceWidth;
  					break;
  				case IGTextRun::kRightToLeft:
  					nextCharAnchor.fX -= spaceWidth;
  					break;
  				case IGTextRun::kTopToBottom:
  					nextCharAnchor.fY += spaceWidth;
  					break;
  				case IGTextRun::kBottomToTop:
  					nextCharAnchor.fY -= spaceWidth;
  					break;
  			}
    }
    else
    {  // extract and display outline for non-space character
  		unsigned long gSize;
    	gSize = GetGlyphOutline(hdc, str[j], GGO_NATIVE, &gm, NULL, NULL, &mat);
  		if (gSize >= 1)
  		{
  			IDeleterForArrayOf<char> gmBuf(new char[gSize]);

  			if (!(char *)gmBuf)
  				GrafDeviceException("Cannot allocate outline buffer");

  			if (GetGlyphOutline(hdc, str[j], GGO_NATIVE, &gm, gSize, (LPVOID)(char *)gmBuf, &mat) == GDI_ERROR)
  				GrafDeviceException("GetGlyphOutline");

  			// parse buffer into glyph
  			IGLoop2D* pNewOne = BufIntoGlyph(gSize, (char *)gmBuf);

  			// the outline extracted is upsidedown, I have to use
  			// reflect it vertically to get correct rendering effect.
  			pNewOne->transformBy(vertRecfMat);

  			// move charactor outline to character anchor position
  			IGrafMatrix anchorMat(nextCharAnchor);
  			pNewOne->transformBy(anchorMat);	

  			fDevice->renderLoop(*pNewOne, tempState);

  			// get extent point of the displayed string
  			SIZE theSize;
  			if (!GetTextExtentPoint32(fDevice->fHDC,textRun.fSubText, j+1, &theSize))
  				GrafDeviceException("GetTextExtentPoint32");
  			
  			// calculate the advancement of the displayed string
  			IGPoint2D advance = IGPoint2D::origin();
  			switch (textRun.attachment())
  			{
  				default:
  				case IGTextRun::kLeftToRight:
  					advance.fX += theSize.cx;
  					break;
  				case IGTextRun::kRightToLeft:
  					advance.fX -= theSize.cx;
  					break;
  				case IGTextRun::kTopToBottom:
  					advance.fY += theSize.cy;
  					break;
  				case IGTextRun::kBottomToTop:
  					advance.fY -= theSize.cy;
  					break;
  			}
  			nextCharAnchor = textRun.anchor() + advance;
  				
  			// an IGLoop2D object is constructed at BufIntoGlyph()
  			delete pNewOne;
  		} //if
    }//else -- non-space character
	} // for loop
}

	
// The following actually should be moved to a separate file


//**********************************************************************
// helper function for text outline extracting
//**********************************************************************
GCoordinate myF16Dot16ToGCoordinate(FIXED a)
{
	long b = *(long*)&a;
	GCoordinate retval = (GCoordinate)b / (GCoordinate)USHRT_MAX;
	return retval;
}

//**********************************************************************
// helper function for text outline extracting
//**********************************************************************
IGLoop2D* BufIntoGlyph(unsigned long size, char *chBuf)
{
	GlyphBuilder builder;
	char* ptr = chBuf;
	TTPOLYGONHEADER* lpPH = (TTPOLYGONHEADER*)NULL;
	int i = 0;
	int bytesParsed = 0;

	builder.startAfresh ();

	short sizeOfPolygonHeader = sizeof(TTPOLYGONHEADER);
	do
	{
		lpPH = (TTPOLYGONHEADER*)ptr;
		i++;
			
		unsigned long sizeOfContour = lpPH->cb;

		GCoordinate lastPtX = myF16Dot16ToGCoordinate((lpPH->pfxStart).x);
		GCoordinate lastPtY = myF16Dot16ToGCoordinate((lpPH->pfxStart).y);

		builder.startNewContour(lastPtX, lastPtY);

		char *cp = ptr + sizeOfPolygonHeader;
		short sizeOfPolyCurveData = (short)(sizeOfContour - sizeOfPolygonHeader);
		short pcBytesParsed = 0;
		short j = 0;

		do
		{
			LPTTPOLYCURVE lpPC = (LPTTPOLYCURVE) cp;
			short numPts = lpPC->cpfx;
			short type = lpPC->wType;

			j++;

			if (type == TT_PRIM_LINE)
			{
				for (short k = 0; k < numPts; k++)
				{
					lastPtX = myF16Dot16ToGCoordinate((lpPC->apfx[k]).x);
					lastPtY = myF16Dot16ToGCoordinate((lpPC->apfx[k]).y);

					builder.addLineTo(lastPtX, lastPtY);
				}
			} // if line segments
			else if (type == TT_PRIM_QSPLINE)
			{
				GCoordinate ptAX = lastPtX;
				GCoordinate ptAY = lastPtY;

				for (short k = 0; k < numPts - 1; k++)
				{
					GCoordinate ptBX = myF16Dot16ToGCoordinate((lpPC->apfx[k]).x);
					GCoordinate ptBY = myF16Dot16ToGCoordinate((lpPC->apfx[k]).y);

					GCoordinate ptCX = myF16Dot16ToGCoordinate((lpPC->apfx[k + 1]).x);
					GCoordinate ptCY = myF16Dot16ToGCoordinate((lpPC->apfx[k + 1]).y);

					if (k < numPts - 2)
					{
						ptCX = (ptBX + ptCX) / 2.0;
						ptCY = (ptBY + ptCY) / 2.0;
					}

					builder.addCurve(ptBX, ptBY, ptCX, ptCY);

					lastPtX = ptAX = ptCX;
					lastPtY = ptAY = ptCY;

				} // for k
		
			} // else if spline segments
			else
			{
				GrafDeviceException("TT outline curve type invalid");
			}

			pcBytesParsed += sizeof(WORD) + sizeof(WORD) + numPts * sizeof(POINTFX);
			cp = ptr + sizeOfPolygonHeader + pcBytesParsed;
			
		} while (pcBytesParsed < sizeOfPolyCurveData);

		bytesParsed += sizeOfContour;
		ptr = ptr + lpPH->cb;
		
	} while (bytesParsed < size);

	IGLoop2D* retval = new IGLoop2D(builder.glyph());
	retval->closeLoop();
	return (retval);
}

//**********************************************************************
// helper class for extracting text outline
// Class name: GlyphBuilder
//**********************************************************************
GlyphBuilder::GlyphBuilder()
{
	fKnots = IRawArray<GParametric>(MAX_PTS_IN_GLYPH + 3);
	fCurvePtArray = IGRPoint2DArray(MAX_PTS_IN_GLYPH);

	startAfresh();
}

GlyphBuilder::~GlyphBuilder()
{
}

void GlyphBuilder::startAfresh()
{
	//
	// Just initialize the counts for valid # of points and knots.
	// We don't need to initialize the arrays. We rely on the
	// premise that only count values in the array are valid.
	//

	fPtCount = 0;
	fKVCount = 0;

	fKnotValue = -1;
	fContourJustStarted = true;

} // GlyphBuilder::startAfresh

void GlyphBuilder::startNewContour(const GCoordinate x, const GCoordinate y)
{
	//
	// ...means that the new contour curve has a c0 discontinuity
	// with the previous contour except for the first one.
	//

	fContourJustStarted = true;

	if (fPtCount == 0)
	{
		fKnotValue ++;
		fKnots.setValue (fKVCount, fKnotValue); fKVCount ++;
		fKnots.setValue (fKVCount, fKnotValue); fKVCount ++;
		fKnots.setValue (fKVCount, fKnotValue); fKVCount ++;

	}
	fCurvePtArray.setPoint (fPtCount, IGRPoint2D (x, y)); fPtCount ++;

} // GlyphBuilder::startNewContour

void GlyphBuilder::addLineTo(const GCoordinate endPtX, const GCoordinate endPtY)
{
	IGRPoint2D startPt = fCurvePtArray.point(fPtCount - 1);
	IGRPoint2D endPt(endPtX, endPtY);
	IGRPoint2D midPt((startPt.fX + endPt.fX) / 2, (startPt.fY + endPt.fY) / 2);

	fCurvePtArray.setPoint(fPtCount, midPt); fPtCount ++;
	fCurvePtArray.setPoint(fPtCount, endPt); fPtCount ++;
	
	addKnotsForThisSegment();

} // GlyphBuilder::addLineTo

void GlyphBuilder::addCurve(
	const GCoordinate x2,
	const GCoordinate y2,
	const GCoordinate x3,
	const GCoordinate y3)
{
	IGRPoint2D pt2(x2, y2);
	IGRPoint2D pt3(x3, y3);

	fCurvePtArray.setPoint(fPtCount, pt2); fPtCount++;
	fCurvePtArray.setPoint(fPtCount, pt3); fPtCount++;

	addKnotsForThisSegment();

} // GlyphBuilder::addCurve

void GlyphBuilder::addKnotsForThisSegment ()
{
	fKnotValue++;

	if (fContourJustStarted)
	{
		fKnots.setValue(fKVCount, fKnotValue); fKVCount++;
		fKnots.setValue(fKVCount, fKnotValue); fKVCount++;
		fKnots.setValue(fKVCount, fKnotValue); fKVCount++;
		
	}
	else
	{
		fKVCount--;
		fKnots.setValue(fKVCount, fKnotValue); fKVCount++;
		fKnots.setValue(fKVCount, fKnotValue); fKVCount++;
		fKnots.setValue(fKVCount, fKnotValue); fKVCount++;
	}

	fContourJustStarted = false;

} // GlyphBuilder::addKnotsForThisSegment

IGLoop2D GlyphBuilder::glyph() const
{
	assert(fPtCount + 3 == fKVCount);
	IRawArray<GParametric> knots(fKVCount);
	IGRPoint2DArray curvePtArray(fPtCount);

	for (int k = 0; k < fPtCount; k ++)
	{
		IGRPoint2D pt = fCurvePtArray.point (k);
		curvePtArray.setPoint (k, pt);
	}

	for (int j = 0; j < fKVCount; j ++)
	{
		GParametric val = fKnots.value (j);
		knots.setValue (j, val);
	}

	IGLoop2D retval = IGLoop2D(3, curvePtArray, knots);

	return retval;

} // GlyphBuilder::glyph

#if TESTING_GLYPHBUILDER
IGLoop2D testGlyphBuilder ()
{
	GlyphBuilder* gBilder;
	gBilder->startAfresh();

	gBilder->startNewContour(6.0 + 100.0, 60.0 + 100.0);
	gBilder->addLineTo(6.0 + 100.0, 69.0 + 100.0);
	gBilder->addLineTo(15.0 + 100.0, 69.0 + 100.0);
	gBilder->addLineTo(15.0 + 100.0, 60.0 + 100.0);


	gBilder->startNewContour(-5.4 + 100.0, -20.6 + 100.0);
	gBilder->addLineTo(-3.2 + 100.0, -12.4 + 100.0);
	gBilder->addCurve(-1.5 + 100.0, -12.0 + 100.0, 1.6 + 100.0, -12.0 + 100.0);
	gBilder->addCurve(3.4 + 100.0, -12.0 + 100.0, 4.7 + 100.0, -10.7 + 100.0);
	gBilder->addCurve(6.0 + 100.0, -9.3 + 100.0, 6.0 + 100.0, -2.1 + 100.0);
	gBilder->addLineTo(6.0 + 100.0, 50.0 + 100.0);
	gBilder->addLineTo(15.0 + 100.0, 50.0 + 100.0);
	gBilder->addLineTo(15.0 + 100.0, -3.3 + 100.0);
	gBilder->addCurve(15.0 + 100.0, -12.2 + 100.0, 12.4 + 100.0, -16.5 + 100.0);
	gBilder->addCurve(9.3 + 100.0, -20.0 + 100.0, 2.2 + 100.0, -20.0 + 100.0);
	gBilder->addCurve(-2.5 + 100.0, -20.0 + 100.0, -5.4 + 100.0, -20.6 + 100.0);

	return gBilder->glyph ();

} // testGlyphBuilder
#endif

