/*----------------------------------------------------------------------------
 *
 *  main.c
 *  ------
 *
 *  Purpose: Provide the startup processing for VisualAge Smalltalk Enterprise
 *
 *  Commandline switches:
 *     -nosplash  Do not display the splash screen
 *     -singleinstance  Allow multiple copies of the executable to run at the
 *                same time
 *
 *  Licensed Materials - Property of IBM
 *
 *  (C) Copyright International Business Machines Corp., 1999
 *  All rights reserved
 *
 *  US Government Users Restricted Rights -
 *  Use, duplication, or disclosure restricted
 *  by GSA ADP Schedule Contract with IBM Corp.
 *
 *  Modifications:
 *    5.5.0.0 24Nov99(jok) Update to 5.1.0 w/2000 copyright
 *    5.5.0.1 19Jan00(jok) Load splash bitmap from file
 *    5.5.0.2 02Mar00(jok) ES40MD30 rebuild
 *
 *----------------------------------------------------------------------------*/

#define PRODUCT_NAME    "VisualAge Smalltalk Enterprise"
#define PRODUCT_VERSION "Version 5.5"
#define PRODUCT_STATUS  ""
#define PRODUCT_EDITION "03/23/2000"
#define PRODUCT_COPYR   "(C) Copyright International Business Machines Corporation  1994, 2000.\nAll Rights Reserved."

#define	INCL_WIN
#define INCL_DOSPROCESS
#define INCL_DOSERRORS
#define INCL_DOSSEMAPHORES
#define INCL_ERRORS
#define INCL_GPIBITMAPS

#include <os2.h>
#include <string.h>
#include "esrc.h"
#include "esuser.h"

/*--------------------------------------------------------------------------*/
/*      DEBUG support                                                       */
/*--------------------------------------------------------------------------*/

#ifdef DEBUG01
#define dbg_printf( x ) { printf x ; fflush( stdout ); }
#else
#define dbg_printf( x )
#endif


/* globals */
PFNWP StaticProc;
HBITMAP hBitmap;
char bitmapPath[MAX_IMAGE_PATH_LENGTH]; /* path to <splash>.bmp */ 


extern EsPrimitiveTable filePrimitiveTable;
extern EsPrimitiveTable WindowProcPrimitiveTable;
extern EsSplashShutDown (EsGlobalInfo *);

/* function prototypes */
BOOLEAN EsParseCommandLineOptions(int argc, char *argv[], char *envp[], EsGlobalInfo * info);


EsDefinePrimitiveTable(EsExePrimitiveTable)
   EsSubTable(WindowProcPrimitiveTable)
   EsSubTable(filePrimitiveTable)
EsEndPrimitiveTable

/**************************************************************************
* 
*  function:  LoadBitmapFile()
*   Load a .bmp file with the same name as the .exe file
* 
*  Input: 
*   hPS - handle of presentation space to make the bitmap compatible with.
* 
*  Returns:
*   handle to the bitmap if successful; otherwise NULL. 
* 
***************************************************************************/ 

HBITMAP LoadBitmapFile( HPS hPS ) { 
   HBITMAP hbm = (HBITMAP) NULL;     /* Setup for error exit */
   FILESTATUS3 fileInfo;            /* Standard file information */ 
   PBITMAPFILEHEADER2 pbfh2; 
   PBITMAPINFOHEADER2 pbmp2; 
   PBYTE pBuffer = NULL; 
   HFILE hFile; 
   ULONG bytesToRead;
   ULONG bytesActuallyRead;
   ULONG ulAction;
   APIRET ulrc;

   dbg_printf( ("Entered LoadBitmapFile\n") );
   ulrc = DosOpen( bitmapPath,      /* Name of file to open */
                   &hFile,          /* Address of file handle */
                   &ulAction,       /* Address of action taken */
                   0,		    /* File size */	
                   FILE_NORMAL,     /* File attribute */
                   FILE_OPEN,       /* Open the file */
                   OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE | OPEN_FLAGS_SEQUENTIAL,
                   (PEAOP2) NULL ); /* Extended attribute buffer */			
   if (ulrc == 0) {
      ulrc = DosQueryFileInfo( hFile,
                               FIL_STANDARD,
                               &fileInfo,
                               sizeof( fileInfo ) );	
      if (ulrc == 0) {
         bytesToRead = fileInfo.cbFile;      /* Filesize */
         dbg_printf( ("\tBitmap size: %d\n", bytesToRead) );
         ulrc = DosAllocMem( (PPVOID) &pBuffer,
                             bytesToRead,
                             PAG_READ | PAG_WRITE | PAG_COMMIT );
         if (ulrc == 0) {
            dbg_printf( ("\tBitmap memory allocated\n") );
            ulrc = DosRead( hFile,
                            (PVOID) pBuffer,
                            bytesToRead,
                            &bytesActuallyRead );
            if ((ulrc == 0) &
                (bytesToRead == bytesActuallyRead)) {
               dbg_printf( ("\tBitmap file read is OK\n") );
               pbfh2 = (PBITMAPFILEHEADER2) pBuffer;
               dbg_printf( ("\tBitmap type = 0x%x\n", pbfh2 -> usType) );
               if ((pbfh2 -> usType == BFT_BMAP) & 
                   (pbfh2 -> cbSize == bytesToRead)) {
                  dbg_printf( ("\tBitmap type and size are OK\n") );
                  pbmp2 = (PBITMAPINFOHEADER2) &pbfh2 -> bmp2; 
                  hbm = GpiCreateBitmap( hPS, 
                                         pbmp2, 
                                         CBM_INIT, 
                                         (PBYTE) pBuffer + pbfh2 -> offBits, 
                                         (PBITMAPINFO2) pbmp2 );
                  } 
               }
            DosFreeMem( pBuffer ); 
            }
         }
      DosClose( hFile );
      }
   return hbm; 
   } 



MRESULT EXPENTRY DrawBitmapProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {				
   RECTL rect;
   HPS hPS;

   dbg_printf( ("Entered DrawBitmapProc\n") );
   switch(msg) {
      case WM_PAINT:
         dbg_printf( ("Got WM_PAINT\n") );
         WinQueryWindowRect( hwnd, &rect );
         hPS = WinBeginPaint( hwnd,
                              (HPS) NULL,
                              (PRECTL) NULL );
         WinDrawBitmap( hPS,
                        hBitmap,
                        NULL,
                        (PPOINTL) &rect,
                        0,
                        0,
                        DBM_NORMAL );
         WinEndPaint( hPS );
         break;

      default:
         dbg_printf( ("Unidentified DrawBitmapProc message = %d\n", msg) );
         return (MRESULT) StaticProc( hwnd, msg, mp1, mp2 );
      }
   return (MRESULT) FALSE;
   }



static void _System splashDialogThread( EsGlobalInfo *globalInfo ) {
   HWND hwndDlgBox;
   HWND hwndStatic;
   HAB hab;
   HMQ hmq;
   QMSG qmsg;
   int displayWidth, displayHeight;
   HPS hPS;
   BITMAPINFOHEADER info;
   int nWidth, nHeight, nX, nY, insetX, insetY;

   dbg_printf( ("Entered splashDialogThread\n") );
   hab = WinInitialize( 0 );
   hmq = WinCreateMsgQueue( hab, 0 );
   hwndDlgBox = WinLoadDlg( HWND_DESKTOP,
                            HWND_DESKTOP,
                            (PFNWP) WinDefDlgProc,
                            (HMODULE) NULL,
                            IDD_SPLASHDIALOG,
                            (PVOID) NULL );
   if (hwndDlgBox != (HWND) NULL) {     /* Got the dialog box handle */
      dbg_printf( ("\thwndDlgBox: %x\n", hwndDlgBox) );
      globalInfo->splashDialogHandle = (void *) hwndDlgBox;     /* Set VM handle to the dialog box */
      globalInfo->splashStatusLineHandle = (void *) NULL; /* WinWindowFromID( hwndDlgBox, IDS_STATUS ); */
      globalInfo->messageLevel = ESVM_MSG_LEVEL_NONE;
      hwndStatic = WinWindowFromID( hwndDlgBox, ID_STATICAREA );
      if (hwndStatic != (HWND) NULL) {
         dbg_printf( ("\thwndStatic: %x\n", hwndStatic) );
         StaticProc = WinSubclassWindow( hwndStatic, DrawBitmapProc );
         hPS = WinGetPS( hwndDlgBox );
	 hBitmap = LoadBitmapFile( hPS );
         WinReleasePS(hPS);
         if (hBitmap != (HBITMAP) NULL) {
            GpiQueryBitmapParameters( hBitmap, &info );
            dbg_printf( ("\tinfo.cx = %d; info.cy = %d\n", info.cx, info.cy) );

            /* Available display space */
            displayWidth = WinQuerySysValue( HWND_DESKTOP, SV_CXSCREEN );
            displayHeight = WinQuerySysValue( HWND_DESKTOP, SV_CYSCREEN );
            dbg_printf( ("\tdisplayWidth = %d; displayHeight = %d\n", displayWidth, displayHeight) );

            /* Inset controls within window by the size of the dialog frame */
	    insetX = WinQuerySysValue( HWND_DESKTOP, SV_CXDLGFRAME );
            insetY = WinQuerySysValue( HWND_DESKTOP, SV_CYDLGFRAME );
            dbg_printf( ("\tinsetX = %d; insetY = %d\n", insetX, insetY) );

            /* Window size and position accounting for the dialog frame */
            nWidth = info.cx + (insetX * 2);
            nHeight = info.cy + (insetY * 2);
            nX = max( (displayWidth - nWidth) / 2, 0 );
            nY = max( (displayHeight - nHeight) / 2, 0 );
            dbg_printf( ("\tnWidth = %d; nHeight = %d; nX = %d; nY = %d\n", nWidth, nHeight, nX, nY) );

            WinSetWindowPos( hwndStatic,  /* Static area handle */ 
                             (HWND) NULL, /* No change in Z-order */
                             insetX,      /* Control x coordinate */
                             insetY,      /* Control y coordinate */
                             info.cx,     /* Resize to bitmap width */
                             info.cy,     /* Resize to bitmap height */
                             SWP_MOVE | SWP_SIZE );  /* Position and size the control */
            WinSetWindowPos( hwndDlgBox,  /* Window handle */ 
                             (HWND) NULL, /* No change in Z-order */
                             nX,          /* Window x coordinate */
                             nY,          /* Window y coordinate */
                             nWidth,      /* Resize to bitmap width */
                             nHeight,     /* Resize to bitmap height */
                             SWP_MOVE | SWP_SIZE | SWP_SHOW );  /* Position, size, and show the window */
            WinShowWindow( hwndDlgBox, TRUE );
            WinUpdateWindow( hwndDlgBox );

            /* NOTE: The splash window gets destroyed in WinDefDlgProc due to an
               EsSplashShutDown call from either the Smalltalk image
               (Sustem>>#splashShutDown) or the main thread (in the case
               of an error starting Smalltalk).  This will cause the
               following while loop to terminate. */
            while (WinGetMsg( hab, &qmsg, (HWND) NULL, 0L, 0L ))
               WinDispatchMsg( hab, &qmsg );
            }
         }   
      globalInfo->splashDialogHandle = (void *) NULL;
      globalInfo->splashStatusLineHandle = (void *) NULL;
	
      if (hBitmap != (HBITMAP) NULL) {
         GpiDeleteBitmap (hBitmap);
         hBitmap = (HBITMAP) NULL;
         }
      }

   WinDestroyMsgQueue( hmq );
   WinTerminate( hab );
   }



/* Answer true if we are running as a Presentation Manager application, false otherwise (VIO mode). */
static BOOLEAN runningPM( void ) {
   PTIB ptib = NULL;
   PPIB ppib = NULL;
   APIRET rc = NO_ERROR;

   rc = DosGetInfoBlocks(&ptib, &ppib);
   if (rc != NO_ERROR)
      return FALSE;

   /* type 3 is PM */
   return (ppib->pib_ultype == 3);
   }



main(int argc, char ** argv, char ** envp) {
   BOOLEAN splashEnabled=TRUE;
   BOOLEAN mutexEnabled=FALSE;
   I_32 rc,i;
   EsGlobalInfo *globalInfo;
   TID tid;
   HMTX startupMutex;
   char mutexNameBuffer[1024] = "\\SEM32\\";
   char *exeName, *exeExtension, *ptr;
   int len;


   /* Walk the command line looking VA specific command line options */
   dbg_printf( ("Entering main, %d commandline arguments\n", argc) );
   for (i=1; i < argc; i++) {
      dbg_printf( ("\t%s\n", argv[i]) );
      if (strcmp(argv[i],"-nosplash") == 0) 
         splashEnabled = FALSE;
      if (strcmp(argv[i],"-singleinstance") == 0)
         mutexEnabled = TRUE;
      }
   dbg_printf( ("splashEnabled: %d\n", splashEnabled) );
   dbg_printf( ("mutexEnabled: %d\n", mutexEnabled) );

   if (mutexEnabled) {
      /* strip the path and extension from the program name */
      exeName = strrchr( argv[0], '\\' );
      if (exeName)
         exeName++;
      else
         exeName = argv[0];

      strcat( mutexNameBuffer, exeName );
      /* uppercase everything */
      for (ptr = mutexNameBuffer; *ptr; ptr++)
         *ptr = toupper(*ptr);
      exeExtension = strrchr( mutexNameBuffer, '.' );
      if (exeExtension && 0 == strcmp( exeExtension, ".EXE" ))
         *exeExtension = '\0';
      strcat( mutexNameBuffer, "SINGLE_START_MUTEX" );

      rc = DosCreateMutexSem( mutexNameBuffer, &startupMutex, 0L, FALSE );
      if(NO_ERROR != rc)
         return 0;	/* not first instance, go away. */
      }

   /* Do not call any Presentation Manager functions if running VIO. */

   globalInfo = EsInitializeImage();
   if (globalInfo == NULL) {
      EsReportError( EsPrimErrNotEnoughMemory, globalInfo );
      DosCloseMutexSem( startupMutex );
      return EsPrimErrNotEnoughMemory;
      }
   globalInfo->imagePrimitives = (U_32) EsExePrimitiveTable;

   if (runningPM() && splashEnabled) {
      strncpy( bitmapPath, argv[0], MAX_IMAGE_PATH_LENGTH );
      bitmapPath[MAX_IMAGE_PATH_LENGTH-1] = '\0'; /* In case it's too long */
      len = strlen(bitmapPath);
      if ((len > 4) && strcmpi( bitmapPath + len - 4, ".exe" ) == 0) {
         strncpy( bitmapPath + len - 3, "bmp", 3 );
         dbg_printf( ("Showing splash from '%s'\n", bitmapPath) );

         /* fork off the thread to keep splash screen alive */
         DosCreateThread( &tid,
                          (PFNTHREAD) splashDialogThread,
                          (ULONG) globalInfo,
                          CREATE_READY,
                          8192 );
         }
      }

   if ((rc = EsInitializeTargetInterface( argc, argv, envp, globalInfo )) != EsPrimErrNoError) {
      EsSplashShutDown( globalInfo );
      EsReportError( rc, globalInfo );
      EsShutdownImage( globalInfo );
      if (mutexEnabled)
         DosCloseMutexSem( startupMutex );
      return rc;
      }

   EsPrintf( "\n%s, %s %s - %s\nVM Timestamp: %s\n%s\n%s\n",
             PRODUCT_NAME,PRODUCT_VERSION,PRODUCT_STATUS,PRODUCT_EDITION,
             EsVMVersionString(),
             PRODUCT_COPYR,
             0,0,0,0);

   if (!EsParseCommandLineOptions( argc, argv, envp, globalInfo )) {
      EsSplashShutDown( globalInfo );
      EsReportError( EsPrimErrImageFileOpenError, globalInfo );
      EsShutdownImage( globalInfo );
      if (mutexEnabled)
         DosCloseMutexSem( startupMutex );
      return EsPrimErrImageFileOpenError;
      }

   rc = EsLoadFileImage( argc, argv, envp, globalInfo );
   if (rc != EsPrimErrNoError) {
      EsSplashShutDown( globalInfo );
      EsReportError ( rc, globalInfo );
      }

   if (rc == EsPrimErrNoError) {
      rc = EsExecuteImage( globalInfo );
      if ((rc != EsPrimErrNoError) && (rc < 9000 || (rc > 9100)))
         EsReportError( rc, globalInfo );
      }

   EsShutDownTargetInterface( globalInfo );
   EsShutdownImage( globalInfo );
   if (mutexEnabled)
      DosCloseMutexSem( startupMutex );
   return rc;
   }