/****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module name:  GPIVIEW
 *
 * Purpose:  This module is called by PTT to test the GPI View APIs.
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               12/04/90, PDVT, Hal Bray
 *                     -  implementation of testcase design
 *
 * SRS References: Section 5.5
 *
 * Contents: InitTest(),
 *           GpiSetDefViewingLimitsPre(),
 *           GpiSetGraphicsFieldPre(),
 *           GpiSetViewingLimitsPre(),
 *           GpiViewApp(),
 *           GpiSetDefViewingLimitsExh1(),
 *           GpiSetDefViewingLimitsExh2(),
 *           GpiSetViewingLimitsExh1(),
 *           GpiSetViewingLimitsExh2(),
 *           GpiSetPageViewportExh()
 *
 * Design Overview:
 *      The GPIVIEW testcase involves testing of the following GPIs:
 *      GpiSetDefViewingLimits(), GpiSetGraphicsField(), GpiSetViewingLimits(),
 *      and GpiSetPageViewport().  The individual testing of each GPI will be
 *      exhaustive.  In addition to the exhaustive tests, three other tests
 *      will be performed:  an Application test will test how each GPI works
 *      with other GPI's, a Return Code test will verify every possible error
 *      return code, and a Precision test will test the precision of the actual
 *      output.
 *
 * Limitations: NONE
 *
 *\\end
****************************************************************************/

#define INCL_WIN
#define INCL_GPI
#define INCL_ERRORS
#define INCL_DEV
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include "pttdlldf.h"   // test case defines
#include "gpi_grid.h"   // Precision Grid defines
#include "std_util.h"   // Standard functions
#include "gpiview.h"    // GPIVIEW defines

LONG    error_count;
CHAR    err_log_line[80];

/****************************************************************************
 *\\ddd
 * Routine Name: InitTest()
 *
 * Purpose: Initialize function name and address array for PTT.
 *
 * System Requirements: NONE
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                      - initial creation
 *               12/04/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs: sel          - pass thru parameter for other function calls
 *         pInitInfo    - pointer to info array
 *
 * Outputs: pInitInfo   - updated with all entry point names and addresses
 *
 * Subroutines Required: InitTestSegVars()
 *
 * Limitations: None
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Initialize the number of tests & test group name in the info structure
 *      For each test routine {
 *          Copy the name of the routine into the info structure
 *          Copy the address of the routine into the info structure
 *      }
 *\\end
 ***************************************************************************/

VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO pInitInfo)
{
    //  Initialize the number of tests & test group name in the info struct.
    pInitInfo->ntests = NTESTS;
    strcpy(pInitInfo->group_name,"GPI View");

    //  Initialize the test routine names and addresses in the info struct.
    strcpy(pInitInfo->test_names[0],"GpiSetDefViewingLimitsPre");
    pInitInfo->test_addrs[0]=GpiSetDefViewingLimitsPre;
    strcpy(pInitInfo->test_names[1],"GpiSetGraphicsFieldPre");
    pInitInfo->test_addrs[1]=GpiSetGraphicsFieldPre;
    strcpy(pInitInfo->test_names[2],"GpiSetViewingLimitsPre");
    pInitInfo->test_addrs[2]=GpiSetViewingLimitsPre;

    strcpy(pInitInfo->test_names[3],"GpiViewApp");
    pInitInfo->test_addrs[3]=GpiViewApp;

    strcpy(pInitInfo->test_names[4],"GpiSetDefViewingLimitsExh1");
    pInitInfo->test_addrs[4]=GpiSetDefViewingLimitsExh1;
    strcpy(pInitInfo->test_names[5],"GpiSetDefViewingLimitsExh2");
    pInitInfo->test_addrs[5]=GpiSetDefViewingLimitsExh2;
    strcpy(pInitInfo->test_names[6],"GpiSetViewingLimitsExh1");
    pInitInfo->test_addrs[6]=GpiSetViewingLimitsExh1;
    strcpy(pInitInfo->test_names[7],"GpiSetViewingLimitsExh2");
    pInitInfo->test_addrs[7]=GpiSetViewingLimitsExh2;
    strcpy(pInitInfo->test_names[8],"GpiSetPageViewportExh");
    pInitInfo->test_addrs[8]=GpiSetPageViewportExh;


    InitTestSegVars(sel);
}

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetDefViewingLimitsPre()
 *
 * Purpose: This PTT entry point begins the Precision testing of
 *          GpiSetDefViewingLimits.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 12/07/90, PDVT, Hal Bray
 *                     - developed test cases
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetDefViewingLimits
 *          GpiMove
 *          GpiResetPS
 *          MapAndWriteString
 *          gpi_grid
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the default viewing limits to be a 5" x 5" rectangle from
 *          coordinates (1",1") to (6",6")
 *      Reset the PS which will change the current viewing limits to the
 *          default viewing limits
 *      Draw a test grid of 1-inch squares to show the exact size of the
 *          default viewing limits
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetDefViewingLimitsPre(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
    RECTL   rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetDefViewingLimitsPre - Entry\n");

        error_count = 0;

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        /*
         * Set the default viewing limit to be a 5-inch x 5-inch square
         * from coordinates (1",1") to (6",6").
         */
        rectangle.xLeft =   delta_x;
        rectangle.yBottom = delta_y;
        rectangle.xRight =  6 * delta_x;
        rectangle.yTop =    6 * delta_y;
        MapRect(rectangle);
        if (GpiSetDefViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetDefViewingLimits", TRUE, "GpiSetDefViewingLimitsPre");
            return;
        }

        //  Print a message on the PS that GpiSetDefViewingLimits was called.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) + (line_space * 3)),
                         "(1) GpiSetDefViewingLimits was called.");

        /*
         * Call GpiResetPS which will reset the viewing limit to the default
         * values, and print the name of the routine -- GpiResetPS.
         */
        if (GpiResetPS(hPS, GRES_ATTRS) == FALSE) {
            BADERROR("GpiResetPS", TRUE, "GpiSetDefViewingLimitsPre");
            return;
        }
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) + line_space),
                         "(2) GpiResetPS was called.");

        //  Print the size of the viewing limit on the PS.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - line_space),
                         "Default viewing limit is 5\" x 5\"");
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 2)),
                         "     at (1\",1\") (6\",6\").");

        /*
         * Draw precision test grid of 1-inch squares, and print the
         * size of the grid on the PS.  The edges of the viewing limit
         * should fall directly on top of the grid lines.
         */
        gpi_grid(hPS, (LONG) INCH);
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 4)),
                         "Precision grid size is 1-inch.");

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetDefViewingLimitsPre                                      -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetDefViewingLimitsPre                                      -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetDefViewingLimitsPre - Exit.\n");

    }  /* end else */

}  /* End GpiSetDefViewingLimitsPre */


/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetGraphicsFieldPre()
 *
 * Purpose: This PTT entry point begins the Precision testing of
 *          GpiSetGraphicsField.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 12/07/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetGraphicsField
 *          GpiMove
 *          MapAndWriteString
 *          gpi_grid
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the graphics field to be a 5" x 5" rectangle from
 *          coordinates (1",1") to (6",6")
 *      Draw a test grid of 1-inch squares to show the exact size of the
 *          graphics field
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetGraphicsFieldPre(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
    RECTL   rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetGraphicsFieldPre - Entry\n");

        error_count = 0;

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        /*
         * Set the graphics field to be a 5-inch x 5-inch square
         * from coordinates (1",1") to (6",6").
         */
        rectangle.xLeft =   delta_x;
        rectangle.yBottom = delta_y;
        rectangle.xRight =  6 * delta_x;
        rectangle.yTop =    6 * delta_y;
        MapRect(rectangle);
        if (GpiSetGraphicsField(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetGraphicsField", TRUE, "GpiSetGraphicsFieldPre");
            return;
        }

        //  Print a message on the PS that GpiSetGraphicsField was called.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) + (line_space * 3)),
                         "GpiSetGraphicsField was called.");

        //  Print the size of the graphics field on the PS.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - line_space),
                         "Graphics Field is 5\" x 5\"");
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 2)),
                         "     at (1\",1\") (6\",6\").");

        /*
         * Draw precision test grid of 1-inch squares, and print the
         * size of the grid on the PS.  The edges of the graphics field
         * should fall directly on top of the grid lines.
         */
        gpi_grid(hPS, (LONG) INCH);
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 4)),
                         "Precision grid size is 1-inch.");

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetGraphicsFieldPre                                         -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetGraphicsFieldPre                                         -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetGraphicsFieldPre - Exit.\n");

    }  /* end else */

}  /* end GpiSetGraphicsFieldPre */

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetViewingLimitsPre()
 *
 * Purpose: This entry point begins Precision testing of GpiSetViewingLimitsPre
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 12/07/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiDefViewingLimits
 *          GpiMove
 *          MapAndWriteString
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the viewing limit to be a 5" x 5" rectangle from
 *          coordinates (1",1") to (6",6")
 *      Draw a test grid of 1-inch squares to show the exact size of the
 *          graphics field
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetViewingLimitsPre(HPS hPS,HWND hWndClient, BOOL SelectionCall)
{
    RECTL   rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetViewingLimitsPre - Entry\n");

        error_count = 0;

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        /*
         * Set the viewing limit to be a 5-inch x 5-inch square
         * from coordinates (1",1") to (6",6").
         */
        rectangle.xLeft =   delta_x;
        rectangle.yBottom = delta_y;
        rectangle.xRight =  6 * delta_x;
        rectangle.yTop =    6 * delta_y;
        MapRect(rectangle);
        if (GpiSetViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetViewingLimits", TRUE, "GpiSetViewingLimitsPre");
            return;
        }

        //  Print a message on the PS that GpiSetViewingLimits was called.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) + (line_space * 3)),
                         "GpiSetViewingLimits was called.");

        //  Print the size of the viewing limit on the PS.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - line_space),
                         "Viewing limit is 5\" x 5\"");
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 2)),
                         "     at (1\",1\") (6\",6\").");

        /*
         * Draw precision test grid of 1-inch squares, and print the
         * size of the grid on the PS.  The edges of the viewing limit
         * should fall directly on top of the grid lines.
         */
        gpi_grid(hPS, (LONG) INCH);
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 4)),
                         "Precision grid size is 1-inch.");

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetViewingLimitsPre                                         -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetViewingLimitsPre                                         -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetViewingLimitsPre - Exit.\n");

    }  /* end else */

}  /* End GpiSetViewingLimitsPre */


/****************************************************************************
 *\\ddd
 * Routine Name: GpiViewApp()
 *
 * Purpose: This PTT entry point begins the application test.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                      - initial creation
 *               12/04/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetDefViewingLimits
 *          GpiSetViewingLimits
 *          GpiSetGraphicsField
 *          GpiMove
 *          GpiResetPS
 *          MapAndWriteString
 *          gpi_grid
 *          fill_page
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the default viewing limits to be a 6-inch x 6-inch square
 *      Set up the current viewing limits to be a 4-inch x 3-inch square
 *      Draw a dark pattern-filled, page-size box to show the size of
 *          the current viewing limits
 *      Call GpiResetPS which will reset the viewing limits to the default
 *      Draw a light pattern-filled, page-size box to show the size of
 *          the default viewing limits
 *      Set up the graphics field to overlap the viewing limits and the
 *          default viewing limits
 *      Draw the test grid (1/2") to show the size of the graphics field
 *          in conjunction with the current viewing limits
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiViewApp(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
    RECTL   rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiViewApp - Entry\n");

        error_count = 0;

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        /*
         * Set the default viewing limit to be a 6-inch x 6-inch square
         * from coordinates (1",1") to (7",7"), and print the name of
         * the called routine -- GpiSetDefViewingLimits.
         */
        rectangle.xLeft =   delta_x;
        rectangle.yBottom = delta_y;
        rectangle.xRight=   7 * delta_x;
        rectangle.yTop =    7 * delta_y;
        MapRect(rectangle);
        if (GpiSetDefViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetDefViewingLimits", TRUE, "GpiViewApp");
            return;
        }
        MapAndWriteString(hPS, (1 * delta_x),
                         ((6 * delta_y) + (delta_y / 4)),
                         "(1) GpiSetDefViewingLimits Called (default limit = 6\" x 6\").");

        /*
         * Set up the current viewing limit to be a 4-inch x 3-inch square
         * from coordinates (2",3") to (6",6") (smaller than the default),
         * and print the name of the routine -- GpiSetViewingLimits.
         */
        rectangle.xLeft =   2 * delta_x;
        rectangle.yBottom = 3 * delta_y;
        rectangle.xRight=   6 * delta_x;
        rectangle.yTop =    6 * delta_y;
        MapRect(rectangle);
        if (GpiSetViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetViewingLimits", TRUE, "GpiViewApp");
            return;
        }
        MapAndWriteString(hPS, (2 * delta_x), ((5 * delta_y) + (line_space * 3)),
                         "(2) GpiSetViewingLimits Called");
        MapAndWriteString(hPS, (2 * delta_x), ((5 * delta_y) + (line_space * 2)),
                         "    (limit = 4\" x 3\").");

        /*
         * Draw a pattern-filled box that would cover the page if no clipping
         * is performed.  This will show the size of the current viewing
         * limits.  (Box is filled with medium density pattern.)
         */
        fill_page(hPS, PATSYM_DENSE7);

        /*
         * Call GpiResetPS which will reset the viewing limits to the default,
         * and print the name of the routine -- GpiResetPS.
         */
        if (GpiResetPS(hPS, GRES_ATTRS) == FALSE) {
            BADERROR("GpiResetPS", TRUE, "GpiViewApp");
            return;
        }
        MapAndWriteString(hPS, delta_x, (delta_y + (delta_y / 4)),
                         "(3) The default viewing limit has been reset.");

        /*
         * Draw a pattern-filled box that would cover the page if no clipping
         * is performed.  This will show the size of the default viewing
         * limits.  (Box is filled with light density pattern.)
         */
        fill_page(hPS, PATSYM_DENSE8);

        /*
         * Set up the graphics field to overlap the viewing limit and the
         * default viewing limit.  The graphics field will be a 3.5-inch X
         * 7-inch rectangle from coordinates (1.5",1.5") to (8.5",5").
         */
        rectangle.xLeft =   delta_x + (delta_x / 2);
        rectangle.yBottom = delta_y + (delta_y / 2);
        rectangle.xRight=   (8 * delta_x) + (delta_x / 2);
        rectangle.yTop =    5 * delta_y;
        MapRect(rectangle);
        if (GpiSetGraphicsField(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetGraphicsField", TRUE, "GpiViewApp");
            return;
        }
        MapAndWriteString(hPS, (delta_x + (delta_x / 2)), (2 * delta_y),
                         "(4) GpiSetGraphicsField Called");
        MapAndWriteString(hPS, (delta_x + (delta_x / 2)), ((2 * delta_y) - line_space),
                         "    (field = 3.5\" x 7\", but def. viewing");
        MapAndWriteString(hPS, (delta_x + (delta_x / 2)), ((2 * delta_y) - (line_space * 2)),
                         "    limit makes this 3.5\" x 5.5\").");

        /*
         * Draw a the test grid (1/2") that would cover the page if no
         * clipping is performed.  This will show the size of the
         * graphics field in conjunction with the viewing limit.
         */
        gpi_grid(hPS, (LONG) HALFINCH);

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiViewApp                                                     -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiViewApp                                                     -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiViewApp - Entry\n");

    }  /* end else */

}  /* End GpiViewApp */


/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetDefViewingLimitsExh1()
 *
 * Purpose: This PTT entry point begins the first exhaustive test for
 *          GpiSetDefViewingLimits.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               11/28/90, PDVT, Hal Bray
 *                     - developed test cases
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetDefViewingLimits
 *          GpiMove
 *          GpiQueryDevice
 *          GpiAssociate
 *          MapAndWriteString
 *          gpi_grid
 *          fill_page
 *
 * Limitations: This test must be run in a Normal Presentation Space
 *              because 'GpiAssociate' is called to re-associate the device
 *              context (which will reset the viewing limit to the default),
 *              and this can not be done in a Micro PS.
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the default viewing limits to be a 5" x 3" rectangle
 *      Print a message that GpiSetDefViewingLimits was called
 *      Draw the test grid (1") to show the size of the default viewing
 *          limits is full page view (no clipping)
 *      Release the current device context and re-associate it with current
 *          device (this will reset the viewing limit to the default)
 *      Draw a filled, page-size box to show the size of the new default
 *          viewing limits
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetDefViewingLimitsExh1(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
    RECTL   rectangle;
    HDC     hDC;                // Handle to the device context

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetDefViewingLimitsExh1 - Entry\n");

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        error_count = 0;

        /*
         * Print a message to the PS and to the log that a Normal PS
         * is needed for this test.
         */
        MapAndWriteString(hPS, (2 * delta_x), ((5 * delta_y) + (delta_y / 2)),
                         "The presentation space MUST be Normal !");
        CWRITELOG(L_LOTRACE,0,"The presentation space MUST be Normal !\n");

        //  Print the initial size of the default viewing limit on the PS.
        MapAndWriteString(hPS, (2 * delta_x), ((4 * delta_y) + (delta_y / 2)),
                         "Initial default viewing limit is no clipping.");

        /*
         * Set the default viewing limit to be a 5-inch x 3-inch square
         * from coordinates (2",1") to (7",4").
         */
        rectangle.xLeft =   2 * delta_x;
        rectangle.yBottom = 1 * delta_y;
        rectangle.xRight =  7 * delta_x;
        rectangle.yTop =    4 * delta_y;
        MapRect(rectangle);
        if (GpiSetDefViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetDefViewingLimits", TRUE, "GpiSetDefViewingLimitsExh1");
            return;
        }

        //  Print a message on the PS that GpiSetDefViewingLimits was called.
        MapAndWriteString(hPS, (2 * delta_x), ((4 * delta_y) - (delta_y / 2)),
                         "(1) GpiSetDefViewingLimits was called.");

        /*
         * Draw a the test grid (1/2") that would cover the page if no
         * clipping is performed.  The initial default viewing limit is
         * no clipping.  Although 'GpiSetDefViewingLimits' was called, the
         * default has not been changed yet.
         */
        gpi_grid(hPS, (LONG) HALFINCH);

        /********************************************************************
         * Reassociate the device context for this PS inorder to reset the
         * viewing limits to the default values.  First, call GpiQueryDevice
         * to get the handle to the device context.  Next, call GpiAssociate
         * (passing it a NULL device context handle) to release the current
         * device context.  This must be done before a new device context can
         * be re-associated with this PS.  Finally, call GpiAssociate again
         * to re-associate it with the device context.
         ********************************************************************/

        //  Get the handle to the device context.
        if ((hDC = GpiQueryDevice(hPS)) == HDC_ERROR) {
            BADERROR("GpiQueryDevice", TRUE, "GpiSetDefViewingLimitsExh1");
            return;
        }

        //  Release the current device context.
        if (GpiAssociate(hPS, NULLHANDLE) == FALSE) {
            BADERROR("GpiAssociate", TRUE, "GpiSetDefViewingLimitsExh1");
            return;
        }

        //  Re-associate the device context with the original hDC.
        if (GpiAssociate(hPS, hDC) == FALSE) {
            BADERROR("GpiAssociate", TRUE, "GpiSetDefViewingLimitsExh1");
            return;
        }

        //  Print a message on the PS that GpiAssociate was called again.
        MapAndWriteString(hPS, (2 * delta_x), (3 * delta_y),
                         "(2) GpiAssociate was called");
        MapAndWriteString(hPS, (2 * delta_x), ((3 * delta_y) - line_space),
                         "   with new hDC.");

        /*
         * Draw a pattern-filled box that would cover the page if no clipping
         * is performed.  This will show the size of the default viewing
         * limits.  (Box is filled with light density pattern.)
         */
        fill_page(hPS, PATSYM_DENSE8);

        //  Print the size of the viewing limit on the PS.
        MapAndWriteString(hPS, (2 * delta_x), ((3 * delta_y) - (line_space * 4)),
                         "New default viewing limit is 5\" x 3\"");
        MapAndWriteString(hPS, (2 * delta_x), ((3 * delta_y) - (line_space * 5)),
                         "   at (2\",1\") (7\",4\").");

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetDefViewingLimitsExh1                                     -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetDefViewingLimitsExh1                                     -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetDefViewingLimitsExh1 - Exit.\n");

    }  /* end else */

}  /* End GpiSetDefViewingLimitsExh1 */


/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetDefViewingLimitsExh2()
 *
 * Purpose: This PTT entry point begins the second exhaustive test for
 *          GpiSetDefViewingLimits.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               11/28/90, PDVT, Hal Bray
 *                     - developed test cases
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetDefViewingLimits
 *          GpiMove
 *          GpiSetDrawingMode
 *          GpiSetInitialSegmentAttrs
 *          GpiOpenSegment
 *          GpiCloseSegment
 *          GpiSetDrawingMode
 *          GpiDrawChain
 *          MapAndWriteString
 *          gpi_grid
 *          fill_page
 *
 * Limitations: This test must be run in a Normal Presentation Space because
 *              GPI routines are called to create and draw a chained segment
 *              (which will reset the viewing limit to the default),
 *              and this can not be done in a Micro PS.
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the default viewing limits to be a 5" x 3" rectangle
 *      Print a message that GpiSetDefViewingLimits was called
 *      Draw the test grid (1") to show the size of the default viewing
 *          limits is full page view (no clipping)
 *      Open a chained segment
 *      Close the segment
 *      Draw a filled, page-size box to show the size of the new default
 *          viewing limits
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetDefViewingLimitsExh2(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
    POINTL  corner_point, cur_point;
    RECTL   rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetDefViewingLimitsExh2 - Entry\n");

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        error_count = 0;

        /*
         * Print a message to the PS and to the log that a Normal PS
         * is needed for this test.
         */
        MapAndWriteString(hPS, (2 * delta_x), ((5 * delta_y) + (delta_y / 2)),
                         "The presentation space MUST be Normal !");
        CWRITELOG(L_LOTRACE,0,"The presentation space MUST be Normal !\n");

        //  Print the initial size of the default viewing limit on the PS.
        MapAndWriteString(hPS, (2 * delta_x), ((4 * delta_y) + (delta_y / 2)),
                         "Initial default viewing limit is no clipping.");

        /*
         * Set the default viewing limit to be a 5-inch x 3-inch square
         * from coordinates (2",1") to (7",4").
         */
        rectangle.xLeft =   2 * delta_x;
        rectangle.yBottom = 1 * delta_y;
        rectangle.xRight =  7 * delta_x;
        rectangle.yTop =    4 * delta_y;
        MapRect(rectangle);
        if (GpiSetDefViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetDefViewingLimits", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        //  Print a message on the PS that GpiSetDefViewingLimits was called.
        MapAndWriteString(hPS, (2 * delta_x), ((4 * delta_y) - (delta_y / 2)),
                         "(1) GpiSetDefViewingLimits was called.");

        /*
         * Draw a the test grid (1/2") that would cover the page if no
         * clipping is performed.  The initial default viewing limit is
         * no clipping.  Although 'GpiSetDefViewingLimits' was called, the
         * default has not been changed yet.
         */
        gpi_grid(hPS, (LONG) HALFINCH);

        /*******************************************************************
         * When the drawing of a chained segment begins or end, the viewing
         * limits are reset to their default values.  To show this, the
         * drawing mode is set to retain, a chained segment is opened, a
         * small box is drawn, and the segment is closed to create the
         * segment.  The segment is then drawn by setting the drawing mode
         * to draw and calling GpiDrawChain.
         ********************************************************************/

        // Set drawing mode to retain.
        if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE) {
            BADERROR("GpiSetDrawingMode", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        // Turn on the chained attribute & turn off the fastchain attribute.
        if (GpiSetInitialSegmentAttrs(hPS, ATTR_CHAINED, ATTR_ON) == FALSE) {
            BADERROR("GpiSetInitialSegmentAttrs", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }
        if (GpiSetInitialSegmentAttrs(hPS, ATTR_FASTCHAIN, ATTR_OFF) == FALSE) {
            BADERROR("GpiSetInitialSegmentAttrs", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        //  Open a segment.
        if (GpiOpenSegment(hPS, SEGMENT_ID1) == FALSE) {
            BADERROR("GpiOpenSegment", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        // Draw a 1" x 1" box at (1",1") (2",2").
        cur_point.x = delta_x;
        cur_point.y = delta_y;
        MapPt(cur_point);
        GpiMove(hPS, &cur_point);
        corner_point.x = 2 * delta_x;
        corner_point.y = 2 * delta_y;
        MapPt(corner_point);
        GpiBox(hPS, DRO_OUTLINE, &corner_point, 0L, 0L);

        //  Close the segment.
        if (GpiCloseSegment(hPS) == FALSE) {
            BADERROR("GpiCloseSegment", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        // Set drawing mode to retain.
        if (GpiSetDrawingMode(hPS, DM_DRAW) == FALSE) {
            BADERROR("GpiSetDrawingMode", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        //  Draw the chained segment.
        if (GpiDrawChain(hPS) == FALSE) {
            BADERROR("GpiDrawChain", TRUE, "GpiSetDefViewingLimitsExh2");
            return;
        }

        //  Print a message on the PS that GpiDrawChain was called.
        MapAndWriteString(hPS, (2 * delta_x), (3 * delta_y),
                         "(2) GpiDrawChain was called.");

        /*
         * Draw a pattern-filled box that would cover the page if no clipping
         * is performed.  This will show the size of the default viewing
         * limits.  (Box is filled with light density pattern.)
         */
        fill_page(hPS, PATSYM_DENSE8);

        //  Print the size of the viewing limit on the PS.
        MapAndWriteString(hPS, (2 * delta_x), (2 * delta_y),
                         "New default viewing limit is 5\" x 3\"");
        MapAndWriteString(hPS, (2 * delta_x), ((2 * delta_y) - line_space),
                         "   at (2\",1\") (7\",4\").");

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetDefViewingLimitsExh2                                     -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetDefViewingLimitsExh2                                     -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetDefViewingLimitsExh2 - Exit.\n");

    }  /* end else */

}  /* End GpiSetDefViewingLimitsExh2 */

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetViewingLimitsExh1()
 *
 * Purpose: This entry point begins the first exhaustive test for
 *          GpiSetViewingLimitsExh.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                      - initial creation
 *               12/04/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetViewingLimits
 *          GpiMove
 *          GpiSetDrawingMode
 *          GpiSetInitialSegmentAttrs
 *          GpiOpenSegment
 *          GpiCloseSegment
 *          GpiSetDrawingMode
 *          GpiCallSegmentMatrix
 *          MapAndWriteString
 *          gpi_grid
 *          fill_page
 *
 * Limitations: This test must be run in a Normal Presentation Space because
 *              GPI routines are called to create and draw a called segment,
 *              and this can not be done in a Micro PS.
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the viewing limit to be a 6" x 6" square
 *      Turn off the 'chained' segment attribute
 *      Open a called segment
 *      Set a smaller viewing limit for this segment
 *      Draw a filled, page-size box to show the size of the viewing limits
 *      Close the segment
 *      Draw the called segment by calling GpiCallSegmentMatrix
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetViewingLimitsExh1(HPS hPS,HWND hWndClient, BOOL SelectionCall)
{
    RECTL       rectangle;
    MATRIXLF    transform_matrix;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetViewingLimitsExh1 - Entry\n");

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        error_count = 0;

        /*
         * Print a message to the PS and to the log that a Normal PS
         * is needed for this test.
         */
        MapAndWriteString(hPS, (2 * delta_x), ((5 * delta_y) + (delta_y / 2)),
                         "The presentation space MUST be Normal !");
        CWRITELOG(L_LOTRACE,0,"The presentation space MUST be Normal !\n");

        /*
         * Set the viewing limit to be a 6-inch x 6-inch square
         * from coordinates (1",1") to (7",7").
         */
        rectangle.xLeft =   delta_x;
        rectangle.yBottom = delta_y;
        rectangle.xRight =  7 * delta_x;
        rectangle.yTop =    7 * delta_y;
        MapRect(rectangle);
        if (GpiSetViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetViewingLimits", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        //  Print a message on the PS that GpiSetViewingLimits was called.
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) + (line_space * 3)),
                         "(1) GpiSetViewingLimits was called (in the");
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) + (line_space * 2)),
                         "    root segment).");

        //  Print the size of the viewing limit on the PS.
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) - line_space),
                         "Root segment viewing limit is 6\" x 6\" at");
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) - (line_space * 2)),
                         "    (1\",1\") (7\",7\").");

        /*
         * Draw a the test grid (1") that would cover the page if no
         * clipping is performed.  This will show the size of the viewing
         * limit set for the root segment.
         */
        gpi_grid(hPS, (LONG) INCH);

        /*******************************************************************
         * Viewing limits specified in called segments override those set in
         * the root segment.  To show this, the drawing mode is set to retain,
         * the chained attribute is turned off, a segment is opened, a small
         * box viewing limit is set, the fill_page routine is called to show
         * the size of the limits, and then the segment is closed.  This
         * creates the called segment.  The segment is then drawn by setting
         * the drawing mode to draw and calling GpiCallSegmentMatrix.
         ********************************************************************/

        // Set drawing mode to retain.
        if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE) {
            BADERROR("GpiSetDrawingMode", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        // Turn off the chained attribute.
        if (GpiSetInitialSegmentAttrs(hPS, ATTR_CHAINED, ATTR_OFF) == FALSE) {
            BADERROR("GpiSetInitialSegmentAttrs", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        //  Open a segment.
        if (GpiOpenSegment(hPS, SEGMENT_ID2) == FALSE) {
            BADERROR("GpiOpenSegment", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        /*
         * Set the viewing limit to be a 5-inch x 4-inch square
         * from coordinates (1",1") to (6",5") for this segment.
         */
        rectangle.xLeft =   delta_x;
        rectangle.yBottom = delta_y;
        rectangle.xRight =  6 * delta_x;
        rectangle.yTop =    5 * delta_y;
        MapRect(rectangle);
        if (GpiSetViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetViewingLimits", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        /*
         * Draw a pattern-filled box that would cover the page if no clipping
         * is performed.  This will show the size of the viewing
         * limits.  (Box is filled with light density pattern.)
         */
        fill_page(hPS, PATSYM_DENSE8);

        //  Close the segment.
        if (GpiCloseSegment(hPS) == FALSE) {
            BADERROR("GpiCloseSegment", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        // Set drawing mode to retain.
        if (GpiSetDrawingMode(hPS, DM_DRAW) == FALSE) {
            BADERROR("GpiSetDrawingMode", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        //  Draw the called segment.
        if (GpiCallSegmentMatrix(hPS, SEGMENT_ID2, 0L,
                                 &transform_matrix,
                                 TRANSFORM_REPLACE) == GPI_ERROR) {
            BADERROR("GpiCallSegmentMatrix", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        //  Print a message on the PS that GpiSetViewingLimits was called.
        MapAndWriteString(hPS, delta_x, (3 * delta_y),
                         "(2) GpiSetViewingLimits was called");
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - line_space),
                         "    (in the called segment).");

        //  Print the size of the viewing limit on the PS.
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 3)),
                         "Called segment viewing limit is 5\" x 4\"");
        MapAndWriteString(hPS, delta_x, ((3 * delta_y) - (line_space * 4)),
                         "   at (1\",1\") (6\",5\").");

        //  Print a message on the PS that the called segment was drawn.
        MapAndWriteString(hPS, delta_x, ((2 * delta_y) - (delta_y / 2)),
                         "(3) Called segment was drawn.");

        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetViewingLimitsExh1                                        -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetViewingLimitsExh1                                        -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetViewingLimitsExh1 - Exit.\n");

    }  /* end else */

}  /* End GpiSetViewingLimitsExh1 */

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetViewingLimitsExh2()
 *
 * Purpose: This entry point begins the second exhaustive test for
 *          GpiSetViewingLimitsExh.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                      - initial creation
 *               12/04/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetViewingLimits
 *          MapAndWriteString
 *          fill_page
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set up a null rectangle with the left boundary greater than the right
 *      Call GpiSetViewingLimits
 *      Draw a filled, page-size box to show the size of the viewing limits
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetViewingLimitsExh2(HPS hPS,HWND hWndClient, BOOL SelectionCall)
{
    RECTL       rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetViewingLimitsExh2 - Entry\n");

        set_delta(hPS);     // Call set_delta to init delta_x & delta_y

        error_count = 0;

        /********************************************************************
         * Print a message on the PS that GpiSetViewingLimits will be called
         * with a null rectangle and a pattern-filled box (that would cover
         * the page if no clipping were performed) will be drawn.  This will
         * show that the entire page is clipped.
         ********************************************************************/
        MapAndWriteString(hPS, delta_x, (4 * delta_y),
                         "In this test, GpiSetViewingLimits will be called");
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) - line_space),
                         "with a NULL rectangle and then a page-sized, pattern-");
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) - (line_space * 2)),
                         "filled box will be drawn.  This will show that all");
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) - (line_space * 3)),
                         "points are clipped.  (This text and the test signature");
        MapAndWriteString(hPS, delta_x, ((4 * delta_y) - (line_space * 4)),
                         "should be the only output -- the box is clipped.)");

        /********************************************************************
         * Call GpiSetViewingLimits with a null rectangle which will clip
         * all points.
         ********************************************************************/

        //  Create a rectangle with the left coordinate greater than the right.
        rectangle.xLeft =   3 * delta_x;
        rectangle.yBottom = 3 * delta_y;
        rectangle.xRight =  2 * delta_x;
        rectangle.yTop =    2 * delta_y;
        MapRect(rectangle);

        if (GpiSetViewingLimits(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetViewingLimits", TRUE, "GpiSetViewingLimitsExh1");
            return;
        }

        /*
         * Draw a pattern-filled box that would cover the page if no clipping
         * is performed.  Nothing will be drawn which will show that the
         * entire page is clipped.
         */
        fill_page(hPS, PATSYM_DENSE8);

        //  Write the proper log messages.
        if ( ! error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetViewingLimitsExh2                                        -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetViewingLimitsExh2                                        -    FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetViewingLimitsExh2 - Exit.\n");

    }
}  /* end GpiSetViewingLimitsExh2 */

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetPageViewportExh()
 *
 * Purpose: This entry point begins exhaustive testing of GpiSetPageViewport.
 *
 * System Requirements: Section 5.5.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                      - initial creation
 *               12/04/90, PDVT, Hal Bray
 *                      - implementation of testcase design
 *
 * Inputs:  hPS           - presentation space handle
 *          hWndClient    - client window handle
 *          SelectionCall - selection indicator
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiSetPageViewport
 *          MapAndWriteString
 *          gpi_grid
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the page viewport to be a rectangle with the width that is one
 *           half the height
 *      Draw a test grid of 1-inch squares to show the size and
 *          transformation of the presentation page to the page viewport
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetPageViewportExh(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
    RECTL   rectangle;

    if (SelectionCall) {
        WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
        *pBasisX = 10000;
        *pBasisY = 10000;
    }
    else {
        /*
         * Display signature to output device & write signature to log file.
         */
        TEST_SIGNATURE(0,0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetPageViewportExh - Entry\n");

        error_count = 0;

        set_delta(hPS);      // Call set_delta to init delta_x & delta_y

        /********************************************************************
         * Set the page viewport to be a rectangle with the width that is
         * one half the height.  This rectangle is not mapped because any
         * page unit that is selected is transformed into pixels by the
         * page viewport.  The transformations of the presentation page
         * space to the device space via the page viewport is called the
         * "Device Transform".  Both scaling and translation takes place
         * during this transform.  This changes the page units (LOENGLISH,
         * HIMETRIC, TWIPS, etc.) into device units (pixels) by multiplying
         * them then by a factor that relates those units to the pixel size.
         * (This is called scaling -- translation also takes place).
         ********************************************************************/
        rectangle.xLeft   = 240;
        rectangle.yBottom = 120;
        rectangle.xRight  = 600;
        rectangle.yTop    = 840;
        if (GpiSetPageViewport(hPS, &rectangle) == FALSE) {
            BADERROR("GpiSetPageViewport", TRUE, "GpiSetPageViewportExh");
            return;
        }

        //  Print a message on the PS that GpiSetPageViewport was called.
        MapAndWriteString(hPS, 0L, (3 * delta_y),
                         "GpiSetPageViewport was called.");

        /*
         * Draw a test grid of 1-inch squares to show the transformations
         * (scaling and translation) of the presentation page space to the
         * device space via the page viewport.
         */
        gpi_grid(hPS, (LONG) INCH);
        MapAndWriteString(hPS, 0L, (2 * delta_y),
                         "Grid squares are 1\" x 1\" before the page");
        MapAndWriteString(hPS, 0L, ((2 * delta_y) - (delta_y / 3)),
                         "   viewport sets the device transform.");

        //  Write the proper log messages.
        if (!error_count) {
            CWRITELOG(L_HDR,0,
"$GpiSetPageViewportExh                                          -    PASS.\n");
        }
        else {
            sprintf(err_log_line,"$ERRORCOUNT = %d.\n", error_count);
            CWRITELOG(L_HDR,0, err_log_line);
            CWRITELOG(L_HDR,0,
"$GpiSetPageViewportExh                                         -     FAIL.\n");
        }

        CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetPageViewportExh - Exit.\n");

    }  /* end else */

}  /* End GpiSetPageViewportExh */

/****************************************************************************
 *\\ddd
 * Routine Name: fill_page()
 *
 * Purpose: This routine draws a pattern-filled box that will cover
 *          the page if no clipping is performed to show the size of
 *          the viewing area.
 *
 * System Requirements: NONE
 *
 * Revision Log: 12/07/90, PDVT, Hal Bray
 *                      - original implementation
 *
 * Inputs:  hPS         - presentation space handle
 *          pattern     - pattern symbol
 *
 * Outputs: NONE
 *
 * Subroutines Required:
 *          GpiBox
 *          GpiMove
 *          GpiSetPattern
 *          MapPt
 *
 * Limitations: NONE
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *      Set the fill pattern with the given pattern symbol
 *      Move the current point to the origin (0,0)
 *      Draw a filled box to the upper right-hand corner
 *\\end
 ***************************************************************************/
VOID fill_page(HPS hPS, LONG pattern)
{
    POINTL  cur_point;
    POINTL  corner_point;

    /********************************************************************
     * Draw a pattern-filled box that will cover the page to show the
     * clipping if it is performed.
     ********************************************************************/
    GpiSetPattern(hPS, pattern);
    cur_point.x = 0;
    cur_point.y = 0;
    MapPt(cur_point);
    GpiMove(hPS, &cur_point);
    corner_point.x = pg_size_x * delta_x;
    corner_point.y = pg_size_y * delta_y;
    MapPt(corner_point);
    GpiBox(hPS, DRO_FILL, &corner_point, 0L, 0L);

}  /* end fill_page */


