#define MAIN_PROGRAM

#define INCL_DOSPROCESS
#define INCL_DEV      // include all DEV definitions
#define INCL_WIN      // include Win definitions
#define INCL_GPI      // include all Gpi definitions
#define INCL_ERRORS   // include all Gre error definitions
#include <string.h>
#include <stdlib.h>
#include <os2.h>
#include <stdio.h>
#include <limits.h>
#include "PTTDLLDF.h"  // test case defines.
#include "common1.h"
#include "common2.h"
#include "gpiseg.h"   // Test Definition File.

/****************************************************************************
 *\\ddd
 * Subtask Name: GREATT
 *
 * Module name: GPISEG.C
 *
 * Purpose: This module contains all common subroutines within the GPISEG
 *          dynamic link library.
 *
 * Revision Log:  9/3/91 - GREATT, Darin C. Hafer, Initial Creation
 *
 * SAD References:
 *
 * SRS References:
 *
 * Contents:  InitTest()
 *            CreateRoom()
 *            RandPels()
 *            OpenAndClose()
 *            GetLineSpacing()
 *            DrawFromOrChain1()
 *            DrawFromOrChain2()
 *            DrawSegment()
 *            DeleteSegment()
 *            LogEnvironment()
 *            Scribble()
 *            SetAndQueryLoop()
 *
 * Limitations:
 *
 *
 *\\end
****************************************************************************/

/****************************************************************************
 *\\ddd
 * Routine Name: InitTest()
 *
 * Purpose:  Initialize function names and address array.
 *
 * System Requirements:
 *
 * Revision Log:
 *
 *
 * Inputs:
 *   sel          // passed through to the InitTestSegVars function
 *   init_info_p  // name and address array used by ptt interface
 *
 * Outputs:
 *   init_info_p  // returned initialized with names and addresses
 *
 * Subroutines Required:
 *   InitTestSegVars()    // used by the ptt interface
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *        Get the number of tests.
 *   for the number of tests {
 *                      Get the test names;
 *                      Get the test addresses;
 *                      }
 *   call InitTestSegVars function
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY InitTest(PVOID Selector, PTESTINITINFO init_info_p)
{
  register USHORT usTestCounter;

  init_info_p->ntests = NTESTS;
  strcpy(init_info_p->group_name,TestCaseName); //Testcase Name

  for (usTestCounter=0 ;usTestCounter< NTESTS; ++usTestCounter){

     init_info_p->test_addrs[usTestCounter]=
                               EntryPointList[usTestCounter].TestAddress;
     strcpy(init_info_p->test_names[usTestCounter],
                               EntryPointList[usTestCounter].TestName);
  }

  InitTestSegVars(Selector);
  return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: CreateRoom()
 *
 * Purpose:  This subroutine will create three segments.  Segment 1 is a lamp
 *           shade, Segment 2 is the Lamp base, and Segment 3 is the table.
 *           The lamp base is a called segment from Segment 3.
 *
 * System Requirements:
 *
 * Revision Log: 8/22/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     EntryName - Name of entry point
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   Open LAMPSHADE to create the lamp shade, Close it.
 *   Open LAMPBASE to create the lamp base, Close it.
 *   Turn the CHAINED attribute off for Seg2.
 *   Open TABLE to create the table.
 *   Call LAMPBASE from within TABLE.
 *   Close TABLE.
 * }
 *\\end
 ***************************************************************************/
INT CreateRoom(HPS hPS, HWND hWndClient)
{
   POINTL CurPos, aptArc[35];
   MATRIXLF segmat;
   LONG   Count;

  Count=ErrorCount;

  /*---------------------------------------\
  |  Open Segment 1 to draw the lamp shade |
  \---------------------------------------*/

  if (!GpiOpenSegment(hPS, LAMPSHADE)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiOpenSegment() LAMPSHADE", NULL);
  }
  GpiSetColor(hPS, CLR_RED);
  GpiBeginArea(hPS, BA_ALTERNATE|BA_BOUNDARY);
  CurPos.x = MapX(140);
  CurPos.y = MapY(370);
  GpiMove(hPS, &CurPos);
  CurPos.x = MapX(260);
  GpiLine(hPS, &CurPos);
  CurPos.x = MapX(320);
  CurPos.y = MapY(300);
  GpiLine(hPS, &CurPos);
  CurPos.x = MapX(80);
  GpiLine(hPS, &CurPos);
  CurPos.x = MapX(140);
  CurPos.y = MapY(370);
  GpiLine(hPS, &CurPos);
  GpiEndArea(hPS);
  if (!GpiCloseSegment(hPS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiCloseSegment() LAMPSHADE", NULL);
  }

  /*---------------------------------------\
  |  Open Segment 2 to draw the lamp base  |
  \---------------------------------------*/

  if (!GpiOpenSegment(hPS, LAMPBASE)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiOpenSegment() LAMPBASE", NULL);
  }
  GpiSetColor(hPS, CLR_BLUE);
  GpiSetPattern(hPS, PATSYM_HALFTONE);
  GpiBeginArea(hPS, BA_ALTERNATE|BA_BOUNDARY);
  CurPos.x = MapX(220);
  CurPos.y = MapY(299);
  GpiMove(hPS, &CurPos);
  CurPos.x = MapX(180);
  GpiLine(hPS, &CurPos);
  aptArc[0].x = MapX(190);
  aptArc[0].y = MapY(295);
  aptArc[1].x = MapX(190);
  aptArc[1].y = MapY(285);
  aptArc[2].x = MapX(180);
  aptArc[2].y = MapY(280);
  aptArc[3].x = MapX(160);
  aptArc[3].y = MapY(275);
  aptArc[4].x = MapX(160);
  aptArc[4].y = MapY(265);
  aptArc[5].x = MapX(180);
  aptArc[5].y = MapY(260);
  aptArc[6].x = MapX(190);
  aptArc[6].y = MapY(245);
  aptArc[7].x = MapX(180);
  aptArc[7].y = MapY(230);
  aptArc[8].x = MapX(140);
  aptArc[8].y = MapY(225);
  aptArc[9].x = MapX(140);
  aptArc[9].y = MapY(205);
  aptArc[10].x =MapX(180);
  aptArc[10].y =MapY(200);
  aptArc[11].x =MapX(180);
  aptArc[11].y =MapY(185);
  aptArc[12].x =MapX(160);
  aptArc[12].y =MapY(170);
  GpiPolyFillet(hPS, 13L, &aptArc[0]); /* Draw left side of base */

  CurPos.x = MapX(240);
  CurPos.y = MapY(170);
  GpiLine(hPS, &CurPos);
  aptArc[0].x = MapX(220);
  aptArc[0].y = MapY(185);
  aptArc[1].x = MapX(220);
  aptArc[1].y = MapY(200);
  aptArc[2].x = MapX(260);
  aptArc[2].y = MapY(205);
  aptArc[3].x = MapX(260);
  aptArc[3].y = MapY(225);
  aptArc[4].x = MapX(220);
  aptArc[4].y = MapY(230);
  aptArc[5].x = MapX(210);
  aptArc[5].y = MapY(245);
  aptArc[6].x = MapX(220);
  aptArc[6].y = MapY(260);
  aptArc[7].x = MapX(240);
  aptArc[7].y = MapY(265);
  aptArc[8].x = MapX(240);
  aptArc[8].y = MapY(275);
  aptArc[9].x = MapX(220);
  aptArc[9].y = MapY(280);
  aptArc[10].x =MapX(210);
  aptArc[10].y =MapY(285);
  aptArc[11].x =MapX(210);
  aptArc[11].y =MapY(295);
  aptArc[12].x =MapX(220);
  aptArc[12].y =MapY(299);
  GpiPolyFillet(hPS, 13L, &aptArc[0]); /* Draw right side of base */
  GpiEndArea(hPS);
  if (!GpiCloseSegment(hPS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiCloseSegment() LAMPBASE", NULL);
  }

  /* Make Seg2 unchained so we can call it */
  if (!GpiSetSegmentAttrs(hPS, LAMPBASE, ATTR_CHAINED, ATTR_OFF)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiSetSegmentAttrs() LAMPBASE", NULL);
  }

  /*---------------------------------------\
  |  Open Segment 3 to draw the table      |
  \---------------------------------------*/
  if (!GpiOpenSegment(hPS, TABLE)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiOpenSegment() TABLE", NULL);
  }
  GpiSetColor(hPS, CLR_GREEN);
  GpiSetPattern(hPS, PATSYM_HALFTONE);
  GpiBeginArea(hPS, BA_ALTERNATE|BA_BOUNDARY);
  aptArc[0].x = MapX(100);
  aptArc[0].y = MapY(200);
  aptArc[1].x = MapX(230);
  aptArc[1].y = MapY(230);
  aptArc[2].x = MapX(290);
  aptArc[2].y = MapY(170);
  aptArc[3].x = MapX(150);
  aptArc[3].y = MapY(130);
  GpiMove(hPS, &aptArc[3]);
  GpiPolyLine(hPS, 4L, &aptArc[0]);
  GpiEndArea(hPS);
  GpiSetColor(hPS, CLR_DARKGREEN);
  GpiSetPattern(hPS, PATSYM_DEFAULT);
  GpiBeginArea(hPS, BA_ALTERNATE|BA_BOUNDARY);
  GpiMove(hPS, &aptArc[0]);
  aptArc[0].x = MapX(100);
  aptArc[0].y = MapY(130);
  aptArc[1].x = MapX(110);
  aptArc[1].y = MapY(120);
  aptArc[2].x = MapX(120);
  aptArc[2].y = MapY(130);
  aptArc[3].x = MapX(120);
  aptArc[3].y = MapY(150);
  aptArc[4].x = MapX(140);
  aptArc[4].y = MapY(120);
  aptArc[5].x = MapX(140);
  aptArc[5].y = MapY(80);
  aptArc[6].x = MapX(150);
  aptArc[6].y = MapY(70);
  aptArc[7].x = MapX(160);
  aptArc[7].y = MapY(80);
  aptArc[8].x = MapX(160);
  aptArc[8].y = MapY(117);
  aptArc[9].x = MapX(270);
  aptArc[9].y = MapY(148);
  aptArc[10].x =MapX(270);
  aptArc[10].y =MapY(110);
  aptArc[11].x =MapX(280);
  aptArc[11].y =MapY(100);
  aptArc[12].x =MapX(290);
  aptArc[12].y =MapY(110);
  aptArc[13].x =MapX(290);
  aptArc[13].y =MapY(170);
  aptArc[14].x =MapX(150);
  aptArc[14].y =MapY(130);
  aptArc[15].x =MapX(100);
  aptArc[15].y =MapY(200);
  GpiPolyLine(hPS, 16L, &aptArc[0]);
  GpiEndArea(hPS);

  if (!GpiCallSegmentMatrix(hPS, LAMPBASE, 0L, &segmat, TRANSFORM_REPLACE)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiCallSegmentMatrix()", NULL);
  }
  if (!GpiCloseSegment(hPS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiCloseSegment() TABLE", NULL);
  }
  return (Count < ErrorCount)?0:1;
}

/****************************************************************************
 *\\ddd
 * Routine Name: RandPels()
 *
 * Purpose:  This subroutine will create several segments, each of which
 *           turns on a pel.
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     Limit - Total number of segments to create.
 *     Points- A RECTL structure that defines the box in which to draw the pels.
 *     Frills- A flag to indicate what options are needed.
 *
 * Outputs:
 *     TRUE/FALSE - if the Open or Close fails, return FALSE.
 *
 * Subroutines Required:
 *
 * Limitations:
 *     GpiLine - If the current position is the same as the endpoint of the
 *     line when you draw it, nothing gets printed to the printer, however,
 *     to the screen, a pel is turned on.  If you increment the distance to
 *     be 1, the pel is printed on the printer.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   For the number of segments requested {
 *      Open the segment.
 *      if Frills & 0x01
 *         Set random color.
 *      if Frills & 0x02
 *         Get a random point within the RECTL structure that was passed.
 *      else
 *         Get a random point for every 50th segment.
 *      Turn that pel/pixel on using GpiLine, not GpiSetPel.
 *      Close the segment.
 *   }
 * }
 *\\end
 ***************************************************************************/

BOOL RandPels(HPS hPS, HWND hWndClient, LONG Limit, RECTL * Points, BYTE Frills)
{
  POINTL  ptlPel;
  LONG    Color, Count;
  LONG    SegID;

  Count = ErrorCount;
  /*----------------------------*/
  /* Frills values:             */
  /*  0 = No Color AND '%50'    */
  /*  1 = Color AND '%50'       */
  /*  2 = No Color and NO '%50' */
  /*  3 = Color and NO '%50'     */
  /*----------------------------*/

  for (SegID=0L; SegID<=Limit; SegID++){
   GpiOpenSegment(hPS, (LONG)SegID);


     if (Frills & 0x01) {
       Color = randL(1L, 15L);
       GpiSetColor(hPS, Color);
     }
     if (Frills & 0x02) {
        ptlPel.x = randL(MapX(Points->xLeft), MapX(Points->xRight));
        ptlPel.y = randL(MapY(Points->yBottom), MapY(Points->yTop));
        GpiMove(hPS, &ptlPel);
        ptlPel.y += 1L;
     } else {
        if (!(SegID%50)) {
           ptlPel.x = randL(MapX(Points->xLeft), MapX(Points->xRight));
           ptlPel.y = randL(MapY(Points->yBottom), MapY(Points->yTop));
           GpiMove(hPS, &ptlPel);
           ptlPel.y += 1L;
        }
     }
     GpiLine(hPS, &ptlPel);

     GpiCloseSegment(hPS);
  }
  return (Count < ErrorCount) ? 0 : 1;
}

/****************************************************************************
 *\\ddd
 * Routine Name: OpenAndClose()
 *
 * Purpose:  This subroutine opens a segment in the current drawing mode,
 *           writes a string of text, and depending on CloseFlag, closes the
 *           segment.
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS      - handle to PS
 *     CloseFlag- A boolean: TRUE means to close the segment, otherwise
 *                leave it opened.
 *     SegID    - The segment identifer.
 *     szBuffer - The string to write.
 *     pPoint   - The coordinates of where to write the string.
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   Open a segment.
 *   Go to point (50, 200) and write a string of text.
 *   if (CloseFlag)
 *     Close the segment.
 * }
 *\\end
 ***************************************************************************/
BOOL OpenAndClose(HPS hPS, BOOL CloseFlag, LONG SegID, CHAR * szBuffer, PPOINTL pPoint)
{
 LONG Count;

 Count = ErrorCount;
 if (!GpiOpenSegment(hPS, SegID)) {
    ErrorCount++;
 }
 GpiCharStringAt(hPS, pPoint, strlen(szBuffer), szBuffer);
 if (CloseFlag)
    if (!GpiCloseSegment(hPS)) {
       ErrorCount++;
    }
 return (Count < ErrorCount) ? 0 : 1;
}

/****************************************************************************
 *\\ddd
 * Routine Name: GetLineSpacing()
 *
 * Purpose:  This subroutine returns the lMaxBaselineExt value of the
 *           current font.
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   Query the font metrics.
 *   return lMaxBaselineExt;
 * }
 *\\end
 ***************************************************************************/
SHORT GetLineSpacing(HPS hPS, HWND hWndClient)
{
 FONTMETRICS fm;

    if (GpiQueryFontMetrics(hPS,(LONG)sizeof(fm),&fm )==FALSE){
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiQueryFontMetrics()", " GetLineSpacing");
    }
    return (SHORT)(fm.lMaxBaselineExt+fm.lExternalLeading);
}

/****************************************************************************
 *\\ddd
 * Routine Name: DrawFromOrChain1()
 *
 * Purpose:  This subroutine tests the functionality of GpiDrawChain or
 *           GpiDrawFrom, depending on the Mode flag.
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     EntryName - Name of entry point
 *     Mode      - Flag to indicate which function to use
 *
 * Outputs:
 *
 * Subroutines Required:
 *     CreateRoom()
 *     RandPels()
 *     OpenAndClose()
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 * TEST - 1
 *   Set drawing mode to RETAIN.
 *   Call CreateRoom().
 *   Depending on the Mode flag, draw the segments using the appropriate
 *     function call.
 * TEST - 2
 *   Set drawing mode to RETAIN again to be sure.
 *   Call RandPels() with the flag set to 0x03.
 *   Depending on the Mode flag, draw the segments using the appropriate
 *     function call.
 * TEST - 3
 *   for (DrawMode=DM_RETAIN; DrawMode<=DM_DRAWANDRETAIN; DrawMode++) {
 *      Set the drawing mode.
 *      for (i=Seg1; i<=Seg5; i++) {
 *         Set initial segment attributes.
 *         if (i<Seg5)
 *            Call OpenAndClose with the TRUE flag.
 *         else {
 *            Call OpenAndClose with the FALSE flag.
 *            Depending on the Mode flag, draw the segments using the
 *              appropriate function call.
 *         }
 *      }
 *      Close segment 5.
 *      Delete Segments 1-5.
 *   }
 * }
 *\\end
 ***************************************************************************/
VOID DrawFromOrChain1(HPS hPS, HWND hWndClient, BYTE Mode)
{
 POINTL ptlPoint;
 RECTL  rclPel;
 LONG   i, DrawMode;


  ErrorCount = 0L;
  srand(1);

  SETENTER(" ", "TEST 1 - SETUP");
    if (!GpiSetDrawingMode(hPS, DM_RETAIN)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
    }
    if (!CreateRoom(hPS, hWndClient)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "CreateRoom()", NULL);
    }
  SETEXIT();

  SUBENTER(Draw the Furniture, NULL);
    if (Mode & CHAIN) {
       if (!GpiResetPS(hPS, GRES_ATTRS)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiResetPS()", NULL);
       }
       if (!GpiDrawChain(hPS)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "TEST 1 - GpiDrawChain()", NULL);
       }
    } else {
       if (!GpiResetPS(hPS, GRES_ATTRS)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiResetPS()", NULL);
       }
       if (!GpiDrawFrom(hPS, LAMPSHADE, TABLE)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "TEST 1 - GpiDrawFrom()", NULL);
       }
    }
  SUBEXIT();
    if (!GpiDeleteSegments(hPS, LAMPSHADE, TABLE)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiDeleteSegments()", NULL);
    }
    GpiSetColor(hPS, CLR_BLACK);
    strcpy(achLogLine, "<=== Lamp Base is called from a segment chain");
    MapAndWriteString(260L, 210L, achLogLine);
    strcpy(achLogLine, "Draw a Chain of pels/pixels");
    MapAndWriteString(30L, 600L, achLogLine);
    sprintf(achLogLine, "Call %s from segment 5", (Mode&CHAIN)?"GpiDrawChain":"GpiDrawFrom");
    MapAndWriteString(420L, 600L, achLogLine);
    strcpy(achLogLine, "while it is still open");
    MapAndWriteString(420L, 580L, achLogLine);

  SETENTER(" ", "TEST 2 - SETUP");
    if (!GpiSetDrawingMode(hPS, DM_RETAIN)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
    }
    rclPel.xLeft = 30L;
    rclPel.yBottom=500L;
    rclPel.xRight= 350L;
    rclPel.yTop  = 580L;
    RandPels(hPS, hWndClient, 800L, &rclPel, 0x03);
  SETEXIT();

  SUBENTER(Draw the Pels, NULL);
    if (Mode & CHAIN) {
       if (!GpiResetPS(hPS, GRES_ATTRS)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiResetPS()", NULL);
       }
       if (!GpiDrawChain(hPS)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "TEST 2 - GpiDrawChain()", NULL);
       }
    } else {
       if (!GpiResetPS(hPS, GRES_ATTRS)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiResetPS()", NULL);
       }
       if (!GpiDrawFrom(hPS, 1L, 800L)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "TEST 2 - GpiDrawFrom()", NULL);
       }
    }
  SUBEXIT();
  if (!GpiDeleteSegments(hPS, 1L, 800L)){
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiDeleteSegments()", NULL);
  }
  if (!GpiSetColor(hPS, CLR_BLACK)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiSetColor()", NULL);
  }

  ptlPoint.x = MapX(420L);
  ptlPoint.y = MapY(580L);

  SUBENTER(TEST 3 - Open and Close segments, NULL);
  for (DrawMode=DM_RETAIN; DrawMode<=DM_DRAWANDRETAIN; DrawMode++) {
     if (!GpiSetDrawingMode(hPS, DrawMode)) {
        ErrorCount++;
        ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
     }
     sprintf(achLogLine, "%s", DrawMode==DM_RETAIN?"DM_RETAIN":"DM_DRAWANDRETAIN");
     ERRSTATUS(FALSE, "DRAW MODE is  ", achLogLine);

     ErrorCount = 0L;
     for (i=Seg1; i<=Seg5; i++) {
        SetInitialSegmentAttrs(hPS, &SegAttrs[0]);
        sprintf(achLogLine, "Segment %ld in %s mode", i,
                   DrawMode==DM_RETAIN?"DM_RETAIN":"DM_DRAWANDRETAIN");
        ptlPoint.y-= syPosition;
        if (i<Seg5) {
           if (!OpenAndClose(hPS, TRUE, i, achLogLine, &ptlPoint))
              ERRSTATUS(TRUE, "OpenAndClose()", NULL);
        } else {
           if (!OpenAndClose(hPS, FALSE, i, achLogLine, &ptlPoint))
              ERRSTATUS(TRUE, "OpenAndClose()", NULL);
        }
     }
     ptlPoint.y-= syPosition;
     if (DrawMode==DM_RETAIN)
        strcpy(achLogLine, "This line was retained.");
     else
        strcpy(achLogLine, "This line was drawn and retained.");

     if (GpiCharStringAt(hPS, &ptlPoint,
                         strlen(achLogLine), achLogLine) != GPI_OK) {
        ErrorCount++;
        ERRSTATUS(TRUE, "GpiCharStringAt()", NULL);
     }
     if (!GpiCloseSegment(hPS)) {
        ErrorCount++;
        ERRSTATUS(TRUE, "GpiCloseSegment()", ltoa(Seg5, achLogLine, 10));
     }
     if (Mode & CHAIN) {
        if (!GpiResetPS(hPS, GRES_ATTRS)) {
           ErrorCount++;
           ERRSTATUS(TRUE, "GpiResetPS()", NULL);
        }
        if (!GpiDrawChain(hPS)) {
           ErrorCount++;
           ERRSTATUS(TRUE, "GpiDrawChain()", NULL);
        }
     } else {
        if (!GpiResetPS(hPS, GRES_ATTRS)) {
           ErrorCount++;
           ERRSTATUS(TRUE, "GpiResetPS()", NULL);
        }
        if (!GpiDrawFrom(hPS, Seg1, Seg5)) {
           ErrorCount++;
           ERRSTATUS(TRUE, "GpiDrawFrom()", NULL);
        }
     }
     if (!GpiDeleteSegments(hPS, Seg1, Seg5)) {
        ErrorCount++;
        ERRSTATUS(TRUE, "GpiDeleteSegments()", NULL);
     }
     ptlPoint.y-= syPosition;
  }
  SUBEXIT();
 return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: DrawFromOrChain2()
 *
 * Purpose:  This subroutine tests the functionality of GpiDrawChain or
 *           GpiDrawFrom, depending on the Mode flag.
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     EntryName - Name of entry point
 *     Mode      - Flag to indicate which function to use
 *
 * Outputs:
 *
 * Subroutines Required:
 *     CreateSegmentChain()
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   for (DrawMode=DM_RETAIN; DrawMode<=DM_DRAWANDRETAIN; DrawMode++) {
 *      for (i=Seg1; i<=Seg8; i++)
 *          make sure the Dynamic flag is set to off if in DRAWANDRETAIN.
 *          randomly set segment attributes to either on or off.
 *      }
 *      Call CreateSegmentChain().
 *      for (Erase=DCTL_ON; Erase>=DCTL_OFF; Erase--)
 *         for (Dynamic=DCTL_ON; Dynamic>=DCTL_OFF; Dynamic--) {
 *            Set the erase draw control.
 *            Set the dynamic draw control.
 *            Depending on the Mode flag, draw the segments using the
 *              appropriate function call.
 *         }
 *      Delete segments 1-8.
 *   }
 * }
 *\\end
 ***************************************************************************/
VOID DrawFromOrChain2(HPS hPS, HWND hWndClient, BYTE Mode)
{
 LONG i, Erase, Dynamic, DrawMode;
 SEGMENTATTRS TempAttrs[8];

  srand(1);
  memcpy((PVOID)TempAttrs, (PVOID)SegAttrs, sizeof(SegAttrs));

  for (DrawMode=DM_RETAIN; DrawMode<=DM_DRAWANDRETAIN; DrawMode++) {
     sprintf(achLogLine, "DrawMode is %s", DrawMode==DM_RETAIN?"DM_RETAIN":"DM_DRAWANDRETAIN");
     LOGINFO(L_DEBUG, "*", achLogLine);
     for (i=Seg1; i<=Seg8; i++) {
        if (DrawMode==DM_DRAWANDRETAIN)
           TempAttrs[i-1].Dynamic = ATTR_OFF;
        else
           TempAttrs[i-1].Dynamic = randL(0L, 1L);
        TempAttrs[i-1].Visible = randL(0L, 1L);
        TempAttrs[i-1].Detectable=randL(0L, 1L);
        TempAttrs[i-1].FastChain = randL(0L, 1L);
     }
     switch (DrawMode) {
       case DM_RETAIN:
         CreateSegmentChain(hPS, hWndClient, TempAttrs, Seg1, Seg8, TYPE_RETAIN);
         break;
       case DM_DRAWANDRETAIN:
         CreateSegmentChain(hPS, hWndClient, TempAttrs, Seg1, Seg8, TYPE_DRAWRETAIN);
         break;
     }
     SUBENTER(Draw the chain, NULL);
     for (Erase=DCTL_ON; Erase>=DCTL_OFF; Erase--) {
        for (Dynamic=DCTL_ON; Dynamic>=DCTL_OFF; Dynamic--) {
           if (!GpiSetDrawControl(hPS, DCTL_ERASE, Erase)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "GpiSetDrawControl() - ERASE", NULL);
           }
           if (!GpiSetDrawControl(hPS, DCTL_DYNAMIC, Dynamic)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "GpiSetDrawControl() - DYNAMIC", NULL);
           }
           if (Mode & CHAIN) {
              if (!GpiResetPS(hPS, GRES_ATTRS)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiResetPS()", NULL);
              }
              if (!GpiDrawChain(hPS)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiDrawChain()", NULL);
              }
           } else {
              if (!GpiResetPS(hPS, GRES_ATTRS)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiResetPS()", NULL);
              }
              if (!GpiDrawFrom(hPS, Seg1, Seg8)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiDrawFrom()", NULL);
              }
           }
        }
     }
     SUBEXIT();
     if (!GpiDeleteSegments(hPS, Seg1, Seg8)) {
        ErrorCount++;
        ERRSTATUS(TRUE, "GpiDeleteSegments()", NULL);
     }
     WriteText(Seg1, Seg8);
     GpiSetColor(hPS, CLR_BLACK);
  }
 return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: DrawSegment()
 *
 * Purpose:  This subroutine tests the functionality of GpiDrawSegment
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     EntryName - Name of entry point
 *
 * Outputs:
 *
 * Subroutines Required:
 *     CreateSegmentChain()
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   i=0;
 *   for (DrawMode=DM_RETAIN; DrawMode<=DM_DRAWANDRETAIN; DrawMode++) {
 *      for (ChainAttr    =DCTL_ON; ChainAttr    >=DCTL_OFF; ChainAttr--)
 *      for (DynamicAttr  =DCTL_ON; DynamicAttr  >=DCTL_OFF; DynamicAttr--)
 *      for (DetectAttr   =DCTL_ON; DetectAttr   >=DCTL_OFF; DetectAttr--)
 *      for (VisibleAttr  =DCTL_ON; VisibleAttr  >=DCTL_OFF; VisibleAttr--)
 *      for (FastChainAttr=DCTL_ON; FastChainAttr>=DCTL_OFF; FastChainAttr--)
 *      for (EraseCtrl    =DCTL_ON; EraseCtrl    >=DCTL_OFF; EraseCtrl--)
 *      for (DynamicCtrl  =DCTL_ON; DynamicCtrl  >=DCTL_OFF; DynamicCtrl--) {
 *         if (i==8) i=0;
 *         Set segment attributes.
 *         i++;
 *         Call CreateSegmentChain() one at a time.
 *         Set the erase draw control.
 *         Set the dynamic draw control.
 *         Draw the segment.
 *      }
 *      Delete segments 1-8.
 *   }
 * }
 *\\end
 ***************************************************************************/
VOID DrawSegment(HPS hPS, HWND hWndClient)
{
 LONG   i=0L, DrawMode, ChainAttr, DynamicAttr, DetectAttr,
        VisibleAttr, FastChainAttr, EraseCtrl, DynamicCtrl, Count=0L;
 SEGMENTATTRS TempAttrs[8];

  memcpy((PVOID)TempAttrs, (PVOID)SegAttrs, sizeof(SegAttrs));
  SUBENTER(Draw the segment, NULL);
  for (DrawMode=DM_RETAIN; DrawMode<=DM_DRAWANDRETAIN; DrawMode++) {
     for (DetectAttr   =ATTR_ON; DetectAttr   >=ATTR_OFF; DetectAttr--)
     for (VisibleAttr  =ATTR_ON; VisibleAttr  >=ATTR_OFF; VisibleAttr--)
     for (ChainAttr    =ATTR_ON; ChainAttr    >=ATTR_OFF; ChainAttr--)
     for (DynamicAttr  =ATTR_ON; DynamicAttr  >=ATTR_OFF; DynamicAttr--)
     for (FastChainAttr=ATTR_ON; FastChainAttr>=ATTR_OFF; FastChainAttr--) {
        // Make sure  1<=i<=8 because the TempAttrs array
        // only has eight items.  This is because CreateSegmentChain
        // only creates eight segments and their values must also be
        // 1 through 8.
        if (i==8) i=0L;
        TempAttrs[i].Chained   = ChainAttr;
        TempAttrs[i].Dynamic   = DynamicAttr;
        TempAttrs[i].Detectable= DetectAttr;
        TempAttrs[i].Visible   = VisibleAttr;
        TempAttrs[i].FastChain = FastChainAttr;
        i++;
        if (DynamicAttr!=ATTR_ON || DrawMode==DM_RETAIN) {
           switch (DrawMode) {
             case DM_RETAIN:
               CreateSegmentChain(hPS, hWndClient, &TempAttrs[i-1], i, i, TYPE_RETAIN);
               break;
             case DM_DRAWANDRETAIN:
               CreateSegmentChain(hPS, hWndClient, &TempAttrs[i-1], i, i, TYPE_DRAWRETAIN);
               break;
           }
        }
        for (EraseCtrl  =DCTL_ON; EraseCtrl  >=DCTL_OFF; EraseCtrl--)
        for (DynamicCtrl=DCTL_ON; DynamicCtrl>=DCTL_OFF; DynamicCtrl--) {
           if (DynamicAttr!=ATTR_ON || DrawMode==DM_RETAIN) {
              if (!GpiSetDrawControl(hPS, DCTL_ERASE, EraseCtrl)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiSetDrawControl() - ERASE", NULL);
              }
              if (!GpiSetDrawControl(hPS, DCTL_DYNAMIC, DynamicCtrl)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiSetDrawControl() - DYNAMIC", NULL);
              }
              Count++;
              if (!GpiResetPS(hPS, GRES_ATTRS)) {
                 ErrorCount++;
                 ERRSTATUS(TRUE, "GpiResetPS()", NULL);
              }
              if (!GpiDrawSegment(hPS, i)) {
                 ErrorCount++;
                 Count--;
                 ERRSTATUS(TRUE, "GpiDrawSegment()", NULL);
                 LogEnvironment(hPS, hWndClient, i);
              }
           }
        } /* for (DynamicCtrl */
        GpiDeleteSegment(hPS, i);
     } /* for (FastChainAttr */
     GpiSetColor(hPS, CLR_BLACK);
  }
  SUBEXIT();
  sprintf(achLogLine, "%ld total segments drawn Successfully", Count);
  LOGINFO(L_DEBUG, "*", achLogLine);
 return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: DeleteSegment()
 *
 * Purpose:  This subroutine tests the functionality of GpiDeleteSegment or
 *           GpiDeleteSegments, depending on the Mode flag.
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     EntryName - Name of entry point
 *     Mode      - Flag to indicate which function to use
 *
 * Outputs:
 *
 * Subroutines Required:
 *     CreateSegmentChain()
 *     SetInitialSegmentAttrs()
 *     OpenAndClose()
 *
 * Limitations:
 *
 *
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 * TEST - 1
 *   Set drawing mode.
 *   Call SetInitialSegmentAttrs().
 *   Call OpenAndClose() to open a segment, write some text, but NOT close it.
 *   Depending on the Mode flag, delete the segment using the appropriate
 *     function call.
 *   Try to draw the segment and force an error.
 * TEST - 2
 *   Set drawing mode.
 *   Call SetInitialSegmentAttrs().
 *   Call OpenAndClose() to open and close segment Seg1.
 *   Set drawing mode.
 *   Call SetInitialSegmentAttrs().
 *   Call OpenAndClose() to open and close segment LONG_MAX.
 *   for (i=1; i<=50; i++) {
 *      Set drawing mode.
 *      Call SetInitialSegmentAttrs().
 *      Call OpenAndClose() to open and close segments with random
 *        anywhere from 2 to LONG_MAX-1L.
 *   }
 *   Depending on the Mode flag, delete the segments using the appropriate
 *     function call.
 * TEST - 3
 *   Call CreateSegmentChain().
 *   Depending on the Mode flag, delete the odd or even segments.
 *   Draw the chain.
 * }
 *\\end
 ***************************************************************************/
VOID DeleteSegment(HPS hPS, HWND hWndClient, BYTE Mode)
{
 LONG   i, aSegIDs[50];
 POINTL ptlPoint = {50, 200};

  SETENTER(" ", "TEST 1 - Opening a Segment");
    if (!GpiSetDrawingMode(hPS, DM_RETAIN)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
    }
    SetInitialSegmentAttrs(hPS, &SegAttrs[0]);
    sprintf(achLogLine, "Segment %ld in TEST 1", Seg1);
    if (!OpenAndClose(hPS, FALSE, Seg1, achLogLine, &ptlPoint))
       ERRSTATUS(TRUE, "OpenAndClose()", NULL);
  SETEXIT();

  SUBENTER(Deleting an opened Segment, NULL);
    if (Mode & SINGLE) {
       if (!GpiDeleteSegment(hPS, Seg1)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiDeleteSegment() - %ld", ltoa(Seg1, achLogLine, 10));
       }
    }
    else {
       if (!GpiDeleteSegments(hPS, Seg1, Seg1)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiDeleteSegments() - %ld", ltoa(Seg1, achLogLine, 10));
       }
    }

    /* This tests the validity of deleting   */
    /* a segment while it is still open.  If */
    /* PMERR_SEG_NOT_FOUND is returned, then */
    /* the delete worked.                    */
    if (!GpiDrawSegment(hPS, Seg1)) {
       error=WinGetLastError(hPS);
       if (ERRORIDERROR(error) != PMERR_SEG_NOT_FOUND) {
          ErrorCount++;
       }
    }
  SUBEXIT();

  SETENTER(" ", "TEST 2 - Creating range of Segments");
    if (!GpiSetDrawingMode(hPS, DM_RETAIN)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
    }
    SetInitialSegmentAttrs(hPS, &SegAttrs[0]);
    sprintf(achLogLine, "Segment %ld in TEST 2", Seg1);
    if (!OpenAndClose(hPS, TRUE, Seg1, achLogLine, &ptlPoint))
       ERRSTATUS(TRUE, "OpenAndClose()", NULL);

    if (!GpiSetDrawingMode(hPS, DM_RETAIN)) {
       ErrorCount++;
       ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
    }
    SetInitialSegmentAttrs(hPS, &SegAttrs[0]);
    sprintf(achLogLine, "Segment %ld in TEST 2", LONG_MAX);
    if (!OpenAndClose(hPS, TRUE, LONG_MAX, achLogLine, &ptlPoint))
       ERRSTATUS(TRUE, "OpenAndClose()", NULL);

    for (i=1L; i<=50L; i++) {
       aSegIDs[i-1] = randL(2L, LONG_MAX-1L);
       if (!GpiSetDrawingMode(hPS, DM_RETAIN)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
       }
       SetInitialSegmentAttrs(hPS, &SegAttrs[0]);
       sprintf(achLogLine, "Segment %ld in TEST 2", aSegIDs[i-1]);
       if (!OpenAndClose(hPS, TRUE, aSegIDs[i-1], achLogLine, &ptlPoint))
          ERRSTATUS(TRUE, "OpenAndClose()", NULL);
    }
  SETEXIT();

  SUBENTER(Delete range of Segments, NULL);
    if (Mode & SINGLE) {
       if (!GpiDeleteSegment(hPS, Seg1)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiDeleteSegment() - %ld", ltoa(Seg1, achLogLine, 10));
       }
       if (!GpiDeleteSegment(hPS, LONG_MAX)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiDeleteSegment() - %ld", ltoa(LONG_MAX, achLogLine, 10));
       }
    } else {
       if (!GpiDeleteSegments(hPS, Seg1, Seg1)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiDeleteSegments() - %ld", ltoa(Seg1, achLogLine, 10));
       }
       if (!GpiDeleteSegments(hPS, LONG_MAX, LONG_MAX)) {
          ErrorCount++;
          ERRSTATUS(TRUE, "GpiDeleteSegments() - %ld", ltoa(LONG_MAX, achLogLine, 10));
       }
    }

    if (Mode & SINGLE) {
       for (i=0; i<=49; i++) {
          if (!GpiDeleteSegment(hPS, aSegIDs[i])) {
             ErrorCount++;
             ERRSTATUS(TRUE, "GpiDeleteSegment() - %ld", ltoa(aSegIDs[i], achLogLine, 10));
          }
       }
    } else {
       if (!GpiDeleteSegments(hPS, 2L, LONG_MAX)) {
           ErrorCount++;
           ERRSTATUS(TRUE, "GpiDeleteSegments()", NULL);
       }
    }
  SUBEXIT();

  SETENTER(" ", "TEST 3 - Creating Segment Chain");
    CreateSegmentChain(hPS, hWndClient, SegAttrs, Seg1, Seg8, TYPE_RETAIN);
  SETEXIT();

  SUBENTER(Deleting Odd Segments, NULL);
    for (i=Seg1; i<=Seg8; i++) {
       if (Mode & SINGLE) {
          if (i&1) {
             if (!GpiDeleteSegment(hPS, i)) {
                ErrorCount++;
                ERRSTATUS(TRUE, "GpiDeleteSegment() - %ld", ltoa(i, achLogLine, 10));
             }
          }
       } else {
          if (!(i&1)) {
             if (!GpiDeleteSegment(hPS, i)) {
                ErrorCount++;
                ERRSTATUS(TRUE, "GpiDeleteSegment() - %ld", ltoa(i, achLogLine, 10));
             }
          }
       }
    }
  SUBEXIT();
  if (!GpiResetPS(hPS, GRES_ATTRS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiResetPS()", NULL);
  }
  if (!GpiDrawChain(hPS)){
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiDrawChain()", NULL);
  }
  WriteText(Seg1, Seg8);
 return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: LogEnvironment()
 *
 * Purpose:
 *
 *
 * System Requirements:
 *
 * Revision Log: 9/05/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *
 * }
 *\\end
 ***************************************************************************/
VOID LogEnvironment(HPS hPS, HWND hWndClient, LONG i)
{
  sprintf(achLogLine, "\tSegment %ld", i);
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "Draw Mode is %ld", GpiQueryDrawingMode(hPS));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "Dynamic is %ld",
                       GpiQuerySegmentAttrs(hPS, i, ATTR_DYNAMIC));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "Detectable is %ld",
                       GpiQuerySegmentAttrs(hPS, i, ATTR_DETECTABLE));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "Visible is %ld",
                       GpiQuerySegmentAttrs(hPS, i, ATTR_VISIBLE));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "Chained is %ld",
                       GpiQuerySegmentAttrs(hPS, i, ATTR_CHAINED));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "FastChain is %ld",
                       GpiQuerySegmentAttrs(hPS, i, ATTR_FASTCHAIN));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "DCTL_ERASE is %ld",
                       GpiQueryDrawControl(hPS, DCTL_ERASE));
  LOGINFO(L_DEBUG, "@", achLogLine);

  sprintf(achLogLine, "DCTL_DYNAMIC is %ld",
                       GpiQueryDrawControl(hPS, DCTL_DYNAMIC));
  LOGINFO(L_DEBUG, "@", achLogLine);

 return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: Scribble()
 *
 * Purpose:  This subroutine fills a buffer with random points and calls
 *           GpiPolyLine to connect the points.
 *
 * System Requirements:
 *
 * Revision Log: 1991 - Initial Creation, David K. McCombs
 *            9/05/91 - Darin C. Hafer, Put this as a subroutine, changed
 *                      the loop counter to 500, used GpiQueryPS to get
 *                      actual width of the Presentation Space, got rid of
 *                      DevQueryCaps call.
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     Top       - An buffer to hold 500 pointl structures.
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   Query the PS size.
 *   for (Loop=0L; Loop < 500; Loop++) {
 *      Initialize buffer with coordinates.
 *   }
 *   Call GpiPolyLine to connect the points.
 * }
 *\\end
 ***************************************************************************/
BOOL Scribble(HPS hPS, HWND hWndClient, PPOINTL Top)
{
 LONG    Loop;
 POINTL  Cm;
 LONG    Count;

   Count=ErrorCount;
   GpiQueryPS(hPS, &sizePS);
   Cm.x = MapX(10L);
   Cm.y = MapY(50L);

   for (Loop=0L; Loop < 500; Loop++) {
      Top[Loop].x=randL(Cm.x, Cm.x+sizePS.cx);
      Top[Loop].y=randL(Cm.y, Cm.y+sizePS.cy);
   }
   GpiSetColor(hPS, CLR_BLUE);
   GpiMove(hPS, &Cm);

   if (GpiPolyLine(hPS, Loop, Top ) == GPI_ERROR)
      ErrorCount++;

 return (Count < ErrorCount)?0:1;
}

/****************************************************************************
 *\\ddd
 * Routine Name: SetAndQueryLoop()
 *
 * Purpose: This subroutine is called from GpiSetAndQryInitialSegmentAttrsExh
 *          and GpiSetAndQrySegmentAttrsExh.  It exhaustively tests setting
 *          and querying segment attributes.
 *
 * System Requirements:
 *
 * Revision Log: 9/26/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *     Initial   - Boolean to flag which calls to use.
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   for (DetectAttr   =ATTR_ON; DetectAttr   >=ATTR_OFF; DetectAttr--)
 *   for (VisibleAttr  =ATTR_ON; VisibleAttr  >=ATTR_OFF; VisibleAttr--)
 *   for (ChainAttr    =ATTR_ON; ChainAttr    >=ATTR_OFF; ChainAttr--)
 *   for (DynamicAttr  =ATTR_ON; DynamicAttr  >=ATTR_OFF; DynamicAttr--)
 *   for (FastChainAttr=ATTR_ON; FastChainAttr>=ATTR_OFF; FastChainAttr--)
 *   for (PropDetect   =ATTR_ON; PropDetect   >=ATTR_OFF; PropDetect--)
 *   for (PropVisible  =ATTR_ON; PropVisible  >=ATTR_OFF; PropVisible--) {
 *      Set DetectAttr.
 *      Query DetectAttr.
 *      Set VisibleAttr.
 *      Query VisibleAttr.
 *      .
 *      .
 *      .
 *      .
 *    }
 * }
 *\\end
 ***************************************************************************/
VOID SetAndQueryLoop(HPS hPS, HWND hWndClient, BOOL Initial)
{
 LONG   DetectAttr,
        VisibleAttr,
        ChainAttr,
        DynamicAttr,
        FastChainAttr,
        PropDetect,
        PropVisible;


   for (DetectAttr   =ATTR_ON; DetectAttr   >=ATTR_OFF; DetectAttr--)
   for (VisibleAttr  =ATTR_ON; VisibleAttr  >=ATTR_OFF; VisibleAttr--)
   for (ChainAttr    =ATTR_ON; ChainAttr    >=ATTR_OFF; ChainAttr--)
   for (DynamicAttr  =ATTR_ON; DynamicAttr  >=ATTR_OFF; DynamicAttr--)
   for (FastChainAttr=ATTR_ON; FastChainAttr>=ATTR_OFF; FastChainAttr--)
   for (PropDetect   =ATTR_ON; PropDetect   >=ATTR_OFF; PropDetect--)
   for (PropVisible  =ATTR_ON; PropVisible  >=ATTR_OFF; PropVisible--) {
      switch ((INT)Initial) {
        case 1:
           if (!GpiSetSegmentAttrs(hPS, Seg1, ATTR_DETECTABLE, DetectAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Detectability", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_DETECTABLE) != DetectAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Detectability", NULL);
           }
           if ( !GpiSetSegmentAttrs(hPS, Seg1, ATTR_VISIBLE, VisibleAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Visibility", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_VISIBLE) != VisibleAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Visibility", NULL);
           }
           if (!GpiSetSegmentAttrs(hPS, Seg1, ATTR_CHAINED, ChainAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Chain", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_CHAINED) != ChainAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Chain", NULL);
           }
           if (!GpiSetSegmentAttrs(hPS, Seg1, ATTR_DYNAMIC, DynamicAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Dynamic", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_DYNAMIC) != DynamicAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Dynamic", NULL);
           }
           if (!GpiSetSegmentAttrs(hPS, Seg1, ATTR_FASTCHAIN, FastChainAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set FastChain", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_FASTCHAIN) != FastChainAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Fast Chain", NULL);
           }
           if (!GpiSetSegmentAttrs(hPS, Seg1, ATTR_PROP_DETECTABLE, PropDetect)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Propagate Detectable", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_PROP_DETECTABLE) != PropDetect) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Propagate Detectable", NULL);
           }
           if (!GpiSetSegmentAttrs(hPS, Seg1, ATTR_PROP_VISIBLE, PropVisible)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Propagate Visible", NULL);
           }
           if (GpiQuerySegmentAttrs(hPS, Seg1, ATTR_PROP_VISIBLE) != PropVisible) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Propagate Visible", NULL);
           }
         break;
        case 0:
           if (!GpiSetInitialSegmentAttrs(hPS, ATTR_DETECTABLE, DetectAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial Detectability", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_DETECTABLE) != DetectAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Detectability", NULL);
           }
           if ( !GpiSetInitialSegmentAttrs(hPS, ATTR_VISIBLE, VisibleAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial Visibility", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_VISIBLE) != VisibleAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Visibility", NULL);
           }
           if (!GpiSetInitialSegmentAttrs(hPS, ATTR_CHAINED, ChainAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial Chain", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_CHAINED) != ChainAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Chain", NULL);
           }
           if (!GpiSetInitialSegmentAttrs(hPS, ATTR_DYNAMIC, DynamicAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial Dynamic", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_DYNAMIC) != DynamicAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Dynamic", NULL);
           }
           if (!GpiSetInitialSegmentAttrs(hPS, ATTR_FASTCHAIN, FastChainAttr)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial FastChain", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_FASTCHAIN) != FastChainAttr) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Fast Chain", NULL);
           }
           if (!GpiSetInitialSegmentAttrs(hPS, ATTR_PROP_DETECTABLE, PropDetect)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial Propagate Detectable", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_PROP_DETECTABLE) != PropDetect) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Propagate Detectable", NULL);
           }
           if (!GpiSetInitialSegmentAttrs(hPS, ATTR_PROP_VISIBLE, PropVisible)) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Set Initial Propagate Visible", NULL);
           }
           if (GpiQueryInitialSegmentAttrs(hPS, ATTR_PROP_VISIBLE) != PropVisible) {
              ErrorCount++;
              ERRSTATUS(TRUE, "Query Initial Propagate Visible", NULL);
           }
         break;
      }
   }
 return;
}

/****************************************************************************
 *\\ddd
 * Routine Name: CreateThread()
 *
 * Purpose:  This subroutine allocates a stack for the asynchronous thread
 *           and creates the thread.
 *
 * System Requirements:
 *
 * Revision Log: 10/01/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *
 * Outputs:
 *
 * Subroutines Required:
 *     NewThread()
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   Allocate space for a stack.
 *   Set a semaphore.
 *   Create the thread.
 * }
 *\\end
 ***************************************************************************/
BOOL CreateThread(HPS hPS, HWND hWndClient)
{

 static PBYTE pbAsyncStack;          /* long pointer to stack for new thread */

  //32 bit
  if (DosAllocMem((PVOID)&pbAsyncStack,STACKSIZE,PAG_READ | PAG_WRITE | PAG_COMMIT)){
        return(FALSE);
  }

  *((PHPS)pbAsyncStack) = hPS;
  *((PHWND)(pbAsyncStack+4)) = hWndClient;

  //32 bit Semaphore
  DosCreateEventSem(NULL,&ulSem,0,TRUE);
  DosResetEventSem(hSem,ulSem);


  if (DosCreateThread( &tidAsync,
                       (PFNTHREAD)NewThread,
//                       (PVOID)pbAsyncStack,threadFlags,STACKSIZE)){
                       (ULONG)pbAsyncStack,threadFlags,STACKSIZE)){
      return(FALSE);
  }
  return(TRUE);
}

/****************************************************************************
 *\\ddd
 * Routine Name: NewThread()
 *
 * Purpose: This subroutine is the asynchronous thread routine.  It creates
 *          4 segments by calling CreateSegmentChain() and draws the dynamic
 *          segments.
 *
 * System Requirements:
 *
 * Revision Log: 10/01/91 - Initial Creation, Darin C. Hafer
 *
 *
 * Inputs:
 *     hPS       - handle to PS
 *     hWndClient- handle to client window
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *   Initialize the thread and get an anchor block.
 *   Turn the fast-chain attribute off for all segments and initialize
 *     segment 1 and 3 to be dynamic.
 *   Create the segment chain.
 *   Wait for the semaphore to be cleared.
 *   Draw the dynamic segments.
 *   for (Move=1; Move<=NumberOfTimes; Move++) {
 *      Set the segment matrix to move over in the x-direction.
 *      Set segment transforms for all four segments.
 *      Reset the PS. (as sometimes required before a GpiDrawDynamics)
 *      Draw the dynamic segments again.
 *   }
 * }
 *\\end
 ***************************************************************************/
VOID  NewThread(HPS hPS, HWND hWndClient)
{
 MATRIXLF MatStruct;
 LONG     Return, Move, NumberOfTimes=1L;
 SEGMENTATTRS TempAttrs[8];

  /*********************************************************************/
  /* Obtain an anchor block handle. If this fails, both threads of the */
  /* application are terminated.                                       */
  /*********************************************************************/
  IsRunning=TRUE;
  habAsync = WinInitialize((USHORT)NULL);
  if (habAsync == (HAB)NULL) {
      DosExit( EXIT_THREAD, 0);
  }

  memcpy((PVOID)TempAttrs, (PVOID)SegAttrs, sizeof(SegAttrs));
  TempAttrs[0].Dynamic   = ATTR_ON;
  TempAttrs[1].FastChain = ATTR_OFF;
  TempAttrs[2].Dynamic   = ATTR_ON;
  TempAttrs[2].FastChain = ATTR_OFF;
  TempAttrs[3].FastChain = ATTR_OFF;
  TempAttrs[4].FastChain = ATTR_OFF;

  CreateSegmentChain(hPS, hWndClient, TempAttrs, Seg1, Seg4, TYPE_RETAIN);


  //32 bit Semaphore
  DosWaitEventSem(ulSem, SEM_INDEFINITE_WAIT);


  if (!GpiResetPS(hPS, GRES_ATTRS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiResetPS()", NULL);
  }
  if (!GpiDrawDynamics(hPS)) {
     ErrorCount++;
  }
  if (!GpiResetPS(hPS, GRES_ATTRS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiResetPS()", NULL);
  }

  NumberOfTimes = 100;

  for (Move=1; Move<=NumberOfTimes; Move++) {
     MatStruct.fxM11 = MAKEFIXED(1, 0);
     MatStruct.fxM12 = MAKEFIXED(0, 0);
     MatStruct.lM13  = 0L;
     MatStruct.fxM21 = MAKEFIXED(0, 0);
     MatStruct.fxM22 = MAKEFIXED(1, 0);
     MatStruct.lM23  = 0L;
     MatStruct.lM31  = MapX(3L*Move);
     MatStruct.lM32  = 0L;
     MatStruct.lM33  = 1L;

     /* if I don't take a nap, it won't work */
     if ((Return=GpiQueryStopDraw(hPS))==GPI_ALTERROR)
        ErrorCount++;
     if (Return)
        DosSleep(100);


     if (!GpiSetSegmentTransformMatrix(hPS, Seg1, 9L, &MatStruct,TRANSFORM_REPLACE))
        ErrorCount++;
     if (!GpiSetSegmentTransformMatrix(hPS, Seg2, 9L, &MatStruct,TRANSFORM_REPLACE))
        ErrorCount++;
     if (!GpiSetSegmentTransformMatrix(hPS, Seg3, 9L, &MatStruct,TRANSFORM_REPLACE))
        ErrorCount++;
     if (!GpiSetSegmentTransformMatrix(hPS, Seg4, 9L, &MatStruct,TRANSFORM_REPLACE))
        ErrorCount++;

     if (!GpiResetPS(hPS, GRES_ATTRS)) {
        ErrorCount++;
        ERRSTATUS(TRUE, "GpiResetPS()", NULL);
     }

     if (!GpiDrawDynamics(hPS)) {
        ErrorCount++;
     }
  }
  IsRunning=FALSE;
return;
}

VOID ResetAndOpen(HPS hPS, HWND hWndClient, LONG DrawMode)
{
  if (!GpiResetPS(hPS, GRES_SEGMENTS)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiResetPS()", NULL);
  }
  if (!GpiSetDrawingMode(hPS, DrawMode)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiSetDrawingMode()", NULL);
  }
  if (!GpiOpenSegment(hPS, Seg1)) {
     ErrorCount++;
     ERRSTATUS(TRUE, "GpiOpenSegment()", NULL);
  }
 return;
}

void Write_Text(HPS hPS, HWND hWndClient, LONG lrange, LONG urange)
{
LONG          lval;
USHORT        Count=0;
CHAR          szBuffer[32];
POINTL        ptText;

   for (lval = lrange; lval <= urange; lval++)
     {
        Count++;
        sprintf(szBuffer, "Segment %ld", lval);

        ptText.x = aptDefine[Count - 1][8].x;
        ptText.y = aptDefine[Count - 1][6].y;

        GpiSetColor(hPS, CLR_BLACK);
        GpiCharStringAt(hPS, &ptText, strlen(szBuffer), &szBuffer[0]);
     }
}
