/***************************************************************************
 *\\ddd
 * Subtask Name:  PDVT
 *
 * Module name:   GPIPATH.C
 *
 * Purpose:       This module is called by the PTT to test GPI Path functions.
 *
 * Revision Log:  01/03/91, PDVT, Kim Cairns
 *                - Original module template
 *                01/11/91, PDVT, Kim Cairns
 *                - Converted to standard utility functions
 *                01/17/91, PDVT, Kim Cairns
 *                - Added RC entry points
 *
 * SRS References:  Section 5.4
 *
 * Contents:
 *               InitTest()
 *               GpiPathPre        - to test path precision
 *               GpiPathApp        - to verify path functionality
 *               GpiFillPathExh    - to test specific GPIs in detail
 *               GpiOutlinePathExh - to test specific GPIs in detail
 *               GpiStrokePathExh  - to test specific GPIs in detail
 *               GpiSetLineExh     - to test specific GPIs in detail
 *               BADERROR          - to report unexpected errors
 *
 * Design Overview:
 *               Paths are used to draw and fill lines and non-rectangular
 *               areas.  This testcase covers the accuracy of the path limits,
 *               the basic functionality of path GPIs and each of the path
 *               drawing functions in detail.
 *
 * Limitations:
 *               This testcase does not cover the use of paths for clipping
 *               purposes; see testcase GPICLIP.
 *
 *\\end
 ***************************************************************************/

#define INCL_WIN      /* win function definitions */
#define INCL_GPI      /* all gpi specific definitions */
#define INCL_ERRORS
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include "PTTDLLDF.h" // test case defines
#include "gpipath.h"  // gpi path dll header file
#include "gpi_grid.h" // gpi standard grid functions header file
#include "std_util.h" // standard screen text functions header file

 /***************************************************************************
 *\\ddd
 * Routine name: InitTest()
 *
 * Purpose:  Initialize function name and address array for ptt.
 *
 * System Requirements:  None
 *
 * Revision Log:  01/03/91, PDVT, Kim Cairns
 *                - initial function template
 *
 * Inputs:
 *             sel            - pass thru parameter for other function call
 *             pInitInfo      - pointer to info array
 *
 * Outputs:
 *             pInitInfo      - updated with all entry points names and addresses
 *
 * Subroutines Required:
 *             InitTestSegVars()
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *   for all test functions in this test case {
 *       pInitInfo->function name = function name[i]
 *       pInitInfo->function address = function address[i]
 *   }
 *   call init set function
 *\\end
 ***************************************************************************/
VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO pInitInfo)
  {

    /**** Initialize total number of tests in test case ****/
    pInitInfo->ntests = NTESTS;

    /**** Load test names and entry point addresses ****/
    strcpy(pInitInfo->group_name,"GPI Path");
    strcpy(pInitInfo->test_names[0],"GpiPathPre");
    pInitInfo->test_addrs[0]=GpiPathPre;
    strcpy(pInitInfo->test_names[1],"GpiPathApp");
    pInitInfo->test_addrs[1]=GpiPathApp;
    strcpy(pInitInfo->test_names[2],"GpiFillPathExh");
    pInitInfo->test_addrs[2]=GpiFillPathExh;
    strcpy(pInitInfo->test_names[3],"GpiOutlinePathExh");
    pInitInfo->test_addrs[3]=GpiOutlinePathExh;
    strcpy(pInitInfo->test_names[4],"GpiStrokePathExh");
    pInitInfo->test_addrs[4]=GpiStrokePathExh;
    strcpy(pInitInfo->test_names[5],"GpiSetLineExh");
    pInitInfo->test_addrs[5]=GpiSetLineExh;

    InitTestSegVars(sel);
  }

 /***************************************************************************
 *\\ddd
 * Routine name:  GpiPathPre()
 *
 * Purpose:  This is an entry point for the PTT that causes GPI Path precision
 *           tests to be run.
 *
 * System Requirements:  Section 5.4.2
 *
 * Revision Log:  01/03/91, PDVT, Kim Cairns
 *                - initial function template
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             gpi_grid()     - standard grid function using GPIs
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *   call gpi_grid to draw 1" x 1" grid
 *
 *   set and map points for triangle
 *   begin path
 *   call GpiPolyLine to create two sides of triangle
 *   call GpiCloseFigure to close triangle
 *   end path
 *   stroke path to create wide lines
 *
 *   set and map points for pentagon
 *   begin path
 *   call GpiPolyLine to explicitly create all five sides of pentagon
 *   end path
 *   outline path to create regular lines
 *
 *   set and map points for rectangle
 *   begin path
 *   call GpiPolyLine to create three sides of rectangle
 *   end path
 *   fill path to implicitly close figure
 *
 *\\end
 ***************************************************************************/

VOID APIENTRY GpiPathPre(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL  ptl, ptlShape[5];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
     TEST_SIGNATURE(0,0);
     LOG_SIGNATURE;
     CWRITELOG(L_LOTRACE, 0, "+ee+ GpiPathPre - Entry.\n");

     lErrorCount = 0L;

     /*
      * This draws a 1" x 1" grid
      */
        gpi_grid (hPS, INCH);

  /*
   * Draw a triangle with wide lines at {(1,1), (2,2), (3,1)}
   */
        ptl.x = delta_x * 1;
        ptl.y = delta_y * 1;
        MapPt(ptl);
        GpiMove (hPS, &ptl);                     // Set starting position

        ptlShape[0].x = delta_x * 2;
        ptlShape[0].y = delta_y * 2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 3;
        ptlShape[1].y = delta_y * 1;
        MapPt(ptlShape[1]);                      // Set other two points

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiPathPre");
          return;
        }
        // Added by TSS...8/1/91 to show wider lines as text indicates.
        if (GpiSetLineWidthGeom(hPS, 5L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", TRUE, "GpiPathPre");
          return;
        }

        GpiPolyLine(hPS, 2L, ptlShape);          // Creates 2 sides of triangle
        if (GpiCloseFigure(hPS) == FALSE)
        {
          BADERROR("GpiCloseFigure", FALSE, NULL);
        }
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiPathPre");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 5/2 + line_space,
           "GpiStrokePath test");
        MapAndWriteString(hPS, 0L, delta_y * 5/2,
           "Triangle should have wide lines and");
        MapAndWriteString(hPS, 0L, delta_y * 5/2 - line_space,
           "touch (1,1), (2,2), (3,1)");

  /*
   * Draw a pentagon at {(2,3.5), (1,5), (3,6), (5,5), (4,3.5)}
   */
        ptl.x = delta_x * 2;
        ptl.y = delta_y * 7/2;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 1;
        ptlShape[0].y = delta_y * 5;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 3;
        ptlShape[1].y = delta_y * 6;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 5;
        ptlShape[2].y = delta_y * 5;
        MapPt(ptlShape[2]);
        ptlShape[3].x = delta_x * 4;
        ptlShape[3].y = delta_y * 7/2;
        MapPt(ptlShape[3]);
        ptlShape[4].x = delta_x * 2;
        ptlShape[4].y = delta_y * 7/2;
        MapPt(ptlShape[4]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiPathPre");
          return;
        }
        GpiPolyLine(hPS, 5L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiPathPre");
          return;
        }
        if (GpiOutlinePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiOutlinePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 6 + line_space*2,
           "GpiOutlinePath test");
        MapAndWriteString(hPS, 0L, delta_y * 6 + line_space,
           "Pentagon should touch (2,3.5), (1,5), (3,6), (5,5), (4,3.5)");

     /*
      * Set fill pattern to halftone
      */
        GpiSetPattern(hPS, PATSYM_HALFTONE);

  /*
   * Draw and fill a rectangle at {(5,2), (5,3), (7,3), (7,2)}
   */
        ptl.x = delta_x * 5;
        ptl.y = delta_y * 2;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5;
        ptlShape[0].y = delta_y * 3;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7;
        ptlShape[1].y = delta_y * 3;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7;
        ptlShape[2].y = delta_y * 2;
        MapPt(ptlShape[2]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiPathPre");
          return;
        }
        GpiPolyLine(hPS, 3L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiPathPre");
          return;
        }
        if (GpiFillPath(hPS, 1L, FPATH_ALTERNATE) == GPI_ERROR)
        {
          BADERROR("GpiFillPath", FALSE, NULL);
        }

        MapAndWriteString(hPS, delta_x * 4, delta_y * 3/2 + line_space,
           "GpiFillPath test");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 3/2,
           "Rectangle should be shaded and");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 3/2 - line_space,
           "touch (5,2), (5,3), (7,3), (7,2)");


    if (lErrorCount) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
       "$GpiPath Precision Test                                         -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiPath Precision Test                                         -    PASS.\n");

      CWRITELOG(L_LOTRACE, 0, "+ex+ GpiPathPre - Exit.\n");

      } /* end the ELSE */
} /* end GpiPathPre */
 /***************************************************************************
 *\\ddd
 * Routine name:  GpiPathApp
 *
 * Purpose:  This is an entry point for the PTT that causes the GPI Path
 *           application test to be run.
 *
 * System Requirements:  Section 5.4.2
 *
 * Revision Log:  01/04/91, PDVT, Kim Cairns
 *                - initial function template
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()   - sets device-specific units
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *   call PathApp, which contains the body of the test
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPathApp(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
        PathApp(hPS, hWndClient);
  } /* end the ELSE */
} /* end GpiPathApp */

 /***************************************************************************
 *\\ddd
 * Routine name:  GpiFillPathExh()
 *
 * Purpose:  This is an entry point for the PTT that causes the GpiFillPath
 *           exhaustive tests to be run.
 *
 * System Requirements:  Section 5.4.2
 *
 * Revision Log:  01/04/91, PDVT, Kim Cairns
 *                - initial function template
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()   - sets device-specific units
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *   Use GpiModifyPath and GpiFillPath to draw an open triangle
 *      with FPATH_ALTERNATE
 *   Use GpiModifyPath and GpiFillPath to draw an open square
 *      with FPATH_WINDING
 *   Use GpiFillPath to draw a pentagon, which is closed automatically
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiFillPathExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL  ptl, ptlShape[5];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
     TEST_SIGNATURE(0,0);
     LOG_SIGNATURE;
     CWRITELOG(L_LOTRACE, 0, "+ee+ GpiFillPathExh - Entry.\n");

     lErrorCount = 0L;

     /*
      * set device-specific units
      */
        set_delta(hPS);

     /*
      * Set fill pattern to halftone
      */
        GpiSetPattern(hPS, PATSYM_HALFTONE);

  /*
   * Draw triangle with FPATH_ALTERNATE
   */
        ptl.x = delta_x * 5/2;
        ptl.y = delta_y * 1;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 3;
        ptlShape[0].y = delta_y * 2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7/2;
        ptlShape[1].y = delta_y * 1;
        MapPt(ptlShape[1]);

        if (GpiSetLineWidthGeom(hPS, 10L) == FALSE)
        {
          lErrorCount += BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }
        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiFillPathExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiFillPathExh");
          return;
        }
        if (GpiModifyPath(hPS, 1L, MPATH_STROKE) == FALSE)
        {
          BADERROR("GpiModifyPath", FALSE, NULL);
        }
        if (GpiFillPath(hPS, 1L, FPATH_ALTERNATE) == GPI_ERROR)
        {
          BADERROR("GpiFillPath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 2,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space,
           "be two sides of a");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space*2,
           "triangle drawn");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space*3,
           "with wide lines.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2,
           "GpiModifyPath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2 - line_space,
           "+ GpiFillPath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2 - line_space*2,
           "Does not close figure.");
        MapAndWriteString(hPS, delta_x * 4, delta_y + line_space,
           "FPATH_ALTERNATE");
        MapAndWriteString(hPS, delta_x * 4, delta_y,
           "Line Join is hollow.");

  /*
   * Draw square with FPATH_WINDING
   */
        ptl.x = delta_x * 5/2;
        ptl.y = delta_y * 3;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5/2;
        ptlShape[0].y = delta_y * 4;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7/2;
        ptlShape[1].y = delta_y * 4;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7/2;
        ptlShape[2].y = delta_y * 3;
        MapPt(ptlShape[2]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiFillPathExh");
          return;
        }
        GpiPolyLine(hPS, 3L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiFillPathExh");
          return;
        }
        if (GpiModifyPath(hPS, 1L, MPATH_STROKE) == FALSE)
        {
          BADERROR("GpiModifyPath", FALSE, NULL);
        }
        if (GpiFillPath(hPS, 1L, FPATH_WINDING) == GPI_ERROR)
        {
          BADERROR("GpiFillPath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 4,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space,
           "be three sides of");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space*2,
           "a square drawn");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space*3,
           "with wide lines.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4,
           "GpiModifyPath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4 - line_space,
           "+ GpiFillPath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4 - line_space*2,
           "Does not close figure.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 3 + line_space,
           "FPATH_WINDING");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 3,
           "Line Join is filled in.");

  /*
   * Draw filled pentagon
   */
        ptl.x = delta_x * 11/4;
        ptl.y = delta_y * 5;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5/2;
        ptlShape[0].y = delta_y * 11/2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 3;
        ptlShape[1].y = delta_y * 6;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7/2;
        ptlShape[2].y = delta_y * 11/2;
        MapPt(ptlShape[2]);
        ptlShape[3].x = delta_x * 13/4;
        ptlShape[3].y = delta_y * 5;
        MapPt(ptlShape[3]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiFillPathExh");
          return;
        }
        GpiPolyLine(hPS, 4L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiFillPathExh");
          return;
        }
        if (GpiFillPath(hPS, 1L, FPATH_ALTERNATE) == GPI_ERROR)
        {
          BADERROR("GpiFillPath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 6,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space,
           "be a complete and");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space*2,
           "filled pentagon.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6,
           "GpiFillPath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6 - line_space,
           "Closes figure automatically.");

    if (lErrorCount != 0) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
       "$GpiFillPath Exhaustive Test                                    -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiFillPath Exhaustive Test                                    -    PASS.\n");

        CWRITELOG(L_LOTRACE, 0, "+ex+ GpiFillPathExh - Exit.\n");
  } /* end the ELSE */
} /* end GpiFillPathExh */

 /***************************************************************************
 *\\ddd
 * Routine name:  GpiOutlinePathExh()
 *
 * Purpose:  This is an entry point for the PTT that causes the GpiOutlinePath
 *           exhaustive tests to be run.
 *
 * System Requirements:  Section 5.4.2
 *
 * Revision Log:  01/04/91, PDVT, Kim Cairns
 *                - initial function template
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *   Use GpiOutlinePath to draw an open triangle
 *   Use GpiOutlinePath to draw an open square
 *   Use GpiCloseFigure and GpiOutlinePath to draw a closed pentagon
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiOutlinePathExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL  ptl, ptlShape[5];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
     TEST_SIGNATURE(0,0);
     LOG_SIGNATURE;
     CWRITELOG(L_LOTRACE, 0, "+ee+ GpiOutlinePathExh - Entry.\n");

     lErrorCount = 0L;

     /*
      * set device-specific units
      */
        set_delta(hPS);

  /*
   * Draw triangle
   */
        ptl.x = delta_x * 5/2;
        ptl.y = delta_y * 1;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 3;
        ptlShape[0].y = delta_y * 2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7/2;
        ptlShape[1].y = delta_y * 1;
        MapPt(ptlShape[1]);

        if (GpiSetLineWidth(hPS, LINEWIDTH_NORMAL) == FALSE)
        {
          BADERROR("GpiSetLineWidth", FALSE, NULL);
        }
        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiOutlinePathExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiOutlinePathExh");
          return;
        }
        if (GpiOutlinePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiOutlinePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 2,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space,
           "be two sides of");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space*2,
           "a triangle.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2,
           "GpiOutlinePath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2 - line_space,
           "Does not close figure.");

  /*
   * Draw square with normal lines
   */
        ptl.x = delta_x * 5/2;
        ptl.y = delta_y * 3;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5/2;
        ptlShape[0].y = delta_y * 4;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7/2;
        ptlShape[1].y = delta_y * 4;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7/2;
        ptlShape[2].y = delta_y * 3;
        MapPt(ptlShape[2]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiOutlinePathExh");
          return;
        }
        GpiPolyLine(hPS, 3L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiOutlinePathExh");
          return;
        }
        if (GpiOutlinePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiOutlinePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 4,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space,
           "be three sides of");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space*2,
           "a square.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4,
           "GpiOutlinePath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4 - line_space,
           "Does not close figure.");

  /*
   * Draw pentagon and close explicitly with GpiCloseFigure
   */
        ptl.x = delta_x * 11/4;
        ptl.y = delta_y * 5;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5/2;
        ptlShape[0].y = delta_y * 11/2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 3;
        ptlShape[1].y = delta_y * 6;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7/2;
        ptlShape[2].y = delta_y * 11/2;
        MapPt(ptlShape[2]);
        ptlShape[3].x = delta_x * 13/4;
        ptlShape[3].y = delta_y * 5;
        MapPt(ptlShape[3]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiOutlinePathExh");
          return;
        }
        GpiPolyLine(hPS, 4L, ptlShape);
        if (GpiCloseFigure(hPS) == FALSE)
        {
          BADERROR("GpiCloseFigure", FALSE, NULL);
        }
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiOutlinePathExh");
          return;
        }
        if (GpiOutlinePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiOutlinePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 6,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space,
           "be a complete");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space*2,
           "pentagon.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6,
           "GpiCloseFigure");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6 - line_space,
           "GpiOutlinePath");

    if (lErrorCount) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
       "$GpiOutlinePath Exhaustive Test                                 -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiOutlinePathExhaustive Test                                  -    PASS.\n");

        CWRITELOG(L_LOTRACE, 0, "+ex+ GpiOutlinePathExh - Exit.\n");
  } /* end the ELSE */
} /* end GpiOutlinePathExh */
 /***************************************************************************
 *\\ddd
 * Routine name:  GpiStrokePathExh()
 *
 * Purpose:  This is an entry point for the PTT that causes the GpiStrokePath
 *           exhaustive tests to be run.
 *
 * System Requirements:  Section 5.4.2
 *
 * Revision Log:  01/04/91, PDVT, Kim Cairns
 *                - initial function template
 *
 *                06/12/91, Tim Seevers
 *                - changed GpiSetLineWidthGeom value from 10 to 15
 *                  per Vish.
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *   Use GpiStrokePath to draw an open triangle with wide lines
 *   Use GpiStrokePath to draw an open square with wide lines
 *   Use GpiCloseFigure and GpiStrokePath to draw a closed pentagon
 *      with wide lines
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiStrokePathExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL  ptl, ptlShape[5];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
     TEST_SIGNATURE(0,0);
     LOG_SIGNATURE;
     CWRITELOG(L_LOTRACE, 0, "+ee+ GpiStrokePathExh - Entry.\n");

     lErrorCount = 0L;

     /*
      * set device-specific units
      */
        set_delta(hPS);

     /*
      * Set fill pattern to halftone
      */
        GpiSetPattern(hPS, PATSYM_HALFTONE);

  /*
   * Draw triangle with wide lines
   */
        ptl.x = delta_x * 5/2;
        ptl.y = delta_y * 1;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 3;
        ptlShape[0].y = delta_y * 2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7/2;
        ptlShape[1].y = delta_y * 1;
        MapPt(ptlShape[1]);

 // TSS....Per Vish, I have changed the GpiSetLineWidthGeom value
 // from 10 to 15.

        if (GpiSetLineWidthGeom(hPS, 15L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }
        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiStrokePathExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiStrokePathExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 2,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space,
           "be two sides of");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space*2,
           "a triangle drawn");
        MapAndWriteString(hPS, 0L, delta_y * 2 - line_space*3,
           "with wide lines.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2,
           "GpiStrokePath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 2 - line_space,
           "Does not close figure.");

  /*
   * Draw square with wide lines
   */
        ptl.x = delta_x * 5/2;
        ptl.y = delta_y * 3;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5/2;
        ptlShape[0].y = delta_y * 4;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 7/2;
        ptlShape[1].y = delta_y * 4;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7/2;
        ptlShape[2].y = delta_y * 3;
        MapPt(ptlShape[2]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiStrokePathExh");
          return;
        }
        GpiPolyLine(hPS, 3L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiStrokePathExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 4,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space,
           "be three sides of");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space*2,
           "a square drawn");
        MapAndWriteString(hPS, 0L, delta_y * 4 - line_space*3,
           "with wide lines.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4,
           "GpiStrokePath");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 4 - line_space,
           "Does not close figure.");

  /*
   * Draw pentagon with wide lines and close explicitly with GpiCloseFigure
   */
        ptl.x = delta_x * 11/4;
        ptl.y = delta_y * 5;
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        ptlShape[0].x = delta_x * 5/2;
        ptlShape[0].y = delta_y * 11/2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 3;
        ptlShape[1].y = delta_y * 6;
        MapPt(ptlShape[1]);
        ptlShape[2].x = delta_x * 7/2;
        ptlShape[2].y = delta_y * 11/2;
        MapPt(ptlShape[2]);
        ptlShape[3].x = delta_x * 13/4;
        ptlShape[3].y = delta_y * 5;
        MapPt(ptlShape[3]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiStrokePathExh");
          return;
        }
        GpiPolyLine(hPS, 4L, ptlShape);
        if (GpiCloseFigure(hPS) == FALSE)
        {
          BADERROR("GpiCloseFigure", FALSE, NULL);
        }
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiStrokePathExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }

        MapAndWriteString(hPS, 0L, delta_y * 6,
           "This shape should");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space,
           "be a complete");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space*2,
           "pentagon drawn");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space*3,
           "with wide lines.");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6,
           "GpiCloseFigure");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6 - line_space,
           "GpiStrokePath");

    if (lErrorCount) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
       "$GpiStrokePath Exhaustive Test                                  -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiStrokePath Exhaustive Test                                  -    PASS.\n");

        CWRITELOG(L_LOTRACE, 0, "+ex+ GpiStrokePathExh - Exit.\n");
  } /* end the ELSE */
} /* end GpiStrokePathExh */

 /***************************************************************************
 *\\ddd
 * Routine name:  GpiSetLineExh()
 *
 * Purpose:  This is an entry point for the PTT that causes the GpiSetLine
 *           exhaustive tests to be run.
 *
 * System Requirements:  Section 5.4.2
 *
 * Revision Log:  01/04/91, PDVT, Kim Cairns
 *                - initial function template
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()
 *
 * Limitations:
 *             This test does not test GpiSetLineWidth because, according to
 *             the IBM documentation, no multiplier greater than one is
 *             supported, thus rendering this API and the corresponding
 *             GpiQueryLineWidth API useless.
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *   Test widths 10, 5, 1, 0 with GpiSetLineWidthGeom
 *   Test all valid line ends with GpiSetLineEnd
 *   Test all valid line joins with GpiSetLineJoin
 *   Test GpiQueryLineWidthGeom
 *   Test GpiQueryLineEnd
 *   Test GpiQueryLineJoin
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetLineExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL  ptl, ptlShape[5];
  LONG    query_value;
  CHAR    out_string[50];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
     TEST_SIGNATURE(0,0);
     LOG_SIGNATURE;
     CWRITELOG(L_LOTRACE, 0, "+ee+ GpiSetLineExh - Entry.\n");

     lErrorCount = 0L;

     /*
      * set device-specific units
      */
        set_delta(hPS);

     /*
      * Set fill pattern to halftone
      */
        GpiSetPattern(hPS, PATSYM_HALFTONE);

  /*
   * Line width test
   */
     // Test 1
        if (GpiSetLineWidthGeom(hPS, 10L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }

        ptl.x = delta_x * 1/2;
        ptl.y = delta_y * 3/2;
        MapPt(ptl);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = delta_x * 1/2;
        ptlShape[0].y = delta_y * 1;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 1;
        ptlShape[1].y = delta_y * 1;
        MapPt(ptlShape[1]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, 0L, delta_y * 3/2 + line_space,
           "Width = 10");


     // Test 2
        if (GpiSetLineWidthGeom(hPS, 5L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 5/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 5/2);
        ptlShape[1].x = MapX(delta_x * 3);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 2, delta_y * 3/2 + line_space,
           "Width = 5");


     // Test 3
        if (GpiSetLineWidthGeom(hPS, 1L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 9/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 9/2);
        ptlShape[1].x = MapX(delta_x * 5);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 4, delta_y * 3/2 + line_space,
           "Width = 1");


     // Test 4
        if (GpiSetLineWidthGeom(hPS, 0L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 13/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 13/2);
        ptlShape[1].x = MapX(delta_x * 7);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 6, delta_y * 3/2 + line_space,
           "Width = 0");

  /*
   * Line End test
   */

     // Setting wide lines so differences are visible
        if (GpiSetLineWidthGeom(hPS, 10L) == FALSE)
        {
          BADERROR("GpiSetLineWidthGeom", FALSE, NULL);
        }

     // Test 1
        if (GpiSetLineEnd(hPS, LINEEND_ROUND) == FALSE)
        {
          BADERROR("GpiSetLineEnd", FALSE, NULL);
        }

        ptl.x = delta_x * 1/2;
        ptl.y = delta_y * 5/2;
        MapPt(ptl);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = delta_x * 1/2;
        ptlShape[0].y = delta_y * 2;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 1;
        ptlShape[1].y = delta_y * 2;
        MapPt(ptlShape[1]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, 0L, delta_y * 5/2 + line_space,
           "End = ROUND");

     // Test 2
        if (GpiSetLineEnd(hPS, LINEEND_SQUARE) == FALSE)
        {
          BADERROR("GpiSetLineEnd", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 5/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 5/2);
        ptlShape[1].x = MapX(delta_x * 3);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 2, delta_y * 5/2 + line_space,
           "End = SQUARE");

     // Test 3
        if (GpiSetLineEnd(hPS, LINEEND_FLAT) == FALSE)
        {
          BADERROR("GpiSetLineEnd", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 9/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 9/2);
        ptlShape[1].x = MapX(delta_x * 5);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 4, delta_y * 5/2 + line_space,
           "End = FLAT");

     // Test 4
        if (GpiSetLineEnd(hPS, LINEEND_DEFAULT) == FALSE)
        {
          BADERROR("GpiSetLineEnd", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 13/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 13/2);
        ptlShape[1].x = MapX(delta_x * 7);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 6, delta_y * 5/2 + line_space,
           "End = DEFAULT");

  /*
   * Join test
   */
     // Test 1
        if (GpiSetLineJoin(hPS, LINEJOIN_MITRE) == FALSE)
        {
          BADERROR("GpiSetLineJoin", FALSE, NULL);
        }

        ptl.x = delta_x * 1/2;
        ptl.y = delta_y * 7/2;
        MapPt(ptl);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = delta_x * 1/2;
        ptlShape[0].y = delta_y * 3;
        MapPt(ptlShape[0]);
        ptlShape[1].x = delta_x * 1;
        ptlShape[1].y = delta_y * 3;
        MapPt(ptlShape[1]);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, 0L, delta_y * 7/2 + line_space,
           "Join = MITRE");

     // Test 2
        if (GpiSetLineJoin(hPS, LINEJOIN_ROUND) == FALSE)
        {
          BADERROR("GpiSetLineJoin", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 5/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 5/2);
        ptlShape[1].x = MapX(delta_x * 3);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 2, delta_y * 7/2 + line_space,
           "Join = ROUND");

     // Test 3
        if (GpiSetLineJoin(hPS, LINEJOIN_BEVEL) == FALSE)
        {
          BADERROR("GpiSetLineJoin", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 9/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 9/2);
        ptlShape[1].x = MapX(delta_x * 5);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 4, delta_y * 7/2 + line_space,
           "Join = BEVEL");

     // Test 4
        if (GpiSetLineJoin(hPS, LINEJOIN_DEFAULT) == FALSE)
        {
          BADERROR("GpiSetLineJoin", FALSE, NULL);
        }

        ptl.x = MapX(delta_x * 13/2);
        GpiMove(hPS, &ptl);

        ptlShape[0].x = MapX(delta_x * 13/2);
        ptlShape[1].x = MapX(delta_x * 7);

        if (GpiBeginPath(hPS, 1L) == FALSE)
        {
          BADERROR("GpiBeginPath", TRUE, "GpiSetLineExh");
          return;
        }
        GpiPolyLine(hPS, 2L, ptlShape);
        if (GpiEndPath(hPS) == FALSE)
        {
          BADERROR("GpiEndPath", TRUE, "GpiSetLineExh");
          return;
        }
        if (GpiStrokePath(hPS, 1L, 0L) == GPI_ERROR)
        {
          BADERROR("GpiStrokePath", FALSE, NULL);
        }
        MapAndWriteString(hPS, delta_x * 6, delta_y * 7/2 + line_space,
           "Join = DEFAULT");


  /*
   * Query geom width test
   */

        query_value = GpiQueryLineWidthGeom(hPS);
        sprintf(out_string, "Returned %ld, expected 10", query_value);

        MapAndWriteString(hPS, 0L, delta_y * 6,
           "GpiQueryLineWidthGeom test");
        MapAndWriteString(hPS, 0L, delta_y * 6 - line_space,
           out_string);

  /*
   * Query line end test
   */

        if (GpiSetLineEnd(hPS, LINEEND_SQUARE) == FALSE)
        {
          BADERROR("GpiSetLineEnd", FALSE, NULL);
        }

        query_value = GpiQueryLineEnd(hPS);
        sprintf(out_string, "Returned %ld, expected 2", query_value);

        MapAndWriteString(hPS, delta_x * 2, delta_y * 5,
           "GpiQueryLineEnd test");
        MapAndWriteString(hPS, delta_x * 2, delta_y * 5 - line_space,
           out_string);

  /*
   * Query Line Join test
   */

        if (GpiSetLineJoin(hPS, LINEJOIN_MITRE) == FALSE)
        {
          BADERROR("GpiSetLineJoin", FALSE, NULL);
        }

        query_value = GpiQueryLineJoin(hPS);
        sprintf(out_string, "Returned %ld, expected 3", query_value);

        MapAndWriteString(hPS, delta_x * 4, delta_y * 6,
           "GpiQueryLineJoin test");
        MapAndWriteString(hPS, delta_x * 4, delta_y * 6 - line_space,
           out_string);

    if (lErrorCount) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
       "$GpiSetLine Exhaustive Test                                     -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiSetLine Exhaustive Test                                     -    PASS.\n");

        CWRITELOG(L_LOTRACE, 0, "+ex+ GpiSetLineExh - Exit.\n");
  } /* end the ELSE */
} /* end GpiSetLineExh */


