/************************************************************************/
/* Module name: GPIMETA.C                                               */
/*                                                                      */
/* Purpose: Exhaustively tests all of the API's involved with           */
/*          Metafiles.                                                  */
/*                                                                      */
/* Author : Timothy S. Seevers                                          */
/*          IBM Information Products Company.                           */
/*          Lexington, KY                                               */
/*                                                                      */
/*                                                                      */
/* Functions:  InitTest                                                 */
/*             GpiMetaApp1                                              */
/*             GpiMetaApp2                                              */
/*             GpiCopyMetaFileExh                                       */
/*             GpiDeleteMetaFileExh                                     */
/*             GpiLoadPlayMetaFileExh1                                  */
/*             GpiLoadPlayMetaFileExh2                                  */
/*             GpiLoadPlayMetaFileExh3                                  */
/*             GpiQueryMetaFileLengthExh                                */
/*             GpiQuerySetMetaFileBitsExh                               */
/*             GpiSaveMetaFileExh                                       */
/*                                                                      */
/* Files Needed:                                                        */
/*                                                                      */
/*             gpimeta.c..........The file you are looking at.          */
/*             gpimeta.h..........Header which defines all entry points.*/
/*             gpimeta.def........Module definition file.               */
/*             gpimeta.mak........Make file                             */
/*             moremeta.h.........Includes: create_gpimeta1 and 2.
    (optional) clr.cmd............Compiles, Links, and Runs the dll     */
/*                                                                      */
/*                                                                      */
/************************************************************************/
#define INCL_WIN
#define INCL_GPI
#define INCL_DOSFILEMGR

#define OPEN_FILE 0x01
#define CREATE_FILE 0x10
#define FILE_ARCHIVE 0x20
#define FILE_EXISTS OPEN_FILE
#define FILE_NOEXISTS CREATE_FILE
#define DASD_FLAG 0
#define INHERIT 0x80
#define WRITE_THRU 0
#define FAIL_FLAG 0
#define SHARE_FLAG 0x10
#define ACCESS_FLAG 0x02
#define FXTOLONG  65536L       // Multiplier to convert FIXED to LONG

#define FILE_SIZE 800L
#define FILE_ATTRIBUTE FILE_ARCHIVE
#define RESERVED 0L


#include <string.h>
#include <stdio.h>
#include <math.h>
#include <os2.h>
#include <io.h>
#include "PTTDLLDF.h"                            // test case defines
#include "std_util.h"
#include "gpimeta.h"                             // Entry point calls
#include "moremeta.h"

PCHAR pGroupName = "GpiMeta";                    // group name of the tests
PCHAR pTestName1 = "GpiMetaApp1";                // spiral
PCHAR pTestName2 = "GpiMetaApp2";                // boxes
PCHAR pTestName3 = "GpiCopyMetaFileExh";         // copy of original
PCHAR pTestName4 = "GpiDeleteMetaFileExh";       // no output
PCHAR pTestName5 = "GpiLoadPlayMetaFileExh1";    // circles
PCHAR pTestName6 = "GpiLoadPlayMetaFileExh2";    // colors
PCHAR pTestName7 = "GpiLoadPlayMetaFileExh3";    // colors
PCHAR pTestName8 = "GpiQueryMetaFileLengthExh";  // no output
PCHAR pTestName9 = "GpiQuerySetMetaFileBitsExh"; // boxes
PCHAR pTestName10= "GpiSaveMetaFileExh";         // no output

// used to get information about the current operating environment
TESTENVINFO testenvinfo;


/**************************************************************
*  Function Name: InitTest(USHORT sel,PTESTINITINFO pInitInfo)*
*                                                             *
*  Purpose:  Establish a link with PTT and initialize pointers*
*            into the shared segment (shared with PTT).       *
*                                                             *
*  Function Calls: InitTestSegVars                            *
*                                                             *
*  Returns:  VOID                                             *
**************************************************************/
VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO pInitInfo){

    pInitInfo->ntests = NTESTS;
    strcpy(pInitInfo->group_name,pGroupName);
    strcpy(pInitInfo->test_names[0],pTestName1);
    pInitInfo->test_addrs[0]=GpiMetaApp1;
    strcpy(pInitInfo->test_names[1],pTestName2);
    pInitInfo->test_addrs[1]=GpiMetaApp2;
    strcpy(pInitInfo->test_names[2],pTestName3);
    pInitInfo->test_addrs[2]=GpiCopyMetaFileExh;
    strcpy(pInitInfo->test_names[3],pTestName4);
    pInitInfo->test_addrs[3]=GpiDeleteMetaFileExh;
    strcpy(pInitInfo->test_names[4],pTestName5);
    pInitInfo->test_addrs[4]=GpiLoadPlayMetaFileExh1;
    strcpy(pInitInfo->test_names[5],pTestName6);
    pInitInfo->test_addrs[5]=GpiLoadPlayMetaFileExh2;
    strcpy(pInitInfo->test_names[6],pTestName7);
    pInitInfo->test_addrs[6]=GpiLoadPlayMetaFileExh3;
    strcpy(pInitInfo->test_names[7],pTestName8);
    pInitInfo->test_addrs[7]=GpiQueryMetaFileLengthExh;
    strcpy(pInitInfo->test_names[8],pTestName9);
    pInitInfo->test_addrs[8]=GpiQuerySetMetaFileBitsExh;
    strcpy(pInitInfo->test_names[9],pTestName10);
    pInitInfo->test_addrs[9]=GpiSaveMetaFileExh;

    InitTestSegVars(sel);
}  // end InitTest



/**************************************************************
*  Function Name: GpiMetaApp1(HPS, HWND, BOOL SelectionCall)  *
*                                                             *
*  Purpose:  General test of metafiles.  A metafile is        *
*            created and played.  The metafile includes       *
*            the use of colors and path.                      *
*                                                             *
*  Function Calls: Creategpimeta1()                           *
*                                                             *
*  API Calls:      GpiLoadMetaFile()                          *
*                  GpiPlayMetaFile()                          *
*                                                             *
*  Returns:  VOID                                             *
**************************************************************/
VOID APIENTRY GpiMetaApp1 (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HAB    habOut;
   HMF    hmfPlay;
   LONG   lPlayRC;
   LONG   playOptionCount;
   LONG   playOptionArray[8];
   LONG   playSegCount;
   LONG   playDescLength;
   CHAR   playDescBuff[251];
   PSZ    pszError;
   USHORT usRC;

   static DEVOPENSTRUC  dop = { NULL, "DISPLAY" };


   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiMetaTestApp1               - Entry\n");

      habOut = WinInitialize (0);
      pszError = Create_gpimeta1();
      CWRITELOG( L_TRACE, 0, pszError );

  //  Load and play the metafile

         hmfPlay = GpiLoadMetaFile( habOut, "gpimeta1.met" );

         if (hmfPlay != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_ORIGINALVIEW;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_RESET;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

         if ( hmfPlay == GPI_ERROR )
             CWRITELOG(L_TRACE,0,"GpiLoadMetaFile               - Failed\n");


         lPlayRC = GpiPlayMetaFile( hPS,
                              hmfPlay,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );
        }

         if ( lPlayRC != GPI_OK ) {
             usRC = ERRORIDERROR( WinGetLastError( habOut ) );
             CWRITELOG(L_TRACE,0,"GpiPlayMetaFile               - Failed\n");
             CWRITELOG(L_HDR,0,
"$GpiMetaApp1                                              - Failed\n");
           }
           else
             CWRITELOG(L_HDR,0,
"$GpiMetaApp1                                              - Passed\n");



  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiMetaTestApp1               - Exit\n");

} // end GpiMetaApp1


/**************************************************************
*  Function Name: GpiMetaApp2HPS, HWND, BOOL SelectionCall)  *
*                                                             *
*  Purpose:  General test of metafiles.  A metafile is        *
*            created and played.  The metafile includes       *
*            the use of GpiBox                                *
*                                                             *
*  Function Calls: Creategpimeta2()                           *
*                                                             *
*  API Calls:      GpiLoadMetaFile()                          *
*                  GpiPlayMetaFile()                          *
*                                                             *
*  Returns:  VOID                                             *
**************************************************************/
VOID APIENTRY GpiMetaApp2 (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HAB    habOut;
   HMF    hmfPlay;
   LONG   lPlayRC;
   LONG   playOptionCount;
   LONG   playOptionArray[8];
   LONG   playSegCount;
   LONG   playDescLength;
   CHAR   playDescBuff[251];
   PSZ    pszError;

   static DEVOPENSTRUC  dop = { NULL, "DISPLAY" };


   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiMetaTestApp2               - Entry\n");

      habOut = WinInitialize( 0 );
      pszError = Create_gpimeta2();
      CWRITELOG( L_TRACE, 0, pszError );

  //  Load and play the metafile

         hmfPlay = GpiLoadMetaFile( habOut, "gpimeta2.met" );

         if (hmfPlay != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_ORIGINALVIEW;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_RESET;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

         if ( hmfPlay == GPI_ERROR )
             CWRITELOG(L_TRACE,0,"GpiLoadMetaFile               - Failed\n");

         lPlayRC = GpiPlayMetaFile( hPS,
                              hmfPlay,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );
        }

         if ( lPlayRC != GPI_OK ) {
             CWRITELOG(L_TRACE,0,"GpiPlayMetaFile               - Failed\n");
             CWRITELOG(L_HDR,0,
"$GpiMetaApp2                                              - Failed\n");
           }
           else
             CWRITELOG(L_HDR,0,
"$GpiMetaApp2                                              - Passed\n");


  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiMetaTestApp2               - Exit\n");

} // end GpiMetaApp2



/***********************************************************************
*  Function Name: GpiCopyMetaFileExh(HPS, HWND, BOOL SelectionCall)    *
*                                                                      *
*  Purpose:  Exhaustively test the GpiCopyMetaFileExh() function call  *
*                                                                      *
*  API Calls:  DevOpenDC()                                             *
*              DevCloseDC()                                            *
*              GpiCopyMetaFile()                                       *
*              GpiPlayMetaFile()                                       *
*                                                                      *
*  Returns:  VOID                                                      *
*                                                                      *
*  Log    :                                                            *
*  06/11/91   D. Scholten  J. Read                                     *
*             Fixed wrong handle in GpiLoadMetaFile                    *
*             Added GpiASsociate to disassociate PS                    *
*             Added Gpi call error checking                            *
*                                                                      *
*                                                                      *
*                                                                      *
************************************************************************/
VOID APIENTRY GpiCopyMetaFileExh (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HPS    hpsMeta;
   HAB    hab;
   HDC    hdcMetafile;
   SIZEL  sizlPage;
   HMF    hmfPicture;
   HMF    hmfPicture2;
   LONG   loadrc = 0L;
   LONG playOptionCount;
   LONG playOptionArray[8];
   LONG playSegCount;
   LONG playDescLength;
   CHAR playDescBuff[251];
   USHORT usRC;

   static DEVOPENSTRUC  dop = { NULL, "DISPLAY" };
   static POINTL        apt;


   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE,0,"GpiCopyMetaFileExh            - Entry\n");
      set_delta(hPS);

  // Open metafile device context

          hdcMetafile = DevOpenDC ( hab, OD_METAFILE, "*", 2L,
                                    (PDEVOPENDATA) &dop, NULLHANDLE);
        if (hdcMetafile == DEV_ERROR)
          CWRITELOG(L_TRACE,0,"DevOpenDC     - failed\n");

  //  Create a presentation space associated with that DC

          sizlPage.cx = 800;
          sizlPage.cy = 1100;
          hpsMeta = GpiCreatePS( hab, hdcMetafile, &sizlPage,
                                  GPIA_ASSOC | PU_LOMETRIC );
        if (hpsMeta == GPI_ERROR)
          CWRITELOG(L_TRACE,0,"GpiCreatePS     - failed\n");

  //  Set current pointer position

    //TSS...added to delta multiplication to account for ps_units...9/3/91
          set_delta(hPS);
          apt.x = 0;
          apt.y = 4*delta_y;
          MapPt( apt );
          if ((GpiMove( hpsMeta, &apt )) == FALSE)
            CWRITELOG(L_TRACE,0,"GpiMove     - failed\n");

  //  Output a character string

        if ((GpiCharString( hpsMeta, 27L, "GpiCopyMetaFileExh - Passed" )) ==
             GPI_ERROR)
          CWRITELOG(L_TRACE,0,"GpiCharString  - failed\n");

  //    Disassociate the presentation space from the DC

         if( GPI_ERROR == GpiAssociate( hpsMeta, (HDC)NULL ) )
         {
            usRC = ERRORIDERROR( WinGetLastError( hab ) );
            CWRITELOG(L_TRACE,0,"disassociate PS in CopyMeta- Failed\n");
         }

  //    Destroy the PS

          if ((GpiDestroyPS( hpsMeta )) == FALSE)
            CWRITELOG(L_TRACE,0,"GpiDestoryPS       - Failed\n");

  //    Close the device context to get the metafile handle

          hmfPicture = DevCloseDC( hdcMetafile );
          if (hmfPicture == DEV_ERROR)
            CWRITELOG(L_TRACE,0,"DevCloseDC         - Failed\n");

  //    Save the metafile to disk

          DosDelete( "text.met" );

         if (GpiSaveMetaFile( hmfPicture, "text.met" ) == FALSE)
            CWRITELOG(L_TRACE,0,"SaveMetaFile       - Failed\n");

  //    Load the newly created metafile

          hmfPicture = GpiLoadMetaFile( hab, "text.met" );
          if (hmfPicture == GPI_ERROR)
            CWRITELOG(L_TRACE,0,"LoadMetaFile       - Failed\n");

  //  Copy the metafile

          hmfPicture2 = GpiCopyMetaFile( hmfPicture );
          if (hmfPicture2 == GPI_ERROR)
            CWRITELOG(L_TRACE,0,"CopyMetaFile2      - Failed\n");

  //    Save the copied metafile to disk

         DosDelete( "text2.met" );

         if (GpiSaveMetaFile( hmfPicture2, "text2.met" ) == FALSE)
            CWRITELOG(L_TRACE,0,"SaveMetaFile2      - Failed\n");

  //    Load the newly copied metafile

          hmfPicture2 = GpiLoadMetaFile( hab, "text2.met" );
          if (hmfPicture2 == GPI_ERROR)
            CWRITELOG(L_TRACE,0,"LoadMetaFile2      - Failed\n");

         if (hmfPicture2 != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_DEFAULT;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_DEFAULT;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;


  //  Play the newly copied metafile

         loadrc = GpiPlayMetaFile( hPS,
                              hmfPicture2,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );

         if ( loadrc != GPI_OK ) {
             usRC = ERRORIDERROR( WinGetLastError( hab ) );
             CWRITELOG(L_TRACE,0,"GpiPlayMetaFile2              - Failed\n");
             CWRITELOG(L_HDR,0,
"$GpiCopyMetaFileExh                                       - Failed\n");
          }

           else
             CWRITELOG(L_HDR,0,
"$GpiCopyMetaFileExh                                       - Passed\n");


  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiCopyMetaFileExh            - Exit\n");
  }
} // end GpiCopyMetaFileExh



/*************************************************************************
*  Function Name: GpiDeleteMetaFileExh(HPS, HWND, BOOL SelectionCall)    *
*                                                                        *
*  Purpose:  Exhaustively test GpiDeleteMetaFile()                       *
*                                                                        *
*  Function Calls:  DevOpenDC()                                          *
*                   DevCloseDC()                                         *
*                   GpiDeleteMetaFile()                                  *
*                   GpiPlayMetaFile()                                    *
*                                                                        *
*  Returns:  VOID                                                        *
**************************************************************************/
VOID APIENTRY GpiDeleteMetaFileExh (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HPS    hpsMeta;
   HAB    hab;
   HDC    hdcMetafile;
   SIZEL  sizlPage;
   HMF    hmfPicture;
   LONG   loadrc;
   LONG   playOptionCount;
   LONG   playOptionArray[8];
   LONG   playSegCount;
   LONG   playDescLength;
   CHAR   playDescBuff[251];

   static DEVOPENSTRUC  dop = { NULL, "DISPLAY" };
   static POINTL        apt;


   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiDeleteMetaFileExh          - Entry\n");
      set_delta(hPS);

  // Open metafile device context

          hdcMetafile = DevOpenDC ( hab, OD_METAFILE, "*", 2L,
                                    (PDEVOPENDATA) &dop, NULLHANDLE);

  //  Create a presentation space associated with that DC

          sizlPage.cx = 800;
          sizlPage.cy = 1100;
          hpsMeta = GpiCreatePS( hab, hdcMetafile, &sizlPage,
                                 GPIA_ASSOC | PU_LOMETRIC );

  //  Set current pointer position

          apt.x = 200;
          apt.y = 200;
          GpiMove( hpsMeta, &apt );

  //  Output a character string

          GpiCharString( hpsMeta, 37L, "If you can see me, testcase Failed.  " );


  //    Disasscociate the presentation space

         if( GPI_ERROR == GpiAssociate( hpsMeta, (HDC)NULL ) )
         {
             ERRORIDERROR( WinGetLastError( hab ) );
             CWRITELOG(L_TRACE,0,"Gpidisassociate               - Failed\n");
         }


  //    Destroy the presentation space

          GpiDestroyPS( hpsMeta );

  //    Close the device context to get the metafile handle

          hmfPicture = DevCloseDC( hdcMetafile );
          if( hmfPicture == DEV_ERROR )
             CWRITELOG(L_TRACE,0,"DevCloseDC                    - Failed\n");

  //    Save the metafile to disk

         DosDelete( "text.met" );
         if( GpiSaveMetaFile( hmfPicture, "text.met" ) == FALSE )
             CWRITELOG(L_TRACE,0,"GpiSaveMetaFile               - Failed\n");

  //  Load the newly created metafile

         hmfPicture = GpiLoadMetaFile( hab, "text.met" );

  //  Delete the Metafile from memory.

         if( GpiDeleteMetaFile( hmfPicture ) == FALSE )
             CWRITELOG(L_TRACE,0,"GpiDeleteSaveMetaFile             - Failed\n");

  //  Play the newly loaded metafile

         if (hmfPicture != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_DEFAULT;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_DEFAULT;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

         loadrc = GpiPlayMetaFile( hPS,
                              hmfPicture,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );
        }

         if ( loadrc == GPI_OK ) {
             CWRITELOG(L_HDR,0,
"$GpiDeleteMetaFileExh                                     - Failed\n");
           }
           else {
             CWRITELOG(L_HDR,0,
"$GpiDeleteMetaFileExh                                     - Passed\n");


             set_delta(hPS);
             apt.x = 0;
             apt.y = 4*delta_y;
             MapPt( apt );
             GpiMove( hPS, &apt );
             GpiCharString( hPS, 24L, "GpiDeleteMetaFile Passed" );
            }  // end the if/then/else


  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiDeleteMetaFileExh          - Exit\n");

} // end GpiDeleteMetaFileExh


/*************************************************************************
*  Function Name: GpiLoadPlayMetaFileExh1(HPS, HWND, BOOL SelectionCall) *
*                                                                        *
*  Purpose:  Exhaustively test GpiLoadMetaFile() and GpiPlayMetaFile()   *
*            Contains: no color, no path, units forced.                  *
*                                                                        *
*  Function Calls:  GpiLoadMetaFile()                                    *
*                   GpiPlayMetaFile()                                    *
*  Returns:  VOID                                                        *
**************************************************************************/
VOID APIENTRY GpiLoadPlayMetaFileExh1 (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HMF  hmfPicture;
   HAB  hab;
   LONG loadrc;
   LONG playOptionCount;
   LONG playOptionArray[8];
   LONG playSegCount;
   LONG playDescLength;
   CHAR playDescBuff[251];


   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiLoadPlayMetaFileExh1       - Entry\n");
      set_delta(hPS);

      hmfPicture = GpiLoadMetaFile( hab, "gpimeta3.met" );

  if (hmfPicture != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_ORIGINALVIEW;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_RESET;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

       loadrc = GpiPlayMetaFile( hPS,
                              hmfPicture,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );

      } // end the if

         if ( loadrc != GPI_OK ) {
             CWRITELOG(L_TRACE,0,"GpiPlayMetaFile               - Failed\n");
             CWRITELOG(L_TRACE,0,"Couldn't load gpimeta3.met.\n");
             CWRITELOG(L_HDR,0,
"$GpiLoadPlayMetaFileExh1                                  - Failed\n");
           }
           else
             CWRITELOG(L_HDR,0,
"$GpiLoadPlayMetaFileExh1                                  - Passed\n");


  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiLoadPlayMetaFileExh1       - Exit\n");

} // end GpiLoadPlayMetaFileExh1


/*************************************************************************
*  Function Name: GpiLoadPlayMetaFileExh2(HPS, HWND, BOOL SelectionCall) *
*                                                                        *
*  Purpose:  Exhaustively test GpiLoadMetaFile() and GpiPlayMetaFile()   *
*            Contains: color, path graphics , nonforced units.           *
*                                                                        *
*  Function Calls:  GpiLoadMetaFile()                                    *
*                   GpiPlayMetaFile()                                    *
*  Returns:  VOID                                                        *
**************************************************************************/
VOID APIENTRY GpiLoadPlayMetaFileExh2 (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HMF  hmfPicture;
   HAB  hab;
   LONG loadrc;
   LONG playOptionCount;
   LONG playOptionArray[8];
   LONG playSegCount;
   LONG playDescLength;
   CHAR playDescBuff[251];

   static POINTL apt;

   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiLoadPlayMetaFileExh2       - Entry\n");
      set_delta(hPS);

      hmfPicture = GpiLoadMetaFile( hab, "gpimeta4.met" );

  if (hmfPicture != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_DEFAULT;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_DEFAULT;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

       loadrc = GpiPlayMetaFile( hPS,
                              hmfPicture,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );

      } // end the if


         apt.x = 0;
         apt.y = ((LONG)(1*delta_y));
         MapPt( apt );
         GpiMove( hPS, &apt );
         GpiCharString( hPS, 43L, "OBJECT'S APPEARANCE MAY DIFFER WHEN PRINTED" );
         GpiCharString( hPS, 30L, " - UNIT MUST BE SET TO PU_PELS" );


         if ( loadrc != GPI_OK ) {
             CWRITELOG(L_TRACE,0,"GpiPlayMetaFile               - Failed\n");
             CWRITELOG(L_TRACE,0,"Couldn't load gpimeta4.met.\n");
             CWRITELOG(L_HDR,0,
"$GpiLoadPlayMetaFileExh2                                  - Failed\n");
           }
           else
             CWRITELOG(L_HDR,0,
"$GpiLoadPlayMetaFileExh2                                  - Passed\n");



  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiLoadPlayMetaFileExh2       - Exit\n");

} // end GpiLoadPlayMetaFileExh2



/*************************************************************************
*  Function Name: GpiLoadPlayMetaFileExh3(HPS, HWND, BOOL SelectionCall) *
*                                                                        *
*  Purpose:  Exhaustively test GpiLoadMetaFile() and GpiPlayMetaFile()   *
*            Contains: color, path graphics , forced units.              *
*                                                                        *
*  Function Calls:  GpiLoadMetaFile()                                    *
*                   GpiPlayMetaFile()                                    *
*  Returns:  VOID                                                        *
**************************************************************************/
VOID APIENTRY GpiLoadPlayMetaFileExh3 (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HMF  hmfPicture;
   HAB  hab;
   LONG loadrc;
   LONG playOptionCount;
   LONG playOptionArray[8];
   LONG playSegCount;
   LONG playDescLength;
   CHAR playDescBuff[251];

   static POINTL apt;

   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiLoadPlayMetaFileExh3       - Entry\n");
      set_delta(hPS);

      apt.x = 0;
      apt.y = delta_y;
      MapPt( apt );
      GpiMove( hPS, &apt );
      GpiCharString( hPS, 31L, "SPOOLER FORMAT MUST BE PM_Q_RAW" );


      hmfPicture = GpiLoadMetaFile( hab, "gpimeta5.met" );

  if (hmfPicture != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_ORIGINALVIEW;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_RESET;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

       loadrc = GpiPlayMetaFile( hPS,
                              hmfPicture,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );

      } // end the if


         if ( loadrc != GPI_OK ) {
             CWRITELOG(L_TRACE,0,"GpiPlayMetaFile               - Failed\n");
             CWRITELOG(L_TRACE,0,"Couldn't load gpimeta5.met.\n");
             CWRITELOG(L_HDR,0,
"$GpiLoadPlayMetaFileExh3                                  - Failed\n");
           }
           else
             CWRITELOG(L_HDR,0,
"$GpiLoadPlayMetaFileExh3                                  - Passed\n");


  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiLoadPlayMetaFileExh3       - Exit\n");

} // end GpiLoadPlayMetaFileExh3


/***************************************************************************
*  Function Name: GpiQueryMetaFileLengthExh(HPS, HWND, BOOL SelectionCall) *
*                                                                          *
*  Purpose:  Exhaustively test GpiQueryMetaFileLengthExh().                *
*                                                                          *
*  Function Calls:  GpiLoadMetaFile()                                      *
*                   GpiQueryMetaFileLength()                               *
*  Returns:  VOID                                                          *
****************************************************************************/
VOID APIENTRY GpiQueryMetaFileLengthExh (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HAB  hab;
   LONG lLength, lLength2;
   HMF  hmfPicture;
   FILE *fmeta;

   static POINTL apt;

   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiQueryMetaFileLengthExh     -Entry\n");
      set_delta(hPS);


   // TSS...added code to query length of file instead of hardcode.

      if( (fmeta = fopen("gpimeta3.met", "r")) == NULL )
          {
          CWRITELOG(L_TRACE,0,"Could not open gpimeta3.met   - Failed\n");
          lLength = -1;
          }
      else
          {
          if( (lLength2 = filelength(_fileno(fmeta))) == -1L )
               CWRITELOG(L_TRACE,0,"filelength function           - Failed\n");

          fclose( fmeta );
          }

      hmfPicture = GpiLoadMetaFile( hab, "gpimeta3.met" );

      if (hmfPicture == (HMF)NULL)
           CWRITELOG(L_TRACE,0,"GpiLoadMetaFile               - Failed\n");

      lLength = GpiQueryMetaFileLength( hmfPicture );

      if ( lLength == lLength2 ) {
           CWRITELOG(L_HDR,0,
"$GpiQueryMetaFileLengthExh                                - Passed\n");

           set_delta(hPS);
           apt.x = 0;
           apt.y = delta_y*4;
           MapPt( apt );
           GpiMove( hPS, &apt );
           GpiCharString( hPS, 29L, "GpiQueryMetaFileLength Passed" );


      } else {

           CWRITELOG(L_HDR,0,
"$GpiQueryMetaFileLengthExh                                - Failed\n");

           set_delta(hPS);
           apt.x = 0;
           apt.y = delta_y*4;
           MapPt( apt );
           GpiMove( hPS, &apt );
           GpiCharString( hPS, 29L, "GpiQueryMetaFileLength Failed" );

      } /* endif */


  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiQueryMetaFileLengthExh     - Exit\n");

} // end GpiQueryMetaFileLengthExh




/****************************************************************************
*  Function Name: GpiQuerySetMetaFileBitsExh(HPS, HWND, BOOL SelectionCall) *
*                                                                           *
*  Purpose:  Exhaustively test GpiQuerySetMetaFileBits()                    *
*                                                                           *
*  Function Calls:  GpiLoadMetaFile()                                       *
*                   GpiQueryMetaFileBits()                                  *
*                   GpiSetMetaFileBits()                                    *
*                   GpiPlayMetaFile()                                       *
*                                                                           *
*                                                                           *
*  Returns:  VOID                                                           *
*****************************************************************************/
VOID APIENTRY GpiQuerySetMetaFileBitsExh (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HAB     hab;
   LONG    lLength;
   HMF     hmfPicture;
   LONG    lOffset = 0L;
   PBYTE   pbData;
   LONG    loadrc;
   LONG    playOptionCount;
   LONG    playOptionArray[8];
   LONG    playSegCount;
   LONG    playDescLength;
   CHAR    playDescBuff[251];
   BOOL    bSuccess;
   ERRORID error;

   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiQuerySetMetaFileBitsExh    - Entry\n" );
      set_delta(hPS);

      hmfPicture = GpiLoadMetaFile( hab, "gpimeta2.met" );

  if (hmfPicture == (HMF)NULL)
           CWRITELOG(L_TRACE,0,"GpiLoadMetaFile               - Failed\n");

      lLength = GpiQueryMetaFileLength( hmfPicture );


   //  Copy a unit of the metafile data into memory.

   //32 bit memory allocation
   if( DosAllocMem((PVOID)&pbData,lLength,
                   PAG_READ | PAG_WRITE | PAG_COMMIT) != 0)
      CWRITELOG(L_TRACE,0,"DosAllocMem                   - Failed\n");

      bSuccess = GpiQueryMetaFileBits( hmfPicture, lOffset, lLength, pbData );
      if ( bSuccess == FALSE )
           CWRITELOG(L_TRACE,0,"GpiQueryMetafFileBits         - Failed\n");

       error = WinGetLastError( hab );

   //  Copy the data in memory back into the metafile.

      bSuccess = GpiSetMetaFileBits( hmfPicture, lOffset, lLength, pbData );
      if ( bSuccess == FALSE )
           CWRITELOG(L_TRACE,0,"GpiSetMetafFileBits           - Failed\n");


  if (hmfPicture != (HMF)NULL) {
            playDescLength = 250;
            playOptionCount = 8L;
            playOptionArray[PMF_SEGBASE] = 0L;
            playOptionArray[PMF_LOADTYPE] = LT_ORIGINALVIEW;
            playOptionArray[PMF_RESOLVE] = RS_DEFAULT;
            playOptionArray[PMF_LCIDS] = LC_LOADDISC;
            playOptionArray[PMF_RESET] = RES_RESET;
            playOptionArray[PMF_SUPPRESS] = SUP_DEFAULT;
            playOptionArray[PMF_COLORTABLES] = CTAB_REPLACE;
            playOptionArray[PMF_COLORREALIZABLE] = CREA_DEFAULT;

       loadrc = GpiPlayMetaFile( hPS,
                              hmfPicture,
                              playOptionCount,
                             (PLONG)playOptionArray,
                              &playSegCount,
                              playDescLength,
                             (PSZ)&playDescBuff[0]
                            );

      } // end the if
      else
         CWRITELOG(L_TRACE,0,"Metafile was not successfully loaded\n");


      if ( loadrc != GPI_OK ) {
         CWRITELOG(L_HDR,0,
"$GpiQueryMetaFileBitsExh                                  - Failed\n");
      } else {
         CWRITELOG(L_HDR,0,
"$GpiQuerySetMetaFileBitsExh                               - Passed\n");
      } /* endif */


  } // end the major else


   CWRITELOG(L_TRACE,0,"GpiQuerySetMetaFileBitsExh    - Exit\n");

   DosFreeMem(pbData);
} // end GpiQuerySetMetaFileBitsExh






/*************************************************************************
*  Function Name: GpiSaveMetaFileExh(HPS, HWND, BOOL SelectionCall)      *
*                                                                        *
*  Purpose:  Exhaustively test GpiSaveMetaFile()                         *
*                                                                        *
*  Function Calls:  GpiLoadMetaFile()                                    *
*                   GpiSaveMetaFile()                                    *
*                   GpiLoadMetaFile()                                    *
*                   GpiPlayMetaFile()                                    *
*                                                                        *
*  Returns:  VOID                                                        *
**************************************************************************/
VOID APIENTRY GpiSaveMetaFileExh (HPS hPS,HWND hWndClient,BOOL SelectionCall){

   HPS    hpsMeta;
   HAB    hab;
   HDC    hdcMetafile;
   SIZEL  sizlPage;
   HMF    hmfPicture;
   USHORT usRC;

   static DEVOPENSTRUC  dop = { NULL, "DISPLAY" };
   static POINTL        apt;


   if (SelectionCall) {
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
      *pBasisX = 1000;
      *pBasisY = 1000;
    }
   else {
      WinSendMsg (hWndClient, WM_GET_ENV_INFO, MPFROMP(&testenvinfo), NULL);

      TEST_SIGNATURE(0,0);
      LOG_SIGNATURE;
      CWRITELOG(L_TRACE, 0, "GpiSaveMetaFileExh            - Entry\n");
      set_delta(hPS);

  // Open metafile device context

          hdcMetafile = DevOpenDC ( hab, OD_METAFILE, "*", 2L,
                                    (PDEVOPENDATA) &dop, NULLHANDLE);

  //  Create a presentation space associated with that DC

          sizlPage.cx = 800;
          sizlPage.cy = 1100;
          hpsMeta = GpiCreatePS( hab, hdcMetafile, &sizlPage,
                                 GPIA_ASSOC | PU_LOMETRIC );

  //  Set current pointer position

          apt.x = 200;
          apt.y = 200;
          GpiMove( hpsMeta, &apt );

  //  Output a character string

          GpiCharString( hpsMeta, 4L, "test" );

  //    Disassociate the presentation space from the DC

         if( GPI_ERROR == GpiAssociate( hpsMeta, (HDC)NULL ) )
         {
            usRC = ERRORIDERROR( WinGetLastError( hab ) );
            CWRITELOG(L_TRACE,0,"disassociate PS in SaveMeta- Failed\n");
         }

  //    Destroy the presentation space

          GpiDestroyPS( hpsMeta );

  //    Close the device context to get the metafile handle

          hmfPicture = DevCloseDC( hdcMetafile );


  //    Save the metafile to disk

         DosDelete( "savetext.met" );
         GpiSaveMetaFile( hmfPicture, "savetext.met" );

  //    Load the newly created metafile

      hmfPicture = GpiLoadMetaFile( hab, "savetext.met" );

      if ( hmfPicture == GPI_ERROR) {
             CWRITELOG(L_HDR,0,
"$GpiSaveMetaFileExh                                       - Failed\n");

             set_delta(hPS);
             apt.x = 0;
             apt.y = delta_y*4;
             MapPt( apt );
             GpiMove( hPS, &apt );
             GpiCharString( hPS, 25L, "GpiSaveMetaFileExh Failed" );

      } else {
             CWRITELOG(L_HDR,0,
"$GpiSaveMetaFileExh                                       - Passed\n");


             set_delta(hPS);
             apt.x = 0;
             apt.y = delta_y*4;
             MapPt( apt );
             GpiMove( hPS, &apt );
             GpiCharString( hPS, 25L, "GpiSaveMetaFileExh Passed" );

      } /* endif */

  } // end the major else

   CWRITELOG(L_TRACE,0,"GpiSaveMetaFileExh            -Exit\n");

} // end GpiSaveMetaFileExh
