/***************************************************************************
 *\\ddd
 * Subtask Name:  PDVT
 *
 * Module name:  gpimark.c
 *
 * Purpose:      This module is called by the PTT to test GPI Marker APIs.
 *
 * Revision Log: 12/03/90, PDVT, Kim Cairns
 *               - Original module template
 *               01/17/91, PDVT, Kim Cairns
 *               - Added RC entry points
 *               8/03/91, Vish (Boca)
 *               - The GpiSetMarkerBox, wherever it was being called, was
 *               using a LONG for the marker-box size where a FIXED variable
 *               was required. So, I changed the sizes to values of FIXED type.
 *               A marked difference can be seen now in the various sizes
 *               of markers that are selected ONLY from a vector font.
 *
 * SRS References:  Section 5.13
 *
 * Contents:
 *               InitTest()
 *               gpi_mark_pre()
 *               gpi_mark_app()
 *               GpiMarkerExh
 *               GpiPolyMarkerExh
 *
 * Design Overview:
 *   This testcase covers the GPI Marker APIs with respect to precision,
 *   basic functionality at the lowest level possible, and return codes that
 *   pertain to the printer driver.
 *
 * Limitations:
 *               None
 *
 *\\end
 ***************************************************************************/

#define INCL_WIN
#define INCL_GPI
#define INCL_ERRORS
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include "PTTDLLDF.h" // test case defines
#include "gpimark.h"  // gpi standard grid dll header file
#include "gpi_grid.h" // gpi standard grid functions header file
#include "std_util.h"  /* screen comments */

#define LCID_MYCOUR    1L
#define LCID_MYHELV    2L
#define LCID_MYTIMES   3L

static SHORT sFontSize[6] = {80, 100, 120, 140, 180, 240};
static CHAR *szFacename[4] = {"System Proportional", "Courier", "Helv", "Tms Rmn"};
static LONG alMatch[4][6];
BOOL BigQueryFonts(HPS hPS);
 /***************************************************************************
 *\\ddd
 * Routine name: InitTest()
 *
 * Purpose:  Initialize function name and address array for ptt.
 *
 * System Requirements:  None
 *
 * Revision Log:  12/03/90, PDVT, Kim Cairns
 *                - initial function design
 *
 * Inputs:
 *             sel            - pass thru parameter for other function call
 *             pInitInfo      - pointer to info array
 *
 * Outputs:
 *             pInitInfo      - updated with all entry points names and addresses
 *
 * Subroutines Required:
 *             InitTestSegVars()
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *   for all test functions in this test case {
 *       pInitInfo->function name = function name[i]
 *       pInitInfo->function address = function address[i]
 *   }
 *   call init set function
 *\\end
 ***************************************************************************/
VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO pInitInfo)
  {

    /**** Initialize total number of tests in test case ****/

    pInitInfo->ntests = NTESTS;

    /**** Load test names and entry point addresses ****/

    strcpy(pInitInfo->group_name,"Gpi Marker");
    strcpy(pInitInfo->test_names[0],"GpiMarkerPre");
    pInitInfo->test_addrs[0]=gpi_mark_pre;
    strcpy(pInitInfo->test_names[1],"GpiMarkerApp");
    pInitInfo->test_addrs[1]=gpi_mark_app;
    strcpy(pInitInfo->test_names[2],"GpiMarkerExh");
    pInitInfo->test_addrs[2]=GpiMarkerExh;
    strcpy(pInitInfo->test_names[3],"GpiPolyMarkerExh");
    pInitInfo->test_addrs[3]=GpiPolyMarkerExh;

    InitTestSegVars(sel);
  }

 /***************************************************************************
 *\\ddd
 * Routine name:  gpi_mark_pre()
 *
 * Purpose:  This is an entry point for the PTT that causes the GPI Marker
 *           precision test to be run.
 *
 * System Requirements:  Section 5.13.2
 *
 * Revision Log:  12/03/90, PDVT, Kim Cairns
 *                - initial function design
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall   - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()    - sets global variables for current device
 *             gpi_grid()     - standard grid function using GPIs
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *  //create grid with 1" squares
 *
 *  gre_grid (hPS, INCH);
 *
 *  //set marker to each different default symbol
 *
 *  GpiSetMarker (hPS, lSymbol);
 *
 *  //position markers at specific points on the grid
 *
 *  GpiMarker (hPS, &ptl);
 *  GpiPolyMarker (hPS, lNumber, aptl);
 *
 *\\end
 ***************************************************************************/

VOID APIENTRY gpi_mark_pre(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  CHAR    out_string[80];
  POINTL  ptl, textptl, aptl[10];
  LONG    cPoints;
  LONG    cHits;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiMarkPre - Entry.\n");

    lErrorCount = 0L;
    cPoints = 1;

    // This function sets delta_x and delta_y for the current device

    set_delta(hPS);

      // This sets the squares of the grid to 1" x 1"

    gpi_grid (hPS, (LONG)INCH);

    // Map point at (1,1) and draw marker

    ptl.x = delta_x;
    ptl.y = delta_y;
    MapPt(ptl);
    cHits =GpiMarker(hPS, &ptl);                        // default at 1,1
    if (cHits == GPI_ERROR)
    {
        lErrorCount+= BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 0;
    textptl.y = delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"1,1 Default");

    // Change marker symbol and draw marker at (3,1)

    if (GpiSetMarker(hPS, MARKSYM_SQUARE) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.x = MapX(3*delta_x);
    cHits =GpiMarker(hPS, &ptl);                        // square at 3,1
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 2*delta_x;
    textptl.y = delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"3,1 Square");

    if (GpiSetMarker(hPS, MARKSYM_EIGHTPOINTSTAR) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }

    ptl.x = MapX(5*delta_x);
    cHits =GpiMarker(hPS, &ptl);                        // 8pt star at 5,1
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 4*delta_x;
    textptl.y = delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"5,1 8pt Star");

    if (GpiSetMarker(hPS, MARKSYM_SIXPOINTSTAR) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = 5*delta_x;
    aptl[0].y = 2*delta_y;
    MapPt(aptl[0]);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // 6pt star at 5,2
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.y = 2*delta_y + 150;
    textptl.x = 4*delta_x;
    MapAndWriteString(hPS,textptl.x,textptl.y,"5,2 6pt Star");

    if (GpiSetMarker(hPS, MARKSYM_DIAMOND) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = MapX(delta_x);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // diamond at 1,2
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 0;
    textptl.y = 2*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"1,2 Diamond");

    if (GpiSetMarker(hPS, MARKSYM_CROSS) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.x = MapX(delta_x);
    ptl.y = MapY(3*delta_y);
    cHits =GpiMarker(hPS, &ptl);                        // cross at 1,3
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 0;
    textptl.y = 3*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"1,3 Cross");

    if (GpiSetMarker(hPS, MARKSYM_SMALLCIRCLE) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.x = MapX(3*delta_x);
    cHits =GpiMarker(hPS, &ptl);                        // circle at 3,3
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 2*delta_x;
    textptl.y = 3*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"3,3 Circle");

    if (GpiSetMarker(hPS, MARKSYM_SOLIDSQUARE) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.x = MapX(5*delta_x);
    cHits =GpiMarker(hPS, &ptl);                        // solid square at 5,3
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 4*delta_x;
    textptl.y = 3*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"5,3 Solid square");

    if (GpiSetMarker(hPS, MARKSYM_SOLIDDIAMOND) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = MapX(5*delta_x);
    aptl[0].y = MapY(4*delta_y);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // Solid Diamond at 5,4
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 4*delta_x;
    textptl.y = 4*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"5,4 Solid Diamond");

    if (GpiSetMarker(hPS, MARKSYM_DEFAULT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = MapX(delta_x);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // explicit default at 1,4
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 0;
    textptl.y = 4*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"1,4 Explicit default");

    if (GpiSetMarker(hPS, MARKSYM_BLANK) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = MapX(delta_x/2);
    aptl[0].y = MapY(11*delta_y/2);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // blank at .5,5.5
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 0;
    textptl.y = 5*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,".5,5.5 Blank");


    if (GpiSetMarker(hPS, MARKSYM_DOT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = MapX(5*delta_x/2);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // Dot at 2.5,5.5
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 2*delta_x;
    textptl.y = 5*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"2.5,5.5 Dot");

    if (GpiSetMarker(hPS, MARKSYM_PLUS) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = MapX(9*delta_x/2);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);           // plus at 4.5,5.5
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 4*delta_x;
    textptl.y = 5*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"4.5,5.5 Plus");

    textptl.x = 0;
    textptl.y = 7*delta_y + 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"3rd and 5th rows plotted with GpiMarker");
    textptl.x = 0;
    textptl.y = 7*delta_y - 150;
    MapAndWriteString(hPS,textptl.x,textptl.y,"1st, 2nd and 4th rows plotted with GpiPolyMarker");

    if (lErrorCount) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
       "$GpiMark Precision Test                                         -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiMark Precision Test                                         -    PASS.\n");

   CWRITELOG(L_LOTRACE,0,"+ex+ GpiMarkPre - Exit.\n");
  }
}
 /***************************************************************************
 *\\ddd
 * Routine name:  gpi_mark_app()
 *
 * Purpose:  This is an entry point for the PTT that causes the GPI Marker
 *           application tests to be run.
 *
 * System Requirements:  Section 5.13.2
 *
 * Revision Log:  12/03/90, PDVT, Kim Cairns
 *                - initial function design
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()    - sets global variables for current device
 *
 * Limitations:
 *             So far, I have been unable to create a vector logical font
 *             to verify GpiSetMarkerBox.
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *  Use GpiSetMarker to set each valid symbol
 *  Call GpiMarker and GpiPolyMarker
 *
 *  Use GpiSetMarkerBox to set size of marker box
 *  Call GpiMarker and GpiPolyMarker
 *
 *  Use GpiSetMarkerSet to set local id of logical font
 *  Call GpiMarker and GpiPolyMarker
 *
 *  Call GpiQueryMarker to query current marker symbol
 *
 *  Call GpiQueryMarkerBox to query current size of marker box
 *
 *  Call GpiQueryMarkerSet to query local id of logical font
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY gpi_mark_app(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  CHAR    out_string[80];
  POINTL  ptl, textptl, aptl[5];
  LONG    cPoints, lSymbol, lcid;
  CHAR    text[30];
  SIZEF   szf;
  LONG    cHits;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {

    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiMarkApp - Entry.\n");

    lErrorCount = 0L;

    // This function sets delta_x and delta_y for the current device

    set_delta(hPS);

    // Test GpiSetMarker with GpiMarker

    ptl.x = delta_x/2;
    ptl.y = 6*delta_y;
    MapPt(ptl);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 0;
    textptl.y = 6*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"Default");

    if (GpiSetMarker(hPS, MARKSYM_PLUS) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.y = MapY(5*delta_y);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.y = 5*delta_y + 300;
    textptl.x = 0;
    MapAndWriteString(hPS,textptl.x,textptl.y,"Plus");

    if (GpiSetMarker(hPS, MARKSYM_DIAMOND) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.y = MapY(4*delta_y);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.y = 4*delta_y + 300;
    textptl.x = 0;
    MapAndWriteString(hPS,textptl.x,textptl.y,"Diamond");

    if (GpiSetMarker(hPS, MARKSYM_SQUARE) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.y = MapY(3*delta_y);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.y = 3*delta_y + 300;
    textptl.x = 0;
    MapAndWriteString(hPS,textptl.x,textptl.y,"Square");

    if (GpiSetMarker(hPS, MARKSYM_DOT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.y = MapY(2*delta_y);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.y = 2*delta_y + 300;
    textptl.x = 0;
    MapAndWriteString(hPS,textptl.x,textptl.y,"Dot");

    textptl.x = 0;
    textptl.y = delta_y+300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarker");
    textptl.y = delta_y;
    textptl.x = 0;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiMarker");

    // Test GpiSetMarker with GpiPolyMarker

    cPoints = 5;
    if (GpiSetMarker(hPS, MARKSYM_SIXPOINTSTAR) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = 3*delta_x/2;
    aptl[0].y = 7*delta_y;
    MapPt(aptl[0]);
    aptl[1].x = 3*delta_x/2;
    aptl[1].y = 6*delta_y;
    MapPt(aptl[1]);
    aptl[2].x = 3*delta_x/2;
    aptl[2].y = 5*delta_y;
    MapPt(aptl[2]);
    aptl[3].x = 3*delta_x/2;
    aptl[3].y = 4*delta_y;
    MapPt(aptl[3]);
    aptl[4].x = 3*delta_x/2;
    aptl[4].y = 3*delta_y;
    MapPt(aptl[4]);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }

    textptl.x = delta_x;
    textptl.y = 6*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"6pt Star");
    textptl.y = 2*delta_y + 300;
    textptl.x = delta_x;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarker");
    textptl.y = 2*delta_y;
    textptl.x = delta_x;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiPolyMarker");

    // Test GpiSetMarkerSet with GpiMarker

    EzfQueryFonts(hPS);
    if (EzfCreateLogFont(hPS, LCID_MYCOUR, FONTFACE_COUR, FONTSIZE_10, (USHORT) NULL))
    {
       if (GpiSetMarkerSet(hPS, LCID_MYCOUR) == FALSE)
       {
        lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
       }
    }
    GpiSetMarker(hPS, MARKSYM_SOLIDDIAMOND);
    ptl.x = 5*delta_x/2;
    ptl.y = 6*delta_y;
    MapPt(ptl);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 5*delta_x/2;
    textptl.y = 6*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"These markers are from non-default fonts");
    textptl.x = 2*delta_x;
    textptl.y = 5*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerSet");
    textptl.x = 2*delta_x;
    textptl.y = 5*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiMarker");

    // Test GpiSetMarkerBox with GpiMarker

    szf.cx = MAKEFIXED(23,0) ;                      // fix - Vish
    szf.cy = MAKEFIXED(23,0) ;
    if (GpiSetMarkerBox(hPS, &szf) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarkerBox", FALSE, NULL);
    }
    if (GpiSetMarker(hPS, MARKSYM_EIGHTPOINTSTAR) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.x = 5*delta_x/2;
    ptl.y = 4*delta_y;
    MapPt(ptl);
    cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = 2*delta_x;
    textptl.y = 3*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerBox");
    textptl.x = 2*delta_x;
    textptl.y = 3*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiMarker");

    // Test GpiSetMarkerSet with GpiPolyMarker

    EzfQueryFonts(hPS);
    if (EzfCreateLogFont(hPS, LCID_MYHELV, FONTFACE_HELV, FONTSIZE_12, (USHORT) NULL))
    {
       if (GpiSetMarkerSet(hPS, LCID_MYHELV) == FALSE)
       {
        lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
       }
    }
    cPoints = 3;
    if (GpiSetMarker(hPS, MARKSYM_SMALLCIRCLE) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = 9*delta_x/2;
    aptl[0].y = 7*delta_y;
    MapPt(aptl[0]);
    aptl[1].x = 9*delta_x/2;
    aptl[1].y = 6*delta_y;
    MapPt(aptl[1]);
    aptl[2].x = 9*delta_x/2;
    aptl[2].y = 5*delta_y;
    MapPt(aptl[2]);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }

    textptl.x = 4*delta_x;
    textptl.y = 4*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerSet");
    textptl.x = 4*delta_x;
    textptl.y = 4*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiPolyMarker");

    // Test GpiSetMarkerBox with GpiPolyMarker

    szf.cx = MAKEFIXED(23,0);                 // fix - Vish
    szf.cy = MAKEFIXED(23,0);
    if (GpiSetMarkerBox(hPS, &szf) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarkerBox", FALSE, NULL);
    }
    cPoints = 2;
    if (GpiSetMarker(hPS, MARKSYM_SOLIDSQUARE) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    aptl[0].x = 13*delta_x/2;
    aptl[0].y = 7*delta_y;
    MapPt(aptl[0]);
    aptl[1].x = 13*delta_x/2;
    aptl[1].y = 6*delta_y;
    MapPt(aptl[1]);
    cHits = GpiPolyMarker(hPS, cPoints, aptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }

    textptl.x = 6*delta_x;
    textptl.y = 5*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerBox");
    textptl.y = 5*delta_y;
    textptl.x = 6*delta_x;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiPolyMarker");

    // Test GpiQueryMarker

    lSymbol = GpiQueryMarker(hPS);
    sprintf(text, "Returned %ld, Expected 8\0", lSymbol);
    textptl.x = 9*delta_x/2;
    textptl.y = 3*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarker");
    textptl.x = 9*delta_x/2;
    textptl.y = 3*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Test GpiQueryMarkerBox

    GpiQueryMarkerBox(hPS, &szf);
    sprintf(text, "Returned %ld,%ld\0", szf.cx, szf.cy);
    textptl.x = 9*delta_x/2;
    textptl.y = 2*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarkerBox");
    textptl.x = 9*delta_x/2;
    textptl.y = 2*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Test GpiQueryMarkerSet

    lcid = GpiQueryMarkerSet(hPS);
    sprintf(text, "Returned %ld, Expected 2\0", lcid);
    textptl.y = delta_y + 300;
    textptl.x = 9*delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarkerSet");
    textptl.y = delta_y;
    textptl.x = 9*delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Return to default marker set

    if (GpiSetMarkerSet (hPS, LCID_DEFAULT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
    }
    GpiDeleteSetId(hPS, LCID_MYCOUR);
    GpiDeleteSetId(hPS, LCID_MYHELV);

    if (lErrorCount){
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
         "$GpiMark Application Test                                       -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiMark Application Test                                       -    PASS.\n");

    CWRITELOG(L_LOTRACE,0,"+ex+ GpiMarkApp - Exit.\n");
  }
}

 /***************************************************************************
 *\\ddd
 * Routine name:  GpiMarkerExh()
 *
 * Purpose:  This is an entry point for the PTT that causes GpiMarker
 *           to be tested exhaustively.
 *
 * System Requirements:  Section 5.13.2
 *
 * Revision Log:  12/03/90, PDVT, Kim Cairns
 *                - initial function design
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()    - sets global variables for current device
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *  // Set new marker attributes
 *
 *  GpiSetMarker(hPS, lSymbol);
 *  GpiSetMarkerBox(hPS, &szf);
 *  GpiSetMarkerSet(hPS, lcid);
 *
 *  // Draw marker after setting each attribute
 *
 *  GpiMarker(hPS, &ptl);
 *
 *  // Query marker attributes for new values
 *
 *  GpiQueryMarker(hPS);
 *  GpiQueryMarkerBox(hPS, &szf);
 *  GpiQueryMarkerSet(hPS);
 *
 *  // Set back to default marker attributes
 *
 *  GpiSetMarker(hPS, lSymbol);
 *  GpiSetMarkerBox(hPS, &szf);
 *  GpiSetMarkerSet(hPS, lcid);
 *
 *  // Draw marker after setting each attribute back to default
 *
 *  GpiMarker(hPS, &ptl);
 *
 *  // Query marker attributes for default values
 *
 *  GpiQueryMarker(hPS);
 *  GpiQueryMarkerBox(hPS, &szf);
 *  GpiQueryMarkerSet(hPS);
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiMarkerExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  CHAR    out_string[80];
  POINTL  ptl, textptl;
  LONG    lSymbol, lcid, rc, i;
  SIZEF   szf;
  BOOL    Success;
  CHAR    text[30];
  LONG    cHits;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {

    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiMarkerExh - Entry.\n");

    lErrorCount = 0L;

    // This function sets delta_x and delta_y for the current device

    set_delta(hPS);

    // Test GpiSetMarker

    if (GpiSetMarker(hPS, MARKSYM_CROSS) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }
    ptl.x = delta_x;
    ptl.y = 6*delta_y;
    MapPt(ptl);
    rc = GpiMarker(hPS, &ptl);
    if (rc == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.x = delta_x/2;
    textptl.y = 6*delta_y+300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarker to CROSS");

    // Test GpiSetMarkerBox

    szf.cx = MAKEFIXED(25,0);                    // fix - Vish
    szf.cy = MAKEFIXED(25,0);
    if (GpiSetMarkerBox(hPS, &szf) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarkerBox", FALSE, NULL);
    }
    ptl.y = MapY(5*delta_y);
    rc = GpiMarker(hPS, &ptl);
    if (rc == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.y = 5*delta_y+300;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerBox");

    // Test GpiSetMarkerSet

    EzfQueryFonts(hPS);
    if (EzfCreateLogFont(hPS, LCID_MYCOUR, FONTFACE_COUR, FONTSIZE_10, (USHORT) NULL))
    {
       if (GpiSetMarkerSet(hPS, LCID_MYCOUR) == FALSE)
       {
        lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
       }
    }
    ptl.y = MapY(4*delta_y);
    rc = GpiMarker(hPS, &ptl);
    if (rc == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    textptl.y = 4*delta_y+300;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerSet to Courier");

    // Test GpiQueryMarker

    lSymbol = GpiQueryMarker(hPS);
    sprintf(text, "Returned %ld, Expected 1\0", lSymbol);
    textptl.y = 3*delta_y + 300;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarker");
    textptl.x = delta_x/2;
    textptl.y = 3*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Test GpiQueryMarkerBox

    Success = GpiQueryMarkerBox(hPS, &szf);
    sprintf(text, "Returned %ld,%ld\0", szf.cx, szf.cy);
    textptl.y = 2*delta_y + 300;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarkerBox");
    textptl.y = 2*delta_y;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Test GpiQueryMarkerSet

    lcid = GpiQueryMarkerSet(hPS);
    sprintf(text, "Returned %ld, Expected 1\0", lcid);
    textptl.y = delta_y + 300;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarkerSet");
    textptl.y = delta_y;
    textptl.x = delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Set Marker Set back to default
    if (GpiSetMarkerSet(hPS, LCID_DEFAULT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
    }

    // Exhaustive test of GpiSetMarker

    textptl.x = 3*delta_x;
    textptl.y = 7*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarker");
    ptl.x = MapX(4*delta_x);
    for (lSymbol = 0; lSymbol <= 11; lSymbol++)
    {
      i = lSymbol + 2;
      ptl.y = MapY(i*delta_y/2);
      if (GpiSetMarker(hPS, lSymbol) == FALSE)
      {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
       }
      cHits = GpiMarker(hPS, &ptl);
      if (cHits == GPI_ERROR)
      {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
      }
    }

    // Set Marker Symbol back to default
    if (GpiSetMarker(hPS, MARKSYM_DEFAULT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }

    // Exhaustive test of GpiSetMarkerBox

    textptl.x = 9*delta_x/2;
    textptl.y = 15*delta_y/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerBox");
    ptl.x = MapX(5*delta_x);
    for (i=1; i <= 12; i++)
    {
     szf.cx = MAKEFIXED(i*3,0);                   // fix - Vish
     szf.cy = MAKEFIXED(i*3,0);
      if (GpiSetMarkerBox(hPS, &szf) == FALSE)
      {
        lErrorCount += BADERROR("GpiSetMarkerBox", FALSE, NULL);
      }
      ptl.y = MapY((i+1)*delta_y/2);
      cHits = GpiMarker(hPS, &ptl);
      if (cHits == GPI_ERROR)
      {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
      }
    }

    // Exhaustive test of GpiSetMarkerSet

    textptl.x = 6*delta_x;
    textptl.y = 7*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarkerSet");
    ptl.x = MapX(6*delta_x);
    EzfQueryFonts(hPS);
    for (i=0; i <= 5; i++)
    {
      if (EzfCreateLogFont(hPS, (LONG)LCID_MYHELV, (USHORT)FONTFACE_HELV, (USHORT)i, (USHORT)NULL))
      {
         if (GpiSetMarkerSet(hPS, LCID_MYHELV) == FALSE)
         {
           lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
         }
      }
      ptl.y = MapY((i+2)*delta_y/2);
      cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    }

    for (i=0; i <= 5; i++)
    {
      if (EzfCreateLogFont(hPS, (LONG)LCID_MYTIMES, (USHORT)FONTFACE_TIMES, (USHORT)i, (USHORT)NULL))
      {
         if (GpiSetMarkerSet(hPS, LCID_MYTIMES) == FALSE)
         {
           lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
         }
      }
      ptl.y = MapY((i+8)*delta_y/2);
      cHits =GpiMarker(hPS, &ptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiMarker", FALSE, NULL);
    }
    }


    if (lErrorCount){
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
         "$GpiMarkerExh                                                   -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiMarkerExh                                                   -    PASS.\n");

    CWRITELOG(L_LOTRACE,0,"+ex+ GpiMarkerExh - Exit.\n");
  }
}

 /***************************************************************************
 *\\ddd
 * Routine name:  GpiPolyMarkerExh()
 *
 * Purpose:  This is an entry point for the PTT that causes GpiPolyMarker
 *           to be tested exhaustively.
 *
 * System Requirements:  Section 5.13.2
 *
 * Revision Log:  12/03/90, PDVT, Kim Cairns
 *                - initial function design
 *
 * Inputs:
 *             hPS            - presentation space handle
 *             hWndClient     - client window handle
 *             SelectionCall  - selection indicator
 *
 * Outputs:
 *             None
 *
 * Subroutines Required:
 *             set_delta()    - sets global variables for current device
 *
 * Limitations:
 *             None
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *
 *  // Query current marker attributes
 *
 *  GpiQueryMarker(hPS);
 *  GpiQueryMarkerBox(hPS, &szf);
 *  GpiQueryMarkerSet(hPS);
 *
 *  // Set new marker attributes
 *
 *  GpiSetMarker(hPS, lSymbol);
 *  GpiSetMarkerBox(hPS, &szf);
 *  GpiSetMarkerSet(hPS, lcid);
 *
 *  // Draw markers after setting each attribute
 *
 *  GpiPolyMarker(hPS, cPoints, aptl);
 *
 *  // Query marker attributes for new values
 *
 *  GpiQueryMarker(hPS);
 *  GpiQueryMarkerBox(hPS, &szf);
 *  GpiQueryMarkerSet(hPS);
 *
 *  // Set default marker attributes
 *
 *  GpiSetMarker(hPS, lSymbol);
 *  GpiSetMarkerBox(hPS, &szf);
 *  GpiSetMarkerSet(hPS, lcid);
 *
 *  // Draw markers after setting each attribute back to default
 *
 *  GpiPolyMarker(hPS, cPoints, aptl);
 *
 *  // Query marker attributes for default values
 *
 *  GpiQueryMarker(hPS);
 *  GpiQueryMarkerBox(hPS, &szf);
 *  GpiQueryMarkerSet(hPS);
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPolyMarkerExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  CHAR    out_string[80];
  POINTL  textptl, aptl[12];
  LONG    lcid, lSymbol, cPoints, i, j;
  SIZEF   szf;
  CHAR    text[130];
  LONG    cHits;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {

    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPolyMarkerExh - Entry.\n");

    lErrorCount = 0L;

    // This function sets delta_x and delta_y for the current device

    set_delta(hPS);

    aptl[0].y = MapY(13*delta_y/2);
    aptl[1].y = MapY(6*delta_y);
    aptl[2].y = MapY(11*delta_y/2);
    aptl[3].y = MapY(5*delta_y);
    aptl[4].y = MapY(9*delta_y/2);
    aptl[5].y = MapY(4*delta_y);
    aptl[6].y = MapY(7*delta_y/2);
    aptl[7].y = MapY(3*delta_y);
    aptl[8].y = MapY(5*delta_y/2);
    aptl[9].y = MapY(2*delta_y);
    aptl[10].y= MapY(3*delta_y/2);
    aptl[11].y= MapY(delta_y);

    // Exhaustive test of GpiSetMarker

    textptl.x = delta_x/2;
    textptl.y = 7*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiSetMarker");
    cPoints = 12;
    for (i=0; i<=11; i++)
    {
      for (j=0; j<=11; j++)
        aptl[j].x = MapX((i+1)*delta_x/2);
      if (GpiSetMarker(hPS, i) == FALSE)
      {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
      }
      cHits = GpiPolyMarker(hPS, cPoints, aptl);
      if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
      cPoints--;
    }

    // Test GpiQueryMarker

    lSymbol = GpiQueryMarker(hPS);
    sprintf(text, "Returned %ld\0", lSymbol);
    textptl.x = 7*delta_x/2;
    textptl.y = 3*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarker");
    textptl.y = 3*delta_y;
    textptl.x = 7*delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);
    MapAndWriteString(hPS,textptl.x,3*delta_y - 300, "Expected 11");

    // Return Marker Symbol to default
    if (GpiSetMarker(hPS, MARKSYM_DEFAULT) == FALSE)
    {
        lErrorCount += BADERROR("GpiSetMarker", FALSE, NULL);
    }

    // Exhaustive test of GpiSetMarkerSet

     GpiSetCharDirection(hPS, CHDIRN_RIGHTLEFT);
    textptl.x = 13*delta_x/2;
    textptl.y = 7*delta_y - line_space;
    MapAndWriteString(hPS,textptl.x,textptl.y,"teSreKraMteSipG");
     GpiSetCharDirection(hPS, CHDIRN_DEFAULT);
    cPoints = 1;
    EzfQueryFonts(hPS);
    for (i=0; i <= 5; i++)
    {
      if (EzfCreateLogFont(hPS, (LONG)LCID_MYHELV, (USHORT)FONTFACE_HELV,(USHORT)i, (USHORT)NULL)) {
         if (GpiSetMarkerSet(hPS, LCID_MYHELV) == FALSE) {
           lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
         }
      }
      aptl[0].x = MapX(13*delta_x/2);
      aptl[0].y = MapY((i+2)*delta_y/2);
      cHits = GpiPolyMarker(hPS, cPoints, aptl);
      if (cHits == GPI_ERROR) {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    }

    for (i=0; i <= 5; i++)
    {
      if (EzfCreateLogFont(hPS, (LONG)LCID_MYTIMES, (USHORT)FONTFACE_TIMES, (USHORT)i, (USHORT)NULL)) {
        if (GpiSetMarkerSet(hPS, LCID_MYTIMES) == FALSE) {
          lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
        }
      }
      aptl[0].x = MapX(13*delta_x/2);
      aptl[0].y = MapY((i+8)*delta_y/2);
      cHits = GpiPolyMarker(hPS, cPoints, aptl);
      if (cHits == GPI_ERROR) {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    }

    // Test GpiQueryMarkerSet

    lcid = GpiQueryMarkerSet(hPS);
    sprintf(text, "Returned %ld, expected 3\0", lcid);
    textptl.x = 5*delta_x/2;
    textptl.y = 2*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarkerSet");
    textptl.y = 2*delta_y;
    textptl.x = 5*delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Return Marker set to default
    if (GpiSetMarkerSet(hPS, LCID_DEFAULT) == FALSE) {
        lErrorCount += BADERROR("GpiSetMarkerSet", FALSE, NULL);
    }

    // Exhaustive test of GpiSetMarkerBox

     GpiSetCharDirection(hPS, CHDIRN_RIGHTLEFT);
    textptl.x = 7*delta_x;
    textptl.y = 7*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"xoBrekraMteSipG");
     GpiSetCharDirection(hPS, CHDIRN_DEFAULT);
    for (i=1; i <= 12; i++)
    {
      szf.cx = MAKEFIXED(i*3,0);          // fix - Vish
      szf.cy = MAKEFIXED(i*3,0);
      if (GpiSetMarkerBox(hPS, &szf) == FALSE) {
        lErrorCount += BADERROR("GpiSetMarkerBox", FALSE, NULL);
      }
      aptl[0].x = MapX(7*delta_x);
      aptl[0].y = MapY((i+1)*delta_y/2);
      cHits = GpiPolyMarker(hPS, cPoints, aptl);
      if (cHits == GPI_ERROR) {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    }

    // Test GpiQueryMarkerBox

    GpiQueryMarkerBox(hPS, &szf);
    sprintf(text, "Returned %ld,%ld\0", szf.cx,szf.cy);

    textptl.x = 3*delta_x/2;
    textptl.y = delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"GpiQueryMarkerBox");
    textptl.y = delta_y;
    textptl.x = 3*delta_x/2;
    MapAndWriteString(hPS,textptl.x,textptl.y,text);

    // Test cPoints=0

    cPoints = 0;
    aptl[0].x = 9*delta_x/2;
    aptl[0].y = 4*delta_y + 300;
    cHits = GpiPolyMarker(hPS, cPoints, aptl);
    if (cHits == GPI_ERROR)
    {
        lErrorCount += BADERROR("GpiPolyMarker", FALSE, NULL);
    }
    textptl.x = 9*delta_x/2 + 300;
    textptl.y = 4*delta_y + 300;
    MapAndWriteString(hPS,textptl.x,textptl.y,"<-cPoints=0");
    textptl.x = 9*delta_x/2;
    textptl.y = 4*delta_y;
    MapAndWriteString(hPS,textptl.x,textptl.y,"No X expected");

    if (lErrorCount) {
       sprintf(out_string,"$ERRORCOUNT = %ld.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,
         "$GpiPolyMarkerExh                                               -    FAIL.\n");
    }
    else
       CWRITELOG(L_HDR,0,
         "$GpiPolyMarkerExh                                               -    PASS.\n");

    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPolyMarkerExh - Exit.\n");
}
}

