#define INCL_DOS
#define INCL_WIN
#define INCL_GPI
#define INCL_ERRORS    // PM error definitions
#define INCL_DEV
#define NUMPOINTS 5000  // number of points for Poly Line Spiral Image
#define NUMREV 7       // number of revolutions for Poly Line Spiral Image
#define PI 3.14159
#define MAINSOURCEFILE              //This main source code.

#include <string.h>
#include <stdlib.h>
#include <os2.h>
#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include "PTTDLLDF.h"  // test case defines
#include "gpi_grid.h"  // function to draw a precision grid on screen
#include "std_util.h"  // Map, Write, and Delta Functions
#include "gpiline.h"   // funtion prototypes

VOID line_type(HPS hPS,HWND hWndClient);

/****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module name:  GPILINE
 *
 * Purpose:  This module is called by PTT to test the LINE APIs.
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               10/29/90, PDVT, Donna Williams
 *                     - test case design
 *               10/31/90, PDVT, Donna Williams
 *                     - implementation of testcase design
 *               01/10/91, PDVT, Donna Williams
 *                     - code review
 *
 * SRS References: Section 5.12
 *
 * Contents: InitTest(),
 *           gpi_line_pre(),
 *           gpi_line_app(),
 *           GpiBoxExh(),
 *           GpiLineExh(),
 *           GpiPolyLineExh()
 *
 * Design Overview:
 *        The GPILINE testcase involves testing of the following GPIs:
 *             GpiMove(),
 *             GpiSetCurrentPosition(),
 *             GpiQueryCurrentPosition(),
 *             GpiSetLineType(),
 *             GpiQueryLineEnd(),
 *             GpiQueryLineJoin(),
 *             GpiQueryLineType(),
 *             GpiBox(),
 *             GpiLine(),
 *             GpiPolyLine().
 *        The individual testing of each GPI will include a functionality
 *        test and an exhaustive return codes test.  Along with the
 *        exhaustive tests, there will be an application routine and
 *        a precision routine.
 *
 * Limitations:  As Noted For Each Function.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\ddd
 * Routine Name: InitTest()
 *
 * Purpose: Initialize function name and address array for PTT
 *
 * System Requirements: None
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                       - initial creation
 *
 * Inputs:  sel        - pass thru parameter for other function calls
 *          pInitInfo  - pointer to info array
 *
 * Outputs: pInitInfo  - updated with all entry point names and addresses
 *
 * Subroutines Required: InitTestSegVars()
 *
 * Limitations:   None.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * for each test routine {
 *   copy the name of the routine into the info structure
 *   copy the address of the routine into the info structure
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO pInitInfo)
  {
    // the number of tests
    pInitInfo->ntests = NTESTS;
    strcpy(pInitInfo->group_name,"GPI Line");

    strcpy(pInitInfo->test_names[0],"GpiLinePre");
    pInitInfo->test_addrs[0]=gpi_line_pre;
    strcpy(pInitInfo->test_names[1],"GpiLineApp");
    pInitInfo->test_addrs[1]=gpi_line_app;

    strcpy(pInitInfo->test_names[2],"GpiBoxExh");
    pInitInfo->test_addrs[2]=GpiBoxExh;
    strcpy(pInitInfo->test_names[3],"GpiLineExh");
    pInitInfo->test_addrs[3]=GpiLineExh;
    strcpy(pInitInfo->test_names[4],"GpiPolyLineExh");
    pInitInfo->test_addrs[4]=GpiPolyLineExh;


    InitTestSegVars(sel);
  }
/****************************************************************************
 *\\ddd
 * Routine Name: gpi_line_pre()
 *
 * Purpose: This PTT entry point executes the precision test by calling GpiBox
 *      GpiLine, and GpiPolyLine.  Output must be verified manually.
 *
 * System Requirements: Section 5.12.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               10/29/90, PDVT, Donna Williams
 *                     - test case design
 *
 * Inputs:  hPS            - presentation space
 *          hWndClient     - client window handle
 *          SelectionCall  - selection indicator
 *
 * Outputs: None.
 *
 * Subroutines Required: CWRITELOG(), LOG_SIGNATURE, TEST_SIGNATURE(),
 *                       set_delta(), gpi_grid(), MapPt().
 *
 * Limitations: None.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 *
 *  //basic functions are called here and measured against standard grid
 *  // display grid
 *  gpi_grid(...);
 *
 *  // move cursor to a position and display a box in one grid space
 *  GpiMove(...);
 *  GpiBox(...);
 *
 *  // move cursor to a new position and draw a line from one corner of
 *  // a grid space to the diagonal corner
 *  GpiMove(...);
 *  GpiLine(...);
 *
 *  // move cursor to a new position and while using GpiPolyLine draw
 *  // a L-Shape in 5 grid spaces
 *  GpiMove(...);
 *  GpiPolyLine(...);
 *
 *\\end
 **********************************************************************/
VOID APIENTRY gpi_line_pre(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL   ptl,
           ptlBox[7];
  CHAR     out_string[100];
  INT      i;
  LONG     lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else{
   LOG_SIGNATURE;
   TEST_SIGNATURE(0,0);
   CWRITELOG(L_LOTRACE,0,"GPILINE Precision Test - Entry.\n");
   set_delta(hPS);
   lErrorCount = 0L;

   // set grid up to have grid squares with dimensions 100x100
   gpi_grid(hPS, (LONG)INCH);
   sprintf(out_string,"The precision grid measurements are one inch x one inch.");
   MapAndWriteString(hPS,delta_x,6*delta_y,out_string);

   // make a box that has dimensions 1"x1" to fill grid square
   ptl.x = delta_x;
   ptl.y = 2 * delta_y;
   MapPt(ptl);
   GpiMove(hPS, &ptl);

   ptl.x = 2 * delta_x;
   ptl.y = 3 * delta_y;
   MapPt(ptl);
   GpiSetPattern(hPS,PATSYM_DIAG3);
   GpiBox(hPS, DRO_FILL, &ptl, 1L, 1L);

   sprintf(out_string,"GpiBox: Box measurements");
   MapAndWriteString(hPS,(delta_x/2),(3*delta_y/2+50),out_string);
   sprintf(out_string,"are one inch by one inch");
   MapAndWriteString(hPS,(delta_x/2),(3*delta_y/2-200),out_string);
   GpiSetPattern(hPS,PATSYM_SOLID);

   // make a line that goes from one corner to another corner of a grid square
   ptl.x = 4 * delta_x;
   ptl.y = delta_y;
   MapPt(ptl);
   GpiMove(hPS, &ptl);

   ptl.x = 3 * delta_x;
   ptl.y = 2 * delta_y;
   MapPt(ptl);
   GpiLine(hPS, &ptl);

   ptl.x = 3 * delta_x;
   ptl.y = 2 * delta_y;
   MapPt(ptl);
   GpiMove(hPS, &ptl);

   ptl.x = 4 * delta_x;
   ptl.y = 3 * delta_y;
   MapPt(ptl);
   GpiLine(hPS, &ptl);

   ptl.x = 4 * delta_x;
   ptl.y = 3 * delta_y;
   MapPt(ptl);
   GpiMove(hPS, &ptl);

   ptl.x = 3 * delta_x;
   ptl.y = 4 * delta_y;
   MapPt(ptl);
   GpiLine(hPS, &ptl);

   ptl.x = 3 * delta_x;
   ptl.y = 4 * delta_y;
   MapPt(ptl);
   GpiMove(hPS, &ptl);

   ptl.x = 4 * delta_x;
   ptl.y = 5 * delta_y;
   MapPt(ptl);
   GpiLine(hPS, &ptl);

   sprintf(out_string,"GpiLine: Lines are drawn from");
   MapAndWriteString(hPS,(5*delta_x/2),(11*delta_y/2+50),out_string);
   sprintf(out_string,"one corner to another");
   MapAndWriteString(hPS,(5*delta_x/2),(11*delta_y/2-200),out_string);

   // initialize array for L-shape
   ptlBox[0].x = (5 * delta_x);
   ptlBox[0].y = (4 * delta_y);
   ptlBox[1].x = (5 * delta_x);
   ptlBox[1].y = delta_y;
   ptlBox[2].x = (7 * delta_x);
   ptlBox[2].y = delta_y;
   ptlBox[3].x = (7 * delta_x);
   ptlBox[3].y = (2 * delta_y);
   ptlBox[4].x = (6 * delta_x);
   ptlBox[4].y = (2 * delta_y);
   ptlBox[5].x = (6 * delta_x);
   ptlBox[5].y = (4 * delta_y);
   ptlBox[6].x = (5 * delta_x);
   ptlBox[6].y = (4 * delta_y);

   // define a path and draw a L-Shape within 4 grid squares
   GpiSetPattern(hPS,PATSYM_DIAG3);
   GpiBeginPath(hPS,1L);
   for(i=0;i<=6;++i)
     MapPt(ptlBox[i]);
   GpiMove(hPS, &ptlBox[0]);
   GpiPolyLine(hPS, 7l, ptlBox);
   GpiCloseFigure(hPS);
   GpiEndPath(hPS);
   GpiFillPath(hPS,1L, FPATH_WINDING);

   sprintf(out_string,"GpiPolyLine:");
   MapAndWriteString(hPS,(5*delta_x+100),(9*delta_y/2+50),out_string);
   sprintf(out_string,"Makes an L shape");
   MapAndWriteString(hPS,(5*delta_x+100),(9*delta_y/2-200),out_string);
   sprintf(out_string,"using 4 grid boxes");
   MapAndWriteString(hPS,(5*delta_x+100),(9*delta_y/2-450),out_string);

   if(!lErrorCount){
     CWRITELOG(L_HDR,0,"$GPILINE Precision Test                                         -    PASS.\n");
   }
   else{
     sprintf(out_string,"$ERRORCOUNT = %d.\n",lErrorCount);
     CWRITELOG(L_HDR,0,out_string);
     CWRITELOG(L_HDR,0,"$GPILINE Precision Test                                         -    FAIL.\n");
   }

   CWRITELOG(L_LOTRACE,0,"GPILINE Precision Test - Exit.\n");
  }  /* end else statement */
}
/****************************************************************************
 *\\ddd
 * Routine Name: gpi_line_app()
 *
 * Purpose: This PTT entry point executes the application test.
 *
 * System Requirements: Section 5.12.2
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               10/29/90, PDVT, Donna Williams
 *                     - testcase design
 *
 * Inputs:  hPS            - presentation space
 *          hWndClient     - client window handle
 *          SelectionCall  - selection indicator
 *
 * Outputs: None.
 *
 * Subroutines Required: CWRITELOG(), LOG_SIGNAUTRE, TEST_SIGNATURE(),
 *                       set_delta(), MapPt(), MapY(), MapX().
 *
 * Limitations: None.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *    // use different APIs with the API calls involved in this testcase
 *    // to verify interaction between APIs
 *    // Draw a grid using the GpiLine() and GpiMove() APIs.
 *  for(tmp.x=0; tmp.x <= value; tmp.x=tmp.x+inc){
 *    GpiMove(hPS,&tmp);
 *    GpiLine(hPS,&tmp);
 *   }
 *  for(tmp.y=0; tmp.y <= value; tmp.y=tmp.y+inc){
 *     GpiMove(hPS,&tmp);
 *     GpiLine(hPS,&tmp);
 *  }
 *    // Draw A Box on the screen
 *    GpiMove(...);
 *    GpiBox(...);
 *    // define a path within the box space and call GpiPolyLine
 *   GpiBeginPath(...);
 *   GpiPolyLine(...);
 *   GpiCloseFigure(...);
 *   GpiEndPath(...);
 *   GpiFillPath(...);
 * }
 *\\end
 **********************************************************************/
VOID APIENTRY gpi_line_app(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  LONG lErrorCount;
  INT i;
  POINTL ptlBox, ptlTriangle[3], tmp,ptPoint;
  CHAR out_string[100];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else{
    set_delta(hPS);
    LOG_SIGNATURE;
    TEST_SIGNATURE(0,0);
    CWRITELOG(L_LOTRACE,0,"GPILINE Application Test - Entry.\n");
    lErrorCount = 0L;

    // construct a grid with GpiLine Calls
    for (tmp.x = 0L; tmp.x < *pBasisX; tmp.x += (delta_x / 2)) {
      ptPoint.x = tmp.x;
      ptPoint.y = 0L;
      MapPt(ptPoint);
      GpiMove(hPS, &ptPoint);
      ptPoint.y = MapY(*pBasisY - 1);
      GpiLine(hPS, &ptPoint);
    }
    for (tmp.y = 0L; tmp.y < *pBasisY; tmp.y += (delta_y / 2)) {
      ptPoint.x = 0L;
      ptPoint.y = tmp.y;
      MapPt(ptPoint);
      GpiMove(hPS, &ptPoint);
      ptPoint.x = MapX(*pBasisX - 1);
      GpiLine(hPS, &ptPoint);
    }

    sprintf(out_string,"This grid is constructed from GpiLine calls and");
    MapAndWriteString(hPS,0L,(6*delta_y),out_string);
    sprintf(out_string,"should not be used for Precision.");
    MapAndWriteString(hPS,0L,(23*delta_y/4),out_string);

    // draw a box
             //TSS...changed pattern from PATSYM_HORIZ...8/1/91
             //TSS...changed pattern from PATSYM_DENSE4..2/10/92
    GpiSetPattern(hPS,PATSYM_HORIZ);
    ptlBox.x = 3 * delta_x;
    ptlBox.y = delta_y;
    MapPt(ptlBox);
    GpiMove(hPS, &ptlBox);
    ptlBox.x = 15 * delta_x / 2;
    ptlBox.y = 11 * delta_y / 2;
    MapPt(ptlBox);
    GpiBox(hPS,DRO_OUTLINEFILL,&ptlBox,1L,1L);
    MapAndWriteString(hPS,(5*delta_x),(3*delta_y/4),"GpiBox");

    // draw a triangle within the box space
    ptlTriangle[0].x = delta_x / 2;
    ptlTriangle[0].y = delta_y;
    ptlTriangle[1].x = 3 * delta_x;
    ptlTriangle[1].y = 11 * delta_y / 2;
    ptlTriangle[2].x = 5 * delta_x + (delta_x / 2);
    ptlTriangle[2].y = delta_y;
    for(i = 0; i <= 3; ++i)
       MapPt(ptlTriangle[i]);
    GpiSetPattern(hPS,PATSYM_DIAG3);
    GpiBeginPath(hPS, 1L);
    GpiMove(hPS, &ptlTriangle[0]);
    GpiPolyLine(hPS, 3L, ptlTriangle);
    GpiCloseFigure(hPS);
    GpiEndPath(hPS);
    GpiFillPath(hPS, 1L, FPATH_ALTERNATE);
    MapAndWriteString(hPS,(2*delta_x),(3*delta_y/4),"GpiPolyLine");

    // TSS...added a second triangle on 2/10/92
    // draw a second triangle within the box space
    ptlTriangle[0].x = delta_x * 5;
    ptlTriangle[0].y = (LONG)(delta_y * 3.5);
    ptlTriangle[1].x = (LONG)(delta_x * 6.25);
    ptlTriangle[1].y = (LONG)(delta_y * 4.5);
    ptlTriangle[2].x = (LONG)(delta_x * 7.5);
    ptlTriangle[2].y = (LONG)(delta_y * 3.5);
    for(i = 0; i <= 3; ++i)
       MapPt(ptlTriangle[i]);
    GpiSetPattern(hPS,PATSYM_DIAG3);
    GpiBeginPath(hPS, 1L);
    GpiMove(hPS, &ptlTriangle[0]);
    GpiPolyLine(hPS, 3L, ptlTriangle);
    GpiCloseFigure(hPS);
    GpiEndPath(hPS);
    GpiFillPath(hPS, 1L, FPATH_ALTERNATE);

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GPILINE Application Test                                       -    PASS.\n");
    }
    else{
       sprintf(out_string,"$ERRORCOUNT = %d.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,"$GPILINE Application Test                                       -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"GPILINE Application Test - Exit.\n");
  } // end else statement
}
/****************************************************************************
 *\\ddd
 * Routine Name: GpiBoxExh()
 *
 * Purpose: This function tests for GpiBox() functionality. This function
 *          will also combine the functionality tests for GpiMove(), and
 *          GpiQueryCurrentPosition().
 *
 * System Requirements: Section 5.12.3
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               10/29/90, PDVT, Donna Williams
 *                     - test case design
 *               11/01/90, PDVT, Donna Williams
 *                     - design implementation
 *               11/19/90, PDVT, Donna Williams
 *                     - redesign
 *
 * Inputs:  hPS            - presentation space
 *          hWndClient     - client window handle
 *          SelectionCall  - selection indicator
 *
 * Outputs: None.
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNAUTRE(), LOG_SIGNATURE,
 *                       set_delta(), MapPt(), BADERROR().
 *
 * Limitations: None.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *    // draw a grid  onto the screen using the GpiBox and GpiMove functions
 *    // this will help in verifying the precision of the Box command
 *    for(ptl.y=0; ptl.y < some_#_<_pBasisY; ptl.y=ptl.y+100){
 *      for(ptl.x=0; ptl.x < some_#_<_pBasisX; ptl.x=ptl.x+100){
 *         GpiMove(hPS,&ptl);       // call with valid parameters
 *         Point.x = ptl.x + 100;
 *         Point.y = ptl.y + 100;
 *         GpiBox(hPS,DRO_OUTLINE,&Point,   // call with valid parameters
 *                  (ULONG)NULL,(ULONG)NULL);
 *      }
 *    }
 *     // TEST 1
 *     // test GpiMove's functionality and report it with BADERROR
 *     GpiMove(hPS,&ptl);      // call with valid parameters
 *     // test GpiQueryCurrentPosition's functionality and report with
 *     // BADERROR.  Also compare values obtained with current
 *     // position values. If there is a difference report with CWRITELOG.
 *     QPoint = GpiQueryCurrentPosition(hPS);
 *     // test GpiBox functionality and report it with BADERROR
 *         GpiBox(hPS,DRO_FILL,&Point,   // call with valid parameters
 *                  (ULONG)NULL,(ULONG)NULL);
 *     // TEST 2
 *     // test GpiMove's functionality and report it with BADERROR
 *     GpiMove(hPS,&ptl);      // call with valid parameters
 *     // test GpiQueryCurrentPosition's functionality and report with
 *     // BADERROR.  Also compare values obtained with current
 *     // position values. If there is a difference report with CWRITELOG.
 *     QPoint = GpiQueryCurrentPosition(hPS);
 *     // test GpiBox functionality and report it with BADERROR
 *         GpiBox(hPS,DRO_OUTLINE,&Point,   // call with valid parameters
 *                  (ULONG)NULL,(ULONG)NULL);
 *     // TEST 3
 *     // test GpiMove's functionality and report it with BADERROR
 *     GpiMove(hPS,&ptl);      // call with valid parameters
 *     // test GpiQueryCurrentPosition's functionality and report with
 *     // BADERROR.  Also compare values obtained with current
 *     // position values. If there is a difference report with CWRITELOG.
 *     QPoint = GpiQueryCurrentPosition(hPS);
 *     // test GpiBox functionality and report it with BADERROR
 *         GpiBox(hPS,DRO_OUTLINEFILL,&Point,   // call with valid parameters
 *                  (ULONG)NULL,(ULONG)NULL);
 *     // exit the GpiBoxExh routine
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiBoxExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL ptl,Point,QPoint;
  CHAR out_string[100];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
   set_delta(hPS);
   LOG_SIGNATURE;
   TEST_SIGNATURE(0,0);
   CWRITELOG(L_LOTRACE,0,"+ee+ GpiBoxExh - Entry.\n");
   lErrorCount = 0L;

   // draw grid onto screen using boxes
   // this test will require a visual check of output
   for (ptl.y = 0L; ptl.y < *pBasisY; ptl.y += (delta_y / 2)){
     for (ptl.x = 0L; ptl.x < *pBasisX; ptl.x += (delta_x / 2)) {
       Point.x = ptl.x;
       Point.y = ptl.y;
       MapPt(Point);
       if (GpiMove(hPS, &Point) == FALSE){
          BADERROR("GpiMove",TRUE,"GpiBoxExh");
          return;
       }
       Point.x = ptl.x + (delta_x / 2);
       Point.y = ptl.y + (delta_y / 2);
       MapPt(Point);
       if (GpiBox(hPS,DRO_OUTLINE,&Point,(ULONG)NULL,(ULONG)NULL) == FALSE){
          BADERROR("GpiBox",TRUE,"GpiBoxExh");
          return;
       }
     }
   }
   sprintf(out_string,"This grid was constructed with GpiBox calls and");
   MapAndWriteString(hPS,0L,(6*delta_y),out_string);
   sprintf(out_string,"should not be used for verifying precision.");
   MapAndWriteString(hPS,0L,(6*delta_y-250),out_string);
   sprintf(out_string,"Tests:  GpiQueryCurrentPosition");
   MapAndWriteString(hPS,0L,(6*delta_y-500),out_string);
   sprintf(out_string,"        GpiMove");
   MapAndWriteString(hPS,0L,(6*delta_y-750),out_string);
   sprintf(out_string,"Tests:  GpiBox");
   MapAndWriteString(hPS,0L,(6*delta_y-1000),out_string);
   /*
    * Test the functionality of the GpiMove call
    */
   ptl.x = delta_x;
   ptl.y = delta_y;
   MapPt(ptl);
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiMove PMERR_OK Test1 - Entry.\n");
   if(!GpiMove(hPS,&ptl)){
      BADERROR("GpiMove",TRUE,"GpiBoxExh");
      return;
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiMove PMERR_OK Test1 - Exit.\n");
   /*
    * Test the functionality of the GpiQueryCurrentPosition call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryCurrentPosition PMERR_OK Test1 - Entry.\n");
   if(!GpiQueryCurrentPosition(hPS,&QPoint)){
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   if((ptl.x != QPoint.x) || (ptl.y != QPoint.y)){
      CWRITELOG(L_FAIL,0,"$GpiQueryCurrentPosition DID NOT Return Correct Values       - FAIL.\n");
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryCurrentPosition PMERR_OK Test1 - Exit.\n");
   /*
    * Test the functionality of the GpiBox call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiBox PMERR_OK Test1 - Entry.\n");
   Point.x = 2 * delta_x;
   Point.y = 2 * delta_y;
   MapPt(Point);
   if(!GpiBox(hPS,DRO_FILL,&Point,(ULONG)NULL,(ULONG)NULL)){
      BADERROR("GpiBox",TRUE,"GpiBoxExh");
      return;
   }
   sprintf(out_string,"Solid Box 1x1");
   MapAndWriteString(hPS,(9*delta_x/4),(7*delta_y/4),out_string);
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiBox PMERR_OK Test1 - Exit.\n");
   /*
    * Test the functionality of the GpiMove call.
    */
   ptl.x = 21 * delta_x / 4;
   ptl.y = 5 * delta_y / 2;
   MapPt(ptl);
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiMove PMERR_OK Test2 - Entry.\n");
   if(!GpiMove(hPS,&ptl)){
      BADERROR("GpiMove",TRUE,"GpiBoxExh");
      return;
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiMove PMERR_OK Test2 - Exit.\n");
   /*
    * Test the functionality of the GpiQueryCurrentPosition call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryCurrentPosition PMERR_OK Test2 - Entry.\n");
   if(!GpiQueryCurrentPosition(hPS,&QPoint)){
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   if((ptl.x != QPoint.x) || (ptl.y != QPoint.y)){
      CWRITELOG(L_FAIL,0,"$GpiQueryCurrentPosition DID NOT Return Correct Values        - FAIL.\n");
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryCurrentPosition PMERR_OK Test2 - Exit.\n");
   /*
    * Test the functionality of the GpiBox call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiBox PMERR_OK Test2 - Entry.\n");
   Point.x = 17 * delta_x / 4;
   Point.y = 3 * delta_y / 2;
   MapPt(Point);
   if(!GpiBox(hPS,DRO_OUTLINE,&Point,(ULONG)NULL,(ULONG)NULL)){
      BADERROR("GpiBox",TRUE,"GpiBoxExh");
      return;
   }
   sprintf(out_string,"Outlined Box 1x1");
   MapAndWriteString(hPS,(21*delta_x/4),(9*delta_y/4),out_string);
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiBox PMERR_OK Test2 - Exit.\n");
   /*
    * Test the functionality of the GpiMove call.
    */
   ptl.x = delta_x / 2;
   ptl.y = 3 * delta_y;
   MapPt(ptl);
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiMove PMERR_OK Test3 - Entry.\n");
   if(!GpiMove(hPS,&ptl)){
      BADERROR("GpiMove",TRUE,"GpiBoxExh");
      return;
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiMove PMERR_OK Test3 - Exit.\n");
   /*
    * Test the functionality of the GpiQueryCurrentPosition call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryCurrentPosition PMERR_OK Test3 - Entry.\n");
   if(!GpiQueryCurrentPosition(hPS,&QPoint)) {
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   if((ptl.x != QPoint.x) || (ptl.y != QPoint.y)){
      CWRITELOG(L_FAIL,0,"$GpiQueryCurrentPosition DID NOT Return Correct Values      - FAIL.\n");
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryCurrentPosition PMERR_OK Test3 - Exit.\n");
   /*
    * Test the functionality of the GpiBox call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiBox PMERR_OK Test3 - Entry.\n");
   Point.x = 5 * delta_x / 2;
   Point.y = 7 * delta_y / 2;
   MapPt(Point);
   GpiSetPattern(hPS,PATSYM_DIAG3);
   if(!GpiBox(hPS,DRO_OUTLINEFILL,&Point,5L,5L)){
      BADERROR("GpiBox",TRUE,"GpiBoxExh");
      return;
   }
   sprintf(out_string,"Filled Box 2x0.5");
   MapAndWriteString(hPS,(delta_x/2),(11*delta_y/4),out_string);
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiBox PMERR_OK Test3 - Exit.\n");
   /*
    * Test the functionality of the GpiMove call.
    */
   ptl.x = 4 * delta_x;
   ptl.y = 4 * delta_y;
   MapPt(ptl);
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiMove PMERR_OK Test4 - Entry.\n");
   if(!GpiMove(hPS,&ptl)){
      BADERROR("GpiMove",TRUE,"GpiBoxExh");
      return;
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiMove PMERR_OK Test4 - Exit.\n");
   /*
    * Test the functionality of the GpiQueryCurrentPosition call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryCurrentPosition PMERR_OK Test4 - Entry.\n");
   if(!GpiQueryCurrentPosition(hPS,&QPoint)){
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   if((ptl.x != QPoint.x) || (ptl.y != QPoint.y)){
      CWRITELOG(L_FAIL,0,"$GpiQueryCurrentPosition DID NOT Return Correct Values      - FAIL.\n");
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryCurrentPosition PMERR_OK Test4 - Exit.\n");
   /*
    * Test the functionality of the GpiBox call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiBox PMERR_OK Test4 - Entry.\n");
   Point.x = 5 * delta_x;
   Point.y = 11 * delta_y / 2;
   MapPt(Point);
   GpiSetPattern(hPS,PATSYM_DIAG4);
   if(!GpiBox(hPS,DRO_OUTLINEFILL,&Point,25L,15L)){
      BADERROR("GpiBox",TRUE,"GpiBoxExh");
      return;
   }
   sprintf(out_string,"Filled Box");
   MapAndWriteString(hPS,(4*delta_x),(15*delta_y/4),out_string);
   sprintf(out_string,"1x1.5");
   MapAndWriteString(hPS,(4*delta_x),(7*delta_y/2),out_string);
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiBox PMERR_OK Test4 - Exit.\n");
   /*
    * Test the functionality of the GpiMove call
    */
   ptl.x = 11 * delta_x / 2;
   ptl.y = 5 * delta_y;
   MapPt(ptl);
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiMove PMERR_OK Test5 - Entry.\n");
   if(!GpiMove(hPS,&ptl)){
      BADERROR("GpiMove",TRUE,"GpiBoxExh");
      return;
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiMove PMERR_OK Test5 - Exit.\n");
   /*
    * Test the functionality of the GpiQueryCurrentPosition call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryCurrentPosition PMERR_OK Test5 - Entry.\n");
   if(!GpiQueryCurrentPosition(hPS,&QPoint)){
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   if((ptl.x != QPoint.x) || (ptl.y != QPoint.y)){
      CWRITELOG(L_FAIL,0,"$GpiQueryCurrentPosition DID NOT Return Correct Values      - FAIL.\n");
      lErrorCount += BADERROR("GpiQueryCurrentPosition",FALSE,NULL);
   }
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryCurrentPosition PMERR_OK Test5 - Exit.\n");
   /*
    * Test the functionality of the GpiBox call.
    */
   CWRITELOG(L_LOTRACE,0,"+ie+ GpiBox PMERR_OK Test5 - Entry.\n");
   Point.x = 15 * delta_x / 2;
   Point.y = 7 * delta_y / 2;
   MapPt(Point);
   GpiSetPattern(hPS,PATSYM_DENSE8);
   if(!GpiBox(hPS,DRO_OUTLINEFILL,&Point,15L,15L)){
      BADERROR("GpiBox",TRUE,"GpiBoxExh");
      return;
   }
   sprintf(out_string,"Filled Box 2x1.5");
   MapAndWriteString(hPS,(11*delta_x/2),(21*delta_y/4),out_string);
   CWRITELOG(L_LOTRACE,0,"+ix+ GpiBox PMERR_OK Test5 - Exit.\n");

   if(!lErrorCount){
     CWRITELOG(L_HDR,0,"$GpiBoxExh                                                      -    PASS.\n");
   }
   else{
     sprintf(out_string,"$ERRORCOUNT = %d\n",lErrorCount);
     CWRITELOG(L_HDR,0,out_string);
     CWRITELOG(L_HDR,0,"$GpiBoxExh                                                      -    FAIL.\n");
   }
   CWRITELOG(L_LOTRACE,0,"+ex+ GpiBoxExh - Exit.\n");
 } /* end else statement */
}
/****************************************************************************
 *\\ddd
 * Routine Name: GpiLineExh()
 *
 * Purpose: This function the functionality of the following APIs:
 *          GpiLine(), GpiSetCurrentPosition(), GpiSetLineType(), and
 *          GpiQueryLineType().
 *
 * System Requirements: Section 5.12.3
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               10/29/90, PDVT, Donna Williams
 *                     - testcase design
 *               11/01/90, PDVT, Donna Williams
 *                     - design implementation
 *               11/19/90, PDVT, Donna Williams
 *                     - redesign
 *
 * Inputs:  hPS            - presentation space
 *          hWndClient     - client window handle
 *          SelectionCall  - selection indicator
 *
 * Outputs: None.
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNAUTRE(), LOG_SIGNATURE,
 *                       set_delta(), MapPt(), BADERROR().
 *
 * Limitations: None.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *       for(every line type){
 *         // test functionality of GpiSetCurrentPosition
 *         GpiSetCurrentPosition(...);
 *        // test functionality of GpiSetLineType
 *        GpiSetLineType(...);
 *        // test functionality of GpiLine
 *        GpiLine(...);
 *        // test functionality of GpiQueryLineType and compare results
 *        GpiQueryLineType(...);
 *        // if there are any errors report these with BADERROR
 *      }
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiLineExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else{
      // call function to perform line functionality test
      line_type(hPS,hWndClient);
  } /* end else statement */
}
/****************************************************************************
 *\\ddd
 * Routine Name: GpiPolyLineExh()
 *
 * Purpose: This function tests GpiPolyLine() functionality.
 *
 * System Requirements: Section 5.12.3
 *
 * Revision Log: 10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               11/01/90, PDVT, Donna Williams
 *                     - design implementation
 *
 * Inputs:  hPS            - presentation space
 *          hWndClient     - client window handle
 *          SelectionCall  - selection indicator
 *
 * Outputs: None.
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNAUTRE(), LOG_SIGNATURE,
 *                       set_delta(), MapPt(), BADERROR(),
 *                       DosAllocMem(), cos(), sin(), DosFreeSeg().
 *
 * Limitations: None.
 *
 *\\end
 ***************************************************************************/
/****************************************************************************
 *\\algorithm
 * {
 *    // set up array for poly line function
 *    GpiPolyLine(...);  // draw spiral
 *    // check value received from GPI and log to PTT_LOG
 *    BADERROR(...);
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPolyLineExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  double    dAngle,
            dScale;
  PPOINTL   pptl;
  SHORT     sIndex;
  LONG      lErrorCount;
  char      out_string[100];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else{
    lErrorCount = 0L;
    set_delta(hPS);
    LOG_SIGNATURE;
    TEST_SIGNATURE(0,0);
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPolyLineExh - Entry.\n");
    /*
     * Test GpiPolyLine Functionality
     */
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyLine PMERR_OK Test - Entry.\n");
    if(!DosAllocMem((PVOID)&pptl,NUMPOINTS * sizeof(POINTL),PAG_READ | PAG_WRITE | PAG_COMMIT)){
      for(sIndex = 0; sIndex < NUMPOINTS; sIndex++){
         dAngle = sIndex * 2 * PI / (NUMPOINTS / NUMREV);
         dScale = 1 - (double)sIndex / NUMPOINTS;
         pptl[sIndex].x = (LONG) ((3000*PI)/2*(1+dScale * cos(dAngle)));
         pptl[sIndex].y = (LONG) ((*pBasisY)/2*(1+dScale * sin(dAngle)));
      }
      for(sIndex = 0; sIndex < NUMPOINTS; sIndex++)
        MapPt(pptl[sIndex]);
      GpiMove(hPS,pptl);
      if(!GpiPolyLine(hPS,NUMPOINTS-1L,pptl+1)){
         BADERROR("GpiPolyLine",TRUE,"GpiPolyLineExh");
         return;
      }
      DosFreeMem((PVOID)pptl);
    }
    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyLine PMERR_OK Test - Exit.\n");
    MapAndWriteString(hPS,(11*delta_x/2),(3*delta_y/4),"Tests: GpiPolyLine");

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiPolyLineExh                                                 -    PASS.\n");
    }
    else{
       sprintf(out_string,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_string);
       CWRITELOG(L_HDR,0,"$GpiPolyLineExh                                                 -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPolyLineExh - Exit.\n");
  } /* end else statement */
}

