#define INCL_GPI
#define INCL_ERRORS    // PM error definitions
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include "PTTDLLDF.h"
#include "arc_util.h"
#include "gpiarc.h"

/****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module name:  GPIARC
 *
 * Purpose:  This module is called by PTT to test the Gpi ARC APIs.
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - created initial module template
 *
 * SRS References: Section 5.3
 *
 * Contents: GpiFullArcExh(),
 *           GpiPartialArcExh(),
 *           GpiPointArcExh(),
 *
 * Limitations:  NONE.
 *
 *\\end
****************************************************************************/

/****************************************************************************
 *\\ddd
 * Routine Name: GpiFullArcExh()
 *
 * Purpose: This PTT entry point performs exhaustive testing of
 *          GpiFullArc.
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS                  - presentation space handle
 *         hWndClient           - client window handle
 *         SelectionCall        - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiSetArcParams
 *      GpiFullArc
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *
 *  GpiFullArc draws three sets of concentric circles
 *  GpiFullArc tests all combinations of DRO_FILL, DRO_OUTLINE, DRO_OUTLINEFILL
 *  GpiFullArc draws a chain of successive same size circles
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiFullArcExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
#define MAX_ARCS     10

  LONG xpos, ypos;              // current x and y position
  POINTL ptl;
  ARCPARAMS arcparam1;
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiFullArcExh - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    set_delta1 (hPS);
    delta_x /= 8;
    delta_y /= 8;

    GpiSetPattern (hPS, PATSYM_DENSE6);
    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", TRUE, "GpiFullArcExh");
        return;
    }

    ptl.x = 5000;
    ptl.y = 6500;
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    // Draw concentric circles each one proportionally larger than the previous

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test1 - Entry.\n");

    for (i = 1; i <= MAX_ARCS; i++) {
        if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(i,0)) == FALSE) {
            lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
            sprintf (out_str, "$GpiFullArc - Unexpected error. (i = %d).\n", i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test1 - Exit.\n");

   // Change position for next sequence

    xpos = 1000;
    ypos = 8000;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    xpos -= delta_x;

    // Draw concentric circles where the leftmost part of every circle is at
    // the same coordinate.

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test2 - Entry.\n");

    for (i = 1; i <= MAX_ARCS; i++) {
        if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(i,0)) == FALSE) {
            lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
            sprintf (out_str, "$GpiFullArc - Unexpected error. (i = %d).\n", i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        ptl.x = xpos + (delta_x * (i+1));
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test2 - Exit.\n");

   // Change position for next sequence

    xpos = 9000;
    ypos = 8000;
    ptl.x = xpos;
    ptl.y = 8000;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    xpos += delta_x;

    // Draw concentric circles where the rightmost part of every circle is at
    // the same coordinate.

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test3 - Entry.\n");

    for (i = 1; i <= MAX_ARCS; i++) {
        if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(i,0)) == FALSE) {
            lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
            sprintf (out_str, "$GpiFullArc - Unexpected error. (i = %d).\n", i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }
        ptl.x = xpos - (delta_x * (i+1));
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test3 - Exit.\n");

    // Draw same size circles with different fills

    xpos = 1600;
    ypos = 2000;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test4 - Entry.\n");

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(4, 0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test4 - Exit.\n");


    xpos += delta_x*9 + delta_x/2;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test5 - Entry.\n");

    if (GpiFullArc (hPS, DRO_FILL, MAKEFIXED(4, 0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test5 - Exit.\n");

    xpos += delta_x*9 + delta_x/2;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test6 - Entry.\n");

    if (GpiFullArc (hPS, DRO_OUTLINEFILL, MAKEFIXED(4, 0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test6 - Exit.\n");

    xpos += delta_x*9 + delta_x/2;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test6 - Entry.\n");

    if (GpiFullArc (hPS, DRO_OUTLINEFILL, MAKEFIXED(4, 0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test6 - Exit.\n");

    xpos += delta_x*9 + delta_x/2;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test7 - Entry.\n");

    if (GpiFullArc (hPS, DRO_FILL, MAKEFIXED(4, 0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test7 - Exit.\n");

    xpos += delta_x*9 + delta_x/2;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test8 - Entry.\n");

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(4, 0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test8 - Exit.\n");

    // Draw same size circles in chain

    xpos = 2000;
    ypos = 3900;
    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test9 - Entry.\n");

    for (i=1; i<=11; i++) {
        if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(5, 0)) == FALSE) {
            lErrorCount += BADERROR("GpiFullArc", FALSE, NULL);
            sprintf (out_str, "$GpiFullArc - Unexpected error. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        ptl.x = xpos+(i*delta_x*4);
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
    }
    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test9 - Exit.\n");

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiFullArcExh                                                  -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiFullArcExh                                                  -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiFullArcExh - Exit.\n");
  }
}

/****************************************************************************
 *\\ddd
 * Routine Name: GpiPartialArcExh()
 *
 * Purpose: This PTT entry point performs exhaustive testing of GpiPartialArc
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiSetArcParams
 *      GpiPartialArc
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *
 *  GpiPartialArc draws three sets of concentric circles
 *  GpiPartialArc draws a chain of successive same size circles
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPartialArcExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  LONG xpos, ypos;              // current x and y position
  int start_angle, sweep_angle;
  int arc_mult;                 // arc multiplier
  POINTL ptl;
  ARCPARAMS arcparam1;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPartialArcExh - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    set_delta1 (hPS);
    delta_x /= 8;
    delta_y /= 8;

    // Initialize x and y position

    xpos = 5000;
    ypos = 5000;

    GpiSetPattern (hPS, PATSYM_DENSE6);
    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);
    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", TRUE, "GpiPartialArcExh");
        return;
    }

    sweep_angle = 15;
    arc_mult = 4;

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPartialArc Test1 - Entry.\n");

    for (arc_mult = 4; arc_mult <= 28; arc_mult += 4) {
        for (start_angle = 0; start_angle < 360; start_angle += sweep_angle) {
            ptl.x = xpos;
            ptl.y = ypos;
            MapPt(ptl);
            GpiMove (hPS, &ptl);

            if (GpiPartialArc (hPS, &ptl, MAKEFIXED(arc_mult,0),
                                          MAKEFIXED(start_angle,0),
                                          MAKEFIXED(sweep_angle,0)) == FALSE) {
                lErrorCount += BADERROR("GpiPartialArc", FALSE, NULL);
                sprintf (out_str, "$GpiPartialArc - Unexpected error. (arcmult = %d; start_angle = %d)\n",
                         arc_mult, start_angle);
                CWRITELOG(L_LOTRACE,0,out_str);
            }
        }
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPartialArc Test1 - Exit.\n");

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiPartialArcExh                                               -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiPartialArcExh                                               -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPartialArcExh - Exit.\n");
  }
}


/****************************************************************************
 *\\ddd
 * Routine Name: GpiPointArcExh()
 *
 * Purpose: This entry point performs exhaustive testing of GpiPointArc.
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiSetArcParams
 *      GpiPointArc
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *
 *  GpiPointArc draws three sets of concentric circles
 *  GpiPointArc draws a chain of successive same size circles
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPointArcExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
#define MAX_ARCS 10

  LONG x_diam, y_diam;
  LONG xpos, ypos;              // current x and y position
  POINTL ptl, aptl[2];
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPointArcExh - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    set_delta1 (hPS);
    delta_x /= 8;
    delta_y /= 8;

    xpos = 5000;
    ypos = 6000;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric circles each one proportionally larger than the previous

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPointArc Test1 - Entry.\n");

    for (i = 1; i <= MAX_ARCS; i++) {
        xpos -= delta_x;
        x_diam += delta_x;
        y_diam += delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = xpos + x_diam;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos + 2*x_diam;
        aptl[1].y = ypos;
        MapPt(ptl);
        MapArrayOfPoints (2, aptl);
        GpiMove (hPS, &ptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 1. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        aptl[0].x = xpos + x_diam;
        aptl[0].y = ypos + y_diam;
        aptl[1].x = xpos;
        aptl[1].y = ypos;
        MapArrayOfPoints (2, aptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 2. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPointArc Test1 - Exit.\n");

   // Change position for next sequence

    xpos = 1000;
    xpos -= delta_x;
    ypos = 8000;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric circles where the leftmost part of every circle is at
    // the same coordinate.

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPointArc Test2 - Entry.\n");

    for (i = 1; i <= MAX_ARCS; i++) {
        x_diam += delta_x;
        y_diam += delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = xpos + x_diam;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos + 2*x_diam;
        aptl[1].y = ypos;
        MapPt(ptl);
        MapArrayOfPoints (2, aptl);
        GpiMove (hPS, &ptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 1. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        aptl[0].x = xpos + x_diam;
        aptl[0].y = ypos + y_diam;
        aptl[1].x = xpos;
        aptl[1].y = ypos;
        MapArrayOfPoints (2, aptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 2. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPointArc Test2 - Exit.\n");

   // Change position for next sequence

    xpos = 9000;
    ypos = 8000;
    xpos += delta_x;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric circles where the rightmost part of every circle is at
    // the same coordinate.

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPointArc Test3 - Entry.\n");

    for (i = 1; i <= MAX_ARCS; i++) {
        x_diam += delta_x;
        y_diam += delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = xpos - x_diam;
        aptl[0].y = ypos + y_diam;
        aptl[1].x = xpos - 2*x_diam;
        aptl[1].y = ypos;
        MapPt(ptl);
        MapArrayOfPoints (2, aptl);
        GpiMove (hPS, &ptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 1. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        aptl[0].x = xpos - x_diam;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos;
        aptl[1].y = ypos;
        MapArrayOfPoints (2, aptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 2. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPointArc Test3 - Exit.\n");

    // Draw same size circles in chain

    x_diam = 7*delta_x;
    y_diam = 7*delta_y;
    xpos = 1200 + 2*x_diam;
    ypos = 2800;

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPointArc Test4 - Entry.\n");

    for (i=1; i<=10; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = xpos - x_diam;
        aptl[0].y = ypos + y_diam;
        aptl[1].x = xpos - 2*x_diam;
        aptl[1].y = ypos;
        MapPt(ptl);
        MapArrayOfPoints (2, aptl);
        GpiMove (hPS, &ptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 1. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        aptl[0].x = xpos - x_diam;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos;
        aptl[1].y = ypos;
        MapArrayOfPoints (2, aptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", FALSE, NULL);
            sprintf (out_str, "$GpiPointArc - Unexpected error on semicircle 2. (i = %d)\n",i);
            CWRITELOG(L_LOTRACE,0,out_str);
        }

        xpos += delta_x*4;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPointArc Test4 - Exit.\n");

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiPointArcExh                                                 -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiPointArcExh                                                 -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPointArcExh - Exit.\n");
  }
}
