#define INCL_GPI
#define INCL_ERRORS    // PM error definitions
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include "PTTDLLDF.h"
#include "arc_util.h"
#include "gpiarc.h"
#include "arc_grid.h"

/****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module name:  GPIARC
 *
 * Purpose:  This module is called by PTT to test the Gpi ARC APIs.
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - created initial module template
 *
 * SRS References: Section 5.3
 *
 * Contents: GpiSetDefArcParamsExh1(),
 *           GpiSetDefArcParamsExh2(),
 *
 * Limitations:  NONE.
 *
 *\\end
****************************************************************************/

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetDefArcParamsExh1()
 *
 * Purpose: This PTT entry point begins exhaustive testing of GpiSetDefArcParams
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *      GpiSetDefArcParams
 *      GpiQueryDefArcParams
 *      GpiResetPS
 *      GpiFullArc
 *      GpiPartialArc
 *      GpiPointArc
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *
 *  Use GpiSetDefArcParams to create a similar display to gpi_arc_pre1 but
 *  using static multiplier so that only arcparams affects display.
 *
 *  GpiResetPS is required after every call to GpiSetDefArcParams to cause
 *  the arcparams to take effect.
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetDefArcParamsExh1(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL ptl;
  POINTL aptl[6];
  ARCPARAMS arcparam1, arcparam2;
  LONG xpos, ypos;              // current x and y position
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetDefArcParamsExh1 - Entry.\n");
    lErrorCount = 0L;

    arc_grid (hPS, (LONG)INCH);

    // GpiFullArc precision
    ptl.x = 2*delta_x;
    ptl.y = 8*delta_y;

    MapAndWriteString1 (hPS, ptl.x - delta_x, ptl.y - 2*delta_y - 2*line_space,
                       "GpiFullArc - 1 & 2");
    MapAndWriteString1 (hPS, ptl.x - delta_x, ptl.y - 2*delta_y - 3*line_space,
                       "inch circles");
    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }


    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    arcparam1.lR = 0L;
    arcparam1.lQ = 2*delta_y;
    arcparam1.lP = 2*delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test2 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }


    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test2 - Exit.\n");

    // GpiPartialArc precision
    MapAndWriteString1 (hPS, 4*delta_x + delta_x/2, 2*delta_y + delta_y/2,
                       "GpiPartialArc - 1 & 2");

    MapAndWriteString1 (hPS, 4*delta_x + delta_x/2, (2*delta_y + delta_y/2) - line_space,
                       "inch semi-circles");

    ptl.x = 3*delta_x;
    ptl.y = 1*delta_y;
    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test3 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }


    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test3 - Exit.\n");

    ptl.x = 3*delta_x;
    ptl.y = 1*delta_y;
    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = 2*delta_y;
    arcparam1.lP = 2*delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test4 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }


    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test4 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test4 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test4 - Exit.\n");

    // GpiPointArc precision
    xpos = 4*delta_x;
    ypos = 4*delta_y;
    MapAndWriteString1 (hPS, xpos, ypos - delta_y/2 - 2*line_space,
                       "GpiPointArc - 1 & 2");

    MapAndWriteString1 (hPS, xpos, ypos - delta_y/2 - 3*line_space,
                       "inch chords");

    for (i = 1; i <= 2; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = ptl.x + i*delta_x;
        aptl[0].y = ptl.y;
        aptl[1].x = ptl.x + i*delta_x;
        aptl[1].y = ptl.y + i*delta_y;
        MapArrayOfPoints (2, aptl);
        MapPt(ptl);
        arcparam1.lR = i*delta_x;
        arcparam1.lQ = 0L;
        arcparam1.lP = 0L;
        arcparam1.lS = i*delta_y;
        MapArc(arcparam1);

        sprintf (out_str, "+ie+ GpiSetDefArcParams Test %d - Entry.\n", i+4);
        CWRITELOG(L_LOTRACE,0,out_str);

        if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
            lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
        }


        if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
            lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh1");
            return;
        }

        sprintf (out_str, "+ie+ GpiQueryDefArcParams Test %d - Entry.\n", i+4);
        CWRITELOG(L_LOTRACE,0,out_str);

        if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
            lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
        }

        if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
            CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
        }

        sprintf (out_str, "+ix+ GpiQueryDefArcParams Test %d - Exit.\n", i+4);
        CWRITELOG(L_LOTRACE,0,out_str);

        GpiMove (hPS, &ptl);

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", TRUE, "GpiSetDefArcParamsExh1");
            return;
        }

        sprintf (out_str, "+ix+ GpiSetDefArcParams Test %d - Exit.\n", i+4);
        CWRITELOG(L_LOTRACE,0,out_str);
    }

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiSetDefArcParamsExh1                                         -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiSetDefArcParamsExh1                                         -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetDefArcParamsExh1 - Exit.\n");
  }
}

/****************************************************************************
 *\\ddd
 * Routine Name: GpiSetDefArcParamsExh2()
 *
 * Purpose: This PTT entry point begins exhaustive testing of GpiSetDefArcParams
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *      GpiSetDefArcParams
 *      GpiQueryDefArcParams
 *      GpiResetPS
 *      GpiFullArc
 *      GpiPartialArc
 *      GpiBeginArea
 *      GpiEndArea
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *
 *  GpiSetArcParams is used to demonstrate drawing direction using GpiFullArc
 *  and GpiPartialArc in conjunction with GpiBeginArea and GpiEndArea which
 *  causes different fills to occur dependent upon drawing direction.
 *
 *  GpiResetPS is required after every call to GpiSetDefArcParams to cause
 *  the arcparams to take effect. This call must be done outside of the
 *  GpiBeginArea/GpiEndArea pair.
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetDefArcParamsExh2(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL ptl;
  ARCPARAMS arcparam1, arcparam2;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiSetDefArcParamsExh2 - Entry.\n");
    lErrorCount = 0L;

    set_delta1 (hPS);
    delta_x /= 2;
    delta_y /= 2;

    // GpiSetDefArcParams / GpiFullArc #1

    ptl.x = 3000;
    ptl.y = 8500;

    MapAndWriteString1 (hPS, ptl.x - delta_x, 9500L,
                       "GpiFullArc");

    MapAndWriteString1 (hPS, ptl.x - 3*delta_x/2, ptl.y - 2*delta_y - 2*line_space,
                       "p*q > r*s");

    MapAndWriteString1 (hPS, ptl.x - 3*delta_x/2, ptl.y - 2*delta_y - 3*line_space,
                       "for both circles");

    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ Drawing direction Test1 - Entry.\n");

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }

    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    if (GpiSetPattern (hPS, PATSYM_DENSE6) == FALSE) {
        lErrorCount += BADERROR("GpiSetPattern", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    if (GpiBeginArea (hPS, BA_WINDING | BA_BOUNDARY) == FALSE) {
        lErrorCount += BADERROR("GpiBeginArea", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    arcparam1.lR = 0L;
    arcparam1.lQ = 2*delta_y;
    arcparam1.lP = 2*delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    if (GpiEndArea (hPS) == FALSE) {
        lErrorCount += BADERROR("GpiEndArea", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ Drawing direction Test1 - Exit.\n");

    // GpiSetDefArcParams / GpiFullArc #2

    ptl.x = 3000;
    ptl.y = 4500;

    MapAndWriteString1 (hPS, ptl.x - 3*delta_x/2, ptl.y - 2*delta_y - 2*line_space,
                       "p*q < r*s");

    MapAndWriteString1 (hPS, ptl.x - 3*delta_x/2, ptl.y - 2*delta_y - 3*line_space,
                       "then p*q > r*s");

    MapPt(ptl);

    arcparam1.lR = delta_x;
    arcparam1.lQ = 0L;
    arcparam1.lP = 0L;
    arcparam1.lS = delta_y;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ Drawing direction Test2 - Entry.\n");

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }

    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    if (GpiSetPattern (hPS, PATSYM_DENSE6) == FALSE) {
        lErrorCount += BADERROR("GpiSetPattern", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    if (GpiBeginArea (hPS, BA_WINDING | BA_BOUNDARY) == FALSE) {
        lErrorCount += BADERROR("GpiBeginArea", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    arcparam1.lR = 0L;
    arcparam1.lQ = 2*delta_y;
    arcparam1.lP = 2*delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    if (GpiEndArea (hPS) == FALSE) {
        lErrorCount += BADERROR("GpiEndArea", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ Drawing direction Test2 - Exit.\n");

    // GpiSetDefArcParams / GpiFullArc #3

    ptl.x = 3000;
    ptl.y = 1500;
    MapAndWriteString1 (hPS, ptl.x - 2*delta_x, ptl.y,
                       "p*q = r*s");

    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = 0L;
    arcparam1.lP = delta_x/2;
    arcparam1.lS = delta_y/2;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ Drawing direction Test3 - Entry.\n");

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }

    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    arcparam1.lR = 0L;
    arcparam1.lQ = 0L;
    arcparam1.lP = delta_x;
    arcparam1.lS = delta_y;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ Drawing direction Test3 - Exit.\n");

    // GpiSetDefArcParams / GpiPartialArc #1

    ptl.x = 5000 + 2*delta_x;
    ptl.y = 7800 + delta_y/2;

    MapAndWriteString1 (hPS, ptl.x - delta_x, 9500L,
                       "GpiPartialArc");

    MapAndWriteString1 (hPS, ptl.x - (3*delta_x)/2, (ptl.y - delta_y/2) - 2*line_space,
                       "p*q > r*s");

    MapAndWriteString1 (hPS, ptl.x - (3*delta_x)/2, (ptl.y - delta_y/2) - 3*line_space,
                       "for both arcs");

    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ Drawing direction Test4 - Entry.\n");

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }

    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    if (GpiSetPattern (hPS, PATSYM_DENSE6) == FALSE) {
        lErrorCount += BADERROR("GpiSetPattern", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    if (GpiBeginArea (hPS, BA_WINDING | BA_BOUNDARY) == FALSE) {
        lErrorCount += BADERROR("GpiBeginArea", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(45,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    ptl.x = 5000 + 2*delta_x;
    ptl.y = 7800 + delta_y/2;
    MapPt(ptl);
    GpiLine (hPS, &ptl);
    GpiMove (hPS, &ptl);

    ptl.x = 5000 + 2*delta_x;
    ptl.y = 7800;
    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = 2*delta_y;
    arcparam1.lP = 2*delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(180,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    if (GpiEndArea (hPS) == FALSE) {
        lErrorCount += BADERROR("GpiEndArea", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ Drawing direction Test4 - Exit.\n");

    // GpiSetDefArcParams / GpiPartialArc #2

    ptl.x = 6000;
    ptl.y = 4000 + delta_y/4;

    MapAndWriteString1 (hPS, ptl.x - (3*delta_x)/2, (ptl.y -delta_y/4) - 2*line_space,
                       "p*q < r*s then");

    MapAndWriteString1 (hPS, ptl.x - (3*delta_x)/2, (ptl.y - delta_y/4) - 3*line_space,
                       "p*q > r*s");

    MapPt(ptl);

    arcparam1.lR = delta_x;
    arcparam1.lQ = 0L;
    arcparam1.lP = 0L;
    arcparam1.lS = delta_y;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ Drawing direction Test5 - Entry.\n");

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }

    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(45,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    ptl.x = 6000;
    ptl.y = 4000 + delta_y/4;
    MapPt(ptl);
    GpiLine (hPS, &ptl);
    GpiMove (hPS, &ptl);

    ptl.x = 6000;
    ptl.y = 4000;
    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = 2*delta_y;
    arcparam1.lP = 2*delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(180,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ Drawing direction Test5 - Exit.\n");

    // GpiSetDefArcParams / GpiPartialArc #3

    ptl.x = 6000;
    ptl.y = 1500 + delta_y/4;

    MapAndWriteString1 (hPS, ptl.x - 2*delta_x, (ptl.y -delta_y/4),
                       "p*q = r*s");

    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = 0L;
    arcparam1.lP = delta_x/2;
    arcparam1.lS = delta_y/2;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ Drawing direction Test6 - Entry.\n");

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetDefArcParams Test1 - Entry.\n");

    if (GpiSetDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetDefArcParams", FALSE, NULL);
    }

    if (GpiResetPS (hPS, GRES_ATTRS) == FALSE) {
        lErrorCount += BADERROR("GpiResetPS", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test1 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test1 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(45,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetDefArcParams Test1 - Exit.\n");

    ptl.x = 6000;
    ptl.y = 1500;
    MapPt(ptl);

    arcparam1.lR = 0L;
    arcparam1.lQ = 0L;
    arcparam1.lP = delta_x;
    arcparam1.lS = delta_y;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test2 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam2) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test2 - Exit.\n");
    CWRITELOG(L_LOTRACE,0,"+ie+ GpiQueryDefArcParams Test3 - Entry.\n");

    if (GpiQueryDefArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiQueryDefArcParams", FALSE, NULL);
    }

    if (memcmp (&arcparam1, &arcparam2, sizeof(ARCPARAMS)) != 0) {
        CWRITELOG(L_LOTRACE,0,"*ERROR - arc parameters do not match expected values.\n");
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiQueryDefArcParams Test3 - Exit.\n");

    GpiMove (hPS, &ptl);

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(180,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiSetDefArcParamsExh2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ Drawing direction Test6 - Exit.\n");

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiSetDefArcParamsExh2                                         -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiSetDefArcParamsExh2                                         -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiSetDefArcParamsExh2 - Exit.\n");
  }
}
