#define INCL_GPI
#define INCL_ERRORS    // PM error definitions
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include "pttdlldf.h"
#include "arc_util.h"
#include "gpiarc.h"


/****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module name:  GPIARC
 *
 * Purpose:  This module is called by PTT to test the Gpi ARC APIs.
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - created initial module template
 *
 * SRS References: Section 5.3
 *
 * Contents: InitTest(),
 *           gpi_arc_app(),
 *           GpiPolyFilletExh(),
 *           GpiPolyFilletSharpExh(),
 *           GpiPolySplineExh()
 *
 * Design Overview:  The GPI Arc test case verifies print drivers arc related
 *              functionality. Tests within this test case consist of pre-
 *              cision, functionality, and exhaustive tests.
 *
 *              The precision tests verify the GPI Arc API's for positioning
 *              on the page and sizing of different arc shapes.
 *
 *              The functionality test verifies the GPI Arc API's as they
 *              function together and in use with other GPI API's such as
 *              GPIBeginPath and GPIEndPath.
 *
 *              Exhaustive testing of the GPI Arc API's involves verifying the
 *              functionality of a given API with respect to the print driver.
 *
 *              The API's GpiQueryArcParams and GpiQueryDefArcParams are
 *              test in conjunction with GpiSetArcParams and GpiSetDefArcParams.
 *
 * Limitations:  NONE.
 *
 *\\end
****************************************************************************/


/****************************************************************************
 *\\ddd
 * Routine Name: InitTest()
 *
 * Purpose: Initialize function name and address array for PTT.
 *
 * System Requirements: NONE
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial function template
 *
 * Inputs:   sel            - pass thru parameter for other function calls
 *           init_info_p    - pointer to info array
 *
 * Outputs:  init_info_p    - updated with all entry point names and addresses
 *
 * Subroutines Required: InitTestSegVars()
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 * for all entry points in this test case {
 *   init_info_p->function name = name of the entry point
 *   init_info_p->function address = address of the entry point
 *  }
 *\\end
 ***************************************************************************/

 VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO init_info_p)
  {
    init_info_p->ntests = NUMARCTESTS;
    strcpy(init_info_p->group_name,"Gpi Arc");
    strcpy(init_info_p->test_names[0],"GpiArcPre1");
    init_info_p->test_addrs[0]=gpi_arc_pre1;
    strcpy(init_info_p->test_names[1],"GpiArcPre2");
    init_info_p->test_addrs[1]=gpi_arc_pre2;
    strcpy(init_info_p->test_names[2],"GpiArcApp");
    init_info_p->test_addrs[2]=gpi_arc_app;
    strcpy(init_info_p->test_names[3],"GpiSetArcParamsExh1");
    init_info_p->test_addrs[3]=GpiSetArcParamsExh1;
    strcpy(init_info_p->test_names[4],"GpiSetArcParamsExh2");
    init_info_p->test_addrs[4]=GpiSetArcParamsExh2;
    strcpy(init_info_p->test_names[5],"GpiSetDefArcParamsExh1");
    init_info_p->test_addrs[5]=GpiSetDefArcParamsExh1;
    strcpy(init_info_p->test_names[6],"GpiSetDefArcParamsExh2");
    init_info_p->test_addrs[6]=GpiSetDefArcParamsExh2;
    strcpy(init_info_p->test_names[7],"GpiFullArcExh");
    init_info_p->test_addrs[7]=GpiFullArcExh;
    strcpy(init_info_p->test_names[8],"GpiPartialArcExh");
    init_info_p->test_addrs[8]=GpiPartialArcExh;
    strcpy(init_info_p->test_names[9],"GpiPointArcExh");
    init_info_p->test_addrs[9]=GpiPointArcExh;
    strcpy(init_info_p->test_names[10],"GpiPolyFilletExh");
    init_info_p->test_addrs[10]=GpiPolyFilletExh;
    strcpy(init_info_p->test_names[11],"GpiPolyFilletSharpExh");
    init_info_p->test_addrs[11]=GpiPolyFilletSharpExh;
    strcpy(init_info_p->test_names[12],"GpiPolySplineExh");
    init_info_p->test_addrs[12]=GpiPolySplineExh;

    InitTestSegVars(sel);
  }

/****************************************************************************
 *\\ddd
 * Routine Name: gpi_arc_app()
 *
 * Purpose: This entry point for PTT begins the application test.
 *
 * System Requirements:  Section 5.3.2
 *
 * Revision Log: 12/15/90, PDVT, Frederik Johannessen
 *                     - created testcase
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiSetArcParams
 *      GpiFullArc
 *      GpiPartialArc
 *      GpiPointArc
 *      GpiPolyFillet
 *      GpiPolyFilletSharp
 *      GpiPolySpline
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *      For sharpness values < 1 the GpiPolyFilletSharp does not behave as
 *      expected. Values < 1 are supposed to return an ellipse. The actual
 *      result is a horizontal line.
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *  Exercise all arc related functions.
 *  GpiSetArcParams to set arcparams to create good size circles
 *  GpiFullArc with GpiPartial arc to create a filled circle containing four
 *    pie shapes
 *  GpiPolyFillet creates two mirror parabolas and GpiPointArc joins three
 *    points of the parabolas
 *  Alternate GpiPolyFilletSharp and GpiPolySpline to create a warped looking
 *    sine wave
 *
 *\\end
 ***************************************************************************/
VOID APIENTRY gpi_arc_app(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  LONG xpos, ypos;              // current x and y position
  POINTL ptl, delta_ptl, aptl[3];
  LONG x_diam, y_diam;
  ARCPARAMS arcparam1;
  int i;
  FIXED afx[2];
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    set_delta1 (hPS);
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"GPIARC Application Test - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    delta_ptl.x = delta_x/8;
    delta_ptl.y = delta_y/8;

    GpiSetPattern (hPS, PATSYM_DENSE6);

    arcparam1.lR = 0L;
    arcparam1.lQ = 10*delta_ptl.y;
    arcparam1.lP = 10*delta_ptl.x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);
    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", TRUE, "GpiArcApp");
        return;
    }

    xpos = 3000;
    ypos = 8000;

    // Start with a circle

    MapAndWriteString1 (hPS, xpos - delta_x , ypos - 10*delta_ptl.y - 2*line_space,
                       "GpiFullArc - filled");
    MapAndWriteString1 (hPS, xpos - delta_x, ypos - 10*delta_ptl.y - 3*line_space,
                       "GpiPartialArc -");
    MapAndWriteString1 (hPS, xpos - delta_x, ypos - 10*delta_ptl.y - 4*line_space,
                       "  4 semi-circles");

    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    if (GpiFullArc (hPS, DRO_FILL, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiArcApp");
        return;
    }

    // Draw 4 partial arcs that encompass and outline circle

    for (i = 60; i <= 360; i += 90) {
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0), MAKEFIXED(i,0), MAKEFIXED(60,0)) == FALSE) {
            lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiArcApp");
            return;
        }
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0), MAKEFIXED((i+60),0), 0L) == FALSE) {
            lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiArcApp");
            return;
        }
    }

    // User GpiPolyFillet with GpiPointArc

    xpos = 7000;
    ypos = 8000;
    x_diam = delta_ptl.x*10;
    y_diam = delta_ptl.y*4;

    MapAndWriteString1 (hPS, xpos - delta_x, ypos - 10*delta_ptl.y - 2*line_space,
                       "GpiPolyFillet -");
    MapAndWriteString1 (hPS, xpos - delta_x, ypos - 10*delta_ptl.y - 3*line_space,
                       "  two mirror parabolas");
    MapAndWriteString1 (hPS, xpos - delta_x, ypos - 10*delta_ptl.y - 4*line_space,
                       "GpiPointArc - joining");
    MapAndWriteString1 (hPS, xpos - delta_x, ypos - 10*delta_ptl.y - 5*line_space,
                       "  3 points of parabolas");

    aptl[0].x = xpos;
    aptl[0].y = ypos - y_diam;
    aptl[1].x = xpos;
    aptl[1].y = ypos + y_diam;
    aptl[2].x = xpos - x_diam;
    aptl[2].y = ypos + y_diam;
    ptl.x = xpos - x_diam;
    ptl.y = ypos - y_diam;
    MapArrayOfPoints (3, aptl);
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    if (GpiPolyFillet (hPS, 3L, aptl) == FALSE) {
        lErrorCount += BADERROR("GpiPolyFillet", TRUE, "GpiArcApp");
        return;
    }

    aptl[0].x = xpos;
    aptl[0].y = ypos - y_diam;
    aptl[1].x = xpos;
    aptl[1].y = ypos + y_diam;
    aptl[2].x = xpos + x_diam;
    aptl[2].y = ypos + y_diam;
    ptl.x = xpos + x_diam;
    ptl.y = ypos - y_diam;
    MapArrayOfPoints (3, aptl);
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    if (GpiPolyFillet (hPS, 3L, aptl) == FALSE) {
        lErrorCount += BADERROR("GpiPolyFillet", TRUE, "GpiArcApp");
        return;
    }

    aptl[0].x = xpos + x_diam;
    aptl[0].y = ypos - y_diam;
    aptl[1].x = xpos - x_diam;
    aptl[1].y = ypos - y_diam;
    MapArrayOfPoints (2, aptl);
    if (GpiPointArc (hPS, aptl) == FALSE) {
        lErrorCount += BADERROR("GpiPointArc", TRUE, "GpiArcApp");
        return;
    }

    // Demonstrate GpiPolyFilletSharp and GpiPolySpline

    delta_ptl.x *= 8;
    delta_ptl.y *= 8;
    MapPt(delta_ptl);

    ptl.x = 1000;
    ptl.y = 3000;

    MapAndWriteString1 (hPS, 3500L, ptl.y+delta_x,
                       "GpiPolyFilletSharp above");

    MapAndWriteString1 (hPS, 3500L, ptl.y-delta_x-2*line_space,
                       "GpiPolySpline below");

    MapPt(ptl);
    GpiMove (hPS, &ptl);

    for (i = 1; i <= 4; i++) {
        GpiQueryCurrentPosition (hPS, &ptl);
        afx[0] = MAKEFIXED(1,0);
        aptl[0].x = ptl.x + delta_ptl.x / 2;
        aptl[0].y = ptl.y + delta_ptl.y;
        aptl[1].x = ptl.x + delta_ptl.x;
        aptl[1].y = ptl.y;
        if (GpiPolyFilletSharp (hPS, 2L, aptl, afx) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFilletSharp", TRUE, "GpiArcApp");
            return;
        }
        GpiQueryCurrentPosition (hPS, &ptl);
        aptl[0].x = ptl.x;
        aptl[0].y = ptl.y - delta_ptl.y;
        aptl[1].x = ptl.x + delta_ptl.x;
        aptl[1].y = ptl.y - delta_ptl.y;
        aptl[2].x = ptl.x + delta_ptl.x;
        aptl[2].y = ptl.y;
        if (GpiPolySpline (hPS, 3L, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPolySpline", TRUE, "GpiArcApp");
            return;
        }
    }

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GPIARC Application Test                                        -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d.\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GPIARC Application Test                                        -    FAIL.\n");
    }

    CWRITELOG(L_LOTRACE,0,"GPIARC Application Test - Exit.\n");

  }
}

/****************************************************************************
 *\\ddd
 * Routine Name: GpiPolyFilletExh()
 *
 * Purpose: This entry point performs exhaustive testing of GpiPolyFillet.
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiPolyFillet
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *  Draw three sets of concentric circles.
 *  Draw a chain that shows intersecting circles.
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPolyFilletExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
#define MAX_ARCS 10

  LONG x_diam, y_diam;
  LONG xpos, ypos;              // current x and y position
  POINTL ptl, aptl[5];
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPolyFilletExh - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    set_delta1 (hPS);
    delta_x /= 8;
    delta_y /= 8;

    xpos = 5000;
    ypos = 6000;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric circles each one proportionally larger than the previous

    for (i = 1; i <= MAX_ARCS; i++) {
        xpos -= delta_x;
        x_diam += delta_x;
        y_diam += delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = xpos;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos + 2*x_diam;
        aptl[1].y = ypos - y_diam;
        aptl[2].x = xpos + 2*x_diam;
        aptl[2].y = ypos + y_diam;
        aptl[3].x = xpos;
        aptl[3].y = ypos + y_diam;
        aptl[4].x = xpos;
        aptl[4].y = ypos;
        MapArrayOfPoints (5, aptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFillet PMERR_OK Test1 - Entry.\n");

        if (GpiPolyFillet (hPS, 5L, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFillet", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFillet PMERR_OK Test1 - Exit.\n");

    }

   // Change position for next sequence

    xpos = 1000;
    xpos -= delta_x;
    ypos = 8000;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric circles where the leftmost part of every circle is at
    // the same coordinate.

    for (i = 1; i <= MAX_ARCS; i++) {
        x_diam += delta_x;
        y_diam += delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = xpos;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos + 2*x_diam;
        aptl[1].y = ypos - y_diam;
        aptl[2].x = xpos + 2*x_diam;
        aptl[2].y = ypos + y_diam;
        aptl[3].x = xpos;
        aptl[3].y = ypos + y_diam;
        aptl[4].x = xpos;
        aptl[4].y = ypos;
        MapArrayOfPoints (5, aptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFillet PMERR_OK Test2 - Entry.\n");

        if (GpiPolyFillet (hPS, 5L, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFillet", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFillet PMERR_OK Test2 - Exit.\n");
    }

   // Change position for next sequence

    xpos = 9000;
    ypos = 8000;
    xpos += delta_x;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric circles where the rightmost part of every circle is at
    // the same coordinate.

    for (i = 1; i <= MAX_ARCS; i++) {
        x_diam += delta_x;
        y_diam += delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = xpos;
        aptl[0].y = ypos + y_diam;
        aptl[1].x = xpos - 2*x_diam;
        aptl[1].y = ypos + y_diam;
        aptl[2].x = xpos - 2*x_diam;
        aptl[2].y = ypos - y_diam;
        aptl[3].x = xpos;
        aptl[3].y = ypos - y_diam;
        aptl[4].x = xpos;
        aptl[4].y = ypos;
        MapArrayOfPoints (5, aptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFillet PMERR_OK Test3 - Entry.\n");

        if (GpiPolyFillet (hPS, 5L, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFillet", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFillet PMERR_OK Test3 - Exit.\n");
    }

    // Draw same size circles in chain

    xpos = 1200;
    ypos = 2800;
    x_diam = 7*delta_x;
    y_diam = 7*delta_y;

    for (i=1; i<=10; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = xpos;
        aptl[0].y = ypos - y_diam;
        aptl[1].x = xpos + 2*x_diam;
        aptl[1].y = ypos - y_diam;
        aptl[2].x = xpos + 2*x_diam;
        aptl[2].y = ypos + y_diam;
        aptl[3].x = xpos;
        aptl[3].y = ypos + y_diam;
        aptl[4].x = xpos;
        aptl[4].y = ypos;
        MapArrayOfPoints (5, aptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFillet PMERR_OK Test4 - Entry.\n");

        if (GpiPolyFillet (hPS, 5L, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFillet", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFillet PMERR_OK Test4 - Exit.\n");

        xpos += delta_x*4;
    }

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiPolyFilletExh                                               -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiPolyFilletExh                                               -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPolyFilletExh - Exit.\n");
  }
}



/****************************************************************************
 *\\ddd
 * Routine Name: GpiPolyFilletSharpExh()
 *
 * Purpose: This PTT entry point performs exhaustive testing on GpiPolyFilletSharp
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *               5/24/91 Vish Thyagarajan
 *                     - removed negative sharpness from testcase
 *                     - changed the MAKEFIXED parameters to check 0.1 to 0.9 sharpness
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiPolyFilletSharp
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *  Draw successively larger parabolas with a fixed sharpness
 *  Exercise all possible sharpness values with using fixed points
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPolyFilletSharpExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  LONG x_diam, y_diam;
  LONG xpos, ypos;              // current x and y position
  POINTL ptl, aptl[2];
  FIXED afx[2];
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPolyFilletSharpExh - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    set_delta1 (hPS);
    delta_x /= 8;
    delta_y /= 8;

    xpos = 5000;
    ypos = 6500;
    x_diam = 0;
    y_diam = 0;

    // Draw concentric parabolas each one proportionally larger than the previous

    for (i = 1; i <= 10; i++) {
        xpos -= delta_x;
        x_diam += delta_x;
        y_diam += 4*delta_y;
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = xpos + x_diam;
        aptl[0].y = ypos + y_diam;
        afx[0] = MAKEFIXED(1,0);
        aptl[1].x = xpos + 2*x_diam;
        aptl[1].y = ypos;
        MapArrayOfPoints (2, aptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFilletSharp PMERR_OK Test1 - Entry.\n");

        if (GpiPolyFilletSharp (hPS, 2L, aptl, afx) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFilletSharp", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFilletSharp PMERR_OK Test1 - Exit.\n");
    }

    MapAndWriteString1 (hPS, 4000L, ypos-2*line_space, "Sharpness:  1;");
    MapAndWriteString1 (hPS, 3600L, ypos-3*line_space, "Current position varies");

    xpos = 1000;
    ypos = 3000;

    for (i = 1; i <= 9; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = 2000;
        aptl[0].y = 5000;
        aptl[1].x = 3000;
        aptl[1].y = 3000;
        MapArrayOfPoints (2, aptl);
        afx[0] = MAKEFIXED(i,0);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFilletSharp PMERR_OK Test2 - Entry.\n");

        if (GpiPolyFilletSharp (hPS, 2L, aptl, afx) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFilletSharp", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFilletSharp PMERR_OK Test2 - Exit.\n");
    }

    MapAndWriteString1 (hPS, 800L, ypos-2*line_space, "Sharpness: 1 to 9");

    xpos = 4000;
    ypos = 5000;

    MapAndWriteString1 (hPS, 3800L, ypos-5*line_space, "Sharpness < 1 Ellipse");
    MapAndWriteString1 (hPS, 3800L, ypos-3*line_space, "Sharpness = 1 Parabola");
    MapAndWriteString1 (hPS, 3800L, ypos-1*line_space, "Sharpness > 1 Hyperbola");

    xpos = 7000;
    ypos = 3000;

    for (i = 1; i <= 9; i++) {
        ptl.x = 7000;
        ptl.y = 3000;
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        aptl[0].x = 8000;
        aptl[0].y = 5000;
        aptl[1].x = 9000;
        aptl[1].y = 3000;
        MapArrayOfPoints (2, aptl);
        afx[0] = MAKEFIXED(0,i*6535);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFilletSharp PMERR_OK Test4 - Entry.\n");

        if (GpiPolyFilletSharp (hPS, 2L, aptl, afx) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFilletSharp", FALSE, NULL);
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFilletSharp PMERR_OK Test4 - Exit.\n");

    }

    MapAndWriteString1 (hPS, 6700L, ypos-2*line_space, "Sharpness: .1 to .9");

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiPolyFilletSharpExh                                          -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiPolyFilletSharpExh                                          -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPolyFilletSharpExh - Exit.\n");
  }
}


/****************************************************************************
 *\\ddd
 * Routine Name: GpiPolySplineExh()
 *
 * Purpose: This PTT entry point performs exhaustive testing on GpiPolySpline
 *
 * System Requirements: Section 5.3.3
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS            - presentation space handle
 *         hWndClient     - client window handle
 *         SelectionCall  - selection indicator
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiPolySpline
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *      MapArrayOfPoints
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *  Draw a figure eight using a single call
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPolySplineExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
#define MAX_PTS 12

  LONG x_diam, y_diam;
  LONG xpos, ypos;              // current x and y position
  POINTL ptl, aptl[MAX_PTS];
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"+ee+ GpiPolySplineExh - Entry.\n");
    lErrorCount = 0L;

    // Delta x and y initialization

    set_delta1 (hPS);

    xpos = 5000;
    ypos = 5000;
    x_diam = 0;
    y_diam = 0;

    // Draw a figure eight

    MapAndWriteString1 (hPS, xpos-(7*delta_x/4), ypos-delta_y, "Figure eight starting at Top,");
    MapAndWriteString1 (hPS, xpos-(7*delta_x/4), ypos-delta_y - line_space, "center marker and going");
    MapAndWriteString1 (hPS, xpos-(7*delta_x/4), ypos-delta_y - 2*line_space, "right.");

    x_diam += delta_x*3;
    y_diam += delta_y*3;
    ptl.x = xpos;
    ptl.y = ypos + y_diam;
    MapPt(ptl);
    GpiMove (hPS, &ptl);
    aptl[0].x = xpos + x_diam;
    aptl[0].y = ypos + y_diam;
    aptl[1].x = xpos + x_diam;
    aptl[1].y = ypos;
    aptl[2].x = xpos;
    aptl[2].y = ypos;
    aptl[3].x = xpos - x_diam;
    aptl[3].y = ypos;
    aptl[4].x = xpos - x_diam;
    aptl[4].y = ypos - y_diam;
    aptl[5].x = xpos;
    aptl[5].y = ypos - y_diam;
    aptl[6].x = xpos + x_diam;
    aptl[6].y = ypos - y_diam;
    aptl[7].x = xpos + x_diam;
    aptl[7].y = ypos;
    aptl[8].x = xpos;
    aptl[8].y = ypos;
    aptl[9].x = xpos - x_diam;
    aptl[9].y = ypos;
    aptl[10].x = xpos - x_diam;
    aptl[10].y = ypos + y_diam;
    aptl[11].x = xpos;
    aptl[11].y = ypos + y_diam;
    MapArrayOfPoints ((int)MAX_PTS, aptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolySpline PMERR_OK Test1 - Entry.\n");

    if (GpiPolySpline (hPS, (LONG)MAX_PTS, aptl) == FALSE) {
        lErrorCount += BADERROR("GpiPolySpline", FALSE, NULL);
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolySpline PMERR_OK Test1 - Exit.\n");

    for (i = 0; i < (int)MAX_PTS; i++) {
        if (GpiMarker (hPS, &aptl[i]) == FALSE) {
            lErrorCount += BADERROR("GpiMarker", TRUE, "GpiPolySplineExh");
            return;
        }
    }

    if(!lErrorCount){
       CWRITELOG(L_HDR,0,"$GpiPolySplineExh                                               -    PASS.\n");
    }
    else{
       sprintf(out_str,"$ERRORCOUNT = %d\n",lErrorCount);
       CWRITELOG(L_HDR,0,out_str);
       CWRITELOG(L_HDR,0,"$GpiPolySplineExh                                               -    FAIL.\n");
    }
    CWRITELOG(L_LOTRACE,0,"+ex+ GpiPolySplineExh - Exit.\n");
  }
}

