/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = PROFILE.C
 *
 * DESCRIPTIVE NAME = Profile calls for Postscript Properties.
 *
 *
 * VERSION = V1.0
 *
 * DATE
 *
 * DESCRIPTION : This file contains functions that read and write data to/from
 *               the OS2SYS.INI file.
 *
 *
 * FUNCTIONS :         ReadFormINIData
 *                     ReadTrayPageINIData
 *                     SavePageTrayINIData
 *                     QueryNextKeyString
 *                     SaveINIGroupData
 *                     TerminateTranslationString
 *
 * NOTES
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/
#pragma pack(1)

#define INCL_PM
#define INCL_DOSMEMMGR
#define INCL_GENPLIB_ERROR
#define INCL_GENPLIB_MEMORY
#define INCL_GENPLIB_LAYOUT

#include <os2.h>
#include <string.h>
#include <genplib.h>
#include "inc\config.h"
#include "inc\ppdialog.h"
#include "inc\pspagtun.h"
#include "inc\dlg.h"
#include "inc\profile.h"
#include "inc\uinames.h"





/****************************************************************************\
** DEFINES START                                                            **
\****************************************************************************/
#define PAPER_INI_NEW          "TRAYPAGEMAP"
#define PAPER_INI_OLD          "PAPERTYPE"
#define USERFORM_INI_OLD       "USERFORMS"

#define ERROR_NOT_ENOUGH_MEMORY   -1

#define OLD_INI_EFFECTS_SIZE      14

/****************************************************************************\
** DEFINES END                                                              **
\****************************************************************************/


/****************************************************************************\
** FUNCTION PROTOTYPES START                                                **
\****************************************************************************/
VOID SaveUserForms( PCHAR, PCHAR, PDLGHDR, PUI_BLOCK );
extern PSZ GetDefaultPageSize( PDESPPD, PBYTE );
extern SHORT szDlmCopy( register PSZ, register PSZ, register SHORT );
extern PFORMSTRUCT GetImageableArea( SHORT, PFORMSTRUCT, PDESPPD, PBYTE );
extern INT _Optlink CompareRealNames( PSZ, PSZ );
extern PUI_ENTRY QueryEntryFromOption( PBYTE, PUI_BLOCK, PBYTE, PINT );
/****************************************************************************\
** FUNCTION PROTOTYPES END                                                  **
\****************************************************************************/


/****************************************************************************\
** EXTERNAL DEFINITIONS START                                               **
\****************************************************************************/
extern PVOID   pProcessHeap;                 // memory.c
/****************************************************************************\
** EXTERNAL DEFINITIONS END                                                 **
\****************************************************************************/
CHAR szKeyEffects[] = "EFFECTS";





/******************************************************************************
 *
 * FUNCTION NAME = ReadFormINIData
 *
 * DESCRIPTION
 * Reads the form data from the INI file.  The application name is
 * the device name, and the key name is PAPERTYPE.
 *
 * INPUT
 * pszKeyApp - Device application name.
 * pStringBuffer - Buffer where the data is stored from the INI.
 * iBuffSize - Size, in bytes of pStringBuffer.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE - INI string exists.
 * RETURN-ERROR  - FALSE - INI string does not exist.
 *
 *****************************************************************************/
BOOL ReadFormINIData( PSZ pszKeyApp, PBYTE pStringBuffer, INT iBuffSize )
{
  return( ReadNewOrOldINIString( pszKeyApp, PAPER_INI_NEW, PAPER_INI_OLD,
                                 pStringBuffer, iBuffSize ));
}
/*---------------------------------------------------------------------------*\
* ReadFormINIData End                                                         *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = ReadTrayPageMapINI
 *
 * DESCRIPTION
 * Reads the mapped tray and page data from the OS2SYS.INI file.  The
 * applicaion name is the printer/device name and the key name is "PAPERTYPE"
 * The data is stored in the form:
 *
 *   "TRAYNAME1,FORMNAME1;TRAYNAME2,FORMNAME2;...\0"
 *
 * where "FORMNAME" is the string name of the form (found in the PPD) that is
 * mapped to TRAYNAME (also found in the PPD).  This function parses the
 * INI string data and stores the data in the PSOURCE structure.  If a buffer
 * is already provided (pReturnBuffer), then the INI data is stored in the
 * return buffer.
 * This function also is backward compatible in regard to the INI string.  This
 * function can read previous forms:
 *
 *   "FORMNAME1;FORMNAME2;...;FORMNAMEn"
 *
 * In this format, the form names are mapped to tray names that are organized
 * according to the PPD.  For example, FORMNAME1 maps to the first tray found
 * in the PPD.
 * The reason this function is used when other INI related functions are used
 * is to provide future compatibility.  It is possible for PPDs to re-organize
 * trays, but tray names (sans translation strings) will remain constant.
 * Translation strings are used for display purposes only and can change.
 * This way, no matter how tray names are re-organized, this function will
 * always correctly map the currently selected forms to the trays.
 *
 * INPUT
 * pDlgHdr - Dialog header.  Includes INI application name.
 * pReturnBuffer - Optional, if not NULL, provides a buffer where the tray/page
 *  INI data is returned.
 * uiBufferSize - If pReturnBuffer is not NULL, this contains the number of
 *  bytes in pReturnBuffer.  Otherwise, this is NULL.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - PSOURCE structure pointer.
 * RETURN-ERROR  - NULL
 *
 *****************************************************************************/
BOOL ReadTrayPageMapINI( PDESPPD pdesPPD, PCNFDATA pCNFData )
{
  #define BUFFER_SIZE   4000                  // Size of memory to allocate

  PUI_BLOCK pFormBlock;
  PUI_BLOCK pTrayBlock;
  CHAR      aSubkey[ MAX_PSIZE ];
  PBYTE     pINIBuffer;
  PBYTE     pCurrINI;
  PCHAR     pTransString;
  BOOL      fNextSubkey;
  INT       iSubkeyID;
  INT       ofsTray;
  BOOL      fRC;
  LONG      ofsArray = 0;
  PSOURCE   pSource = pCNFData->u.iv.pSourcePaper;

  /*
  ** Before reading any INI data, first verify if the data already exists.
  ** If so, then there is no need to re-read the data again.
  */
  if (CountPSourceList( pCNFData ) == 0)
  {
    /*
    ** Query the font count.
    */
    PrfQueryProfileData( HINI_SYSTEMPROFILE, (PSZ) pCNFData->szKeyApp,
                         (PSZ) FONTCOUNT_INI, (PLONG) &pCNFData->lFontCount,
                         (PULONG) &ofsTray );

    /*
    ** Query the form and tray blocks, respectively.
    */
    pFormBlock = QueryBlockFromKeyword( &pdesPPD->stUIList,
                                        pdesPPD->pPSStringBuff,
                                        UINAME_PAGESIZE, NULL );
    pTrayBlock = QueryBlockFromKeyword( &pdesPPD->stUIList,
                                        pdesPPD->pPSStringBuff,
                                        UINAME_INPUTSLOT, NULL );

    if (pFormBlock != NULL && pTrayBlock != NULL)
    {
      if ((pINIBuffer = (PCHAR) GplMemoryAlloc( (HMCB) 0, BUFFER_SIZE ))
           != NULL)
      {
        /*
        ** Read the INI data.
        */
        fRC = ReadFormINIData( pCNFData->szKeyApp, pINIBuffer, BUFFER_SIZE );

        /*
        ** If reading the old INI, then mark it, it may be needed.
        */
        if (fRC == READ_OLD_INI_DATA)
        {
          pCNFData->ulFlags |= OLD_INI_USED;
        }
      }
      else
      {
        fRC = FALSE;
        GplErrSetError( ERROR_NOT_ENOUGH_MEMORY );
      }

      if (fRC != NO_INI_READ)
      {
        pCurrINI = pINIBuffer;

        /*
        ** With INI string in hand, parse the string into sub-strings,
        ** and map all relavent data.
        */
        do
        {
          iSubkeyID = 0;
          ofsTray = -1;

          /*
          ** This loop runs through each INI string group:
          ** TRAY,FORM;
          ** The tray is always the first INI string.
          ** The form is always the second INI string.
          */
          do
          {
            fNextSubkey = QueryNextSubkey( &pCurrINI, (PSZ) aSubkey,
                                           MAX_PSIZE );

            switch (iSubkeyID)
            {
            // The tray is the first subkey.
            case 0:
                 /*
                 ** For the current INI, the string is a tray name.  For the
                 ** old INI, the string is a form name, so fall through this
                 ** case to the next one (1 below).
                 */
                 if (fRC == CURR_INI_READ)
                 {
                   QueryEntryFromOption( pdesPPD->pPSStringBuff, pTrayBlock,
                                         aSubkey, &ofsTray );
                   break;
                 }
                 else
                 {
                   ofsTray = ofsArray;

                   /*
                   ** For the old INI format, if no form was selected for a
                   ** given tray, "None" could be stored in that tray.  For
                   ** the new format, no form is stored for the tray.
                   ** For the new format, remove any references to "None".
                   */
                   if (!strcmp( aSubkey, "None" ))
                   {
                     aSubkey[ 0 ] = 0;
                   }
                   /*
                   ** The old INI format stored the translation string as
                   ** well as the current key name.  Remove the T.S. for the
                   ** new format.
                   */
                   else if ((pTransString = strchr( aSubkey, '/' )) != NULL)
                   {
                     *pTransString = 0;
                   }
                 }

            // The form is the second subkey.
            case 1:
                 /*
                 ** A tray must exist (> -1) in order to assign a form to
                 ** it.
                 */
                 if (ofsTray > -1 && aSubkey[ 0 ] != 0)
                 {
                   strcpy( pSource->szPaperName[ ofsTray ], aSubkey );
                 }
            }

            iSubkeyID++;
          } while (fNextSubkey == TRUE );

          ofsArray++;

        } while (*pCurrINI != 0);
      }
      else
      {
        /*
        ** No INI exists, so read the default page and tray and associate the
        ** both of them.
        */
        pCurrINI = pdesPPD->pPSStringBuff +
                   pTrayBlock->uiEntry[ pTrayBlock->usDefaultEntry ].ofsOption;
        QueryEntryFromOption( pdesPPD->pPSStringBuff, pTrayBlock,
                              pCurrINI, &ofsTray );

        // @V4.0168335
//        pCurrINI = pdesPPD->pPSStringBuff +
//                   pFormBlock->uiEntry[ pFormBlock->usDefaultEntry ].ofsOption;
        pCurrINI = (PBYTE) GetDefaultPageSize( pdesPPD,
                                               pdesPPD->pPSStringBuff );
        strcpy( pCNFData->u.iv.pSourcePaper->szPaperName[ ofsTray ],
                pCurrINI );
      }

      /*
      ** If the old INI was read, then save the data under the current INI
      ** (see header).
      */
      if (fRC == OLD_INI_READ)
      {
        *pINIBuffer = 0;

        for (ofsTray = 0 ; ofsTray < INPBINS ; ofsTray++)
        {
          if (pSource->szPaperName[ ofsTray ][ 0 ] != 0)
          {
            pCurrINI = (PBYTE) (pdesPPD->pPSStringBuff +
                                pTrayBlock->uiEntry[ ofsTray ].ofsOption);
            SaveINIGroupData( pINIBuffer, pCurrINI );
            SaveINIGroupData( pINIBuffer, pSource->szPaperName[ ofsTray ] );
            SaveINIGroupData( pINIBuffer, NULL );
          }
        }

        PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pCNFData->szKeyApp,
                             (PSZ) PAPER_INI_NEW, (PVOID) pINIBuffer,
                             strlen( pINIBuffer ) + 1 );
      }

      if (pINIBuffer != NULL)
      {
        GplMemoryFree( pINIBuffer );
      }
    }
    else
    {
      fRC = TRUE;
    }
  }
  else
  {
    fRC = TRUE;
  }

  if (pCNFData->jobProperties.szFormName[ 0 ] == 0)
  {
    PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pCNFData->szKeyApp,
                           PAPERSOURCE_INI, NULL,
                           (PVOID) pCNFData->jobProperties.szFormName,
                           (LONG) sizeof( pCNFData->jobProperties.
                                  szFormName ) );
  }

  return( fRC );
}
/*---------------------------------------------------------------------------*\
* ReadTrayPageINIData End                                                     *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = CountPSourceList
 *
 * DESCRIPTION
 * Counts the number of forms that are in the PSOURCE->szPaperName array.
 * Each element in the array represents a tray and each string in an array (if
 * any) represents the form that is associated with it.
 * This function returns the number of form strings that are found in the
 * array list.
 *
 * INPUT
 * pCNFData - Current properties settings.  This includes the tray/form
 *  map settings in PSOURCE->szPaperName.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - Number of forms associated with trays.
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
LONG CountPSourceList( PCNFDATA pCNFData )
{
  LONG    lIndex;
  LONG    lCount = 0;
  PSOURCE pSource = pCNFData->u.iv.pSourcePaper;

  /*
  ** Run through the list and count the number of form/tray mappings.
  */
  for (lIndex = 0 ; lIndex < INPBINS ; lIndex++)
  {
    /*
    ** If a string exists, increment the count.
    */
    if (pSource->szPaperName[ lIndex ][ 0 ] != 0)
    {
      lCount++;
    }
  }

  return( lCount );
}
/*---------------------------------------------------------------------------*\
* CountPSourceList End                                                        *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = LoadUserForms
 *
 * DESCRIPTION
 * Reads OS2SYS.INI for a list of user-defined forms and their associated
 * real forms.  The data is read from the INI and stored in a UNQFRM structure,
 * which is used throughout Printer/Job Properties.  The user-defined form
 * name is a name generated by the user.  The real form name is the printer
 * defined form name that the user-defined form is mapped.
 *
 * INPUT
 * pDlgHdr - Dialog header.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - Pointer to a form UI block.
 * RETURN-ERROR  - NULL
 *
 *****************************************************************************/
PUI_BLOCK LoadUserForms( PDLGHDR pDlgHdr )
{
  #define BUFFER_SIZE      4000

  PBYTE     pINIBuffer;
  PBYTE     pCurrINI;
  INT       ofsFormEntry;
  CHAR      aUserFormStr[ MAX_PSIZE ];
  CHAR      aRealFormStr[ MAX_PSIZE ];
  PUI_BLOCK pFormBlock;
  PSZ       pKeyApp = (PSZ) pDlgHdr->pCNFData->szKeyApp;
  PUNQFRM   pUniqueForm = pDlgHdr->stUnqFrmLst;
  INT       iRC;
  PCHAR     pSlash;

  pFormBlock = QueryBlockFromKeyword( &pDlgHdr->pdesPPD->stUIList,
                                      pDlgHdr->pItemsBuff, UINAME_PAGESIZE,
                                      NULL );

  if ((pINIBuffer = (PBYTE) GplMemoryAlloc( (HMCB) 0, BUFFER_SIZE )) != NULL)
  {
    /*
    ** Read the user-defined INI entry.
    */
    if ( ( iRC = ReadNewOrOldINIString( pKeyApp, USERFORM_INI_NEW,
                 USERFORM_INI_OLD, pINIBuffer, BUFFER_SIZE ) ) != NO_INI_READ )
    {
      /*
      ** Query the form block.
      */
      if (pFormBlock != NULL)
      {
        pCurrINI = pINIBuffer;

        while (*pCurrINI != 0)
        {
          /*
          ** Query the INI, first for the user-defined string, then for the
          ** read form name.
          */
          QueryNextSubkey( &pCurrINI, aUserFormStr, MAX_PSIZE );
          QueryNextSubkey( &pCurrINI, aRealFormStr, MAX_PSIZE );

          if ( iRC == OLD_INI_READ )
          {
            if ( ( pSlash = strchr( aRealFormStr, '/' ) ) != NULL )
            {
              *pSlash = 0;
            }

            /* Since old format was UserFormName1,RealFormName1;;UFN2,RFN2;;
            ** bump past extra ; so will work
            */
            if ( *pCurrINI == ';' )
            {
              pCurrINI++;
            }
          }

          /*
          ** With the given real form name, query the form block to see if
          ** that form matches with what is in the block.  If so, copy
          ** the data.
          */
          if (QueryEntryFromOption( pDlgHdr->pItemsBuff, pFormBlock,
                                    aRealFormStr, &ofsFormEntry ) != NULL &&
              aUserFormStr[ 0 ] != 0)
          {
            strcpy( pUniqueForm->aFormName, aUserFormStr );
            pUniqueForm->sPrtHndlMap = (SHORT) ofsFormEntry;
            pUniqueForm++;
          }
        }
      }
    }

    GplMemoryFree( pINIBuffer );
  }

  return ( pFormBlock );
}
/*---------------------------------------------------------------------------*\
* LoadUserForms End                                                           *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = SavePageTrayINIData
 *
 * DESCRIPTION
 *
 * Saves the form-tray mapping from Printer Properties under two INI keys:
 * PAPER_INI_NEW and PAPER_INI_OLD.  This is so that if the system copies the
 * INI files to an older driver, the current form-tray selection will still
 * be valid.
 *
 * For PAPER_INI_NEW, the format is as follows:
 *
 *   "TRAYNAME1,FORMNAME1;TRAYNAME2,FORMNAME2;...\0"
 *
 * where "FORMNAME" is the string name of the form (found in the PPD) that is
 * mapped to TRAYNAME (also found in the PPD).  This form-tray mapping is
 * set by the user in Printer Properties.  This is saved under the key string
 * "PAPERTYPE".
 *
 * For PAPER_INI_OLD, the format is as follows:
 *
 *   "FORMNAME1;FORMNAME2;;FORMNAME4;...\0"
 *
 * where FORMNAME is the string name that is found in the PPD.  Each semicolon
 * represents a tray, so FORMNAME1 goes in the first tray, while FORMNAME4
 * goes in the fourth tray (the third tray has not been assigned a form, that is
 *     there is no FORMNAME3).
 *
 * INPUT
 * hDlg - Dialog handle.
 * pDlgHdr - Dialog header structure.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID SavePageTrayINIData( HWND hDlg, PDLGHDR pDlgHdr )
{
  #define BUFFER_SIZE    4000

  PUI_BLOCK pTrayBlock;
  PUI_BLOCK pFormBlock;
  PBYTE     pNewINIString;
  PBYTE     pOldINIString;
  PSZ       pString;
  LONG      lLoop;
  LONG      ofsString;
  PTMENTRY  pTMEntry = (PTMENTRY) pDlgHdr->pPtr;

  /*
  ** Allocate a temporary buffer for the new INI string.
  */
  if ((pNewINIString = GplMemoryAlloc( (HMCB) 0, BUFFER_SIZE )) != NULL)
  {
    /*
    ** Since the INI buffer is        large, rather than allocating more
    ** memory for the old buffer, divide the new buffer in half and assign
    ** the old pointer to the second half.
    */
    pOldINIString = pNewINIString + (ULONG) (BUFFER_SIZE / 2);
    *pNewINIString = *pOldINIString = 0;

    /*
    ** Query the list of available trays and forms.
    */
    pTrayBlock = QueryBlockFromKeyword( &pDlgHdr->pdesPPD->stUIList,
                                        pDlgHdr->pItemsBuff, UINAME_INPUTSLOT,
                                        NULL );
    pFormBlock = QueryBlockFromKeyword( &pDlgHdr->pdesPPD->stUIList,
                                        pDlgHdr->pItemsBuff, UINAME_PAGESIZE,
                                        NULL );

    /*
    ** This loop will run through the list of available trays and forms and
    ** store whatever forms were assigned to any trays.
    */
    for (lLoop = 0 ; lLoop < (INT) pDlgHdr->usNumTMEntries ; lLoop++)
    {
      // NOFORMS_HANDLE means that no forms were assigned to the current tray.
      if (pTMEntry->lFormID != NOFORMS_HANDLE 
          // Check for constraints (if tray is not installed, we don't want to save   //@V1.0VV04
          // it's form settings).  If this is not called, in job properties dialog    //@V1.0VV04
          // trays that are not actually installed are available (undesirable!)       //@V1.0VV04
          && QueryCnstListBlockList( pDlgHdr, pTrayBlock, lLoop,                      //@V1.0VV04
              QUERY_INST_OPT_GROUP, BLOCK_SEARCH_ALL) == FALSE                        //@V1.0VV04
         )
      {
        /*
        ** Get the current tray name and save it under the new INI format.
        */
        pString = (PSZ) (pDlgHdr->pItemsBuff +
                         pTrayBlock->uiEntry[ lLoop ].ofsOption);
        SaveINIGroupData( pNewINIString, pString );

        // IS_UNQ_FORMID means that the form is a user-defined form.
        if (!IS_UNQ_FORMID( pTMEntry->lFormID ))
        {
          pString = (PSZ) (pDlgHdr->pItemsBuff + pFormBlock->uiEntry
                           [ pTMEntry->lFormID ].ofsOption);
          SaveINIGroupData( pNewINIString, pString );
          SaveINIGroupData( pNewINIString, NULL );
        }
        // Query the printer-defined form name.
        else if (pTMEntry->lFormID != NOFORMS_HANDLE)
        {
          pTMEntry->lFormID = CONV_LBID_TO_UNQID( pTMEntry->lFormID );
          pString = (PSZ) (pDlgHdr->stUnqFrmLst[ pTMEntry->lFormID ].
                           aFormName);
          SaveINIGroupData( pNewINIString, pString );
          SaveINIGroupData( pNewINIString, NULL );
        }
        else
        {
          pString = NULL;
        }
      }
      else
      {
        pString = NULL;
      }

      /*
      ** For the old INI, do not store the tray name, store the form name
      ** only.  Remember, that the number of form names in the INI cannot
      ** exceed INPBINS (the total number of trays supported in the old
      ** driver).
      */
      if (lLoop < INPBINS)
      {
        // Store the form name (if one exists).
        if (pString != NULL)
        {
          /*
          ** We can't use SaveINIGroupData() for two reasons:
          ** 1) The function truncates all translation strings.  We need to
          **    save the T.S.
          ** 2) Every time SaveINIGroupData() is called, a comma is
          **    automatically added prior to the string to indicate a
          **    substring.  Commas are not allowed in the old format.
          ** GetOldFormName() does convert current strings to the old format,
          ** so call it and append the old string manually to the old INI
          ** string.
          */
          ofsString = strlen( pOldINIString );
          GetOldFormName( pDlgHdr->pdesPPD, (PSZ) pString,
                          (PSZ) (pOldINIString + ofsString) );
        }

        // If NULL is provided, the function inserts a semicolon.
        SaveINIGroupData( pOldINIString, NULL );
      }

      pTMEntry++;
    }

    /*
    ** Write the new form-tray mapping string to the INI.
    */
    PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pDlgHdr->pCNFData->szKeyApp,
                         (PSZ) PAPER_INI_NEW, (PVOID) pNewINIString,
                         strlen( pNewINIString ) + 1 );

    /*
    ** Write the old form-tray mapping string to the INI.
    */
    PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pDlgHdr->pCNFData->szKeyApp,
                         (PSZ) PAPER_INI_OLD, (PVOID) pOldINIString,
                         strlen( pOldINIString ) + 1 );

    // Save all user-defined forms.
    SaveUserForms( pNewINIString, pOldINIString, pDlgHdr, pFormBlock );

    /*
    ** Save the current form setting.
    */
    PrfWriteProfileString( HINI_SYSTEMPROFILE,
                           (PSZ) pDlgHdr->pCNFData->szKeyApp,
                           PAPERSOURCE_INI,
                           (PSZ) pDlgHdr->pCNFData->jobProperties.szFormName );

    /*
    ** Write the font count.
    */
    PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pDlgHdr->pCNFData->szKeyApp,
                         (PSZ) FONTCOUNT_INI,
                         (PVOID) &pDlgHdr->pCNFData->lFontCount,
                         sizeof( LONG ) );

    GplMemoryFree( pNewINIString );
  }
  else
  {
    GplErrSetError( ERROR_NOT_ENOUGH_MEMORY );
  }
}
/*---------------------------------------------------------------------------*\
* SavePageTrayINIData End                                                     *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = SaveUserForms
 *
 * DESCRIPTION
 * Saves all user-defined forms under two INI keys:  USERFORM_INI_NEW and
 * USERFORM_INI_OLD.  This is to provide backward compatibility with older
 * drivers that use USERFORM_INI_OLD.
 *
 * For USERFORM_INI_NEW, the format is as follows:
 *
 *    "USERFORMNAME1,REALFORMNAME1;USERFORMNAME2,REALFORMNAME2;...\0"
 *
 * where USERFORMNAME is the form name the user defined and REALFORMNAME is
 * the printer-defined form name that is associated with it.
 *
 * For USERFORM_INI_OLD, the format is:
 *
 *    "USERFORMNAME1;REALFORMNAME1;;USERFORMNAME2;REALFORMNAME2;;...\0"
 *
 * INPUT
 * hDlg - Dialog handle.
 * pDlgHdr - Dialog header structure.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID SaveUserForms( PCHAR pNewINIBuff, PCHAR pOldINIBuff, PDLGHDR pDlgHdr,
                    PUI_BLOCK pFormBlock )
{
  LONG    lLoop;
  PSZ     pRealFormName;
  LONG    ofsString;
  PUNQFRM pUniqueForm = pDlgHdr->stUnqFrmLst;

  // Initialize both buffers.
  *pNewINIBuff = *pOldINIBuff = 0;

  /*
  ** This loop runs through the list of user-defined forms and saves them in
  ** the INI; both in the new and old format.
  */
  for (lLoop = 0 ; lLoop < NUM_OF_UNQ_FORMS ; lLoop++)
  {
    /*
    ** User-defined form?
    */
    if (pUniqueForm->aFormName[ 0 ] != 0 &&
        (pUniqueForm->sFlag & UNQ_FORM_DELETE) == 0)
    {
      // New INI - save user-defined form name first.
      SaveINIGroupData( pNewINIBuff, pUniqueForm->aFormName );

      /*
      ** Old INI - save user-defined form name first and add a semicolon
      ** (use NULL).
      */
      strcat( pOldINIBuff, pUniqueForm->aFormName );
      SaveINIGroupData( pOldINIBuff, NULL );

      // Query the matching real form name.
      pRealFormName = (PSZ) (pDlgHdr->pItemsBuff + pFormBlock->uiEntry
                             [ pUniqueForm->sPrtHndlMap ].ofsOption);

      /*
      ** New INI - save real form name and terminate substring (by inserting
      ** a semicolon).
      */
      SaveINIGroupData( pNewINIBuff, pRealFormName );
      SaveINIGroupData( pNewINIBuff, NULL );

      /*
      ** Old INI - save real form name and insert two semicolons to indicate
      ** the end of the substring.  Remember to include the translation
      ** string.
      */
      ofsString = strlen( pOldINIBuff );
      GetOldFormName( pDlgHdr->pdesPPD, pRealFormName,
                      (PSZ) (pOldINIBuff + ofsString) );
      SaveINIGroupData( pOldINIBuff, NULL );
      SaveINIGroupData( pOldINIBuff, NULL );

    }

    pUniqueForm++;
  }

  // Save new INI string.
  PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pDlgHdr->pCNFData->szKeyApp,
                       (PSZ) USERFORM_INI_NEW, (PVOID) pNewINIBuff,
                       strlen( pNewINIBuff ) + 1 );

  // Save old INI string.
  PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pDlgHdr->pCNFData->szKeyApp,
                       (PSZ) USERFORM_INI_OLD, (PVOID) pOldINIBuff,
                       strlen( pOldINIBuff ) + 1 );
}
/*---------------------------------------------------------------------------*\
* SaveUserForms End                                                           *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = SaveINIGroupData
 *
 * DESCRIPTION
 * Appends a given string to a buffer in the following format:
 *
 *     STRING11,STRING12,...STRING1N;STRING21,STRING22,...,STRING2N;...
 *
 * If a string is provided, a comma is first appended to the end of the buffer
 * to separate the new string.  The string is then appended to the buffer.  If
 * no string is provided, then the semicolon is appended to the string,
 * thereby separating the groups of strings from other groups.
 * The above string is NULL-terminated.
 *
 * INPUT
 * pINIString - Buffer that contains groups of strings.  This is where the
 *   string will be stored in the above format.
 * pStringToAppend - Contains the string to append in pINIString.  If this is
 *   NULL, then a semicolon is appended to pINIString.
 *
 * OUTPUT
 * pINIString is appended with a new string, or with a semicolon.
 *
 * RETURN-NORMAL - None.
 * RETURN-ERROR  - None.
 *
 *****************************************************************************/
VOID SaveINIGroupData( PBYTE pINIString, PBYTE pStringToAppend )
{
  INT iStringLen;

  if (pStringToAppend != NULL)
  {
    iStringLen = strlen( pINIString );
    if (iStringLen != 0 && *(pINIString + iStringLen - 1) != ';')
    {
      strcat( pINIString, "," );
    }
    strcat( pINIString, pStringToAppend );
    TerminateTranslationString( pINIString );
  }
  else
  {
    strcat( pINIString, ";" );
  }
}
/*---------------------------------------------------------------------------*\
* SaveINIGroupData End                                                        *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = TerminateTranslationString
 *
 * DESCRIPTION
 * Searches for a slash in a given string.  If a slash is found, then a
 * translation string is to follow.  For this case, the slash is replaced with
 * a 0-terminator.
 *
 * INPUT
 * pString - String to query for a slash.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - Returns a pointer to the slash, if one is found.
 * RETURN-ERROR  - No slash is found.
 *
 *****************************************************************************/
PBYTE TerminateTranslationString( PBYTE pString )
{
  PBYTE pTransString = NULL;

  if (pTransString = strchr( pString, '/' ))
  {
    *pTransString = 0;
  }

  return( pTransString );
}
/*---------------------------------------------------------------------------*\
* TerminateTranslationString End                                              *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = ReadNewOrOldINIString
 *
 * DESCRIPTION
 * Reads an INI string from the OS2SYS.INI file.  The application name is the
 * printer name.  The caller must provide the current key name for the
 * application and (optionally) may provide an older key name.
 * The current key profile is read.  If the key doesn't exist and the caller
 * provided an older key name, the older key name is read.  If an older key
 * name is not provided (NULL), then the function returns.
 * The reason for being able to read older INI key data is to provide
 * compatiblity with older driver INI strings.  Older INI strings contain
 * translation strings which, officially, should not be included in the INI
 * since translation strings can be changed at any time.  This also provides
 * backward compatibility so that an older driver can be used (and hence, the
 * older INI key can be accessed) if the user decides to go back to an older
 * version.  The older INI data is not modified.
 *
 * INPUT
 * pszAppName - INI application name.  This cannot be NULL.
 * pszCurrKeyName - INI key name that is used by the current driver.  This
 *  cannot be NULL.
 * pszOldKeyName - INI key name that was used by an older version of the
 *  driver.  This can be NULL.
 * iBuffSize - Totoal byte size of pStringBuffer.
 *
 * OUTPUT
 * pStringBuffer - If an INI string exists, the INI string is stored here.
 *  The max size is provided by iBuffSize.
 *
 * RETURN-NORMAL - CURR_INI_READ - The current INI string was read.
 *                 OLD_INI_READ - The current INI string could not be accessed
 *                   but there was an older INI string and that was read.
 * RETURN-ERROR  - NO_INI_READ - No INI data was read.  Either no INI strings
 *                   (current or earlier) existed or not enough INI data was
 *                   provided.
 *
 *****************************************************************************/
INT ReadNewOrOldINIString( PSZ pszAppName, PSZ pszCurrKeyName,
                           PSZ pszOldKeyName, PBYTE pStringBuffer,
                           INT iBuffSize )
{
  LONG  lINIData;
  ULONG ulNewINIRead;
  INT   iRC = NO_INI_READ;             // Function return code

  /*
  ** At minimum, an application name and current key name must be provided.
  */
  if (pszAppName != NULL && pszCurrKeyName != NULL)
  {
    lINIData = QueryINIKeyToRead( pszAppName );

    if (lINIData == READ_NEW_INI_DATA &&
        (ulNewINIRead = PrfQueryProfileString( HINI_SYSTEMPROFILE,
                                               (PSZ) pszAppName,
                                               pszCurrKeyName, NULL,
                                               (PVOID) pStringBuffer,
                                               (LONG) iBuffSize )) != 0)
    {
      iRC = CURR_INI_READ;
    }

    if (pszOldKeyName != NULL &&
        (lINIData == READ_OLD_INI_DATA || ulNewINIRead == 0))
    {
      /*
      ** The current key could not be found.  Now, try to read the old
      ** key name from the INI.  If that could not be found, return the
      ** NO_INI_READ code.
      */
      if (PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pszAppName,
                                 pszOldKeyName, NULL, (PVOID) pStringBuffer,
                                 (LONG) iBuffSize ) != 0)
      {
        iRC = OLD_INI_READ;
      }
    }
  }

  return( iRC );
}
/*---------------------------------------------------------------------------*\
* ReadNewOrOldINIString End                                                   *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = QueryNextSubkey
 *
 * DESCRIPTION
 * Reads the following INI string format:
 *
 *   "Mainkey1,Subkey1a,Subkey1b,...;Mainkey2,Subkey2a,Subkey2b,...;
 *
 * This function reads the next available string in the above format up to,
 * but not including, the semicolon or comma.  The string is copied to another
 * buffer.  The pointer to the above string format is incremented to point to
 * the next available string.
 * The function returns a code, indicating if the end of a subkey is found.
 * This is true if a semicolon is found.  If a comma is found, then the
 * substring still exists.
 *
 * INPUT
 * ppInputString - Double pointer to a string in the above format.
 * iRSLength - Size of buffer where string is to be stored.  If pInputString
 *  is NULL, this value is ignored.
 *
 * OUTPUT
 * pReturnString - Buffer where string is stored.  If this is NULL, then
 *  ppInputString is incremented over the next string.
 *
 * RETURN-NORMAL - TRUE - More substrings exit.
 *                 FALSE - End of the substring encountered.
 * RETURN-ERROR  - None.
 *
 *****************************************************************************/
BOOL QueryNextSubkey( PSZ *ppInputString, PSZ pReturnString,
                      INT iRSLength )
{
  BOOL bRC = FALSE;             // Return code

  /*
  ** If pReturnString is NULL, then increment the string length size to a
  ** large value.  The string length size is decremented as the data is copied.
  */
  if (pReturnString == NULL)
  {
    iRSLength = 0xFFFF;
  }
  else
  {
    *pReturnString = 0;
  }

  /*
  ** Copy the string, one character at a time, until a delimiter is found.
  */
  while (**ppInputString != ',' && **ppInputString != ';' &&
         **ppInputString != 0 && iRSLength-- > 0)
  {
    /*
    ** Copy the character from the source to destination buffer, provided
    ** that a destination buffer exists.
    */
    if (pReturnString != NULL)
    {
      *(pReturnString++) = **ppInputString;
    }
    (*ppInputString)++;
  }

  if (pReturnString != NULL)
  {
    *pReturnString = 0;
  }

  /*
  ** If a substring delimiter is found (either a semicolon or 0), then
  ** set the return code to TRUE.
  */
  if (**ppInputString != 0 && *((*ppInputString)++) != ';')
  {
    bRC = TRUE;
  }

  return( bRC );
}
/*---------------------------------------------------------------------------*\
* QueryNextSubkey End                                                         *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = ReadPrtPropOptionsINI
 *
 * DESCRIPTION
 * Reads from the PP_OTHER_FEATURES ("PP_OTHERFEATURES") key from OS2SYS.INI.
 * This key contains TRUE/FALSE type values for other additional features for
 * Printer Properties.
 *
 * Each value in this key makes up a DOUBLEWORD group.  The first WORD in the
 * group contains a INISUBKEY_... ID (found in PROFILE.H) that maps to the
 * specific group to query.  The second WORD is either TRUE or FALSE.
 *
 * For Mode Switch only, this function reads the old INI data first (the
 * "EFFECTS" key), if the old INI data is the latest.
 *
 * INPUT
 * pKeyApp - The application name for the desired printer in the INI.
 * usINISubkeyID - This is a "INISUBKEY_" ID that is found in PROFILE.H.
 *
 * OUTPUT
 * pfRetVal - Returns the TRUE/FALSE value.
 *
 * RETURN-NORMAL - TRUE - INI substring exists.
 * RETURN-ERROR  - FALSE - INI substring or INI app doesn't exit.
 *
 *****************************************************************************/
BOOL ReadPrtPropOptionsINI( PSZ pKeyApp, USHORT usINISubkeyID,
                            PBOOL pfRetVal )
{
  USHORT ausINIBuff[ MAX_PSIZE ];    // Buffer with INI data groups
  PSZ    pINI;
  ULONG  ulDataSize;                 // Byte size of INI groups
  ULONG  ulLoop;                     // Loop counter
  BOOL   fRC = FALSE;                // Return code

  if (pfRetVal != NULL && pKeyApp != NULL && usINISubkeyID != 0)
  {
    *pfRetVal = ausINIBuff[ 0 ] = 0;

    /*
    ** The old driver stored the mode switch value in "Effects".  We need
    ** to read and parse this key to query the current modeswitch value
    ** from the old INI.
    */
    if (usINISubkeyID == INISUBKEY_MODESWITCH &&
        QueryINIKeyToRead( pKeyApp ) == READ_OLD_INI_DATA)
    {
      PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pKeyApp, szKeyEffects,
                             NULL, (PVOID) ausINIBuff, sizeof( ausINIBuff ) );
      fRC = TRUE;
    }

    if (ausINIBuff[ 0 ] == 0)
    {
      /*
      ** The INI size is recorded in bytes.  Therefore, query the INI size.
      ** If the size is not 0 and it is and the size is an even value (each
      ** group makes up a doubleword value), then everything is OK.
      */
      ulDataSize = sizeof( ausINIBuff );
      if (PrfQueryProfileData( HINI_SYSTEMPROFILE, pKeyApp, PP_OTHER_FEATURES,
                               (PVOID) ausINIBuff, &ulDataSize ) == TRUE &&
          (ulDataSize % 2) == 0)
      {
        /*
        ** Run through the list and find a matching ID.  The ID is found in the
        ** first word of the group.
        */
        for (ulLoop = 0 ; ulLoop < ulDataSize / sizeof( USHORT ) ;
             ulLoop += 2)
        {
          if (ausINIBuff[ ulLoop ] == usINISubkeyID)
          {
            *pfRetVal = (BOOL) ausINIBuff[ ulLoop + 1 ];
            fRC = TRUE;
            break;
          }
        }
      }
      else
      {
        PrfQueryProfileData( HINI_SYSTEMPROFILE, pKeyApp, PP_OTHER_FEATURES,
                             NULL, 0 );
      }
    }
    else                    // READ_OLD_INI_DATA
    {
      /*
      ** The old Modeswitch data is kept in "Effects".  Read that string
      ** and parse.
      */
      pINI = (PSZ) ausINIBuff;

      // Modeswitch is located at offset 9.
      for (ulLoop = 0 ; ulLoop < 9 ; ulLoop++)
      {
        QueryNextSubkey( &pINI, NULL, 0 );
      }

      *pfRetVal = (BOOL) (*pINI - '0');
    }
  }

  return( fRC );
}
/*---------------------------------------------------------------------------*\
* ReadPrtPropOptionsINI End                                                   *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = SavePrtPropOptionsINI
 *
 * DESCRIPTION
 * Saves TRUE/FALSE type values from Printer Properties into OS2SYS.INI.
 * The application for this INI is PP_OTHER_FEATURES ("PP_OTHERFEATURES").
 * The key for this INI is broken down into doubleword groups, with the first
 * word being a unique ID and the second word being 1 (TRUE) or 0 (FALSE).
 * With a given ID, the function looks throught the list of entries taken
 * from the INI.  If a matching ID is found, then the value is updated,
 * otherwise, the new ID and matching value are appended.  In either case,
 * the INI data is written back.
 *
 * For Mode Switching, the old driver stored the value in the "EFFECTS" key,
 * and it is the 9th field (each field is separated by semicolons).
 * This function stores the mode switching value in the old field.  It is
 * either "0", or "1" ASCII.
 *
 * INPUT
 * pKeyApp - The application name for the desired printer in the INI.
 * usINISubkeyID - This is a "INISUBKEY_" ID that is found in PROFILE.H.
 * fSetFlag - Either TRUE or FALSE.  If TRUE, the 1 is written, otherwise,
 *   0 is written.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None.
 * RETURN-ERROR  - None.
 *
 *****************************************************************************/
VOID SavePrtPropOptionsINI( PSZ pKeyApp, USHORT usINISubkeyID, BOOL fSetFlag )
{
  USHORT ausINIBuff[ MAX_PSIZE ];   // INI data buffer
  PSZ    pINI;
  ULONG  ulDataSize;                // Size of INI group data
  ULONG  ulLoop = 0;                // Loop counter

  if (pKeyApp != NULL && usINISubkeyID != 0)
  {
    /*
    ** We need to store the mode switch in the old INI area, which is
    ** "Effects".  The problem is that this field is part of several other
    ** fields.  Modeswitch is located at offset 10 and is either "0", or
    ** "1" (ASCII).
    */
    if (usINISubkeyID == INISUBKEY_MODESWITCH &&
        QueryINIKeyToRead( pKeyApp ) == READ_OLD_INI_DATA)
    {
      pINI = (PSZ) ausINIBuff;

      /*
      ** Retrieve the old INI data.
      */
      PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pKeyApp, szKeyEffects,
                             NULL, (PVOID) ausINIBuff, sizeof( ausINIBuff ) );

      /*
      ** If the old INI data exists, then increment the array counter past
      ** the first eight fields.  The ninth field will be the modeswitch.
      ** Some values may be of variable length, but there are always
      ** semicolons after each field, so count each semicolon.
      */
      if (ausINIBuff[ 0 ] != 0)
      {
        ulLoop = 0;

        // Modeswitch is located at offset 9.
        while (ulLoop < 9)
        {
          QueryNextSubkey( &pINI, NULL, 0 );
          ulLoop++;
        }

        *pINI = (CHAR) (fSetFlag + '0');

        PrfWriteProfileString( HINI_SYSTEMPROFILE, (PSZ) pKeyApp,
                               szKeyEffects, (PSZ) ausINIBuff );
      }
    }

    ausINIBuff[ 0 ] = 0;

    /*
    ** Query the INI data and store it in the INI buffer.
    */
    ulDataSize = sizeof( ausINIBuff );
    if (PrfQueryProfileData( HINI_SYSTEMPROFILE, pKeyApp,
                               PP_OTHER_FEATURES, (PVOID) ausINIBuff,
                               &ulDataSize ) == TRUE)
    {
      /*
      ** Run through the array to see if the ID is already in the buffer.
      */
      for (ulLoop = 0 ; ulLoop < ulDataSize / 2 ; ulLoop += 2)
      {
        if (ausINIBuff[ ulLoop ] == usINISubkeyID)
        {
          break;
        }
      }
    }

    /*
    ** The INI ID may or may not be in the buffer.  If it is, then the FOR
    ** loop will break when the ID is found.  If not, then append the ID and
    ** TRUE/FALSE binary value.
    */
    ausINIBuff[ ulLoop ] = usINISubkeyID;
    ausINIBuff[ ulLoop + 1 ] = (USHORT) fSetFlag;

    PrfWriteProfileData( HINI_SYSTEMPROFILE, pKeyApp, PP_OTHER_FEATURES,
                         NULL, 0 );

    PrfWriteProfileData( HINI_SYSTEMPROFILE, pKeyApp, PP_OTHER_FEATURES,
                         (PVOID) ausINIBuff, (ulLoop + 2) * sizeof( USHORT ) );

  }
}
/*---------------------------------------------------------------------------*\
* SavePrtPropOptionsINI End                                                   *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = QueryINIKeyToRead
 *
 * DESCRIPTION
 * Reads the INI data from the "Effects" INI application in OS2SYS.INI. and
 * counts the number of fields, or subkeys, that are found in this INI string.
 * Since each field is separated by a semicolon, use QueryNextSubkey() to count
 * the number of subkeys in the string.
 * This count will continue until the end of the buffer.
 *
 * Both the old driver (Warp and earlier), as well as the new driver (Merlin and
 * later), both uses the "EFFECTS" key in OS2SYS.INI.  However, the old driver
 * contains no more than OLD_INI_EFFECTS_SIZE fields, while the new driver
 * will contain more.  Therefore, if the number of current fields exceeds
 * OLD_INI_EFFECTS_SIZE, then the "EFFECTS" INI is being used by the new driver.
 *
 * INPUT
 * pszAppName - INI application name.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - READ_NEW_INI_DATA (default).  INI key has been modified by
 *                 the new driver.
 *                 READ_OLD_INI_DATA. INI key has been modified by the old
 *                 driver.
 * RETURN-ERROR  - None.
 *
 *****************************************************************************/
LONG QueryINIKeyToRead( PSZ pszAppName
                      )
{
  CHAR aINIString[ CCHMAXPATH ];
  PSZ  pINIString = (PSZ) aINIString;
  LONG lNumOfSubkeys = 0;
  LONG lRC = READ_NEW_INI_DATA;

  *pINIString = 0;

  /*
  ** Read the "Effects" INI key list.
  */
  PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pszAppName,
                         szKeyEffects, NULL,
                         (PVOID) aINIString, sizeof( aINIString ) - 1 );

  // Key exists?
  if (*pINIString != 0)
  {
    /*
    ** This loop will count the number of fields (subkeys) in "Effects".  We
    ** don't need to save any data, just count the number of fields.  The
    ** old driver uses no more than OLD_INI_EFFECTS_SIZE number of fields.
    ** The new driver uses a greater number.
    */
    while (*pINIString != 0)
    {
      QueryNextSubkey( &pINIString, NULL, 0 );
      lNumOfSubkeys++;
    }

    /*
    ** If the number of fields does not exceed the largest size for the old
    ** (Warp) driver, then return a code indicating to read the old INI data.
    */
    if (lNumOfSubkeys <= OLD_INI_EFFECTS_SIZE)
    {
      lRC = READ_OLD_INI_DATA;
    }
  }

  return( lRC );
}
/*---------------------------------------------------------------------------*\
* QueryINIKeyToRead End                                                       *
\*---------------------------------------------------------------------------*/
