/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Tray dialog control Functions                           */
/* SOURCE NAME    : TRAYS.C                                                 */
/* AUTHOR         : MFR                                                     */
/*                  Mark H.                                                 */
/* DATE WRITTEN   : 9/26/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/

// os2 includes
#define INCL_DOS
#define INCL_PM
#include <os2.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

// @DBCS
#define INCL_VMANDDI
#include <ddi.h>
#include <pmddi.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"
#include "trays.h"

// Used to help distinguish "add, delete, and modify" buttons for
// "Paper size" page from other pages
/* @170406 SIZE_INDICATOR is defined in both forms.c and in trays.c.  I dont
** know     it has the same name but it is used to make the common button ids
** different for the connection, tray, and form page.  connect.c does not use
** this define so it is 0, trays.c uses 1 and forms.c uses 2.
*/
#define TRAY_INDICATOR  0x00010000

/****************************************************************************/
/* PROCEDURE NAME : CreateUniqueTrayID                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10-18-93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG
CreateUniqueTrayID (PDEVICEINFO pDevice)
{
   ULONG       ulTrayID   = 0;
   PUSERTRAY   pUserTray;
   BOOL        bIDClash   = FALSE;
#ifdef DEBUG
   PDEBUGINFO  pDbg       = &globals.DebugInfo;
#endif

   DBPRINTIF ((pDbg->bCREATEUNIQUETRAYID, "%s(): Enter\n", __FUNCTION__));

   // if we have user defined Trays to check against
   if (pDevice->pUserDefData->usNumTrays > 0)
   {
      // sanity check pointers
      assertF (pDevice->pUserDefData->pUserTRAYS);

      // as long as we don't exceed some large max ID number
      // we'll keep trying to assign a unique ID
      while (ulTrayID < MAX_USER_DEF_ID)
      {
         // assign Tray list pointer to local variable
         pUserTray = pDevice->pUserDefData->pUserTRAYS;

         // for each entry in linked list of user defined Trays
         while ((pUserTray != NULL) && !bIDClash)
         {
            if (pUserTray->tiUser.ulTrayID == (ulTrayID | DEF_TYPE_USER))
            {
               bIDClash = TRUE;
               break;
            }

            pUserTray = pUserTray->pNextTray;
         }

         // we had a clash try next ID
         if (bIDClash)
         {
            ulTrayID++;
            bIDClash = FALSE;
         }
         else
         {
            // we have found a match let's get out of while loop
            break;
         }
      }
   }

   DBPRINTIF ((pDbg->bCREATEUNIQUETRAYID, "%s(): Exit; ID = %lu | DEF_TYPE_USER\n",
               __FUNCTION__, ulTrayID));

   return ulTrayID | DEF_TYPE_USER;

} /* end CreateUniqueTrayID */

/****************************************************************************/
/* PROCEDURE NAME : AddUserDefinedTrayEntry                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
AddUserDefinedTrayEntry (PDEVICEINFO pDevice, PTRAYINFO pNewTrayDef)
{
   PUSERTRAY  pUserTray;
   BOOL       bSuccess   = TRUE;
   BOOL       rc;
#ifdef DEBUG
   PDEBUGINFO pDbg       = &globals.DebugInfo;
#endif

  DBPRINTIF ((pDbg->bADDUSERDEFINEDTRAYENTRY,"%s(): Enter; pNewTrayDef = %x\n", __FUNCTION__, pNewTrayDef));

  // allocate user Tray struct and add to current list
  pUserTray = (PUSERTRAY)GplMemoryAlloc (globals.pvSharedHeap,
                                         sizeof (USERTRAY));

  assertF (pUserTray);

  // if memory is allocated all right
  if (pUserTray)
  {
     USHORT us;

     // Since we are altering global data, request a global semaphore
     // for the changes we need to make...
     rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
     assertT (rc);

     // Copy in new Tray entry from one passed in
     pUserTray->pNextTray              = NULL;
     pUserTray->tiUser.ulTrayID        = pNewTrayDef->ulTrayID;
     pUserTray->tiUser.ulNameID        = pNewTrayDef->ulNameID;
     pUserTray->tiUser.ulTrayType      = pNewTrayDef->ulTrayType;
     strcpy (pUserTray->tiUser.szTrayName, pNewTrayDef->szTrayName);
     pUserTray->tiUser.ulCmdSelectTray = pNewTrayDef->ulCmdSelectTray;

     for (us = 0; us < pUserTray->tiUser.ulCmdSelectTray; us++)
        pUserTray->tiUser.szCmdSelectTray[us] = pNewTrayDef->szCmdSelectTray[us];

     DBPRINTIF ((pDbg->bADDUSERDEFINEDTRAYENTRY,"AddUserDefinedTrayEntry(): Adding Tray = %s\n", pUserTray->tiUser.szTrayName));

     // Add new Tray structure to linked list
     if (pDevice->pUserDefData->pUserTRAYS == NULL)
     {
        // first Tray being defined add at top of list
        pDevice->pUserDefData->pUserTRAYS = pUserTray;
     }
     else
     {
        // add at end of list
        pDevice->pUserDefData->pLastUserTRAYS->pNextTray = pUserTray;
     }

     pDevice->pUserDefData->pLastUserTRAYS = pUserTray;
     pDevice->pUserDefData->usNumTrays++;

     rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
     assertT (rc);
  }

  DBPRINTIF ((pDbg->bADDUSERDEFINEDTRAYENTRY,"%s(): Exit; bSuccess = %d\n",
              __FUNCTION__, bSuccess));

#ifndef DEBUG
  rc++;                  // Avoid compiler warnings
#endif

  return bSuccess;

} /* end AddUserDefinedTrayEntry */

/****************************************************************************/
/* PROCEDURE NAME : DeleteUserDefinedTrayEntry                              */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
DeleteUserDefinedTrayEntry (PDEVICEINFO pDevice, ULONG ulTrayID)
{
   PUSERTRAY   pCurrent;
   PUSERTRAY   pLast    = NULL;
   BOOL        bSuccess = FALSE;
   BOOL        rc;
#ifdef DEBUG
   PDEBUGINFO  pDbg     = &globals.DebugInfo;
#endif

  DBPRINTIF ((pDbg->bDELETEUSERDEFINEDTRAYENTRY,"%s(): Enter\n", __FUNCTION__));

  assertF (pDevice);

  if (ulTrayID & DEF_TYPE_USER)
  {
     /* Since we are altering global data, request a global semaphore
     ** for the changes we need to make...
     */
     rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
     assertT (rc);

     // search for Tray ID in linked list and once found delete entry
     // from list and fix chain
     pCurrent = pDevice->pUserDefData->pUserTRAYS;

     while (pCurrent != NULL)
     {
        // matching TrayID
        if (pCurrent->tiUser.ulTrayID == ulTrayID)
        {
           // we are in middle of list chain
           if (pLast != NULL)
           {
              pLast->pNextTray = pCurrent->pNextTray;
           }
           else
           {
              // deleting first entry
              pDevice->pUserDefData->pUserTRAYS = pCurrent->pNextTray;
           }

           GplMemoryFree (pCurrent);

           pDevice->pUserDefData->usNumTrays--;

           bSuccess = TRUE;

           break;
        }

        // advnce list pointers
        pLast = pCurrent;
        pCurrent = pCurrent->pNextTray;
     }

     rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
     assertT (rc);
  }
  else
  {
     // Inform user that only user defined forms are deletable
     WinMessageBox ((HWND)HWND_DESKTOP,
                    (HWND)NULL,
                    globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_MSG_TRAY],
                    globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_TRAY],
                    1L,
                    MB_OK | MB_INFORMATION);
  }

  DBPRINTIF ((pDbg->bDELETEUSERDEFINEDTRAYENTRY,"%s(): Exit\n", __FUNCTION__));

#ifndef DEBUG
  rc++;                  // Avoid compiler warnings
#endif

  return bSuccess;

} /* end DeleteUserDefinedTrayEntry */

/****************************************************************************/
/* PROCEDURE NAME : FreeUserDefinedTrayList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
FreeUserDefinedTrayList (PDEVICEINFO pDevice)
{
  PUSERTRAY   pDelete,
              pNext;
  BOOL        rc;
#ifdef DEBUG
  PDEBUGINFO  pDbg      = &globals.DebugInfo;
#endif

  DBPRINTIF ((pDbg->bFREEUSERDEFINEDTRAYLIST,"%s(): Enter\n", __FUNCTION__));

  assertF (pDevice);

  if (!pDevice || !pDevice->pUserDefData)
     return;

  /* Since we are altering global data, request a global semaphore
  ** for the changes we need to make...
  */
  rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
  assertT (rc);

  // search for Tray ID in linked list and once found delete entry
  // from list and fix chain
  pDelete = pDevice->pUserDefData->pUserTRAYS;

  while (pDelete != NULL)
  {
     pNext = pDelete->pNextTray;

     GplMemoryFree (pDelete);

     pDelete = pNext;
  }

  // reset linked list values
  pDevice->pUserDefData->bTraysInit     = FALSE;
  pDevice->pUserDefData->pUserTRAYS     = NULL;
  pDevice->pUserDefData->pLastUserTRAYS = NULL;
  pDevice->pUserDefData->usNumTrays     = 0;

  rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
  assertT (rc);

  DBPRINTIF ((pDbg->bFREEUSERDEFINEDTRAYLIST,"%s(): Exit\n", __FUNCTION__));

#ifndef DEBUG
  rc++;                  // Avoid compiler warnings
#endif

} /* FreeUserDefinedTrayList */

/****************************************************************************/
/* PROCEDURE NAME : SaveUserDefinedTrayList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
SaveUserDefinedTrayList (PSZ pszAppName, PDEVICEINFO pDevice)
{
   USHORT           usTraysProcessed    = 0;
   USHORT           usTotalTrays;
   PUSERTRAY        pCurrent;
   CHAR             szTemp[LEN_STRING];
   PEXTTRAYINFOSAVE pTray;
   PBYTE            pbBuffer;
   ULONG            ulDataSize;
   BOOL             bSuccess            = TRUE;

   // write out Num trays to INI
   DBPRINTF (("SaveUserDefinedTrayList(): Enter\n"));

   assertF (pDevice);
   usTotalTrays = pDevice->pUserDefData->usNumTrays;

   _itoa (usTotalTrays, szTemp, 10);

   PrfWriteProfileString (HINI_SYSTEMPROFILE,
                          pszAppName,
                          (PSZ)KEYNAME_NUM_USER_TRAYS,
                          (PVOID)szTemp);

   // allocate memory to hold all trayinfo and cast to TRAYINFO struct
   ulDataSize = usTotalTrays * sizeof (EXTTRAYINFO_1_1);

   // if we have trays strucs to allocate for
   if (ulDataSize > 0)
   {
      pbBuffer = GplMemoryAlloc (0, ulDataSize);
      assertF (pbBuffer);

      if (pbBuffer)
      {
         pTray = (PEXTTRAYINFOSAVE)pbBuffer;

         // set up start of list pointer
         pCurrent = pDevice->pUserDefData->pUserTRAYS;

         // while we have trays in linked list to process
         while (pCurrent != NULL && usTraysProcessed <= usTotalTrays)
         {
            // copy out to memory area
            pTray->ulTrayID   = pCurrent->tiUser.ulTrayID;
            pTray->ulTrayCap  = pCurrent->tiUser.ulTrayCap;
            pTray->ulNameID   = pCurrent->tiUser.ulNameID;
            memcpy (pTray->szTrayName,
                    pCurrent->tiUser.szTrayName,
                    sizeof (pTray->szTrayName));

#ifdef DEBUG
{
   ULONG ulFrom = sizeof (pCurrent->tiUser.szTrayName),
         ulTo   = sizeof (pTray->szTrayName);

   if (ulFrom > ulTo)
   {
      DBPRINTF (("***WARNING*** %s () loosing data copied from szTrayName!\n",
                 __FUNCTION__));
   }
}
#endif
            pTray->szTrayName[sizeof (pTray->szTrayName) - 1] = '\0';

            pTray->ulTrayType      = pCurrent->tiUser.ulTrayType;
            pTray->ulCmdSelectTray = min (pCurrent->tiUser.ulCmdSelectTray,
                                          sizeof (pTray->szCmdSelectTray));
            memcpy (pTray->szCmdSelectTray,
                    pCurrent->tiUser.szCmdSelectTray,
                    sizeof (pTray->szCmdSelectTray));

#ifdef DEBUG
{
   ULONG ulFrom = sizeof (pCurrent->tiUser.szCmdSelectTray),
         ulTo   = sizeof (pTray->szCmdSelectTray);

   if (ulFrom > ulTo)
   {
      DBPRINTF (("***WARNING*** %s () lose data copied from szCmdSelectTray!\n",
                 __FUNCTION__));
   }
}
#endif

            /* @TBD - Need to handle
            */
            pTray->ulCompatibleForms  = 0;
            pTray->pulCompatibleForms = NULL;

            // increment counter
            usTraysProcessed++;

            // advance buffer pointer
            pTray++;

            // advance list pointer
            pCurrent = pCurrent->pNextTray;
         }

         PrfWriteProfileData (HINI_SYSTEMPROFILE,
                              pszAppName,
                              (PSZ)KEYNAME_USER_TRAYS,
                              (PVOID)pbBuffer,
                              ulDataSize );

         PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                pszAppName,
                                (PSZ)KEYNAME_VERSION_KEY_USER_TRAYS,
                                (PVOID)KEYNAME_VERSION_DATA_USER_TRAYS);

         GplMemoryFree (pbBuffer);

         bSuccess = TRUE;
      }
      else
      {
         bSuccess = FALSE;
      }
   }

   if (FALSE == bSuccess)
   {
      // overwrite any old entries with NULL
      PrfWriteProfileData (HINI_SYSTEMPROFILE,
                           pszAppName,
                           (PSZ)KEYNAME_USER_TRAYS,
                           (PVOID)NULL,
                           0);

      PrfWriteProfileData (HINI_SYSTEMPROFILE,
                           pszAppName,
                           (PSZ)KEYNAME_VERSION_KEY_USER_TRAYS,
                           (PVOID)NULL,
                           0);
   }

   DBPRINTF (("SaveUserDefinedTrayList(): Exit\n"));

   return bSuccess;

} /* SaveUserDefinedTrayList */

/****************************************************************************/
/* PROCEDURE NAME : ReadUserDefinedTrayList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ReadUserDefinedTrayList (PSZ pszAppName, PDEVICEINFO pDevice)
{
   PTRAYINFO  pTray, pBase;
   ULONG      ulDataSize;
   BOOL       bSuccess            = TRUE;
   CHAR       szTemp[LEN_STRING];
   USHORT     usTraysProcessed    = 0;
   USHORT     usNumUserTrays      = 0;
   BOOL       bRC;
   PBYTE      pbInput             = (PBYTE)NULL;
#ifdef DEBUG
   PDEBUGINFO pDbg                = &globals.DebugInfo;
#endif

   DBPRINTIF ((pDbg->bREADUSERDEFINEDTRAYLIST, "%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);

   if (!pDevice->pUserDefData->bTraysInit)
   {
      ulDataSize = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                          pszAppName,
                                          (PSZ)KEYNAME_NUM_USER_TRAYS,
                                          (PSZ)NULL,
                                          (PVOID)szTemp,
                                          LEN_STRING);

      DBPRINTF (("ReadUserDefinedTrayList(); ulDataSize = %d\n", ulDataSize));

      // If we have an INI entry
      if (ulDataSize != 0)
      {
         // read Num trays from INI
         usNumUserTrays = atoi (szTemp);

         DBPRINTF (("ReadUserDefinedTrayList(); usNumTrays = %d\n", usNumUserTrays));

         if (usNumUserTrays > 0)
         {
            // allocate memory to hold all tray info and cast to TRAYINFO struct
            ulDataSize = usNumUserTrays * sizeof (TRAYINFO);

            pBase = GplMemoryAlloc (0, ulDataSize);
            assertF (pBase);

            if (pBase)
            {
               ULONG   ulEntrySize;

               // read in profile data
               bRC = PrfQueryProfileSize (HINI_SYSTEMPROFILE,
                                          pszAppName,
                                          (PSZ)KEYNAME_USER_TRAYS,
                                          &ulDataSize);
               assertF (bRC);

               if (bRC)
               {
                  ulEntrySize = ulDataSize / usNumUserTrays;

                  switch (ulEntrySize)
                  {
                  case SIZEOF_EXTTRAYINFO_1_1:
                  {
                     PEXTTRAYINFO_1_1 pTrayInput;

                     DBPRINTF (("%d user defined trays are in 1.1 format!\n", usNumUserTrays));

                     ulDataSize = usNumUserTrays * sizeof (EXTTRAYINFO_1_1);

                     pbInput = GplMemoryAlloc (0, ulDataSize);
                     assertF (pbInput);

                     bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                                pszAppName,
                                                (PSZ)KEYNAME_USER_TRAYS,
                                                (PVOID)pbInput,
                                                &ulDataSize);
                     assertF (bRC);

                     // if data read in correctly
                     if (bRC)
                     {
                        // init current entry pointer to each Tray structure
                        pTray      = pBase;
                        pTrayInput = (PEXTTRAYINFO_1_1)pbInput;

                        // while we have Trays in linked buffer to process
                        while (usTraysProcessed < usNumUserTrays)
                        {
                           // Translate from external format to internal format
                           pTray->ulTrayID   = pTrayInput->ulTrayID;
                           pTray->ulTrayCap  = pTrayInput->ulTrayCap;
                           pTray->ulDJPid    = 0;
                           pTray->ulNameID   = pTrayInput->ulNameID;
                           memcpy (pTray->szTrayName,
                                   pTrayInput->szTrayName,
                                   sizeof (pTrayInput->szTrayName));
                           pTray->szTrayName[sizeof (pTrayInput->szTrayName) - 1] = '\0';
                           pTray->ulTrayType      = pTrayInput->ulTrayType;
                           pTray->ulCmdSelectTray = pTrayInput->ulCmdSelectTray;
                           memcpy (pTray->szCmdSelectTray,
                                   pTrayInput->szCmdSelectTray,
                                   sizeof (pTrayInput->szCmdSelectTray));
                           // @TBD
                           pTray->ulCompatibleForms  = 0;
                           pTray->pulCompatibleForms = NULL;

                           AddUserDefinedTrayEntry (pDevice, pTray);

                           // increment counter
                           usTraysProcessed++;

                           // advance buffer pointer
                           pTray++;
                           pTrayInput++;
                        }

                        // free memory
                        bRC = GplMemoryFree (pBase);
                        assertT (bRC);
                     }
                     else
                     {
                        bSuccess = FALSE;
                     }
                     break;
                  }

                  case SIZEOF_EXTTRAYINFO_1_5:
                  {
                     PEXTTRAYINFO_1_5 pTrayInput;

                     DBPRINTF (("%d user defined trays are in 1.5 format!\n", usNumUserTrays));

                     ulDataSize = usNumUserTrays * sizeof (EXTTRAYINFO_1_5);

                     pbInput = GplMemoryAlloc (0, ulDataSize);
                     assertF (pbInput);

                     bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                                pszAppName,
                                                (PSZ)KEYNAME_USER_TRAYS,
                                                (PVOID)pbInput,
                                                &ulDataSize);
                     assertF (bRC);

                     // if data read in correctly
                     if (bRC)
                     {
                        // init current entry pointer to each Tray structure
                        pTray = pBase;
                        pTrayInput = (PEXTTRAYINFO_1_5)pbInput;

                        // while we have Trays in linked buffer to process
                        while (usTraysProcessed < usNumUserTrays)
                        {
                           // Translate from external format to internal format
                           pTray->ulTrayID   = pTrayInput->ulTrayID;
                           pTray->ulTrayCap  = pTrayInput->ulTrayCap;
                           pTray->ulDJPid    = 0;
                           pTray->ulNameID   = pTrayInput->ulNameID;
                           memcpy (pTray->szTrayName,
                                   pTrayInput->szTrayName,
                                   sizeof (pTrayInput->szTrayName));
                           pTray->szTrayName[sizeof (pTrayInput->szTrayName) - 1] = '\0';
                           pTray->ulTrayType      = pTrayInput->ulTrayType;
                           pTray->ulCmdSelectTray = pTrayInput->ulCmdSelectTray;
                           memcpy (pTray->szCmdSelectTray,
                                   pTrayInput->szCmdSelectTray,
                                   sizeof (pTrayInput->szCmdSelectTray));
                           // @TBD
                           pTray->ulCompatibleForms  = 0;
                           pTray->pulCompatibleForms = NULL;

                           AddUserDefinedTrayEntry (pDevice, pTray);

                           // increment counter
                           usTraysProcessed++;

                           // advance buffer pointer
                           pTray++;
                           pTrayInput++;
                        }

                        // free memory
                        bRC = GplMemoryFree (pBase);
                        assertT (bRC);
                     }
                     else
                     {
                        bSuccess = FALSE;
                     }
                     break;
                  }

                  case SIZEOF_EXTTRAYINFO_1_21:
                  {
                     PEXTTRAYINFO_1_21 pTrayInput;

                     DBPRINTF (("%d user defined trays are in 1.21 format!\n", usNumUserTrays));

                     ulDataSize = usNumUserTrays * sizeof (EXTTRAYINFO_1_21);

                     pbInput = GplMemoryAlloc (0, ulDataSize);
                     assertF (pbInput);

                     bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                                pszAppName,
                                                (PSZ)KEYNAME_USER_TRAYS,
                                                (PVOID)pbInput,
                                                &ulDataSize);
                     assertF (bRC);

                     // if data read in correctly
                     if (bRC)
                     {
                        // init current entry pointer to each Tray structure
                        pTray      = pBase;
                        pTrayInput = (PEXTTRAYINFO_1_21)pbInput;

                        // while we have Trays in linked buffer to process
                        while (usTraysProcessed < usNumUserTrays)
                        {
                           // Translate from external format to internal format
                           pTray->ulTrayID   = pTrayInput->ulTrayID;
                           pTray->ulTrayCap  = pTrayInput->ulTrayCap;
                           pTray->ulDJPid    = 0;
                           pTray->ulNameID   = pTrayInput->ulNameID;
                           memcpy (pTray->szTrayName,
                                   pTrayInput->szTrayName,
                                   sizeof (pTrayInput->szTrayName));
                           pTray->szTrayName[sizeof (pTrayInput->szTrayName) - 1] = '\0';
                           pTray->ulTrayType      = pTrayInput->ulTrayType;
                           pTray->ulCmdSelectTray = pTrayInput->ulCmdSelectTray;
                           memcpy (pTray->szCmdSelectTray,
                                   pTrayInput->szCmdSelectTray,
                                   sizeof (pTrayInput->szCmdSelectTray));
                           // @TBD
                           pTray->ulCompatibleForms  = 0;
                           pTray->pulCompatibleForms = NULL;

                           AddUserDefinedTrayEntry (pDevice, pTray);

                           // increment counter
                           usTraysProcessed++;

                           // advance buffer pointer
                           pTray++;
                           pTrayInput++;
                        }

                        // free memory
                        bRC = GplMemoryFree (pBase);
                        assertT (bRC);
                     }
                     else
                     {
                        bSuccess = FALSE;
                     }
                     break;
                  }

                  default:
                  {
                     DBPRINTF (("%d user defined Trays are in ***UNKNOWN*** format! ulEntrySize = %d ulDataSize = %d\n",
                                usNumUserTrays, ulEntrySize, ulDataSize));

                     assertstring ("Unknown size!\n");
                     bRC =GplMemoryFree (pBase);
                     assert (bRC);
                     return FALSE;
                  }
                  }
               }
            }
            else
            {
               bSuccess = FALSE;
            }
         }
      }
      else
      {
         DBPRINTF (("ReadUserDefinedTrayList(): No INI data for user-defined trays\n"));
      }

      // indicate we have read in tray data
      pDevice->pUserDefData->bTraysInit = TRUE;
   }

   // Clean up
   if (pbInput)
   {
      bRC = GplMemoryFree (pbInput);
      assertT (bRC);
   }

   DBPRINTIF(( pDbg->bREADUSERDEFINEDTRAYLIST, "ReadUserDefinedTrayList(): Exit\n"));

   return bSuccess;

} /* ReadUserDefinedTrayList */

/****************************************************************************/
/* PROCEDURE NAME : TrayNameFromID                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/23/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
TrayNameFromID (PDRIVERINFO pDriver,
                PDEVICEINFO pDevice,
                PSZ         pszTrayName,
                ULONG       ulTrayID)
{
   ULONG      ulDefType;
   BOOL       bTrayNameFound = FALSE;
   ULONG      i;
   ULONG      ulNumDefined;
   PTRAYINFO  pTrays;
#ifdef DEBUG
   PDEBUGINFO pDbg           = &globals.DebugInfo;
#endif

   DBPRINTIF ((pDbg->bTRAYNAMEFROMID, "%s(): Enter\n", __FUNCTION__));

   // Find out how Tray defined (in driver, in device, by user)
   ulDefType = MAKEULONG (0, HIUSHORT (ulTrayID));

   switch (ulDefType)
   {
   case DEF_TYPE_DRIVER:
   {
      ulNumDefined = pDriver->ulNumTrays;
      pTrays       = pDriver->pTRAYS;

      // For now only driver table definitions allowed
      // loop through table of Trays until matching Tray ID
      // found
      for (i = 0; i < ulNumDefined; i++)
      {
         if (ulTrayID == pTrays[i].ulTrayID)
         {
            // Now we must determine if the string is in our stringtable
            // or a it is a string the user defined in hcInfo structure
            if (pTrays[i].ulNameID == TRAY_STRING_UNLISTED)
            {
               strcpy (pszTrayName, pTrays[i].szTrayName);
            }
            else
            {
               ULONG ulIdx;

               ulIdx = pTrays[i].ulNameID - STRING_TABLE_BASE;

               assertF (globals.pbStringTable[ulIdx]);

               strcpy (pszTrayName, globals.pbStringTable[ulIdx]);
            }

            bTrayNameFound = TRUE;
            break;
         }
      }
      break;
   }

   case DEF_TYPE_DEVICE:
   {
      /* @TBD */
      break;
   }

   case DEF_TYPE_USER:
   {
      PUSERTRAY pUserTray;

      // assign Tray list pointer to local variable
      pUserTray = pDevice->pUserDefData->pUserTRAYS;

      // for each entry in linked list of user defined Trays
      while (pUserTray != NULL)
      {
         if (pUserTray->tiUser.ulTrayID == ulTrayID)
         {
            strcpy (pszTrayName, pUserTray->tiUser.szTrayName);

            bTrayNameFound = TRUE;

            break;
         }

         pUserTray = pUserTray->pNextTray;
      }

      break;
   }

   default:
   {
      assertstring ("Should not be here!\n");
      break;
   }
   }

   DBPRINTIF ((pDbg->bTRAYNAMEFROMID, "%s(): Tray Name Found = %s\n", __FUNCTION__, pszTrayName));

   DBPRINTIF ((pDbg->bTRAYNAMEFROMID, "%s(): Exit\n", __FUNCTION__));
   assertF (bTrayNameFound);

   return bTrayNameFound;

} /* end TrayNameFromID */

/****************************************************************************/
/* PROCEDURE NAME : AddTrayNameToTrayInfo                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/01/96                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
AddTrayNameToTrayInfo (PTRAYINFO pTrayInfo)
{
   DBPRINTIF ((TRUE, "%s(): Enter, ulNameID = %d\n", __FUNCTION__, pTrayInfo->ulNameID));

   // Now we must determine if the string is in our stringtable
   // or a it is a string the user defined in hcInfo structure
   if (pTrayInfo->ulNameID == TRAY_STRING_UNLISTED)
   {
      // string already there!
      DBPRINTIF ((FALSE, "%s(): Adding '%s' (UNLISTED)\n", __FUNCTION__, pTrayInfo->szTrayName));
   }
   else
   {
      ULONG ulIdx;

      ulIdx = pTrayInfo->ulNameID - STRING_TABLE_BASE;

      assertF (globals.pbStringTable[ulIdx]);

      strcpy (pTrayInfo->szTrayName, globals.pbStringTable[ulIdx]);

      DBPRINTIF ((FALSE, "%s(): Adding '%s' from string table\n", __FUNCTION__, pTrayInfo->szTrayName));
   }

   DBPRINTIF ((FALSE, "%s(): Exit\n", __FUNCTION__));

   return TRUE;

} /* end AddTrayNameToTrayInfo */

/****************************************************************************/
/* PROCEDURE NAME : TrayInfoFromID                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/23/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
TrayInfoFromID (PDRIVERINFO pDriver,
                PDEVICEINFO pDevice,
                ULONG       ulTrayID,
                PPTRAYINFO  pTray)
{
  ULONG      ulDefType;
  BOOL       bTrayInfoFound = FALSE;
  ULONG      i;
  ULONG      ulNumDefined;
  PTRAYINFO  pTrays;

  DBPRINTIF ((FALSE, "%s(): Enter\n", __FUNCTION__));

  // Find out how Tray defined (in driver, in device, by user)
  ulDefType = MAKEULONG (0, HIUSHORT(ulTrayID));

  switch (ulDefType)
  {
  case DEF_TYPE_DRIVER:  // No hi word value
  {
     ulNumDefined = pDriver->ulNumTrays;
     pTrays       = pDriver->pTRAYS;

     // For now only driver table definitions allowed
     // loop through table of trays until matching tray ID
     // found
     for (i = 0; i < ulNumDefined; i++)
     {
        if (ulTrayID == pTrays[i].ulTrayID)
        {
           *pTray = &(pTrays[i]);

           bTrayInfoFound = TRUE;

           break;
        }
     }

     break;
  }

  case DEF_TYPE_DEVICE:
  {
     /* @TBD */
     break;
  }

  case DEF_TYPE_USER:
  {
     PUSERTRAY pUserTray;

     // assign Tray list pointer to local variable
     pUserTray = pDevice->pUserDefData->pUserTRAYS;

     // for each entry in linked list of user defined Trays
     while (pUserTray != NULL)
     {
        if (pUserTray->tiUser.ulTrayID == ulTrayID)
        {
           *pTray = &pUserTray->tiUser;

           bTrayInfoFound = TRUE;

           break;
        }

        pUserTray = pUserTray->pNextTray;
     }

     break;
  }

  default:
  {
     assertstring ("Should not be here!\n");
     break;
  }
  }

  assertF (bTrayInfoFound);

  DBPRINTIF ((FALSE, "%s(): Exit\n", __FUNCTION__));

  return bTrayInfoFound;

} /* end TrayInfoFromID */

/****************************************************************************/
/* PROCEDURE NAME : FillTrayList                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/22/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
FillTrayList (HWND         hwndDlg,
              PDLGINSTANCE pdi,
              PDEVICEINFO  pDevice,
              ULONG        ulListboxID)
{
   USHORT     usNumTrays;
   USHORT     i;
   ULONG      ulTrayID;
   CHAR       szTrayName[LEN_STRING];
   PUSERTRAY  pUserTray;
   USHORT     usMatchIdx                = 0;
#ifdef DEBUG
   PDEBUGINFO pDbg                      = &globals.DebugInfo;
#endif

  DBPRINTIF ((pDbg->bFILLTRAYLIST, "%s(): Enter\n", __FUNCTION__));

  // initial pointer validation
  assertF (pdi);
  assertF (pDevice);

  // free existing list
  WinSendDlgItemMsg (hwndDlg,
                     ulListboxID,
                     LM_DELETEALL,
                     MPVOID,
                     MPVOID);

  /************************************/
  /* DRIVER TRAYS                     */
  /************************************/

  // get # Trays defined
  usNumTrays = pdi->pDevice->usNumTrays;

  // add each Tray defined into listbox
  for (i = 0; i < usNumTrays; i++)
  {
     ulTrayID = pdi->pDevice->pulTRAYS[i];

     // if Tray ID found and string returned
     if (TrayNameFromID (pdi->pDriver, pDevice, szTrayName, ulTrayID))
     {
        SHORT sIdx;

        sIdx = (SHORT)WinSendDlgItemMsg (hwndDlg,
                                         ulListboxID,
                                         LM_INSERTITEM,
                                         MPFROM2SHORT (LIT_END, 0),
                                         MPFROMP (szTrayName));

        // Set Item Handle to be the Tray ID
        WinSendDlgItemMsg (hwndDlg,
                           ulListboxID,
                           LM_SETITEMHANDLE,
                           MPFROMSHORT (sIdx),
                           MPFROMLONG (ulTrayID));

        DBPRINTIF ((pDbg->bFILLTRAYLIST, "%s(): Adding Tray: '%s'; ID=%lu\n", __FUNCTION__, szTrayName, ulTrayID));
     }
  }

  /************************************/
  /* User Defined Trays               */
  /************************************/

  // get # Trays defined in linked list by user
  usNumTrays = pDevice->pUserDefData->usNumTrays;

  // assign base pointer to linked list
  pUserTray = pDevice->pUserDefData->pUserTRAYS;

  // add each Tray defined into listbox
  for (i = 0; i < usNumTrays; i++)
  {
     // value easier to dereference
     ulTrayID = pUserTray->tiUser.ulTrayID;

     // if Tray ID found and string returned
     if (TrayNameFromID (pdi->pDriver, pDevice, szTrayName, pUserTray->tiUser.ulTrayID))
     {
        SHORT sIdx;

        sIdx = (SHORT)WinSendDlgItemMsg (hwndDlg,
                                         ulListboxID,
                                         LM_INSERTITEM,
                                         MPFROM2SHORT (LIT_END, 0),
                                         MPFROMP (szTrayName));

        // Set Item Handle to be the Tray ID
        WinSendDlgItemMsg (hwndDlg,
                           ulListboxID,
                           LM_SETITEMHANDLE,
                           MPFROMSHORT (sIdx),
                           MPFROMLONG (ulTrayID));

        DBPRINTIF ((pDbg->bFILLTRAYLIST, "%s(): Adding Tray: '%s'; ID=%lu\n", __FUNCTION__, szTrayName, ulTrayID));
     }

     // increment list pointer
     pUserTray = pUserTray->pNextTray;
  }

  // Select entry with matching tray ID
  WinSendDlgItemMsg (hwndDlg,
                     ulListboxID,
                     LM_SELECTITEM,
                     MPFROMLONG (usMatchIdx),
                     MPFROMLONG (TRUE));

  DBPRINTIF ((pDbg->bFILLTRAYLIST, "%s(): Highliting Tray index: (%d)\n", __FUNCTION__, usMatchIdx));

  DBPRINTIF ((pDbg->bFILLTRAYLIST, "%s(): Exit\n", __FUNCTION__));

} /* end FillTrayList */

/****************************************************************************/
/* PROCEDURE NAME : IsTrayNameUnique                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/31/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
IsTrayNameUnique (PDLGINSTANCE pdi, PSZ pszNewName, PSZ pszIgnore)
{
   PDEVICEINFO pDevice;
   USHORT      usNumTrays;
   USHORT      i;
   ULONG       ulTrayID;
   CHAR        szTrayName[TRAYNAME_SIZE+2];
   PUSERTRAY   pUserTray;
   BOOL        bIsUnique = TRUE;

   DBPRINTF (("IsTrayNameUnique(): Enter\n"));

   // initial pointer validation
   assertF (pdi);

   // easier pointer dereference
   pDevice = pdi->pDevice;
   assertF (pDevice);

   /************************************/
   /* Driver Defined Trays             */
   /************************************/

   // get # Trays defined
   usNumTrays = pDevice->usNumTrays;

   // check name passed in against each Tray defined into listbox
   for (i = 0; i < usNumTrays; i++)
   {
      ulTrayID = pdi->pDriver->pTRAYS[i].ulTrayID;

      // if Tray ID found and string returned
      if (TrayNameFromID (pdi->pDriver, pDevice, szTrayName, ulTrayID))
      {
         if (0 == strncmp (szTrayName, pszNewName, TRAYNAME_SIZE))
         {
            if (0 != strncmp (szTrayName, pszIgnore, TRAYNAME_SIZE))
            {
               bIsUnique = FALSE;
               break;
            }
         }
      }
   }

   /************************************/
   /* User Defined Trays               */
   /************************************/

   // Still need to keep checking new Tray name for uniqueness
   if (bIsUnique)
   {
      // get # Trays defined in linked list by user
      usNumTrays = pDevice->pUserDefData->usNumTrays;

      // assign base pointer to linked list
      pUserTray = pDevice->pUserDefData->pUserTRAYS;

      // add each Tray defined into listbox
      for (i = 0; i < usNumTrays; i++)
      {
         // Get local copy of Tray ID
         ulTrayID = pUserTray->tiUser.ulTrayID;

         // if Tray ID found and string returned
         if (TrayNameFromID (pdi->pDriver, pDevice, szTrayName, ulTrayID))
         {
            if (0 == strncmp (szTrayName, pszNewName, TRAYNAME_SIZE))
            {
               if (0 != strncmp (szTrayName, pszIgnore, TRAYNAME_SIZE))
               {
                  bIsUnique = FALSE;
                  break;
               }
            }
         }

         // increment list pointer
         pUserTray = pUserTray->pNextTray;
      }
   }

   DBPRINTF (("IsTrayNameUnique(): Exit; bIsUnique = %d\n", bIsUnique));

   return bIsUnique;

} /* end IsTrayNameUnique */

/****************************************************************************/
/* PROCEDURE NAME : UpdateTrayDlg                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 6/01/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
UpdateTrayDlg (HWND hwnd, PDLGINSTANCE pdi, ULONG ulTrayID)
{
   PTRAYINFO  pTray;
   BOOL       bRC;
#ifdef DEBUG
   BOOL       bShow = FALSE;
#endif

  DBPRINTIF ((bShow, "%s(): Enter\n", __FUNCTION__));

  bRC = TrayInfoFromID (pdi->pDriver, pdi->pDevice, ulTrayID, &pTray);

  assertF (bRC);
  assertF (pTray);

  if (bRC)
  {
     CHAR  szTemp[LEN_STRING];
     ULONG ulControlID;
     ULONG ulIdx;

     TrayNameFromID (pdi->pDriver, pdi->pDevice, szTemp, ulTrayID);

     DBPRINTIF ((bShow, "%s(): Tray ID='%lu'; NAME='%s'\n",
                 __FUNCTION__, ulTrayID, szTemp));

     WinSetDlgItemText (hwnd, IDE_DEF_TRAY_NAME, szTemp);

     ConvertHexCmdToAscii (pTray->ulCmdSelectTray,
                           pTray->szCmdSelectTray,
                           (PSZ)szTemp);

     WinSetDlgItemText (hwnd, IDE_DEF_TRAY_SELECT_CMD, szTemp);

     WinSetDlgItemShort (hwnd,
                         IDE_TRAY_ID,
                         LOUSHORT (ulTrayID),
                         FALSE);

     if (HIUSHORT (ulTrayID))
        ulIdx = DIALOG_STRING_TRAY_USERID - STRING_TABLE_BASE;
     else
        ulIdx = DIALOG_STRING_TRAY_ID - STRING_TABLE_BASE;
     WinSetWindowText (WinWindowFromID (hwnd, IDT_TRAY_ID),
                       globals.pbStringTable[ulIdx]);

     // set radio button for tray type
     switch (pTray->ulTrayType)
     {
     case TRAY_TYPE_AUTO:
     {
        ulControlID = IDR_DEF_TRAY_TYPE_AUTO;
        break;
     }

     case TRAY_TYPE_MANUAL:
     {
        ulControlID = IDR_DEF_TRAY_TYPE_MANUAL;
        break;
     }

     default:  // default to automatic
     {
        ulControlID = IDR_DEF_TRAY_TYPE_AUTO;
        assertstring ("Should not be here!\n");
        break;
     }
     }

     WinSendDlgItemMsg (hwnd, ulControlID, BM_SETCHECK, (MPARAM)TRUE, 0);
  }

  DBPRINTIF ((bShow, "%s(): Exit\n", __FUNCTION__));

} /* end UpdateTrayDlg */

/****************************************************************************/
/* PROCEDURE NAME : AddDefinedTray                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 6/02/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
SHORT
AddDefinedTray (HWND hwnd, PDLGINSTANCE pdi, ULONG ulTrayID)
{
   TRAYINFO  UserTray;
   CHAR      szTemp[LEN_STRING];
   SHORT     sIdx;
   ULONG     ulChkState;

   DBPRINTF (("AddDefinedTray(): Enter\n"));

   WinQueryWindowText (WinWindowFromID (hwnd, IDE_DEF_TRAY_NAME),
                       LEN_STRING-2, szTemp);

   strcpy (UserTray.szTrayName, szTemp);

   UserTray.ulCmdSelectTray = ConvertAsciiCmdToHex (WinWindowFromID (hwnd,
                                                                     IDE_DEF_TRAY_SELECT_CMD),
                                                    UserTray.szCmdSelectTray,
                                                    LEN_CMD);

   // @TBD - validate entries needed to create tray definition

   // Create a unique User Defined Tray ID and place in definition
   UserTray.ulTrayID = CreateUniqueTrayID (pdi->pDevice);

   DBPRINTF (("DefineTrayDlgProc(): Tray ID='%lu'; NAME='%s'\n", UserTray.ulTrayID, UserTray.szTrayName));

   // @TBD - query tray feed type and stick into structure
   ulChkState = WinQueryButtonCheckstate (hwnd, IDR_DEF_TRAY_TYPE_AUTO);

   if (ulChkState)
   {
      UserTray.ulTrayType = TRAY_TYPE_AUTO;
   }
   else
   {
      UserTray.ulTrayType = TRAY_TYPE_MANUAL;
   }

   // Add new tray definition to linked list
   AddUserDefinedTrayEntry (pdi->pDevice, &UserTray);

   // update tray listbox with new entry
   sIdx = (SHORT)WinSendDlgItemMsg (hwnd,
                                    IDL_DEF_TRAYS,
                                    LM_INSERTITEM,
                                    MPFROM2SHORT (LIT_END, 0),
                                    MPFROMP (UserTray.szTrayName));

   // Set Item Handle to be the Tray ID
   WinSendDlgItemMsg (hwnd,
                      IDL_DEF_TRAYS,
                      LM_SETITEMHANDLE,
                      MPFROMSHORT (sIdx),
                      MPFROMLONG (UserTray.ulTrayID));

   WinSendDlgItemMsg (hwnd,
                      IDL_DEF_TRAYS,
                      LM_SELECTITEM,
                      MPFROMLONG (sIdx),
                      MPFROMLONG (TRUE));

   // update listbox on connection page with new entry
   sIdx = (SHORT)WinSendDlgItemMsg (pdi->hwndConnectDlg,
                                    IDL_FORM_SOURCE,
                                    LM_INSERTITEM,
                                    MPFROM2SHORT (LIT_END, 0),
                                    MPFROMP (UserTray.szTrayName));

   // Set Item Handle to be the Tray ID
   WinSendDlgItemMsg (pdi->hwndConnectDlg,
                      IDL_FORM_SOURCE,
                      LM_SETITEMHANDLE,
                      MPFROMSHORT (sIdx),
                      MPFROMLONG (UserTray.ulTrayID));

   WinSendDlgItemMsg (pdi->hwndConnectDlg,
                      IDL_FORM_SOURCE,
                      LM_SELECTITEM,
                      MPFROMLONG (sIdx),
                      MPFROMLONG (TRUE));

   DBPRINTF (("AddDefinedTray(): Exit\n"));

   return sIdx;

} /* end AddDefinedTray */

/****************************************************************************/
/* PROCEDURE NAME : DeleteDefinedTray                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 6/07/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
DeleteDefinedTray (HWND hwnd, PDLGINSTANCE pdi, SHORT sIdx)
{
   ULONG ulTrayID;
   BOOL  bRC;

   // Get Item Handle (Tray ID)
   ulTrayID = (ULONG)WinSendDlgItemMsg (hwnd,
                                        IDL_DEF_TRAYS,
                                        LM_QUERYITEMHANDLE,
                                        MPFROMSHORT (sIdx),
                                        MPVOID);

   // Find tray entry in linked list or in driver table and delete
   bRC = DeleteUserDefinedTrayEntry (pdi->pDevice, ulTrayID);

   // if delete was a success remove tray name from listbox
   if (bRC)
   {
      // remove listbox entry
      WinSendDlgItemMsg (hwnd,
                         IDL_DEF_TRAYS,
                         LM_DELETEITEM,
                         MPFROMSHORT (sIdx),
                         MPVOID);

      WinSendDlgItemMsg (hwnd,
                         IDL_DEF_TRAYS,
                         LM_SELECTITEM,
                         MPFROMLONG (0),
                         MPFROMLONG (TRUE));

      // remove listbox entry
      WinSendDlgItemMsg (pdi->hwndConnectDlg,
                         IDL_FORM_SOURCE,
                         LM_DELETEITEM,
                         MPFROMSHORT (sIdx),
                         MPVOID);

      WinSendDlgItemMsg (pdi->hwndConnectDlg,
                         IDL_FORM_SOURCE,
                         LM_SELECTITEM,
                         MPFROMLONG (0),
                         MPFROMLONG (TRUE));
   }

   return bRC;

} /* end DeleteDefinedTray */

/****************************************************************************/
/* PROCEDURE NAME : DefineTrayDlgProc                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/22/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY
DefineTrayDlgProc (HWND hwnd, ULONG  msg, MPARAM mp1, MPARAM mp2)
{
   PDLGINSTANCE  pdi;
   ULONG         ulTrayID = 0;
   SHORT         sIdx;
   BOOL          bRC;

   switch (msg)
   {
   case WM_CONTROLPOINTER:
   {
      ULONG ulCtlID = LONGFROMMP (mp1);

      pdi = (PDLGINSTANCE)WinQueryWindowULong (hwnd, QWL_USER);

      if (ulCtlID != (ULONG)DID_STATIC)
      {
         if (  ulCtlID == DID_ADD
            || ulCtlID == DID_DELETE
            || ulCtlID == DID_MODIFY
            )
         {
            ulCtlID |= TRAY_INDICATOR;
         }

         QueryContextHelp (pdi->pHelpPrnProp,
                           pdi->hwndPrnPropTxtHelp,
                           ulCtlID);
      }

      break;
   }

   case WM_INITDLG:
   {
      ULONG ulIdx;

      pdi = (PDLGINSTANCE)mp2;

      // store the pointer to instance data
      WinSetWindowULong (hwnd, QWL_USER, (ULONG)pdi);

      // We are going to fill in a lot of strings from a similar location
      // in stringtable so lets get a base index to reference from
      ulIdx = STRING_TABLE_DLG_BASE - STRING_TABLE_BASE;

      // Fill in text fields in dialogs from stringtable
      WinSetDlgItemText (hwnd, DID_ADD,    globals.pbStringTable[ulIdx + DIALOG_TEXT_ADD]);
      WinSetDlgItemText (hwnd, DID_DELETE, globals.pbStringTable[ulIdx + DIALOG_TEXT_DELETE]);
      WinSetDlgItemText (hwnd, DID_MODIFY, globals.pbStringTable[ulIdx + DIALOG_TEXT_MODIFY]);

      // Set up easy index into stringtable's context help msgs
      ulIdx = STRING_TABLE_MSG_BASE - STRING_TABLE_BASE;

      // add context help for "Tray" Printer Properties notebook page
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEFINE_TRAY,          globals.pbStringTable[ulIdx + MSG_TRAY_PAGE     ]);
      AddContextHelp (pdi->pHelpPrnProp, IDL_DEF_TRAYS,            globals.pbStringTable[ulIdx + MSG_TRAY_LIST     ]);
      AddContextHelp (pdi->pHelpPrnProp, IDE_DEF_TRAY_NAME ,       globals.pbStringTable[ulIdx + MSG_TRAY_NAME     ]);
      AddContextHelp (pdi->pHelpPrnProp, IDE_DEF_TRAY_SELECT_CMD,  globals.pbStringTable[ulIdx + MSG_TRAY_CMD      ]);
      AddContextHelp (pdi->pHelpPrnProp, IDR_DEF_TRAY_TYPE_AUTO,   globals.pbStringTable[ulIdx + MSG_TRAY_AUTOMATIC]);
      AddContextHelp (pdi->pHelpPrnProp, IDR_DEF_TRAY_TYPE_MANUAL, globals.pbStringTable[ulIdx + MSG_TRAY_MANUAL   ]);

      // Add "Add", "Delete", and "Modify" messages for "Tray" page
      AddContextHelp (pdi->pHelpPrnProp, DID_ADD    | TRAY_INDICATOR, globals.pbStringTable[ulIdx + MSG_ADD_TRAY_BUTTON]);
      AddContextHelp (pdi->pHelpPrnProp, DID_DELETE | TRAY_INDICATOR, globals.pbStringTable[ulIdx + MSG_DEL_TRAY_BUTTON]);
      AddContextHelp (pdi->pHelpPrnProp, DID_MODIFY | TRAY_INDICATOR, globals.pbStringTable[ulIdx + MSG_MOD_TRAY_BUTTON]);

#ifdef DEBUG
      WinEnableControl (hwnd, IDE_TRAY_ID, TRUE);
      WinShowWindow (WinWindowFromID (hwnd, IDE_TRAY_ID), TRUE);
#endif

      FillTrayList (hwnd, pdi, pdi->pDevice, IDL_DEF_TRAYS);

      // return since we handled init
      return 0;
   }

   case WM_CONTROL:
   {
      // get instance data
      pdi = (PDLGINSTANCE)WinQueryWindowULong (hwnd, QWL_USER);

      switch (SHORT1FROMMP (mp1))
      {
      case IDL_DEF_TRAYS:
      {
         // get currently selected Tray
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_TRAYS,
                                                 LM_QUERYSELECTION,
                                                 0L,
                                                 0L));

         // verify user selected listbox field
         if (sIdx != LIT_NONE)
         {
            // Get Item Handle (Tray ID)
            ulTrayID = (ULONG)WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_TRAYS,
                                                 LM_QUERYITEMHANDLE,
                                                 MPFROMSHORT (sIdx),
                                                 MPVOID);

            UpdateTrayDlg (hwnd, pdi, ulTrayID);
         }
         break;
      }
      }

      break;
   }

   case WM_COMMAND:
   {
      // get instance data
      pdi = (PDLGINSTANCE)WinQueryWindowULong (hwnd, QWL_USER);

      switch (SHORT1FROMMP (mp1))
      {
      case DID_ADD:
      {
         CHAR szTemp[TRAYNAME_SIZE+2];
         PSZ  pszNull = "";

         WinQueryWindowText (WinWindowFromID (hwnd, IDE_DEF_TRAY_NAME),
                             LEN_STRING-2,
                             szTemp);

         // Validate tray name is unique before saving
         if (IsTrayNameUnique (pdi, szTemp, pszNull))
         {
            AddDefinedTray (hwnd, pdi, ulTrayID);
         }
         else
         {
            WinMessageBox ((HWND)HWND_DESKTOP,
                           (HWND)NULL,
                           globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_ADD_MSG_TRAY],
                           globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_ADD_TRAY],
                           1L,
                           MB_OK | MB_INFORMATION);
         }

         break;
      }

      case DID_DELETE:
      {
         // get currently selected Tray to be deleted
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_TRAYS,
                                                 LM_QUERYSELECTION,
                                                 0L,
                                                 0L));

         // verify user selected a tray to delete
         if (sIdx != LIT_NONE)
         {
            DeleteDefinedTray (hwnd, pdi, sIdx);
         }
         else
         {
            WinMessageBox ((HWND)HWND_DESKTOP,
                           (HWND)NULL,
                           globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_SELECT_MSG_TRAY],
                           globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_TRAY],
                           1L,
                           MB_OK | MB_INFORMATION);
         }
         break;
      }

      case DID_MODIFY:
      {
         CHAR szTemp[LEN_STRING];

         // get currently selected Form to be deleted
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_TRAYS,
                                                 LM_QUERYSELECTION,
                                                 0L,
                                                 0L));

         // verify user selected a form to delete
         if (sIdx != LIT_NONE)
         {
            // Get Item Handle (FORM ID)
            ulTrayID = (ULONG)WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_TRAYS,
                                                 LM_QUERYITEMHANDLE,
                                                 MPFROMSHORT (sIdx),
                                                 MPVOID);

            // if tray is a user-defined tray we can modify it
            if (ulTrayID & DEF_TYPE_USER)
            {
               CHAR szIgnore[LEN_STRING];

               // retrieve old tray name we are modifying
               TrayNameFromID (pdi->pDriver, pdi->pDevice, szIgnore, ulTrayID);

               WinQueryWindowText (WinWindowFromID (hwnd, IDE_DEF_FORM_NAME),
                                   LEN_STRING-2,
                                   szTemp);

               // Validate new tray name is unique ignoring
               // the old name before we continue the modify oper.
               if (IsTrayNameUnique (pdi, szTemp, szIgnore))
               {
                  AddDefinedTray (hwnd, pdi, ulTrayID);

                  // Find form entry in linked list or in driver table and delete
                  bRC = DeleteUserDefinedTrayEntry (pdi->pDevice, ulTrayID);

                  // if delete was a success
                  if (bRC)
                  {
                     // remove form name from listbox
                     WinSendDlgItemMsg (hwnd,
                                        IDL_DEF_TRAYS,
                                        LM_DELETEITEM,
                                        MPFROMSHORT (sIdx),
                                        MPVOID);

                     // remove form name from listbox
                     WinSendDlgItemMsg (pdi->hwndConnectDlg,
                                        IDL_FORM_SOURCE,
                                        LM_DELETEITEM,
                                        MPFROMSHORT (sIdx),
                                        MPVOID);
                  }
                  else
                  {
                     WinMessageBox ((HWND)HWND_DESKTOP,
                                    (HWND)NULL,
                                    globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_MSG_TRAY],
                                    globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_TRAY],
                                    1L,
                                    MB_OK | MB_INFORMATION);
                  }
               }
               else
               {
                  WinMessageBox ((HWND)HWND_DESKTOP,
                                 (HWND)NULL,
                                 globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_ADD_MSG_TRAY],
                                 globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_TRAY],
                                 1L,
                                 MB_OK | MB_INFORMATION);
               }
            }
            else
            {
               WinMessageBox ((HWND)HWND_DESKTOP,
                              (HWND)NULL,
                              globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_MSG_TRAY],
                              globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_TRAY],
                              1L,
                              MB_OK | MB_INFORMATION);
            }
         }

         break;
      }

      case DID_UNDO:
      {
         // Free Current tray list
         FreeUserDefinedTrayList (pdi->pDevice);

         // read in original tray list from INI file
         bRC = ReadUserDefinedTrayList (pdi->szAppName, pdi->pDevice);
         assertF (bRC);

         // Fill in all known Trays on Tray and connection page
         FillTrayList (hwnd, pdi, pdi->pDevice, IDL_DEF_TRAYS);
         FillTrayList (pdi->hwndConnectDlg, pdi, pdi->pDevice, IDL_FORM_SOURCE);
         break;
      }

      case DID_DEFAULTS:
      {
         // Free Current tray list
         FreeUserDefinedTrayList (pdi->pDevice);

         // Fill in all known Trays on Tray and connection page
         FillTrayList (hwnd, pdi, pdi->pDevice, IDL_DEF_TRAYS);
         FillTrayList (pdi->hwndConnectDlg, pdi, pdi->pDevice, IDL_FORM_SOURCE);
         break;
      }
      }

      return 0;
   }
   }

   // default for most cases
   return WinDefDlgProc (hwnd, msg, mp1, mp2);

} /* end DefineTrayDlgProc */
