/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Omni Driver Helper Functions                            */
/* SOURCE NAME    : HELPER.C                                                */
/* AUTHOR         : MATTR & MARKH                                           */
/* DATE WRITTEN   : 9/26/93                                                 */
/* DESCRIPTION    : helper functions including INI file access helpers      */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @100713 - 10/04/94 - MFR [IBM] - Need to default to A4 for most          */
/*                                  countries but USA, Canada, and Brazil.  */
/*                                  Also default to CM not inches.          */
/* @168907 - 09/19/96 - Shih [IBM]- Validate device fonts exist             */
/* @178797 - 05/22/97 - MJH [IBM] - Need to default to A4 for Brazil        */
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/* @JPN2H99- 08/01/99 - UCC [IBMJ]- Fix duplex printing problem and H/W nup */
/*                                  problem. 2nd half'99 new device support.*/
/*                                                                          */
/****************************************************************************/
#define INCL_DEV
#define INCL_DOS
#define INCL_SPL
#define INCL_SPLDOSPRINT
#define INCL_ERRORS
#define INCL_WINSEI
#define INCL_WINERRORS
#define INCL_PM

// DLLBLD is defined to build the dll version omni.drv
#ifdef DLLBLD
#define INCL_DOSFILEMGR
#endif

#include <os2.h>

// @DBCS
#define INCL_VMANDDI
#include <ddi.h>
#include <pmddi.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include <ctype.h>
#include <string.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"

// @TTY
#define ADDR_TYPE_LPT           1
#define ADDR_TYPE_COM           2
#define ADDR_TYPE_FILE          3
#define ADDR_TYPE_OTHER         4
#define ADDR_TYPE_UNC           5
#define ADDR_TYPE_NULL         -1


/****************************************************************************/
/* PROCEDURE NAME : AllocateDeviceHandle                                    */
/* AUTHOR         : Matt                                                    */
/* DATE WRITTEN   : 09-15-95                                                */
/* DESCRIPTION    : Allocates Device handle if not already allocated.       */
/*                  Validates all function pointers and updates state       */
/*                  data.                                                   */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @157983 - Mark H. - Move "Free handle" processing here on re-allocs      */
/* @JPN2H99- UCC[IBMJ]-DJP printing problem fix. Change code not to terminat*/
/*                     job when same jobproperties within same job.         */
/****************************************************************************/
BOOL
AllocateDeviceHandle (PDDC pddc, BOOL fNewJob)
{
   PFNSUBCLASS    pDevFuncs  = NULL;
   BOOL           fOldHandle = FALSE;

   // if we have a valid Device Context
   if (pddc)
   {
      // if we have a valid Device Block and Device Instance
      if (  pddc->pdb
         && pddc->pdb->pDevice
//       && pddc->pdb->bNewJobProp                          // @JPN2H99
         )
      {
         // Retrieve pointer to table of our device's subclassed functions
         pDevFuncs = pddc->pdb->pDevice->pSubclassedFunctions;

         if (  pddc->pdb->pvDeviceCookie
            && fNewJob
            )
         {
               // Terminate job, so that resolution etc. can change
               if (  pddc->StateData.fCalledDeviceInitJob
                  && pDevFuncs->pfnTermJob
                  )
               {
                  pDevFuncs->pfnTermJob (pddc, pddc->pdb->pvDeviceCookie);
               }
         }

         if (pddc->pdb->pvDeviceCookie)
         {
            fOldHandle = TRUE;

            // @157983  - If we are "reallocating" due to DJPs we'll
            // now call the device to free old handle here instead of
            // at all the places in ESCAPE.C

            // Free the old handle, to prepare for new one
            pDevFuncs->pfnDeviceQuery ((PVOID)pddc,
                                       DEVICE_FREE_HANDLE,
                                       (PVOID)pddc->pdb->pvDeviceCookie,
                                       (BOOL)fOldHandle);
         }

         // If Device has subclassed functions including "Device Query"
         if (  pDevFuncs
            && pDevFuncs->pfnDeviceQuery
            )
         {
            PVOID    pvHandle;
            PVOID    pvOldHandle;

            // Check for unsupported Dither algorithms and replace
            if (pddc->pdb->pJobProperties->ulAlgorithm == HT_LEVEL)
            {
               // replace w/ magic squares as a default algorithm
               pddc->pdb->pJobProperties->ulAlgorithm = HT_MAGIC_SQUARES;
            }

            if (fOldHandle)
               pvOldHandle = pddc->pdb->pvDeviceCookie;
            else
               pvOldHandle = NULL;

            // Call Device Query and tell device to allocate it's handle
            pvHandle = (PVOID)pDevFuncs->pfnDeviceQuery ((PVOID)pddc,
                                                         DEVICE_ALLOC_HANDLE,
                                                         pddc->pdb->hmcbHeap,
                                                         pvOldHandle);

            pddc->pdb->pvDeviceCookie = pvHandle;
         }

         if (   fNewJob
            &&  pddc->StateData.fCalledDeviceInitJob
            &&  pDevFuncs->pfnInitJob
            )
         {
            pDevFuncs->pfnInitJob (pddc, pddc->pdb->pvDeviceCookie);
         }
      }
   }

   return FALSE;

} /* end AllocateDeviceHandle */

/****************************************************************************/
/* PROCEDURE NAME : BuildAppName                                            */
/* AUTHOR         : Monte                                                   */
/* DATE WRITTEN   : xx-xx-93                                                */
/* DESCRIPTION    : build the application name used in INI file access      */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL DRVENTRY
BuildAppName (PSZ   pszAppname,
              ULONG ulAppnameSize,
              PSZ   pszPrintername,
              PSZ   pszDrivername,
              PSZ   pszDevicename)
{

  assertF (pszPrintername);
  assertF (strlen (pszPrintername));

  assertF (pszDrivername);
  assertF (strlen (pszDrivername));

  assertF (pszDevicename);
  assertF (strlen (pszDevicename));

  assertF (ulAppnameSize >   8
                           + strlen (pszDevicename)
                           + strlen (pszPrintername)
                           + strlen (pszDrivername));

  sprintf (pszAppname,
           "PM_DD_%s,%s.%s",
           pszPrintername,
           pszDrivername,
           pszDevicename);

  return TRUE;

} /* end BuildAppName */

/****************************************************************************/
/* PROCEDURE NAME : GetCountryInfo                                          */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 01-21-95                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL DRVENTRY
GetCountryInfo (PCOUNTRYCODE pcc, PCOUNTRYINFO pci)
{
   ULONG        ulcb;
   ULONG        cCP;
   APIRET       apiret;

   // @100713 - need to get country code to determine what default
   // paper size to get (A4 is default in some countries, Letter
   // in others) also used for CM or Inch measures
   pcc->country = PrfQueryProfileInt (HINI_USERPROFILE,
                                      APPNAME_COUNTRYCODE,
                                      KEYNAME_COUNTRYCODE,
                                      COUNTRYCODE_USA);

   // Known OS/2 Codepages for OS/2 Warp are:
   //
   // 437 = United States
   // 850 = Multilingual
   // 852 = Latin 2 (Czech, Hung, Poland, Yugo)
   // 857 = Turkish
   // 860 = Potuguese
   // 861 = Iceland
   // 863 = Canadian French
   // 865 = Nordic
   // 932 = Japan
   // 934 = Korea
   // 936 = P.R. of China
   // 938 = Taiwan
   // 942 = Japan SAA
   // 944 = Korea SAA
   // 946 = P.R. of China SAA
   // 943 = Japan (future codepage)

   apiret = DosQueryCp (sizeof (pcc->codepage), &pcc->codepage, &cCP);

   // We only want current process primary codepage, therefore
   // we except NO_ERROR and ERROR_CPLIST_TOO_SMALL.  This means
   // that we'll get codepage 437 for US but still get the return
   // code ERROR_CPLIST_TOO_SMALL since codepage 850 is available.
   if (apiret == ERROR_CPLIST_TOO_SMALL || apiret == NO_ERROR)
   {
      apiret = DosGetCtryInfo (sizeof (COUNTRYINFO), pcc, pci, &ulcb);

      if (apiret == NO_ERROR)
      {
         DBPRINTF (("%s(): PCC: CodePage=%d, Country=%d\n",
                    __FUNCTION__, pcc->codepage, pcc->country));
         DBPRINTF (("%s(): PCI: CodePage=%d, Country=%d\n",
                    __FUNCTION__, pci->codepage, pci->country));

        return TRUE;
      }
#if DEBUG
      else
      {
         DBPRINTF (("%s(): DosGetCtryInfo failed rc=%d, default to 850\n",
                    __FUNCTION__, apiret));
      }
#endif
   }
#if DEBUG
   else
   {
      DBPRINTF (("%s(): DosQueryCp failed rc=%d, default to 850\n",
                 __FUNCTION__, apiret));
   }
#endif

  // Apply some European defaults, since something failed
  pci->country = 850;
  pci->szDecimal[0] = ',';

  return FALSE;

} /* end GetCountryInfo */

/****************************************************************************/
/* PROCEDURE NAME : GetDriverDriveAndPath                                   */
/* AUTHOR         : Matt                                                    */
/* DATE WRITTEN   : 08-02-94                                                */
/* DESCRIPTION    : Retrieves Drivers Full path and Drive letter w/o actual */
/*                  driver name for use in finding files relative to the    */
/*                  current installed directory                             */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
GetDriverDriveAndPath (PSZ pszDriveAndPath)
{
   PSZ pszTemp;

   DBPRINTF (("%s(): globals.szModule = '%s'\n", __FUNCTION__, globals.szModule));

   strcpy (pszDriveAndPath, globals.szModule);

   DBPRINTF (("%s(): pszDriveAndPath  = '%s'\n", __FUNCTION__, pszDriveAndPath));

   pszTemp = strchr (pszDriveAndPath, '.');
   DBPRINTF (("%s(): pszTemp = '%s'\n", __FUNCTION__, pszTemp));

   pszTemp++;
   DBPRINTF (("%s(): pszTemp = '%s'\n", __FUNCTION__, pszTemp));

   strcpy (pszTemp, "HLP");
   DBPRINTF (("%s(): pszTemp = '%s'\n", __FUNCTION__, pszTemp));

   DBPRINTF (("%s(): pszDriveAndPath  = '%s'\n", __FUNCTION__, pszDriveAndPath));

   return TRUE;

} /* end GetDriverDriveAndPath */

/****************************************************************************/
/* PROCEDURE NAME : SetDriverDataDefaults                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10-28-93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/*                                                                          */
/****************************************************************************/
BOOL DRVENTRY
SetDriverDataDefaults (PDEVICEINFO    pDevice,
                       PJOBPROPERTIES pJobProperties,
                       PSZ            pszDeviceName,
                       ULONG          ulCountryCode)
{
   BOOL        bRC = TRUE;
   ULONG       cb,
               cb2;                                          // @EXJOB
#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
///pDbg->bSETDRIVERDATADEFAULTS;
#endif

   // validate parms as valid pointers
   assertF (pDevice);
   assertF (pJobProperties);

   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "%s(): Enter\n", __FUNCTION__));

   // Clear out the memory
   memset (pJobProperties, 0, sizeof (JOBPROPERTIES));

   // Copy the properties structure in the pDevice block
   *pJobProperties = pDevice->DeviceDefaults;

   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS,
               "%s(): Setting Metric Conn ID = '%d'\n",
               __FUNCTION__,
               pJobProperties->ulDefConnID));

   // Change default form connection if country code
   // indicates a "non-metric" country (i.e. "Letter" paper
   // not "A4" paper size)
   if (  ulCountryCode == COUNTRYCODE_USA
      || ulCountryCode == COUNTRYCODE_CANADA
      )
   {
      pJobProperties->ulDefConnID = pJobProperties->ulDefNonMetricConnID;
      DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS,
                  "%s(): Replacing with Non-metric Conn ID = '%d'\n",
                  __FUNCTION__,
                  pJobProperties->ulDefConnID));
   }

   // Overwrite the first two fields with the proper data since it is not
   // required that they be filled in in the device declaration.

   // @EXJOB
   //pJobProperties->cb = sizeof (JOBPROPERTIES);
   SafeStrNCpy (pJobProperties->achDriver,
                APP_NAME,
                sizeof (pJobProperties->achDriver));
   pJobProperties->ulSig = JOBPROP_SIG;

// begin @EXJOB
   cb  = GetJobpropertiesSize (pDevice, JP_SIZE_ALL);
   cb2 = GetJobpropertiesSize (pDevice, JP_SIZE_PLUGIN);

   // set current version cb size as default
   pJobProperties->cb = cb;

   // Check if there is enough room for device jobdata
   if (cb2 <= (cb - sizeof (JOBPROPERTIES)))
   {
      PFNSUBCLASS ppfnDevice = pDevice->pSubclassedFunctions;

      if (  ppfnDevice
         && ppfnDevice->pfnDeviceQuery
         )
      {
         PBYTE pbJobDevice = (PBYTE)pJobProperties + cb - cb2;

         memset (pbJobDevice, 0, cb2);

         ppfnDevice->pfnDeviceQuery (pDevice,
                                     DEVICE_QUERY_EXTJOBDATA,
                                     &cb2,
                                     pbJobDevice,
                                     DEVICE_EXTJOBDATA_SET_DEFAULT);
      }
   }
// end @EXJOB
#ifdef DEBUG
   DebugOutputJobProperties (pDbg->bSETDRIVERDATADEFAULTS,
                             "Job Properties are set to:",
                             &pDevice->DeviceDefaults);
#endif
   
   // We validate and initialize Layout control block
   GplLayoutInitCB( &(pJobProperties->LayoutCB), GL_SET_DEFAULT_LAYOUT,
                     (ULONG) 0X00);

   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS,
               "%s(): Exit; bRC = %d\n", __FUNCTION__,
               bRC));

   return bRC;

} /* end SetDriverDataDefaults */

BOOL DRVENTRY
SetPrinterPropDefaults (PDEVICEINFO    pDevice,
                        PPRNPROPERTIES pPrnProperties,
                        PSZ            pszDeviceName)
{
   BOOL        bRC  = TRUE;
#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   // validate parms as valid pointers
   assertF (pDevice);
   assertF (pPrnProperties);

   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "SetPrinterPropDefaults(): Enter\n"));

// pPrnProperties->cb = sizeof (JOBPROPERTIES);             // @JPN2H99
   pPrnProperties->cb = sizeof (PRNPROPERTIES);             // @JPN2H99
   SafeStrNCpy (pPrnProperties->achDriver,
                APP_NAME,
                sizeof (pPrnProperties->achDriver));
   pPrnProperties->ulSig = PRINTPROP_SIG;

   pPrnProperties->ulInstalledMemory = pDevice->pMemory->ulBase;

   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "Printer Properties are set to:\n"));
   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "------------------------------\n"));
   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "ulInstalled Memory = %lu\n", pPrnProperties->ulInstalledMemory));
   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "------------------------------\n"));

   DBPRINTIF ((pDbg->bSETDRIVERDATADEFAULTS, "SetPrinterPropDefaults(): Exit; bRC = %d\n", bRC ));

   return bRC;

} /* end SetPrinterPropDefaults */

/****************************************************************************/
/* PROCEDURE NAME : FindDeviceNameInDriver                                  */
/* AUTHOR         : Mark H                                                  */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL DRVENTRY
FindDeviceNameInDriver (PSZ          pszDeviceName,
                        PDRIVERINFO *pDriverMatch,
                        PDEVICEINFO *pDeviceMatch)
{
   register INT iDriver, iDevice;
   PDEVICEINFO  pDevice;

   // validate input parms before dereference
   assertF (pszDeviceName);
   assertF (*pszDeviceName);

   if (pszDeviceName && *pszDeviceName)
   {
      // search through all driver structs we know about
      for (iDriver = 0; iDriver < globals.ulTotalDrivers; iDriver++)
      {
         // search through all device structs within this driver struct
         for (iDevice = globals.ppDrivers[iDriver]->ulNumDevices,
              pDevice = globals.ppDrivers[iDriver]->pDevices;
              iDevice;
              iDevice--, pDevice++)
         {
            // compare device name strings for a match
            if (0 == strcmp (pDevice->pszDeviceName, pszDeviceName))
            {
               // Matched device name passed in to one supported

               // If requested we can supply the pDevice at this time
               if (pDriverMatch)
                  if (*pDriverMatch == NULL)
                     *pDriverMatch = globals.ppDrivers[iDriver];
                  else
                     DBPRINTF (("FindDeviceNameInDriver: pDriver Already filled!\n"));

               // If requested we can also supply the pDevice at this time
               if (pDeviceMatch)
                  if (*pDeviceMatch == NULL)
                     *pDeviceMatch = pDevice;
                  else
                     DBPRINTF (("FindDeviceNameInDriver: pDevice Already filled!\n"));

               // Search for the device name was successful
               return TRUE;
            }
         }
      }
   }

   // Report to debugger the failure to find the device name
   DBPRINTF (("FindDeviceNameInDriver: '%s' is not a valid device name!\n",
               pszDeviceName));

   // Search for this device name failed
   return FALSE;

} /* end FindDeviceNameInDriver */

/***************************************************************************
 *
 * FUNCTION NAME = ReturnDriverData
 *
 * DESCRIPTION   = This takes a copy of the driver data that we are not sure
 *                 about (from the application) and returns a copy that we
 *                 are sure about.
 *
 * This function will return valid driver data.  It does this by allocating
 * a copy of driver data and initializing it to its default state.  It is the
 * caller's responsibility to free this data.  If driver data was given to
 * us, then we will copy over that data onto our initialized data up to either
 * the size that they gave us or our current size.
 *
 * NOTE:
 *    We should always have a DRIVDATA passed to us.  If the application did
 *    not give us any, then the spooler will give us the queue's job
 *    properties (DevOpenDC calls SplQmSetup which verifys the parameters).
 *
 * INPUT         = pvHeap              - a heap
 *                 pInput              - input job properties
 *                 pszLogicalAddress   - the logical address that we are
 *                                       printing to
 *                 fDevPostDeviceModes - Did DevPostDeviceModes call us?
 *
 * OUTPUT        = PDRIVDATA - A copy of job properties that we are sure about
 *
 * @TBD - use Appname built here and fill in pdb if present and change
 *        enable.c to use ours if provided - mfr
 * @JPN2H99- 08/01/99 - UCC [IBMJ]- Printer properties option page support.
 **************************************************************************/
PDRIVDATA
ReturnDriverData (PDEVICEBLOCK pdb,
                  PDLGINSTANCE pdi,
                  HMCB         hmcbHeap,
                  PDRIVDATA    pInput,
                  PSZ          pszLogicalAddress,
                  BOOL         fDevPostDeviceModes,
                  PDEVICEINFO  pDevicedummy,
                  PDRIVERINFO  pDriverdummy)

// pDevicedummy and pDriverdummy only used by DLLBLD version of omni

{
   USHORT       rc;
   PDRIVDATA    pOutput                  = NULL; /* Driver definition        */
   PDRIVERINFO  pDriver                  = NULL; // move here don't call
   PDEVICEINFO  pDevice                  = NULL; // move here don't call
   PSZ          pszPrinterName           = NULL;
   PSZ          pszAppName               = NULL;
   ULONG        ulMemorySize             = 0L;   /* receive size of segment  */
   ULONG        ulFlags;
   ULONG        ulSizeForName;
   BOOL         fMustInitializeToUnknown = TRUE;
   BOOL         fCanCopy                 = FALSE;
   CHAR         achDeviceName[33];
   PSZ          pszGoodDeviceName        = NULL;
   ULONG        ulCountryCode;
   PPRNPROPERTIES pPrnProperties;                           // @JPN2H99

// begin @EXJOB
   ULONG        ulPropertySize;
   ULONG        ulDrivDataSize;

   if (fDevPostDeviceModes)
   {
      pDevice        = pdi->pDevice;
      pPrnProperties = &pdi->PrnProperties;                 // @JPN2H99
   }
   else
   {
      pDevice        = pdb->pDevice;
      pPrnProperties = pdb->pPrnProperties;                 // @JPN2H99
   }

   /* @begin 221528
   ** We need a device to get device job properties
   */
   if (!pDevice)
   {
      if (pInput)
      {
         /* There are job properties to read.
         ** The question is can we access the device name?
         */
         ulMemorySize = (PCHAR)pInput->abGeneralData-(PCHAR)pInput;
         rc = DosQueryMem (pInput, &ulMemorySize, &ulFlags);

         if (  !rc                                                // Success
            && (!(ulFlags & (PAG_FREE   |          PAG_GUARD)))   // Things we don't want
            && ( (ulFlags & (PAG_COMMIT | PAG_READ          )))   // Things we do want
            )
         {
            /* At least we know that we have data to look at.
            ** Now, see if we were given enough to look at the device name.
            */
            ulSizeForName = (PCHAR)pInput->abGeneralData-(PCHAR)pInput;

            // See if we can look at the device name in the job properties
            if (  (ulMemorySize >= ulSizeForName)
               && (pInput->cb   >= ulSizeForName)
               )
            {
               /* We can see the name */
               if (FindDeviceNameInDriver (pInput->szDeviceName, NULL,NULL))
               {
                  /* We were given a valid device name.                  */
                  pDevice = PDeviceFromDeviceName (pInput->szDeviceName);
               }
            }
         }
      }

      if (!pDevice)
      {
         /* There is no input job properties or we could not read the
         ** device name from the input job properties.
         ** Try to find the device name and printer name
         ** from the logical address instead...
         */
         FindDeviceName (hmcbHeap,
                         achDeviceName,
                         pszLogicalAddress,
                         fDevPostDeviceModes);

         // Did we return a valid name for our driver?
         if (  *achDeviceName
            && FindDeviceNameInDriver (achDeviceName, NULL, NULL)
            )
         {
            /* We were given a valid device name. */
            pDevice = PDeviceFromDeviceName (achDeviceName);
            pszGoodDeviceName = achDeviceName;
            fMustInitializeToUnknown = FALSE;
         }
         else
         {
            DBPRINTF (("ReturnDriverData: We must initialize to an unknown device!\n"));

            // Default to first device we find in this driver
            // TBD: at least try to init to device of correct class (i.e HP, Canon,
            // Epsn etc)
            pDevice = PDeviceFromDeviceName(globals.ppDrivers[0]->pDevices->pszDeviceName);
         }
      }

      if (!pDevice)
      {
         DBPRINTF (("ReturnDriverData: Still no device found. Critical error!\n"));

         GplErrSetError (PMERR_DEVICE_DRIVER_ERROR_1);
      }
   }
   // @end 221528

   if (pDevice)
   {
      ulPropertySize = GetJobpropertiesSize (pDevice, JP_SIZE_ALL);
   }
   else
   {
      // Sorry! we can't get proper size.
      ulPropertySize = sizeof (JOBPROPERTIES);
   }

   if (fDevPostDeviceModes)
   {
      pdi->ulJobProperties = ulPropertySize;
   }

   ulDrivDataSize = DRIVDATA_HEADER + ulPropertySize;

   pOutput = (PDRIVDATA)GplMemoryAlloc (hmcbHeap, ulDrivDataSize);

   if (!pOutput)
      return pOutput;

   pOutput->cb       = ulDrivDataSize;
   pOutput->lVersion = DRIVERDATA_VERSION_CURRENT;

// Moved to after checking pInput
// pOutput = (PDRIVDATA)GplMemoryAlloc (hmcbHeap, DRIVERDATA_SIZE);
//
// if (!pOutput)
//    return pOutput;
//
// // Set up the driver data that we have just allocated.
// // Initialize current size and version fields
// // The 'szDeviceName' field will be filled in when we know what the name is.
// // The 'abGeneralData' field will be filled in after we know the device name.
// pOutput->cb       = DRIVERDATA_SIZE;
// pOutput->lVersion = DRIVERDATA_VERSION;
// end @EXJOB

   // See how much of the old data we can copy, unless the given pointer is NULL
   if (pInput)
   {
      // Find out how big the memory is.  Keep in mind that for tiled
      // memory this value will be 65536.
// begin @EXJOB
      ulMemorySize = ulDrivDataSize;
//    ulMemorySize = DRIVERDATA_SIZE;
// end @EXJOB

      // Get size of DriverData memory block passed in to us by application
      // NOTE: Be careful using DosQueryMem().  From testing it only seems to
      //       work correctly if the memory was allocated using DosAllocMem().
      rc = DosQueryMem (pInput, &ulMemorySize, &ulFlags);

      if (  !rc
#if 0
          /* The headers (bsememf.h) define a flag fPERM which is
          ** (PAG_EXECUTE | PAG_READ | PAG_WRITE).  Lotus 123G (and
          ** probably other apps) use this flag to allocate their data,
          ** so ignore PAG_EXECUTE!
          */
         && (!(ulFlags & (PAG_FREE   | PAG_EXECUTE | PAG_GUARD)))   // Things we don't want
#else
         && (!(ulFlags & (PAG_FREE   |               PAG_GUARD)))   // Things we don't want
#endif
         && ( (ulFlags & (PAG_COMMIT | PAG_READ               )))   // Things we do want
         )
      {
         if (pInput->cb)
         {
//            fCanCopy = TRUE;

            /* At least we know that we have data to copy.
            ** Now, see if we were given enough to look at the device name.
            */
            ulSizeForName = (PCHAR)pInput->abGeneralData-(PCHAR)pInput;

            // See if we can look at the device name in the job properties
            if (  (ulMemorySize >= ulSizeForName)
               && (pInput->cb   >= ulSizeForName)
               )
            {
               /* We can see the name */
               if (FindDeviceNameInDriver (pInput->szDeviceName, NULL, NULL))
               {
                  /* We were given a valid device name.
                  */
                  fMustInitializeToUnknown = FALSE;
                  pszGoodDeviceName = pInput->szDeviceName;
                  fCanCopy = TRUE; // We can try to copy
               }
            }
               
            // See if we can look at the entire thing
            
            //For Network: At 1st the input Job Properties length may be 
            //incorrect for old driver (before @236580 defect fix it was not 
            //rewritten in SafeCopyJobData()). At 2nd if input Job Properties
            //are smaller then Ouput so for a Job Properties rest the validiation 
            //and initialization are incorrect. Let's do in beginning safe copy
            //input job data to output, then the validiation and initialization 
            //of them and if they are incorrect return Default Job Properties 
            //(see changes below) 
//            if (  pszGoodDeviceName
//               && (ulMemorySize >= ulSizeForName)
//               && (pInput->cb   >= ulSizeForName)
//               )
//            {
//               pDriver = PDriverFromDeviceName (pszGoodDeviceName, &pDevice);
//               assertF (pDriver);
//
//               fCanCopy = ValidateDriverData ((PJOBPROPERTIES)(&pInput->abGeneralData),
//                                              pPrnProperties,
//                                              pDriver,
//                                              pDevice);
//            }
         }
      }

      /* Here, we were given job properties.  If the above tests failed,
      ** then post a warning.
      */
      if (FALSE == fCanCopy)
         GplErrSetWarning (PMERR_INV_DRIVER_DATA);
   }

   /* See if we were not given a device name.
   */
   if (fMustInitializeToUnknown)
   {
      DBPRINTF (("ReturnDriverData: We must initialize to an unknown device!\n"));

      /* Try to find that device name and printer name
      ** from the logical address...
      */
      FindDeviceName (hmcbHeap,
                      achDeviceName,
                      pszLogicalAddress,
                      fDevPostDeviceModes);

      // Did we return a valid name for our driver?
      if (  *achDeviceName
         && FindDeviceNameInDriver (achDeviceName, NULL, NULL)
         )
      {
         pszGoodDeviceName = achDeviceName;
      }
      else
      {
         GplErrSetError (PMERR_DEVICE_DRIVER_ERROR_1);
      }
   }

   // By now we *should* have a device name
   if (pszGoodDeviceName)
   {
      strcpy (pOutput->szDeviceName, pszGoodDeviceName);

      /* Find the spooler's printer name */
      pszPrinterName = FindPrinterName (hmcbHeap,
                                        pszGoodDeviceName,
                                        pszLogicalAddress,
                                        fDevPostDeviceModes);
      assertF (pszPrinterName);

      if (pszPrinterName)
      {
         BOOL  bRC;
// begin @EXJOB
         ULONG          ulRead;
// end @EXJOB

         // allocate from heap; conserve stack space
         pszAppName = GplMemoryAlloc (hmcbHeap, LEN_APPNAME);
         assertF (pszAppName);

         // build the INI file Application name for accessing the INI file for this device
         bRC = BuildAppName (pszAppName,
                             LEN_APPNAME,
                             pszPrinterName,
                             APP_NAME,
                             pszGoodDeviceName);
         assertF (bRC);

         // If we don't have a printer name or
         // we have a printer name but could not find default job properties
         // Then we must initialize Driver Data to default values

// begin @EXJOB
         ulRead = ulPropertySize;
         bRC = ReadDefaultJobProperties (pszAppName,
                                         (PVOID)&pOutput->abGeneralData,
                                         &ulRead);
// end @EXJOB

         if (  ((PSZ)NULL == pszPrinterName)
// begin @EXJOB
            || !bRC
            || (ulRead != ulPropertySize)
//          || !ReadDefaultJobProperties (pszAppName, (PVOID)&pOutput->abGeneralData)
// end @EXJOB
            )
         {
            // If we don't have a device name at this point get one
            if (pDevice == NULL)
            {
               pDevice = PDeviceFromDeviceName (pszGoodDeviceName);
               assertF (pDevice);
            }

            // @100713 - need to get country code to determine what default
            // paper size to get (A4 is default in some countries, Letter
            // in others) - MFR
            if (fDevPostDeviceModes)
            {
               // We can use country info in the dialog instance data
               ulCountryCode = pdi->ctryInfo.country;
            }
            else
            {
               // We can use country info in the physical device block
               ulCountryCode = pdb->ctryInfo.country;
            }

            // Call function to get default Driver Data (job Proprties)
            SetDriverDataDefaults (pDevice,
                                   (PJOBPROPERTIES)(&pOutput->abGeneralData),
                                   pszGoodDeviceName,
                                   ulCountryCode);
         }

#ifndef DEBUG
      bRC++;          // Avoid compiler warnings
#endif
      }
   }

   if (!pszGoodDeviceName)
   {
      /* Big time failure!  We do not know what device to initialize our
      ** job properties to.  What can we do now???
      ** Maybe we should default to our first device...
      */
      GplErrSetError (PMERR_DEVICE_DRIVER_ERROR_1);

      assertstring ("***ERROR***ReturnDriverData: Initialising to %s\n");

      // Default to first device we find in this driver
      // TBD: at least try to init to device of correct class (i.e HP, Canon,
      // Epsn etc)
      pDevice = PDeviceFromDeviceName (globals.ppDrivers[0]->pDevices->pszDeviceName);

      SetDriverDataDefaults (pDevice,
                             (PJOBPROPERTIES)(&pOutput->abGeneralData),
                             pszGoodDeviceName,
                             COUNTRYCODE_USA);

      strcpy (pOutput->szDeviceName, globals.ppDrivers[0]->pDevices[0].pszDeviceName);
   }

   // Shall we overwrite what we have initialized with what was given to us?
   if (fCanCopy)
   {
      
      //Properties length by ourselves (in case incorrect length), do SafeCopy
      //JobData(Input to Output), validate them and if they are incorrect return 
      //the Output back
      PDRIVDATA    pDefaultOutput           = NULL; /* Driver default definition */
      PJOBPROPERTIES pInputJobProp = (PJOBPROPERTIES)(&pInput->abGeneralData);
               
      pDefaultOutput = (PDRIVDATA)GplMemoryAlloc (hmcbHeap, ulDrivDataSize);
      assertF (pDefaultOutput);
      
      if (pDefaultOutput)
         memcpy ((PBYTE)pDefaultOutput, (PBYTE)pOutput, ulDrivDataSize);
      
      if ((pInput->cb - ulSizeForName) < pInputJobProp->cb)
         pInputJobProp->cb = pInput->cb - ulSizeForName;
               
      /* Don't copy more than the size of out driver data
      ** OR is specified in the structure
      ** OR that we can access!
      */
// begin @EXJOB
//    min = ((USHORT)pInput->cb <= (USHORT)ulMemorySize) ? (USHORT)pInput->cb
//                                                       : (USHORT)ulMemorySize;
//    min = (DRIVERDATA_SIZE <= min) ? DRIVERDATA_SIZE : min;
//    memcpy ((PBYTE)pOutput, (PBYTE)pInput, min);

      // Copy Driver Data Header
      memcpy ((PBYTE)pOutput, (PBYTE)pInput, DRIVDATA_HEADER);

      SafeCopyJobData (pDevice,
                       (PBYTE)&pInput->abGeneralData,
                       ulPropertySize,
                       (PBYTE)&pOutput->abGeneralData);

// end @EXJOB

      /* Now, change some of the information back to our current settings.
      */
// begin @EXJOB
      pOutput->cb       = ulDrivDataSize;
      pOutput->lVersion = DRIVERDATA_VERSION_CURRENT;
//    pOutput->cb       = DRIVERDATA_SIZE;
//    pOutput->lVersion = DRIVERDATA_VERSION;
// end @EXJOB

      pDriver = PDriverFromDeviceName (pszGoodDeviceName, &pDevice);
      assertF (pDriver);

      fCanCopy = ValidateDriverData ((PJOBPROPERTIES)(&pOutput->abGeneralData),
                                      pPrnProperties,
                                      pDriver,
                                      pDevice);
                                      
      if (!fCanCopy && pDefaultOutput) //Ops! We could not copy !
         memcpy ((PBYTE)pOutput, (PBYTE)pDefaultOutput, ulDrivDataSize);

      // Free the DefaultOutput
      GplMemoryFree (pDefaultOutput);
   }

   // If we have a device block structure to fill in
   // If we have made successful calls to FindPrinterName()
   // PDriverFromDeviceName()
   // we should place these in our device block so we do
   // not make redundant calls later in OS2_PM_DRV_ENABLE
   // NOTE: for OS2_PM_DRV_DEVMODE we have no pdb to fill in
   if (pdb)
   {
      // If we found a matching device we found the driver too
      if (pDriver)
      {
         pdb->pDriver = pDriver;
      }

      // If we found a matching device
      if (pDevice)
      {
         pdb->pDevice = pDevice;
      }

      // If FindPrinterName() called successfully
      if (pszPrinterName)
      {
         if (pdb->pszPrinterName)
            // Free the existing string
            GplMemoryFree (pdb->pszPrinterName);

         pdb->pszPrinterName = pszPrinterName;
      }
   }
   else
   {
      if (pszPrinterName)
         GplMemoryFree (pszPrinterName);
   }

   /* Clean up */
   if (pszAppName)
      GplMemoryFree (pszAppName);

   // NOTE: caller MUST free pOutput
   return pOutput;

} /* end ReturnDriverData */

/****************************************************************************/
/* PROCEDURE NAME : PDeviceFromDeviceName                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10-28-93                                                */
/* DESCRIPTION    : return the pointer to the global structure PDEVICEINFO  */
/*                  given a device name.                                    */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
PDEVICEINFO DRVENTRY
PDeviceFromDeviceName (PSZ pszDeviceName)
{
   register INT iDriver, iDevice;
   PDEVICEINFO  pdiMatch;
   PDEVICEINFO  pDevice;
#ifdef DEBUG
   PDEBUGINFO   pDbg = &globals.DebugInfo;
#endif

  // validate input parms before dereference
  assertF (pszDeviceName);
  assertF (*pszDeviceName);

  // @TBD - if pszDevicename == NULL we need way to default
  DBPRINTIF ((pDbg->bPDEVICEFROMDEVICENAME,
              "PDeviceFromDeviceName(): Enter; pszDeviceName = %s\n",
              pszDeviceName));

  // Initilaize return value
  pdiMatch = NULL;

  for (iDriver = 0; iDriver < globals.ulTotalDrivers; iDriver++)
  {
     for (iDevice = globals.ppDrivers[iDriver]->ulNumDevices,
             pDevice = globals.ppDrivers[iDriver]->pDevices;
          iDevice;
          iDevice--, pDevice++)
     {
         // compare device name strings for a match
         // @TBD - may want to create a no-case compare ???
         if (0 == strcmp (pDevice->pszDeviceName, pszDeviceName))
         {
            // Found a device name supported by this driver that
            // matches one passed in
            pdiMatch = pDevice;

            // get out of loop
            return pdiMatch;
        }
     }
  }

  // @TBD - if device name is NULL or not found perhaps popup a
  // list of supported devices to choose from for user - MFR, MARKV

  DBPRINTIF ((pDbg->bPDEVICEFROMDEVICENAME,
              "PDeviceFromDeviceName(): Exit; pdiMatch = %x\n",
              pdiMatch));

  return (PDEVICEINFO)NULL;

} /* end PDeviceFromDeviceName */

PDRIVERINFO DRVENTRY
PDriverFromDeviceName (PSZ pszDeviceName, PDEVICEINFO *pDeviceMatch)
{
   register INT iDriver, iDevice;
   PDRIVERINFO  pDriverMatch = (PDRIVERINFO)NULL;
   PDEVICEINFO  pDevice;
#ifdef DEBUG
   PDEBUGINFO   pDbg         = &globals.DebugInfo;
   BOOL         bShow        = pDbg->bPDRIVERFROMDEVICENAME;
#endif

   DBPRINTIF ((bShow,
               "PDriverFromDeviceName(): Enter; pszDeviceName = %s\n",
               pszDeviceName));

   // validate input parms before dereference
   assertF (pszDeviceName);
   if (pszDeviceName)
   {
      assertF (*pszDeviceName);
   }

   // search through all drivers (and devices) for matching device name
   for (iDriver = 0; iDriver < globals.ulTotalDrivers; iDriver++)
   {
      for (iDevice = globals.ppDrivers[iDriver]->ulNumDevices,
              pDevice = globals.ppDrivers[iDriver]->pDevices;
           iDevice;
           iDevice--, pDevice++)
      {
         // compare device name strings for a match
         // @TBD - may want to create a no-case compare ???
         if (0 == strcmp (pDevice->pszDeviceName, pszDeviceName))
         {
            // Found a device name supported by this driver that
            // matches one passed in
            pDriverMatch = globals.ppDrivers[iDriver];

            // If requested we can also supply the pDevice at this time
            if (pDeviceMatch)
            {
                *pDeviceMatch = pDevice;
            }

            // get out of loop
            return pDriverMatch;
         }
      }
   }

   // @TBD - if pszDevicename == NULL we need way to default
   // @TBD - if device name is NULL or not found perhaps popup a
   // list of supported devices to choose from for user - MFR, MARKV

   DBPRINTIF ((bShow,
               "PDriverFromDeviceName(): Exit; pDeviceMatch = %x\n",
               pDeviceMatch));

   return (PDRIVERINFO)NULL;

} /* end PDriverFromDeviceName */

//---------------------------------------------------------------------------------------------------------------------
// strip from the ends of string given in parameter 1 all characters in string in parameter 2
// strip left, right, or both according to pszOption which can be "LEFT", "RIGHT", or "BOTH"
// similar in syntax and function to Rexx strip(), except this function modifies the string at psz
// where rexx returns a new string and leaves the one passed as an argument intact.

PSZ DRVENTRY
Strip (PSZ psz, PSZ pszOption, PSZ pszStripSet)
{
   assertF (psz);
   assertF (pszOption);
   assertF (strchr ("LRBlrb", *pszOption));
   assertF (pszStripSet);
   assertF (strlen (pszStripSet));

   if (strchr( "LlBb", *pszOption))
   {
      // strip on left
      while (*psz && strchr (pszStripSet, *psz))
      {
         strcpy (psz, psz+1);
      }
   }

   if (strchr ( "RrBb", *pszOption))
   {
      // strip on right
      while (*psz && strchr (pszStripSet, *(psz + strlen (psz) - 1)))
      {
         *(psz+strlen(psz)-1) = 0;
      }
   }

   return psz;
}

#if 0

/****************************************************************************/
/* PROCEDURE NAME : StrToken                                                */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Copies substring from string2 to string1.  The          */
/*                  substring to be copied is identified by its position    */
/*                  of occurrence in string1 as separated by a token.       */
/*                  The copy will proceed up to the next token or an        */
/*                  end-of-string character is encountered                  */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Advance position in string2 to requested occurrence */
/*                      (occ) of the token (tok).                           */
/*                  (b) Copy string2 to string1 until token or end of       */
/*                      string found.                                       */
/*                  (c) Always return a valid NULL terminated string even   */
/*                      on error.                                           */
/*                                                                          */
/* PARAMETERS:      (CHAR *) string1 - destination string                   */
/*                  (CHAR *) string2 - source string                        */
/*                  (CHAR)   tok     - token used as occurrence separator   */
/*                  (USHORT) occ     - occurrence of substring to copy      */
/*                                                                          */
/* RETURN VALUES:   (BOOL) ErrorFound == TRUE  - Failure, error found       */
/*                                    == FALSE - Success, no errors         */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
StrToken (CHAR *string1, CHAR *string2, CHAR tok, USHORT occ)
{
   USHORT i          = 0,
          j          = 0;
   BOOL   ErrorFound = FALSE;

   occ--;

   while (occ > 0 && string2[i] != '\0')
   {
      if (string2[i] == tok)
      {
         occ--;
      }

      i++;
   }

   if (string2[i] != '\0')
   {
      while (string2[i] != tok && string2[i] != '\0')
      {
         string1[j] = string2[i];

         i++;
         j++;
      }
      string1[j] = '\0';
   }
   else
   {
      string1[j] = '\0';
      ErrorFound = TRUE;
   }

   return ErrorFound;

} /* end StrToken */

#endif

/****************************************************************************/
/* Function:    ConvertHMMtoPels                                            */
/* AUTHOR  :    MFR                                                         */
/*                                                                          */
/* Description: Takes a value in hundreds of Millimeters to Pels based on   */
/*              dpi resolution passed in.                                   */
/*                                                                          */
/*                                                                          */
/* Returns:     VOID                                                        */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*                                                                          */
/* TAG      DATE        DEVELOPER [COMPANY]   DESCRIPTION                   */
/* ---      ----        -------------------   -----------                   */
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
VOID
ConvertHMMtoPels (ULONG  ulHMMValue,
                  ULONG  ulDpi,
                  PULONG pulPelValue,
                  BOOL   bRoundUp)
{
   double  dblInchValue;
   double  dblInches;

///DBPRINTF (("ConvertHMMtoPels(): Enter\n"));

   dblInches = ((double)ulHMMValue) / ((double)HMM_PER_INCH);

   if (bRoundUp)
      dblInchValue = dblInches + (double)0.9;
   else
      dblInchValue = dblInches;

   *pulPelValue = (ULONG)(dblInchValue * (double)ulDpi);

///DBPRINTF (("Inch value = %lf pel value = %d\n", dblInchValue, *pulPelValue));
///DBPRINTF (("ConvertHMMtoPels(): Exit\n"));

} /* end ConvertMMtoPels */

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
ULONG
RoundDivide (ULONG ulVal, ULONG ulDivide)
{
   // Round up..
   ulVal += ulDivide - 1;

   return ulVal/ulDivide;
}

/***************************************************************************
 *
 * FUNCTION NAME = newton f_lsqrt
 *
 * DESCRIPTION   = Evaluate a square root using Newton's approximation
 *                 method.  Uses a fixed number of guesses.
 *
 * INPUT         = ulIn - integer to square root
 *
 * OUTPUT        = A fixed point representation of the square root.
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/
ULONG
newton_f_lsqrt(ULONG ulIn)
{
   /*
   **    Returns an approximation to the square root of ulIn.  Value is
   **  as a fixed point number (16 bits of integer, 16 bits of fraction).
   */
   ULONG ulGuess;

   if (ulIn == 0)
      return 0;                          /* Easy one!                         */

   ulGuess = (ulIn >> 16) + 0x4000;

   /*
   ** Next guess = (In / guess + guess) / 2
   */
   ulGuess = (ulIn / ulGuess + ulGuess) / 2;

   /*
   ** And one more time for good measure
   */
   ulGuess = (ulIn / ulGuess + ulGuess) / 2;

   /*
   ** And one more time for good measure
   */
   ulGuess = (ulIn / ulGuess+ulGuess) / 2;

   return  ulGuess;
}

/***************************************************************************
 *
 * FUNCTION NAME = f_lsqrt
 *
 * DESCRIPTION   = Returns the square root of the unsigned long
 *                 argument.  The value is returned as a FIXED number!
 *                 Be warned.
 *
 * INPUT         = ulIn - integer to square root
 *
 * OUTPUT        = FIXED square root
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/
ULONG
f_lsqrt (ULONG ulIn)
{
   /*
   **    Steps in the calculation are:
   **  1/ Normalize the input value (set top bit to 1)
   **  2/ Approximate the square root using Newton's method
   **  3/ Remove the effects of the normalization.
   */

   /*
   **  Number of bits shifted for normalisation
   */
   INT cNormal;

   if (!ulIn)
      return 0L;

   /*
   **     Perform the normalization.  The reason for this is to maintain
   **  maximum precision.
   */
   cNormal = 0;

   while (!(ulIn & 0xff000000))
   {
      ulIn <<= 8;                        /* 8 bits at a time                  */
      cNormal += 8;
   }

   /*
   **  Now for the remaining bits
   */
   while (!(ulIn&0x80000000))
   {
      ulIn <<= 1;
      cNormal++;
   }

   if (cNormal & 0x01)
   {
      /*
      **   NOTE:  the shift was by an odd number of bits.  This is not
      ** acceptable, since later we need to shift right by HALF the number
      ** of places that we shifted left above.  So,  reduce the count and
      ** shift the value one place to the right.
      */
      --cNormal;
      ulIn >>= 1;
   }

   /*
   **     Value is normalised to maximise precision.  Now proceed to use
   **  Newton's method to generate the square root.
   */
   ulIn = newton_f_lsqrt (ulIn);

   /*
   **    Now have a fixed version of the square root.   Unnormalize it to
   **  compensate for what was done above.
   */
   cNormal = cNormal/2-16;

   if (cNormal > 0)
      ulIn >>= cNormal;
   else
   {
      if (cNormal < 0)
         ulIn <<= -cNormal;
   }

   return ulIn;
}

/****************************************************************************/
/* PROCEDURE NAME : GetpResFromResID                                        */
/* AUTHOR         : Mark H                                                  */
/* DATE WRITTEN   : 11/16/93                                                */
/* DESCRIPTION    : Finds the DPI resolution value based on the             */
/*                  resolution ID passed in the job properties.             */
/*                                                                          */
/*                                                                          */
/* PARAMETERS     : (ULONG)  ulResID                                        */
/*                                                                          */
/*                                                                          */
/* RETURNS        : (BOOL)   bSuccess                                       */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* TAG      DATE        DEVELOPER [COMPANY]   DESCRIPTION                   */
/* ---      ----        -------------------   -----------                   */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
PRESINFO DRVENTRY
GetpResFromResID (PDRIVERINFO    pDriver,
                  PDEVICEINFO    pDevice,
                  PJOBPROPERTIES pJobProperties)
{
   PRESINFO  pResInfo       = NULL;
   ULONG     ulResID;
   ULONG     ulNumDefined;
   ULONG     i;

   if (!pJobProperties)
   {
      // Error.  Default to 0
      ulResID = 0;
   }
   else
   {
      ulResID = pJobProperties->ulDefResID;
   }

   // Ask the device to handle it if it wants it
   if (  pDevice
      && pDevice->pSubclassedFunctions
      && pDevice->pSubclassedFunctions->pfnQueryResInfo
      )
   {
      PFNQUERYRESINFO pfnQueryResInfo;

      pfnQueryResInfo = pDevice->pSubclassedFunctions->pfnQueryResInfo;
      pResInfo = pfnQueryResInfo (pDevice->usDeviceID, pJobProperties);
   }

   // Use the standard OMNI way
   if (!pResInfo)
   {
      ulNumDefined = pDriver->ulNumRes;
      pResInfo     = pDriver->pRES;

      // For now only driver table definitions looking for a match
      for (i = 0; i < ulNumDefined; i++)
         if (ulResID == pResInfo[i].ulResID)
            return &pResInfo[i];

        // did not find a match
        // default to first in list
        DBPRINTF (("Did not find a matching resolution for %d!\n", ulResID));
        assertstring ("Did not find a matching resolution!\n");
   }

   assertF (pResInfo);

   return pResInfo;

} /* end GetpResFromResID */

/****************************************************************************/
/* PROCEDURE NAME : GetpPrintModeFromID                                     */
/* AUTHOR         : Mark H.                                                 */
/* DATE WRITTEN   : 02/23/94                                                */
/* DESCRIPTION    : Finds the Print Mode structure based on the             */
/*                  print mode ID passed in the job properties.             */
/*                                                                          */
/*                                                                          */
/* PARAMETERS     : (PDRIVERINFO) pDriver                                   */
/* PARAMETERS     : (ULONG)       ulPrintModeID                             */
/*                                                                          */
/*                                                                          */
/* RETURNS        : (BOOL)   bSuccess                                       */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* TAG      DATE        DEVELOPER [COMPANY]   DESCRIPTION                   */
/* ---      ----        -------------------   -----------                   */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
PPRINTMODE DRVENTRY
GetpPrintModeFromID (PDRIVERINFO    pDriver,
                     PDEVICEINFO    pDevice,
                     PJOBPROPERTIES pJobProperties)
{
   PPRINTMODE pPrintMode     = NULL;
   ULONG      ulPrintModeID;
   ULONG      ulNumDefined;
   ULONG      i;

   if (!pJobProperties)
   {
      // Error.  Default to 0
      ulPrintModeID = 0;
   }
   else
   {
      ulPrintModeID = pJobProperties->ulDefPrintModeID;
   }

   // Ask the device to handle it if it wants it
   if (  pDevice
      && pDevice->pSubclassedFunctions
      && pDevice->pSubclassedFunctions->pfnQueryPrintMode
      )
   {
      PFNQUERYPRINTMODE       pfnQueryPrintMode;

      pfnQueryPrintMode = pDevice->pSubclassedFunctions->pfnQueryPrintMode;
      pPrintMode = pfnQueryPrintMode (pDevice->usDeviceID, pJobProperties);
   }

   // Use the standard OMNI way
   if (!pPrintMode)
   {
      ulNumDefined = pDriver->ulNumPrintModes;
      pPrintMode   = pDriver->pPrintModes;

      // For now only driver table definitions looking for a match
      for (i = 0; i < ulNumDefined; i++)
         if (ulPrintModeID == pPrintMode[i].ulPrintModeID)
            return &pPrintMode[i];

      /* Did not find a match... default to first in list
      ** This is an error condition!!! We need to report it!!!
      */
      DBPRINTF (("Did not find a matching Print Mode for %d!\n", ulPrintModeID ));
      assertstring ("Did not find a matching Print Mode!\n");
   }

   return pPrintMode;

} /* end GetpPrintModeFromID */

/***************************************************************************
 *
 * FUNCTION NAME = FindDeviceName
 *
 * DESCRIPTION   = Given a logical address (spooler printer name, spooler
 *                 device name. port name, or a UNC name) find the associated
 *                 OMNI device name.
 *
 * This function takes a logical address (queue name, device name, port name,
 * or file name) and tries to find a PLOTTER.device name.  It will place the
 * device name string in pszDeviceName.  If there are multiple plotter devices
 * attached to the printer (spooler term) then we will return the first one
 * because we have know way of determining the proper one to select.
 *
 * INPUT         = pvHeap              - Heap handle
 *                 pszDeviceName       - OUTPUT parameter
 *                 pszLogicalAddress   - Logical address
 *                 fDevPostDeviceModes - Is DevPostDeviceNames calling us?
 *                                       We need a different strategy if so.
 *
 * OUTPUT        = VOID
 *                 pszDeviceName     - Device name that we found.  If no device
 *                                     name, then an error occured.
 *****************************************************************************/
VOID
FindDeviceName (HMCB  hmcbHeap,
                PSZ   pszDeviceName,
                PSZ   pszLogicalAddress,
                BOOL  fDevPostDeviceModes)
{
   SPLERR       rc;
   PBYTE        pbQueue         = NULL;
   ULONG        cbBytes;
   ULONG        cbNeeded;
   ULONG        cReturned;
   ULONG        cTotal;
   PPRQINFO3    prq;
   PPRDINFO3    prd;
   PSZ          pszDrivers      = NULL;
   PSZ          pszComputerName = NULL;
   PSZ          pszTemp         = NULL;
   PSZ          pszComma;
   register INT i;

   *pszDeviceName = 0;
   cbNeeded = 0;
   cbBytes  = 0;

   DBPRINTF (("FindDeviceName called with :\n\tlogaddr = %s,\n\tfDPM = %d\n",
              pszLogicalAddress, fDevPostDeviceModes));

   /* We have a problem because the spooler's printer name can be the same
   ** as it's device name.  That is     we need to know who is calling us.
   ** For the DevPostDeviceModes call, we are given a device name.  And,
   ** in the spooled case, we are given a queue name.
   */
   if (fDevPostDeviceModes)
   {
      /* See if it is a valid device name */
      rc = SplQueryDevice ((PSZ)NULL,           // Computer name
                           pszLogicalAddress,   // Device Name
                           3,                   // Level
                           pbQueue,             // Buffer
                           cbBytes,             // is size
                           &cbNeeded);          // # bytes needed
      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         /* It is a valid device name */
         // Allocate the memory
         cbBytes = cbNeeded;
         pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
         if (!pbQueue)
            // Error!
            return;

         rc = SplQueryDevice ((PSZ)NULL,           // Computer name
                              pszLogicalAddress,   // Device Name
                              3,                   // Level
                              pbQueue,             // Buffer
                              cbBytes,             // is size
                              &cbNeeded);          // # bytes needed
         if (rc)
            // Error!
            return;

         prd = (PPRDINFO3)pbQueue;
         pszDrivers = prd->pszDrivers;
         goto done;
      }

      /* It wasn't a valid device name... Fall through to see if it might
      ** be a queue name instead.
      */
      cbNeeded = 0;
      cbBytes  = 0;
      GplMemoryFree (pbQueue);
      pbQueue = NULL;
   }

   if ('\\' == pszLogicalAddress[0] && '\\' == pszLogicalAddress[1])
   {
      /* This is a network print queue in the form of \\server\queue
      */
      pszTemp = (PSZ)GplMemoryAlloc (hmcbHeap, strlen (pszLogicalAddress)+1);
      if (pszTemp)
         strcpy (pszTemp, pszLogicalAddress);
      else
         // Error!
         return;

     /* Set up pointers ... */
     pszComputerName = pszTemp;
     /* Get rid of the queue name in the \\server\queue string */
     for (i = 2; pszComputerName[i]; i++)
        if ('\\' == pszComputerName[i])
        {
           pszComputerName[i] = 0;
           pszLogicalAddress = pszComputerName+i+1;
           break;
        }

     /* pszComputerName   should now be \\server
     ** pszLogicalAddress should now be queue
     */
   }

   /* See if it is a queue or a port
   */
   rc = SplQueryQueue ((PSZ)NULL,                // Computer name
                       pszLogicalAddress,        // Queue Name
                       3,                        // Level
                       pbQueue,                  // Buffer
                       cbBytes,                  // is size
                       &cbNeeded);               // # bytes needed
   if (NERR_QNotFound == rc)
   {
      // Must be a port
      cbNeeded = 0;
      cbBytes  = 0;

      // Find out how much memory to allocate
      rc = SplEnumDevice ((PSZ)NULL,             // Computer name
                          3,                     // Level
                          pbQueue,               // Buffer
                          cbBytes,               // is size
                          &cReturned,            // Number returned
                          &cTotal,               // Total available
                          &cbNeeded,             // # bytes needed
                          (PVOID)NULL);          // Reserved
      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         // Allocate the memory
         cbBytes = cbNeeded;
         pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
         if (!pbQueue)
            // Error!
            return;

         rc = SplEnumDevice ((PSZ)NULL,          // Computer name
                             3,                  // Level
                             pbQueue,            // Buffer
                             cbBytes,            // is size
                             &cReturned,         // Number returned
                             &cTotal,            // Total available
                             &cbNeeded,          // # bytes needed
                             (PVOID)NULL);       // Reserved
         if (rc)
            // Error!
            return;
      }
      else if (rc)
         // Error!
         return;

      // The data returned by the spooler is valid.  Setup the pointer.
      prd = (PPRDINFO3)pbQueue;

      /*
      ** First we should check if the logical address is null
      */
      if (!pszLogicalAddress || !pszLogicalAddress[0])
      {
         BOOL  fExit = FALSE;
         CHAR  achTest[100];

         // Build a test string
         sprintf (achTest, "%s.%s", APP_NAME, pszDeviceName);

         for (i = 0; !fExit && i < cReturned; i++)
         {
            BOOL  fContinue = TRUE;

            pszComma = strchr (prd->pszDrivers, ',');
            if (pszComma)
               *pszComma = 0;
            else
               fContinue = FALSE;

            do
            {
               if (0 == strcmp (achTest, prd->pszDrivers))
               {
                  pszDrivers = prd->pszPrinterName;
                  fExit = TRUE;
                  break;
               }

               if (pszComma)
               {
                  prd->pszDrivers = pszComma+1;
                  pszComma = strchr (prd->pszDrivers, ',');
                  if (pszComma)
                     *pszComma = 0;
               }
               else
                  fContinue = FALSE;
            } while (fContinue);
            prd++;
         }
      }
      /*
      ** Next Test for a PORT name first
      */
      else
      {
         for (i = 0; i < cReturned; i++)
         {
            if (0 == strcmp (prd->pszLogAddr, pszLogicalAddress))
            {
               pszDrivers = prd->pszDrivers;
               break;
            }
            prd++;
         }
      }

      /* If it was not a port name, then perhaps it is a filename.
      ** File names that are passed in are fully qualified -- look for a ':'
      */
      if (':' == pszLogicalAddress[1] &&
          !pszDrivers                  )
      {
         prd = (PPRDINFO3)pbQueue;        // Reset pointer

         for (i = 0; i < cReturned; i++)
         {
            if (0 == strcmp (prd->pszLogAddr, "FILE"))
            {
               pszDrivers = prd->pszDrivers;
               break;
            }
            prd++;
         }
      }
   }
   else if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
   {
      PSZ    pszDevice = NULL;
      ULONG  ulSize;

      // Must be a queue
      cbBytes = cbNeeded;

      // Allocate some memory
      pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
      if (!pbQueue)
      {
         GplErrSetError (PMERR_INSUFFICIENT_MEMORY);
         return;
      }

      rc = SplQueryQueue ((PSZ)NULL,             // Computer name
                          pszLogicalAddress,     // Queue Name
                          3,                     // Level
                          pbQueue,               // Buffer
                          cbBytes,               // is size
                          &cbNeeded);            // # bytes needed
      if (rc)
         // Error!
         return;

      /* Now that we have information about the queue.  All we need here
      ** is the device name.
      */
      prq = (PPRQINFO3)pbQueue;
      ulSize = strlen (prq->pszPrinters)+1;
      pszDevice = GplMemoryAlloc (hmcbHeap, ulSize);
      if (pszDevice)
         strcpy (pszDevice, prq->pszPrinters);
      else
      {
         // Error!
         GplErrSetError (PMERR_INSUFFICIENT_MEMORY);
         return;
      }

      /* This can be a comma seperated list of printers so remove the first
      ** comma!      do we take the first one?  Because each of the printers
      ** should have exactly the same information (courtesy of the spooler).
      */
      if ((pszComma = strchr (pszDevice, ',')) != (PSZ)NULL)
      {
         *pszComma = 0;
      }

      // Free the memory.  We are done with it now.
      GplMemoryFree ((PBYTE)pbQueue);
      pbQueue = NULL;
      cbBytes = 0;

      rc = SplQueryDevice ((PSZ)NULL,            // Computer name
                           pszDevice,            // Printer name
                           3,                    // Level
                           pbQueue,              // Buffer
                           cbBytes,              // Size of buffer
                           &cbNeeded);           // # bytes needed

      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         // Allocate the memory
         cbBytes = cbNeeded;
         pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
         if (!pbQueue)
            // Error!
            return;

         rc = SplQueryDevice ((PSZ)NULL,         // Computer name
                              pszDevice,         // Printer name
                              3,                 // Level
                              pbQueue,           // Buffer
                              cbBytes,           // Size of buffer
                              &cbNeeded);        // # bytes needed
         if (rc)
            // Error!
            return;
      }
      else if (rc)
         // Error!
         return;

      prd = (PPRDINFO3)pbQueue;
      if (0 == strcmp (prd->pszPrinterName, pszDevice))
         pszDrivers = prd->pszDrivers;

      // Clean up
      if (pszDevice)
         GplMemoryFree ((PBYTE)pszDevice);
   }
   else if (rc)
      // Error!
      return;

done:
   if (pszDrivers)
   {
      PSZ  pszPrinter, pszDriver, pszDevicename;
      BOOL fDriverFound;

      /* Here, we are given a list of printer drivers '.' device name.
      ** We have to go through each element in the list and see if the
      ** printer driver is us and the device name is ours.
      */
      pszPrinter = pszDrivers;
      while (*pszPrinter)
      {
         fDriverFound = FALSE;
         pszDriver = pszPrinter;
         while (*pszPrinter)
         {
            if ('.' == *pszPrinter)
               break;
            pszPrinter++;
         }
         if ('.' == *pszPrinter)
         {
            fDriverFound  = TRUE;
            *pszPrinter   = 0;
            pszPrinter++;
            pszDevicename = pszPrinter;
         }
         if (fDriverFound && *pszDevicename)
         {
            /* See if there is another driver after this one.  If so then
            ** remove the comma.
            */
            while (*pszPrinter && ',' != *pszPrinter)
               pszPrinter++;
            if (',' == *pszPrinter)
            {
               *pszPrinter = 0;
               pszPrinter++;
            }

            if (0 == strcmp (pszDriver, APP_NAME))
            {
               /* Its a match.  We are going to take the first one that
               ** we find.  Why??? Because we aren't mind readers.
               ** If we are in this code and we are looking at the following
               ** example:
               **    APP_NAME.Device 10,EPSON.LQ-2550,APP_NAME.Device 3
               ** we can ignore all other printer drivers but our own.
               ** However, if there is more than one APP_NAME we have no
               ** way of determining which device the app really wants.
               */
               strcpy (pszDeviceName, pszDevicename);
               *pszPrinter = 0;
               *pszDriver  = 0;
            }
         }
      }
   }

   // Clean up
   // Free the memory.  We are done with it now.
   if (pbQueue)
      GplMemoryFree ((PBYTE)pbQueue);
   if (pszTemp)
      GplMemoryFree ((PBYTE)pszTemp);

} /* FindDeviceName */

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME = FindLogAddress                                            */
/*                                                                           */
/*    typedef struct _PRDINFO3                                               */
/*    {                                                                      */
/*       PSZ     pszPrinterName;                                             */
/*       PSZ     pszUserName;                                                */
/*       PSZ     pszLogAddr;                                                 */
/*       USHORT  uJobId;                                                     */
/*       USHORT  fsStatus;                                                   */
/*       PSZ     pszStatus;                                                  */
/*       PSZ     pszComment;                                                 */
/*       PSZ     pszDrivers;                                                 */
/*       USHORT  time;                                                       */
/*       USHORT  usTimeOut;                                                  */
/*    } PRDINFO3;                                                            */
/*                                                                           */
/*****************************************************************************/
BOOL
FindLogAddress (HMCB  hmcb,
                PSZ   pszPrinterName,
                PSZ  *ppszLogicalAddress)
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  BOOL       bSuccess              = FALSE;
  SPLERR     splErr;
  PPRDINFO3  prdiDevices           = (PPRDINFO3)NULL;
  ULONG      ulcbDevices           = 0L;
  ULONG      ulcbNeeded            = 0L;
  ULONG      ulReturned, ulTotal;
  ULONG      ul;

  // Find out how much memory to allocate
  splErr = SplEnumDevice ((PSZ)NULL,             // Computer name
                          3,                     // Level
                          (PBYTE)prdiDevices,    // Buffer
                          ulcbDevices,           // is size
                          &ulReturned,           // Number returned
                          &ulTotal,              // Total available
                          &ulcbNeeded,           // # bytes needed
                          (PVOID)NULL);          // Reserved

  if (splErr == ERROR_MORE_DATA || splErr == NERR_BufTooSmall)
  {
     prdiDevices = (PPRDINFO3)GplMemoryAlloc (hmcb, ulcbNeeded);
     assertF (prdiDevices);

     if (prdiDevices)
     {
        ulcbDevices = ulcbNeeded;
        ulcbNeeded  = 0;

        splErr = SplEnumDevice ((PSZ)NULL,             // Computer name
                                3,                     // Level
                                (PBYTE)prdiDevices,    // Buffer
                                ulcbDevices,           // is size
                                &ulReturned,           // Number returned
                                &ulTotal,              // Total available
                                &ulcbNeeded,           // # bytes needed
                                (PVOID)NULL);          // Reserved

        assertF (splErr == NO_ERROR);

        if (splErr == NO_ERROR)
        {
           for (ul = 0; ul < ulTotal; ul++)
           {
              if (strcmp (prdiDevices[ul].pszPrinterName, pszPrinterName) == 0)
              {
                 *ppszLogicalAddress = (PSZ)GplMemoryAlloc (hmcb,
                                                            strlen (prdiDevices[ul].pszLogAddr) + 1);

                 strcpy (*ppszLogicalAddress, prdiDevices[ul].pszLogAddr);

                 bSuccess = TRUE;
                 break;
              }
           }
        }
     }
  }

  // Clean up
  if (prdiDevices)
  {
     GplMemoryFree (prdiDevices);
  }

  return bSuccess;

} /* end FindLogAddress */

/****************************************************************************/
/* PROCEDURE NAME : FindPrinterName                                         */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 03-28-94                                                */
/* DESCRIPTION    : This function takes a logical address (spooler printer  */
/*                  name,  spooler device name, port name, or a UNC name)   */
/*                  and returns the string containing the spooler's printer */
/*                  name.                                                   */
/*                                                                          */
/*                  NOTE: It is up to the caller to free this string        */
/*                        (it is allocated off of the heap).                */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      INPUT: pvHeap              - A heap handle              */
/*                         pszDeviceName       - The device name            */
/*                         (used if the logical address is a file name)     */
/*                         pszLogicalAddress   - The logical address        */
/*                         fDevPostDeviceModes - Is DevPostDeviceNames      */
/*                                               calling us?                */
/*                         (We need a different strategy if so)             */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:  PSZ - A string containing the spooler printer name.      */
/*                       If an error occured, then NULL.                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* @98155 - 09/17/94 - Matt Rutkowski [IBM] - Succeed if Logical address    */
/*                                            is NULL                       */
/*                                                                          */
/****************************************************************************/
PSZ
FindPrinterName (HMCB  hmcbHeap,
                 PSZ   pszDeviceName,
                 PSZ   pszLogicalAddress,
                 BOOL  fDevPostDeviceModes)
{
   SPLERR       rc;
   PBYTE        pbQueue         = NULL;
   ULONG        cbBytes;
   ULONG        cbNeeded;
   ULONG        cReturned;
   ULONG        cTotal;
   PPRQINFO3    prq;
   PPRDINFO3    prd;
   PSZ          pszPrinter      = NULL;
   PSZ          pszRet          = NULL;
   PSZ          pszComputerName = NULL;
   PSZ          pszTemp         = NULL;
   PSZ          pszComma;
   register INT i;

   cbNeeded = 0;
   cbBytes  = 0;

   if (!pszLogicalAddress)  // @98155 - 123G 2.1 fix for LAN printing
      goto hack;

#ifdef DEBUG
   DBPRINTF (("FindPrinterName called with:\n"));
   if (pszDeviceName)
      DBPRINTF (("\tdevice name = %s,\n", pszDeviceName));
   if (pszLogicalAddress)
      DBPRINTF (("\tlogaddr = %s,\n", pszLogicalAddress));
   DBPRINTF (("\tfDPM = %d\n", fDevPostDeviceModes));
#endif

   /* We have a problem because the spooler's printer name can be the same
   ** as it's device name.  That is     we need to know who is calling us.
   ** For the DevPostDeviceModes call, we are given a device name.  And,
   ** in the spooled case, we are given a queue name.
   */
   if (fDevPostDeviceModes)
   {
      /* See if it is a valid device name */
      rc = SplQueryDevice ((PSZ)NULL,           // Computer name
                           pszLogicalAddress,   // Device Name
                           3,                   // Level
                           pbQueue,             // Buffer
                           cbBytes,             // is size
                           &cbNeeded);          // # bytes needed
      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         /* It is a valid device name */
         pszPrinter = pszLogicalAddress;
         goto done;
      }
      /* It wasn't a valid device name... Fall through to see if it might
      ** be a queue name instead.
      */
   }

   if ('\\' == pszLogicalAddress[0] && '\\' == pszLogicalAddress[1])
   {
      /* This is a network print queue in the form of \\server\queue
      */
      pszTemp = (PSZ)GplMemoryAlloc (hmcbHeap, strlen (pszLogicalAddress)+1);
      if (pszTemp)
         strcpy (pszTemp, pszLogicalAddress);
      else
         // Error!
         return NULL;

     /* Set up pointers ... */
     pszComputerName = pszTemp;
     /* Get rid of the queue name in the \\server\queue string */
     for (i = 2; pszComputerName[i]; i++)
        if ('\\' == pszComputerName[i])
        {
           pszComputerName[i] = 0;
           pszLogicalAddress = pszComputerName+i+1;
           break;
        }

     /* pszComputerName   should now be \\server
     ** pszLogicalAddress should now be queue
     */
   }

   /* See if it is a queue name.  If not then it is a port name.
   */
   rc = SplQueryQueue ((PSZ)NULL,                // Computer name
                       pszLogicalAddress,        // Queue Name
                       3,                        // Level
                       pbQueue,                  // Buffer
                       cbBytes,                  // is size
                       &cbNeeded);               // # bytes needed
   if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
   {
      // Must be a queue
      // Allocate some memory
      cbBytes = cbNeeded;
      pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
      if (!pbQueue)
         // Error!
         return NULL;

      rc = SplQueryQueue ((PSZ)NULL,             // Computer name
                          pszLogicalAddress,     // Queue Name
                          3,                     // Level
                          pbQueue,               // Buffer
                          cbBytes,               // is size
                          &cbNeeded);            // # bytes needed
      if (rc)
         // Error!
         return NULL;

      prq = (PPRQINFO3)pbQueue;
      pszPrinter = prq->pszPrinters;

      /* This can be a comma seperated list of printers so remove the first
      ** comma!      do we take the first one?  Because each of the printers
      ** should have exactly the same information (courtesy of the spooler).
      */
      if ((pszComma = strchr (pszPrinter, ',')) != (PSZ)NULL)
      {
         *pszComma = 0;
      }
   }
   else if (NERR_QNotFound == rc)
   {
hack:
      // Must be a port
      cbNeeded = 0;
      cbBytes  = 0;

      // Find out how much memory to allocate
      rc = SplEnumDevice ((PSZ)NULL,             // Computer name
                          3,                     // Level
                          pbQueue,               // Buffer
                          cbBytes,               // is size
                          &cReturned,            // Number returned
                          &cTotal,               // Total available
                          &cbNeeded,             // # bytes needed
                          (PVOID)NULL);          // Reserved
      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         // Allocate the memory
         cbBytes = cbNeeded;
         pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
         if (!pbQueue)
            // Error!
            return NULL;

         rc = SplEnumDevice ((PSZ)NULL,          // Computer name
                             3,                  // Level
                             pbQueue,            // Buffer
                             cbBytes,            // is size
                             &cReturned,         // Number returned
                             &cTotal,            // Total available
                             &cbNeeded,          // # bytes needed
                             (PVOID)NULL);       // Reserved
         if (rc)
            // Error!
            return NULL;
      }
      else if (rc)
         // Error!
         return NULL;

      if (pszLogicalAddress)
      {
         prd = (PPRDINFO3)pbQueue;

         /* Test for a PORT name first
         */
         for (i = 0; i < cReturned; i++)
         {
            if (0 == strcmp (prd->pszLogAddr, pszLogicalAddress))
            {
               pszPrinter = prd->pszPrinterName;
               break;
            }
            prd++;
         }
      }

#if 0
      /* If it was not a port name, then perhaps it is a filename.
      ** File names that are passed in are fully qualified -- look for a ':'
      */
      if (':' == pszLogicalAddress[1] &&
          !pszPrinter                  )
#else
      if (!pszPrinter)
#endif
      {
         BOOL  fExit = FALSE;

         prd = (PPRDINFO3)pbQueue;            // Reset pointer

         // FILE name
         for (i = 0; !fExit && i < cReturned; i++)
         {
#if 0
            if (0 == strcmp (prd->pszLogAddr, "FILE"))
#endif
            {
               BOOL  fContinue = TRUE;
               CHAR  achTest[100];

               // Build a test string
               sprintf (achTest, "%s.%s", APP_NAME, pszDeviceName);

               pszComma = strchr (prd->pszDrivers, ',');
               if (pszComma)
                  *pszComma = 0;
               else
                  fContinue = FALSE;

               do
               {
                  if (0 == strcmp (achTest, prd->pszDrivers))
                  {
                     pszPrinter = prd->pszPrinterName;
                     fExit = TRUE;
                     break;
                  }

                  if (pszComma)
                  {
                     prd->pszDrivers = pszComma+1;
                     pszComma = strchr (prd->pszDrivers, ',');
                     if (pszComma)
                        *pszComma = 0;
                  }
                  else
                     fContinue = FALSE;
               } while (fContinue);
            }
            prd++;
         }
      }
   }
   else if (rc)
      // Error!
      return NULL;

done:
   if (pszPrinter)
   {
      pszRet = (PSZ)GplMemoryAlloc (hmcbHeap, strlen (pszPrinter)+1);
      if (pszRet)
         strcpy (pszRet, pszPrinter);
   }

   // Clean up
   // Free the memory.  We are done with it now.
   if (pbQueue)
      GplMemoryFree ((PBYTE)pbQueue);
   if (pszTemp)
      GplMemoryFree ((PBYTE)pszTemp);

   if (pszRet)
      DBPRINTF (("FindPrinterName returns %s\n", pszRet));
   else
      DBPRINTF (("FindPrinterName returns nothing!\n"));

   return pszRet;
}

/***************************************************************************
 *
 * FUNCTION NAME = StringInCommaList
 *
 * DESCRIPTION   = This function takes a string and a list of strings and
 *                 returns true if the string is in the list
 *
 * INPUT         = PSZ pszStr  - string to search for
 *                 PSZ pszList - a comma separated list of strings
 *
 * OUTPUT        = TRUE  - string found
 *                 FALSE - string not found
 *
 **************************************************************************/
BOOL
StringInCommaList (PSZ pszStr, PSZ pszList)
{
   PSZ   pszComma,
         pszTest;

   pszTest = pszComma = pszList;
   do
   {
      // Get the next comma from the substring
      pszComma = strchr (pszComma, ',');
      if (pszComma)
      {
         // Zap comma to make pszTest point to the substring
         *pszComma = 0;
         // Move comma pointer to next substring
         pszComma++;
      }

      if (0 == strcmp (pszStr, pszTest))
         return TRUE;

   } while (pszComma);

   return FALSE;
}

/***************************************************************************
 *
 * FUNCTION NAME = PrinterNameFromDeviceName
 *
 * DESCRIPTION   = This function will take a device name and try to return
 *                 the spooler printer name that has the given device name.
 *
 * INPUT         = HMCB hmcbHeap      - heap for allocations
 *                 PSZ  pszDeviceName - device name to search for
 *
 * OUTPUT        = PSZ                - spooler printer name
 *                 NULL if error
 *
 * NOTE: It is the caller's responsibility to free the returned string.
 *
 **************************************************************************/
PSZ
PrinterNameFromDeviceName (HMCB   hmcbHeap,
                           PSZ    pszDeviceName,
                           PSZ   *ppszLogAddr)
{
   SPLERR       rc;
   PBYTE        pbQueue         = NULL;
   CHAR         achTest[50];
   ULONG        cbBytes;
   ULONG        cbNeeded;
   ULONG        cReturned;
   ULONG        cTotal;
   PPRDINFO3    prd;
   PSZ          pszRet          = NULL;
   register INT i;

   DBPRINTF (("PrinterNameFromDeviceName called with:\n"));
   if (pszDeviceName)
      DBPRINTF (("\tdevice name = %s\n", pszDeviceName));
   else
      return NULL;

   sprintf (achTest, "%s.%s", APP_NAME, pszDeviceName);

   cbBytes  = 0;
   cbNeeded = 0;

   // Ask for a list of all the printers in the system
   rc = SplEnumDevice ((PSZ)NULL,          // Computer name
                       3,                  // Level
                       pbQueue,            // Buffer
                       cbBytes,            // is size
                       &cReturned,         // Number returned
                       &cTotal,            // Total available
                       &cbNeeded,          // # bytes needed
                       (PVOID)NULL);       // Reserved
   if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
   {
      // Allocate some memory
      cbBytes = cbNeeded;
      pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
      assertF (pbQueue);
      if (!pbQueue)
         // Error!
         goto done;

      rc = SplEnumDevice ((PSZ)NULL,          // Computer name
                          3,                  // Level
                          pbQueue,            // Buffer
                          cbBytes,            // is size
                          &cReturned,         // Number returned
                          &cTotal,            // Total available
                          &cbNeeded,          // # bytes needed
                          (PVOID)NULL);       // Reserved
      if (rc)
         // Error!
         goto done;

      prd = (PPRDINFO3)pbQueue;

      for (i = 0; i < cReturned; i++)
      {
         if (StringInCommaList (achTest, prd->pszDrivers))
         {
            pszRet = GplMemoryAlloc (hmcbHeap, strlen (prd->pszPrinterName) + 1);
            assertF (pszRet);

            strcpy (pszRet, prd->pszPrinterName);

            *ppszLogAddr = GplMemoryAlloc (hmcbHeap, strlen (prd->pszLogAddr) + 1);
            assertF (*ppszLogAddr);

            strcpy (*ppszLogAddr, prd->pszLogAddr);
            break;
         }

         prd++;
      }
   }

done:
   // Clean up
   // Free the memory.  We are done with it now.
   if (pbQueue)
      GplMemoryFree ((PBYTE)pbQueue);

   return pszRet;
}

/***************************************************************************
 *
 * FUNCTION NAME = LogAddrFromOmniDefaultPrinter
 *
 * DESCRIPTION   =
 *
 * INPUT         = HMCB  hmcbHeap      -
 *                 PSZ   pszLogAddress -
 *                 ULONG ulType        -
 *
 * OUTPUT        = PSZ   pszLogAddress -
 *
 * RETURN-NORMAL = TRUE
 * RETURN-ERROR  = FALSE
 *
 **************************************************************************/
BOOL
LogAddrFromOmniDefaultPrinter (HMCB  hmcbHeap,
                               PSZ   pszLogAddress,
                               ULONG ulType)
{
   CHAR         achName[10];
   PBYTE        pbQueue      = NULL;
   ULONG        cbNeeded;
   ULONG        cbBytes;
   PPRDINFO3    prd;
   PPRQINFO3    prq;
   BOOL         bRet         = FALSE;          // Assume an error
   APIRET       rc;

   if (OD_DIRECT == ulType)
   {
      /* OD_DIRECT
      */
      /* Ask for the system default printer name
      */
      if (PrfQueryProfileString (HINI_USERPROFILE,
                                 "PM_SPOOLER",
                                 "PRINTER",
                                 "",
                                 achName,
                                 sizeof (achName)))
      {
         // Remove trailing ';'
         achName[strlen (achName) - 1] = 0;

         cbBytes = cbNeeded = 0;

         /* Find out the size for level 3 info on this device name
         */
         rc = SplQueryDevice ((PSZ)NULL,             // Computer name
                              achName,               // Printer device name
                              3,                     // Level
                              pbQueue,               // Buffer
                              cbBytes,               // is size
                              &cbNeeded);            // # bytes needed
         if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
         {
            cbBytes = cbNeeded;
            pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
            if (!pbQueue)
               // Error!
               goto done;

            /* Get the info!
            */
            rc = SplQueryDevice ((PSZ)NULL,             // Computer name
                                 achName,               // Printer device name
                                 3,                     // Level
                                 pbQueue,               // Buffer
                                 cbBytes,               // is size
                                 &cbNeeded);            // # bytes needed
            if (rc)
               // Error!
               goto done;

            prd = (PPRDINFO3)pbQueue;

            /* We have no idea what the proper device name is to initialize
            ** to, so we just check to see if one of the devices in the
            ** list is our driver...
            */
            if (strstr (prd->pszDrivers, APP_NAME))
            {
               strcpy (pszLogAddress, prd->pszLogAddr);
               bRet = TRUE;
            }
         }
      }
   }
   else
   {
      /* OD_QUEUED, OD_INFO, ...
      */
      /* Ask for the system default queue name
      */
      if (PrfQueryProfileString (HINI_USERPROFILE,
                                 "PM_SPOOLER",
                                 "QUEUE",
                                 "",
                                 achName,
                                 sizeof (achName)))
      {
         // Remove trailing ';'
         achName[strlen (achName) - 1] = 0;

         cbBytes = cbNeeded = 0;

         // Find out the size of the queue's info
         rc = SplQueryQueue ((PSZ)NULL,                // Computer name
                             achName,                  // Queue Name
                             3,                        // Level
                             pbQueue,                  // Buffer
                             cbBytes,                  // is size
                             &cbNeeded);               // # bytes needed
         if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
         {
            // Allocate the memory
            cbBytes = cbNeeded;
            pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
            if (!pbQueue)
               // Error!
               goto done;

            // Query the queue's info
            rc = SplQueryQueue ((PSZ)NULL,                // Computer name
                                achName,                  // Queue Name
                                3,                        // Level
                                pbQueue,                  // Buffer
                                cbBytes,                  // is size
                                &cbNeeded);               // # bytes needed
            if (rc)
               // Error!
               goto done;

            prq = (PPRQINFO3)pbQueue;

            // See if this queue is driven by us...
            if (strstr (prq->pszDriverName, APP_NAME))
            {
               strcpy (pszLogAddress, achName);
               bRet = TRUE;
            }
         }
      }
   }

done:
   if (pbQueue)
      GplMemoryFree (pbQueue);

   assertF (bRet);

   return bRet;
}

/***************************************************************************
 *
 * FUNCTION NAME = LogAddrFromOmniDevice
 *
 * DESCRIPTION   =
 *
 * INPUT         = HMCB  hmcbHeap      -
 *                 PSZ   pszLogAddress -
 *                 PSZ   pszDeviceName -
 *                 ULONG ulType        -
 *
 * OUTPUT        = PSZ   pszLogAddress -
 *
 * RETURN-NORMAL = TRUE
 * RETURN-ERROR  = FALSE
 *
 **************************************************************************/
BOOL
LogAddrFromOmniDevice (HMCB  hmcbHeap,
                       PSZ   pszLogAddress,
                       PSZ   pszDeviceName,
                       ULONG ulType)
{
   CHAR         achTest[100];
   ULONG        cbNeeded;
   ULONG        cbBytes;
   ULONG        cReturned;
   ULONG        cTotal;
   PPRDINFO3    prd;
   PPRQINFO3    prq;
   PBYTE        pbQueue      = NULL;
   BOOL         bRet         = FALSE;          // Assume an error
   APIRET       rc;
   register INT i;

   if (!pszDeviceName || !*pszDeviceName)
      return FALSE;

   // Set up the test string
   sprintf (achTest, "%s.%s", APP_NAME, pszDeviceName);

   if (OD_DIRECT == ulType)
   {
      /* OD_DIRECT
      */
      cbBytes  = 0;
      cbNeeded = 0;

      // Ask for the size of a list of all the printers in the system
      rc = SplEnumDevice ((PSZ)NULL,          // Computer name
                          3,                  // Level
                          pbQueue,            // Buffer
                          cbBytes,            // is size
                          &cReturned,         // Number returned
                          &cTotal,            // Total available
                          &cbNeeded,          // # bytes needed
                          (PVOID)NULL);       // Reserved
      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         // Allocate some memory
         cbBytes = cbNeeded;
         pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
         if (!pbQueue)
            // Error!
            goto done;

         // Get the list!
         rc = SplEnumDevice ((PSZ)NULL,          // Computer name
                             3,                  // Level
                             pbQueue,            // Buffer
                             cbBytes,            // is size
                             &cReturned,         // Number returned
                             &cTotal,            // Total available
                             &cbNeeded,          // # bytes needed
                             (PVOID)NULL);       // Reserved
         if (rc)
            // Error!
            goto done;

         prd = (PPRDINFO3)pbQueue;

         for (i = 0; i < cReturned; prd++, i++)
         {
            /* Test to see if the device name is in the list of drivers.
            ** As long as the logical address is not a user entered file
            ** name (key word: FILE), take it!
            */
            if (strstr (prd->pszDrivers, achTest)   &&
                0 != strcmp (prd->pszLogAddr, "FILE"))
            {
               strcpy (pszLogAddress, prd->pszLogAddr);
               bRet = TRUE;
               break;
            }
         }
      }
   }
   else
   {
      /* OD_QUEUED, OD_INFO, ...
      */
      cbBytes  = 0;
      cbNeeded = 0;

      // Ask for the size of a list of the queues in the system
      rc = SplEnumQueue ((PSZ)NULL,                // Computer name
                         3,                        // Level
                         pbQueue,                  // Buffer
                         cbBytes,                  // is size
                         &cReturned,               // Number returned
                         &cTotal,                  // Total available
                         &cbNeeded,                // # bytes needed
                         (PVOID)NULL);             // Reserved
      if (ERROR_MORE_DATA == rc || NERR_BufTooSmall == rc)
      {
         // Allocate the memory
         cbBytes = cbNeeded;
         pbQueue = (PBYTE)GplMemoryAlloc (hmcbHeap, cbNeeded);
         if (!pbQueue)
            // Error!
            goto done;

         // Get the list!
         rc = SplEnumQueue ((PSZ)NULL,                // Computer name
                            3,                        // Level
                            pbQueue,                  // Buffer
                            cbBytes,                  // is size
                            &cReturned,               // Number returned
                            &cTotal,                  // Total available
                            &cbNeeded,                // # bytes needed
                            (PVOID)NULL);             // Reserved

         if (rc)
            // Error!
            goto done;

         prq = (PPRQINFO3)pbQueue;

         for (i = 0; i < cReturned; prq++, i++)
         {
            /* If this queue has our device as its default driver then
            ** use it!
            */
            if (strstr (prq->pszDriverName, achTest))
            {
               strcpy (pszLogAddress, prq->pszName);
               bRet = TRUE;
               break;
            }
         }
      }
   }

done:
   if (pbQueue)
      GplMemoryFree (pbQueue);

   assertF (bRet);

   return bRet;
}

#ifndef DLLBLD
/***************************************************************************
 *
 * FUNCTION NAME = NumberOfDeviceNames
 *
 * DESCRIPTION   = Return how many devices are in our list of drivers & devices
 *
 * INPUT         = NONE
 *
 * OUTPUT        = ULONG - number of device names
 *
 * RETURN-NORMAL = TRUE
 * RETURN-ERROR  = FALSE
 *
 **************************************************************************/
ULONG
NumberOfDeviceNames (VOID)
{
   register INT iDriver, iDevice;
   ULONG        ulCount = 0;
   PDEVICEINFO  pDevice;

   for (iDriver = 0; iDriver < globals.ulTotalDrivers; iDriver++)
   {
//      for (iDevice = Drivers[iDriver]->ulNumDevices,
//              pDevice = Drivers[iDriver]->pDevices;
      for (iDevice = globals.ppDrivers[iDriver]->ulNumDevices,
              pDevice = globals.ppDrivers[iDriver]->pDevices;
           iDevice;
           iDevice--, pDevice++)
      {
         ulCount++;
      }
   }

   return ulCount;
}

#endif

/***************************************************************************
 *
 * FUNCTION NAME = IsItADriverName
 *
 * DESCRIPTION   = Check that the physical driver name is supported.
 *
 * INPUT         = pDriverName - name of device
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = TRUE
 * RETURN-ERROR  = FALSE
 *
 **************************************************************************/
BOOL
IsItADriverName (PSZ pszDriverName)
{
   return 0 == strcmp (pszDriverName, APP_NAME);
}

/****************************************************************************/
/* PROCEDURE NAME : ValidateDriverData                                      */
/* AUTHOR         : Mark H. and MFR                                         */
/* DATE WRITTEN   : 9/23/93                                                 */
/*   RE-WRITTEN   : 7/20/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @TBD - change 256, 255 to macro constants                                */
/*                                                                          */
/* @TBD - need to load/unload string table based on memory page faults      */
/* on an address range - MFR                                                */
/* @JPN2H99- 08/01/99 - UCC [IBMJ]- Printer properties option page support. */
/*                                                                          */
/****************************************************************************/
BOOL
ValidateDriverData (PJOBPROPERTIES pJobProperties,
                    PPRNPROPERTIES pPrnProperties,          //@JPN2H99
                    PDRIVERINFO    pDriver,
                    PDEVICEINFO    pDevice)
{
   BOOL         bRet = TRUE;
   register INT i;

   DBPRINTF (("%s() Enter: Driver = '%s', Device = '%s'\n",
              __FUNCTION__,
              pDriver->pszDriverName,
              pDevice->pszDeviceName));

   if (strcmp (pJobProperties->achDriver, APP_NAME))
   {
      DBPRINTF (("%s(): achDriver = '%s' It is not %s!\n",
                 __FUNCTION__,
                 pJobProperties->achDriver,
                 APP_NAME));
      bRet = FALSE;
   }

   if (JOBPROP_SIG != pJobProperties->ulSig)
   {
      DBPRINTF (("%s(): ulSig = '%x' It is not %x!\n",
                 __FUNCTION__,
                 pJobProperties->ulSig,
                 JOBPROP_SIG));
      bRet = FALSE;
   }

   if (  (pJobProperties->ulOrientation != ORIENTATION_PORTRAIT)
      && (pJobProperties->ulOrientation != ORIENTATION_LANDSCAPE)
      )
   {
      DBPRINTF (("%s(): ulOrientation = '%d' \n", __FUNCTION__,
                 pJobProperties->ulOrientation));
      bRet = FALSE;
   }

   if (  pJobProperties->ulCopies > MAX_UNCOLLATEDCOPIES
      || pJobProperties->ulCopies == 0
      )
   {
      DBPRINTF (("%s(): ulCopies = '%d' is out of range!!!\n",
                 __FUNCTION__,
                 pJobProperties->ulCopies ));
      bRet = FALSE;
   }

   /* Validate the current connection id against the list of possible ids.
   ** If it is not found, then use the first one.
   */
   if (  !(pJobProperties->ulDefConnID & DEF_TYPE_USER)
      && (pDevice->usNumConnects)
      )
   {
      for (i = 0; i < pDevice->usNumConnects; i++)
         if (pDevice->pulCONNECTS[i] == pJobProperties->ulDefConnID)
            break;

      if (i >= pDevice->usNumConnects)
      {
         DBPRINTF (("%s(): ulDefConnID = '%d' is out of range!!!\n",
                    __FUNCTION__,
                    pJobProperties->ulDefConnID ));

         bRet = FALSE;
      }
   }

   /* Validate the current non-metric connection id against the list of
   ** possible ids.  If it is not found, then use the first one.
   */
   if (  !(pJobProperties->ulDefNonMetricConnID & DEF_TYPE_USER)
      && (pDevice->usNumConnects)
      )
   {
      for (i = 0; i < pDevice->usNumConnects; i++)
         if (pDevice->pulCONNECTS[i] == pJobProperties->ulDefNonMetricConnID)
            break;

      if (i >= pDevice->usNumConnects)
      {
         DBPRINTF (("%s(): ulDefNonMetricConnID = '%d' is out of range!!!\n",
                    __FUNCTION__,
                    pJobProperties->ulDefNonMetricConnID ));
         bRet = FALSE;
      }
   }

   /* Validate the current font id against the list of possible ids.
   ** If it is not found, then use the first one.
   */
   if (pDevice->usNumFonts)
   {
      LONG   lNumFonts;

      lNumFonts = ReturnNumberOfDeviceFonts (pDriver, pDevice);

      if (0 <= (LONG)pJobProperties->ulDefFontID)
      {
         USHORT       usNumFonts;
         ULONG        ulNumFontsTotal;
         PFONTINFO2   pFonts;
         ULONG        ulFontID;
         register INT j;

         // Old style.
         DBPRINTF (("%s(): Warning: Old style font match id %d\n",
                    __FUNCTION__,
                    pJobProperties->ulDefFontID));

         bRet            = FALSE;

         ulNumFontsTotal = pDriver->ulNumFonts;
         pFonts          = pDriver->pFONTS;
         usNumFonts      = pDevice->usNumFonts;

         for (i = 0; i < usNumFonts; i++)
         {
            if (pJobProperties->ulDefFontID == pDevice->pulFONTS[i])
            {
               ulFontID = pDevice->pulFONTS[i];

               for (j = 0; j < ulNumFontsTotal; j++)
               {
                  if (ulFontID == pFonts[j].ulFontID)
                  {
                     bRet = TRUE;

                     pJobProperties->ulDefFontID = (ULONG)pFonts[j].lDevMatch;

                     DBPRINTF (("%s(): ulDefFontID is now = '%d'.\n",
                                __FUNCTION__,
                                pJobProperties->ulDefFontID));
                  }
               }
            }
         }
      }
      else
      {
         // New style.
//       if (-(LONG)pJobProperties->ulDefFontID > lNumFonts)            //@JPN2H99
         if (-(LONG)pJobProperties->ulDefFontID > pDriver->ulNumFonts)  //@JPN2H99
         {
            DBPRINTF (("%s(): ulDefFontID = '%d' is out of range!!!  There are %d fonts.\n",
                       __FUNCTION__,
                       pJobProperties->ulDefFontID,
                       lNumFonts));
            bRet = FALSE;
         }
      }
   }
// begin @JPN2H99
   if (  pPrnProperties
      && (pPrnProperties->fOmniOptions & OMNI_OPTION_DEVICEFONT_OFF)
      )
   {
      // If user don't want to use device font,
      // validation check can/should return TRUE.
      bRet = TRUE;
   }
// end @JPN2H99

   /* Validate the current resolution id against the list of possible ids.
   ** If it is not found, then use the first one.
   */
   if (pDevice->usNumRes)
   {
      for (i = 0; i < pDevice->usNumRes; i++)
         if (pDevice->pulRES[i] == pJobProperties->ulDefResID)
            break;

      if (i >= pDevice->usNumRes)
      {
         DBPRINTF (("%s(): ulDefResID = '%d' is out of range!!!\n",
                    __FUNCTION__,
                    pJobProperties->ulDefResID));
         bRet = FALSE;
      }
   }

   /* Validate the current print mode id against the list of possible ids.
   ** If it is not found, then use the first one.
   */
   if (pDevice->usNumPrintModes)
   {
      for (i = 0; i < pDevice->usNumPrintModes; i++)
         if (pDevice->pulPrintModes[i] == pJobProperties->ulDefPrintModeID)
            break;

      if (i >= pDevice->usNumPrintModes)
      {
         DBPRINTF (("%s(): ulDefPrintModeID = '%d' is out of range!!!\n",
                    __FUNCTION__,
                    pJobProperties->ulDefPrintModeID));

         bRet = FALSE;
      }
   }

   // Check dither algorithm, we no longer support certain ones
   // so replace them with one we do
   if (pJobProperties->ulAlgorithm == HT_LEVEL)
   {
      pJobProperties->ulAlgorithm = HT_MAGIC_SQUARES;
      DBPRINTF (("%s(): HT_LEVEL not supported\n", __FUNCTION__ ));
      DBPRINTF (("%s(): kindly replace with HT_MAGIC_SQUARES and pass validate\n",
                 __FUNCTION__));
   }

    
    // We validate and initialize Layout control block

    
    // This condition is FALSE if we have uppgrated old driver without Layout features
    // The problem is solved by @246532 fix
//    if (((PBYTE)(&(pJobProperties->LayoutCB))-(PBYTE)pJobProperties)<pJobProperties->cb)
       GplLayoutInitCB( &(pJobProperties->LayoutCB), GL_SET_DEFAULT_LAYOUT,
                        (ULONG) 0X00);

#if 0
   // @TBD - still need print mode, conn ID, ulDefFormID, ulDefFontID, and ulDefResID

   if (pJobProperties->)
   {
      DBPRINTF (("ValidateDriverData(): \n"));
      bRet = FALSE;
   }
#endif

   DBPRINTF (("%s() Exit :\n", __FUNCTION__));

   return bRet;

} /* end ValidateDriverData */

/****************************************************************************/
/* PROCEDURE NAME : ValidatePrinterProperties                               */
/* AUTHOR         : Mark H                                                  */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ValidatePrinterProperties (PPRNPROPERTIES pPrnProperties)
{
   BOOL bRet = TRUE;

   if (strcmp (pPrnProperties->achDriver, APP_NAME))
   {
      DBPRINTF (("ValidatePrinterProperties: achDriver = %s It is not %s!\n",
                 pPrnProperties->achDriver,
                 APP_NAME));
      bRet = FALSE;
   }

   if (PRINTPROP_SIG != pPrnProperties->ulSig)
   {
      DBPRINTF (("ValidatePrinterProperties: ulSig = %x It is not %x!\n",
                 pPrnProperties->ulSig,
                 PRINTPROP_SIG));
      bRet = FALSE;
   }

#if 0
   // @TBD - need to test ulInstalledMemory

   if (pPrnProperties->)
   {
      DBPRINTF (("ValidatePrinterProperties:\n"));
      bRet = FALSE;
   }
#endif

   return bRet;
}

/****************************************************************************/
/* PROCEDURE NAME : ReadDefaultJobProperties                                */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansiblejob data support              */
/*                                                                          */
/****************************************************************************/
BOOL
// begin @EXJOB
  ReadDefaultJobProperties (PSZ pszAppName, PVOID pVoid, PULONG pulRead)
//ReadDefaultJobProperties (PSZ pszAppName, PVOID pVoid)
// end @EXJOB
{
// begin @EXJOB
// ULONG  ul;
// end @EXJOB
   BOOL   bRC;

   DBPRINTF (("Reading default job properties from %s\n", pszAppName));

   // retrieve drivdata from os2sys.ini
// begin @EXJOB
// ul = DRIVERDATA_SIZE;
// end @EXJOB

   bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                              pszAppName,
                              KEYNAME_DRIVERDATA,
                              pVoid,
// begin @EXJOB
                              pulRead);
//                            &ul);
// end @ENJOB
#ifdef DEBUG
   if (!bRC)
      DBPRINTF (("***** FAILED *****\n"));
#endif

   return bRC;

} /* end ReadDefaultJobProperties */

/****************************************************************************/
/* PROCEDURE NAME : SavePrinterProperties                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
SavePrinterProperties (PSZ            pszAppName,
                       PPRNPROPERTIES pPrnProps)
{
   BOOL           bSuccess = TRUE;

   DBPRINTIF ((TRUE, "%s(): Enter\n", __FUNCTION__));

   assertF (pszAppName);
   assertF (pPrnProps);

   pPrnProps->cb = PRINTERPROP_SIZE;

   bSuccess = PrfWriteProfileData (HINI_SYSTEMPROFILE,
                                   pszAppName,
                                   KEYNAME_PRN_PROPS,
                                   pPrnProps,
                                   PRINTERPROP_SIZE);

   DBPRINTIF ((TRUE,"%s(): Exit\n", __FUNCTION__));

   return bSuccess;

} /* SavePrinterProperties */

/****************************************************************************/
/* PROCEDURE NAME : ReadPrinterProperties                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ReadPrinterProperties (PSZ            pszAppName,
                       PPRNPROPERTIES pPrnProps,
                       PDEVICEINFO    pDevice)
{
   BOOL           bSuccess      = TRUE;
   ULONG          ulDataSize;
   ULONG          ulProfileSize = PRINTERPROP_SIZE;

   DBPRINTF (("ReadPrinterProperties(): Enter\n"));

   assertF (pszAppName);
   assertF (pPrnProps);
   assertF (pDevice);

   memset (pPrnProps, 0, PRINTERPROP_SIZE);
   ulDataSize = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                     pszAppName,
                                     KEYNAME_PRN_PROPS,
                                     pPrnProps,
                                     &ulProfileSize);

   DBPRINTF (("ReadPrinterProperties(): ulDataSize = %lu\n", ulDataSize));

   // If we don't have an INI entry initialize fields
   if (ulDataSize == 0                      ||
       !ValidatePrinterProperties (pPrnProps))
      SetPrinterPropDefaults (pDevice,
                              pPrnProps,
                              pDevice->pszDeviceName);

   DBPRINTF (("ReadPrinterProperties(): Exit\n"));

   return bSuccess;

} /* ReadPrinterProperties */

#define MAX_PATH_SIZE                   256             // size of driver name DCR1399
#define EA_NAME_VERSION                 ".VERSION"      // attribute name DCR1399
#define SIZE_EA_NAME_VERSION            (sizeof(EA_NAME_VERSION) - 1)  // DCR1399

#pragma pack (1)
typedef struct _FEAL_VER {
  ULONG  cbList;                          /* size of the structure             */
  ULONG  oNextEntryOffset;                /*                                   */
  BYTE   fEA;                             /*                                   */
  BYTE   cbName;                          /*                                   */
  USHORT cbValue;                         /*                                   */
  CHAR   szEAName[SIZE_EA_NAME_VERSION+1];/* fixed size EA name                */
  USHORT usEAT_Type;                      /* Type of the EA                    */
  USHORT usCodePage;                      /* code page field                   */
  USHORT usValueCount;                    /* total fileds                      */
  USHORT usVerASCIIType;                  /* type of EAT assume ANSI           */
  USHORT usSizeVerName;                   /* name of version name              */
  CHAR   szVersionName[MAX_VERNAME_SIZE]; /* name of the version               */
} FEAL_VER;
#pragma pack()

/*
** The GEAL_VER must be packed on 4-byte boundaries so that it can be used
** with DosQueryPathInfo
*/
#pragma pack(4)
typedef struct _GEAL_VER {
   ULONG cbList;
   ULONG oNextEntryOffset;
   BYTE  cbName;
   CHAR  szEAName[SIZE_EA_NAME_VERSION+1];
} GEAL_VER;
#pragma  pack()

/****************************************************************************/
/* PROCEDURE NAME : GetEAVersion                                            */
/* AUTHOR         : Mark H                                                  */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
GetEAVersion (PSZ pszStrVersion, ULONG cbLength)
{
   FEAL_VER feaVer;
   GEAL_VER geaVer;
   EAOP2    eaop2;
   USHORT   usVerNameCount;               // length of version data
   CHAR     szFileName[MAX_PATH_SIZE];    // driver file name
   APIRET   rc;

   strcpy (pszStrVersion, "???.???");

   if (PrfQueryProfileString (HINI_USERPROFILE,
                              "PM_DEVICE_DRIVERS",
                              APP_NAME,
                              "",
                              szFileName,
                              sizeof (szFileName)))
   {
      /*
      ** Fill in GEALIST
      */
      strcpy (geaVer.szEAName, EA_NAME_VERSION);
      geaVer.cbName = SIZE_EA_NAME_VERSION;
      geaVer.cbList = sizeof (GEAL_VER);

      /*
      ** Initialize FEALIST
      */
      feaVer.cbList = sizeof(FEAL_VER);

      /*
      ** set EAOP2
      */
      geaVer.oNextEntryOffset = 0;

      eaop2.fpGEA2List = (PGEA2LIST)&geaVer;
      eaop2.fpFEA2List = (PFEA2LIST)&feaVer;
      eaop2.oError = 0L;

      rc = DosQueryPathInfo (szFileName,      /* File path                  */
                             3,               /* info level                 */
                             (PBYTE)&eaop2,   /* EAOP2 structure            */
                             sizeof (eaop2)); /* Size of EAOP2              */

      if (!rc &&                                    /* call succeeded       */
         (feaVer.cbList > sizeof(feaVer.cbList)) && /* has return value     */
         (feaVer.cbValue) &&                        /* value not zero       */
         (feaVer.cbName == SIZE_EA_NAME_VERSION))   /* name size must be = to .VERSION */
      {
         switch (feaVer.usEAT_Type)      /* see what type we got            */
         {
         case EAT_ASCII:                 /* ASCII type                      */
         {
            /*
            ** If we got EAT_ASCII out front, then following the type will
            ** be USHORT of count and follow by the name
            */
            usVerNameCount = (USHORT)feaVer.usCodePage + 1;

#ifdef DEBUG
            *((PSZ)((PUSHORT)&feaVer.usValueCount) + usVerNameCount) = '\0';
#endif

            DBPRINTF (("%s: Version is '%s', length is %d\n",
                       __FUNCTION__,
                       (PSZ)((PUSHORT)&feaVer.usValueCount),
                       usVerNameCount));

            if (usVerNameCount < cbLength - 1)
            {
               SafeStrNCpy (pszStrVersion,
                            (PSZ)((PUSHORT)&feaVer.usValueCount),
                            usVerNameCount);
            }
            break;
         }

         case EAT_MVMT:
         case EAT_MVST:
         {
            if (!feaVer.usValueCount)
              break;

            usVerNameCount = (USHORT)feaVer.usSizeVerName + 1;

#ifdef DEBUG
            *(feaVer.szVersionName + usVerNameCount) = '\0';
#endif

            DBPRINTF (("%s: Version is '%s', length is %d\n",
                       __FUNCTION__,
                       feaVer.szVersionName,
                       usVerNameCount));

            if (usVerNameCount < cbLength - 1)
            {
               SafeStrNCpy (pszStrVersion,
                            feaVer.szVersionName,
                            usVerNameCount);
            }
            break;
         }
         }
      }
   }

} /* end GetEAVersion */

// begin @EXJOB
/****************************************************************************/
/* PROCEDURE NAME : GetJobpropertiesSize                                    */
/* AUTHOR         : Yu-Chih,Shih                                            */
/* DATE WRITTEN   : 10/01/98                                                */
/* DESCRIPTION    : Return current version JobProperies size.               */
/*                                                                          */
/* PARAMETERS:                                                              */
/*    pDevice: Target device's DEVICEINFO                                   */
/*    fType  : Request type.                                                */
/*             JP_SIZE_CORE:   Omni defined Jobdata = sizeof(JOBPROPERTIES) */
/*             JP_SIZE_PLUGIN: Device specific Job data size.               */
/*             JP_SIZE_ALL:    JP_SIZE_CORE + JP_SIZE_PLUGIN                */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG
GetJobpropertiesSize (PDEVICEINFO pDevice, ULONG fType)
{
   ULONG ulSize = 0;

   if (fType & JP_SIZE_CORE)
   {
      ulSize = sizeof(JOBPROPERTIES);
   }

   // Only OMNI_CAP_EXTJOBDATA device can report JP_SIZE_PLUGIN
   if (pDevice->ulOmniDeviceTechnology & OMNI_CAP_EXTJOBDATA)
   {
      if (fType & JP_SIZE_PLUGIN)
      {
         BOOL        rc;
         ULONG       ulExtData;
         PFNSUBCLASS ppfnDevice = pDevice->pSubclassedFunctions;

         if (  ppfnDevice
            && ppfnDevice->pfnDeviceQuery
            )
         {
            rc = ppfnDevice->pfnDeviceQuery (pDevice,
                                             DEVICE_QUERY_EXTJOBDATA,
                                             &ulExtData,
                                             (PBYTE)NULL,
                                             DEVICE_EXTJOBDATA_CURRENT_SIZE);
            if (TRUE == rc)
            {
               ulSize += ulExtData;
            }
         }
      }
   }

   return ulSize;
}

/****************************************************************************/
/* PROCEDURE NAME : HasDeviceJobProperties                                  */
/* AUTHOR         : Yu-Chih,Shih                                            */
/* DATE WRITTEN   : 10/01/98                                                */
/* DESCRIPTION    : Device specific JobProperties validation check.         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
HasDeviceJobProperties (PDEVICEINFO    pDevice,
                        PJOBPROPERTIES pJobProperties)
{
   ULONG                      ulSize;
   PDEVICEJOBPROPERTIESFOOTER pDeviceJobProperties;
   PFNSUBCLASS                ppfnDevice = pDevice->pSubclassedFunctions;
   BOOL                       rc         = FALSE;

   if (  ppfnDevice
      && ppfnDevice->pfnDeviceQuery
      )
   {
      PBYTE  pbTmp;

      ulSize = sizeof (DEVICEJOBPROPERTIESFOOTER);

      pbTmp = ( (PBYTE)pJobProperties
              + pJobProperties->cb
              - ulSize);
      pDeviceJobProperties = (PDEVICEJOBPROPERTIESFOOTER)pbTmp;

      rc = ppfnDevice->pfnDeviceQuery (pDevice,
                                       DEVICE_QUERY_EXTJOBDATA,
                                       &ulSize,
                                       pDeviceJobProperties,
                                       DEVICE_EXTJOBDATA_DATA_VALID);
   }

   return rc;
}

/****************************************************************************/
/* PROCEDURE NAME : GetDeviceJobProperties                                  */
/* AUTHOR         : Yu-Chih,Shih                                            */
/* DATE WRITTEN   : 10/01/98                                                */
/* DESCRIPTION    : Returns device specific jobdata start address.          */
/*                                                                          */
/*                                                                          */
/*   JOBPROPERTIES                                                          */
/*   A  +--------------+    A                                               */
/*   |  |cb            |    |                                               */
/*   |  |              |    |                                               */
/*   |  |              |   Defined by OMNI base code                        */
/*   |  | OMNI Defined |    |                                               */
/*   cb +--------------+ A  V  A                                            */
/*   |  | Device       | |     |                                            */
/*   |  |  Specific    | cb2  Defined by plug-in code                       */
/*   |  |cb2           | |     |                                            */
/*   V  +--------------+ V     V                                            */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*    NULL: INVALID                                                         */
/*    OTHER:Start address of device jobdata                                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @JPN2H99- 08/01/99 - UCC [IBMJ]- Try to retrieve default data when       */
/*                                  validation chack fails                  */
/*                                                                          */
/****************************************************************************/
PBYTE
GetDeviceJobProperties (PDEVICEINFO    pDevice,
                        PJOBPROPERTIES pJobProperties)
{
   ULONG       cb, cb2;
   PBYTE       pJobDevice;
   PFNSUBCLASS ppfnDevice = pDevice->pSubclassedFunctions;
   BOOL        rc         = FALSE;

   cb  = pJobProperties->cb;
   cb2 = (ULONG)*((PBYTE)pJobProperties + cb - sizeof(ULONG));
   pJobDevice =  ((PBYTE)pJobProperties + cb - cb2);

   // Validation check;
   if (!HasDeviceJobProperties (pDevice, pJobProperties))
   {
// begin @JPN2H99
      if( cb > sizeof(JOBPROPERTIES))
      {
         // See if we can have enough room to set default
         // Get device extended jobproperties length
         rc = ppfnDevice->pfnDeviceQuery (pDevice,
                                          DEVICE_QUERY_EXTJOBDATA,
                                          &cb2,
                                          (PBYTE)NULL,
                                          DEVICE_EXTJOBDATA_CURRENT_SIZE);

         // Try to see if enough room.
         // We have to trust cb; application may only allocate cb in size
         // to treat jobproperties.
         if ( (TRUE == rc)
            &&(cb2 < (cb - sizeof(JOBPROPERTIES)))
            )
         {
            // We have enough room!!
            PBYTE pbJobDevice = (PBYTE)pJobProperties + cb - cb2;

            memset (pbJobDevice, 0, cb2);

            rc = ppfnDevice->pfnDeviceQuery (pDevice,
                                             DEVICE_QUERY_EXTJOBDATA,
                                             &cb2,
                                             pbJobDevice,
                                             DEVICE_EXTJOBDATA_SET_DEFAULT);
         }
      }

      if ( !rc)
// end @JPN2H99
      pJobDevice = NULL;
   }

   return pJobDevice;
}

/****************************************************************************/
/* PROCEDURE NAME : SafeCopyJobData                                         */
/* AUTHOR         : Yu-Chih,Shih                                            */
/* DATE WRITTEN   : 10/01/98                                                */
/* DESCRIPTION    : Copy JobProperties.                                     */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @JPN2H99 - 08/16/99 - UCC [IBM] - Add code to set default when Trg > Src */
/*                                                                          */
/****************************************************************************/
ULONG
SafeCopyJobData (PDEVICEINFO pDevice,
                 PBYTE       pSource,
                 ULONG       ulTarget,
                 PBYTE       pTarget)
{
   ULONG ulCopySize,
         ulSizeCore,
         ulSizePlugin,
         cbSrc;
   PFNSUBCLASS ppfnDevice = pDevice->pSubclassedFunctions;

   cbSrc        = ((PJOBPROPERTIES)pSource)->cb;
   ulSizeCore   = GetJobpropertiesSize (pDevice, JP_SIZE_CORE);
   ulSizePlugin = GetJobpropertiesSize (pDevice, JP_SIZE_PLUGIN);

   if (  (cbSrc == ulTarget)
      && (cbSrc == ulSizeCore)
      )
   {
      ulCopySize = cbSrc;
      memcpy (pTarget, pSource, ulCopySize);
   }
   else
   {
      // Has device specific JobProperties...
      // Check validation of device job propertties
      if (FALSE == HasDeviceJobProperties (pDevice, (PJOBPROPERTIES)pSource))
      {
         ULONG min;
         // Don't copy device job data if validation check fails
         // Copy core job data only and left device job data default
         min = (ulTarget >= cbSrc)      ? cbSrc      : ulTarget;
         min = (min      >= ulSizeCore) ? ulSizeCore : min;

// begin @JPN2H99
         memcpy (pTarget, pSource, min);
         ulCopySize = min;

         // If ulSizePlugin is not 0, try to see if we can have enough room.
         if (  ulSizePlugin
            &&(ulSizePlugin <= (ulTarget - ulSizeCore))
            )
         {
            // We have enough room!! Try to transfer default data.
            PBYTE pbJobDevice = (PBYTE)pTarget + ulTarget - ulSizePlugin;

            memset (pbJobDevice, 0, ulSizePlugin);

            ppfnDevice->pfnDeviceQuery (pDevice,
                                        DEVICE_QUERY_EXTJOBDATA,
                                        &ulSizePlugin,
                                        pbJobDevice,
                                        DEVICE_EXTJOBDATA_SET_DEFAULT);

            ((PJOBPROPERTIES)pTarget)->cb = ulTarget;
            ulCopySize += ulSizePlugin;
         }
// end @JPN2H99
         cbSrc = ulTarget = min;
      }
// begin @JPN2H99
      else
      {
         memcpy (pTarget, pSource, cbSrc);
         ulCopySize = cbSrc;
      }
// end @JPN2H99

//      if (ulTarget == cbSrc)
//      {
//         memcpy (pTarget, pSource, cbSrc);
//
//         ulCopySize = cbSrc;
//      }
//      else
      if (ulTarget != cbSrc)
      {
         ULONG cbDevice;
         PBYTE pJobDevice;
         PBYTE pTrg = pTarget;

         ulCopySize = ulTarget;

         // Case 1: ulTarget > cbSrc
         // Source is an old version job data
         //
         // Src(OLD)           Trg(NEW)
         // +--------+-------->+-------+
         // |OMNI    |         |OMNI   |
         // +--------+-------->|       |<-+
         // |DEVICE  |\        +-------+  |-Default
         // +--------+ \------>|       |<-+
         //           \        |DEVICE |
         //            \------>+-------+
         //
         if (ulTarget > cbSrc)
         {
            cbDevice = *(PULONG)(pSource + cbSrc - sizeof (ULONG));

            // Copy OMNI defined Jobdata
            memcpy (pTarget, pSource, cbSrc - cbDevice);

            // Copy Device specific
            pTrg       = pTarget + ulTarget - cbDevice;
            pJobDevice = pSource + cbSrc    - cbDevice;
            memcpy (pTrg, pJobDevice, cbDevice);
         }
         // Case 2: ulTarget < cbSrc
         // Source is newer version job data
         //
         // Src(NEW)           Trg(OLD)
         // +--------+-------->+-------+
         // |OMNI    |         |OMNI   |
         // |        |------->/+-------+
         // +--------+  /----/ |DEVICE |
         // |        |-/ /---->+-------+
         // |DEVICE  |  /
         // +--------+-/
         //
         else
         {
            // Copy OMNI defined Jobdata
            memcpy (pTarget, pSource, ulSizeCore);

            // Device specific Jobproperties is growable from
            // bottom to top!! Just copy ulTarget in size.
            cbDevice   = ulTarget - ulSizeCore;
            pJobDevice = pSource + cbSrc    - cbDevice;
            pTrg       = pTarget + ulTarget - cbDevice;
            memcpy (pTrg, pJobDevice, cbDevice);
         }
         // cb must always be ulTarget as we allocated
         ((PJOBPROPERTIES)pTarget)->cb                         = ulTarget;
         *(PULONG)((PBYTE)pTarget + ulTarget - sizeof (ULONG)) = ulSizePlugin;
      }
   }

   ((PJOBPROPERTIES)pTarget)->cb = ulSizeCore;   
   return ulCopySize;
}
// end @EXJOB

/****************************************************************************/
/* PROCEDURE NAME : SafeStrNCpy                                             */
/* AUTHOR         : Mark H                                                  */
/* DATE WRITTEN   : 07/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
PSZ
SafeStrNCpy (register PSZ   pszOut,
             register PSZ   pszIn,
             register ULONG cbLength)
{
   PSZ    pszRet = pszOut;

   assertF (pszIn);
   assertF (pszOut);

   if (  !pszIn
      || !pszOut
      )
      return NULL;

   if (0 == cbLength)
   {
      assertT (0 == cbLength);

      return pszRet;
   }

   while (  1 < cbLength
         && *pszIn
         )
   {
      *pszOut++ = *pszIn++;

      cbLength--;
   }

   *pszOut = '\0';

   return pszRet;
}
