/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Form Dialog Control Functions                           */
/* SOURCE NAME    : FORMS.C                                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/26/93                                                 */
/* DESCRIPTION    : Contains all routines that manage Paper Sizes within    */
/*                  the Omni driver.  This includes all routines that       */
/*                  add, delete, modify and view paper sizes from the       */
/*                  Printer Properties notebook (Paper Size page).          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @100713 - 10/04/94 - MFR [IBM] - Need to default to A4 for most          */
/*                                  countries but USA, Canada, and Brazil.  */
/*                                  Also default to CM not inches.          */
/*                                                                          */
/* @147476 - 03/08/96 - MFR [IBM] - we were reporting the incorrect margins */
/*                                  when job properties had orientation     */
/*                                  set to Landscape.                       */
/* @153333 - 04/29/96 - MFR [IBM] - Rotated landscape margins clockwise     */
/*                                  so envelopes report absolute margins to */
/*                                  AmiPro correctly too.                   */
/* @3677   - 06/24/98 - UCC [IBMJ]  Change to send printer landscape code.  */
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/*                                          support                         */
/****************************************************************************/

// os2 includes
#define INCL_DOS
#define INCL_PM
#define INCL_WIN
#define INCL_DEV
#include <os2.h>

// @DBCS
#define INCL_VMANDDI
#include <ddi.h>
#include <pmddi.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"

#define isdigit(c)          ('0' <= c && c <= '9')

// Used to help distinguish "add, delete, and modify" buttons for
// "Paper size" page from other pages
/* @170406 SIZE_INDICATOR is defined in both forms.c and in trays.c.  I dont
** know     it has the same name but it is used to make the common button ids
** different for the connection, tray, and form page.  connect.c does not use
** this define so it is 0, trays.c uses 1 and forms.c uses 2.
*/
#define SIZE_INDICATOR  0x00020000

// Globals
CLASSINFO   eci;        // Data about the *public* entry field control
                        // Note, we don't change it. So it should be safe as
                        // global data

ULONG
HinchToHmm (ULONG ulHinch)
{
   float f;

   if (0 == ulHinch)
      return 0;

   f = (float)ulHinch;
   f /= 100.0;         // --> inches
   f *= 2.54*1000.0;   // --> hundreds of millimeters

   return (ULONG)f;
}

ULONG
HmmToHinch (ULONG ulHmm)
{
   float f;

   if (0 == ulHmm)
      return 0;

   f = (float)ulHmm;
   f /= 2.54*10.0;     // --> inches

   return (ULONG)f;
}

/****************************************************************************/
/* PROCEDURE NAME : GetEntryText                                            */
/* AUTHOR         : Matt R. and Mark H.                                     */
/* DATE WRITTEN   : 4/11/94                                                 */
/* DESCRIPTION    : Reads floating point (inches or cm) from entry fields   */
/*                  the Omni driver has subclassed.                         */
/*                  Used mainly for showing paper sizes and margins in      */
/*                  the Printer Properties notebook.                        */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwndDialog : Dialog's window handle that       */
/*                                        contains the entryfield we are    */
/*                                        going to update.                  */
/*                  (USHORT) id         : ID of the entryfield to update.   */
/*                  (CHAR)   cDecimal   : decimal separator for current     */
/*                                        country.                          */
/*                  (BOOL)   bUseInches : if TRUE format 'ulVal' for inches */
/*                                        else format for centimeters.      */
/*                                                                          */
/* RETURN VALUES:   None.                                                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*          12/11/98 - MarkH - added NLS code to replace decimal separator  */
/*                             to the correct one for the current country   */
/*                             we are running on.                           */
/****************************************************************************/
ULONG
GetEntryValue (HWND   hwndDialog,
               USHORT id,
               CHAR   cDecimal,
               BOOL   bUseInches)
{
    HWND    hwnd;
    CHAR    achText[100];
    float   f;

    hwnd = WinWindowFromID (hwndDialog, id);

    // Convert to inches or CM only when we have some data.
    // WinQueryWindowText return code equals the length of the
    // text string.  LEU 10/97
    if (WinQueryWindowText (hwnd, sizeof (achText), achText))
    {
       PSZ pszSeparator = strchr (achText, cDecimal);

       if (pszSeparator)
       {
          *pszSeparator = '.';
       }

       // Scanf returns the count of items scanned in.  If it returns a
       // zero, then it cannot parse the requested item, in this case a
       // float (ex: if a "." is in the window)
       if (sscanf (achText, "%f", &f))
       {
          if (bUseInches)
             f *= 25.4;   // --> inches
          else
             f *= 10.0;   // --> millimeters

          f *= 100.0;         // In hundreds of a millimeter
       }
       else
          // Default to zero if we cannot scanf fails and
          f = 0;
    }
    else
       // Default to zero if we there is no String in window
       f = 0;

    return (ULONG)f;
}

/****************************************************************************/
/* PROCEDURE NAME : SetEntryText                                            */
/* AUTHOR         : Matt R. and Mark H.                                     */
/* DATE WRITTEN   : 4/11/94                                                 */
/* DESCRIPTION    : Formats and writes floating point (inches or cm )       */
/*                  to entry fields the Omni driver has subclassed.         */
/*                  Used mainly for showing paper sizes and margins in      */
/*                  the Printer Properties notebook.                        */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwndDialog : Dialog's window handle that       */
/*                                        contains the entryfield we are    */
/*                                        going to update.                  */
/*                  (USHORT) id         : ID of the entryfield to update.   */
/*                  (CHAR)   cDecimal   : decimal separator for current     */
/*                                        country.                          */
/*                  (ULONG)  ulVal      : value to place in entry field.    */
/*                  (BOOL)   bUseInches : if TRUE format 'ulVal' for inches */
/*                                        else format for centimeters.      */
/*                                                                          */
/* RETURN VALUES:   None.                                                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @NLS01 - 01/22/95 - MattR - added NLS code to replace decimal separator  */
/*                             to the correct one for the current country   */
/*                             we are running on.                           */
/****************************************************************************/
VOID
SetEntryText (HWND   hwndDialog,
              USHORT id,
              CHAR   cDecimal,
              ULONG  ulVal,
              BOOL   bUseInches)
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  CHAR    achText[32];
  float   fValue;
  PSZ     pszDecimal;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Place ulValue into a local variable in floating pt. format
  // this value represents a size in hundreths of a millimeter.
  fValue = (float)ulVal;

  // Convert value to scalar from hundreths of a millimeter to
  // centimeters preserving remainder.
  fValue /= 1000.0;

  // If value passed in is to be dispayed in inches convert
  // value to inches otherwise leave in centimeters
  if (bUseInches)
     fValue /= 2.54;

  // format floating point value into text string with 2 decimal places
  sprintf (achText, "%.2f", fValue);

  // replace decimal point with one appropriate for current processes
  // country/codepage
  pszDecimal = strchr( achText, '.' );
  *pszDecimal = cDecimal;

  // Display formatted value into entryfield using ID passed in
  WinSetWindowText (WinWindowFromID (hwndDialog, id), achText);

} /* end SetEntryText */

BOOL
GetCheck (HWND hwndDlg, USHORT id)
{
   return (BOOL)WinSendDlgItemMsg (hwndDlg,
                                   id,
                                   BM_QUERYCHECK,
                                   (MPARAM)NULL,
                                   (MPARAM)NULL) == 1;
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY
SubClassedEntryWndProc (HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   PCHRMSG  p;
   BOOL     bFilter = FALSE;

   switch (msg)
   {
   case WM_CHAR:
      if (SHORT1FROMMP (mp1) &  KC_VIRTUALKEY)
      {
         switch (SHORT2FROMMP (mp2))
         {
         case VK_ESC:
         case VK_TAB:
         case VK_BACKTAB:
         case VK_INSERT:
         case VK_DELETE:
         case VK_BACKSPACE:
         case VK_NEWLINE:
         case VK_ENTER:
         case VK_END:
         case VK_HOME:
         case VK_UP:
         case VK_DOWN:
         case VK_LEFT:
         case VK_RIGHT:
         case VK_F1:
            // Allow these keys through to the entry field
            return (*eci.pfnWindowProc) (hwnd, msg, mp1, mp2);
         }
      }

      p = CHARMSG (&msg);                          /* addres char structure  */
      if ((p->fs & (KC_KEYUP|KC_CHAR))==KC_CHAR)   /* ONLY key up in transit */
      {
         if (!isdigit(p->chr) && p->chr != '.')
            bFilter = TRUE;
      }
   }

   if (bFilter)
   {
      WinAlarm (HWND_DESKTOP,WA_WARNING);    /* beep */
      return (MRESULT)(TRUE);                /* fail it */
   }
   else
      return (*eci.pfnWindowProc) (hwnd, msg, mp1, mp2);
}

/****************************************************************************/
/* PROCEDURE NAME : CreateUniqueFormID                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10-18-93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG
CreateUniqueFormID (PDEVICEINFO pDevice)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   ULONG       ulFormID = 0;
   PUSERFORM   pUserForm;
   BOOL        bIDClash = FALSE;
#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bCREATEUNIQUEFORMID, "%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);

   // if we have user defined forms to check against
   if (pDevice->pUserDefData->usNumForms > 0)
   {
      // sanity check pointers
      assertF (pDevice->pUserDefData->pUserFORMS);

      // as long as we don't exceed some large max ID number
      // we'll keep trying to assign a unique ID
      while (ulFormID < MAX_USER_DEF_ID)
      {
         // assign form list pointer to local variable
         pUserForm = pDevice->pUserDefData->pUserFORMS;

         // for each entry in linked list of user defined forms
         while ((pUserForm != NULL) && !bIDClash)
         {
            if (pUserForm->fiUser.ulFormID == ( ulFormID | DEF_TYPE_USER))
            {
               bIDClash = TRUE;
               break;
            }

            pUserForm = pUserForm->pNextForm;
         }

         // we had a clash try next ID
         if (bIDClash)
         {
            ulFormID++;
            bIDClash = FALSE;
         }
         else
         {
            // we have found a match let's get out of while loop
            break;
         }
      }
   }

   DBPRINTIF ((pDbg->bCREATEUNIQUEFORMID, "CreateUniqueFormID(): Exit; ulFormID = %d\n",
               ulFormID | DEF_TYPE_USER));

   return ulFormID | DEF_TYPE_USER;

} /* end CreateUniqueFormID */

/****************************************************************************/
/* PROCEDURE NAME : AddUserDefinedFormEntry                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
AddUserDefinedFormEntry (PDEVICEINFO pDevice, PFORMINFO2 pNewFormDef)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   PUSERFORM   pUserForm;
   BOOL        bSuccess = TRUE;
   BOOL        rc;
#ifdef DEBUG
   PDEBUGINFO  pDbg     = &globals.DebugInfo;
   BOOL        bShow    = pDbg->bADDUSERDEFINEDFORMENTRY;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((bShow, "%s(): Enter; pNewFormDef = %x\n", __FUNCTION__, pNewFormDef));

   assertF (pDevice);
   assertF (pNewFormDef);

   // allocate user form struct and add to current list
   pUserForm = (PUSERFORM)GplMemoryAlloc (globals.pvSharedHeap,
                                          sizeof (USERFORM));

   assertF (pUserForm);

   // if memory is allocated all right
   if (pUserForm)
   {
      USHORT us;

      /* Since we are altering global data, request a global semaphore
      ** for the changes we need to make...
      */
      rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
      assertT (rc);

      // Copy in new Form entry from one passed in
      pUserForm->pNextForm                 = NULL;
      pUserForm->fiUser.ulFormID           = pNewFormDef->ulFormID;
      pUserForm->fiUser.ulFormClass        = pNewFormDef->ulFormClass;
      pUserForm->fiUser.ulNameID           = pNewFormDef->ulNameID;
      pUserForm->fiUser.hcInfo             = pNewFormDef->hcInfo;
      pUserForm->fiUser.usLenCmdSelectForm = pNewFormDef->usLenCmdSelectForm;

      for (us = 0; us < pUserForm->fiUser.usLenCmdSelectForm; us++)
         pUserForm->fiUser.szCmdSelectForm[us] = pNewFormDef->szCmdSelectForm[us];

      // init values not yet established
      pUserForm->fiUser.hcInfo.xPels        =  0;
      pUserForm->fiUser.hcInfo.yPels        =  0;
      pUserForm->fiUser.hcInfo.flAttributes =  0;

      DBPRINTIF ((bShow, "AddUserDefinedFormEntry(): Adding Form = %s; ID=%lu\n",
                  pUserForm->fiUser.hcInfo.szFormname,
                  pUserForm->fiUser.ulFormID));

      // Add new form structure to linked list
      if (pDevice->pUserDefData->pUserFORMS == NULL)
      {
         // first form being defined add at top of list
         pDevice->pUserDefData->pUserFORMS = pUserForm;
      }
      else
      {
         assertF (pDevice->pUserDefData->pLastUserFORMS);

         // add at end of list
         pDevice->pUserDefData->pLastUserFORMS->pNextForm = pUserForm;
      }

      pDevice->pUserDefData->pLastUserFORMS = pUserForm;
      pDevice->pUserDefData->usNumForms++;

      rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
      assertT (rc);
   }

   DBPRINTIF ((bShow, "AddUserDefinedFormEntry(): Exit; bSuccess = %d\n", bSuccess));

#ifndef DEBUG
   rc++;       // Avoid compiler warnings
#endif

   return bSuccess;

} /* end AddUserDefinedFormEntry */

/****************************************************************************/
/* PROCEDURE NAME : DeleteUserDefinedFormEntry                              */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES  :                                                         */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
DeleteUserDefinedFormEntry (PDEVICEINFO pDevice, ULONG ulFormID)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   PUSERFORM   pCurrent;
   PUSERFORM   pLast    = NULL;
   BOOL        bSuccess = FALSE;
   BOOL        rc;
#ifdef DEBUG
   PDEBUGINFO  pDbg     = &globals.DebugInfo;
#endif

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  DBPRINTIF ((pDbg->bDELETEUSERDEFINEDFORMENTRY,
              "%s(): Enter; ulFormID = %d\n",
              __FUNCTION__, ulFormID));

  assertF (pDevice);

  if (!IsFormPartOfAConnection (pDevice, ulFormID))
  {
     if (ulFormID & DEF_TYPE_USER)
     {
        /* Since we are altering global data, request a global semaphore
        ** for the changes we need to make...
        */
        rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
        assertT (rc);

        // search for Form ID in linked list and once found delete entry
        // from list and fix chain
        pCurrent = pDevice->pUserDefData->pUserFORMS;

        while (pCurrent != NULL)
        {
           // matching FormID
           if (pCurrent->fiUser.ulFormID == ulFormID)
           {
              // we are in middle of list chain
              if (pLast != NULL)
              {
                 pLast->pNextForm = pCurrent->pNextForm;
              }
              else
              {
                 // deleting first entry
                 pDevice->pUserDefData->pUserFORMS = pCurrent->pNextForm;
              }

              GplMemoryFree (pCurrent);

              pDevice->pUserDefData->usNumForms--;

              bSuccess = TRUE;
              break;
           }

           // advnce list pointers
           pLast    = pCurrent;
           pCurrent = pCurrent->pNextForm;
        }

        rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
        assertT (rc);
     }
     else
     {
        // Inform user that only user defined forms are deletable
        WinMessageBox ((HWND)HWND_DESKTOP,
                       (HWND)NULL,
                       globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_MSG_FORM],
                       globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_FORM],
                       1L,
                       MB_OK | MB_INFORMATION);
     }
  }
  else
  {
     // Inform user that only unconnected forms are deletable
     WinMessageBox ((HWND)HWND_DESKTOP,
                    (HWND)NULL,
                    globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_FORM_IN_USE],
                    globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_FORM],
                    1L,
                    MB_OK | MB_INFORMATION);
  }

  DBPRINTIF ((pDbg->bDELETEUSERDEFINEDFORMENTRY, "DeleteUserDefinedFormEntry(): Exit\n"));

#ifndef DEBUG
   rc++;       // Avoid compiler warnings
#endif

  return bSuccess;

} /* end DeleteUserDefinedFormEntry */

/****************************************************************************/
/* PROCEDURE NAME : FreeUserDefinedFormList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
FreeUserDefinedFormList (PDEVICEINFO pDevice)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   PUSERFORM  pDelete, pNext;
   BOOL       rc;
#ifdef DEBUG
   PDEBUGINFO pDbg            = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bFREEUSERDEFINEDFORMLIST,"%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);

   if (!pDevice || !pDevice->pUserDefData)
      return;

   /* Since we are altering global data, request a global semaphore
   ** for the changes we need to make...
   */
   rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
   assertT (rc);

   // search for FORM ID in linked list and once found delete entry
   // from list and fix chain
   pDelete = pDevice->pUserDefData->pUserFORMS;

   while (pDelete != NULL)
   {
      pNext = pDelete->pNextForm;

      GplMemoryFree (pDelete);

      pDelete = pNext;
   }

   // reset linked list values
   pDevice->pUserDefData->bFormsInit     = FALSE;
   pDevice->pUserDefData->pUserFORMS     = NULL;
   pDevice->pUserDefData->pLastUserFORMS = NULL;
   pDevice->pUserDefData->usNumForms     = 0;

   rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
   assertT (rc);

   DBPRINTIF ((pDbg->bFREEUSERDEFINEDFORMLIST,"FreeUserDefinedFormList(): Exit\n"));

#ifndef DEBUG
   rc++;       // Avoid compiler warnings
#endif

} /* FreeUserDefinedFormList */

/****************************************************************************/
/* PROCEDURE NAME : SaveUserDefinedFormList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
SaveUserDefinedFormList (PSZ pszAppName, PDEVICEINFO pDevice)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   USHORT           usFormsProcessed   = 0;
   USHORT           usTotalForms;
   PUSERFORM        pCurrent;
   CHAR             szTemp[LEN_STRING];
   PEXTFORMINFOSAVE pForm;
   PBYTE            pbBuffer;
   ULONG            ulDataSize;
   BOOL             bSuccess           = FALSE;

#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bSAVEUSERDEFINEDFORMLIST,"%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);

   // write out Num Forms to INI
   usTotalForms = pDevice->pUserDefData->usNumForms;

   _itoa (usTotalForms, szTemp, 10);

   PrfWriteProfileString (HINI_SYSTEMPROFILE,
                          pszAppName,
                          (PSZ)KEYNAME_NUM_USER_FORMS,
                          (PVOID)szTemp);

   // allocate memory to hold all Forminfo and cast to FORMINFO2 struct
   ulDataSize = usTotalForms * sizeof (EXTFORMINFO_1_1);

   // if we have form strucs to allocate for
   if (ulDataSize > 0)
   {
      pbBuffer = GplMemoryAlloc (0, ulDataSize);

      assertF (pbBuffer);

      if (pbBuffer)
      {
         pForm = (PEXTFORMINFOSAVE)pbBuffer;

         // set up start of list pointer
         pCurrent = pDevice->pUserDefData->pUserFORMS;

         // while we have Forms in linked list to process
         while (pCurrent != NULL && usFormsProcessed <= usTotalForms)
         {
            // copy out to memory area
            pForm->ulFormID           = pCurrent->fiUser.ulFormID;
            pForm->ulFormCap          = pCurrent->fiUser.ulFormCap;
            pForm->ulFormClass        = pCurrent->fiUser.ulFormClass;
            pForm->ulNameID           = pCurrent->fiUser.ulNameID;

            pForm->usLenCmdSelectForm = min (pCurrent->fiUser.usLenCmdSelectForm,
                                             sizeof (pForm->szCmdSelectForm));
            memcpy (pForm->szCmdSelectForm,
                    pCurrent->fiUser.szCmdSelectForm,
                    sizeof (pForm->szCmdSelectForm));
            assertT (pCurrent->fiUser.usLenCmdSelectForm > sizeof (pForm->szCmdSelectForm));

            pForm->hcInfo = pCurrent->fiUser.hcInfo;
            pForm->bHasBeenSetup      = pCurrent->fiUser.bHasBeenSetup;

            // increment counter
            usFormsProcessed++;

            // advance buffer pointer
            pForm++;

            // advance list pointer
            pCurrent = pCurrent->pNextForm;
         }

         PrfWriteProfileData (HINI_SYSTEMPROFILE,
                              pszAppName,
                              (PSZ)KEYNAME_USER_FORMS,
                              (PVOID)pbBuffer,
                              ulDataSize);

         PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                pszAppName,
                                (PSZ)KEYNAME_VERSION_KEY_USER_FORMS,
                                (PVOID)KEYNAME_VERSION_DATA_USER_FORMS);

         GplMemoryFree (pbBuffer);

         bSuccess = TRUE;
      }
      else
      {
         bSuccess = FALSE;
      }
   }

   if (FALSE == bSuccess)
   {
      // overwrite any old entries with NULL
      PrfWriteProfileData (HINI_SYSTEMPROFILE,
                           pszAppName,
                           (PSZ)KEYNAME_USER_FORMS,
                           (PVOID)NULL,
                           0);

      PrfWriteProfileData (HINI_SYSTEMPROFILE,
                           pszAppName,
                           (PSZ)KEYNAME_VERSION_KEY_USER_FORMS,
                           (PVOID)NULL,
                           0);
   }

   DBPRINTIF ((pDbg->bSAVEUSERDEFINEDFORMLIST,"SaveUserDefinedFormList(): Exit\n"));

   if (0 == usTotalForms)
      // Function succeeded... dont count on bSuccess
      return TRUE;
   else
      return bSuccess;

} /* SaveUserDefinedFormList */

/****************************************************************************/
/* PROCEDURE NAME : ReadUserDefinedFormList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ReadUserDefinedFormList (PSZ pszAppName, PDEVICEINFO pDevice)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   PBYTE       pbInput             = (PBYTE)NULL;
   PFORMINFO2  pForm, pBase;
   ULONG       ulDataSize;
   BOOL        bSuccess            = TRUE;
   CHAR        szTemp[LEN_STRING];
   USHORT      usFormsProcessed    = 0;
   USHORT      usNumUserForms      = 0;
   BOOL        bRC;
   APIRET      rc;

#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bREADUSERDEFINEDFORMLIST, "%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);

   if (!pDevice->pUserDefData->bFormsInit)
   {
      ulDataSize = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                          pszAppName,
                                          (PSZ)KEYNAME_NUM_USER_FORMS,
                                          (PSZ)NULL,
                                          (PVOID)szTemp,
                                          LEN_STRING);

      DBPRINTF (("ReadUserDefinedFormList(): ulDataSize = %d\n", ulDataSize));

      // If we have an INI entry
      if (ulDataSize != 0)
      {
         // write out Num Forms to INI
         usNumUserForms = atoi (szTemp);

         DBPRINTF (("ReadUserDefinedFormList(): usNumForms = %d\n", usNumUserForms));

         if (usNumUserForms > 0)
         {
            // allocate memory to hold all Forminfo and cast to FORMINFO2 struct
            ulDataSize = usNumUserForms * sizeof (FORMINFO2);

            pBase = GplMemoryAlloc (0, ulDataSize);
            assertF (pBase);

            if (pBase)
            {
               ULONG  ulEntrySize;

               // read in profile data
               bRC = PrfQueryProfileSize (HINI_SYSTEMPROFILE,
                                          pszAppName,
                                          KEYNAME_USER_FORMS,
                                          &ulDataSize);
               assertF (bRC);

               if (bRC)
               {
                  ulEntrySize = ulDataSize / usNumUserForms;

                  if (SIZEOF_EXTFORMINFO_1_1 == ulEntrySize)
                  {
                     PEXTFORMINFO_1_1  pFormInput;

                     DBPRINTF (("%d user defined forms are in 1.1 format!\n",
                                usNumUserForms));

                     ulDataSize = usNumUserForms * sizeof (EXTFORMINFO_1_1);

                     pbInput = GplMemoryAlloc (0, ulDataSize);
                     assertF (pbInput);

                     bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                                pszAppName,
                                                (PSZ)KEYNAME_USER_FORMS,
                                                (PVOID)pbInput,
                                                &ulDataSize);
                     assertF (bRC);

                     // if data read in correctly
                     if (bRC)
                     {
                        // init current entry pointer to each Form structure
                        pForm      = pBase;
                        pFormInput = (PEXTFORMINFO_1_1)pbInput;

                        // while we have Forms in linked buffer to process
                        while (usFormsProcessed < usNumUserForms)
                        {
                           // Translate from external format to internal format
                           pForm->ulFormID           = pFormInput->ulFormID;
                           pForm->ulFormCap          = pFormInput->ulFormCap;
                           pForm->ulDJPid            = 0;
                           pForm->ulFormClass        = pFormInput->ulFormClass;
                           pForm->ulNameID           = pFormInput->ulNameID;
                           pForm->usLenCmdSelectForm = pFormInput->usLenCmdSelectForm;
                           memcpy (pForm->szCmdSelectForm,
                                   pFormInput->szCmdSelectForm,
                                   sizeof (pFormInput->szCmdSelectForm));
                           pForm->hcInfo             = pFormInput->hcInfo;
                           pForm->bHasBeenSetup      = pFormInput->bHasBeenSetup;

                           AddUserDefinedFormEntry (pDevice, pForm);

                           // increment counter
                           usFormsProcessed++;

                           // advance buffer pointer
                           pForm++;
                           pFormInput++;
                        }

                        // free memory
                        rc = GplMemoryFree (pBase);
                        assertT (rc);
                     }
                     else
                     {
                        bSuccess = FALSE;
                     }
                  }
                  else if (SIZEOF_EXTFORMINFO_1_5 == ulEntrySize)
                  {
                     PEXTFORMINFO_1_5  pFormInput;

                     DBPRINTF (("%d user defined forms are in 1.5 format!\n",
                                usNumUserForms));

                     ulDataSize = usNumUserForms * sizeof (EXTFORMINFO_1_5);

                     pbInput = GplMemoryAlloc (0, ulDataSize);
                     assertF (pbInput);

                     bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                                pszAppName,
                                                (PSZ)KEYNAME_USER_FORMS,
                                                (PVOID)pbInput,
                                                &ulDataSize);
                     assertF (bRC);

                     // if data read in correctly
                     if (bRC)
                     {
                        // init current entry pointer to each Form structure
                        pForm      = pBase;
                        pFormInput = (PEXTFORMINFO_1_5)pbInput;

                        // while we have Forms in linked buffer to process
                        while (usFormsProcessed < usNumUserForms)
                        {
                           // Translate from external format to internal format
                           pForm->ulFormID           = pFormInput->ulFormID;
                           pForm->ulFormCap          = pFormInput->ulFormCap;
                           pForm->ulDJPid            = pFormInput->ulDJPid;
                           pForm->ulFormClass        = pFormInput->ulFormClass;
                           pForm->ulNameID           = pFormInput->ulNameID;
                           pForm->usLenCmdSelectForm = pFormInput->usLenCmdSelectForm;
                           memcpy (pForm->szCmdSelectForm,
                                   pFormInput->szCmdSelectForm,
                                   sizeof (pFormInput->szCmdSelectForm));
                           pForm->hcInfo             = pFormInput->hcInfo;
                           pForm->bHasBeenSetup      = pFormInput->bHasBeenSetup;

                           AddUserDefinedFormEntry (pDevice, pForm);

                           // increment counter
                           usFormsProcessed++;

                           // advance buffer pointer
                           pForm++;
                           pFormInput++;
                        }

                        // free memory
                        rc = GplMemoryFree (pBase);
                        assertT (rc);
                     }
                     else
                     {
                        bSuccess = FALSE;
                     }
                  }
                  else if (SIZEOF_EXTFORMINFO_1_21 <= ulEntrySize)
                  {
                     PEXTFORMINFO_1_21  pFormInput;

#ifdef DEBUG
                     if (SIZEOF_EXTFORMINFO_1_21 == ulEntrySize)
                     {
                        DBPRINTF (("%d user defined forms are in 1.21 format!\n",
                                   usNumUserForms));
                     }
                     else
                     {
                        DBPRINTF (("%d user defined forms are in ***UNKNOWN** format! ulEntrySize = %d ulDataSize = %d\n",
                                   usNumUserForms,
                                   ulEntrySize,
                                   ulDataSize));
                        assertT (SIZEOF_EXTFORMINFO_1_21 != ulEntrySize);
                     }
#endif

                     ulDataSize = usNumUserForms * ulEntrySize;

                     pbInput = GplMemoryAlloc (0, ulDataSize);
                     assertF (pbInput);

                     bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                                pszAppName,
                                                (PSZ)KEYNAME_USER_FORMS,
                                                (PVOID)pbInput,
                                                &ulDataSize);
                     assertF (bRC);

                     // if data read in correctly
                     if (bRC)
                     {
                        // init current entry pointer to each Form structure
                        pForm      = pBase;
                        pFormInput = (PEXTFORMINFO_1_21)pbInput;

                        // while we have Forms in linked buffer to process
                        while (usFormsProcessed < usNumUserForms)
                        {
                           // Translate from external format to internal format
                           pForm->ulFormID           = pFormInput->ulFormID;
                           pForm->ulFormCap          = pFormInput->ulFormCap;
                           pForm->ulDJPid            = pFormInput->ulDJPid;
                           pForm->ulFormClass        = pFormInput->ulFormClass;
                           pForm->ulNameID           = pFormInput->ulNameID;
                           pForm->usLenCmdSelectForm = pFormInput->usLenCmdSelectForm;
                           memcpy (pForm->szCmdSelectForm,
                                   pFormInput->szCmdSelectForm,
                                   sizeof (pFormInput->szCmdSelectForm));
                           pForm->hcInfo             = pFormInput->hcInfo;
                           pForm->bHasBeenSetup      = pFormInput->bHasBeenSetup;

                           AddUserDefinedFormEntry (pDevice, pForm);

                           // increment counter
                           usFormsProcessed++;

                           // advance buffer pointer
                           pForm++;

                           pFormInput = (PEXTFORMINFO_1_21)( (PBYTE)pFormInput
                                                           + ulEntrySize
                                                           );
                        }

                        // free memory
                        rc = GplMemoryFree (pBase);
                        assertT (rc);
                     }
                     else
                     {
                        bSuccess = FALSE;
                     }
                  }
                  else
                  {
                     DBPRINTF (("%d user defined forms are in ***UNKNOWN** format! ulEntrySize = %d ulDataSize = %d\n",
                                usNumUserForms,
                                ulEntrySize,
                                ulDataSize));

                     assertstring ("Unknown size!\n");

                     rc = GplMemoryFree (pBase);
                     assertT (rc);

                     bSuccess = FALSE;
                  }
               }
            }
            else
            {
               bSuccess = FALSE;
            }
         }
      }
      else
      {
         DBPRINTF (("ReadUserDefinedFormList(): No INI data\n"));
      }

      // indicate we have read in form data
      pDevice->pUserDefData->bFormsInit = TRUE;
   }

   // Clean up
   if (pbInput)
   {
      rc = GplMemoryFree (pbInput);
      assertT (rc);
   }

   DBPRINTIF ((pDbg->bREADUSERDEFINEDFORMLIST,"ReadUserDefinedFormList(): Exit\n"));

#ifndef DEBUG
   rc++;       // Avoid compiler warnings
#endif

   return bSuccess;

} /* ReadUserDefinedFormList */

/****************************************************************************/
/* PROCEDURE NAME : FormNameFromID                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/23/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
FormNameFromID (PDRIVERINFO pDriver,
                PDEVICEINFO pDevice,
                PSZ         pszFormName,
                ULONG       ulFormID)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   ULONG       ulDefType;
   BOOL        bFormNameFound = FALSE;
   ULONG       i;
   ULONG       ulNumDefined;
   PFORMINFO2  pForms;
#ifdef DEBUG
   PDEBUGINFO  pDbg           = &globals.DebugInfo;
#endif

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  DBPRINTIF ((pDbg->bFORMNAMEFROMID,
              "%s(): Enter; Form ID = %lu\n",
              __FUNCTION__,
              ulFormID));

  // Find out how form defined (in driver, in device, by user )
  ulDefType = MAKEULONG (0, HIUSHORT (ulFormID));

  switch (ulDefType)
  {
  case DEF_TYPE_DRIVER:  // No hi word value
  {
     DBPRINTIF ((pDbg->bFORMNAMEFROMID, "%s(): DEF_TYPE_DRIVER\n", __FUNCTION__));

     ulNumDefined = pDevice->ulNumForms;
     pForms       = pDevice->pFORMS;

     // For now only driver table definitions allowed
     // loop through table of forms until matching form ID
     // found
     for (i = 0; i < ulNumDefined; i++)
     {
        DBPRINTIF ((pDbg->bFORMNAMEFROMID,
                    "%s(): Comparing to Form ID=%lu\n",
                    __FUNCTION__, pForms[i].ulFormID));

        if (ulFormID == pForms[i].ulFormID)
        {
           // Now we must determine if the string is in our stringtable
           // or a it is a string the user defined in hcInfo structure
           if (pForms[i].ulNameID == FORM_STRING_UNLISTED)
           {
              SafeStrNCpy (pszFormName,
                           pForms[i].hcInfo.szFormname,
                           FORMNAME_SIZE + 1);
           }
           else
           {
              ULONG ulIdx;

              ulIdx = pForms[i].ulNameID - STRING_TABLE_BASE;

              assertF (globals.pbStringTable[ulIdx]);

              SafeStrNCpy (pszFormName,
                           globals.pbStringTable[ulIdx],
                           FORMNAME_SIZE + 1);
           }

           bFormNameFound = TRUE;
           break;
        }
     }
     break;
  }

  case DEF_TYPE_DEVICE:
     break;

  case DEF_TYPE_USER:
  {
     PUSERFORM pUserForm;

     DBPRINTIF ((pDbg->bFORMNAMEFROMID, "FormNameFromID(): DEF_TYPE_USER\n"));

     // assign form list pointer to local variable
     pUserForm = pDevice->pUserDefData->pUserFORMS;

     // for each entry in linked list of user defined forms
     while (pUserForm != NULL)
     {
        if (pUserForm->fiUser.ulFormID == ulFormID)
        {
           SafeStrNCpy (pszFormName,
                        pUserForm->fiUser.hcInfo.szFormname,
                        FORMNAME_SIZE + 1);

           bFormNameFound = TRUE;
           break;
        }

        pUserForm = pUserForm->pNextForm;
     }
     break;
  }

  default:
     assertstring ("Not handled!\n");
     break;
  }

  DBPRINTIF ((pDbg->bFORMNAMEFROMID,
              "FormNameFromID(): Exit; pszFormName = %s\n",
              pszFormName));
  assertF (bFormNameFound);

  return bFormNameFound;

} /* end FormNameFromID */

/****************************************************************************/
/* PROCEDURE NAME : FormIDFromName                                          */
/* AUTHOR         :                                                         */
/* DATE WRITTEN   : 06/14/94                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG
FormIDFromName (PDEVICEINFO pDevice,
                PSZ         pszName)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   ULONG        i;
   ULONG        ulMatchIdx   = 0;
   ULONG        ulNumDefined;
   ULONG        ulIdx;
   PFORMINFO2   pForms;
#ifdef DEBUG
   PDEBUGINFO   pDbg         = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bFORMIDFROMNAME, "%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);
   assertF (pszName);

   ulNumDefined = pDevice->ulNumForms;
   pForms       = pDevice->pFORMS;
   assertF (pForms);

   assertF (pszName);

   if (strlen (pszName) > 0)
   {
      for (i = 0; i < ulNumDefined; i++)
      {
         ulIdx = pForms[i].ulNameID - STRING_TABLE_BASE;

         if (strcmp (pszName, globals.pbStringTable[ulIdx]) == 0)
         {
            ulMatchIdx = i;
            break;
         }
      }
   }

   DBPRINTIF ((pDbg->bFORMIDFROMNAME,
               "%s(): Exit; ulFormID = %d\n",
               __FUNCTION__,
               pForms[ulMatchIdx].ulFormID));

   return pForms[ulMatchIdx].ulFormID;

} /* end of FormIDFromName */

/****************************************************************************/
/* PROCEDURE NAME : FormInfoFromID                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/23/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
FormInfoFromID (PDRIVERINFO pDriver,
                PDEVICEINFO pDevice,
                ULONG       ulFormID,
                PFORMINFO2  pForm)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   ULONG       ulDefType;
   BOOL        bFormInfoFound = FALSE;
   ULONG       i;
   ULONG       ulNumDefined;
   PFORMINFO2  pForms;
#ifdef DEBUG
   PDEBUGINFO  pDbg           = &globals.DebugInfo;
#endif

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  DBPRINTIF ((pDbg->bFORMINFOFROMID,
              "FormInfoFromID(): Enter; ulFormID = %d\n",
              ulFormID));

  assertF (pDevice);

  // Find out how form defined (in driver, in device, by user )
  ulDefType = MAKEULONG (0, HIUSHORT (ulFormID));

  switch (ulDefType)
  {
  case DEF_TYPE_DRIVER:  // No hi word value
  {
     DBPRINTIF ((pDbg->bFORMINFOFROMID,"FormInfoFromID(): DEF_TYPE_DRIVER\n"));

     ulNumDefined = pDevice->ulNumForms;
     pForms       = pDevice->pFORMS;

     // For now only driver table definitions allowed
     // loop through table of forms until matching form ID
     // found
     for (i = 0; i < ulNumDefined; i++)
     {
        if (ulFormID == pForms[i].ulFormID)
        {
           // Copy form information into pointer
           *pForm = pForms[i];

           bFormInfoFound = TRUE;
           break;
        }
     }
     break;
  }

  case DEF_TYPE_DEVICE:
     // @TDB Something goes here, right???
     assertstring ("Not handled!\n");
     break;

  case DEF_TYPE_USER:
  {
     PUSERFORM pUserForm;

     DBPRINTIF ((pDbg->bFORMINFOFROMID, "FormInfoFromID(): DEF_TYPE_USER\n"));

     // assign form list pointer to local variable
     pUserForm = pDevice->pUserDefData->pUserFORMS;

     // for each entry in linked list of user defined forms
     while (pUserForm != NULL)
     {
        if (pUserForm->fiUser.ulFormID == ulFormID)
        {
           // Copy form information into pointer
           *pForm = pUserForm->fiUser;

           bFormInfoFound = TRUE;
           break;
        }

        pUserForm = pUserForm->pNextForm;
     }
     break;
  }

  default:
     assertstring ("Not handled!\n");
     break;
  }

  DBPRINTIF ((pDbg->bFORMINFOFROMID,
              "FormInfoFromID(): Exit; pForm = %x\n",
              pForm));
  assertF (bFormInfoFound);

  if (!bFormInfoFound)
  {
     DBPRINTF (("%s(): failure; ulFormID = %d\n", __FUNCTION__, ulFormID));
  }

  return bFormInfoFound;

} /* end FormInfoFromID */

PFNWP         pfnwpEntryFieldProc;

/* This function is not in the multi-threaded library (but it is in the
** single-threaded version).  So I created a define for it.
*/
#define isdigit(c) ('0' <= c && c <= '9')

MPARAM
NumericSubProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
{
   if (WM_CHAR == msg)
   {
      PCHRMSG p;

      /* if this is a char msg */
      if (SHORT1FROMMP (mp1) &  KC_VIRTUALKEY)
      {
         switch (SHORT2FROMMP (mp2))
         {
         case VK_ESC:
         case VK_TAB:
         case VK_BACKTAB:
         case VK_INSERT:
         case VK_DELETE:
         case VK_BACKSPACE:
         case VK_NEWLINE:
         case VK_ENTER:
         case VK_END:
         case VK_HOME:
         case VK_UP:
         case VK_DOWN:
         case VK_LEFT:
         case VK_RIGHT:
         case VK_F1:
            // Allow these keys through to the entry field
            // call previous entry field proc
            return WinDefDlgProc (hwnd, msg, mp1, mp2);
         }
      }

      p = CHARMSG (&msg);                          /* addres char structure */
      if ((p->fs & (KC_KEYUP|KC_CHAR))==KC_CHAR)   /* ONLY key down transit */
      {
         if (!isdigit(p->chr))                     /* if not numeric ONLY */
         {                                         /* make sure to ignore */
                                                   /* control keys */
            WinAlarm (HWND_DESKTOP,WA_WARNING);    /* beep */
            return (MRESULT)(TRUE);                /* continue */
         }
      }
   }

   return (*pfnwpEntryFieldProc) (hwnd, msg, mp1, mp2);
}


#define TYPE_WORLD        1
#define TYPE_DEVICE       2

LONG
ReturnLength (HPS hps, LONG lDeviceLength, INT iType)
{
   POINTL  ptlConv[3];
   LONG    lXWorld,
           lYWorld;

   ptlConv[0].x = 0;
   ptlConv[0].y = 0;
   ptlConv[1].x = 0;
   ptlConv[1].y = lDeviceLength;
   ptlConv[2].x = lDeviceLength;
   ptlConv[2].y = 0;

   if (TYPE_WORLD == iType)
   {
      if (!GpiConvert (hps, CVTC_DEVICE, CVTC_WORLD, 3L, ptlConv))
        /* if error */
        return 0;
   }
   else
   {
      if (!GpiConvert (hps, CVTC_WORLD, CVTC_DEVICE, 3L, ptlConv))
        /* if error */
        return 0;
   }

   ptlConv[1].x -= ptlConv[0].x;
   ptlConv[1].y -= ptlConv[0].y;

   ptlConv[2].x -= ptlConv[0].x;
   ptlConv[2].y -= ptlConv[0].y;

   // check for rotation
   if (ptlConv[1].x)
      lYWorld = (LONG) (( f_lsqrt(ptlConv[1].x * ptlConv[1].x +
                                  ptlConv[1].y * ptlConv[1].y)  +
                                  32768) >> 16 );
   else
      lYWorld = (LONG)abs (ptlConv[1].y);

   // check for rotation
   if (ptlConv[2].y)
      lXWorld = (LONG) (( f_lsqrt(ptlConv[2].x * ptlConv[2].x +
                                  ptlConv[2].y * ptlConv[2].y)  +
                                  32768) >> 16 );
   else
      lXWorld = (LONG)abs (ptlConv[2].x);

   return max (lXWorld, lYWorld);
}

#define ARROW_SPACING_MAJOR  6
#define ARROW_SPACING_MINOR  3
#define ARROW_LINE_LENGTH    (2*ARROW_SPACING_MAJOR)
#define TYPE_XMAJOR          1
#define TYPE_YMAJOR          2

VOID
DrawArrow (HPS    hps,
           ULONG  xBegin,
           ULONG  yBegin,
           ULONG  xEnd,
           ULONG  yEnd,
           ULONG  ulAmount,
           BOOL   bUseInches,
           USHORT usType)
{
   POINTL       aptlTriangle[4];
   LONG         lArrowSpacingMinor;
   LONG         lArrowSpacingMajor;
   LONG         lArrowLineLength;

   lArrowSpacingMajor = ReturnLength (hps, ARROW_SPACING_MAJOR, TYPE_WORLD);
   lArrowSpacingMinor = ReturnLength (hps, ARROW_SPACING_MINOR, TYPE_WORLD);
   lArrowLineLength   = ReturnLength (hps, ARROW_LINE_LENGTH,   TYPE_WORLD);

   GpiSetColor    (hps, CLR_RED);
   GpiSetPattern  (hps, PATSYM_SOLID);
   GpiSetLineType (hps, LINETYPE_SOLID);

   // Draw the first arrow head
   if (TYPE_XMAJOR == usType)
   {
      aptlTriangle[0].x = xBegin - 1;
      aptlTriangle[0].y = yBegin;
      aptlTriangle[1].x = xBegin - 1 - lArrowSpacingMajor;
      aptlTriangle[1].y = yBegin + lArrowSpacingMinor;
      aptlTriangle[2].x = xBegin - 1 - lArrowSpacingMajor;
      aptlTriangle[2].y = yBegin - lArrowSpacingMinor;
      aptlTriangle[3].x = xBegin - 1 - lArrowLineLength;
      aptlTriangle[3].y = yBegin;
   }
   else if (TYPE_YMAJOR == usType)
   {
      aptlTriangle[0].x = xBegin;
      aptlTriangle[0].y = yBegin - 1;
      aptlTriangle[1].x = xBegin + lArrowSpacingMinor;
      aptlTriangle[1].y = yBegin - 1 - lArrowSpacingMajor;
      aptlTriangle[2].x = xBegin - lArrowSpacingMinor;
      aptlTriangle[2].y = yBegin - 1 - lArrowSpacingMajor;
      aptlTriangle[3].x = xBegin;
      aptlTriangle[3].y = yBegin - 1 - lArrowLineLength;
   }
   else
      // Error
      assertT (TYPE_XMAJOR != usType);

   GpiBeginPath (hps, 1);
   GpiSetCurrentPosition (hps, aptlTriangle);
   GpiPolyLine (hps, 2, &aptlTriangle[1]);
   GpiEndPath (hps);
   GpiFillPath (hps, 1, FPATH_ALTERNATE);

   // Draw arrow line
   GpiSetCurrentPosition (hps, aptlTriangle);
   GpiLine (hps, &aptlTriangle[3]);

   // Draw the second arrow head
   if (TYPE_XMAJOR == usType)
   {
      aptlTriangle[0].x = xEnd + 1;
      aptlTriangle[0].y = yEnd;
      aptlTriangle[1].x = xEnd + 1 + lArrowSpacingMajor;
      aptlTriangle[1].y = yEnd + lArrowSpacingMinor;
      aptlTriangle[2].x = xEnd + 1 + lArrowSpacingMajor;
      aptlTriangle[2].y = yEnd - lArrowSpacingMinor;
      aptlTriangle[3].x = xEnd + 1 + lArrowLineLength;
      aptlTriangle[3].y = yEnd;
   }
   else if (TYPE_YMAJOR == usType)
   {
      aptlTriangle[0].x = xEnd;
      aptlTriangle[0].y = yEnd + 1;
      aptlTriangle[1].x = xEnd + lArrowSpacingMinor;
      aptlTriangle[1].y = yEnd + 1 + lArrowSpacingMajor;
      aptlTriangle[2].x = xEnd - lArrowSpacingMinor;
      aptlTriangle[2].y = yEnd + 1 + lArrowSpacingMajor;
      aptlTriangle[3].x = xEnd;
      aptlTriangle[3].y = yEnd + 1 + lArrowLineLength;
   }
   else
      // Error
      assertT (TYPE_XMAJOR != usType);

   GpiBeginPath (hps, 1);
   GpiSetCurrentPosition (hps, aptlTriangle);
   GpiPolyLine (hps, 2, &aptlTriangle[1]);
   GpiEndPath (hps);
   GpiFillPath (hps, 1, FPATH_ALTERNATE);

   // Draw arrow line
   GpiSetCurrentPosition (hps, aptlTriangle);
   GpiLine (hps, &aptlTriangle[3]);
}

VOID
DrawRuler (HPS     hps,
           PPOINTL pptlStart,
           PPOINTL pptlEnd,
           ULONG   ulNumUnits,
           BOOL    bUseInches,
           USHORT  usType)
{
   POINTL       ptl;
   float        fTick,
                fTickInc,
                fTickStart,
                fTickEnd;
   LINEBUNDLE   LineBundle;
   AREABUNDLE   AreaBundle;
   register INT i;

   if (0 == ulNumUnits)
      ulNumUnits = 10;

   LineBundle.lColor = CLR_BLACK;
   LineBundle.usType = LINETYPE_SOLID;
   GpiSetAttrs (hps, PRIM_LINE, LBB_COLOR | LBB_TYPE, 0L, &LineBundle);
   AreaBundle.lColor   = CLR_YELLOW;
   AreaBundle.usSymbol = PATSYM_SOLID;
   GpiSetAttrs (hps, PRIM_AREA, ABB_COLOR | ABB_SYMBOL, 0L, &AreaBundle);

   // convert device -> world
   GpiConvert (hps, CVTC_DEVICE, CVTC_WORLD, 1, pptlStart);
   GpiConvert (hps, CVTC_DEVICE, CVTC_WORLD, 1, pptlEnd);

   // Draw ruler
   GpiMove (hps, pptlStart);                      // 1
   GpiBox (hps, DRO_OUTLINEFILL, pptlEnd, 0, 0);  // 2
   if (TYPE_YMAJOR == usType)
   {
      /* +-2
      ** | |
      ** | |
      ** 1-+
      */
      fTickStart = pptlEnd->x;
      fTickEnd   = ((float)pptlEnd->x - pptlStart->x)/2.0 + pptlStart->x;
      fTickInc   = ((float)pptlEnd->y - pptlStart->y)/ulNumUnits;
      fTick      = pptlStart->y + fTickInc;
      for (i = 1; i < ulNumUnits; i++)
      {
         ptl.x = fTickStart; ptl.y = fTick;
         GpiMove (hps, &ptl);
         ptl.x = fTickEnd;   ptl.y = fTick;
         GpiLine (hps, &ptl);
         fTick += fTickInc;
      }
   }
   else
   {
      /* +----2
      ** 1----+
      */
      fTickStart = pptlEnd->y;
      fTickEnd   = ((float)pptlEnd->y - pptlStart->y)/2.0 + pptlStart->y;
      fTickInc   = ((float)pptlEnd->x - pptlStart->x)/ulNumUnits;
      fTick      = pptlStart->x + fTickInc;
      for (i = 1; i < ulNumUnits; i++)
      {
         ptl.x = fTick; ptl.y = fTickStart;
         GpiMove (hps, &ptl);
         ptl.x = fTick; ptl.y = fTickEnd;
         GpiLine (hps, &ptl);
         fTick += fTickInc;
      }
   }
}

VOID
DrawDogEar (HPS hps, LONG lRight, LONG lTop, LONG lXSize, LONG lYSize)
{
   POINTL     aptl[8];
   LINEBUNDLE LineBundle;
   AREABUNDLE AreaBundle;

   // Set the two anchor points
   aptl[7].x = lRight;
   aptl[7].y = lTop;
   aptl[0].x = aptl[7].x - lXSize;
   aptl[0].y = aptl[7].y - lYSize;

   aptl[1].x = aptl[0].x;
   aptl[1].y = aptl[7].y;
   aptl[2].x = aptl[7].x;
   aptl[2].y = aptl[0].y;
   aptl[3].x = aptl[0].x;
   aptl[3].y = aptl[0].y;
   aptl[4].x = aptl[7].x;
   aptl[4].y = aptl[7].y;
   aptl[5].x = aptl[7].x;
   aptl[5].y = aptl[0].y;
   aptl[6].x = aptl[0].x;
   aptl[6].y = aptl[7].y;

   // Set line color to white & fill color to white
   LineBundle.lColor = CLR_WHITE;
   LineBundle.usType = LINETYPE_SOLID;
   GpiSetAttrs (hps, PRIM_LINE, LBB_COLOR | LBB_TYPE, 0L, &LineBundle);
   AreaBundle.lColor   = CLR_WHITE;
   AreaBundle.usSymbol = PATSYM_SOLID;
   GpiSetAttrs (hps, PRIM_AREA, ABB_COLOR | ABB_SYMBOL, 0L, &AreaBundle);

   // Draw top half of the "ear"
   GpiBeginArea (hps, BA_BOUNDARY | BA_ALTERNATE);
   GpiMove (hps, &aptl[4]);
   GpiPolyLine (hps, 3, &aptl[5]);
   GpiEndArea (hps);

   // Set line color to black & fill color to white
   LineBundle.lColor = CLR_BLACK;
   LineBundle.usType = LINETYPE_SOLID;
   GpiSetAttrs (hps, PRIM_LINE, LBB_COLOR | LBB_TYPE, 0L, &LineBundle);
   AreaBundle.lColor   = CLR_WHITE;
   AreaBundle.usSymbol = PATSYM_SOLID;
   GpiSetAttrs (hps, PRIM_AREA, ABB_COLOR | ABB_SYMBOL, 0L, &AreaBundle);

   // Draw bottom half of the "ear"
   GpiBeginArea (hps, BA_BOUNDARY | BA_ALTERNATE);
   GpiMove (hps, aptl);
   GpiPolyLine (hps, 3, &aptl[1]);
   GpiEndArea (hps);
}

MRESULT EXPENTRY
FormInfoWndProc (HWND hwnd, ULONG  msg, MPARAM mp1, MPARAM mp2)
{
#define INCH 100                     // LOMETRIC units equivalent to an inch
   static HDC   hdc;
   static HPS   hps   = (HPS)NULL;
   static SIZEL sizel = {100*INCH, 100*INCH};
   PFORMINFO    pFI;

   switch (msg)
   {
   case WM_CREATE:
   {
      RECTL     rectl;

      hdc = WinOpenWindowDC (hwnd);
      assertF (hdc);

      hps = GpiCreatePS (WinQueryAnchorBlock (hwnd),
                         hdc,
                         &sizel,
                         GPIF_SHORT | GPIA_ASSOC | PU_LOENGLISH);
      assertF (hps);

      GpiSetPageViewport (hps, &rectl);
      break;
   }

   case WM_USER_INFO:
      pFI = (PFORMINFO)mp1;

      WinSetWindowULong (hwnd, QWL_USER, (ULONG)pFI);   // Store the handle
      WinQueryWindowRect (hwnd, &pFI->rectlWnd);
      WinInvalidateRect (hwnd, &pFI->rectlWnd, TRUE);     // Force a redraw

      // Setup the page
      pFI->rectlPage.xLeft   = pFI->rectlWnd.xLeft   + 40;
      pFI->rectlPage.yBottom = pFI->rectlWnd.yBottom + 40;
      pFI->rectlPage.xRight  = pFI->rectlWnd.xRight  - 20;
      pFI->rectlPage.yTop    = pFI->rectlWnd.yTop    - 20;
      GpiConvert (hps, CVTC_DEVICE, CVTC_WORLD, 2, (PPOINTL)&pFI->rectlPage);

      if (pFI->cx && pFI->cy)
      {
         ULONG    ulX, ulY, ulXf, ulYf, ulMax;
         float    f;

         ulX  = pFI->cx;
         ulXf = RoundDivide (ulX, 1000);
         ulY  = pFI->cy;
         ulYf = RoundDivide (ulY, 1000);
         ulMax = max (ulXf, ulYf);

#if 0
         PRINT_VAR (ulX);
         PRINT_VAR (ulXf);
         PRINT_VAR (ulY);
         PRINT_VAR (ulYf);
         PRINT_VAR (ulMax);
#endif

         f  = (float)pFI->rectlPage.xRight - (float)pFI->rectlPage.xLeft;
         DBPRINTIF (( 0, "f = %f\n", f));

         f *= ((float)ulX/1000.0)/(float)ulMax;
         DBPRINTIF (( 0, "f = %f\n", f));

         f += (float)pFI->rectlPage.xLeft;
         DBPRINTIF (( 0, "f = %f\n", f));

         pFI->rectlPage.xRight = f;
         f  = (float)pFI->rectlPage.yTop - (float)pFI->rectlPage.yBottom;
         DBPRINTIF (( 0, "f = %f\n", f));

         f *= ((float)ulY/1000.0)/(float)ulMax;
         DBPRINTIF (( 0, "f = %f\n", f));

         f += (float)pFI->rectlPage.yBottom;
         DBPRINTIF (( 0, "f = %f\n", f));

         pFI->rectlPage.yTop = f;
      }

      // Setup the page size
      pFI->sizelPage.cx = pFI->rectlPage.xRight - pFI->rectlPage.xLeft   + 1;
      pFI->sizelPage.cy = pFI->rectlPage.yTop   - pFI->rectlPage.yBottom + 1;

      // Setup the page clip
      pFI->rectlClip.xLeft   = pFI->rectlPage.xLeft;
      pFI->rectlClip.yBottom = pFI->rectlPage.yBottom;
      pFI->rectlClip.xRight  = pFI->rectlPage.xRight;
      pFI->rectlClip.yTop    = pFI->rectlPage.yTop;

      if (pFI->cx && pFI->cy)
      {
         if ( pFI->ulLeft <= pFI->cx )
            pFI->rectlClip.xLeft   += pFI->sizelPage.cx*HmmToHinch (pFI->ulLeft)/HmmToHinch (pFI->cx);
         if ( pFI->ulBottom <= pFI->cy )
            pFI->rectlClip.yBottom += pFI->sizelPage.cy*HmmToHinch (pFI->ulBottom)/HmmToHinch (pFI->cy);
         if ( pFI->ulRight <= pFI->cx )
            pFI->rectlClip.xRight  -= pFI->sizelPage.cx*HmmToHinch (pFI->ulRight)/HmmToHinch (pFI->cx);
         if ( pFI->ulTop <= pFI->cy )
            pFI->rectlClip.yTop    -= pFI->sizelPage.cy*HmmToHinch (pFI->ulTop)/HmmToHinch (pFI->cy);
      }
      break;

   case WM_PAINT:
   {
      RECTL  rclInvalid;

      pFI = (PFORMINFO)WinQueryWindowULong (hwnd, QWL_USER);

      hps = WinBeginPaint ( hwnd, hps, &rclInvalid );

      // fill client with background color
      WinFillRect ( hps, &rclInvalid, CLR_WHITE );

      if (pFI)
      {
         POINTL  ptlStart, ptlEnd;
         LONG    ulX, ulY, ulMax;

         GpiSetColor (hps, CLR_DARKGREEN);

         ulX = pFI->bUseInches ?
                  RoundDivide (HmmToHinch (pFI->cx),  100) :
                  RoundDivide (pFI->cx,              1000) ;
         ulY = pFI->bUseInches ?
                  RoundDivide (HmmToHinch (pFI->cy),  100) :
                  RoundDivide (pFI->cy,              1000) ;
         ulMax = max (ulX, ulY);

         ptlStart.x = 40;
         ptlStart.y = 0;
         ptlEnd.x = pFI->rectlWnd.xRight-20;
         ptlEnd.y = 20;
         DrawRuler (hps,
                    &ptlStart,
                    &ptlEnd,
                    ulMax,
                    pFI->bUseInches,
                    TYPE_XMAJOR);

         ptlStart.x = 0;
         ptlStart.y = 40;
         ptlEnd.x = 20;
         ptlEnd.y = pFI->rectlWnd.yTop-20;
         DrawRuler (hps,
                    &ptlStart,
                    &ptlEnd,
                    ulMax,
                    pFI->bUseInches,
                    TYPE_YMAJOR);

         // @TBD - validate that user entered margins are > than
         // physical limits imposed by hardware if not reject them - MFR
         // Clipped page
         if ((pFI->cx  &&
              pFI->cy   ) &&
             (pFI->ulLeft   ||
              pFI->ulBottom ||
              pFI->ulRight  ||
              pFI->ulTop     ))
         {
            // Make unprintable area look unprintable
            GpiSetColor (hps, CLR_PALEGRAY);
            GpiBeginPath (hps, 1);
            GpiSetCurrentPosition (hps, (PPOINTL)&pFI->rectlPage);
            GpiBox (hps, DRO_OUTLINE, (PPOINTL)&pFI->rectlPage.xRight, 0, 0);
            GpiSetCurrentPosition (hps, (PPOINTL)&pFI->rectlClip);
            GpiBox (hps, DRO_OUTLINE, (PPOINTL)&pFI->rectlClip.xRight, 0, 0);
            GpiEndPath (hps);
            GpiFillPath (hps, 1, FPATH_ALTERNATE);

            GpiSetColor (hps, CLR_RED);
            GpiSetLineType (hps, LINETYPE_DASHDOUBLEDOT);

            GpiSetCurrentPosition (hps, (PPOINTL)&pFI->rectlClip);
            GpiBox (hps, DRO_OUTLINE, (PPOINTL)&pFI->rectlClip.xRight, 0, 0);

            if ( pFI->ulBottom && pFI->ulBottom <= pFI->cy)
               // Draw Arrow - yBottom
               DrawArrow (hps,
                          pFI->rectlPage.xLeft + (pFI->rectlPage.xRight - pFI->rectlPage.xLeft)/2,
                          pFI->rectlPage.yBottom,
                          pFI->rectlPage.xLeft + (pFI->rectlPage.xRight - pFI->rectlPage.xLeft)/2,
                          pFI->rectlClip.yBottom,
                          pFI->ulBottom,
                          pFI->bUseInches,
                          TYPE_YMAJOR);

            if (pFI->ulRight && pFI->ulRight <= pFI->cx)
               // Draw Arrow - xRight
               DrawArrow (hps,
                          pFI->rectlClip.xRight,
                          pFI->rectlPage.yTop - (pFI->rectlPage.yTop - pFI->rectlPage.yBottom)/2,
                          pFI->rectlPage.xRight,
                          pFI->rectlPage.yTop - (pFI->rectlPage.yTop - pFI->rectlPage.yBottom)/2,
                          pFI->ulRight,
                          pFI->bUseInches,
                          TYPE_XMAJOR);

            if (pFI->ulTop && pFI->ulTop <= pFI->cy)
               // Draw Arrow - yTop
               DrawArrow (hps,
                          pFI->rectlPage.xLeft + (pFI->rectlPage.xRight - pFI->rectlPage.xLeft)/2,
                          pFI->rectlClip.yTop,
                          pFI->rectlPage.xLeft + (pFI->rectlPage.xRight - pFI->rectlPage.xLeft)/2,
                          pFI->rectlPage.yTop,
                          pFI->ulTop,
                          pFI->bUseInches,
                          TYPE_YMAJOR);

            if (pFI->ulLeft && pFI->ulLeft <= pFI->cx)
               // Draw Arrow - xLeft
               DrawArrow (hps,
                          pFI->rectlPage.xLeft,
                          pFI->rectlPage.yTop - (pFI->rectlPage.yTop - pFI->rectlPage.yBottom)/2,
                          pFI->rectlClip.xLeft,
                          pFI->rectlPage.yTop - (pFI->rectlPage.yTop - pFI->rectlPage.yBottom)/2,
                          pFI->ulLeft,
                          pFI->bUseInches,
                          TYPE_XMAJOR);
         }

         // Full Page
         if (pFI->cx && pFI->cy)
         {
            GpiSetColor (hps, CLR_BLACK);
            GpiSetLineType (hps, LINETYPE_SOLID);
            GpiSetCurrentPosition (hps, (PPOINTL)&pFI->rectlPage);
            GpiBox (hps, DRO_OUTLINE, (PPOINTL)&pFI->rectlPage.xRight, 0, 0);

            // Draw page "dog ear"
            DrawDogEar (hps,
                        pFI->rectlPage.xRight,
                        pFI->rectlPage.yTop,
                        (pFI->rectlPage.xRight - pFI->rectlPage.xLeft  )/8,
                        (pFI->rectlPage.yTop   - pFI->rectlPage.yBottom)/8);
         }
      }

      WinEndPaint (hps);
      break;
   }

   case WM_DESTROY:
      GpiDestroyPS (hps);
      break;

   default:
      return WinDefWindowProc (hwnd, msg, mp1, mp2);
   }

   return 0;
}

/****************************************************************************/
/* PROCEDURE NAME : FillFormList                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/22/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/*                                                                          */
/****************************************************************************/
VOID
FillFormList (HWND         hwndDlg,
              PDLGINSTANCE pdi,
              PDEVICEINFO  pDevice,
              ULONG        ulListboxID)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   ULONG      ulNumForms;
   USHORT     i;
   ULONG      ulFormID;
   CHAR       szFormName[FORMNAME_SIZE + 1];
   PUSERFORM  pUserForm;
   USHORT     usMatchIdx                  = 0;
   BOOL       bMatchFound                 = FALSE;
   ULONG      ulDefFormID;
#ifdef DEBUG
   PDEBUGINFO pDbg                        = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/

   DBPRINTIF ((pDbg->bFILLFORMLIST, "FillFormList(): Enter\n"));

   // initial pointer validation
   assertF (pdi);
   assertF (pDevice);

   // free existing list
   WinSendDlgItemMsg (hwndDlg,
                      ulListboxID,
                      LM_DELETEALL,
                      MPVOID,
                      MPVOID);

   // Get default form ID from default form connection for hilite
   // @EXJOB
   GetIDsFromConnID (pdi->pDriver,
                     pDevice,
                     pdi->pJobProperties->ulDefConnID,
                     NULL,
                     &ulDefFormID,
                     NULL);

   /************************************/
   /* Driver Defined Forms             */
   /************************************/

   // get # forms defined
   ulNumForms = pDevice->ulNumForms;

   // add each form defined into listbox
   for (i = 0; i < ulNumForms; i++)
   {
      ulFormID = pDevice->pFORMS[i].ulFormID;

      // if Form ID found and string returned
      if (FormNameFromID (pdi->pDriver, pdi->pDevice, szFormName, ulFormID))
      {
         SHORT sIdx;

         sIdx = (SHORT)WinSendDlgItemMsg (hwndDlg,
                                          ulListboxID,
                                          LM_INSERTITEM,
                                          MPFROM2SHORT (LIT_END, 0),
                                          MPFROMP (szFormName));

         // Set Item Handle to be the Form ID
         WinSendDlgItemMsg (hwndDlg,
                            ulListboxID,
                            LM_SETITEMHANDLE,
                            MPFROMSHORT (sIdx),
                            MPFROMLONG (ulFormID));

         DBPRINTIF ((pDbg->bFILLFORMLIST,
                     "FillFormList(): Adding Form: '%s'; ID=%d\n",
                     szFormName,
                     ulFormID));

         // If we are to highlite correct index
         if (!bMatchFound)
         {
            // If we match default form ID
            if (ulFormID == ulDefFormID)
            {
               usMatchIdx  = sIdx;
               bMatchFound = TRUE;
            }
         }
      }
   }

   /************************************/
   /* User Defined Forms               */
   /************************************/

   // get # Forms defined in linked list by user
   ulNumForms = pDevice->pUserDefData->usNumForms;

   // assign base pointer to linked list
   pUserForm = pDevice->pUserDefData->pUserFORMS;

   // add each Form defined into listbox
   for (i = 0; i < ulNumForms; i++)
   {
      // Get local copy of Form ID
      ulFormID = pUserForm->fiUser.ulFormID;

      // if Form ID found and string returned
      if (FormNameFromID (pdi->pDriver, pdi->pDevice, szFormName, ulFormID))
      {
         SHORT sIdx;

         sIdx = (SHORT)WinSendDlgItemMsg (hwndDlg,
                                          ulListboxID,
                                          LM_INSERTITEM,
                                          MPFROM2SHORT (LIT_END, 0),
                                          MPFROMP (szFormName));

         // Set Item Handle to be the Form ID
         WinSendDlgItemMsg (hwndDlg,
                            ulListboxID,
                            LM_SETITEMHANDLE,
                            MPFROMSHORT (sIdx),
                            MPFROMLONG (ulFormID));

         DBPRINTIF ((pDbg->bFILLFORMLIST,
                     "FillFormList(): Adding Form: %s (%d)\n",
                     szFormName, ulFormID));

         // If we are to highlite correct index
         if (!bMatchFound)
         {
            // If we match default form ID
            if (ulFormID == ulDefFormID)
            {
               usMatchIdx = sIdx;
               bMatchFound = TRUE;
            }
         }
      }

      // increment list pointer
      pUserForm = pUserForm->pNextForm;
   }

   // @TBD - highlite index may need to check if no match found
   // what to do as default then
   WinSendDlgItemMsg (hwndDlg,
                      ulListboxID,
                      LM_SELECTITEM,
                      MPFROMLONG (usMatchIdx),
                      MPFROMLONG (TRUE));

   DBPRINTIF ((pDbg->bFILLFORMLIST,
               "FillFormList(): Highliting Form index: (%d)\n",
               usMatchIdx));

   DBPRINTIF ((pDbg->bFILLFORMLIST, "FillFormList(): Exit\n"));

} /* end FillFormList */

/****************************************************************************/
/* PROCEDURE NAME : FillFormClassList                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 6/29/95                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
FillFormClassList (PDLGINSTANCE pdi, HWND hwnd, ULONG ulLBID)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   SHORT  sIdx;
   ULONG  ul, ulID;

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   // initial pointer validation
   assertF (pdi);

   // free existing list
   WinSendDlgItemMsg (hwnd,
                      ulLBID,
                      LM_DELETEALL,
                      MPVOID,
                      MPVOID);

   for (ul = STRING_TABLE_FORM_CLASS_BASE; ul < STRING_TABLE_FORM_CLASS_END; ul++)
   {
      sIdx = (SHORT)WinSendDlgItemMsg (hwnd,
                                       ulLBID,
                                       LM_INSERTITEM,
                                       MPFROM2SHORT (LIT_END, 0),
                                       MPFROMP (globals.pbStringTable[ul-STRING_TABLE_BASE]));

      ulID = ul - (STRING_TABLE_FORM_CLASS_BASE);

      // Set Item Handle
      WinSendDlgItemMsg (hwnd,
                         ulLBID,
                         LM_SETITEMHANDLE,
                         MPFROMSHORT (sIdx),
                         MPFROMLONG (ulID));
   }

} /* end FillFormClassList */


/****************************************************************************/
/* PROCEDURE NAME : FormInfoFromName                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 11/19/93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
FormInfoFromName (PDRIVERINFO pDriver,
                  PDEVICEINFO pDevice,
                  PSZ         pszFormName,
                  PFORMINFO2  pFormMatch)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   BOOL        bSuccess       = FALSE;
   ULONG       ul;
   ULONG       ulNumDefined;
   PFORMINFO2  pForms;
#ifdef DEBUG
   PDEBUGINFO  pDbg           = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bFORMINFOFROMNAME, "FormInfoFromName(): Enter\n"));

   assertF (pDevice);
   assertF (pFormMatch);

   /************************************/
   /* Driver Defined Forms             */
   /************************************/
   ulNumDefined = pDevice->ulNumForms;
   pForms       = pDevice->pFORMS;

   // For now only driver table definitions allowed
   // loop through table of forms until matching form ID
   // found
   for (ul = 0; ul < ulNumDefined; ul++)
   {
      // Now we must determine if the string is in our stringtable
      // or a it is a string the user defined in hcInfo structure
      if (pForms[ul].ulNameID == FORM_STRING_UNLISTED)
      {
         if (0 == strcmp (pszFormName, pForms[ul].hcInfo.szFormname))
         {
            // Copy form information into pointer
            *pFormMatch = pForms[ul];

            bSuccess = TRUE;
            break;
         }
      }
      else
      {
         ULONG ulIdx;

         ulIdx = pForms[ul].ulNameID - STRING_TABLE_BASE;

         assertF (globals.pbStringTable[ulIdx]);

         if (0 == strcmp( pszFormName, globals.pbStringTable[ulIdx]))
         {
            SafeStrNCpy (pForms[ul].hcInfo.szFormname,
                         globals.pbStringTable[ulIdx],
                         FORMNAME_SIZE + 1);

            // Copy form information into pointer
            *pFormMatch = pForms[ul];

            bSuccess = TRUE;
            break;
         }
      }
   }

   /************************************/
   /* User Defined Forms               */
   /************************************/

   // search user defined forms if no match found yet
   if (!bSuccess)
   {
      PUSERFORM pUserForm;

      // assign form list pointer to local variable
      pUserForm = pDevice->pUserDefData->pUserFORMS;

      // for each entry in linked list of user defined forms
      while (pUserForm != NULL)
      {
         if (0 == strcmp (pszFormName, pUserForm->fiUser.hcInfo.szFormname))
         {
            // Copy form information into pointer
            *pFormMatch = pUserForm->fiUser;

            bSuccess = TRUE;
            break;
         }

         pUserForm = pUserForm->pNextForm;
      }
   }

   DBPRINTIF ((pDbg->bFORMINFOFROMNAME, "FormInfoFromName(): Exit\n"));

   return bSuccess;

} /* end FormInfoFromName */

/****************************************************************************/
/* PROCEDURE NAME : GetDefaultFormInfo                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 11/19/93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
GetDefaultFormInfo (PDEVICEBLOCK pdb,
                    ULONG        ulDefFormID,
                    ULONG        ulDefResID,
                    PFORMINFO2   pFormMatch)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   BOOL          bSuccess       = FALSE;
   ULONG         ul;
   ULONG         ulNumDefined;
   PFORMINFO2    pForms;
   PDRIVERINFO   pDriver        = pdb->pDriver;
   PDEVICEINFO   pDevice        = pdb->pDevice;
#ifdef DEBUG
   PDEBUGINFO    pDbg           = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bGETDEFAULTFORMINFO, "GetDefaultFormInfo(): Enter\n"));

   assertF (pDevice);
   assertF (pFormMatch);

   /************************************/
   /* Device Defined Forms             */
   /************************************/
   ulNumDefined = pDevice->ulNumForms;
   pForms       = pDevice->pFORMS;

   // For now only driver table definitions allowed
   // loop through table of forms until matching form ID
   // found
   for (ul = 0; ul < ulNumDefined; ul++)
   {
      // Now we must determine if the string is in our stringtable
      // or a it is a string the user defined in hcInfo structure
      if (pForms[ul].ulFormID == ulDefFormID)
      {
         // Copy form information into pointer
         *pFormMatch = pForms[ul];

         DBPRINTIF ((pDbg->bGETDEFAULTFORMINFO, "GetDefaultFormInfo: Match! %s\n",
                     pFormMatch->hcInfo.szFormname));

         bSuccess = TRUE;
         break;
      }
   }

   /************************************/
   /* User Defined Forms               */
   /************************************/

   // search user defined forms if no match found yet
   if (!bSuccess)
   {
      PUSERFORM pUserForm;

      // assign form list pointer to local variable
      pUserForm = pDevice->pUserDefData->pUserFORMS;

      // for each entry in linked list of user defined forms
      while (pUserForm != NULL)
      {
         if (pUserForm->fiUser.ulFormID == ulDefFormID)
         {
            // Copy form information into pointer
            *pFormMatch = pUserForm->fiUser;

            bSuccess = TRUE;
            break;
         }

         pUserForm = pUserForm->pNextForm;
      }
   }

   // if we found a default form
   if (bSuccess)
   {
      CalculateFormSize (pdb, pdb->pJobProperties, pFormMatch);

      // set HCAPS_CURRENT since this is default form
      pFormMatch->hcInfo.flAttributes |= HCAPS_CURRENT;
   }
   else
   {
      // @TBD - should have already taken care of this when validating
      // DRIVDATA, so this should really not be a problemo
      *pFormMatch = pForms[0];

      CalculateFormSize (pdb, pdb->pJobProperties, pFormMatch);

      // set HCAPS_CURRENT since this is default form
      pFormMatch->hcInfo.flAttributes |= HCAPS_CURRENT;
   }

   FormNameFromID (pDriver,
                   pDevice,
                   pFormMatch->hcInfo.szFormname,
                   ulDefFormID);

   DBPRINTIF ((pDbg->bGETDEFAULTFORMINFO, "GetDefaultFormInfo(): Exit\n"));

   return bSuccess;

} /* end GetDefaultFormInfo */

/****************************************************************************/
/* PROCEDURE NAME : CalculateFormSize                                       */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 02/11/93                                                */
/* DESCRIPTION    : Used to calculate form dimensions and margins and       */
/*                  fill them into the OS/2 HCINFO structure.   This        */
/*                  structure returns HardCopy form information to user     */
/*                  when DevQueryHardcopyCaps() is called.                  */
/*                                                                          */
/*                  These values also help us to display form/margin        */
/*                  info in our printer properties.                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* 03/08/96 - Matt Rutkowski -  defect @147476 tells us we were reporting   */
/*            the incorrect margins when job properties had orientation     */
/*            set to Landscape.                                             */
/*                                                                          */
/****************************************************************************/
VOID
CalculateFormSize (PDEVICEBLOCK   pdb,
                   PJOBPROPERTIES pJobProperties,
                   PFORMINFO2     pForm)
{
   PRESINFO pResInfo;
   ULONG    ulXRes,
            ulYRes;
   LONG     cx = 0,
            cy = 0;
   PFNSUBCLASS ppfnDevice = pdb->pDevice->pSubclassedFunctions;

   // Check to see if we have calculated this form already...
   if (pForm->bHasBeenSetup)
      return;

   // Some device have to adjust its hardcopy caps if orientation
   // is changed. We give it a chance to change its value before
   // returning information back to user.
   if (ppfnDevice->pfnDeviceQuery)
   {
      ppfnDevice->pfnDeviceQuery ((PVOID)pdb,
                                  DEVICE_QUERY_HCINFO,
                                  (PVOID)pForm);
   }

   // Calculate printable region (x,y directions)
   cx = pForm->hcInfo.xRightClip - pForm->hcInfo.xLeftClip;
   cy = pForm->hcInfo.yTopClip   - pForm->hcInfo.yBottomClip;

   // Get current dpi from Resolution passed in
   pResInfo = GetpResFromResID (pdb->pDriver, pdb->pDevice, pJobProperties);
   assertF (pResInfo);

   ulXRes = pResInfo->ulXRes;
   ulYRes = pResInfo->ulYRes;

   
   if (pdb->ulNupPages == 2)
   {
       LONG       lDeltaY    = 0;
       PFORMINFO2 pLargeForm = pdb->pLargeFormInfo;

       //
       // Set equival margins and calculate
       //
       lDeltaY = pForm->hcInfo.yBottomClip +
                 pForm->hcInfo.yTopClip -
                 pForm->hcInfo.cy;

       //
       // Set equivalent margins
       //
       if (lDeltaY > 0)
       {
          pdb->ulTopMove  = lDeltaY;
          pForm->hcInfo.yTopClip  -= lDeltaY;
          ConvertHMMtoPels (pdb->ulTopMove,
                            ulYRes,
                            (PULONG)&(pdb->ulTopMove),
                            FALSE);
       }
       else
       {
          pForm->hcInfo.yBottomClip  -= lDeltaY;
       }

       // Calculate new printable region (x,y directions)
       cx = pForm->hcInfo.xRightClip - pForm->hcInfo.xLeftClip;
       cy = pForm->hcInfo.yTopClip   - pForm->hcInfo.yBottomClip;

       memcpy (pLargeForm, pForm, sizeof (*pLargeForm));

       //
       // Calculate small page
       //
       pForm->hcInfo.cx = cy / 2;
       pForm->hcInfo.cy = cx;
       pForm->hcInfo.xRightClip = pForm->hcInfo.cx;
       pForm->hcInfo.yTopClip   = pForm->hcInfo.cy;
       pForm->hcInfo.xLeftClip  = pForm->hcInfo.yBottomClip = 0;

       ConvertHMMtoPels (cx,
                         ulXRes,
                         (PULONG)&(pLargeForm->hcInfo.xPels),
                         FALSE);
       ConvertHMMtoPels (cy,
                         ulYRes,
                         (PULONG)&(pLargeForm->hcInfo.yPels),
                         FALSE);

       if (pLargeForm->hcInfo.yPels % 2)
          pLargeForm->hcInfo.yPels--;

       pForm->hcInfo.xPels      = pLargeForm->hcInfo.yPels / 2;
       pForm->hcInfo.yPels      = pLargeForm->hcInfo.xPels;

       pLargeForm->hcInfo.cx          = RoundDivide (pLargeForm->hcInfo.cx,          100);
       pLargeForm->hcInfo.cy          = RoundDivide (pLargeForm->hcInfo.cy,          100);
       pLargeForm->hcInfo.xLeftClip   = RoundDivide (pLargeForm->hcInfo.xLeftClip,   100);
       pLargeForm->hcInfo.yBottomClip = RoundDivide (pLargeForm->hcInfo.yBottomClip, 100);
       pLargeForm->hcInfo.xRightClip  = RoundDivide (pLargeForm->hcInfo.xRightClip,  100);
       pLargeForm->hcInfo.yTopClip    = RoundDivide (pLargeForm->hcInfo.yTopClip,    100);
       pLargeForm->bHasBeenSetup      = TRUE;

    } 
    else // Not booklet
    {
        // Set height/width dynamically based on device resolution
        ConvertHMMtoPels (cx,
                          ulXRes,
                          (PULONG)&(pForm->hcInfo.xPels),
                          FALSE);
        ConvertHMMtoPels (cy,
                          ulYRes,
                          (PULONG)&(pForm->hcInfo.yPels),
                          FALSE);
   }

   if (ORIENTATION_LANDSCAPE == pdb->pJobProperties->ulOrientation)
   {
     // Swap imageable areas
     SWAP (pForm->hcInfo.cx,         pForm->hcInfo.cy);
     SWAP (pForm->hcInfo.xPels,      pForm->hcInfo.yPels);

     // Swap margins
     // @147476 - Fixed here - MFR
     // @153333 - Rotated margins clockwise so envelopes work as well
     // this is especially for AmiPro which does absolute pos. on paper
     // to achieve correct margins
     SWAP (pForm->hcInfo.xLeftClip,   pForm->hcInfo.yTopClip);
     SWAP (pForm->hcInfo.xLeftClip,   pForm->hcInfo.yBottomClip);
     SWAP (pForm->hcInfo.yBottomClip, pForm->hcInfo.xRightClip);

     // now adjust values since HCINFO reports distance from Bot-Left corner
     pForm->hcInfo.yTopClip    = pForm->hcInfo.cy - pForm->hcInfo.yTopClip;
     pForm->hcInfo.yBottomClip = pForm->hcInfo.cy - pForm->hcInfo.yBottomClip;
   }

   // Account for data being stored in hundredths of a millimeter now.
   pForm->hcInfo.cx          = RoundDivide (pForm->hcInfo.cx,          100);
   pForm->hcInfo.cy          = RoundDivide (pForm->hcInfo.cy,          100);
   pForm->hcInfo.xLeftClip   = RoundDivide (pForm->hcInfo.xLeftClip,   100);
   pForm->hcInfo.yBottomClip = RoundDivide (pForm->hcInfo.yBottomClip, 100);
   pForm->hcInfo.xRightClip  = RoundDivide (pForm->hcInfo.xRightClip,  100);
   pForm->hcInfo.yTopClip    = RoundDivide (pForm->hcInfo.yTopClip,    100);
   pForm->bHasBeenSetup      = TRUE;

} /* end CalculateFormSize */


/****************************************************************************/
/* PROCEDURE NAME : IsFormNameUnique                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/31/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
IsFormNameUnique (PDLGINSTANCE pdi, PSZ pszNewName, PSZ pszIgnore)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   PDEVICEINFO pDevice;
   ULONG       ulNumForms;
   USHORT      i;
   ULONG       ulFormID;
   CHAR        szFormName[FORMNAME_SIZE + 1];
   PUSERFORM   pUserForm;
   BOOL        bIsUnique                    = TRUE;
#ifdef DEBUG
   PDEBUGINFO  pDbg                         = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bISFORMNAMEUNIQUE, "%s(): Enter\n", __FUNCTION__));

   // initial pointer validation
   assertF (pdi);

   // easier pointer dereference
   pDevice = pdi->pDevice;
   assertF (pDevice);

   /************************************/
   /* Driver Defined Forms             */
   /************************************/

   // get # forms defined
   ulNumForms = pDevice->ulNumForms;

   // check name passed in against each form defined into listbox
   for (i = 0; i < ulNumForms; i++)
   {
      ulFormID = pDevice->pFORMS[i].ulFormID;

      // if Form ID found and string returned
      if (FormNameFromID (pdi->pDriver, pDevice, szFormName, ulFormID))
      {
         if (0 == strncmp (szFormName, pszNewName, FORMNAME_SIZE + 1))
         {
            if (0 != strncmp (szFormName, pszIgnore, FORMNAME_SIZE + 1))
            {
               bIsUnique = FALSE;
               break;
            }
         }
      }
   }

   /************************************/
   /* User Defined Forms               */
   /************************************/

   // Still need to keep checking new form name for uniqueness
   if (bIsUnique)
   {
      // get # Forms defined in linked list by user
      ulNumForms = pDevice->pUserDefData->usNumForms;

      // assign base pointer to linked list
      pUserForm = pDevice->pUserDefData->pUserFORMS;

      // add each Form defined into listbox
      for (i = 0; i < ulNumForms; i++)
      {
         // Get local copy of Form ID
         ulFormID = pUserForm->fiUser.ulFormID;

         // if Form ID found and string returned
         if (FormNameFromID (pdi->pDriver, pDevice, szFormName, ulFormID))
         {
            if (0 == strncmp (szFormName, pszNewName, FORMNAME_SIZE + 1))
            {
               if (0 != strncmp (szFormName, pszIgnore, FORMNAME_SIZE + 1))
               {
                  bIsUnique = FALSE;
                  break;
               }
            }
         }

         // increment list pointer
         pUserForm = pUserForm->pNextForm;
      }
   }

   DBPRINTIF ((pDbg->bISFORMNAMEUNIQUE,
               "%s(): Exit; bIsUnique = %d\n",
               __FUNCTION__,
               bIsUnique));

   return bIsUnique;

} /* end IsFormNameUnique */

/****************************************************************************/
/* PROCEDURE NAME : ValidateFormSizeAndMargins                              */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 1/16/95                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ValidateFormSizeAndMargins (PDLGINSTANCE pdi, PFORMINFO pFI)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   BOOL        bValidForm = FALSE;

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/

   // initial pointer validation
   assertF (pdi);

   // check for overlapping clipping (i.e. no imageable area)
   if (((pFI->ulLeft+pFI->ulRight)< pFI->cx) &&
       ((pFI->ulTop+pFI->ulBottom)< pFI->cy)  )
   {
      bValidForm = TRUE;
   }
   else
   {
      WinMessageBox ((HWND)HWND_DESKTOP,
                     (HWND)NULL,
                     "Margins overlap leaving no place on page to print to",
                     "Margin Error",
                     1L,
                     MB_OK | MB_INFORMATION);
   }

   //  pFI->Form.hcInfo.cx          = pFI->cx;
   //  pFI->Form.hcInfo.cy          = pFI->cy;
   //  pFI->Form.hcInfo.xLeftClip   = pFI->ulLeft;
   //  pFI->Form.hcInfo.yBottomClip = pFI->ulBottom;
   //  pFI->Form.hcInfo.xRightClip  = pFI->cx - pFI->ulRight;
   //  pFI->Form.hcInfo.yTopClip    = pFI->cy - pFI->ulTop;

   // Validate Margins are equal to OR greater than printer's physical
   // margins (i.e. will not allow margins to shrink below what the
   // printer is physically able to print to)


   return bValidForm;

} /* end ValidateFormSizeAndMargins */

/****************************************************************************/
/* PROCEDURE NAME : AddDefinedForm                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/31/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
AddDefinedForm (HWND hwnd, PDLGINSTANCE pdi, PFORMINFO pFI, PSZ pszOldName)
{
   CHAR        szTemp[LEN_STRING];
   SHORT       sIdx                = -1;
   BOOL        bFormAdded          = FALSE;
   BOOL        bValidMargins;
#ifdef DEBUG
   PDEBUGINFO  pDbg                = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bADDDEFINEDFORM, "%s(): Enter\n", __FUNCTION__));

   WinQueryWindowText (WinWindowFromID (hwnd, IDE_DEF_FORM_NAME),
                       LEN_STRING-2,
                       szTemp);

   // Validate form name is unique before saving
   if (IsFormNameUnique (pdi, szTemp, pszOldName))
   {
      strcpy (pFI->Form.hcInfo.szFormname, szTemp);

      pFI->Form.usLenCmdSelectForm = ConvertAsciiCmdToHex (WinWindowFromID (hwnd,
                                                                            IDE_DEF_FORM_CMD),
                                                           pFI->Form.szCmdSelectForm,
                                                           LEN_CMD);

      // Validate that the new form size and margins are realizable
      bValidMargins = ValidateFormSizeAndMargins (pdi, pFI);

      // If new margins are within hardware limits
      if (bValidMargins)
      {
         // adopt new form information and create a new form
         pFI->Form.hcInfo.cx          = pFI->cx;
         pFI->Form.hcInfo.cy          = pFI->cy;
         pFI->Form.hcInfo.xLeftClip   = pFI->ulLeft;
         pFI->Form.hcInfo.yBottomClip = pFI->ulBottom;
         pFI->Form.hcInfo.xRightClip  = pFI->cx - pFI->ulRight;
         pFI->Form.hcInfo.yTopClip    = pFI->cy - pFI->ulTop;

         // Retrieve form class, get currently selected listbox entry's index
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDD_FORM_CLASS,
                                                 LM_QUERYSELECTION,
                                                 0L,
                                                 0L));

         // verify user selected listbox some value in the listbox
         if (sIdx != LIT_NONE)
         {
            ULONG ulClass;

            // Get Item Handle (currently selected ID for Tray, Form etc. )
            ulClass = (ULONG)WinSendDlgItemMsg (hwnd,
                                                IDD_FORM_CLASS,
                                                LM_QUERYITEMHANDLE,
                                                MPFROMSHORT (sIdx),
                                                MPVOID);
            pFI->Form.ulFormClass = ulClass;
         }

         // Create a unique User Defined Form ID and place in definition
         pFI->Form.ulFormID = CreateUniqueFormID (pdi->pDevice);

         // @TBD validate form dimensions - MFR

         // Add new form definition to linked list
         AddUserDefinedFormEntry (pdi->pDevice, &pFI->Form);

         // @TBD - reset pFI->Form to initial values once a form is added

         // update listbox on form page with new entry
         sIdx = (SHORT)WinSendDlgItemMsg (hwnd,
                                          IDL_DEF_FORM_LIST,
                                          LM_INSERTITEM,
                                          MPFROM2SHORT (LIT_END, 0),
                                          MPFROMP (pFI->Form.hcInfo.szFormname));

         DBPRINTIF ((pDbg->bADDDEFINEDFORM,
                     "%s(): LM_INSERTITEM '%s' at Index %d\n",
                     __FUNCTION__,
                     pFI->Form.hcInfo.szFormname,
                     sIdx));

         // Set Item Handle to be the Form ID
         WinSendDlgItemMsg (hwnd,
                            IDL_DEF_FORM_LIST,
                            LM_SETITEMHANDLE,
                            MPFROMSHORT (sIdx),
                            MPFROMLONG (pFI->Form.ulFormID));

         WinSendDlgItemMsg (hwnd,
                            IDL_DEF_FORM_LIST,
                            LM_SELECTITEM,
                            MPFROMLONG (sIdx),
                            MPFROMLONG (TRUE));

         // Indicate form was added for caller return code
         bFormAdded = TRUE;

         // update listbox on connection page with new entry
         sIdx = (SHORT)WinSendDlgItemMsg (pdi->hwndConnectDlg,
                                          IDL_FORMS,
                                          LM_INSERTITEM,
                                          MPFROM2SHORT (LIT_END, 0),
                                          MPFROMP (pFI->Form.hcInfo.szFormname));

         // Set Item Handle to be the Form ID
         WinSendDlgItemMsg (pdi->hwndConnectDlg,
                            IDL_FORMS,
                            LM_SETITEMHANDLE,
                            MPFROMSHORT (sIdx),
                            MPFROMLONG (pFI->Form.ulFormID));

         WinSendDlgItemMsg (pdi->hwndConnectDlg,
                            IDL_FORMS,
                            LM_SELECTITEM,
                            MPFROMLONG (sIdx),
                            MPFROMLONG (TRUE));
      }
   }
   else
   {
      // unable to add new form since name is not unique
      WinMessageBox ((HWND)HWND_DESKTOP,
                     (HWND)NULL,
                     globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_ADD_MSG_FORM],
                     globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_ADD_FORM],
                     1L,
                     MB_OK | MB_INFORMATION );
   }

   DBPRINTIF ((pDbg->bADDDEFINEDFORM,
               "%s(): Exit; sIdx = %d\n",
               __FUNCTION__,
               sIdx));

   return bFormAdded;

} /* end AddDefinedForm */


/****************************************************************************/
/* PROCEDURE NAME : DeleteDefinedForm                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/31/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
DeleteDefinedForm (HWND hwnd, PDLGINSTANCE pdi, SHORT sIdx)
{
   ULONG ulFormID;
   BOOL  bRC;

   // Get Item Handle (FORM ID)
   ulFormID = (ULONG)WinSendDlgItemMsg (hwnd,
                                        IDL_DEF_FORM_LIST,
                                        LM_QUERYITEMHANDLE,
                                        MPFROMSHORT (sIdx),
                                        MPVOID);

   // Find form entry in linked list or in driver table and delete
   bRC = DeleteUserDefinedFormEntry (pdi->pDevice, ulFormID);

   // if delete was a success remove form name from listbox
   if (bRC)
   {
      // remove listbox entry
      WinSendDlgItemMsg (hwnd,
                         IDL_DEF_FORM_LIST,
                         LM_DELETEITEM,
                         MPFROMSHORT (sIdx),
                         MPVOID);

      WinSendDlgItemMsg (hwnd,
                         IDL_DEF_FORM_LIST,
                         LM_SELECTITEM,
                         MPFROMLONG (0),
                         MPFROMLONG (TRUE));

      // remove listbox entry
      WinSendDlgItemMsg (pdi->hwndConnectDlg,
                         IDL_FORMS,
                         LM_DELETEITEM,
                         MPFROMSHORT (sIdx),
                         MPVOID);

      WinSendDlgItemMsg (pdi->hwndConnectDlg,
                         IDL_FORMS,
                         LM_SELECTITEM,
                         MPFROMLONG (0),
                         MPFROMLONG (TRUE));
   }

  return bRC;

} /* end DeleteDefinedForm */

/****************************************************************************/
/* PROCEDURE NAME : UpdateDefinedFormDlg                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/31/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/*                                                                          */
/****************************************************************************/
VOID
UpdateDefinedFormDlg (HWND hwnd, PDLGINSTANCE pdi, PFORMINFO pFI)
{
   ULONG  ulIdx;
   CHAR   szHexCmd[32];

   DBPRINTF (("%s(): Enter\n", __FUNCTION__));

#ifdef DEBUG
   // @EXJOB
   DebugOutputJobProperties (FALSE, "UpdateDefinedFormDlg():", pdi->pJobProperties);
   DebugOutputFormInfo (FALSE, "UpdateDefinedFormDlg(): New Form:", pdi->pFI);
#endif

   DBPRINTF (("UpdateDefinedFormDlg(): bUseInches = %d\n", pdi->bUseInches));

   pFI->bUseInches = pdi->bUseInches; // @100713

   // Notify the form drawing window of the current form metrics
   WinSendMsg (WinWindowFromID (hwnd, IDD_CLIPPING_WINDOW),
               WM_USER_INFO,
               MPFROMP (pFI),
               (MPARAM)0);

   // We do not want to overflow the form name size
   WinSendMsg (WinWindowFromID (hwnd, IDE_DEF_FORM_NAME),
               EM_SETTEXTLIMIT,
               (MPARAM)(sizeof (pFI->Form.hcInfo.szFormname) - 1),
               MPVOID);

   // Fill in the entry fields...
   WinSetWindowText (WinWindowFromID (hwnd, IDE_DEF_FORM_NAME),
                     pFI->Form.hcInfo.szFormname);

   ConvertHexCmdToAscii (pFI->Form.usLenCmdSelectForm,
                         pFI->Form.szCmdSelectForm,
                         (PSZ)szHexCmd);

   WinSetWindowText (WinWindowFromID(hwnd, IDE_DEF_FORM_CMD), szHexCmd);

   DBPRINTF (("UpdateDefinedFormDlg: ulFormID = %lu\n", pFI->Form.ulFormID));

   WinSetDlgItemShort (hwnd, IDD_DEF_FORM_ID,
                       LOUSHORT (pFI->Form.ulFormID),
                       FALSE);

   // Display current form class in combobox
   SelectItemFromHandle (pdi, hwnd, IDD_FORM_CLASS, pFI->Form.ulFormClass);

   // Indicate if the form is "built-in" or "user-defined"
   if (HIUSHORT (pFI->Form.ulFormID))
      ulIdx = DIALOG_STRING_FORM_USERID - STRING_TABLE_BASE;
   else
      ulIdx = DIALOG_STRING_FORM_ID - STRING_TABLE_BASE;

   // Display "Built-in" or "user-defined" on dialog
   WinSetWindowText (WinWindowFromID (hwnd, IDD_DEF_FORM_TYPE),
                     globals.pbStringTable[ulIdx]);

   // @TBD - validate that user entered margins are > than
   // physical limits imposed by hardware if not reject them - MFR
   pFI->bEntryChanging = TRUE;
   SetEntryText (hwnd,
                 IDD_DEF_FORM_WIDTH,
                 pdi->ctryInfo.szDecimal[0],
                 pFI->cx,
                 pdi->bUseInches);

   SetEntryText (hwnd,
                 IDD_DEF_FORM_HEIGHT,
                 pdi->ctryInfo.szDecimal[0],
                 pFI->cy,
                 pdi->bUseInches);

   SetEntryText (hwnd,
                 IDD_DEF_FORM_LEFT_CLIP,
                 pdi->ctryInfo.szDecimal[0],
                 pFI->ulLeft,
                 pdi->bUseInches);

   SetEntryText (hwnd,
                 IDD_DEF_FORM_BOT_CLIP,
                 pdi->ctryInfo.szDecimal[0],
                 pFI->ulBottom,
                 pdi->bUseInches);

   SetEntryText (hwnd,
                 IDD_DEF_FORM_RIGHT_CLIP,
                 pdi->ctryInfo.szDecimal[0],
                 pFI->ulRight,
                 pdi->bUseInches);

   SetEntryText (hwnd,
                 IDD_DEF_FORM_TOP_CLIP,
                 pdi->ctryInfo.szDecimal[0],
                 pFI->ulTop,
                 pdi->bUseInches);

   pFI->bEntryChanging = FALSE;

   DBPRINTF (("UpdateDefinedFormDlg(): Exit\n"));

} /* end UpdateDefinedFormDlg */

/****************************************************************************/
/* PROCEDURE NAME : DefineFormDlgProc                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/22/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY
DefineFormDlgProc (HWND hwnd, ULONG  msg, MPARAM mp1, MPARAM mp2)
{
   PFORMINFO      pFI;
   PDLGINSTANCE   pdi;
   SHORT          sIdx;
   ULONG          ulFormID;
   BOOL           bRC;

   switch (msg)
   {
   case WM_CONTROLPOINTER:
   {
      ULONG ulCtlID = LONGFROMMP (mp1);

      pdi = (PDLGINSTANCE)WinQueryWindowULong (hwnd, QWL_USER);

      if (ulCtlID != (ULONG)DID_STATIC)
      {
        if (ulCtlID == DID_ADD || ulCtlID == DID_DELETE ||
            ulCtlID == DID_MODIFY)
        {
           ulCtlID |= SIZE_INDICATOR;
        }

        QueryContextHelp (pdi->pHelpPrnProp,
                          pdi->hwndPrnPropTxtHelp,
                          ulCtlID);
      }
      break;
   }

   case WM_INITDLG:
   {
      ULONG ulIdx;

      pdi = (PDLGINSTANCE)mp2;
      pFI = pdi->pFI;

      // store the pointer to instance data
      WinSetWindowULong (hwnd, QWL_USER, (ULONG)pdi);

      // We are going to fill in a lot of strings from a similar location
      // in stringtable so lets get a base index to reference from
      ulIdx = STRING_TABLE_DLG_BASE - STRING_TABLE_BASE;

      // Fill in text fields in dialogs from stringtable
      WinSetDlgItemText (hwnd, DID_ADD,
                         globals.pbStringTable[ulIdx + DIALOG_TEXT_ADD]);
      WinSetDlgItemText (hwnd, DID_DELETE,
                         globals.pbStringTable[ulIdx + DIALOG_TEXT_DELETE]);
      WinSetDlgItemText (hwnd, DID_MODIFY,
                         globals.pbStringTable[ulIdx + DIALOG_TEXT_MODIFY]);

      // @100713 - end A4/CM default
      WinSendDlgItemMsg (hwnd,
                         IDD_DEF_FORM_INCHES,
                         BM_SETCHECK,
                         MPFROMSHORT (pdi->bUseInches),
                         NULL);

      WinSendDlgItemMsg (hwnd,
                         IDD_DEF_FORM_CENTIMETERS,
                         BM_SETCHECK,
                         MPFROMSHORT (!pdi->bUseInches),
                         NULL);

      pFI->bUseInches = pdi->bUseInches;

      FillFormList (hwnd, pdi, pdi->pDevice, IDL_DEF_FORM_LIST);

      // Fill in form class list (i.e. Sheet, Envelope, Postcard etc.)
      FillFormClassList (pdi, hwnd, IDD_FORM_CLASS);

      // Update all form attributes on the page (to reflect current form)
      UpdateDefinedFormDlg (hwnd, pdi, pFI);

#ifdef DEBUG
      WinEnableControl (hwnd, IDD_DEF_FORM_ID, TRUE);
      WinShowWindow (WinWindowFromID (hwnd, IDD_DEF_FORM_ID), TRUE);
#endif

      // Set up easy index into stringtable's context help msgs
      ulIdx = STRING_TABLE_MSG_BASE - STRING_TABLE_BASE;

      // add context help for "Sizes" Printer Properties notebook page
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEFINE_FORM, globals.pbStringTable[ulIdx + MSG_FORM_PAGE]);
      AddContextHelp (pdi->pHelpPrnProp, IDL_DEF_FORM_LIST, globals.pbStringTable[ulIdx + MSG_FORM_LIST]);
      AddContextHelp (pdi->pHelpPrnProp, IDE_DEF_FORM_NAME, globals.pbStringTable[ulIdx + MSG_FORM_NAME]);
      AddContextHelp (pdi->pHelpPrnProp, IDE_DEF_FORM_CMD, globals.pbStringTable[ulIdx + MSG_FORM_CMD]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_WIDTH, globals.pbStringTable[ulIdx + MSG_FORM_WIDTH]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_HEIGHT, globals.pbStringTable[ulIdx + MSG_FORM_HEIGHT]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_LEFT_CLIP, globals.pbStringTable[ulIdx + MSG_FORM_LEFT_CLIP]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_RIGHT_CLIP, globals.pbStringTable[ulIdx + MSG_FORM_RIGHT_CLIP]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_TOP_CLIP, globals.pbStringTable[ulIdx + MSG_FORM_TOP_CLIP]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_BOT_CLIP, globals.pbStringTable[ulIdx + MSG_FORM_BOT_CLIP]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_INCHES, globals.pbStringTable[ulIdx + MSG_FORM_INCHES]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_CENTIMETERS, globals.pbStringTable[ulIdx + MSG_FORM_CENTIMETERS]);
      AddContextHelp (pdi->pHelpPrnProp, IDT_DEF_FORM_TYPE, globals.pbStringTable[ulIdx + MSG_FORM_TYPE_TITLE]);
      AddContextHelp (pdi->pHelpPrnProp, IDD_DEF_FORM_TYPE, globals.pbStringTable[ulIdx + MSG_FORM_TYPE_VALUE]);

      // Add "Add", "Delete", and "Modify" messages for "Paper Size" page
      AddContextHelp (pdi->pHelpPrnProp, DID_ADD | SIZE_INDICATOR, globals.pbStringTable[ulIdx + MSG_ADD_SIZE_BUTTON]);
      AddContextHelp (pdi->pHelpPrnProp, DID_DELETE | SIZE_INDICATOR, globals.pbStringTable[ulIdx + MSG_DEL_SIZE_BUTTON]);
      AddContextHelp (pdi->pHelpPrnProp, DID_MODIFY | SIZE_INDICATOR, globals.pbStringTable[ulIdx + MSG_MOD_SIZE_BUTTON]);

      // return 0 since focus was not changed
      return 0;
   }

   case WM_CONTROL:
   {
      // get instance data
      pdi = (PDLGINSTANCE)WinQueryWindowULong (hwnd, QWL_USER);
      pFI = pdi->pFI;

      // new form selected from list those defined for this device
      if (SHORT1FROMMP(mp1) == IDL_DEF_FORM_LIST)
      {
         // Get index of facename selected by user
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_FORM_LIST,
                                                 LM_QUERYSELECTION,
                                                 0L,0L));

         // as long as we verify an entry in the list was indeed highlited
         if (sIdx != LIT_NONE)
         {
            // Get Item Handle (FORM ID)
            ulFormID = (ULONG)WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_FORM_LIST,
                                                 LM_QUERYITEMHANDLE,
                                                 MPFROMSHORT (sIdx),
                                                 MPVOID);

            // If we really have a new form with new information to copy in
            if (ulFormID != pFI->Form.ulFormID)
            {
               DBPRINTF(( "%s(): IDL_DEF_FORM_LIST: sIdx = %d; ulFormID = %lu\n", __FUNCTION__, sIdx, ulFormID ));

               pFI->ulSize   = sizeof (FORMINFO);
               pFI->bNoEdit  = TRUE;

               // If we are able to get form info for currently selected ID
               if (FormInfoFromID (pdi->pDriver, pdi->pDevice, ulFormID, &pFI->Form))
               {
                  // copy all information about this form into temp. structure
                  FormNameFromID (pdi->pDriver,
                                  pdi->pDevice,
                                  pFI->Form.hcInfo.szFormname,
                                  pFI->Form.ulFormID);

                  pFI->cx       = pFI->Form.hcInfo.cx;
                  pFI->cy       = pFI->Form.hcInfo.cy;
                  pFI->ulLeft   = pFI->Form.hcInfo.xLeftClip;
                  pFI->ulBottom = pFI->Form.hcInfo.yBottomClip;
                  pFI->ulRight  = pFI->cx - pFI->Form.hcInfo.xRightClip;
                  pFI->ulTop    = pFI->cy - pFI->Form.hcInfo.yTopClip;

                  // Update the clipping window example with new values
                  UpdateDefinedFormDlg (hwnd, pdi, pFI);
               }
            }
         }
      }

      if (EN_CHANGE == SHORT2FROMMP (mp1) && !pFI->bEntryChanging)
      {
         BOOL    bNotify;

         bNotify = TRUE;
         switch (SHORT1FROMMP (mp1))
         {
         case IDD_DEF_FORM_WIDTH:
            pFI->cx = GetEntryValue (hwnd,
                                     SHORT1FROMMP (mp1),
                                     pdi->ctryInfo.szDecimal[0],
                                     pdi->bUseInches);
            break;
         case IDD_DEF_FORM_HEIGHT:
            pFI->cy = GetEntryValue (hwnd,
                                     SHORT1FROMMP (mp1),
                                     pdi->ctryInfo.szDecimal[0],
                                     pdi->bUseInches);
            break;
         case IDD_DEF_FORM_LEFT_CLIP:
            pFI->ulLeft = GetEntryValue (hwnd,
                                         SHORT1FROMMP (mp1),
                                         pdi->ctryInfo.szDecimal[0],
                                         pdi->bUseInches);
            break;
         case IDD_DEF_FORM_RIGHT_CLIP:
            pFI->ulRight = GetEntryValue (hwnd,
                                          SHORT1FROMMP (mp1),
                                          pdi->ctryInfo.szDecimal[0],
                                          pdi->bUseInches);
            break;
         case IDD_DEF_FORM_TOP_CLIP:
            pFI->ulTop = GetEntryValue (hwnd,
                                        SHORT1FROMMP (mp1),
                                        pdi->ctryInfo.szDecimal[0],
                                        pdi->bUseInches);
            break;
         case IDD_DEF_FORM_BOT_CLIP:
            pFI->ulBottom = GetEntryValue (hwnd,
                                           SHORT1FROMMP (mp1),
                                           pdi->ctryInfo.szDecimal[0],
                                           pdi->bUseInches);
            break;
         default:
            bNotify = FALSE;
            break;
         }

         if (bNotify)
         {
            pFI->bUseInches = pdi->bUseInches; // @100713

            WinSendMsg (WinWindowFromID (hwnd, IDD_CLIPPING_WINDOW),
                        WM_USER_INFO,
                        MPFROMP (pFI),
                        (MPARAM)0);
         }
      }
      else if (((BN_CLICKED    == SHORT2FROMMP (mp1)) ||
                (BN_DBLCLICKED == SHORT2FROMMP (mp1))  )           &&
               ((IDD_DEF_FORM_INCHES      == SHORT1FROMMP (mp1)) ||
                (IDD_DEF_FORM_CENTIMETERS == SHORT1FROMMP (mp1))  ) )
      {
         pdi->bUseInches = GetCheck (hwnd, IDD_DEF_FORM_INCHES);

         DBPRINTF (("BN_CLICKED: mp1 = %d, bShouldUseInch = %d\n", SHORT1FROMMP (mp1), pdi->bUseInches));

         UpdateDefinedFormDlg (hwnd, pdi, pFI);
      }
      break;
   }

   case WM_COMMAND:
   {
      // get instance data
      pdi = (PDLGINSTANCE) WinQueryWindowULong( hwnd, QWL_USER );
      pFI = pdi->pFI;

      switch (SHORT1FROMMP (mp1))
      {
      case DID_UNDO:
      {
         // Free Current form list
         FreeUserDefinedFormList (pdi->pDevice);

         // read in original form list from INI file
         bRC = ReadUserDefinedFormList (pdi->szAppName, pdi->pDevice);
         assertF (bRC);

         // Fill in all known Forms on Form and connection page
         FillFormList (hwnd, pdi, pdi->pDevice, IDL_DEF_FORM_LIST);
         FillFormList (pdi->hwndConnectDlg, pdi, pdi->pDevice, IDL_FORMS);
         break;
      }

      case DID_DEFAULTS:
      {
         // Free Current form list
         FreeUserDefinedFormList (pdi->pDevice);

         // Fill in all known Forms on Form and connection page
         FillFormList (hwnd, pdi, pdi->pDevice, IDL_DEF_FORM_LIST);
         FillFormList (pdi->hwndConnectDlg, pdi, pdi->pDevice, IDL_FORMS);
         break;
      }

      case DID_ADD:
      {
         CHAR szTemp[LEN_STRING];
         PSZ  pszNull = "";

         WinQueryWindowText (WinWindowFromID (hwnd, IDE_DEF_FORM_NAME),
                             LEN_STRING-2, szTemp);

         // Validate form name is unique before saving
         AddDefinedForm (hwnd, pdi, pFI, pszNull);
         break;
      }

      case DID_DELETE:
      {
         // get currently selected Form to be deleted
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_FORM_LIST,
                                                 LM_QUERYSELECTION,
                                                 0L,0L));

         // verify user selected a form to delete
         if (sIdx != LIT_NONE)
         {
            DeleteDefinedForm (hwnd, pdi, sIdx);
         }
         else
         {
            WinMessageBox ((HWND)HWND_DESKTOP,
                           (HWND)NULL,
                           globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_SELECT_MSG_FORM],
                           globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_DELETE_FORM],
                           1L,
                           MB_OK | MB_INFORMATION);
         }
         break;
      }

      case DID_MODIFY:
      {
         // get currently selected Form to be modified
         sIdx = SHORT1FROMMR (WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_FORM_LIST,
                                                 LM_QUERYSELECTION,
                                                 0L,0L));

         // verify user selected a form to modify
         if (sIdx != LIT_NONE)
         {
            // Get Item Handle (FORM ID)
            ulFormID = (ULONG)WinSendDlgItemMsg (hwnd,
                                                 IDL_DEF_FORM_LIST,
                                                 LM_QUERYITEMHANDLE,
                                                 MPFROMSHORT (sIdx),
                                                 MPVOID);

            // if form is a user-defined form we can modify it
            if (ulFormID & DEF_TYPE_USER)
            {
               // if form is not currently part of a connection
               // we can still modify it
               if (!IsFormPartOfAConnection (pdi->pDevice, ulFormID))
               {
                  CHAR szIgnore[LEN_STRING];

                  // retrieve old form name we are modifying
                  FormNameFromID (pdi->pDriver, pdi->pDevice, szIgnore, ulFormID);

                  // Validate new form name is unique ignoring
                  // the old name before we continue the modify oper.
                  if (AddDefinedForm (hwnd, pdi, pFI, szIgnore))
                  {
                     // Find form entry in linked list or in driver table and delete
                     if (DeleteUserDefinedFormEntry (pdi->pDevice, ulFormID))
                     {
                        // remove form name from listbox
                        WinSendDlgItemMsg (hwnd,
                                           IDL_DEF_FORM_LIST,
                                           LM_DELETEITEM,
                                           MPFROMSHORT (sIdx),
                                           MPVOID);

                        // remove form name from listbox
                        WinSendDlgItemMsg (pdi->hwndConnectDlg,
                                           IDL_FORMS,
                                           LM_DELETEITEM,
                                           MPFROMSHORT (sIdx),
                                           MPVOID);
                     }
                     else
                     {
                        WinMessageBox ((HWND)HWND_DESKTOP,
                                       (HWND)NULL,
                                       globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_MSG_FORM],
                                       globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_FORM],
                                       1L,
                                       MB_OK | MB_INFORMATION);
                     }
                  }
               }
               else
               {
                  // Inform user that only unconnected forms are deletable
                  WinMessageBox ((HWND)HWND_DESKTOP,
                                 (HWND)NULL,
                                 globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_FORM_IN_USE],
                                 globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_FORM],
                                 1L,
                                 MB_OK | MB_INFORMATION);
               }
            }
            else
            {
               WinMessageBox ((HWND)HWND_DESKTOP,
                              (HWND)NULL,
                              globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_MSG_FORM],
                              globals.pbStringTable[STRING_TABLE_WARN_BASE - STRING_TABLE_BASE + WARNING_MODIFY_FORM],
                              1L,
                              MB_OK | MB_INFORMATION);
            }
         }
         break;
      }
      }

      return 0;
   }
   }

#ifndef DEBUG
   bRC++;       // Avoid compiler warnings
#endif

  // default for most cases
  return WinDefDlgProc (hwnd, msg, mp1, mp2);

} /* end DefineFormDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : IsUserDefinedForm                                       */
/* AUTHOR         : MJH                                                     */
/* DATE WRITTEN   : 4/29/97                                                 */
/* DESCRIPTION    : Returns TRUE if the form is a user defined form (from   */
/*                  printer properties dialog).                             */
/*                                                                          */
/* PARAMETERS:    pForm - Form Info structure from pddc->pdb->pForm         */
/*                                                                          */
/* RETURN VALUES: BOOLEAN TRUE for user defined, FALSE for normal form.     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @176300 - wrote it                                                       */
/****************************************************************************/
BOOL
IsUserDefinedForm (PFORMINFO2 pForm)
{
   if (DEF_TYPE_USER == MAKEULONG (0, HIUSHORT (pForm->ulFormID)))
      return TRUE;
   else
      return FALSE;
}
