/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
#define INCL_DOS
#define INCL_SPL
#define INCL_SPLFSE
#define INCL_SPLDOSPRINT
#define INCL_DEV
#define INCL_GPIBITMAPS    // To pull in the proper include files...
#define INCL_GPIPRIMITIVES
#define INCL_WINMENUS
#define INCL_ERRORS
#include <os2.h>

#ifndef SPLDATA_NOAUTORETRY
#define SPLDATA_NOAUTORETRY 0x4000
#endif

//#define INCL_WINP_FSRS
//#include <pmwinp.h>

#define INCL_DDIDEFS
#define INCL_DDIPATHS
#define INCL_GRE_LINES
#define INCL_GRE_DEVICE
#define INCL_GRE_DEVMISC2
#define INCL_DDIMISC
#define INCL_VMANDDI            // 2.2 Engine stuff
#define INCL_GRE_DCS
#define INCL_GRE_CLIP
#include <ddi.h>                // 2.2 Engine stuff
#include <pmddi.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"

// define new 2.2 escapes not yet in header files
// TBD - remove when in header files
#ifndef DEVESC_QUERYACTUALRESOLUTION
  #define DEVESC_QUERYACTUALRESOLUTION  3L
#endif

#ifndef DEVESC_QUERYPATTERNLIMITS
  #define DEVESC_QUERYPATTERNLIMITS     4L
#endif

#ifndef DEVESC_QUERYHALFTONESUPPORT
  #define DEVESC_QUERYHALFTONESUPPORT   5L
#endif

// New Merlin escap codes for sending separator pages
#ifndef DEVESC_SEP
  #define DEVESC_SEP   16305L
#endif

#ifndef HALFTONES_FOR_AREAS
  #define HALFTONES_FOR_AREAS     0x00000001L
#endif

#ifndef HALFTONES_FOR_BITBLT
  #define HALFTONES_FOR_BITBLT    0x00000002L
#endif

#ifndef PATTERN_LIMITS
typedef struct _PATTERN_LIMITS  {   /* patlim */
            USHORT usHorzLimit;
            USHORT usVertLimit;
        } PATTERN_LIMITS;
#endif

/* This is not what I was looking for.  I wanted to test to see if the
** struct QMJOBINFO was in the spooler's header file but C dont like
** that :~(.  So see if a bidi MACRO constant is defined instead.
*/
#ifndef PMSPLB_INCLUDED

    ULONG APIENTRY SplQmGetJobID (HSPL   hspl,
                                  ULONG  ulLevel,
                                  PVOID  pvBuf,
                                  ULONG  cbBuf,
                                  PULONG pcbNeeded);

   /*
    * structure returned by SplQmGetJobID
    */
   typedef struct _QMJOBINFO {
      ULONG ulJobID;
      PSZ   pszComputerName;
      PSZ   pszQueueName;
    } QMJOBINFO;
   typedef QMJOBINFO *PQMJOBINFO;

   #define PRJ_PAGESSPOOLED_PARMNUM     20

   typedef struct _PRJINFO4      /* prj4 */
   {
      USHORT uJobId;
      USHORT uPriority;
      PSZ    pszUserName;
      USHORT uPosition;
      USHORT fsStatus;
      ULONG  ulSubmitted;
      ULONG  ulSize;
      PSZ    pszComment;
      PSZ    pszDocument;
      PSZ    pszSpoolFileName;
      PSZ    pszPortName;
      PSZ    pszStatus;
      ULONG  ulPagesSpooled;
      ULONG  ulPagesSent;
      ULONG  ulPagesPrinted;
      ULONG  ulTimePrinted;
      ULONG  ulExtendJobStatus;
      ULONG  ulStartPage;
      ULONG  ulEndPage;
   } PRJINFO4;
   typedef PRJINFO4 *PPRJINFO4;

   ULONG  APIENTRY PrtNewPage (HFILE  hFile,
                               ULONG  ulPageNumber);

#endif

#define INCL_GENPLIB_LAYOUT
#include <genplib.h>

/* Function prototypes...
*/
// VOID            ReinitializeWithJobProps  (PDDC         pddc,        // @JPN2H99
BOOL            ReinitializeWithJobProps  (PDDC         pddc,           // @JPN2H99
                                           PDRIVDATA    pNewDrivData);
VOID            CreateAndSetDeviceSurface (PDDC         pddc);
LONG            CreateJournalFile         (PDDC         pddc);
VOID            StartRecording            (PDDC         pddc);
VOID            ErasePage                 (PDDC         pddc);

/****************************************************************************/
/* PROCEDURE NAME : StartTheThread                                          */
/* AUTHOR         : Mark Hamzy & Matt Rutkowski                             */
/* DATE WRITTEN   : 09-15-95                                                */
/* DESCRIPTION    : Utility function to help starting/updating a 2nd        */
/*                  output thread.                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @?????? - 06/10/96 - MFR - Updated to validate thread/spooler handles    */
/*                            This helps abort processing.                  */
/*                                                                          */
/****************************************************************************/
BOOL
StartTheThread (PDDC pddc, LHANDLE lHandle, HFILE hFile)
{
   BOOL        rc = FALSE;
   ABORT_DATA  AbortData;

   if (pddc->pdb->fCalledGplThreadStart)
      return TRUE;

   assertT (pddc->pdb->hThread == (HTHREAD)NULL);
   assertT (!hFile && !lHandle);

   // Validate our thread handle (i.e. from GplThreadCreateInstance() )
   if (pddc->pdb->hThread)
   {
      // Validate we have a valid file handle
      // This may not be true if DEVESC_ABORTDOC called.
      // We have an "hFile" for PrtOpen() and "lHandle" for SplQmOpen()
      if (hFile || lHandle)
      {
         AbortData.hmcbHeap           = pddc->pdb->hmcbHeap;
         AbortData.bSendString        = FALSE;
         AbortData.pszAbortString     = (PSZ)NULL;
         AbortData.bSendRepeatBlock   = FALSE;
         AbortData.uchAbortChar       = 0;
         AbortData.ulAbortRepeatCount = 0;
         AbortData.bSendBlock         = FALSE;
         AbortData.pbAbortBlock       = (PBYTE)NULL;
         AbortData.ulAbortBlockSize   = 0;

/////////// Allocate device handle!!! before calling for data! - MFR
/////////AllocateDeviceHandle (pddc, FALSE);

         // Call the subclassed function...
         rc = pddc->pdb->pDevice->pSubclassedFunctions->pfnAbortJob (pddc,
                                                                     &AbortData,
                                                                     pddc->pdb->pvDeviceCookie);

         // Create a second print thread
         rc = GplThreadStart (pddc->pdb->hThread,                 // Instance handle
                              lHandle,                            // Handle to spooler
                              hFile,                              // Handle to device
                              0,                                  // Buffer size: Ask for defaults
                              0,                                  // Max buffers: Ask for defaults
                              AbortData.uchAbortChar,             // Abort char
                              AbortData.ulAbortRepeatCount,       // Abort char count
                              AbortData.ulAbortBlockSize,         // Abort buffer size
                              AbortData.pbAbortBlock,             // Abort block
                              AbortData.pszAbortString,           // Abort string
                              pddc->pdb->szLogAddress,            // Abort address
                              globals.pbStringTable[STRING_TABLE_ERROR_BASE - STRING_TABLE_BASE + ERROR_OFFLINE],
                              pddc->pdb->pDrivData->szDeviceName, // Device description
                              pddc->pdb->ulType);                 // DC type
         assertF (rc);

         pddc->pdb->fCalledGplThreadStart = TRUE;
      }
   }

   return rc;

} /* end StartTheThread  */

/****************************************************************************/
/* PROCEDURE NAME : EndTheThread                                            */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 09-18-98                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
BOOL
EndTheThread (PDDC pddc)
{
   APIRET   rc;

   if (!pddc->pdb->hThread)
      return FALSE;

   if (!pddc->pdb->fCalledGplThreadStart)
      return TRUE;

   rc = GplThreadEnd (pddc->pdb->hThread);
   assertF (rc);

   pddc->pdb->fCalledGplThreadStart = FALSE;

#ifndef DEBUG
   rc++;          // Avoid compiler warnings
#endif

   return TRUE;
}

/****************************************************************************/
/* PROCEDURE NAME : FormFeedSeen                                            */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 09-15-95                                                */
/* DESCRIPTION    : Returns TRUE if a form feed character is in the last    */
/*                  part of the output buffer.  Note: this is not 100%      */
/*                  accurate since there is no history kept to make sure    */
/*                  that we are only checking a valid command and not in    */
/*                  the middle of a command.                                */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
BOOL
FormFeedSeen (PDDC  pddc,
              PBYTE pbData,
              LONG  cInCount)
{
   PCMDINFO    pCmds = pddc->pdb->pDevice->pCmds;
   BOOL        rc = FALSE;
#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   // Is there enough data to look for a page eject command?
   if (!pCmds || pCmds->ulCmdPageEject >= cInCount)
      return FALSE;

   /* Are there commands?
   ** Is there a page eject command?
   ** Is the page eject command at the end of the data stream?
   */

   if (  pCmds->ulCmdPageEject
      && 0 == memcmp (pbData + cInCount - pCmds->ulCmdPageEject,
                      pCmds->pszCmdPageEject,
                      pCmds->ulCmdPageEject)
      )
      rc = TRUE;

   /* Or is the page eject command before end printing command
   ** because of CMVC defect #230424 - PTT: at the end blank sheet
   */
   if (  pCmds->ulCmdTerm
      && 0 == memcmp (pbData + cInCount - pCmds->ulCmdTerm,
                      pCmds->pszCmdTerm,
                      pCmds->ulCmdTerm)
      && pCmds->ulCmdPageEject
      && 0 == memcmp (pbData + cInCount - pCmds->ulCmdPageEject -
                      pCmds->ulCmdTerm,
                      pCmds->pszCmdPageEject,
                      pCmds->ulCmdPageEject)
      )
      rc = TRUE;

   DBPRINTIF ((pDbg->bDEVESC_RAWDATA, "%s() returns %s\n", __FUNCTION__, rc ? "TRUE" : "FALSE"));

   return rc;

} /* end FormFeedSeen */

/****************************************************************************/
/* PROCEDURE NAME : SpoolerEntryPointExists                                 */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 09-15-95                                                */
/* DESCRIPTION    : Checks to see if the exported function name exists in   */
/*                  the spooler that is installed on the operating system.  */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
BOOL
SpoolerEntryPointExists (PFN *ppfnEntryPoint, PSZ pszEntryName)
{
   BOOL      fExists             = FALSE;
   HMODULE   hmodSpl;
   CHAR      achObjectName[50];
   APIRET    rc;

   if (!ppfnEntryPoint || !pszEntryName)
      return FALSE;

   if (*ppfnEntryPoint)
      // Already filled in!
      return TRUE;

   rc = DosLoadModule (achObjectName,
                       sizeof (achObjectName),
                       "PMSPL",
                       &hmodSpl);
   assertT (rc);
   if (NO_ERROR != rc)
      return FALSE;

   rc = DosQueryProcAddr (hmodSpl,
                          0,
                          pszEntryName,
                          ppfnEntryPoint);
   assertT (rc);

   fExists = NO_ERROR == rc;

   rc = DosFreeModule (hmodSpl);
   assertT (rc);

   return fExists;
}

/****************************************************************************/
/* PROCEDURE NAME : IncrementSpoolerPageNum                                 */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 09-15-95                                                */
/* DESCRIPTION    : Notifies the spooler that the page number has been      */
/*                  incremented.                                            */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
BOOL
IncrementSpoolerPageNum (PDDC pddc)
{
   PRJINFO4    prjInfo4;
   ULONG       ulSizeNeeded;
   PQMJOBINFO  pqmJobInfo   = NULL;
   BOOL        fSuccess     = FALSE;
   APIRET      rc;

   DBPRINTF (("%s(): Enter hspl = %08X\n", __FUNCTION__, pddc->pdb->hspl));

   if (!pddc->pdb->hspl)
      return TRUE;

   // We need the spooler API to report page num
   if (!SpoolerEntryPointExists (&globals.pfnSplQmGetJobID,
                                 "SPLQMGETJOBID"))
   {
      // Entry point not found (old spooler) return FALSE
      DBPRINTF(( "%s(): SplQmGetJobID() does not exist, old spooler!!!\n", __FUNCTION__ ));
      return FALSE;
   }

   /* query the QMJOBINFO struct
   ** First try a large buffer...
   */
   ulSizeNeeded = 4096;
   pqmJobInfo = GplMemoryAlloc (pddc->pdb->hmcbHeap, ulSizeNeeded);
   assertF (pqmJobInfo);
   if (!pqmJobInfo)
      return FALSE;

   rc = globals.pfnSplQmGetJobID (pddc->pdb->hspl,    // Spooler handle
                                  0,                  // Level 0
                                  (PVOID)pqmJobInfo,  // buffer
                                  ulSizeNeeded,
                                  &ulSizeNeeded);
   if (ERROR_MORE_DATA == rc)
   {
      // Retry with the correct size
      rc = GplMemoryFree (pqmJobInfo);
      assertT (rc);

      pqmJobInfo = GplMemoryAlloc (pddc->pdb->hmcbHeap, ulSizeNeeded);
      assertF (pqmJobInfo);
      if (!pqmJobInfo)
         return FALSE;

      rc = globals.pfnSplQmGetJobID (pddc->pdb->hspl,    // Spooler handle
                                     0,                  // Level 0
                                     (PVOID)pqmJobInfo,  // buffer
                                     ulSizeNeeded,
                                     &ulSizeNeeded);
      assertT (rc);

      DBPRINTIF(( (BOOL)(pqmJobInfo), "%s(): SplQmGetJobID( %lu ), returns Job ID = %d\n",
                  __FUNCTION__, pddc->pdb->hspl, pqmJobInfo->ulJobID ));

      if (rc)
         goto done;
   }

   if (NO_ERROR == rc)
   {
      /* Now that we have valid QMJOBINFO data,
      ** we know the job ID #
      */
      rc = SplQueryJob (pqmJobInfo->pszComputerName,
                        pqmJobInfo->pszQueueName,
                        pqmJobInfo->ulJobID,
                        4,
                        (PVOID)&prjInfo4,
                        sizeof (prjInfo4),
                        &ulSizeNeeded);

      if (NO_ERROR == rc || ERROR_MORE_DATA == rc)
      {
         DBPRINTF(( "%s(): Before: SplSetJob(), ulPagesSpooled=%d\n", __FUNCTION__, prjInfo4.ulPagesSpooled ));

         // Increment the number of pages spooled
         prjInfo4.ulPagesSpooled++;

         // Update the spooler
         rc = SplSetJob (pqmJobInfo->pszComputerName,
                         pqmJobInfo->pszQueueName,
                         pqmJobInfo->ulJobID,
                         4,
                         (PVOID)&(prjInfo4.ulPagesSpooled),
                         sizeof (prjInfo4.ulPagesSpooled),
                         PRJ_PAGESSPOOLED_PARMNUM);

         DBPRINTF(( "%s(): After: SplSetJob(), ulPagesSpooled=%d\n", __FUNCTION__, prjInfo4.ulPagesSpooled ));
      }

      assertF (NO_ERROR == rc || ERROR_MORE_DATA == rc);
   }
   else
   {
      // Error!
      assertT (rc);
      goto done;
   }

   fSuccess = TRUE;
done:

   // Clean up
   if (pqmJobInfo)
   {
      rc = GplMemoryFree (pqmJobInfo);
      assertT (rc);
   }

   DBPRINTF (("%s(): Exit, return = %d\n", __FUNCTION__, fSuccess));

   return fSuccess;

} /* end IncrementSpoolerPageNum */

/****************************************************************************/
/* PROCEDURE NAME : UpdateSpoolJobProperties                                */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 09-15-95                                                */
/* DESCRIPTION    : Updates the job properties that have been kept with     */
/*                  the print job.                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
VOID
UpdateSpoolJobProperties (HSPL      hspl,
                          PDRIVDATA pDrivData)
{
   PRJINFO3         prjInfo3;
   QMJOBINFO        qmJobInfo;
   ULONG            ulSizeNeeded = sizeof (qmJobInfo);
   APIRET           rc;

   DBPRINTF (("%s(): Enter hspl = %08X\n", __FUNCTION__, hspl));

   if (!hspl)
      return;

   // Clear out the structures
   memset (&prjInfo3, 0, sizeof (prjInfo3));
   memset (&qmJobInfo, 0, sizeof (qmJobInfo));

   // Only query the job id.  Dont care about the strings
   rc = globals.pfnSplQmGetJobID (hspl,               // Spooler handle
                                  0,                  // Level 0
                                  (PVOID)&qmJobInfo,  // buffer
                                  ulSizeNeeded,
                                  &ulSizeNeeded);

   prjInfo3.pDriverData = pDrivData;

   // Update only the job properties
   rc = SplSetJob ((PSZ)NULL,
                   (PSZ)NULL,
                   qmJobInfo.ulJobID,
                   3,
                   (PVOID)prjInfo3.pDriverData,
                   sizeof (prjInfo3.pDriverData),
                   PRJ_DRIVERDATA_PARMNUM);
   assertT (rc);

#ifndef DEBUG
   rc++;          // Avoid compiler warnings
#endif

   DBPRINTF (("%s(): Exit\n", __FUNCTION__));
}

/****************************************************************************/
/* PROCEDURE NAME : OmniDeviceOpen                                          */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 01-27-96                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
APIRET
OmniDeviceOpen (PVOID  pvDDC,
                PVOID  pvHandle,
                PVOID  pvLogAddress,
                PVOID  pvHFile,
                PVOID  pvAction,
                ULONG  cbFile,
                ULONG  ulDefAttrs,
                ULONG  ulDefAction,
                ULONG  ulDefAccess,
                PVOID  pvEAInfo)
{
   APIRET apiret     = NO_ERROR;
   ULONG  ulWork;
   PHFILE pHFile     = (PHFILE)pvHFile;
   PSZ    pszLogAddr = (PSZ)pvLogAddress;
   ULONG  ulAttrs    = FILE_NORMAL;
   ULONG  ulAction   = OPEN_ACTION_REPLACE_IF_EXISTS
                     | OPEN_ACTION_CREATE_IF_NEW;
   ULONG  ulAccess   = OPEN_ACCESS_WRITEONLY
                     | OPEN_SHARE_DENYREADWRITE;
   ULONG  ulResponse = MBID_RETRY;  // return code from SplMessageBox

   DBPRINTF (("%s: Enter:PrtOpen (%s)\n", __FUNCTION__, pszLogAddr));

   while (ulResponse == MBID_RETRY || ulResponse == MBID_IGNORE)
   {
      apiret = PrtOpen (pszLogAddr,
                        pHFile,
                        &ulWork,
                        0,
                        ulAttrs,
                        ulAction,
                        ulAccess,
                        0);

      if (apiret != NO_ERROR)
      {
         DBPRINTF(( "%s: PrtOpen(%s) FAILED! rc=%d\n", __FUNCTION__, pszLogAddr, apiret ));

         // SPLINFO_DDERROR          Presentation driver error
         // SPLINFO_INFORMATION      Information only, no error.
         // SPLINFO_WARNING          Warning.
         // SPLINFO_ERROR            Recoverable error.
         // SPLINFO_USERINTREQD      This flag is optional. It shows that recovery requires action
         //                          from the user.
         // SPLDATA_PRINTERJAM           Printer is jammed, offline, or not turned on
         // SPLDATA_UNEXPECTERROR        Unexpected DOS error
         // SPLDATA_OTHER                Any other error

         ulResponse = SplMessageBox (pszLogAddr,
                                     SPLINFO_DDERROR
                                     | SPLINFO_USERINTREQD
                                     | SPLINFO_ERROR,
                                     SPLDATA_PRINTERJAM,
                                     globals.pbStringTable[STRING_TABLE_ERROR_BASE - STRING_TABLE_BASE + ERROR_OFFLINE],
                                     pszLogAddr,
                                     HWND_DESKTOP,
                                     MB_ABORTRETRYIGNORE);

#ifdef DEBUG
         {
            CHAR  szTemp[32];

            DebugInterpretSplMsgRC (ulResponse, (PSZ)szTemp);

            DBPRINTF (("%s: SplMessageBox() ulResponse = %d (%s)\n", __FUNCTION__,
                       ulResponse, szTemp));
         }
#endif
      }
      else
      {
         DBPRINTF (("%s: Exit: PrtOpen(%s) Success, HFile = %x\n", __FUNCTION__,
                    pszLogAddr, *pHFile));

         return apiret;
      }
   }

   DBPRINTF (("%s: SplMessageBox() 'MBID_ABORT' requested from user\n", __FUNCTION__,
              pszLogAddr, *pHFile));

   // Tell Queue processor to abort job and not play metafile or send raw data
   // This causes the spooler to initiate an asynchronous  DEVESC_ABORTDOC
   ulResponse = SplControlDevice (NULL,         // pszComputerName,
                                  pszLogAddr,   // pszPortName,
                                  PRD_DELETE);  // ulControl

   // make sure HFILE is NULL so we don't try to start using a bogus file
   *pHFile = 0;

   // May need to add a 3-5 second delay to allow spooler to start a thread
   // and call DEVESC_ABORTDOC, especially in client/server environment
   DosSleep (3000);

   DBPRINTF (("%s: SplControlDevice (PRD_DELETE(%d)); rc = %d\n", __FUNCTION__,
              PRD_DELETE, ulResponse));

   DBPRINTF (("%s: Exit\n", __FUNCTION__));

   return apiret;

} /* end OmniDeviceOpen */


/****************************************************************************/
/* PROCEDURE NAME : OmniDeviceClose                                         */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 01-27-96                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
APIRET
OmniDeviceClose (PVOID pv, HFILE hFile)
{
   APIRET apiret;

   DBPRINTF (("%s Entry: PrtClose(%x)\n", __FUNCTION__, hFile));

   apiret = PrtClose (hFile);
   assertT (apiret != NO_ERROR);

   return apiret;

} /* end OmniDeviceClose */

/****************************************************************************/
/* PROCEDURE NAME : OmniSpoolerOpen                                         */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 03-24-96                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
HSPL
OmniSpoolerOpen (PVOID       pvDDC,
                 PVOID       pvHandle,
                 PSZ         pszToken,
                 LONG        lCount,
                 PQMOPENDATA pqmdopData)
{
   HSPL   hspl = NULLHANDLE;

   hspl = SplQmOpen (pszToken, lCount, pqmdopData);
   assertF (hspl);

   return hspl;

} /* end OmniSpoolerOpen */


/****************************************************************************/
/* PROCEDURE NAME : OmniSpoolerStart                                        */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 03-24-96                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
OmniSpoolerStartJob (PVOID pvDDC,
                     PVOID pvHandle,
                     HSPL  hspl,
                     PSZ   pszDocumentName)
{
   BOOL   bRC = SPL_ERROR;

   if (hspl)
   {
      // queue manager start document
      bRC = SplQmStartDoc (hspl, pszDocumentName);
      assertF (bRC);
   }

   return bRC;

} /* end OmniSpoolerStartJob */


/****************************************************************************/
/* PROCEDURE NAME : OmniSpoolerEnd                                          */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 03-24-96                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
LONG
OmniSpoolerEndJob (PVOID pvDDC,
                   PVOID pvHandle,
                   HSPL  hspl)
{
   LONG   lJobID;

   lJobID = (LONG)SplQmEndDoc (hspl);
   assertF (lJobID != SPL_ERROR);

   return lJobID;

} /* end OmniSpoolerEndJob */

/****************************************************************************/
/* PROCEDURE NAME : OmniSpoolerClose                                        */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 03-24-96                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
OmniSpoolerClose (PVOID pvDDC,
                  PVOID pvHandle,
                  HSPL  hspl)
{
   BOOL   bRC = SPL_ERROR;

   if (hspl)
   {
      bRC = SplQmClose (hspl);
      assertF (bRC);
   }

   return bRC;

} /* end OmniSpoolerClose */


/****************************************************************************/
/* PROCEDURE NAME : ReinitializeWithJobProps                                */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 11-19-95                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @JPN2H99- 08/01/99 - UCC [IBMJ]- Fix duplex printing problem and H/W nup */
/*                                  problems.                               */
/****************************************************************************/
// VOID                                                     // @JPN2H99
BOOL                                                        // @JPN2H99
ReinitializeWithJobProps (PDDC pddc, PDRIVDATA pNewDrivData)
{
   PDEVICEBLOCK   pdb            = pddc->pdb;
   PDRIVERINFO    pDriver        = pddc->pdb->pDriver;
   PDEVICEINFO    pDevice        = pddc->pdb->pDevice;
   ULONG          ulDefaultFormID,
                  ulDefaultTrayID,
                  ulDefaultMediaID;
   PTRAYINFO      pTrayInfoLcl;
   PMEDIAINFO     pMediaInfoLcl;
   APIRET         rc;
   BOOL           fCalledUserDefTray = FALSE,
                  fCalledUserDefForm = FALSE,
                  fCalledUserDefConn = FALSE,
                  fCalledUserDefFont = FALSE;
   BOOL           bRC;
// begin @JPN2H99
   PDRIVDATA      pDrivTemp;
   BOOL           bRetCode;
   ULONG          ulJobPhaseOld, ulJobPhaseNew;
   PJOBPROPERTIES pJobOld, pJobNew;

//   if (pdb->pDrivData)
//   {
//      rc = GplMemoryFree (pdb->pDrivData);
//      assertT (rc);
//   }
//
//   pdb->pDrivData = ReturnDriverData (pdb,
     pDrivTemp = ReturnDriverData (   pdb,
// end @JPN2H99
                                      (PDLGINSTANCE)NULL,
                                      pdb->hmcbHeap,
                                      pNewDrivData,
                                      pdb->szLogAddress,
                                      FALSE,
                                      pDevice,
                                      pDriver);
// begin @JPN2H99
   // If dynamic job properties is passed in with same setting as
   // previous one, prevent it to reset jobproperties !!
   // If not it will cause duplex print or H/W N-Up not to work
   // correctly !!
   pJobOld = (PJOBPROPERTIES)&pdb->pDrivData->abGeneralData;
   pJobNew = (PJOBPROPERTIES)&pDrivTemp->abGeneralData;
   ulJobPhaseOld = pJobOld->ulJobPhaseControl;
   ulJobPhaseNew = pJobNew->ulJobPhaseControl;
   pJobOld->ulJobPhaseControl =
   pJobNew->ulJobPhaseControl = 0L;

   // Compare old data with new data.
   if ( 0 == memcmp(pdb->pDrivData, pDrivTemp, pDrivTemp->cb))
   {
      rc = GplMemoryFree (pDrivTemp);
      assertT (rc);
      bRetCode = FALSE;
      pJobOld->ulJobPhaseControl = ulJobPhaseOld;
   }
   else
   {
      rc = GplMemoryFree (pdb->pDrivData);
      assertT (rc);
      pdb->pDrivData = pDrivTemp;
      pJobNew->ulJobPhaseControl = ulJobPhaseNew;
      bRetCode = TRUE;
   }
// end @JPN2H99

   pdb->dop.pdriv      = (PDRIVDATA)pdb->pDrivData;
   pdb->pJobProperties = (PJOBPROPERTIES)&pdb->pDrivData->abGeneralData;

#ifdef DEBUG
   DebugOutputJobProperties (TRUE,
                             "ReinitializeWithJobProps called!!!!!!!!",
                             pdb->pJobProperties);
#endif

   pdb->pResInfo = GetpResFromResID (pdb->pDriver,
                                     pdb->pDevice,
                                     pdb->pJobProperties);
   pdb->pPrintMode = GetpPrintModeFromID (pdb->pDriver,
                                          pdb->pDevice,
                                          pdb->pJobProperties);

   // @224087: During some applications work when STARTDOC_WPROP is
   // called pddc->pdb->pDevice->pUserDefData doesn't contain data,
   // although user wants to print on User Defined Form, Tray, etc.
   // thus, read in user defined values from INI file to linked list

   if (!pdb->pDevice->pUserDefData->bTraysInit)
   {
      bRC = ReadUserDefinedTrayList (pdb->szAppName, pdb->pDevice);
      assertF (bRC);
      fCalledUserDefTray = TRUE;
   }

   if (!pdb->pDevice->pUserDefData->bFormsInit)
   {
      bRC = ReadUserDefinedFormList (pdb->szAppName, pdb->pDevice);
      assertF (bRC);
      fCalledUserDefForm = TRUE;
   }

   if (!pdb->pDevice->pUserDefData->bConnsInit)
   {
      bRC = ReadUserDefinedConnectList (pdb->szAppName, pdb->pDevice);
      assertF (bRC);
      fCalledUserDefConn = TRUE;
   }

   if (!pdb->pDevice->pUserDefData->bFontsInit)
   {
      bRC = ReadUserDefinedFontList (pdb->szAppName, pdb->pDevice);
      assertF (bRC);
      fCalledUserDefFont = TRUE;
   }

   GetIDsFromConnID (pdb->pDriver,
                     pdb->pDevice,
                     pdb->pJobProperties->ulDefConnID,
                     &ulDefaultTrayID,
                     &ulDefaultFormID,
                     &ulDefaultMediaID);

   // Get current tray information for this connection
   TrayInfoFromID (pdb->pDriver,
                   pdb->pDevice,
                   ulDefaultTrayID,
                   &pTrayInfoLcl);

   *pdb->pTrayInfo = *pTrayInfoLcl;

   TrayNameFromID (pdb->pDriver,
                   pdb->pDevice,
                   pdb->pTrayInfo->szTrayName,
                   ulDefaultTrayID);

   GetDefaultFormInfo (pdb,
                       ulDefaultFormID,
                       pdb->pJobProperties->ulDefResID,
                       pdb->pFormInfo);

   // Get current tray information for this connection
   MediaInfoFromID (pdb->pDriver,
                    pdb->pDevice,
                    ulDefaultMediaID,
                    &pMediaInfoLcl);

   *pdb->pMediaInfo = *pMediaInfoLcl;

   MediaNameFromID (pdb->pDriver,
                    pdb->pDevice,
                    pdb->pMediaInfo->szMediaName,
                    ulDefaultMediaID);

   GetDefaultFontInfo (pdb,
                       pdb->pJobProperties->ulDefFontID,
                       &(pdb->pFontInfo));

   switch (pddc->pdb->ulType)
   {
   case OD_DIRECT:
   case OD_QUEUED:
   {
      // If this is a raw data job, then there is no need for the following...
      if (!pddc->pdb->fRawOnly)
      {
         PDEVICESURFACE pds;

         pds = (PDEVICESURFACE)pddc->pdb->pDeviceSurface;

         // Clean up matrix dither info
         if (pddc->pmdi)
         {
            rc = GplMemoryFree (pddc->pmdi);
            assertT (rc);
         }
         pddc->pmdi = NULL;

         // Clean up surface bitmap info
         if (pds->SurfaceBmapInfo.pBits)
         {
            rc = GplMemoryFree (pds->SurfaceBmapInfo.pBits);
            assertT (rc);
         }
         pds->SurfaceBmapInfo.pBits = NULL;
      }
      break;
   }
   }

   SetupDeviceSurface (pddc);

   // @224087: free lists if we have full them in this function

   if (fCalledUserDefConn)
       FreeConnectionList (pdb->pDevice);
   if (fCalledUserDefTray)
       FreeUserDefinedTrayList (pdb->pDevice);
   if (fCalledUserDefForm)
       FreeUserDefinedFormList (pdb->pDevice);
   if (fCalledUserDefFont)
       FreeUserDefinedFontList (pdb->pDevice);

#ifndef DEBUG
   rc++;          // Avoid compiler warnings
#endif
   return bRetCode;                                         // @JPN2H99

} /* end ReinitializeWithJobProps */

/****************************************************************************/
/* PROCEDURE NAME : CreateAndSetDeviceSurface                               */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 11-19-95                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/* NOTE: This function is destructive in the sense that after calling this, */
/*       you need to erase the page.                                        */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
CreateAndSetDeviceSurface (PDDC pddc)
{
   PDEVICESURFACE pds = (PDEVICESURFACE)pddc->pdb->pDeviceSurface;
   APIRET         rc;

   if (pddc->StateData.fUseBitmap)
   {
      if (GRE_22 <= globals.ulGreVersion)
      {
         // 2.2 Engine or above
         assertF (pds);

         assertF (pds->SurfaceBmapInfo.ulBytesPerLine);
         if (!pds->SurfaceBmapInfo.pBits)
         {
            pds->SurfaceBmapInfo.pBits = GplMemoryAlloc (0,
                                                         pds->SurfaceBmapInfo.ulBytesPerLine
                                                         * pddc->bmp.cy);
            assertF (pds->SurfaceBmapInfo.pBits);
         }
      }
   }

   // @166185 - Make sure we have a SetDeviceSurface
   if (!globals.pfnSetDeviceSurface)
   {
      CHAR    achObjectName[50];

      // We need to get the set device surface function from someone
      if (GRE_22 <= globals.ulGreVersion)
      {
         // 2.2 Engine or above
         if (!globals.hModuleGre)
         {
            rc = DosLoadModule (achObjectName,
                                sizeof (achObjectName),
                                "PMGRE",
                                &globals.hModuleGre);
            assertT (rc);

            rc = DosQueryProcAddr (globals.hModuleGre,
                                   0,
                                   "SETDEVICESURFACE",
                                   &globals.pfnSetDeviceSurface);
            assertT (rc);
         }
      }
      else
      {
         // Not the 2.2 Engine
         assertF (*globals.achPrePath);

         if (!globals.hModulePre)
         {
            rc = DosLoadModule (achObjectName,
                                sizeof (achObjectName),
                                globals.achPrePath,
                                &globals.hModulePre);
            assertT (rc);

            rc = DosQueryProcAddr (globals.hModulePre,
                                   0,
                                   "SETDEVICESURFACE",
                                   &globals.pfnSetDeviceSurface);
            assertT (rc);
         }
      }
   }

   // Call SetDeviceSurface
   if (globals.pfnSetDeviceSurface)
   {
      if (GRE_22 <= globals.ulGreVersion)
         // 2.2 Engine or above
         rc = globals.pfnSetDeviceSurface (pddc->pdb->hdc,
                                           pddc->pdb->pDeviceSurface);
      else
         // Call shadow DC
         rc = globals.pfnSetDeviceSurface (pddc->hdcShadow,
                                           pddc->pdb->pDeviceSurface);
      assertF (rc);
   }

#ifndef DEBUG
   rc++;          // Avoid compiler warnings
#endif

} /* end CreateAndSetDeviceSurface */

/****************************************************************************/
/* PROCEDURE NAME : CreateJournalFile                                       */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 11-19-95                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
LONG
CreateJournalFile (PDDC pddc)
{
   ULONG    ulSize;
   LONG     lrc;

   // create the journal file only if no abort was processed
   if (!pddc->pdb->fAbortDocCalled)
   {
     // We may have already allocated an hjournal if the app
     // improperly called DEVESC_STARTDOC multiple times
     if (!pddc->hJournal)
     {
        // The driver is responsible for allocating the memory!!
        // ... but first query the size.
        ulSize = GplJournalCreateInstance (NULL, NULL, 0);
        assert (ulSize);
        if (!ulSize)
           return DEV_ERROR;

        // allocate the memory
        pddc->hJournal = GplMemoryAlloc (pddc->pdb->hmcbHeap, ulSize);
        assert (pddc->hJournal);
        if (NULL == pddc->hJournal)
           return DEV_ERROR;

        memset (pddc->hJournal, 0, ulSize);
     }

     lrc = GplJournalCreateInstance (pddc->hJournal,
                                     &pddc->IJournal,
                                     0);
     assertF (lrc);

     return DEV_OK;
   }

#ifndef DEBUG
   lrc++;          // Avoid compiler warnings
#endif

   // abort doc called or create failed
   return DEV_ERROR;

} /* end CreateJournalFile */

/****************************************************************************/
/* PROCEDURE NAME : StartRecording                                          */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 11-19-95                                                */
/* DESCRIPTION    : Created for defect @166185                              */
/*                                                                          */
/* PARAMETERS:    PDDC - presentation device drawing context handle         */
/*                                                                          */
/* RETURN VALUES: NONE                                                      */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @203613 - split StartRecordingOrErasePage into two functions.            */
/*                                                                          */
/****************************************************************************/
VOID
StartRecording (PDDC pddc)
{
   PFNSUBCLASS    pfnTbl        = NULL;
   APIRET         rc;

   pfnTbl = pddc->pdb->pDevice->pSubclassedFunctions;

   switch (pddc->pdb->ulType)
   {
   case OD_DIRECT:
   case OD_QUEUED:
   {
      pddc->pdb->fRecording = TRUE;                                      

      // If we were journalling and we are not journalling anymore
      if (pddc->StateData.fBanding)
      {
         if (  !pfnTbl->pfnBandingSupport
            || !pfnTbl->pfnBandingSupport (pddc->pdb->pvDeviceCookie,
                                           BANDINGSUPPORT_CREATE,
                                           &pddc->hJournal)
            )
         {
            // Restart journalling after the ResetDC2 call!
            rc = CreateJournalFile (pddc);
            assertT (DEV_ERROR == rc);
         }
      }
      break;
   }
   }

#ifndef DEBUG
      rc++;          // Avoid compiler warnings
#endif

} /* StartRecording */

/****************************************************************************/
/* PROCEDURE NAME : ErasePage                                               */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 9-15-98                                                 */
/* DESCRIPTION    : Created for defect @203613                              */
/*                                                                          */
/* PARAMETERS:    PDDC - presentation device drawing context handle         */
/*                                                                          */
/* RETURN VALUES: NONE                                                      */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @206862 - sleeper     exposed.  bounds accumulation was being turned off */
/*                                                                          */
/****************************************************************************/
VOID
ErasePage (PDDC pddc)
{
   RECTL  rectlFull;
   APIRET rc;

   // Set the full page to be visible and erase it
   rectlFull.xLeft   = 0;
   rectlFull.yBottom = 0;
   rectlFull.xRight  = pddc->bmp.cx;
   rectlFull.yTop    = pddc->bmp.cy;

   // @TBD perhaps GplJournalSetVisRegion should be changed to
   //      turn back on bounding instead.
   rc = GplJournalSetVisRegion (pddc->pdb->hdc,
                                &rectlFull,
                                0,                         // lBandDCOriginX
                                0,                         // lBandDCOriginY
                                TRUE,                      // Erase PS
                                SETUPDC_VISRGN
                                | SETUPDC_ACCUMBOUNDSON    // we need bounding! @206862
                                | SETUPDC_ORIGIN
                                | SETUPDC_RECALCCLIP);
   assertF (rc);

#ifndef DEBUG
      rc++;          // Avoid compiler warnings
#endif
}

/****************************************************************************/
/* PROCEDURE NAME : Escape                                                  */
/* AUTHOR         : Matt Rutkowski, Mark Hamzy                              */
/* DATE WRITTEN   : 09-12-93                                                */
/* DESCRIPTION    : Processes the call DevEscape() from PM applications.    */
/*                                                                          */
/*  NOTE: Todo : investigate better handling of asynch. entry by ABORTDOC   */
/*  It has been suggested that we enterdriver() on every escape but         */
/*  ABORTDOC. Returning HDC_BUSY causes spooler never to abort the job.     */
/*  The scenario is an app calling us with ABORTDOC for an HDC from a       */
/*  separate thread that is printing to the same HDC.                       */
/*  Also possible is the scenario that an ABORTDOC comes in after pddc is   */
/*  freed up so that we can not access any part of it or we'll trap -MFR    */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES: DEVESC_ERROR         (-1L) == Failure                     */
/*                DEV_OK                (1L) == Success                     */
/*                DEVESC_NOTIMPLEMENTED (0L) == Escape code not implemented */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @159049 - Matt Rutkowski - 7/8/96 - Fixed # pages spooled logic for both */
/*           "Queued" and "Direct" jobs so the numbers are correct in       */
/*           Print Queue.                                                   */
/* @157983 - Mark H. - 7/23/96 - Fixed DJP re-allocating device handles     */
/*           on NEWFRAME calls, so devices can change resolutions correctly */
/* @3632   - 04/08/98 - UCC [IBMJ]- Add 2 flag for DevFnt sup use.          */
/* @JPN2H99- 08/01/99 - UCC [IBMJ]- Fix duplex printing problem and H/W nup */
/*                                  problems. Pre-rawdata, post-rawdata supp*/
/****************************************************************************/
LONG ENGENTRY
Escape (HDC       hdc,
        LONG      lEscape,
        LONG      cInCount,
        PBYTE     pInData,
        PLONG     pcOutCount,
        PBYTE     pOutData,
        PDDC      pddc,
        ULONG     ulFunction)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   BOOL           fOK;
   REGREC         regrec;
   ULONG          ulException;
   LONG           lrc;
   LONG           lWork;
   ULONG          ulWork;
   APIRET         rc;
   HSTD           hstd;
   PCHAR          pchSplBuffer  = NULL;
   PFNSUBCLASS    pfnTbl        = NULL;
   PDRIVERINFO    pDriver       = NULL;
   PDEVICEINFO    pDevice       = NULL;
   PDEVICEBLOCK   pdb;

#ifdef DEBUG
   PDEBUGINFO     pDbg = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* EXCEPTION HANDLER                                                      */
   /*------------------------------------------------------------------------*/
   REGISTERHANDLER (regrec, globals.hModule);

   ulException = setjmp (regrec.jmp);

   // If we have encountered an exception
   if (ulException)
   {
      // Free-up any memory we have allocated
      if (pchSplBuffer)
      {
         rc = GplMemoryFree (pchSplBuffer);
         assertT (rc);
      }

      CheckForTermination (&regrec, ulException, pddc);

      // Set appropriate error result for DevEscape() call for application
      lrc = DEVESC_ERROR;
      goto depart;
   }

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/

   // NOTE: Mutex Semaphores are not allowed for DevEscape() processing
   // since we must always process the asynchronous DEVESC_ABORTDOC
   // call even in the middle of another DevEscape call.

   // Retrieve pointer to subclassed functions from our device files (e.g. SAMP_PCL.C)
   if (pddc)
      if (pddc->pdb)
         if (pddc->pdb->pDevice)
            if (pddc->pdb->pDriver)
            {
               // Set up the pointers for easier dereference/readability
               pdb      = pddc->pdb;
               pDriver  = pdb->pDriver;
               pDevice  = pdb->pDevice;
               pfnTbl   = pdb->pDevice->pSubclassedFunctions;
            }

   assertF (pddc);
   assertF (pdb);
   assertF (pDriver);
   assertF (pDevice);
   assertF (pfnTbl);

   // Find out which DevEscpe() escape code was requested/sent from application
   switch (lEscape)
   {
   case DEVESC_STARTDOC:
#ifndef DISABLE_DJP_SUPPORT
   case DEVESC_STARTDOC_WPROP:
#endif
   {
      if (DEVESC_STARTDOC_WPROP == lEscape)
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, ">>>> Entry: DEVESC_STARTDOC_WPROP <<<<\n"));
      else
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, ">>>> Entry: DEVESC_STARTDOC <<<<\n"));

      // new document for this DC
      pdb->fAbortDocCalled = FALSE;

      // ignore multiple DEVESC_STARTDOCs it's not an error
      if (pdb->fStartDocCalled && DEVESC_STARTDOC_WPROP != lEscape)
         break;

      if (DEVESC_STARTDOC_WPROP == lEscape)
// begin @JPN2H99
         pdb->bNewJobProp = pdb->fStartDocCalled ?
                            ReinitializeWithJobProps (pddc, (PDRIVDATA)pOutData) :
                            TRUE;
//       ReinitializeWithJobProps (pddc, (PDRIVDATA)pOutData);
// end @JPN2H99

      CreateAndSetDeviceSurface (pddc);

      if (DEVESC_STARTDOC_WPROP == lEscape)
         GreResetDC2 (pdb->hdc, 0);

      // Call after the ResetDC2
      StartRecording (pddc);
      ErasePage (pddc);

      // ignore multiple DEVESC_STARTDOCs it's not an error
      if (pdb->fStartDocCalled)
      {
         // @157983 - But first reinitialize the device
         if (DEVESC_STARTDOC_WPROP == lEscape)
            AllocateDeviceHandle (pddc, TRUE);
         break;
      }

      // pInData is document name; copy if non null
      assertF (cInCount ? (BOOL)pInData:TRUE);
      SafeStrNCpy (pdb->szDocumentName,
                   pInData ? pInData : "",
                   sizeof (pdb->szDocumentName));

      switch (pddc->pdb->ulType)
      {
      case OD_DIRECT:
      case OD_QUEUED:
      {
         //------------------------------------------------------
         // Query device for it's function table at this time
         // giving device a chance to subclass Spl calls
         //------------------------------------------------------
         AllocateDeviceHandle (pddc, FALSE);

         if (pfnTbl->pfnDeviceQuery)
         {
            BOOL             fGenerated;

            if (pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_RAW_JOB)
               fGenerated = TRUE;
            else
               fGenerated = FALSE;

            pfnTbl->pfnDeviceQuery (pddc,
                                    DEVICE_QUERY_FUNCTIONS,
                                    pdb->pDevFuncs,
                                    pdb->pvDeviceCookie,
                                    pdb->ulType,
                                    pdb->ulDataType,
                                    fGenerated);
         }
         break;
      }
      }

      switch (pdb->ulType)
      {
      case OD_DIRECT:
      {
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "  OD_DIRECT, "));

         // validate data type is not Standard (i.e. metafile)
         if (PM_Q_STD == pdb->ulDataType)
         {
            DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "PM_Q_STD - Impossible!!!\n"));

            // Change data type to force Raw data path
            pdb->ulDataType = PM_Q_RAW;
         }

         if (PM_Q_RAW == pdb->ulDataType)
         {
            DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "PM_Q_RAW\n"));

            // Should we wait for user intervention? (i.e. Manual Paper Feed)
            if (  TRAY_TYPE_MANUAL == pdb->pTrayInfo->ulTrayType
               && (  pdb->ulNupPages != 2                                                  
                  || (pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_RAW_JOB) 
                  )
               )
            {
               SplMessageBox (pdb->szLogAddress,
                              SPLINFO_DDERROR
                              | SPLINFO_INFORMATION,
                              SPLDATA_FORMCHGREQD
                              | SPLDATA_NOAUTORETRY,
                              globals.pbStringTable[STRING_TABLE_DLG_BASE - STRING_TABLE_BASE + DIALOG_INSERT_MANUAL],
                              globals.pbStringTable[STRING_TABLE_DLG_BASE - STRING_TABLE_BASE + DIALOG_PAPER_REQUEST],
                              HWND_DESKTOP,
                              MB_OK);
            }

            // Open output port (Printer driver version of DosOpen)
            rc = pdb->pDevFuncs->pfnDeviceOpen (pddc,
                                                pdb->pvDeviceCookie,
                                                pdb->szLogAddress,
                                                &pdb->hspl,
                                                &ulWork,
                                                0,
                                                FILE_NORMAL,
                                                OPEN_ACTION_REPLACE_IF_EXISTS
                                                | OPEN_ACTION_CREATE_IF_NEW,
                                                OPEN_ACCESS_WRITEONLY
                                                | OPEN_SHARE_DENYREADWRITE,
                                                0);

            assertT (rc);

            // If DEVESC_ABORTDOC was called (user intervention) hspl will be NULL
            assertF (pdb->hspl);
         }

         // If "DeviceOpen" function (i.e. PrtWrite) worked we'll
         // have a file handle to use otherwise we do not continue
         // @159049 - MFR - Moved call to after SplQmOpen() otherwise
         // spooler handle was NULL
         if (pdb->hspl)
         {
            if (OD_DIRECT == pddc->pdb->ulType)
            {
               if (SpoolerEntryPointExists (&globals.pfnPrtNewPage,
                                            "PRTNEWPAGE"))
               {
                  LONG lRet;

                  lRet = globals.pfnPrtNewPage (pddc->pdb->hspl, ++pddc->pdb->ulPage);

                  DBPRINTF (("DEVESC_STARTDOC: PrtNewPage( %lu, %lu ), rc = %d\n", pdb->hspl, pdb->ulPage, lRet));

#ifndef DEBUG
                  lRet++;          // Avoid compiler warnings
#endif
               }
            }

            /*---------------------------------------*/
            /* Create/Update the 2nd print thread    */
            /*---------------------------------------*/
            StartTheThread (pddc, 0, pdb->hspl);

            //
            
            // if Booklet buffer is enabled but isn't initialized,
            // we'll initialize it
            //
            if (  !GplBookletEnabled (pddc->pdb->hThread)
               && !(pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_RAW_JOB)
               )
            {
               if (  (pddc->pdb->ulNupPages == 2)
                  || (  (pddc->pdb->pJobProperties->ulCopies > 1)
                     && !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COPIES)
                     )
                  || ( (pddc->pdb->pJobProperties->ulImageoption & IMAGE_OPTION_COLLATE)
                     && !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COLLATE)
                     )
                  )
               {
                  GplBookletInit (pddc->pdb->hThread,
                                  globals.hModule,
                                  pddc->pdb->pszRangeOfPages,
                                  pddc->pdb->szDocumentName,
                                  (((pddc->pdb->ulNupPages == 2) && !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)) ? BOOKLET_NUP : 0)  
                                  | (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED ? BOOKLET_BOOKLET : 0)                                 
                                  | ((pddc->pdb->pJobProperties->ulImageoption & IMAGE_OPTION_COLLATE) ? 0 : BOOKLET_UNCOLLATE)      
                                  | (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT ? 0 : BOOKLET_LANDSCAPE)
                                  | (pddc->pdb->pJobProperties->ulDuplexOptions == DUPLEX_NONE ? 0 : BOOKLET_DUPLEX)
                                  | ((pddc->pdb->bFlags & OMNI_BOOKLET_UPPERTRAY) ? BOOKLET_UPPERTRAY : 0)
                                  | BOOKLET_OMNI,
                                  (  !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COPIES)                                       //@HWCOLLATE
                                  || ( (pddc->pdb->pJobProperties->ulImageoption & IMAGE_OPTION_COLLATE)                             //@HWCOLLATE
                                     && !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COLLATE)                                   //@HWCOLLATE
                                     )                                                                                               //@HWCOLLATE
                                  ) ? pddc->pdb->pJobProperties->ulCopies : 1 );                                                     //@HWCOLLATE

                  GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_SETUP);      
               }
            }
            

            /*---------------------------------------*/
            /* See if PrtWrite is subclassed         */
            /*---------------------------------------*/
            if (pdb->pDevFuncs->pfnDeviceWrite)
            {
               assertF (pdb->hThread);
               GplThreadHookDeviceWrites (pdb->hThread,
                                          (PFN)pdb->pDevFuncs->pfnDeviceWrite,
                                          pdb->pvDeviceCookie);
            }

            /*---------------------------------------*/
            /* See if InitJob is subclassed          */
            /*---------------------------------------*/
            if (!pdb->fRawOnly && pfnTbl->pfnInitJob)
            {
               
               if (OmniBookletEnabled (pddc))     
               {
                  //
                  // We change small page's Form, Tray and Media info from
                  // large page's data.
                  //
                  SWAPP (pddc->pdb->pLargeFormInfo, pddc->pdb->pFormInfo);

                  pfnTbl->pfnInitJob (pddc, pdb->pvDeviceCookie);

                  SWAPP (pddc->pdb->pLargeFormInfo, pddc->pdb->pFormInfo);
               }
               else
               {
                  pfnTbl->pfnInitJob (pddc, pdb->pvDeviceCookie);
               }
               

               pddc->StateData.fCalledDeviceInitJob = TRUE;

               CreateAndSetDeviceSurface (pddc); // Can be destructive, so we need an erase
               ErasePage (pddc);
            }
// Begin @JPN2H99
            else if ( !(pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_RAW_JOB)
                    && pfnTbl->pfnPreRawData
                    )
            {
               pfnTbl->pfnPreRawData (pddc, pdb->pvDeviceCookie);
            }
// end @JPN2H99
         }

         break;
      } /* end case - OD_DIRECT */

      case OD_QUEUED:
      {
         PJOBPROPERTIES   pJobProperties;

         /* Alter our job properties so that on the second pass of printing,
         ** will will not add an extra form feed character.
         */
         pJobProperties = pdb->pJobProperties;
         pJobProperties->ulJobPhaseControl |= JPC_OMNI_GENERATED_JOB;
                  DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "  OD_QUEUED, "));

         // Perform different queue operations based on data type
         switch (pdb->ulDataType)
         {
         case PM_Q_STD:
         {
            DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "PM_Q_STD\n"));

            // starts the recording of GPI calls into a metafile
            fOK = SplStdStart (pdb->hdc);
            assert (fOK);

            // this creates a queued print job it need only be
            // done once per DC instance and closed in enable.c
            // at BEGIN_CLOSE_DC time.
            if (!pdb->hspl)
            {
               // Call function to perform SplQmOpen
               // this call can be subclassed in device files
               pdb->hspl = pdb->pDevFuncs->pfnSpoolerOpen (pddc,
                                                           pdb->pvDeviceCookie,
                                                           "*",
                                                           9,
                                                           (PQMOPENDATA)&pdb->dop);

               // Call function to perform SplQmStart
               // this call can be subclassed in device files
               pdb->pDevFuncs->pfnSpoolerStartJob (pddc,
                                                   pdb->pvDeviceCookie,
                                                   pdb->hspl,
                                                   pdb->szDocumentName);
            }
            break;
         }

         case PM_Q_RAW:
         {
            DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "PM_Q_RAW\n"));

            // We have spooled this job into a printer-dependant datastream
            pJobProperties->ulJobPhaseControl |= JPC_OMNI_GENERATED_RAW_JOB;

            // this creates a queued print job it need only be
            // done once per DC instance and closed in enable.c
            // at BEGIN_CLOSE_DC time.
            if (!pdb->hspl)
            {
               // Call function to perform SplQmOpen/StartDoc
               // this call can be subclassed in device files
               pdb->hspl = pdb->pDevFuncs->pfnSpoolerOpen (pddc,
                                                           pdb->pvDeviceCookie,
                                                           "*",
                                                           9,
                                                           (PQMOPENDATA)&pdb->dop);

               // Call function to perform SplQmOpen/StartDoc
               // this call can be subclassed in device files
               pdb->pDevFuncs->pfnSpoolerStartJob (pddc,
                                                   pdb->pvDeviceCookie,
                                                   pdb->hspl,
                                                   pdb->szDocumentName);
            }

            assertF (pdb->hspl);

            /*---------------------------------------*/
            /* Create/Update the 2nd print thread    */
            /*---------------------------------------*/
            StartTheThread (pddc, pdb->hspl, 0);

            
            //
            // if Booklet buffer is enabled but isn't initialized,
            // we'll initialize it
            //
            if (!GplBookletEnabled (pddc->pdb->hThread))
            {
               if ( (pddc->pdb->ulNupPages == 2)
                  || (  (pddc->pdb->pJobProperties->ulCopies > 1)
                     && !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COPIES)
                     )
                  || (  (pddc->pdb->pJobProperties->ulImageoption & IMAGE_OPTION_COLLATE)
                     && !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COLLATE)
                     )
                  )
               {
                  GplBookletInit (pddc->pdb->hThread,
                                  globals.hModule,
                                  pddc->pdb->pszRangeOfPages,
                                  pddc->pdb->szDocumentName,
                                  (((pddc->pdb->ulNupPages == 2) && !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)) ? BOOKLET_NUP : 0)  
                                  | (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED ? BOOKLET_BOOKLET : 0)                                 
                                  | ((pddc->pdb->pJobProperties->ulImageoption & IMAGE_OPTION_COLLATE) ? 0 : BOOKLET_UNCOLLATE)      
                                  | (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT ? 0 : BOOKLET_LANDSCAPE)
                                  | (pddc->pdb->pJobProperties->ulDuplexOptions == DUPLEX_NONE ? 0 : BOOKLET_DUPLEX)
                                  | ((pddc->pdb->bFlags & OMNI_BOOKLET_UPPERTRAY) ? BOOKLET_UPPERTRAY : 0)
                                  | BOOKLET_OMNI,
                                  (  !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COPIES)                                       //@HWCOLLATE
                                  || (  (pddc->pdb->pJobProperties->ulImageoption & IMAGE_OPTION_COLLATE)                            //@HWCOLLATE
                                     && !(pdb->pDevice->ulOmniDeviceTechnology & OMNI_CAP_COLLATE)                                   //@HWCOLLATE
                                     )                                                                                               //@HWCOLLATE
                                  ) ? pddc->pdb->pJobProperties->ulCopies : 1 );                                                     //@HWCOLLATE

                  GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_SETUP);      
               }
            }
            

            if (pfnTbl->pfnInitJob)
            {
               
               if (OmniBookletEnabled (pddc))  
               {
                  //
                  // We change small page's Form, Tray and Media info from
                  // large page's data.
                  //
                  SWAPP (pddc->pdb->pLargeFormInfo, pddc->pdb->pFormInfo);

                  pfnTbl->pfnInitJob (pddc, pdb->pvDeviceCookie);

                  SWAPP (pddc->pdb->pLargeFormInfo, pddc->pdb->pFormInfo);
               }
               else
               {
                  pfnTbl->pfnInitJob (pddc, pdb->pvDeviceCookie);
               }
               

               pddc->StateData.fCalledDeviceInitJob = TRUE;

               CreateAndSetDeviceSurface (pddc); // Can be destructive so we need an erase
               ErasePage (pddc);
            }
            break;
         }
         }
         break;
      } /* end case - OD_QUEUED */

      case OD_MEMORY:
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "  OD_MEMORY case\n"));
         break;

      case OD_METAFILE:
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "  OD_METAFILE case\n"));
         break;

      case OD_INFO:
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "  OD_INFO case\n"));
         break;

      default:
         DBPRINTIF ((pDbg->bDEVESC_STARTDOC, "  OD_???? unknown case\n"));
         break;
      }

      // this app clearly indicated start of document;
      // other apps may not be so tidy
      pdb->fStartDocCalled = TRUE;

      // Success!
      lrc = DEV_OK;

      DBPRINTIF ((pDbg->bDEVESC_STARTDOC, ">>>> Exit: DEVESC_STARTDOC <<<<\n"));

      break;
   } /* end case - DEVESC_STARTDOC */

   case DEVESC_SEP:
   case DEVESC_RAWDATA:
   {
      DBPRINTIF ((pDbg->bDEVESC_RAWDATA, "DEVESC_RAWDATA"));

      if (!pdb->fStartDocCalled)
      {
         // error to do DEVESC_RAWDATA before DEVESC_STARTDOC
         RAISEEXCEPTION (XCPT_USER);
      }

      // Indicate that Raw Data escape added data to the
      // current page so we do not later play journal file
      pdb->fRawDataOnPage = TRUE;

      // Is this the second phase of spooling?
      if (!(pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_ALREADY_SPOOLED))
      {
         // No.  Remove the "our job" marker since its not ours anymore.
         pdb->pJobProperties->ulJobPhaseControl &= ~JPC_OMNI_GENERATED_JOB;
      }

      switch (pdb->ulType)
      {
      case OD_DIRECT:
      {
         DBPRINTIF ((pDbg->bDEVESC_RAWDATA, "  OD_DIRECT case "));

         switch (pdb->ulDataType)
         {
         case PM_Q_RAW:
         {
            // check hspl handle or fAbortDocCalled when deciding
            // to call PrtWrite
            if (pdb->hspl)
            {
               if (!(pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_JOB))
               {
                  /* For print jobs that were not generated by us, we
                  ** must check to see if there is a page eject command
                  ** at the end of every raw data buffer sent to us.
                  */
                  if (FormFeedSeen (pddc, pInData, cInCount))
                     pdb->pJobProperties->ulJobPhaseControl |= JPC_FORM_FEED_SEEN;
                  else
                     pdb->pJobProperties->ulJobPhaseControl &= ~JPC_FORM_FEED_SEEN;
               }

               assertF (pdb->hThread);
               rc = GplThreadOutput (pdb->hThread,
                                     pInData,
                                     cInCount,
                                     THREAD_DT_BINARY);
               assertF (rc);

               DBPRINTIF ((pDbg->bDEVESC_RAWDATA, "data = %x, size = %d\n", pInData, cInCount));
            }
            else
            {
               // must have called abortdoc if hspl is zero
               assert (pdb->fAbortDocCalled);
            }

            break;
         }

         case PM_Q_STD:
            // impossible case: this situation subjected to override in enable.c
            assertstring ("impossible case: this situation subjected to override in enable.c\n");
            break;
         }

         break;
      }

      case OD_QUEUED:
      {
         DBPRINTIF ((pDbg->bDEVESC_RAWDATA, "  OD_QUEUED case "));

         switch (pdb->ulDataType)
         {
         case PM_Q_RAW:
         {
            assert (pdb->hspl);
            assert (cInCount);
            assert (pInData);

            if (!pInData || !cInCount)
               // No data!
               break;

            assertF (pdb->hThread);
            rc = GplThreadOutput (pdb->hThread,
                                  pInData,
                                  cInCount,
                                  THREAD_DT_BINARY);
            assertF (rc);

            DBPRINTIF ((pDbg->bDEVESC_RAWDATA, "data = %x, size = %d\n", pInData, cInCount));

            /* For print jobs that were not generated by us, we
            ** must check to see if there is a page eject command
            ** at the end of every raw data buffer sent to us.
            */
            if (FormFeedSeen (pddc, pInData, cInCount))
               pdb->pJobProperties->ulJobPhaseControl |= JPC_FORM_FEED_SEEN;
            else
               pdb->pJobProperties->ulJobPhaseControl &= ~JPC_FORM_FEED_SEEN;
            break;
         }

         case PM_Q_STD:
            break;

         default:
            assertstring ("unhandled pdb->ulDataType!\n");
            break;
         }

         break;
      }

      default:
         assertstring ("unhandled pdb->ulType!\n");
         break;
      }

      // Success!
      lrc = DEV_OK;
      break;
   } /* end case DEVESC_RAWDATA */

   case DEVESC_ENDDOC:
   {
      PFNSUBCLASS ppfnDevice;

      DBPRINTIF ((pDbg->bDEVESC_ENDDOC, ">>>> Entry: DEVESC_ENDDOC <<<<\n"));

      // ignore enddoc if no startdoc; it's not an error
      if (!pdb->fStartDocCalled)
         break;

      pdb->fRecording = FALSE;                                                  

      if (pdb->fAbortDocCalled)
      {
         if (pdb->hThread)
            // Notify 2nd thread that the abort doc condition is over
            GplThreadResetAbortDoc (pdb->hThread);

         if (  !pfnTbl->pfnBandingSupport
            || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                           BANDINGSUPPORT_ABORT,
                                           pddc->hJournal)
            )
         {
            if (pddc->hJournal)
               // Notify journalling that the abort doc condition is over
               GplJournalResetAbortDoc (pddc->hJournal);
         }
      }
      else
      {
         /* Test to see if we need to add a page eject command to the
         ** data stream.
         ** Only for direct print jobs though!
         */
         if (  (  (PM_Q_RAW == pdb->ulDataType)
               && (OD_DIRECT == pdb->ulType)
               )
            && pdb->fRawDataOnPage
            && !(pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_JOB)
            )
         {
            BOOL   bFFControlType;
            BOOL   bFormFeedSeen;

            /* This is a raw data job.
            ** This job was not generated by us.
            ** Check to see if we need to add a page eject command.
            */
            bFFControlType = pdb->pJobProperties->bFFControlType;
            bFormFeedSeen  = pdb->pJobProperties->ulJobPhaseControl & JPC_FORM_FEED_SEEN;
            DBPRINTIF ((pDbg->bDEVESC_ENDDOC, "bFFControlType = %d\n", bFFControlType));
            DBPRINTIF ((pDbg->bDEVESC_ENDDOC, "bFormFeedSeen  = %s\n", bFormFeedSeen ? "TRUE" : "FALSE"));

            /* Should we add a form feed?
            ** FF_CTL_COMPULSORY  means always
            ** FF_CTL_CONDITIONAL means only if one was not seen
            ** FF_CTL_NONE        means never
            */
            if ( (FF_CTL_COMPULSORY  == bFFControlType)     ||
                 ( (FF_CTL_CONDITIONAL == bFFControlType) &&
                   (!bFormFeedSeen)                        ) )
            {
               PCMDINFO   pCmds = pDevice->pCmds;

               assertF (pdb->hThread);

               if (pCmds && pCmds->ulCmdPageEject)
                  rc = GplThreadOutput (pdb->hThread,
                                        pCmds->pszCmdPageEject,
                                        pCmds->ulCmdPageEject,
                                        THREAD_DT_BINARY);
               assertF (rc);
            }
// Begin @JPN2H99
            if ( !(pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_RAW_JOB)
               &&  pfnTbl->pfnPostRawData
               )
            {
               pfnTbl->pfnPostRawData (pddc, pdb->pvDeviceCookie);
            }
// end @JPN2H99
         }

         // NOTE: @159049 - MFR - Removed PrtNewPage call from this section
         // since for "Queued" jobs we show # pages spooled (past tense)
         // so we do need call here, but for "Direct" jobs we show the
         // current page being sent (present tense) so we instead make
         // additional call to increment page during STARTDOC not ENDDOC
         if (  (OD_QUEUED == pdb->ulType)
            && (PM_Q_RAW  == pdb->ulDataType)
            )
         {
            DBPRINTF (("DEVESC_ENDDOC: IncrementSpoolerPageNum()\n"));

            IncrementSpoolerPageNum (pddc);
         }

         switch (pdb->ulType)
         {
         case OD_DIRECT:
         {
            DBPRINTIF ((pDbg->bDEVESC_ENDDOC, "  OD_DIRECT case\n"));

            if (!pdb->fRawDataOnPage)
            {
               
               if (OmniBookletEnabled (pddc))    
               {
                  pddc->ptlPrintHead.y = pddc->pdb->pLargeFormInfo->hcInfo.yPels - 1;
                  pddc->lHeaderY = (pddc->ptlPrintHead.y + 1) / 2 - 1;

                  if (  (  (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                        && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT)
                        && !(pddc->ulBookletPage % 2)
                        )
                     || (  (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                        && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_LANDSCAPE)
                        && (pddc->ulBookletPage % 2)
                        )
                     || (  !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                        && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_LANDSCAPE)
                        && !(pddc->ulBookletPage % 2)
                        )
                     || (  !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                        && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT)
                        && (pddc->ulBookletPage % 2)
                        )
                     )
                  {
                     pddc->ptlPrintHead.y = pddc->lHeaderY;
                  }
               }
               

               if (pddc->StateData.fBanding)
               {
                  if (  !pfnTbl->pfnBandingSupport
                     || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                                    BANDINGSUPPORT_PLAY,
                                                    pddc->hJournal)
                     )
                  {
                     if (pddc->hJournal)
                     {
                        fOK = GplJournalPlayInstance (pddc->hJournal,
                                                      PLAY_JOURNAL_BAND
                                                      | PLAY_DESTRUCTIVE
                                                      | ((pDevice->ulRasterInfo == RASTER_TOP_TO_BOTTOM)
                                                            ? PLAY_TOP_TO_BOTTOM
                                                            : PLAY_BOTTOM_TO_TOP));
                        assertF (fOK);
                     }
                  }
               }
               else
               {
                  RECTL  rctlPage;

                  rctlPage.xLeft   = 0;
                  rctlPage.yTop    = pdb->pFormInfo->hcInfo.yPels-1;
                  rctlPage.xRight  = pdb->pFormInfo->hcInfo.xPels-1;
                  rctlPage.yBottom = 0;
                  fOK = NewFrame (pddc, &rctlPage);
               }

               assert (fOK);

               GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_ENDDOC);     

            }

            // Call the subclassed term function if we have called the init.
            if (  pddc->StateData.fCalledDeviceInitJob
               && pfnTbl->pfnTermJob
               )
            {
                pfnTbl->pfnTermJob (pddc, pdb->pvDeviceCookie);
            }

            if (pddc->StateData.fBanding)
            {
               if (  !pfnTbl->pfnBandingSupport
                  || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                                 BANDINGSUPPORT_DELETE,
                                                 pddc->hJournal)
                  )
               {
                  // If we have a journal file clean it up
                  // We may not if DEVESC_ABORTDOC called.
                  if (pddc->hJournal)
                  {
                     rc = GplJournalDeleteInstance (pddc->hJournal);

                     // Driver must free memory!
                     rc = GplMemoryFree (pddc->hJournal);
                     assertT (rc);

                     pddc->hJournal = NULL;
                  }
               }
            }

            if (pdb->hThread)
            {
               // Flush the 2nd thread
               rc = GplThreadFlushBuffer (pdb->hThread, TRUE);
               assertF (rc);

               if (FALSE == rc)
                  lrc = DEVESC_ERROR;
            }

            GplBookletReplayOutput (pddc->pdb->hThread); 

            if (pcOutCount)
            {
               // Pat Davis said "If you don't return anything set the count to 0."
               *pcOutCount = 0;
            }

            break;
         } /* end case - OD_DIRECT */

         case OD_QUEUED:
         {
            DBPRINTIF ((pDbg->bDEVESC_ENDDOC, "  OD_QUEUED, "));

            switch (pdb->ulDataType)
            {
            case PM_Q_RAW:
            {
               PJOBPROPERTIES   pJobProperties;

               DBPRINTIF(( pDbg->bDEVESC_ENDDOC, "PM_Q_RAW\n" ));

               if (!pdb->fRawDataOnPage)
               {
                  
                  if (OmniBookletEnabled (pddc))    
                  {
                     pddc->ptlPrintHead.y = pddc->pdb->pLargeFormInfo->hcInfo.yPels - 1;
                     pddc->lHeaderY = (pddc->ptlPrintHead.y + 1) / 2 - 1;

                     if (  (  (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                           && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT)
                           && !(pddc->ulBookletPage % 2)
                           )
                        || (  (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                           && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_LANDSCAPE)
                           && (pddc->ulBookletPage % 2)
                           )
                        || (  !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                           && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_LANDSCAPE)
                           && !(pddc->ulBookletPage % 2)
                           )
                        || (  !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                           && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT)
                           && (pddc->ulBookletPage % 2)
                           )
                        )
                     {
                        pddc->ptlPrintHead.y = pddc->lHeaderY;
                     }
                  }
                  

                  if (pddc->StateData.fBanding)
                  {
                     if (  !pfnTbl->pfnBandingSupport
                        || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                                       BANDINGSUPPORT_PLAY,
                                                       pddc->hJournal)
                        )
                     {
                        if (pddc->hJournal)
                        {
                           fOK = GplJournalPlayInstance (pddc->hJournal,
                                                         PLAY_JOURNAL_BAND
                                                         | PLAY_DESTRUCTIVE
                                                         | ((pDevice->ulRasterInfo == RASTER_TOP_TO_BOTTOM)
                                                               ? PLAY_TOP_TO_BOTTOM
                                                               : PLAY_BOTTOM_TO_TOP));
                           assertF (fOK);
                        }
                     }
                  }
                  else
                  {
                     RECTL  rctlPage;

                     rctlPage.xLeft   = 0;
                     rctlPage.yTop    = pdb->pFormInfo->hcInfo.yPels-1;
                     rctlPage.xRight  = pdb->pFormInfo->hcInfo.xPels-1;
                     rctlPage.yBottom = 0;
                     fOK = NewFrame (pddc, &rctlPage);
                  }

                  assert (fOK);

                  GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_ENDDOC);     
               }

               // Call the subclassed term function if we have called the init.
               if (pddc->StateData.fCalledDeviceInitJob &&
                   pfnTbl->pfnTermJob                    )
               {
                  pfnTbl->pfnTermJob (pddc, pdb->pvDeviceCookie);
               }

               if (pddc->StateData.fBanding)
               {
                  if (  !pfnTbl->pfnBandingSupport
                     || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                                    BANDINGSUPPORT_DELETE,
                                                    pddc->hJournal)
                     )
                  {
                     if (pddc->hJournal)
                     {
                        rc = GplJournalDeleteInstance (pddc->hJournal);

                        // Driver must free memory!
                        rc = GplMemoryFree (pddc->hJournal);
                        assertT (rc);

                        pddc->hJournal = NULL;
                     }
                  }
               }

               if (pdb->hThread)
               {
                  // Flush the 2nd thread
                  rc = GplThreadFlushBuffer (pdb->hThread, TRUE);
                  assertF (rc);
                  if (FALSE == rc)
                     lrc = DEVESC_ERROR;
               }

               GplBookletReplayOutput (pddc->pdb->hThread); 

               /* Alter our job properties so that on the second pass
               ** of printing,  will will not add an extra form feed
               ** character.
               */
               pJobProperties = pddc->pdb->pJobProperties;
               pJobProperties->ulJobPhaseControl |= JPC_OMNI_ALREADY_SPOOLED;
               UpdateSpoolJobProperties (pdb->hspl, pddc->pdb->pDrivData);

               // Call function to perform SplQmEndDoc and retrieves Job ID
               // this call can be subclassed in device files
               lWork = pdb->pDevFuncs->pfnSpoolerEndJob (pddc,
                                           pdb->pvDeviceCookie,
                                           pdb->hspl);

               // if there is room at the pOutData pointer, then give the caller the job id (lWork)
               if (pcOutCount && pOutData && sizeof (ULONG) <= *pcOutCount)
               {
                  // there is room
                  *(PULONG)pOutData = lWork;
                  *pcOutCount = sizeof (ULONG);
               }
               else if (pcOutCount && pOutData && sizeof (USHORT) <= *pcOutCount)
               {
                  *(PUSHORT)pOutData = lWork;
                  *pcOutCount = sizeof (USHORT);
               }

               break;
            }

            case PM_Q_STD:
            {
               LONG           lStart;
               LONG           lStdLength;
               LONG           lLength;

               DBPRINTIF ((pDbg->bDEVESC_ENDDOC, "PM_Q_STD\n"));

               // stop recording and get handle to job
               hstd = SplStdStop (pdb->hdc);
               assert (hstd);

               // get length of metafile as a count of bytes
               lStdLength = SplStdQueryLength (hstd);

               // allocate some DDC heap for use as a buffer
#define MAX_LEN_SPLBITBUFFER    0x1000

               lLength = min (MAX_LEN_SPLBITBUFFER, lStdLength);
               pchSplBuffer = (PCHAR)GplMemoryAlloc (pdb->hmcbHeap, lLength);
               assert (pchSplBuffer);

               // loop as necessary
               lStart = 0;
               while (lStart < lStdLength)
               {
                  fOK = SplStdGetBits (hstd, lStart, lLength, pchSplBuffer);
                  assert (fOK);

                  fOK = SplQmWrite (pdb->hspl, lLength, pchSplBuffer);
                  assert (fOK);

                  lStart += lLength;
                  lLength = min (lLength, lStdLength-lStart);
               }

               // delete the spooler metafile data
               fOK = SplStdDelete (hstd);
               assert (fOK);

               // Call function to perform SplQmEndDoc and retrieves Job ID
               // this call can be subclassed in device files
               lWork = pdb->pDevFuncs->pfnSpoolerEndJob (pddc,
                                                         pdb->pvDeviceCookie,
                                                         pdb->hspl);

               // if there is room at the pOutData pointer, then give the caller the job id (lWork)
               if (pcOutCount && pOutData && sizeof (ULONG) <= *pcOutCount)
               {
                  // there is room
                  *(PULONG)pOutData = lWork;
                  *pcOutCount = sizeof (ULONG);
               }
               else if (pcOutCount && pOutData && sizeof (USHORT) <= *pcOutCount)
               {
                  *(PUSHORT)pOutData = lWork;
                  *pcOutCount = sizeof (USHORT);
               }

               fOK = GplMemoryFree (pchSplBuffer);
               assertT (fOK);
               pchSplBuffer = NULL;
               break;
            }
            }
            break;
         }

         case OD_INFO:
            DBPRINTIF ((pDbg->bDEVESC_ENDDOC, "DEVESC_ENDOC: OD_INFO\n"));
            break;

         default:
            assertstring ("DEVESC_ENDOC: unknown type\n");
            break;
         }
      } /* end else (not abortdoc) */

      ppfnDevice = pddc->pdb->pDevFuncs;
      assertF (ppfnDevice);

      // perform job close work based on DC Type
      switch (pddc->pdb->ulType)
      {
      case OD_DIRECT:
      {
        if (pddc->pdb->hspl)
        {
           if (ppfnDevice->pfnDeviceClose)
           {
              ppfnDevice->pfnDeviceClose (pddc->pdb->pvDeviceCookie,
                                          pddc->pdb->hspl);
           }
           else
           {
              rc = PrtClose (pddc->pdb->hspl);
           }

           pddc->pdb->hspl = 0;
        }
        break;
      }

      case OD_QUEUED:
      {
         // close spool buffer for queued PM_Q_STD data
         if (pddc->pdb->ulDataType == PM_Q_STD)
         {
            rc = SplStdClose (pddc->pdb->hdc);
            assertF (rc);
         }

         // if a queued job was ever created for this DC, close spooler handle now
         if (pddc->pdb->hspl)
         {
            // Call function to perform SplQmClose
            // this call can be subclassed in device files
            lWork = ppfnDevice->pfnSpoolerClose (pddc,
                                                 pddc->pdb->pvDeviceCookie,
                                                 pddc->pdb->hspl);

            // reset to zero for assurance
            pddc->pdb->hspl = 0;
         }
         break;
      }

      case OD_INFO:
      case OD_MEMORY:
         break;
      }

      // this doc is over
      // clean up this ddc in case there is another startdoc/enddoc pair down the road
      pdb->fStartDocCalled  = FALSE;
      pdb->fAbortDocCalled  = FALSE;
      pdb->ulPage           = 0;
      memset (pdb->szDocumentName, 0, sizeof (pdb->szDocumentName));

      // End the second thread
      EndTheThread (pddc);

      // Success!
      lrc = DEV_OK;

      DBPRINTIF ((pDbg->bDEVESC_ENDDOC, ">>>> Exit: DEVESC_ENDDOC <<<<\n"));
      break;
   } /* end case DEVESC_ENDDOC */

   case DEVESC_NEWFRAME:
#ifndef DISABLE_DJP_SUPPORT
   case DEVESC_NEWFRAME_WPROP:
#endif
   {
      pdb->fRecording = FALSE;                                                  

      if (DEVESC_NEWFRAME_WPROP == lEscape)
         DBPRINTIF ((pDbg->bDEVESC_NEWFRAME, ">>>> Entry: DEVESC_NEWFRAME_WPROP <<<<\n"));
      else
         DBPRINTIF ((pDbg->bDEVESC_NEWFRAME, ">>>> Entry: DEVESC_NEWFRAME <<<<\n"));

      // call routine to dump shadow DC bits to printer
      if (  (OD_DIRECT == pdb->ulType)
         || (  (OD_QUEUED == pdb->ulType)
            && (PM_Q_RAW  == pdb->ulDataType)
            )
         )
      {
         // Should we wait for user intervention?
         if (  (OD_DIRECT        == pdb->ulType)
            && (TRAY_TYPE_MANUAL == pdb->pTrayInfo->ulTrayType)
            
            && (  pdb->ulNupPages != 2
               || pdb->pJobProperties->ulJobPhaseControl & JPC_OMNI_GENERATED_RAW_JOB
               )
            
            )
         {
            // Flush the 2nd thread
            rc = GplThreadFlushBuffer (pdb->hThread, TRUE);
            assertF (rc);

            SplMessageBox (pdb->szLogAddress,
                           SPLINFO_DDERROR     | SPLINFO_INFORMATION,
                           SPLDATA_FORMCHGREQD | SPLDATA_NOAUTORETRY,
                           globals.pbStringTable[STRING_TABLE_DLG_BASE - STRING_TABLE_BASE + DIALOG_INSERT_MANUAL],
                           globals.pbStringTable[STRING_TABLE_DLG_BASE - STRING_TABLE_BASE + DIALOG_PAPER_REQUEST],
                           HWND_DESKTOP,
                           MB_OK);
         }

         if (!pdb->fRawDataOnPage)
         {
            
            if (OmniBookletEnabled (pddc))    
            {
               pddc->ptlPrintHead.y = pddc->pdb->pLargeFormInfo->hcInfo.yPels - 1;
               pddc->lHeaderY = (pddc->ptlPrintHead.y + 1) / 2 - 1;

               if (  (  (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                     && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT)
                     && !(pddc->ulBookletPage % 2)
                     )
                  || (  (pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                     && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_LANDSCAPE)
                     && (pddc->ulBookletPage % 2)
                     )
                  || (  !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                     && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_LANDSCAPE)
                     && !(pddc->ulBookletPage % 2)
                     )
                  || (  !(pddc->pdb->bFlags & OMNI_BOOKLET_ENABLED)
                     && (pddc->pdb->pJobProperties->ulOrientation == ORIENTATION_PORTRAIT)
                     && (pddc->ulBookletPage % 2)
                     )
                  )
               {
                  pddc->ptlPrintHead.y = pddc->lHeaderY;
               }
            }
            

            if (pddc->StateData.fBanding)
            {
               if (  !pfnTbl->pfnBandingSupport
                  || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                                 BANDINGSUPPORT_PLAY,
                                                 pddc->hJournal)
                  )
               {
                  if (pddc->hJournal)
                  {
                     fOK = GplJournalPlayInstance (pddc->hJournal,
                                                   PLAY_JOURNAL_BAND
                                                   | PLAY_DESTRUCTIVE
                                                   | ((pDevice->ulRasterInfo == RASTER_TOP_TO_BOTTOM)
                                                         ? PLAY_TOP_TO_BOTTOM
                                                         : PLAY_BOTTOM_TO_TOP));
                     assertF (fOK);
                  }
               }
            }
            else
            {
               // called for single band pages
               RECTL  rctlPage;

               rctlPage.xLeft   = 0;
               rctlPage.yBottom = 0;
               rctlPage.xRight  = pdb->pFormInfo->hcInfo.xPels-1;
               rctlPage.yTop    = pdb->pFormInfo->hcInfo.yPels-1;
               fOK = NewFrame (pddc, &rctlPage);
            }

            assert (fOK);

            
            pddc->bBlankPageRasterized = TRUE;

            pddc->ulBookletPage++;
            GplBookletNewPage (pddc->pdb->hThread);

            // Call the subclassed function...
            if (pfnTbl->pfnNextPage)
            {
               if (OmniBookletEnabled (pddc))    
               {
                  GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_EJECT);

                  SWAPP (pddc->pdb->pLargeFormInfo, pddc->pdb->pFormInfo);

                  pddc->ptlPrintHead.x = 0;
                  pddc->ptlPrintHead.y = 0;

                  pfnTbl->pfnNextPage (pddc, pdb->pvDeviceCookie);

                  SWAPP (pddc->pdb->pLargeFormInfo, pddc->pdb->pFormInfo);

                  GplBookletResetOutput (pddc->pdb->hThread);
               }
               else
               {
                  // Call the subclassed function...
                  GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_EJECT);
                  pfnTbl->pfnNextPage (pddc, pdb->pvDeviceCookie);
                  GplBookletResetOutput (pddc->pdb->hThread);
               }
            }
            
         }

         // @159049 - MFR - Moved page increment to after "playing" of
         // journal file since we can't say we've printed a page before
         // we've banded!
         if ((OD_QUEUED == pdb->ulType)    &&
             (PM_Q_RAW  == pdb->ulDataType) )
         {
            DBPRINTF(( "DEVESC_NEWFRAME: IncrementSpoolerPageNum()\n" ));

            IncrementSpoolerPageNum (pddc);
         }
         else if (OD_DIRECT == pdb->ulType)
         {
            if (SpoolerEntryPointExists (&globals.pfnPrtNewPage,
                                         "PRTNEWPAGE"))
            {
               globals.pfnPrtNewPage (pdb->hspl, ++pdb->ulPage );

               DBPRINTF(( "DEVESC_NEWFRAME: PrtNewPage( %lu, %lu )\n", pdb->hspl, pdb->ulPage ));
            }
         }
      }

      // This is our job again...
      pdb->pJobProperties->ulJobPhaseControl |= JPC_OMNI_GENERATED_JOB;

      // @157983, @166185 - Changed order, reset surface THEN device handle
      if (DEVESC_NEWFRAME_WPROP == lEscape)
      {
         // Re initialize the pddc
         // @JPN2H99
//       ReinitializeWithJobProps (pddc, (PDRIVDATA)pOutData);
         pdb->bNewJobProp = ReinitializeWithJobProps (pddc, (PDRIVDATA)pOutData);

         // Create a new surface with the new information
         CreateAndSetDeviceSurface (pddc);  // Can be destructive so we need an erase

         GreResetDC2 (pdb->hdc, 0);
      }

      // Call after the ResetDC2
      StartRecording (pddc);
      ErasePage (pddc);

      if (  (DEVESC_NEWFRAME_WPROP == lEscape)
         && (  (OD_DIRECT == pdb->ulType)
            || (  (OD_QUEUED == pdb->ulType)
               && (PM_Q_RAW  == pdb->ulDataType)
               )
            )
         )
      {
         // if new jobprop has been set, terminate the job  // @JPN2H99
         // and eat the setup data for 2 up print.          // @JPN2H99
         // If not, no necessary to realloct device handle  // @JPN2H99
         // again!                                          // @JPN2H99
         if ( pddc->pdb->bNewJobProp )                      // @JPN2H99
         {
            GplBookletSetOutput (pddc->pdb->hThread, BOOKLET_OUTPUT_IGNORE); 
            AllocateDeviceHandle (pddc, TRUE);
         }
      }

      // Success!
      lrc = DEV_OK;

      DBPRINTIF ((pDbg->bDEVESC_NEWFRAME, ">>>> Exit: DEVESC_NEWFRAME <<<<\n"));

      break;
   } /* end case DEVESC_NEWFRAME */

   case DEVESC_ABORTDOC:
   {
      DBPRINTIF ((pDbg->bDEVESC_ABORTDOC, ">>>> Entry: DEVESC_ABORTDOC <<<<\n"));

      pdb->fStartDocCalled = FALSE;

      // abortdocs can arrive more than once
      if (!pdb->fAbortDocCalled)
      {
         pdb->fAbortDocCalled  = TRUE;

         if (pdb->hThread)
            // Tell second thread that an abort doc occured!
            GplThreadAbortDoc (pdb->hThread);

         if (  !pfnTbl->pfnBandingSupport
            || !pfnTbl->pfnBandingSupport (pdb->pvDeviceCookie,
                                           BANDINGSUPPORT_ABORT,
                                           pddc->hJournal)
            )
         {
            if (pddc->hJournal)
               // Notify journalling that an abort doc occured!
               GplJournalAbortDoc (pddc->hJournal);
         }

         // Do not do a close since T1 is still writing and using the
         // file handle!
         switch (pdb->ulType)
         {
         case OD_QUEUED:
         {
            fOK = SplQmAbortDoc (pdb->hspl);
            assert (fOK);
            break;
         }
         }
      }

      // Success!
      lrc = DEV_OK;

      DBPRINTIF ((pDbg->bDEVESC_ABORTDOC, ">>>> Exit: DEVESC_ABORTDOC <<<<\n"));
      break;
   } /* end case DEVESC_ABORTDOC */

   case DEVESC_SETABORTPROC: //@SAP
   {
      /*-----------------------------------------------
      ** Support for SetAbortProc
      ** Store the address and HDC for later callback
      **-----------------------------------------------
      */
      if (cInCount != sizeof (SETABORTPROC))
      {
         lrc = DEVESC_ERROR;
      }
      else
      {
         pdb->SetAbortProc = *((PSETABORTPROC)pInData);
         lrc = DEV_OK;
      }
      break;
   }

   case DEVESC_QUERYPDL:
   {
      PIQUERYPDL     pIQueryPDL  = (PIQUERYPDL)pInData;
      POQUERYPDL     pOQueryPDL  = (POQUERYPDL)pOutData;
      PPDL           pPDL        = (PPDL)NULL;
      INT            iSizeNeeded;
      INT            iNumPDLs    = 0;

      DBPRINTF ((">>>> Entry: DEVESC_QUERYPDL <<<<\n"));

      assertF (cInCount);
      assertF (pInData);
      assertF (pcOutCount);
      assertF (pOutData);
      if (!cInCount   ||
          !pInData    ||
          !pcOutCount ||
          !pOutData    )
      {
         lrc = DEVESC_ERROR;
         assertT (lrc == DEVESC_ERROR);
         break;
      }

      // Does the device support this query?
      if (!pfnTbl || !pfnTbl->pfnQueryPDL)
      {
         lrc = DEVESC_ERROR;
         assertT (lrc == DEVESC_ERROR);
         break;
      }

      // Ask how many PDLs it supports
      if (!pfnTbl->pfnQueryPDL (QUERYPDL_QUERY_NUMBER,
                                (PPDL)NULL,
                                pDevice->usDeviceID,
                                &iNumPDLs))
      {
         lrc = DEVESC_ERROR;
         assertT (lrc == DEVESC_ERROR);
         break;
      }

      iSizeNeeded = sizeof (OQUERYPDL) + ((iNumPDLs - 1) * sizeof (PDL));

      // What to do?
      switch (pIQueryPDL->lCmd)
      {
      case QUERYPDL_CMD_QUERY_SIZE:
      {
         DBPRINTF (("DEVESC_QUERYPDL: QUERYPDL_CMD_QUERY_SIZE iSizeNeeded = %d, iNumPDLs = %d\n",
                    iSizeNeeded, iNumPDLs));
         pOQueryPDL->lBufSize  = iSizeNeeded;
         pOQueryPDL->lMaxCount = iNumPDLs;
         pOQueryPDL->lCount    = 0;
         lrc = DEV_OK;
         break;
      }

      case QUERYPDL_CMD_QUERY_PDL:
      {
         INT          iLeft;
         LONG         lCount;
         LONG         lStart;
         LONG         lEnd;
         register INT i;

         lrc = DEV_OK;

         iLeft  = *pcOutCount;
         lCount = pIQueryPDL->lCount;
         // NOTE: Spec is 0 based for some reason known only to Lee
         lStart = pIQueryPDL->lStart + 1;
         lEnd   = min (iNumPDLs, (lStart + lCount - 1));

         DBPRINTF (("DEVESC_QUERYPDL: QUERYPDL_CMD_QUERY_PDL size = %d, count = %d, start = %d\n",
                    iLeft, lCount, lStart));

         // Initialize the header
         pOQueryPDL->lBufSize  = sizeof (OQUERYPDL) - sizeof (PDL);
         pOQueryPDL->lMaxCount = iNumPDLs;
         pOQueryPDL->lCount    = 0;

         iLeft -= sizeof (OQUERYPDL) - sizeof (PDL);
         pPDL   = pOQueryPDL->aPDL;
         for (i = lStart;
              (i <= lEnd) && (sizeof (PDL) <= iLeft);
              i++)
         {
            memset (pPDL, 0, sizeof (PDL));
            pfnTbl->pfnQueryPDL (QUERYPDL_QUERY_PDL,
                                 pPDL,
                                 pDevice->usDeviceID,
                                 i);

            pOQueryPDL->lCount++;
            pOQueryPDL->lBufSize += sizeof (PDL);

            pPDL++;
            iLeft -= sizeof (PDL);
         }
         break;
      }

      default:
      {
         assertstring ("Unknown cmd!\n");
         lrc = DEVESC_ERROR;
         break;
      }
      }

      DBPRINTF ((">>>> Exit: DEVESC_QUERYPDL <<<<\n"));
      break;
   }

   case DEVESC_MACRO:
   {
      PESCMACRO      pMacroEsc = (PESCMACRO)pInData;
      BOOL           fFound    = FALSE;

      DBPRINTF ((">>>> Entry: DEVESC_MACRO <<<<\n"));

      assertF (pMacroEsc);
      if (!pMacroEsc)
      {
         lrc = DEVESC_ERROR;
         break;
      }

      /* check pMacroEsc->sPDL to make sure that it
      ** is compatible to the current value that would be returned
      ** by DEVESC_QUERYPRINTERLANGUAGE.
      */
      if (pfnTbl && pfnTbl->pfnQueryPDL)
      {
         INT            iNumPDLs = 0;
         register INT   i;

         // Ask how many PDLs it supports
         if (!pfnTbl->pfnQueryPDL (QUERYPDL_QUERY_NUMBER,
                                   (PPDL)NULL,
                                   pDevice->usDeviceID,
                                   &iNumPDLs))
         {
            lrc = DEVESC_ERROR;
            assertT (lrc == DEVESC_ERROR);
            break;
         }

         for (i = 1; i <= iNumPDLs; i++)
         {
            fFound = pfnTbl->pfnQueryPDL (QUERYPDL_ISOK_PDL,
                                          &pMacroEsc->sPDL,
                                          pDevice->usDeviceID,
                                          i);
            assertF (fFound);
            DBPRINTF (("Querying PDL #%d, returns %s\n", i, (fFound ? "TRUE" : "FALSE")));

            if (fFound)
               break;
         }
      }
      else
         // Let it fall through
         fFound = TRUE;

      if ((OD_DIRECT == pdb->ulType)        ||
          ( (OD_QUEUED == pdb->ulType)    &&
            (PM_Q_RAW  == pdb->ulDataType) ) )
      {
         assertF (pdb->hThread);

         if (fFound)
         {
            rc = GplThreadOutput (pdb->hThread,
                                  pMacroEsc->abMacro,
                                  pMacroEsc->ulMacroLength,
                                  THREAD_DT_BINARY);
            assertF (rc);
         }
#ifdef DEBUG
         else
         {
            DBPRINTF (("DEVESC_MACRO: Not a valid macro for this device!\n"));
         }
#endif
      }

      lrc = DEV_OK;
      DBPRINTF ((">>>> Exit: DEVESC_MACRO <<<<\n"));
      break;
   }

   case DEVESC_QUERYHALFTONESUPPORT:
   {
      /*-------------------------------------------------------------*/
      /* WORKAROUND FOR PAGEMAKER 3.01RC4 and below  - MFR           */
      /*                                                             */
      /* Pagemaker supplies a pointer to an uninitialised long       */
      /* in the plOutData parameter. This workaround corrects        */
      /* this problem using the assumption that the pcOutCount       */
      /* field should be four.                                       */
      /*-------------------------------------------------------------*/

      DBPRINTIF ((pDbg->bDEVESC_QUERYHALFTONESUPPORT, ">>>> Entry: DEVESC_QUERYHALFTONESUPPORT <<<<\n"));

      // return success unless we cannot pOutData is not large enough to hold
      // data we need to return - MFR
      lrc = DEV_OK;

      if (*pOutData == 4L)
      {
         *pOutData = HALFTONES_FOR_AREAS | HALFTONES_FOR_BITBLT;
      }
      else
      {
         if ((ULONG)*pOutData > (ULONG)4L)
         {
            DBPRINTIF(( pDbg->bDEVESC_QUERYHALFTONESUPPORT, "DEVESC_QUERYHALFTONESUPPORT: *pOutData = %d and should be = 4\n", *pOutData ));
            *pcOutCount = 4L;
            *pOutData = HALFTONES_FOR_AREAS | HALFTONES_FOR_BITBLT;
         }
         else
         {
            if ((ULONG)*pOutData == (ULONG)2L)
            {
               DBPRINTIF ((pDbg->bDEVESC_QUERYHALFTONESUPPORT, "DEVESC_QUERYHALFTONESUPPORT: *pOutData = %d and should be = 4\n", *pOutData));
               *pOutData = LOUSHORT (HALFTONES_FOR_AREAS | HALFTONES_FOR_BITBLT);
            }
            else
            {
               DBPRINTIF ((pDbg->bDEVESC_QUERYHALFTONESUPPORT, "DEVESC_QUERYHALFTONESUPPORT: *pOutData = %d and should be = 4\n", *pOutData));
               lrc = DEVESC_ERROR;
            }
         }
      }

      DBPRINTIF ((pDbg->bDEVESC_QUERYHALFTONESUPPORT, ">>>> Exit: DEVESC_QUERYHALFTONESUPPORT <<<<\n"));
      break;
   }

   case DEVESC_QUERYPATTERNLIMITS:
   {
      PATTERN_LIMITS PatLim;

      DBPRINTIF(( pDbg->bDEVESC_QUERYPATTERN, ">>>> Entry: DEVESC_QUERYPATTERNLIMITS <<<<\n" ));

      /*-------------------------------------------------------------*/
      /* @TBD - needs to be dynamic based on resolution of device    */
      /*-------------------------------------------------------------*/
      PatLim.usHorzLimit = 8;
      PatLim.usVertLimit = 8;

      *pcOutCount = sizeof(PatLim);
      memcpy (pOutData, &PatLim, *pcOutCount);

      lrc = DEV_OK;

      DBPRINTIF ((pDbg->bDEVESC_QUERYPATTERN, ">>>> Exit: DEVESC_QUERYPATTERNLIMITS <<<<\n"));
      break;
   }

   case DEVESC_QUERYSIZE:
   case DEVESC_QUERYJOBPROPERTIES:
   case DEVESC_SETJOBPROPERTIES:
   case DEVESC_DEFAULTJOBPROPERTIES:
   {
      ULONG              ulCode;
      PDRIVDATA          pDrivData;
      PDRIVDATA          pGoodDrivData = (PDRIVDATA)NULL;
      DLGINSTANCE        di;
      PDLGINSTANCE       pdi           = &di;

      // This api is not supported on pre-DAX engines
#ifndef DISABLE_DJP_SUPPORT
      if (GRE_234 > globals.ulGreVersion)
#endif
      {
         assertstring ("Not supported on pre-DAX engine!\n");
         lrc = DEVESC_NOTIMPLEMENTED;
         break;
      }

      lrc = DEV_OK;

      /* Take the input job properties and make sure the size is set
      ** correctly.
      */
      pDrivData  = (PDRIVDATA)pOutData;
      pDrivData->cb = min (*pcOutCount, pDrivData->cb);

      memset (pdi, 0, sizeof (*pdi));
      GetCountryInfo (&pdi->ctryCode, &pdi->ctryInfo);

      pGoodDrivData = ReturnDriverData (pdb,
                                        pdi,
                                        pdb->hmcbHeap,
                                        pDrivData,
                                        pdb->pszPrinterName,
                                        TRUE,
                                        pDevice,
                                        pDriver);
      assertF (pGoodDrivData);

      /* Is the size of the input driver data is not the same as
      ** the size of our good data?
      */
      if (pDrivData->cb != pGoodDrivData->cb)
         lrc = DEV_PROP_BUF_TOO_SMALL;

      /* Or, if the two don't compare, then fail the call.
      */
      if (0 != memcmp (pDrivData, pGoodDrivData, pGoodDrivData->cb))
         lrc = DEV_INV_INP_JOBPROPERTIES;

      if (DEVESC_QUERYSIZE == lEscape)
         ulCode = DEVPE_QUERYSIZE;
      else if (DEVESC_QUERYJOBPROPERTIES == lEscape)
         ulCode = DEVPE_QUERYJOBPROPERTIES;
      else if (DEVESC_SETJOBPROPERTIES == lEscape)
         ulCode = DEVPE_SETJOBPROPERTIES;
      else if (DEVESC_DEFAULTJOBPROPERTIES == lEscape)
         ulCode = DEVPE_DEFAULTJOBPROPERTIES;

      if (DEV_OK == lrc)
         lrc = InnerDeviceQuery (ulCode,
                                 pdb,
                                 pGoodDrivData,
                                 cInCount,
                                 pInData,
                                 *pcOutCount,
                                 pOutData);

      // Clean up
      if (pGoodDrivData)
      {
         rc = GplMemoryFree (pGoodDrivData);
         assertT (rc);
      }
      break;
   }

   case DEVESC_GETJOBID:
   {
      lrc = DEV_ERROR;

      if (!SpoolerEntryPointExists (&globals.pfnSplQmGetJobID,
                                    "SPLQMGETJOBID"))
         // We need the spooler API!
         break;

      if (!pcOutCount)
         // We need this pointer!
         break;

      /* There should be two cases:
      **    1) INPUT:  pcOutCount points to a LONG
      **               pOutData is NULL.
      **       OUTPUT: pcOutCount contains the size needed
      **
      **    2) INPUT:  pcOutCount points to a LONG that has the size allocated
      **               pOutData points to the block of memory
      **       OUTPUT: pOutData is filled in.
      */
      rc = globals.pfnSplQmGetJobID (pdb->hspl,
                                     0,
                                     pOutData,
                                     *pcOutCount,
                                     pcOutCount);

      if (NO_ERROR == rc || ERROR_MORE_DATA == rc)
         lrc = DEV_OK;
      else
      {
         assertF (NO_ERROR == rc || ERROR_MORE_DATA == rc);

         lrc = DEV_ERROR;
      }
      break;
   }

   case DEVESC_QUERYESCSUPPORT:
   {
      LONG  lCmd;

      DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, ">>>> Entry: DEVESC_QUERYESCSUPPORT <<<<\n"));

      assert (pInData);

      // NOTE: PageMaker gives us a cInCount of 2 not 4 as we expect - MFR
      if (cInCount != sizeof (LONG))
      {
         DBPRINTIF ((cInCount != sizeof (LONG), "cInCount != sizeof(LONG): cInCount = %d\n", cInCount));
         lCmd = (LONG)LOUSHORT (*(PLONG)pInData);
      }
      else
      {
         lCmd = *(PLONG)pInData;
      }

      switch (lCmd)
      {
      case DEVESC_QUERYESCSUPPORT:        //      0L 0000x
      case DEVESC_SETABORTPROC:           //   8001L 1F41x                    @SAP
      case DEVESC_QUERYPDL:               //   8002L 1F42x
      case DEVESC_STARTDOC:               //   8150L 1FD6x
      case DEVESC_ENDDOC:                 //   8151L 1FD7x
      case DEVESC_ABORTDOC:               //   8153L 1FD9x
      case DEVESC_NEWFRAME:               //  16300L 3FACx
      case DEVESC_RAWDATA:                //  16303L 3FAFx
      case DEVESC_MACRO:                  //  16307L 3FB3x
      case DEVESC_QUERYPATTERNLIMITS:     //      4L 0004x
      case DEVESC_QUERYHALFTONESUPPORT:   //      5L 0005x
      {
         DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, "DEVESC_QUERYESCSUPPORT: Escape Supported = '%lu'\n", lCmd));
         lrc = DEV_OK;
         break;
      }

      case DEVESC_STARTDOC_WPROP:         //  57342L DFFEx
      case DEVESC_NEWFRAME_WPROP:         //  57343L DFFFx
      case DEVESC_QUERYSIZE:              //   8162L 1FE2x
      case DEVESC_QUERYJOBPROPERTIES:     //   8163L 1FE3x
      case DEVESC_SETJOBPROPERTIES:       //   8164L 1FE4x
      case DEVESC_DEFAULTJOBPROPERTIES:   //   8165L 1FE5x
      {
         DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, "DEVESC_QUERYESCSUPPORT: Escape Supported = '%lu'\n", lCmd));

         // This api is not supported on pre-DAX engines
#ifndef DISABLE_DJP_SUPPORT
         if (GRE_234 > globals.ulGreVersion)
#endif
         {
            assertstring ("Not supported on pre-DAX engine!\n");
            lrc = DEVESC_NOTIMPLEMENTED;
            break;
         }

         lrc = DEV_OK;
         break;
      }

      case DEVESC_GETSCALINGFACTOR:       //      1L
      case DEVESC_QUERYVIOCELLSIZES:      //      2L
      case DEVESC_QUERYACTUALRESOLUTION:  //      3L
      // case DEVESC_QUERYPATTERNLIMITS:  //      4L
      // case DEVESC_QUERYHALFTONESUPPORT://      5L
      case DEVESC_GETCP:                  //   8000L
      case DEVESC_NEXTBAND:               //   8152L
      case DEVESC_DRAFTMODE:              //  16301L
      case DEVESC_FLUSHOUTPUT:            //  16302L
      case DEVESC_CHAR_EXTRA:             //  16998L
      case DEVESC_BREAK_EXTRA:            //  16999L
      case DEVESC_SETMODE:                //  16304L
      // Note that the definition  DEVESC_DBE_FIRST and                       //@DEVFNT-IBMJ
      // DEVESC_DBE_LAST are for reservation of escape                        //@DEVFNT-IBMJ
      // range, not escape code.                                              //@DEVFNT-IBMJ
      //case DEVESC_DBE_FIRST:            //  24450L                          //@DEVFNT-IBMJ
      //case DEVESC_DBE_LAST:             //  24455L                          //@DEVFNT-IBMJ
      case DEVESC_STD_JOURNAL:            //  32600L
      {
         DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, "DEVESC_QUERYESCSUPPORT: Escape Not Implemented = '%lu'\n", lCmd));
         lrc = DEVESC_NOTIMPLEMENTED;
         break;
      }

      default:
      {
         lrc = DEVESC_NOTIMPLEMENTED;
         DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, "DEVESC_QUERYESCSUPPORT: Unknown Escape = '%lu'\n", lCmd));
         break;
      }
      }

      DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, ">>>> Exit: DEVESC_QUERYESCSUPPORT <<<<\n"));
      break;
   } /* end case DEVESC_QUERYESCSUPPORT */

   // devescapes not implemented
   case DEVESC_GETSCALINGFACTOR:       //      1L
   case DEVESC_QUERYVIOCELLSIZES:      //      2L
   case DEVESC_QUERYACTUALRESOLUTION:  //      3L
   case DEVESC_GETCP:                  //   8000L
   case DEVESC_NEXTBAND:               //   8152L
   case DEVESC_DRAFTMODE:              //  16301L
   case DEVESC_FLUSHOUTPUT:            //  16302L
   case DEVESC_CHAR_EXTRA:             //  16998L
   case DEVESC_BREAK_EXTRA:            //  16999L
   case DEVESC_SETMODE:                //  16304L
   // Note that the definition  DEVESC_DBE_FIRST and                           //@DEVFNT-IBMJ
   // DEVESC_DBE_LAST are for reservation of escape                            //@DEVFNT-IBMJ
   // range, not escape code.                                                  //@DEVFNT-IBMJ
   //case DEVESC_DBE_FIRST:              //  24450L                            //@DEVFNT-IBMJ
   //case DEVESC_DBE_LAST:               //  24455L                            //@DEVFNT-IBMJ
   case DEVESC_STD_JOURNAL:              //  32600L
   {
      DBPRINTIF ((pDbg->bDEVESC_QUERYESCSUPPORT, "DevEscape = %lu; returning NOTIMPLEMENTED!!!\n",
                  lEscape));
      lrc = DEVESC_NOTIMPLEMENTED;
      break;
   }

   default:
   {
      DBPRINTF (("!!! Warning: DEVESC_ Unknown = '%lu' or 0x'%x'\n", lEscape, lEscape));
      DBPRINTF (("!!! returning DEVESC_NOTIMPLEMENTED!!!\n"));
      lrc = DEVESC_NOTIMPLEMENTED;
                     // @96652
                     // NOTE: MIRROR'ed apps (like WordPerfect 5.2)
                     //       Sends us PM_GETPHYSPAGESIZE escape
                     //       from Windows.
                     //       The Mirror's code will call DevQueryHCCaps
                     //       as long as we don't return DEV_OK.  - MFR
                     // NOTE: CorelDraw also sends us unknown escapes - MFR
                     // CONCERNS: some apps may loop until success and
                     // hang system like PM Draw - check this @TBD
      break;
   }
   }

depart:  // NOTE: Mutex Semaphores are not allowed for DevEscape() processing
          // since we must always process the asynchronous DEVESC_ABORTDOC
          // call even in the middle of another DevEscape call.

   UNREGISTERHANDLER (regrec);

#ifndef DEBUG
   fOK++;          // Avoid compiler warnings
#endif

   return lrc;

} /* end Escape */
