/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                                                                          */
/* PROGRAM NAME   : Font Test Help Functions                                */
/* AUTHOR         : Matthew F. Rutkowski                                    */
/* FILENAME       : FTUTIL.C                                                */
/* DATE WRITTEN   : 08-04-92                                                */
/* DESCRIPTION    : This module contains functions that help Font Test      */
/*                  fill dialog listboxes, perform string manipulation,     */
/*                  process INI file requests, File reads, and error        */
/*                  assertion.                                              */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* Modification/Change log                                                  */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* @DBCS    -          - DBCS enabling Tag                                  */
/*                                                                          */
/* @MAXLINE - 11/02/93 - Matt Rutkowski [IBM] - Allows browse and print of  */
/*                       huge files where previously files upto 6000 lines  */
/*                       were only allowed.                                 */
/*                                                                          */
/*                       Added 2 new special keys:                          */
/*                       CTRL-PAGEUP   - Browse top of file                 */
/*                       CTRL-PAGEDOWN - Browse bottom of file              */
/*                                                                          */
/* @74644   - 11/03/93 - Matt Rutkowski [IBM] - New text format option      */
/*                       "Print Line Numbers" added as a suggestion that    */
/*                       was logged as a defect.                            */
/*                                                                          */
/* @TTTTT   - 05/12/94 - Matt Rutkowski [IBM] - Add "Title text" option     */
/*                                                                          */
/* @100698  - 09/28/94 - Matt Rutkowski [IBM] - recalculate char box size   */
/*                       when user selects a new printer resolution.        */
/*                                                                          */
/*                                                                          */
/****************************************************************************/

#define INCL_DEV
#define INCL_DOSFILEMGR
#define INCL_DOSMEMMGR
#define INCL_DOSPROCESS
#define INCL_ERRORS
#define INCL_GPI
#define INCL_GPIERRORS
#define INCL_SPL
#define INCL_SPLDOSPRINT
#define INCL_WIN

#include <os2.h>
#include <stdlib.h>
#include <stdio.h>

#include "ft32.h"
#include "ftdlg.h"

/*------------------------------------------------------*/
/* prototypes                                           */
/*------------------------------------------------------*/

ERRORID  AssertPM( BOOL, HAB, PSZ );
VOID     ReplaceTabs( PSZ, PSZ, PULONG, CHAR, ULONG );
BOOL     StrToken( CHAR *, CHAR *, CHAR, USHORT );
USHORT   FillFormsListBox( HAB, HWND, LONG );
VOID     FillHCCapsListBox( HWND, USHORT );
BOOL     ReadInTextFile ( HAB, USHORT, ULONG, BOOL );
ULONG    CountLineFeeds ( CHAR FAR *, ULONG );
VOID     SetupTextFileLinePtrs ( CHAR FAR *, ULONG );
VOID     DeleteIniFileData( HWND, ULONG );
VOID     WriteIniFileData( PPGMDATA, ULONG );
VOID     ReadIniFileData( PPGMDATA, HWND );

/*------------------------------------------------------*/
/* external globals                                     */
/*------------------------------------------------------*/
extern PGMDATA    PgmData;         // current program data
extern PSZ        pFileBuff;
extern CHAR       DbcsEnv[MAXSTR];

/*------------------------------------------------------*/
/* globals                                              */
/*------------------------------------------------------*/

/****************************************************************************/
/* PROCEDURE NAME : WriteIniFileData                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/5/92                                                  */
/* DESCRIPTION    : Writes out data for the app to the OS2.INI file for     */
/*                  the keyname(s) passed in.                               */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Check all known keyname flags against the KeyNames  */
/*                      input parameter.  For all matched save the          */
/*                      corresponding keyname's data to OS2.INI.            */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (ULONG)  KeyNames - flag field that indicates which     */
/*                                      Font Test keynames need to be       */
/*                                      written to the OS2.INI file         */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID WriteIniFileData( PPGMDATA pPgmData, ULONG KeyNames )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ULONG ulDataLen;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-----------------*/
  /* Version         */
  /*-----------------*/
  if( KeyNames & FT_KEYID_VERSION )
  {
    CHAR szVersion[8];

    sprintf( szVersion, "%d.%d", FT_MAJOR_VERSION, FT_MINOR_VERSION );

    PrfWriteProfileString( HINI_USER,
                           FT_APPNAME,
                           FT_KEYNAME_VERSION,
                           szVersion );
  } /* end if */

  /*-----------------*/
  /* Color           */
  /*-----------------*/
  if( KeyNames & FT_KEYID_COLOR )
  {
     ulDataLen = (ULONG)sizeof(COLORDATA);

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_COLOR,
                          &(pPgmData->ColorData),
                          ulDataLen );
  } /* end if */

  /*-----------------*/
  /* Queue           */
  /*-----------------*/
  if( KeyNames & FT_KEYID_QUEUE )
  {
     ulDataLen = (ULONG)sizeof(QUEUEDATA);

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_QUEUE,
                          &(pPgmData->QueueData),
                          ulDataLen );

     ulDataLen = (ULONG)((PDRIVDATA)(pPgmData->QueueData.pDrivData)->cb);

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_JOBPROPSIZE,
                          &ulDataLen,
                          sizeof(ULONG) );

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_JOBPROP,
                          pPgmData->QueueData.pDrivData,
                          ulDataLen );
  } /* end if */

  /*-----------------*/
  /* Font            */
  /*-----------------*/
  if( KeyNames & FT_KEYID_FONT )
  {
     ulDataLen = (ULONG)sizeof(FONTDATA);

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_FONT,
                          &(pPgmData->FontData),
                          ulDataLen );
  } /* end if */

  /*-----------------*/
  /* Format Data     */
  /*-----------------*/
  if( KeyNames & FT_KEYID_FORMAT )
  {
     ulDataLen = (ULONG)sizeof(FORMATDATA);

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_FORMAT,
                          &(pPgmData->FmtData),
                          ulDataLen );
  } /* end if */

  /*-----------------*/
  /* Title Data      */
  /*-----------------*/
  if( KeyNames & FT_KEYID_TITLE )
  {
     ulDataLen = (ULONG)sizeof(TEXTDATA);

     PrfWriteProfileData( HINI_USER,
                          FT_APPNAME,
                          FT_KEYNAME_TITLE,
                          &(pPgmData->TxtTitle),
                          ulDataLen );
  } /* end if */

} /* end WriteIniFileData */

/****************************************************************************/
/* PROCEDURE NAME : DefaultFormatOptions                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/7/95                                                  */
/* DESCRIPTION    : Deletes data for the app in the OS2.INI file for the    */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID DefaultFormatOptions( PFORMATDATA pfd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  pfd->bPrintCharBoxes  = FALSE;
  pfd->bPrintDualColumn = FALSE;
  pfd->bPrintFileName   = FALSE;
  pfd->bPrintTime       = FALSE;
  pfd->bPrintDate       = FALSE;
  pfd->bPrintPageNumbers= FALSE;
  pfd->bPrintTextBorder = FALSE;
  pfd->bPrintLineNumbers= FALSE;
  pfd->bPrintTitleText  = FALSE;  // @TTTTT

} /* end DefaultFormatOptions */

/****************************************************************************/
/* PROCEDURE NAME : DeleteIniFileData                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/5/92                                                  */
/* DESCRIPTION    : Deletes data for the app in the OS2.INI file for the    */
/*                  keyname(s) passed in.                                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Check all known keyname flags against the KeyNames  */
/*                      input parameter.  For all matched delete the        */
/*                      corresponding keyname's data from OS2.INI.          */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwndClient - window handle for client window   */
/*                  (ULONG)  KeyNames   - flag field that indicates which   */
/*                                        Font Test keynames need to be     */
/*                                        deleted from the OS2.INI file     */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID DeleteIniFileData( HWND hwndClient, ULONG KeyNames )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-----------------*/
  /* Version         */
  /*-----------------*/
  if( KeyNames & FT_KEYID_VERSION )
  {
     PrfWriteProfileString(HINI_USER,FT_APPNAME,FT_KEYNAME_VERSION, NULL );

  } /* end if */

  /*-----------------*/
  /* Color           */
  /*-----------------*/
  if( KeyNames & FT_KEYID_COLOR )
  {
    PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_COLOR, NULL, 0L );

    // Restore default colors
    PgmData.ColorData.WinTextColor     = CLR_BLACK;
    PgmData.ColorData.WinBackColor     = CLR_WHITE;
    PgmData.ColorData.PrnTextColor     = CLR_BLACK;
    PgmData.ColorData.PrnBackColor     = CLR_WHITE;
    PgmData.ColorData.TextBorderColor  = CLR_BLACK;
    PgmData.ColorData.TextTrailerColor = CLR_BLACK;
    PgmData.ColorData.TextTitleColor   = CLR_BLACK;  // @TTTTT
    PgmData.ColorData.SearchHighliteColor = CLR_RED; // @TTTTT
  } /* end if */

  /*-----------------*/
  /* Queue           */
  /*-----------------*/
  if( KeyNames & FT_KEYID_QUEUE )
  {
    PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_QUEUE, NULL, 0L );
    PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_JOBPROPSIZE, NULL, 0L );
    PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_JOBPROP, NULL, 0L );

    PgmData.QueueData.QueueName[0] = '\0';
    PgmData.QueueData.QueueDesc[0] = '\0';
    PgmData.QueueData.pDrivData = NULL;
  } /* end if */

  /*-----------------*/
  /* Font            */
  /*-----------------*/
  if( KeyNames & FT_KEYID_FONT )
  {
    PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_FONT, NULL, 0L );

    // use default data ( Courier 8 pt font )
    strcpy( PgmData.FontData.fm.szFacename, FT_DEFAULT_FONT );
    PgmData.FontData.fm.lMatch      = FT_DEFAULT_LMATCH;
    PgmData.FontData.fm.fsSelection = 0;
    PgmData.FontData.PointSize      = FT_DEFAULT_POINTSIZE;
    PgmData.FontData.FontScalable   = FALSE;
  } /* end if */

  /*-----------------*/
  /* Format Data     */
  /*-----------------*/
  if( KeyNames & FT_KEYID_FORMAT )
  {
     PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_FORMAT, NULL, 0L );

     DefaultFormatOptions( &PgmData.FmtData );

  } /* end if */

  /*-----------------*/
  /* Title Data      */
  /*-----------------*/
  if( KeyNames & FT_KEYID_TITLE )
  {
     PrfWriteProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_TITLE, NULL, 0L );

     PgmData.TxtTitle.pszText[0] = '\0';
     PgmData.TxtTitle.usTextJust = TEXT_JUSTIFY_LEFT;

  } /* end if */

  // Redraw client window w/ defaults
  WinInvalidateRect ( hwndClient, NULL, TRUE );

} /* end DeleteIniFileData */


/****************************************************************************/
/* PROCEDURE NAME : ReadIniFileData                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/5/92                                                  */
/* DESCRIPTION    : Reads in app data from the OS2.INI file for all         */
/*                  keynames known.  If a given keyname is not found or     */
/*                  it's data is invalid default data is set.               */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Read all Key Name data from the OS2.INI file.       */
/*                  (b) validate each key name's data as read in            */
/*                  (c) delete any key data that is invalid.                */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HWND) hwnd - caller's window handle                    */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID ReadIniFileData( PPGMDATA pPgmData, HWND hwnd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  CHAR   Version[MAXSTR];
  CHAR   Buffer[MAXSTR];
  ULONG  ulDataLen;
  USHORT MajorVersion, MinorVersion;
  BOOL   bSuccess;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-----------------*/
  /* Version         */
  /*-----------------*/
  ulDataLen = PrfQueryProfileString( HINI_USER, FT_APPNAME,
                                      FT_KEYNAME_VERSION, NULL,Buffer,
                                      MAXSTR);

  // If we have INI file data for fontest
  if( ulDataLen > 0 )
  {
    // Parse string for major.minor parts
    StrToken( Version,Buffer, '.', 1 );
    MajorVersion = (USHORT)atoi( Version );

    StrToken( Version,Buffer, '.', 2 );
    MinorVersion = (USHORT)atoi( Version );

    // major version is outdated OR minor version unrecognized
    if( MajorVersion != FT_MAJOR_VERSION  ||
        MinorVersion < FT_MINOR_VERSION_1 ||
        MinorVersion > FT_MINOR_VERSION )
    {
      // ini data is of an older version so reset it
      DeleteIniFileData( hwnd, FT_KEYID_ALL );

    } /* end if */
    else
    {
      // Read-in INI file data setting up program's default settings
      // correct/adjust for older minor versions of programs data strucs

      /*-----------------*/
      /* Color           */
      /*-----------------*/
      ulDataLen = sizeof(COLORDATA);

      // read in Color data from user INI file
      bSuccess = PrfQueryProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_COLOR,
                                      &pPgmData->ColorData, &ulDataLen );

      // read of data was successful
      if( !bSuccess )
      {
        // No data in INI file or call failed so setup defaults
        DeleteIniFileData( hwnd, FT_KEYID_COLOR );

      } /* end if */
      else
      {
        // size of color data matches current FT_MINOR_VERSION size
        if( ulDataLen != sizeof(COLORDATA) )
        {
          // @74644 - See if the INI data is in a version format we recognize
          switch( MinorVersion )
          {
            case FT_MINOR_VERSION_1:
            case FT_MINOR_VERSION_2:
            case FT_MINOR_VERSION_3:
              // New color option in v1.4 for "Title Text"
              pPgmData->ColorData.TextTitleColor      = CLR_BLACK; // @TTTTT
              pPgmData->ColorData.SearchHighliteColor = CLR_RED;
            break;

            // case v1.4 but struc not big enough
            case FT_MINOR_VERSION_4:
              // New color option in v1.4 for "Title Text"
              pPgmData->ColorData.TextTitleColor      = CLR_BLACK; // @TTTTT
              pPgmData->ColorData.SearchHighliteColor = CLR_RED;
            break;

            default:
              // Unknown data for version reset to defaults
              DeleteIniFileData( hwnd, FT_KEYID_COLOR );
            break;

          } /* end switch */

        } /* end if */

      } /* end else */

      /*-----------------*/
      /* Font            */
      /*-----------------*/
      ulDataLen = sizeof(FONTDATA);

      bSuccess = PrfQueryProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_FONT,
                                      &pPgmData->FontData, &ulDataLen );

      // See if the INI data is in same format we expect
      if( !bSuccess )
      {
        DeleteIniFileData( hwnd, FT_KEYID_FONT );
      } /* end if */
      else
      {
        if( ulDataLen != sizeof(FONTDATA) )
        {
          // @74644
          switch( MinorVersion )
          {
            default:
              DeleteIniFileData( hwnd, FT_KEYID_FONT );
            break;

          } /* end switch */

        } /* end if */

      } /* end else */

      /*-----------------*/
      /* Queue           */
      /*-----------------*/
      ulDataLen = sizeof(QUEUEDATA);

      bSuccess = PrfQueryProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_QUEUE,
                                &pPgmData->QueueData, &ulDataLen );

      // See if the INI data is in same format we expect
      if ( !bSuccess || (ulDataLen != sizeof(QUEUEDATA)) )
      {
        DeleteIniFileData( hwnd, FT_KEYID_QUEUE );
      } /* end if */
      else
      {

        ULONG ul = sizeof(ULONG);

        bSuccess = PrfQueryProfileData( HINI_USER,
                                  FT_APPNAME,
                                  FT_KEYNAME_JOBPROPSIZE,
                                  &ulDataLen,
                                  &ul );

        // able to read size of Driver Data
        if( bSuccess )
        {
          // allocate memory for driver data
          DosAllocMem( (PPVOID)&(pPgmData->QueueData.pDrivData),
                        ulDataLen,
                        PAG_READ | PAG_WRITE | PAG_COMMIT );

          // read in driver data
          bSuccess = PrfQueryProfileData( HINI_USER,
                                    FT_APPNAME,
                                    FT_KEYNAME_JOBPROP,
                                    pPgmData->QueueData.pDrivData,
                                    &ulDataLen );

          // Prf call failed
          if( !bSuccess )
          {
            // free Driver Data memory and reset pointer to NULL
            DosFreeMem( pPgmData->QueueData.pDrivData );
            pPgmData->QueueData.pDrivData = (PDRIVDATA)NULL;
          } /* end if */

        } /* end if */

      } /* end else */

      /*-----------------*/
      /* Format Data     */
      /*-----------------*/
      ulDataLen = (ULONG)sizeof(FORMATDATA);

      bSuccess = PrfQueryProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_FORMAT,
                                &pPgmData->FmtData, &ulDataLen );


      // See if the INI data is in same format we expect
      if( !bSuccess )
      {
        DeleteIniFileData( hwnd, FT_KEYID_FORMAT );
      } /* end if */
      else
      {
        if( ulDataLen != sizeof(FORMATDATA) )
        {

          // @74644
          switch( MinorVersion )
          {
            case FT_MINOR_VERSION_1:
            case FT_MINOR_VERSION_2:

              // added new text format "Line Numbers" in v1.3
              // accept what we read in and init line number BOOL
              pPgmData->FmtData.bPrintLineNumbers = FALSE;

            case FT_MINOR_VERSION_3:

              // Added text format option "Title Text" in v1.4
              pPgmData->FmtData.bPrintTitleText  = FALSE;  // @TTTTT

            break;

            default:
              DeleteIniFileData( hwnd, FT_KEYID_FORMAT );
            break;

          } /* end switch */

        } /* end if */

      } /* end else */

      /*-----------------*/
      /* Title           */
      /*-----------------*/
      ulDataLen = sizeof(TEXTDATA);

      bSuccess = PrfQueryProfileData( HINI_USER, FT_APPNAME, FT_KEYNAME_TITLE,
                                &pPgmData->TxtTitle, &ulDataLen );

      // See if the INI data is in same format we expect
      if ( !bSuccess || (ulDataLen != sizeof(TEXTDATA)) )
      {
        DeleteIniFileData( hwnd, FT_KEYID_TITLE );
      } /* end if */

    } /* end else */

  } /* end if */
  else
  {
    // ini data does not exist so set defaults
    DeleteIniFileData( hwnd, FT_KEYID_ALL );

  } /* end else */

  // Data should now be in latest format, update INI version to latest level
  WriteIniFileData( pPgmData, FT_KEYID_VERSION  );

} /* end ReadIniFileData */


/****************************************************************************/
/* PROCEDURE NAME : StrToken                                                */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Copies substring from string2 to string1.  The          */
/*                  substring to be copied is identified by its position    */
/*                  of occurrence in string1 as separated by a token.       */
/*                  The copy will proceed up to the next token or an        */
/*                  end-of-string character is encountered                  */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Advance position in string2 to requested occurrence */
/*                      (occ) of the token (tok).                           */
/*                  (b) Copy string2 to string1 until token or end of       */
/*                      string found.                                       */
/*                  (c) Always return a valid NULL terminated string even   */
/*                      on error.                                           */
/*                                                                          */
/* PARAMETERS:      (CHAR *) string1 - destination string                   */
/*                  (CHAR *) string2 - source string                        */
/*                  (CHAR)   tok     - token used as occurrence separator   */
/*                  (USHORT) occ     - occurrence of substring to copy      */
/*                                                                          */
/* RETURN VALUES:   (BOOL) ErrorFound == TRUE  - Failure, error found       */
/*                                    == FALSE - Success, no errors         */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL StrToken( CHAR *string1, CHAR *string2, CHAR tok, USHORT occ )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  USHORT i = 0, j= 0;
  BOOL   ErrorFound = FALSE;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  occ--;

  while ( occ > 0 && string2[i] != '\0' )
  {
    if( string2[i] == tok )
    {
       occ--;
    } /* end if */

    i++;

  } /* endwhile */

  if( string2[i] != '\0' )
  {
    while( string2[i] != tok && string2[i] != '\0' )
    {

      string1[j] = string2[i];

      i++;
      j++;

    } /* end while */

    string1[j] = '\0';

  } /* end then */
  else
  {
    string1[j] = '\0';
    ErrorFound = TRUE;

  } /* end else */

  return( ErrorFound );

} /* end StrToken */


/****************************************************************************/
/* PROCEDURE NAME : FillCapsListBox                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Displays all known capabilities supported by the        */
/*                  DevQueryCaps call in a listbox.  The caps have already  */
/*                  been queried and placed in an array before the call.    */
/*                  This procedure attempts to interpret the caps data      */
/*                  for the user so they don't have to look it up in        */
/*                  an OS/2 programming reference book.                     */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Loop through all device capabilites returned from   */
/*                      DevQueryCaps();                                     */
/*                  (b) Interpret each capability and insert information    */
/*                      about capability into caller's listbox.             */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd - caller's window handle where we send    */
/*                                  insert messages                         */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID FillCapsListBox( HWND hwnd, PLONG plCaps )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  SHORT  i;
  SHORT  j;
  SHORT  sNumSubCaps;
  CHAR   szCapName[MAXSTR];
  CHAR   szCapString[MAXSTR];
  CHAR   szSubCapNames[10][MAXSTR];
  HWND   hwndLB = WinWindowFromID( hwnd, IDL_CAPSLIST );

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  WinSendDlgItemMsg( hwnd, IDL_CAPSLIST, LM_DELETEALL, NULL, NULL );

  for( i = CAPS_START; i < CAPS_END; i++ )
  {
    // For each capabilty we start with no sub-capabilities
    sNumSubCaps = 0;

    // Decide which Device Capabilty to report on
    switch (i)
    {
      case CAPS_FAMILY:
      {
        strcpy( szCapName, "CAPS_FAMILY" );
        switch(plCaps[i])
        {
          case OD_QUEUED:
            sprintf( szSubCapNames[sNumSubCaps],
                     "OD_QUEUED (%d)", plCaps[i] );
          break;

          case OD_DIRECT:
            sprintf( szSubCapNames[sNumSubCaps],
                     "OD_DIRECT (%d)", plCaps[i] );
          break;

          case OD_INFO:
            sprintf( szSubCapNames[sNumSubCaps],
                     "OD_INFO (%d)", plCaps[i] );
          break;

          case OD_METAFILE:
            sprintf( szSubCapNames[sNumSubCaps],
                     "OD_METAFILE (%d)", plCaps[i] );
          break;

          case OD_METAFILE_NOQUERY:
            sprintf( szSubCapNames[sNumSubCaps],
                     "OD_METAFILE_NOQUERY (%d)", plCaps[i] );
          break;

          case OD_MEMORY:
            sprintf( szSubCapNames[sNumSubCaps],
                     "OD_MEMORY (%d)", plCaps[i] );
          break;

          default:
            sprintf( szSubCapNames[sNumSubCaps],
                     "Unknown CAPS_FAMILY (%d)", plCaps[i] );
          break;
        } /* end switch */

        // we always have 1 sub-capability
        sNumSubCaps++;

      } /* end case */
      break;
      case CAPS_IO_CAPS:
      {
        strcpy( szCapName, "CAPS_IO_CAPS" );
        switch( plCaps[i] )
        {
          case CAPS_IO_DUMMY:
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_IO_DUMMY (%d)", plCaps[i] );
          break;

          case CAPS_IO_SUPPORTS_OP:
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_IO_SUPPORTS_OP (%d)", plCaps[i] );
          break;

          case CAPS_IO_SUPPORTS_IP:
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_IO_SUPPORTS_IP (%d)", plCaps[i] );
          break;

          case CAPS_IO_SUPPORTS_IO:
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_IO_SUPPORTS_IO (%d)", plCaps[i] );
          break;

          default:
            sprintf( szSubCapNames[sNumSubCaps],
                     "Unknown CAPS_IO_CAPS (%d)", plCaps[i] );
          break;

        } /* end switch */

        // we always have 1 sub-capability
        sNumSubCaps++;

      } /* end case */
      break;
      case CAPS_TECHNOLOGY:
      {
        strcpy( szCapName, "CAPS_TECHNOLOGY" );
        switch( plCaps[i] )
        {
          case CAPS_TECH_UNKNOWN:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_TECH_UNKNOWN (%d)", plCaps[i] );
          }
          break;
          case CAPS_TECH_VECTOR_PLOTTER:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_TECH_VECTOR_PLOTTER (%d)", plCaps[i] );
          }
          break;
          case CAPS_TECH_RASTER_DISPLAY:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_TECH_RASTER_DISPLAY (%d)", plCaps[i] );
          }
          break;
          case CAPS_TECH_RASTER_PRINTER:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_TECH_RASTER_PRINTER (%d)", plCaps[i] );
          }
          break;
          case CAPS_TECH_RASTER_CAMERA:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_TECH_RASTER_CAMERA (%d)", plCaps[i] );
          }
          break;
          case CAPS_TECH_POSTSCRIPT:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_TECH_POSTSCRIPT (%d)", plCaps[i] );
          }
          break;
          default:
          {
            sprintf( szSubCapNames[sNumSubCaps],
                     "Unknown CAPS_TECHNOLOGY (%d)", plCaps[i] );
          }
          break;
        } /* end switch */

        // We always have 1 sub-capability
        sNumSubCaps++;

      } /* end case */
      break;
      case CAPS_DRIVER_VERSION:
      {
           CHAR  szTemp[33];
           float fTemp;

           strcpy( szCapName, "CAPS_DRIVER_VERSION" );

           _ltoa( plCaps[i], szTemp, 16 );
           fTemp = ((float)atoi( szTemp ))/100.0;

           sprintf( szSubCapNames[sNumSubCaps], "%1.2f", fTemp );

           sNumSubCaps++;
      } /* end case */
      break;
      case CAPS_WIDTH:
           strcpy( szCapName, "CAPS_WIDTH" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_HEIGHT:
           strcpy( szCapName, "CAPS_HEIGHT" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_WIDTH_IN_CHARS:
           strcpy( szCapName, "CAPS_WIDTH_IN_CHARS" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Characters", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_HEIGHT_IN_CHARS:
           strcpy( szCapName, "CAPS_HEIGHT_IN_CHARS" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Characters", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_HORIZONTAL_RESOLUTION:
           strcpy( szCapName, "CAPS_HORIZONTAL_RESOLUTION" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels/meter", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_VERTICAL_RESOLUTION:
           strcpy( szCapName, "CAPS_VERTICAL_RESOLUTION" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels/meter", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_CHAR_WIDTH:
           strcpy( szCapName, "CAPS_CHAR_WIDTH" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels (CharBox Width)", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_CHAR_HEIGHT:
           strcpy( szCapName, "CAPS_CHAR_HEIGHT" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels (CharBox Height)", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_SMALL_CHAR_WIDTH:
           strcpy( szCapName, "CAPS_SMALL_CHAR_WIDTH" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels (small CharBox Width)", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_SMALL_CHAR_HEIGHT:
           strcpy( szCapName, "CAPS_SMALL_CHAR_HEIGHT" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels (small CharBox Height)", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_COLORS:
           strcpy( szCapName, "CAPS_COLORS" );
           sprintf( szSubCapNames[sNumSubCaps], "Number Supported = %d", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_COLOR_PLANES:
           strcpy( szCapName, "CAPS_COLOR_PLANES" );
           sprintf( szSubCapNames[sNumSubCaps], "Number Supported = %d", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_COLOR_BITCOUNT:
           strcpy( szCapName, "CAPS_COLOR_BITCOUNT" );
           sprintf( szSubCapNames[sNumSubCaps], "%d adjacent bits/pel/plane", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_COLOR_TABLE_SUPPORT:
           strcpy( szCapName, "CAPS_COLOR_TABLE_SUPPORT" );
           if( plCaps[i] & CAPS_COLTABL_RGB_8  )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_COLTABL_RGB_8" );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_COLTABL_RGB_8_PLUS  )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_COLTABL_RGB_8_PLUS" );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_COLTABL_TRUE_MIX )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_COLTABL_TRUE_MIX" );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_COLTABL_REALIZE )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_COLTABL_REALIZE" );
             sNumSubCaps++;
           }
      break;
      case CAPS_MOUSE_BUTTONS:
           strcpy( szCapName, "CAPS_MOUSE_BUTTONS" );
           sprintf( szSubCapNames[sNumSubCaps], "   Number supported = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_FOREGROUND_MIX_SUPPORT:
           sprintf( szCapName, "CAPS_FOREGROUND_MIX_SUPPORT = 0x%x", plCaps[i] );

           /*-------------------------------------*/
           /*  CAPS_FM_OR                      1L */
           /*  CAPS_FM_OVERPAINT               2L */
           /*  CAPS_FM_XOR                     8L */
           /*  CAPS_FM_LEAVEALONE             16L */
           /*  CAPS_FM_AND                    32L */
           /*  CAPS_FM_GENERAL_BOOLEAN        64L */
           /*-------------------------------------*/

           if( plCaps[i] & CAPS_FM_OR )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FM_OR (0x%x)", CAPS_FM_OR );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FM_OVERPAINT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FM_OVERPAINT (0x%x)", CAPS_FM_OVERPAINT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FM_XOR )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FM_XOR (0x%x)", CAPS_FM_XOR );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FM_LEAVEALONE )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FM_LEAVEALONE (0x%x)", CAPS_FM_LEAVEALONE );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FM_AND )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FM_AND (0x%x)", CAPS_FM_AND );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FM_GENERAL_BOOLEAN )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FM_GENERAL_BOOLEAN (0x%x)", CAPS_FM_GENERAL_BOOLEAN );
             sNumSubCaps++;
           }
         break;

      case CAPS_BACKGROUND_MIX_SUPPORT:
           sprintf( szCapName, "CAPS_BACKGROUND_MIX_SUPPORT = 0x%x", plCaps[i] );

           /*-------------------------------------*/
           /*  CAPS_BM_OR                      1L */
           /*  CAPS_BM_OVERPAINT               2L */
           /*  CAPS_BM_XOR                     8L */
           /*  CAPS_BM_LEAVEALONE             16L */
           /*  CAPS_BM_AND                    32L */
           /*  CAPS_BM_GENERAL_BOOLEAN        64L */
           /*  CAPS_BM_SRCTRANSPARENT        128L */
           /*  CAPS_BM_DESTTRANSPARENT       256L */
           /*-------------------------------------*/

           if( plCaps[i] & CAPS_BM_OR )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_OR (0x%x)",
                      CAPS_BM_OR );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_OVERPAINT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_OVERPAINT (0x%x)",
                      CAPS_BM_OVERPAINT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_XOR )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_XOR (0x%x)",
                      CAPS_BM_XOR );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_LEAVEALONE )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_LEAVEALONE (0x%x)",
                      CAPS_BM_LEAVEALONE );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_AND )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_AND (0x%x)",
                      CAPS_BM_AND );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_GENERAL_BOOLEAN )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_GENERAL_BOOLEAN (0x%x)",
                      CAPS_BM_GENERAL_BOOLEAN );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_SRCTRANSPARENT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_SRCTRANSPARENT (0x%x)",
                      CAPS_BM_SRCTRANSPARENT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BM_DESTTRANSPARENT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_BM_DESTTRANSPARENT (0x%x)",
                      CAPS_BM_DESTTRANSPARENT );
             sNumSubCaps++;
           }
         break;
      case CAPS_VIO_LOADABLE_FONTS:
           strcpy( szCapName, "CAPS_VIO_LOADABLE_FONTS" );
           sprintf( szSubCapNames[sNumSubCaps], "Loadable fonts for VIO = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_WINDOW_BYTE_ALIGNMENT:
           strcpy( szCapName, "CAPS_WINDOW_BYTE_ALIGNMENT" );
           if( plCaps[i] & CAPS_BYTE_ALIGN_REQUIRED )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_BYTE_ALIGN_REQUIRED" );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BYTE_ALIGN_RECOMMENDED )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_BYTE_ALIGN_RECOMMENDED" );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_BYTE_ALIGN_NOT_REQUIRED )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_BYTE_ALIGN_NOT_REQUIRED" );
             sNumSubCaps++;
           }
         break;
      case CAPS_BITMAP_FORMATS:
           strcpy( szCapName, "CAPS_BITMAP_FORMATS" );
           sprintf( szSubCapNames[sNumSubCaps], "Bitmap formats supported = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_RASTER_CAPS:
           sprintf( szCapName, "CAPS_RASTER_CAPS = 0x%x", plCaps[i] );
           if( plCaps[i] & CAPS_RASTER_BITBLT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                     "CAPS_RASTER_BITBLT (0x%x)",
                     CAPS_RASTER_BITBLT);
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_RASTER_BANDING )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_RASTER_BANDING (0x%x)",
                      CAPS_RASTER_BANDING );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_RASTER_BITBLT_SCALING )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_RASTER_BITBLT_SCALING (0x%x)",
                      CAPS_RASTER_BITBLT_SCALING );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_RASTER_SET_PEL )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_RASTER_SET_PEL (0x%x)",
                      CAPS_RASTER_SET_PEL );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_RASTER_FONTS )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_RASTER_FONTS (0x%x)",
                      CAPS_RASTER_FONTS );
             sNumSubCaps++;
           }
         break;
      case CAPS_MARKER_HEIGHT:
           strcpy( szCapName, "CAPS_MARKER_HEIGHT" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_MARKER_WIDTH:
           strcpy( szCapName, "CAPS_MARKER_WIDTH" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_DEVICE_FONTS:
           strcpy( szCapName, "CAPS_DEVICE_FONTS" );
           sprintf( szSubCapNames[sNumSubCaps], "Device Specific Fonts = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_GRAPHICS_SUBSET:
           strcpy( szCapName, "CAPS_GRAPHICS_SUBSET" );
           sprintf( szSubCapNames[sNumSubCaps], "Graphics Support = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_GRAPHICS_VERSION:
           strcpy( szCapName, "CAPS_GRAPHICS_VERSION" );
           sprintf( szSubCapNames[sNumSubCaps], "Graphics Version = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_GRAPHICS_VECTOR_SUBSET:
           strcpy( szCapName, "CAPS_GRAPHICS_VECTOR_SUBSET" );
           sprintf( szSubCapNames[sNumSubCaps], "Graphics Vector Support = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_DEVICE_WINDOWING:
           strcpy( szCapName, "CAPS_DEVICE_WINDOWING" );
           if( plCaps[i] & CAPS_DEV_WINDOWING_SUPPORT )
           {
             strcpy( szSubCapNames[sNumSubCaps],
                     "CAPS_DEV_WINDOWING_SUPPORT" );
             sNumSubCaps++;
           }
         break;
      case CAPS_ADDITIONAL_GRAPHICS:
           strcpy( szCapName, "CAPS_ADDITIONAL_GRAPHICS" );

           /*-----------------------------------------------*/
           /* CAPS_VDD_DDB_TRANSFER               1L        */
           /* CAPS_GRAPHICS_KERNING_SUPPORT       2L        */
           /* CAPS_FONT_OUTLINE_DEFAULT           4L        */
           /* CAPS_FONT_IMAGE_DEFAULT             8L        */
           /* CAPS_SCALED_DEFAULT_MARKERS        64L        */
           /* CAPS_COLOR_CURSOR_SUPPORT         128L        */
           /* CAPS_PALETTE_MANAGER              256L        */
           /* CAPS_COSMETIC_WIDELINE_SUPPORT    512L        */
           /* CAPS_DIRECT_FILL                 1024L        */
           /* CAPS_REBUILD_FILLS               2048L        */
           /* CAPS_CLIP_FILLS                  0x00001000   */
           /* CAPS_ENHANCED_FONTMETRICS        0x00002000   */
           /* CAPS_TRANSFORM_SUPPORT           0x00004000   */
           /* CAPS_EXTERNAL_16_BITCOUNT        0x00008000   */
           /*-----------------------------------------------*/

           if( plCaps[i] & CAPS_VDD_DDB_TRANSFER )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_VDD_DDB_TRANSFER (0x%x)",
                      CAPS_VDD_DDB_TRANSFER );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_GRAPHICS_KERNING_SUPPORT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_GRAPHICS_KERNING_SUPPORT (0x%x)",
                      CAPS_GRAPHICS_KERNING_SUPPORT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FONT_OUTLINE_DEFAULT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FONT_OUTLINE_DEFAULT (0x%x)",
                      CAPS_FONT_OUTLINE_DEFAULT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_FONT_IMAGE_DEFAULT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_FONT_IMAGE_DEFAULT (0x%x)",
                      CAPS_FONT_IMAGE_DEFAULT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_SCALED_DEFAULT_MARKERS )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_SCALED_DEFAULT_MARKERS (0x%x)",
                      CAPS_SCALED_DEFAULT_MARKERS );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_COLOR_CURSOR_SUPPORT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_COLOR_CURSOR_SUPPORT (0x%x)",
                      CAPS_COLOR_CURSOR_SUPPORT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_PALETTE_MANAGER )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_PALETTE_MANAGER (0x%x)",
                      CAPS_PALETTE_MANAGER );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_COSMETIC_WIDELINE_SUPPORT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_COSMETIC_WIDELINE_SUPPORT (0x%x)",
                      CAPS_COSMETIC_WIDELINE_SUPPORT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_DIRECT_FILL )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_DIRECT_FILL (0x%x)",
                      CAPS_DIRECT_FILL );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_REBUILD_FILLS )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_REBUILD_FILLS (0x%x)",
                      CAPS_REBUILD_FILLS );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_CLIP_FILLS )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_CLIP_FILLS (0x%x)",
                      CAPS_CLIP_FILLS );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_ENHANCED_FONTMETRICS )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_ENHANCED_FONTMETRICS (0x%x)",
                      CAPS_ENHANCED_FONTMETRICS );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_TRANSFORM_SUPPORT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_TRANSFORM_SUPPORT (0x%x)",
                      CAPS_TRANSFORM_SUPPORT );
             sNumSubCaps++;
           }
           if( plCaps[i] & CAPS_EXTERNAL_16_BITCOUNT )
           {
             sprintf( szSubCapNames[sNumSubCaps],
                      "CAPS_EXTERNAL_16_BITCOUNT (0x%x)",
                      CAPS_EXTERNAL_16_BITCOUNT );
             sNumSubCaps++;
           }
         break;
      case CAPS_PHYS_COLORS:
           strcpy( szCapName, "CAPS_PHYS_COLORS" );
           sprintf( szSubCapNames[sNumSubCaps], "Max Colors = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_COLOR_INDEX:
           strcpy( szCapName, "CAPS_COLOR_INDEX" );
           sprintf( szSubCapNames[sNumSubCaps], "Max color table index = %d", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_GRAPHICS_CHAR_WIDTH:
           strcpy( szCapName, "CAPS_GRAPHICS_CHAR_WIDTH" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels (graphics charbox width)", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_GRAPHICS_CHAR_HEIGHT:
           strcpy( szCapName, "CAPS_GRAPHICS_CHAR_HEIGHT" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels (graphics charbox height)", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_HORIZONTAL_FONT_RES:
           strcpy( szCapName, "CAPS_HORIZONTAL_FONT_RES" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels/inch (horz device res)", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_VERTICAL_FONT_RES:
           strcpy( szCapName, "CAPS_VERTICAL_FONT_RES" );
           sprintf( szSubCapNames[sNumSubCaps], "%d Pels/inch (vert device res)", plCaps[i] );
           sNumSubCaps++;
         break;
      case CAPS_DEVICE_FONT_SIM:
           strcpy( szCapName, "CAPS_DEVICE_FONT_SIM" );
           sprintf( szSubCapNames[sNumSubCaps], "%d", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_LINEWIDTH_THICK:
           strcpy( szCapName, "CAPS_LINEWIDTH_THICK" );
           sprintf( szSubCapNames[sNumSubCaps], "%d", plCaps[i] );
           sNumSubCaps++;
      break;
      case CAPS_DEVICE_POLYSET_POINTS:
           strcpy( szCapName, "CAPS_DEVICE_POLYSET_POINTS" );
           sprintf( szSubCapNames[sNumSubCaps], "%d", plCaps[i] );
           sNumSubCaps++;
      break;

      default:
           sprintf( szCapName, "Unrecognized Capabilty found = %2d", i );
      break;

    } /* end switch */

    // Insert the capability in the listbox
    sprintf( szCapString, "[%2d] %s", i, szCapName );

    WinInsertLboxItem( hwndLB, LIT_END, szCapString );

    // For every subcaps print it out
    for ( j=0; j < sNumSubCaps; j++ )
    {
      // indent a little to visually indicate subcap
      // instaead of a regular capability
      sprintf( szCapString, "         %s", szSubCapNames[j] );

      WinInsertLboxItem( hwndLB, LIT_END, szCapString );

    } /* endfor */

  } /* end for */

} /* end FillCapsListBox */



/****************************************************************************/
/* PROCEDURE NAME : FillHCCapsListBox                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Displays all known capabilities supported by the        */
/*                  DevQueryHardcopyCaps call in a listbox.  The caps       */
/*                  have already been queried and placed in an array before */
/*                  the call.                                               */
/*                  This procedure attempts to interpret the caps data      */
/*                  for the user so they don't have to look it up in        */
/*                  an OS/2 programming reference book.                     */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Use index passed in to index into (HCINFO) data     */
/*                      structure.                                          */
/*                  (b) Interpret fields of (HCINFO) structure and display  */
/*                      interpreted data in a dialog listbox                */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd  - callers window handle where we send    */
/*                                   listbox insert messages                */
/*                  (USHORT) index - index into (HCINFO) structure for      */
/*                                   form we are interpreting/addding       */
/*                                   information for.                       */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID FillHCCapsListBox( HWND hwnd, USHORT index )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  CHAR szText[MAXSTR];
  PHCINFO phci = PgmData.PrinterData.pHcInfo;
  HWND hwndLB = WinWindowFromID( hwnd, IDL_HCCAPSLIST );

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  WinSendDlgItemMsg( hwnd, IDL_HCCAPSLIST, LM_DELETEALL, NULL, NULL );

  /*---------------*/
  /* cx            */
  /*---------------*/
  sprintf( szText, "cx (width)  = %d mm", phci[index].cx );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* cy            */
  /*---------------*/
  sprintf( szText, "cy (height) = %d mm", phci[index].cy );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* xLeftClip     */
  /*---------------*/
  sprintf( szText, "xLeftClip = %d mm", phci[index].xLeftClip );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* yBottomClip   */
  /*---------------*/
  sprintf( szText, "yBottomClip = %d mm", phci[index].yBottomClip );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* xRightClip    */
  /*---------------*/
  sprintf( szText, "xRightClip = %d mm", phci[index].xRightClip );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* yTopClip      */
  /*---------------*/
  sprintf( szText, "yTopClip = %d mm", phci[index].yTopClip );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* xPels         */
  /*---------------*/
  sprintf( szText, "xPels (imageable area) = %d pels", phci[index].xPels );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* yPels         */
  /*---------------*/
  sprintf( szText, "yPels (imageable area) = %d pels", phci[index].yPels );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  /*---------------*/
  /* flAttributes  */
  /*---------------*/
  sprintf( szText, "flAttributes = %d", phci[index].flAttributes );
  WinInsertLboxItem( hwndLB, LIT_END, szText );

  if( phci[index].flAttributes & HCAPS_CURRENT )
  {
    sprintf( szText, "     HCAPS_CURRENT (0x%x)", HCAPS_CURRENT );
    WinInsertLboxItem( hwndLB, LIT_END, szText );

  } /* end if */

  if( phci[index].flAttributes & HCAPS_SELECTABLE )
  {
    sprintf( szText, "     HCAPS_SELECTABLE (0x%x)", HCAPS_SELECTABLE );
    WinInsertLboxItem( hwndLB, LIT_END, szText );

  } /* end if */

} /* end FillHCCapsListBox */


/****************************************************************************/
/* PROCEDURE NAME : FillFormsListBox                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Uses forms info already returned from DevQueryHCCaps    */
/*                  to enumerate all forms known by the device in a         */
/*                  listbox.                                                */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Loop THROUGH each entry in the (HCINFO) structure   */
/*                      list returned from DevQueryHardCopyCaps(); and      */
/*                      insert the names of each form returned in a listbox */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HAB)    hab            - caller's anchor block handle  */
/*                  (HWND)   hwnd           - caller's window handle        */
/*                  (LONG)   lFormsReturned - Number of forms in (HCINFO)   */
/*                                            list                          */
/*                                                                          */
/* RETURN VALUES:   (USHORT) CurrentIndex  - Index into (HCINFO) list of    */
/*                                           the form marked HCCAPS_CURRENT */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
USHORT FillFormsListBox( HAB hab, HWND hwnd, LONG lFormsReturned )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  LONG    i = 0;
  LONG    lCurrentIndex = -1; // default to invalid index
  PHCINFO phci = PgmData.PrinterData.pHcInfo;
  HWND    hwndLB = WinWindowFromID( hwnd, IDL_FORMSLIST );

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  WinSendDlgItemMsg( hwnd, IDL_FORMSLIST, LM_DELETEALL, NULL, NULL );
  while( i < lFormsReturned )
  {

    WinInsertLboxItem( hwndLB, LIT_END, phci[i].szFormname );

    if( phci[i].flAttributes & HCAPS_CURRENT )
    {
      lCurrentIndex = i;

    } /* endif */

    i++;

  } /* end while */

  // error if no form marked current
  if( lCurrentIndex == -1 )
  {
    // use first form in list as workaround
    lCurrentIndex = 0;

    AssertPM( TRUE, hab, ERR_NO_CURRENT_FORM );

  } /* end if */

  return( (USHORT)lCurrentIndex );

} /* end FillFormsListBox */



/****************************************************************************/
/* PROCEDURE NAME : ReadInTextFile                                          */
/* AUTHOR         : Matt R.                                                 */
/* DATE WRITTEN   : 8/3/92                                                  */
/* DESCRIPTION    : This function accepts an open text file handle and      */
/*                  reads it into memory.  It then counts the number of     */
/*                  lines in the file, and then initializes an array        */
/*                  of pointers to the lines in the file.                   */
/*                                                                          */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) call DosQueryFileInfo(); to get files size.         */
/*                  (b) call DosAllocMem(); to allocate memory to hold file */
/*                  (c) call DosRead(); to read in file                     */
/*                  (d) call CountLineFeeds(); to count number of CR LF in  */
/*                      the text file for display purposes.                 */
/*                  (e) Initialize pointers to the beginning of each text   */
/*                      line based on CR/LF                                 */
/*                  (f) call DosClose(); to close file                      */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HAB)    hab               - anchor block handle        */
/*                  (USHORT) hInFile           - file handle to open        */
/*                  (ULONG)  cbFile            - number of bytes in file    */
/*                  (BOOL)   FileIsAlreadyOpen - indicates if file has      */
/*                                               previously been opened     */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:   (BOOL)   TRUE  - Success (file read in)                 */
/*                           FALSE - Failure - file not read in             */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL ReadInTextFile ( HAB hab,
                      USHORT hInFile,
                      ULONG cbFile,
                      BOOL FileIsAlreadyOpen )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  BOOL         bFileTruncated = FALSE; // If file truncated (too big)
  FILESTATUS3  FileInfoBuf;
  ULONG        ulBytesRead;
  ULONG        ulLineCount = 0L;       // current lines in file
  SHORT        sNumBuffers;            // @MAXLINE: # of lines in file/32k

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // If a file is already open
  if( FileIsAlreadyOpen )
  {
    // free the memory that hold the file
    if( pFileBuff )
    {
       DosFreeMem( pFileBuff );
    } /* end if */

    if( pLinePtrs )
    {
      DosFreeMem( pLinePtrs );
    } /* end if */

  } /* end if */

  // Reset global values for new file
  ulTotalLines = 0;

  // Find size of file to be read in
  DosQueryFileInfo( hInFile, FIL_STANDARD, &FileInfoBuf, sizeof(FILESTATUS3) );

  // Allocate as much memory as needed to hold file
  if( DosAllocMem( (PPVOID)&pFileBuff, (FileInfoBuf.cbFile + 1L) ,
                    PAG_READ | PAG_WRITE | PAG_COMMIT ) == NO_ERROR )
  {
    USHORT  usTemp;    // @MAXLINES
    ldiv_t ldivTemp;   // @MAXLINES

    // Init local line count
    ulLineCount = 0L;

    // read the file into the buffer
    if( DosRead( hInFile, pFileBuff, FileInfoBuf.cbFile, &ulBytesRead) )
    {
       AssertPM( TRUE, hab, ERR_DOSREAD );

       return(FALSE);

    } /* end if */

    // count the number of lines in the file
    ulLineCount = CountLineFeeds ( pFileBuff, FileInfoBuf.cbFile );

    ulTotalLines = ulLineCount;

    // @MAXLINES - How many 32k line buffers do we need to display file
    ldivTemp = ldiv( ulTotalLines, MAXLINES );  // @MAXLINES
    sNumBuffers = ldivTemp.quot;

    // @MAXLINES - how many extra buffers for remaining lines
    if( ldivTemp.rem )
    {
      sNumBuffers++;

    } /* end if */

// @MAXLINES
//
//  if ( ulTotalLines > MAXLINES )
//  {
//    AssertPM( TRUE, hab, ERR_TRUNCATE );
//
//    // Read as many lines as we can in
//    ulTotalLines = MAXLINES;
//    bFileTruncated = TRUE;
//
//  } /* end if */

    // @MAXLINES - allocate memory to hold line pointers
    if( DosAllocMem( (PPVOID)&pLinePtrs, ( (ulLineCount + 1) * sizeof(PSZ)),
                      PAG_READ | PAG_WRITE | PAG_COMMIT ) == NO_ERROR )
    {
      /* initialize the line ptr array to point to lines in the file */
      SetupTextFileLinePtrs ( pFileBuff, ulTotalLines );

      // Close the input file since we've read it all in.
      DosClose( hInFile );

      /* ensure that there's a carriage return at the end of the file */
      *( pFileBuff + FileInfoBuf.cbFile ) = '\n';

      return(TRUE);

    } /* end if */
    else
    {
      AssertPM( TRUE, hab, ERR_DOSALLOCMEM );

      // Close the input file since we failed to set up line ptrs
      DosClose( hInFile );

      return(FALSE);

    } /* end else */

  } /* end if */
  else
  {
    AssertPM( TRUE, hab, ERR_DOSALLOCMEM );

    // @MAXLINES - Close the input file since alloc failed
    DosClose( hInFile );

    return(FALSE);

  } /* end else */

} /* end ReadInTextFile */


/****************************************************************************/
/* PROCEDURE NAME : CountLineFeeds                                          */
/* AUTHOR         : Matt R.                                                 */
/* DATE WRITTEN   : 8/3/92                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* This function returns a count of the number of \n terminated lines       */
/* in a raw ASCII buffer                                                    */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (USHORT) msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG CountLineFeeds ( CHAR FAR * buffptr, ULONG ulFileSize )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ULONG ulCount;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // for every byte in the file
  for ( ulCount = 0L; ulFileSize-- ; buffptr ++ )
  {
      // count each EOL character representing lines in file
      if ( *buffptr == '\n')
          ulCount++;

  } /* end for */

  // Check for files with no line end character
  if ( ulCount == 0)
  {
    ulCount++;  // always 1 line
  } /* end if */

  return ( ulCount );

} /* end CountLineFeeds */


/****************************************************************************/
/* PROCEDURE NAME : SetupTextFileLinePtrs                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/6/92                                                  */
/* DESCRIPTION    : This function accepts a pointer to a raw ASCII text     */
/*                  buffer and initializes an array of pointer to lines in  */
/*                  the buffer.                                             */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) search through text file buffer for line end        */
/*                      marker and if found place the next byte's address   */
/*                      into an array of line pointers.                     */
/*                                                                          */
/* PARAMETERS:      (CHAR FAR *) buffptr   - pointer to memory buffer where */
/*                                           text file is stored            */
/*                  (ULONG)      linecount - number of line (CR/LF) in text */
/*                                           file                           */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID SetupTextFileLinePtrs ( CHAR FAR * buffptr, ULONG linecount )
{

  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
//USHORT       CurLineCount = 0;
  ULONG        ulCurLineCount = 0; // @MAXLINES

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  pLinePtrs [ulCurLineCount++] = buffptr;

  while( --linecount != 0 )
  {
    while( *buffptr++ != '\n' );

    pLinePtrs [ulCurLineCount++] = buffptr;

  } /* end while */

} /* end SetupTextFileLinePtrs */



/****************************************************************************/
/* PROCEDURE NAME : AssertPM                                                */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/5/92                                                  */
/* DESCRIPTION    : Used to perform error checking by assertion             */
/*                  causes a popup message box to appear.                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Determine what type of assertion message box and    */
/*                      icon to display (Information, error or warning).    */
/*                                                                          */
/*                  (b) call WinGetLastError(); to retrieve last system     */
/*                      error message set.                                  */
/*                                                                          */
/*                  (c) call WindMessageBox(); to display error message     */
/*                      passed in and error code from (b).                  */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (BOOL) bDisplayError - error condition result passed    */
/*                                         from caller.  If this evaluated  */
/*                                         to TRUE an error occurred so     */
/*                                         this tells us whether to display */
/*                                         a message box or not             */
/*                  (HAB)  hab           - caller's anchor block handle     */
/*                                         to retrieve last error           */
/*                  (PSZ)  message       - message string to display in     */
/*                                         message box.                     */
/*                                                                          */
/* RETURN VALUES:   (ERRORID) ErrorCode  == PM Error Code (pmerr.h)         */
/*                                       == NO_ERROR                        */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*  4/24/93 - Matt Rutkowski [IBM] - added ability to use AssertPM for      */
/*            displaying information to user as well as errors.             */
/*                                                                          */
/****************************************************************************/
ERRORID AssertPM( BOOL bDisplayError, HAB habErr, PSZ message )
{
  static       ULONG ulErrCount = 0;

  if( bDisplayError == TRUE )
  {
    ERRORID    ErrorCode;
    CHAR       BodyText[MAXSTR];
    CHAR       TitleText[MAXSTR];
    CHAR       Buffer[32];
    ULONG      ulIcon= MB_ICONEXCLAMATION;
    ULONG      ulSound = WA_ERROR;

    TitleText[0] = '\0';

    // Get last set PM Error Code (set by: PMWIN, PMGPI, or Device Driver)
    ErrorCode = WinGetLastError( habErr );
    ErrorCode &= 0xffff;
    _ltoa( (LONG)ErrorCode, Buffer, 16 );
    sprintf( TitleText, "Error Code = %s", Buffer );


    // Test for special conditions we don't want to freak out over
    switch( ErrorCode )
    {
      case PMERR_SPL_PRINT_ABORT: sprintf( BodyText, "%s", ERR_JOB_ABORTED );
      break;
      default: sprintf( BodyText, "%s", message );
      break;
    } /* end switch */

    // Display a Message Box w/ Caller's Message and OS/2 Error ID
    WinMessageBox( (HWND)HWND_DESKTOP,
                   (HWND)HWND_DESKTOP,
                   (PSZ)BodyText,
                   (PSZ)TitleText,
                   1L, MB_OK | ulIcon );
    ulErrCount++;

    if( ErrorCode == PMERR_OK )
      ErrorCode = PMERR_UNKNOWN_ERROR_CODE;

    return( ErrorCode );

  } /* end if */

  return( PMERR_OK );

} /* end AssertPM */

/****************************************************************************/
/* PROCEDURE NAME : ReplaceTabs                                             */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 01/25/95                                                */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:      */
/*                  */
/*                                                                          */
/* RETURN VALUES:   */
/*                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID ReplaceTabs( PSZ pszIn, PSZ pszOut, PULONG pulLength, CHAR chReplace, ULONG ulReplaceCount )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ULONG ul;  // loop control variable

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Initialize string to NULL character string
  *pszOut = '\0';

  // @DBCS - while replacing chars look for double-byte char sets
  // replace tab characters with eight spaces
  for ( *pulLength = 0 ;(*pszIn != '\n') &&
      ( *pulLength < MAX_CHAR_PER_LINE );
        pszIn += CHARSIZE(*pszIn) )
  {
     if ( *pszIn == '\t' )
     {
        for ( ul=0 ; ul < ulReplaceCount ; ul++ )
        {
          *pszOut = ' ';
          pszOut++;

        } /* end for */

        *pulLength += ulReplaceCount - 1;

     } /* end if */
     else
     {
        if (CHARSIZE(*pszIn) == SIZEDBCS)
        {
           *((PUSHORT)pszOut) = *(PUSHORT)pszIn;
           pszOut += SIZEDBCS;
           (*pulLength)++;
        }
        else
        {
           *pszOut = *pszIn;
           pszOut++;

        } /* end else */

     } /* end else */

    (*pulLength)++;

  } /* end for */

  // If CR terminated
  if ( *(pszOut-1) == 0xd )
  {
     // NULL terminate string replacing CR char
     *(pszOut-1) = '\0';
     (*pulLength)--;

  } /* end if */
  else
  {
     // NULL terminate string
     *(pszOut) = '\0';

  } /* end else */

} /* end ReplaceTabs */


