/*
 *  linux/fs/msdos/namei.c
 *
 *  Written 1992,1993 by Werner Almesberger
 *
 *  Windows95/Windows NT compatible extended MSDOS filesystem
 *    by Gordon Chaffee Copyright (C) 1995
 */

#ifdef __IBMC__
#pragma strings(readonly)
#endif

#ifdef MODULE
#include <linux/module.h>
#endif

#ifdef OS2
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>		// From the "Developer Connection Device Driver Kit" version 2.0

#include <string.h>                // Standard MS Visual C++ string.h

#include <os2/types.h>
#include <os2/StackToFlat.h>
#include <os2/devhlp32.h>
#include <os2/os2proto.h>

#endif

#ifndef OS2
#include <asm/segment.h>
#endif

#include <linux/sched.h>

#ifdef OS2
#include <linux/ext2_fs.h>
#include <linux/fs_proto.h>
#include <linux/ext2_proto.h>
#endif

#include <linux/xmsdos_fs.h>
#ifndef OS2
#include <linux/kernel.h>
#endif
#include <linux/errno.h>
#ifndef OS2
#include <linux/string.h>
#endif
#include <linux/stat.h>

#include "msbuffer.h"

#if 0
#define PRINTK(x) printk x
#else
#define PRINTK(x)
#endif

#ifdef DEBUG_LOG
extern char debug_com;
extern char debug_vfat_com;             // output VFAT debug info to COM port
#define KERNEL_PRINTF(x) { if (debug_vfat_com) kernel_printf(x);}
#define KERNEL_PRINTF1(x) { if (debug_com) kernel_printf(x);}
#else
#define KERNEL_PRINTF(x)
#endif

#ifdef DEBUG

static void dump_fat(struct super_block *sb,int start)
{
	printk("[");
	while (start) {
		printk("%d ",start);
        	start = fat_access(sb,start,-1);
		if (!start) {
			printk("ERROR");
			break;
		}
		if (start == -1) break;
	}
	printk("]\n");
}

static void dump_de(struct msdos_dir_entry *de)
{
	int i;
	unsigned char *p = (unsigned char *) de;
	printk("[");

	for (i = 0; i < 32; i++, p++) {
		printk("%02x ", *p);
	}
	printk("]\n");
}

#endif

/* MS-DOS "device special files" */

static const char *reserved_names[] = {
	"CON     ","PRN     ","NUL     ","AUX     ",
	"LPT1    ","LPT2    ","LPT3    ","LPT4    ",
	"COM1    ","COM2    ","COM3    ","COM4    ",
	NULL };


/* Characters that are undesirable in an MS-DOS file name */
  
static char bad_chars[] = "*?<>|\"";
static char bad_if_strict[] = "+=,; ";

static int xmsdos_find(struct inode *dir,const char *name,int len,
		      int find_long,int new_filename,int is_dir,
		      struct slot_info *sinfo_out);

/* Checks the validity of an long MS-DOS filename */
/* Returns negative number on error, 0 for a normal
 * return, and 1 for . or .. */

static int xmsdos_valid_longname(const char *name, int len, int dot_dirs)
{
	const char **reserved;
	unsigned char c;
	int i;

KERNEL_PRINTF("xmsdos_valid_longname");

	if (IS_FREE(name)) return -EINVAL;
	if (name[0] == '.' && (len == 1 || (len == 2 && name[1] == '.'))) {
		if (!dot_dirs) return -EEXIST;
		return 1;
	}

	if (len >= 256) return -EINVAL;
	for (i = 0; i < len; i++) {
		c = name[i];
		if (strchr(bad_chars,c)) return -EINVAL;
	}
	if (len == 3 || len == 4) {
		for (reserved = reserved_names; *reserved; reserved++)
			if (!strncmp(name,*reserved,8)) return -EINVAL;
	}
	return 0;
}

static int xmsdos_valid_shortname(char conv,const char *name,int len,
				 int dot_dirs)
{
	const char *walk, **reserved;
	unsigned char c;
	int space;
KERNEL_PRINTF("xmsdos_valid_shortname");

	if (IS_FREE(name)) return -EINVAL;
	if (name[0] == '.' && (len == 1 || (len == 2 && name[1] == '.'))) {
		if (!dot_dirs) return -EEXIST;
		return 1;
	}

	space = 1; /* disallow names starting with a dot */
	c = 0;
	for (walk = name; len && walk-name < 8;) {
	    	c = *walk++;
		len--;
		if (conv != 'r' && strchr(bad_chars,c)) return -EINVAL;
		if (conv == 's' && strchr(bad_if_strict,c)) return -EINVAL;
  		if (c >= 'A' && c <= 'Z' && (conv == 's' || conv == 'x')) return -EINVAL;
		if (c < ' ' || c == ':' || c == '\\') return -EINVAL;
		if (c == '.') break;
		space = c == ' ';
	}
	if (space) return -EINVAL;
	if ((conv == 's' || conv == 'x') && len && c != '.') {
		c = *walk++;
		len--;
		if (c != '.') return -EINVAL;
	}
	while (c != '.' && len--) c = *walk++;
	if (c == '.') {
		if (len >= 4) return -EINVAL;
		while (len > 0 && walk-name < (MSDOS_NAME+1)) {
			c = *walk++;
			len--;
			if (conv != 'r' && strchr(bad_chars,c)) return -EINVAL;
			if (conv == 's' && strchr(bad_if_strict,c))
				return -EINVAL;
			if (c < ' ' || c == ':' || c == '\\' || c == '.')
				return -EINVAL;
			if (c >= 'A' && c <= 'Z' && (conv == 's' || conv == 'x')) return -EINVAL;
			space = c == ' ';
		}
		if (space) return -EINVAL;
		if ((conv == 's' || conv == 'x') && len) return -EINVAL;
	}
	for (reserved = reserved_names; *reserved; reserved++)
		if (!strncmp(name,*reserved,8)) return -EINVAL;

	return 0;
}

/* Takes a short filename and converts it to a formatted MS-DOS filename.
 * If the short filename is not a valid MS-DOS filename, an error is 
 * returned.  The formatted short filename is returned in 'res'.
 */

static int xmsdos_format_name(char conv,const char *name,int len,char *res,
  int dot_dirs)
{
	char *walk;
	const char **reserved;
	unsigned char c;
	int space;

KERNEL_PRINTF("xmsdos_format_name");

	if (IS_FREE(name)) return -EINVAL;
	if (name[0] == '.' && (len == 1 || (len == 2 && name[1] == '.'))) {
		if (!dot_dirs) return -EEXIST;
		memset(res+1,' ',10);
		while (len--) *res++ = '.';
		return 0;
	}

	space = 1; /* disallow names starting with a dot */
	c = 0;
	for (walk = res; len && walk-res < 8; walk++) {
	    	c = *name++;
		len--;
		if (conv != 'r' && strchr(bad_chars,c)) return -EINVAL;
		if (conv == 's' && strchr(bad_if_strict,c)) return -EINVAL;
		if (c >= 'A' && c <= 'Z' && (conv == 's' || conv == 'x')) return -EINVAL;
		if (c < ' ' || c == ':' || c == '\\') return -EINVAL;
		if (c == '.') break;
		space = c == ' ';
		*walk = c >= 'a' && c <= 'z' ? c-32 : c;
	}
	if (space) return -EINVAL;
	if ((conv == 's' || conv == 'x') && len && c != '.') {
		c = *name++;
		len--;
		if (c != '.') return -EINVAL;
	}
	while (c != '.' && len--) c = *name++;
	if (c == '.') {
		while (walk-res < 8) *walk++ = ' ';
		while (len > 0 && walk-res < MSDOS_NAME) {
			c = *name++;
			len--;
			if (conv != 'r' && strchr(bad_chars,c)) return -EINVAL;
			if (conv == 's' && strchr(bad_if_strict,c))
				return -EINVAL;
			if (c < ' ' || c == ':' || c == '\\' || c == '.')
				return -EINVAL;
			if (c >= 'A' && c <= 'Z' && (conv == 's' || conv == 'x')) return -EINVAL;
			space = c == ' ';
			*walk++ = c >= 'a' && c <= 'z' ? c-32 : c;
		}
		if (space) return -EINVAL;
		if ((conv == 's' || conv == 'x') && len) return -EINVAL;
	}
	while (walk-res < MSDOS_NAME) *walk++ = ' ';
	for (reserved = reserved_names; *reserved; reserved++)
		if (!strncmp(res,*reserved,8)) return -EINVAL;

	return 0;
}

static char skip_chars[] = "^+=/[]:',.?*\\<>|\" ";

/* Given a valid longname, create a unique shortname.  Make sure the
 * shortname does not exist
 */
static int xmsdos_create_shortname(struct inode *dir, const char *name,
     int len, char *name_res)
{
	const char *ip, *ext_start, *end;
	char *p;
	int valid;
	int sz, extlen, baselen, totlen;
	char msdos_name[13];
	char base[9], ext[4];
	int i;
	int res;
	int ino;
	int spaces;
	int count;
	char buf[8];
	struct slot_info sinfo;
#ifdef OS2
        int k;
#endif

KERNEL_PRINTF("xmsdos_create_shortname");

	PRINTK(("Entering xmsdos_create_shortname: name=%s, len=%d\n", name, len));
	sz = 0;			/* Make compiler happy */
	valid = 1;
	if (len <= 12) {
		/* Do a case insensitive search if the name would be a valid
		 * shortname if is were all capitalized */
		for (i = 0, p = msdos_name, ip = name; i < len; i++, p++, ip++)
		{
			if (*ip >= 'A' && *ip <= 'Z') {
				*p = *ip + 32;
			} else {
				*p = *ip;
			}
		}
KERNEL_PRINTF("xmsdos_create_shortname 1");
		res = xmsdos_format_name('x', __StackToFlat(msdos_name), len, name_res, 1);
KERNEL_PRINTF("xmsdos_create_shortname 2");
		if (res > -1) {
			PRINTK(("xmsdos_create_shortname 1\n"));
			res = xmsdos_find(dir, __StackToFlat(msdos_name), len, 0, 0, 0, __StackToFlat(&sinfo));
KERNEL_PRINTF("xmsdos_create_shortname 3");
			ino = sinfo.ino;
			PRINTK(("xmsdos_create_shortname 2\n"));
KERNEL_PRINTF("xmsdos_create_shortname 4");
			if (res > -1) return -EEXIST;
KERNEL_PRINTF("xmsdos_create_shortname 5");
			return 0;
		}
	}
KERNEL_PRINTF("xmsdos_create_shortname 6");
	PRINTK(("xmsdos_create_shortname 3\n"));
	/* Now, we need to create a shortname from the long name */
	ext_start = end = &name[len];
	while (--ext_start >= name) {
		if (*ext_start == '.') {
			if (ext_start == end - 1) {
				sz = len;
				ext_start = NULL;
			}
			break;
		}
	}
KERNEL_PRINTF("xmsdos_create_shortname 7");
	if (ext_start == name - 1) {
		sz = len;
		ext_start = NULL;
	} else if (ext_start) {
		sz = ext_start - name;
		ext_start++;
	}
KERNEL_PRINTF("xmsdos_create_shortname 8");
	for (baselen = i = 0, p = base, ip = name; i < sz && baselen < 8; i++)
	{
		if (!strchr(skip_chars, *ip)) {
			if (*ip >= 'A' && *ip <= 'Z') {
				*p = *ip + 32;
			} else {
				*p = *ip;
			}
			p++; baselen++;
		}
		ip++;
	}
KERNEL_PRINTF("xmsdos_create_shortname 9");
	if (baselen == 0) {
		baselen = 3;
		*p++ = 'd';
		*p++ = 'o';
		*p++ = 't';
	}
		
	spaces = 8 - baselen;

	if (ext_start) {
		extlen = end - ext_start;
		if (extlen > 3) extlen = 3;
		for (i = 0, p = ext, ip = ext_start; i < extlen; i++)
		{
			if (!strchr(skip_chars, *ip)) {
				if (*ip >= 'A' && *ip <= 'Z') {
					*p = *ip + 32;
				} else {
					*p = *ip;
				}
				p++;
			}
			ip++;
		}
	} else {
		extlen = 0;
	}
#ifdef DEBUGLOG
  if (debug_vfat_com) kernel_printf("xmsdos_create_shortname 10 baselen: %u extlen %u", baselen, extlen);
#endif
	ext[extlen] = '\0';
	base[baselen] = '\0';
	count = 0;

KERNEL_PRINTF1("xmsdos_create_shortname 100");

#ifndef OS2
	strcpy(msdos_name, base);
#else
        k = 0;
        while (base[k]) { msdos_name[k] = base[k]; k++;}
        msdos_name[k] = '\0';
#endif



KERNEL_PRINTF1("xmsdos_create_shortname 11");
	msdos_name[baselen] = '.';

#ifndef OS2
	strcpy(&msdos_name[baselen+1], ext);
#else
        k = 0;
        while (ext[k]) { msdos_name[baselen+1+k] = ext[k]; k++;}
        msdos_name[baselen+1+k] = '\0';
#endif


KERNEL_PRINTF1("xmsdos_create_shortname 12");
	totlen = baselen + extlen + 1;
	res = xmsdos_find(dir, __StackToFlat(msdos_name), totlen, 0, 0, 0, __StackToFlat(&sinfo));
KERNEL_PRINTF1("xmsdos_create_shortname 13");
	if (res < 0) {
KERNEL_PRINTF1("xmsdos_create_shortname 14");
		res = xmsdos_format_name('x', __StackToFlat(msdos_name), totlen, name_res, 1);
KERNEL_PRINTF1("xmsdos_create_shortname 15");
		return res;
	}

	while (res > -1) {

KERNEL_PRINTF1("xmsdos_create_shortname 13.1");

		/* Create the next shortname to try */
		count++;
KERNEL_PRINTF1("xmsdos_create_shortname 13.2");
		if (count == 10000000) return -EEXIST;
KERNEL_PRINTF1("xmsdos_create_shortname 13.3");
		sprintf(__StackToFlat(buf), "%d", count);
KERNEL_PRINTF1("xmsdos_create_shortname 13.4");
		sz = strlen(buf);
KERNEL_PRINTF1("xmsdos_create_shortname 13.5");
		if (sz + 1 > spaces) {
			baselen = baselen - (sz + 1 - spaces);
			spaces = sz + 1;
		}
KERNEL_PRINTF1("xmsdos_create_shortname 16");
		strncpy(msdos_name, base, baselen);

		msdos_name[baselen] = '~';
KERNEL_PRINTF1("xmsdos_create_shortname 17");
#ifndef OS2
		strcpy(&msdos_name[baselen+1], buf);
#else
                k = 0;
                while (buf[k]) { msdos_name[baselen+1+k] = buf[k]; k++;}
                msdos_name[baselen+1+k] = '\0';
if (debug_com) kernel_printf("xmsdos_create_shortname 17.1: %s", __StackToFlat(msdos_name));
                
#endif
		msdos_name[baselen+sz+1] = '.';
KERNEL_PRINTF1("xmsdos_create_shortname 18");
#ifndef OS2
		strcpy(&msdos_name[baselen+sz+2], ext);
#else
                k = 0;
                while (ext[k]) { msdos_name[baselen+sz+2+k] = ext[k]; k++;}
                msdos_name[baselen+sz+2+k] = '\0';
if (debug_com) kernel_printf("xmsdos_create_shortname 18.1: %s", __StackToFlat(msdos_name));
#endif

KERNEL_PRINTF1("xmsdos_create_shortname 19");
		totlen = baselen + sz + 2 + extlen;
		res = xmsdos_find(dir, __StackToFlat(msdos_name), totlen, 0, 0, 0, __StackToFlat(&sinfo));
KERNEL_PRINTF1("xmsdos_create_shortname 20");
	}
	res = xmsdos_format_name('x', __StackToFlat(msdos_name), totlen, name_res, 1);
KERNEL_PRINTF1("xmsdos_create_shortname 21");
	return res;
}

static loff_t xmsdos_find_free_slots(struct inode *dir,int slots)
{
	struct super_block *sb = dir->i_sb;
	loff_t offset, curr;
	struct msdos_dir_entry *de;
	struct buffer_head *bh;
	struct inode *inode;
	int ino;
	int row;
	int done;
	int res;
	int added;

KERNEL_PRINTF("xmsdos_find_free_slots");

	PRINTK(("xmsdos_find_free_slots: find %d free slots\n", slots));
	offset = curr = 0;
	bh = NULL;
	row = 0;
	ino = xmsdos_get_entry(dir,__StackToFlat(&curr),__StackToFlat(&bh),__StackToFlat(&de));

	for (added = 0; added < 2; added++) {
		while (ino > -1) {
			done = IS_FREE(de->name);
			if (done) {
				inode = iget(sb,ino);
				if (inode) {
					/* Directory slots of busy deleted files aren't available yet. */
					done = !MSDOS_I(inode)->i_busy;
					/* PRINTK(("inode %d still busy\n", ino)); */
				}
				iput(inode);
			}
			if (done) {
				row++;
				if (row == slots) {
					brelse(bh);
					/* printk("----- Free offset at %d\n", offset); */
					return offset;
				}
			} else {
				row = 0;
				offset = curr;
			}
			ino = xmsdos_get_entry(dir,__StackToFlat(&curr),__StackToFlat(&bh),__StackToFlat(&de));
		}

		if (dir->i_ino == MSDOS_ROOT_INO) return -ENOSPC;
		if ((res = xmsdos_add_cluster(dir)) < 0) return res;
		ino = xmsdos_get_entry(dir,__StackToFlat(&curr),__StackToFlat(&bh),__StackToFlat(&de));
	}
	/* Should never get here, but if it does */
	printk("xmsdos_find_free_slots: Unable to find any\n");
	return -ENOSPC;
}
		
static int xmsdos_build_slots(struct inode *dir,const char *name,int len,
     int find_long, int new_filename,
     struct msdos_dir_slot *ds, struct msdos_dir_slot *ds_mask,
     int *slots, int *is_long)
{
	struct msdos_dir_slot *ps, *ps_mask;
	struct msdos_dir_entry *de, *de_mask;
	char msdos_name[MSDOS_NAME];
	int res;
	int slot;
	int i;
	const char *ip;
	loff_t offset;
	unsigned char alias_checksum;

	PRINTK(("Entering xmsdos_build_slots: name=%s, len=%d\n", name, len));
	de = (struct msdos_dir_entry *) ds;
	de_mask = (struct msdos_dir_entry *) ds_mask;
	*slots = 1;
	*is_long = 0;

	memset(ds_mask, 0, sizeof(struct msdos_dir_slot) * MSDOS_SLOTS);


        if (len == 1 && name[0] == '.') {
		strncpy(de->name, MSDOS_DOT, MSDOS_NAME);
		memset(de_mask, 0xff, MSDOS_NAME);
	} else if (len == 2 && name[0] == '.' && name[1] == '.') {
		strncpy(de->name, MSDOS_DOT, MSDOS_NAME);
		memset(de_mask, 0xff, MSDOS_NAME);
	} else {
		PRINTK(("xmsdos_build_slots 4\n"));
		res = xmsdos_valid_shortname('x', name, len, 1);
		if (res > -1) {
			PRINTK(("xmsdos_build_slots 5a\n"));
			res = xmsdos_format_name('x', name, len, de->name, 1);
			PRINTK(("xmsdos_build_slots 5b\n"));
			memset(de_mask->name, 0xff, MSDOS_NAME);
			PRINTK(("xmsdos_build_slots 5c\n"));
		} else {
			PRINTK(("xmsdos_build_slots 5A: %s (len=%d) is an invalid shortname\n", name, len));
			if (new_filename) {
				unsigned char sum;

				PRINTK(("xmsdos_build_slots 5Z\n"));
				res = xmsdos_create_shortname(dir, name, len, __StackToFlat(msdos_name));
				PRINTK(("xmsdos_build_slots 5Y\n"));
				if (res < 0) {
					return res;
				}

				for (sum = i = 0; i < 11; i++) {
					sum = (((sum&1)<<7)|((sum&0xfe)>>1)) + msdos_name[i];
				}
				PRINTK(("xmsdos_build_slots 5X: msdos_name=%s\n", msdos_name));
				alias_checksum = sum;
			} else {
				alias_checksum = 0;
			}

			if (!find_long) return -EINVAL;
			res = xmsdos_valid_longname(name, len, 1);
			if (res < 0) return res;

			*is_long = 1;
			*slots = (len + 12) / 13;
			PRINTK(("xmsdos_build_slots 6: slots=%d\n",*slots));
			for (ps = ds, slot = *slots, ps_mask = ds_mask;
			     slot > 0; slot--, ps++, ps_mask++)
			{
				int end, j;

				PRINTK(("xmsdos_build_slots 6a\n"));
				ps->id = slot; ps_mask->id = 0xff;
				ps->attr = ATTR_EXT; ps_mask->attr = 0xff;
				ps->reserved = 0; ps_mask->reserved = 0xff;
				ps->alias_checksum = alias_checksum;
				ps_mask->alias_checksum = 0;
				ps->start[0] = 0; ps_mask->start[0] = 0xff;
				ps->start[1] = 0; ps_mask->start[1] = 0xff;
				PRINTK(("xmsdos_build_slots 6b: name=%s\n",name));
				offset = (slot - 1) * 13;
				ip = &name[offset];
				j = offset;
				end = 0;
				for (i = 0; i < 10; i += 2) {
					if (!end && j == len) {
						end = 1;
						ps->name0_4[i] = 0; ps_mask->name0_4[i] = 0xff;
						ps->name0_4[i+1] = 0; ps_mask->name0_4[i] = 0xff;
						continue;
					} else if (end) {
						ps->name0_4[i] = 0xff; ps_mask->name0_4[i] = 0xff;
						ps->name0_4[i+1] = 0xff; ps_mask->name0_4[i+1] = 0xff;
						continue;
					}
					ps->name0_4[i] = *ip; ps_mask->name0_4[i] = 0xff;
					ps->name0_4[i+1] = 0; ps_mask->name0_4[i+1] = 0xff;
					j++; ip++;
				}
				PRINTK(("xmsdos_build_slots 6c\n"));
				for (i = 0; i < 12; i += 2) {
					if (!end && j == len) {
						end = 1;
						ps->name5_10[i] = 0;
						ps->name5_10[i+1] = 0;
						continue;
					} else if (end) {
						ps->name5_10[i] = 0xff; ps_mask->name5_10[i] = 0xff;
						ps->name5_10[i+1] = 0xff; ps_mask->name5_10[i+1] = 0xff;
						continue;
					}
					ps->name5_10[i] = *ip; ps_mask->name5_10[i] = 0xff;
					ps->name5_10[i+1] = 0; ps_mask->name5_10[i+1] = 0xff;
					j++; ip++;
				}
				PRINTK(("xmsdos_build_slots 6d\n"));
				for (i = 0; i < 4; i += 2) {
					if (!end && j == len) {
						end = 1;
						ps->name11_12[i] = 0;
						ps->name11_12[i+1] = 0;
						continue;
					} else if (end) {
						ps->name11_12[i] = 0xff; ps_mask->name11_12[i] = 0xff;
						ps->name11_12[i+1] = 0xff; ps_mask->name11_12[i+1] = 0xff;
						continue;
					}
					ps->name11_12[i] = *ip; ps_mask->name11_12[i] = 0xff;
					ps->name11_12[i+1] = 0; ps_mask->name11_12[i+1] = 0xff;
					j++; ip++;
				}
			}
			PRINTK(("xmsdos_build_slots 6e\n"));
			ds[0].id |= 0x40;

			if (new_filename) {
				de = (struct msdos_dir_entry *) ps;
				de_mask = (struct msdos_dir_entry *) ps_mask;

				PRINTK(("xmsdos_build_slots 10\n"));
				strncpy(de->name, msdos_name, MSDOS_NAME);
				memset(de_mask->name, 0xff, MSDOS_NAME);
			}
		}
	}
	return 0;
}

/* Given a shortname offset, see if there is an associated longname.  Returns
 * the number of slots in the longname if one is found, else 0 */
static int xmsdos_get_longname(struct inode *dir,loff_t short_offset,
     unsigned char checksum, loff_t *pos_out)
{
	struct super_block *sb = dir->i_sb;
	struct msdos_dir_slot *ps;
	struct msdos_dir_entry *de;
	struct buffer_head *bh;
	loff_t offset, temp;
	int id, res, slots;

KERNEL_PRINTF("xmsdos_get_longname");

	/* printk("Short offset: %d\n", short_offset); */
	if (short_offset == 0) {
		return 0;
	}

	slots = 0;
	id = 1;
	bh = NULL;
	offset = short_offset - sizeof(struct msdos_dir_slot);
	while (offset > 0) {
		temp = offset;
		res = xmsdos_get_entry(dir,__StackToFlat(&temp),__StackToFlat(&bh),__StackToFlat(&de));
		if (res < 0) goto finish;
		ps = (struct msdos_dir_slot *) de;
		if (ps->alias_checksum != checksum) goto finish;
		if ((ps->id &~ 0x40) != id) goto finish;
		if (IS_FREE(de->name)) goto finish;
		if (ps->id & 0x40) {
			*pos_out = offset;
			slots = ps->id &~ 0x40;
			/* printk("Found a longname for the shortname: long_offset=%ld\n", offset); */
			goto finish;
		}
		offset -= sizeof(struct msdos_dir_slot);
		id++;
	}
 finish:
	if (bh) brelse(bh);
	return slots;
}

/*
#undef OS2Stack
*/
#define OS2Stack 1
 
#ifdef OS2Stack
char *kmalloc( long int size, unsigned int flags)
{ char *p;

  if (DevHlp32_VMAlloc(size, VMDHA_NOPHYSADDR, VMDHA_SWAP, (void **)(__StackToFlat(&p))) != NO_ERROR) p = NULL;

/*if (p==NULL) kernel_printf("kmalloc failed"); else kernel_printf("kmalloc 0x%x", p);*/

  return p;
}

void kfree(char *p)
{
/*kernel_printf("kfree");*/
   DevHlp32_VMFree(p);
}
#define GFP_KERNEL 0
#endif

static int xmsdos_find(struct inode *dir,const char *name,int len,
    int find_long, int new_filename,int is_dir,struct slot_info *sinfo_out)
{
	struct super_block *sb = dir->i_sb;
	const char *ip;
	char *op, *op_mask;
	int res;
#ifndef OS2Stack
	struct msdos_dir_slot ds[MSDOS_SLOTS], ds_mask[MSDOS_SLOTS];
#else
	struct msdos_dir_slot *ds, *ds_mask;
#endif
	struct msdos_dir_slot *ps;
	struct msdos_dir_entry *de, *de_mask;
	struct buffer_head *bh;
	int i, slot, slots;
	loff_t offset, start;
	int match;
	int is_long;
	unsigned char alias_checksum = 0;


KERNEL_PRINTF("xmsdos_find");

if (debug_vfat_com) {
   kernel_printf("xmsdos_find %s", name);
} /* endif */

#ifdef OS2Stack
	ds = (struct msdos_dir_slot *) kmalloc(sizeof(struct msdos_dir_slot)*MSDOS_SLOTS, GFP_KERNEL);
	if (ds == NULL) return -ENOMEM;

	ds_mask = (struct msdos_dir_slot *) kmalloc(sizeof(struct msdos_dir_slot)*MSDOS_SLOTS, GFP_KERNEL);
	if (ds_mask == NULL)
        {
           kfree((char*)ds);
           return -ENOMEM;
        }
#endif

	PRINTK(("xmsdos_find 1: name=%s, len=%d\n",name,len));
KERNEL_PRINTF("xmsdos_find 1");

#ifdef OS2Stack
	res = xmsdos_build_slots(dir, name, len, find_long, new_filename,
				ds, ds_mask, __StackToFlat(&slots), __StackToFlat(&is_long));
	if (res < 0) goto cleanup;
#else
	res = xmsdos_build_slots(dir, name, len, find_long, new_filename,
				__StackToFlat(&ds), __StackToFlat(&ds_mask), __StackToFlat(&slots), __StackToFlat(&is_long));
	if (res < 0) return res;
#endif

KERNEL_PRINTF("xmsdos_find 2");
	de = (struct msdos_dir_entry *) ds;
	de_mask = (struct msdos_dir_entry *) ds_mask;

	PRINTK(("xmsdos_find 7\n"));
	offset = start = 0;
	bh = NULL;
KERNEL_PRINTF("xmsdos_find 02");
	sinfo_out->ino = xmsdos_get_entry(dir,__StackToFlat(&offset),__StackToFlat(&bh),__StackToFlat(&de));
KERNEL_PRINTF("xmsdos_find 3");
	while (sinfo_out->ino > -1 && slots > 0) {
		match = 1;
KERNEL_PRINTF("xmsdos_find 4");
		ps = (struct msdos_dir_slot *) de;
		alias_checksum = ps->alias_checksum;
KERNEL_PRINTF("xmsdos_find 5");
		for (slot = 0; slot < slots; slot++) {
			ip = (char *) de;
			ps = (struct msdos_dir_slot *) de;
KERNEL_PRINTF("xmsdos_find 6");
			if (is_long && ps->alias_checksum != alias_checksum) {
				printk("Checksums don't match 1\n");
				match = 0;
				start = offset;
				break;
			}
KERNEL_PRINTF("xmsdos_find 7");
			for (i = 0, ip = (char *) de, op = (char *) &ds[slot], op_mask = (char *) &ds_mask[slot];
			     i < sizeof(struct msdos_dir_entry);
			     i++, ip++, op++, op_mask++)
			{
#if 0
				if (is_long && de->attr == ATTR_EXT)
					printk("%02x?%02x ",
					       (unsigned char) *ip,
					       (unsigned char) *op);
#endif
				if ((*ip & *op_mask) != (*op & *op_mask)) {
					start = offset;
					match = 0;
					break;
				}
			}
KERNEL_PRINTF("xmsdos_find 70");
#if 0
			if (is_long && de->attr == ATTR_EXT) printk("\n");
#endif
			if ((!is_long && !match) ||
			    (is_long && (match || slot == 0)))
			{
KERNEL_PRINTF("xmsdos_find 71");
				sinfo_out->ino = xmsdos_get_entry(dir,__StackToFlat(&offset),__StackToFlat(&bh),__StackToFlat(&de));
KERNEL_PRINTF("xmsdos_find 72");
				/* if (ino >=0 && de->attr == ATTR_EXT) dump_de(de); */
			}
			if (!match) {
				break;
			}
			if (sinfo_out->ino == -1) {
				match = 0;
				goto breakout;
			}
KERNEL_PRINTF("xmsdos_find 73");
		}
KERNEL_PRINTF("xmsdos_find 8");
		if (match) {
			unsigned char sum;

			for (sum = i = 0; i < 11; i++) {
				sum = (((sum&1)<<7)|((sum&0xfe)>>1)) + de->name[i];
			}
			if (is_long) {
				if (sum != alias_checksum) {
					PRINTK(("Checksums don't match %d != %d\n", sum, alias_checksum));
					match = 0;
				}
			} else {
				int long_slots;
				long_slots = xmsdos_get_longname(dir, offset - sizeof(struct msdos_dir_entry), sum, __StackToFlat(&start));
				if (long_slots > 0) {
					slots = long_slots;
					is_long = 1;
				}
			}
KERNEL_PRINTF("xmsdos_find 9");				
			if (match) {
				PRINTK(("name: %s, alias: %c%c%c%c%c%c%c%c%c%c%c\n",
					name,
					de->name[0], de->name[1], de->name[2],
					de->name[3], de->name[4], de->name[5],
					de->name[6], de->name[7], de->name[8],
					de->name[9], de->name[10]));
				PRINTK(("xmsdos_find 10\n"));
				res = CF_LE_W(de->start);

				sinfo_out->shortname_offset = offset - sizeof(struct msdos_dir_slot);
				sinfo_out->longname_offset = start;
				sinfo_out->is_long = is_long;
				if (is_long) {
					sinfo_out->long_slots = slots;
					slots++;
				} else {
					sinfo_out->long_slots = 0;
				}
KERNEL_PRINTF("xmsdos_find 10");
				sinfo_out->total_slots = slots;
				if (new_filename) {
					if (bh) brelse(bh);
#ifndef OS2Stack
					return -EEXIST;
#else
                                        res = -EEXIST;
                                        goto cleanup;
#endif

				}
				if (bh) brelse(bh);
#ifndef OS2Stack
				return res;
#else
                                goto cleanup;
#endif
			}
		}
	}
 breakout:
KERNEL_PRINTF("xmsdos_find ");
	PRINTK(("breakout\n"));

	if (bh) brelse(bh);
	if (new_filename) {
#ifdef OS2
        unsigned short  time;
        unsigned short  date;
#endif
		PRINTK(("xmsdos_find: create file 1\n"));
		if (is_long) slots++;
KERNEL_PRINTF("xmsdos_find 11");
		offset = xmsdos_find_free_slots(dir, slots);
		if (offset < 0) {
#ifndef OS2Stack
			return offset;
#else
                        res = offset;
                        goto cleanup;
#endif
		}

KERNEL_PRINTF("xmsdos_find 12");
		PRINTK(("xmsdos_find: create file 2\n"));
		/* Now create the new entry */
		bh = NULL;
		for (slot = 0, ps = ds; slot < slots; slot++, ps++) {
			PRINTK(("xmsdos_find: create file 3, slot=%d\n",slot));
			sinfo_out->ino = xmsdos_get_entry(dir,__StackToFlat(&offset),__StackToFlat(&bh),__StackToFlat(&de));
			if (sinfo_out->ino < 0) {
				PRINTK(("xmsdos_find: problem\n"));
#ifndef OS2Stack
				return sinfo_out->ino;
#else
				res = sinfo_out->ino;
                                goto cleanup;
#endif
			}
KERNEL_PRINTF("xmsdos_find 13");
			memcpy(de, ps, sizeof(struct msdos_dir_slot));
KERNEL_PRINTF("xmsdos_find 14");
			mark_buffer_dirty(bh, 1);
		}
KERNEL_PRINTF("xmsdos_find 15");
		PRINTK(("xmsdos_find: create file 4\n"));
		dir->i_ctime = dir->i_mtime = CURRENT_TIME;
		dir->i_dirt = 1;

		PRINTK(("xmsdos_find: create file 5\n"));

		memset(de->unused, 0, sizeof(de->unused));
#ifndef OS2
		date_unix2dos(dir->i_mtime,&de->time,&de->date);
#else
	        date_unix2dos(dir->i_mtime,__StackToFlat(&time), __StackToFlat(&date));
                de->time = time;
                de->date = date;

#endif
		de->start = 0;
		de->size = 0;
		de->attr = is_dir ? ATTR_DIR : ATTR_ARCH;
		de->lcase = CASE_LOWER_BASE | CASE_LOWER_EXT;
KERNEL_PRINTF("xmsdos_find 16");
		mark_buffer_dirty(bh, 1);
		brelse(bh);

		sinfo_out->is_long = (slots > 1) ? 1 : 0;
		if (sinfo_out->is_long) {
			sinfo_out->long_slots = slots - 1;
		} else {
			sinfo_out->long_slots = 0;
		}
		sinfo_out->total_slots = slots;
		sinfo_out->shortname_offset = offset - sizeof(struct msdos_dir_slot);
		sinfo_out->longname_offset = offset - sizeof(struct msdos_dir_slot) * slots;
#ifndef OS2Stack
		return 0;
#else
                res = 0;
                goto cleanup;
#endif
	}
#ifndef OS2Stack
	return -ENOENT;
#else
        res = -ENOENT;
cleanup:
        kfree((char*)ds);
        kfree((char*)ds_mask);
        return res;
#endif
}

int xmsdos_lookup(struct inode *dir,const char *name,int len,
    struct inode **result)
{
	int res, ino;
	struct inode *next;
	struct slot_info sinfo;

if (debug_vfat_com) kernel_printf("xmsdos_lookup dir=0x%x name=%s len=%d result = 0x%x", dir, name, len, result);

/*
return -ENOENT;
*/
	
	PRINTK (("xmsdos_lookup: name=%s, len=%d\n", name, len));

	*result = NULL;

KERNEL_PRINTF("xmsdos_lookup 1");

	if (!dir) return -ENOENT;
KERNEL_PRINTF("xmsdos_lookup 2");
	if (!S_ISDIR(dir->i_mode)) {
KERNEL_PRINTF("xmsdos_lookup 3");
		iput(dir);
		return -ENOENT;
	}
	PRINTK (("xmsdos_lookup 2\n"));
KERNEL_PRINTF("xmsdos_lookup 4");
	if (len == 1 && name[0] == '.') {
KERNEL_PRINTF("xmsdos_lookup");
        	*result = dir;
		return 0;
	}
	if (len == 2 && name[0] == '.' && name[1] == '.') {
KERNEL_PRINTF("xmsdos_lookup 5");
		ino = xmsdos_parent_ino(dir,0);
		iput(dir);
		if (ino < 0) return ino;
		if (!(*result = iget(dir->i_sb,ino))) return -EACCES;
		return 0;
	}
	PRINTK (("xmsdos_lookup 3\n"));
KERNEL_PRINTF("xmsdos_lookup 6");
	if ((res = xmsdos_find(dir,name,len,1,0,0,__StackToFlat(&sinfo))) < 0) {
KERNEL_PRINTF("xmsdos_lookup 7");
		iput(dir);
		return res;
	}
KERNEL_PRINTF("xmsdos_lookup 8");
	PRINTK (("xmsdos_lookup 4.5\n"));
	if (!(*result = iget(dir->i_sb,sinfo.ino))) {
KERNEL_PRINTF("xmsdos_lookup 9");
		iput(dir);
		return -EACCES;
	}
	PRINTK (("xmsdos_lookup 5\n"));
KERNEL_PRINTF("xmsdos_lookup 10");
	if (MSDOS_I(*result)->i_busy) { /* mkdir in progress */
KERNEL_PRINTF("xmsdos_lookup 11");
		iput(*result);
		iput(dir);
		return -ENOENT;
	}
	PRINTK (("xmsdos_lookup 6\n"));
KERNEL_PRINTF("xmsdos_lookup 12");
	while (MSDOS_I(*result)->i_old) {
KERNEL_PRINTF("xmsdos_lookup 13");
		next = MSDOS_I(*result)->i_old;
		iput(*result);
		if (!(*result = iget(next->i_sb,next->i_ino))) {
			fs_panic(dir->i_sb,"xmsdos_lookup: Can't happen");
			iput(dir);
			return -ENOENT;
		}
	}
KERNEL_PRINTF("xmsdos_lookup 14");
	iput(dir);
	return 0;
}


static int xmsdos_create_entry(struct inode *dir,const char *name,int len,
    int is_dir, struct inode **result)
{
	struct super_block *sb = dir->i_sb;
	int res,ino;
	loff_t offset;
	struct buffer_head *bh;
	struct msdos_dir_entry *de;
	struct slot_info sinfo;

KERNEL_PRINTF("xmsdos_create");

	PRINTK(("xmsdos_create_entry 1\n"));
	res = xmsdos_find(dir, name, len, 1, 1, is_dir, __StackToFlat(&sinfo));
	if (res < 0) {
		return res;
	}

	offset = sinfo.shortname_offset;

	PRINTK(("xmsdos_create_entry 2\n"));
	bh = NULL;
	ino = xmsdos_get_entry(dir, __StackToFlat(&offset), __StackToFlat(&bh), __StackToFlat(&de));
	if (ino < 0) {
		PRINTK(("xmsdos_mkdir problem\n"));
		if (bh) brelse(bh);
		return ino;
	}
	PRINTK(("xmsdos_create_entry 3\n"));

	if ((*result = iget(dir->i_sb,ino)) != NULL)
		xmsdos_read_inode(*result);
	brelse(bh);
	if (!*result) return -EIO;
	(*result)->i_mtime = (*result)->i_atime = (*result)->i_ctime =
	    CURRENT_TIME;
	(*result)->i_dirt = 1;

	return 0;
}

int xmsdos_create(struct inode *dir,const char *name,int len,int mode,
	struct inode **result)
{
	int res;

KERNEL_PRINTF("xmsdos_create");

	if (!dir) return -ENOENT;

	lock_creation();
	if ((res = xmsdos_create_entry(dir,name,len,0,result)) < 0) {
		printk("xmsdos_create: unable to get new entry\n");
		unlock_creation();
		iput(dir);
		return res;
	}

	unlock_creation();
	iput(dir);
	return res;
}

int xmsdos_create_a_dotdir(struct inode *dir,struct inode *parent,
     struct buffer_head *bh,
     struct msdos_dir_entry *de,int ino,const char *name, int isdot)
{
	struct super_block *sb = dir->i_sb;
	struct inode *dot;
#ifdef OS2
        unsigned short  time;
        unsigned short  date;
#endif
KERNEL_PRINTF("xmsdos_create_a_dotdir");

	PRINTK(("xmsdos_create_a_dotdir 1\n"));

	/*
	 * XXX all times should be set by caller upon successful completion.
	 */
	dir->i_ctime = dir->i_mtime = CURRENT_TIME;
	dir->i_dirt = 1;
	memcpy(de->name,name,MSDOS_NAME);
	memset(de->unused, 0, sizeof(de->unused));
	de->lcase = 0;
	de->attr = ATTR_DIR;
	de->start = 0;
#ifndef OS2
	date_unix2dos(dir->i_mtime,&de->time,&de->date);
#else
	date_unix2dos(dir->i_mtime,__StackToFlat(&time), __StackToFlat(&date));
        de->time = time;
        de->date = date;
#endif

	de->size = 0;
	mark_buffer_dirty(bh, 1);
	if ((dot = iget(dir->i_sb,ino)) != NULL)
		xmsdos_read_inode(dot);
	if (!dot) return -EIO;
	dot->i_mtime = dot->i_atime = dot->i_ctime = CURRENT_TIME;
	dot->i_dirt = 1;
	if (isdot) {
		dot->i_size = dir->i_size;
		MSDOS_I(dot)->i_start = MSDOS_I(dir)->i_start;
		dot->i_nlink = dir->i_nlink;
	} else {
		dot->i_size = parent->i_size;
		MSDOS_I(dot)->i_start = MSDOS_I(parent)->i_start;
		dot->i_nlink = parent->i_nlink;
	}

	iput(dot);

	PRINTK(("xmsdos_create_a_dotdir 2\n"));
	return 0;
}

int xmsdos_create_dotdirs(struct inode *dir, struct inode *parent)
{
	struct super_block *sb = dir->i_sb;
	int res;
	struct buffer_head *bh;
	struct msdos_dir_entry *de;
	loff_t offset;

KERNEL_PRINTF("xmsdos_create_dotdirs");

	PRINTK(("xmsdos_create_dotdirs 1\n"));
	if ((res = xmsdos_add_cluster(dir)) < 0) return res;

	PRINTK(("xmsdos_create_dotdirs 2\n"));
	offset = 0;
	bh = NULL;
	if ((res = xmsdos_get_entry(dir,__StackToFlat(&offset),__StackToFlat(&bh),__StackToFlat(&de))) < 0) return res;
	
	PRINTK(("xmsdos_create_dotdirs 3\n"));
	res = xmsdos_create_a_dotdir(dir, parent, bh, de, res, MSDOS_DOT, 1);
	PRINTK(("xmsdos_create_dotdirs 4\n"));
	if (res < 0) {
		brelse(bh);
		return res;
	}
	PRINTK(("xmsdos_create_dotdirs 5\n"));

	if ((res = xmsdos_get_entry(dir,__StackToFlat(&offset),__StackToFlat(&bh),__StackToFlat(&de))) < 0) {
		brelse(bh);
		return res;
	}
	PRINTK(("xmsdos_create_dotdirs 6\n"));

	res = xmsdos_create_a_dotdir(dir, parent, bh, de, res, MSDOS_DOTDOT, 0);
	PRINTK(("xmsdos_create_dotdirs 7\n"));
	brelse(bh);

	return res;
}

int xmsdos_mkdir(struct inode *dir,const char *name,int len,int mode)
{
	struct inode *inode;
	int res;

KERNEL_PRINTF("xmsdos_mkdir");

	lock_creation();
	if ((res = xmsdos_create_entry(dir,name,len,1,__StackToFlat(&inode))) < 0) {
		unlock_creation();
		iput(dir);
		return res;
	}

	dir->i_nlink++;
	inode->i_nlink = 2; /* no need to mark them dirty */
	MSDOS_I(inode)->i_busy = 1; /* prevent lookups */

	res = xmsdos_create_dotdirs(inode, dir);
	unlock_creation();
	MSDOS_I(inode)->i_busy = 0;
	iput(inode);
	iput(dir);
	if (res < 0) {
		if (xmsdos_rmdir(dir,name,len) < 0)
			fs_panic(dir->i_sb,"rmdir in mkdir failed");
	}
	return res;
}

/* XXX: This needs to be changed to ignore long filename components */
static int xmsdos_empty(struct inode *dir)
{
	struct super_block *sb = dir->i_sb;
	loff_t pos;
	struct buffer_head *bh;
	struct msdos_dir_entry *de;

KERNEL_PRINTF("xmsdos_empty");

	if (dir->i_count > 1)
		return -EBUSY;
	if (MSDOS_I(dir)->i_start) { /* may be zero in mkdir */
		pos = 0;
		bh = NULL;
		while (xmsdos_get_entry(dir,__StackToFlat(&pos),__StackToFlat(&bh),__StackToFlat(&de)) > -1) {
			/* Skip extended filename entries */
			if (de->attr == ATTR_EXT) continue;

			if (!IS_FREE(de->name) && strncmp(de->name,MSDOS_DOT,
			    MSDOS_NAME) && strncmp(de->name,MSDOS_DOTDOT,
			    MSDOS_NAME)) {
				brelse(bh);
				return -ENOTEMPTY;
			}
		}
		if (bh)
			brelse(bh);
	}
	return 0;
}

static int xmsdos_rmdir_free_ino(struct inode *dir,struct buffer_head *bh,
     struct msdos_dir_entry *de,int ino)
{
	struct super_block *sb = dir->i_sb;
	struct inode *inode;
	int res;

KERNEL_PRINTF("xmsdos_rmdir_free_ino");

	if (ino < 0) return -EINVAL;
	if (!(inode = iget(dir->i_sb,ino))) return -ENOENT;
	if (!S_ISDIR(inode->i_mode)) {
		iput(inode);
		return -ENOTDIR;
	}
	if (dir->i_dev != inode->i_dev || dir == inode) {
		iput(inode);
		return -EBUSY;
	}
	res = xmsdos_empty(inode);
	if (res) {
		iput(inode);
		return res;
	}
	inode->i_nlink = 0;
	inode->i_ctime = dir->i_ctime = dir->i_mtime = CURRENT_TIME;
	dir->i_nlink--;
	inode->i_dirt = dir->i_dirt = 1;
	de->name[0] = DELETED_FLAG;
	mark_buffer_dirty(bh, 1);
	iput(inode);

	return 0;
}

static int xmsdos_unlink_free_ino(struct inode *dir,struct buffer_head *bh,
     struct msdos_dir_entry *de,int ino,int nospc)
{
	struct super_block *sb = dir->i_sb;
	struct inode *inode;

KERNEL_PRINTF("xmsdos_unlink_free_ino");

	if (!(inode = iget(dir->i_sb,ino))) return -ENOENT;
	if (!S_ISREG(inode->i_mode) && nospc) {
		iput(inode);
		return -EPERM;
	}
	inode->i_nlink = 0;
	inode->i_ctime = dir->i_ctime = dir->i_mtime = CURRENT_TIME;
	MSDOS_I(inode)->i_busy = 1;
	inode->i_dirt = dir->i_dirt = 1;
	de->name[0] = DELETED_FLAG;
	mark_buffer_dirty(bh, 1);

	iput(inode);
	return 0;
}

static int xmsdos_remove_entry(struct inode *dir,struct slot_info *sinfo,
     struct buffer_head **bh,struct msdos_dir_entry **de,
     int is_dir,int nospc)
{
	struct super_block *sb = dir->i_sb;
	loff_t offset;
	int res, i;

KERNEL_PRINTF("xmsdos_remove_entry");

	/* remove the shortname */
	offset = sinfo->shortname_offset;
	res = xmsdos_get_entry(dir, __StackToFlat(&offset), bh, de);
	if (res < 0) return res;
	if (is_dir) {
		res = xmsdos_rmdir_free_ino(dir,*bh,*de,res);
	} else {
		res = xmsdos_unlink_free_ino(dir,*bh,*de,res,nospc);
	}
	if (res < 0) return res;
		
	/* remove the longname */
	offset = sinfo->longname_offset;
	for (i = sinfo->long_slots; i > 0; --i) {
		res = xmsdos_get_entry(dir, __StackToFlat(&offset), bh, de);
		if (res < 0) {
			printk("xmsdos_remove_entry: problem 1\n");
			continue;
		}
		(*de)->name[0] = DELETED_FLAG;
		(*de)->attr = 0;
		mark_buffer_dirty(*bh, 1);
	}
	return 0;
}


int xmsdos_rmdirx(struct inode *dir,const char *name,int len)
{
	struct super_block *sb = dir->i_sb;
	int res;
	struct buffer_head *bh;
	struct msdos_dir_entry *de;
	struct slot_info sinfo;

KERNEL_PRINTF("xmsdos_rmdirx");

	bh = NULL;
	res = -EPERM;
	if (name[0] == '.' && (len == 1 || (len == 2 && name[1] == '.')))
		goto rmdir_done;
	res = xmsdos_find(dir,name,len,1,0,0,__StackToFlat(&sinfo));

	if (res >= 0 && sinfo.total_slots > 0) {
		res = xmsdos_remove_entry(dir,__StackToFlat(&sinfo),__StackToFlat(&bh),__StackToFlat(&de),1,0);
		if (res > 0) {
			res = 0;
		}
	} else {
		printk("Problem in xmsdos_rmdirx\n");
	}

rmdir_done:
	brelse(bh);
	return res;
}

int xmsdos_rmdir(struct inode *dir,const char *name,int len)
{
	int res;

KERNEL_PRINTF("xmsdos_rmdir");

	res = xmsdos_rmdirx(dir, name, len);
	iput(dir);
	return res;
}

static int xmsdos_unlinkx(
	struct inode *dir,
	const char *name,
	int len,
	int nospc)	/* Flag special file ? */
{
	struct super_block *sb = dir->i_sb;
	int res;
	struct buffer_head *bh;
	struct msdos_dir_entry *de;
	struct slot_info sinfo;

KERNEL_PRINTF("xmsdos_unlinkx");

	bh = NULL;
	if ((res = xmsdos_find(dir,name,len,1,0,0,__StackToFlat(&sinfo))) < 0)
		goto unlink_done;

	if (res >= 0 && sinfo.total_slots > 0) {
		res = xmsdos_remove_entry(dir,__StackToFlat(&sinfo),__StackToFlat(&bh),__StackToFlat(&de),0,nospc);
		if (res > 0) {
			res = 0;
		}
	} else {
		printk("Problem in xmsdos_unlinkx: res=%d, total_slots=%d\n",res, sinfo.total_slots);
	}

unlink_done:
	brelse(bh);
	return res;
}

int xmsdos_unlink(struct inode *dir,const char *name,int len)
{
	int res;

KERNEL_PRINTF("xmsdos_unlink");

	res = xmsdos_unlinkx (dir,name,len,1);
	iput(dir);
	return res;
}
/*
	Special entry for umsdos
*/
int xmsdos_unlink_umsdos(struct inode *dir,const char *name,int len)
{
	int res;
KERNEL_PRINTF("xmsdos_unlink_umsdos");

	res = xmsdos_unlinkx (dir,name,len,0);
	iput(dir);
	return res;
}

int xmsdos_rename(struct inode *old_dir,const char *old_name,int old_len,
	struct inode *new_dir,const char *new_name,int new_len)
{
	struct super_block *sb = old_dir->i_sb;
	struct buffer_head *old_bh,*new_bh,*dotdot_bh;
	struct msdos_dir_entry *old_de,*new_de,*dotdot_de;
	loff_t old_offset,new_offset,old_longname_offset;
	int old_slots,old_ino,new_ino,dotdot_ino;
	struct inode *old_inode, *new_inode, *dotdot_inode;
	int res, is_dir, i;
	int locked = 0;
	struct slot_info sinfo;

KERNEL_PRINTF("xmsdos_rename");

	PRINTK(("xmsdos_rename 1\n"));
	if (old_dir == new_dir && old_len == new_len &&
	    strncmp(old_name, new_name, old_len) == 0)
		return 0;

	old_bh = new_bh = NULL;
	old_inode = new_inode = NULL;
	res = xmsdos_find(old_dir,old_name,old_len,1,0,0,__StackToFlat(&sinfo));
	PRINTK(("xmsdos_rename 2\n"));
	if (res < 0) goto rename_done;

	old_slots = sinfo.total_slots;
	old_longname_offset = sinfo.longname_offset;
	old_offset = sinfo.shortname_offset;
	old_ino = sinfo.ino;
	res = xmsdos_get_entry(old_dir, __StackToFlat(&old_offset), __StackToFlat(&old_bh), __StackToFlat(&old_de));
	PRINTK(("xmsdos_rename 3\n"));
	if (res < 0) goto rename_done;

	if (!(old_inode = iget(old_dir->i_sb,old_ino))) goto rename_done;
	is_dir = S_ISDIR(old_inode->i_mode);

	res = xmsdos_find(new_dir,new_name,new_len,1,0,is_dir,__StackToFlat(&sinfo));

	PRINTK(("xmsdos_rename 4\n"));
	if (res > -1) {
		int is_dir;

		PRINTK(("xmsdos_rename 5\n"));
		/* Filename currently exists.  Need to delete it */
		new_offset = sinfo.shortname_offset;
		res = xmsdos_get_entry(new_dir, __StackToFlat(&new_offset), __StackToFlat(&new_bh), __StackToFlat(&new_de));
		PRINTK(("xmsdos_rename 6\n"));
		if (res < 0) goto rename_done;

		if (!(new_inode = iget(new_dir->i_sb,res)))
			goto rename_done;
		is_dir = S_ISDIR(new_inode->i_mode);
		iput(new_inode);
		if (is_dir) {
			PRINTK(("xmsdos_rename 7\n"));
			res = xmsdos_rmdirx(new_dir,new_name,new_len);
			PRINTK(("xmsdos_rename 8\n"));
			if (res < 0) goto rename_done;
		} else {
			PRINTK(("xmsdos_rename 9\n"));
			res = xmsdos_unlinkx(new_dir,new_name,new_len,1);
			PRINTK(("xmsdos_rename 10\n"));
			if (res < 0) goto rename_done;
		}
	}

	PRINTK(("xmsdos_rename 11\n"));
	lock_creation(); locked = 1;
	res = xmsdos_find(new_dir,new_name,new_len,1,1,is_dir,__StackToFlat(&sinfo));

	PRINTK(("xmsdos_rename 12\n"));
	if (res < 0) goto rename_done;

	new_offset = sinfo.shortname_offset;
	new_ino = sinfo.ino;
	res = xmsdos_get_entry(new_dir, __StackToFlat(&new_offset), __StackToFlat(&new_bh), __StackToFlat(&new_de));
	PRINTK(("xmsdos_rename 13\n"));
	if (res < 0) goto rename_done;

	new_de->attr = old_de->attr;
	new_de->time = old_de->time;
	new_de->date = old_de->date;
	new_de->start = old_de->start;
	new_de->size = old_de->size;

	if (!(new_inode = iget(new_dir->i_sb,new_ino))) goto rename_done;
	PRINTK(("xmsdos_rename 14\n"));

	/* At this point, we have the inodes of the old file and the
	 * new file.  We need to transfer all information from the old
	 * inode to the new inode and then delete the slots of the old
	 * entry
	 */

	xmsdos_read_inode(new_inode);
	MSDOS_I(old_inode)->i_busy = 1;
	cache_inval_inode(old_inode);
	PRINTK(("msdos_rename 15: old_slots=%d\n",old_slots));
	old_inode->i_dirt = 1;

	/* remove the old entry */
	for (i = old_slots; i > 0; --i) {
		res = xmsdos_get_entry(old_dir, __StackToFlat(&old_longname_offset), __StackToFlat(&old_bh), __StackToFlat(&old_de));
		if (res < 0) {
			printk("xmsdos_unlinkx: problem 1\n");
			continue;
		}
		old_de->name[0] = DELETED_FLAG;
		old_de->attr = 0;
		mark_buffer_dirty(old_bh, 1);
	}
	PRINTK(("xmsdos_rename 15b\n"));

	mark_buffer_dirty(new_bh, 1);
	iput(new_inode);
	/* XXX: There is some code in the original MSDOS rename that
	 * is not duplicated here and it might cause a problem in
	 * certain circumstances.
	 */
	
	if (S_ISDIR(old_inode->i_mode)) {
		if ((res = xmsdos_scan(old_inode,MSDOS_DOTDOT,__StackToFlat(&dotdot_bh),
		    __StackToFlat(&dotdot_de),__StackToFlat(&dotdot_ino))) < 0) goto rename_done;
		if (!(dotdot_inode = iget(old_inode->i_sb,dotdot_ino))) {
			brelse(dotdot_bh);
			res = -EIO;
			goto rename_done;
		}
		dotdot_de->start = MSDOS_I(dotdot_inode)->i_start =
		    MSDOS_I(new_dir)->i_start;
		dotdot_inode->i_dirt = 1;
		mark_buffer_dirty(dotdot_bh, 1);
		old_dir->i_nlink--;
		new_dir->i_nlink++;
		/* no need to mark them dirty */
		dotdot_inode->i_nlink = new_dir->i_nlink;
		iput(dotdot_inode);
		brelse(dotdot_bh);
	}

	if (res > 0) res = 0;

rename_done:
	if (locked) unlock_creation();
	if (old_bh) brelse(old_bh);
	if (new_bh) brelse(new_bh);
	if (old_inode) iput(old_inode);
	iput(old_dir);
	iput(new_dir);
	return res;
}

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 8
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -8
 * c-argdecl-indent: 8
 * c-label-offset: -8
 * c-continued-statement-offset: 8
 * c-continued-brace-offset: 0
 * End:
 */
