/****************************************************************************
*
*  mo640nt.cpp
*
*  Utility to convert FAT-Superfloppies with more than 512 bytes per sector
*  for usage with N512DASD.FLT - WinNT release
*
*  Release: 0.9a
*  Update:  01.08.2001
*  (C)      Marcel Mller 1999
*
****************************************************************************/


#include <windows.h>

#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <ctype.h>
#include <stdlib.h>


static int ssdest = 11;
static int rserial = 0;
static char* dev = NULL;


char* strdupcat(const char* s1, const char* s2)
{	register size_t l1 = strlen(s1);
	register size_t l2 = strlen(s2) +1;
	char* r = (char*)malloc(l1 + l2);
	if	(r)
   {	memcpy(r, s1, l1);
		memcpy(r + l1, s2, l2);
	}
	return r;
}


int sscheck(int ss)
{  switch(ss)
   {case 256:
      return 8;
    case 512:
      return 9;
    case 1024:
      return 10;
    case 2048:
      return 11;
    default:
      return -1;  // error
}  }

int main(int argc, char** argv)
{  if (argc < 2)
   {  puts("MO640 V 0.9\n"
           "(C) Marcel Mller 1999-2001\n\n"
           "usage: MO640 <driveletter>: [options]\n\n"
           "options: /s#  change logical sector size to #, default 2048");
      return EXIT_FAILURE;
   }

   // parse command line
   while (--argc)
   {  char* cp = argv[argc];
      switch (cp[0])
      {case '/':
       case '-':
         switch (toupper(cp[1]))
         {case 'N':
            rserial = 1;
            continue;
          case 'S':
            ssdest = 0;
            sscanf(cp+2, "%u", &ssdest);
            ssdest = sscheck(ssdest);
            if (ssdest == -1)
            {  fprintf(stderr, "Sectorsize %u not supported.\n", ssdest);
               return 10;
            }
            continue;
          default:
            fprintf(stderr, "Unrecognized option %s.\n", cp);
            return 11;
         }
       default:
         if (dev)
         {  fprintf(stderr, "Command line syntax error at %s.\n", cp);
            return 11;
         }
         dev = cp;
   }  }

   char *devname = strdupcat("\\\\.\\", argv[1]);
      
   // open device
   HANDLE f = CreateFile(devname, GENERIC_READ|GENERIC_WRITE, FILE_SHARE_READ, NULL,
    OPEN_EXISTING, 0, NULL);
   if (f == INVALID_HANDLE_VALUE)
   {  fprintf(stderr, "Failed to open device %s for writing. - Code %li.\n", devname, GetLastError());
      free(devname);
      return 9;
   }

   try
   {  // read 1st block
      #pragma pack(1)
      struct bootblock
      {  char           branch[3];
         char           oemid[8];
         unsigned short bps;
         unsigned char  spc;
         unsigned short res;
         unsigned char  nfats;
         unsigned short ndirs;
         unsigned short nsects;
         unsigned char  media;
         unsigned short spf;
         unsigned short spt;
         unsigned short nsides;
         unsigned long  nhid;
         unsigned long  lnsects;
         unsigned short physdrv;
         unsigned char  sig;
         unsigned long  serial;
         char           name[11];
         char           fsid[8];
         char           bootcode[448];
         unsigned short magic;
         char           space[3*512];
      } boot;

      DWORD l1;
      if (ReadFile(f, &boot, 2048, &l1, NULL) == FALSE)
      {  fprintf(stderr, "Failed to read sector 0. - Code %li.\n", GetLastError());
         throw 6;
      }

      // checkpoint
      if (boot.magic != 0xaa55)
      {  fprintf(stderr, "Error: illagal magic code %x. - Sector 0 seems to contain no valid data.\n", boot.magic);
         throw 5;
      }
      if (strncmp(boot.fsid, "FAT", 3))
      {  fprintf(stderr, "Error: illegal filesystem-ID %.8s. - This tool is only designed for FAT superfloppy volumes.\n", boot.fsid);
         throw 4;
      }

      // display parameters
      printf("\nBoot block information:\n"
       "  File system ID:        %.8s\n"
       "  OEM ID:                %.8s\n"
       "  Bytes per sector:      %i\n"
       "  Sectors per cluster:   %i\n"
       "  Reserved sectors:      %i\n"
       "  Number of FATs:        %i\n"
       "  Size of root directory:%i\n"
       "  Total sectors (short): %i\n"
       "  Media byte:            %X\n"
       "  Sectors per FAT:       %i\n"
       "  Sectors per track:     %i\n"
       "  Heads:                 %i\n"
       "  Hidden sectors:        %lu\n"
       "  Total sectors (long):  %lu\n"
       "  Serial number:         %lX\n"
       "  Volume name:           %.11s\n"
      , boot.fsid, boot.oemid, boot.bps, boot.spc, boot.res, boot.nfats, boot.ndirs, boot.nsects, boot.media
      , boot.spf, boot.spt, boot.nsides, boot.nhid, boot.lnsects, boot.serial, boot.name);

      // checkpoint 2
      int shift = sscheck(boot.bps);
      if (shift == -1)
      {  fprintf(stderr, "Sector size %u cannot be converted.\n", boot.bps);
         throw 3;
      }
      // log2 of conversion ratio
      shift -= ssdest;
      if (shift == 0)
      {  printf("\nDevice has %u bytes per sector. No changes made.\n", boot.bps);
         throw 0;
      }

      // direction
      if (shift > 0)
      {  // reduce sector size
         boot.bps >>= shift;
         boot.spc <<= shift;
         boot.res <<= shift;
         boot.spf <<= shift;
         boot.nsects <<= shift;
         boot.nhid <<= shift;
         boot.lnsects <<= shift;

      } else
      {  // increase sector size
         shift = -shift;
         int mask = (1 << shift) -1;
         // check
         if ( boot.spc & mask || boot.res & mask || boot.spf & mask
             || boot.nsects & mask || boot.nhid & mask || boot.lnsects & mask )
         {  fprintf(stderr, "Conversion impossible, because not all entries are multiples of %u.\n", mask+1);
            throw 2;
         }
         boot.bps <<= shift;
         boot.spc >>= shift;
         boot.res >>= shift;
         boot.spf >>= shift;
         boot.nsects >>= shift;
         boot.nhid >>= shift;
         boot.lnsects >>= shift;
      }

      // serial number
      if (rserial)
         boot.serial = rand() + (rand() << 11) + (rand() << 22);

      // reset file pointer
      if (SetFilePointer(f, 0, NULL, FILE_BEGIN) == -1)
      {  fprintf(stderr, "Failed to reset file pointer - Code %li.\n", GetLastError());
         throw 1;
      }
      // save
      if (WriteFile(f, &boot, 2048, &l1, NULL) == FALSE)
      {  fprintf(stderr, "Failed to write sector 0. - Code %li.\n", GetLastError());
         throw 1;
      }

      // message
      printf("\nParameters changed:\n"
       "  Bytes per sector:      %i\n"
       "  Sectors per cluster:   %i\n"
       "  Reserved sectors:      %i\n"
       "  Sectors per FAT:       %i\n"
       "  Hidden sectors:        %lu\n"
       "  Total sectors (short): %i\n"
       "  Total sectors (long):  %lu\n"
       "  Serial number:         %lX\n"
       "\nDevice successfully converted.\n"
      , boot.bps, boot.spc, boot.res, boot.spf, boot.nhid, boot.nsects, boot.lnsects, boot.serial);

      throw EXIT_SUCCESS;
   } catch (int r)
   {  // free device
      CloseHandle(f);
      return r;
}  }
