/****************************************************************************
*
*  mo640.cpp
*
*  Utility to convert FAT-Superfloppies with more than 512 bytes per sector
*  for usage with N512DASD.FLT
*
*  Release: 1.0
*  Update:  16.02.2002
*  (C)      Marcel Mller 1999-2002
*
****************************************************************************/


#define INCL_BASE
#define INCL_DOSDEVIOCTL
#include <os2.h>

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <stddef.h>
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>


static int ssdest = 9;
static bool rserial = false;
static bool halterr = false;
static bool force = true; // removable flag is not checked by OS2DASD !
static char* dev = NULL;

// Error handling
struct Error_dat
{  int err;
   APIRET rc;
   char buf[1024];
};

struct Error_ref
:  Error_dat
{  explicit Error_ref(Error_dat &ref) : Error_dat(ref) {*ref.buf = 0;}
};

struct Error
:  Error_dat
{  static const char head[2][8];
   explicit Error(int err, APIRET rc, const char* fmt, ...);
   Error(Error &e) : Error_dat(e) { *e.buf = 0; }
   Error(Error_ref ref) : Error_dat(ref) {}
   ~Error();
   operator Error_ref() { return Error_ref(*this); }
   operator int() { return err; }
};

const char Error::head[2][8] = {"Warning", "Error"};

Error::Error(int err, APIRET rc, const char* fmt, ...)
{  this->err = err;
   this->rc = rc;
   va_list va;
   va_start((void*)va, fmt);
   vsprintf(buf, fmt, va);
   va_end(va);
}

Error::~Error()
{  if (!*buf)
      return;
   fprintf(stderr, "%s: %s", head[err>0], buf, rc);
   if (rc != NO_ERROR)
      fprintf(stderr, " - Code %lu.", rc);
   fputc('\n', stderr);
   if (halterr && err > 0)
   {  fputs("\nPress any key to continue.\n", stderr);
      getch();
}  }

// check sector size and return log2(size)
int sscheck(int ss)
{  switch(ss)
   {case 256:
      return 8;
    case 512:
      return 9;
    case 1024:
      return 10;
    case 2048:
      return 11;
    default:
      return -1;  // error
}  }

// bootblock layout
#pragma pack(1)
struct bootblock
{  char           branch[3];
   char           oemid[8];
   unsigned short bps;
   unsigned char  spc;
   unsigned short res;
   unsigned char  nfats;
   unsigned short ndirs;
   unsigned short nsects;
   unsigned char  media;
   unsigned short spf;
   unsigned short spt;
   unsigned short nsides;
   unsigned long  nhid;
   unsigned long  lnsects;
   unsigned short physdrv;
   unsigned char  sig;
   unsigned long  serial;
   char           name[11];
   char           fsid[8];
   char           bootcode[448];
   unsigned short magic;
};

int main(int argc, char** argv)
{  puts("MO640 V 1.0\n"
        "(C) Marcel Mller 1999-2002\n");

   HFILE dh = (HFILE)-1;
   bool locked = false;
   ULONG l1,l2;
   APIRET rc;
   unsigned char buf1[8];
   unsigned char buf2[8];

   try
   {  // parse command line
      while (--argc)
      {  char* cp = argv[argc];
         switch (cp[0])
         {case '/':
          case '-':
            switch (toupper(cp[1]))
            {case 'N':
               rserial = true;
               continue;
             case 'F':
               force = true;
               continue;
             case 'E':
               halterr = true;
               continue;
             case 'S':
               ssdest = 0;
               sscanf(cp+2, "%u", &ssdest);
               ssdest = sscheck(ssdest);
               if (ssdest == -1)
                  throw Error(10, NO_ERROR, "Sectorsize %u not supported.", ssdest);
               continue;
             default:
               throw Error(11, NO_ERROR, "Unrecognized option %s.", cp);
            }
          default:
            if (dev)
               throw Error(11, NO_ERROR, "Command line syntax error at %s.", cp);
            dev = cp;
      }  }

      if (!dev)
         throw Error(12, NO_ERROR,
          "usage: MO640 <driveno>: [options]\n"
          "options: /s#  change logical sector size to #, default 512.");

      // open device
      rc = DosOpen((PUCHAR)dev, &dh, &l1, 0, 0, OPEN_ACTION_OPEN_IF_EXISTS,
       OPEN_FLAGS_DASD|OPEN_FLAGS_NO_CACHE|OPEN_FLAGS_FAIL_ON_ERROR|OPEN_SHARE_DENYREADWRITE|OPEN_ACCESS_READWRITE,
       NULL);
      if (rc != NO_ERROR)
         throw Error(9, rc, "Failed to get handle for DASD %s.", dev);

      // lock device
      *buf1 = 0; l1 = 1; *buf2 = 0; l2 = 1;
      rc = DosDevIOCtl(dh, IOCTL_DISK, DSK_LOCKDRIVE, buf1, sizeof buf1, &l1, buf2, sizeof buf2, &l2);
      if (rc != NO_ERROR)
         throw Error(8, rc, "Failed to lock drive %s.", dev);
      locked = true;

      // read bootblock
      bootblock boot;
      rc = DosRead(dh, &boot, sizeof boot, &l1);
      if (rc != NO_ERROR)
         throw Error(7, rc, "Failed to read sector 0 of %s.", dev);

      // checkpoint - signatures
      if (boot.magic != 0xaa55)
         throw Error(5, NO_ERROR, "Illagal magic code %x. - Sector 0 seems to contain no valid data.", boot.magic);

      if (strncmp(boot.fsid, "FAT", 3))
         throw Error(4, NO_ERROR, "Illegal filesystem-ID %.8s. - This tool is only designed for FAT superfloppy volumes.", boot.fsid);

      // display parameters
      printf("Boot block information:\n"
       "  File system ID:        %.8s\n"
       "  OEM ID:                %.8s\n"
       "  Bytes per sector:      %i\n"
       "  Sectors per cluster:   %i\n"
       "  Reserved sectors:      %i\n"
       "  Number of FATs:        %i\n"
       "  Size of root directory:%i\n"
       "  Total sectors (short): %i\n"
       "  Media byte:            %X\n"
       "  Sectors per FAT:       %i\n"
       "  Sectors per track:     %i\n"
       "  Heads:                 %i\n"
       "  Hidden sectors:        %lu\n"
       "  Total sectors (long):  %lu\n"
       "  Serial number:         %lX\n"
       "  Volume name:           %.11s\n\n"
      , boot.fsid, boot.oemid, boot.bps, boot.spc, boot.res, boot.nfats, boot.ndirs, boot.nsects, boot.media
      , boot.spf, boot.spt, boot.nsides, boot.nhid, boot.lnsects, boot.serial, boot.name);

      // checkpoint 2 - removable?
      if (!force)
      {  *buf1 = 0; l1 = 2; l2 = 1;
         rc = DosDevIOCtl(dh, IOCTL_DISK, DSK_BLOCKREMOVABLE, buf1, sizeof buf1, &l1, buf2, sizeof buf2, &l2);
         if (rc != NO_ERROR)
            throw Error(6, rc, "Failed to check removable property.", dev);
         if (*buf2)
            throw Error(6, NO_ERROR, "You do not really want to convert a fixed disk.", dev);
      }

      // checkpoint 3
      int shift = sscheck(boot.bps);
      if (shift == -1)
         throw Error(3, NO_ERROR, "Sector size %u cannot be converted.", boot.bps);

      // log2 of conversion ratio
      shift -= ssdest;
      if (shift == 0)
         throw Error(0, NO_ERROR, "Device %s has %u bytes per sector. No changes made.", dev, boot.bps);

      // direction
      if (shift > 0)
      {  // reduce sector size
         boot.bps >>= shift;
         boot.spc <<= shift;
         boot.res <<= shift;
         boot.spf <<= shift;
         boot.nsects <<= shift;
         boot.nhid <<= shift;
         boot.lnsects <<= shift;

      } else
      {  // increase sector size
         shift = -shift;
         int mask = (1 << shift) -1;
         // check
         if ( boot.spc & mask || boot.res & mask || boot.spf & mask
             || boot.nsects & mask || boot.nhid & mask || boot.lnsects & mask )
            throw Error(2, NO_ERROR, "Conversion impossible, because not all entries are multiples of %u.", mask+1);
         boot.bps <<= shift;
         boot.spc >>= shift;
         boot.res >>= shift;
         boot.spf >>= shift;
         boot.nsects >>= shift;
         boot.nhid >>= shift;
         boot.lnsects >>= shift;
      }

      // serial number
      if (rserial)
         boot.serial = rand() + (rand() << 11) + (rand() << 22);

      // save bootblock
      rc = DosSetFilePtr(dh, 0, FILE_BEGIN, &l1);
      if (rc != NO_ERROR)
         throw Error(1, rc, "Failed to reset file pointer.");
      rc = DosWrite(dh, &boot, sizeof boot, &l1);
      if (rc != NO_ERROR)
         throw Error(1, rc, "Failed to write sector 0 of %s.", dev);

      // message
      printf("Parameters changed:\n"
       "  Bytes per sector:      %i\n"
       "  Sectors per cluster:   %i\n"
       "  Reserved sectors:      %i\n"
       "  Sectors per FAT:       %i\n"
       "  Hidden sectors:        %lu\n"
       "  Total sectors (short): %i\n"
       "  Total sectors (long):  %lu\n"
       "  Serial number:         %lX\n"
       "\nDevice successfully converted.\n"
      , boot.bps, boot.spc, boot.res, boot.spf, boot.nhid, boot.nsects, boot.lnsects, boot.serial);

      // redetermine media
      *buf1 = 0; l1 = 1; *buf2 = 0; l2 = 1;
      rc = DosDevIOCtl(dh, IOCTL_DISK, DSK_REDETERMINEMEDIA, buf1, sizeof buf1, &l1, buf2, sizeof buf2, &l2);
      if (rc != NO_ERROR)
         Error(-1, rc, "Failed to redetermine media.");

      // done!
      throw Error(0, NO_ERROR, "");

   } catch (Error &e)
   {  // unlock device
      if (dh != -1)
      {  if (locked)
         {  *buf1 = 0; l1 = 1; *buf2 = 0; l2 = 1;
            rc = DosDevIOCtl(dh, IOCTL_DISK, DSK_UNLOCKDRIVE, buf1, sizeof buf1, &l1, buf2, sizeof buf2, &l2);
            if (rc != NO_ERROR)
               Error(-1, rc, "Failed to unlock drive %s.", dev);
         }
         // close
         rc = DosClose(dh);
         if (rc != NO_ERROR)
            Error(-1, rc, "Failed to close drive %s.", dev);
      }
      return e;
}  }

