/* V_MULTI.CMD - Voice/Fax Answer Script for FaxWorks/PMfax (REXX) */
/* 
                  Keller Group Inc.  June 19, 1997
   You may freely use or modify this script for use with licensed
        products which are developed by Keller Group Inc.
      For new scripts or to share your scripts with others,
                 see http://www.kellergroup.com

   Modification history:
      05May97 - Added SysSleep to paging to make sure line is hung up.
      06Jun97 - NOTE: Uses new FxRxFind command (requires 3.01.01 !!)
      19Jun97 - Added reset command ('*')
      19Jun97 - Changed termination after FOD calls - no log entry
*/
/*******************************************************************
V_MULTI.CMD - Multiple Mailbox (Standalone) voice/fax mailbox system

For use with the following Keller software configurations:
  Stand-alone (single-line or multiline)
  LAN Shared Mode
  LAN Private Mode (with fax server as router & retrieval from server log)

Features:

  Callers can:
     select a mailbox using an extension
     hear both a "company" greeting and the extension's "personal" greeting
     send a fax (to an extension, or by default to mailbox 0),
     leave a voice message (to an extension, or by default to mailbox 0),
     listen to an announcement, or
     enter extension and private password to enter remote retrieval mode

  In the password-protected remote retrieval mode, you can do:
     greeting modification - remotely update your personal greeting message
     password modification - remotely modify your personal access code
     voice retrieval - get voice messages by phone
     fax retrieval - get fax messages, or send them to a fax number
     log retrieval - get a fax of your log (or the last part of it)
     "fax on demand" from log - retrieve voice and fax items from your log

  New features:
     20May97 - Switched to FxRxFind Next/Prev functions to allow retrieval
               of old messages (not just new ones) using the PrevMsg command.


DESCRIPTION:
-----------
This script is a "multiple mailbox" voice/fax system which will work
with a standalone copy of PMfax/FaxWorks.  When messages or faxes are
received, the notes field in the log will indicate the mailbox
extension.  Mailbox users can call and retrieve their messages and
faxes from the system for their specific mailbox extension.

The script first plays your outgoing message (OGM).  The caller can
enter a "mailbox extension" followed by the '#' key to hear that
mailbox's personal outgoing message and then record a message or send
a fax to that mailbox.  If the caller doesn't enter any extension or
enters '0#' or '#', the caller can record a message to the
"receptionist" at mailbox '0'.

The caller can press the # or * key to skip the outgoing message and
immediately record their message, otherwise the script will provide a
beep and record the caller's message if no further action is taken.

If the call is from a fax device and your fax hardware detects the
caller's fax (CNG) tones, the script automatically receives the fax.

If the caller enters '2' during playing of the outgoing message, the
script plays the "announcement" and then returns to the outgoing
message again so that the caller can leave a message on the same
call.  If a mailbox extension was entered, the mailbox's personal
announcement is played, otherwise the system's main announcement is
played.

If the caller enters '9' during or immediately after the playing of
the OGM and then enters the user-configurable access code followed by
the # key, the script enters its "remote retrieval" mode so the
caller can hear received voice messages, have received faxes sent to
them via fax, and other features.  When in this mode, the keys on the
telephone do the following:

   1 - HELP       - Play help message which describes these commands
   2 - SET CODE   - Changes LAN user's private access code (default: 42)
   3 - GREETING   - Enter new outgoing message (change greeting)

   4 - PREV MSG   - Play previous voice message
   5 - REPLAY MSG - Play current voice message (again)
   6 - NEXT MSG   - Play next voice message

   7 - GET FAXES  - Retrieve new fax messages by fax
   8 - GET LOG    - Retrieve fax log by fax
   9 - FOD LOG    - Retrieve voice/fax msgs from log ("fax-on-demand")

   0 - TOP        - Return to top level and play outgoing message again

   * - RESET      - Marks all "new" messages/faxes as "retrieved"


In summary, for multiple mailbox use:

1) You will want to create an outgoing message (OGM) which instructs
your callers to "Enter the mailbox extension number followed by the #
key".  When they record a message or send a fax, it will than appear
in the log with a Notes entry of "mbx1234" where 1234 is the
extension number.

2) To call in and retrieve the messages and/or faxes for extension
1234, enter 1234# during or just after the OGM, then press 9 to enter
the remote retrieval mode.  Enter the access code (which defaults to
42, but can be changed by each user once in the remote retrieval
mode).  The system will beep once for each new voice message.  You
can then use '5' to hear the first voice message, '4' to hear older
voice messages (even ones that aren't "new" anymore), and other keys
to change your access code, change your mailbox outgoing message, and
get your faxes.  You will only get those messages and faxes which
match the extension.


USAGE INSTRUCTIONS:
------------------
You must be using the program with supported voice/fax hardware to
use this script.  Consult the README.DOC file for tested voice/fax
hardware and recommended configuration settings for your hardware.
Before using this script, first test your system with our built-in
voice answering machine feature by enabling "Voice" and setting the
"Answer script" to * (an asterisk character) on the Voice page of the
Settings notebook.  If you are using supported voice/fax hardware
with the proper configuration settings, you can then use
"Fax/Receive/All calls" mode to answer calls, play the outgoing
message (OGM), and record a voice message (or receive a fax).

Then, to change to this advanced script, specify the full pathname of
this file in the "Answer script" field on the Voice page of the
settings notebook.

NOTE: The script expects to find the LIB_FOD.CMD file and its Wave
audio files in the fax program (EXE) directory.  The script also uses
Wave files from the c:\mmos2\sounds directory for special effects.

SCRIPT VARIABLES:
----------------
The following script variable is used by this script.  Script
variables are set on the Voice page of the Settings Notebook.

General:

	OGM - Pathname of the main "outgoing message" which is played
	to the caller.  By default, the program sets OGM to be the
	file OGM.WAV in the program directory.
	Example: OGM = C:\FAX\OGM.WAV
	[Mailbox users can also place a personal MBXnnnn.WAV file in the
        program directory or record one using remote access command '3'.]

	CODE - Access code for retrieving messages or changing
	settings from the log.  By default, the program
	sets CODE to be 42.  You can change this to be any number of up
	to 5 digits in length.
	Example: CODE = 42
	[Mailbox users can also set a personal access code for their mailbox
	using the '2' command when they call in, and this personal value
	is saved in the MBXnnnn.COD file in the program directory.  If no
	personal access code is set, the default value is 42.]

	CALLER - (optional, default "Person") - Name of the person/company
	who will be using the access code to retrieve messages/faxes
	by phone.  This is used for addressing faxes when the caller
	uses the access code for remote retrieval.  
	Example: CALLER = Your Company Inc.

	LOGMAXLINE - (optional, default 56) - Maximum number of log
	entries, from the end of the log, to be sent to the caller
	when they retrieve the log by fax.  If set to 0, then
	the entire log is sent.
	Example: LOGMAXLINE = 120


VOICE PROMPT FILES:
------------------
The script expects to find the following Wave audio files in the fax
program (EXE) directory.  The script also uses Wave files from the
c:\mmos2\sounds directory for special effects.

To record or modify your own voice prompt files, use your OS/2
Multimedia microphone and record voice prompts using the "Fax/New
message" command then save them to a .WAV file with the "Fax/Save
file/Wave" command.  You can also use the OS/2 Digital Audio program
to record, save, edit and modify the Wave files (be sure to use
"Type" of Mono, 8-bit, 11.025 kHz).

	OGM.WAV - The main "outgoing message" as recorded with the
	Utilities/Outgoing message command.

	ENTR_COD.WAV - Short: "Enter code."
	Long: "Please enter your access code and the # key."

	ENTR_CMD.WAV - Short: "Enter command."
	Long: "To play current message press 5, to play next message
	press 6, to play previous message press 4...."

	ENTR_MSG.WAV - Short: "Enter message, then press # key."
	Long: "Please leave your message at the beep.  Press # key to end."

	ANNOUNC.WAV - Short: "No announcement."
	Long: "Our annoucement for this week is...."

	HELP.WAV - "Press 3 to change outgoing message, 4 for previous..."

	NO_MSGS.WAV - Short: "No more messages."

And used by any fax retrieval actions (LIB_FOD.CMD procedure):

	ENTR_DOC.WAV - Short: "Enter document number and # key, or
	press the # key if done."
	Long: "Please enter the document number followed by the # key,
	or press the # key if done.  To receive an index of available
	documents, enter document 1000."

	HOW_FAX.WAV - Short: "Enter 1 if calling from fax machine, 2
	if not."
	Long: "If you are calling from your fax machine and are ready
	to receive the fax press 1, to send the fax to a different
	number press 2." 

	ENTR_FAX.WAV - Short: "Enter fax number and # key."
	Long: "Please enter your fax number followed by the # key,
	including 1 and area code if this is not a local call." 

	NO_FILE.WAV - Short: "File not found."
	Long: "That document was not found, please try again."

	WAIT.WAV - Short: "Please wait."
	Long: "Please wait while we process your request."

	THANKYOU.WAV - Short: "Thank you."

TO MODIFY THE SCRIPT:
--------------------
The script is REXX and uses calls from the Keller REXX API which are
automatically loaded into the OS/2 REXX environment by the retail
versions of Keller's fax products (version 3.0 and later).  See the
Reference Manual for documentation of the FxRx and FxLn calls.  You
may modify the script as desired, and then use it as the "Answer
script" with Keller fax software products.

Note - For retrieval, this script uses the READ/N_READ mode of
FxRxFind and the new "NOTES=" parameter of version 3.01.01 and later.
Using READ/N_READ mode means it will report all faxes/messages for
the matching mailbox userid which have not been previously retrieved
via phone (even those that have been viewed at the computer).  If
desired, you can change this to use RCVD/N_RCVD mode of FxRxFind,
which means that it will report only those faxes which have been
received but not already viewed at the computer and not previously
retrieved via phone (but this is less reliable since "error" faxes
won't be included).  You can also retrieve any desired fax, whether
or not it has been viewed or retrieved, using the "fax log" and
"fax-on-demand" features in this script which allow you to retrieve
any fax by its index number.  In version 3 of the fax program,
viewing a received fax changes its status from Rcvd to Read, but
printing of faxes does NOT change the status, so you can auto-print
all received faxes at the computer and still retrieve them as "Rcvd"
faxes by phone. 

TO USE WITH MULTILINE VERSIONS:
------------------------------
This script can be used with multiline versions of Keller fax
software.  All lines are handled identically, so calls can be
received on any of your "receive" lines.  If you prefer to have
different lines be handled differently you can modify the script to
test the LID variable.  For example, to have line 1 immediately jump
to the "please enter your message" prompt, you could add a line like
the following after the "call FxLnInit" line in the script: "if LID =
1 then signal voice" where "voice" is a label in the script.

*******************************************************************/

  call FxLnInit

  VOXDIR = FxRxPath( 'EXE' )	/* get voice prompts from exe dir */
  LIBDIR = FxRxPath( 'EXE' )	/* get REXX subroutines from exe dir */
  FODPROCEDURE = LIBDIR||'lib_fod.cmd'  /* use this external procedure */

  /* default values for script variables */
  if CALLER = 'CALLER' then CALLER = 'Person'
  if LOGMAXLINE = 'LOGMAXLINE' then LOGMAXLINE = 56

  DEFAULTCODE = '42'		/* default access code */

hello:
  LREC. = ""				/* be sure this is clean */
  USERID = '0'				/* default/receptionist is mbx 0 */
  USERDIR = VOXDIR||'MBX'||USERID	/* just add extension later */

  /* If using Brooktrout, you may need to use certain lines for fax since */
  /* it has trouble hearing calling fax CNG tones when you are using      */
  /* voice and DTMF commands. For example... */
  /* if LID = 1 then signal fax */

  call FxLnMsg 'OGM'				/* show in status window */
	if result = 'NOTOK' then signal byebye
  call FxLnPlay OGM				/* play greeting prompt */
	if result = 'NOTOK' then signal byebye
	if result = 'FAX' then signal fax
	if result = 'DATA' then signal data

  digits = GetExtension( 5, 10, 5 )		/* get extension or number */
	if digits = 'NOTOK' then signal byebye
	if digits = 'FAX' then signal fax
	if digits = 'DATA' then signal data

  if digits = '' | digits = '0' then signal voice
  if digits = 2 then signal announce		/* use key value... */
  if digits = 9 then signal accesscode

  /* else they must have entered an extension... */
  USERID = digits
  USERDIR = VOXDIR||'MBX'||USERID		/* just add extension later */
  if stream(USERDIR||'.WAV', 'C', 'QUERY EXISTS' ) <> ''
  then call FxLnPlay USERDIR||'.WAV'

  call FxLnDtmf 1, 3, 'digits'			/* get key, if any... */
	if result = 'NOTOK' then signal byebye
	if result = 'FAX' then signal fax
	if result = 'DATA' then signal data
	if result <> 'DTMF' then signal voice

  if digits = 2 then signal announce		/* use key value... */
  if digits = 3 then signal fax
  if digits = 9 then signal accesscode


voice:					/* take voice message, hangup */
  call FxLnMsg "Record Msg"
  call FxLnTone 1850
	if result = 'NOTOK' then signal byebye
  call FxLnDtmf 	/* flush buffer */
  call FxLnRecord
	if result = 'NOTOK' then signal byebye
	if result = 'FAX' then signal fax
	if result = 'DATA' then signal data
  /* caller can hang up or press a key to end recording */
  call FxLnTone 1850, 1850, 1850
  if USERID <> '' then do
    LREC.!Notes = 'mbx'||USERID		/* tag the message with a note */
  end
  signal out


fax:					/* receive fax and hangup */
  call FxLnReceive
  if USERID <> '' then do
    LREC.!Notes = 'mbx'||USERID		/* tag the fax with a note */
  end
  signal out


data:					/* report data call, hangup */
  call FxLnMsg 'Data call, oh well...'
  LREC.!Status = '*Data!'   /* log status of call */
  signal out


announce:				/* play annoucement, restart */
  call FxLnMsg "Announcement"
  if stream(USERDIR||'.ANN', 'C', 'QUERY EXISTS' ) <> ''
  then call FxLnPlay USERDIR||'.ANN'
  else call FxLnPlay VOXDIR||'ANNOUNC.WAV'
  signal hello


accesscode:
  call FxLnMsg 'Code ???'		/* get & check access code */
  call FxLnPlay VOXDIR||'ENTR_COD.WAV'
	if result = 'NOTOK' then signal byebye
  digits = GetNumber( 6, 10 )
  if digits <> FetchCode() then signal evil

  call FxLnMsg 'Code Mode'
  call FxLnVout 'Valid code entered'
  call FxRxBeep 354, 393, 295, 354, 0, 393

  call FxRxFind 'MSG', 'N_READ', 'NOTES=mbx'||USERID	/* # msgs */
  if result = 'NOTOK' then do
	call FxLnPlay 'c:\mmos2\sounds\BOING.wav'
	signal byebye
  end
  n = result

  /* if no messages... stay in command mode for other commands */
  if n = 0 then do
        call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	    if result = 'NOTOK' then signal byebye
  end

  i = n				/* Play 1 beep for each voice message */
  call FxLnMsg 'Code Mode' n 'messages'
  call FxLnVout 'Beep' n 'times'
  do while i > 0
    call FxLnTone 2100
    i = i - 1
  end

  call FxLnVout 'Enter message loop'
  i = 0

  /* select the first voice message, or last one if no new msgs */
  if n > 0 then do
    call FxRxFind 'MSG', 'READ', 'NOTES=mbx'||USERID, 'LREC'
    if result <> 'NOTOK' then do
      call FxLnVout 'Found by READ' LREC.!ID
      i = LREC.!ID
    end
  end
  else do
    call FxRxFind 'MSG', 'PREV', 'NOTES=mbx'||USERID, 'LREC'
    if result <> 'OK' then do
      call FxLnVout 'Found by PREV' LREC.!ID
      i = LREC.!ID
    end
  end

  do forever
    call FxLnMsg 'Waiting for next command'
    call FxLnPlay VOXDIR||'ENTR_CMD.WAV'
    call FxLnDtmf 1, 30, 'digits'
    if result <> 'DTMF' then do
      signal goodbye
    end
    else do
      call FxLnMsg 'Code Mode command' digits
      call FxLnVout 'Command' digits 'entered'


      if digits = '1' then do			/* 1 = Play help msg */
        call FxLnVout 'Help message'
        call FxLnPlay VOXDIR||'HELP.WAV'
	  if result = 'NOTOK' then signal byebye
        iterate
      end


      if digits = '2' & USERID > '0' then do	/* 2 = change access code */
        /* Only access codes for MBX users are changed through this. */
        /* For the fax server, use the CODE script variable on Voice page. */
        call FxLnVout 'Change access code for' USERID
        call FxLnMsg 'Code Mode command' digits
        call FxLnPlay VOXDIR||'ENTR_COD.WAV'
          if result = 'NOTOK' then signal byebye
        code1 = GetNumber( 6, 10 )
        call FxLnPlay VOXDIR||'ENTR_COD.WAV'
          if result = 'NOTOK' then signal byebye
        code2 = GetNumber( 6, 10 )
        if code1 = code2 then do
          call SaveCode code1
          call FxLnPlay VOXDIR||'THANKYOU.WAV'
        end
        else do
          call FxLnVout 'New code mismatch:' code1 'vs' code2
          call FxLnPlay 'c:\mmos2\sounds\BOO.wav'
        end
        iterate
      end


      if digits = '3' then do			/* 3 = Enter outgoing msg */
        call FxLnVout 'Enter outgoing message for' USERID
        call FxLnPlay VOXDIR||'ENTR_MSG.WAV'
	  if result = 'NOTOK' then signal byebye
        call FxLnTone 1850
	  if result = 'NOTOK' then signal byebye
        call FxLnDtmf		/* flush buffer */
        if USERID = '0'
        then call FxLnRecord OGM
        else call FxLnRecord USERDIR||'.WAV'
 	  if result = 'NOTOK' then signal byebye
        call FxLnTone 1850, 1850, 1850
        call FxLnDtmf		/* flush buffer */
        if USERID = '0'
        then call FxLnPlay OGM
        else call FxLnPlay USERDIR||'.WAV'
	  if result = 'NOTOK' then signal byebye
        iterate
      end


      if digits = '4' then do			/* 4 = Previous msg */
        call FxLnVout 'Play previous message' i
        if i = 0 
	then call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	else do
	  call FxRxFind 'MSG', 'PREV', 'NOTES=mbx'||USERID, 'LREC'
	  if result = 'OK'
	  then do
	    call FxLnVout 'Find PREV returns' result
	    call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	  end
	  else do
	    call FxLnVout 'Find PREV returns' result
	    call FxLnVout 'Found by PREV' LREC.!ID
	    i = LREC.!ID
	    call FxLnMsg 'Code Mode command' digits 'Msg' i
	    call FxLnPlay i
	  end
        end
        iterate
      end


      if digits = '5' then do			/* 5 = Replay msg */
        call FxLnVout 'Replay message' i
        if i = 0 
	then call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	else do
	  call FxLnMsg 'Code Mode command' digits 'Msg' i
	  call FxLnPlay i
        end
        iterate
      end


      if digits = '6' then do			/* 6 = Next message */
        call FxLnVout 'Play next message' i
        if i = 0 
	then call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	else do
	  call FxRxFind 'MSG', 'NEXT', 'NOTES=mbx'||USERID, 'LREC'
	  if result = 'OK'
	  then do
	    call FxLnVout 'Find NEXT returns' result
	    call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	  end
	  else do
	    call FxLnVout 'Find NEXT returns' result
	    call FxLnVout 'Found by NEXT' LREC.!ID
	    i = LREC.!ID
	    call FxLnMsg 'Code Mode command' digits 'Msg' i
	    call FxLnPlay i
	  end
        end
        iterate
      end


      if digits = '7' then do			/* 7 = send rcvd faxes */
        call FxLnVout 'Send rcvd faxes'
        call FxLnMsg 'Code Mode command' digits
        call FxRxFind 'FAX', 'N_READ', 'NOTES=mbx'||USERID  /* # new faxes */
        if result = 'NOTOK' then do
          call FxLnPlay 'c:\mmos2\sounds\BOING.wav'
          signal byebye
        end
        nfax = result
        if nfax = 0 then do
          call FxLnPlay VOXDIR||'NO_MSGS.WAV'
          if result = 'NOTOK' then signal byebye
        end
        else do
          ifax = nfax		/* Get faxes into list and beep per fax */
          faxlist = ''
          flist = ''
          call FxLnMsg 'Code Mode' nfax 'faxes'
          call FxLnVout 'FaxBeep' nfax 'times'
          do while ifax > 0
            call FxLnTone 2100
            ifax = ifax - 1
            call FxRxFind 'FAX', 'READ', 'NOTES=mbx'||USERID, 'LREC'
            call FxLnVout 'Found' LREC.!ID
            /* Note use of filenames rather than ID values.  This is done */
            /* because FxLnSend, which could be used in the FOD procedure,*/
            /* cannot be given IDs from an FxRxSelect LAN user directory. */
            faxlist = AddToCsvList( FxRxIndexToFax( LREC.!ID ), faxlist )
            flist = flist LREC.!ID
          end
          /* call our FOD procedure to do the faxing */
          FODPARAMS = "FxRxPath( 'LOG' ), VOXDIR, , CALLER, faxlist," ,
            "'As you requested, your received faxes are attached:' flist," ,
            "'Remote fax retrieval...'"
          interpret 'call' "'"||FODPROCEDURE||"'" FODPARAMS
          if result = 'BYEBYE' then signal BYEBYE
          if result = 'OUT' then signal OUT
        end
        iterate
      end


      if digits = '8' then do			/* 8 = send log */
        call FxLnVout 'Send log'
        call FxLnMsg 'Code Mode command' digits
        call FxLnPlay VOXDIR||'WAIT.WAV'
	  if result = 'NOTOK' then signal byebye
        call SendLog
          if result = 'BYEBYE' then signal BYEBYE
          if result = 'OUT' then signal OUT
        iterate
      end


      if digits = '9' then do			/* 9 = FOD from log */
        call FxLnVout 'FOD from log'
        call FxLnMsg 'Code Mode command' digits
        FODPARAMS = "FxRxPath( 'LOG' ), VOXDIR, , CALLER, , ," ,
               "'FOD Retrieval...'"
        interpret 'call' "'"||FODPROCEDURE||"'" FODPARAMS
          if result = 'BYEBYE' then signal BYEBYE
          if result = 'OUT' then signal OUT
        iterate
      end


      if digits = '0' then do			/* 0 = back to top level */
        call FxLnVout 'Return to HELLO'
	signal hello
      end


      if digits = '*' then do			/* * = reset */
        /* This is used to mark as messages and faxes in the log as  */
	/* having been "previously retrieved" so that you don't need */
	/* to get them all via phone.  You can still go back through */
	/* them from the end using "Prev Msg" and "fax-on-demand".   */
        call FxLnVout 'Reset' i
	call FxLnMsg 'Reset command' digits
        call FxLnPlay VOXDIR||'WAIT.WAV'
	  if result = 'NOTOK' then signal byebye
	LREC. = ""		/* forget any previous position in log */
	do forever
	  call FxRxFind 'ANY', 'NEXT', 'NOTES=mbx'||USERID, 'LREC'
          if result = 'OK' | result = 'NOTOK' then leave
	end
	/* Select last voice message as the current message */
	LREC. = ""
	call FxRxFind 'MSG', 'PREV', 'NOTES=mbx'||USERID, 'LREC'
	if result = 'OK'
	then i = 0
	else i = LREC.!ID
	call FxLnVout 'Reset find PREV returns' result
        iterate
      end


      if digits = '#' then do
        signal goodbye
      end


      /* otherwise, if none were matched... */
      call FxLnVout 'Unknown command' i
      call FxLnPlay 'c:\mmos2\sounds\EEERRUPP.wav'
      iterate

    end
  end


goodbye:
  call FxLnVout 'Exit Code Mode'
  call FxLnPlay VOXDIR||'THANKYOU.WAV'
  call FxLnPlay 'c:\mmos2\sounds\DRUMROLL.wav'
  signal byebye


evil:
  call FxLnVout 'Bad code entered:' digits 'vs' FetchCode()
  call FxLnPlay 'c:\mmos2\sounds\BOO.wav'
  call FxRxBeep 1318, 1396, 1318, 1090, 1318, 1396, 1318, 1090
  signal byebye


byebye:					/* called for error cases */
  LREC.!Status = ''			/* don't log anything */
  signal out


out:					/* clean up and quit */
  call FxLnTerm
  exit


/************************ Procedures ************************/


GetNumber: procedure
  /* Gets DTMF input until # or * key or "digits" are entered. */
  /* TimeOut is seconds to wait for each digit (default 15).       */
  parse arg digits, TimeOut
  number = ''
  if TimeOut = '' then TimeOut = 15  /* default to 15 seconds */
  do digits
    call FxLnDtmf 1, TimeOut, 'newkey'
      if result <> 'DTMF' | newkey = '#' | newkey = '*' then leave
    number = number||newkey
  end
  return number


GetExtension: procedure
  /* Gets DTMF input until # or * key or "digits" are entered. */
  /* TimeOut is seconds to wait for first digit (default 15).   */
  /* NextTime is seconds to wait for subsequent digits (default TimeOut)*/
  /* Returns 'FAX' or 'DATA' or 'NOTOK' results or the number. */
  parse arg digits, TimeOut, NextTime
  number = ''
  if TimeOut = '' then TimeOut = 15  /* default to 15 seconds */
  if NextTime = '' then NextTime = TimeOut
  do digits
    call FxLnDtmf 1, TimeOut, 'newkey'
      if result = 'FAX' | result = 'DATA' | result = 'NOTOK' then return result
      if result <> 'DTMF' | newkey = '#' | newkey = '*' then leave
    number = number||newkey
    TimeOut = NextTime
  end
  return number


TmpFile: procedure
  /* Generate a unique base file name using seconds since midnight */
  return 'TMP'||time( S )


AddToCsvList: procedure
  /* returns a comma-separated value list with item added */
  parse arg item, list
  if list = ''
    then list = item
    else list = list||','||item
  return list


SendLog:	/* sends the log file by fax */
  tmp1 = VOXDIR||TmpFile()||'.TXT'
  tmp2 = VOXDIR||TmpFile()||'.FAX'
  log = FxRxPath( 'LOG' )||'fax.log'

  call LineOut tmp1, 'Fax Log   ' Date('W') ' '||Date('N') ' ' Time('C')
  call LineOut tmp1, ''
  call LineOut tmp1, ,
       'Type ID      Date    Time  Pg Status Elpsd  Name/RemoteID/CallerID'
  call LineOut tmp1, ''

  if LOGMAXLINE <= 0 then do /* send the entire fax log */
    do while Lines( log ) > 0
      record = LineIn( log )		/* read a record from log */
      if FxRxParseLog( record, 'lrec' ) = OK then do
        call LogRpt tmp1
      end
    end
  end
  else do /* send the last x lines from the log */
    head = 1
    tail = 1
    do while Lines( log ) > 0	/* collect last x lines */
      buf.head = LineIn( log )
      head = head + 1
      if head > LOGMAXLINE then head = 1
      if head = tail then tail = tail + 1
      if tail > LOGMAXLINE then tail = 1
    end
    if tail <> head & FxRxParseLog( buf.head, 'lrec' ) = OK then do
      call LogRpt tmp1
    end
    do while tail <> head
      if FxRxParseLog( buf.tail, 'lrec' ) = OK then do
        call LogRpt tmp1
      end
      tail = tail + 1
      if tail > LOGMAXLINE then tail = 1
    end
  end
  rc = LineOut( log )			/* close the files */
  rc = LineOut( tmp1 )
  call FxRxTextToFax tmp1, tmp2		/* convert text to fax */
  /* call our FOD procedure to do the faxing */
  FODPARAMS = "FxRxPath( 'LOG' ), VOXDIR, , CALLER, tmp2," ,
          "'As you requested, the log is attached.', 'Log retrieval...'"
  interpret 'call' "'"||FODPROCEDURE||"'" FODPARAMS
  fodresult = result
  call FxLnVout 'Deleting:' tmp1
  'DEL "'||tmp1||'"'
  call FxLnVout 'Deleting:' tmp2
  'DEL "'||tmp2||'"'
  return fodresult


LogRpt:
  /* Using global lrec., write a report line to a file */
  parse arg Rfile

  if BitTest( lrec.!Flags, 256 ) then mtype = 'MSG'
  else if BitTest( lrec.!Flags, 512 ) then mtype = 'Txt'
  else if BitTest( lrec.!Flags, 1024 ) then mtype = 'Dat'
  else mtype = 'Fax'

  if lrec.!Company = ''
  then namestr = lrec.!Name
  else namestr = lrec.!Name||', '||lrec.!Company

  if lrec.!Pages = 0
  then pagestr = ''    /* don't show 0 pages */
  else pagestr = lrec.!Pages
      
  call LineOut Rfile, mtype left( lrec.!ID, 6 ) ,
         left( lrec.!Date, 9 ) left( lrec.!Time, 5 ) ,
         right( pagestr, 2 ) left( lrec.!Status, 6 ) ,
         right( lrec.!Elapsed, 5) ' '||strip( left( namestr, 46 ) )
  return


BitTest: procedure
  /* Given a number and a bit number, return 1 if bit is set */
  parse arg value, bit
  return (value % bit) // 2

SaveCode:
  /* Save access code to file MBXnnnnn.COD in user's log directory */
  parse arg newcode
  call LineOut USERDIR||'.cod', newcode, 1
  call LineOut USERDIR||'.cod'		/* close file */
  return

FetchCode:
  /* Returns the access code for the current USERID */
  if USERID = 0
  then return CODE
  newcode = LineIn( USERDIR||'.cod', 1 )
  call LineOut USERDIR||'.cod'		/* close file */
  if newcode = '' then newcode = DEFAULTCODE
  return newcode
