/* V_MBX.CMD - Voice/Fax Answer Script for FaxWorks/PMfax (REXX) */
/* 
                  Keller Group Inc.  June 19, 1997
   You may freely use or modify this script for use with licensed
        products which are developed by Keller Group Inc.
      For new scripts or to share your scripts with others,
                 see http://www.kellergroup.com

   Modification history:
      05May97 - Added SysSleep to paging to make sure line is hung up.
      06Jun97 - Changed to use new FxRxFind command (requires 3.01.01 !!)
                [Unlike original release, you can now move back through
                 old voice mail messages with the PrevMsg command.]
      19Jun97 - Changed termination after FOD calls - no log entry
      19Jun97 - Added reset command ('*')
*/
/*******************************************************************
V_MBX.CMD - Advanced voice/fax mailbox system

For use with any of the following Keller software configurations:
  Stand-alone (single-line or multiline)
  LAN Shared Mode
  LAN Private Mode (with fax server as router & retrieval from server log)

Features:

  Callers can:
     send a fax,
     leave a voice message,
     listen to an announcement, or
     enter the special password to enter remote retrieval mode

  In the password-protected remote retrieval mode, you can do:
     greeting modification - remotely update your greeting message
     voice retrieval - get voice messages by phone
     fax retrieval - get fax messages, or send them to a fax number
     log retrieval - get a fax of your log (or the last part of it)
     "fax on demand" from log - retrieve items from log using index

  You can also set configuration values to provide:
     forwarding by fax - forward all received faxes to a fax number
     forwarding by voice - forward all received voice messages to phone #
     forwarding by email - forward received fax and/or voice via email
       and also...
     paging by beeper - notify you by pager when voice or fax is received
     paging by voice - notify you by calling and playing a voice message
     paging by fax - notify you by sending a fax to a designated fax # 

See also:
    V_MULTI.CMD - Multiple Mailbox (Standalone) voice/fax mailbox system
    V_MBX4.CMD - 4-line Version of V_MBX.CMD voice/fax mailbox system
    V_MBXLAN.CMD - Advanced voice/fax mailbox system (LAN multiuser)

DESCRIPTION:
-----------

This script plays your outgoing message (OGM).  The caller can press
the # or * key to skip the OGM and immediately record their message,
otherwise the script will provide a beep and record the caller's
message if no special action is taken.

If the call is from a fax device and your fax hardware detects the
caller's fax (CNG) tones, the script automatically receives the fax.

If the caller enters '2' during playing of the OGM, the script plays
the "announcement" and then returns to the OGM again so that the
caller can leave a message on the same call.

If the caller enters '9' during or immediately after the playing of
the OGM and then enters the user-configurable access code followed by
the # key, the script enters its "remote retrieval" mode so the
caller can hear received voice messages, have received faxes sent to
them via fax, and other features.  When in this mode, the keys on the
telephone do the following:

   1 - HELP       - Play help message which describes these commands
   3 - GREETING   - Enter new outgoing message (change greeting)

   4 - PREV MSG   - Play previous voice message
   5 - REPLAY MSG - Play current voice message (again)
   6 - NEXT MSG   - Play next voice message

   7 - GET FAXES  - Retrieve new fax messages by fax
   8 - GET LOG    - Retrieve fax log by fax
   9 - FOD LOG    - Retrieve voice/fax msgs from log ("fax-on-demand")

   * - RESET      - Marks all "new" messages/faxes as "retrieved"

Other features are controlled by setting "script variables" as
described below.  These values can cause the script to do automatic
forwarding by fax, voice or email and automatic paging via beeper,
voice or fax.

USAGE INSTRUCTIONS:
------------------
You must be using the program with supported voice/fax hardware to
use this script.  Consult the README.DOC file for tested voice/fax
hardware and recommended configuration settings for your hardware.
Before using this script, first test your system with our built-in
voice answering machine feature by enabling "Voice" and setting the
"Answer script" to * (an asterisk character) on the Voice page of the
Settings notebook.  If you are using supported voice/fax hardware
with the proper configuration settings, you can then use
"Fax/Receive/All calls" mode to answer calls, play the outgoing
message (OGM), and record a voice message (or receive a fax).

Then, to change to this advanced script, specify the full pathname of
this file in the "Answer script" field on the Voice page of the
settings notebook.

NOTE: The script expects to find the LIB_FOD.CMD file and its Wave
audio files in the fax program (EXE) directory.  The script also uses
Wave files from the c:\mmos2\sounds directory for special effects.

SCRIPT VARIABLES:
----------------
The following script variable is used by this script.  Script
variables are set on the Voice page of the Settings Notebook.

General:

	OGM - Pathname of the "outgoing message" file which is played
	to the caller.  By default, the program sets OGM to be the
	file OGM.WAV in the program directory.
	Example: OGM = C:\FAX\OGM.WAV

	CODE - Access code for retrieving messages or changing
	settings.  By default, the program sets CODE to be 42.  You
	can change this to be any number of up to 5 digits in length.
	Example: CODE = 42

	CALLER - (optional, default "Person") - Name of the person
	who will be using the access code to retrieve messages/faxes
	by phone.  This is used for addressing faxes when the
	caller uses the access code for remote retrieval.
	Example: CALLER = Jim Smith

	LOGMAXLINE - (optional, default 56) - Maximum number of log
	entries, from the end of the log, to be sent to the caller
	when they retrieve the log by fax.  If set to 0, then
	the entire log is sent.
	Example: LOGMAXLINE = 120

	EMAILCMD - (optional) - The command line to be executed to send a
	fax, voice or text file via email.  We append the file pathname
	to the end of this command line and execute it to tell your email
	software to send the file.  If the EMAILPARAM script variable is
	set, then it is appended to the command line following the file
	pathname (which can be used to add additional parameters).
	Email support requires the use of an email package which provides
	a command line interface and is configured to automatically
	send the email which is submitted to it.
	Example: To demo by inserting into our log rather than emailing:
	       EMAILCMD = c:\fax\fxrcv -rcvd
	Example for Post Road Mailer version 2.0 using its PRMFAX.EXE:
	       EMAILCMD = c:\postroad\prmfax
	       EMAILPARAM = "yourid@ibm.net" "EMAIL FAX"

Forwarding:

	FWDFAX - (optional) - A fax number.  If specified, received
	faxes are automatically forwarded to the specified fax
	number.  All faxes are also kept in the fax log.
	Example: FWDFAX = 1 612 555-5555

	FWDVOICE - (optional) - A voice number.  If specified,
	received voice messages are automatically forwarded to the
	specified number.  Add comma characters to the end of the
	number if additional pauses are desired before the message is
	played.  This can be used to pass a voice message along to a
	second answering machine or a cellular phone, but delivery is
	not guaranteed since dialing delays, busy signals and other
	events may interfere.
	Example: FWDVOICE = 555-5555

	FWDEMAIL - (optional) - Set to FAX, VOICE or ALL.  If
	specified (and if the EMAILCMD script variable is also
	specified), received faxes, voice messages or both will be
	forwarded by emailing the file using the EMAILCMD command
	line.
	Example: FWDEMAIL = ALL

Paging:

	PAGEBEEPFAX - (optional) - If paging is desired upon receipt of each
	fax, this script variable is defined and is set to the dial string
	which to be called for paging.  (See Paging below.)

	PAGEBEEPVOICE - (optional) - If paging is desired upon receipt of
	each voice message, this script variable is defined and is set to
	the dial string which to be called for paging.  (See Paging below.)

	PAGEVOICE - (optional) - A voice number.  If specified,
	the system calls this number and plays the RCVD_FAX.WAV or
	RCVD_MSG.WAV voice prompt file.  Add comma characters to the
	end of the number if additional pauses are desired before the
	message is played.
	Example: PAGEVOICE = 555-5555

	PAGEFAX - (optional) - A fax number.  If specified, the
	system sends a fax to this number to say that a fax or voice
	message was received.  
	Example: PAGEFAX = 1 612 555-5555

PAGING:
------
You can have the script call your pager if you wish to be notified
when a call or fax is received.  To enable paging, define and set the
PAGEBEEPFAX or PAGEBEEPVOICE script variables on the Voice page of
the Settings notebook.

The value of the script variable is the dial string for calling and
using your paging system.  The value should include:

	1) the telephone number of your paging system,

	2) several commas (each comma delays for 2 seconds - enter
	the number of commas required for the proper delay for your
	paging system),

	3) your Pager ID followed by another comma (if your paging
	system uses IDs to identify a specific pager - paging systems
	which use a unique telephone number for each pager will not
	require this step),

	4) the page message (your telephone number or some number
	to show you that you have received a fax or voice message).

For example, for a paging system which does not use Pager IDs, the
value might be of the form "PAGEBEEPVOICE=555-5555,,,,555-1234".  For
a paging system which uses Pager IDs where your Pager ID is 123456,
the value might be "PAGEBEEPVOICE=1-800-555-5555,,,,123456,555-1234".

VOICE PROMPT FILES:
------------------
The script expects to find the LIB_FOD.CMD file and the following
Wave audio files in the fax program (EXE) directory.  The script also
uses Wave files from the c:\mmos2\sounds directory for special effects.

To record or modify your own voice prompt files, use your OS/2
Multimedia microphone and record voice prompts using the "Fax/New
message" command then save them to a .WAV file with the "Fax/Save
file/Wave" command.  You can also use the OS/2 Digital Audio program
to record, save, edit and modify the Wave files (be sure to use
"Type" of Mono, 8-bit, 11.025 kHz).

	OGM.WAV - The main "outgoing message" as recorded with the
	Utilities/Outgoing message command.

	ENTR_COD.WAV - Short: "Enter code."
	Long: "Please enter your access code and the # key."

	ENTR_CMD.WAV - Short: "Enter command."
	Long: "To play current message press 5, to play next message
	press 6, to play previous message press 4...."

	ENTR_MSG.WAV - Short: "Enter message, then press # key."
	Long: "Please leave your message at the beep.  Press # key to end."

	ANNOUNC.WAV - Short: "No announcement."
	Long: "Our annoucement for this week is...."

	HELP.WAV - "Press 3 to change outgoing message, 4 for previous..."

	NO_MSGS.WAV - Short: "No more messages."

	RCVD_FAX.WAV - Short: "A fax has been received."

	RCVD_MSG.WAV - Short: "A voice message has been received."

And used by any fax retrieval actions (LIB_FOD.CMD procedure):

	ENTR_DOC.WAV - Short: "Enter document number and # key, or
	press the # key if done."
	Long: "Please enter the document number followed by the # key,
	or press the # key if done.  To receive an index of available
	documents, enter document 1000."

	HOW_FAX.WAV - Short: "Enter 1 if calling from fax machine, 2
	if not."
	Long: "If you are calling from your fax machine and are ready
	to receive the fax press 1, to send the fax to a different
	number press 2." 

	ENTR_FAX.WAV - Short: "Enter fax number and # key."
	Long: "Please enter your fax number followed by the # key,
	including 1 and area code if this is not a local call." 

	NO_FILE.WAV - Short: "File not found."
	Long: "That document was not found, please try again."

	WAIT.WAV - Short: "Please wait."
	Long: "Please wait while we process your request."

	THANKYOU.WAV - Short: "Thank you."

TO MODIFY THE SCRIPT:
--------------------
The script is REXX and uses calls from the Keller REXX API which are
automatically loaded into the OS/2 REXX environment by the retail
versions of Keller's fax products (version 3.0 and later).  See the
Reference Manual for documentation of the FxRx and FxLn calls.  You
may modify the script as desired, and then use it as the "Answer
script" with Keller fax software products.

Note - For retrieval, this script uses the READ/N_READ mode of
FxRxFind and the new Next/Prev FxRxFind support of version 3.01.01
and later.  Using READ/N_READ mode means it will report all
faxes/messages which have not been previously retrieved via phone
(even those that have been viewed at the computer).  If desired, you
can change this to use RCVD/N_RCVD mode of FxRxFind, which means that
it will report only those faxes which have been received but not
already viewed at the computer and not previously retrieved via phone
(but this is less reliable since "error" faxes won't be included).
You can also retrieve any desired fax, whether or not it has been
viewed or retrieved, using the "fax log" and "fax-on-demand" features
in this script which allow you to retrieve any fax by its index
number.  In version 3 of the fax program, viewing a received fax
changes its status from Rcvd to Read, but printing of faxes does NOT
change the status, so you can auto-print all received faxes at the
computer and still retrieve them as "Rcvd" faxes by phone. 

TO USE WITH MULTILINE VERSIONS:
------------------------------
This script can be used with multiline versions of Keller fax
software.  All lines are handled identically, so calls can be
received on any of your "receive" lines.  If you prefer to have
different lines be handled differently you can modify the script to
test the LID variable.  For example, to have line 1 immediately jump
to the "please enter your message" prompt, you could add a line like
the following after the "call FxLnInit" line in the script: "if LID =
1 then signal voice" where "voice" is a label in the script.

*******************************************************************/

  call FxLnInit
  /* need to use REXXUTIL for SysSleep function in paging (6/97) */
  call RxFuncAdd 'SysLoadFuncs','RexxUtil','SysLoadFuncs'
  call SysLoadFuncs

  VOXDIR = FxRxPath( 'EXE' )	/* get voice prompts from exe dir */
  LIBDIR = FxRxPath( 'EXE' )	/* get REXX subroutines from exe dir */
  FODPROCEDURE = LIBDIR||'lib_fod.cmd'  /* use this external procedure */

  /* default values for script variables */
  if CALLER = 'CALLER' then CALLER = 'Person'
  if LOGMAXLINE = 'LOGMAXLINE' then LOGMAXLINE = 56
  if EMAILCMD = 'EMAILCMD' then EMAILCMD = ''
  if EMAILPARAM = 'EMAILPARAM' then EMAILPARAM = ''

  if FWDFAX = 'FWDFAX' then FWDFAX = ''
  if FWDVOICE = 'FWDVOICE' then FWDVOICE = ''
  if FWDEMAIL = 'FWDEMAIL' then FWDEMAIL = ''

  if PAGEBEEPFAX = 'PAGEBEEPFAX' then PAGEBEEPFAX = ''
  if PAGEBEEPVOICE = 'PAGEBEEPVOICE' then PAGEBEEPVOICE = ''
  if PAGEVOICE = 'PAGEVOICE' then PAGEVOICE = ''
  if PAGEFAX = 'PAGEFAX' then PAGEFAX = ''


hello:
  LREC. = ""				/* be sure this is clean */
  call FxLnMsg 'OGM'				/* show in status window */
	if result = 'NOTOK' then signal byebye
  call FxLnPlay OGM				/* play greeting prompt */
	if result = 'NOTOK' then signal byebye
	if result = 'FAX' then signal fax
	if result = 'DATA' then signal data

  call FxLnDtmf 1, 3, 'digits'			/* get key, if any... */
	if result = 'NOTOK' then signal byebye
	if result = 'FAX' then signal fax
	if result = 'DATA' then signal data
	if result <> 'DTMF' then signal voice

  if digits = 2 then signal announce		/* use key value... */
  if digits = 9 then signal accesscode


voice:					/* take voice message, hangup */
  call FxLnMsg "Record Msg"
  call FxLnTone 1850
	if result = 'NOTOK' then signal byebye
  call FxLnDtmf		/* flush buffer */
  call FxLnRecord
	if result = 'NOTOK' then signal byebye
	if result = 'FAX' then signal fax
	if result = 'DATA' then signal data
  /* caller can hang up or press a key to end recording */
  call FxLnTone 1850, 1850, 1850
  call Forwarding 'voice'
  call Paging 'voice'
  signal out


fax:					/* receive fax and hangup */
  call FxLnReceive
  if result = 'OK' then do
    call Forwarding 'fax'
    call Paging 'fax'
  end
  signal out


data:					/* report data call, hangup */
  call FxLnMsg 'Data call, oh well...'
  LREC.!Status = '*Data!'   /* log status of call */
  signal out


announce:				/* play annoucement, restart */
  call FxLnMsg "Announcement"
  call FxLnPlay VOXDIR||'ANNOUNC.WAV'
    if result = 'NOTOK' then signal byebye
  signal hello


accesscode:
  call FxLnMsg 'Code ???'		/* get & check access code */
  call FxLnPlay VOXDIR||'ENTR_COD.WAV'
	if result = 'NOTOK' then signal byebye
  digits = GetNumber( 6, 10 )
  if digits <> CODE then signal evil

  call FxLnMsg 'Code Mode'
  call FxLnVout 'Valid code entered'
  call FxRxBeep 354, 393, 295, 354, 0, 393

  call FxRxFind 'MSG', 'N_READ'		/* get # of new voice msgs */
  if result = 'NOTOK' then do
	call FxLnPlay 'c:\mmos2\sounds\BOING.wav'
	signal byebye
  end
  n = result

  /* if no messages... stay in command mode for other (fax) commands */
  if n = 0 then do
        call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	    if result = 'NOTOK' then signal byebye
  end

  i = n				/* Play 1 beep for each voice message */
  call FxLnMsg 'Code Mode' n 'messages'
  call FxLnVout 'Beep' n 'times'
  do while i > 0
    call FxLnTone 2100
    i = i - 1
  end

  call FxLnVout 'Enter message loop'
  i = 0

  /* select the first voice message, or last one if no new msgs */
  if n > 0 then do
    call FxRxFind 'MSG', 'READ', 'LREC'
    if result <> 'NOTOK' then do
      call FxLnVout 'Found by READ' LREC.!ID
      i = LREC.!ID
    end
  end
  else do
    call FxRxFind 'MSG', 'PREV', 'LREC'
    if result <> 'OK' then do
      call FxLnVout 'Found by PREV' LREC.!ID
      i = LREC.!ID
    end
  end

  do forever
    call FxLnMsg 'Waiting for next command'
    call FxLnPlay VOXDIR||'ENTR_CMD.WAV'
    call FxLnDtmf 1, 30, 'digits'
    if result <> 'DTMF' then do
      signal goodbye
    end
    else do
      call FxLnMsg 'Code Mode command' digits
      call FxLnVout 'Command' digits 'entered'


      if digits = '1' then do			/* 1 = Play help msg */
        call FxLnVout 'Help message'
        call FxLnPlay VOXDIR||'HELP.WAV'
	  if result = 'NOTOK' then signal byebye
        iterate
      end


      if digits = '3' then do			/* 3 = Enter outgoing msg */
        call FxLnVout 'Enter outgoing message'
        call FxLnPlay VOXDIR||'ENTR_MSG.WAV'
	  if result = 'NOTOK' then signal byebye
        call FxLnTone 1850
	  if result = 'NOTOK' then signal byebye
        call FxLnDtmf		/* flush buffer */
        call FxLnRecord OGM
	  if result = 'NOTOK' then signal byebye
        call FxLnTone 1850, 1850, 1850
        call FxLnDtmf		/* flush buffer */
        call FxLnPlay OGM
	  if result = 'NOTOK' then signal byebye
        iterate
      end


      if digits = '4' then do			/* 4 = Previous msg */
        call FxLnVout 'Play previous message' i
        if i = 0 
	then call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	else do
	  call FxRxFind 'MSG', 'PREV', 'LREC'
	  if result = 'OK'
	  then do
	    call FxLnVout 'Find PREV returns' result
	    call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	  end
	  else do
	    call FxLnVout 'Find PREV returns' result
	    call FxLnVout 'Found by PREV' LREC.!ID
	    i = LREC.!ID
	    call FxLnMsg 'Code Mode command' digits 'Msg' i
	    call FxLnPlay i
	  end
        end
        iterate
      end


      if digits = '5' then do			/* 5 = Replay msg */
        call FxLnVout 'Replay message' i
        if i = 0 
	then call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	else do
	  call FxLnMsg 'Code Mode command' digits 'Msg' i
	  call FxLnPlay i
        end
        iterate
      end


      if digits = '6' then do			/* 6 = Next message */
        call FxLnVout 'Play next message' i
        if i = 0 
	then call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	else do
	  call FxRxFind 'MSG', 'NEXT', 'LREC'
	  if result = 'OK'
	  then do
	    call FxLnVout 'Find NEXT returns' result
	    call FxLnPlay VOXDIR||'NO_MSGS.WAV'
	  end
	  else do
	    call FxLnVout 'Find NEXT returns' result
	    call FxLnVout 'Found by NEXT' LREC.!ID
	    i = LREC.!ID
	    call FxLnMsg 'Code Mode command' digits 'Msg' i
	    call FxLnPlay i
	  end
        end
        iterate
      end


      if digits = '7' then do			/* 7 = send rcvd faxes */
        call FxLnVout 'Send rcvd faxes'
        call FxLnMsg 'Code Mode command' digits
        call FxRxFind 'FAX', 'N_READ'		/* get # of new faxes */
        if result = 'NOTOK' then do
          call FxLnPlay 'c:\mmos2\sounds\BOING.wav'
          signal byebye
        end
        nfax = result
        if nfax = 0 then do
          call FxLnPlay VOXDIR||'NO_MSGS.WAV'
          if result = 'NOTOK' then signal byebye
        end
        else do
          ifax = nfax		/* Get faxes into list and beep per fax */
          faxlist = ''
          flist = ''
          call FxLnMsg 'Code Mode' nfax 'faxes'
          call FxLnVout 'FaxBeep' nfax 'times'
          do while ifax > 0
            call FxLnTone 2100
            ifax = ifax - 1
            call FxRxFind 'FAX', 'READ', 'LREC'
            call FxLnVout 'Found' LREC.!ID
            faxlist = AddToCsvList( LREC.!ID, faxlist )
            flist = flist LREC.!ID
          end
          /* call our FOD procedure to do the faxing */
          FODPARAMS = "FxRxPath( 'LOG' ), VOXDIR, , CALLER, faxlist," ,
            "'As you requested, your received faxes are attached:' flist," ,
            "'Remote fax retrieval...'"
          interpret 'call' "'"||FODPROCEDURE||"'" FODPARAMS
          if result = 'BYEBYE' then signal BYEBYE
          if result = 'OUT' then signal OUT
        end
        iterate
      end


      if digits = '8' then do			/* 8 = send log */
        call FxLnVout 'Send log'
        call FxLnMsg 'Code Mode command' digits
        call FxLnPlay VOXDIR||'WAIT.WAV'
	  if result = 'NOTOK' then signal byebye
        call SendLog
          if result = 'BYEBYE' then signal BYEBYE
          if result = 'OUT' then signal OUT
        iterate
      end


      if digits = '9' then do			/* 9 = FOD from log */
        call FxLnVout 'FOD from log'
        call FxLnMsg 'Code Mode command' digits
        FODPARAMS = "FxRxPath( 'LOG' ), VOXDIR, , CALLER, , ," ,
               "'FOD Retrieval...'"
        interpret 'call' "'"||FODPROCEDURE||"'" FODPARAMS
          if result = 'BYEBYE' then signal BYEBYE
          if result = 'OUT' then signal OUT
        iterate
      end


      if digits = '*' then do			/* * = reset */
        /* This is used to mark as messages and faxes in the log as  */
	/* having been "previously retrieved" so that you don't need */
	/* to get them all via phone.  You can still go back through */
	/* them from the end using "Prev Msg" and "fax-on-demand".   */
        call FxLnVout 'Reset' i
	call FxLnMsg 'Reset command' digits
        call FxLnPlay VOXDIR||'WAIT.WAV'
	  if result = 'NOTOK' then signal byebye
	LREC. = ""		/* forget any previous position in log */
	do forever
	  call FxRxFind 'ANY', 'NEXT', 'LREC'
          if result = 'OK' | result = 'NOTOK' then leave
	end
	/* Select last voice message as the current message */
	LREC. = ""
	call FxRxFind 'MSG', 'PREV', 'LREC'
	if result = 'OK'
	then i = 0
	else i = LREC.!ID
	call FxLnVout 'Reset find PREV returns' result
        iterate
      end


      if digits = '#' then do
        signal goodbye
      end


      /* otherwise, if none were matched... */
      call FxLnVout 'Unknown command' i
      call FxLnPlay 'c:\mmos2\sounds\EEERRUPP.wav'
      iterate

    end
  end


goodbye:
  call FxLnVout 'Exit Code Mode'
  call FxLnPlay 'c:\mmos2\sounds\DRUMROLL.wav'
  signal byebye


evil:
  call FxLnVout 'Bad code entered:' digits 'vs' CODE
  call FxLnPlay 'c:\mmos2\sounds\BOO.wav'
  call FxRxBeep 1318, 1396, 1318, 1090, 1318, 1396, 1318, 1090
  signal byebye


byebye:					/* called for error cases */
  LREC.!Status = ''			/* don't log anything */
  signal out


out:					/* clean up and quit */
  call FxLnTerm
  exit


/************************ Procedures ************************/


GetNumber: procedure
  /* Gets DTMF input until # or * key or "digits" are entered. */
  /* TimeOut is seconds to wait for each digit (default 15).       */
  parse arg digits, TimeOut
  number = ''
  if TimeOut = '' then TimeOut = 15  /* default to 15 seconds */
  do digits
    call FxLnDtmf 1, TimeOut, 'newkey'
      if result <> 'DTMF' | newkey = '#' | newkey = '*' then leave
    number = number||newkey
  end
  return number


TmpFile: procedure
  /* Generate a unique base file name using seconds since midnight */
  return 'TMP'||time( S )


AddToCsvList: procedure
  /* returns a comma-separated value list with item added */
  parse arg item, list
  if list = ''
    then list = item
    else list = list||','||item
  return list


SendLog:	/* sends the log file by fax */
  tmp1 = VOXDIR||TmpFile()||'.TXT'
  tmp2 = VOXDIR||TmpFile()||'.FAX'
  log = FxRxPath( 'LOG' )||'fax.log'

  call LineOut tmp1, 'Fax Log   ' Date('W') ' '||Date('N') ' ' Time('C')
  call LineOut tmp1, ''
  call LineOut tmp1, ,
       'Type ID      Date    Time  Pg Status Elpsd  Name/RemoteID/CallerID'
  call LineOut tmp1, ''

  if LOGMAXLINE <= 0 then do /* send the entire fax log */
    do while Lines( log ) > 0
      record = LineIn( log )		/* read a record from log */
      if FxRxParseLog( record, 'lrec' ) = OK then do
        call LogRpt tmp1
      end
    end
  end
  else do /* send the last x lines from the log */
    head = 1
    tail = 1
    do while Lines( log ) > 0	/* collect last x lines */
      buf.head = LineIn( log )
      head = head + 1
      if head > LOGMAXLINE then head = 1
      if head = tail then tail = tail + 1
      if tail > LOGMAXLINE then tail = 1
    end
    if tail <> head & FxRxParseLog( buf.head, 'lrec' ) = OK then do
      call LogRpt tmp1
    end
    do while tail <> head
      if FxRxParseLog( buf.tail, 'lrec' ) = OK then do
        call LogRpt tmp1
      end
      tail = tail + 1
      if tail > LOGMAXLINE then tail = 1
    end
  end
  rc = LineOut( log )			/* close the files */
  rc = LineOut( tmp1 )
  call FxRxTextToFax tmp1, tmp2		/* convert text to fax */
  /* call our FOD procedure to do the faxing */
  FODPARAMS = "FxRxPath( 'LOG' ), VOXDIR, , CALLER, tmp2," ,
          "'As you requested, the log is attached.', 'Log retrieval...'"
  interpret 'call' "'"||FODPROCEDURE||"'" FODPARAMS
  fodresult = result
  call FxLnVout 'Deleting:' tmp1
  'DEL "'||tmp1||'"'
  call FxLnVout 'Deleting:' tmp2
  'DEL "'||tmp2||'"'
  return fodresult


LogRpt:
  /* Using global lrec., write a report line to a file */
  parse arg Rfile

  if BitTest( lrec.!Flags, 256 ) then mtype = 'MSG'
  else if BitTest( lrec.!Flags, 512 ) then mtype = 'Txt'
  else if BitTest( lrec.!Flags, 1024 ) then mtype = 'Dat'
  else mtype = 'Fax'

  if lrec.!Company = ''
  then namestr = lrec.!Name
  else namestr = lrec.!Name||', '||lrec.!Company

  if lrec.!Pages = 0
  then pagestr = ''    /* don't show 0 pages */
  else pagestr = lrec.!Pages
      
  call LineOut Rfile, mtype left( lrec.!ID, 6 ) ,
         left( lrec.!Date, 9 ) left( lrec.!Time, 5 ) ,
         right( pagestr, 2 ) left( lrec.!Status, 6 ) ,
         right( lrec.!Elapsed, 5) ' '||strip( left( namestr, 46 ) )
  return


BitTest: procedure
  /* Given a number and a bit number, return 1 if bit is set */
  parse arg value, bit
  return (value % bit) // 2


Forwarding:
  parse arg mode /* 'voice' or 'fax' */

  if mode = 'voice' then do

    /* voice forwarding */
    if FWDVOICE <> '' then do
      call DeliverVoice LREC.!ID FWDVOICE
    end
    /* email forwarding */
    if EMAILCMD <> '' then do	/* using email? */
      call FxLnVout 'FWDEMAIL=' FWDEMAIL 'EMAILCMD=' EMAILCMD
      if FWDEMAIL = 'ALL' | FWDEMAIL = 'VOICE' then do
        cmdstring = EMAILCMD '"'||FxRxIndexToFax( LREC.!ID )||'"' EMAILPARAM
        cmdstring
        call FxLnVout 'rc=' rc 'for' cmdstring
      end
    end
  end

  else if mode = 'fax' then do
    /* fax forwarding */
    if FWDFAX <> '' then do
      /* Send with default cover sheet and default "From" information */
      call FxRxQueue LREC.!ID, ,
         'TO='||CALLER||' at fax: '||FWDFAX||',,'||FWDFAX, ,
         'INFO="Forwarded fax is attached.",*,"Forwarding fax...",*,1,2', ,
         'FROM=*,*,*,*'
    end
    /* email forwarding */
    if EMAILCMD <> '' then do	/* using email? */
      call FxLnVout 'FWDEMAIL=' FWDEMAIL 'EMAILCMD=' EMAILCMD
      if FWDEMAIL = 'ALL' | FWDEMAIL = 'FAX' then do
        cmdstring = EMAILCMD '"'||FxRxIndexToFax( LREC.!ID )||'"' EMAILPARAM
        cmdstring
        call FxLnVout 'rc=' rc 'for' cmdstring
      end
    end

  end
  return


DeliverVoice: procedure expose LREC.
  /* given a message (id# or file) and phone number, call the number */
  /* and play message.  Protects LREC record while doing so.         */
  parse arg msg fwdnum
  call FxLnVout 'DELIVERVOICE =' msg fwdnum
  call FxLnMsg 'Voice delivery of' msg 'to' fwdnum
  logtmp = FxRxFormatLog( 'LREC' )	/* save the record */
  do 3   /* retry in case of no dialtone, etc. */
    SysSleep( 5 )			/* wait for line to hangup */
    call FxLnLine 'DIAL' fwdnum
    if result = OK then do
      call FxLnPlay msg
      leave
    end
  end
  call FxRxParseLog logtmp, 'LREC'	/* restore record  */
  call FxLnVout 'Leaving DeliverVoice'
  return


DeliverBeeps: procedure
  /* Given a phone number, call the number */
  parse arg BeepNum
  call FxLnMsg 'Delivery beeps to' BeepNum
  do 3   /* retry in case of no dialtone, etc. */
    SysSleep( 5 )			/* wait for line to hangup */
    call FxLnLine 'DIAL' BeepNum
    if result = OK then leave
  end
  return


Paging:
  parse arg mode	/* 'voice' or 'fax' */

  call FxLnVout 'PAGING =' mode PAGEFAX PAGEVOICE
  if PAGEFAX <> '' then do
    if mode = 'fax'
    then tmp = 'A fax was received.'
    else tmp = 'A voice message was received.'
    call FxRxQueue 'TO='||CALLER||' at fax: '||PAGEFAX||',,'||PAGEFAX, ,
       'INFO='||'"'||tmp||'"'||',*,"Paging by fax...",*,1,2', ,
       'FROM=*,*,*,*'
  end

  if PAGEVOICE <> '' then do
    if mode = 'fax' then do
      call DeliverVoice VOXDIR||'RCVD_FAX.WAV' PAGEVOICE
    end
    else if mode = 'voice' then do
      call DeliverVoice VOXDIR||'RCVD_MSG.WAV' PAGEVOICE
    end
  end

  if mode = 'fax' & PAGEBEEPFAX <> '' then do
    call DeliverBeeps PAGEBEEPFAX
  end
  if mode = 'voice' & PAGEBEEPVOICE <> '' then do
    call DeliverBeeps PAGEBEEPVOICE
  end

  return
