/****************************************************************************
  Macro to wrap paragraphs.

  If a column block is marked, the text is wrapped without disturbing the
  surrounding text.

  For other block types, text is wrapped until end-of-block.

  If not block, the normal WrapPara command is called.

  If argument "quote" is passed, block is quoted by "quote_chars".

  Mar 29, 2001 SEM
        Turn off undo for hidden buffer.
  Oct 14, 1998 SEM
        Allow 'quote character' to be specified by setting variable
        quote_chars to the desired value.

  Aug 24, 1998 SEM
        Did not co-exist with Chris Lee's select macro.  The problem appears
        to be that we were switching to a non-hidden buffer, causing select
        to think that it was an editing buffer.  By adding the _HIDDEN_ parm
        to CreateBuffer, the problem seems to have gone away.

  Jul 14, 1998 SEM
        Quote the wrapped area, if "quote" is passed as first argument.
        Quoting of Column blocks is not supported.
 ***************************************************************************/

string quote_chars[] = ">"          // can be more than a single char; if "\", should be
                                    // quoted, as in "\\"
proc WrapColumnBlock()
    integer
        id,                         // work buffer id
        curr_id,
        block_beg_col,
        save_leftmargin,
        save_rightmargin,
        save_autoindent,
        save_wrapend

    curr_id = GetBufferId()

    GotoBlockBegin()
    block_beg_col = CurrCol()
    id = CreateBuffer("", _HIDDEN_)
    SetUndoOff()
    CopyBlock()                 // Copy block to temp buffer

    /**************************************************************
      The number of lines in the column may become less than what
      it was - so we must fill the old block with spaces.
     **************************************************************/
    PushBlock()                 // Save block settings
    GotoBufferId(curr_id)       // Back to original file
    CopyBlock(_OVERWRITE_)      // And get the block back
    FillBlock(' ')              // Wipe it out
    GotoBufferid(id)            // Back to where we were
    PopBlock()                  // And get our block marked again

    /**************************************************************
      Prepare to wrap - we need to set the left/right margins to
      1 and the width of the column.  We also need to preserve the
      old settings.
     **************************************************************/
    save_leftmargin = Set(LeftMargin, 1)
    GotoBlockEnd()
    save_rightmargin = Set(RightMargin, CurrCol())
    save_autoindent = Set(AutoIndent, Off)
    save_wrapend = Set(ParaEndStyle, 0)
    BegFile()
    repeat
    until not WrapPara()
    UnMarkBlock()           // We need to re-mark the block
    BegFile()
    MarkColumn()
    EndFile()
    GotoColumn(Query(RightMargin))

    /*************************************************************
      And finally, go back to the original file, and copy the block in.
     *************************************************************/
    GotoBufferId(curr_id)
    CopyBlock(_OVERWRITE_)
    AbandonFile(id)
    GotoBlockEnd()
    Down()
    GotoColumn(block_beg_col)

    // Restore saved settings

    Set(LeftMargin, save_leftmargin)
    Set(RightMargin, save_rightmargin)
    Set(AutoIndent, save_autoindent)
    Set(ParaEndStyle, save_wrapend)
end

proc main()
    integer
        start_line,                 // starting line if we are quoting
        quoting,                    // boolean flag
        save_rightmargin,           // fudged rightmargin if quoting
        blocktype = isCursorInBlock()

    quoting = FALSE
    if Query(MacroCmdLine) == "quote" and blocktype <> _COLUMN_
        quoting = TRUE
        save_rightmargin = Set(RightMargin, Query(RightMargin) - Length(quote_chars))
    endif

    if blocktype == 0
        start_line = CurrLine()
        WrapPara()
    else
        start_line = Query(BlockBegLine)
        Set(Marking, off)               // Stop marking

        if blocktype == _COLUMN_        // If Column block, only wrap the column
            WrapColumnBlock()
        else                            // Otherwise, wrap the entire block
            GotoBlockEnd()
            AddLine()
            GotoBlockBegin()
            BegLine()
            // while we can find something to wrap within the block, and we can
            // wrap it, keep repeating.
            while lFind("[~ \t]", "lx") and WrapPara()
            endwhile
            if CurrLineLen() == 0
                DelLine()
            endif
        endif
    endif

    if quoting
        Set(RightMargin, save_rightmargin)
        PushBlock()
        PushPosition()
        if PosFirstNonWhite() == 0 or (blocktype <> 0 and not isCurrLineInBlock())
            Up()
        endif
        MarkLine(start_line, CurrLine())
        lReplace("^", quote_chars, "glxn")
        PopPosition()
        PopBlock()
    endif

end

