/*************************************************************************
  State       Save/Restore editor state information (files/windows/etc)

  Author:     SemWare

  Date:       Summer 1995 (Initial version)

              Nov 1995 SEM: Fix bug that occurs when:
                    -Display boxed, Mouse is enabled, and cursor is in
                        bottom most window.

              Nov 1996 SEM: close windows on unnamed buffers

              Feb 1997 non_inclusive (MarkChar) blocks are not properly
                    restored, e.g, the last character is not marked.  Bug
                    fix submitted by CJA.

                    CJA: save keyboard macros

              Apr 21, 1997 SEM: Try to restore w32 rows/cols (DOS legacy of
                    80 columns was still being enforced).

              Jul 15, 1997 SEM: Fix handling of filenames with embedded
                    spaces by quoting them.

              Sep  4, 1997 SEM: Use MAX_WINDOWS local constant for # of windows.

              Jan 26, 1998 SEM: More fixes on spaces in filenames.

              Feb 10, 2006 SEM: Apply bug fix submitted by Colin
                Liebenrood.  In RestoreState, the string 's' needed to
                be set at the maximum string length.  Thanks for the
                fix Colin!

              Aug 17, 2006 SEM: Problems with filenames containing
              spaces.  Fix is to slightly change the state file
              format.  Thanks to Jim Wilson for the bug report.

              Dec 01, 2008 SEM: Added ShowLineNumbers

              Mar 02, 2010 SEM: Fix endless loop problem when a file cannot be loaded.
                Thanks to Richard Blackburn for the report.

              Dec 06, 2010 SEM: Only use DLL's (isMaximized) if running under Windows

              Dec 21, 2010 SEM: Add BrowseMode (readonly mode)

              Jan 04, 2011 SEM: Updated to be backwards compatible with v1.3 and
                v1.2 state files.

              Sep 30, 2013 SEM: Add a menu to specify global/local/state filename

              Sep 29, 2015 SEM: Fix a couple of issues when: file not found and the screen
                size has changed.  Thanks to Richard Blackburn for the bug reports.

              Sep 09, 2017 SEM: Save/restore font

              May 10, 2019 SEM: Add -d option - delete state file

              May 28, 2019 SEM: enforce default global state file location

              Apr 13, 2020 SEM: local option of SaveState being
                ignored.  Thanks to Tom Collins for the report.

              Apr 24, 2020 SEM: clean up convoluted code in main - thanks to
                Tom Collins for bringing it to my attention.

  Overview:

  This macro allows you to interactively save the editing environment
  for the current session, or restore the previously-saved
  environment.

  Keys:       (none)

  Usage notes:

  The State macro can come in handy when you need to interrupt your
  current editing session to view another set of files.  Without
  exiting from the editor, you can save the environment for the
  current session; load additional files and/or change certain editing
  options; then automatically restore the previous environment so you
  can return to the set of files and options as they were previously.

  When you execute this macro, the editor displays a prompt box from
  which you can select "Save State" to save the current editing
  environment, or select "Restore State" to restore the
  previously-saved environment.

  Command line format:

  state [-g|-l] [-r|-s] [-ffilename]

  where:
      -g  use global state file (default)
      -l  use local state file
      -r  restore
      -s  save
      -ffilename  alternate file-path (default is tsestate.dat)
      -q  don't complain if state file not found
      -d  delete any existing state file

  Things that TSE saves via other means:

      - recent files
      - history

  Things saved via this macro:
      - bookmarks
      - windows
      - files
      - common-modes
      - block
      - keyboard macros

  Format of the saved information:

  New in 1.4:  added browse_mode - under [files]

  [state v1.4]
  [files]
  line row pos xoffset binary_mode display_mode browse_mode filename
  [bookmarks]
  bookmark line row pos xoffset filename
  [windows]
  curr_windowid display_boxed show_statusline show_mainmenu show_helpline statusline_at_top screen_rows screen_cols mouse_enabled
  line row pos xoffset windowid x1 y1 width height filename
  [block]
  type begline begcol endline endcol filename
  [modes]
  insert auto wordwrap rightmargin leftmargin tabtype tabwidth backups state eoltype zoomed maximized line-numbers
  [keymac]
  filename-of-keymac-file (less extension)

  New in 1.3:
    files:
        1.2 order
        line row pos xoffset filename binary_mode display_mode
        filename=5, binary_mode=6, display_mode=7

        1.3 order
        line row pos xoffset binary_mode display_mode filename
        binary_mode=5, display_mode=6, filename=7

        'filename' is moved to the end.

    modes: added maximized, line-numbers

  In 1.2:
  files:

  Copyright 1995-1997 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

 ******************************************************************************/

dll "<user32.dll>"
    integer proc IsMaximized0(integer hwnd) : "IsZoomed"
end

constant MAXPATH = 255

constant
    FILES_STATE =           0x01,
    BOOKMARKS_STATE =       0x02,
    WINDOWS_STATE =         0x04,
    BLOCK_STATE =           0x08,
    MODES_STATE =           0x10,
    KEYMAC_STATE =          0x20,
    FONT_STATE =            0x40,
    ALL_STATE = FILES_STATE | BOOKMARKS_STATE | WINDOWS_STATE | BLOCK_STATE | MODES_STATE | KEYMAC_STATE | FONT_STATE,

    SAVE_STATE =    1,
    RESTORE_STATE = 2,
    DELETE_STATE  = 3,
    MAX_WINDOWS = 20

integer state_buffer, curr_id

string
    raw_fn[] = "tsestate.dat",
    state_version[] = "[state v1.5 Sep 09 2017];[state v1.4 Dec 21 2010];[state v1.3 Aug 15 2006];[state v1.2 Jul 15 1997]",
    files_str[] = "[files]",
    bookmarks_str[] = "[bookmarks]",
    windows_str[] = "[windows]",
    block_str[] = "[block]",
    modes_str[] = "[modes]",
    keymac_str[] = "[keymac]",
    font_str[] = "[font]"

proc Error(string msg)
    if curr_id
        GotoBufferId(curr_id)
        AbandonFile(state_buffer)
    endif
    Warn(msg)
    Halt
end

integer proc IsMaximized(integer hwnd)
    if WhichOS() == _LINUX_
        return (FALSE)
    endif
    return (IsMaximized0(hwnd))
end

// BOOLEAN: Set new screen rows
integer proc SetNewRowsCols(integer rows, integer cols)
    SetVideoRowsCols(rows, cols)
    return (Query(ScreenRows) == rows and Query(ScreenCols) == cols)
end

string proc QuotedCurrFilename()
    return (QuotePath(CurrFilename()))
end

// create state buffer, set global curr_id, leave us in state buffer
proc CreateStateBuffer()
    curr_id = GetBufferId()
    state_buffer = CreateTempBuffer()
    if state_buffer == 0
        Error("Can't create state buffer")
    endif
    BinaryMode(-2)
end

integer proc GotoStateBuffer()
    return (GotoBufferId(state_buffer))
end

proc SaveWindowInfo()
    if Length(CurrFilename()) == 0
        Error("Can't save state: no filename!")
    endif
    AddLine(Format(CurrLine(); CurrRow(); CurrPos(); CurrXOffset();
            WindowId(); Query(WindowX1); Query(WindowY1);
            Query(WindowCols); Query(WindowRows); QuotedCurrFilename()),
            state_buffer)
end

proc AddStateId(string state_id)
    GotoBufferId(curr_id)
    AddLine(state_id, state_buffer)
end

proc GetZoomed(integer zoomed)
    if zoomed
        ZoomWindow()
    endif
end

/*------------------------------------------------------------------------
  Save State main routine

  ShowLineNumbers is a little tricky.  In order to handle it
  properly:
  * save the current value of ShowLineNumbers
  * set it off
  * build the state file
  * after state file is built, restore ShowLineNumbers value

 ------------------------------------------------------------------------*/
proc SaveState(integer to_save, string state_fn)
    string keymac_fn[MAXPATH], font_name[64] = ""
    integer
        i, cwid, id, line, cpos, xoff, row, zoomed, msglevel, show_line_numbers,
        point_size, font_flags

    show_line_numbers = Set(ShowLineNumbers, off)

    zoomed = isZoomed()
    if zoomed
        ZoomWindow()
    endif

    // close windows with unnamed buffers
    if (to_save & WINDOWS_STATE) and NumWindows() > 1
        id = WindowId()
        for i = 1 to MAX_WINDOWS
            if GotoWindow(i) and Length(CurrFilename()) == 0
                CloseWindow()
                if i == id
                    id = WindowId()
                endif
            endif
        endfor
        GotoWindow(id)
    endif

    CreateStateBuffer()
    InsertText(GetToken(state_version, ';', 1))

    if to_save & FONT_STATE
        AddStateId(font_str)
        GetFont(font_name, point_size, font_flags)
        AddLine(Format(QuotePath(font_name); point_size; font_flags), state_buffer)
    endif

    if to_save & MODES_STATE
        // save modes info
        AddStateId(modes_str)
        AddLine(Format(Query(Insert); Query(AutoIndent); Query(WordWrap);
                Query(RightMargin); Query(LeftMargin); Query(TabType); Query(TabWidth);
                Query(MakeBackups); Query(SaveState); Query(EOLType); zoomed;
                iif(Query(CurrVideoMode) == _MAXIMIZED_, 1, 0); show_line_numbers),
                state_buffer)
    endif

    if to_save & WINDOWS_STATE and CurrFilename() <> ""
        // save windows info
        AddStateId(windows_str)
        AddLine(Format(WindowId(); Query(DisplayBoxed);
                Query(ShowStatusLine); Query(ShowMainMenu);
                Query(ShowHelpLine); Query(StatusLineAtTop);
                Query(ScreenRows); Query(ScreenCols); Query(MouseEnabled)), state_buffer)
        cwid = WindowId()
        for i = 1 to MAX_WINDOWS
            GotoWindow(i)
            if WindowId() == i and WindowId() <> cwid
                SaveWindowInfo()
            endif
        endfor
        GotoWindow(cwid)
        SaveWindowInfo()
    endif

    if to_save & FILES_STATE
        // save loaded files info
        AddStateId(files_str)
        do NumFiles() + (BufferType() <> _NORMAL_) times
            if Length(CurrFilename())
                AddLine(Format(CurrLine();
                               CurrRow();
                               CurrPos();
                               CurrXOffset();
                               BinaryMode();
                               DisplayMode();
                               BrowseMode();
                               QuotedCurrFilename()), state_buffer)
            endif
            PrevFile(_DONT_LOAD_)
        enddo
    endif

    if to_save & BOOKMARKS_STATE
        // save bookmark info
        AddStateId(bookmarks_str)
        for i = Asc('a') to Asc('z')
            if isBookMarkSet(Chr(i))
                GetBookMarkInfo(Chr(i), id, line, cpos, xoff, row)
                if GotoBufferId(id)
                    AddLine(Format(Chr(i); line; row; cpos; xoff; QuotedCurrFilename()), state_buffer)
                endif
            endif
        endfor
    endif

    if to_save & BLOCK_STATE and isBlockMarked()
        // save block info
        PushPosition()
        AddStateId(block_str)
        GotoBufferId(Query(BlockId))
        AddLine(Format(isBlockMarked(); Query(BlockBegLine); Query(BlockBegCol);
                Query(BlockEndLine); Query(BlockEndCol); QuotedCurrFilename()), state_buffer)
        PopPosition()
    endif

    if to_save & KEYMAC_STATE
        // save keyboard macros
        keymac_fn = SplitPath(state_fn, _DRIVE_|_NAME_)
        msglevel = Set(MsgLevel, _NONE_)
        if SaveKeyMacro(keymac_fn)
            AddStateId(keymac_str)
            AddLine(keymac_fn, state_buffer)
        endif
        Set(MsgLevel, msglevel)
    endif

    // save the state buffer
    GotoStateBuffer()
    SaveAs(state_fn, _OVERWRITE_)
    GotoBufferId(curr_id)
    AbandonFile(state_buffer)
    GetZoomed(zoomed)
    Set(ShowLineNumbers, show_line_numbers)
    Set(MacroCmdLine, "true")
end

integer proc mEditFile(string switches, string fn, integer flag)
    return (EditFile(iif(switches == "", "", " ") + QuotePath(fn), flag))
end

integer proc SearchForState(string state_str)
    GotoStateBuffer()
    return (lFind(state_str, "g^"))
end

integer proc GetString(var string s)
    s = GetText(1, CurrLineLen())
    return (s[1] <> "[")
end

integer proc verify_version(var string state_version_no)
    integer i
    string s[255]

    state_version_no = ""
    for i = 1 to NumTokens(state_version, ';')
        s = GetToken(state_version, ';', i)
        if GetText(1, Length(s)) == s
            s = GetToken(s, ' ', 2)
            state_version_no = DelStr(s, 1, 1)
            break
        endif
    endfor

    return (state_version_no in "1.5", "1.4", "1.3", "1.2")
end

proc fix_output_window()
    Set(Attr, Query(TextAttr))
    clrscr()
    SignOn()
end

//*** Restore State main routine **********************************
proc RestoreState(integer to_restore, string state_fn)
    string fn[MAXPATH], s[MAXPATH] = "only to keep sc happy", tmp_fn[MAXPATH], font_name[64], font_name2[64] = ""
    string state_version_no[5] = ""
    integer ok, curr_win_id, w, x1, y1, cols, rows, mouse
    integer line, row, cpos, xoff, binary_mode = 0, display_mode = 0, browse_mode = 0
    integer block_type, msglevel
    integer wins_restored = 0, first_id = 0, zoomed = 0, maximized = 0
    integer window_error = FALSE
    integer new_file_flag = FALSE
    integer show_line_numbers
    integer buf_id
    integer point_size, font_flags, point_size2, font_flags2

    CreateStateBuffer()
    // load the state buffer
    PushBlock()
    ok = InsertFile(state_fn, _DONT_PROMPT_)
    PopBlock()
    if not ok or NumLines() <= 1
        Error("Empty state file")
    endif

    if not verify_version(state_version_no)
        Error("State file version mismatch")
    endif

    GotoBufferId(curr_id)
    OneWindow()
    GotoStateBuffer()

    if to_restore & MODES_STATE
        // Restore modes
        if SearchForState(modes_str) and Down() and GetString(s)
            Set(Insert,     Val(GetToken(s, ' ', 1)))
            Set(AutoIndent, Val(GetToken(s, ' ', 2)))
            Set(WordWrap,   Val(GetToken(s, ' ', 3)))
            Set(RightMargin,Val(GetToken(s, ' ', 4)))
            Set(LeftMargin, Val(GetToken(s, ' ', 5)))
            Set(TabType,    Val(GetToken(s, ' ', 6)))
            Set(TabWidth,   Val(GetToken(s, ' ', 7)))
            Set(MakeBackups,Val(GetToken(s, ' ', 8)))
            Set(SaveState,  Val(GetToken(s, ' ', 9)))
            Set(EOLType,    Val(GetToken(s, ' ',10)))
            zoomed = Val(GetToken(s, ' ', 11))

            if state_version_no in "1.5", "1.4", "1.3"
                maximized = Val(GetToken(s, ' ', 12))
                show_line_numbers = Val(GetToken(s, ' ', 13))
            endif
        endif
    endif

    if to_restore & FILES_STATE
        // load any files found
        if SearchForState(files_str)
            while Down() and GetString(s)
                line = Val(GetFileToken(s, 1))
                row =  Val(GetFileToken(s, 2))
                cpos = Val(GetFileToken(s, 3))
                xoff = Val(GetFileToken(s, 4))

                if state_version_no in "1.5", "1.4"
                    binary_mode  = Val(GetFileToken(s, 5))
                    display_mode = Val(GetFileToken(s, 6))
                    browse_mode  = Val(GetFileToken(s, 7))

                    fn = GetFileToken(s, 8)
                elseif state_version_no == "1.3"
                    binary_mode  = Val(GetFileToken(s, 5))
                    display_mode = Val(GetFileToken(s, 6))

                    fn = GetFileToken(s, 7)
                elseif state_version_no == "1.2"
                    fn = GetFileToken(s, 5)

                    binary_mode  = Val(GetFileToken(s, 6))
                    display_mode = Val(GetFileToken(s, 7))
                endif

                if StrFind("^\<unnamed-[0-9]+\>$", fn, "x")
                    //??? 12 May 2017  3:49 pm SEM - this is where unwanted <unamed-nnn> files created
                    new_file_flag = True
                    if GetBufferId(fn) == 0
                        PushLocation()
                        NewFile()
                        tmp_fn = fn
                        fn = CurrFilename()
                        PopLocation()
                        lReplace(tmp_fn, fn, "gn")
                    endif
                endif
                Message("Loading ", fn)

                if binary_mode and GotoBufferId(GetBufferId(fn))
                    if not FileChanged()
                        AbandonFile(GotoStateBuffer())
                    else
                        binary_mode = 0
                    endif
                endif

/*------------------------------------------------------------------------
  Ideally, <unnamed-n> type files should be loaded via the
  AddFileToRing() code below.  But AddFileToRing() adds the
  current path, and we don't want that.  So, we'll just load it
  via EditFile().
 ------------------------------------------------------------------------*/
                if new_file_flag
                    buf_id = EditThisFile(fn)
                elseif line <= 1 and row <= 1 and cpos <= 1 and xoff == 0
                    buf_id = mEditFile(Format("-b", binary_mode), fn, _DONT_PROMPT_|_DONT_LOAD_)
                else
                    buf_id = mEditFile(Format("-b", binary_mode), fn, _DONT_PROMPT_)
                    if buf_id <> 0
                        DisplayMode(display_mode)
                        GotoLine(line)
                        ScrollToRow(row)
                        GotoPos(cpos)
                        GotoXOffset(xoff)
                    endif
                endif

                if first_id == 0 and buf_id <> 0
                    first_id = GetBufferId()
                endif

                if browse_mode
                    BrowseMode(true)
                endif

                GotoStateBuffer()
            endwhile

        endif
    endif

    if to_restore & BOOKMARKS_STATE
        // set any bookmarks
        if SearchForState(bookmarks_str)
            Message("Restoring bookmarks")
            while Down() and GetString(s)
                mEditFile("", GetFileToken(s, 6), _DONT_PROMPT_)

                PushPosition()
                GotoLine(Val(GetToken(s, ' ', 2)))
                ScrollToRow(Val(GetToken(s, ' ', 3)))
                GotoPos(Val(GetToken(s, ' ', 4)))
                GotoXOffset(Val(GetToken(s, ' ', 5)))
                PlaceMark(GetToken(s, ' ', 1))
                PopPosition()

                GotoStateBuffer()
            endwhile

        endif
    endif

    if (to_restore & BLOCK_STATE) and not isBlockMarked()
        // restore block
        if SearchForState(block_str) and Down() and GetString(s)
            block_type = Val(GetToken(s, ' ', 1))
            if mEditFile("", GetFileToken(s, 6), _DONT_PROMPT_)
                PushPosition()
                GotoLine(Val(GetToken(s, ' ', 2)))
                if block_type == _COLUMN_
                    GotoColumn(Val(GetToken(s, ' ', 3)))
                else
                    GotoPos(Val(GetToken(s, ' ', 3)))
                endif
                Mark(block_type)

                GotoLine(Val(GetToken(s, ' ', 4)))
                if block_type == _COLUMN_
                    GotoColumn(Val(GetToken(s, ' ', 5)))
                else
                    GotoPos(Val(GetToken(s, ' ', 5)))
                endif

                if block_type == _NONINCLUSIVE_     //02/07/97 need this to not skip marking last char
                   Right()
                endif

                Mark(block_type)
                PopPosition()
            endif
        endif
    endif

    if to_restore & FONT_STATE
        if SearchForState(font_str) and Down() and GetString(s)
            font_name  = GetFileToken(s, 1)
            point_size = Val(GetFileToken(s, 2))
            font_flags = Val(GetFileToken(s, 3))
            GetFont(font_name2, point_size2, font_flags2)
            if font_name2 <> font_name or point_size2 <> point_size or font_flags2 <> font_flags
                Setfont(font_name, point_size, font_flags)
            endif
        endif
    endif

    if to_restore & WINDOWS_STATE
        // restore windows
        if SearchForState(windows_str) and Down() and GetString(s)
            curr_win_id = Val(GetToken(s, ' ', 1))
            Set(DisplayBoxed, Val(GetToken(s, ' ', 2)))
            Set(ShowStatusLine, Val(GetToken(s, ' ', 3)))
            Set(ShowMainMenu, Val(GetToken(s, ' ', 4)))
            Set(ShowHelpLine, Val(GetToken(s, ' ', 5)))
            Set(StatusLineAtTop, Val(GetToken(s, ' ', 6)))
            OneWindow()

            rows = Val(GetToken(s, ' ', 7))
            cols = Val(GetToken(s, ' ', 8))
            mouse = Val(GetToken(s, ' ', 9))

            if (IsMaximized(GetWinHandle()) and not maximized) or
                    (not IsMaximized(GetWinHandle()) and maximized)
                ExecMacro("togmax")
            endif

            if rows <> Query(ScreenRows) or cols <> Query(ScreenCols)
                SetNewRowsCols(rows, cols)
            endif

            if rows <> Query(ScreenRows)
                fix_output_window()
                Warn("Can't restore windows: saved screen height (", rows, ") current max (", Query(ScreenRows), ").")
                window_error = TRUE
            elseif cols <> Query(ScreenCols)
                fix_output_window()
                Warn("Can't restore windows: saved screen width (", cols, ") current max (", Query(ScreenCols), ").")
                window_error = TRUE
            elseif mouse <> Query(MouseEnabled)
                fix_output_window()
                Warn("Can't restore windows: mouse state changed")
                window_error = TRUE
            endif

            if not window_error
                ClearEditWindows()

            // First pass: build window structure
            // line-1 row-2 pos-3 xoff-4 wid-5 x1-6 y1-7 width-8 height-9 name-10
                while Down() and GetString(s)
                    w    = Val(GetToken(s, ' ', 5))
                    x1   = Val(GetToken(s, ' ', 6))
                    y1   = Val(GetToken(s, ' ', 7))
                    cols = Val(GetToken(s, ' ', 8))
                    rows = Val(GetToken(s, ' ', 9))

                    if w <> curr_win_id
                        x1 = x1 - 1
                        y1 = y1 - 1
                        cols = cols + 2
                        rows = rows + 2
                        if y1 + rows + 1 < Query(ScreenRows) and
                                Query(DisplayBoxed) and Query(MouseEnabled)
                            rows = rows + 1
                        endif
                    endif

                    MakeEditWindow(w, x1, y1, cols, rows)

                endwhile

                FinishEditWindows()
            endif

        // Second pass: Place files in windows
            SearchForState(windows_str)
            Down()

            loop
                PushPosition()

                GotoStateBuffer()
                if Down() and GetString(s)
                    PopPosition()
                else
                    PopPosition()
                    break
                endif

                if not window_error
                    GotoWindow(Val(GetToken(s, ' ', 5)))
                endif
                mEditFile("", GetFileToken(s, 10), _DONT_PROMPT_)
                GotoLine(Val(GetToken(s, ' ', 1)))
                ScrollToRow(Val(GetToken(s, ' ', 2)))
                GotoPos(Val(GetToken(s, ' ', 3)))
                GotoXOffset(Val(GetToken(s, ' ', 4)))

                wins_restored = GetBufferId()

            endloop

            GotoWindow(curr_win_id)
            UpdateDisplay()

        endif
    endif

    if to_restore & KEYMAC_STATE
        // restore keyboard macros
        if SearchForState(keymac_str) and Down() and GetString(s)
            msglevel = Set(MsgLevel, _NONE_)
            LoadKeyMacro(s)
            Set(MsgLevel, msglevel)
        endif
    endif

    if not wins_restored or window_error
        GotoBufferId(iif(first_id, first_id, curr_id))
        mEditFile("", CurrFilename(), 0)
    endif
    AbandonFile(state_buffer)
    GetZoomed(zoomed)
    if show_line_numbers
        Set(ShowLineNumbers, on)
    endif
    Set(MacroCmdLine, "true")

    if NumFiles() == 0
        fix_output_window()
    endif
end

menu WhereMenu()
    "Use &Global state file"
    "Use &Local state file"
    "Specify a state &Filename"
end

menu StateMenu()
    "&Save State"
    "&Restore State"
    "", , DIVIDE
//    "State &Filename" [raw_fn], GetStateFn(), _DONT_CLOSE_
//    "&Directory"      [state_dir], GetStateDir(), _DONT_CLOSE_
//
    "&Cancel"
end

proc main()
    constant c_global = 1, c_local = 2
    integer i, action, quiet, save_where
    string s[MAXPATH], state_fn[MAXPATH], cmdline[MAXPATH] = Query(MacroCmdLine)

    Set(MacroCmdLine, "false")      // assume failure

    action = 0
    quiet = 0
    state_fn = ""
    save_where = iif(Query(SaveState) == _LOCAL_, c_local, c_global)
    // parse the command line
    for i = 1 to NumFileTokens(cmdline)
        s = GetFileToken(cmdline, i)
        case s[1:2]
            when "-g"   save_where = c_global
            when "-l"   save_where = c_local
            when "-s"   action = SAVE_STATE
            when "-r"   action = RESTORE_STATE
            when "-f"
                if Length(s) > 2
                    state_fn = s[3:sizeof(state_fn)]
                else
                    s = GetFileToken(cmdline, i + 1)
                    if s <> "" and s[1] <> "-"
                        state_fn = s
                        i = i + 1
                    endif
                endif
            when "-q"   quiet = TRUE
            when "-d"   action = DELETE_STATE
        endcase
    endfor

    // if no action specified, ask the user
    if action == 0
        case StateMenu()
            when 0 return ()
            when 1 action = SAVE_STATE
            when 2 action = RESTORE_STATE
        endcase

        case WhereMenu()
            when 0 return ()
            when 1 /* global = true */  save_where = c_global
            when 2 /* local = true  */  save_where = c_local
            when 3 /* get_filename = true */
                if not AskFilename("State Filename:", state_fn, _FULL_PATH_) or length(state_fn) == 0
                    return ()
                endif
        endcase
    endif

    // make sure what have a state file to save/restore
    if state_fn == ""
        state_fn = iif(save_where == c_local, raw_fn, LoadDir() + raw_fn)
    else
        if save_where == c_global and SplitPath(state_fn, _PATH_) == ""
            state_fn = LoadDir() + state_fn
        endif
    endif

    case action
        when SAVE_STATE
            SaveState(ALL_STATE, state_fn)

        when RESTORE_STATE
            if FileExists(state_fn)
                RestoreState(ALL_STATE, state_fn)
            elseif not quiet
                Warn("State file not found: ", state_fn)
            endif

        when DELETE_STATE
            EraseDiskFile(state_fn)
    endcase
    PurgeMacro(CurrMacroFilename())
end

