/*************************************************************************
  SL          Enables multiple statuslines when multiple windows displayed

  Author:     SemWare

  Date:       Feb 1997
              Feb 20, 1988 SEM: fix pbms with non-bordered single window
              Oct 12, 2000 SEM: fix pbms with non-bordered single window
              Mar  8, 2001 SEM: Add 'B' for Browse Mode - thanks to Bill Stewart
                    for bringing it to our attention
              Jul 16, 2018 SEM: Adjust for line numbers.
              Jul 06, 2019 SEM: Use built-in function to adjust for line numbers

  Overview:

  This macro allows the editor to display a statusline for each window
  when multiple windows are open.  To use it, select it from the
  Potpourri, or run "sl" from the ExecMacro command (default:
  <ctrl x>)

  Copyright 1992-2001 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

*************************************************************************/

/**************************************************************************
  Replace window title lines with a statusline for each window

  Requires statusline on TOP
 **************************************************************************/

integer
    last_numwins = 1,           // 10/12/2000 SEM - This MUST be initialized to 1, otherwise, macro fails
    statusline_taken,
    command_count,
    last_update_count,
    new_file_window,
    last_rows, last_cols,
    this_nwins

proc UpdateStateVars()
    last_rows = Query(ScreenRows)
    last_cols = Query(ScreenCols)
    this_nwins = NumWindows()
end

integer proc isDisplayBoxed()
    return (isZoomed() or NumWindows() > 1 or Query(DisplayBoxed) == 2 or
            (Query(DisplayBoxed) and Query(MouseEnabled)))
end

string NEWFILE_MSG[] = " Editing new file: "
/**************************************************************************
  Display a statusline for a file, even if windowed
 **************************************************************************/
proc ShowStatusLine(integer win)
    constant BASEPART = 24
    string sl[255], fn[255], locked_changed[2], line_col[25]
    integer cols, startcol, endcol, adj
#if 0
    integer nwins, zoomed, mouse,
#endif

    if Length(CurrFilename()) == 0
        return ()
    endif
#if 0
    zoomed = isZoomed()
    nwins = NumWindows()
    mouse = Query(MouseEnabled)
#endif

    adj = GetLineNumberLength()
    startcol = 1
    endcol = Query(WindowCols) - 1
    endcol = Min(endcol, sizeof(sl))

    sl = Format(" ":endcol)

    if win == new_file_window
        new_file_window = 0
        sl[1:Length(NEWFILE_MSG) + Length(CurrFilename())] = NEWFILE_MSG + CurrFilename()
        goto show_what_we_got
    endif

#if 0
    if nwins > 1
        if mouse
            sl[1:3] = "[x]"
            startcol = 5
        endif
        if zoomed
            if mouse
                sl[endcol - 2:3] = "[]"
                endcol = endcol - 4
            else
                sl[endcol] = "Z"
                endcol = endcol - 2
            endif
        else
            sl[endcol - 2:3] = "[]"
            endcol = endcol - 4
            sl[endcol] = Str(WindowId())
            endcol = endcol - 2
              NoOp()
        endif
    endif
#endif

    cols = endcol - startcol + 1

    locked_changed = iif(query(bufferflags) & (_READONLY_LOCKING_ +
                         _HANDLE_LOCKING_),"+"," ") +
                         iif(FileChanged(), '*', ' ')
    fn = CurrFilename()

    if cols - BASEPART >= Length(fn) + sizeof(locked_changed)
        sl[startcol:cols] = Format(
                "L ",
                CurrLine():-adj,
                "C ",
                CurrCol():-5,
                iif(Query(Insert), "I", " "),
                iif(Query(AutoIndent), "A", " "),
                iif(Query(WordWrap), "W", " "),
                iif(BrowseMode(), "B", iif(Query(LineDrawing), "L", " ")),
                iif(Query(KbdMacroRecording), "R", " "),
                "   ",
                locked_changed + fn)
    else
        if cols < Length(fn) + sizeof(locked_changed)
            fn = SplitPath(fn, _DRIVE_) + SplitPath(fn, _NAME_|_EXT_)
            if cols < length(fn) + sizeof(locked_changed)
                fn = SplitPath(fn, _NAME_|_EXT_)
                if cols < Length(fn) + sizeof(locked_changed)
                    fn = fn[1:(cols - sizeof(locked_changed) - 3)] + "..."
                endif
            endif
        endif

        sl[endcol - Length(fn) - sizeof(locked_changed) + 1:Length(fn) +
            sizeof(locked_changed)] = locked_changed + fn
        cols = cols - Length(fn) - sizeof(locked_changed)
        line_col = Format("[", CurrLine(), ":", CurrCol(), "]")
        if cols > Length(line_col)
            sl[startcol:Length(line_col)] = line_col
        endif
    endif

    show_what_we_got:
    VGotoXYAbs(Query(WindowX1) - adj, Query(WindowY1) - 1)
    PutStr(sl)
    PutAttr(Query(StatusLineAttr), adj + 1)
end

/**************************************************************************
  Use our statusline instead of the standard one
 **************************************************************************/
proc AfterUpdateStatusline()
    integer i, cw, attr
#if 0   // #ifdef WIN32
    string c[1], a[1]
#endif

    if isDisplayBoxed() and
        (NumWindows() <> this_nwins or last_rows <> Query(ScreenRows) or last_cols <> Query(ScreenCols) or
        (statusline_taken and last_update_count <> command_count))
#if 0 // #def WIN32
    doit_anyway:
#endif
        last_update_count = command_count
        cw = WindowId()
        attr = Set(Attr, Query(StatusLineAttr))
        for i = 1 to 9
            GotoWindow(i)
            if WindowId() == i
                ShowStatusLine(i)
            endif
        endfor
        Set(Attr, attr)
        GotoWindow(cw)
        UpdateStateVars()
    else
#if 0 // #ifdef WIN32
        c = ''
        a = ''
        VGotoXYAbs(Query(WindowX1), Query(WindowY1) - 1)
        GetStrAttr(c, a, 1)
        if Asc(a[1]) <> Query(StatusLineAttr) and Asc(a[1]) <> Query(MsgAttr)
            goto doit_anyway
        endif
#endif
    endif
end

proc GrabStatusline()
    if not statusline_taken
        Set(ShowStatusLine, off)
        Set(StatusLineUpdating, off)
        Hook(_AFTER_UPDATE_STATUSLINE_, AfterUpdateStatusline)
        statusline_taken = TRUE
    endif
end

proc ReleaseStatusline()
    if statusline_taken
        Set(ShowStatusLine, on)
        Set(StatusLineUpdating, on)
        UnHook(AfterUpdateStatusline)
        statusline_taken = FALSE
    endif
end

// BOOLEAN: was 2nd window created by compiler?
integer proc isCompileMac()
    integer ok, id

    PushPosition()
    id = WindowId()
    GotoWindow(2)
    ok = Length(CurrFilename()) == 0
    GotoWindow(id)
    PopPosition()
    return (ok)
end

/**************************************************************************
  Trap window changes, and other changes that might cause us to gain or
  release statusline control.
 **************************************************************************/
proc AfterCommand()
    integer cmd, window_rows

    command_count = command_count + 1
    if NumWindows() > 1
        if not statusline_taken and last_numwins == 1 and NumWindows() == 2 and not isCompileMac()
            window_rows = Query(WindowRows)
            if Query(ShowStatusLine)
                window_rows = window_rows + 1
            endif
            if Query(ShowHelpLine)
                window_rows = window_rows + 1
            endif
            if Query(ShowMainMenu)
                window_rows = window_rows + 1
            endif

            cmd = Query(ScreenRows) - window_rows

            GrabStatusline()
            PrevWindow()
            OneWindow()
            if cmd == 1
                VWindow()
            else
                HWindow()
            endif
        endif
    else
        if isDisplayBoxed()
            if not statusline_taken
                GrabStatusline()
                OneWindow()
            endif
        else
            if statusline_taken
                ReleaseStatusline()
                OneWindow()
            endif
        endif
    endif
    last_numwins = NumWindows()
end

/**************************************************************************
  Use this hook to trap "Editing new file".
  If we do not trap it, our statusline does not get painted immediately
  after a new file has been edited.
 **************************************************************************/
proc AfterFirstEdit()
    if Query(BufferFlags) & _NEWFILE_ and NumLines() == 0
        UpdateDisplay()
        new_file_window = WindowId()
    endif
end

proc WhenLoaded()
    if Query(StatusLineAtTop) == FALSE
        PurgeMacro(CurrMacroFilename())
        Warn("StatusLine must be at top to use this macro")
        return ()
    endif

    Hook(_AFTER_COMMAND_, AfterCommand)
    Hook(_ON_FIRST_EDIT_, AfterFirstEdit)
    OneWindow()                         // we can ONLY start with one-window
    UpdateStateVars()
    if isDisplayBoxed()
        GrabStatusline()
        last_update_count = -1
        Set(CurrWinBorderType, Query(CurrWinBorderType))
    endif
end

proc WhenPurged()
    ReleaseStatusline()
    OneWindow()
    UpdateDisplay()
end

